/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <sys/types.h>
#include <sys/file.h>
#ifdef ultrix
#include <fcntl.h>
#else
#include <sys/fcntl.h>
#endif
#include <stdio.h>
#include <unistd.h>
#include "audio.h"
#include "rplayd.h"
#include "spool.h"
#include "sound.h"
#include "ulaw.h"
#ifdef __386BSD__
#include <machine/sblast.h>
#endif
#ifdef sgi
#include <audio.h>
#endif

#undef MAX
#define MAX(a,b)	((a)>(b)?(a):(b))

#ifdef sgi
static ALport	audio_fd = (ALport)-1;
static ALconfig	sgitheconfig;
static short	audio_buf[AUDIO_BUFSIZE];
#else
static int	audio_fd = -1;
static char	audio_buf[AUDIO_BUFSIZE];
#endif
static int	audio_size;
int		audio_timeout = AUDIO_TIMEOUT;

/*
 * initialize the audio device
 */
void	audio_init()
{
#ifdef sgi
	long	sgisampfmt	= AL_SAMPLE_16;
	long	sginchannels	= AL_MONO;
	long	sgipvbuf[4];
	long	sgipvbuflen;

	sgipvbuf[0] = AL_INPUT_RATE;
	sgipvbuf[1] = AL_RATE_8000;
	sgipvbuf[2] = AL_OUTPUT_RATE;
	sgipvbuf[3] = AL_RATE_8000;
	sgipvbuflen = 4;

	ALsetparams(AL_DEFAULT_DEVICE, sgipvbuf, sgipvbuflen);

	sgitheconfig = ALnewconfig();
	ALsetwidth(sgitheconfig, sgisampfmt);
	ALsetchannels(sgitheconfig, sginchannels);
#endif
}

/*
 * open the audio device
 */
void	audio_open()
{
	if (debug)
	{
		report("opening %s\n", AUDIO_DEVICE);
	}
#ifdef sgi
	audio_fd = ALopenport("rplay", "w", sgitheconfig);
	if ((int)audio_fd < 0 && debug)
#else
	audio_fd = open(AUDIO_DEVICE, O_WRONLY|O_NDELAY, 0);
	if (audio_fd < 0 && debug)
#endif
	{
		report("%s open: %s\n", AUDIO_DEVICE, sys_errlist[errno]);
	}
}

/*
 * close the audio device
 */
void	audio_close()
{
#ifdef sgi
	if ((int)audio_fd != -1)
	{
		ALcloseport(audio_fd);
	}
	audio_fd = (ALport)-1;
#else
	if (audio_fd > 0)
	{
		if (debug)
		{
			report("closing %s\n", AUDIO_DEVICE);
		}
		close(audio_fd);
	}
	audio_fd = -1;
#endif
}

/*
 * write a buffer from the spool to the audio device
 */
void	audio_write()
{
	int		i, n;
	int		total, val, vol;
	SPOOL		*sp;

	while (audio_size != AUDIO_BUFSIZE && spool_size)
	{
		total = 0;
		for (i = 0; i < SPOOL_SIZE && audio_size != AUDIO_BUFSIZE; i++)
		{
			sp = &spool[i];
			if (sp->state == SPOOL_PLAY)
			{
				vol = sp->curr_attrs->volume;
				if (spool_prio && spool_size > 1)
				{
					vol -= (spool_prio - sp->rp->priority) >> 1;
					vol = MAX(vol, RPLAY_MIN_VOLUME);
				}
				val = ulaw_to_linear((unsigned char)*sp->ptr)*vol;
				total += val >> 7;
				if (sp->ptr == sp->end)
				{
					if (sp->curr_count > 1)
					{
						sp->curr_count--;
					}
					else
					{
						sp->curr_sound++;
						if (sp->curr_sound == sp->rp->nsounds)
						{
							if (sp->list_count > 1)
							{
								sp->list_count--;
								sp->curr_attrs = sp->rp->attrs;
								sp->curr_sound = 0;
							}
							else
							{
								spool_clear(sp);
								spool_size--;
								continue;
							}
						}
						else
						{
							sp->curr_attrs = sp->curr_attrs->next;
						}
						sp->curr_count = sp->curr_attrs->count;
					}
					sp->ptr = sp->sound[sp->curr_sound]->start;
					sp->end = sp->sound[sp->curr_sound]->stop;
				}
				else
				{
					sp->ptr++;
				}
			}
		}
#ifdef __386BSD__
		/*
		 * convert signed short to unsigned 8 bit value.
		 */
		audio_buf[audio_size++] = (total ^ 0x8000) >> 8;
#else
#ifdef sgi
		audio_buf[audio_size++] = total;
#else
		audio_buf[audio_size++] = linear_to_ulaw(total);
#endif
#endif
	}
#ifdef sgi
	if ((int)audio_fd < 0 && audio_size)
#else
	if (audio_fd < 0 && audio_size)
#endif
	{
		audio_open();
	}
#ifdef sgi
	if ((int)audio_fd > 0)
#else
	if (audio_fd > 0)
#endif
	{
#ifdef sgi
		ALwritesamps(audio_fd, audio_buf, audio_size);
#else /* sgi */
		n = write(audio_fd, audio_buf, audio_size);
#ifdef __386BSD__
		if (spool_size == 0)
		{
			ioctl(audio_fd, DSP_IOCTL_FLUSH, 0);
		}
#endif /* __386BSD__ */
		if (n < 0 && debug)
		{
			report("write: %s\n", sys_errlist[errno]);
		}
#endif /* sgi */
	}
	audio_size = 0;
}
