;;; -*- Package: Toolset; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolset)

(DEFUN PARSE-FEATURES (RA-INSTANCE) 
  "Transform feature specifications to their executable form:
	(invoke dragon 'verb '(arg arg arg) 'return-value-type)"
  (if (not (slot-empty-p ra-instance 'display-features))
      (LET*
       ((FEATURE-SPEC-VECTOR (SLOT-VALUE RA-INSTANCE 'DISPLAY-FEATURES))
	(EVALABLE-FEATURE-VECTOR (MAKE-ARRAY (LENGTH FEATURE-SPEC-VECTOR)))
	(FEATURE-FETCH-TYPE-VECTOR (MAKE-ARRAY (LENGTH FEATURE-SPEC-VECTOR
						       )))
	(FEATURE-RESULT-TYPE-VECTOR (MAKE-ARRAY (LENGTH 
						 FEATURE-SPEC-VECTOR
						 )))
	EVAL-FORM FEATURE-FETCH-TYPE FEATURE-RESULT-TYPE FEATURE)

       (DOTIMES
	(I (LENGTH FEATURE-SPEC-VECTOR))
	(SETF FEATURE (AREF FEATURE-SPEC-VECTOR I))

	(cond 
	 ((listp feature)
	  (CASE (FIRST FEATURE)
		(ASK-USER
		 (SETF EVAL-FORM 
		       (translate-invoke-form feature))
		 (CASE (LENGTH FEATURE)
		       (2 (SETF FEATURE-RESULT-TYPE 'USUAL-3-VAL))
		       (3 (SETF FEATURE-RESULT-TYPE (THIRD FEATURE)))
		       (OTHERWISE
			(ERROR
			 "ERROR: specification for ASK-USER feature ~S in RA ~S does not have the correct number of arguments." 
			 FEATURE (SLOT-VALUE RA-INSTANCE 'UNIQUE-NAME))))
		 (SETF FEATURE-FETCH-TYPE 'ASK-USER))
		(ASK
		 (setf eval-form (translate-invoke-form feature))
		 (CASE (LENGTH FEATURE)
		       (3 (SETF FEATURE-RESULT-TYPE 'USUAL-3-VAL))
		       (4 (SETF FEATURE-RESULT-TYPE (FOURTH FEATURE)))
		       (OTHERWISE 
			(ERROR "ERROR: specification for ASK feature
				   ~S in RA ~S does not have the correct
				   number of arguments." 
			       FEATURE (SLOT-VALUE
					RA-INSTANCE
					'UNIQUE-NAME))))
		 (SETF FEATURE-FETCH-TYPE 'IDB))
		(JUDGE (setf eval-form
			     (translate-invoke-form feature))
		       (CASE (LENGTH FEATURE)
			     (2 (SETF FEATURE-RESULT-TYPE 'USUAL-9-VAL))
			     (3 (IF (LISTP (THIRD FEATURE))
				    (SETF FEATURE-RESULT-TYPE
					  'USUAL-9-VAL)
				    (SETF FEATURE-RESULT-TYPE 
					  (THIRD FEATURE))))
			     (4 (SETF FEATURE-RESULT-TYPE
				      (fourth FEATURE)))
			     (OTHERWISE 
			      (ERROR "ERROR: specification for JUDGE feature ~S in RA ~S does not have the correct number of arguments." 
				     FEATURE 
				     (SLOT-VALUE RA-INSTANCE 'UNIQUE-NAME))))
		       (SETF FEATURE-FETCH-TYPE 'RA))
		(INVOKE (setf eval-form 
			      (translate-invoke-form feature))
			(CASE (LENGTH FEATURE)
			      (3 (SETF FEATURE-RESULT-TYPE 'USUAL-9-VAL))
			      (OTHERWISE (setf feature-result-type
					       (eval (car (last feature))))))
			(SETF FEATURE-FETCH-TYPE 'RA))
		(number
		 ;; the rest of the feature is a Lisp form or variable
		 ;; which will return a number when evaluated at runtime
		 (setf eval-form (cadr feature)
		       feature-result-type 'number
		       feature-fetch-type 'lisp-form))
		(symbol
		 ;; the rest of the feature is a Lisp form or variable
		 ;; which will return a symbol when evaluated at runtime
		 (setf eval-form (cadr feature)
		       feature-result-type 'symbol
		       feature-fetch-type 'lisp-form))
		(OTHERWISE
		 ;; if the list starts with the name of a confidence set
		 ;; then the rest of it is a lisp form or variable returning
		 ;; a member of that set when evaluated 
		 (if (confidence-set-p (car feature))
		     (setf eval-form (cadr feature)
			   feature-result-type (cadr feature)
			   feature-fetch-type 'lisp-form)
		     ;; if not a confidence set, unknown form -- error
		     (error
		      "ERROR: ~S is not a recognized feature type for an RA."
		      feature)))))
		 
	 ((symbolp feature)
	  (setf feature-fetch-type 'symbol)
	  (setf feature-result-type 'symbol)
	  (setf eval-form `',feature))
	 ((numberp feature)
	  (setf feature-fetch-type 'number)
	  (setf feature-result-type 'number)
	  (setf eval-form feature))
       
       ;; if its not an ASK, ASK-USER, JUDGE, INVOKE or a symbol or number

	 (t (ERROR 
	  "ERROR: ~S does not indicate a recognized feature type for a RA" 
				FEATURE)))
	(SETF (AREF EVALABLE-FEATURE-VECTOR I) EVAL-FORM)
	(SETF (AREF FEATURE-FETCH-TYPE-VECTOR I) FEATURE-FETCH-TYPE)
	(SETF (AREF FEATURE-RESULT-TYPE-VECTOR I) FEATURE-RESULT-TYPE))

       (SETF (SLOT-VALUE RA-INSTANCE 'FEATURES) EVALABLE-FEATURE-VECTOR)
       (SETF (SLOT-VALUE RA-INSTANCE 'FEATURE-FETCH-TYPES)
	     FEATURE-FETCH-TYPE-VECTOR)
       (SETF (SLOT-VALUE RA-INSTANCE 'FEATURE-VALUE-TYPES)
	     FEATURE-RESULT-TYPE-VECTOR))

    ;; if there aren't any features
      (setf (slot-value ra-instance 'features) nil
	    (slot-value ra-instance 'feature-fetch-types) nil
	    (slot-value ra-instance 'feature-value-types) nil)))


(DEFUN PARSE-TEST (TEST expected-feature TEST-NUMBER TRANSFORM) 
                               "parse a test spec into executable form"

   (LET
    ;; the check-feature form will return a value of this type,
    ;; either a number, a symbol, or a member of a confidence set
    ((feature-type (if (confidence-set-p expected-feature)
		       (return-generic-instance expected-feature)
		       expected-feature))
		       
     (CHECK-FEATURE-FORM (IF (NULL transform)
                             `(CHECK-FEATURE ,TEST-NUMBER)
                             `(FETCH-TRANSFORMED-FEATURE ,TEST-NUMBER))))
 
    (COND
       ((SYMBOLP TEST)
       
       ;; ? --> T

        (COND
           ((OR (EQ TEST '?) (EQ TEST '*))
            T)
       
       ;; nil -- > bomb

           ((NULL TEST)
            (ERROR 
  "ERROR NIL is not a legitimate test spec. You probably mean (EQ NIL)"
                   ))
       
       ;; symbol --> (EQ feature (quote symbol))    DOESN'T
       ;; deal with if-any, if-all contexts.

           (T `(EQ ,CHECK-FEATURE-FORM ',TEST))))

       ((NUMBERP TEST)
       
       ;; number --> (= feature number)

	(if (eq feature-type 'number)
	    `(= ,CHECK-FEATURE-FORM ,TEST)
	    (error "A number has been given as a test specification (meaning (= feature number)), but the feature in question is of type ~S, not number."
		   feature-type)))

       ((LISTP TEST)
        (CASE
         (CAR TEST)
       
	 ;; (GT | LT |GE|LE |EQ thing) --> depends on whether the input feature
	 ;; is a number or confidence value, the thing to be compared to the
	 ;; feature might be a number, a confidence value, or a variable
	 ;; name or lisp form returning a number or a confidence value

         ((GE LE GT LT EQ)
          (IF (NOT (= (LENGTH TEST)
                      2))
              (ERROR "ERROR: ~S A single argument must follow the comparative in this context." 
                     TEST))
          (COND
       
       ;;  (GT | LT |GE|LE|EQ number) --> ( >|<|>=|<=|=  feature
       ;; number)

             ((eq feature-type 'number)
	      (if (and (not (NUMBERP (SECOND TEST)))
		       (not (symbolp (second test)))
		       (not (listp (second test))))
		  (error "The argument to a comparison test with a feature of type NUMBER must be either a number or a variable or Lisp form whose value is a number."))
              (CASE (CAR TEST)
		    (EQ `(= ,check-feature-form ,(cadr test)))
                    (GT `(> ,CHECK-FEATURE-FORM ,(CADR TEST)))
                    (LT `(< ,CHECK-FEATURE-FORM ,(CADR TEST)))
                    (GE `(>= ,CHECK-FEATURE-FORM ,(CADR TEST)))
                    (LE `(<= ,CHECK-FEATURE-FORM ,(CADR TEST)))))

             ((confidence-set-p feature-type)
	      
	      ;;  (GT|LT|GE|LE|EQ) confidence-value --> (APPLY
	      ;; special-lambda feature confidence-value) is wrapped
	      ;; inside a form to check aliases
	      
	      ;;   The special-lambda is found in the similarly named
	      ;; slot in the confidence set that has been specified or
	      ;; defaulted.
	      
	      (check-alias-compare-form (car test)
				   check-feature-form
				   (cadr test)
				   feature-type))))

         (RANGE
          (IF (NOT (= (LENGTH TEST)
                      3))
              (ERROR 
          "ERROR: ~S two arguments must be specified for a range test." 
                     TEST))
          (COND
       
       ;; (RANGE number1 number2) ---> (number1<=feature<=number2)

             ((eq feature-type 'number)
	      (if (and (not (NUMBERP (SECOND TEST)))
		       (not (symbolp (second test)))
		       (not (listp (second test)))
		       (not (NUMBERP (third TEST)))
		       (not (symbolp (third test)))
		       (not (listp (third test))))
		  (error "The arguments to a range test with a feature of type NUMBER must be either numbers or variables or Lisp forms whose value is a number."))

	      `(<= ,(SECOND TEST) ,CHECK-FEATURE-FORM
		   ,(THIRD TEST)))

             ((confidence-set-p feature-type)
	      (IF
               (and (MEMBER-CONFIDENCE (second TEST) FEATURE-TYPE)
		    (MEMBER-CONFIDENCE (THIRD TEST) FEATURE-TYPE))
	       ;; if both ends of the range are specified, we can
	       ;; take the range either way, to get lowest<=feature<=highest
	       ;; if not, we have to take the arguments in the order
	       ;; given, as we don't know their runtime values yet
               (IF
		(confidence-compare 'ge (THIRD TEST)
				    (SECOND TEST) feature-type)
		`(and ,(check-alias-compare-form 'le
						 (second test)
						 check-feature-form
						 feature-type)
		      ,(check-alias-compare-form 'le
						 check-feature-form
						 (third test)
						 feature-type))
		`(and ,(check-alias-compare-form 'le
						 check-feature-form
						 (second test)
						 feature-type)
		      ,(check-alias-compare-form 'le
						 (third test)
						 check-feature-form
						 feature-type)))

	       ;; we don't have one or the other of the values to
	       ;; compare yet, so we use them in the order given
	       `(and ,(check-alias-compare-form 'le
						 (second test)
						 check-feature-form
						 feature-type)
		      ,(check-alias-compare-form 'le
						 check-feature-form
						 (third test)
						 feature-type)))
	      (T (ERROR "ERROR: ~S RANGE tests are supported for numbers and confidences only, this feature has a result-type of ~S." 
                       TEST feature-type)))))

         (OTHERWISE 
       
	  ;; check to see if it is a list of symbols, if so make it
	  ;;  (MEMBER feature symbol-list), else error

                (IF (NOT (FIND-IF-NOT #'SYMBOLP TEST))
                    `(MEMBER ,CHECK-FEATURE-FORM ',TEST)
                    (ERROR 
                         "ERROR ~S is not recognized syntax for a test" 
                           TEST)))))
       (T (ERROR "ERROR ~S is not recognized syntax for a test; must be a symbol, number, or list." 
                 TEST)))))


(DEFUN PARSE-CONFIDENCE (CONFIDENCE-SPEC)
   (LET
    NIL
    (COND
       
       ;; if it's nil, return 'NOT-SPECIFIED

       ((NULL CONFIDENCE-SPEC)
        'NOT-SPECIFIED)
       
       ;; if is's a symbol, return it quoted

       ((SYMBOLP CONFIDENCE-SPEC)
        `',CONFIDENCE-SPEC)

       ;; if it's a number (number-valued conf-set), return it

       ((numberp confidence-spec)
	confidence-spec)
       
       ;; if it's a JUDGE , ASK, or INVOKE transform to
       ;; "INVOKE" form
       ;; if it's any other list, just return it

       ((listp confidence-spec)
	(if
	    (member (car confidence-spec) '(judge ask ask-user invoke))
	    (translate-invoke-form confidence-spec)
	    confidence-spec))
       
       ;; else freak out

       (T (ERROR "ERROR unrecognized type of confidence specifier ~S " 
                 CONFIDENCE-SPEC)))))


(DEFUN TRANSFORM-PATTERN-REFERENCES (TREE) "Transforms a form, changing user-visible PATTERN I form of references to patterns to internal form."
   (COND
      ((SYMBOLP TREE)
       TREE)
      ((LISTP TREE)
       (CASE (CAR TREE)
             (PATTERN `(NTH ,(CADR TREE) PATTERNS))
             (OTHERWISE (CONS (TRANSFORM-PATTERN-REFERENCES
                               (CAR TREE))
                              (TRANSFORM-PATTERN-REFERENCES
                               (CDR TREE))))))
      (T TREE)))


(DEFUN TRANSFORM-CONFIDENCE-REFERENCES (TREE) "transforms a form, changing the user-visible CONF form of references to associated confidences into internal form"
   (COND
      ((SYMBOLP TREE)
       TREE)
      ((LISTP TREE)
       (CASE (CAR TREE)
             (CONF `(AREF ASSOCIATED-CONFIDENCES ,(CADR TREE)))
             (OTHERWISE (CONS (TRANSFORM-CONFIDENCE-REFERENCES
                               (CAR TREE))
                              (TRANSFORM-CONFIDENCE-REFERENCES
                               (CDR TREE))))))
      (T TREE)))
