;;;-*-Mode:LISP; Package:(PCL LISP 1000); Base:10; Syntax:Common-lisp -*-
;;;
;;; *************************************************************************
;;; Copyright (c) 1985, 1986, 1987, 1988 Xerox Corporation.
;;; All rights reserved.
;;;
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted.  Any distribution of this
;;; software or derivative works must comply with all applicable United
;;; States export control laws.
;;; 
;;; This software is made available AS IS, and Xerox Corporation makes no
;;; warranty about the software, its performance or its conformity to any
;;; specification.
;;; 
;;; Any person obtaining a copy of this software is requested to send their
;;; name and post office or electronic mail address to:
;;;   CommonLoops Coordinator
;;;   Xerox PARC
;;;   3333 Coyote Hill Rd.
;;;   Palo Alto, CA 94304
;;; (or send Arpanet mail to CommonLoops-Coordinator.pa@Xerox.arpa)
;;;
;;; Suggestions, comments and requests for improvements are also welcome.
;;; *************************************************************************
;;;
;;;
;;; This file defines the initialization and related protocols.
;;; 

(in-package 'pcl)

(defmethod *make-instance ((class standard-class) &rest initargs)
  (setq initargs (*default-initargs class initargs))
  ;;check-initargs
  (let ((instance (apply #'allocate-instance class initargs)))
    (apply #'*initialize-instance instance initargs)
    instance))

;;;
;;; This method definition appears at the end of the file high.lisp
;;; where is is possible to have it.
;;; 
;(defmethod *make-instance ((class-name symbol) &rest initargs)
;  (apply #'*make-instance (find-class class-name) initargs))

(defvar *default-initargs-flag* (list nil))

(defmethod *default-initargs ((class standard-class) supplied-initargs)
  ;; This implementation of default initargs is critically dependent
  ;; on all-default-initargs not having any duplicate initargs in it.
  (let ((all-default (class-all-default-initargs class))
	(miss *default-initargs-flag*))
    (flet ((getf* (plist key)
	     (do ()
		 ((null plist) miss)
	       (if (eq (car plist) key)
		   (return (cadr plist))
		   (setq plist (cddr plist))))))
      (labels ((default-1 (tail)
		 (if (null tail)
		     nil
		     (if (eq (getf* supplied-initargs (caar tail)) miss)
			 (list* (caar tail)
				(funcall (cadar tail))
				(default-1 (cdr tail)))
			 (default-1 (cdr tail))))))
	(append supplied-initargs (default-1 all-default))))))


(defmethod *initialize-instance ((instance object) &rest initargs)
  (apply #'shared-initialize instance t initargs))

(defmethod reinitialize-instance ((instance object) &rest initargs)
  ;; ***
  ;; *** Later we need to do initarg checking here.
  ;; ***
  (apply #'shared-initialize instance nil initargs)
  instance)


(defmethod update-instance-for-different-class ((previous object)
						(current object)
						&rest initargs)
  ;; ***
  ;; *** Later we need to do initarg checking here.
  ;; ***
  ;;
  ;; First we must compute the newly added slots.  The spec defines
  ;; newly added slots as "those local slots for which no slot of
  ;; the same name exists in the previous class."
  (let ((added-slots '())
	(current-slotds (class-slots (class-of current)))
	(previous-slot-names (mapcar #'slotd-name
				     (class-slots (class-of previous)))))
    (dolist (slotd current-slotds)
      (if (and (not (memq (slotd-name slotd) previous-slot-names))
	       (eq (slotd-allocation slotd) ':instance))
	  (push (slotd-name slotd) added-slots)))
    (apply #'shared-initialize current added-slots initargs)))

(defmethod update-instance-for-redefined-class ((instance object)
						added-slots
						discarded-slots
						property-list
						&rest initargs)
  (declare (ignore discarded-slots property-list))
  ;; ***
  ;; *** Later we need to do initarg checking here.
  ;; ***
  (apply #'shared-initialize instance added-slots initargs))


(defmethod shared-initialize ((instance object) slot-names &rest initargs)
  ;;
  ;; initialize the instance's slots in a two step process
  ;;   1) A slot for which one of the initargs in initargs can set
  ;;      the slot, should be set by that initarg.  If more than
  ;;      one initarg in initargs can set the slot, the leftmost
  ;;      one should set it.
  ;;
  ;;   2) Any slot not set by step 1, may be set from its initform
  ;;      by step 2.  Only those slots specified by the slot-names
  ;;      argument are set.  If slot-names is:
  ;;       T
  ;;            any slot not set in step 1 is set from its
  ;;            initform
  ;;       <list of slot names>
  ;;            any slot in the list, and not set in step 1
  ;;            is set from its initform
  ;;
  ;;       ()
  ;;            no slots are set from initforms
  ;;
  (let* ((class (class-of instance))
	 (slotds (class-slots class)))
    (dolist (slotd slotds)
      (let ((slot-name (slotd-name slotd))
	    (slot-initargs (slotd-initargs slotd)))
	(flet ((from-initargs ()
		 ;; Try to initialize the slot from one of the initargs.
		 ;; If we succeed return T, otherwise return nil.
		 (doplist (initarg val)
			  initargs
		   (when (memq initarg slot-initargs)
		     (setf (slot-value instance slot-name) val)
		     (return 't))))
	       (from-initforms ()
		 ;; Try to initialize the slot from its initform.  This
		 ;; returns no meaningful value.
		 (if (and slot-names
			  (or (eq slot-names 't)
			      (memq slot-name slot-names))
			  (not (slot-boundp instance slot-name)))
		     (let ((initfunction (slotd-initfunction slotd)))
		       (when initfunction
			 (setf (slot-value instance slot-name)
			       (funcall initfunction)))))))
	  
	  (or (from-initargs)
	      (from-initforms))))))
  instance)


#||
;;;
;;; This is another version of check-initargs called from initialization
;;; methods.
;;; 
;;; (check-initargs class instance initargs generic-function-name)
;;; initargs is computed by default-initargs
;;;
(defun check-initargs (class instance initargs generic-function-name)
  (let ((invalid-key nil)
	(allow-other-keywords nil)
	(keywords (mapcan #'slotd-initargs
			  (class-slots class)))
	(effective-methods
	 (ecase generic-function-name
	   (make-instance
	    (union (compute-applicable-methods #'allocate-instance class)
		   (compute-applicable-methods #'*initialize-instance instance)
		   (compute-applicable-methods #'shared-initialize instance)))
	   (reinitialize-instance
	    (union (compute-applicable-methods #'reinitialize-instance
					       instance)
		   (compute-applicable-methods #'shared-initialize instance)))
	   (update-instance-for-different-class
	    (union (compute-applicable-methods
		    #'update-instance-for-different-class class
		    instance)
		   (compute-applicable-methods #'shared-initialize instance)))
	   (update-instance-for-redefined-class
	    (union (compute-applicable-methods
		    #'update-instance-for-redefined-class
		    instance)
		   (compute-applicable-methods #'shared-initialize
					       instance))))))
    (dolist (method methods)
      (multiple-value-bind (keys allow-other-keys)
	   (function-keywords method)
	(cond (allow-other-keys
	       (setq allow-other-keywords t)
	       (return t)))
	(setq keywords (union keys keywords))))
    (unless allow-other-keywords
      (iterate ((key (*list-elements initargs :by #'cddr)))
	(unless (memq key keywords)
	  (return (setq invalid-key key)))))
   invalid-key))
||#
