;;; -*- Mode:Common-Lisp; Package:POS; Base:10 -*-
;;; $Id: qgraph.lisp,v 1.3 1992/07/12 17:25:09 bert Exp $

(in-package 'pos)

;;; There are four types of objects that can be graphed :
;;; - Time qualitative datasets  (a set of (timeLmark, yLmark) points)
;;; - Phase qualitative datasets
;;; - Time envelope datasets   (i.e., a set of (x, ylo, yhi) points)
;;; - Phase envelope datasets  (a set of (xlo, xhi, ylo, yhi) points)
;;; - Line datasets  (i.e., a set of points)

;;; Assumptions :
;;; - Assume that min values are always <= max values in T/PEnv.
;;; - Assume a single qualitative dataset.

;;; Log scaling :
;;;  The data itself is stored non-scaled.  The -xlimits and -ylimits
;;;  reflect the axis as if the data were scaled.  Screenx and Screeny
;;;  take an optional log parameter that will scale the data.

;;; To do :
;;;         Clipping for tquals (should happen in draw-qual-box).
;;;         Qual scaling in Y.
;;;         Draw-data should pass a window for zooming.
;;;         Compute true scale for qual regions w/o blindly using the
;;;           lmark endpoints (i.e., look at the data rather than the lmarks)

;;; Change Log:
;;; 20Dec90 - Stable version to symbolics.
;;; 18Jan91 - New datatype TQ3.
;;; 21Jan91 - Fixed find-qual-limits-TQ3 to check data slot for Y limits.
;;; 22Jan91 - Made single points print as larger circles.
;;;  7Feb91 - Made default for rangestyle 1 and intervalstyle 5
;;; 18Jun91 - Switched storage from arrays to queues. This makes it easier
;;;           to dynamically add elements to a structure.

(defparameter *BDR* 3)         ; distance from border to first thing to be displayed
(defparameter *TCK* 20)        ; length of a tick mark spacings in Y ?
(defparameter *TCKMARKLEN* 8)  ; length of a tickmark.
(defparameter *LABEL-WIDTH* 7) ; -ddd.dd
(defparameter *ZERO-THRESHOLD* 1.0e-7) ; how close to measure for a 0 axis
(defparameter *INTER-LEGEND-SPACING* 12)

(defparameter *QualAxisNumLines* 3)  ; Number of lines that can be stacked on the qual axis
(defparameter *QualAxisNumNames* 3)  ; Number of names that can be stacked on the qual axis
(defparameter *QualAxisLineSpacing* 3) ; Vertical pixel space for qual line separation
(defparameter *QualAxisNameSpacing* 2) ; Vertical pixel space between names

;;; Graph structure.
;;; There is one of these for each graph to be displayed.
;;; Note that there may be multiple datasets in a graph.

(defstruct (qgraph (:print-function qgraph-printer))
  (Id   (gentemp "QG-"))
  (Name NIL)         ; An sexpr that quickly identifies the graph.
  (Documentation NIL)
  (Timestamp NIL)    ; Creation date
  (XLimits NIL)      ; A list of (lowx highx minf inf) or NIL.  
                     ; If this slot is NIL then the plotting bounds
                     ; will be plotted automatically.
  (YLimits NIL)      ; ditto
  (XQualScaling NIL) ; If T, the X axis is qualitatively scaled
                     ; (range values are not used).  What happens
                     ; if numerical datasets are also plotted is
                     ; undefined.
  (YQualScaling NIL) ; ditto
  (LnX NIL)          ; If T, scale X logarithmically.
  (LnY NIL)          ; If T, scale Y logarithmically.

  (XUnitText "X")    ; Text for the axis labels
  (YUnitText "Y")

  (QualPlottingMethod 'BOX) ; BOX  => LL corner of box labeled with qual vals
                            ; AXIS => Axis has qual section to label 
                            ; PLAIN-BOX => unlabeled boxes with no axis labeling
                            ; BOX-CENTER => qdir labeled with qual vals
  (QualAxisNumLines  *QualAxisNumLines*)  ; The number of range lines for the
                                          ; the qualitative axis.
  (QualAxisNumNames  *QualAxisNumNames*)  ; The number of rows used for labeling
                                          ; the qualitative axis.
  (Tickflag  T)             ; If T then ticks are used to mark numeric grid labels
                            ; If NIL then lines are used.
  (ZeroWidth 2)             ; pixel width of the X=0 or Y=0 axis
  (GridWidth 1)             ; Width of grid lines and tick marks

  (Window NIL)       ; qgraph-window struct.

  ;; Internal limits and inf markers
  (-xlimits NIL)     ; (min max minf inf)
  (-ylimits NIL)     ; ditto

  ;; Line style and color information for each data type.
  ;; There is a list in each slot; each entry gives
  ;; the value for the nth object of a particular type.
  ;; Note that Styles is for non-qual plots and Range and
  ;; Interval styles are only for qual plots.
  (Colors '("red" "green" "blue" "yellow"))
  (Styles '((1 0) (2 1)))  ; Line styles for non-qualitative plots
;#  (RangeStyle 1)           ; Line style for range boxes of the Qualitative plot
;#  (IntervalStyle 5)        ; Line style for interval boxes (NIL = none)
;#  (RangeColor "red")
;#  (IntervalColor "black")
  
  ;; The actual data.  There is a list of structures in each slot
  ;; which gives data for the nth object to be plotted.
  ;; Objects are typed so that the functions know what they are.
  (Data NIL)

  (-QualDataset NIL) ; Should point to the qualitative dataset which will
                     ; be used for qualitative axis scaling.  This slot is
                     ; used by qualitative axis plotting for type AXIS only
                     ; and should not be used elsewhere.
)

(defun qgraph-printer (qgraph stream ignore)
  (declare (ignore ignore))
  (format stream "#<qgraph ~a: Name: ~a: YunitText=~a XUnitText=~a>" 
	  (qgraph-id qgraph) (qgraph-name qgraph)
	  (qgraph-YUnitText qgraph) (qgraph-XUnitText qgraph)))

;;; An object composed of lines.
(defstruct (Line (:print-function line-printer))
  (Id (gentemp"LI-"))
  (Name NIL)        ; a symbol
  (TrendName NIL)   ; a string
  (Documentation NIL) ; a string
  (Timestamp NIL)
  (Data NIL)        ; a queue of (x y) entries
)

(defun Line-printer (Line stream ignore)
  (declare (ignore ignore))
  (format stream "#<Line ~a: Name:~a>"
	  (line-id line) (line-Name line)))


;;; Time envelope
(defstruct (TEnv (:print-function tenv-printer))
  (Id (gentemp "TE-"))
  (Name NIL)        ; a symbol
  (TrendName NIL)   ; a string
  (Documentation NIL) ; a string
  (Timestamp NIL)
  (Data NIL)        ; a queue of (x yl yu) entries
)

(defun tenv-printer (tenv stream ignore)
  (declare (ignore ignore))
  (format stream "#<TEnv ~a: Name:~a>"
	  (tenv-id tenv) (tenv-Name tenv)))


;;; Phase envelope
(defstruct (PEnv (:print-function PEnv-printer))
  (Id (gentemp "PE-"))
  (Name NIL)        ; a symbol
  (TrendName NIL)   ; a string
  (Documentation NIL) ; a string
  (Timestamp NIL)
  (Data NIL)        ; a queue of (xl xu yl yu) entries
)

(defun PEnv-printer (penv stream ignore)
  (declare (ignore ignore))
  (format stream "#<Penv ~a: Name:~a>"
	  (penv-Id penv) (penv-Name penv)))


;;; Time qualitative
(defstruct (TQual (:print-function TQual-printer))
  (Id   (gentemp "TQ-"))
  (Name NIL)        ; a symbol
  (TrendName NIL)   ; a string
  (Timestamp NIL)
  (Documentation NIL) ; a string
  (Lmarks NIL)      ; (Timelmarks Varlmarks), each a list of (Lmark l u)
  (Data NIL)        ; a list of (Lmark-or-int Lmark-or-int qdir)
)

(defun TQual-printer (TQual stream ignore)
  (declare (ignore ignore))
  (format stream "#<TQual ~a: Name:~a>"
	  (TQual-Id TQual) (TQual-Name TQual)))


;;; Time qualitative for Q3 (subject to change)
(defstruct TQ3
  (Id   (gentemp "TQ3-"))
  (Name NIL)        ; a symbol
  (TrendName NIL)   ; a string
  (Documentation NIL) ; a string
  (Timestamp NIL)
  (Lmarks NIL)      ; (Timelmarks Varlmarks), each a list of (Lmark l u)
  (Data NIL)        ; a list of (Lmark Range qdir)
                    ; where Range is a list of (L H) where either may be
                    ; a number or a Lmark.  This differs from TQual in that
                    ; intervals are not recorded.
)

;;; Phase qualitative
(defstruct PQual
  (Id (gentemp "PQ-"))
  (Name NIL)        ; a symbol
  (TrendName NIL)   ; a string
  (Documentation NIL) ; a string
  (Timestamp NIL)
  (Lmarks NIL)      ; a list of (Xlmarks Ylmarks) each of which
                    ; is a list of (Lmark l u)
  (Data NIL)        ; a list of (Lmark-or-int Lmark-or-int)
)


;;; Window structure

(defstruct QgraphWindow 
  (WindowX NIL)     ; "Absolute" screen location
  (WindowY NIL)     
  (Xsize   NIL)     ; Number of pixels in X and Y
  (YSize   NIL)

                    ; these next slots are scaled logarithmically
                    ; if need be
  (MinX    NIL)     ; min values in user coords
  (MinY    NIL)
  (MaxX    NIL)     ; max values in user coords
  (MaxY    NIL)
  
  (RightBorder NIL)   ; border structures for each border
  (BottomBorder NIL)
  (LeftBorder  NIL)
  (TopLeftBorder NIL)

  (YTickSpacing   *TCK*)  ; space between tick marks in pixels
  (XTickSpacing   NIL)    ; depends on size of labels

  (XUsable     NIL)   ; Usable # pixels in X and Y 
  (YUsable     NIL)   ; after subtracting out the Axis and Label spaces
  (XNUsable    NIL)   ; # pixels in X and Y after removing 
  (YNUsable    NIL)   ; infinite regions

  (Xpixels/Unit NIL)  ; # pixels per unit in X and Y
  (YPixels/Unit NIL)

  (XInfW       NIL)   ; Widths and heights of infinite regions.
  (XMinfW      NIL)
  (YInfH       NIL)
  (YMinfH      NIL)

  (Font           NIL)
  (FontCharHeight NIL)
  (FontCharWidth  NIL)

  (gridjuke    (make-gridjuke))
)


;;; This is for storing grid offsets for log scaling.
(defstruct gridjuke
  (array (make-array '(101)))
  (base 0)
  (step 0)
  (n  0)
  (cur 0))



;;; Return the TrendName slot of an object.
(defun Object-TrendName (obj)
  (typecase obj
    (Line  (Line-TrendName obj))
    (TEnv  (TEnv-TrendName obj))
    (PEnv  (PEnv-TrendName obj))
    (TQual (TQual-TrendName obj))
    (TQ3   (TQ3-Trendname obj))    ; BKay 18Jan91
    (PQual (PQual-TrendName obj))
    (T (error "~%Can't operate on ~a" obj))))


;;;------------------------------------------------------------
;;; Find the limits of a dataset
;;; Inputs:  obj - a dataset type.
;;;          axis - X or Y.
;;; Returns: (min max minf inf)
;;;------------------------------------------------------------

(defun find-limits (obj axis)
  (typecase obj
    (Line  (find-limits-Line obj axis))
    (TEnv  (find-limits-TEnv obj axis))
;    (PEnv  (find-limits-PEnv obj axis))
    (TQual (find-limits-TQual obj axis))
    (TQ3   (find-limits-TQ3 obj axis))   ; BKay 18Jan91
;    (PQual (find-limits-PQual obj axis))
    (T (error "~%Can't operate on ~a" obj))))

(defun find-limits-line (o axis)
  (let (min max
        (i (if (eq axis 'X) 0 1))
	(data-op (typecase o
		   (Line #'Line-Data)
		   (TEnv #'TEnv-Data)
		   (T    (error "Expected Line or TEnv, got ~a" o)))))
    (setf min (elt (qtop (funcall data-op o)) i))
    (setf max min)
    (dolist (point (cdr (qlist (funcall data-op o))))
      (setf min (min min (elt point i)))
      (setf max (max max (elt point i))))
    (list min max NIL NIL)))

(defun find-limits-TEnv (o axis)
  (if (eq axis 'X) 
      (find-limits-line o axis)
      (let (min max)
	(setf min (elt (qtop (TEnv-Data o)) 1))
	(setf max (elt (qtop (TEnv-Data o)) 2))
	(dolist (point (cdr (qlist (TEnv-Data o))))
	  (setf min (min min (elt point 1)))
	  (setf max (max max (elt point 2))))
	(list min max NIL NIL))))

;;; BKay 14Feb91
;;; Look on the y-axis for points that are really used rather than looking
;;; at the whole axis.
;;;
;(defun find-limits-TQual (o axis)
;  (if (eq axis 'X) 
;      (find-qual-limits (first (TQual-Lmarks o)))

(defun find-limits-TQual (o axis)
  (find-qual-limits (if (eq axis 'X) (first (TQual-Lmarks o))
		                     (second (TQual-Lmarks o)))))

;;; BKay 18Jan91
(defun find-limits-TQ3 (o axis)
  (if (eq axis 'X)
      (find-qual-limits  (first (TQ3-Lmarks o)))
      ;; BKay 21Jan91
      ;; For Q3 stuff, we need to look at the ranges themselves
      ;; since every range doesn't have a corresponding Lmark
      (let ((limits (find-qual-limits (second (TQ3-Lmarks o)))))
	(dolist (point (TQ3-Data o))
	  (let ((range (second point)))
	    (if (or (eq (first range) '-inf) (eq (second range) '-inf)) 
		(setf (third limits) T))
	    (if (or (eq (first range) '+inf) (eq (second range) '+inf))
		(setf (fourth limits) T)) 
	    (when (numberp (first range))
	      (if (or (null (first limits)) (< (car range) (first limits)))
		  (setf (first limits) (first range)))
	      (if (null (second limits))
		  (setf (second limits) (first range))))
	    (when (numberp (second range))
	      (if (or (null (second limits)) (> (cadr range) (second limits)))
		  (setf (second limits) (second range)))
	      (if (null (first limits))
		  (setf (first limits) (second range))))))
	limits)))


;;;------------------------------------------------------------------
;;; Find the limits of a qualitative axis by looking only at the
;;; landmarks.
;;; Inputs:  lmarks - a list of (name lower upper) for each lmark
;;;                   in the qspace.
;;; Returns: (min max minf inf)
;;; Notes:   Assumes that INF entry is (INF +inf +inf) and MINF
;;;          entry is (MINF -inf -inf).
;;;          This also guarantees that at least (0,0) is in the range.
;;; Changes: Changed names for minf and inf to minf-flag and inf-flag
;;;          so tha there are no collisions with Q3.
;;;------------------------------------------------------------------

(defun find-qual-limits (Lmarks)
  (let ((min NIL) (max NIL) (minf-flag NIL) (inf-flag NIL))
    (dolist (Lmark-entry Lmarks)
      ;; Check low bound for lmark
      (cond
	((eq (second Lmark-entry) '-inf)
	 (setf minf-flag T))
	((eq (second Lmark-entry) '+inf)
	 (setf inf-flag T))
	(T
	 (when (numberp (second Lmark-entry))
	   (if (null min)
	       (setf min (second Lmark-entry))
	       (setf min (min min (second Lmark-entry))))
	   (if (null max)
	       (setf max (second Lmark-entry))
	       (setf max (max max (second Lmark-entry)))))))
;      (format t "~%After ~a  limits are ~a ~a ~a ~a" Lmark-entry min max minf inf)
      (cond
	((eq (third Lmark-entry) '+inf)
	 (setf inf-flag T))
	((eq (third Lmark-entry) '-inf)
	 (setf minf-flag T))
	(T
	 (when (numberp (third Lmark-entry))
	   (if (null min)
	       (setf min (third Lmark-entry))
	       (setf min (min min (third Lmark-entry))))
	   (if (null max)
	       (setf max (third Lmark-entry))
	       (setf max (max max (third Lmark-entry)))))))
;      (format t "~%After ~a  limits are ~a ~a ~a ~a" Lmark-entry min max minf inf)
      )
    (list (if (null min) 0 min)
	  (if (null max) 0 max)
	  minf-flag inf-flag)))



;;;------------------------------------------------------------------
;;; Find the limits of a qgraph structure.
;;; Inputs:  s     - the qgraph structure.
;;;          axis  - X or Y.
;;; Returns: (min max minf inf)
;;;------------------------------------------------------------------

(defun find-limits-of-plot (s axis)
  (if (null (qgraph-data s))
      '(0 10 nil nil)
      (let ((bounds (find-limits (car (qgraph-data s)) axis)))
	(dolist (obj (cdr (qgraph-data s)))
	  (let ((new-bounds (find-limits obj axis)))
	    (replace-limits bounds new-bounds)))
	bounds)))


;;;---------------------------------------------------------------------
;;; Replace the limits of a limit structure if the new bounds demand it.
;;; Inputs:  bounds      - current bounds (min max minf inf).
;;;          new-bounds  - another bound set.
;;; Returns: none, but modifies bounds with new elements as needed.
;;;---------------------------------------------------------------------

(defun replace-limits (bounds new-bounds)
  (if (< (car new-bounds) (car bounds)) (setf (car bounds) (car new-bounds)))
  (if (> (cadr new-bounds) (cadr bounds)) (setf (cadr bounds) (cadr new-bounds)))
  (if (caddr new-bounds) (setf (caddr bounds) (caddr new-bounds)))
  (if (cadddr new-bounds) (setf (cadddr bounds) (cadddr new-bounds))))


;;;---------------------------------------------------------------------
;;; Find the first (and only, hopefully!) qualitative dataset in a
;;; list of datasets.
;;; Returns:  the qual dataset (either TQual or PQual) or NIL.
;;;---------------------------------------------------------------------

(defun find-qual-dataset (objs)
  (cond 
   ((null objs) NIL)
   ((typep (car objs) (or 'TQual 'PQual 'TQ3)) (car objs)) ; BKay 18Jan91 - added TQ3
   (T (find-qual-dataset (cdr objs)))))
  


;;; Main function
;;; For now, assume that numeric scaling is to be done.
(defun qgraph (s)
  (setf (qgraph--QualDataset s) (find-qual-dataset (qgraph-data s)))

  ;; THIS DOESN'T SPEED THINGS UP
  ;; Bounds are computed only if
  ;; 1.  the [XY]limits slot is set  -OR-
  ;; 2.  the -[XY]limits slot is not set
  ;; The second case would mean that bounds are only recomputed when
  ;; needed (i.e., if we repaint the screen, there will be no overhead).
;  (when (or (null (qgraph--xlimits s))
;	    (null (qgraph--ylimits s))
;	    (qgraph-xlimits s)
;	    (qgraph-ylimits s)) 
;    (compute-plotting-bounds s))
  (compute-plotting-bounds s)
  (draw-axes s)
  (draw-data s)
  (draw-legend s)
)


;;;---------------------------------------------------------------------
;;; Figure out how much space is needed for each plotting region, etc.
;;;---------------------------------------------------------------------

(defun compute-plotting-bounds (s)
  ;; Compute the numeric limits of the data, if necessary.
  (setf (qgraph--xlimits s)
	(if (qgraph-Xlimits s)
	    (copy-list (qgraph-Xlimits s))
	    (find-limits-of-plot s 'X)))
  (setf (qgraph--ylimits s)
	(if (qgraph-Ylimits s)
	    (copy-list (qgraph-Ylimits s))
	    (find-limits-of-plot s 'Y)))

  ;; Take logs of the bounds if this is a log axis
  (when (qgraph-LnX s)
    (setf (first (qgraph--xlimits s)) (log (first (qgraph--xlimits s)) 10))
    (setf (second (qgraph--xlimits s)) (log (second (qgraph--xlimits s)) 10)))
  (when (qgraph-LnY s)
    (setf (first (qgraph--ylimits s)) (log (first (qgraph--ylimits s)) 10))
    (setf (second (qgraph--ylimits s)) (log (second (qgraph--ylimits s)) 10)))

  ;; Our calculation of plotting constants is in part controlled by whether or
  ;; not there is a RANGE of values on an axis.  If there isn't, meaning the
  ;; axis looks something like (minf 0 inf), we want to simplify the gridding.
  (let ((single-point-x (= (car (qgraph--xlimits s)) (cadr (qgraph--xlimits s))))
	(single-point-y (= (car (qgraph--ylimits s)) (cadr (qgraph--ylimits s))))
	(w (qgraph-Window s)))

    ;; Pad data limits by 10%.  This gives some slop for printing ticks.
    ;; [The pad is zero for single point ranges, so there's no problem here.]
    (let (pad)
      (setf pad (/ (- (second (qgraph--xlimits s)) (first (qgraph--xlimits s)))
		   20.0))
      (decf (first (qgraph--xlimits s)) pad)
      (incf (second (qgraph--xlimits s)) pad)

      (setf pad (/ (- (second (qgraph--ylimits s)) (first (qgraph--ylimits s)))
		   20.0))
      (decf (first (qgraph--ylimits s)) pad)
      (incf (second (qgraph--ylimits s)) pad)

      ;; Move min and max values into the window structure for xscreen and yscreen
      (setf (QgraphWindow-MinX (Qgraph-window s)) (car (qgraph--xlimits s)))
      (setf (QgraphWindow-MaxX (Qgraph-window s)) (cadr (qgraph--xlimits s)))
      (setf (QgraphWindow-MinY (Qgraph-window s)) (car (qgraph--ylimits s)))
      (setf (QgraphWindow-MaxY (Qgraph-window s)) (cadr (qgraph--ylimits s))))

    ;; Compute the size of axis labels so we know how much space to leave for them.
    ;; In the X direction, this is the WIDTH of the longest trend name or the
    ;; axis name.  In the Y direction, this is the WIDTH of the longest lmark
    ;; name.
    (let (max-x-axis-legend-width
	  max-y-axis-lmark-width
	  ;; Next two vars are for precomputing X-axis exponent.
	  (expX 0)
	  larger)

      ;; Precompute the X axis exponent to see if we need to include it.
      ;;
      ;; Grid display power is computed by taking the log
      ;; of the largest number and rounding to a multiple of 3.
      (when (and (null (qgraph-LnX s)) 
		 (not (zerop (car (qgraph--xlimits s))))   ; don't do it if only 0
		 (not (zerop (cadr (qgraph--xlimits s)))))   
	(setf larger (max (abs (car (qgraph--xlimits s)))
			  (abs (cadr (qgraph--xlimits s)))))
	(setf expX (* (floor (/ (log larger 10) 3.0)) 3)))

      ;; Determine maximum X-axis label (i.e. trendname) width in pixels.
      (setf max-x-axis-legend-width 
	    ;; Include exponent max size only if we are using them
	    (* (+ (if (zerop expX) 1 6)
		  (length (qgraph-XUnitText s))) (QgraphWindow-FontCharWidth w)))
      (dolist (o (qgraph-Data s))
	(setf max-x-axis-legend-width 
	      (max max-x-axis-legend-width
		   (* (length (Object-TrendName o)) 
		      (QgraphWindow-FontCharWidth w)))))

      ;; Determine maximum Y-axis label (i.e. Lmark names in Y) width in pixels.
      ;; in BOX mode, this is 0 as no lmarks are displayed on the axis.
      ;; In axis mode, we will need to check how much space is needed for
      ;; overlapping qual interval markers.  This has not been done yet
      (setf max-y-axis-lmark-width 0)
;     This is shut off since axis scaling is only on X for now
;      (when (and (qgraph--QualDataset s) (eq (qgraph-QualPlottingMethod s) 'AXIS))
;	(dolist (ylmark-entry (second (TQual-Lmarks (qgraph--QualDataset s))))
;	  (setf max-y-axis-lmark-width
;		(max max-y-axis-lmark-width
;		     (* (length (format nil "~a" (first ylmark-entry))) ; need the string here
;			(QgraphWindow-FontCharWidth w))))))

      ;; Determine plotting bounds for each display region.
      (setf (QgraphWindow-TopLeftBorder w)
	    (make-CornerBorder :Yentries `((Border       ,*BDR*)
					   (LabelTop     ,(* 1.5 (QgraphWindow-FontCharHeight w)))
					   (LabelBottom  ,*BDR*)
					   (TopOfData    ,(+ *INTER-LEGEND-SPACING*
							     (QgraphWindow-FontCharHeight w)))
					   (TopOfLegendNameB  0))
			       :Xentries `((Border       ,*BDR*)
					   (Label        0))))
      (setf (QgraphWindow-LeftBorder w)
	    (make-SideBorder :entries `((Border          ,*BDR*)
					(Lmark           ,max-y-axis-lmark-width)
					(LmarkR          0)
					(PadOutsideQline 4)
					(Qline           0)
					(PadInsideQline  4)
					(Number          ,(* *LABEL-WIDTH* (QgraphWindow-FontCharWidth w)))
					(NumberRight     0)
					(Tick            ,*TCKMARKLEN*)
					(Axis            0))))
      (setf (QgraphWindow-BottomBorder w)
	    (make-SideBorder :entries `((Border          ,(+ 2 *BDR*))
					(QualAxisBotName ,(if (and (qgraph--QualDataset s) 
								   (eq (qgraph-QualPlottingMethod s) 'AXIS))
							      (* (qgraph-QualAxisNumNames s)
								 (+ (QgraphWindow-FontCharHeight w)
								    *QualAxisNameSpacing*))
							      0))
					(QualAxisTopName ,(if (and (qgraph--QualDataset s) 
								   (eq (qgraph-QualPlottingMethod s) 'AXIS))
							      (* (qgraph-QualAxisNumLines s)
								 *QualAxisLineSpacing*)
							      0))
					(QualAxisTop     ,(if (and (qgraph--QualDataset s) 
								   (eq (qgraph-QualPlottingMethod s) 'AXIS))
							      3
							      0))
					(Qline           0)
					(PadInsideQline  4)
					(Number          ,(QgraphWindow-FontCharHeight w))
					(Tick            ,*TCKMARKLEN*)
					(Axis            0))))
      (setf (QgraphWindow-RightBorder w)
	    (make-SideBorder :entries `((Border          ,*BDR*)
					(TrendNameR      ,max-x-axis-legend-width)
					(TrendNameL      ,*BDR*)
					(TickR           ,*TCKMARKLEN*)
					(TickL           0)
					(RightData       0))))
      )
    (setf (QgraphWindow-XUsable w)
	  (- (QgraphWindow-Xsize w) (border-to 'axis (QgraphWindow-LeftBorder w))
	     (border-to 'RightData (QgraphWindow-RightBorder w))))
    (setf (QgraphWindow-YUsable w)
	  (- (QgraphWindow-Ysize w)
	     (border-to 'TopOfData (QgraphWindow-TopLeftBorder w) 'Y)
	     (border-to 'axis (QgraphWindow-BottomBorder w))))


    ;; Figure out how much space should be alloted to minf and inf
    (setf (QgraphWindow-XMinfW w)
	  (cond
	    ;; If there's a single numeric point then split the usable
	    ;; space in two if both minf and inf are present.
	    (single-point-x
	     (cond
	       ((and (third (qgraph--xlimits s)) (fourth (qgraph--xlimits s)))
		(/ (QgraphWindow-XUsable w) 2))
	       ((third (qgraph--xlimits s))
		(QgraphWindow-XUsable w))
	       (T
		0)))
	    ;; Take 10% of distance if there is a range.
	    (T
	     (if (third (qgraph--xlimits s))
		 (/ (QgraphWindow-XUsable w) 10)
		 0))))
    (setf (QgraphWindow-XInfW w)
	  (cond
	    (single-point-x
	     (cond
	       ((and (third (qgraph--xlimits s)) (fourth (qgraph--xlimits s)))
		(/ (QgraphWindow-XUsable w) 2))
	       ((fourth (qgraph--xlimits s))
		(QgraphWindow-XUsable w))
	       (T
		0)))
	    (T
	     (if (fourth (qgraph--xlimits s))
		 (/ (QgraphWindow-XUsable w) 10)
		 0))))
    (setf (QgraphWindow-YMinfH w)
	  (cond
	    (single-point-y
	     (cond
	       ((and (third (qgraph--ylimits s)) (fourth (qgraph--ylimits s)))
		(/ (QgraphWindow-YUsable w) 2))
	       ((third (qgraph--ylimits s))
		(QgraphWindow-YUsable w))
	       (T
		0)))
	    (T
	     (if (third (qgraph--ylimits s))
		 (/ (QgraphWindow-YUsable w) 10)
		 0))))
    (setf (QgraphWindow-YInfH w)
	  (cond
	    (single-point-y
	     (cond
	       ((and (third (qgraph--ylimits s)) (fourth (qgraph--ylimits s)))
		(/ (QgraphWindow-YUsable w) 2))
	       ((fourth (qgraph--ylimits s))
		(QgraphWindow-YUsable w))
	       (T
		0)))
	    (T
	     (if (fourth (qgraph--ylimits s))
		 (/ (QgraphWindow-YUsable w) 10)
		 0))))
    
    (setf (QgraphWindow-XNUsable w)
	  (- (QgraphWindow-XUsable w)
	     (QgraphWindow-XMinfW w) (QgraphWindow-XInfW w)))
    (setf (QgraphWindow-YNUsable w)
	  (- (QgraphWindow-YUsable w)
	     (QgraphWindow-YMinfH w) (QgraphWindow-YInfH w)))
    (if (or (< (QgraphWindow-XNUsable w) 0)
	    (< (QgraphWindow-YNUsable w) 0))
	(error "QGRAPH: Window is too small to draw data"))
    (setf (QgraphWindow-XPixels/Unit w)
	  (if single-point-x
	      1
	      (/ (QgraphWindow-XNUsable w)
		 (- (second (qgraph--Xlimits s))
		    (first (qgraph--Xlimits s))))))
    (setf (QgraphWindow-YPixels/Unit w)
	  (if single-point-y
	      1
	      (/ (QgraphWindow-YNUsable w)
		 (- (second (qgraph--Ylimits s))
		    (first (qgraph--Ylimits s))))))
    
    ;; Compute x tickspacing based on char width
    (setf (QgraphWindow-XTickSpacing w) (* (QgraphWindow-FontCharWidth w)
					   *LABEL-WIDTH*))
    ))


;;; Draw the coordinate axes
(defun draw-axes (s)
  (with-qplot-gcon ()
    (draw-numeric-axes s)
    (draw-qualitative-axes s)))

;;; This is very similar to DrawGridAndAxis in Xgraph.
(defun draw-numeric-axes (s)
  (let ((w (qgraph-window s))
	(expX 0) (expY 0)
	larger yincr ystart xincr xstart
	xspot yspot
	)

    ;; Grid display power is computed by taking the log
    ;; of the largest number and rounding to a multiple of 3.
    (when (and (null (qgraph-LnX s)) 
	       (not (zerop (car (qgraph--xlimits s))))   ; don't do it if only 0
	       (not (zerop (cadr (qgraph--xlimits s)))))   
      (setf larger (max (abs (car (qgraph--xlimits s)))
			(abs (cadr (qgraph--xlimits s)))))
      (setf expX (* (floor (/ (log larger 10) 3.0)) 3)))
    (when (and (null (qgraph-LnY s))
	       (not (zerop (car (qgraph--ylimits s))))   ; don't do it if only 0
	       (not (zerop (cadr (qgraph--ylimits s)))))
      (setf larger (max (abs (car (qgraph--ylimits s)))
			(abs (cadr (qgraph--ylimits s)))))
      (setf expY (* (floor (/ (log larger 10) 3.0)) 3)))
    
    ;; Draw the axis labels
    (setf xspot (+ (border-to 'Label (QgraphWindow-TopLeftBorder w) 'X)
		   (QgraphWindow-WindowX w)))
    (setf yspot (+  (border-to 'LabelBottom (QgraphWindow-TopLeftBorder w) 'Y)
		    (QgraphWindow-WindowY w)))
    (if (not (zerop expY))
	(progn
	  (qgraph-plot-string (format nil "~ax10" (qgraph-YUnitText s) expY)
			      xspot yspot 'LOWERLEFT w)
	  (qgraph-plot-string (format nil "~d" expY)
			      (+ xspot (* (length (format nil "~ax10" 
							  (qgraph-YUnitText s) expY))
					  (QgraphWindow-FontCharWidth w)))
			      (- yspot (/ (QgraphWindow-FontCharWidth w) 1.5))
			      'LOWERLEFT w))
        (qgraph-plot-string (qgraph-YUnitText s)
			    xspot yspot 'LOWERLEFT w))
    (setf xspot (- (+ (QgraphWindow-WindowX w) (QgraphWindow-XSize w))
		   (border-to 'TrendNameR (QgraphWindow-RightBorder w))))
    (setf yspot (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
		   (border-to 'Number (QgraphWindow-BottomBorder w))))
    (if (not (zerop expX))
	(progn
	  (qgraph-plot-string (format nil "~d" expX)
			      xspot
			      (- yspot (/ (QgraphWindow-FontCharWidth w) 1.5))
			      'RIGHT w)
	  (qgraph-plot-string (format nil "~ax10" (qgraph-XUnitText s) expX)
			      (- xspot (* (length (format nil "~d" expX))
					  (QgraphWindow-FontCharWidth w)))
			      yspot 'RIGHT w))
        (qgraph-plot-string (qgraph-XUnitText s)
			    xspot yspot 'RIGHT w))

    ;; Now draw the grid labels
    (setf yincr (/ (QgraphWindow-YTickSpacing w) (QgraphWindow-YPixels/Unit w)))
    (setf ystart (init-grid (car (qgraph--ylimits s)) yincr (qgraph-LnY s)
			    (QgraphWindow-gridjuke w)))
    (do ((yindex ystart (step-grid (QgraphWindow-gridjuke w))))
      ((>= yindex (second (qgraph--ylimits s))))
      
      (qgraph-plot-string (write-value yindex expY (qgraph-LnY s))
			  (+ (QgraphWindow-WindowX w) 
			     (border-to 'NumberRight (QgraphWindow-LeftBorder w)))
			  (screeny w yindex)
			   'RIGHT w))
    
    ;; Label inf and minf as needed
    (when (third (qgraph--ylimits s))
      (qgraph-plot-string "-inf"
			  (+ (QgraphWindow-WindowX w) 
			     (border-to 'NumberRight (QgraphWindow-LeftBorder w)))
			  (screeny w '-inf)
			  'RIGHT w))
    (when (fourth (qgraph--ylimits s))
      (qgraph-plot-string "+inf"
			  (+ (QgraphWindow-WindowX w) 
			     (border-to 'NumberRight (QgraphWindow-LeftBorder w)))
			  (screeny w '+inf)
			  'RIGHT w))
    ;; Do single points as a special case
    (when (= (first (qgraph--ylimits s)) (second (qgraph--ylimits s)))
      (qgraph-plot-string (write-value (first (qgraph--ylimits s)) expY (qgraph-LnY s))
			  (+ (QgraphWindow-WindowX w) 
			     (border-to 'NumberRight (QgraphWindow-LeftBorder w)))
			  (screeny w (first (qgraph--ylimits s)))
			   'RIGHT w))

    (setf xincr (/ (QgraphWindow-XTickSpacing w) (QgraphWindow-XPixels/Unit w)))
    (setf xstart (init-grid (car (qgraph--xlimits s)) xincr (qgraph-LnX s)
			    (QgraphWindow-gridjuke w)))
	  
    (do ((xindex xstart (step-grid (QgraphWindow-gridjuke w))))
      ((>= xindex (second (qgraph--xlimits s))))
      
      (qgraph-plot-string (write-value xindex expX (qgraph-LnX s))
			  (screenx w xindex) 
			  (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
			     (border-to 'Number (QgraphWindow-BottomBorder w)))
			  'BOTTOM w))
    ;; Do single points as a special case
    (when (= (first (qgraph--xlimits s)) (second (qgraph--xlimits s)))
      (qgraph-plot-string (write-value (first (qgraph--xlimits s)) expX (qgraph-LnX s))
			  (screenx w (first (qgraph--xlimits s))) 
			  (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
			     (border-to 'Number (QgraphWindow-BottomBorder w)))
			  'BOTTOM w))

    ;; Label inf and minf as needed
    (when (third (qgraph--xlimits s))
      (qgraph-plot-string "-inf"
			  (screenx w '-inf)
			  (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
			     (border-to 'Number (QgraphWindow-BottomBorder w)))
			  'BOTTOM w))
    (when (fourth (qgraph--xlimits s))
      (qgraph-plot-string "+inf"
			  (screenx w '+inf)
			  (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
			     (border-to 'Number (QgraphWindow-BottomBorder w)))
			  'BOTTOM w))
    
    ;; Now draw the tick marks
    (setf yincr (/ (QgraphWindow-YTickSpacing w) (QgraphWindow-YPixels/Unit w)))
    (setf ystart (init-grid (car (qgraph--ylimits s)) yincr (qgraph-LnY s)
			    (QgraphWindow-gridjuke w)))
    (do ((yindex ystart (step-grid (QgraphWindow-gridjuke w))))
      ((>= yindex (second (qgraph--ylimits s))))
      (qgraph-plot-tick-or-grid-y yindex s))
    (when (third (qgraph--ylimits s))
      (qgraph-plot-tick-or-grid-y '-inf s))
    (when (fourth (qgraph--ylimits s))
      (qgraph-plot-tick-or-grid-y '+inf s))
    ;; Handle single-point axes specially
    (when (= (first (qgraph--ylimits s)) (second (qgraph--ylimits s)))
      (qgraph-plot-tick-or-grid-y (first (qgraph--ylimits s)) s))

    (setf xincr (/ (QgraphWindow-XTickSpacing w) (QgraphWindow-XPixels/Unit w)))
    (setf xstart (init-grid (car (qgraph--xlimits s)) xincr (qgraph-LnX s)
			    (QgraphWindow-gridjuke w)))
    (do ((xindex xstart (step-grid (QgraphWindow-gridjuke w))))
      ((>= xindex (second (qgraph--xlimits s))))
      (qgraph-plot-tick-or-grid-x xindex s))
    (when (third (qgraph--xlimits s))
      (qgraph-plot-tick-or-grid-x '-inf s))
    (when (fourth (qgraph--xlimits s))
      (qgraph-plot-tick-or-grid-x '+inf s))
    ;; Handle single-point axes specially
    (when (= (first (qgraph--xlimits s)) (second (qgraph--xlimits s)))
      (qgraph-plot-tick-or-grid-x (first (qgraph--xlimits s)) s))
    ))


(defun qgraph-plot-tick-or-grid-y (yindex s)    
  (let* ((w (qgraph-window s))
	 (width (if (and (not (member yindex '(-inf +inf)))
			 (< (abs yindex) *ZERO-THRESHOLD*)
			 (not (qgraph-LnY s)))
		    (qgraph-ZeroWidth s)
		    (qgraph-GridWidth s))))
    (if (qgraph-tickflag s)
	(progn
	  (pos:qplot-hline (+ (QgraphWindow-WindowX w) 
			      (border-to 'Tick (QgraphWindow-LeftBorder w)))
			   (screeny w yindex)
			   *TCKMARKLEN*
			   :thickness width)
	  (pos:qplot-hline (- (+ (QgraphWindow-WindowX w) (QgraphWindow-XSize w))
			      (border-to 'RightData (QgraphWindow-RightBorder w)))
			   (screeny w yindex)
			   *TCKMARKLEN*
			   :thickness width))
	(pos:qplot-line (+ (QgraphWindow-WindowX w) 
			   (border-to 'Tick (QgraphWindow-LeftBorder w)))
			(screeny w yindex)
			(- (+ (QgraphWindow-WindowX w) (QgraphWindow-XSize w))
			   (border-to 'RightData (QgraphWindow-RightBorder w)))
			(screeny w yindex)
			:thickness width))))


(defun qgraph-plot-tick-or-grid-x (xindex s)    
  (let* ((w (qgraph-window s))
	 (width (if (and (not (member xindex '(-inf +inf)))
			 (< (abs xindex) *ZERO-THRESHOLD*)
			 (not (qgraph-LnX s)))
		    (qgraph-ZeroWidth s)
		    (qgraph-GridWidth s))))
    (if (qgraph-tickflag s)
	(progn
	  (pos:qplot-vline (screenx w xindex)
			   (+ (- (border-to 'TopOfData (QgraphWindow-TopLeftBorder w)
					    'Y)
				 *TCKMARKLEN*)
			      (QgraphWindow-WindowY w))
			   *TCKMARKLEN*
			   :thickness width)
	  (pos:qplot-vline (screenx w xindex)
			   (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
			      (border-to 'Axis (QgraphWindow-BottomBorder w)))
			   *TCKMARKLEN*
			   :thickness width))
	(pos:qplot-line (screenx w xindex)
			(+ (border-to 'TopOfData (QgraphWindow-TopLeftBorder w)
				      'Y)
			   (QgraphWindow-WindowY w))
			(screenx w xindex)
			(- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
			   (border-to 'Tick (QgraphWindow-BottomBorder w)))
			:thickness width))))


;;; Draw a qualitative axis, if specified.
;;;
(defun draw-qualitative-axes (s)
  (when (qgraph--QualDataset s)
    (let* ((tqual (qgraph--QualDataset s))
	   (lmarks (TQual-Lmarks tqual)))
      (when (eq (qgraph-QualPlottingMethod s) 'AXIS)
	;; Just do this for the x axis right now.
	(qplot-qual-axis (car lmarks) s)))))


;;; This just works for the x axis

(defun qplot-qual-axis (lmarks s)
  (let ((w (qgraph-window s))
	(line-occupancy (make-array (list (qgraph-QualAxisNumLines s)) :initial-element 0))
	(name-occupancy (make-array (list (qgraph-QualAxisNumNames s)) :initial-element 0)))
    (dolist (lmark lmarks)
      (let* ((xstart (screenx w (second lmark)))
	     (xend   (screenx w (third lmark)))
	     (yline  (find-free-entry xstart xend line-occupancy 'LINE w))
	     (yname  (find-free-entry xstart
				      (+ xstart
					 (* (length (format nil "~a" (first lmark)))
					    (QgraphWindow-FontCharWidth w)))
				      name-occupancy 'NAME w)))
	(pos:qplot-hline xstart yline (- xend xstart))
	(pos:qplot-vline xstart yline (- yname yline))
	(qgraph-plot-string (format nil "~a" (first lmark))
			    xstart yname 'TOP w)))))



(defun find-free-entry (start end table mode w)
  (let (yentry
	(max-index (1- (array-dimension table 0))))
    (do
      ((i 0 (incf i)))
      ((or (> i max-index)
	   (> start (svref table i)))
       (if (> i max-index)
	   (progn
	     (setf yentry (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w)
			     (- (border-to (if (eq mode 'LINE) 'QualAxisTop 'QualAxisTopName)
					   (QgraphWindow-BottomBorder w)))))
	     (setf i 0))
	   (setf yentry (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w)
			   (- (border-to (if (eq mode 'LINE) 'QualAxisTop 'QualAxisTopName)
					 (QgraphWindow-BottomBorder w)))
			   (* i (if (eq mode 'LINE) 
				    *QualAxisLineSpacing* 
				    (+ (QgraphWindow-FontCharHeight w) *QualAxisNameSpacing*))))))
       (setf (svref table i) end)
       yentry))))

;;; Allow alignment keys for plotting strings
(defun qgraph-plot-string (str x y just w)
  (let (rx ry)
    (case just
      (LEFT       (setf rx x) (setf ry (+ y (/ (QgraphWindow-FontCharHeight w) 2))))
      (LOWERLEFT  (setf rx x) (setf ry y))
      (TOP        (setf rx (- x (/ (* (length str) 
				      (QgraphWindow-FontCharWidth w))
				   2)))
		  (setf ry (+ y (QgraphWindow-FontCharHeight w))))
      (BOTTOM     (setf rx (- x (/ (* (length str) 
				      (QgraphWindow-FontCharWidth w))
				   2)))
		  (setf ry y))
      (RIGHT      (setf rx (- x (* (length str) 
				   (QgraphWindow-FontCharWidth w))))
		  (setf ry (+ y (/ (QgraphWindow-FontCharHeight w) 2))))
      (LOWERRIGHT (setf rx (- x (* (length str) 
				   (QgraphWindow-FontCharWidth w))))
		  (setf ry y))
      (T          (error "QGRAPH-PLOT-STRING: Unimplemented just ~a" just)))
    (pos:qplot-string str rx ry)))


(defun write-value (val exp log)
  (cond
   (log
    (if (= val (floor val)) #+symbolics (format nil "~,0E" (expt 10 val))
	                    ;; Stupid TI doesn't roundoff correctly when
	                    ;; using "~,0E".  For example printing
	                    ;; 9.99999e+21 it gives 9.e+21
	                    #+ti        (fix-roundoff-for-ti (expt 10 val))
			    #-(or symbolics ti) (format nil "~,0E"
							(expt 10 val))

                            (string-right-trim '(#\Space)
					       (format nil "~,2G"
						       (expt 10 (- val
								   (floor val)))))))
   (T
    (if (< exp 0)
	(do ((idx exp (incf idx)))
          ((>= idx 0))
	  (setf val (* val 10.0)))
;	  (format t "~%val = ~a" val)
        (do ((idx 0 (incf idx)))
	  ((>= idx exp))
	  (setf val (/ val 10.0))))
    (format nil "~,2F" val))))


;;; The TI does round correctly with "~,1E" so use this and strip out the digit after 
;;; the decimal point.
(defun fix-roundoff-for-ti (num)
  (let* ((str (format nil "~,1E" num))
	 (dp  (1+ (position #\. str))))
    (concatenate 'string (subseq str 0 dp)
		 (subseq str (1+ dp)))))


;;; A (hopefully) clever border manipulation program.
;;; The idea here is that rather than hard-coding border
;;; information into the program, we build an ordered
;;; list of (name pixels) entries so that when we
;;; need a distance, we just ask for it and it is computed.
;;;
;;; There are two types of border records -- sides and corners.
;;; Sides are ordered from the edge of the window inwards.
;;; Corners are ordered similarly, but they have an
;;; X and a Y component.


(defstruct SideBorder 
  entries)

(defstruct CornerBorder
  Xentries
  Yentries)


;;; Compute the distance from the border edge to the named entry.

(defun border-to (name border &optional (direction nil))
  (do ((entries (if (typep border 'SideBorder)
		    (SideBorder-entries border)
		    (if (eq direction 'X)
			(CornerBorder-Xentries border)
		        (CornerBorder-Yentries border)))
		(cdr entries))
       (distance 0))
      ((or (null entries) (eq name (first (car entries))))
       (if (null entries)
	   (error "BORDER-TO: No entry ~a in border ~a (direction ~a)" name border direction)
	   distance))
    (incf distance (second (car entries)))))




(defun screenx (w x)
  (+ (QgraphWindow-WindowX w) (border-to 'Axis (QgraphWindow-LeftBorder w))
     (cond
       ((eq x '-inf)
	0)
       ((eq x '+inf)
	(QgraphWindow-XUsable w))
       (T
	(+ (QgraphWindow-XMinfW w)
	   (* (- x
		 (QgraphWindow-MinX w)) (QgraphWindow-Xpixels/Unit w)))))))


(defun screeny (w y)
  (- (+ (QgraphWindow-WindowY w) (QgraphWindow-YSize w))
     (border-to 'Axis (QgraphWindow-BottomBorder w))
     (cond
       ((eq y '-inf)
	0)
       ((eq y '+inf)
	(QgraphWindow-YUsable w))
       (T
	(+ (QgraphWindow-YMinfH w)
	   (* (- y
		 (QgraphWindow-MinY w)) (QgraphWindow-Ypixels/Unit w)))))))
  

;;; This stuff is lifted from xgraph.  It supposedly
;;; Sets up reasonable grids if we are doing log scaling.  
;;; I don't pretend to understand it.

(defun init-grid (low step log g)
  (setf (gridjuke-n g) 0)
  (setf (gridjuke-cur g) 0)
  (setf (svref (gridjuke-array g) 0) 0.0)
  (cond
   (log
    (let ((ratio (expt 10 step))
	  x)
      (setf (gridjuke-base g) (floor low))
      (setf (gridjuke-step g) (ceiling step))  ; the step is one order of magnitude
      (incf (gridjuke-n g))
;      (format t "~%low=~a   step=~a" low step)
;      (format t "~%ratio=~a   base=~a   step=~a" ratio (gridjuke-base g) (gridjuke-step g))
      (cond
       ((<= ratio 3.0)
	(cond
	 ((> ratio 2.0)
	  (add-grid 3.0 g))
	 ((> ratio 1.333)
	  (add-grid 2.0 g) (add-grid 5.0 g))
	 ((> ratio 1.25)
	  (add-grid 1.5 g) (add-grid 2.0 g) (add-grid 3.0 g)
	  (add-grid 5.0 g) (add-grid 7.0 g))
	 (T
	  (setf x 1.0)
	  (do ()
	    ((not (and (< x 10.0) (>= (/ (+ x 0.5) (+ x 0.4)) ratio))))
	    (add-grid (+ x .1) g) (add-grid (+ x .2) g) 
	    (add-grid (+ x .3) g) (add-grid (+ x .4) g) 
	    (add-grid (+ x .5) g) 
	    (incf x 0.5))
	  (when (not (= (floor x) x)) 
	    (add-grid (incf x 0.5) g))
	  (do ()
	    ((not (and (< x 10.0) (>= (/ (+ x 1.0) (+ x 0.5)) ratio))))
	    (add-grid (+ x 0.5) g) (add-grid (+ x 1.0) g)
	    (incf x 1.0))
	  (do ()
	    ((not (and (< x 10.0) (>= (/ (+ x 1.0) x) ratio))))
	    (add-grid (+ x 1.0) g)
	    (incf x 1.0))
	  (when (= x 7.0)
            (decf (gridjuke-n g))
	    (setf x 6.0))
	  (when (< x 7.0)
	    (add-grid (+ x 2.0) g))
	  (when (= x 10.0)
	    (decf (gridjuke-n g)))))
	(setf x (- low (gridjuke-base g)))
	(setf (gridjuke-cur g) -1)
	;; Find out which juke we should start with.
	(do ()
	    ((not (>= x (svref (gridjuke-array g) (1+ (gridjuke-cur g))))))
	  (incf (gridjuke-cur g)))))))
   (T
    (setf (gridjuke-step g) (roundup step))
    (setf (gridjuke-base g) (* (floor (/ low (gridjuke-step g)))
			       (gridjuke-step g)))))
  (step-grid g))


(defun add-grid (val g)
  (setf (svref (gridjuke-array g) (gridjuke-n g)) (log val 10))
  (incf (gridjuke-n g))
  ;; Marks the end of the list.
  (setf (svref (gridjuke-array g) (gridjuke-n g)) 1000))

(defun step-grid (g)
  (when (>= (incf (gridjuke-cur g)) (gridjuke-n g))
    (setf (gridjuke-cur g) 0)
    (incf (gridjuke-base g) (gridjuke-step g)))
  (+ (gridjuke-base g) (svref (gridjuke-array g) (gridjuke-cur g))))


;;; Round up the number (actually stepsize) such that
;;; val <= 10^x*(1, 2 or 5)
;;;
(defun roundup (val)
  (let* ((exp (floor (log val 10)))
	 (val/10**exp (/ val (expt 10 exp))))
    (setf val/10**exp
	  (cond
	   ((> val/10**exp 5.0) 10.0)
	   ((> val/10**exp 2.0)  5.0)
	   ((> val/10**exp 1.0)  2.0)
	   (T              1.0)))
    (* val/10**exp (expt 10 exp))))



(defun draw-data (s)
  (dotimes (data-ptr (length (qgraph-data s)))
    (draw-one-dataset data-ptr s)))


(defun draw-one-dataset (data-ptr s)
  (typecase (nth data-ptr (qgraph-data s))
    (Line (draw-data-line data-ptr s))
    (TEnv  (draw-data-tenv data-ptr s))
    (TQual (draw-data-tqual data-ptr s))
    (TQ3   (draw-data-tq3 data-ptr s))
    ))


;;; Get an entry from the color or style slot.  Wraparound if
;;; there aren't that many items.
;;; Inputs:  index - A number >= 0.
;;;          list  - A list.
;;; Returns: An element.
;;;
(defun nth-wrapped (index list)
  (nth (mod index (length list)) list))


;;; Determine the clipping directions for a point.
;;;
(defun clip-direction (x y w)
  (let ((code NIL))
    (if (< x (QgraphWindow-MinX w))
	(push 'LEFT code)
	(if (> x (QgraphWindow-MaxX w)) (push 'RIGHT code)))
    (if (< y (QgraphWindow-MinY w))
	(push 'BOTTOM code)
	(if (> y (QgraphWindow-MaxY w)) (push 'TOP code)))
    code))


;;; Do the clipping. 
;;; Inputs:  ox, oy     - The left point of the segment.
;;;          oclip-code - The clip code for the left point.
;;;          x, y       - The right point of the segment.
;;;          clip-code  - The clip code for the right point.
;;;          w          - A qgraphWindow structure.
;;; Returns: #1     : A plot flag - T if the segment should be plotted
;;;                   and NIL if not.
;;;          #2, #3 : The leftmost point
;;;          #4, #5 : The rightmost point
;;; Notes:   The points are returned even if the segment is completely clipped.
;;;
(defun clip-segment (ox oy oclip-code x y clip-code w)
  (do
   (cd tx ty)
   ((and (null oclip-code) (null clip-code))
    )
    (when (and oclip-code clip-code)
      (return))
    (setf cd (if oclip-code
		 oclip-code
		 clip-code))
    (cond
      ((member 'LEFT cd)
       (setf ty (+ oy (/ (* (- y oy) (- (QgraphWindow-MinX w) ox)) (- x ox))))
       (setf tx (QgraphWindow-MinX w)))
      ((member 'RIGHT cd)
       (setf ty (+ oy (/ (* (- y oy) (- (QgraphWindow-MaxX w) ox)) (- x ox))))
       (setf tx (QgraphWindow-MaxX w)))
      ((member 'BOTTOM cd)
       (setf tx (+ ox (/ (* (- x ox) (- (QgraphWindow-MinY w) oy)) (- y oy))))
       (setf ty (QgraphWindow-MinY w)))
      ((member 'TOP cd)
       (setf tx (+ ox (/ (* (- x ox) (- (QgraphWindow-MaxY w) oy)) (- y oy))))
       (setf ty (QgraphWindow-MaxY w))))
    (cond
      ((eq cd oclip-code)
       (setf ox tx)
       (setf oy ty)
       (setf oclip-code (clip-direction ox oy w)))
      (T
       (setf x tx)
       (setf y ty)
       (setf clip-code (clip-direction x y w)))))
    (values-list (if (and (null oclip-code) (null clip-code))
		     (list T ox oy x y)
		     (list NIL ox oy x y))))

    
;;; Plot a marker.  We assume that the marker is on the plotting surface
;;; (i.e., clip-direction has already been run).
;;; Inputs:  style - A style spec (an atom or 2 element list).
;;;          type  - A type (circles, markers, ...).
;;;          x, y  - The location.
;;;          w     - The window.
;;;          color - The color.
;;; Returns: None, but plots the marker.
;;;
(defun plot-marker (style type x y w color)
  (funcall (case type
	     ((markers circles) #'pos:qplot-circle)
	     (T                 #'pos:qplot-circle))
	   (screenx w x) (screeny w y)
	   (if (consp style) (second style) 1)
	   #-(or symbolics ti) :alu #-(or symbolics ti)
	   color))


;;; Plot an errorbar.  This consists of the centerpoint and
;;; a vertical error bar (with cross bars if unclipped).
;;; Inputs:  clip-dot   - NIL if dot should be plotted.
;;;          x, y       - The location of the dot (already log-scaled if need be).
;;;          p          - The point record from the line structure.
;;;          w          - The QgraphWindow.
;;;          color      - The color.
;;;          qg         - The qgraph structure.
;;; Returns: Nothing, but plots the error bar.
;;;
(defun plot-errorbar (clip-dot x y p w color qg)
  (when (null clip-dot)
    (pos:qplot-circle (screenx w x) (screeny w y) 2
		      #-(or symbolics ti) :alu #-(or symbolics ti)
		      color))
  (let* ((data-len (length P))
	 (top-y  (if (= data-len 3) (+ (elt P 1) (elt P 2)) (elt P 3)))
	 (bot-y  (if (= data-len 3) (- (elt P 1) (elt P 2)) (elt P 2)))
	 scrx
	 cc1 cc2)
    (when (qgraph-LnY qg)
      (setf top-y (log top-y 10))
      (setf bot-y (log bot-y 10)))
    (setf cc1 (clip-direction x top-y w))
    (setf cc2 (clip-direction x bot-y w))
    (multiple-value-bind (plot xu yu xl yl)
	(clip-segment x top-y cc1 x bot-y cc2 w)
      (when plot
	(setf scrx (screenx w xl))
	(pos:qplot-line (screenx w xl) (screeny w yl)
			(screenx w xu) (screeny w yu)
			#+symbolics :thickness #+symbolics 0
			#-(or symbolics ti) :alu #-(or symbolics ti) color
			:dashed nil)
	;; Add the bars to the top and/or bottom if they are on the screen.
	(when (= top-y yu)
	  (pos:qplot-line (- scrx *BDR*) (screeny w yu)
			  (+ scrx *BDR*) (screeny w yu)))
	(when (= bot-y yl)
	  (pos:qplot-line (- scrx *BDR*) (screeny w yl)
			  (+ scrx *BDR*) (screeny w yl)))))))


;;; last-segment-only Finds the last segment of the line and draws
;;; only that.
;;; Modified by BKay 2Jun92 to ignore lines with no points.
;;;
(defun draw-data-line (data-ptr s &key (last-segment-only NIL))
  (let* ((d       (nth data-ptr (qgraph-data s)))
	 (w       (qgraph-window s))  ; pass in for zooming
	 (color   (nth-wrapped data-ptr (qgraph-colors s)))
	 (style   (nth-wrapped data-ptr (qgraph-styles s)))
	 (marker-style (or (member style '(markers circles))
			   (and (consp style)
				(member (first style)
					'(markers circles)))))
	 (dashed  nil)
	 ox oy x y
	 oclip-code clip-code plot
	 Pn Pn-1)
    (when (> (qlength (Line-data d)) 0)

      ;; Qplot-line doesn't know how to handle a dash spec of (1 0) (i.e.,
      ;; no dashes).  For no dashes, the dashed keyword must be NIL.  This
      ;; next line checks for a 0 second arg in the style and sets dashed
      ;; appropriately.
      (setf dashed (and (not marker-style) (not (eq style 'errorbars))
			(not (zerop (second style)))))
      ;; Set up the graphics context (only neede for lispview)
      (with-qplot-gcon (:color  color
			:dashed dashed
			:dash-pattern (if dashed style '(10 10)))

	(cond
	  ;; For "last segment only" mode, we find the last two points of
	  ;; the line.  If we are using markers, then only the last one is needed.
	  ;; If there's only one datapoint, then ignore the "n-1st" one.
	  (last-segment-only
	   (setf Pn (car (queue-tail (Line-data d))))
	   (setf Pn-1 (if (and (> (qlength (Line-data d)) 1)
			       (not marker-style)
			       (not (eq style 'errorbars)))
			  (elt (qlist (Line-data d)) (- (qlength (Line-data d)) 2))
			  nil))
	   (cond
	     (Pn-1
	      (setf x (elt Pn-1 0))
	      (setf y (elt Pn-1 1)))
	     (t
	      (setf x (elt Pn 0))
	      (setf y (elt Pn 1)))))
	  (t
	   (setf x (elt (qtop (Line-Data d)) 0))
	   (setf y (elt (qtop (Line-Data d)) 1))))
	
	(when (qgraph-LnX s) (setf x (log x 10)))
	(when (qgraph-LnY s) (setf y (log y 10)))
	
	;; Determine clipping code for (x, y)
	;; LEFT, RIGHT, TOP, BOTTOM, or NIL
	;; This must happen AFTER the point is log-scaled since the axes
	;; limits are stored this way.
	(setf clip-code (clip-direction x y w))
	(cond
	  ((and marker-style (null clip-code))
	   ;; Plots the first point if this is a marker.
	   ;; If it's a true line, the loop will take care of it.
	   (plot-marker style (first marker-style) x y w color))

	  ((eq style 'errorbars)
	   (plot-errorbar clip-code x y (if last-segment-only
					    (car (queue-tail (Line-Data d)))
					    (qtop (Line-data d)))
			  w color s)))

	;; If we were just printing a single marker or there isn't a line to draw
	;; then don't plot the line segments.
	(unless (and last-segment-only (null Pn-1))
	  (dolist (point (if last-segment-only
			     (queue-tail (Line-data d))
			     (cdr (qlist (Line-Data d)))))
	    (setf ox x)
	    (setf oy y)
	    (setf oclip-code clip-code)
	    (setf x (elt point 0))
	    (setf y (elt point 1))
	    (when (qgraph-LnX s) (setf x (log x 10)))
	    (when (qgraph-LnY s) (setf y (log y 10)))
	    (setf clip-code (clip-direction x y w))
	    (cond
	      ((and marker-style (null clip-code))
	       (plot-marker style (first marker-style) x y w color))
	      ((eq style 'errorbars)
	       (plot-errorbar clip-code x y point w color s))
	      (T
	       (multiple-value-setq (plot ox oy x y)
		 (clip-segment ox oy oclip-code x y clip-code w))
	       (when plot
		 (pos:qplot-line (screenx w ox) (screeny w oy)
				 (screenx w x)  (screeny w y)
				 #+symbolics :thickness #+symbolics 0
				 #-(or symbolics ti) :alu #-(or symbolics ti) color
				 :dashed dashed
				 ;; This is necessary because a dash-pattern of
				 ;; (X 0) screws up lispview even when dashed = NIL
				 :dash-pattern (if dashed style '(10 10))))))
	    ))))))

					

;;; Modified by BKay 2Jun92 to ignore lines with no points.
;;;
(defun draw-data-tenv (data-ptr s &key (last-segment-only nil))
  (let ((d       (nth data-ptr (qgraph-data s)))
	(w       (qgraph-window s))  ; pass in for zooming
	(color   (nth-wrapped data-ptr (qgraph-colors s)))
	(style   (nth-wrapped data-ptr (qgraph-styles s)))
	(dashed  nil)
	ox oy1 oy2 x y1 y2
	oclip-code1 oclip-code2 clip-code1 clip-code2
	plot)
    (when (> (qlength (Line-data d)) 0)
      ;; Qplot-line doesn't know how to handle a dash spec of (1 0) (i.e.,
      ;; no dashes).  For no dashes, the dashed keyword must be NIL.  This
      ;; next line checks for a 0 second arg in the style and sets dashed
      ;; appropriately.
      (setf dashed (not (zerop (second style))))
      
      ;; Set up the graphics context (only neede for lispview)
      (with-qplot-gcon (:color color
			:dashed dashed
			:dash-pattern (if dashed style '(10 10)))
	(cond
	  ((and last-segment-only
		(> (qlength (TEnv-data d)) 1))
	   (let ((Pn-1 (elt (qlist (TEnv-data d)) (- (qlength (TEnv-data d)) 2))))
	     (setf x (elt Pn-1 0))
	     (setf y1 (elt Pn-1 1))
	     (setf y2 (elt Pn-1 2))))
	  (T
	   (setf x  (elt (qtop (TEnv-Data d)) 0))
	   (setf y1 (elt (qtop (TEnv-Data d)) 1))
	   (setf y2 (elt (qtop (TEnv-Data d)) 2))))
	
	(when (qgraph-LnX s) (setf x (log x 10)))
	(when (qgraph-LnY s) (setf y1 (log y1 10)) (setf y2 (log y2 10)))
	
	(setf clip-code1 (clip-direction x y1 w))
	(setf clip-code2 (clip-direction x y2 w))
	
	(dolist (point (if last-segment-only
			   (queue-tail (TEnv-data d))
			   (cdr (qlist (TEnv-Data d)))))
	  (setf ox x)
	  (setf oy1 y1)
	  (setf oy2 y2)
	  (setf oclip-code1 clip-code1)
	  (setf oclip-code2 clip-code2)
	  (setf x  (elt point 0))
	  (setf y1 (elt point 1))
	  (setf y2 (elt point 2))
	  (when (qgraph-LnX s) (setf x (log x 10)))
	  (when (qgraph-LnY s) (setf y1 (log y1 10))  (setf y2 (log y2 10)))
	  (setf clip-code1 (clip-direction x y1 w))
	  (setf clip-code2 (clip-direction x y2 w))
	  
	  (multiple-value-setq (plot ox oy1 x y1)
	    (clip-segment ox oy1 oclip-code1 x y1 clip-code1 w))
	  (when plot
	    (pos:qplot-line (screenx w ox) (screeny w oy1)
			    (screenx w x)  (screeny w y1)
			    #+symbolics :thickness #+symbolics 0
			    #-(or symbolics ti) :alu #-(or symbolics ti) color
			    :dashed dashed
			    ;; This is necessary because a dash-pattern of
			    ;; (X 0) screws up lispview even when dashed = NIL
			    :dash-pattern (if dashed style '(10 10))))
	  
	  (multiple-value-setq (plot ox oy2 x y2)
	    (clip-segment ox oy2 oclip-code2 x y2 clip-code2 w))
	  (when plot
	    (pos:qplot-line (screenx w ox) (screeny w oy2)
			    (screenx w x)  (screeny w y2)
			    #+symbolics :thickness #+symbolics 0
			    #-(or symbolics ti) :alu #-(or symbolics ti) color
			    :dashed dashed
			    :dash-pattern (if dashed style '(10 10)))))
	))))


(defun draw-data-tqual (data-ptr s)
  (let* ((d              (nth data-ptr (qgraph-data s)))
	 (w              (qgraph-window s))  ; pass in for zooming
	 ;; Color and style entries should be a two-element list, but if they
	 ;; aren't then just use the same entry for ranges and intervals.
	 (color-entry    (nth-wrapped data-ptr (qgraph-colors s)))
	 (style-entry    (nth-wrapped data-ptr (qgraph-styles s)))
	 (range-color    (if (consp color-entry) (first color-entry) color-entry))
	 (interval-color (if (consp color-entry) (second color-entry) color-entry))
	 ;; style is a number representing dot frequency
	 (range-style    (if (consp style-entry) (first style-entry) style-entry))
	 (interval-style (if (consp style-entry) (second style-entry) style-entry))
	x y)
    (dolist (point (TQual-Data d))
      (cond
	;; A time point.
	((atom (first point))
	 (setf x (get-range-from-lmarks (first point) (first (tqual-lmarks d))))
	 (setf y (get-range-from-lmarks (second point) (second (tqual-lmarks d))))
	 (draw-qual-box x y range-style range-color w 
			:qdir (third point)
			:print-coord (case (qgraph-QualPlottingMethod s)
				       (BOX        'LL)
				       (BOX-CENTER 'CENTER)
				       (T          NIL))
			:xname (first point) :yname (second point)
			:point-markers T
			:LnX (qgraph-LnX s) :LnY (qgraph-LnY s))
	 )
	;; A time interval.
	(T
	 (setf x (get-range-from-lmarks (first point) (first (tqual-lmarks d))))
	 (setf y (get-range-from-lmarks (second point) (second (tqual-lmarks d))))
	 (draw-qual-box x y interval-style interval-color w 
			:LnX (qgraph-LnX s) :LnY (qgraph-LnY s))
	 )))))


;;; BKay 18Jan91
(defun draw-data-tq3 (data-ptr s)
  (let ((d       (nth data-ptr (qgraph-data s)))
	(w       (qgraph-window s))  ; pass in for zooming
	(range-color    (or (first (nth-wrapped data-ptr (qgraph-colors s)))
			    "red"))
	(interval-color (or (second (nth-wrapped data-ptr (qgraph-colors s)))
			    "red"))
	;; style is a number representing dot frequency
	(range-style    (or (first (nth-wrapped data-ptr (qgraph-styles s)))
			    1))
	(interval-style (or (first (nth-wrapped data-ptr (qgraph-styles s)))
			    1))
	x y qdir
	(prev-x NIL)
	(prev-y NIL))
    ;; There are only time points in the TQ3-Data slot, so figure out the
    ;; intervals as needed.
    (dolist (point (TQ3-Data d))
      (setf x (get-range-from-lmarks (first point) (first (tq3-lmarks d))))
      (setf y (second point))
      (draw-qual-box x y range-style range-color w 
		     :qdir qdir
		     :print-coord (case (qgraph-QualPlottingMethod s)
				    (BOX        'LL)
				    (BOX-CENTER 'CENTER)
				    (T          NIL))
		     :xname (first point) :yname (second point)
		     :point-markers T
		     :LnX (qgraph-LnX s) :LnY (qgraph-LnY s))
      (when (and prev-x interval-style) ; we don't do it unless there's an interval request
	(draw-qual-box (list (first prev-x) (second x))
		       (list (qgraph-emin (first prev-y) (first y))
			     (qgraph-emax (second prev-y) (second y)))
		       interval-style interval-color w 
		       :LnX (qgraph-LnX s) :LnY (qgraph-LnY s)))
      (setf prev-x x)
      (setf prev-y y))))


;;; BKay 18Jan91
;;; Get a range (L H) for a lmark or an interval given the lmark set in 
;;; TQual-lmarks format.
;;;
(defun get-range-from-lmarks (lmark-or-int lmarks)
  (let* ((low-lmark (if (listp lmark-or-int) (first lmark-or-int) lmark-or-int))
	 (up-lmark  (if (listp lmark-or-int) (second lmark-or-int) lmark-or-int)))
    (list (second (assoc low-lmark lmarks))
	  (third  (assoc up-lmark  lmarks)))))


;;; BKay 18Jan91 
;;; BKay 21Jan91 - renamed from emin and emax
(defun qgraph-emin (e1 e2)
  (cond
    ((and (numberp e1) (numberp e2))      (min e1 e2))
    ((or (eq e1 '-inf) (eq e2 '-inf))     '-inf)
    ((numberp e1)                         e1)
    ((numberp e2)                         e2)
    (T                                    '+inf)))

(defun qgraph-emax (e1 e2)
  (cond
    ((and (numberp e1) (numberp e2))      (max e1 e2))
    ((or (eq e1 '+inf) (eq e2 '+inf))     '+inf)
    ((numberp e1)                         e1)
    ((numberp e2)                         e2)
    (T                                    '-inf)))

;;; BKay 18Jan91  Changed calling sequence so that xrange, yrange, and point-qdir are passed in.
;;;      22Jan91  Added point-markers key.
;;; Inputs: xrange - the range of x (L H)
;;;         yrange - the range of y (L H)
;;;         style  - a style number or NIL
;;;         color  - a color spec
;;;         w      - a window structure
;;;         qdir   - inc, dec, std, or NIL (for no display)
;;;         print-coord - LL -> lower left of bx, CENTER -> centered, NIL -> ignore
;;;         point-markers - T if zerowidth&height points show be specially marked
;;;         xname  - the name of the x coord (usually a Lmark or interval)
;;;         yname  - the name of the y coord
;;;         LnX    - log scaling on X
;;;         LnY    - log scaling on Y
;;;
(defun draw-qual-box (xrange yrange style color w 
		      &key  (qdir NIL) (print-coord nil) (point-markers NIL)
		      (xname NIL) (yname NIL) 
		      (LnX nil) (LnY nil))
  (when (not (null style))
    (let* ((low-x  (first xrange))
	   (up-x   (second xrange))
	   (low-y  (first yrange))
	   (up-y   (second yrange))
	   plot)
      (when LnX
	(setf low-x (log low-x 10))
	(setf up-x  (log up-x 10)))
      (when LnY
	(setf low-y (log low-y 10))
	(setf up-y  (log up-y 10)))

      (multiple-value-setq (plot low-x up-x low-y up-y)
	(clip-tqual-box low-x up-x low-y up-y w))

      (when plot
	(setf low-x (screenx w low-x))
	(setf up-x  (screenx w up-x))
	(setf low-y (screeny w low-y))
	(setf up-y  (screeny w up-y))
	;; vline draws downwards.  hline draws right
	(qgraph-draw-dots-vline low-x up-y  (abs (- up-y low-y))
				style color)
	(qgraph-draw-dots-hline low-x up-y  (abs (- up-x low-x))
				style color)
	(qgraph-draw-dots-vline up-x  up-y  (abs (- up-y low-y))
				style color)
	(qgraph-draw-dots-hline low-x low-y (abs (- up-x low-x))
				style color)
	;; Print the symbol only for ranges, not intervals
	(when qdir
	  (pos:qplot-symbol (/ (+ low-x up-x) 2) (/ (+ low-y up-y) 2) qdir))
	;; Add a crossbar to the range if it is only a line
	(when (and (= up-y low-y) (not (= up-x low-x)))  ; not if a point
	  (qgraph-draw-dots-vline low-x (- up-y *BDR*) (* 2 *BDR*)
				  style color)
	  (qgraph-draw-dots-vline up-x (- up-y *BDR*) (* 2 *BDR*)
				  style color))
	(when (and (= up-x low-x) (not (= up-y low-y)))
	  (qgraph-draw-dots-hline (- low-x *BDR*)  low-y (* 2 *BDR*)
				  style color)
	  (qgraph-draw-dots-hline (- up-x *BDR*) up-y (* 2 *BDR*)
				  style color))
	;; Draw the dot for a zero length and width point a little bigger
	(when (and point-markers (eql (first xrange) (second xrange))
		   (eql (first yrange) (second yrange)))
	  (pos:qplot-circle low-x low-y 1 :thickness 1 
			    #-(or symbolics ti) :alu #-(or symbolics ti) color))
	(when print-coord
	  (case print-coord
	    ;; Addition of *BDR* gives spacing between qdir symbol and coord.
	    (LL 	  (qgraph-plot-string (format nil "~a,~a" xname yname)
					      (+ low-x *BDR*) low-y 'LEFT w))
	    (CENTER (qgraph-plot-string (format nil "~a,~a" xname yname)
					(+ (/ (+ low-x up-x) 2) *BDR*) (/ (+ low-y up-y) 2)
					'LEFT w))))
	))))


;;; Do a quick and dirty clip operation for qual boxes.
;;;
(defun clip-tqual-box (leftx rightx lowy upy w)
  (let ((llccode (qual-clip-direction leftx lowy w))
	(lrccode (qual-clip-direction rightx lowy w))
	(ulccode (qual-clip-direction leftx upy w))
	(urccode (qual-clip-direction rightx upy w)))
    (qual-clip-box leftx rightx lowy upy llccode lrccode ulccode urccode w)))

    

;;; Compute the clipping direction assuming that values include +/-inf
;;;
(defun qual-clip-direction (x y w)
  (let ((code NIL))
    (if (or (and (eq x '-inf) (null (QgraphWindow-XMinfW w)))
	    (and (numberp x) (< x (QgraphWindow-MinX w))))
	(push 'LEFT code)
	(if (or (and (eq x '+inf) (null (QgraphWindow-XInfW w)))
		(and (numberp x) (> x (QgraphWindow-MaxX w))))
	    (push 'RIGHT code)))
    (if (or (and (eq y '-inf) (null (QgraphWindow-YMinfH w)))
	    (and (numberp y) (< y (QgraphWindow-MinY w))))
	(push 'BOTTOM code)
	(if (or (and (eq y '+inf) (null (QgraphWindow-YInfH w)))
		(and (numberp y) (> y (QgraphWindow-MaxY w))))
	    (push 'TOP code)))
    code))


;;; This takes into account +/-inf x and y values.
;;; It assumes that we are clipping segments of a box that is aligned with the
;;; axes.
;;;
(defun qual-clip-box (leftx rightx lowy upy llcc lrcc ulcc urcc w)
  (loop for i in llcc
	do (case i
	     (bottom      (setf lowy  (QgraphWindow-MinY w)))
	     (left        (setf leftx (QgraphWindow-MinX w)))
	     ((right top) (return-from qual-clip-box
			    (values NIL leftx rightx lowy upy)))))
  (loop for i in lrcc
	do (case i
	     (bottom     (setf lowy   (QgraphWindow-MinY w)))
	     (right      (setf rightx (QgraphWindow-MaxX w)))
	     ((left top) (return-from qual-clip-box
			   (values NIL leftx rightx lowy upy)))))
  (loop for i in ulcc
	do (case i
	     (top    (setf upy   (QgraphWindow-MaxY w)))
	     (left   (setf leftx (QgraphWindow-MinX w)))
	     ((right bottom) (return-from qual-clip-box
			       (values NIL leftx rightx lowy upy)))))
  (loop for i in urcc
	do (case i
	     (top    (setf upy    (QgraphWindow-MaxY w)))
	     (right  (setf rightx (QgraphWindow-MaxX w)))
	     ((left bottom) (return-from qual-clip-box
			      (values NIL leftx rightx lowy upy)))))
  (values T leftx rightx lowy upy))



(defun qgraph-draw-dots-vline (x1 y1 len style color)
  (if (= style 1) 
      (pos:qplot-vline x1 y1 len #-(or symbolics ti) :alu #-(or symbolics ti) color)
      (do 
	((y y1 (+ y style)))
	((>= y (+ y1 len)))
	(pos:qplot-dot x1 y #-(or symbolics ti) :alu #-(or symbolics ti) color))))

(defun qgraph-draw-dots-hline (x1 y1 len style color)
  (if (= style 1) 
      (pos:qplot-hline x1 y1 len #-(or symbolics ti) :alu #-(or symbolics ti) color)
      (do 
	((x x1 (+ x style)))
	((>= x (+ x1 len)))
	(pos:qplot-dot x y1 #-(or symbolics ti) :alu #-(or symbolics ti)  color))))

	 
(defun draw-legend (s)
  (do* ((datasets (qgraph-data s) (cdr datasets))
	(ptr      0               (incf ptr))
	(w        (qgraph-window s))		
	(x        (- (+ (QgraphWindow-WindowX w) (QgraphWindow-XSize w))	
		     (border-to 'TrendNameR (QgraphWindow-RightBorder w))))
	(y        (+ (QgraphWindow-WindowY w) 
		     (border-to 'TopOfLegendNameB (QgraphWindow-TopLeftBorder w) 'Y))
		  (+ y (+ *INTER-LEGEND-SPACING* (QgraphWindow-FontCharHeight w)))))
       ((null datasets))
    (draw-legend-string (typecase (car datasets)
			  (Line (Line-TrendName (car datasets)))
			  (TEnv (Tenv-TrendName (car datasets)))
			  (TQual (Tqual-TrendName (car datasets)))
			  )
			;; DESIGN DECISION: TQuals will only show their
			;; interval legends.  This could change.
			(if (typep (car datasets) 'tqual)
			    (second (nth ptr (qgraph-colors s)))
			    (nth-wrapped ptr (Qgraph-colors s)))
			(if (typep (car datasets) 'tqual)
			    (second (nth ptr (qgraph-styles s)))
			    (nth-wrapped ptr (Qgraph-styles s)))
			;; The pixel-length is in the RightBorder
			(cadr (assoc 'TrendNameR
				     (SideBorder-Entries
				      (QgraphWindow-RightBorder w))))
			w x y)))
      

;;; Length is in pixels.
;;;
(defun draw-legend-string (str color style pixellen w x y)
  (when str
    (draw-legend-name str w (- x pixellen) y)
    (draw-legend-stylecolor pixellen style color w x y)))


(defun draw-legend-name (str w x y)
  (qgraph-plot-string str x y 'LOWERLEFT w
		      ))

(defun draw-legend-stylecolor (pixellen style color w x y)
  (cond
    ((typep style 'number)
     ;; Tqual styles are numbers and graph as points.
     (qgraph-draw-dots-hline (- x pixellen)
			     (- y 
				(QgraphWindow-FontCharHeight w)
				(/ *INTER-LEGEND-SPACING* 3))
			     pixellen
			     style color))
    (T
     ;; Otherwise, they graph as broken lines
     (let ((dashed (not (zerop (second style)))))
       (pos:qplot-hline (- x pixellen)
			(- y 
			   (QgraphWindow-FontCharHeight w)
			   (/ *INTER-LEGEND-SPACING* 3))
			pixellen
			:dashed dashed
			:dash-pattern (if dashed style '(10 10))
			#-(or symbolics ti) :alu #-(or symbolics ti) color)))
    ))

