;;; -*- Syntax: Common-lisp; Package: POS; Mode:Common-Lisp; Base:10  -*-
;;; Copyright 1988 David Throop at the University of Texas at Austin

(in-package :pos)

#+:ccl
(use-package :ccl :pos)


#+symbolics
(use-package :scl :pos)

#+symbolics
(import '(self send defmethod))


;;; This file is device independent; all code should be CommonLisp.

;;; EVERYTHING BETWEEN THE ############'S  IS NO LONGER NECESSARY AND
;;; HAS BEEN REMOVED.  IT WAS USED BY PS-STRING WHICH HAS BEEN REDEFINED.
;;; BKay 22Apr92
;;;########################################################
;;; Special handling to get strings output with the right escapes for postscript.
;;; Fortunately, postscript uses the same escape character, the backslash, as does
;;; Common-Lisp.  However, a different set of characters need to be slashified.  Some things
;;; that Lisp slashifies are turned to alphanumeric, (by giving them the same
;;; characteristics as #\a) and some that it treats as other things need to be slashified
;;; (by giving them the same characteristics as #\").

;;; Anytime a string is written to the postscript file, the *ps-out-readtable* is used to
;;; convert it to a string with the proper slashes.

;(defparameter *ps-out-readtable* (copy-readtable))	; Copy from  standard LISP readtable.
;
;(mapc #'(lambda (char)                          ; Set these three chars to alphanumeric
;            (set-syntax-from-char
;              char #\a *ps-out-readtable*))
;       (list #\/ #\" #\| ))
;
;(mapc #'(lambda (char)                          ; Set these four chars to be slashified.
;            (set-syntax-from-char
;              char #\" *ps-out-readtable*))
;        (list #\\ #\( #\) #\%))
;
;;;########################################################

(defmacro fp (&rest args)
  `(format *postscript-output-file* ,@args))

(defun compress-commands ()
  (fp " /settings save def~&~
	/as {arc stroke} def             /af {arc fill} def              /b  /closepath load def~&~
	/c  {0 360 arc fill} def         /d  {moveto (\\56) show} def    /f  /setfont load def~&~
	/g  /setgray load def            /gs /gsave load def             /gr /grestore load def~&~
	/i  /fill load def               /k  {currentpoint newpath moveto} def~&~
	/l  /lineto load def             /m  /moveto load def            /n  /newpath load def~&~
	/o  {0 360 arc} def              /p  /show load def              /pc /currentpoint load def~&~
	/q  /rotate load def             /r  {rlineto stroke} def        /rl /rlineto load def~&~
	/rm /rmoveto load def            /s  /stroke load def            /sc /scale load def~&~
        /sd /setdash load def~&~
	/sn {scale newpath} def          /t  /translate load def         /w  /setlinewidth load def~&"))

(defun restore-environment ()
  (fp "~&settings restore~&"))

;; THIS IS SYMBOLICS-DEPENDENT AND HAS BEEN REPLACED WITH THE FOLLOWING CODE.
;; HOPEFULLY, IT WORKS IDENTICALLY.  BKay 22Apr92
;; Given a string, insert the proper slashes.  Shove the string through FORMAT, using
;; the ~S directives.  Use a special readtable, to suppress LISP's own
;; slashification.  Then trim off the doublequotes.  QPLOT-STRING is sometimes given
;; a symbol to plot, so the STRING argument here sometimes isn't.
;
;(defun ps-string (string)               
;  (let ((*readtable* *ps-out-readtable*))
;    (typecase  string
;      (string (let ((cv-string (format nil "~s" string)))
;                (subseq cv-string 1 (1- (length cv-string)))))
;      ;; Suppress package prefixes for symbols and suppress ||s for symbols with
;      ;; lowercase.
;      (symbol (let ((*package* (symbol-package string)))
;                (format nil "~a" string)))      
;      (t (format nil "~s" string)))))

;;; Given an object, generate a string slashified for postscript.
;;;
(defun ps-string (object)		
    ;; Convert the following:
    ;; (  -> \(                 ; Balancing parens for emacs )(
    ;; )  -> \)   
    ;; %  -> \%
    (coerce (loop for c in (coerce (format nil "~a" object) 'list)
		  nconc (if (or (eq c #\() (eq c #\)) (eq c #\%))
			    (list #\\ c)
			    (list c)))
	    'string))


(defun ps-trim (number &key (string (string (format nil "~,1f" number))))
  (let ((len (length string)))
    (if (and (equal #\0 (aref string (1- len)))
	     (> len 1))
	(subseq string 0 (- len 2))
	string)))

(defmacro ps-trim-y (number)
  `(ps-trim (- yscreen ,number)))

;;;  BASIC DRAWING FUNCTIONS - all of these functions take the same arguments as does the
;;;  corresponding dw::dynamic-lisp-listener method (although there are some optional
;;;  arguments that a dynamic window takes that I haven't implemented - none of the code in
;;;  QSIM uses these options.)

;;; PS-DRAW-STRING uses SI:CHANGE-STRING-CHARACTER-STYLE, which is a zetalisp function.
;;; Unfortuneately, character information is not standard in Common-Lisp.  If this code is
;;; ported to a system that does not implement character information, the call (and its Nil
;;; list argument) can be removed.  PS-DRAW-STRING tolerates nonstrings in its String arg.
	
(defun ps-draw-string (string x y &key (alu *black*) (font *plain-font*)
		       &aux (*print-case* :upcase))
  (unless (zerop (length (format nil "~a" string)))
    (checkgray alu)
    (get-font-for-string string font)
    (fp " ~a ~a m (~a) p~&" (ps-trim x) (ps-trim (- yscreen y))
	    (strip-font-info (ps-string string)))))	; Slashify #\\, #\( and #\).

;;; PS-DRAW-SUPERSCIPTED-STRING is similar to PS-DRAW-STRING; see the
;;; comments at QPLOT-SUPERSCRIPTED-STRING for explanation.

(defun ps-draw-superscipted-string
       (string superscript x y superscript-height xadv alu font superscript-font
				     &aux (*print-case* :upcase))
  (unless (zerop (length (format nil "~a" string)))
    (checkgray alu)
    (get-font-for-string string font)
    (fp " ~a ~a m (~a) p " (ps-trim x) (ps-trim (- yscreen y))
	    (strip-font-info (ps-string string)))
    (get-font-for-string string  superscript-font)
    (format *postscript-output-file* " ~a ~a rm (~a) p ~&"  ; ~& added by BKay 11Oct91
	    (ps-trim xadv)(ps-trim superscript-height)
	    (strip-font-info (ps-string superscript )))))

(defun ps-draw-line (x1 y1 x2 y2 &key (thickness 1)(alu *black*) )
  (let ((x1s (ps-trim x1))(y1s (ps-trim (- yscreen y1)))
	(xrs (ps-trim (- x2 x1)))(yrs (ps-trim (- y1 y2))))
  (checkwidth thickness)
  (checkgray alu)
  (unless (and (string-equal xrs "0")(string-equal yrs "0")))
    (fp
	    "~a ~a m ~a ~a r ~&"  ; Added ~& to break up output BKay 11Oct91
	       x1s y1s xrs yrs)))

(defun ps-draw-dashed-line (x1 y1 x2 y2 &key (thickness 1)(alu *black*) (dash-pattern '(4 4)))
  (checkwidth thickness)
  (checkgray alu)
  (fp
	  " gs [~s ~s] 2 sd n ~a ~a m  ~a ~a l s gr~&"
	  (first dash-pattern)(second dash-pattern)(ps-trim x1)
	  (ps-trim (- yscreen y1)) (ps-trim x2) (ps-trim (- yscreen y2))))

;;; ARROW-HEAD-LENGTH is set equal to GRAPHICS::*DEFAULT-ARROW-LENGTH*.
;;; ARROW-BASE-WIDTH is set equal to GRAPHICS::*DEFAULT-ARROW-WIDTH*

(defun ps-draw-vector (x1 y1 x2 y2 &key
		       (arrow-head-length 10)	
		       (arrow-base-width 5)	
		       (alu *black*) (filled nil)
		       (shaftthick 1))
;;;  (format t "Points before : x1 ~d  y1 ~d   x2 ~d  y2 ~d~%" x1 y1 x2 y2)
  (multiple-value-bind  (points xbas ybas)
      (triangle-point-translation x1 y1 x2 y2 arrow-base-width arrow-head-length)
;;;	x1 y1 x2 y2 (* shaftthick arrow-base-width)
;;;	(* shaftthick  arrow-head-length))
;;; (format t "Points after triangle translation : points : ~a   xbas : ~d  ybas ~d~%"  points xbas ybas)
    (ps-draw-line x1 y1 xbas ybas
		  :thickness shaftthick :alu alu)
    (ps-draw-polygon points :filled filled :alu alu)))

(defun triangle-point-translation (from-x from-y to-x to-y
				   arrow-base-width arrow-head-length
				   &aux (halfbase (/ arrow-base-width 2.0)))
  (let* ((dy (- to-y from-y))
	 (dx (float (- to-x from-x)))
	 (alpha (atan dy dx))
	 (beta (atan halfbase arrow-head-length))
	 (len (sqrt (float (+ (* arrow-head-length arrow-head-length)
			      (* halfbase halfbase)))))
	 (xt (- to-x (* len (cos (- alpha beta)))))
	 (yt (- to-y (* len (sin (- alpha beta)))))	
	 (xb (- to-x (* len (cos (+ alpha beta)))))
	 (yb (- to-y (* len (sin (+ alpha beta)))))
	 (xbas (/ (+ xt xb) 2))
	 (ybas (/ (+ yt yb) 2)))
    (values (list to-x to-y xt yt xb yb)  xbas ybas)))


(defvar *pi-degrees* (/ 180 pi))

(defun ps-draw-circle (x y rad &key (alu *black*) (thickness 1)(filled nil)
		       (start-angle 0)(end-angle (* 2 pi)))
  (checkwidth thickness)
  (checkgray alu)
  ;; The NEWPATH (n) has to be output to stop connecting line from current point.
  (if  (= (abs (coerce (-  end-angle start-angle) 'single-float)) 6.2831855)
       (fp					; This is part of image compression
	 "n ~a ~a ~a o ~:[s~;i~] "
	 (ps-trim x) (ps-trim (- yscreen y)) (ps-trim rad)  filled)
       (fp
	 "n ~a ~a ~a ~a ~a ~:[as~;af~]~&"
	 (ps-trim x) (ps-trim (- yscreen y)) (ps-trim rad) (ps-trim (* end-angle -1 *pi-degrees*))
	 (ps-trim (* start-angle -1 *pi-degrees*)) filled)))

(defun ps-draw-oval (center-x center-y x-radius y-radius
		     &key (alu *black*) (thickness 1)(filled nil))
  ;; The oval is completely implemented in terms of lines and half circles.
  (let ((dif (- x-radius y-radius)))
    (cond  ((> x-radius y-radius)
	    (cond (filled			; When filled, just one thick line
		   (ps-draw-line (- center-x dif) center-y (+ center-x dif) center-y
				 :thickness (* 2 y-radius) :alu alu))
		  ((not filled)
		   (ps-draw-line (- center-x dif) (+ center-y y-radius) (+ center-x dif) (+ center-y y-radius)
				 :thickness thickness :alu alu)
		   (ps-draw-line (- center-x dif) (- center-y y-radius) (+ center-x dif) (- center-y y-radius)
				 :thickness thickness :alu alu)))
	    (ps-draw-circle (+ center-x dif) center-y y-radius :thickness thickness :alu alu
			    :filled filled
			    :start-angle (/ pi -2) :end-angle (/ pi 2))
	    (ps-draw-circle (- center-x dif) center-y y-radius :thickness thickness :alu alu
			    :filled filled
			    :start-angle (/ pi 2) :end-angle (* 1.5 pi)))
	   ((= x-radius y-radius)
	    (ps-draw-circle center-x center-y x-radius
			    :alu alu
			    :thickness thickness :filled filled))
	   ((< x-radius y-radius)
	    (cond (filled 
		   (ps-draw-line center-x (- center-y dif) center-x (+  center-y dif)
				 :thickness (* 2 x-radius) :alu alu))
		  ((not filled)
		   (ps-draw-line (+ center-x x-radius) (- center-y dif) (+ center-x x-radius) (+ center-y dif)
				 :thickness thickness :alu alu)
		   (ps-draw-line (- center-x x-radius) (- center-y dif) (- center-x x-radius) (+ center-y dif)
				 :thickness thickness :alu alu)))
	    (ps-draw-circle center-x (+ center-y dif) x-radius :thickness thickness :alu alu
			    :filled filled
			    :start-angle 0 :end-angle pi)
	    (ps-draw-circle center-x (- center-y dif) x-radius :thickness thickness :alu alu
			    :filled filled
			    :start-angle pi :end-angle (* pi 2))))))



(defun ps-draw-ellipse (center-x center-y x-radius y-radius
			&key (alu *black*) (thickness 1)(filled nil)
			(start-angle 0)(end-angle (* 2 pi))
			(rotation 0))
  (flet ((rotat-and-dg (angle)
	   (*  *pi-degrees* (+ pi angle))))
	   
    (checkwidth (max thickness 1))
    (checkgray alu)
    ;; The NEWPATH has to be output to stop connecting line from current point.
;;;    (fp "gs ~,2f ~,2f t 1~&"
;;;	    (ps-trim center-x) (ps-trim (- yscreen center-y)))
    (fp "gs ~a ~a t 1~&" (ps-trim center-x) (ps-trim (- yscreen center-y)))
    (unless (zerop rotation)			; Rotation in postscript is opposite
;;;      (fp "~,1f q "	; Symbolics graphics rotation
;;;	      (ps-trim (* -1 *pi-degrees* rotation))))
      (fp "~a q "	; Symbolics graphics rotation
	(ps-trim (* -1 *pi-degrees* rotation))))
;;;    (fp
;;;	    "~,4f sn 0 0 ~,1f ~,1f ~,1f ~:[as~;af~] gr "
;;;	    (/ y-radius x-radius) (ps-trim x-radius) (ps-trim (rotat-and-dg start-angle))
;;;	    (ps-trim (rotat-and-dg end-angle)) filled)))
    (fp
	    "~,4f sn 0 0 ~a ~a ~a ~:[as~;af~] gr "
	    (/ y-radius x-radius) (ps-trim x-radius) (ps-trim (rotat-and-dg start-angle))
	    (ps-trim (rotat-and-dg end-angle)) filled) ))

;; Draw a 1 pixel point on the screen.

(defun ps-draw-point (x y &key (alu *black*) (font *plain-font*))
  (checkgray alu)
  (get-font-for-string "." font)
  (fp
	  "~a ~a d " (ps-trim (- x 1.5))(ps-trim (- yscreen y 1))))	;56 octal = 46

(defun ps-draw-box (left top right bottom &key (thickness 1)(alu *black*) (filled nil))
  (checkwidth thickness)
  (checkgray alu)
  (fp
	  "~&~a ~a m       % drawing a box~&~
            ~a ~a l ~a ~a l ~a ~a l  b ~(~a~) ~&"	; "s" or "fill" must be lowercase
	                                                ; ~& added by BKay 11Oct91
	  (ps-trim left) (ps-trim (- yscreen top)) (ps-trim left) (ps-trim (- yscreen bottom))
	  (ps-trim right) (ps-trim (- yscreen bottom)) (ps-trim right) (ps-trim (- yscreen top))
	  (if filled 'i 's)))


(defun ps-draw-rectangle (width height x y &key (alu *black*) )
  (checkgray alu)
  (fp
	  "~&~a ~a m  0 ~a rl  ~a 0 rl  0 ~a rl  b i "
	  (ps-trim x) (ps-trim (- yscreen y)) (ps-trim (- height))
	  (ps-trim width) (ps-trim height)))

(defun ps-draw-lines (points &key (alu *black*) (thickness 1)(dashed nil)(dash-pattern '(4 4)))
  (checkwidth thickness)
  (checkgray alu)
  (when dashed
    (fp " gs [~s ~s] 2 sd n " (first dash-pattern)(second dash-pattern)))
  (fp "n ")
  (fp "~a ~a m "
	  (ps-trim (first points)) (ps-trim (- yscreen (second points))))
  (do ((points (cddr points)(cddr points)))
      ((null points))
    (fp "~a ~a l "
	    (ps-trim (first points)) (ps-trim (- yscreen (second points)))))
  (let ((closure-p (and (= (first points)(nth (- (length points) 2) points))
			(= (second points)(car (last points))))))
    (fp "~:[~; b~] s ~:[~; gr~]~&"
	    closure-p dashed)))


;;; PS-DRAW-POLYGON draws a polygon, POINTS is a list of its vertices.
;;; By default, the interior of the polygon is filled - this is the
;;; extension over PS-DRAW-LINES. 
 
(defun ps-draw-polygon (points &key (filled t)(alu *black*) (thickness 1))	
  (checkwidth thickness)
  (checkgray alu)
  (fp " n ~a ~a m "
	  (ps-trim (first points)) (ps-trim (- yscreen (second points))))
  (do ((points (cddr points)(cddr points)))
       ((null points))
    (fp "~a ~a l "
	    (ps-trim (first points)) (ps-trim (- yscreen (second points)))))
  (fp "b ~:[s~;i~]~&"
			filled))


(defun ps-draw-ring (x y minor-r major-r  &key (alu *black*) )
  (checkgray alu)
  (checkwidth (- major-r minor-r))
  (fp		; The NEWPATH (n) has to be output to stop 
	  "n ~a ~a ~a 0 360 arc s~&"	; connecting line from current point.
	   (ps-trim x) (ps-trim (- yscreen y)) (ps-trim (/ (+ major-r minor-r) 2.0))))


;;; PS-NEW-BEHAVIOR should be called to start the page. It turns the output so that it
;;; prints in landscape mode on the printer (for :standard output).  It scales the output so
;;; that it doesn't overflow the right margin.  For :Text-Format-insert it computes a different
;;; scaling, based on *Text-Format-text-width*.

(defun ps-new-behavior (&key (box nil))
  (setf *current-linewidth* 1				; These are reset after every showpage.
	*current-grayval* 0)
  (case *postscript-style*
    (:Text-Format-insert
      (let* ((*x-scale* (/ (* 72.0 *Text-Format-text-width*)  xscreen) )
	     (*y-scale* *x-scale*))
	(fp "~,2f ~,2f scale~&" *x-scale* *y-scale*)
	(fp 
		"~&% Drawing a figure for Text-Format.   Width is ~f. Height should be ~,2f inches~&~
	      % xscreen = ~a,  yscreen = ~a~& n~&"
		*Text-Format-text-width* (/ (* yscreen *y-scale*) 72.0)
		xscreen yscreen)
	(when box
	  (ps-draw-box 0 yscreen xscreen 0))))
    (:ms-word
      (let* ((*x-scale* (/ (* 72.0 *Text-Format-text-width*)  xscreen) )
	     (*y-scale* *x-scale*))
	(fp
		      "~2&~,3f wp$x mul ~,3f inch div ~,2f wp$y mul ~,2f inch div scale~2&"
		      *x-scale* *Text-Format-text-width* *y-scale* (/ (* yscreen *y-scale*) 72.0))
	(fp 
		"~&% Drawing a figure for MS-WORD.   Width is ~f. Height should be ~,2f inches~&~
	      % xscreen = ~a,  yscreen = ~a   No rotation, no translation~&n "
		*Text-Format-text-width* (/ (* yscreen *y-scale*) 72.0)
		xscreen yscreen)))
    ((:standard :bounded)
     (fp 
	     "~&% 			                     New page~&")
     (fp "~2&n~&~,3f inch ~,3f inch t~
		~&~,3f q~&~,3f ~,3f sc~2&n~&"
	     *x-translation* *y-translation* *rotation* *x-scale* *y-scale*))
    (t (error "Unknown *postscript-style*: ~s" *postscript-style*))))

    
(defun ps-end-image ()
  (case  *postscript-style*
    ((:Text-Format-insert :ms-word) nil)			; No action need be taken for TEXT-FORMAT images.
    ((:standard :bounded)
      (fp "~&showpage~&"))
    (t (error "Unknown *postscript-style*: ~s" *postscript-style*))))


;;; helpful references: sample of Symbol typeface p 251 Red PS manual.
;;; The correspondence between the Symbol typeface and the (octal) ASCII collating
;;;   sequence is on p256 of the blue PS manual.
;;; ASCII character set p206 blue PS manual

;; Given a character code, print that character at that point on the
;; page, in the given pointsize.

(defun ps-special-char (x y ascii &key (size *symbol-ps-size*)(alu *black*) )
  (let* ((font :|Symbol|)
	 (*print-case* :upcase)	; otherwise you load "symbol" instead of "Symbol"
	 (pair (list font size))
	 (familiar-font (assoc pair *used-fonts* :test #'equal))
	 (gatom (if familiar-font
		    (second familiar-font)
		    (gentemp (format nil "~:@(~a-~)" font)))))
    (enforce-string font familiar-font size gatom pair))
  (checkgray alu)
  (fp "~&~a ~a m (\\~o) p ~&" (ps-trim x) (ps-trim (- yscreen y)) ascii))


