;;; -*- Mode:Common-Lisp; Package:POS; Syntax:COMMON-LISP; Default-character-style:(FIX BOLD NORMAL); Base:10 -*-


(in-package :pos)

;;; This file contains three parts :
;;; - Font control code for lispview.
;;; - Window management objects and methods.
;;; - Printing to a file via the "print" button.
;;; - Graphics output functions.



;;;-------------------------------------------------------------------
;;;
;;; Font Control
;;;
;;;-------------------------------------------------------------------

;;; Return the size of the font in lispview terms.
;;;
(defun get-x-font-size (pos-size)
       (case pos-size
		(:tiny		8)
                (:very-small	8)
                (:smaller	10)
                (:small		10)
                (:normal	12)
                (:large		14)
                (:larger	18)
                (:very-large	24)
                (t		12) ))


;;; This is a list of the fonts already created.
;;;
(defparameter *realized-fonts* NIL)


;;; We use symbolics-style font names.  This converts them to lispview font
;;; objects, checking to see if the object already exists.
;;;
(defun get-x-font (pos-font) ; (family face size)
  (let ((font-entry (assoc pos-font *realized-fonts* :test #'equal)))
    (cond
      (font-entry (second font-entry))
      (T
       (let ((font (make-instance 'lv:font
				  :family (case (first pos-font)
					    (:swiss  'swiss)
					    (:fix    'courier)
					    (t       'times))
				  :weight (case (second pos-font)
					    (:italic      :medium)
					    (:bold        :bold)
					    (:bold-italic :bold)
					    (t            :medium))
				  :slant  (case (second pos-font)
					    (:italic      :italic)
					    (:bold        :roman)
					    (:bold-italic :italic)
					    (t            :roman))
				  :point-size (get-x-font-size
					       (third pos-font)))))
	 (push (list pos-font font) *realized-fonts*)
	 font)))))
					 


;;;--------------------------------------------------------------------------
;;;
;;; Lispview window management
;;;
;;;--------------------------------------------------------------------------

;;; Here is the basic idea :
;;;
;;; Each qplot-display is a Lispview object containing a plotting window and a 
;;; list of drawing requests.  The POS functions are designed to simply push a
;;; set of drawing requests onto the commands list for the qplot-display.
;;; Thereafter, whenever a damage event is received, the list is "evaluated"
;;; and the display is redrawn.  Since the commands are already hanging off
;;; of the window, it is also easy to have a "print" button that writes the
;;; plot to a file.
;;;
;;; One side-effect of this method is that the pos interface routines for
;;; lispview just copy themselves directly onto the commands slot of the
;;; display without any modification.  This is so that the commands will be
;;; "clean" enough that both the postscripter and lispview can interpret
;;; them.  The only exception to this rule is that the command qplot-end-display
;;; is not copied.  This command is a no-op for lispview and is automatically
;;; added to the end of a "Print."  This permits a plot to be added to after
;;; it has already been drawn.
;;;
;;; From lispview's perspective, a pos command is controlled by the switches
;;; image-to-postscript-p, image-to-screen-p, and use-lv-call as follows :
;;;
;;; postscript?   screen?   lv-call?      operation (printer & display)
;;; -----------   -------   --------      ---------
;;;      T          T          T          disallowed
;;;      T          T          F          print & push onto window
;;;      T          F          T          disallowed
;;;      T          F          F          print & ignore
;;;      F          T          T          ignore & display
;;;      F          T          F          ignore & push onto window
;;;      F          F          T          ignore & ignore
;;;      F          F          F          ignore & ignore
;;;      
;;; Note that this scheme covers any image disposal option
;;; (screen, file, both, or none).  The lispview display code makes sure to
;;; shut off postscript? and turn on lv-call? before plotting which permits
;;; displaying only.
;;;
;;; To implement this, each pos command is implemented in the following
;;; form :
;;; (defun cmd <args>
;;;   (when (image-to-postscript-p)
;;;     <print to the file> )
;;;   (when (image-to-screen-p)
;;;     #+:lispview (if (use-lv-call *qplot-output*)
;;;                     <draw to screen>
;;;                     <push request onto commands slot of window>)
;;;     #+others    <normal display command>
;;;    ))
;;;
;;; For those pos commands that themselves call pos commands, this form
;;; looks like :
;;; (defun cmd <args>
;;;   (when (image-to-postscript-p)
;;;     (with-plotting-to-screen-inhibited
;;;        <print to the file> ))
;;;   (when (image-to-screen-p)
;;;     #+:lispview (if (use-lv-call *qplot-output*)
;;;                     <draw to screen>
;;;                     <push request onto commands slot of window>)
;;;     #+others    (with-plotting-to-postscript-inhibited
;;;                     <normal display command> )
;;;    ))
;;;
;;; Graphics contexts :
;;; 
;;; Lispview runs more efficiently when drawing functions can take their graphics
;;; context from an existing graphics-context object rather than setting the
;;; context in each call.  To facilitate this capability, the pos-gcon and gcon
;;; slots of a qplot-display are used.  The gcon slot always contains the current
;;; graphics context for the display.  The posgcon slot is an easier-to-access
;;; version of the graphics context which contains the posgcon structure defined
;;; in qplot-primitives.lisp.  All lispview POS primitives (they generally
;;; start with "lv-") check their arguments against this structure in order to
;;; prevent needless resetting of the graphics context (which is very expensive).
;;; The graphics context may be changed via an argument to qplot-new-behavior or
;;; it can be temporarily rebound using with-qplot-gcon.
;;; This macro uses the lispview with-graphics-context macro to temporarily reset
;;; the gcon an pos-gcon contexts.   This macro is a no-op when compiled
;;; without lispview.
;;;
;;; Window hierarchy :
;;;
;;; The POS functions use the stream of *qplot-output* as the destination for
;;; drawing commands.  Under lispview, *qplot-output* is bound to a qplot-display
;;; object.  This object contains a slots for the command queue, the graphics context,
;;; a variety of flags, and the lispview windows used.  The hierarchy of windows is
;;;        base-window   - This window holds the sub windows.
;;;            panel     - This is the window with the buttons in it.
;;;            window    - This is the window that POS draws in (it's of type qplot-windw).
;;; Whenever a damage-event of type repaint-qplot is received by "window",
;;; it calls the receive-event method for qplot-windows which redraws the
;;; display (it calls redraw-qplot which does the actual work).
;;;
;;; Status flag:
;;;
;;; The status slot of the qplot-display is used to control synchronization between
;;; display redraw and other commands that use the with-plotting-to-lispview-forced
;;; macro for immediate drawing.  The status flag is set to :busy whenever the display
;;; is being redrawn and it is set to :idle when redrawing is complete.
;;; With-plotting-to-lispview-forced waits until this flag is set to :idle before
;;; proceeding.  In theory, the use-lv-call flag could be used for this purpose,
;;; except that there is a period of time between window creation and the first
;;; damage event where use-lv-call is not set (and should not be).  At the end of
;;; qplot-end-display, therefore status is set to :busy.  This is the only place that
;;; status is set outside of redraw-qplot.



;;; Define a class for damage events.
;;;
(defclass repaint-qplot (lv:damage-interest) ())


;;; A structure that holds the current printer settings
;;;
(defstruct printerSettings
  (filename (namestring (merge-pathnames (user::pwd) *ps-file*)))
  (style    :Standard)
  (text-width 6.0)
  (screen-width 1037)
  (screen-height 700)
  (append        NIL) ; Determines what to do if the file exists
  )

(defparameter *printersettings-defaults* (make-printersettings))


;;; Define an object that holds qplots.
;;;
(defclass qplot-display ()
  ((window   :accessor window      ; A qplot-window object
	     :initform nil)
   (base-window :accessor base-window ; The base window holding control panel and window.
                :initform nil)        ; This is never referenced -- but maybe it will be needed
                                      ; someday.
   (panel    :accessor panel          ; The control panel with print button.
	     :initform nil)
   (commands :accessor commands       ; A list of drawing commands
	     :initform (make-q))
   (status   :accessor status         ; :busy => redrawing the display.  Otherwise, :idle.
	     :initform :idle)
   (use-lv-call :accessor use-lv-call ; T if we should graph, NIL to store.
		:initform nil)
   (gcon     :accessor gcon           ; Default graphics context
	     :initform (make-instance
			'lv:graphics-context
			:foreground (lv:find-color :name "black")
			:font       (get-x-font *plain-font*)))
   (pos-gcon :accessor pos-gcon  ; An easy-to-access version of gcon
	     :initform (make-posgcon :color        "black"
				     :font         *plain-font*
				     :dashed       nil
				     :dash-pattern '(10 10)
				     :thickness    1))
   (printersettings :accessor printersettings  ; A printerSettings structure
		    :initform (make-printerSettings))
   (size     :accessor size        ; The requested size of the window
	     :initform nil         ; this is either :fit or (X Y)
	     :initarg  :size)
   (drawsize :accessor drawSize    ; The actual space taken up by the drawing
	     :initform nil)
   ))


;;; This is a specialization of a window.
;;;
(defclass qplot-window (lv:window)
  ((qplot-display :initarg :qplot-display  ; The parent qplot-display object
		  :accessor qplot-display)
   )
  (:default-initargs
      :interests (list (make-instance 'repaint-qplot))))
   

;;; Send the qplot commands to the display.
;;;
(defmethod lv:receive-event ((w qplot-window) (i repaint-qplot) event)
  (declare (ignore event))
  ;; This is a hack that shouldn't be necessary as far as the lispview
  ;; manual suggests.  The problem is that the subwindows of a base
  ;; window do not receive resize infofor some reason.  This code
  ;; ensures that they do.
  (let* ((qd (qplot-display w))
	 (br-panel (lv:bounding-region (panel qd)))
	 (panel-height (lv:region-height br-panel))
	 (br-base (lv:bounding-region (base-window qd)))
	 (br-wind (lv:bounding-region (window qd))))
    (when (>= (lv:region-width br-base)
	      (lv:region-width br-wind))
      (setf (lv:region-width br-wind) (lv:region-width br-base)))
    (when (>= (lv:region-height br-base)
	      (+  panel-height (lv:region-height br-wind)))
      (setf (lv:region-height br-wind) (- (lv:region-height br-base) 25)))
    (when (or (>= (lv:region-width br-base)
		  (lv:region-width br-wind))
	      (>= (lv:region-height br-base)
	      (+ panel-height (lv:region-height br-wind))))
      (setf (lv:bounding-region (window qd)) br-wind)))
		    
  (redraw-qplot (qplot-display w)))


;;; This handles the actual redrawing of the display.
;;; The commands slot contains lists of the form (pos-command-name args).
;;;
(defmethod redraw-qplot ((qd qplot-display))
  ;; Force subsequent pos commands to plot to the window,
  ;; but not to the file (if we are in option :both)
  (with-plotting-to-postscript-inhibited
      (let ((old-output *qplot-output*))
	(unwind-protect
	     (progn
	       (setf (status qd) :busy)
	       (setf (use-lv-call qd) T)
	       (let ((*qplot-output* qd))
		 (lv:with-output-buffering (lv:display (window *qplot-output*))
		   (dolist (c (qlist (commands qd)))
		     (apply (car c) (cdr c))))))
	  (progn
	    (setf (use-lv-call qd) NIL)
	    (setf (status qd) :idle)
	    (setf *qplot-output* old-output))))))
  

;;; This determines the correct size that a window should be.
;;; It does this by looking at the size slot of the display.
;;; If it contains the keyword :fit then determine-plotting-size
;;; will figure out how big the area is.  Otherwise size should
;;; contain the required x, y size.
;;;
(defmethod lv-window-size ((qd qplot-display))
  (cond
    ((consp (size qd))
     (values-list (size qd)))
    ((eq (size qd) :fit)
     (determine-plotting-size qd))
    (T
     (error "Bad Window size ~a for ~a" (size qd) qd))))

(defparameter *determine-plotting-size-extensions* NIL)


;;; This determines the size of a window based on the commands
;;; stored on the window.  It stores it in the drawSize slot.
;;; Whenever new pos commands are defined, a size calculation should be
;;; added.
;;;
(defmethod determine-plotting-size ((qd qplot-display))
  (cond
    ((drawSize qd))
    (T
     ;; This is a translation of Kee Kimbrall's code for GetWindowSize
     ;; in xbatch.c
     (let ((width 300)
	   (height 200))
       (dolist (c (qlist (commands qd)))
;	 (format t "~%Doing ~a" c)
	 (case (car c)
	   ((qplot-dot qplot-symbol qplot-special-char)
	    (setf width (max width (second c)))
	    (setf height (max height (third c))))
	   (qplot-hline
	    (setf width (max width (+ (second c) (fourth c))))
	    (setf height (max height (third c))))
	   ((qplot-vline)
	    (setf width (max width (second c)))
	    (setf height (max height (+ (third c) (fourth c)))))
	   (qplot-line
	    (setf width (max width (second c) (fourth c)))
	    (setf height (max height (third c) (fifth c))))
	   ((qplot-lines qplot-polygon)
	    (setf width
		  (apply #'max
			 (cons width
			       (loop for pt in (second c)
				     for i from 1 upto (length (second c))
				     if (oddp i) collect pt))))
	    (setf height
		  (apply #'max
			 (cons height
			       (loop for pt in (second c)
				     for i from 1 upto (length (second c))
				     if (evenp i) collect pt)))))
	   (qplot-string
	    (setf width (max width (third c)))
	    (setf height (max height (fourth c))))
	   (qplot-circle
	    (setf width (max width (+ (second c) (fourth c))))
	    (setf height (max height (+ (third c) (fourth c)))))
	   (qplot-ring
	    (setf width (max width (+ (second c) (fifth c))))
	    (setf height (max height (+ (third c) (fifth c)))))
	   (qplot-ellipse
	    (setf width (max width (+ (second c) (fourth c))))
	    (setf height (max height (+ (third c) (fifth c)))))
	   (qplot-box
	    (setf width (max width (+ (second c) (fourth c))))
	    (setf height (max height (+ (third c) (fifth c)))))
	   (qplot-vector
	    (setf width (max width (second c) (fourth c)))
	    (setf height (max height (third c) (fifth c))))
	   ((qplot-new-behavior qplot-end-display)
	    nil)
	   (t
	    ;; If it's not in this list then check the extensions
	    ;; slot.  An entry looks like (cmdname widthfctn heightfctn).
	    ;; Each function takes one argument, the command.
	    (let ((entry (assoc (car c)
				*determine-plotting-size-extensions*)))
	      (cond
		(entry
		 (setf width (max width (funcall (second entry) c)))
		 (setf height (max height (funcall (third entry) c))))
		(t
		 (error "POS command ~a unrecognized" c)))))
	   )
;	 (format t "~%   width = ~a   height = ~a" width height)
	 )
       (incf width 20)
       (incf height 20)
       (setf (drawSize qd) (list width height))
       (values width height)))))


;;; Clear the window.
;;;
(defmethod clear-window ((w qplot-window))
  (lv:clear w))


;;; Generate an icon for the window.
;;; This should probably be an x-bitmap rather than a sun icon.
;;;
(defun lv-make-icon (string)
  (let ((icon-pathname
	 (lp:translate-logical-pathname "XPOS:XPOS;xpos.icon")))
    (make-instance
     'lv:icon
     :label
     (if (probe-file icon-pathname)
	 (list string (make-instance
		       'lv:image
		       :filename icon-pathname
		       :format :sun-icon))
	 string))))





;;;-------------------------------------------------------------------
;;;
;;; Printing to a file.
;;;
;;;-------------------------------------------------------------------


;;; Postscript output the window.
;;; This just brings up a menu to do the job.
;;; Notes:  For some reason, the blanco X server uses a very
;;;         pale gray for the widgets, although they look
;;;         just dandy on the sparc2.  So, I've upped the
;;;         color to darkslategray (which appears to be what
;;;         you get on the sparc2s anyway.
;;;
(defmethod lv-postscript ((qd qplot-display))
  (let (popup panel filename append style text-width
	      screen-width screen-height
	      apply defaults message)
    (setf popup
	  (make-instance 'lv:popup-window
			 :width 500
			 :height 220
			 :label "Print options")
	  panel
	  (make-instance 'lv:panel
			 :parent popup
			 :left 0 :top 0)
	  filename
	  (make-instance 'lv:text-field
			 :left 0
			 :top 10
			 :parent panel
			 :foreground (lv:find-color :name :darkslategray)
			 :label "PostScript filename:"
			 :value (printersettings-filename
				 (printersettings qd)))
	  append
	  (make-instance 'lv:exclusive-setting
			 :parent panel
			 :label "Append if file exists:"
			 :choices '("Yes" "No")
			 :foreground (lv:find-color :name :darkslategray)
			 :left 0
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region filename)))
			 :value (if (printersettings-append
				     (printersettings qd))
				    "Yes"
				    "No"))
	  style
	  (make-instance 'lv:exclusive-setting
			 :parent panel
			 :label "PostScript style:"
			 ;; The handling of these choices by xpos is really screwy.
			 ;; I've commented out all but the most useful.
			 :choices '("Standard" "Bounded" ; "Text-format-insert" "MS"
				    )
			 :foreground (lv:find-color :name :darkslategray)
			 :left 0
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region append)))
			 :value (lv-stringify
				 (printersettings-style
				  (printersettings qd))))
	  text-width
	  (make-instance 'lv:text-field
			 :parent panel
			 :label "Formatted text width (in):"
			 :foreground (lv:find-color :name :darkslategray)
			 :displayed-value-length 5
			 :left 0
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region style)))
			 :value (lv-stringify
				 (printersettings-text-width
				  (printersettings qd))))
	  screen-width
	  (make-instance 'lv:numeric-field
			 :parent panel
			 :label "Screen width: "
			 :foreground (lv:find-color :name :darkslategray)
			 :displayed-value-length 4
			 :left 0
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region text-width)))
			 :min-value 0
			 :max-value 1037
			 :value (printersettings-screen-width
				 (printersettings qd)))
	  screen-height
	  (make-instance 'lv:numeric-field
			 :parent panel
			 :label "Screen height:"
			 :foreground (lv:find-color :name :darkslategray)
			 :displayed-value-length 4
			 :left 0
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region screen-width)))
			 :min-value 0
			 :max-value 700
			 :value (printersettings-screen-height
				 (printersettings qd)))
	  apply
	  (make-instance 'lv:command-button
			 :parent panel
			 :label "Do it"
			 :foreground (lv:find-color :name :darkslategray)
			 :right (- 10
				   (truncate (lv:region-width
					      (lv:bounding-region panel))
					     2))
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region screen-height)))
			 )
	  defaults
	  (make-instance 'lv:command-button
			 :parent panel
			 :label "Defaults"
			 :foreground (lv:find-color :name :darkslategray)
			 :left (+ 10
				  (truncate (lv:region-width
					     (lv:bounding-region panel))
					    2))
			 :top (+ 10 (lv:region-bottom
				     (lv:bounding-region screen-height)))
			 )
	  message
	  (make-instance 'lv:message
		    :label ""
		    :foreground (lv:find-color :name :darkslategray)
		    :parent panel
		    :top (+ (lv:region-bottom
			     (lv:bounding-region defaults))
			    10)
		    :left 0)
	  )
    (setf (lv:command defaults) #'(lambda ()
				    (setf (lv:label message) "")
				    (lv-set-printersettings-defaults
				     filename append style text-width 
				     screen-width screen-height)))
    (setf (lv:command apply)    #'(lambda ()
				    (setf (lv:label message) "")
				    (lv-do-it-button
				     qd popup panel 
				     filename append style text-width
				     screen-width screen-height message)))
    ))


;;; Convert numbers and keywords to strings.
;;;
(defun lv-stringify (object)
  (etypecase object
    (float   (format nil "~,2F" object))
    (symbol  (subseq (format nil "~@(~a~)" object) 1))))


;;; Reset the panel settings to their defaults.
;;;
(defun lv-set-printersettings-defaults (filename append style text-width
					screen-width screen-height)
  (setf (lv:value filename)
	(printersettings-filename *printersettings-defaults*)
	(lv:value append)
	(if (printersettings-append *printersettings-defaults*)
	    "Yes" "No")
	(lv:value style)
	(lv-stringify (printersettings-style *printersettings-defaults*))
	(lv:value text-width)
	(lv-stringify (printersettings-text-width *printersettings-defaults*))
	(lv:value screen-width)
	(printersettings-screen-width *printersettings-defaults*)
	(lv:value screen-height)
	(printersettings-screen-height *printersettings-defaults*)))


;;; Store the settings in the qplot-display.
;;; It is assumed that the settings have already been checked.
;;;
(defun lv-store-printersettings (w filename append style text-width
				   screen-width screen-height)
  (let ((ps (printersettings w)))
    (setf (printersettings-filename ps)
	  (lv:value filename)
	  
	  (printersettings-append ps)
	  (if (string-equal (lv:value append) "Yes") T NIL)
	  
	  (printersettings-style ps)
	  (read-from-string (concatenate 'string ":" (lv:value style)))
	  
	  (printersettings-text-width ps)
	  (read-from-string (lv:value text-width))
	  
	  (printersettings-screen-width ps)
	  (lv:value screen-width)
	  
	  (printersettings-screen-height ps)
	  (lv:value screen-height))))


;;; Check if print parameters are OK and then print if they are.
;;; This function is lucid-specific due to the use of condition
;;; handlers.
;;;
(defun lv-do-it-button (qd popup panel filename append style text-width
			screen-width screen-height message)
  (user::handler-case
   (cond
     (T ;(lv-filename-check (lv:value filename) panel)
      (lv-store-printersettings qd filename append style text-width
				screen-width screen-height)
      (lv-postscript-to-file qd)
      (lv:withdraw popup
		   (lv:root-canvas lv:*default-display*))))
  (file-error (c)
	      (setf (lv:label message) (format nil "~a" c))
	      )))


;;; Actually print to a file.
;;;
(defmethod lv-postscript-to-file ((qd qplot-display))
  (let* ((ps (printersettings qd))
	 ;; Here's what's going on :
	 ;; The device-interface call might be reentrant since
	 ;; we may have made this plot as part of a series.
	 ;; Therefore, we setup new special vars for everything that
	 ;; might matter.
	 ;;
	 ;; Here is how styles are defaulted :
	 ;; All modes      :      xscreen, yscreen, *text-format-text-width*,
	 ;;                       are set as per menu.
	 ;; :standard, :ms :      *rotation* = 90,
	 ;;                       *x-translation* x *y-translation* = 8.5 x 0.5
	 ;; :text-format-insert : *rotation* = 0, X x Y = 1 x 1.
	 ;; :bounded            : *rotation* = 0, X x Y = 1 x 1.
	 ;;                       *bounding-box* (llx lly urx ury)
	 ;;
	 ;; Scaling is set elsewhere.  What all this means is that
	 ;; bounded mode should give you something usable by
	 ;; psfig.
	 ;;
	 ;; This is, of course, all very stupid.  Someone should really rewrite
	 ;; the xpos handling of these switches to be more sane.
	 ;;
	 (*postscript-output-file* NIL)
	 (*text-format-text-width* (printersettings-text-width ps))
	 (*postscript-style*       (printersettings-style ps))
	 (xscreen                  (printersettings-screen-width ps))
	 (yscreen                  (printersettings-screen-height ps))
	 (*postscript-bound*       NIL)
	 (*rotation*               (if (member *postscript-style*
					       '(:bounded :text-format-insert))
				       0 90))
	 (*x-translation*          (if (member *postscript-style*
					       '(:bounded :text-format-insert))
				       1 8.5))
	 (*y-translation*          (if (member *postscript-style*
					       '(:bounded :text-format-insert))
				       1 0.5))
	 (*bounding-box*           (if (eq *postscript-style* :bounded)
				       (list lmargin (- yscreen bmargin)
					     (- xscreen rmargin) 0)
				       nil))
	 (*picture*                (printersettings-filename ps))
	 (*print-case*             :upcase)
	 (*image-disposal*         :ps)
	 )
    (with-plotting-to-postscript-forced 
      (with-open-file (*postscript-output-file* *picture*
		       :direction :output
		       :if-exists (if (printersettings-append ps)
				      :append
				      :error))                    
	(start-postscript-file)
	(dolist (c (qlist (commands qd)))
	  (apply (car c) (cdr c)))
	;; The command list does not include this command, so call it by hand.
	(qplot-end-display)
	(end-postscript-file)
	))))


    


;;;--------------------------------------------------------------------------
;;;
;;; Lispview drawing commands
;;;
;;;--------------------------------------------------------------------------


;;; QPLOT-NEW-BEHAVIOR:
;;; A call to this function will clear the window if it isn't brand new.
;;;
(defmethod lv-new-behavior ((w qplot-window) box)
  (declare (ignore box))
  (clear-window w))


;;; QPLOT-END-DISPLAY
;;;
(defmethod lv-end-display ((w qplot-window))
  NIL)


;;; QPLOT-DOT:
;;;
(defmethod lv-qplot-dot ((w qplot-window) x y alu color)
  (let ((rx (round x)) (ry (round y)))
    (with-slots ((qd qplot-display)) w
       (if (equal (get-color alu color) (posgcon-color (pos-gcon qd)))
	   (lv:draw-line w rx ry rx ry :gc (gcon qd))
	   (lv:draw-line w rx ry rx ry :gc (gcon qd)
			      :foreground (lv:find-color :name
							 (get-color alu color)))))))



;;; QPLOT-HLINE:
;;;
(defmethod lv-qplot-hline ((w qplot-window) x y length
			   alu color thickness dashed dash-pattern)
  (lv-qplot-line w x y (+ x length) y
		 alu color thickness dashed dash-pattern))


;;; QPLOT-VLINE:
;;;
(defmethod lv-qplot-vline ((w qplot-window) x y length
			   alu color thickness dashed dash-pattern)
  (lv-qplot-line w x y x (+ y length)
		 alu color thickness dashed dash-pattern))


;;; QPLOT-LINE:
;;;
(defmethod lv-qplot-line ((w qplot-window) x1 y1 x2 y2
			  alu color thickness dashed dash-pattern)
  (let ((rx1 (round x1)) (ry1 (round y1))
	(rx2 (round x2)) (ry2 (round y2)))
    (with-slots ((qd qplot-display)) w
       (cond
	 ((and (eq (get-color alu color) (posgcon-color (pos-gcon qd)))
	       (eq thickness    (posgcon-thickness (pos-gcon qd)))
	       (eq dashed       (posgcon-dashed (pos-gcon qd)))
	       (eq dash-pattern (posgcon-dash-pattern (pos-gcon qd))))
	  (lv:draw-line w rx1 ry1 rx2 ry2 :gc (gcon qd)))
	 ((eq (get-color alu color) (posgcon-color (pos-gcon qd)))
	   (lv:draw-line w rx1 ry1 rx2 ry2 :gc (gcon qd)
			 :line-width thickness
			 :line-style (if dashed :dash) ; assume :solid is default
			 :dashes dash-pattern))
	 (T
	  (lv:draw-line w rx1 ry1 rx2 ry2 :gc (gcon qd)
			:line-width thickness
			:line-style (if dashed :dash) ; assume :solid is default
			:dashes dash-pattern
			:foreground (lv:find-color :name
						   (get-color alu color))))))))


;;; QPLOT-LINES:
;;;
(defmethod lv-qplot-lines ((w qplot-window) points
			   alu color thickness dashed dash-pattern)
  (labels ((d-lines (f-x f-y pts) 
	     (with-slots ((qd qplot-display)) w
	       (cond ((null pts) nil)
		     (t
		      (let ((rf-x (round f-x)) (rf-y (round f-y))
			    (rs-x (round (car pts))) (rs-y (round (cadr pts))))
			(cond ((and (equal (get-color alu color) (posgcon-color (pos-gcon qd)))
				    (eq    thickness (posgcon-thickness (pos-gcon qd)))
				    (eq    dashed    (posgcon-dashed (pos-gcon qd)))
				    (eq    dash-pattern    (posgcon-dash-pattern (pos-gcon qd))))
			       (lv:draw-line w rf-x rf-y rs-x rs-y
					     :gc (gcon qd)))
			      ((and (equal (get-color alu color) (posgcon-color (pos-gcon qd)))
				    (eq    dashed    (posgcon-dashed (pos-gcon qd)))
				    (eq    dash-pattern    (posgcon-dash-pattern (pos-gcon qd))))
			       (lv:draw-line w rf-x rf-y rs-x rs-y
					     :gc (gcon qd) :line-width thickness))
			      (T
			       (lv:draw-line w rf-x rf-y rs-x rs-y
					     :gc (gcon qd)
					     :line-width thickness
					     :line-style (if dashed :dash)
					     :dashes dash-pattern
					     :foreground (lv:find-color :name
									(get-color alu color)))))
			(d-lines (car pts) (cadr pts) (cddr pts))))))))
    (cond ((oddp (length points))  (error "List must be even"))
	  (t     (d-lines (car points) (cadr points) (cddr points))))))


;;; QPLOT-STRING:
;;;
(defmethod lv-qplot-string ((w qplot-window) string x y
			    alu color font)
  (let ((rx (round x)) (ry (round y)))
    (with-slots ((qd qplot-display)) w
       (cond ((and (eq (get-color alu color) (posgcon-color (pos-gcon qd)))
		   (equal font (posgcon-font (pos-gcon qd))))
	      (lv:draw-string w rx ry string :gc (gcon qd)))
	     ((eq (get-color alu color) (posgcon-color (pos-gcon qd)))
	      (lv:draw-string w rx ry string :gc (gcon qd) :font (get-x-font font)))
	     (T
	      (lv:draw-string w rx ry string :gc (gcon qd)
			      :font (get-x-font font)
			      :foreground (lv:find-color :name
							 (get-color alu color))))))))


;;; QPLOT-CIRCLE:
;;;
(defmethod lv-qplot-circle ((w qplot-window) x y radius
			    alu color filled start-angle end-angle
			    thickness)
  (let ((rx (round (- x radius))) (ry (round (- y radius)))
	(rr (* 2 radius))
	(sa (* start-angle (/ 360 (* 2 pi))))
	(ea (* end-angle (/ 360 (* 2 pi)))))
    (with-slots ((qd qplot-display)) w
       (if (and (equal (get-color alu color) (posgcon-color (pos-gcon qd)))
		(eq thickness (posgcon-thickness (pos-gcon qd))))
	   (lv:draw-arc w rx ry rr rr sa ea :fill-p filled :gc (gcon qd))
	   (lv:draw-arc w rx ry rr rr sa ea :fill-p filled :gc (gcon qd)
			:line-width thickness
			:foreground (lv:find-color :name
						   (get-color alu color)))))))



;;; QPLOT-RING:
;;;
(defmethod lv-qplot-ring ((w qplot-window) x y minor-r major-r
			  alu color)
  (let* ((thickness (round (- major-r minor-r)))
	 (r         (round (+ minor-r (/ thickness 2))))
	 (rx (round (- x r))) (ry (round (- y r)))
	 (rr (* 2 r)))
    (with-slots ((qd qplot-display)) w
       (cond
	 ((equal (get-color alu color) (posgcon-color (pos-gcon qd)))
	  (lv:draw-arc w rx ry rr rr 0 360 :line-width thickness
		       :gc (gcon qd)))
	 (T
	  (lv:draw-arc w rx ry rr rr 0 360
		       :line-width thickness :gc (gcon qd)
		       :foreground (lv:find-color :name
						  (get-color alu color))))))))


;;; QPLOT-ELLIPSE:
;;;
(defmethod lv-qplot-ellipse ((w qplot-window) center-x center-y x-radius y-radius
			     alu color filled start-angle end-angle thickness
			     rotation)
  (declare (ignore rotation))
  (let ((rx (round (- center-x x-radius))) (ry (round (- center-y y-radius)))
	(rrx (* 2 x-radius)) (rry (* 2 y-radius))
	(sa (* start-angle (/ 360 (* 2 pi))))
	(ea (* end-angle (/ 360 (* 2 pi)))))
    (with-slots ((qd qplot-display)) w
       (cond
	 ((and (equal (get-color alu color) (posgcon-color (pos-gcon qd)))
	       (eq    thickness (posgcon-thickness (pos-gcon qd))))
	   (lv:draw-arc w rx ry rrx rry sa ea :fill-p filled :gc (gcon qd)))
	 ((equal (get-color alu color) (posgcon-color (pos-gcon qd)))
	  (lv:draw-arc w rx ry rrx rry sa ea :fill-p filled :gc (gcon qd)
		       :line-width thickness))
	 (T
	  (lv:draw-arc w rx ry rrx rry sa ea :fill-p filled :gc (gcon qd)
		       :line-width thickness
		       :foreground (lv:find-color :name
						  (get-color alu color))))))))



;;; QPLOT-POLYGON:
;;; LispView doesn't have a filled polygon as yet, so this is just an
;;; outline of a polygon for now.
;;;
(defmethod lv-qplot-polygon ((w qplot-window) points
			     alu color filled thickness)
  (declare (ignore filled))
  (lv-qplot-lines w (append points (list (car points) (cadr points)))
		  alu color thickness nil '(10 10)))



;;; QPLOT-BOX:
;;;
(defmethod lv-qplot-box ((w qplot-window) xpos ypos xwide yheight
			 alu color thickness filled)
  (let ((rx (round xpos)) (ry (round ypos))
	(rw (round xwide)) (rh (round yheight)))
    (with-slots ((qd qplot-display)) w
       (cond
	 ((and (equal (get-color alu color) (posgcon-color (pos-gcon qd)))
	       (eq    thickness (posgcon-thickness (pos-gcon qd))))
	  (lv:draw-rectangle w rx ry rw rh :fill-p filled :gc (gcon qd)))
	 ((equal (get-color alu color) (posgcon-color (pos-gcon qd)))
	  (lv:draw-rectangle w rx ry rw rh :fill-p filled :gc (gcon qd)
			     :line-width thickness))
	 (T
	  (lv:draw-rectangle w rx ry rw rh :fill-p filled :gc (gcon qd)
			     :line-width thickness 
			     :foreground (lv:find-color :name
							(get-color alu color))))))))


;;; A set of bitmaps for arrows.  These look much better than using a set
;;; of draw functions to make the arrows.
;;; A function is required to set these up because they need to create an
;;; XView display, and if this is done in something that is a load image,
;;; the load image will work incorrectly.
;;; 
(defparameter *lv-up-arrow* NIL)
(defparameter *lv-down-arrow* NIL)
(defparameter *lv-left-arrow* NIL)
(defparameter *lv-right-arrow* NIL)

;;; MAKE THIS CALL IN THE REALIZE SECTION THAT SETS UP THE INITIAL WINDOW?

(defun setup-special-chars ()
  (declare (special *lv-up-arrow*
		    *lv-down-arrow*
		    *lv-left-arrow*
		    *lv-right-arrow*))
  (setf *lv-up-arrow*
	(make-instance
	 'lv:image
	 :data (make-array '(9 9)
			   :element-type 'bit
			   :initial-contents '(#*000010000
					       #*000111000
					       #*001111100
					       #*011111110
					       #*000010000
					       #*000010000
					       #*000010000
					       #*000010000
					       #*000010000))))
  (setf *lv-down-arrow*
	(make-instance
	 'lv:image
	 :data (make-array '(9 9)
			   :element-type 'bit
			   :initial-contents '(#*000010000
					       #*000010000
					       #*000010000
					       #*000010000
					       #*000010000
					       #*011111110
					       #*001111100
					       #*000111000
					       #*000010000))))

  (setf *lv-left-arrow*
	(make-instance
	 'lv:image
	 :data (make-array '(9 9)
			   :element-type 'bit
			   :initial-contents '(#*000000000
					       #*000100000
					       #*001100000
					       #*011100000
					       #*111111111
					       #*011100000
					       #*001100000
					       #*000100000
					       #*000000000))))

  (setf *lv-right-arrow*
	(make-instance
	 'lv:image
	 :data (make-array '(9 9)
			   :element-type 'bit
			   :initial-contents '(#*000000000
					       #*000001000
					       #*000001100
					       #*000001110
					       #*111111111
					       #*000001110
					       #*000001100
					       #*000001000
					       #*000000000))))
  )

;;; QPLOT-SYMBOL:
;;;
(defmethod lv-qplot-symbol ((w qplot-window) x y symbol
			    alu color)
  (let ((rx (round x)) (ry (round y))
	(xoff *symbol-x-offset*) (yoff 7) ; *symbol-y-offset* is OK for postscript but
                                          ; wrong for lispview.
	(color-match (equal (get-color alu color) (posgcon-color (pos-gcon (qplot-display w))))))
    (when (null *lv-up-arrow*)
      (setup-special-chars))
    (case symbol
      ((std :std)     (lv-qplot-circle w rx ry 2 alu color nil 0 (* 2 pi) 1))
      ((inc :inc)     (if color-match
			  (lv:copy-area *lv-up-arrow* w 0 0 9 9 (- rx 4) (- ry 4)
					:operation boole-ior)
			  (lv:copy-area *lv-up-arrow* w 0 0 9 9 (- rx 4) (- ry 4)
					:operation boole-ior
					:foreground (lv:find-color :name
							(get-color alu color)))))
      ((dec :dec)     (if color-match
			  (lv:copy-area *lv-down-arrow* w 0 0 9 9 (- rx 4) (- ry 4)
					:operation boole-ior)
			  (lv:copy-area *lv-down-arrow* w 0 0 9 9 (- rx 4) (- ry 4)
					:operation boole-ior
					:foreground (lv:find-color :name
							(get-color alu color)))))
      ((ign :ign)     (lv-qplot-string w "*" (+ rx xoff) (+ ry yoff)
				       alu color *plain-font*))
      ((left :left)   (if color-match
			  (lv:copy-area *lv-left-arrow* w 0 0 9 9 (- rx 4) (- ry 4))
			  (lv:copy-area *lv-left-arrow* w 0 0 9 9 (- rx 4) (- ry 4)
					:foreground (lv:find-color :name
							(get-color alu color)))))
      ((right :right) (if color-match
			  (lv:copy-area *lv-right-arrow* w 0 0 9 9 (- rx 4) (- ry 4))
			  (lv:copy-area *lv-right-arrow* w 0 0 9 9 (- rx 4) (- ry 4)
					:foreground (lv:find-color :name
							(get-color alu color)))))
      (t	      (lv-qplot-string w "?" (+ rx xoff) (+ ry yoff)
				       alu color *plain-font*)))
    ))



;;; QPLOT-SPECIAL-CHAR:
;;;
(defmethod lv-qplot-special-char ((w qplot-window) x y char ascii
				  size font-info alu color)
  (declare (ignore size ascii))
  (lv-qplot-string w char x y alu color font-info))



;;; QPLOT-VECTOR:
;;;
(defmethod lv-qplot-vector ((w qplot-window) from-x from-y to-x to-y
			    alu color arrow-head-length arrow-base-width
			    filled #+lucid thickness dashed dash-pattern
			    shaftthick)
  (declare (ignore thickness))
  (multiple-value-bind (points xbas ybas)
      (triangle-point-translation from-x from-y to-x to-y
				  arrow-base-width arrow-head-length)
    (lv-qplot-line w
		   from-x from-y xbas ybas
		   alu color shaftthick dashed dash-pattern)
    (lv-qplot-polygon w points alu color filled shaftthick)))


;;; QVEC:
;;;
#|
(defmethod lv-qvec ((w qplot-window) x1 y1 x2 y2
		    &key alu color)
  (lv-qplot-vector w x1 y1 x2 y2 :arrow-head-length 4
		                 :arrow-base-width  4
				 :shaftthick        2
				 :alu               alu
				 :color             color))



|#