;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-
(in-package :qsim) ;changed DJC

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;;                                                                         ;;;
;;;                      OCCURRENCE BRANCH ELIMINATION                      ;;;
;;;                                                                         ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;   THIS FILE CONTAINS THE CODE REQUIRED TO ELIMINAT OCCURRENCE BRANCHING. 
;;;   THERE ARE 3 SECTIONS OF THIS FILE:
;;;
;;;         2.  CORE OCCURRENCE BRANCH CODE
;;;         3.  CODE TO DETERMINE THE QUALITAITVE EQUIVALENCE OF TWO STATES
;;;         4.  CODE TO DISPLAY THE OCCURRENCE BRANCH STRUCTURES





;;;  The following additions are not currently used:
  
;(defother qde hash-map)
;(defother lmark matching-lms)
;(defother state abstraction-of)



;;*******************************************************************************
;;;
;;;                   SECTION 2


;;;
;;;   DEFINITIONS
;;;
;;;   EQUIVALENCE-SET:	Equivalence Set
;;;
;;;		A set of qualitatively equivalent states. 
;;;
;;;   EQUIVALENCE-NODE:	Equivalence Node
;;;
;;;             This structure maintains information about a set of states which have
;;;             been determined to be qualitaitvely equivalent.  Slots on this strucure are:
;;;
;;;		     EQUIV-SET          a list of equiv states
;;;                  EQUIV-LMARKS       an alist indexed by lmarks in the ABSTRACT state
;;;                                        listing matching lmarks.  The other lmarks are in the
;;;                                        qspaces of states in the EQUIV-SET
;;;		     ABSTRACT-STATE     An abstraction of the states in EQUIV-SET.  This
;;;                                        will point to one of the states in EQUIV-SET if an
;;;                                        abstraction has not yet been required (i.e. the
;;;                                        elimination of an lmark)
;;;                  COMBINED-FROM      A list of other EQUIVALENCE-NODES which contain an
;;;                                        EQUIV-SET which is a subset of this EQUIV-SET
;;;                  USED?              A boolean slot.  It is true if there is an aggregate
;;;                                        state which has been formed which points to this 
;;;                                        node.  This can be used later to eliminate the
;;;                                        needless creation of equivalence-nodes
;;;                  SUCCESSOR-NODE     Currently not used.
;;;                  TEMINAL?           A boolean slot.  It is true if this equivalence node is
;;;                                        a combination of terminal states.
;;;          
;;;
;;;
;;;   EQUIVALENCE-NODE-LIST:   Equivalent Node List
;;;
;;;             This is an ordered list of EQUIVALENCE-NODEs.  They are ordered so that
;;;             the states abstracted by the EQUIVALENCE-NODEs earlier in the list
;;;             are successors of the states abstracted by nodes which follow
;;;             them in the list.  This allows for an aggregate interval to abstract
;;;             different amounts of the tree.  The EQUIVALENCE-NODE at the
;;;             beginning of the list abstracts as much of the tree as possible.  
;;;             Multiple equivalence nodes abstracting different amounts of the tree
;;;             occur when there are multiple occurrence branches within a tree.
;;;
;;;             For example, if the behavior tree is of the following form: 
;;;             (Note that this is just a "skeleton" of the tree with intervening
;;;             states omitted.
;;;
;;;
;;;
;;;
;;;                                   |-----3
;;;                        |----2-----|-----3 
;;;                        |          |-----3
;;;                        |
;;;                        |          |-----3
;;;                  1-----|----2-----|-----3
;;;                        |          |-----3
;;;                        |
;;;                        |          |-----3
;;;                        |----2-----|-----3
;;;                                   |-----3
;;;
;;;              The states marked 3 are qualitatively equivalent as are the states 
;;;              marked 2.  Starting at state 1 it is possible to abstract all the
;;;              way to state 3 or to simply abstract to state 2.  The 
;;;              EQUIVALENCE-NODE-LIST for this example from state 1 would
;;;              be  (EQ-3 EQ-2).  WHere these equievalence nodes abstract
;;;              states 3 and 2 respectively.  
;;;
;;;              The resulting levels of abstraction in the behavior tree
;;;              which could be displayed are:
;;;
;;;                         |--|
;;;                   1-----|  |-----3
;;;                         |--|
;;;
;;;              or
;;;
;;;                   
;;;                         |--|         |----3
;;;                   1-----|  |-----2---|----3
;;;                         |--|         |----3
;;;
;;;




;;;  
;;;  PERFORM-OCC-BRANCH-AGGREGATION
;;;
;;;  Paramters: a STATE (assumed to be a time interval)
;;;
;;;  Returns:  An EQUIVALENCE-NODE-LIST for STATE.
;;;            If the states grouped under a single EQUIVALENCE-NODE are
;;;            combined, then a SISO subgraph is formed starting from the STATE
;;;            paramter.
;;;
;;;  Descr:    This function will recursively call itself on each of its children.
;;;            It will then atempt to combine the equivalence-node-lists returned
;;;            by its children.
  
;;;            It is assumed that states passed to this function are
;;;            time intervals.  It then creates an EQUIVALENCE-NODE with itself as the
;;;            only member and combines this with the equivalence nodes which results
;;;            from combining its children's nodes.
;;;
;;;            This function will also create aggregate states and attatch them to the
;;;            behavior tree.
;;;
;;;            If state has more than one child, and there exists an EQUIVALENCE-NODE
;;;            in what is returned by COMBINE-EQUIVALENCE-PACKETS, then an aggregate
;;;            state is created for each EQUIVALENCE-NODE returned.  The aggregate
;;;            state is rooted in STATE and it "terminates" in the ABSTRACT-STATE
;;;            in the EQUIVALENCE-PACKET.  If this ABSTRACT-STATE has not been
;;;            expanded (i.e. simulated) then it needs to be placed on a list to be
;;;            expanded later.

(defun perform-occ-branch-aggregation (state)
  (cond ((null state) nil)
	((time-interval-state state)
	 (let* ((children (successor-states state))
		(equivalence-packets
		  (combine-equivalence-node-lists
		    (mapcar #'(lambda (child-state)
				(extend-histories (perform-occ-branch-aggregation child-state) state))
			    children))))
	   (when (and (cdr children)
		      equivalence-packets)
	     (create-aggregate-interval state equivalence-packets))
	   (if equivalence-packets
	       (cons (create-equiv-node state)
		     equivalence-packets)
	       (list (create-equiv-node state)))))
	((term&combine? state) (list (create-equiv-node state)))
	((time-point-state state) (extend-histories (perform-occ-branch-aggregation (car (successor-states state)))
						 state))
	(t (error "State did not match one of the options in the COND."))))


;;;
;;;   COMBINE-EQUIVALENCE-NODE-LISTS
;;;
;;;      Parameters:  EQUIVALENCE-NODE-LIST-SET: A set of equivalence node lists.  Each
;;;                      element of the set is an equivalence node list for a different child
;;;                      of the current state.
;;;
;;;           (<equiv-list-ch-1> <equiv-list-ch-2> <equiv-list-ch-3> ...)
;;;
;;;      Returns:  A new EQUIVALENCE-NODE-LIST which results from combining the sets passed.
;;;
;;;      Descr:  A new equivalence node can only be formed if there is a list of equivalence
;;;              nodes which are qualitatively equivalent such that one and only one
;;;              node is contributed from each child.
;;;
;;;              Furthermore, each list of nodes from each child is ordered upon receipt.
;;;              This function will only generate a single new node if the order of
;;;              two new nodes in the resulting list is unambiguos.
;;;
;;;              For example:  If the following set of lists were passed  where a number
;;;                 represents an equivalence node (i.e not a state) and numerical equality
;;;                 is used in comparisons:
;;;
;;;                             ((1 2 3) (2 1 3) (2 1 3))
;;;
;;;                 The resulting node would be:
;;;
;;;                                     (1 3)
;;;
;;;                 Node 1 would be matched first.  Node 2 would not be matched, however, because
;;;                 the pointer into this list would have passed this node.
;;;
;;;                 Due to the crisscross in the ordering.  It is expected that this condition
;;;                 is an error in itself as it implies cycles within a set.
;;;
;;;              This methodology not only increase efficency, but it eliminates the 
;;;              possiblility of an ambiguos ordering.  To justify this decision, it must be
;;;              shown that this condition cannot occur otherwise, the decision of
;;;              which nodes can be combined would result in an arbitrary decision based upon
;;;              the ordering of the children and would cause different results if an equivalent
;;;              tree with different ordering of its children were processed. 
;;;
;;;      Algorithm:  This function uses the first equivalence-node-list as the reference
;;;                  list against which the others are compared.  It then creates a list
;;;                  of pointers into the rest of the equivalence-node-lists (*EQUIV-SET-PT-LIST*)
;;;                  It will step through the nodes in the REF set and advance the pointers
;;;                  in the *EQUIV-SET-PT-LIST* as matches are found.
;;;
;;;     Calls: GROUP-EQUIV-NODE-SETS
;;;            EQUIV-NODE-UNION



(defun combine-equivalence-node-lists (equivalence-node-list-set)
  (cond ((null equivalence-node-list-set) nil)
	((null (cdr equivalence-node-list-set))    ; if only one set is passed then return it
	 (car equivalence-node-list-set))
	(t
	 (let* ((ref-equivalence-node-list (car equivalence-node-list-set))
		(rest-equivalence-node-lists (cdr equivalence-node-list-set)))
	   (setf equiv-set-pt-list (copy-tree rest-equivalence-node-lists))
	     (remove-nil
	       (mapcar #'(lambda (cur-ref-node)
			   (equivalence-node-union cur-ref-node
						   (group-equiv-node-sets cur-ref-node 
									  equiv-set-pt-list)))
		       ref-equivalence-node-list))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;   GROUP-EQUIV-NODE-SETS
;;;
;;;      Parameters:  REF-NODE
;;;                   EQUIV-SET-PT-LIST
;;;
;;;      Returns:  A list with members of the following form:
;;;
;;;                         (<equiv-node> <matching-lmarks>)
;;;
;;;                There will be an entry in the list for each set within the 
;;;                EQUIV-SET-PT-LIST (i.e. one for each child)
;;;
;;;      Descr:  This function will receive a reference equivalence node (REF-NODE) and 
;;;              a list of pointers into the lists of equivalence node sets from the
;;;              COMBINE-EQUIVALENCE-NODE-LISTS routine.  It will try to locate an
;;;              equivalence node in each of the sets which matches (i.e. qualitaitvely
;;;              equivalent) the REF-NODE.  It will return a list of these nodes with the
;;;              matching-Lmark list for each node if they are
;;;              found.  Otherwise, it will return nil.  It will advance the pointer for each
;;;              equivalence node set to the element foloowing the one that is returned.  
;;;              THis ensures that a strict order is maintained amongst the equivalence
;;;              nodes in a set.
;;
;;;              The list of pointers will not be advanced until a match is found for
;;;              each of the children (i.e. sets) in the list.  Up until this point, the
;;;              new values for the pointers is maintained in NEW-EQUIV-SET-PT-LIST.  The
;;;              variable *CUR-EQUIV-SET-PT* is used by FIND-EQUIV-NODE to pass the new
;;;              value of the pointer for that particular set.
;;;
;;;      Called By: COMBINE-EQUIVALENCE-NODE-LISTS
;;;
;;;      Calls:  FIND-EQUIV-NODE



(defun group-equiv-node-sets (ref-node equiv-set-pt-list)
  (block children-match
    (let* ((new-equiv-set-pt-list nil)
	   (return-val
	     (mapcar #'(lambda (equiv-set-pt)
			       (prog1
				 (cond ((find-equiv-node ref-node equiv-set-pt))
				       (t (return-from children-match nil)))
				 (setf new-equiv-set-pt-list
				       (cons *cur-equiv-set-pt* new-equiv-set-pt-list))))
		     equiv-set-pt-list)))
      ; advance the pointers
      (mapl #'(lambda (equiv-set-pt-l new-equiv-set-pt)
		      (setf (car equiv-set-pt-l) (car new-equiv-set-pt)))
	    equiv-set-pt-list (reverse new-equiv-set-pt-list))
      return-val)))

;;; 
;;;    FIND-EQUIV-NODE
;;;
;;;      Paramters: REF-NODE
;;;                 EQUIV-SET-PT  - a pointer into a list of equivalence nodes
;;;
;;;      Returns:  An EQUIVALENCE NODE from the EQUIV-SET-PT which is qualitatively
;;;                equivalenct to the reference node.  Along with the matching lmarks
;;;                It will take the following form:
;;;
;;;                      (<equiv-node> <matching-lmarks>)
;;;
;;;                It will also set *CUR-EQUIV-SET-PT* to the node following the
;;;                returned node in the equivalence set list passed.
;;;
;;;      Called By:  GROUP-EQUIV-NODE-SETS
;;;

(defun find-equiv-node (ref-node equiv-set-pt)
  (let (return-val)
    (cond ((null equiv-set-pt) nil) 
	  ((setf return-val (qual-equiv-equivalence-nodes ref-node (car equiv-set-pt)))
	   (setf *cur-equiv-set-pt* (cdr equiv-set-pt))
	   return-val)
	  (t 
	   (find-equiv-node ref-node (cdr equiv-set-pt))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;
;;; EQUIVALENCE-NODE-UNION
;;;
;;;    Parameters: REF-EQUIV-NODE - the equivalence node against which the other equivalence
;;;                   nodes where matched.
;;;                EQUIV-NODE-LMARK-MATCH-LIST -this contains a list with the
;;;                   rest of the equivalence nodes.  FOr each equivalence node a lmark-match
;;;                   list is provided.  This is an alist referenced by lmarks from the
;;;                   REF-EQUIV-NODE with equivalent landmarks within the respective
;;;                   equivalence node.
;;;
;;; Returns: A new EQUIVALENCE-NODE whose STATE-SET is the union of the STATE-SETS
;;;          of the EQUIVALENCE-NODEs in the input parameter list and whose
;;;          ABSTRACT-STATE is an abstraction of all of the states in the
;;;          above union.

(defun equivalence-node-union (ref-node equiv-node-lmark-match-list)
  (when equiv-node-lmark-match-list
    (let ((equiv-nodes (mapcar #'car equiv-node-lmark-match-list)) 
	  (equiv-lmark-lists (expand-lmark-match-list equiv-node-lmark-match-list))
	  (new-eq-node nil))
      (when *deb-equiv-node* (format t "~%~% IN EQUIVALENCE-NODE-UNION ~%")
	    (format t " equiv-nodes = ~a~%~%equiv-lmark-lists = ~a~% ref-node = ~a~%" 
		    equiv-nodes equiv-lmark-lists ref-node))
      (setf new-eq-node
	    (make-equivalence-node :name (genname 'EQ)
				   :combined-from (get-combined-from-list equiv-nodes)
				   :equiv-set (apply #'append
						     (mapcar #'equivalence-node-equiv-set
							     (cons ref-node equiv-nodes)))
				   :equiv-lmarks (combine-equiv-lmark-lists 
						   (cons (equivalence-node-equiv-lmarks ref-node)
							 equiv-lmark-lists))
				   :abstract-state (equivalence-node-abstract-state ref-node)
				   :terminal? (equivalence-node-terminal? ref-node)
				   :histories (equivalence-node-histories ref-node)
				   ))
      (set (equivalence-node-name new-eq-node) new-eq-node))))


;;;
;;;  GET-COMBINED-FROM-LIST
;;;
;;;    Paramters:  EQUIV-NODE-LIST
;;;
;;;    Returns:  A list of the equivalence nodes passed it and all of the equivalence
;;;              nodes from which these were combined (i.e. the combined-from slot.)
;;;             
;;;              Currenlty, this information is not used.

(defun get-combined-from-list (equiv-node-list)
  (cond ((null equiv-node-list) nil)
	(t (cons (car equiv-node-list)
		 (append (get-combined-from-list (equivalence-node-combined-from 
						    (car equiv-node-list)))
			 (get-combined-from-list (cdr equiv-node-list)))))))





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;  EXPAND-LMARK-MATCH-LIST
;;;
;;;     Paramters: EQUIV-NODE-LMARK-MATCH-LIST - IT is a list with elements of the
;;;                  following form:
;;;
;;;                    (<equiv-node> <lmark-match-alist>)  
;;;
;;;     Descr:  The lmark-match list contains a list of landmark pairs.  THe first is
;;;             a landmark in the abstract state of the REF-NODE while the second
;;;             is a landmark in the abstract state of equivalence node to which the
;;;             LMARK-MATCH-LIST is attatched.  This equivalence node has a set of
;;;             landmarks which are in turn equivalent to this landmark of the
;;;             abstract-state.
;;;
;;;             This function will expand the lmark-match-list so that it includes not only
;;;             the landmark in the abstract state of the equivalence node, but all
;;;             of the landmarks which are in turn equivalent to this landmark.

(defun expand-lmark-match-list (equiv-node-lmark-match-list)
  (cond ((null equiv-node-lmark-match-list) nil)
	((null (cdar equiv-node-lmark-match-list))     ; there are no matching lmarks for this
                                                       ; equivalence node
	 (expand-lmark-match-list (cdr equiv-node-lmark-match-list)))
	(t
	 (let ((cur-equiv-node (caar equiv-node-lmark-match-list))
	       (cur-lmark-match-list (cadar equiv-node-lmark-match-list)))
	   (cons (mapcar #'(lambda (lmark-pair)
				   (let ((ref-lmark (car lmark-pair))
					 (matching-lmark (cadr lmark-pair)))
				     (cons ref-lmark
					   (get-equiv-lmarks cur-equiv-node
							     matching-lmark))))
			 cur-lmark-match-list)
		 ;recursive call
		 (expand-lmark-match-list (cdr equiv-node-lmark-match-list)))))))

;;;
;;;   GET-EQUIV-LMARKS
;;;
;;;     Paramters:  EQUIV-NODE
;;;                 LMARK       - a landmark from the abstract state of the EQUIV-NODE
;;; 
;;;     Returns:  The set of landmarks which are equivalenct to the LMARK passed for the 
;;;               designated EQUIV-NODE.
;;;
;;      Called By:  EXPAND-LMARK-MATCH-LIST

(defun get-equiv-lmarks (equiv-node lmark)
  (cond ((assoc lmark (equivalence-node-equiv-lmarks equiv-node)))
	(t (list lmark))))



;;;
;;;   COMBINE-EQUIV-NODE-LMARK-LISTS
;;;
;;;      Paramters:  EQUIV-LMARK-LISTS -  a list of equivalent lmark lists.  Each list
;;;                    is for a different equivalence node in the set of nodes being
;;;                    combined. 
;;;
;;;                    Each EQUIV-LMARK-LIST (i.e. which applies to a single equivalence node)
;;;                    matches a landmark from the abstract state of the REF-NODE to a 
;;;                    list of landmarks from the states within the equivalence node to which
;;;                    this list applies.
;;;
;;;      Descr:  This function will extract from the equiv-lmark lists all of the
;;;              landmarks which are used to index into any of the lists.  With this
;;;              set it will step through and create a list of equivalent
;;;              landmarks for  each of these landmarks.
;;;
;;;              This function can be thought of as taking a set of assoc lists and finding
;;;              their union such that every index in the original set of alists is in the
;;;              union and if an index appears more than once in the original set, then the
;;;              union is taken of the sets to which the index points.
;;;


(defun combine-equiv-lmark-lists (equiv-lmark-lists)
  (let ((ref-lmarks nil))
    ; Extract a complete set of the index landmarks
    (mapc #'(lambda (equiv-lmark-list)
		    (mapc #'(lambda (equiv-lmarks)
				    (unless (member (car equiv-lmarks) ref-lmarks :test #'equal)
				      (setf ref-lmarks 
					    (cons (car equiv-lmarks) ref-lmarks))))
			  equiv-lmark-list))
	  equiv-lmark-lists)
    ;  Find the set to which each of these  indeces should point.
    (mapcar #'(lambda (cur-ref-lmark)
		(cons cur-ref-lmark
		      (apply #'append
			     (remove-nil 
			       (mapcar #'(lambda (equiv-lmark-list)
					   (cdr (assoc cur-ref-lmark
						       equiv-lmark-list 
						       :test #'equal)))
				       equiv-lmark-lists)))))
	    ref-lmarks)))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;   CREATE-AGGREGATE-INTERVAL
;;; 
;;;      Paramters: START-STATE  - interval from which the aggregate branches
;;;                 EQUIV-NODE-SET - an ordered set of equivalence nodes
;;;      
;;;      Descr:  This function will create an aggregate node.  It will be connected to
;;;              the start state.  The setf of equivalence nodes represent different
;;;              levels of abstraction.  This is necessary in case a set of
;;;              equivalence nodes branch into another set of equivalent nodes.  The set
;;;              is ordered such that the states farthest from the START-STATE are at the
;;;              end of the set.  THis will be reversed since the most abstract description
;;;              will be presented to the user first.
;;;
;;;              If the combination of the states has not required a new abstract state
;;;              to be created (i.e. no landmarks have been eliminated) then the portion
;;;              of the tree extending beyond the ABSTRACT STATE for a given equivalence
;;;              node can be used as the behavior extending beyond this aggregate.  If
;;;              a new abstract state has been created due to the elimination of
;;;              landmarks, then simulation must be performed for this abstract state.




(defun create-aggregate-interval (start-state equiv-node-set)
  (let* ((ordered-equiv-node-set (reverse (filter-equiv-node-set equiv-node-set)))
	 (new-aggregate (make-aggregate-interval 
			  :name (genname 'AGG)
			  :start-state start-state
			  :equiv-nodes (mapcar #'create-equiv-node-copy ordered-equiv-node-set))))
    (setf (aggregate-interval-cur-level new-aggregate)
	  (aggregate-interval-equiv-nodes new-aggregate))
    (setf (state-aggregates start-state) new-aggregate)
    (mark-used equiv-node-set)
    (setf *aggregate-intervals* (cons new-aggregate *aggregate-intervals*))
    (set (aggregate-interval-name new-aggregate) new-aggregate)
    (format *Qsim-Report* "~&Created an aggregate interval (~a) starting at ~a.  The following states " 
	    new-aggregate start-state)
    (format *Qsim-Report* "have been abstracted: ~%~a"
	    (mapcan #'equivalence-node-equiv-set ordered-equiv-node-set))))
		    
 

(defun mark-used (equiv-node-list)
  (cond ((null equiv-node-list) nil)
	(t (setf (equivalence-node-used? (car equiv-node-list)) t)
	   (mark-used (cdr equiv-node-list)))))

;;;
;;;   CREATE-EQUIV-NODE
;;;
;;;     Paramters:  STATE
;;;
;;;     Returns:  An equivalence node with STATE as the only state in the EQUIV-SET.
;;;               STATE is also used as the ABSTRACT-STATE


(defun create-equiv-node (state)
  (let ((new-eq-node
	  (make-equivalence-node :name (genname 'EQ)
				 :equiv-set (list state)
				 :terminal? (when (term&combine? state) t)
				 :abstract-state  state
				 :histories (initialize-histories (state-qvalues state)))))
    (set (equivalence-node-name new-eq-node) new-eq-node)))

(defun create-equiv-node-copy (equiv-node)
  (let ((new-eq-node
	  (make-equivalence-node :name (genname 'EQ)
				 :equiv-set (equivalence-node-equiv-set equiv-node)
				 :terminal? (equivalence-node-terminal? equiv-node)
				 :abstract-state  (equivalence-node-abstract-state equiv-node)
				 :histories (equivalence-node-histories equiv-node))))
    (set (equivalence-node-name new-eq-node) new-eq-node)))


;;;
;;;  INITIALIZE-HISTORIES
;;;
;;;  Will receive a list of qvalues and it will convert this to the
;;;  proper format for a history.


(defun initialize-histories (qvalues)
  (mapcar #'(lambda (qval)
	      (cons (car qval)
		    (list (cdr qval))))
	  qvalues))



;;;  Will determine if the two histories are equivalent using the
;;;  landmark matching set provided in LMARK-MATCH-SET

(defun equiv-histories (histories1 histories2 lmark-match-set)
  (if *match-histories*
      (mapc #'(lambda (hist1 hist2)
		(mapc #'(lambda (qval1 qval2)
			  (if (qval-match qval1 qval2 lmark-match-set)
			      t
			      (return-from equiv-histories nil)))
		      (cdr hist1) (cdr hist2)))
	    (cdr histories1) (cdr histories2))
      t)
  histories1)


;;;  Will extend the histories in the list of equivalence nodes
;;;  (EQUIV-NODE-LIST) by the values in STATE

(defun extend-histories (equiv-node-list state)
  (when equiv-node-list
    (mapcar #'(lambda (equiv-node)
		(setf (equivalence-node-histories equiv-node)
		      (add-qvals-to-histories (equivalence-node-histories equiv-node)
					      (state-qvalues state)))
		equiv-node)
	    equiv-node-list)))


;;;  Will add the QVALUES to the HISTORIES

(defun add-qvals-to-histories (histories qvalues)
  (if histories
      (mapcar #'(lambda (history var-qval)
		  (let ((history-vals (cdr history))
			(qval (cdr var-qval)))
		    (if (and history-vals
			     (qval-equal (car history-vals)
					 qval))
			history
			(cons (car history)
			      (cons
				qval
				history-vals)))))
	      histories qvalues)
      (initialize-histories qvalues)))
  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;                       MISC FUNCTIONS 
;;;
;;;
;;;
;;; QUAL-EQUIV-EQUIVALENCE-NODES
;;;
;;; Parameters: EQUIV-NODE1:  The equivalence node against which the other one
;;;                must be matched.
;;;             EQUIV-NODE2:
;;;
;;; Returns:    Determines if two EQUIV-NODE1 and EQUIV-NODE2 are 
;;;             qualitatively equivalent.  It will return an alist  indexed
;;;             by an lmark in the EQUIV-NODE1 matching it to an lmark in
;;;             EQUIV-NODE2.
;;;   
;;;            Two EQUIVALENCE-NODES are considered to be qualitatively equivalent if the
;;;            ABSTRACT-STATEs connected to them are qualitatively equivalent.
;;;
;;;     The histories for the two equivalence nodes must be identical for
;;;     the match to occur if *match-histories* is set.

(defun qual-equiv-equivalence-nodes (equiv-node1 equiv-node2)
    (let ((lmark-set
	    (qual-equiv-states (equivalence-node-abstract-state equiv-node1)
			       (equivalence-node-abstract-state equiv-node2))))
      (when (and lmark-set
		 (equiv-histories (equivalence-node-histories equiv-node1)
				  (equivalence-node-histories equiv-node2)
				  lmark-set))
	(list equiv-node2 (remove-nil lmark-set)))))





(defun filter-equiv-node-set (equiv-node-set)
  "This function filters equivalence nodes in an equivalence node set which
   have nodes of the same size (i.e. same number of states) immediately preceding
   them in the list.  This is because the earlier node already abstracts all
   of the branches in the succeeding node."
  (do* ((ref-node-size (equiv-node-size (car equiv-node-set)))
	(equiv-node-list (cdr equiv-node-set) (cdr equiv-node-list))
	(return-nodes (list (car equiv-node-set)))
	(next-node-size (equiv-node-size (car equiv-node-list))
			(equiv-node-size (car equiv-node-list))))
       ((null equiv-node-list) (nreverse return-nodes))
    (unless (eq next-node-size ref-node-size)
      (setq return-nodes (cons (car equiv-node-list)
			       return-nodes))
      (setq ref-node-size next-node-size))))




(defun earlier-interval (term-node other-nodes)
  "Returns T if there is an interval eq-node in OTHER-NODES which matches TERM-NODE"
  (cond ((null other-nodes) nil)
	((equal (length (equivalence-node-equiv-set term-node))
		(length (equivalence-node-equiv-set (car other-nodes)))))
	(t (earlier-interval term-node (cdr other-nodes)))))


(defun term-eq-node? (eq-node)
  "Returns true if this is a terminal equivalence node"
  (equivalence-node-terminal? eq-node))





;;;
;;;   TERM&COMBINE?
;;;
;;;      Parameters:  STATE
;;;
;;;      Returns:  True if the state is a terminal state and it is valid to combine this state
;;;                with other terminal states to form an aggregate.  This function currently
;;;                determines this in a very superficial fashion.  It uses the fact
;;;                that the successor states is nill to determine if it is a final state
;;;                due to the fact that "final" was not in the status list for a model
;;;                obtained from David Franke run via CC.

  
(defun term&combine? (state)
  (let ((next-child-status (state-status state)))
    (and (null (successor-states state))
	 (not (member 'cycle next-child-status))
	 (not (inconsistent-p state))
	 (not (incomplete-p state)))))



;;; 
;;;  LMARK-EQUAL-ROBUST
;;;
;;;    Descr:  This function is a robust feature of lmark-equal.  Before calling
;;;            lmark-equal it ensures that bother values are lmarks.

(defun lmark-equal-robust (lm1 lm2)
  (and (lmark-p lm1)
       (lmark-p lm2)
       (lmark-equal lm1 lm2)))

;;;
;;;   Used to keep track of where I have taken the cdr of a list to remove time as a
;;;   variable.

(defun remove-time (list)
  (cdr list))




;;;  Returns the number of states which are abstracted by this equivalence node
(defun equiv-node-size (equiv-node)
  (if equiv-node
      (length (equivalence-node-equiv-set equiv-node))
      0))


;;;  Can be used to remove all aggregate intervals from a tree.  Useful
;;;  for debugging.
(defun erase-aggregates (&optional (state *initial-state*))
  "This function will remove all of the aggregate intervals from the
   tree which follows from the designated state."
  (setf (state-aggregates state) nil)
  (mapc #'erase-aggregates (successor-states state)))



;;*******************************************************************************
;;;
;;;         SECTION 3


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   QUALITATIVE EQUIVALENT STATE FUNCTIONS
;;;
;;;   This file contains code to determine if two states are qualitaitvely
;;;   equivalent.  This is used in the elimination of occurrence brnaching
;;;   via the creation of aggregate intervals.

;;;
;;;  QUAL-EQUIV-STATES
;;;
;;;    Parameters:  STATE1 STATE2
;;;  
;;;    Returns:  Nil if the states do not qualitatively match.  Otherwise
;;;              it returns an alist of the matching landmarks in the
;;;              qsapces for each variable.

(defun qual-equiv-states (state1 state2)
  (block qual-match
    (when (compatible-status state1 state2)
      (let ((comp-lmark-matches
	      (apply #'append                                  ;;; probably can change to mapcan
		     (mapcar #'(lambda (cur-qval1 cur-qval2
						  cur-qsp1 cur-qsp2)
				 (let ((cur-lmark-matches
					 (qspace-match (cdr cur-qsp1)
						       (cdr cur-qsp2))))
				   (cond ((null cur-lmark-matches)
					  (if *deb-qual-equiv* 
					      (format t "~%~% Qspaces failed to match~%"))
					  (return-from qual-match nil))
					 ((qval-match (cdr cur-qval1)
						      (cdr cur-qval2)
						      cur-lmark-matches)
					  cur-lmark-matches)
					 (t (if *deb-qual-equiv* (format t "~%~%Qvals failed to match"))
					    (return-from qual-match nil)))))
			     (remove-time (state-qvalues state1))
			     (remove-time (state-qvalues state2))
			     (remove-time (state-qspaces state1))
			     (remove-time (state-qspaces state2))))))
	(if *deb-qual-equiv* (format t "~%Calling cvals~%"))
	(when (and (cvals-set-match (state-cvalues state1)
				    (state-cvalues state2)
				    (remove-nil comp-lmark-matches)))
	  comp-lmark-matches)))))


;;;
;;;   COMPATIBLE-STATUS
;;;
;;;      Paramters:  two states
;;;
;;;      Returns:   True if the statuses of the two states are compatible for combination.
;;;         
;;;                 NEEDS EXPANSION.  CURRENTLY ONLY A SIMPLE CAOMPATABLITY CHECK IS DONE.

(defun compatible-status (state1 state2)
  (null (set-difference (state-status state1)
			(state-status state2))))




;;;
;;;  QSPACE-MATCH
;;; 
;;;  parameters:   QSPACE1 QSPACE2
;;;
;;;  Returns:  A list of matching landmarks which do not exist
;;;            in the intersection of the landmarks if the two
;;;            qspaces match.  Otherwise nil


(defun qspace-match (qspace1 qspace2)
  (when (equal (length qspace1)
	       (length qspace2))
	 (block matching
		(mapcar #'(lambda (lm1 lm2)
			    (cond ((lmark-equal lm1 lm2) nil)
				  ((lmark-equiv-qsp lm1 lm2) 
				   (list lm1 lm2))
				  (t (return-from matching nil))))
			qspace1 qspace2))))


;;;
;;;  LMARK-EQUIV-QSP
;;;
;;;    Paramters:  LM1 LM2
;;;
;;;    Descr:  This function is designed to work with QSPACE-MATCH.  It determines if
;;;            two lmarks are equivalent.  Normally, it would need to check to
;;;            see that the lamrks are defined within the same region.  Since qspace
;;;            match steps through the qspaces, a mismatch in where two lmarks are
;;;            defined will be detected.
;;;
;;;            It uses the where-defined field of the lmark to detemrine if it is an 
;;;            original lmark.  If it is it expacts the values to be equal, otherwise the
;;;            values must be defined for the same reason.

(defun lmark-equiv-qsp (lm1 lm2)
  (cond ((or (null (lmark-where-defined lm1))
	     (null (lmark-where-defined lm2)))
	 (lmark-equal lm1 lm2))
	(t (match-lmark-why (lmark-why-defined lm1)
			    (lmark-why-defined lm2)))))

(defun match-lmark-why (lm1-why lm2-why)
  (equal lm1-why lm2-why))

;;;
;;;  QVAL-MATCH
;;;
;;;   Paramters:  QVAL1 QVAL2 LMARK-MATCHES
;;;
;;;   Returns:  This function checks to see if the values passed
;;;;            as paramters qualitatively match.  It would have been
;;;;            sufficient to simply check the qmag's of all of the
;;;             variables, but this allows us to possible detect a mismatch
;;;             in the qdir's earlier.  It will matchthe mags and then
;;;             the qdirs.

(defun qval-match (qval1 qval2 lmark-matches)
  (and (qmag-match (qval-qmag qval1)
		   (qval-qmag qval2)
		   lmark-matches)
       (equal (qval-qdir qval1)
	      (qval-qdir qval2))))

;;;   QMAG-MATCH
;;;
;;;   Paramters:   QMAG1 QMAG2
;;;                LMARK-MATCHES   A list of lmark pairs which identifies
;;;                                lamrks from different qspaces which have 
;;;                                been determined to match
;;;
;;;   Returns:  T if the QMAGS match, otherwise nil

(defun qmag-match (qmag1 qmag2 lmark-matches)
  (cond ((and (point-p qmag1)
	      (point-p qmag2))
	 (check-lm-match qmag1 qmag2 lmark-matches))
	((and (interval-p qmag1)
	      (interval-p qmag2))
	 (and (check-lm-match (car qmag1)
			      (car qmag2) lmark-matches)
	      (check-lm-match (cadr qmag1)
			      (cadr qmag2) lmark-matches)))
	(t nil)))


(defun check-lm-match (lm1 lm2 lmark-matches)
  (let ((matching-lm nil))
    (or (lmark-equal lm1 lm2)
	(when (setf matching-lm (assoc lm1 lmark-matches))
	  (lmark-equal lm2 (cadr matching-lm))))))


;;;
;;;  CVALS-SET-MATCH
;;;
;;;  Parameters:  CVALS-SET1 CVALS-SET2
;;;               COMP-LMARK-MATCHES      the lmark matches for all of the variables
;;;
;;;  Returns:  True if all of the corresponding values in
;;;            sets of corresponding values either have equal lmarks
;;;            or ones that match  pair in comp-lmark-matches
;;;
;;;     The corresponding value attribute is of the form:
;;;
;;;        (<constraint> <cvals> <cvals> ...)
;;;     with <cvals> being of the form
;;;
;;;        ((
;;;

(defun cvals-set-match (cvals-set1 cvals-set2 comp-lmark-matches)
  (or (and (null cvals-set1) (null cvals-set2))
      (when (equal (length cvals-set1)
		   (length cvals-set2))
	(block matching
	  (mapcar #'(lambda (cvals-list1 cvals-list2)
		      (cond ((cvals-list-match (cdr cvals-list1) (cdr cvals-list2) comp-lmark-matches) t)
			    (t (return-from matching nil))))
		  cvals-set1 cvals-set2)))))

  

;;;
;;;  THis function will determine if the cvals for a particular constraint
;;;  are compatible.  The two cvals lists do not need to be of the same
;;;  size.  A conflict occurs if there is a cvals set in each of the
;;;  lists which differ in only one variable.  
;;;
;;;  At some point, it should be modified to return a composite cvals set.
;;;  This would be the uniun of the cvals passed to it.


(defun cvals-list-match (cvals-list1 cvals-list2 comp-lmark-matches)
  (do* ((cv-list1 cvals-list1 (cdr cv-list1))
	(cv1 (car cv-list1) (car cv-list1))
	(conflict-found nil))
       ((or conflict-found 
	    (null cv1))
	(not conflict-found))
    (do* ((cv-list2 cvals-list2 (cdr cv-list2))
	  (cv2 (car cv-list2) (car cv-list2)))
	 ((or (null cv2)
	      (setf conflict-found
		    (conflict-cvals cv1 cv2 comp-lmark-matches)))))))

(defun tmp-1 (t1) (not t1))

(defun conflict-cvals (cv1 cv2 comp-lmark-matches)
  (do ((lms1 cv1 (cdr lms1))
       (lms2 cv2 (cdr lms2))
       (conflicts 0))
      ((or (null lms1)
	   (eq conflicts 1))
       (if (eq conflicts 1) 
	   t
	   nil))
    (if (not (or (lmark-equal (car lms1) (car lms2))
		 (let ((lm2-match (cadr (assoc (car lms1)
					  comp-lmark-matches))))
		   (when lm2-match
		     (lmark-equal (car lms1)
				  lm2-match)))))
	(setf conflicts (1+ conflicts)))))


;(defun cvals-list-match (cvals-list1 cvals-list2 comp-lmark-matches)
;  (block more-matching
;	 (mapcar #'(lambda (lm-list1 lm-list2)
;		     (mapcar #'(lambda (lm1 lm2)
;				 (cond ((check-lm-match lm1 lm2 comp-lmark-matches) t)
;				       (t (return-from more-matching nil))))
;				 lm-list1 lm-list2))
;		 cvals-list1 cvals-list2)))

;;; 
;;;   REMOVE-NIL
;;;
;;;   Will return a list with all nil values removed

(defun remove-nil (list)
  (remove-if #'null list))


;;;
;;;  REMOVE-ASSOC
;;;
;;;    Paramters:  ALIST-MATCH   an association list indexed by variabels
;;;                              within the qde.  The a-list contains a list
;;;                              of matching landmarks within the qspaces
;;;                              of the respective variables for two states
;;;                              being compared.
;;;
;;;    Returns:  Returns a single list of the matching landmarks for
;;;              for all of the variables by stripping away the variable
;;;              name and appending the lists.

(defun remove-assoc (match-alist)
  (cond ((null match-alist) nil)
	(t (append (cdar match-alist)
		   (remove-assoc (cdr match-alist))))))



;;*******************************************************************************
;;;
;;;                        SECTION 4



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;     This file contains code which is used to support the display of
;;;     aggregate intervals.  THis includes the menu functions and the
;;;     functions required to display the rectangles in the behavior tree.





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;
;;;  OCC-BRANCH-DISPLAY-MENU
;;;
;;;    The following options are provided:
;;;
;;;          - Expand an aggregate behavior
;;;
;;;          - Toggle occurrence branching display on and off
;;;
;;;          - Reset all of the aggregate intervals in the behavior
;;;            tree to their highest level of abstraction.
;;;
;;;          - Undo the last aggregate interval expansion.
;;;
;;;     This menu is designed to allow the user to search through the space
;;;     of abstractions of the behavior description for the appropriate level
;;;     of description.  Once occurrence branching aggregation has been performed
;;;     the highest level of abstraction will be presented to the user.  The
;;;     user can then select aggregate intervals to be expanded into more
;;;     detail.  A stack is maintained of the aggregate intervals which are
;;;     expanded.  The user is allowed to step back through this stack of
;;;     expansions to move backward up the hiearchy of abstractions.  The
;;;     user is also able to reset the abstraction level to the highest level.
;;;        

(defun occ-branch-display-menu (visible-aggregates all-aggregates)
  (let ((cmd (get-occ-branch-command)))
    (case cmd
      ((#\P #\p) (format t "~%~%THIS FUNCTIONALITY IS NOT PROVIDED YET.~%~%") nil)
      ((#\A #\a) (let ((agg (get-aggregate-to-expand visible-aggregates)))
		   (when agg 
			 (push agg *agg-stack*)
			 (expand-aggregate agg)
			 (list (list 'recalc-beh nil)))))
      ((#\T #\t) (setf *filter-occ-branching* (not *filter-occ-branching*)) 
		 (list (list 'recalc-beh nil)))
      ((#\R #\r) (mapc #'reset-aggregate all-aggregates)
		 (list (list 'recalc-beh nil)))
      ((#\U #\u) (cond (*agg-stack* (unexpand-aggregate (pop *agg-stack*))
				    (list (list 'recalc-beh nil)))
		       (t (format *Qsim-Report* "~&No action on the stack.  Cannot undo.")
			  nil)))
      (t (format *Qsim-Report* "~&~A is an invlaid command.  Back to the basic prompt." cmd)))))

(defun get-occ-branch-command ()
  (format *Qsim-Report* "~&A=Aggregate Beh Expansion, T=Toggle Display Aggregates, ~
                           R=Reset to highest level, U=Undo last expansion: ")
  (clear-input)
  (read-char))

;;;
;;;  This function will prompt the user for an aggregate interval number.  This number
;;;  will be converted to an aggregate interval.  If this aggregate interval is 
;;;  visible at this level of abstraction, then it will be returned.

(defun get-aggregate-to-expand (visible-aggregates)
  (format *Qsim-Report* "~& Select the aggregate behavior to expand (~a" 
	  (get-index-string (aggregate-interval-name (car visible-aggregates))))
  (mapc #'(lambda (agg)
	    (format *Qsim-Report* " ~a" (get-index-string (aggregate-interval-name agg))))
	(cdr visible-aggregates))
  (format *Qsim-report* "): ")
  (let* ((input (read))
	 (aggregate (and (typep input 'fixnum)
			 (convert-to-valid-aggregate input visible-aggregates))))
    (if (not aggregate)
	(progn
	  (format *Qsim-Report* "~& Invalid aggregate number.  Return to the top level.")
	  nil)
	aggregate)))

(defun print-visible-agg-numbers (agg-list)
  "Print the numbers of the aggregate intervals in agg-list."
  (mapc #'(lambda (agg)
	    (format *Qsim-Report* "~a" (get-index-string (aggregate-interval-name agg))))
	agg-list))


;;;  Convert AGG-NUM to an aggregate interval.  If it is in the list of
;;;  visible-aggregates then return this structure.  Otherwise, return nil

(defun convert-to-valid-aggregate (agg-num visible-aggregates)
  (let ((aggregate (eval (find-symbol (gen-symbol-name 'agg agg-num *dashp*) 'qsim))))
    (car (member aggregate visible-aggregates :test #'equal))))


;;;
;;;  Receives a list of behaviors and it will step through them and return a
;;;  list of the aggregate states which are included in this list.

(defun find-aggregates-in-behaviors (behaviors)
  (remove-duplicates (mapcan #'(lambda (beh)
				 (remove-if-not #'(lambda (state) (typep state 'aggregate-interval))
						beh))
			     behaviors)
		     :test #'equal))

;;;
;;;  This function will find all of the aggregate behaviors which follow from
;;;  this state anywhere in the behavior tree.
;;;  Modified so that it will accept either a single state or a list of states.
;;;  DJC  23 OCt 91


(defun get-all-aggregates (state-or-states)
  (let ((states (if (listp state-or-states)
		    state-or-states
		    (list state-or-states))))
    (mapcan #'(lambda (state)
		(remove-nil
		  (mapnode #'(lambda (state)
			       (values (when (aggregate-interval-p state)
					 state)
				       t))
			   state)))
	    states)))

;(defun get-all-aggregates (state-or-states)
;  (cond ((null state-or-states) nil)
;	((atom state-or-states)
;	 (if (state-aggregates state-or-states)
;	     (cons (state-aggregates state-or-states) (mapcan #'get-all-aggregates
;							      (successor-states state-or-states)))
;	     (mapcan #'get-all-aggregates (successor-states state-or-states))))
;	(t (let ((state (car state-or-states)))
;	     (if (state-aggregates state)
;		 (append (cons (state-aggregates state)
;			       (mapcan #'get-all-aggregates
;				       (successor-states state)))
;			 (get-all-aggregates (cdr state-or-states)))
;		 (append (mapcan #'get-all-aggregates (successor-states state))
;			 (get-all-aggregates (cdr state-or-states))))))))

;;;
;;;  THis function will plot an aggregate interval in the behavior tree.

(defun qplot-agg-node (agg x y)
  (if (not (typep agg 'aggregate-interval))
	   (error "Function called with something that is not an aggregate interval"))
  (let ((name (get-index-string (aggregate-interval-name agg))))
    (qplot-box (- x 2)(- y 5) 14 10)    ;; x and y are adjusted to center the box
    (qplot-string name
		   x  (+ y 5) :font axis-font)))

;;;
;;;  Expands the aggregate-interval to the next lower level of abstraction.
;;;  This entails moving the cur-level pointer in the aggregate-interval
;;;  structure further down the equiv-nodes list.

(defun expand-aggregate (agg-interval)
  (setf (aggregate-interval-cur-level agg-interval)
	(cdr (aggregate-interval-cur-level agg-interval))))


;;;
;;;  Resets the aggregate-interval to the highest level of abstraction.
;;;  The cur-level pointer must be reset to the begining of the
;;;  equiv-nodes list.

(defun reset-aggregate (agg-interval)
   (setf (aggregate-interval-cur-level agg-interval)
	 (aggregate-interval-equiv-nodes agg-interval)))

;;;
;;;  Moves the cur-level pointer backwards to the previous node int
;;;  in the equiv-nodes list.  THis is the inverse of the
;;;  expand-aggregate function.

(defun unexpand-aggregate (agg-interval)
  (if (equal (aggregate-interval-cur-level agg-interval)
	     (aggregate-interval-equiv-nodes agg-interval))
      (format *Qsim-Report* "~&Aggregate behavior is at the top level.  Cannot backup further.")
      (backup-cur-level-pointer (aggregate-interval-equiv-nodes agg-interval)
				  agg-interval)))


(defun backup-cur-level-pointer (equiv-nodes-list agg-interval)
  (if (equal (aggregate-interval-cur-level agg-interval)
	     (cdr equiv-nodes-list))
      (setf (aggregate-interval-cur-level agg-interval)
	    equiv-nodes-list)
      (backup-cur-level-pointer (cdr equiv-nodes-list)
				   agg-interval)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;    code for plotting a behavior


;;;  Extracts the histories from AGG-INTERVAL
(defun aggregate-interval-histories (agg-interval)
  (let ((eq-node (car (aggregate-interval-cur-level agg-interval))))
    (when eq-node
      (equivalence-node-histories eq-node))))

;;;  Returns the size of the largest history in the AGG-INTERVAL
(defun max-history-size (agg-interval)
  (let ((max-hist 0))
    (dolist (cur-history (cdr (aggregate-interval-histories agg-interval)) max-hist)
      (when (> (length (cdr cur-history)) max-hist)
	(setf max-hist (length (cdr cur-history)))))))


(defun size-of-agg-histories-in-beh (beh)
  (cond ((null beh) 0)
	((typep (car beh) 'aggregate-interval)
	 (+ (- (max-history-size (car beh)) 1)
	    (size-of-agg-histories-in-beh (cdr beh))))
	(t (size-of-agg-histories-in-beh (cdr beh)))))



(defun first-qval-in-history (param agg-interval)
  "Returns the first value for PARAM in the AGG-INTERVAL history"
  (second (assoc param (aggregate-interval-histories agg-interval) :test #'equal)))


(defun qvals-in-history (param agg-interval)
  "Returns the history of an individual PARAM in the AGG-INTERVAL"
  (cdr (assoc param (aggregate-interval-histories agg-interval) :test #'equal)))


