;; -*- Mode:Common-Lisp; Package: Qsim; Syntax:Common-Lisp; Base:10 -*-

;;; $Id: numsim.lisp,v 1.6 1992/07/07 11:58:39 bert Exp $

(in-package 'qsim)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Do a numerical simulation of a QDE using expected values for function
;;;; envelopes, constants, and initial values for state vars.
;;;; See the description of the function call NUMSIM for more info.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;; See the file NQ:NQ;structures.lisp for the definitions of the
;;;; numsim structures and related qde and state slots.

;;;; This file contains the following parts:
;;;; 1. Propagation functions used to complete a numerical state that is
;;;;    is consistent with a QDE (and a QSIM state, if desired).
;;;; 2. Support functions for simulation.
;;;; 3. Top-level numsim call.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; 1. Propagation functions used to complete a numerical state that is
;;;;    is consistent with a QDE (and a QSIM state, if desired).
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Numerical values are propagated through the constraint network by
;;; repeatedly scanning through a list of incomplete constraints and
;;; removing any constraint that has all its values set.  If this
;;; process succeeds, then all the constraints have been satisfied.
;;; If it fails, then there is some constraint that does not hold,
;;; or there was not enough information to propagate values to every
;;; model variable.


;;; Propagation structures
;;;
(defstruct npVar
  "A variable used for numeric propagation in numsim"
  name              ; a symbol
  value             ; a number
  )

(defstruct npCon
  "A constraint used for numeric propagation in numsim"
  name             ; the QSIM constraint-name
  vals             ; a list of (varname value) pairs
  )


;;;---------------------------------------------------------------------
;;; Propagate a set of initial values through the constraint network.
;;; Inputs:  qde             - The qde.
;;;          initial-values  - A list of initial value entries (see comments
;;;                            to numsim).
;;;          function-envelopes - :lb, :ub, or NIL (= :expected).
;;;          use-state       - A reference state or NIL.
;;;          match-state     - T or NIL.
;;;          expected-values - A list of values to tack to the front of the
;;;                            QDE expected-values clause.
;;;          svars, cvars, ovars - The state, constant, and other var lists.
;;; Returns: A list of (var value) entries for every variable in the qde.
;;;---------------------------------------------------------------------
;;;
(defun numsim-propagate (qde initial-values function-envelopes
			 use-state match-state
			 expected-values
			 svars cvars ovars)
  (let* ((*expected-values* (append expected-values (qde-expected-values qde)))
	 (*npCons* (mapcar #'(lambda (x)
			       (make-npCon :name (constraint-name x)
					   :vals (mapcar #'(lambda (y)
							     (list y NIL))
							 (cdr (constraint-name x)))))
			   (remove-if #'(lambda (x)
					  (or (not (constraint-active-p x))
					      (case (first (constraint-name x))
						(D/DT     T)
						(CONSTANT (= (length (constraint-name x))
							     2)))))
				      (qde-constraints qde))))
	 (dvars    ;; A list of the derivative variables.
	           ;; Partition-qde should probably generate this rather than
	           ;; doing it here.
	           (loop for c in (qde-constraints qde)
			 when (and (constraint-active-p c)
				   (eq (first (constraint-name c)) 'D/DT))
			 collect (third (constraint-name c))))
	 (*npvars* NIL)			 
	 (*qde*    qde)
	 (*state* use-state)
	 (*function-envelopes*  function-envelopes)
	 (all-vars (append svars cvars ovars))
	 completion-status)
    (declare (special *npVars* *npCons* *qde* *state*
		      *function-envelopes* *expected-values*))

    ;; First set any defaults.
    (setf *npVars* (np-initialize-network initial-values qde use-state
					  all-vars svars cvars ovars dvars))

    (setf completion-status (np-satisfy-constraints *npCons*))

    (case completion-status
      (INCOMPLETE   (error "The constraint network is incomplete.~%~ 
                            The following constraints were left incomplete : ~%~ 
                            ~:{~%~a ~30T ~a ~}~%~
                            The variables defined are~%~
                            ~{~%~a~}"
			   (mapcar #'(lambda (v)
				       (list (npCon-name v) (npCon-vals v)))
				   *npCons*)
			   (mapcar #'(lambda (x)
				       (list (npvar-name x) (npvar-value x)))
				   *npVars*)))
      (COMPLETE     (when (and match-state use-state)
		      ;; Make sure that all variables are consistent with the state.
		      (loop for v in all-vars
			    with error = nil
			    do (let ((state-range (np-get-range v use-state nil))
				     (value (npVar-value
					       (find v *npVars* :key #'npVar-name))))
				 (when (null (np-var-in-range value state-range))
				   (format *Qsim-report*
					   "Variable ~a (=~a) is not in the range ~a."
					   v value state-range)
				   (setf error T)))
			    finally (when error
				      (error "Initial state is not consistent with ~a."
					     use-state))))

		    ;; Return the values for all the variables.
                    (loop for i in *npVars*
			  collect (list (npVar-name i) (npVar-value i))))
      )))





;;;---------------------------------------------------------------------
;;; Create npVar structures from the :initial-values keyword argument
;;; to numsim.
;;; Inputs:  initial-values - A list of initial value entries.
;;;          qde            - The qde.
;;;          use-state      - A reference state (or NIL).
;;;          all-vars       - A list of all the qde variables.
;;;          svars, cvars, ovars, dvars - The state, constant, other,
;;;                           and derivative vars (note that these sets
;;;                           don't cleanly partition the variable space).
;;; Returns: A list of npVar structs.
;;;---------------------------------------------------------------------
;;;
(defun np-initialize-network (initial-values qde use-state
			      all-vars svars cvars ovars dvars)
  (declare (special *expected-values*) (ignore ovars))
  (let ((init-npVars NIL))

    ;; First set any defaults.
    (setf init-npVars
	  (loop for default in (loop for iv in initial-values
				     when (keywordp (first iv))
				     collect iv)
		for var-set = (case (first default)
				(:default    all-vars)
				(:state      svars)
				(:constant   cvars)
				(:derivative dvars)
				(T         (error "Unknown default type ~a"
						  (first default))))
		nconc (cond
			
			;; (<default> :from-state <accessor>)
			((eq (second default) :from-state)
			 (when use-state
			   (loop for var in var-set
				 with value = nil
				 when (setf value (find-expected-value-for-qmag
						   var (qmag (qval var use-state))
						   qde use-state
						   (third default) *expected-values*))
				 collect (make-npVar :name var
						     :value value))))
			
			;; (<default> <number>)
			((numberp (second default))
			 (loop for var in var-set
			       collect (make-npVar :name var :value (second default))))
			
			;; (<default> <lmark> <accessor>) (unlikely, but possible)
			(T
			 (loop for var in var-set
			       with value = nil
			       when (setf value (find-expected-value-for-lmarkname
						 var (second default)
						 qde use-state
						 (third default) *expected-values*))
			       collect (make-npVar :name var
						   :value value))))))

    ;; Now set all the other initial-values.
    (loop for iv-entry in initial-values
	  when (not (keywordp (first iv-entry)))
	  do (let* ((var   (first iv-entry))
		    (npVar (cond
			     ((find var init-npVars :key #'npVar-name))
			     (T
			      (car (push (make-npVar :name var)
					 init-npVars)))))
		    value)
	       (cond
		((numberp (second iv-entry))
		 (setf (npVar-value npVar) (second iv-entry)))
		((eq (second iv-entry) :from-state)
		 (setf value (find-expected-value-for-qmag
			      var (qmag (qval var use-state))
			      qde use-state
			      (third iv-entry) *expected-values*))
		 (if value
		     (setf (npVar-value npvar) value)
		     (when (not (member :none (third iv-entry)))
		       (error "No value found for ~a.  Its qmag is ~a."
			      iv-entry (qmag (qval var use-state))))))
		((null (second iv-entry))
		 (setf (npVar-value npvar) NIL))
		(T
		 (setf value (find-expected-value-for-lmarkname
			      var (second iv-entry)
			      qde use-state
			      (third iv-entry) *expected-values*))
		 (if value
		     (setf (npVar-value npvar) value)
		     (error "No value found for ~a."
			    iv-entry))))))
    init-npVars))



;;;; Finding ranges for variables and comparing numbers and ranges
;;;;

;;;---------------------------------------------------------------------
;;; Given a variable, attempt to find its range using the given state.
;;; Inputs:  var      - A variable name.
;;;          state    - A state.
;;;          at-lmark - If T, then the variable must be at a lmark to
;;;                     succeed.
;;; Returns: A list (LB UB) where the entries are drawn from the
;;;          extended reals (+/-inf) or NIL if one can't be found
;;;          (only possible if at-lmark is T).
;;; Notes: We can't really look up the range of a var in the initial-ranges
;;;        slot of the QDE because we have no guarantee that this value
;;;        jibes with the state since the state may have been eliminated
;;;        if these values had been used.  
;;;---------------------------------------------------------------------
;;;
(defun np-get-range (var state at-lmark)
  (let ((qmag (qmag (qval var state)))
	(qspace (alookup var (state-qspaces state))))

    (cond
     ;; If it's at 0 the just return 0.
     ((eq qmag *zero-lmark*)
      (list 0 0))
     (T
      (cond
       ((lmark-p qmag)
	(cond 
	  ;; If there is Q2 info for the state, then use it.
	  ((state-bindings state)
	   (np-get-q2-range var qmag state))
	
	  ;; Otherwise find an infinite interval.
	  (T
	   (case (qmag-order qmag *zero-lmark* qspace)
	     (+   '(0 +inf))
	     (-   '(-inf 0))
	     (T   (error "Qmag order says not + or - : Can't happen!"))))))

       ((null at-lmark)
	(cond
	 ((state-bindings state)
	  (list (first (np-get-q2-range var (first qmag) state))
		(second (np-get-q2-range var (second qmag) state))))
	       
	 ((null at-lmark)
	  (case (qmag-order (first qmag) *zero-lmark* qspace)
	    ((+ 0)  '(0 +inf))
	    (-      '(-inf 0))
	    (T      (error "Qmag order says not +/-/0 : Can't happen!"))))
	 
	 (T
	  NIL))))))))


(defun np-get-q2-range (var lmark state)
  (car (cdr (assoc lmark
		   (cdr (assoc var (state-bindings state)))))))



;;;---------------------------------------------------------------------
;;; Check to see if val is within range.
;;; Inputs:  val   - a number.
;;;          range - a list of (LB UB).
;;; Returns: T if val is in range and NIL otherwise.
;;;---------------------------------------------------------------------
;;;
(defun np-var-in-range (val range)
  (and (e<= (first range) val)
       (e<= val (second range))))
	



;;;; Satisfying the constraint network
;;;;

;;;---------------------------------------------------------------------
;;; Satisfy a set of constraints.
;;; Inputs:  *npCons*  - A list of npCon structures.
;;; Returns: COMPLETE or INCOMPLETE.
;;;          It also updates *npCons* (a special variable) to retain any
;;;          incomplete constraints.
;;;---------------------------------------------------------------------
;;;
(defun np-satisfy-constraints (*npCons*)
  (declare (special *npCons*))
  (if (null *npCons*)
      'complete
    (loop with done = nil
	  while (null done)
	  do (loop for con in *npCons*
		   with changes = nil
		   do (let ((constatus (np-solve-constraint con)))
			(case constatus
			  (FAILED     (error "Constraint ~A failed.~%Final ~
                                                        values were ~a"
					     (npCon-name con)
					     (npCon-vals con)))
			  (SATISFIED  (setf *npCons* (delete con *npCons*))
				      (format t "~%Removing ~a"
					      (npcon-name con))
				      (setf changes T))
			  ))
		   finally  (cond
			     ((null *npCons*) (setf done 'COMPLETE))
			     ((null changes)  (setf done 'INCOMPLETE))))
	  finally (return done))))


;;;---------------------------------------------------------------------
;;; Attempt to solve a constraint using the given vars.
;;; Returns: FAILED    - if the constraint is inconsistent.
;;;          SATISFIED - if the constraint can now be satisfied.
;;;          NIL       - otherwise.
;;;---------------------------------------------------------------------
;;;
(defun np-solve-constraint (npCon)
  (declare (special *npVars*))
  (let ((unset-vars (remove-duplicates
		     (loop for vv in (npCon-vals npCon)
			   when (null (second vv))
			   collect (first vv)))))

    ;; Try and get values for all the unset vars for this constraint
    (loop for var in unset-vars
	  with npVar = nil
	  do (when (setf npVar (find var *npVars* :key #'npVar-name))
	       (setf (second (assoc var (npCon-Vals npCon)))
		     (npVar-value npVar))))

    ;; Now check the constraint for consistency
    (np-check-constraint npCon)))


;;;---------------------------------------------------------------------
;;; Check a constraint for consistency.
;;; Returns: FAILED    - if the constraint is inconsistent.
;;;          SATISFIED - if the constraint can now be satisfied.
;;;          NIL       - otherwise.
;;; Note:  It is assumed that D/DT constraints won't show up here.
;;;---------------------------------------------------------------------
;;;
(defun np-check-constraint (npCon)
  (declare (special *np-constraints*))
  (let ((checker (alookup (first (npCon-name npCon))
			  *np-constraints*)))
    (if checker
	(funcall checker npCon)
	(error "Don't know about the constraint ~a" (npCon-name npCon)))))



(defun np-add-check (npCon)
  (declare (special *npVars*))
  (let ((A (second (first (npCon-Vals npCon))))
	(B (second (second (npCon-Vals npCon))))
	(C (second (third (npCon-Vals npCon)))))
    (cond
     ((and A B C)
      (if (= (+ A B) C) 'SATISFIED 'FAILED))
     ((or (and (null A) (null B))
	  (and (null A) (null C))
	  (and (null B) (null C)))
      nil)
     (T
      (cond
       ((and A B)
	(push (make-npVar :name (first (third (npCon-vals npCon)))
			  :value (+ A B))
	      *npVars*))
       ((and A C)
	(push (make-npVar :name (first (second (npCon-vals npCon)))
			  :value (- C A))
	      *npVars*))
       ((and B C)
	(push (make-npVar :name (first (first (npCon-vals npCon)))
			  :value (- C B))
	      *npVars*)))
      'SATISFIED))))


(defun np-mult-check (npCon)
  (declare (special *npVars*))
  (let ((A (second (first (npCon-Vals npCon))))
	(B (second (second (npCon-Vals npCon))))
	(C (second (third (npCon-Vals npCon)))))
    (cond
     ((and A B C)
      (if (= (* A B) C) 'SATISFIED 'FAILED))
     ((or (and (null A) (null B))
	  (and (null A) (null C))
	  (and (null B) (null C)))
      nil)
     ((and A B)
      (push (make-npVar :name (first (third (npCon-vals npCon)))
			:value (* A B))
	    *npVars*)
      'SATISFIED)
     ((and A C)
      (if (or (and (= a 0) (not (= c 0)))
	      (and (= c 0) (not (= a 0))))
	  'FAILED
	  (let ()
	    (push (make-npVar :name (first (second (npCon-vals npCon)))
			      :value (if (and (zerop a) (zerop c))
					 0 (/ C A)))
		  *npVars*)
	    'SATISFIED)))
     ((and B C)
      (if (or (and (= b 0) (not (= c 0)))
	      (and (= c 0) (not (= b 0))))
	  'FAILED
	  (let ()
	    (push (make-npVar :name (first (first (npCon-vals npCon)))
			      :value (if (and (zerop b) (zerop c))
					 0 (/ C B)))
		  *npVars*)
	    'SATISFIED))))))


(defun np-sum-zero-check (npCon)
  (declare (special *npVars*))
  (let* ((vals (mapcar #'second (npCon-vals npCon)))
	 (len (length vals))
	 (num-filled (loop for i in vals count i)))
    (cond
     ((= num-filled len)
      (if (zerop (apply #'+ vals)) 'SATISFIED 'FAILED))
     ((< (1+ num-filled) len)
      nil)
     (T
      (let ((emptyvar (find-if #'(lambda (x) (null (second x)))
			       (npCon-Vals npCon))))
	(push (make-npVar :name emptyvar
			  :value (- (apply #'+ (delete nil vals))))
	      *npVars*)
	'SATISFIED)))))

 
(defun np-minus-check (npCon)
  (declare (special *npVars*))
  (let ((A (second (first (npCon-Vals npCon))))
	(B (second (second (npCon-Vals npCon)))))
    (cond
     ((and A B)
      (if (= (- A) B) 'SATISFIED 'FAILED))
     ((and (null A) (null B))
      nil)
     (A
      (push (make-npVar :name (first (second (npCon-Vals npCon)))
			:value (- A))
	    *npVars*)
      'SATISFIED)
     (B
      (push (make-npVar :name (first (second (npCon-Vals npCon)))
			:value (- B))
	    *npVars*)
      'SATISFIED))))


(defun np-zero-std-check (npCon)
  (declare (special *npVars*))
  (let ((A (second (first (npCon-Vals npCon)))))
    (cond
     (A
      (if (zerop A) 'SATISFIED 'FAILED))
     (T
      (push (make-npVar :name (first (first (npCon-Vals npCon)))
			:value 0)
	    *npVars*)
      'SATISFIED))))


;;; The use of the values slot is non-standard here.
;;; The list is ((A <NIL or number>) (LMARK nil))
;;; Note:  It is assumed that the only CONSTANT constraints that
;;;        we see have the LMARK argument.
;;;
(defun np-constant-check (npCon)
  (declare (special *npVars* *expected-values* *qde* *state*))
  (let* ((A      (second (first (npCon-Vals npCon))))
	 (Blmark (second (second (npCon-Vals npCon))))
	 (B      (find-expected-value-for-lmarkname 
		  (first (npCon-Vals npCon)) Blmark
		  *qde* *state*
		  (list :expected :lb :ub) *expected-values*)))
    (cond
     ((and A B)
      (if (= A B) 'SATISFIED 'FAILED))
     (T
      (push (make-npVar :name (first (first (npCon-Vals npCon)))
			:value B)
	    *npVars*)
      'SATISFIED))))


(defun np-M-check (npCon)
  (declare (special *npVars* *function-envelopes* *qde*))
  (let* ((A                 (second (first (npCon-Vals npCon))))
	 (B                 (second (second (npCon-Vals npCon))))
	 (function-accessor (if *function-envelopes*
				(if (eq *function-envelopes* :LB)
				    'lower-envelope
				    'upper-envelope)
			        'expected-function))
	 (inverse-accessor  (if *function-envelopes*
				(if (eq *function-envelopes* :LB)
				    'lower-inverse
				    'upper-inverse)
			        'expected-inverse))
	 (envs              (lookup-set (npCon-name npCon)
					(qde-m-envelopes *qde*) :test #'equal))
	 (F                 (if envs (lookup function-accessor envs)))
	 (Finv              (if envs (lookup inverse-accessor envs))))
    (cond
     ((and A B)
      (if (= (funcall F A) B) 'SATISFIED 'FAILED))
     ((and (null A) (null B))
      nil)
     (A
      (push (make-npVar :name (first (second (npCon-Vals npCon)))
			:value (funcall F A))
	    *npVars*)
      'SATISFIED)
     (B
      (push (make-npVar :name (first (first (npCon-Vals npCon)))
			:value (funcall Finv B))
	    *npVars*)
      'SATISFIED))))

  
(defparameter *np-constraints*
  `((ADD      . ,#'np-add-check)
    (MULT     . ,#'np-mult-check)
    (SUM-ZERO . ,#'np-sum-zero-check)
    (ZERO-STD . ,#'np-zero-std-check)
    (CONSTANT . ,#'np-constant-check)
    (MINUS    . ,#'np-minus-check)
    (M+       . ,#'np-M-check)
    (M-       . ,#'np-M-check)
;    (S+       . ,#'np-S+-check)
;    (S-       . ,#'np-S--check)
    ))



;;;-----------------------------------------------------------------------
;;; Try and find a value for a variable at a landmark.
;;; Inputs:  var          - The name of a var.
;;;          lmark        - A lmark name (unfortunate, but this might be user-entered).
;;;          qde          - A qde.
;;;          state        - A (possibly nil) state.
;;;          accessor     - A list (or atom) of :lb, :ub, :expected, or NIL that
;;;                         determines where lmark values come from.
;;;          expected-values - A list from which the :expected option looks
;;;                         for lmark values.
;;; Returns: A numeric value for the var or NIL.
;;;-----------------------------------------------------------------------
;;;
(defun find-expected-value-for-lmarkname (var lmark qde state
					      accessor expected-values)
  (if (eq lmark 0)
      ;; This is a little dangerous.  The name for *ZERO-LMARK* is the fixnum 0.
      ;; This could change on someone's whim.
      0
      (loop for access-type in (cond ((null accessor) (list :expected))
				     ((atom accessor) (list accessor))
				     (T               accessor))
	    with value = nil
	    while (null value)
	    do (case access-type
		 (:expected
		  (setf value (lookup `(,var ,lmark)
				      expected-values :test #'equal)))
		 ((:lb :ub)
		  (setf value 
			(cond
			 ((and state (state-bindings state)
			       (np-get-range var state T)))
			 (T
			  (lookup `(,var ,lmark) (qde-initial-ranges qde)
				  :test #'equal))))
		  (when value
		    (setf value (if (eq access-type :lb)
				    (first value)
				    (second value)))
		    (if (member value '(+inf -inf))
			(setf value nil)))))
	    finally (return value))))


;;;-----------------------------------------------------------------------
;;; Try and find a value for a variable at a qmag.
;;; Inputs:  var          - The name of a var.
;;;          qmag         - A qmag.
;;;          qde          - A qde.
;;;          state        - A  state.
;;;          accessor     - A list (or atom) of :lb, :ub, :expected, or NIL that
;;;                         determines where lmark values come from.
;;;          expected-values - A list from which the :expected option looks
;;;                         for lmark values.
;;; Returns: A numeric value for the var or NIL.
;;; Note:    This could return +/-inf.
;;;-----------------------------------------------------------------------
;;;
(defun find-expected-value-for-qmag (var qmag qde state
					 accessor expected-values)
  (if (not (lmark-p qmag))
      nil
      (find-expected-value-for-lmarkname var (lmark-name qmag)
					 qde state
					 accessor expected-values)))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; 2. Support functions for simulation.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;-----------------------------------------------------------------------
;;; Initialize a numsim simulation.  
;;; Inputs:  qde             - A qde.
;;;          initial-values  - A list of initial value entries
;;;          simvars         - A list of vars to simulate or :all.
;;;          function-envelopes - One of :lb, :ub, :expected, or NIL.
;;;          use-state       - A reference state (or NIL).
;;;          match-state     - T or NIL.
;;;          expected-values - A list of expected values for landmarks
;;;          start           - Simulation start time.
;;;          simStep         - Simulation time step.
;;;          saveResults     - T or NIL.
;;;          ns              - An optional numsim struct.
;;;          method          - Simulation function.
;;;          fixedstep       - T or NIL.
;;; Returns: An initialized numsim struct complete with sv and ov set for
;;;          t0.
;;;-----------------------------------------------------------------------
;;;
(defun setup-numsim (qde initial-values
			 simvars function-envelopes
			 use-state match-state
			 expected-values
			 &key
			 (start 0) (simStep 0.1)
			 (saveResults T) (ns (make-numsim))
			 (method #'rk45-run) (fixedstep nil))
  (let ((*qde* qde)
	(*function-accessor* (case function-envelopes
			       (:lb             'lower-envelope)
			       (:ub             'upper-envelope)
			       ((:expected NIL) 'expected-function)
			       (T    (error "Unknown envelope designatior ~a" 
					    function-envelopes))))
	(*inverse-accessor* (case function-envelopes
			       (:lb             'lower-inverse)
			       (:ub             'upper-inverse)
			       ((:expected NIL) 'expected-inverse)
			       (T    (error "Unknown envelope designatior ~a" 
					    function-envelopes))))
	value-list
	svars cvars ovars cons)
    (declare (special *qde* *function-accessor* *inverse-accessor*))

    ;; Setup sim parameters.
    (setf (numsim-start ns) start)
    (setf (numsim-simStep ns) simStep)
    (setf (numsim-method ns) method)
    (setf (numsim-fixedstep ns) fixedStep)

    ;; Partition the variables.
    (multiple-value-setq (svars ovars cvars cons)
      (partition-qde *qde*))
    (setf (numsim-svars ns) svars)
    (setf simvars
	  (if (eq simvars :all)
	      ovars
	      (mapcan #'(lambda (s)
			  (cond
			    ((member s svars) NIL)
			    ((member s cvars) NIL)
			    ((member s ovars) (list s))
			    (T
			     (error "~a in :simvars is not a QDE variable"
				    s))))
		      simvars)))
    (setf (numsim-ovars ns) simvars)
    (setf (numsim-cvars ns) cvars)
    (setf (numsim-sv ns) (make-array (list (length svars))))
    (setf (numsim-ov ns) (if simvars (make-array (list (length simvars)))
			             NIL))
    (setf (numsim-cv ns) (if cvars (make-array (list (length cvars)))
			           NIL))

    ;; Determine values for all the state variables and constants
    ;; based on the user-supplied information.
    (setf value-list (numsim-propagate qde initial-values function-envelopes
				       use-state match-state
				       expected-values
				       svars cvars ovars))

    ;; Generate (and compile) lisp functions for the state and other vars.
    (multiple-value-bind (state-fctn other-fctn)
	(create-state-and-other-functions cons svars ovars cvars simvars)
      (setf (numsim-stateFctn ns) state-fctn)
      (setf (numsim-otherFctn ns) other-fctn))

    ;; Initialize the rk-struct.
    (setf (numsim-rk ns) (rk45-init (numsim-stateFctn ns) (length svars)
				    *rk-relerr* *rk-abserr*
				    start 1 (numsim-cv ns)))

    ;; Initialize the result structure.
    (when saveResults
      (setf (numsim-results ns) (init-numsim-results svars simvars)))
    
    ;; Setup the initial state, constant, and "other" vectors.
    (loop for c in cvars
	  for ci from 0 upto (1- (length cvars))
	  do (setf (svref (numsim-cv ns) ci)
		   (lookup c value-list)))
    (loop for s in svars
	  for si from 0 upto (1- (length svars))
	  do (setf (svref (numsim-sv ns) si)
		   (lookup s value-list)))
    (numsim-compute-other-vars ns)
    ns))


;;;-----------------------------------------------------------------------
;;; Do a numeric simulation for some period of time.
;;; Inputs:  ns        - A numsim struct.
;;;          new-time  - The time to simulate until.
;;; Returns: Nothing.  In fact, this function doesn't store anything in the
;;;          numsim.results slot either.
;;; Note:  Numsim.fixedStep is only needed for numerical methods that use a
;;;        fixed timestep.  If the method doesn't need it (as rk45 doesn't)
;;;        it isn't used and new-time is all that is needed.
;;;-----------------------------------------------------------------------
;;;
(defun numsim-until (ns new-time)
  (let ((method (numsim-method ns)))
    (when (> new-time (numsim-time ns))
      (cond ((numsim-fixedstep ns)
	     (let* ((time      (numsim-time ns))
		    (fixedStep   (numsim-fixedStep ns))
		    (limit     (floor (/ (- new-time time) fixedStep))))
	       ;; Break up the simulation into intervals no bigger than fixedStep.
	       (dotimes (k limit)
		 (setq time (+ time fixedStep))
		 ;; compute the new system values
		 (funcall method (numsim-sv ns) (numsim-rk ns) time)
		   
		 ;; Compute the new other variables
		 (numsim-compute-other-vars ns))

	       ;; Simulate out the last fractional part.
	       (when (> new-time time)
		 (funcall method (numsim-sv ns) (numsim-rk ns) new-time)
		 (numsim-compute-other-vars ns))))
	    (T
	     (funcall method (numsim-sv ns) (numsim-rk ns) new-time)
	     (numsim-compute-other-vars ns)))
      (setf (numsim-time ns) new-time))))



;;;-----------------------------------------------------------------------
;;; See if a stop condition is met by the data.
;;; Inputs:  ns        - A numsim structure.
;;;          condition - A list of (op var number) entries.
;;; Returns: T if the condition is met and NIL otherwise.
;;;-----------------------------------------------------------------------
;;;
(defun stop-condition-met (ns condition)
  (let ((varpos NIL))
    (dolist (cond condition nil)
      (cond
	((setf varpos (position (second cond) (numsim-svars ns)))
	 (if (funcall (first cond) (svref (numsim-sv ns) varpos) (third cond))
	     (return T)))
	((setf varpos (position (second cond) (numsim-ovars ns)))
	 (if (funcall (first cond) (svref (numsim-ov ns) varpos) (third cond))
	     (return T)))
	(T
	 NIL)))))


;;;-----------------------------------------------------------------------
;;; Setup the numsim.results structure.
;;; This structure is a list of (svar-results ovar-results) where each
;;; entry of the results sublists is a queue.
;;;-----------------------------------------------------------------------
;;;
(defun init-numsim-results (svars ovars)
  (mapcar #'(lambda (varlist)
	      (mapcar #'(lambda (var) (declare (ignore var))
			  (make-queue)) varlist))
	      (list svars ovars)))


;;;-----------------------------------------------------------------------
;;; Store results in the results structure (actually it's a list).
;;; Inputs:  sv      - A state vector.
;;;          ov      - An "other" vector.
;;;          time    - The simulation time.
;;;          results - The numsim.results slot.
;;; Returns: Nothing.
;;;-----------------------------------------------------------------------
;;;
(defun store-results (sv ov time results)
  (mapc #'(lambda (varlist vector)
	    (dotimes (pos (length vector))
	      (qpush (list time (svref vector pos)) (nth pos varlist))))
	results (list sv ov)))


;;;-----------------------------------------------------------------------
;;; Compute the values of the state vars given the values of the
;;; state and constant vectors.
;;; Inputs:  ns - The numsim struct.
;;; Returns: Nothing.
;;; Notes:   The called functions args are (tin sv sv' cv) where tin is
;;;          ignored and sv' gets the result of the function application.
;;;-----------------------------------------------------------------------
;;;
(defun numsim-compute-other-vars (ns)
  (when (numsim-ovars ns)
    (funcall (numsim-otherFctn ns) NIL (numsim-sv ns) (numsim-ov ns) (numsim-cv ns))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; 3. Top-level numsim call.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;-----------------------------------------------------------------------
;;; Top-level simulation call.
;;; Given a qde (or state), an ODE system is extracted by using the 
;;; expected-values and expected-functions/inverses of the QDE.  This information,
;;; together with user-specified initial conditions will then be used to run 
;;; the simulation.
;;; In addition, if enough information is given to extract meaningful transition
;;; rules, the simulation will faithfully pass between transition regions.
;;;
;;; New clauses and their uses :
;;;
;;; 1. expected-values : This clause is similar to the initial-values clause.
;;;    It is a top-level qde clause.  Its format is 
;;;
;;;          (expected-values <vlv> ... )   
;;;
;;;    where <vlv> = ((<varname> <lmarkname>) <number>)
;;;    where <number> is any expression that evals to a number.  This clause
;;;    gives a value to a landmark of a variable for the numsim simulation.
;;;    Normally, all constants and state variables should have expected-values 
;;;    entries for their values at t=0.
;;;
;;; 2. expected-function, expected-inverse : These clauses are similar to the
;;;    upper-envelope and upper-inverse clauses associated with monotonic 
;;;    constraints.  They appear as subclauses of the M-envelopes qde clause.
;;;    For each monotonic constraint, both an expected-function and an
;;;    expected-inverse function are required.  [Is this really necessay?
;;;    Can it be shown that only one is needed?]
;;;
;;; Inputs:  state-or-qde  - A qsim state or qde.
;;;          start         - Simulation start time.
;;;          stop          - Simulation stop time.
;;;          simStep       - Simulation step size.
;;;          simvars       - A list of variables to simulate, or :all for all vars.
;;;                          (note that the state vars are simulated even if
;;;                           simvars = NIL)
;;;          function-envelopes - One of :lb, :ub, :expected, or NIL.  :ub or :lb
;;;                          select the lower or upper envelope clauses from
;;;                          the qde m-envelopes clause.  NIL or :expected
;;;                          uses the expected clauses of m-envelopes.
;;;          use-state     - If this keyword is set to a state then this state is
;;;                          the state referred to in the match-state argument and
;;;                          the :from-state keyword in the intial-values argument.
;;;          match-state   - If T, then the numsim initial state must be in the state
;;;                          space defined by use-state. 
;;;          initial-values - A list of (<varname> <value> {<accessor>}*) entries
;;;                          which specify initial values for variables.  
;;;                          Allowable forms are :
;;;                             (<varname> <number>)
;;;                             (<varname> <lmarkname> <accessor>)
;;;                             (<varname> :from-state <accessor>)
;;;                          The <accessor> arg determines how numeric values for
;;;                          landmarks are located.  In general, this is a list of
;;;                          :lb, :ub, :expected, and :none keywords.  These are
;;;                          processed in list order to find the numeric value.
;;;                          :Lb and :ub look in the Q2 ranges
;;;                          for the value and select the lower or upper bounds
;;;                          (see the note below for further info on HOW it searches).
;;;                          :Expected looks in the expected-values list (see below).
;;;                          :None prevents a lisp error when no value is found for
;;;                          the variable.
;;;                          :From-state uses the lmark that the variable is at in
;;;                          use-state to determine the value.  It is an error if the
;;;                          variable is not at a landmark or if no value can be found
;;;                          using <accessor>.
;;;                          There are also a set of default arguments.  They are of
;;;                          the form (<default> <value> <accessor>) where <default>
;;;                          is one of :state, :constant, :derivative, or :default.
;;;                          These forms will attempt to set every variable of
;;;                          the corresponding type in the model
;;;                          based on <value> and <accessor>, but if it can't, it will
;;;                          simply leave it alone.  They are always performed before
;;;                          specific variable initializations, and the specific variable
;;;                          initializations can override them.  This option is useful
;;;                          for setting commonly set groups of variables automatically.
;;;                          Finally, using the form
;;;                            (<varname> NIL)
;;;                          will unset the value of variable.  This is also useful
;;;                          in combination with :default.
;;;          expected-values - A list of ((var lmark) val) that is logically appended
;;;                          to the front of the QDE expected-values clause.  This
;;;                          lets you specify expected values for landmarks without
;;;                          redefining the QDE.
;;;          saveResults   - T if data should be stored.  NIL otherwise.
;;;          method        - Simualtion method (actually, the name of the sim fctn).
;;;          fixedStep     - A stepsize if this is a fixed step size method.  Otherwise NIL.
;;;          stop-condition - A list of (op var number) entries.  If var op number is
;;;                           true for any entry then simulation stops.
;;; Returns: A numsim structure or NIL.  Also, if numsim was called with a state,
;;;          the numsim structure will be set to hang off of it at the state.numsim-struct
;;;          slot.
;;; Usage Notes:  
;;;          - To create an initial state, numsim needs values for the state 
;;;            variables and constants of the model.  These values are supplied
;;;            through the :initial-values arg.  You need only specify enough
;;;            values to guarantee that the state variables and constants have values.
;;;            Numsim will propagate whatever you give it to try and complete 
;;;            the initial state.  
;;;          - While the above process specifies an initial state suitable for
;;;            simulation, the system may not in fact be a member of the QDE family.
;;;            This is beacuse the QDE can also constrain the possible initial state
;;;            via corresponding values and initial ranges on parameters.  To
;;;            guarantee that the numerical model matches the QDE, it is important
;;;            to match it against a QSIM initial state.  This is done by using the
;;;            :use-state and :match-state args which will ensure that the initial
;;;            state is consistent with a specific QSIM initial state.
;;;            Guaranteeing that a numsim state matches a QSIM state is in general
;;;            an optimization problem, and numsim does not have an optimizer.  It
;;;            is therefore up to the user to solve this problem (by varying
;;;            the :initial-values clauses) until a suitable initial state is found.
;;;          - A strategy that seems to work is to start by defaulting the state,
;;;            constant, and derivative variables in the model and then to determine
;;;            where a confict occurs and redefine some variables to fix it by
;;;            overriding a default value.
;;;          - A note on how :ub and :lb work in the initial-values argument ::
;;;            These keywords look for range info on the landmark of a variable.
;;;            They find the range as follows :
;;;                  1) If the lmark is 0, then 0 is returned.
;;;                  2) Failing that, if use-state has been run through Q2, then the
;;;                     state-bindings slot is used.
;;;                  3) Failing that, the intitial-ranges clause of the QDE is used.
;;;                  4) Otherwise, no value is found.
;;;            Note that if Q2 has not been run, the initial-ranges may not match
;;;            the QSIM state.
;;;          - It is usually a mistake to use the :default initialization keyword
;;;            because it sets too many variables.
;;;          - For the purposes of default initializations a state variable is the
;;;            first argument of a D/DT constraint, a derivative variable is the
;;;            second argument of a D/DT constraint, and a cosntant is a variable
;;;            mentioned in a CONSTANT or ZERO-STD constraint or the QDE
;;;            INDEPENDENT clause.
;;;-----------------------------------------------------------------------
;;;
(defun numsim (state-or-qde &key (start 0) (stop 10) (simStep .1)
			    (simvars :all) (function-envelopes NIL)
			    (use-state NIL) (match-state NIL) 
			    (initial-values NIL) (expected-values nil)
			    (saveResults T) (method #'rk45-run) (fixedStep NIL) 
			    (stop-condition NIL))
  (let* ((qde (if (state-p state-or-qde) (state-qde state-or-qde) state-or-qde))
	 ns)
    (when (null qde)
      (error "NUMSIM: No QDE specified"))

    ;; Setup the simulation structure.
    (setf ns (setup-numsim qde initial-values
			   simvars function-envelopes
			   use-state match-state
			   expected-values
			   :start start :simStep simStep
			   :saveResults saveResults
			   :method method :fixedstep fixedstep))

    ;; Store the initial results.
    (store-results (numsim-sv ns) (numsim-ov ns) start (numsim-results ns))
  
    (do ((time start (+ time simStep)))
	((>= time stop) NIL)
      (numsim-until ns time)
      (store-results (numsim-sv ns) (numsim-ov ns) time (numsim-results ns))
      (when (and stop-condition (stop-condition-met ns stop-condition))
	(return NIL)))
    (when (state-p state-or-qde)
      (setf (state-numsim-struct state-or-qde) ns))
    ns))

