;;; -*- Mode:Common-Lisp; Package: Qsim; Syntax:Common-Lisp; Base:10 -*-

;;; $Id: numeric-eqn-gen.lisp,v 1.2 1992/03/26 18:34:52 bert Exp $

(in-package 'qsim)

;;;======================================================================
;;; Equation generation from a QDE
;;;
;;; A structure called an ExprTableEntry is defined to maintain info
;;; for translating a QDE into a pair of lisp functions : one to
;;; evaluate the derivative of the state variables of the system and
;;; one to evaluate the values of the "other" variables of the system.
;;;
;;; The process consists of "unwrapping" each constraint to form a
;;; set of subexpressions that describe each model variable (e.g.
;;; (ADD A B C) -> B = C - A, among other things).  These unwrapped
;;; expressions are then combined into full expressions consisting only
;;; of state variables and constants.  Finally, substitutions are made
;;; to replace these variable types with actual constants and vector
;;; references.
;;;======================================================================


(defparameter *trace-function-gen* nil)   ; Display the lisp functions as they are compiled
(defparameter *trace-exprtable-gen* nil)  ; Display the exprTable.

;;; An ExprTableEntry keeps track of the possible expansions for a variable.
;;; variables are either STATE, CONSTANT, or OTHER (not STATE or CONSTANT).
;;; The expansion for a state variable is its derivative and the expansion
;;; for an other variable is its value.  Constants expand to themselves.
;;;
(defstruct (ExprTableEntry (:print-function exprTableEntry-printer))
  "Variable Expression generator structure"
  (varname     nil)   ; name of the variable
  (vtype       nil)   ; STATE OTHER CONSTANT
  (subexprList nil)   ; list of possible expressions based on single constraints
  (expr        nil)   ; expression based on state vars and constants
  (subexpr     nil)   ; the entry in subexprList that is used in expr
  )

(defun exprTableEntry-printer (ete stream ignore)
  (declare (ignore ignore) (special *detailed-printing*))
  (if *detailed-printing*
      (format stream "~%#<exprTableEntry varname = ~a   vtype = ~a~%~16T ~
              subexprlist = ~a~%~16T expr = ~a~%~16T subexpr = ~a>"
	      (exprTableEntry-varname ete) (exprTableEntry-vtype ete)
	      (exprTableEntry-subexprList ete) (exprTableEntry-expr ete)
	      (exprTableEntry-subexpr ete))
      (format stream "#<exprTableEntry ~a>" (exprTableEntry-varname ete))))


;;;-----------------------------------------------------------------------
;;; Partition the variables in a QDE into state, constant, and other vars.
;;; Remove those constraints (and vars) that won't help in equation
;;; generation.
;;; Inputs:  qde - A qde.
;;; Returns: #1 - A list of the state varnames.
;;;          #2 - A list of the "other" varnames.
;;;          #3 - A list of the constant varnames.
;;;          #4 - A list of the constraints that apply (names not structs)
;;; Notes: - CONSTANT and ZERO-STD vars are considered constants.
;;;        - Discrete vars are ignored.
;;;-----------------------------------------------------------------------
;;;
(defun partition-qde (qde)
  (let* ((cvars (remove-duplicates
		 (append (qde-independent qde)
			 (mapcar #'second
				 (append (find-qde-cons-by-name 'constant qde)
					 (find-qde-cons-by-name 'zero-std qde))))))
	 (svars (remove-duplicates
		 (mapcar #'second (find-qde-cons-by-name 'd/dt qde))))
	 (ovars (remove-if #'(lambda (x) (or (member x svars)
					     (member x cvars)
					     (eq x 'time)))
			   (mapcar #'variable-name 
				   (remove-if #'(lambda (x) (variable-discrete-p x))
					      (qde-variables qde)))))
	 (cons  (mapcan #'(lambda (x)
			    (if (and (constraint-active-p x)
				     (not (member (car (constraint-name x))
						  '(constant zero-std))))
				(list (constraint-name x))
			        NIL))
			(qde-constraints qde))))
    (values svars ovars cvars cons)))


;;;-----------------------------------------------------------------------
;;; Find all the "op" constraints in a qde.
;;; Inputs:  op  - The name of a constraint (e.g., d/dt, mult, etc.).
;;;          qde - A qde.
;;; Returns: A list of constraints (as sexprs, not structs) of type "op"
;;;          or NIL.
;;;-----------------------------------------------------------------------
;;;
(defun find-qde-cons-by-name (op qde)
  (find-all op (mapcan #'(lambda (x)
			   (if (constraint-active-p x)
			       (list (constraint-name x))
			       NIL))
		       (qde-constraints qde))
	    :key #'first))


;;;-----------------------------------------------------------------------
;;; Create a pair of lisp functions that will evaluate an ODE based on
;;; a QDE.
;;; Inputs:  cons  - the (active) constraints of a qde (names, not structs).
;;;          svars - the state variable for the system.
;;;          ovars - the other variables for the system.
;;;          cvars - the constant variables for the system.
;;;          simvars - those ovars that will be simulated.
;;; Outputs: #1 - a function that computes y' given y (a state vector)
;;;               and c (a constant vector)
;;;          #2 - a function that computes onew given y and c
;;; Notes: It is assumed that constraints doesn't contain any implicit
;;;        "constant" constraints.  Also, vars that are ZERO-STD or
;;;        CONSTANT are assumed to have been cataloged as constants.
;;;        Note that the function partition-qde should take
;;;        care of all this.
;;;-----------------------------------------------------------------------
;;;
(defun create-state-and-other-functions (cons svars ovars cvars simvars)
  (let ((exprTable (create-exprTable cons svars ovars cvars)))
    (create-expressions exprTable)
    (when *trace-exprtable-gen*
      (let ((*detailed-printing* T))
	(format t "~%Exprtable:~%~A" exprTable)))
    (values (make-state-fctn svars cvars exprTable)
	    (make-other-fctn simvars svars cvars exprTable))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 1. Build the expression table
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;-----------------------------------------------------------------------
;;; Make an exprTable and fill it with subexprs.
;;; Inputs:  cons  - a list of constraints (actually constraint-names).
;;;          svars - a list of the state vars.
;;;          ovars - a list of the other vars.
;;;          cvars - a list of the constant vars.
;;; Returns: a list of exprTableEntrys.
;;;-----------------------------------------------------------------------
;;;
(defun create-exprTable (cons svars ovars cvars)
  (let ((table nil)
	applicable-cons) ; These are cons that mention a particular var
    ;; For state vars we are only interested in d/dt constraints.
    (dolist (var svars)
      ;; This selects those cons of the form (d/dt var dVar)
      (setf applicable-cons (remove-if #'(lambda (x) (not (eq (second x) var)))
				       (find-all 'd/dt cons :key #'first)))
      (when (> (length applicable-cons) 1)
	(error "State var ~a has more than one derivative" var))
      (pushnew (make-exprTableEntry :varname     var
				    :vtype       'state
				    :subexprList (list (third
							(first applicable-cons))))
	       table))

    ;; Constants are stored as themselves
    (dolist (var cvars)
      (pushnew (make-exprTableEntry :varname     var
				    :vtype       'constant
				    ) ; don't need the other fields
	       table))

    ;; Other vars may have multiple subexprs
    (dolist (var ovars)
      (setf applicable-cons (remove-if #'(lambda (x) (eq (car x) 'd/dt))
				       (find-cons-with-var var cons)))
      (pushnew (make-exprTableEntry :varname   var
				    :vtype     'other
				    :subexprList (mapcan #'(lambda (x)
							     (form-subexprs x
									    var))
							 applicable-cons))
	       table))
    table))
				     
								
;;;-----------------------------------------------------------------------
;;; Rearrange a constraint so that it solves for var.
;;; Inputs:  con - the constraint (actually, constraint-name).
;;;          var - the var to solve for.
;;; Returns: A list of subexpressions, solved for var.
;;; NOTE: There may be more than one way to solve for var if var
;;;       appears in more than one argument to the constraint.
;;;-----------------------------------------------------------------------
;;;
(defun form-subexprs (con var)
  (let ((subexprs nil))
    (case (first con)
      ((m+ m- s+ s-) (when (eq (third con) var)
		       (pushnew `(FUNCTION ,con ,(second con)) subexprs))
                     (when (eq (second con) var)
		       (pushnew `(INVERSE ,con ,(third con)) subexprs)))
      (ADD           (when (eq (second con) var)
		       (pushnew `(- ,(fourth con) ,(third con)) subexprs))
		     (when (eq (third con) var)
		       (pushnew `(- ,(fourth con) ,(second con)) subexprs))
		     (when (eq (fourth con) var)
		       (pushnew `(+ ,(second con) ,(third con)) subexprs)))
      (MULT          (when (eq (second con) var)
		       (pushnew `(/ ,(fourth con) ,(third con)) subexprs))
		     (when (eq (third con) var)
		       (pushnew `(/ ,(fourth con) ,(second con)) subexprs))
		     (when (eq (fourth con) var)
		       (pushnew `(* ,(second con) ,(third con)) subexprs)))
      (MINUS         (when (eq (second con) var)
		       (pushnew `(- ,(third con)) subexprs))
		     (when (eq (third con) var)
		       (pushnew `(- ,(second con)) subexprs)))
      (SUM-ZERO      ;; This gets expanded into (sum-zero otherargs)
                     ;; Note that this changes the semantics of sum-zero
                     ;; from its use as a qsim constraint.
                     (pushnew `(sum-zero ,.(remove var (cdr con)))
			      subexprs))
      (T             (format *Qsim-Report*
			     "~&FORM-SUBEXPRS: Don't know about ~a constraint ~
                                -- it is being ignored" con)))
    subexprs))


;;;-----------------------------------------------------------------------
;;; Create expressions for each variable in terms of svars and cvars.
;;; Enter the expression in the expr slot of the exprTableEntry for the var.
;;; Inputs:  exprTable - the expression table
;;; Returns: none.
;;; Updates: the expr slot of entries in the exprTable.
;;;-----------------------------------------------------------------------
;;;
(defun create-expressions (exprTable)
  (dolist (v exprTable)
    (when (null (expand-var (exprTableEntry-varname v) exprTable nil
			    :state-eqn (eq 'state (exprTableEntry-vtype v))))
      (error "CREATE-EXPRESSIONS: Couldn't create an expression for ~a"
	     (exprTableEntry-varname v)))))


;;;-----------------------------------------------------------------------
;;; Expand a variable into an expression.
;;; Inputs:  var       - the name of the variable to expand.
;;;          exprTable - the expression table.
;;;          path      - a list of vars we are already expanding
;;;                      (this prevents recursive expansions).
;;;          state-eqn - T if we want a state equation (derivative) for this
;;;                      var.  NIL otherwise.
;;; Returns: the expansion for the variable or NIL if none is found.
;;; Updates: the expr slot of entries in the exprTable.
;;;-----------------------------------------------------------------------
;;;
(defun expand-var (var exprTable path &key (state-eqn NIL))
  (let ((v (find var exprTable :key #'exprTableEntry-varname)))
    (cond
      ;; Constants and state variables (when we aren't interested in their
      ;; deriv) return themselves.
      ((eq (exprTableEntry-vtype v) 'constant)
       var)
      ((and (eq (exprTableEntry-vtype v) 'state)
	    (null state-eqn))
       var)

      ;; Vars in path are recursive calls and therefore fail to expand.
      ((member var path)
       NIL)

      ;; Otherwise, keep trying subexprs for var until one expands
      (T
       (do ((subexprs (exprTableEntry-subexprList v) (cdr subexprs))
	    (new-path (cons var path)))
	   ((or (null subexprs) (exprTableEntry-expr v))
	    ;; If there was no expansion then this result will be NIL.
	    ;; Otherwise, it will be a valid expression for the var.
	    (exprTableEntry-expr v))
	 (setf (exprTableEntry-expr v)
	       (expand-subexpr (car subexprs) exprTable new-path))
	 ;; If there was an expansion, store the unexpanded subexprList
	 ;; entry that succeeded.
	 (when (exprTableEntry-expr v)
	   (setf (exprTableEntry-subexpr v) (car subexprs)))
	 )))))


;;;-----------------------------------------------------------------------
;;; Expand a subexpression into an expression.
;;; Inputs:  subexpr   - a subexpression.
;;;          exprTable - an exprTable.
;;;          path      - a list of vars being expanded.
;;; Returns: the expanded subexpr or NIL.
;;;-----------------------------------------------------------------------
;;;
(defun expand-subexpr (subexpr exprTable path)
  (let ((expr NIL))
    (cond
      ;; Variables require an expand-var call.
      ((atom subexpr)
       (expand-var subexpr exprTable path))

      ;; M and S functions (stored as (FUNCTION/INVERSE constraint var))
      ;; expand the var argument.
      ((member (first subexpr) '(FUNCTION INVERSE))
       (setf expr (expand-var (third subexpr) exprTable path))
       (if expr
	   `(,(first subexpr) ,(second subexpr) ,expr)
	   NIL))

      ;; Otherwise expand the arguments and return the result.
      (T
       (do ((vars (cdr subexpr) (cdr vars))
	    (result (list (first subexpr))))
	   ((or (null vars)
		(null (setf expr (expand-var (car vars) exprTable path))))
	    (if (null vars)
		(nreverse result)
		NIL))
	 (push expr result))))))
    


;;;-----------------------------------------------------------------------
;;; Find all instances of item in seq.
;;; Inputs: item - an item (normally an atom).
;;;         seq  - a sequence.
;;;Returns: all entries in seq that match item.
;;;-----------------------------------------------------------------------
;;;
(defun find-all (item seq &key (test #'eql) (key #'identity))
  (mapcan #'(lambda (x) (if (funcall test item (funcall key x))
			    (list x)
			    NIL))
	  seq))
	  

;;;-----------------------------------------------------------------------
;;; Find the constraints that contain var.
;;; Inputs:  var  - A var name.
;;;          cons - A list of constraints (as sexprs, not structs).
;;; Returns: Those elements of cons that mention var.
;;;-----------------------------------------------------------------------
;;;
(defun find-cons-with-var (var cons)
  (mapcan #'(lambda (con)
	      (if (member var (cdr con))
		  (list con)
		  NIL))
	  cons))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 2.  Build an executable lisp expression for state vars and other vars.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;-----------------------------------------------------------------------
;;; Make a function that computes the derivatives of all state vars.
;;; The form of the function is
;;;      (lambda (tin y yp c)
;;;        (declare (ignore tin))
;;;        (setf (svref yp <pos>) <expression>)
;;;         ... )
;;; Inputs:  svars     - the state vars.
;;;          cvars     - the constant vars.
;;;          exprTable - the exprTable (with exprs slots filled in).
;;; Returns: a compiled function as described above.
;;;-----------------------------------------------------------------------
;;;
(defun make-state-fctn (svars cvars exprTable)
  (let* ((ign-decl (if (null cvars) '(declare (ignore tin c))
		                    '(declare (ignore tin))))
	 (fctn
	   `(lambda (tin y yp c)
	      ,ign-decl
	      ,.(mapcar #'(lambda (var)
			    `(setf (svref yp ,(position var svars))
				   ,(build-lisp-expression var exprTable
							   svars cvars
							   :state-eqn T)))
			svars)))
	 (fname (gentemp "F")))
    (when *trace-function-gen*
      (format t "~%Creating function ~a" fname)
      (pprint fctn))
    (compile fname fctn)))


;;;-----------------------------------------------------------------------
;;; Make a function that computes the values of all other vars.
;;; The form of the function is
;;;      (lambda (tiny onew c)
;;;        (declare (ignore tin))
;;;        (setf (svref onew <pos>) <expression>)
;;;         ... )
;;; Inputs:  ovars     - the other vars to be simulated.
;;;          svars     - the state vars.
;;;          cvars     - the constant vars.
;;;          exprTable - the exprTable (with exprs slots filled in).
;;; Returns: A compiled function as described above or NIL if no ovars.
;;;-----------------------------------------------------------------------
;;;
(defun make-other-fctn (ovars svars cvars exprTable)
  (cond
    (ovars
     (let* ((ign-decl (if (null cvars) '(declare (ignore tin c))
			               '(declare (ignore tin))))
	    (fctn
	      `(lambda (tin y onew c)
		 ,ign-decl
		 ,.(mapcar #'(lambda (var)
			       `(setf (svref onew ,(position var ovars))
				      ,(build-lisp-expression var exprTable
							      svars cvars)))
			   ovars)))
	    (fname (gentemp "F")))
       (when *trace-function-gen*
	 (format t "~%Creating function ~a" fname)
	 (pprint fctn))
       (compile fname fctn)))
    (T
     (when *trace-function-gen*
	 (format t "~%No 'other' vars -- not compiling a function"))
     nil)))

;;;-----------------------------------------------------------------------
;;; Substitute in vector references and function envelopes into an
;;; expression.
;;; Inputs:  expr      - the expr to build the expression for.
;;;          exprTable - the exprTable.
;;;          svars     - the state vars.
;;;          cvars     - the constant vars.
;;;          state-eqn - T when this is called with a svar as expr and we
;;;                      want to construct a state (derivative) expr.
;;; Returns: the fully expanded lisp expression.
;;;-----------------------------------------------------------------------
;;;
(defun build-lisp-expression (expr exprTable svars cvars &key (state-eqn nil))
  (cond
    ((and (atom expr) (member expr svars) (null state-eqn))
     `(svref y ,(position expr svars)))
    ((and (atom expr) (member expr cvars))
     `(svref c ,(position expr cvars)))
    ((atom expr)
     (build-lisp-expression (exprTableEntry-expr
			     (find expr exprTable :key #'exprTableEntry-varname))
			    exprTable svars cvars))
    ((member (first expr) '(FUNCTION INVERSE))
     (build-lisp-expression-for-envelope expr exprTable svars cvars))
    (T
     `(,(first expr) ,.(mapcar #'(lambda (arg)
				   (build-lisp-expression arg exprTable
							  svars cvars))
			(cdr expr))))))


;;;-----------------------------------------------------------------------
;;; Build a function expression using envelopes from *qde*.
;;; Inputs:  expr      - an expression of the form
;;;                      (FUNCTION/INVERSE constraint var).
;;;          exprTable - the exprTable.
;;;          svars     - the state vars.
;;;          cvars     - the constant vars.
;;; Returns: the fully expanded lisp expression.
;;;-----------------------------------------------------------------------
;;;
(defun build-lisp-expression-for-envelope (expr exprTable svars cvars)
  (declare (special *qde* *function-accessor* *inverse-accessor*))
  (let* ((envs (lookup-set (second expr) (qde-m-envelopes *qde*)
			   :test #'equal))
	 (env  (if envs (lookup (if (eq (first expr) 'FUNCTION)
				     *function-accessor*
				     *inverse-accessor*)
				envs))))
    (when (null env)
      (error "No envelope for ~a under slot ~a"
	     expr (if (eq (first expr) 'FUNCTION)
		      *function-accessor*
		      *inverse-accessor*)))
    `(funcall (function ,env)
      ,(build-lisp-expression (third expr) exprTable svars cvars))))




;;;-----------------------------------------------------------------------
;;; This defines the sum-zero operator for use in evaluable functions.
;;;-----------------------------------------------------------------------
;;;
(defun sum-zero (&rest args)
  (- (apply #'+ args)))

