;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: intervals.lisp,v 1.1 1991/03/26 21:37:58 clancy Exp $
;  Copyright (c) 1988, Benjamin Kuipers and Daniel Berleant.
(in-package 'QSIM)

; This is the Q2 interval range reasoner, that adds the capability of reasoning
; with incomplete quantitative knowledge (in the form of intervals) about 
; qualitative behaviors (by propagation across constraints).
;
; The code consists of several chapters:
;   1.  Top-level call to Q2 as a global filter on QSIM states.
;   2.  Generation of equations from a behavior.
;   3.  Index equations.
;   4.  Access functions for variable-landmark pairs (VLPs) and ranges.
;   5.  Propagation loop.
;   6.  Zip-up and Zip-down quantity spaces.
;   7.  RANGER* --- the quantitative range interpreter
;   8.  RANGER  --- a simpler interval-arithmetic interpreter
;   9.  Interval-arithmetic routines.
;   10.  Display and trace functions.


; Information for quantitative range reasoning is stored on QDE.other in the form
; illustrated by the following examples:
;  (other					
;    (m-envelopes
;      ((M+ amount outflow) (upper-envelope (lambda (x) x))
;                           (upper-inverse  (lambda (y) y))
;			    (lower-envelope (lambda (x) (if (< x 20) (* x .5)
;							             (+ (* x .2) 6))))
;			    (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
;							             (* 5 (- y 6)))))))
;                           (slope (0.2 1.0))
;                           (curvature (-0.1 0.1))
;    (initial-ranges ((inflow if*)  (5 10))
;		    ((amount full) (50 100))
;		    ((time t0)     (0 0))))


; Control parameters.

;      The global data structures for RANGER:
(defparameter *bindings* nil)			; current state of ranges
(defparameter *eqn-index* nil)			; index of equations
(defparameter *M-envelopes* nil)		; envelopes for monotonic functions
(defparameter *agenda* nil)			; agenda of equations to propagate
(defparameter *state* nil)			; current state

(defparameter *epsilon* 1.0E-6)			; minimum change to propagate
(defparameter infminf (list *inf-lmark* *minf-lmark*))

(defparameter *trace-Q2-check* nil)		; trace global filter use
(defparameter *trace-eqn-indexing* nil)		; trace equation index creation
(defparameter *trace-q2-agenda* nil)	       	; trace addition of equations to agenda
(defparameter *trace-consider-eqn* nil)		; show when each equation is considered
(defparameter *trace-ranger* nil)		; trace the range interpreter
(defparameter *trace-range-update* nil)		; trace changes to intervals
(defparameter *trace-single-char* t)		; single-char trace output
						
; Access macros for intervals: (lo hi).

(defmacro lo (int)
  `(car ,int))

(defmacro hi (int)
  `(cadr ,int))

; 1.  Top-level call to Q2 as a global filter on QSIM states.
;
;   It is therefore interleaved with qualitative simulation.
;   Runs on time-points (not intervals), if the switch is on, and there is
;   some useful range information to work with.
;     Normally, this is called without the range-init argument, but it can be
;     called with explicit new information.

(defun quantitative-range-reasoning (current-state &key (assert-ranges nil))
  (cond ((null *check-quantitative-ranges*) current-state)
	((not (qpointp (state-time current-state))) current-state)
	((member (car (state-status current-state))
		 '(inconsistent incomplete)) current-state)
	(t (trace-Q2-global-filter current-state)
	   (get-globals-from-state current-state)
	   (setup-equations-from-state current-state)
	   (setup-agenda)
	   (assimilate-asserted-ranges current-state assert-ranges)

	   (cond ;((null *bindings*) current-state)
		 ((propagate-ranges)
		  (store-globals-in-state current-state)
		  current-state)
		 (t (prune-inconsistent-state current-state
					      "Quantitative ranges")
		    (store-globals-in-state current-state)
		      nil)))))

(defun get-globals-from-state (state)
  (setq *state* state)
  (setq *agenda* nil)
  (setq *eqn-index* (state-eqn-index state))
  (setq *bindings* (state-bindings state))
  (setq *m-envelopes* (cdr (assoc 'M-envelopes (qde-other (state-qde state)))))
  t)

; Given a state, construct all equations and index them on *eqn-index*.
; This is logically prior to all range propagation.
;   Make equations from corresponding values only at initial and transition-result states.
;   Make equations from current values at all time-points.
;   Assert identical values of variables and landmarks with the same names across transitions.

(defun setup-equations-from-state (state)
  (if *trace-single-char* (format *QSIM-Trace* "["))
  (if (member (car (state-justification state))
	      '(initialized-with transition-from
		one-of-several-completions-of unique-completion-of))
      (make-eqns-from-cvtuples state))
  (make-eqns-from-values state)
  (let ((ostate (predecessor-of-state state)))
    (if (and ostate (eql (car (state-justification state)) 'transition-from))
	(create-equalities-across-transition ostate state)))
  (if *trace-single-char* (format *QSIM-Trace* "]"))
  t)

; This is created to assert (= (V1 LM1) (V2 LM2)) about landmarks with the same name,
; represented by different LMARK structures, that appear across region transitions.

(defun create-equalities-across-transition (state1 state2)	; qde1 -> qde2
  (dolist (v1 (qde-variables (state-qde state1)))
    (let ((v2 (find-if #'(lambda (v) (eql (variable-name v)
					  (variable-name v1)))
		       (qde-variables (state-qde state2)))))
      (if v2
	  (dolist (lm1 (cdr (assoc (variable-name v1) (state-qspaces state1))))
	    (let ((lm2 (find-if #'(lambda (lm) (and (not (eql lm lm1))
						    (eql (lmark-name lm)
							 (lmark-name lm1))))
				(cdr (assoc (variable-name v2) (state-qspaces state2))))))
	      (if lm2 (index-equations
			(transition-eqns v1 lm1 v2 lm2)))))))))

(defun store-globals-in-state (state)
  (setf (state-eqn-index state) *eqn-index*)
  (setf (state-bindings state) *bindings*)
  t)  

(defun setup-agenda ()
  (dolist (var (qde-variables (state-qde *state*)))
    (add-to-agenda `(zip-up   ',var))
    (add-to-agenda `(zip-down ',var)))
  (dolist (var-entry *eqn-index*)
    (dolist (vl-entry (cdr var-entry))
      (dolist (eqn (cdr vl-entry))
	(add-to-agenda eqn))))
  t)

; This assumes that user-asserted ranges are on state.assert-ranges,
; QDE.initial-ranges, or provided as an explicit argument.

(defun assimilate-asserted-ranges (state ranges)
  (let ((from-arg (cleanup-user-asserted-ranges ranges state))
	(from-qde (cleanup-user-asserted-ranges
		    (lookup-set 'initial-ranges (qde-other (state-qde state))) state))
	(from-state (cleanup-user-asserted-ranges
		      (state-assert-ranges state) state)))
    (dolist (item from-qde)   (add-to-agenda `(update-range ',(car item) ',(cadr item))))
    (dolist (item from-state) (add-to-agenda `(update-range ',(car item) ',(cadr item))))
    (dolist (item from-arg)   (add-to-agenda `(update-range ',(car item) ',(cadr item))))))


;;; In CLEANUP-USER-ASSERTED-RANGES 
;;; Entry := ((var lmark) (lb  ub)) where var and lmark are symbols
;;; Returns list of entry2 where
;;; Entry2 := ((<var> <lmark>)(lb ub))
;;; The previous definition was bombing when an lmark had the same name as the var. -drt 22feb90

(defun cleanup-user-asserted-ranges (entries state)	; ((<var> <lmark-name>) (lb ub))
  (loop with valist = (qde-var-alist (state-qde state))
	for ((varname lmarkname) bounds) in entries
	for var = (alookup varname valist)
	for lmark = (cleanup-landmark-description varname lmarkname state)
	when lmark
	  collect `((,var ,lmark) ,bounds)))

; Clean up a landmark-name by finding the LMARK structure in the qspace.
; Translate (<var> (AT <time>)) into a landmark if possible; else leave be.

(defun cleanup-landmark-description (varname lmdesc state)
  (cond ((symbolp lmdesc)
	 (find lmdesc (lookup-set varname (state-qspaces state)) :key #'lmark-name)
	 ; add (error "No landmark ~a in qspace for ~a in ~a." lmdesc varname state) ?
	 )
	((and (at-expressionp lmdesc)
	      (= (length lmdesc) 2))		; only handling one case
	 (or (lmark-at-time varname (cadr lmdesc) state)
	     lmdesc))
	(t (error "Can't handle landmark description ~a = ~a." varname lmdesc))))


(defun lmark-at-time (varname time-name state)
  (let ((time nil))
    (cond ((null state) nil)
	  ((and (setq time (qval-qmag (cdr (assoc 'time (state-qvalues state)))))
		(atom time)
		(eql time-name (lmark-name time)))
	   (let ((qmag (qval-qmag (cdr (assoc varname (state-qvalues state))))))
	     (if (typep qmag 'lmark) qmag nil)))
	  (t (lmark-at-time varname time-name (predecessor-of-state state))))))


;   2.  Generation of equations from a behavior.

; Make equations from values at the current state.
;   Notice that vls are now (<variable> <lmark>), where both are structures, not names.

(defun make-eqns-from-values (state)
  (let ((vls (all-vls-from-values state)))
    (dolist (con (qde-constraints (state-qde state)))
      (let ((vltuple (mapcar #'(lambda (v) (assoc v vls))
                             (constraint-variables con))))
        (case (constraint-typename con)
          (add      (index-equations (apply #'add-eqns vltuple)))
          (minus    (index-equations (apply #'minus-eqns vltuple)))
          (mult     (index-equations (apply #'mult-eqns vltuple)))
          (sum-zero (index-equations (apply #'sum-zero-eqns vltuple)))
          (d/dt     (index-equations (make-d/dt-eqns con state)))
          ((M+ M-)  (index-equations (make-M-MVT-eqns con state))
                    (index-equations (apply #'fn-eqns (cons (constraint-name con) vltuple))))
          ((S+ S-)  (index-equations (make-S-MVT-eqns con state))
                    (if (within-S-curve con state)
                        (index-equations
			  (apply #'fn-eqns (cons (constraint-name con) vltuple)))))

	  ((zero-std unreachable)  nil)
          ((constant increasing decreasing non-constant non-increasing non-decreasing)
                    nil)
          (t        (format *QSIM-trace* "~%Can't MAKE-EQNS-FROM-VALUES ~a." con)))))))

; Make equations from corresponding values at the initial state.

(defun make-eqns-from-cvtuples (state)
  (dolist (con (qde-constraints (state-qde state)))
    (let ((cvalues (cdr (assoc con (state-cvalues state))))
	  (params (constraint-variables con)))
      (case (constraint-typename con)
	(add      (dolist (cvtuple cvalues)
		    (index-equations 
		      (apply #'add-eqns (mapcar #'list params cvtuple)))))
	(minus    (dolist (cvtuple cvalues)
		    (index-equations (apply #'minus-eqns (mapcar #'list params cvtuple)))))
	(mult     (dolist (cvtuple cvalues)
		    (index-equations (apply #'mult-eqns (mapcar #'list params cvtuple)))))
	(sum-zero (dolist (cvtuple cvalues)
		    (index-equations (apply #'sum-zero-eqns (mapcar #'list params cvtuple)))))
	(d/dt     nil)
	((M+ M-)  (dolist (cvtuple cvalues)
		    (index-equations
		      (apply #'fn-eqns
			     (cons (constraint-name con)
				   (mapcar #'list params cvtuple))))))
	((S+ S-)  (if (within-S-curve con state)
		      (dolist (cvtuple cvalues)
			(index-equations
			  (apply #'fn-eqns
				 (cons (constraint-name con)
				       (mapcar #'list params cvtuple)))))))

	((zero-std unreachable)  nil)
	((constant increasing decreasing non-constant non-increasing non-decreasing)
	 nil)
	(t        (format *QSIM-trace* "`%Can't MAKE-EQNS-FROM-CVTUPLES ~a." con))))))

; Create a list of VLs, one for each variable in the state.
; Since a VL is of the form (var desc), retrieve with assoc.

(defun all-vls-from-values (state)
  (mapcar #'(lambda (var) (vl-from-value var state))
          (qde-variables (state-qde state))))

; Generate VL from variable, value, and state.

(defun vl-from-value (var state)
  (let ((val (qmag (cdr (assoc (variable-name var) (state-qvalues state)))))
        (time-point (qval-qmag (cdr (car (state-qvalues state))))))
    (cond ((lmark-p val) (list var val))
          (t (index-equations 
               (at-t-eqns var time-point (car val) (cadr val)))
             (list var (list 'at time-point))))))

; Equations are relations among VLPs:  (variable value-description) pairs.
; A value-description is a landmark, (at <time-point>), or (at (tp1 tp2)).
;   ((ADD x y z) (x1 y1 z1) ... )  =>  (= (z z1) (+ (x x1) (y y1))) ...
;   ((SUM-ZERO x1 x2 ... xn)  ...  (p1 p2 ... pn)  ...  )
;           =>   (= (x1 p1) (- 0 (SUM (x2 p2) ... (xn pn))))

; These equations are given VLPs.  Exclude tuples with all zeros or any infinity.

(defun bad-cvtuple (&rest tuple)
  (or (every #'(lambda (vl) (eql (cadr vl) *zero-lmark*)) tuple)
      (some  #'(lambda (vl) (member (cadr vl) infminf))   tuple)))

(defun add-eqns (a b c)
  (unless (bad-cvtuple a b c)
    (list `(= ,c (+ ,a ,b))
          `(= ,b (- ,c ,a))
          `(= ,a (- ,c ,b)))))

(defun mult-eqns (a b c)
  (unless (bad-cvtuple a b c)
    (list `(= ,c (* ,a ,b))
          `(= ,b (/ ,c ,a))
          `(= ,a (/ ,c ,b)))))

(defun minus-eqns (a b)
  (unless (bad-cvtuple a b)
    (list `(= ,b (- 0 ,a))
          `(= ,a (- 0 ,b)))))

(defun sum-zero-eqns (&rest vltuple)
  (unless (apply #'bad-cvtuple vltuple)
    (mapcar #'(lambda (var)
		`(= ,var (- 0 (SUM . ,(remove var vltuple)))))
	    vltuple)))

(defun fn-eqns (fn p q)
  (unless (or (member (cadr p) infminf) (member (cadr q) infminf))
    (list `(= ,q (function ,fn ,p))
          `(= ,p (inverse  ,fn ,q)))))

; These equations are given variables and value descriptions, separately.

; Define d(a,b) = b-a.  (The relation d(a,b)=-d(b,a) follows automatically.)

(defun d-eqns (x x1 x2)
  (let ((a   (list x x1))
        (b   (list x x2))
        (dab (list x (list 'd x1 x2))))
    (list `(= ,dab (- ,b ,a))
          `(= ,b   (+ ,a ,dab))
          `(= ,a   (- ,b ,dab)))))

; Generate the equations representing x(t1) = (a b), defining the term (x (at t1))

(defun at-t-eqns (x t1 a b)
  (nconc 
    (unless (eql a *minf-lmark*)
      (list `(> (,x (at ,t1)) (,x ,a))
            `(< (,x ,a) (,x (at ,t1)))))
    (unless (eql b *inf-lmark*)
      (list `(< (,x (at ,t1)) (,x ,b))
            `(> (,x ,b) (,x (at ,t1)))))))

; To declare that two variables have the same value.  Used primarily across transitions.

(defun transition-eqns (v1 lm1 v2 lm2)
  (list `(= (,v1 ,lm1) (,v2 ,lm2))
	`(= (,v2 ,lm2) (,v1 ,lm1))))

; The Mean Value Theorem tells us that, given a constraint (d/dt L R),
; two time-points t0 and t1, and the values a=L(t0) and b=L(t1),
; that there is a point t* in (t0,t1) such that R(t*) = (b-a)/(t1-t0).
;
; This gives us the equations for propagation:
;   d(a,b)   = d(t0,t1) * R(t*)
;   d(t0,t1) = d(a,b) / R(t*)
;   R(t*)    = d(a,b) / d(t0,t1)


; Create MVT equations from (d/dt x y)

(defun make-d/dt-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
	 (time-var (car (qde-variables (state-qde state))))	; variable TIME
         (params (constraint-variables con))
         (x (car params))
         (y (cadr params)))
    (unless (null gfather)
      (d/dt-eqns x
                 (cadr (vl-from-value x gfather))   ; x(t1)
                 (cadr (vl-from-value x state))     ; x(t2)
                 y
                 (cadr (vl-from-value y gfather))   ; y(t1)
                 (cadr (vl-from-value y state))     ; y(t2)
		 time-var
                 (qval-qmag (cdar (state-qvalues gfather)))	; t1
                 (qval-qmag (cdar (state-qvalues state)))	; t2
		 ;; The qmags of the grandfather, father, and current states. BKay 17Mar90
		 (qval-qmag (alookup (variable-name y) (state-qvalues gfather)))
		 (qval-qmag (alookup (variable-name y) (state-qvalues father)))
		 (qval-qmag (alookup (variable-name y) (state-qvalues state)))))))

(defun d/dt-eqns (x x1 x2 y y1 y2 time t1 t2 y1mag y12mag y2mag)
  (nconc 
    (d-eqns x x1 x2)
    (d-eqns time t1 t2)
    (list `(= (,x (d ,x1 ,x2)) (,y (defint ,y ,t1 ,t2)))
          `(= (,y (defint ,y ,t1 ,t2)) (,x (d ,x1 ,x2)))
          `(= (,y (defint ,y ,t1 ,t2)) (* (,time (d ,t1 ,t2))  (,y (at (,t1 ,t2)))))
          `(= (,time (d ,t1 ,t2))  (/ (,y (defint ,y ,t1 ,t2)) (,y (at (,t1 ,t2))))))
    ;; span-of only makes sense if the last interval was monotonic wrt y.
    ;; This might not be the case if y is an ignore-qdirs variable.
    ;; In particular, if qmag(y@t1) = qmag(y@t2), but qmag(y@(t1 t2)) is
    ;; different then we know that y@(t1 t2) is not in the span of y from t1 to t2
    ;; so this equation should not be included.  BKay 17Mar90
    (if (and (equal y1mag y2mag) (not (equal y1mag y12mag)))
	nil
	(list `(= (,y (at (,t1 ,t2))) (span-of (,y ,y1) (,y ,y2)))))))

; Direct-state-predecessor takes only predecessors under normal qsim prediction.

(defun direct-state-predecessor (state)
  (let ((just (state-justification state)))
    (cond ((member (car just) '(unique-successor-of
				 one-of-several-successors-of))
	   (cadr just)))))


; When y=f(x) for f in M+/M-/S+/S-, the Mean Value Theorem implies relations
; between x(t1), x(t2), y(t1), y(t2) and f'.

(defun make-M-MVT-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
         (params (constraint-variables con))
         (x (car params))
         (y (cadr params)))
    (unless (null gfather)
      (M-eqns x
              (cadr (vl-from-value x gfather))   ; x(t1)
              (cadr (vl-from-value x state))     ; x(t2)
              y
              (cadr (vl-from-value y gfather))   ; y(t1)
              (cadr (vl-from-value y state))     ; y(t2)
              con))))

(defun make-S-MVT-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
         (params (constraint-variables con))
         (x (car params))
         (y (cadr params)))
    (unless (or (null gfather)
                (not (within-S-curve con state gfather)))
      (M-eqns x
              (cadr (vl-from-value x gfather))   ; x(t1)
              (cadr (vl-from-value x state))     ; x(t2)
              y
              (cadr (vl-from-value y gfather))   ; y(t1)
              (cadr (vl-from-value y state))     ; y(t2)
              con))))

(defun within-S-curve (con state1 &optional state2)	; (S+ x y (a b) (c d))
  ; return T if state(s) are within the monotonic portion of the S+/S-.
  (let* ((x (car (constraint-variables con)))
	 (a (caar (constraint-bend-points con)))
	 (b (caadr (constraint-bend-points con)))
	 (qspace (cdr (assoc (variable-name x) (state-qspaces state1))))
	 (val1 (qval-qmag (cdr (assoc (variable-name x) (state-qvalues state1)))))
	 (val2 (if state2
		   (qval-qmag (cdr (assoc (variable-name x) (state-qvalues state2)))))))
    (and (member (qmag-order val1 a qspace) '(+ 0))
	 (member (qmag-order val1 b qspace) '(- 0))
	 (or (null val2)
	     (and (member (qmag-order val2 a qspace) '(+ 0))
		  (member (qmag-order val2 b qspace) '(- 0)))))))

(defun M-eqns (x x1 x2 y y1 y2 con)
  (let ((slope (list 'slope (constraint-name con))))
    (nconc
      (d-eqns x x1 x2)
      (d-eqns y y1 y2)
      (list `(= (,y (d ,y1 ,y2)) (* (,x (d ,x1 ,x2)) ,slope))
            `(= (,x (d ,x1 ,x2)) (/ (,y (d ,y1 ,y2)) ,slope))))))

;   3.  Index equations.
;
; Search through a list of expressions, and index them under every <vl> appearing in them.

(defun index-equations (eqn-list)
  (dolist (eqn eqn-list) (index-equation eqn eqn)))

(defun index-equation (eqn subtree)
  (cond ((numberp subtree) nil)
	((atom subtree) (error "Bad atom in expression:  ~a."  subtree))
	((vlp subtree) (store-into-eqn-index subtree eqn))
	(t (case (car subtree)
	     ((+ - * / span-of sum square)
	      (dolist (arg (cdr subtree))
		(index-equation eqn arg)))
	     ((= > >= < <= function inverse sqrt)
	      (index-equation eqn (third subtree)))
	     ((abs)
	      (index-equation eqn (second subtree)))
	     ((slope) nil)
	     (t (error "Bad expression type:  ~a"  subtree))))))

; Store a new equation on *eqn-index*.  Create indexing structure as needed.
; Do nothing if <eqn> is already there.

(defun store-into-eqn-index (vl eqn)
  (let ((var-entry nil)
	(vl-entry nil))
    (cond ((setq var-entry (assoc (car vl) *eqn-index*)))
	  (t (setq var-entry (list (car vl)))
	     (pushnew var-entry *eqn-index*)))
    (cond ((setq vl-entry (assoc (cadr vl) (cdr var-entry) :test #'equal)))
	  (t (setq vl-entry (list (cadr vl)))
	     (nconc var-entry (list vl-entry))))
    (cond ((member eqn (cdr vl-entry) :test #'equal))
	  (t (nconc vl-entry (list eqn))))
    (trace-eqn-index-entry vl eqn)
    eqn))


;   4.  Access functions for variable-landmark pairs (VLPs) and ranges.
;
; The basic object whose value is described by a range is a var.landmark pair.
; This is generalized to specify a variable, and a name for a quantity, where there
; are three types of names:  landmarks, differences between landmarks, and values
; taken on at particular times.
;         <vl>  ::=  (<var> <valname>)
;     <valname> ::=  <landmark>  |  (d <landmark> <landmark>)  |
;                    (at <time-point>)  |  (at (<time-point> <time-point>))

;    => Is there a potential bug with a variable or landmark named "D" or "AT"?

(defun vlp (exp)				; Is <exp> of type <vl>?
  (and (listp exp)
       (= (length exp) 2)
       (atom (first exp))			; eventually -> <variable>
       (or (atom (second exp))			; eventually -> <lmark>
	   (d-expressionp (second exp))
	   (at-expressionp (second exp))
	   (defint-expressionp (second exp)))))

(defun d-expressionp (exp)			; matches (d <landmark> <landmark>)?
  (and (= (length exp) 3)			; The "landmarks" can now be
       (eql (first exp) 'd)			; AT-expressions
       (or (atom (second exp))
	   (at-expressionp (second exp)))
       (or (atom (third exp))
	   (at-expressionp (third exp)))))

(defun at-expressionp (exp)			; matches (at <time-point>)  |
  (and (= (length exp) 2)			;         (at (<time-point> <time-point>))
       (eql (first exp) 'at)
       (or (atom (second exp))
	   (= (length (second exp)) 2)
	   (atom (first (second exp)))
	   (atom (second (second exp))))))

(defun defint-expressionp (exp)			; matches (defint <var> <tp1> <tp2>)
  (and (= (length exp) 4)
       (eql (first exp) 'defint)
       (atom (second exp))
       (atom (third exp))
       (atom (fourth exp))))

; The following access functions use find-or-create retrieval to make
; sure that every <vl> starts with the range (-inf +inf).  Retrieval and
; updating clobber the actual list cell where the range is stored.

(defun get-range-binding (var lm)
  (let ((var-entry nil)
	(vl-entry nil))
    (cond ((setq var-entry (assoc var *bindings*)))
	  (t (setq var-entry (list var))
	     (pushnew var-entry *bindings*)))
    (cond ((setq vl-entry (assoc lm (cdr var-entry) :test #'equal)))
	  (t (setq vl-entry (create-vl-entry var lm))
	     (nconc var-entry (list vl-entry))
	     (zip-down var)
	     (zip-up var)))
    (cdr vl-entry)))

(defun get-range (vl)
  (car (get-range-binding (car vl) (cadr vl))))

; A brand-new entry for a landmark is created with range (-inf +inf).
; An entry for (d <l1> <l2>) is created with an initial range (0 +inf), because l1<l2.

(defun create-vl-entry (var lm)
  (declare (ignore var))
  (cond ((lmark-p lm) (if (numberp (lmark-name lm))
			  (list lm (int (lmark-name lm) (lmark-name lm)))
			  (list lm (int '-inf '+inf))))
	((d-expressionp lm)      (list lm (int '-inf '+inf)))
	((at-expressionp lm)     (list lm (int '-inf '+inf)))
	((defint-expressionp lm) (list lm (int '-inf '+inf)))
	(t (error "Impossible case."))))

; Probably the D case should follow the AT case, letting one more step of propagation
; refine the trivial initial value of (0 +inf).  The make+store-d-eqns must go at
; equation creation time, then.

; 5.  Propagation loop.
;
;   *BINDINGS*  ::=  alist of (<var> . alist of (<landmark> <range>))
;        encodes the current state of knowledge about the value of <landmark>
;
;   *EQN-INDEX* ::=  alist of (<var> . alist of (<landmark> . <equations>))
;        indexes the equations whose RHS includes <landmark>.
;
;   *AGENDA*    ::=  (priority) queue of <equations> to propagate across.

; This is an initial version of the driver for this process.

(defun propagate-ranges ()
  (do ((N 0 (+ N 1))
       (eqn nil))
      (nil)
    (cond ((null *agenda*) (return t))		; done
	  (t (setq eqn (pop *agenda*))
	     (or (catch 'contradiction
		   (trace-consider eqn)
		   (cond ((member (car eqn) '(zip-up zip-down update-range)) (eval eqn))
			 (t (ranger* eqn)))
		   t)				; t = normal termination
		 (return nil))))))

; Update-range takes the intersection between the old range and the new one,
; and stores the result, detecting a null intersection.

(defun update-range (vl range)
  (let ((binding (get-range-binding (car vl) (cadr vl)))
	(intersection nil))
    (cond ((null range) (error "Invalid range for ~a:  ~a." vl range))
	  ((span>= range (car binding)))	; do nothing
	  ((insignificant-difference range (car binding))
	   (trace-ignore-update vl (car binding) range))
	  ((setq intersection (iintersect range (car binding)))
	   (trace-range-update vl (car binding) intersection)
	   (setf (car binding) intersection)
	   (follow vl))
	  (t (trace-range-update-failure vl (car binding) range)
	     (throw 'contradiction nil)))
    (car binding)))

; Once a range is updated, add all expressions that depend on that range
; to the agenda.

(defun follow (vl)
  (let ((equations (cdr (assoc (cadr vl)
			       (cdr (assoc (car vl) *eqn-index*))
			       :test #'equal))))
    (dolist (eqn equations) (add-to-agenda eqn))
    (add-to-agenda `(zip-up ',(car vl)))
    (add-to-agenda `(zip-down ',(car vl)))
    equations))

(defun add-to-agenda (equation)			; => replace with priority queue later.
  (trace-q2-agenda equation)
  (pushnew equation *agenda* :test #'equal))

; 6.  Zip-up and Zip-down quantity spaces.
;
; When the range associated with a landmark changes, the implications can 
; quickly be "zipped" up and down the quantity space.
;   This shrinks ranges as lower-bounds zip upward and upper-bounds zip downward.
;   Contradiction detected when a range upper bound is not greater than the
;   upward-propagating lower bound, and vice versa.

(defun zip-up (var)
  (let ((lb '-inf))
    (dolist (landmark (lookup-set (variable-name var) (state-qspaces *state*)))
      (cond ((member landmark infminf))
	    (t (let* ((binding (get-range-binding var landmark))
		      (nlb (lo (car binding))))
		 (cond ((e<= (hi (car binding)) lb)
			(trace-zip-failure var landmark 'up binding lb)
			(throw 'contradiction nil)))
		 (cond ((equalp lb nlb))
		       ((or (eql lb '-inf) (and (numberp nlb) (< lb nlb)))
			(setq lb nlb))
		       ((or (eql nlb '-inf) (and (numberp lb) (> lb nlb)))
			(trace-zip var landmark 'up (lo (car binding)) (hi (car binding))
				   lb (hi (car binding)))
			(setf (lo (car binding)) lb)
			(follow (list var landmark)))
		       (t (error "Impossible case.")))))))))

(defun zip-down (var)
  (let ((ub '+inf))
    (dolist (landmark (reverse (lookup-set (variable-name var) (state-qspaces *state*))))
      (cond ((member landmark infminf))
	    (t (let* ((binding (get-range-binding var landmark))
		      (nub (hi (car binding))))
		 (cond ((e<= ub (lo (car binding)))
			(trace-zip-failure var landmark 'down binding ub)
			(throw 'contradiction nil)))
		 (cond ((equalp ub nub))
		       ((or (eql ub '+inf) (and (numberp nub) (> ub nub)))
			(setq ub nub))
		       ((or (eql nub '+inf) (and (numberp ub) (< ub nub)))
			(trace-zip var landmark 'down (lo (car binding)) (hi (car binding))
				   (lo (car binding)) ub)
			(setf (hi (car binding)) ub)
			(follow (list var landmark)))
		       (t (error "Impossible case.")))))))))

; => zip-down is more expensive, because of the consing in reversing the qspace.


;   7.  RANGER* --- the quantitative range interpreter
;
; This range-expression interpreter evaluates expressions and updates ranges.
; It differs from RANGER in that:
;   - it must look up values indexed under <vl> in a two-level alist;
;   - it must handle (d <l> <l>) expressions.  
;   - it handles >, >=, <, <=, treating the strict and non-strict inequalities identically.

(defun ranger* (exp)
  (let ((value
	  (cond ((numberp exp) (int exp exp))
		((atom exp) (error "Bad expression:  ~a."  exp))
		((vlp exp) (get-range exp))
		(t (case (car exp)
		     (=         (update-range (cadr exp)
					      (ranger* (caddr exp))))
		     ((<= <)    (update-range (cadr exp)
					      (int '-inf (hi (ranger* (caddr exp))))))
		     ((>= >)    (update-range (cadr exp)
					      (int (lo (ranger* (caddr exp))) '+inf)))
		     (+         (plus-handle (cdr exp)))
		     (-         (neg-handle (cdr exp)))
		     (*         (mult-handle (cdr exp)))
		     (/         (idiv (ranger* (cadr exp))
				      (ranger* (caddr exp))))
		     (abs       (iabs (ranger* (cadr exp))))
		     (sum       (isum (mapcar #'ranger* (cdr exp))))
		     (span-of   (span-of (ranger* (cadr exp))
				         (ranger* (caddr exp))))
		     (square (isquare (ranger* (second exp))))
		     (sqrt  (isquart (ranger* (second exp))
				      (ranger* (third exp))))
		     (function  (apply-fn (cadr exp)
					  (ranger* (caddr exp))))
		     (inverse   (apply-finv (cadr exp)
				 	    (ranger* (caddr exp))))
		     (slope     (explicit-bounds (car exp) (cadr exp)))
		     (curvature (explicit-bounds (car exp) (cadr exp)))
		     (t         (error "Bad expression:  ~a"  exp)))))))
    (trace-ranger exp value)
    value))

; RANGER* is the only interface between the range propagator and the incomplete
; quantitative knowledge representation and arithmetic package.  It can be replaced
; by either of two strategies:
;  1. Redefine:  iintersect, iplus, idiff, itimes, idiv, iabs, span-of
;                (??) apply-fn, apply-finv
;  2. Redefine RANGER* to call an alternate representation and access functions.
;     (This seems the more sensible.)



;;; Give "*" arity of 2 or more.

(defun mult-handle (args)
  (itimes (ranger* (car args))
	  (if (cddr args)
	      (mult-handle (cdr args))
	      (ranger* (cadr args)))))

;;; Give "+" arity of 2 or more.

(defun plus-handle (args)
  (iplus (ranger* (car args))
	  (if (cddr args)
	      (plus-handle (cdr args))
	      (ranger* (cadr args)))))

;;; Give "-" unfixed arity - 1 arg is negation, 2 args are subtraction,
;;; >2 args substracts the sum of the trailing args from the first arg.

(defun neg-handle (args) 
  (cond ((cddr args)(idiff (ranger* (first args))
			   (plus-handle (cdr args))))
	((cdr args)(iplus (ranger* (first args))
			  (iminus (ranger* (second args)))))
	(t (iminus (ranger* (first args))))))

;   8.  RANGER  --- a simple interval-arithmetic interpreter
;
; Here we have a range interpreter for algebraic expressions.
;   <bindings>  ::=  alist of (<variable> <interval>)

(defun range-interpreter (exp bindings)
  (let ((*bindings* bindings))
    (declare (special *bindings*))
    (ranger exp)))

(defun ranger (exp)
  (declare (special *bindings*))		; expects bindings on global variable
  (let ((value
	  (cond ((symbolp exp) (or (cadr (assoc exp *bindings*))
				   (error "No binding for ~a in ~a." exp *bindings*)))
		((numberp exp) (int exp exp))
		(t (case (car exp)
		     (+ (iplus (ranger (cadr exp))
			       (ranger (caddr exp))))
		     (- (idiff (ranger (cadr exp))
			       (ranger (caddr exp))))
		     (* (itimes (ranger (cadr exp))
				(ranger (caddr exp))))
		     (/ (idiv (ranger (cadr exp))
			      (ranger (caddr exp))))
		     (abs (iabs (ranger (cadr exp))))
		     (span-of (span-of (ranger (cadr exp))
				       (ranger (caddr exp))))
		     (function (apply-fn (cadr exp)
					 (ranger (caddr exp))))
		     (inverse  (apply-finv (cadr exp)
					   (ranger (caddr exp))))
		     (t (error "Bad expression:  ~a"  exp)))))))
    (trace-ranger exp value)
    value))


; 9.  Interval-arithmetic routines.
;
; An interval is defined as a pair (lo hi), where lo <= hi, describing a real value
; that must lie within the interval.
;   Intervals are defined over the extended real number line, [-inf,+inf],
;   but must include finite values, so +inf cannot be a lower bound, nor -inf an upper bound.

; This makes the set of intervals closed under division, branching on the cases:
;       - zero is in the interior of the interval
;       - zero is an upper or lower bound
;       - zero is outside the interval.


(defun int (lo hi)
  (cond ((or (not (or (numberp lo) (eql lo '-inf)))
	     (not (or (numberp hi) (eql hi '+inf)))
	     (and (numberp lo) (numberp hi) (> lo hi)))
	 (error "Illegal interval bounds:  [~a, ~a]." lo hi))
	(t (list lo hi))))

; Testing sign

(defun ipos (range)
  (and (numberp (lo range))
       (> (lo range) 0)))

(defun ineg (range)
  (and (numberp (hi range))
       (< (hi range) 0)))

(defun iabs (range)
  (cond ((ipos range) range)
	((ineg range) (iminus range))
	(t					; (<= (lo range) 0 (hi range))
	 (int 0 (cond ((eql (lo range) '-inf) '+inf)
		      ((eql (hi range) '+inf) '+inf)
		      (t (max (abs (lo range)) (abs (hi range)))))))))

; Addition, subtraction, negation, and absolute-value are straight-forward.

(defun iplus (xrange yrange)
  (int (if (and (numberp (lo xrange)) (numberp (lo yrange)))  (+ (lo xrange) (lo yrange))  '-inf)
       (if (and (numberp (hi xrange)) (numberp (hi yrange)))  (+ (hi xrange) (hi yrange))  '+inf)))

(defun iminus (range)
  (int (if (eql (hi range) '+inf)  '-inf  (- (hi range)))
       (if (eql (lo range) '-inf)  '+inf  (- (lo range)))))

(defun idiff (xrange yrange)
  (iplus xrange (iminus yrange)))

(defun isum (L)
  (let* ((lbs (mapcar #'car L))
	 (ubs (mapcar #'cadr L))
	 (lb (if (member '-inf lbs)  '-inf  (apply #'+ lbs)))
	 (ub (if (member '+inf ubs)  '+inf  (apply #'+ ubs))))
    (int lb ub)))


; Multiplication and division.
;   The code is defined in terms of e*, emin, and emax, "extended" functions
;   that can cope with the values +inf and -inf.

(defun itimes (xrange yrange)
  (cond ((and (ipos xrange) (ipos yrange))
	 (int (e* (lo xrange) (lo yrange)) (e* (hi xrange) (hi yrange))))
	((and (ineg xrange) (ineg yrange))
	 (int (e* (hi xrange) (hi yrange)) (e* (lo xrange) (lo yrange))))
	((and (ipos xrange) (ineg yrange))
	 (int (e* (hi xrange) (lo yrange)) (e* (lo xrange) (hi yrange))))
	((and (ineg xrange) (ipos yrange))
	 (int (e* (lo xrange) (hi yrange)) (e* (hi xrange) (lo yrange))))
	(t (int (emin (e* (lo xrange) (hi yrange))
		      (e* (hi xrange) (lo yrange)))
		(emax (e* (lo xrange) (lo yrange))
		      (e* (hi xrange) (hi yrange)))))))

; Decompose x/y = x*(1/y) unless y = [0,0] or lb(y) < 0 < ub(y).

(defun idiv (xrange yrange)
  (cond ((and (numberp (lo yrange))		;   - the most common case
	      (numberp (hi yrange))
	      (or (< 0 (lo yrange)) (< (hi yrange) 0)))
	 (itimes xrange (int (/ 1 (hi yrange)) (/ (lo yrange)))))
	((equal yrange '(0 0))
	 (int '-inf '+inf))
	((and (or (eql (hi yrange) '+inf) (> (hi yrange) 0))
	      (or (eql (lo yrange) '-inf) (< (lo yrange) 0)))
	 (int '-inf '+inf))
	(t (itimes xrange
		   (int (cond ((eql (hi yrange) '+inf) 0)
			      ((= (hi yrange) 0) '-inf)
			      (t (/ 1 (hi yrange))))
			(cond ((eql (lo yrange) '-inf) 0)
			      ((= (lo yrange) 0) '+inf)
			      (t (/ 1 (lo yrange)))))))))

(defun iinv (range)				; 1/x
  (cond ((and (numberp (lo range))		;   - the most common case
	      (numberp (hi range))
	      (or (< 0 (lo range)) (< (hi range) 0)))
         (int (/ 1 (hi range)) (/ (lo range))))
	((equal range '(0 0))
	 (int '-inf '+inf))
	((and (or (eql (hi range) '+inf) (> (hi range) 0))
	      (or (eql (lo range) '-inf) (< (lo range) 0)))
	 (int '-inf '+inf))
	(t (int (cond ((eql (hi range) '+inf) 0)
		      ((= (hi range) 0) '-inf)
		      (t (/ 1 (hi range))))
		(cond ((eql (lo range) '-inf) 0)
		      ((= (lo range) 0) '+inf)
		      (t (/ 1 (lo range))))))))

;;; ISQUARE does not check for malformed RANGE.

(defun isquare (range)
  (let ((lo (lo range))
	(hi (hi range)))
    (cond ((and (numberp lo)(>= lo 0))		; completely nonnegative interval
	   (list (* lo lo)(e* hi hi)))
	  ((and (numberp hi)(<= hi 0))		; completely nonpositive interval
	    (list (* hi hi )(e* lo lo)))	
	  (t (list 0 (emax (e* hi hi)		; interval crosses 0.
			   (e* lo lo)))))))

;;; In ISQUART, currange is the currently known range of an expression
;;; EXP, and EXP = plusorminus sqrt(sqrange).  The central problem for
;;; combining square roots across ranges is that the sqrt of a range has
;;; to disconnected intervals.  Suppose we know that exp1 = sqrt(exp2),
;;; and we already know (from other constraints) that exp1 = [-10 3.5]
;;; and exp2 = [9 16].  The sqrt relationship tells us that we can
;;; refine exp1 to [-4 -3] OR [3 3.5].  However, we have no way of
;;; representing disjunt intervals.  In this case, we set exp1's new
;;; range to [-4 3.5].

;;; Fortunately, in most cases, exp1 will have an unambiguous sign, and
;;; we can select the correct root.

;;; The SQRT relation in Q2 has a unique syntax, (csqrt exp1 exp2).  The
;;; interpretation is "The sqrt(exp2), using exp1, if possible, to
;;; determine the correct root."

(defun isquart (currange sqrange)
  (let ((lo (lo currange))
	(hi (hi currange))
	(slo (lo sqrange))
	(shi (hi sqrange)))
    (when (or (and (numberp shi)(minusp shi))
	      (eq shi '-inf))
      (error "~s can't be the square root of ~s" currange sqrange))
    (cond ((and (numberp lo)(>= lo (e-inverse (esqrt (emax slo 0)))))	; ranges overlap only on + side.
	   (list (esqrt (emax slo 0))(esqrt shi)))
	  ((and (numberp hi)(<= hi (e-inverse (esqrt (emin shi 0)))))	; ranges overlap only on - side.
	   (list (e-inverse (esqrt shi))
		 (e-inverse (esqrt (max slo 0)))))	
	  (t (let ((shisqrt (esqrt shi)))	; ranges overlap on both sides.
	       (list (emax lo (e-inverse shisqrt))
		     (emin hi shisqrt)))))))

; E* is an extended scalar multiplication, handling the values -inf and +inf.

(defun e* (a b)
  (cond ((and (numberp a) (numberp b)) (* a b))	;   - most common case
	((eql a '+inf) (cond ((eql b '+inf) '+inf)
			     ((eql b '-inf) '-inf)
			     ((> b 0) '+inf)
			     ((= b 0) 0)
			     ((< b 0) '-inf)))
	((eql a '-inf) (cond ((eql b '+inf) '-inf)
			     ((eql b '-inf) '+inf)
			     ((> b 0) '-inf)
			     ((= b 0) 0)
			     ((< b 0) '+inf)))
	((> a 0) (cond ((eql b '+inf) '+inf)
		       ((eql b '-inf) '-inf)))
	((= a 0) 0)
	((< a 0) (cond ((eql b '+inf) '-inf)
		       ((eql b '-inf) '+inf)))))

(defun emax (a b)
  (cond ((and (numberp a) (numberp b)) (max a b))
	((eql a '+inf) '+inf)
	((eql b '+inf) '+inf)
	((eql a '-inf) b)
	((eql b '-inf) a)))

(defun emin (a b)
  (cond ((and (numberp a) (numberp b)) (min a b))
	((eql a '-inf) '-inf)
	((eql b '-inf) '-inf)
	((eql a '+inf) b)
	((eql b '+inf) a)))

(defun e<= (a b)
  (cond ((and (numberp a) (numberp b)) (<= a b))
	((eql a '-inf) t)
	((eql b '+inf) t)))

;;; ESQRT returns the positive, extended sqrt of a value.

(defun esqrt (val)
  (cond ((and (numberp val)
	      (>= val 0))
	 (sqrt val))
	((eq val '+inf) '+inf)
	(t (error "Can't take square root of ~s" val))))

(defun e-inverse (value)
  (cond ((numberp value)
	 (- value))
	((eq value '-inf) '+inf)
	((eq value '+inf) '-inf)))



; Monotonic functions are interpreted according to their computable envelopes.
; The computable envelopes are guaranteed to be called with numerical arguments.
; They must return numerical values, or NIL where they are undefined.

(defun apply-fn (f range)
  (multiple-value-bind (function upper-env lower-env)
      (get-envelopes f)
    (case function 
      ((M+ S+ m0+)
       (int (or (and (numberp (lo range)) lower-env (funcall lower-env (lo range))) '-inf)
	    (or (and (numberp (hi range)) upper-env (funcall upper-env (hi range))) '+inf)))
      ((M- S- m0-)
       (int (or (and (numberp (hi range)) lower-env (funcall lower-env (hi range))) '-inf)
	    (or (and (numberp (lo range)) upper-env (funcall upper-env (lo range))) '+inf)))
      (t (error "Unknown Function ~S" function)))))

(defun apply-finv (f y)
  (multiple-value-bind (function upper-inv lower-inv)
      (get-envelopes f :inverse t)
    (case function 
      ((M+ S+ m0+)
       (int (or (and (numberp (lo y)) upper-inv (funcall upper-inv (lo y))) '-inf)
	    (or (and (numberp (hi y)) lower-inv (funcall lower-inv (hi y))) '+inf)))
      ((M- S- m0-)
       (int (or (and (numberp (hi y)) lower-inv (funcall lower-inv (hi y))) '-inf)
	    (or (and (numberp (lo y)) upper-inv (funcall upper-inv (lo y))) '+inf)))
      (t (error  "Unknown Function ~S" function)))))



(defun get-envelopes (fspec &key (inverse))
  (let ((up (if inverse 'upper-inverse 'upper-envelope))
	(down (if inverse 'lower-inverse 'lower-envelope)))
    (cond ((and (symbolp fspec)
		(q-function-p fspec))
	   (if inverse
	       (values (get-function-type fspec)
		       (get-function-upper-inverse fspec)
		       (get-function-lower-inverse fspec))
	       (values (get-function-type fspec)
		       (get-function-upper-envelope fspec)
		       (get-function-lower-envelope fspec))))
	  ((and (consp fspec)
		(member (car fspec) '(m+ m0+ m- m0-)))
	   (values (car fspec)
		   (cadr (assoc up
				(cdr (assoc fspec *M-envelopes* :test #'equal))))
		   (cadr (assoc down
				(cdr (assoc fspec *M-envelopes* :test #'equal))))))
	  (t (error "Don't know this function spec ~s" fspec)))))

;;; Functions for finding information about qde-functions - named
;;; monotonic functions.  - drt 

(defun get-function-type (fnname)
  (or (getf (alookup fnname (qde-functions *current-qde*)) :type)
      (error "Couldn't find ~s" fnname)))

(defun get-function-upper-envelope (fnname)
  (or (getf (alookup fnname (qde-functions *current-qde*)) :upper-envelope)
      (error "Couldn't find ~s" fnname)))

(defun get-function-lower-envelope (fnname)
  (or (getf (alookup fnname (qde-functions *current-qde*)) :lower-envelope)
      (error "Couldn't find ~s" fnname)))

(defun get-function-upper-inverse (fnname)
  (or (getf (alookup fnname (qde-functions *current-qde*)) :upper-inverse)
      (error "Couldn't find ~s" fnname)))

(defun get-function-lower-inverse (fnname)
  (or (getf (alookup fnname (qde-functions *current-qde*)) :lower-inverse)
      (error "Couldn't find ~s" fnname)))

(defun get-function-signmaps (fnname)
  (or (getf (alookup fnname (qde-functions *current-qde*)) :signmaps)
      (error "Couldn't find ~s" fnname)))

(defun q-function-p (fnname)
  (or (member fnname (qde-functions *current-qde*) :key #'car)
	    (eq fnname 'function)))

; Generalization to S+/S- is not straight-forward (or for that matter to M+/M- with
; finite asymptotes) since it is unclear how to handle a bounded range for the
; envelope function or its inverse.

; Various explicitly-asserted bounds for f may be retrieved:
;    (slope (M+ x y))   or   (curvature (M+ x y))

(defun explicit-bounds (property f)
  (let ((bounds (lookup property
			(cdr (assoc f *M-envelopes* :test #'equal)))))
    (cond ((null bounds) (int '-inf '+inf))
	  (t bounds))))


; Check whether one interval includes another.

(defun span>= (xrange yrange)				; range xrange spans range yrange
  (and (e<= (lo xrange) (lo yrange))
       (e<= (hi yrange) (hi xrange))))

(defun span-of (xrange yrange)				; smallest interval containing both xrange and y.
  (int (emin (lo xrange) (lo yrange))
       (emax (hi xrange) (hi yrange))))

; Intersecting two intervals gives a smaller interval, or NIL if they are disjoint.

(defun iintersect (xrange yrange)
  (let ((lb (emax (lo xrange) (lo yrange)))
	(ub (emin (hi xrange) (hi yrange))))
    (cond ((and (numberp lb) (numberp ub) (> lb ub)) nil)	; empty intersection!
	  (t (int lb ub)))))

; An update is insignificant if the change falls below some threshold.
; This is necessary to avoid infinite convergent sequences.

(defun insignificant-difference (xrange yrange)
  (and (if (and (numberp (lo xrange)) (numberp (lo yrange)))
	   (< (abs (- (lo xrange) (lo yrange))) *epsilon*)
	   (equal (lo xrange) (lo yrange)))
       (if (and (numberp (hi xrange)) (numberp (hi yrange)))
	   (< (abs (- (hi xrange) (hi yrange))) *epsilon*)
	   (equal (hi xrange) (hi yrange)))))

; 10.  Display and trace functions.

(defun show-bindings (&optional (state nil) &aux (*detailed-printing* nil))
  (dolist (var-entry
	    (cond (state (state-bindings state))
		  (t *bindings*)))
    (format *QSIM-Trace* "~% ~a:" (car var-entry))
    (dolist (vl-entry (cdr var-entry))
      (format *QSIM-Trace* "~%~6t~a: ~16t~a" (car vl-entry) (cadr vl-entry))))
  t)

;;; If varname present, SHOW-EQUATIONS prints out only the equations
;;; associated with the variable of that name.  Otherwise, prints out
;;; all the Q2 equations for the state.

(defun show-equations (&optional (state nil) (varname))
  (loop with *detailed-printing* = nil
	for (var . var-entry) in
	    (cond (state (state-eqn-index state))
		  (t *eqn-index*))
	when (or (null varname)
		 (equal varname (variable-name var)))
	  do (format *QSIM-Trace* "~% ~a:" var)
	     (dolist (vl-entry var-entry)
	      (format *QSIM-Trace* "~%~6t~a:" (car vl-entry))
	      (dolist (eqn  (cdr vl-entry))
		(format *QSIM-Trace* "~%~12t~a" eqn))))
  (values))

; There is a menu for controlling these variables in front-end.lisp.

(defun trace-Q2-global-filter (state)
  (if *trace-Q2-check*
      (format *QSIM-Trace* "~%Starting quantitative check on ~a." state)))

(defun trace-eqn-index-entry (vl eqn)
  (if *trace-eqn-indexing*
      (format *QSIM-Trace* "~%  Indexing ~a  ->  ~a." vl eqn)))

(defun trace-constraint-for-indexing (con)
  (if *trace-eqn-indexing*
      (format *QSIM-Trace* "~%Creating equations from ~a." con)))

(defun trace-q2-agenda (eqn)
  (if *trace-q2-agenda*
      (format *QSIM-Trace* "~%Add to agenda:  ~a." eqn)))

(defun trace-consider (eqn)
  (if *trace-consider-eqn*
      (format *QSIM-Trace* "~%Considering:  ~a." eqn))
  (if *trace-single-char* (format *QSIM-Trace* ".")))

(defun trace-ranger (exp value)
  (if *trace-ranger*
      (format *QSIM-Trace* "~%Ranger:  ~a  ->  ~a" exp value)))

(defun trace-range-update (vl old new)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%Updating:  ~a from ~a -> ~a." vl old new))
  (if *trace-single-char* (format *QSIM-Trace* "!")))

(defun trace-ignore-update (vl old new)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%Insignificant update ignored ~a:  ~a ~~ ~a." vl old new))
  (if *trace-single-char* (format *QSIM-Trace* "~~")))

(defun trace-range-update-failure (vl old new)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%Contradiction between values for ~a:  ~a vs. ~a." vl old new))
  (if *trace-single-char* (format *QSIM-Trace* "#")))

(defun trace-zip (var lm dir olb oub nlb nub)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%ZIP-~a (~a ~a):  (~a ~a) -> (~a ~a)."
	      dir var lm olb oub nlb nub)))

(defun trace-zip-failure (var lm dir range bound)
  (if *trace-range-update*
      (format *QSIM-Trace* "~%ZIP-~a (~a ~a) = ~a.  Contradiction with bound ~a."
	      dir var lm range bound))
  (if *trace-single-char* (format *QSIM-Trace* "#")))

; Single character trace output:
;   []      begin/end equation creation from state
;   .       evaluation of an equation
;   !       range update
;   ~       insignificant change to range
;   #       state found inconsistent

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;           Notes about fixes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; => The bug is that transition states are not properly filtered in the
;    current situation, because Q2 can only work after landmarks have been
;    created, and transition-state landmarks are only created after
;    the global-filter-list has been run.
;
; => The interaction with state-transitions is a bit wierd anyway.
;
; => There is a partially-included bug fix that creates (= <lm1> <lm2>) equations
;    for LMARK structures of the same name, but not EQ, across region transitions.

; Need:
;
;  - recognize and handle higher-order derivative combinations:
;      (d/dt x v)  (d/dt v a)
;  - recognize simultaneous equations to solve directly.
;     (I+S minimal complete subset analysis?)

; Inefficiencies:
;  - assimilated-asserted-ranges asserts all user-provided ranges at each state.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;         Changes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; - Make-new-state has &key :assert-ranges argument.
;   This provides a way to assert quantitative range constraints by hand, with either:
;     (quantitative-range-reasoning <state> :assert-ranges <range-specs>)
;     (make-initial-state <state-or-qde> :assert-ranges <range-specs>)
;   Make-initial-state and make-modified-state do not have this feature.
;   However, they are now defined in terms of the more general make-new-state.
;
; - It is now possible to specify a value for (variable (time t2)).
;     e.g. in assert-ranges:   ((<var> (AT <time>)) (lb ub))
;     is equivalent to ((<var> <lmark>) (lb ub)), where <var>(<time>) = <lmark>.
;     If <var>(<time>) is not a landmark, nothing happens!
;
; - Envelope functions may now be undefined (returning NIL), in which case
;   they provide no constraint.
;
; - Non-landmark values of the form (V (at T1)) are handled.
;   The MVT forms are handled more generally.
;   Equations are generated more cleanly.
;   Equation generation is not as redundant as before.
;   VLs are (<variable> <lmark>) rather than (<variable-name> <lmark-name>)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;         INCOMPATIBLE Changes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; - replaced unnamed second argument to quantitative-range-reasoning with
;   &key :assert-ranges argument.
;      (I don't believe anyone but me used that second arg anyway.)

