;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-
;;;  $Id: equations.lisp,v 1.13 1992/08/05 18:21:06 bert Exp $

;;; Equations.lisp.
;;; Copyright (c) 1990 Benjamin Kuipers.
;;; Modified 6/11/91 by DLD to change (X (AT T1)) to (X (AT S-2)) since
;;; (X (AT T1)) is not well-defined at transitions.
;;; Note that this change leads to differences between the description
;;; of the form (AT T1 T2) described on p. 18 of AI Tech Report 90-122.
;;; A vl pair such as (X (AT T1 T2)) is now represented as (X (AT S1))
;;; where S1 is the interval state between T1 and T2.
;;; Further Modified 4Sept91 by BKay to use vls that are (<var> <lmark>)
;;; instead of (varname lmarkname)
;;; Modified 5May92 by BKay to include multivariate M constraints
;;; (see below)

;;; Q2 Multivariate M constraints.
;;; ==============================
;;;
;;;
;;; M-envelopes clause format
;;; -------------------------
;;;
;;; The multivariate M constraint looks like this :
;;;
;;;   (((M s1 ... sn) v1 ... vn y) (cv11 ... cv1n y1) ...)
;;;
;;; As with M+/M- constraints, m-envelopes clauses are needed for
;;; each M constraint.  Their form is :
;;;
;;; (((M s1 ... sn) v1 ... vn y)
;;;  <upper-env> 
;;;  <lower-en1>)
;;;
;;; where
;;;  <upper-env> is a list of (upper-envelope (f v1 ... vn))
;;;  <lower-env> is a list of (lower-envelope (f v1 ... vn))
;;;
;;; example m-envelopes clause :
;;;  (m-envelopes
;;;     (((M + -) A B C)
;;;      (upper-envelope (lambda (a b) (f1 a b)))
;;;      (lower-envelope (lambda (a b) (g1 a b))))
;;;  )
;;;
;;; Note that this definition is different from M+, say, in that only one
;;; envelope set (for f(A, B) = C) is defined rather than the inverse
;;; functions (B = g(A, C)  and  A = h(B, C)) as well.  The M constraint is a
;;; "one-way" constraint in this sense, since it defines only the partials
;;; of y with respect to each vi but not dvi/dvj.  To get the effect of
;;; an (M+ A C) , you need two (M +) constraints : ((M +) A C) and ((M +) C A).
;;;
;;; Comparison with M+ :
;;;  (m-envelopes
;;;     ((M+ A C)
;;;      (upper-envelope (lambda (A) (f1 A)))
;;;      (lower-envelope (lambda (A) (f2 A)))
;;;      (upper-inverse  (lambda (B) (f1inv B)))
;;;      (lower-inverse  (lambda (B) (f2inv B)))))
;;;
;;;  (m-envelopes
;;;     (((M +) A C)
;;;      (upper-envelope (lambda (A) (f1 A)))
;;;      (lower-envelope (lambda (A) (f2 A))))
;;;     (((M +) C A)
;;;      (upper-envelope (lambda (C) (f2inv A)))
;;;      (lower-envelope (lambda (C) (f1inv A)))))
;;;
;;;  note that the lower-inverse clause of (M+ A C) and upper-envelope
;;;  clause of ((M +) C A) are the same function.
;;;
;;; For partials that are zero, it is assumed that the functional envelopes ignore
;;; the variable.  For example:
;;; 
;;;  (m-envelopes
;;;     (((M + 0) A B C)
;;;      (upper-envelope (lambda (a b) (declare (ignore b)) (f3 a)))
;;;      (lower-envelope (lambda (a b) (declare (ignore b)) (g3 a)))))
;;;
;;; Here, since dC/dB is zero, there is no relation between the two variables.
;;; Therefore calculations of C do not use the value of B.


(in-package :qsim)


; The code consists of several chapters.
;   1.  Generate equations from a behavior.
;   2.  Primitive terms:  landmarks, differences, etc.
;   3.  Generate equations from values and cvtuples.
;   4.  Generate equations using the Mean Value Theorem.
;   5.  Create all equivalent variants of a given equation.
;   6.  Index equations.

(defparameter *enable-equation-generation* t)	; or suppress.

(defparameter *trace-eqn-creation* nil)
(defparameter *trace-eqn-indexing* nil)		; trace equation index creation
(defparameter *trace-single-char* t)		; single-char trace output

; Equation generation is invoked functionally, by specifying a state,
; puts the equations on the global *eqn-index*, and stores them in state.eqn-index when done.

(defparameter *eqn-index* nil)			; where to store generated equations.

;   1.  Generate equations from a behavior.
;
; Given a state, construct all equations and index them on *eqn-index*.
; After all equations are created, store them back into (state-eqn-index state).
; This is logically prior to all range propagation.
;   Make equations from corresponding values only at initial and transition-result states.
;   Make equations from current values at all time-points.
;   Assert identical values of variables and landmarks with the same names across transitions.

(defun setup-equations-from-state (state)
  (when *enable-equation-generation*
    (let ((*eqn-index* (state-eqn-index state)))
      (declare (special *eqn-index*))
      (if *trace-single-char* (format *QSIM-Trace* "["))
      (if (member (car (state-justification state))
		  '(initialized-with transition-from
				     one-of-several-completions-of unique-completion-of))
	  (make-eqns-from-cvtuples state))
      (make-eqns-from-values state)
      (if *trace-single-char* (format *QSIM-Trace* "]"))
      (setf (state-eqn-index state) *eqn-index*)
    t)))


;   2.  Primitive terms:  landmarks, differences, etc.
;
; The basic object whose value is described by a range is a var.landmark pair (aka vl or vlp).
; This is generalized to specify a variable, and a name for a quantity, where there
; are three types of names:  landmarks, differences between landmarks, and values
; taken on at particular times.
;         <vl>  ::=  (<var> <lmarkexp>)
;     <lmarkexp ::=  <lmark>  |  (d <lark> <lmark>)  |
;                    (at <state>)

;    => Is there a potential bug with a variable or landmark named "D" or "AT"?
;       ANSWER: I don't think so, because variables will be structures now, not symbols.

; To help distinguish vl usage from other usages of a similar representation,
; the following two accessor functions are used when a vl is being referenced.
;
(defmacro vl-var (vl)
  `(first ,vl))

(defmacro vl-lmark (vl)
  `(second ,vl))

(defun vlp (exp)				; Is <exp> of type <vl>?
  (and (listp exp)
       (= (length exp) 2)
       (variable-p (first exp))
       (or (lmark-p (second exp))
	   (d-expressionp (second exp))
	   (at-expressionp (second exp))
	   (defint-expressionp (second exp)))))

(defun d-expressionp (exp)			; matches (d <landmark> <landmark>)?
  (and (= (length exp) 3)			; The "landmarks" can now be
       (eql (first exp) 'd)			; AT-expressions
       (or (lmark-p (second exp))
	   (at-expressionp (second exp)))
       (or (lmark-p (third exp))
	   (at-expressionp (third exp)))))

(defun at-expressionp (exp)			; matches (at <state>)  |
  (and (= (length exp) 2)			;         (at (<state> <state>)) NO
       (eql (first exp) 'at)                    ; the second form isn't created anywhere
       (state-p (second exp))))
;       (or (atom (second exp))
;	   (= (length (second exp)) 2)          ; Are these three lines needed?
;	   (atom (first (second exp)))
;	   (atom (second (second exp))))))

(defun defint-expressionp (exp)			; matches (defint <var> <tp1> <tp2>)
  (and (= (length exp) 4)
       (eql (first exp) 'defint)
       (variable-p (second exp))
       (atom (third exp))
       (atom (fourth exp))))


; Create a list of VLs, one for each variable in the state.
; Since a VL is of the form (var desc), retrieve with assoc.

(defun all-vls-from-values (state)
  (mapcar #'(lambda (pair) (vl-from-varname-and-value (car pair) (cdr pair) state))
	  (state-qvalues state)))

; Generate VL from variable name, value, and state.
; This is one of the few functions that takes a varname rather than a <var>.
; Inputs:  varname - that SYMBOL name of a variable
;          qval    - a qval
;          state   - a state
;
(defun vl-from-varname-and-value (varname qval state)
  (cond
    ;; If null value, return nil.
    ((null qval) nil)
    (T
     (let ((var (alookup varname (qde-var-alist (state-qde state)))))
       (if (atom (qval-qmag qval))
	   ;; If qmag is an atom, return (<VAR> <QMAG>).
	   (list var (qval-qmag qval))
	   ;; Otherwise, return (<VAR> (AT <STATE>))
	   (progn
	     (index-equations (at-t-eqns var state (car (qmag qval)) (cadr (qmag qval))))
	     (list var (list 'AT state))))))))

;    (t (let* ((time-point (qval-qmag (cdr (car (state-qvalues state)))))
;	      (tpnames (lmnames time-point)))
;	 (at-t-eqns varname tpnames (car (qval-qmag qval)) (cadr (qval-qmag qval)))
;	 (list varname (list 'AT tpnames))))))
   

#| Unused as of 4Sept91
(defun lmnames (qmag)
  (cond ((atom qmag) (lmark-name qmag))
	(t (list (lmark-name (car qmag)) (lmark-name (cadr qmag))))))
|#


;;; This function name is definitly a misnomer.  What it does is to generate
;;; a vl given a <var> and a state.
;;;
(defun vl-from-value (var state)
  (vl-from-varname-and-value (variable-name var)
			     (cdr (assoc (variable-name var)
					 (state-qvalues state)))
			     state))


;   3.  Generate equations from values and cvtuples.

; Make equations from values at the current state.
;   Notice that vls are now (<variable> <lmark>), where both are structures, not names.

;;; Return the typename of a constraint as an atom.
;;; This is the same as constraint-typename for all but the M constraint.
;;; This should probably be moved to structures.lisp.
;;; Added BKay 28Apr92
;;;
(defun constraint-atomic-typename (con)
  (let ((type (constraint-typename con)))
    (if (atom type) type (first type))))


(defun make-eqns-from-values (state)
  (let ((vls (all-vls-from-values state)))
    (dolist (con (qde-constraints (state-qde state)))
      (when (constraint-active-p con)
	(trace-constraint-for-indexing state con)
	(let ((vltuple (mapcar #'(lambda (v) (assoc v vls :test #'equal))
			       (constraint-variables con))))
	  (case (constraint-atomic-typename con)
	    (add      (index-equations (apply #'add-eqns vltuple)))
	    (mult     (index-equations (apply #'mult-eqns vltuple)))
	    (minus    (index-equations (apply #'minus-eqns vltuple)))
	    (equal    (index-equations (apply #'equal-eqns vltuple)))
	    ;; Next 2 clauses added BKay 15Jun92
	    (abs-value (index-equations (apply #'abs-value-eqns vltuple)))
	    (sum      (index-equations (sum-eqns (constraint-name con) vltuple)))
	    (sum-zero (index-equations (apply #'sum-zero-eqns vltuple)))
	    (d/dt     (index-equations (make-d/dt-eqns con state)))
	    ;; Added BKay 28Apr92
	    (M        (index-equations (make-MvM-MVT-eqns con state))
		      (index-equations (fn-MvM-eqns (constraint-name con)
						    vltuple)))
	    ((M+ M-)  (index-equations (make-M-MVT-eqns con state))
	              (index-equations (apply #'fn-eqns
					      (cons (constraint-name con)
						    vltuple))))
	    ((S+ S-)  (index-equations (make-S-MVT-eqns con state))
	              (if (within-S-curve con state)
			  (index-equations
			   (apply #'fn-eqns (cons (constraint-name con)
						  vltuple)))))

	    ((zero-std unreachable)  nil)
	    ((constant increasing decreasing non-constant non-increasing non-decreasing)
	     nil)
	    (t        (format *QSIM-trace* "~%Can't MAKE-EQNS-FROM-VALUES ~a." con))))))))

; Make equations from corresponding values at the initial state.

(defun make-eqns-from-cvtuples (state)
  (dolist (con (qde-constraints (state-qde state)))
    (when (constraint-active-p con)
      (trace-constraint-for-indexing state con)
      (let ((cvalues (cdr (assoc con (state-cvalues state))))
	    (params  (constraint-variables con)))
	(case (constraint-atomic-typename con)
	  (add      (dolist (cvtuple cvalues)
		      (index-equations 
			(apply #'add-eqns (cv-vls params cvtuple)))))
	  (mult     (dolist (cvtuple cvalues)
		      (index-equations (apply #'mult-eqns (cv-vls params cvtuple)))))
	  (minus    (dolist (cvtuple cvalues)
		      (index-equations (apply #'minus-eqns (cv-vls params cvtuple)))))
	  (equal    (dolist (cvtuple cvalues)
		      (index-equations (apply #'equal-eqns (cv-vls params cvtuple)))))
	  ;; Next 2 clauses added BKay 15Jun92
	  (abs-value (dolist (cvtuple cvalues)
		       (index-equations (apply #'abs-value-eqns (cv-vls params cvtuple)))))
	  (sum      (dolist (cvtuple cvalues)
		      (index-equations (sum-eqns (constraint-name con)
						 (cv-vls params cvtuple)))))
	  (sum-zero (dolist (cvtuple cvalues)
		      (index-equations (apply #'sum-zero-eqns (cv-vls params cvtuple)))))
	  (d/dt     nil)
	  ;; Added BKay 28Apr92
	  (M        (dolist (cvtuple cvalues)
		      (index-equations
			(fn-MvM-eqns (constraint-name con)
				     (cv-vls params cvtuple)))))
	  ((M+ M-)  (dolist (cvtuple cvalues)
		      (index-equations
			(apply #'fn-eqns
			       (cons (constraint-name con)
				     (cv-vls params cvtuple))))))
	  ((S+ S-)  (if (within-S-curve con state)
			(dolist (cvtuple cvalues)
			  (index-equations
			    (apply #'fn-eqns
				   (cons (constraint-name con)
					 (cv-vls params cvtuple)))))))

	  ((zero-std unreachable)  nil)
	  ((constant increasing decreasing non-constant non-increasing non-decreasing)
	   nil)
	  (t        (format *QSIM-trace* "`%Can't MAKE-EQNS-FROM-CVTUPLES ~a." con)))))))


(defun cv-vls (params cvtuple)			; make vls with symbols
  (mapcar #'(lambda (p lm) (list p lm))
	  params
	  cvtuple))

; => Create equations only for *new* cvtuples, since last time-point.

; => Provide an optional way to scan back to generate previous MVT equations.


; Equations are relations among VLPs:  (variable value-description) pairs.
; A value-description is a landmark, (at pstate), or (at istate).
;   ((ADD x y z) (x1 y1 z1) ... )  =>  (= (z z1) (+ (x x1) (y y1))) ...
;   ((SUM-ZERO x1 x2 ... xn)  ...  (p1 p2 ... pn)  ...  )
;           =>   (= (x1 p1) (- 0 (SUM (x2 p2) ... (xn pn))))

; These equations are given VLPs.  Exclude tuples with all zeros or any infinity.

(defun bad-cvtuple (&rest tuple)
  (or (every #'(lambda (vl) (eql vl *zero-lmark*)) tuple)
      (some  #'(lambda (vl) (member vl (list *minf-lmark* *inf-lmark*))) tuple)))

(defun add-eqns (a b c)
  (unless (bad-cvtuple a b c)
    (eqn-equivalents `(= ,c (+ ,a ,b)))))

(defun mult-eqns (a b c)
  (unless (bad-cvtuple a b c)
    ;; If we have a form like (mult a b c) then we know it's a
    ;; squared relation.  Eqn-equivalents requires that the RHS of an
    ;; equation be a three-tuple of (op arg1 arg2), so we stick "a" on
    ;; twice.
    (nconc (when (equal a b)
	     (eqn-equivalents `(= ,c (square ,a ,a))))
	   (eqn-equivalents `(= ,c (* ,a ,b))))))

(defun minus-eqns (a b)
  (unless (bad-cvtuple a b)
    (eqn-equivalents `(= ,b (- 0 ,a)))))

(defun equal-eqns (a b)
  (unless (bad-cvtuple a b)
    (eqn-equivalents `(= ,a ,b))))

(defun abs-value-eqns (x y)
  (unless (bad-cvtuple x y)
    (eqn-equivalents `(= ,y (abs ,x)))))

(defun sum-zero-eqns (&rest vltuple)
  (unless (apply #'bad-cvtuple vltuple)
    (eqn-equivalents (cons 'SUM-ZERO vltuple))))

;;; The sum constraint name looks like:
;;; ((SUM s1 ... sn) x1 ... xn y)
;;; where si=+,0,- and the semantics are
;;; y=sum[si*xi]
(defun sum-eqns (con-name vltuple)
  (unless (apply #'bad-cvtuple vltuple)
    (eqn-equivalents (list 'SUM (second (first con-name)) vltuple))))

(defun fn-eqns (fn p q)
  (unless (or (member (vl-lmark p) (list *minf-lmark* *inf-lmark*))
	      (member (vl-lmark q) (list *minf-lmark* *inf-lmark*)))
    (eqn-equivalents `(= ,q (function ,fn ,p)))))

;;; Inputs:  fn       - A list of the form ((M s1 ... sn-1) v1 ... vn y)
;;;          vl-tuple - A list of vls (v1, ..., vn, y).
;;; Returns: a list containing a single equation of the form 
;;;          (= y (M-function fn . tuples-other-than-y))
;;; Note:    M functions do not assert all their "inverses".
;;; Added BKay 28Apr92
;;;
(defun fn-MvM-eqns (fn vl-tuple)
  (unless (some #'(lambda (vl) (member (vl-lmark vl)
				       (list *minf-lmark* *inf-lmark*)))
		vl-tuple)
    `((= ,(car (last vl-tuple))
         (M-function ,fn ,.(subseq vl-tuple 0 (1- (length vl-tuple))))))))

; These equations are given variables and value descriptions, separately.

; Define d(a,b) = b-a.  (The relation d(a,b)=-d(b,a) follows automatically.)

(defun d-eqns (x x1 x2)
  (let ((a   (list x x1))
        (b   (list x x2))
        (dab (list x (list 'd x1 x2))))
    (eqn-equivalents  `(= ,dab (- ,b ,a)) )))

; Generate the equations representing x(t1) = (a b), defining the term (x (at <state>))

(defun at-t-eqns (x state a b)
  (nconc 
    (unless (eql a *MINF-LMARK*)
      (eqn-equivalents `(< (,x ,a) (,x (at ,state)))))
    (unless (eql b *INF-LMARK*)
      (eqn-equivalents `(< (,x (at ,state)) (,x ,b))))))

; To declare that two variables have the same value.  Used primarily across transitions.

(defun transition-eqns (v1 lm1 v2 lm2)
  (eqn-equivalents `(= (,v1 ,lm1) (,v2 ,lm2))))

;   4.  Generate equations using the Mean Value Theorem.
;
; The Mean Value Theorem tells us that, given a constraint (d/dt L R),
; two time-points t0 and t1, and the values a=L(t0) and b=L(t1),
; that there is a point t* in (t0,t1) such that R(t*) = (b-a)/(t1-t0).
;
; This gives us the equations for propagation:
;   d(a,b)   = d(t0,t1) * R(t*)
;   d(t0,t1) = d(a,b) / R(t*)
;   R(t*)    = d(a,b) / d(t0,t1)


; Create MVT equations from (d/dt x y)

(defun make-d/dt-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
         (params  (constraint-variables con))
         (x (car params))
         (y (cadr params)))
    (unless (null gfather)
      (d/dt-eqns x
                 (cadr (vl-from-value x gfather))   ; x(t1)
                 (cadr (vl-from-value x state))     ; x(t2)
                 y
                 (cadr (vl-from-value y gfather))   ; y(t1)
                 (cadr (vl-from-value y state))     ; y(t2)
		 (alookup 'time (qde-var-alist (state-qde state)))
                 (qval-qmag (cdar (state-qvalues gfather)))  ; t1
                 (qval-qmag (cdar (state-qvalues state)))    ; t2
		 ;; The qmags of the grandfather, father, and current states. BKay 17Mar90
		 (qval-qmag (alookup (variable-name y) (state-qvalues gfather)))
		 (qval-qmag (alookup (variable-name y) (state-qvalues father)))
		 (qval-qmag (alookup (variable-name y) (state-qvalues state)))
		 father))))

(defun d/dt-eqns (x x1 x2 y y1 y2 time t1 t2 y1mag y12mag y2mag father)
  (nconc 
    (d-eqns x x1 x2)
    (d-eqns time t1 t2)
    (eqn-equivalents `(= (,x (d ,x1 ,x2)) (,y (defint ,y ,t1 ,t2))))
    ;; This equation only works if we know y, therefore, we generate only
    ;; one equivalent equation.  This is because y is existentially
    ;; (rather than universally) quantified, so we can only USE its value,
    ;; but we aren't allowed to compute it. BKay 5Aug92
    (list `(= (,y (defint ,y ,t1 ,t2)) (* (,time (d ,t1 ,t2)) (,y (at ,father))))
	  `(= (,time (d ,t1 ,t2)) (/ (,y (defint ,y ,t1 ,t2)) (,y (at ,father)))))

    ;; Span-of only makes sense if the last interval was monotonic wrt y.
    ;; This might not be the case if y is an ignore-qdirs variable.
    ;; In particular, if qmag(y@t1) = qmag(y@t2), but qmag(y@(t1 t2)) is
    ;; different then we know that y@(t1 t2) is not in the span of y from t1 to t2
    ;; so this equation should not be included.  BKay 17Mar90
    (if (and (equal y1mag y2mag) (not (equal y1mag y12mag)))
	nil
	(list `(= (,y (at ,father)) (span-of (,y ,y1) (,y ,y2)))))))

; Direct-state-predecessor takes only predecessors under normal qsim prediction.

(defun direct-state-predecessor (state)
  (let ((just (state-justification state)))
    (cond ((member (car just) '(successor-of unique-successor-of 
				 one-of-several-successors-of))
	   (cadr just)))))


; When y=f(x) for f in M+/M-/S+/S-, the Mean Value Theorem implies relations
; between x(t1), x(t2), y(t1), y(t2) and f'.

(defun make-M-MVT-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
         (params (constraint-variables con))
         (x (car params))
         (y (cadr params)))
    (unless (null gfather)
      (M-eqns x
              (cadr (vl-from-value x gfather))   ; x(t1)
              (cadr (vl-from-value x state))     ; x(t2)
              y
              (cadr (vl-from-value y gfather))   ; y(t1)
              (cadr (vl-from-value y state))     ; y(t2)
              con))))

(defun make-S-MVT-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
         (params (constraint-variables con))
         (x (car params))
         (y (cadr params)))
    (unless (or (null gfather)
                (not (within-S-curve con state gfather)))
      (M-eqns x
              (cadr (vl-from-value x gfather))   ; x(t1)
              (cadr (vl-from-value x state))     ; x(t2)
              y
              (cadr (vl-from-value y gfather))   ; y(t1)
              (cadr (vl-from-value y state))     ; y(t2)
              con))))

(defun within-S-curve (con state1 &optional state2)	; (S+ x y (a b) (c d))
  ; return T if state(s) are within the monotonic portion of the S+/S-.
  (let* ((x (car (constraint-variables con)))
	 (a (caar (constraint-bend-points con)))
	 (b (caadr (constraint-bend-points con)))
	 (qspace (cdr (assoc (variable-name x) (state-qspaces state1))))
	 (val1 (qval-qmag (cdr (assoc (variable-name x) (state-qvalues state1)))))
	 (val2 (if state2
		   (qval-qmag (cdr (assoc (variable-name x) (state-qvalues state2)))))))
    (and (member (qmag-order val1 a qspace) '(+ 0))
	 (member (qmag-order val1 b qspace) '(- 0))
	 (or (null val2)
	     (and (member (qmag-order val2 a qspace) '(+ 0))
		  (member (qmag-order val2 b qspace) '(- 0)))))))


;;; Multivariate M Mean Value Theorem equations.
;;; This produces a set of n equations for an n-ary M constraint
;;; (((M s1 .. sn) v1 ... vn y)) of the form
;;; dy/dvi = d(y(t1),y(t2))/d(vi(t1),vi(t2))
;;; Bear in mind that these equations are only useful if there is a "slope"
;;; clause in the m-envelopes QDE clause.
;;;
(defun make-MvM-MVT-eqns (con state)
  (let* ((father (direct-state-predecessor state))
         (gfather (if father (direct-state-predecessor father)))
         (params (constraint-variables con))
         (y (car (last params))))
    (unless (null gfather)
      (loop for param in params
	    with y-at-gfather  = (cadr (vl-from-value y gfather))
	    with y-at-father   = (cadr (vl-from-value y father))
	    when (not (eq param y))
	    nconc (M-eqns param 
			  (cadr (vl-from-value param gfather))   ; vi(t1)
			  (cadr (vl-from-value param state))     ; vi(t2)
			  y
			  y-at-gfather 				 ; y(t1)
			  y-at-father                     	 ; y(t2)
			  con)))))


(defun M-eqns (x x1 x2 y y1 y2 con)
  (let ((slope (list 'slope (constraint-name con))))
    (nconc
      (d-eqns x x1 x2)
      (d-eqns y y1 y2)
      (eqn-equivalents `(= (,y (d ,y1 ,y2)) (* (,x (d ,x1 ,x2)) ,slope))))))

;   5.  Create all equivalent variants of a given equation.

(defun eqn-equivalents (eqn)
  (trace-eqn-creation eqn)
  (remove-if #'(lambda (eqn) (or (eql (cadr eqn) 0)
				 (member (car (cadr eqn)) '(span-of slope curvature abs))))
	     (ecase (car eqn)
	       ((> >= < <=)
		(list eqn
		      (list (sublis '((> . <) (>= . <=) (< . >) (<= . >=)) (car eqn))
			    (caddr eqn)
			    (cadr eqn))))
	       ;; eqn = (SUM (s1 ... sn) vltuple)
	       (sum      (let ((signs (append (second eqn) (list '+)))
			       (vltuple (third eqn)))
			   (format *qsim-report* "~%Warning: SUM Q2 constraint has not been tested.  Report any errors to qsim-bugs@cs.utexas.edu")
			   (loop for sign   in signs
				 for lhs    in vltuple
				 for argpos from 0 upto (length signs)
				 collect
				 `(= ,lhs ,(case sign
					     (0   0)
					     (+   (sum-vls-in-sum argpos
								  signs
								  vltuple))
					     (-   `(- ,(sum-vls-in-sum argpos
							               signs
							               vltuple))))))))
	       (sum-zero (mapcar #'(lambda (var)
				     `(= ,var (- 0 (SUM . ,(remove var (cdr eqn))))))
				 (cdr eqn)))
	       (= (let ((LHS (cadr eqn))	; (= LHS (op A B))
			(RHS (caddr eqn))
			(op  (car (caddr eqn)))
			(A   (cadr (caddr eqn)))
			(B   (caddr (caddr eqn))))
		    (case op
		      (+ (list eqn
			       `(= ,A (- ,LHS ,B))
			       `(= ,B (- ,LHS ,A))))
		      (- (list eqn
			       `(= ,A (+ ,LHS ,B))
			       `(= ,B (- ,A ,LHS))))
		      (* (list eqn
			       `(= ,A (/ ,LHS ,B))
			       `(= ,B (/ ,LHS ,A))))
		      (/ (list eqn
			       `(= ,A (* ,LHS ,B))
			       `(= ,B (/ ,A ,LHS))))
		      (function (list eqn
				      `(= ,B (inverse ,A ,LHS))))
		      (square (list eqn
				    `(= ,a (sqrt ,a ,LHS))))
		      (t (list eqn
			       `(= ,RHS ,LHS)))
		      )))
	       )))


;;; Sum up the elements other than the one at argpos using the
;;; rule that each element is equal to si*xi.
;;; Inputs:  argpos  - The position of the arg to exclude.
;;;          signs   - A list of signs.
;;;          vltuple - The tuple of vls.
;;; Returns: A list of the form (SUM si*xi ...) with i=argpos
;;;          excluded.
;;;
(defun sum-vls-in-sum (argpos signs vltuple)
  `(SUM .
     ,(loop for sign in signs
            for vl   in vltuple
            for pos  from 0 upto (length vltuple)
            unless (= pos argpos)
            collect (case sign
		      (0   0)
		      (+   vl)
		      (-   `(- ,vl))))))

; This will create equations with (V (AT istate)) on the lhs.


;   6.  Index equations.
;
;   *EQN-INDEX* ::=  alist of (<var> . alist of (<lmark> . <equations>))
;        indexes the equations whose RHS includes <lmark>.
;
; Search through a list of expressions, and index them under every <vl> appearing in them.

(defun index-equations (eqn-list)
  (dolist (eqn eqn-list) (index-equation eqn eqn)))

(defun index-equation (eqn subtree)
  (cond ((numberp subtree) nil)
	((atom subtree) (error "Bad atom in expression:  ~a."  subtree))
	((vlp subtree) (store-into-eqn-index subtree eqn))
	(t (ecase (car subtree)
	     ((+ - * / span-of sum square)
	      (dolist (arg (cdr subtree))
		(index-equation eqn arg)))
	     ((= > >= < <= function inverse sqrt)
	      (index-equation eqn (third subtree)))
	     ;; Added BKay 5May92
	     ((M-function)
	      (dolist (arg (cddr subtree))
		(index-equation eqn arg)))
	     ((abs)
	      (index-equation eqn (second subtree)))
	     ((slope) nil)))))

; Store a new equation on *eqn-index*.  Create indexing
; structure as needed.  Do nothing if <eqn> is already there.

;;; This seems identical in function to the old call.  BKay 27May92
(defun store-into-eqn-index (vl eqn)
  (declare (special *eqn-index*))
  (let ((var-entry (or (assoc (vl-var vl) *eqn-index*)
		       (car (push (list (vl-var vl)) *eqn-index*)))))
    (let ((vl-entry (assoc (vl-lmark vl) (cdr var-entry) :test #'equal)))
      (when (null vl-entry)
	(setq vl-entry (list (vl-lmark vl)))
	(nconc var-entry (list vl-entry)))
      ;; make sure that all equal eqns are eq!
      (when (not (member eqn (cdr vl-entry) :test #'equal))
	(nconc vl-entry (list eqn))
	(trace-eqn-index-entry vl eqn))
      eqn)))


#| old code
(defun store-into-eqn-index (vl eqn)
  (declare (special *eqn-index*))
  (let ((var-entry nil)
	(vl-entry nil))
    (cond ((setq var-entry (assoc (vl-var vl) *eqn-index*)))
	  (t (setq var-entry (list (vl-var vl)))
	     (pushnew var-entry *eqn-index* :test #'equal)))
    (cond ((setq vl-entry (assoc (vl-lmark vl) (cdr var-entry) :test #'equal)))
	  (t (setq vl-entry (list (vl-lmark vl)))
	     (nconc var-entry (list vl-entry))))
    (cond ((member eqn (cdr vl-entry) :test #'equal))
	  (t (nconc vl-entry (list eqn))))
    (trace-eqn-index-entry vl eqn)
    eqn))
|#

; Trace functions.

(defun trace-eqn-index-entry (vl eqn)
  (if *trace-eqn-indexing*
      (format *QSIM-Trace* "~%  Indexing ~a  ->  ~a." vl eqn)))

(defun trace-constraint-for-indexing (state con)
  (if *trace-eqn-creation*
      (format *QSIM-Trace* "~%Creating equations at ~a from ~a." state con)))

(defun trace-eqn-creation (eqn)
  (if *trace-eqn-creation*
      (format *QSIM-trace* "~% Created equation ~a." eqn)))
