;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-
;;; $Id: time-plot.lisp,v 1.15 1992/07/02 13:14:42 bert Exp $

(in-package 'QSIM)

;;; This file containst the functions used for drawing the time plots of
;;; qsim behaviors.  Calling order looks like this: [not all functions
;;; in this file]

;;; time-plot-behavior
;;;     qplot-new-behavior
;;;     qplot-label
;;;     display-behavior
;;;         allocate-h-axis
;;;         plot-parameter
;;;             allocate-v-axis
;;;             get-box-label
;;;             must-retrieve
;;;             qsymbol-vloc
;;;             qplot-frame
;;;                 plot-v-axis
;;;                 plot-h-axis
;;;                 get-q2-ranges
;;;         plot-state-tree
;;;     qplot-end-display




; Modified 1/24/90 D.B.,  4 Oct 90, 25 Nov 91 Mallory

(defun TIME-PLOT-BEHAVIOR (initials refpoints behaviors layout qspaces
			   bmax btotal bnum trees 
			   &key (display-block (find-display-block initials)))
  (let* ((truncate? (display-block-truncate? display-block))
	 (beh (if truncate? (truncate-behavior (nth (1- bnum) behaviors)
						truncate?)
		  (nth (1- bnum) behaviors))))
    (qplot-new-behavior)
    (qplot-label (nth (1- bnum) behaviors) bnum btotal)
    (display-behavior refpoints beh layout qspaces)
    (when truncate? (format *Qsim-Report* 
			    "The behavior displayed has been truncated at ~a"
			    truncate?))
    (let ((*plot-small-state-tree* trees))
      (plot-small-state-tree initials bmax btotal bnum))
    (qplot-end-display)))

; A behavior to be plotted is specified in two parts:
;  - reference points:  an alist of (name state)
;  - behavior:  a list of states

(defun DISPLAY-BEHAVIOR (ref beh layout &optional qspaces)
  (declare (ignore qspaces))
  (setf *behavior* beh)
  (do ((xalloc (allocate-h-axis ref beh *xsize*))
       (rows layout (cdr rows))
       (ypos tmargin (+ ypos *ysize* ysep)))
      ((null rows))
    (do ((varnames (car rows) (cdr varnames))
         (varname nil)
         (xpos lmargin (+ xpos *xsize* xsep)))
        ((null varnames))
      (setq varname (car varnames))
      (if varname
          (plot-parameter varname ref beh xpos (+ ypos *plot-offset*) *xsize* *ysize* xalloc)))))


;;; PLOT-PARAMETER is the function that draws one of the little boxes in
;;; the time display, the label below it, and everything that goes
;;; inside it.  Qval will be NIL for some states when behavior goes thru
;;; multiple QDEs.

;;; Modified so that it will handle aggregate-intervals.
;;; 15 May 1991   DJC
;;; Modified so that it can print intervals explicitly
;;; 04 Jun 1991   PF

(defun PLOT-PARAMETER
       (param refpoints behavior xpos ypos xsize ysize xalloc &key (label))
  (loop with yalloc = (allocate-v-axis param behavior ysize)
	with behavior-xalloc = (member (first behavior) xalloc :key #'car)
        with param-found-in-behavior-p	; Flag gets set T if parameter appears
	= (plot-refpoints param refpoints	;  in either refpoints or behavior.
			  xpos ypos xalloc yalloc)
        for ostate = nil then state
        for state in behavior
	;; state--x-offset added, PF 03/06/91
	for state--x-offset in behavior-xalloc
	;; for x-offset = (lookup state xalloc)
	for x-offset = (second state--x-offset)
        for count from 0
        for qval = (if (agg-interval-p state)
		       (first-qval-in-history param state)
		       (qval param state))
        for ox = nil then x		; ox,oy inherit value of x,y
        for oy = nil then y		;  from previous loop cycle.
        for x = (when qval (+ xpos x-offset))
        for y = (when qval
                  (+ ypos (qsymbol-vloc qval yalloc :param param
                                        :n count :behavior behavior)))
	;; Variables needed to display an interval as a vertical bar
	for qmag = (when qval (qval-qmag qval))
	for interval-p = (when qmag (qmag-interval-p qmag))
	for y1 = (when interval-p (+ ypos (must-retrieve-lmark (first  qmag) yalloc)))
	for y2 = (when interval-p (+ y 7))
	for y3 = (when interval-p (- y 7))
	for y4 = (when interval-p (+ ypos (must-retrieve-lmark (second qmag) yalloc)))
        ;; Keep flag of whether the parameter occurs in this behavior at all.
	do (setf param-found-in-behavior-p
		 (or param-found-in-behavior-p qval))
        when (and (numberp ox)(numberp x))	; Connect states with dots 
	do (if (perturbed-p ostate)	;  or arrow.
	       (perturb-link ox oy x y)
	       (connect-with-dots ox oy x y))
        when qval
	do (qplot-point-label x y param state behavior)
	;; Plot the vertical bar
	(when (and interval-p *plot-intervals*)
	  (qplot-line x y1 x y2 :dashed t :dash-pattern '(1 3))
	  (qplot-line x y3 x y4 :dashed t :dash-pattern '(1 3)))
	(qplot-symbol x y (qdir qval))
	when (and param-found-in-behavior-p
		  (agg-interval-p state))
	  do (do* ((num-times (1- (max-history-size state)) (1- num-times))
		   (qvals-in-hist (qvals-in-history param state))
		   (qvals (if (cdr qvals-in-hist)
			      (cdr qvals-in-hist)
			      qvals-in-hist)
			  (if (cdr qvals)
			      (cdr qvals)
			      qvals))
		   (qval (car qvals) (car qvals)))
		  ((= num-times 0))
	       (setq ox x)
	       (setq oy y)
	       (setq x (round (+ x Hshare)))
	       (setq y (+ ypos (qsymbol-vloc qval yalloc :param param
					     :n count :behavior behavior)))
	       (when (and (numberp ox) (numberp x))
		 (connect-with-dots ox oy x y))
	       (qplot-symbol x y (qdir qval)))
        finally (when param-found-in-behavior-p
                  (qplot-frame param refpoints behavior xpos ypos
                               xsize ysize xalloc yalloc :label
                               (or label (get-box-label param behavior))))))

;(defun PLOT-PARAMETER
;       (param refpoints behavior xpos ypos xsize ysize xalloc &key (label))
;  (loop with yalloc = (allocate-v-axis param behavior ysize)
;        with param-found-in-behavior-p		; Flag gets set T if parameter appears
;	  = (plot-refpoints param refpoints	;  in either refpoints or behavior.
;			    xpos ypos xalloc yalloc)
;        for ostate = nil then state
;        for state in behavior
;	for x-offset = (lookup state xalloc)
;        for count from 0
;        for qval = (if (agg-interval-p state)
;		       (first-qval-in-history param state)
;		       (qval param state))
;        for ox = nil then x		  ; ox,oy inherit value of x,y
;        for oy = nil then y               ;  from previous loop cycle.
;        for x = (when qval (+ xpos x-offset))
;        for y = (when qval
;                  (+ ypos (qsymbol-vloc qval yalloc :param param
;                                        :n count :behavior behavior)))
;        ;; Keep flag of whether the parameter occurs in this behavior at all.
;	do (setf param-found-in-behavior-p
;		 (or param-found-in-behavior-p qval))
;        when (and (numberp ox)(numberp x))	; Connect states with dots 
;          do (if (perturbed-p ostate)		;  or arrow.
;                 (perturb-link ox oy x y)
;                 (connect-with-dots ox oy x y))
;        when qval
;          do (qplot-point-label x y param state behavior)
;	     (qplot-symbol x y (qdir qval))
;	when (agg-interval-p state)
;	  do (do* ((num-times (1- (max-history-size state)) (1- num-times))
;		   (qvals-in-hist (qvals-in-history param state))
;		   (qvals (if (cdr qvals-in-hist)
;			      (cdr qvals-in-hist)
;			      qvals-in-hist)
;			  (if (cdr qvals)
;			      (cdr qvals)
;			      qvals))
;		   (qval (car qvals) (car qvals)))
;		  ((= num-times 0))
;	       (setq ox x)
;	       (setq oy y)
;	       (setq x (round (+ x Hshare)))
;	       (setq y (+ ypos (qsymbol-vloc qval yalloc :param param
;					     :n count :behavior behavior)))
;	       (when (and (numberp ox) (numberp x))
;		 (connect-with-dots ox oy x y))
;	       (qplot-symbol x y (qdir qval)))
;        finally (when param-found-in-behavior-p
;                  (qplot-frame param refpoints behavior xpos ypos
;                               xsize ysize xalloc yalloc :label
;                               (or label (get-box-label param behavior))))))



(defun QPLOT-POINT-LABEL (x y param s behavior)  ; For putting information about a plotted point by that point.
  (declare (ignore x y param  s behavior)))      ; D.B. 5/90


(defun PLOT-REFPOINTS (param refpoints xpos ypos xalloc yalloc)
  (loop for (symbol state) in refpoints		; plot reference points
	with non-empty-behavior = nil		; flag if this param is found in any refpoint
	for qval = (alookup param (state-qvalues state))
	for x = (when qval
		  (+ xpos (must-retrieve state xalloc)))
	for y = (when qval
		  (+ ypos (qsymbol-vloc qval yalloc :state state)))
	when qval
	  do (setq non-empty-behavior t)
	     (qplot-symbol x y (qdir qval))
	finally (return non-empty-behavior)))


; The vertical axis must be laid out separately for each plot, allocating
; a location to each landmark in the important section of the qspace.

; The allocation is an alist of (landmark coord) where the TOP of the plot
; has coord=0.

; Eventually this will call the specific v-axis allocation algorithm
; appropriate to the situation. For example, numerical scaling.

(defun ALLOCATE-V-AXIS (param behavior ysize)
  (let ((qspace (qspace-from-behavior param behavior)))
    (allocate-v-axis-with-qualitative-scaling qspace ysize)))



(defun GET-BOX-LABEL (param behavior)
  (or (box-label param (car behavior))
      (do ((L behavior (cdr L)))
          ((null L) nil)
        (cond ((and (eql (car (filtered-successor-slot-contents (car L))) 'transition-identity)
                    (cdr L))
               (return (get-box-label param (cdr L))))))
      param))



(defun MUST-RETRIEVE (key alist)                ; lookup with error on failure.
  (let ((pair (assoc key alist)))
    (cond (pair (second pair))
          (t (error "No value for ~a in ~a." key alist)))))



(defun QSYMBOL-VLOC (qval yalloc &key (param nil) n behavior state)
  (declare (ignore state))
  (let ((qmag (qval-qmag qval)))
    (cond ((null qmag) 0.)
	  ((qmag-point-p qmag) (must-retrieve-lmark qmag yalloc))
	  ((member nil qmag) 0.)
	  (t (guess-vloc qval yalloc param n behavior)))))


;;; QPLOT-FRAME draws the box for each parameter plot, labels it, and
;;; draws in the horizontal and vertical axis.  Its only called if the
;;; parameter has been founds somewhere in the behavior.
 
(defun QPLOT-FRAME (param refpoints behavior xpos ypos xsize ysize xalloc yalloc &key (label))
  (qplot-box xpos ypos xsize (- ysize labelspace) :thickness *border-width*)
  (plot-v-axis yalloc (get-q2-ranges param behavior) xpos ypos)
  (plot-h-axis refpoints behavior xalloc xpos ypos)
  (qplot-box-label xpos (+ ypos ysize) label))



; Plot horizontal and vertical axis markings.
;   yalloc = alist of (landmark vertical-position)
;   ranges = alist of (landmark (lo hi))

;Modified 10/29/90 Dan B.
(defun PLOT-V-AXIS (yalloc ranges xloc yloc &key qvar state)
  (let ((xtick   (+ xloc vtickpoint))
        (xlm     (+ xloc vlabelpoint))
        (zeropos (lookup *zero-lmark* yalloc)))
    (if zeropos
        (qplot-hline (+ xloc (* lshare Hshare))
                     (+ yloc zeropos)
                     (- *xsize* (* Hshare (+ lshare rshare)))))
    (dolist (pair yalloc)
      (let ((y-offset (second pair))
	    (lmark (first pair)))
	(qplot-hline xtick (+ yloc y-offset) tick-length)
	(qplot-hline xloc  (+ yloc y-offset) tick-length)
	(qplot-axis-label (+ 2 xlm) (+ yloc y-offset 5) (get-axis-label lmark ranges :qvar qvar))
	(qplot-axis-label-elaboration (+ 2 xlm) (+ yloc y-offset 5) lmark qvar state)))))

; For user to customize own code with. Dan B. 10/29/90.
(defun qplot-axis-label-elaboration (x y lmark param state)
  (declare (ignore x y lmark param state) ) nil)
  

; xalloc = alist of (state x-offset)

;(defun PLOT-H-AXIS (ref beh xalloc xloc yloc)
;  (let ((ytick (+ yloc htickpoint)))
;    (mapc #'(lambda (pair)			; Plot Ref points
;              (plot-h-axis-point (car pair) (second pair)
;		  (+ xloc (lookup (second pair) xalloc))
;		  ytick))
;          ref)
;    (mapc #'(lambda (state)			; Plot states at timepoints
;              (let* ((time-qmag (qmag (state-time state))))
;                (when (atom time-qmag)
;		  (plot-h-axis-point (lmark-name time-qmag) state
;		      (+ xloc (lookup state xalloc))
;		      ytick))))
;          beh)))

;;; Modified to handle aggregate-intervals
;;; 15 May 1991  DJC

(defun PLOT-H-AXIS (ref beh xalloc xloc yloc)
  (let ((ytick (+ yloc htickpoint)))
    (mapc #'(lambda (pair)			; Plot Ref points
              (plot-h-axis-point (car pair) (second pair)
		  (+ xloc (lookup (second pair) xalloc))
		  ytick nil))
          ref)
    (do* ((states beh (cdr states))
	  (allocs xalloc (cdr allocs))
	  (alloc (car allocs) (car allocs))
	  (offset (second alloc) (second alloc))
	  (state (car states) (car states))
	  (time-index 0)
	  (transition-p (transition-follows state (cadr states))
			(transition-follows state (cadr states))))
	 ((null states) t)
      (unless (agg-interval-p state)
	(let* ((time-qmag (qmag (state-time state))))
	  (if (qmag-point-p time-qmag)
	      (let ((h-axis-label (h-axis-label beh time-qmag)))
		(cond (*envisionment*
		       (plot-h-axis-point (format nil "T~a" time-index) state
					  (+ xloc offset)
					  ;(+ xloc (lookup state xalloc))
					  ytick transition-p)
		       (unless transition-p (incf time-index)))
		      (t (plot-h-axis-point h-axis-label state
					    (+ xloc (lookup state xalloc)) ytick
					    transition-p))))))))))
  



(defun h-axis-label (beh time-qmag) beh (lmark-name time-qmag)) ;Hook for future improvement.

                ; factor on label position


(defun GET-Q2-RANGES (varname behavior)
  (cdr (find varname (state-bindings (car (last behavior)))
             :key #'(lambda (item) (if (symbolp (car item))
                                       (car item)
                                       (variable-name (car item)))))))

(defun QPLOT-LABEL (behavior bnum btotal &key (xpos lmargin)(ypos 25)(yadv 15))
  (let* ((*print-pretty* nil)
	 (*detailed-printing* nil)) 
    (loop for fn in  *Functions-for-label*
	      do (incf ypos (* yadv (funcall fn xpos ypos yadv
					      behavior bnum btotal))))))


(defun QPRINT-FINAL-STATE (xpos ypos yadv behavior bnum btotal)
  (declare (ignore yadv bnum btotal))
  (let ((final (car (last behavior))))
    (qplot-string (format nil "Final state: ~a, ~a, ~a."
			  ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
                          (state-status final) (filtered-successor-slot-contents final)
                          (state-time-label final))
                  xpos ypos)
    1))


(defun QPRINT-LIST-BEHAVIOR (xpos ypos yadv behavior bnum btotal)
  (declare (ignore yadv))
  (qplot-string (format nil "Behavior ~a of ~a:    ~a."
			bnum btotal (mapcar #'(lambda (state)
						;; Modified DJC 06/3/91
						(if (aggregate-interval-p state) 
						    (aggregate-interval-name state)
						    (state-name state)))
					    behavior))
		xpos ypos)
  1)


(defun QPRINT-STRUCTURE (xpos ypos yadv behavior bnum btotal)
  (declare (ignore yadv bnum btotal))
  (let* ((qde (state-qde (car behavior)))
	(text (or (car (qde-text qde))
		  (qde-name qde))))
    
    (qplot-string (format nil "Structure: ~a." text)
                  xpos ypos))
  1)


(defun QPRINT-JUSTIFICATION (xpos ypos yadv behavior bnum btotal)
  ;; If the initial-state was incomplete, the state-text (ie, the text describing
  ;; the initialization) will be on the (incomplete) parent of *initial-state*.
  (declare (ignore yadv bnum btotal))
  (let ((justification (or (state-text (car behavior))
			   (state-justification (car behavior)))))
    (when (and (listp justification)
               (eq (car justification) 'one-of-several-completions-of)  
               (typep (second justification) 'state)
               (state-text (second justification)))
      (setq  justification (state-text (second justification))))
    (qplot-string (format nil "Initialization: ~a  (~a)"
                          justification (state-name (car behavior)))
                  xpos ypos))
  1)
    

(defun QPRINT-ACC-STATS (xpos ypos yadv behavior bnum btotal)
  (declare (ignore yadv bnum btotal))
  (let* ((sd2-clause (qde-curvature-at-steady (state-qde (car behavior))))
	 (sd2-long (and sd2-clause (> (length sd2-clause) 3))))
    (when *perform-acc-analysis*
      (when sd2-clause
        (qplot-string (format nil "Curvatures: ~a" (subseq sd2-clause 0 3))
		      xpos ypos))
      (when sd2-long
        (qplot-string (format nil " continued- ~a" (cdddr sd2-clause))
		      xpos ypos)))
    (cond ((or (null  *perform-acc-analysis*)
	       (null sd2-clause))
	   0)
	  ((null sd2-long) 1)
	  (t 2))))



(defun get-axis-label (lmark ranges &key qvar)         ; modified 10/17/89 by D.B. 10/29/90. 1/31/91.
  (declare (ignore qvar))
  (let ((range ;; This checks for
	       ;; 1. The same lmark
	       ;; 2. The same lmark name
	       ;; 3. Two different lmarks with the same name.
	       ;;    This last change is needed for transitions where a new lmark is
	       ;;    generated in the new region that represents a similarly named one
	       ;;    in the old region.
               (or (lookup lmark ranges)
		   (lookup (lmark-name lmark) ranges) 	; BJK 11-26-90
		   (lookup (lmark-name lmark) ranges
			   :test #'(lambda (key item)  ; there are things other than lmarks in the alist.
				     (and (lmark-p item)
					  (eq key (lmark-name item)))))))) ;BKay 21May92
    (cond ((null range) (lmark-name lmark))
          (*pretty-numbers* (format nil "~a ~a" (lmark-name lmark) (range-to-string range)))
          (t                (format nil "~a ~a" (lmark-name lmark) range)))))

(defun GET-V-LABEL (lmark ranges &key qvar) (get-axis-label lmark ranges :qvar qvar))


(defun PLOT-H-AXIS-POINT (name state xoffset ytick transition-p &optional yname)
  ;; Yname should be deleted but who knows whose code assumes it's there.
  (declare (ignore yname state))
  (cond ((atom name)
	 (qplot-vline xoffset ytick tick-length)
	 (or transition-p	; Substituted for (transition-precedes state)
				; when parameter transition-p was added. -Mallory 9 Oct 90
	     (qplot-h-axis-label (- xoffset *time-label-offset*)
				 (+ ytick tick-length *axis-font-height*)
				 name)))))


;;; Currently, h-axis labels are done the same as v-axis labels.

(defun QPLOT-H-AXIS-LABEL (coord1 coord2 label)
  (qplot-axis-label coord1 coord2 label))


;;; Determine the label for a qplot box.  Scan across transitions, if necessary.

(defun BOX-LABEL (param state)
  (let ((var  (alookup param (qde-var-alist (state-qde state)))))
    (cond ((null var) param)
	  (*literal-parameter-names*  param)
	  (t (variable-title var)))))
