;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-


;;; TABLES

;;; Functions for printing tables of values, such as the qvalues of several states.


(in-package :qsim)


(defparameter *TABLE-SPECS* '(125 20 17)
  "Specifications for tables printed by Print-Table and Print-Completions.
  A list of 3 integers specifying
    table width (typically the screen width minus 1),
    label width (a label is printed at the left end of each line), and
    column width (width of one column of data)
  respectively, in characters.")


(defun MAKE-TABLE-FORMATS ()

  "Compute table formats and number of columns in a table printed by Print-Table or
  Print-Completions.  Takes input from *table-specs*.  Returns 3 values: the format for
  printing a label and individual values, the format for printing a label and the first
  block of a list of values, and the number of columns in a table."

  (let* ((table-width  (first  *table-specs*))
	 (label-width  (second *table-specs*))
	 (column-width (third  *table-specs*))
	 (num-columns  (truncate (/ (- table-width label-width 1) column-width)))
	 (tab-stops    (loop repeat num-columns
			     for tab from (+ label-width 2) by column-width
			     collect tab)))
    (flet ((make-format (list? tabs)
	      ;; No, its not readable, but thats "format" for you.
	      (format nil "~~% ~~A~:[~;~~1{~]~{~~~DT~~A~~^~^~}~:[~;~~}~]" list? tabs list?)))
      (values (make-format nil tab-stops)
	      (make-format t   tab-stops)
	      num-columns))))


(defun PRINT-COMPLETIONS (nstate initial-values after-prop after-filtering)
  
  ;; Print variable values initially, after propagation, and after filtering.
  ;; Separate variables into those with a single value after filtering
  ;; and those with multiple values.

  ;; Modified for more room on printout (larger tab stops) and to sort output
  ;; lines depending on whether they contain multiple values after filtering.

  (format *Qsim-Trace* "~%Completing state ~A of ~:@(~A~):~%~%Variables:"
	  (state-name nstate) (qde-name (state-qde nstate)))

  (let ((*detailed-printing* nil)
	(line-format (make-table-formats)))
    
    (multiple-value-bind (multi-valued multi-f-vals)
	(print-satisfaction nstate initial-values after-prop after-filtering
			    line-format)
      (when multi-valued
	(dolist (line multi-valued)
	  (format *Qsim-Trace* line-format (first line) (second line)
		  (third line) (fourth line) "Varies"))
	(format *Qsim-Trace* "~%~%Variables with multiple values:")
	(print-table multi-f-vals
		     (mapcar #'first multi-valued) nil)))
    ;; Print constraints
    (print-constraints nstate)))


(defun PRINT-SATISFACTION (nstate initial-values after-prop after-filtering line-format)

  (format *Qsim-Trace* line-format "Name:" "Qspace:" "Initial:" "Propagation:" "Satisfaction:")
  (loop with all-single = (null (rest after-filtering))
	for pair  in (state-qspaces nstate)
	for varname = (car pair)
	for qspace =  (cdr pair)
	for i-val =   (alookup varname initial-values)
	for p-val =   (alookup varname after-prop)
	for f-vals =  (mapcar #'(lambda (lst) (or (alookup varname lst) ""))
			      after-filtering)
	for p-prt =   (cond ((eql 'no-pvals-survived (variable--pvals (qval-variable i-val)))
			     "None survived")
			    ((equalp i-val p-val) "Same")
			    (t p-val))
	for collectp = (not (or all-single (all-same-p f-vals)))
	if (not collectp)
	  do (format *Qsim-Trace* line-format varname qspace i-val p-prt
		     (cond ((equalp p-prt "None survived") "None survived")
			   ((equalp (first f-vals) p-val) "Same")
			   (t (first f-vals))))
	if collectp 
	  collect (list varname qspace i-val p-prt) into multi-valued
	if collectp 
	  collect f-vals into  multi-f-vals
	finally (return (values multi-valued multi-f-vals))))


(defun PRINT-CONSTRAINTS (nstate)
  (format *Qsim-Trace* "~%~%Constraints:")
  (dolist (con (qde-constraints (state-qde nstate)))
    (when (constraint-active-p con)
      (let ((cvals  (alookup con (state-cvalues nstate)))
	    (*detailed-printing* nil)
	    (tuples (constraint--tuples con)))
	(format *Qsim-Trace* "~% ~A  ~A ~A"
		con
		(or cvals "")
		(if (eql 'no-tuples-survived tuples) "<-- No tuples survived" ""))))))


(defun ALL-SAME-P (lst &optional (test #'equal))

  "Return non-nil if all elements of lst are the same (using test) as the first element of
  lst, nil if lst contains at least 2 different elements (using test)."
  (every #'(lambda (elt) (funcall test (first lst) elt))
	 (rest lst)))


(defun PRINT-TABLE (rows labels headings
		    &key (check-uniform-rows t) (check-duplicate-cols t) (test #'equal))
  
  "Print a table of values (typically qvalues) in blocks of n columns each, where n is
  computed from *table-specs*.
     Rows is a list of sublists of values.  Each sublist is a set of values for a
  particular row, with one value for each column.
    Labels label the rows.
    Headings label the columns.  If no headings are supplied, columns will be numbered.
    Check-uniform-rows controls the handling of rows which have all values the same.  If
  check-uniform-rows is non-nil (the default), only one value from each such row is
  printed; if nil, they are treated like any other row.
    Check-duplicate-cols controls the handling of columns which have all values the same
  as some other column.  If check-duplicate-cols is non-nil (the default) such columns are
  identified and only one representative of each set of such columns is printed; if nil,
  they are treated like any other column."

  (unless headings
      (setq headings (loop for i from 1 upto (length (first rows))
			   collect (format nil "-- ~D --" i))))
  (multiple-value-bind
    (indiv-format table-format num-columns) (make-table-formats)
    ;; Find and print rows with all the same value
    (when check-uniform-rows
      (multiple-value-setq (rows labels)
	(find-and-print-uniform-rows rows labels test indiv-format)))
    ;; Find and list columns which equal some other column
    (when (and check-duplicate-cols rows)
      (find-duplicate-columns rows headings test))

      ;; Print rows with varying values
      (print-varying-rows rows labels headings num-columns table-format)))


(defun PRINT-VARYING-ROWS (rows labels headings num-columns table-format)
  (loop						; over blocks
    while (first rows)
    do (format *Qsim-Trace* "~%")
    (when headings
      (format *Qsim-Trace* table-format "" headings)
      (setf headings (nthcdr num-columns headings)))
    (loop					; over lines
      for label    in labels
      for row-cons on rows
      for row      in rows
      do (format *Qsim-Trace* table-format label row)
      ;; Point the current element of rows to the rest of that row still to be printed
      ;; (starting with the element following the num-columns elements just printed):
      (setf (first row-cons) (nthcdr num-columns row)))))


(defun FIND-AND-PRINT-UNIFORM-ROWS (rows labels test indiv-format)
  (let* (uniform-rows uniform-labels)
    (multiple-value-setq
      (rows labels uniform-rows uniform-labels)
      (remove-uniform-rows rows labels test))
    (when uniform-rows
      (format *Qsim-Trace* "~%~%Same value in all columns:")
      (loop
	    for value in uniform-rows
	    for label in uniform-labels
	    do (format *Qsim-Trace* indiv-format label value)))
    (values rows labels)))


(defun FIND-DUPLICATE-COLUMNS (rows headings test)

  "Find all duplicate columns (identical to some other column) in rows.
   Remove from rows all but the last of each set of duplicate columns.
   Remove corresponding column headings from headings.
   Print lists of identical columns."
  (let (duplicate-cols)
    (multiple-value-setq (rows headings)
      (remove-duplicate-rows (transpose rows) headings test))
    (setq rows (transpose rows)
	  duplicate-cols (loop for heading in headings
			       when (listp heading)
				 collect heading)
	  headings (mapcar #'(lambda (heading)
			       (if (atom heading)
				   heading
				   (first (last heading))))
			   headings))
    (if duplicate-cols
	(format *Qsim-Trace* "~%~%Lists of identical columns:~{~%  ~A~}"
		duplicate-cols))))


(defun REMOVE-UNIFORM-ROWS (rows labels test)

  "Separate rows into uniform and varying rows (according to test) and separate
  corresponding labels accordingly."
  (loop
    for row   in rows
    for label in labels
    when (all-same-p row test)
       collect (first row) into uniform-rows and
       collect label       into uniform-labels
    else
       collect row   into varying-rows and
       collect label into varying-labels
    finally (return (values varying-rows varying-labels
			    uniform-rows uniform-labels))))


(defun REMOVE-DUPLICATE-ROWS (rows labels test)

  "Remove duplicate rows from a list of rows.  Compress labels into a list of atoms
  (for unduplicated rows) or lists of labels (for duplicated rows)."
  (let* ((work (mapcar #'(lambda (label row) (cons label row))
		       labels rows))
	 (result (remove-duplicates work
	           :test #'(lambda (r1 r2)
			     (if (every test (rest r1) (rest r2))	; r1 will be removed
				 (setf (first r2) (nconc (listify (first r1))
							 (listify (first r2)))))))))
    (values (mapcar #'rest  result)
	    (mapcar #'first result))))


(defun TRANSPOSE (table)

  "Transpose a table (matrix) about its major diagonal: a[i,j] -> a[j,i]:
  list of rows -> list of columns or vice-versa."
  (loop
    for i from 0 to (1- (length (first table)))
    collect (mapcar #'(lambda (row) (nth i row)) table)))


(defun LISTIFY (arg)
  "Return arg as a list."
  (if (listp arg) arg (list arg)))


(defun TEST-PRINT-TABLE ()

  "Simple test for Print-Table with two uniform rows and two equal columns."
  (print-table
    '((a a a a a a a)
      (1 3 5 3 9 1 5)
      (2 4 6 4 8 0 6)
      (x x x x x x x)
      (b c d c f g d))
    '(r1 r2 r3 r4 r5) nil
    :test #'eql))


;;; Modified by Mallory 30 Oct 90 so test defaults to qval-equal rather than equal,
;;; which sometimes fails to detect equal qvals.

(defun SHOW-QVAL-DIFFS (&optional (states *initial-state*)
			(check-uniform-rows t) (check-duplicate-cols t)
			(test #'qval-equal))
  
  "Print a table showing all qvals in (1) all states in states (if states is a list of
  states or state names) or (2) all final states in all behaviors (finished or not)
  starting in states (if states is a single state).  The handling of uniform rows or
  repeated columns of values is controlled by the parameters check-uniform-rows and
  check-duplicate-cols.  See documentation for function Print-Table."

  (cond ((atom states)
	 (setq states (get-final-states states)))
	((not (state-p (first states)))		; Assume list of state names
	 (setq states (mapcar #'eval states)))
	(t nil))
  (let ((var-names (mapcar #'first (state-qvalues (first states))))
	(*detailed-printing* nil)
	(*short-qval-printing* t))
    (print-table
      (transpose (mapcar #'(lambda (state)
			     (mapcar #'rest (state-qvalues state)))
			 states))
      var-names
      (mapcar #'state-name states)
      :check-uniform-rows check-uniform-rows
      :check-duplicate-cols check-duplicate-cols
      :test test)))


;;; Modifed by A.Farquhar 11 June 90
;;; Previously used (rest (state-successors state)), which caused problems with cycles.
;;; Now uses successor-states.

(defun GET-FINAL-STATES (&optional (state *initial-state*))

  "Find the leaves of the tree of states beginning at state and continuing through the
  successors of state until all states with no successor are found."

  (let* ((successors (successor-states state)))	; successors, completions, and transitions
    (if (or (null successors)
	    (equal successors '(nil)))
	(list state)
	(mapcan #'get-final-states successors))))


