;;;  -*- Mode:Common-Lisp; Package: QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  Dan Dvorak, 1989.

(in-package 'QSIM)

;;;  Adds backjump (intelligent backtracking) and temporary learning of
;;;  bad patterns to form-all-states, recursive-form-states.

(defother variable level-when-set)
(defother variable level-when-used)

(defparameter *num-completions* 0)


;;;-----------------------------------------------------------------------------
;;;  Function:  (form-all-states  qde)
;;;
;;;  Given:     the qde
;;;
;;;  Returns:   a list of consistent value assignments for each variable.
;;;
;;;  Purpose:   This function, which is run after Waltz filtering, takes the
;;;             surviving tuples of each constraint and forms all possible
;;;             consistent value completions.
;;;
;;;  Design:    The basic idea in this algorithm is that we pick some constraint,
;;;             bind its variables using the next tuple of the constraint, and
;;;             then move on to another unprocessed constraint that is connected
;;;             to one of the processed constraints.  Of the candidate constraints
;;;             on this "boundary", we always take the one with the fewest tuples.
;;;             By moving in a "connected order" from constraints having few
;;;             tuples to constraints having many tuples, value completions are
;;;             formed recursively with a minimum amount of backtracking.
;;;             Here are the basic steps:
;;;
;;;             1.  For every constraint having only one tuple, bind its
;;;                 variables "permanently" since they must be the same for all
;;;                 completions.
;;;             2.  From the list of constraints processed in step 1, form the
;;;                 initial boundary consisting of directly-connected constraints
;;;                 that were not processed in step 1.
;;;             3.  Find the constraint having the fewest tuples in the boundary
;;;                 and delete it from the boundary.  Watch out for two special
;;;                 cases here -- the boundary could be nil either because there
;;;                 were no 1-tuple constraints or because ALL constraints had
;;;                 only one tuple.  In the latter case we're all done.
;;;             4.  Call (recursive-form-states constraint boundary).  The
;;;                 remaining steps are inside this recursive function.
;;;             5.  Temporarily bind any unbound variables of this constraint.
;;;                 Add to the boundary the unprocessed constraints that are
;;;                 directly connected to these newly bound variables.
;;;             6.  If boundary is empty, then we're all done.  Otherwise, find
;;;                 the constraint having the fewest tuples in the boundary and
;;;                 remove it.  Call (recursive-form-states constraint boundary).
;;;
;;;  Note:      This function could have been written in a very simple recursive
;;;             way to bind the variables of the first constraint, then do the
;;;             second constraint, etc, backtracking whenever an inconsistency
;;;             occurred.  This would probably be fine for small constraint
;;;             networks, but the amount of backtracking grows exponentially
;;;             with the number of constraints.
;;;-----------------------------------------------------------------------------

(defparameter *all-states* nil)

(defun form-all-states (qde)
  (let* ((variables    (qde-variables qde))
	 (constraints  (qde-constraints qde))
	 (starting-constraints nil)
	 (num-remaining-constraints (length constraints)))
    (declare (special variables))

    (setq *all-states* nil)

    ;; Step 0 -- Prepare 'variable' structures (leaving "time" variable alone).
    ;; Any variable which has only one possible value can have its qval set
    ;; permanently here.  This is in fact necessary for discrete variables
    ;; since they don't necessarily participate in constraints.
    (dolist (var (cdr variables))
      (let ((pvals (variable--pvals var)))
	(setf (variable--qval var)           nil
	      (variable-level-when-set  var) nil
	      (variable-level-when-used var) nil
	      (variable-done-p var)          nil)
	(if (singleton-p pvals)
	    (setf (variable--qval var) (car pvals)
		  (variable-level-when-set var) 0))))

    ;; Step 1 -- For all 1-tuple constraints, permanently bind the associated
    ;;           variables.
    (dolist (con constraints)
      (if (not (constraint-active-p con))
	  (progn
	    (setf (constraint-done con) t)	; want to skip inactive constraints
	    (decf num-remaining-constraints))
	  (progn
	    (setf (constraint-done con) nil)
	    (let ((tuples (constraint--tuples con)))
	      (when (singleton-p tuples)
		(setf (constraint-done con) t)
		(push con starting-constraints)
		(decf num-remaining-constraints)
		;; Bind variable values from this tuple.
		(mapc #'(lambda (var tval)
			  (let ((qval (variable--qval var)))
			    (if (null qval)
				(setf (variable--qval var) tval
				      (variable-level-when-set var) 0)
				(if (not (qval-equal qval tval))
				    (error "Inconsistent values ~a and ~a for ~a."
					   qval tval (variable-name var))))))
		      (constraint-variables con)
		      (first tuples)))))))
	
    ;; If all constraints were processed above (rare, but possible),
    ;; then form the single state and return.
    (when (= 0 num-remaining-constraints)
      (recursive-form-states nil 1)
      (return-from form-all-states *all-states*))

    ;; Step 2 -- From the list of constraints processed in step 1, form the
    ;;           initial boundary consisting of directly-connected constraints
    ;;           that were not processed in step 1.
    (let ((ordered-constraints nil)
	  (boundary nil)
	  nextcon)

      ;; Build initial boundary.
      (dolist (con starting-constraints)
	(dolist (ncon (constraint-neighbors con))
	  (if (not (constraint-done ncon))
	      (pushnew ncon boundary))))

      ;; Loop over remaining constraints to build an ordered list of
      ;; constraints for recursive formation of states.
      (dotimes (n num-remaining-constraints)

	;; If a boundary (still) exists ...
	(if boundary

	    ;; then find the constraint in the boundary having the fewest tuples
	    ;; and update the boundary accordingly.
	    (multiple-value-setq (nextcon boundary) (find-best-constraint boundary))

	    ;; else either there was no initial boundary (because there was no
	    ;; 1-tuple constraint) or else the constraint network consists of
	    ;; two or more unconnected networks, and we must now jump over to
	    ;; another network.
	    (multiple-value-setq (nextcon boundary) (find-best-constraint-2 constraints)))

	(push nextcon ordered-constraints))

      ;; Go form the states.
      (let ((ordered-constraints (nreverse ordered-constraints))
	    (level 0))
	;(pprint ordered-constraints)
	(dolist (con ordered-constraints)
	  (incf level)
	  (dolist (var (constraint-variables con))
	    (cond ((null (variable-level-when-set var))
		   (setf (variable-level-when-set var) level))
		  ((null (variable-level-when-used var))
		   (setf (variable-level-when-used var) level)))))

	(catch 'no-more-completions
	  (recursive-form-states ordered-constraints 1)))

      ;; Return the states, if any, created in the recursive function.
      *all-states*)))



;;;-----------------------------------------------------------------------------
;;;  Function:  (recursive-form-states  remaining-constraints  my-level)
;;
;;;  Given:     -- a list of constraints remaining to be checked in forming
;;;                a state;
;;;             -- the current recursion depth.
;;;
;;;  Returns:   two values:
;;;             -- an integer equal to the depth that should be backjumped to,
;;;                else its own depth if no backjumping needed;
;;;             -- the smallest depth that this call or any deeper call
;;;                depended on
;;;
;;;  Side Effects:  The last (deepest) recursive call forms a list of value
;;;                 completions and pushes that list onto *all-states*.
;;;
;;;  Design:    If no more constraints, push value completions onto *all-states*
;;;             and return.  Otherwise, take the first remaining constraint,
;;;             and check/bind its variables.  If there is a contradiction
;;;             such that one of its variables cannot be bound, then return.
;;;             Otherwise, do a recursive call with the tail of the list of
;;;             constraints.  
;;;
;;;             3/27/89: Intelligent backtracking ("backjumping") added.
;;;             If none of the tuples of a given constraint are compatible with
;;;             the current bindings, then backjump to the nearest constraint
;;;             (in terms of recursion depth) that has bound a value that
;;;             conflicts with the current constraint.
;
;for each tuple of the constraint
;   if tuple is compatible with all previous settings
;   then
;      if tuple matches a temporary disallowed pattern, then skip this tuple.
;      if tuple matches a chain of disallowed patterns (either starting at
;         this level or being tracked from a previous level)
;         then
;            if this is end of chain,
;               then skip this tuple,
;               else include this chain for next deeper r-f-s call
;
;   then if tuple does not match any temporary disallowed pattern from a
;              previous backjump) or any permanent disallowed pattern in the
;              branches being tracked
;        then call r-f-s and save return-depth, dependent-depth
;        if return-depth < my-depth
;        then return with return-depth and updated dependent depth
;        if return-depth = my-depth
;        then (0) I have been backjumped to;
;             (1) save my current tuple as a temporary disallowed pattern,
;                 with nil for any vals not depended upon at time of backjump;
;                 For each var that I set, if its depended-on-level is <= level
;                 of backjumper, then the var's current value is used in the
;                 pattern, else NIL is used.  If no such var found, error.
;             (2) remember that a backjump occurred.
;( else tuple is either incompatible or matches a disallowed pattern, so skip it)
;
;
;if no tuple of this constraint produced a completion
;   then 
;      (2) backjump to closest conflicting dependency.
;   else
;      (1) make each temporary disallowed pattern into a chain running from
;          smallest-dependent-depth to my depth, making sure to use as much
;          of a common branch that starts at s-d-d as possible.
;;;-----------------------------------------------------------------------------

(defun recursive-form-states (remaining-constraints my-level)
  (declare (special variables *all-states*))
  (declare (optimize speed))

  ;; If no more constraints ...
  (when (null remaining-constraints)
    ;; then push the state values onto *all-states* and return.
    (push (mapcar #'(lambda (var)
		      (cons (variable-name var) (variable--qval var)))
		  variables)
	  *all-states*)
    (when (= 0 (mod *num-completions* 100))
      (multiple-value-bind (sec min hour)
	  (get-decoded-time)
	(declare (ignore sec))
	(format t "~%~4d completions at ~2d:~2,'0d" *num-completions* hour min)))
    (incf *num-completions*)
    ;; Return normally (no backjump).
    (return-from recursive-form-states (values my-level nil)))

  ;; Take the next constraint on the list.
  (let* ((con  (car remaining-constraints))
	 (vars (constraint-variables con))
	 (bad-patterns nil)
	 (closest-level-with-conflict 0)
	 (old-num-completions *num-completions*))

    ;; For each tuple of this constraint ...
    (dolist (tuple (constraint--tuples con))
      (block do-this-tuple

	;; For each qval of this tuple ...
	(do* ((tvals tuple       (cdr tvals))
	      (cvars vars        (cdr cvars))
	      (var   (car cvars) (car cvars)))
	     ((endp tvals))
	  (if (= my-level (variable-level-when-set var))
	      ;; then set its value
	      (setf (variable--qval var) (car tvals))
	      ;; else it is already set,so check for compatible value.
	      (unless (qval-equal (car tvals) (variable--qval var))
		(if (> (variable-level-when-set var) closest-level-with-conflict)
		    (setq closest-level-with-conflict (variable-level-when-set var)))
		(return-from do-this-tuple))))

	;; The only way to get here is if the tuple is compatible with
	;; all prior settings.  Now, see if this tuple is covered by any
	;; known bad pattern.  First, we check the temporary bad patterns
	;; that have (just) been created while in this call.

	(when bad-patterns
	  (dolist (pattern bad-patterns)
	    (when (do ((tvals   tuple    (cdr tvals))
		       (pvals   pattern  (cdr pvals)))
		      ((endp tvals) t)
		    (and (car pvals)
			 (not (eq (car tvals) (car pvals)))
			 (return nil)))
	      ;; We've matched a bad pattern, so skip this tuple.
	      (return-from do-this-tuple))))

	;; Tuple has survived all tests, so make recursive call.
	(multiple-value-bind (return-level backjumper-level)
	    (recursive-form-states (cdr remaining-constraints) (1+ my-level))

	  (cond
	    ;; If return-depth < my-depth, then a backjump is "passing through".
	    ((< return-level my-level)
	     (return-from recursive-form-states (values return-level backjumper-level)))

	    ;; If return-level = my-level, then this is target of backjump.
	    ;; First, if all set vars were depended on, then no point in
	    ;; saving the pattern since it will never recur.
	    ((= return-level my-level)
	     (when (and (> (count-if #'(lambda (var)
					 (= my-level (variable-level-when-set var)))
				     vars) 1)
			(dolist (var vars nil)
			  (if (and (= my-level (variable-level-when-set var))
				   (variable-level-when-used var)
				   (> (variable-level-when-used var) backjumper-level))
			      (return t))))
	       ;; Construct the bad pattern and save for this invocation.
	       (push (mapcar #'(lambda (tval var)
				 (cond ((not (= my-level (variable-level-when-set var))) nil)
				       ((> (variable-level-when-used var) backjumper-level) nil)
				       (t tval)))
			     tuple
			     vars)
		     bad-patterns)))

	    ;; Normal return (just proceed to next tuple).
	    (t nil))
	  )					; end of multiple-value-bind
	)					; end of block do-this-tuple
      )						; end of dolist tuple

    ;; All tuples have been tried.  If none of them yielded a completion,
    ;; then backjump to my nearest dependent level.
    (if (= *num-completions* old-num-completions)
	;; No completions, so backjump.
	(return-from recursive-form-states
	  (values closest-level-with-conflict my-level))
	;; At least one completion, so normal return (no backjump).
	(return-from recursive-form-states
	  (values my-level nil)))
    )
  )

(defother constraint num-vars-set)


;;; Returns 2 values:  constraint having fewest tuples in boundary,
;;; and updated boundary.


(defun find-best-constraint (boundary)
  (let ((min-num-tuples most-positive-fixnum)
	(min-constraint nil)
	(max-num-vars-set 0)
	(min-num-vars-unset most-positive-fixnum)
	(best-constraints nil)
	(best-constraints-2 nil))

    ;; PASS 1: For each constraint on boundary, determine how many
    ;;         of its vars are already set.  Remember highest number.
    (dolist (con boundary)
      (let* ((vars         (constraint-variables con))
	     (num-vars-set (count-if #'(lambda (var) (variable-done-p var)) vars)))
	(cond ((> num-vars-set max-num-vars-set)
	       (setq max-num-vars-set num-vars-set)
	       (setq best-constraints (list con)))
	      ((= num-vars-set max-num-vars-set)
	       (push con best-constraints)))))

    (when (singleton-p best-constraints)
      (let ((bestcon (car best-constraints)))
	(format t "~%~27a because ~d of its vars are already set." bestcon max-num-vars-set)
	(return-from find-best-constraint
	  (values bestcon (update-boundary boundary bestcon)))))

    ;; PASS 2: Of all the "best" constraints, prefer the one(s) with
    ;;         the fewest dangling variables.
    (dolist (con best-constraints)
      (let* ((vars           (constraint-variables con))
	     (num-vars-unset (count-if-not #'(lambda (var) (variable-done-p var)) vars)))
	(cond ((< num-vars-unset min-num-vars-unset)
	       (setq min-num-vars-unset num-vars-unset)
	       (setq best-constraints-2 (list con)))
	      ((= num-vars-unset min-num-vars-unset)
	       (push con best-constraints-2)))))

    (when (singleton-p best-constraints-2)
      (let ((bestcon (car best-constraints-2)))
	(format t "~%~27a because it has only ~d dangling vars." bestcon min-num-vars-unset)
	(return-from find-best-constraint
	  (values bestcon (update-boundary boundary bestcon)))))

    ;; PASS 3: Of all the "best" constraints, prefer the one with
    ;;         the fewest tuples.
      
    (dolist (con best-constraints-2)
      (let ((num-tuples (length (constraint--tuples con))))
	(when (< num-tuples min-num-tuples)
	  (setq min-num-tuples num-tuples
		min-constraint con))))

    (format t "~%~27a because it has fewest tuples (~d)." min-constraint min-num-tuples)

    (return-from find-best-constraint
      (values min-constraint (update-boundary boundary min-constraint)))
    )
  )

;;;  Delete con from current boundary and add neighbors of con to boundary.

(defun update-boundary (boundary con)
  (setf (constraint-done con) t)
  (dolist (var (constraint-variables con))
    (setf (variable-done-p var) t))
  (setq boundary (delete con boundary :count 1))
  (dolist (ncon (constraint-neighbors con))
    (if (not (constraint-done ncon))
	(pushnew ncon boundary)))
  boundary)


;;; Of the unprocessed constraints, return the one having the fewest tuples.

(defun find-best-constraint-2 (constraints)
  (let ((min-num-tuples most-positive-fixnum)
	(min-constraint nil)
	(boundary nil)
	n)
    (dolist (con constraints)
      (when (and (not (constraint-done con))
		 (< (setq n (length (constraint--tuples con))) min-num-tuples))
	(setq min-num-tuples n
	      min-constraint con)))
    (format t "~%~27a because it has fewest TUPLES (~d)." min-constraint min-num-tuples)
    (setf (constraint-done min-constraint) t)
    (dolist (var (constraint-variables min-constraint))
      (setf (variable-done-p var) t))
    (dolist (con (constraint-neighbors min-constraint))
      (if (not (constraint-done con))
	  (pushnew con boundary)))
    (values min-constraint boundary)))


