;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)

(defun describe-ign-qval ()
  (format *qsim-report*
"~2%Ignore-Qvals allows one to specify a set of variables whereby distinctions
arising from states of these variables will be ignored.  This is useful for
ignoring distinctions arising from intermediate variables and other variables
one chooses not to focus attention on.  Include an (ignore-qvals V1 V2 . . . )
clause in the Other clause of the QDE to ignore distinctions arising from
variables V1, V2, . . .~2%"))

;;;  This example is taken out of the Water-Heater example to illustrate the
;;;  use of Ignore-Qvals.  For details of the full example, refer to
;;;  "examples;water-heater".

(define-QDE WATER-HEATER
  (text "Water heater with thermostatic control")
  (quantity-spaces
    (Heat       (0 Room Lo Hi inf))
    (TempWater  (0 Room Lo Hi inf))
    (TempRoom   (0 Room inf))
    (TempDiff   (0 Lo Hi inf))
    (HFoutI     (0 Lo Hi inf))			; Heat loss via insulation
    (HFoutW     (0 W1 W2 W3 W4 inf))		; Heat loss via water flow
    (HFout      (0 H1 H2 H3 H4 H5 H6 inf))	; Heat loss total.
    (netHF      (minf 0 inf))
    (H-power    (0 On))				; Heater power
    (S-power    (0 On))				; Source power
    (HFin1      (0 On))
    (HFin2      (0 On))
    (HFin       (0 On1 On2))
    (WaterFlow  (0 WF1 WF2 inf)))			; Water flow
  (discrete-variables
    (switch  (open closed stuck-open stuck-closed))
    (H1      (good bad))
    (H2      (good bad)))
  (constraints
    (mode ((switch closed) (switch stuck-closed))
	  ((M+ S-power H-power) (0 0) (On On)))
    (mode ((switch open) (switch stuck-open))
	  ((zero-std H-power)))
    (mode ((H1 good))
	  ((M+ H-power HFin1) (0 0) (On On)))
    (mode ((H1 bad))
	  ((zero-std HFin1)))
    (mode ((H2 good))
	  ((M+ H-power HFin2) (0 0) (On On)))
    (mode ((H2 bad))
	  ((zero-std HFin2)))
    ((ADD  HFin1 HFin2 HFin)            (0 0 0) (On 0 On1) (0 On On1) (On On On2))
    ((M+   Heat TempWater)              (0 0) (Room Room) (Lo Lo) (Hi Hi) (inf inf))
    ((ADD  TempDiff TempRoom TempWater) (0 Room Room) (0 inf inf) (Lo Room Lo) (Hi Room Hi))
    ((M+   TempDiff HFoutI)             (0 0) (Lo Lo) (Hi Hi) (inf inf))
    ((MULT WaterFlow TempDiff HFoutW)   (WF1 Lo W1) (WF1 Hi W2) (WF2 Lo W3) (WF2 Hi W4))
    ((ADD  HFoutI HFoutW HFout)         (Lo 0 H1) (Hi 0 H2) (Lo W1 H3) (Hi W2 H4) (Lo W3 H5) (Hi W4 H6))
    ((ADD  netHF HFout HFin)            (0 H3 On1) (0 H5 On2))
    ((D/DT Heat netHF)))
  (transitions
     ((and (switch (open   std)) (TempWater (Lo dec)))   turn-switch-on)
     ((and (switch (closed std)) (TempWater (Hi inc)))   turn-switch-off))
  (independent S-power WaterFlow TempRoom)
  (history TempWater)
  (layout (TempWater  WaterFlow    H-power)
	  (TempDiff   HFoutW       HFin1)
	  (nil        HFoutI       HFin2)
	  (netHF      HFout        HFin))
  (other (ignore-qvals HFoutW HFoutI HFout netHF))
  )

;;; H1 & H2 both bad, Temp drops to low, heater switch turns on but no heat produced,
;;; so water temperature drops to room temperature.

(defun bad-heaters-w-o-ign-qval ()
  (setf (qde-other water-heater) nil)
  (let ((*state-limit* 210)
	(start (make-initial-state water-heater
				   '(;; ----------------------- Discrete Variables.
				     (switch     (open nil))
				     (H1         (bad  nil))
				     (H2         (bad nil))
				     ;; ----------------------- Independent Variables.
				     (S-power    (On std))
				     (WaterFlow  (WF2 std))
				     (TempRoom   (Room std))
				     ;; ----------------------- History Variables.
				     (TempWater  (Hi nil))
				     )
				   "H1 & H2 bad, hot, hiflow")))
    (declare (special *state-limit*))
    (format *qsim-report*
"~2%This example is taken out of the Water-Heater example to illustrate the use of
Ignore-Qvals.  For details of the full example, refer to `examples;water-heater'.
In this example, both heaters, H1 & H2, are bad and water temperature drops to room
temperature.  Many behaviors are generated due to uninteresting distinctions (23
behaviors with 200 states).  With an appropriate Ignore-Qvals clause, only one
behavior will be genarated.  Refer to `examples;ignore-qvals-example' for details
of this example.~2%")
    (qsim start)
    (qsim-display start)
    (setf (qde-other water-heater) '((ignore-qvals HFoutW HFoutI HFout netHF)))
    t))

(defun bad-heaters-w-ign-qval ()
  (let ((*state-limit* 210)
	(start (make-initial-state water-heater
				   '(;; ----------------------- Discrete Variables.
				     (switch     (open nil))
				     (H1         (bad  nil))
				     (H2         (bad nil))
				     ;; ----------------------- Independent Variables.
				     (S-power    (On std))
				     (WaterFlow  (WF2 std))
				     (TempRoom   (Room std))
				     ;; ----------------------- History Variables.
				     (TempWater  (Hi nil))
				     )
				   "H1 & H2 bad, hot, hiflow")))
    (declare (special *state-limit*))
    (format *qsim-report*
"~2%This example is taken out of the Water-Heater example to illustrate the use of
Ignore-Qvals.  For details of the full example, refer to `examples;water-heater'.
In this example, both heaters, H1 & H2, are bad and water temperature drops to room
temperature.  Without an appropriate Ignore-Qvals clause, many behaviors are
generated due to uninteresting distinctions (23 behaviors with 200 states).  See
this by running Bad-heater-w-o-ign-qval.  With an appropriate Ignore-Qvals clause,
however, only one behavior is genarated.  Refer to `examples;ignore-qvals-example'
for details of this example.~2%")
    (qsim start)
    (qsim-display start)
    t))

;;;  Operating region transition functions.  Not sure that they are needed here.

(defun turn-switch-on (water-heater-state)
  (create-transition-state :from-state   water-heater-state
			   :to-qde       water-heater
			   :assert       '((switch  (closed std)))
			   :inherit-qmag '(S-power WaterFlow TempRoom TempWater)
			   :inherit-qdir nil))


(defun turn-switch-off (water-heater-state)
  (create-transition-state :from-state   water-heater-state
			   :to-qde       water-heater
			   :assert       '((switch  (open std)))
			   :inherit-qmag '(S-power WaterFlow TempRoom TempWater)
			   :inherit-qdir nil))


