;;; -*- Syntax: Common-lisp; Package: qsim -*-
(in-package :qsim)

; Tanaka heart regulation model, for Q.  This is a modified version of the
; Heart Regulation model that appeared in MIT LCS TM-280, thanks to the
; help of Dr. Hiroshi Tanaka of the University of Tokyo.  
;   One interesting feature this uses is the "unreachable values" clause 
;   in QDE.others, to make an open-interval legal domain.

(define-QDE HEART-REGULATION
 (text "Heart model homeostasis via sympathetic stimulation (H.Tanaka)")
  (quantity-spaces
    (SS (0 SS* INF))
    (AB (0 AB% AB* AB%% INF))
    (AS (0 AS* INF))
    (BB (0 BB% BB* BB%% INF))
    (BS (0 BS* INF))
    (PL (0 PL* INF))
    (PLA (0 PLA* INF))
    (BV (0 BV% BV* BV%% INF))
    (SC (0 SC% SC* SC%% INF))
    (HR (0 HR* INF))
    (HRA (0 HRA* INF))
    (HRX (0 HRX% HRX* HRX%% INF))
    (SV (0 SV* INF))
    (CO (0 CO* INF))
    (PVR (0 PVR* INF))
    (PVRA (0 PVRA* INF))
    (PVRX (0 PVRX% PVRX* PVRX%% INF))
    (MSFP (0 MSFP* INF))
    (MSP (0 MSP* INF))
    (MDP (0 MDP* INF))
    (DSS (MINF 0 INF))
    (EDV (0 EDV* INF))
    (EF (0 EF* INF))
    (IN (0 IN* INF))
    (INA (0 INA* INF))
    (INB (0 INB* INF))
    (INC (0 INC* INF))
    (INX (0 INX% INX* INX%% INF)))
  (constraints
    ((ADD SS AB AS)       (ss* ab* as*))
    ((ADD SS BB BS)       (ss* bb* bs*))
    ((M+ PLA AS)          (pla* as*)         (0 0) (inf inf))
    ((MULT MSFP SC BV)    (msfp* sc* bv*))
    ((ADD BV PLA PL)      (bv* pla* pl*))
    ((M+ HRA BS)          (hra* bs*)         (0 0) (inf inf))
    ((ADD HRA HRX HR)     (hra* hrx* hr*))
    ((M+ PVRA AS)         (pvra* as*)        (0 0) (inf inf))
    ((ADD PVRA PVRX PVR)  (pvra* pvrx* pvr*))
    ((M+ EDV PL)          (edv* pl*)         (0 0) (inf inf))
    ((M+ INA EDV)         (ina* edv*)        (0 0) (inf inf))
    ((M+ INC BS)          (inc* bs*)         (0 0) (inf inf))
    ((ADD INC INX INB)    (inc* inx* inb*))
    ((ADD INA INB IN)     (ina* inb* in*))
    ((M+ EF IN)           (ef* in*)          (0 0) (inf inf))
    ((MULT EDV EF SV)     (edv* ef* sv*))
    ((MULT HR SV CO)      (hr* sv* co*))
    ((MULT CO PVR MDP)    (co* pvr* mdp*))
    ((ADD MDP MSFP MSP)   (mdp* msfp* msp*))
    ((M- MDP DSS)         (mdp* 0)           (0 inf)  (inf minf))
    ((D/DT SS DSS)) )
  (independent ab bb sc bv inx hrx pvrx)
  (history ss)
 (layout
   (SS AB AS BB BS)
   (SC BV nil INX IN)
   (BV  nil PL EDV EF)	
   (HRX  nil HR SV CO)
   (SC nil PVRX PVR)
   (MSFP MDP MSP DSS))
 (print-names
   (SS "sympathetic stimulation")
   (AB "alpha stimulation base")
   (AS "alpha stimulation")
   (BB "beta stimulation base")
   (BS "beta stimulation")
   (PL "preload")
   (BV "blood volume")
   (SC "systemic compliance")
   (PLa "preload(alpha)")
   (HR "heart rate")
   (HRa "heart rate(beta)")
   (HRx "heart rate(other)")
   (SV "stroke volume")
   (CO "cardiac output")
   (PVR "peripheral vascular resistance")
   (PVRa "PVR(alpha)")
   (PVRx "PVR(other)")
   (MSFP "mean syst. filling pressure")
   (MDP "mean dynamic pressure")
   (MSP "mean systemic pressure")
   (DSS "derivative(symp.stim.)")
   (EDV "end diastolic volume")
   (EF "ejection fraction")
   (IN "inotropic state")
   (INa "inotropic(preload)")
   (INb "inotropic(partial sum)")
   (INc "inotropic(beta)")
   (INx "inotropic(other)"))
 (other
   (unreachable-values (ss 0)))
 )

; Beta stimulation can be reduced by certain drugs ("beta blockers") or
; increased by excitement or fright.

(defun beta-blocker ()
  (let* ((normal (make-initial-state heart-regulation
				     '((SS (SS* std))
				       (BV (BV* STD))
				       (AB (AB* STD))
				       (BB (BB* STD))
				       (INX (INX* STD))
				       (HRX (HRX* STD))
				       (SC (SC* STD))
				       (PVRX (PVRX* STD)))))
	 (init (make-modified-state normal
				    '((SS (SS* NIL))
				      (BV (BV* STD))
				      (AB (AB* STD))
				      (BB (BB% STD))
				      (INX (INX* STD))
				      (HRX (HRX* STD))
				      (SC (SC* STD))
				      (PVRX (PVRX* STD)))
				    "Beta blocker")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun beta-stimulation ()
  (let* ((normal (make-initial-state heart-regulation
				     '((SS (SS* std))
				       (BV (BV* STD))
				       (AB (AB* STD))
				       (BB (BB* STD))
				       (INX (INX* STD))
				       (HRX (HRX* STD))
				       (SC (SC* STD))
				       (PVRX (PVRX* STD)))))
	 (init (make-modified-state normal
				    '((SS (SS* NIL))
				      (BV (BV* STD))
				      (AB (AB* STD))
				      (BB (BB%% STD))
				      (INX (INX* STD))
				      (HRX (HRX* STD))
				      (SC (SC* STD))
				      (PVRX (PVRX* STD)))
				    "Beta stimulation")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))


; Simulating the effects of high and low blood volume.

(defun high-blood-volume ()
  (let* ((normal (make-initial-state heart-regulation
				     '((SS (SS* std))
				       (BV (BV* STD))
				       (AB (AB* STD))
				       (BB (BB* STD))
				       (INX (INX* STD))
				       (HRX (HRX* STD))
				       (SC (SC* STD))
				       (PVRX (PVRX* STD)))))
	 (init (make-modified-state normal
				    '((SS (SS* NIL))
				      (BV (BV%% STD))
				      (AB (AB* STD))
				      (BB (BB* STD))
				      (INX (INX* STD))
				      (HRX (HRX* STD))
				      (SC (SC* STD))
				      (PVRX (PVRX* STD)))
				    "High blood volume")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun low-blood-volume ()
  (let* ((normal (make-initial-state heart-regulation
				     '((SS (SS* std))
				       (BV (BV* STD))
				       (AB (AB* STD))
				       (BB (BB* STD))
				       (INX (INX* STD))
				       (HRX (HRX* STD))
				       (SC (SC* STD))
				       (PVRX (PVRX* STD)))))
	 (init (make-modified-state normal
				    '((SS (SS* NIL))
				      (BV (BV% STD))
				      (AB (AB* STD))
				      (BB (BB* STD))
				      (INX (INX* STD))
				      (HRX (HRX* STD))
				      (SC (SC* STD))
				      (PVRX (PVRX* STD)))
				    "Low blood volume")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))
