;;; -*- Package: QSIM; Syntax: Common-lisp -*-
(in-package :qsim)

; Double Heat-Flow Example from AI Journal paper, 1984.

(define-QDE Double-Heat-Flow
  (text "Container with heat flow in and out.")
  (quantity-spaces
   (T-air    (minf airtemp flametemp inf))
   (T        (minf airtemp flametemp inf))
   (T-source (minf airtemp flametemp inf))
   (dTA      (minf 0 inf))
   (dTS      (minf 0 inf))
   (outflow  (minf 0 inf))
   (inflow   (minf 0 inf))
   (netflow  (minf 0 inf)))
  (constraints
   ((add T-air dTa T)    (airtemp 0 airtemp))
   ((add T dTS T-source) (flametemp 0 flametemp))
   ((M+ dTA outflow)     (0 0) (inf inf) (minf minf))
   ((M+ dTS inflow)      (0 0) (inf inf) (minf minf))
   ((add outflow netflow inflow))
   ((d/dt T netflow))
   ((constant T-air))
   ((constant T-source)))
  (layout
    (T-air T T-source nil)
    (dTA nil dTS)
    (outflow nil inflow)
    (nil netflow nil))
  )

; Unfortunately, in spite of appearances, T-air, T, and T-source do not
; share a quantity space:  they only have similarly spelled landmarks.
; This is why the add constraints need those odd correspondences.

; The three cases explored here depend on the ability to branch on an underspecified
; initial state:
;  - completely unknown initial state
;  - initial temp between airtemp and flametemp
;  - initial temp = airtemp

(defun HF-unknown-temp ()
  (let ((init (make-new-state :from-qde Double-Heat-Flow
			      :assert-values '((T-air  (airtemp std))
					       (T-source (flametemp std))
					       (T      ((minf inf) nil)))
			      :text "Container with unknown temperature.")))
    (qsim init)
    (qsim-display init)))

(defun HF-intermediate-temp ()
  (let ((init (make-new-state :from-qde Double-Heat-Flow
			      :assert-values '((T-air  (airtemp std))
					       (T-source (flametemp std))
					       (T      ((airtemp flametemp) nil)))
			      :text "Container starts between air and flame temp.")))
    (qsim init)
    (qsim-display init)))

(defun HF-airtemp ()
  (let ((init (make-new-state :from-qde Double-Heat-Flow
			      :assert-values '((T-air  (airtemp std))
					       (T-source (flametemp std))
					       (T      (airtemp nil)))
			      :text "Container starts at air temperature.")))
    (qsim init)
    (qsim-display init)))
