;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Default-character-style:(FIX ROMAN NORMAL); Base:10 -*-

; The standard Bathtub example.  The basic first-order system that inspires many
; other equilibrium systems.  The base case, "fill-a-bathtub", fills an empty
; bathtub and branches three ways according to the possibility of overflow.  The
; two other cases show how a bathtub at steady state responds to a step change
; in the inflow.

; Some syntactic improvements.
;  - Quantity-spaces take an optional third argument:  a documentation string.
;    (This replaces the "print-names" clause.)
;  - The constant constraint normally replaces the "independent" clause (except in TSA).
;  - The "history" clause is not used, and may be omitted.
;  - The "layout" may be omitted, and provided as a keyword argument to qsim-display.
;  - Use make-new-state, with keyword arguments, rather than make-initial-state or
;    make-modified-state.

(in-package :qsim) ;changed DJC

(define-QDE bathtub
  (text "Standard bathtub with open drain.")
  (quantity-spaces
    (amount    (0 FULL)      "amt(water,tub)")
    (level     (0 TOP inf)   "level(water,tub)")
    (pressure  (0 inf)       "pressure(water,bottom(tub))")
    (outflow   (0 inf)       "flow(water,tub->out)")
    (inflow    (0 if* inf)   "flow(water,in->tub)")
    (netflow   (minf 0 inf)  "net flow(water,out->tub)"))
  (constraints
    ((M+ amount level)              (0 0) (full top))
    ((M+ level pressure)            (0 0) (inf inf))
    ((M+ pressure outflow)          (0 0) (inf inf))
    ((ADD netflow outflow inflow))
    ((d/dt amount netflow))
    ((constant inflow)))

  (layout (nil nil amount nil)
	  (nil nil level)
	  (nil nil pressure)
	  (nil inflow outflow)
	  (nil netflow nil))
  )

; fill the bathtub.

(defun fill-a-bathtub ()
  (let ((initial-state
	  (make-new-state :from-qde bathtub
			  :assert-values '((inflow (if* std))
					   (amount (0 nil)))
			  :text "Filling at constant rate, starting from empty")))
    (qsim initial-state)
    (qsim-display initial-state)))

;; These examples show
;;    How to make a step change from an initial state,
;;    How to extract the generated landmark values from an initial state.

;; INCREASE-FLOW generates three behaviors - similar to the behaviors
;; from FILL-A-BATHTUB.  The tub either finds a new steady state, or
;; hits steadystate at exactly the top of the tub, or overflows.


(defun increase-flow ()
  ; NORMAL is the steady state - everything steady, inflow = outflow,
  ; the level somewhere between 0 and TOP.
  (let* ((normal (make-new-state :from-qde bathtub
				 :assert-values '((amount    ((0 FULL) std))
						  (level     ((0 TOP) std))
						  (pressure  ((0 inf) std))
						  (inflow    (if* std))
						  (netflow   (0 std)))))
	 ; We generate STEP-UP as a modification of NORMAL.  Inflow is
	 ; increased.  We want to set Amount, Level, and Pressure at
	 ; their NORMAL values, and have them increase from there.  We
	 ; do this using the call to QMAG, which extracts the landmark
	 ; value for Amount from the NORMAL data-structure.  The values
	 ; for Level, Pressure etc. are generated by propigation.
	 ;
	 (step-up (make-new-state :from-state normal
				  :inherit '(amount)	        ; amount keeps normal value
				  :perturb '((inflow +))	; increase inflow
				  :text "Increased Inflow")))

    ;              Run QSIM, starting  at the modified state
    (qsim step-up)
    ;              Display the behavior, and include the normal state for reference.
    (qsim-display step-up :reference-states `((normal ,normal)))
    ))

;; DECREASE-FLOW generates only one behavior.  The level falls from its
;; NORMAL value to some value intermediate between 0 and Normal.

(defun decrease-flow ()
  (let* ((normal (make-new-state :from-qde bathtub
				 :assert-values '((amount    ((0 FULL) std))
						  (level     ((0 TOP) std))
						  (pressure  ((0 inf) std))
						  (inflow    (if* std))
						  (netflow   (0 std)))))
	 (step-down (make-new-state :from-state normal
				    :inherit '(amount)
				    :perturb '((inflow -))
				    :text "Decreased Inflow")))
    (qsim step-down)
    (qsim-display step-down :reference-states `((normal ,normal)))
    ))


; These are underspecified initializations, to test the multiple initial-state handling.

(defun fill-underspecified-bathtub ()
  (let ((initial-state
	  (make-new-state :from-qde bathtub
			  :assert-values '((inflow (if* std))
					   ; (amount (0 nil))
					   )
			  :text "Filling at constant rate, starting anywhere.")))
    (qsim initial-state)
    (qsim-display initial-state)
    ))

(defun totally-underspecified-bathtub ()
  (let* ((sim (make-sim :state-limit 50))
	 (initial-state
	   (make-new-state :from-qde bathtub
			   :assert-values '(	; (inflow (if* std))
					    ; (amount (0 nil))
					    )
			   :text "Totally underspecified bathtub."
			   :sim sim)))
    (qsim initial-state)
    (qsim-display initial-state)
    ))
