/*
 *  'formnewstate' takes the candidate state existing in param[] and
 *  constraint[] and stores it as a new state if it survives all
 *  state filtering ("no change" and "cycle" filters).  If the new
 *  state is not quiescent and not divergent, then it is also added
 *  to the agenda of states to be expanded from.
 *
 *  This task of checking and installing a candidate state is potentially
 *  time-consuming, so 'formnewstate' does the following:
 *
 *  (1)  It does as much as possible "in parallel".  For example, as it is
 *	 iterating through the set of parameters tentatively installing
 *	 their values in a new state, it is also testing for "no change",
 *	 divergence, and quiescence, and installing new landmarks.
 *  (2)  It tries to eliminate logically unnecessary tests.  For example,
 *	 if no new landmarks were created then check for a cycle, but don't
 *	 bother checking for new correspondences.
 */

#include  <stdio.h>
#include  "defs.h"
#define   EXTERN  extern
#include  "global.h"

char  *prqspace(), *timetostring();


formnewstate()
{
    register struct param  *p;
    register struct qval   *prevqvalp, *cqvalp;
    int  i, j, x, qmagx, qdir, myqmagx[MAXARGS];
    short  *lmpsave;
    unsigned int  checksum;
    Boolean  nopredecessor, nochange, quiescence, divergence;
    Boolean  newlmarks, newcorr;
    struct state       *statep, *prevp;
    struct constraint  *conp;
    struct corr        *corrp, *corrstop, *crpsave;
    struct scorr       *scrpsave;
    short  *lmarkp;
    struct qspace *qspsave[MAXPARAMS];


    /*  Install candidate as a full-fledged new state.  However, we will
     *  discard this candidate if it is a "no change" or if it identifies
     *  a cycle.
     */
    if (stp >= stpmax)
    {
	printf("Exceeded limit of %d states\n", MAXSTATES);
	exit(1);
    }
    statep   = stp;
    lmpsave  = lmp;
    nopredecessor = (curstatep == (struct state *) 0)  ?  TRUE : FALSE;

    statep->time     = curtime + 1;
    statep->npreds   = nopredecessor ? 0 : 1;
    statep->predp[0] = curstatep;
    statep->nsuccs   = 0;
    statep->qvalp    = qvp;

    nochange   = interval(curtime);   /* test for no-change only at interval */
    quiescence = TRUE;
    divergence = FALSE;
    newlmarks  = FALSE;

    /*  If this is the initial state then don't try to check for no change */
    if (nopredecessor)
	nochange = FALSE;
    else
	cqvalp = curstatep->qvalp;

#ifdef TRACE
    if (trace & TR_STATES)
	printf("Tentative state %d, time = %s, nochange = %d\n",
		(int)(statep - &state[0]), timetostring(curtime), nochange);
#endif

    if ((qvp + nparams) >= qvpmax)
    {
	printf("Exceeded limit of %d qval's\n", MAXQVALS);
	exit(1);
    }

    /*  Install state values and, in the process, create new landmarks and
     *  check for no-change, quiescence and divergence.
     */
    checksum = 0;
    for (i = 0;  i < nparams;  i++, qvp++)
    {
	p = &param[i];
	qspsave[i] = p->qspacep;
	p->newx = BIGNUM;
	qvp->qspacep =                         p->qspacep;
	qvp->qmagx   =  qmagx  =  p->qmagx  =  p->mypvalp->qmagx;
	qvp->qdir    =  qdir   =  p->qdir   =  p->mypvalp->qdir;

	/*  The state checksum computed in the following statement generates
	 *  a number based solely on the parameters' qmagx and qdir.  It is
	 *  used later as a fast test during the check for a cycle.  This
	 *  can be eliminated if it isn't worth the (small) overhead.
	 */
	checksum = checksum + (checksum<<2) + (qdir<<4) + qmagx;

	if (nochange)
	{
	    if ((qmagx != cqvalp->qmagx) || (qdir != cqvalp->qdir))
	    {
		nochange = FALSE;
#ifdef TRACE
		if (trace & TR_STATES)
		{   printf("mismatch on %s, so nochange=FALSE\n", p->namep);
		    printf("  qmag: %d :: %d, qdir: %d :: %d\n",
			qmagx, cqvalp->qmagx, qdir, cqvalp->qdir);
		}
#endif
	    }
	    else cqvalp++;
	}

	if (qdir != STD)
	    quiescence = FALSE;
	else
	{
	    /* Since qdir is STD, see if new landmark needs to be created. */
	    if (interval(qmagx))
	    {
		/*  CREATE NEW LANDMARK  */
		newlmarks = TRUE;
		nochange = FALSE;
#ifdef TRACE
		if (trace & TR_LANDMARKS)
		    printf("existing qspace for %s is %s\n",
			p->namep, prqspace(p->qspacep));
#endif
		if (qsp >= qspmax)
		{
		    printf("Exceeded limit of %d qspaces\n", MAXQSPACES);
		    exit(1);
		}

		/*  Increment number of landmarks  */
		qsp->nlmarks = p->qspacep->nlmarks + 1;
		qsp->zerox   = p->qspacep->zerox;
		qsp->minf    = p->qspacep->minf;
		qsp->inf     = p->qspacep->inf;

		/*  Compute index of new landmark in new qspace.  Its value
		 *  is saved in 'param' for later use in adjusting correspond-
		 *  ing values.
		 */

		/*  Adjust qmag index since this is now a point value  */
		p->newx = p->qmagx = qvp->qmagx = ++qmagx;

		/*  Adjust index of the zero landmark if necessary  */
		if (qsp->zerox >= qvp->qmagx)
		    qsp->zerox += 2;

		/*  Copy first part of old qspace to new qspace  */
		qsp->lmarkp = lmp;
		lmarkp = p->qspacep->lmarkp;
		x = qvp->qmagx >> 1;
		for (j = 0;  j < x;  j++)
		    *lmp++ = *lmarkp++;

		/*  Create new landmark name  */
		sprintf(symp, "%s%d", p->prnamep, p->suffix++);
		symbol[nsymbols] = symp;
		symp += strlen(symp) + 1;

		/*  Insert new landmark at index 'x'  */
		*lmp++ = nsymbols++;

		/*  Copy remaining part of old qspace to new qspace  */
		for ( ;  j < p->qspacep->nlmarks;  j++)
		    *lmp++ = *lmarkp++;

		if (lmp > lmpmax)
		{
		    printf("Exceeded limit of %d landmarks\n", MAXLMARKS);
		    exit(1);
		}

		/*  Point the parameter's qval to the new qspace  */
		p->qspacep = qvp->qspacep = qsp++;
#ifdef TRACE
		if (trace & TR_LANDMARKS)
		    printf("   new landmark for %s in %s\n",
			p->namep, prqspace(qvp->qspacep));
#endif
	    }
	}

	/*  Test for divergence only if qmag is a point  */
	if (point(qmagx))
	    if (((qmagx == 0) && (qdir == DEC)) ||
		((qmagx>>1 == qvp->qspacep->nlmarks - 1) && (qdir == INC)))
		    divergence = TRUE;
    }

    /*  If this state is same as previous state, then don't save it.  */
    if (nochange)
    {   /* restore qvp to original value, i.e., discard the qvals. */
	qvp = statep->qvalp;
	lmp = lmpsave;
#ifdef TRACE
	if (trace & TR_STATES)
	    puts("No-change filter: state matches its immediate predecessor");
#endif
	return;
    }

    statep->checksum = checksum;


    /*  CHECK FOR CYCLE:
     *  If the candidate state is at a time point and it has no new landmarks
     *  and it has a predecessor (i.e., it is not the starting state), then
     *  check through its predecessors (at time points only) for an identical
     *  state.  If found, update predecessor and successor lists to show that
     *  a cycle has occurred (and discard the candidate state).
     */

    if ( point(statep->time)  &&  !newlmarks  &&  !nopredecessor )
    {
	/*  Get pointer to predecessor at next earliest time point  */
	prevp = curstatep->predp[0];

	/*  Loop through predecessors at previous time points  */
	do
	{
	      
	    /*  Checksum is used for fast rejection  */
	    if (checksum == prevp->checksum)
	    {
		/*  Loop over all parameters, test for exact match */
		prevqvalp = prevp->qvalp;
		for (i = 0;  i < nparams;  i++, prevqvalp++)
		{
		    p = &param[i];
		    if (p->qspacep != prevqvalp->qspacep)
			goto out;
		    if (p->qdir != prevqvalp->qdir)
			goto nextstate;
		    qmagx = p->qmagx;
		    if (interval(qmagx))
			goto nextstate;
		    if (qmagx != prevqvalp->qmagx)
			goto nextstate;
		}

		/*  Candidate state at this time point matches a predecessor
		 *  at an earlier time point.  So, indicate a cycle by making
		 *  previous state a successor of current state and by making
		 *  current state a predecessor of previous state.  Note that:
		 *  -- curstatep points to current state,
		 *  -- statep points to the candidate state (which is
		 *     an immediate successor of current state), and
		 *  -- prevp points the matching previous state.
		 */

		if (curstatep->nsuccs >= MAXNEIGHBORS)
		{
		    printf("Exceeded limit of %d successors\n",
			MAXNEIGHBORS);
		    exit(1);
		}
		curstatep->succp[curstatep->nsuccs++] = prevp;

		if (prevp->npreds >= MAXNEIGHBORS)
		{
		    printf("Exceeded limit of %d predecessors\n",
			MAXNEIGHBORS);
		    exit(1);
		}
		prevp->predp[prevp->npreds++] = curstatep;
#ifdef TRACE
		if (trace & TR_STATES)
		    puts("Cycle filter: state matches a predecessor");
#endif
		/*
		 *  Note that we return here since we are not forming
		 *  a new state.  Discard the qvals.
		 */
		qvp = statep->qvalp;
		return;
	    }
nextstate:  continue;
	}
	while ((prevp->npreds > 0)  &&  (prevp = prevp->predp[0]->predp[0]));
    }
out:

    /*  Candidate state has survived all tests, so make it official.  */
    stp++;
    nstates++;
    if (!nopredecessor)
    {
	if (curstatep->nsuccs >= MAXNEIGHBORS)
	{
	    printf("Exceeded limit of %d successors\n", MAXNEIGHBORS);
	    exit(1);
	}
	curstatep->succp[curstatep->nsuccs++] = statep;
    }

#ifdef  TRACE
    if (trace & TR_STATES)
	prstate( statep );
#endif

    /*  INSTALL CORRESPONDING VALUES:
     *  Now, for each constraint, install its corresponding values into this
     *  state.  If the constraint's arguments are all point values and are
     *  not already corresponding values, and the constraint is not d//dt or
     *  MULT(f,g,h) where h = 0, then add the new corresponding values.
     */
    newcorr = FALSE;
    crpsave  = crp;
    scrpsave = scrp;
    statep->scorrp = scrp;

    /*  Loop over each constraint  */
    for (i = 0;  i < nconstraints;  i++, scrp++)
    {
	if (scrp >= scrpmax)
	{
	    printf("Exceeded limit of %d state correspondences\n", MAXSCORR);
	    exit(1);
	}

	conp = &constraint[i];
	scrp->ncorrs = conp->ncorrs;
	scrp->corrp  = conp->corrp;

	/*  Derivatives don't have corresponding values, so skip them.  */
	if (conp->type == DERIV)  continue;

	/*  Copy all the existing correspondences of this constraint and
	 *  adjust any cval indices due to a new landmark in a qspace.
	 */
	corrp    = conp->corrp;
	corrstop = corrp + conp->ncorrs;
	scrp->corrp  = crp;
	while (corrp < corrstop)
	{
	    /*  Loop over all arguments of this correspondence  */
	    for (j = 0;  j < conp->nargs;  j++)
	    {
		x = corrp->cvalx[j];
		if (x >= conp->paramp[j]->newx)
		{
		    crp->cvalx[j] = x + 2;
		    newcorr = TRUE;
		}
		else
		    crp->cvalx[j] = x;
	    }
#ifdef TRACE
	    if (trace & TR_CVALS)
		prcorr( conp, crp );
#endif
	    crp++;  corrp++;
	}

	/*  The correspondences for this constraint have been installed.
	 *  Now see if a new correspondence should be created based on
	 *  the parameter values in this state.
	 */

	/*  If MULT(f,g,h) and h==0 then don't create new correspondence.  */
	if ((conp->type == MULT) &&
	 (conp->paramp[2]->qmagx == conp->paramp[2]->qspacep->zerox))
	    continue;

	/*  Loop over the arguments of this constraint  */
	for (j = 0;  j < conp->nargs;  j++)
	{
	    /*  Save qmags in a more readily accessible form for future use. */
	    myqmagx[j] = x = conp->paramp[j]->qmagx;

	    /*  Make sure that all arguments are point values.  */
	    if (interval(x)) goto nextcon;
	}

	/*  All of the arguments for this constraint are point values, so
	 *  we need to see if they match any of the existing corresponding
	 *  values.  If not, then we will add it.
	 */

	/*  Loop over all existing corresponding values for this constraint */
	corrp = scrp->corrp;
	corrstop = corrp + scrp->ncorrs;
	for ( ;  corrp < corrstop;  corrp++)
	{
	    /*  Loop over all arguments of this constraint  */
	    for (j = 0;  j < conp->nargs;  j++)
		if (myqmagx[j] != corrp->cvalx[j])
		    goto nextcorr;

	    /*  Parameters match an existing correspondence, so just proceed
	     *  to the next constraint.
	     */
/*
 printf("Current qmagx's are exact match for:\n");
 prcorr(conp,corrp);
*/
	    goto nextcon;

nextcorr:   continue;
	}

	/*  No match found, so create new corresponding value.  */
#ifdef TRACE
	if (trace & TR_CVALS)
	{
	    printf("Before adding correspondence, ncorrs=%d\n", scrp->ncorrs);
	    /*  prcon( conp );  */
	}
#endif


	/*  Add the new set of corresponding values  */
	for (j = 0;  j < conp->nargs;  j++)
	    crp->cvalx[j] = myqmagx[j];

	scrp->ncorrs++;
	crp++;
	newcorr = TRUE;

	if (crp >= crpmax)
	{
	    printf("Exceeded limit of %d sets of correspondences\n", MAXCORR);
	    exit(1);
	}

#ifdef TRACE
	if (trace & TR_CVALS)
	{
	    printf("New correspondence below, ncorrs=%d, crp=%x\n",
		 scrp->ncorrs, crp-1);
	    prcorr( conp, crp - 1 );
	}
#endif

nextcon: continue;
    }

    /*  Restore the old qspace for each parameter so that any other states
     *  formed by 'formstates' will be using the qspaces of the state
     *  that we are predicting from.
     */
    for (i = 0;  i < nparams;  i++)
    {
	p = &param[i];
	p->qspacep = qspsave[i];
    }

    /*  If there have been no changes to any of the correspondences, then
     *  we can save memory space by pointing this state to its predecessor's
     *  correspondences (and then restoring the old values of crp and scrp).
     */
    if ( !newcorr  &&  curstatep )
    {
	statep->scorrp = curstatep->scorrp;
	crp  = crpsave;
	scrp = scrpsave;
    }

    /*  Add new state to agenda only if not quiescent and not divergent  */
    if (quiescence  &&  !(trace & TR_QUIESCENT))
    {
#ifdef TRACE
	if (trace & TR_STATES)
	    puts("State is quiescent, so it is left off agenda");
#endif
	return;
    }

    if (divergence  &&  !(trace & TR_DIVERGENT))
    {
#ifdef TRACE
	if (trace & TR_STATES)
	    puts("State is divergent, so it is left off agenda");
#endif
	return;
    }

    addagenda(statep);
}
