(in-package :qsim)


(defmacro varname (v)
  `(if (variable-p ,v)
       (variable-name ,v)
       ,v))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Functions used to canonicalize various structures
;;;
;;;
		     
(defmacro canonicalize-lm (lm qspace)
  `(cond ((lmark-p ,lm)
	  ,lm)
         ((and ,qspace
	   (lmark-find ,lm ,qspace)))
         (t (cerror "continue" "Cannot find a canonical form of ~a.  Qspace is ~a"
	     ,lm ,qspace))))

(defun canonicalize-qvals (qvals &optional (state nil))
  "This function will receive a list of qval pairs.  Each qval-pair can be in a number of
formats.  These formats include:
     qval-pair --> (var . qval)
     qval-pair --> (var qval)
     var       --> variable structure
     var       --> symbol
     qval      --> (qmag qdir)
     qval      --> qval-structure
     qmag      -->  lm
     qmag      -->  (lm lm)
     qdir      --> inc | std | dec
     qdir      --> (inc | std | dec)+
     lm        --> lmark-structure
     lm        --> symbol
     qval-strcuture: a structure of type qval with a qmag slot and a qdir slot.
It will return a qval pair in the following canonical format.
     qval-pair --> (var qval)
     var       --> symbol
     qval      --> (qmag qdir)
     qmag      --> lm | (lm lm)
     lm        --> lmark structure
     qdir      --> inc | std | dec | (inc | std | dec)+."
  (mapcar #'(lambda (qval-pair)
	      (let* ((var (car qval-pair))
		     (qval (if (listp (cdr qval-pair))
			       (cadr qval-pair)
			       (cdr qval-pair)))
		     (canonical-varname (if (variable-p var)
					    (variable-name var)
					    var)))
		(list canonical-varname
		      (canonicalize-qval qval (when state (qspace canonical-varname state))))))
	    qvals))


(defun canonicalize-qval (qval &optional (qspace nil))
  (let ((qmag (if (qval-p qval)
		  (qmag qval)
		  (canonicalize-qmag (car qval) qspace)))
	(qdir (if (qval-p qval)
		  (qdir qval)
		  (cadr qval))))
    (list qmag qdir)))


(defun canonicalize-qmag (qmag &optional (qspace nil))
  (cond ((null qmag) nil)
	((listp qmag)
	 (list (canonicalize-lm (car qmag) qspace)
	       (canonicalize-lm (cadr qmag) qspace)))
	(t (canonicalize-lm qmag qspace))))


(defun var-equal (v1 v2)
  (equal (varname v1)
	 (varname v2)))

(defun landmark-gt (a b qspace)
  (member a (cdr (member b qspace))))

(defun landmark-ge (a b qspace)
  (member a (member b qspace)))



;;;  FUNCTIONS FROM QDEFS.LISP

;;  Variables used in the successor explanation code.
;;  DJC 11/25/92
(defparameter *explain-successors* nil)
(defparameter *successors-explanation* nil)


;;;  FUNCTIONS FROM QUTILS.LISP


(defun pprint-at (SEXP col &optional (stream t))
  "Pretty print SEXP starting at column COL.  Note that in CLtL2, this
is probably easier to do with (pprint-index...)."
  (with-input-from-string
      (in (with-output-to-string (out)
	    (write sexp :stream out :case :downcase :pretty t)))
    (loop
     ;; not the best way to get the spaces!  Separate writes feels
     ;; slower under emacs, however.
     with spaces = (make-string col :initial-element #\space)
     for line = (read-line in nil nil)
     while line
     do
     (format stream "~&~a~a~%" spaces line)))
  (values))


;;;  TAKEN FROM STATES.LISP


(defun get-ignore-qdirs (&key state (sim (state-sim state))
			      (qde (if state
				       (state-qde state)
				       (sim-qde sim))))
  "Compiles a list of QDIRs being ignored using both the SIM and the QDE."
  (union (cdr (assoc 'ignore-qdirs (qde-other qde)))
	 (sim-ignore-qdirs sim)))

(defparameter *unreachables-early* nil
  "Eliminates unrechable values at an earlier point")

(defun eliminate-unreachable-values (var qvals)
   (if *unreachables-early*
       (delete-if #'(lambda (qval)
		      (member (qmag qval) (variable-unreachable-values var)))
		  qvals)
       qvals))

;;; Added DJC 03/16/93
(defun check-status (state check)
  "Checks to see if there is an intersection between (state-status STATE) and
the statuses in CHECK.  Check can either be an atom or a list of atoms.  It will
return the first status entry from state-status that intersects CHECK.  For each
state-status entry it checks to see if it is a list.  If it is, then is
uses the car of this list.  Thus, status entries can be made as a list with
additional information in the cdr."
  (let ((check-list (listify check)))
    (loop for entry in (state-status state)
	  for search-entry = (if (listp entry) (car entry) entry)
	  when (member search-entry check-list)
	  return entry)))


(defun record-successors-explanation (state)
  (when *explain-successors*
    (setf (state-successors-explanation state)
	  (nreverse *successors-explanation*))))



(defun get-all-states (&optional (initial-state *initial-state*)
				 &key (successor-function #'get-successors))
  "This function returns a list of all of the states in the graph/tree rooted in
*initial-state*."
  (cond ((null initial-state) nil)
	(t (cons initial-state
		 (mapcan #'(lambda (state)
			     (get-all-states state :successor-function successor-function))
			 (funcall successor-function initial-state))))))

;;;-----------------------------------------------------------------------------
;;; This function permits specialized copying of the other slot of
;;; a state.  Some slots do not want a full copy-tree, and so using this
;;; call rather than a copy-tree will copy just the right part of
;;; those structures.  It should be in place of copy-tree when
;;; copying the state.other slot.
;;; Inputs:  other-slots  - The contents of the state-other slot.
;;; Returns: a copy of other-slots for use in a new state.
;;; Note: This function is meant to behave like copy-tree.
;;; BKay 23May92
;;;   Modified to have an explicit list of dont-copy slots.
;;;   The default should probably be *not* to copy, but this preserves
;;;   compatibility.  BJK 8-5-92
;;;-----------------------------------------------------------------------------
;;;
(defvar *state-other-dont-copy*		; don't copy these other-slots
  '(predecessors stability-test abstracted-constraints
    corresponding-states corresponding-landmarks
    fast-completions slow-successors))

(defun copy-state-other (other-slots)
  (loop for slot in other-slots
	unless (member (car slot) *state-other-dont-copy*)
	collect (case (car slot)
		  (eqn-index (list 'eqn-index (copy-eqn-index (second slot))))
		  ;; Add other slot-specific copying functions here
		  (T         (copy-tree slot)) ; default = copy
		  )))

(defun new-state-from-old-state (state qvalues &optional (text nil))
  (let* ((nstate 
	   (make-state :qde           (state-qde state)
		       :qvalues       qvalues
		       :qspaces       (copy-alist (state-qspaces state))
		       :cvalues       (copy-alist (state-cvalues state))
		       :other         (copy-state-other (state-other state)) ; BKay 26May92
		       :text          text
		       :justification `(initialized-with ,qvalues)
		       :name          (genname 'S)
		       )))

    (setf (state-sim nstate) (or *current-sim*       ; added DJC 07/15/91
				 (state-sim state))) ; added BJK 08/05/92
    ;; The statement to set time (below) is not actually necessary because
    ;; time is included in the qvalues list.
    ;; (setf (state-time nstate) (state-time state))
    (set (state-name nstate) nstate)
    nstate))

(defun successor-state-from-predecessor (state qvalues)
  ;; No need to copy-alist qvalues here since caller is providing a fresh alist.
  (let* ((nstate
	  (make-state :qde       (state-qde state)
		      :qvalues   qvalues
		      :qspaces   (copy-alist (state-qspaces state))
		      :cvalues   (copy-alist (state-cvalues state))
		      :other     (copy-state-other (state-other state)) ; BKay 26May92
		      :name      (genname 'S)
		      )))
  
    (setf (state-sim nstate) *current-sim*)     ; added DJC 07/15/91
    (set (state-name nstate) nstate)    ;; Note: SET, not SETF
    nstate))


;;; Check whether a state should go onto the agenda or not.

(defparameter *statuses-to-filter-from-agenda* '(inconsistent self-intersection cycle final-state))

(defun filter-for-agenda (state)
  (let ((status (state-status state)))
    (cond ((null status) (list state))
	  ((intersection status *statuses-to-filter-from-agenda*)
	   nil)
	  ((intersection status '(incomplete transition OK))
	   (error "Agenda shouldn't see state ~a with status ~a."
		  state (state-status state)))
	  ((member 'quiescent status) (when *quiescent-ok-for-agenda*
					(list state)))
	  ((member 'complete status) (list state))
	  (t (error "State ~a has unknown status ~a."
		    state (state-status state))))))

(defparameter *prune-inconsistent-states* t)

(defun filter-pvals-for-explanation (state)
  ;; Used in successor explanation.  It filters the pvals according to
  ;; a template provided by the user.
  ;; Ensure that all of the template values are possible values and
  ;; delete those possible values for the variables that are in the
  ;; template that do not match the template
  (when *explain-successors*
    (let ((qvals-template (if (equal *explain-successors* T)
			      nil
			      *explain-successors*))
	  (vars (if (equal *explain-successors* T)
		    nil
		    (mapcar #'car *explain-successors*))))
      (loop for qv-template-pair in qvals-template
	    for var-name in vars
	    for var = (var var-name state)
	    for qval-template = (cadr qv-template-pair)
	    for pvals = (variable--pvals var)
	    do (setf (variable--pvals var)
		     (remove-if-not #'(lambda (pval)
					(match-qval-template qval-template pval))
				    (variable--pvals var)))
	    do (unless (variable--pvals var)
		 (pushnew (list 'transition-table qval-template pvals)
			  *successors-explanation*))
	    ))))

(defun match-qval-template (template qval)
  (and (or (null (car template))
	   (qmag-equal (car template)
		       (qmag qval)))
       (or (null (cadr template))
	   (equal (cadr template)
		  (qdir qval)))))
	

(defun end-simulation-p (sim)
  (or (and (sim-state-limit sim) (> (sim-state-count sim) (sim-state-limit sim)))
      (if (equal (sim-simulation-type sim) :breadth-first)
	  (let ((time-bound (sim-time-limit sim))
		(states (sim-agenda sim)))
	    (and (every #'(lambda (state)
			    (later-time? (qmag (state-time state))
					 time-bound
					 :or-equal))
			states)
		 (sim-time-limit sim)))
	  (let ((time (state-time (first (sim-agenda sim)))))
	    (and (sim-time-limit sim)
		 (qpointp time)
		 (eq (lmark-name (qmag time)) (sim-time-limit sim))
		 (sim-time-limit sim))))))



;;;  FUNCTIONS TAKEN FROM GLOBAL-FILTERS.LISP

(defun add-global-state-filter (filter)
  "Adds FILTER to the global state filter list.  It adds it to the end of the list."
  (unless (member filter *global-state-filters*)
    (setf *global-state-filters*
	  (append *global-state-filters*
		  (list filter)))))

(defun delete-global-state-filter (filter)
  "Deletes FILTER from the global state filter list."
  (when (member filter *global-state-filters*)
    (setf *global-state-filters*
	  (delete filter *global-state-filters*))))

(defun add-global-state-analysis (filter)
  "Adds FILTER to the global state analysis list.  It adds it to the end of the list."
  (unless (member filter *global-state-analysis*)
    (setf *global-state-analysis*
	  (append *global-state-analysis*
		  (list filter)))))

(defun delete-global-state-analysis (filter)
  "Deletes FILTER from the global state analysis list."
  (when (member filter *global-state-analysis*)
    (setf *global-state-analysis*
	  (delete filter *global-state-analysis*))))




;;;  TAKEN FROM STRUCUTRES.LISP

(defun copy-qval (qval)
  (make-qval :variable (qval-variable qval)
	     :qmag (if (listp (qval-qmag qval))
		       (copy-list (qval-qmag qval))
		       (qval-qmag qval))
	     :qdir (if (listp (qval-qdir qval))
		       (copy-list (qval-qdir qval))
		       (qval-qdir qval))
	     :other (qval-other qval)))

(defun COPY-QVALUES (qvalues)
  (mapcar #'(lambda (qvalue)
	      (cons (car qvalue) (copy-qval (cdr qvalue))))
	  qvalues))


