(in-package :qsim)

(defun aggregate-state? (state)
  (declare (ignore state))
  nil)

;;;  FROM XEDGE.LISP

(defun user-qval (qval)
  "Return a list (qmag qdir) of symbols for qmag and qdir."
  (list (user-qmag qval)
	(order-qdir (qdir qval))))

(defun order-qdir (qdir)
  (cond ((atom qdir) qdir)
	(t (nconc (and (member 'inc qdir)
		       (list 'inc))
		  (and (member 'std qdir)
		       (list 'std))
		  (and (member 'dec qdir)
		       (list 'dec))))))


(defun qmag-equal (qmag1 qmag2)
  (cond ((and (point-p qmag1)
	      (point-p qmag2))
	  (robust-lmark-equal qmag1 qmag2))
	((and (interval-p qmag1)
	      (interval-p qmag2))
	 (and (robust-lmark-equal (car qmag1)
				  (car qmag2))
	      (robust-lmark-equal (cadr qmag1)
				  (cadr qmag2))
	      (or (equal (third qmag1) (third qmag2))
		  (and (null qmag1) (equal qmag2 :oo))
		  (and (null qmag2) (equal qmag1 :oo)))))
	 (t nil)))

(defun QVAL-PRINTER (qval stream ignore)
  (declare (ignore ignore))  ;;  added DJC porting to the Sun
  "Print-function for structure qval."
  (let ((qmag (qval-qmag qval))
	(qdir (qval-qdir qval)))
    (cond (*detailed-printing*
	   (format stream "#<Qval ~(~A~): (~A ~A)>"
		   (variable-name (qval-variable qval)) (print-qmag qmag) qdir))
	  (*short-qval-printing*
	   (format stream "~A,~A"
		   (cond ((null qmag) "?")
			 ((listp qmag) (print-qmag (substitute "?" nil qmag)))
			 (t qmag))
		   (case qdir
		     (std "=")
		     (inc %up-arrow-char)
		     (dec %down-arrow-char)
		     (ign "*")
		     ((nil) "?")
		     (t (format nil "~(~A~)" qdir)))))
	  (t ; Ordinary printing
	   (format stream "(~A ~(~A~))" (print-qmag qmag) qdir)))))

(defun print-qmag (qmag)
  (if (or (atom qmag) 
	  (null (caddr qmag)))
      qmag
      (format nil "~a~a ~a~a"
	      (if (lower-boundary-closed? qmag)
		  "[" "(")
	      (car qmag)
	      (cadr qmag)
	      (if (upper-boundary-closed? qmag)
		  "]" ")"))))
		  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Functions that are called from these QSIM baseline functions that have
;;;  been modified.  They should be included in the file that has the appropriate function.


(defun trans-abs-qdir-for-display (qdir)
  (cond ((qdir-equal qdir '(inc std dec))
	 :inc-std-dec)
	((qdir-equal qdir '(inc std))
	 :inc-std)
	((qdir-equal qdir '(dec std))
	 :dec-std)))


(defun abstracted-interval? (state var)
  (let ((qmag (qmag (qval var state)))
	(qspace (qspace var state)))
    (and (interval-p qmag)
	 (not (robust-lmark-equal (cadr qmag)
				  (succ (car qmag) qspace))))))

(defun mark-if-chatter-box (states)
  (mapcar #'(lambda (state)
	      (let ((chvars (chatter-vars state)))
		(when chvars (record-chatter-box-info state nil nil nil chvars))))
	  states))


(defun check-qdir-combinations (state)
  "This function takes a state with abstracted qdirs and it checks all possible
combinations of the abstracted qdirs and determines what the possible
successors of this state should be.  This determines if there should
be any successors of the state with the abstracted qdir."
  (let* ((chvars (chatter-vars state))
	 (ch-dirs (mapcar #'(lambda (var)
			      (qdir (qval var state)))
			  chvars))
	 (poss-dir-combs (cross-product ch-dirs))
	 (comps
	  (loop for poss-dir-set in poss-dir-combs
		for init-state = (state-copy state)
		do (mapcar #'(lambda (var qdir)
			       (setf (qdir (qval var init-state))
				     qdir))
		     chvars poss-dir-set)
		collect (list init-state (successors-of-state init-state)) into ret-val
		finally (return ret-val))))
    (mapcar #'(lambda (poss-dir-set succs)
		(format t "~%Qdir Set: ~a~%  init-state   : ~a~%   successors :~a"
			poss-dir-set (car succs) (cdr succs)))
	    poss-dir-combs comps)
    comps))



(defun cross-product (lists)
  (cond ((null lists) nil)
	((null (cdr lists))
	 (mapcar #'(lambda (elem)
		     (list elem))
		 (car lists)))
	(t (let ((rem-x-prod (cross-product (cdr lists))))
	     (mapcan #'(lambda (elem)
			 (mapcar #'(lambda (x-prod)
				     (cons elem x-prod))
				 rem-x-prod))
		     (car lists))))))
  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;;                                                                       ;;;
;;;   SOURCE CODE MODS                                                    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defparameter *display-chatter-box* t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Taken from TIME-PLOT.LISP


(defun QSYMBOL-VLOC (qval yalloc &key (param nil) n behavior state abs-qmag)
  (declare (ignore state))
  (let ((qmag (qval-qmag qval)))
    (cond ((null qmag) 0.)
	  ((qmag-point-p qmag) (must-retrieve-lmark qmag yalloc))
	  ((member nil qmag) 0.)
	  (abs-qmag (/ (+ (must-retrieve-lmark (car qmag) yalloc)
		     (must-retrieve-lmark (cadr qmag) yalloc))
		  2))
	  (t (guess-vloc qval yalloc param n behavior)))))


;;; PLOT-PARAMETER is the function that draws one of the little boxes in
;;; the time display, the label below it, and everything that goes
;;; inside it.  Qval will be NIL for some states when behavior goes thru
;;; multiple QDEs.

;;; Modified so that it will handle aggregate-intervals.
;;; 15 May 1991   DJC
;;; Modified so that it can print intervals explicitly
;;; 04 Jun 1991   PF

(defun PLOT-PARAMETER
       (param refpoints behavior xpos ypos xsize ysize xalloc &key (label))
  (loop with yalloc = (allocate-v-axis param behavior ysize)
	with behavior-xalloc = (member (first behavior) xalloc :key #'car)
        with param-found-in-behavior-p	; Flag gets set T if parameter appears
	= (plot-refpoints param refpoints	;  in either refpoints or behavior.
			  xpos ypos xalloc yalloc)
        for ostate = nil then state
        for state in behavior
	for qspace = (qspace param state)
	;; state--x-offset added, PF 03/06/91
	for state--x-offset in behavior-xalloc
	;; for x-offset = (lookup state xalloc)
	for x-offset = (second state--x-offset)
        for count from 0
        for qval = (if (agg-interval-p state)
		       (first-qval-in-history param state)
		       (qval param state))
        for ox = nil then x		; ox,oy inherit value of x,y
        for oy = nil then y		;  from previous loop cycle.
        for x = (when qval (+ xpos x-offset))
        for y = (when qval
                  (+ ypos (qsymbol-vloc qval yalloc :param param
                                        :n count :behavior behavior
					:abs-qmag (abs-qmag (qmag qval) qspace))))
	;; Variables needed to display an interval as a vertical bar
	for qmag = (when qval (qval-qmag qval))
	for interval-p = (when qmag (qmag-interval-p qmag))
	for y1 = (when interval-p (+ ypos (must-retrieve-lmark (first  qmag) yalloc)))
	for y2 = (when interval-p (+ y 7))
	for y3 = (when interval-p (- y 7))
	for y4 = (when interval-p (+ ypos (must-retrieve-lmark (second qmag) yalloc)))
        ;; Keep flag of whether the parameter occurs in this behavior at all.
	do (setf param-found-in-behavior-p
		 (or param-found-in-behavior-p qval))
        when (and (numberp ox)(numberp x))	; Connect states with dots 
	do (if (perturbed-p ostate)	;  or arrow.
	       (perturb-link ox oy x y)
	       (connect-with-dots ox oy x y))
        when qval
	do (qplot-point-label x y param state behavior)
	;; Plot the vertical bar
	(when (or (and interval-p *plot-intervals*)
		  (abstracted-interval? state param)) ;; modified DJC to display
	                                             ;; abstracted intervals
	  (qplot-line x y1 x y2 :dashed t :dash-pattern '(1 3))
	  (qplot-line x y3 x y4 :dashed t :dash-pattern '(1 3)))
	(qplot-symbol x y (if (abs-qdir (qdir qval))
				     (trans-abs-qdir-for-display (qdir qval))
				     (qdir qval)))
	when (and param-found-in-behavior-p
		  (agg-interval-p state))
	  do (do* ((num-times (1- (max-history-size state)) (1- num-times))
		   (qvals-in-hist (qvals-in-history param state))
		   (qvals (if (cdr qvals-in-hist)
			      (cdr qvals-in-hist)
			      qvals-in-hist)
			  (if (cdr qvals)
			      (cdr qvals)
			      qvals))
		   (qval (car qvals) (car qvals)))
		  ((= num-times 0))
	       (setq ox x)
	       (setq oy y)
	       (setq x (round (+ x Hshare)))
	       (setq y (+ ypos (qsymbol-vloc qval yalloc :param param
					     :n count :behavior behavior)))
	       (when (and (numberp ox) (numberp x))
		 (connect-with-dots ox oy x y))
	       (qplot-symbol x y (if (abs-qdir (qdir qval))
				     (trans-abs-qdir-for-display (qdir qval))
				     (qdir qval))))
        finally (when param-found-in-behavior-p
                  (qplot-frame param refpoints behavior xpos ypos
                               xsize ysize xalloc yalloc :label
                               (or label (get-box-label param behavior))))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   Taken from QPLOT.LISP


(defun QPLOT-STATE (state x y &optional status)
  "Plot and perhaps number a state in a state tree diagram."
  ;; Status is optional, so it may be overwritten.
  (unless status (setq status (state-status state)))
  (let* ((ret-val nil)
	 (size 3)				        	; BJK:  10-25-90
	 (index (cond ((member 'cycle status)                   ; DJC:  03-15-92
		       ;; If state ends a cycle, also record in the state index
		       ;; the state that begins the cycle.
		       (concatenate 'string
				    (get-index-string (state-name state))
				    " = "
				    (get-index-string (state-name (cadr (state-successors state))))))
		      ((member 'cross-edge (state-status state))
		       (concatenate 'string
				    (get-index-string (state-name state))
				    " = "
				    (get-index-string (state-name (state-cross-edge-identity state)))))
		      (t
		       (get-index-string (state-name state)))))
	 (fill   (time-point-p (state-time state))))
    ;; Print state index (name stripped of leading "s-") near state symbol.
    (case *plot-state-indices*
      (:right-of-node
       (qplot-string index (+ x 5) (- y 1) :font axis-font))
      (:above-node
       (qplot-string index
		     (- x 5) (- y 5) :font axis-font))
      (:at-node
       (qplot-string index
		     (- x 5) (+ y 4) :font axis-font))) 
    ;; Plot state symbol, unless index takes its place.
    (unless (eq *plot-state-indices* :at-node)
      (cond
	;; Plot a box when the state is an aggregate state
	((aggregate-state? state)
	 (qplot-box (- x 2) (- y 5) 14 10)
	 (setq ret-val 10))
	;; Added DJC with the Cross edge detect code  03/6/92
	;; Final state displayed as a small disc inside a larger circle.
	((intersection status '(QUIESCENT FINAL-STATE))
	 (qplot-circle x y (+ size 1))
	 (qplot-circle x y (- size 1) :alu *black* :filled t))
	;; Cycle displayed as small circle inside large circle.
	((member 'CYCLE status) (qplot-circle x y (+ size 1))
	                        (qplot-circle x y (- size 1)))
	;; Cross edge displayed as a circle with an X through it
	((member 'CROSS-EDGE status) (qplot-circle x y (+ size 1))
		                     (qplot-line (- x (* .5 size)) (- y (* .5 size)) 
						 (+ x (* .5 size)) (+ y (* .5 size))))
	;; Transition state displayed as large circle with vertical line through it.
	((member 'TRANSITION status) (qplot-circle x y (+ size 1))
		                     (qplot-line x (- y size) x (+ y size)))
	;; Display a box when a state is a chatter abstraction state and is not
	;; also one of the above statuses  DJC
	((and *display-chatter-box*
	      (chatter-box-state? state))
	 (qplot-box (- x 2) (- y 4) 5 7 :filled fill)
	 (when (and (not (member 'OK status))
		    (null (successor-states state)))
	   (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3))))
	;; Unfinished state displayed with trailing dashed line.
	((null status)
	 (qplot-circle x y size :filled fill)
	 (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3)))
	;; Normal intermediate state.
	((member 'OK status)
	 (qplot-circle x y size :alu *black* :filled fill))
	;; Abstraction of chattering states displayed as a star
	((member 'CHATTER status)
	 (qplot-symbol x y 'ign))
	;;(qplot-box (- x size) (- y size) (* 2 size) (* 2 size) :thickness 1)
	;;(qplot-circle x y (+ size 1))
	;;(qplot-line (- x size) (- y size) (+ x size) (+ y size))
	;;(qplot-line (- x size) (+ y size) (+ x size) (- y size))
	;; States aggregating occurence branching displayed as boxes
	((member 'OCCURRENCE-BRANCHING status)
	 (qplot-box (- x size) (- y size) (* 2 size) (* 2 size) :thickness 2))
	;; Inconsistent state is plotted normally, later marked with "X".
	((intersection status '(INCONSISTENT SELF-INTERSECTION))
	 (qplot-circle x y size :alu *black* :filled fill))
	;; Unfinished state displayed with trailing dashed line.
	((member 'COMPLETE status)
	 (qplot-circle x y size :filled fill)
	 (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3)))
	;; Unknown status displayed as "?".
	(T (qplot-circle x y (+ size 5))		
	   (qplot-symbol x y '?)))     
      (cond ((intersection status '(INCONSISTENT SELF-INTERSECTION))
	     ;; Inconsistent state is marked with "X", in addition to whatever else.
	     (qplot-line (- x size -10) (- y size) (+ x size 10) (+ y size))
	     (qplot-line (- x size -10) (+ y size) (+ x size 10) (- y size)))))  
    ;; To mark a state in the display add (MARK-STATE <marking-type>) to the status slot.
    ;; Currently the <marking-type> can either be :arrow for an arrow or a string
    ;; added DJC 05/4/92
    (when *mark-states*   
      (let ((marking (cadar (member 'MARK-STATE status 
				    :test #'(lambda (s1 s2)
					      (when (listp s2)
						(equal s1 (car s2))))))))
	(cond ((equal marking :arrow)
	       (qplot-symbol x (+ y (* 3 size)) :inc))
	      ((stringp marking)
	       (unless (equal *plot-state-indices* :above-node)
		 (qplot-string marking (- x 3) (- y 5) :font axis-font))))))
    ret-val))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   Taken from TRANSITIONS.LISP


;;;
;;; THe changes to CREATE-TRANSITION-STATE and CREATE-NEW-LANDMARK
;;; were required so that abstracted qdirs would be maintained across
;;; transitions.

(defun CREATE-TRANSITION-STATE
       (&key from-state to-qde assert inherit-qmag inherit-qdir
	     inherit-ranges assert-ranges text)
  (pushnew to-qde (display-block-qdes-in-beh-tree *current-display-block*))
  ;; Add any qdirs which are specified as ignored in the second qde
  ;; This will cause these qdirs to always be ignored even if the
  ;; simulation transitions back to the original qde (if the variable
  ;; is a shared variable in the two qde's.)  DJC 10/07/92
  (when (assoc 'ignore-qdirs (qde-other to-qde))
    (setf (sim-ignore-qdirs (state-sim from-state))
	  (get-ignore-qdirs :state from-state :qde to-qde)))

  ;; STEP 1:  Make a new/initial state of the to-qde, having same state-time.
  (let ((nstate (make-state :qde       to-qde
			    :name      (genname 'S)
			    :text      (format nil "~a: Transition from ~a to ~a"
					       text
					       (qde-name (state-qde from-state))
					       (qde-name to-qde))
			    :justification `(transition-from ,from-state ,to-qde)
			    ;; This next line changed by BKay 3Sept91
			    ;; state.eqn-index and state.assert-ranges
			    ;; should not be carried into the next state.
;			    :other     (copy-tree (state-other from-state))
			    :other     (copy-state-other
					(remove-if #'(lambda (slot)
						       (member (car slot)
							       '(bindings
								 eqn-index
								 assert-ranges)))
						   (state-other from-state)))
			    )))
    (setf (state-predecessors nstate) (list from-state))
    (set (state-name nstate) nstate)

    ;; STEP 2:  Union qspaces of from-state and to-qde for the new state.
    (union-qspaces-at-transition from-state nstate to-qde)

    ;; STEP 3:  Union corresponding values of identical constraints in the
    ;;		from-state and to-qde.
    (union-cvalues-at-transition from-state nstate to-qde)

    ;; STEP 4:  Convert user values now that qspaces have been union'ed.
    (setf (state-qvalues nstate)
	  (get-transition-qvalues from-state nstate assert inherit-qmag inherit-qdir))

    ;; STEP 5:  Add the appropriate Q2 info.  This info will be passed via the
    ;;          state.assert-ranges slot.  Added BKay 15Jun92
    (set-transition-ranges nstate from-state inherit-ranges assert-ranges)
	  
    ;; STEP 6:  Complete the state.
    (cond ((complete-states-from-partial-state nstate))
	  (*prune-uncompletable-transitions*
	   (prune-inconsistent-state from-state "No completions upon transition")) 
	  (t (error "Inconsistent transition result ~a had no completions." nstate)))

    ;; If NSTATE has an abstracted qdir (i.e. is a chatter box time point) then
    ;; add a chatter-box-info structure to it and/or its completions.
    (mark-if-chatter-box (cons nstate (successor-states nstate)))
    
    ;; STEP 7:  Return new state.
    (if (member 'Inconsistent (state-status nstate))
	(values nil nstate)			; This VALUES clause is just a hook for
	nstate)))				; tracing the outcome of failed transitions.



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Taken from GLOBAL-FILTERS.LISP


(defun qmag-equivalent (qm1 qm2)
  (and qm1
       qm2
       (or (equal qm1 qm2)
	   (cond ((and (qmag-point-p qm1)
		       (qmag-point-p qm2))
		  (qmag-equivalent (car  (lmark-where-defined qm1))
				   (car  (lmark-where-defined qm2)))
		  (qmag-equivalent (cadr (lmark-where-defined qm1))
				   (cadr (lmark-where-defined qm2))))
		 ((and (qmag-interval-p qm1)
		       (qmag-interval-p qm2))
		  (qmag-equivalent (car  qm1) (car  qm2))
		  (qmag-equivalent (cadr qm1) (cadr qm2)))
		 ((and (qmag-point-p qm1)
		       (qmag-interval-p qm2))
		  (qmag-equivalent (lmark-where-defined qm1) qm2))
		 ((and (qmag-interval-p qm1)
		       (qmag-point-p qm2))
		  (qmag-equivalent (lmark-where-defined qm2) qm1))
		 ))))

	      

(defun create-new-lmark (state qv-pair qs-pair why)
  (let* ((qval (cdr qv-pair))
	 (nqval (copy-qval qval))		; newly created for the state's qval for var.
	 (var  (qval-variable qval))
	 (oqmag (qmag qval))
	 new-lmark)
    (unless (abstracted-qdir (qdir qval))    ; Added for chatter box abstraction DJC
      (setf new-lmark
	    (create-and-insert-lmark-at-point oqmag qs-pair var state why))
      (setf (qmag nqval)       new-lmark)	   ; Set the qmag of the new qval to the new lmark
      (setf (cdr qv-pair) nqval)		   ; Set the qval in the state to the new qval
      new-lmark)))



;;;  TAKEN FROM STRUCUTRES.LISP

(defun COPY-QVALUES (qvalues)
  (mapcar #'(lambda (qvalue)
	      (cons (car qvalue) (copy-qval (cdr qvalue))))
	  qvalues))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Taken from  STATES.LISP



(defun complete-states-from-partial-state (nstate)
  (let* ((start-time        (get-internal-run-time))
	 (*current-qde*               (state-qde nstate))
	 (initial-values    (copy-qvalues (state-qvalues nstate)))
	 (current-state     nstate)
	 (*filtering-condition* :completions)
	 after-propagation
	 new-states
	 value-completions1 value-completions2)
    (declare (special current-state *current-qde*))		; make visible to cfilter, etc.

    ;; Sets the qdirs for variables whose qdirs are to be
    ;; ignored.  It will access both the SIM and the QDE to
    ;; derive this list. DJC 10/07/92 
    (set-ignore-qdirs nstate (get-ignore-qdirs :state nstate))
    
    (constraint-net-for-state nstate)		; Install the constraint network
    (setq after-propagation (propagate-if-possible nstate))
    
    ;; Compute possible values for each variable.
    (dolist (var (qde-variables *current-qde*))
      (setf (variable--pvals var)
	    (eliminate-unreachable-values var (all-qvalues (variable--qval var)))))
            ;; (all-qvalues (variable--qval var))

    (setq value-completions1 (cfilter *current-qde* #'check-qsim-constraint))
    (setq value-completions2 (filter-alists-at-init *current-qde* nstate value-completions1))

    ;;  NEW-STATES will be non-nil iff nstate had multiple completions.
    ;;  Install the completion(s) in Nstate or as its successors.
    (setf new-states
	  (install-value-completions nstate initial-values after-propagation value-completions2))

    ;;  Will reduce the number of completions by combining states that which differ
    ;;  only in the qdir of potentially chattering variables.
    ;;  DJC 02/5/93
    (abstract-initial-state-completions nstate)
    
    ;; Possibly reduce the number of new states if two or more states are
    ;; similar (with respect to occurrence branching and "ignored" variables).
    (setq new-states (filter-for-similarity nstate new-states))

    (print-initialization-time-trace start-time (or new-states nstate))    
    (if *show-completion-steps*
	(print-completions nstate initial-values after-propagation value-completions2))
    
    (cond ((member 'inconsistent (state-status nstate)) nil)
	  ((member 'incomplete   (state-status nstate)) new-states)
	  (t nstate))))



;;;  Used to be called DELETE-INCONSISTENT-STATE name changed to MARK-INCONSISTENT-STATE.
;;;     Modified to eliminate the *delete-inconsistent-branches* flag.  All inconsistent
;;;     branches are maintained in the tree and *show-inconsistent-successors* determines
;;;     if they are visible to the modeler.  The visibility is controled via the 
;;;     SUCCESSOR-STATES function.  This function no longer deletes the inconsistent
;;;     states.  DJC 01/30/91


(defun mark-inconsistent-state (state predecessor reason)
  (when (and state (state-name state))
    (let* ((justification (state-justification state))
	   (successors (when (typep predecessor 'state)
			 ;; modified to call filtered-successor-slot-contents instead 
			 ;;  of state-successor   02/13/91   DJC
			 (filtered-successor-slot-contents predecessor)))
	   (cross-edge-preds (state-cross-edge-predecessors state)))
      (pushnew `inconsistent (state-status state))
      (pushnew `(inconsistent ,reason) (state-status state) :test #'equal)
      (pruning-trace state justification successors predecessor)
      (ecase (car justification)
	((unique-successor-of one-of-several-successors-of successor-of
			      perturbation-of spliced-in separated-branch-of)
	 (clear-pred predecessor))
	(one-of-several-completions-of
	 (clear-initial state)
	 (clear-pred predecessor)
	 (when (eq (car (state-justification predecessor)) 'transition-from)
	   (clear-pred (state-predecessor predecessor))))
	;; If the state  is not a member of the successors list, it
	;;  will be filtered and never installed.
	(transition-from
	 (when (member state successors)
	   (clear-pred predecessor) ))
	((initialized-with root-of-separated-tree copy-of)
	 (clear-initial state)))
      (mapcar #'(lambda (state)
		  (mark-inconsistent-state state (state-predecessor state)
					   "cross edge state pruned"))
	      cross-edge-preds)
      (when *cleanup-inconsistent-states*
	(when predecessor
	  (setf (state-successors predecessor)
		(delete state (state-successors predecessor))))
	(state-cleanup state)))))
  
(defun qsimulate-state (S sim)
  (let ((*current-qde* (state-qde s))
	(*successors-explanation* nil))
    ;; This function is wrapped with a *current-qde* call because 
    ;; q-continue calls this and the local binding of
    ;; *current-qde* by qsim will have been broken.  BKay 29May92
    (declare (special *current-qde*))
    (when (member 'incomplete (state-status S))
      (trace-qsimulate-incompletes s)
      ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
      (add-states-to-agenda (successor-states S) sim :type :depth-first)
      (return-from qsimulate-state))
    
    (cond ((not (member 'complete (state-status S)))
	   (error "State ~a should be complete by now." S))
	  ((intersection '(incomplete OK) (state-status S))
	   (error "State ~a should not have status ~a by now." S (state-status S))))
    
    (trace-qsimulate-count S (sim-agenda sim))
    
    (unless (member 'GF (state-status S))
      (apply-global-state-analysis  S)
      (apply-global-state-filters   S)
      (pushnew 'GF (state-status S)))
    
    (if (member 'inconsistent (state-status S))
	(return-from qsimulate-state))
    
    (let ((tstates (remove S (filter-for-transitions S))))
      (if tstates
	  (add-states-to-agenda tstates sim)))
    
    (when (eligable-for-successors S)
      (let ((new-states (successors-of-state S)))
	(dolist (ns new-states)
	  (apply-global-state-analysis ns)
	  (apply-global-state-filters  ns)
	  (pushnew 'GF (state-status ns)))      
	(add-states-to-agenda new-states sim)
	(record-successors-explanation S)))
    ))


(defun filter-for-agenda (state)
  (unless (null (state-name state))   ;; occurs when inconsistent states are cleaned up
    (let ((status (state-status state)))
      (cond ((null status) (list state))
	    ((check-status state *statuses-to-filter-from-agenda*) ; DJC 03/16/93
	     nil)
	    ((check-status state '(incomplete transition OK))
	     (error "Agenda shouldn't see state ~a with status ~a."
		    state (state-status state)))
	    ((member 'quiescent status) (when *quiescent-ok-for-agenda*
					  (list state)))
	    ((member 'complete status) (list state))
	    (t (error "State ~a has unknown status ~a."
		      state (state-status state)))))))
  


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   Taken from CONSTRAINTS.LISP


;;;-----------------------------------------------------------------------------
;;;  Function:  (P-successors  qval  qspace  var)
;;;             (I-successors  qval  qspace  var)
;;;
;;;  Purpose:   Given a qualitative value from the current state, create and
;;;             return all possible next values of the variable.  This code is
;;;             justified by Appendix A in [Kuipers, 1986].
;;;
;;;  Terminology change:  P- & I-transitions ====> P- & I-successors
;;;             [Kuipers, 1986] refers to this step as P- and I-transitions,
;;;             but the word "transitions" is also used to refer to region
;;;             transitions, which is a jump from one QDE to another.  Thus,
;;;             to avoid confusion, the word "transitions" will now refer only
;;;             to region transitions, and we'll use the more suggestive word
;;;             "successors" here.  After all, the successor values generated
;;;             here give rise to successor states.  -- DD
;;;
;;;  P-successors are possible next values when moving from a time-point
;;;  to a time-interval:  t0 --> (t0 t1).
;;;  11-21:  experimentally added a new qdir:  IGN.
;;;-----------------------------------------------------------------------------

;;;
;;;  The following transitions are supported for abstract qdirs.
;;;
;;;  ((0 A) (inc std dec))  -->  ((0 A) (inc std dec))
;;;                         -->  (A inc)
;;;                         -->  (A std)
;;;                         -->  (0 dec)
;;;                         -->  (0 std)
;;;  ((0 A) (inc std))      -->  ((0 A) (inc std))
;;;                         -->  ((0 A) dec)
;;;                         -->  (A inc)
;;;                         -->  (A std)
;;;  ((0 A) (std dec))      -->  ((0 A) (std dec))
;;;                         -->  ((0 A) inc)
;;;                         -->  (A dec)
;;;                         -->  (A std)


(defun P-successors (qval qspace var)
  (let ((qmag    (qmag qval))
	(qdir    (qdir qval)))
    (cond ((qmag-point-p qmag)
	   ;; qmag is a point value
	   (cond ((listp qdir)
		  (nconc (when (or (member 'inc qdir)
				   (and (member 'std qdir)
					(not (eql qmag (car (last qspace))))))
			   (list (make-qval :variable var
					    :qmag (list qmag (succ qmag qspace))
					    :qdir 'inc)))
			 (when (or (member 'dec qdir)
				   (and (member 'std qdir)
					(not (eql qmag (first qspace)))))
			   (list (make-qval :variable var
					    :qmag (list (pred qmag qspace) qmag)
					    :qdir 'dec)))
			 (when (member 'std qdir)
			   (make-qval :variable var
				      :qmag qmag
				      :qdir 'std))))
		 ;;(cerror "Continue" "Qmag is a point value and it is in a chatter box.")
		 ;;(list (make-qval :variable (qval-variable qval)
		 ;;:qmag (qval-qmag qval)
		 ;;:qdir '(inc std dec)))
		 ((eql qdir 'inc) (list (make-qval :variable var
						   :qmag (list qmag (succ qmag qspace))
						   :qdir 'inc)))
		 ((eql qdir 'dec) (list (make-qval :variable var
						   :qmag (list (pred qmag qspace) qmag)
						   :qdir 'dec)))
		 ((eql qdir 'std) (delete nil
					  (list (if (not (eql qmag (car (last qspace))))
						    (make-qval :variable var
							       :qmag (list qmag (succ qmag qspace))
							       :qdir 'inc))
						(if (not (eql qmag (first qspace)))
						    (make-qval :variable var
							       :qmag (list (pred qmag qspace) qmag)
							       :qdir 'dec))
						(make-qval :variable var
							   :qmag qmag
							   :qdir 'std))))
		 ((eql qdir 'ign) (delete nil
					  (list (if (not (eql qmag (car (last qspace))))
						    (make-qval :variable var
							       :qmag (list qmag (succ qmag qspace))
							       :qdir 'ign))
						(if (not (eql qmag (first qspace)))
						    (make-qval :variable var
							       :qmag (list (pred qmag qspace) qmag)
							       :qdir 'ign))
						(make-qval :variable var
							   :qmag qmag
							   :qdir 'ign))))))
	  ;; qmag is an interval
	  (t (cond ((eql qdir 'std) (list (make-qval :variable var :qmag qmag :qdir 'inc)
					  (make-qval :variable var :qmag qmag :qdir 'dec)
					  (make-qval :variable var :qmag qmag :qdir 'std)))
		   (t  (list qval)))))))


;;;-----------------------------------------------------------------------------
;;;  I-successors are possible next values when moving from a time-interval
;;;  to a time-point:  (t0 t1) --> t1.
;;;-----------------------------------------------------------------------------

(defun I-successors (qval qspace var)
  (declare (ignore qspace))
  (let ((qmag (qmag qval))
	(qdir (qdir qval)))
    (cond ((qmag-point-p qmag)
	   ;; qmag is a point value
	   (cond ((eql qdir 'std) (list qval))
		 ((eql qdir 'ign) (list (make-qval :variable var :qmag qmag :qdir 'ign)))
		 (t  (error "Can't be (~a, ~a) over an interval." qmag qdir))))
	  ;; qmag is an interval
	  (t (cond ((listp qdir)
		    (nconc (list (make-qval :variable (qval-variable qval)
					    :qmag qmag
					    :qdir qdir))
			   (if (member 'inc qdir)
			       (list (make-qval :variable (qval-variable qval)
						:qmag (cadr qmag)
						:qdir 'inc)
				     (make-qval :variable (qval-variable qval)
					      :qmag (cadr qmag)
					      :qdir 'std))
			       ;; inc is not a member and std is
			       (when (member 'std qdir)
				 (list (make-qval :variable (qval-variable qval)
						  :qmag qmag
						  :qdir 'inc))))
			   (if (member 'dec qdir)
			       (list (make-qval :variable (qval-variable qval)
						:qmag (car qmag)
						:qdir 'dec)
				     (make-qval :variable (qval-variable qval)
						:qmag (car qmag)
						:qdir 'std))
			       ;; dec is not a member and std is
			       (when (member 'std qdir)
				 (list (make-qval :variable (qval-variable qval)
						  :qmag qmag
						  :qdir 'dec))))
			   ))
		   ((eql qdir 'inc) (list (make-qval :variable var :qmag (cadr qmag) :qdir 'std)
					  (make-qval :variable var :qmag (cadr qmag) :qdir 'inc)
					  (make-qval :variable var :qmag qmag :qdir 'std)
					  qval))
		   ((eql qdir 'dec) (list (make-qval :variable var :qmag (car qmag) :qdir 'std)
					  (make-qval :variable var :qmag (car qmag) :qdir 'dec)
					  (make-qval :variable var :qmag qmag :qdir 'std)
					  qval))
		   ((eql qdir 'std) (list qval))
		   ((eql qdir 'ign) (list (make-qval :variable var :qmag (cadr qmag) :qdir 'ign)
					  (make-qval :variable var :qmag (car qmag)  :qdir 'ign)
					  qval)))))))



;;;
;;;   TEST-P-SUCCESSOR and TEST-I-SUCCESSOR
;;;
;;;   These two functions are designed to test whether two sets of qvalues
;;;   are consistent with respect to continuity.  P-successor test the successor
;;;   of a time point state while I-successors a time interval state.
;;;
;;;   These two functions have been modified to use a table to encode valid
;;;   sequences.  THe table is made up of five entries.  The first four
;;;   can have the following values:
;;;
;;;      Entry 1:     point | int       Whether QMAG1 is a point or interval qmag
;;;      Entry 2:     inc | std | dec   Value of QDIR1.
;;;      Entry 3:     point | int       Same as one for QMAG2.
;;;      Entry 4:     inc | std |dec    Value of QDIR2.
;;;
;;;   QVAL1 is the current qvalue while QVAL2 is the successor qvalue.
;;;
;;;   The fifth entry is a lisp expression that can use the following
;;;   arguments:  QMAG1, QMAG2, and QSPACE.
;;;
;;;   This table is inserted into an array with the fifth entry being the
;;;   array value and the first four entries being used to derive the
;;;   index into the array with the following translation:
;;;
;;;      int     0
;;;      point   1
;;;
;;;      inc     0
;;;      std     1
;;;      dec     2
;;;
;;;   The value of QVAL1 and QVAL2 are used to index into the array.  The
;;;   array entry must evaluate to TRUE, if the two qvalues are consistent
;;;   with respect to continuity.
;;;
;;;   This change has been made to simplify changing the valid conditions as
;;;   various abstraction techniques are developed.  Currently, this table
;;;   is designed to handle abstracted QMAGs.  For abstracted QDIRs as well
;;;   as the qdir IGN, multiple checks are made to make sure that at least
;;;   one set of the possible qdirs are valid.  For abstracted qmags, the
;;;   form that is evaluated has been modified to handle this possiblity.
;;;
;;;   DJC  03/29/93

(defparameter *p-successors-table*
  '((point	inc	int	inc             (qmag-includes
						   qmag2 (list qmag1 (succ qmag1 qspace))
						   qspace))
    (point	dec	int	dec		(qmag-includes
						 qmag2 (list (pred qmag1 qspace) qmag1)
						 qspace))
    (point	std	point   std		(eql qmag1 qmag2))
    (point	std	int	inc		(qmag-includes
						 qmag2 (list qmag1 (succ qmag1 qspace))
						 qspace))
    (point      std     int	dec		(qmag-includes
						 qmag2 (list (pred qmag1 qspace) qmag1)
						 qspace))
    
    (int	std	nil     nil		(equal qmag1 qmag2))
    (int	inc	nil	inc		(or (qmag-includes qmag1 qmag2 qspace)
						    (qmag-includes qmag2 qmag1 qspace)))
    (int	dec	nil	dec		(or (qmag-includes qmag1 qmag2 qspace)
						    (qmag-includes qmag2 qmag1 qspace)))
    ))


(defparameter *i-successors-table*
  '((point	std	nil	std		(equal qmag1 qmag2))
    
    (int	inc	int	inc		(equal qmag2 qmag1))
    (int	inc	point	inc		(or (eql qmag2 (cadr qmag1))
						    (qmag-includes qmag1 qmag2 qspace)))
    (int	inc	int	std		(equal qmag2 qmag1))
    (int	inc	point  	std		(or (eql qmag2 (cadr qmag1))
						    (eql qmag2 (succ (car qmag1) qspace))
						    (qmag-includes qmag1 qmag2 qspace)))
    (int	dec	int	dec		(equal qmag2 qmag1))
    (int	dec	point	dec		(or (eql qmag2 (car qmag1))
						    (qmag-includes qmag1 qmag2 qspace)))
    (int	dec	int	std		(equal qmag2 qmag1))
    (int	dec	point  	std		(or (eql qmag2 (car qmag1))
						    (eql qmag2 (pred (cadr qmag1) qspace))
						    (qmag-includes qmag1 qmag2 qspace)))
    ))

(defun qmag-index (qmag)
  " Provides the following mapping from a qmag.  It is also valid to pass the symbol
point or int (for interval).

        qmag-interval    0
        qmag-point       1"
  (cond ((null qmag)
	 '(0 1))
	((or (eql qmag 'int)
	     (qmag-interval-p qmag))
	 '(0))
	(t       '(1))))

(defun qdir-index (qdir)
  "Provides a mapping for QDIRs."
  (let ((qd (if (or (equal qdir 'ign)
		    (null qdir))
		'(inc std dec)
		(listify qdir))))
    (mapcar #'(lambda (qdir)
		(case qdir
		  (inc 0)
		  (std 1)
		  (dec 2)))
	    qd)))

(defun combinations (lists)
  "Receives a list of four lists.  It will return a list of lists each with four elements.
The returned lists will be all of the combinations of the entries in the initial
four lists.  It actually will work for lists of any size."
  (when lists
    (let ((cdr-combs (combinations (cdr lists))))
      (mapcan #'(lambda (entry)
		  (if cdr-combs
		      (mapcar #'(lambda (cdr-comb)
				  (cons entry cdr-comb))
			      cdr-combs)
		      (list (list entry))))
	      (car lists)))))


;;;
;;;  MAKE-ARRAY-ENTRY
;;;
;;;  Will take an entry from the tables above, use the first four elements of the
;;;  entry to find the indeces for the array entry and then it will insert the
;;;  fifth element of the entry into the array.

(defun make-array-entry (entry array)
  (let ((indeces (combinations (list (qmag-index (car entry))
				     (qdir-index (second entry))
				     (qmag-index (third entry))
				     (qdir-index (fourth entry))))))
    (mapcar #'(lambda (index)
		(setf (apply #'aref array index)
		      (fifth entry))
		(apply #'aref array index))
	    indeces)))

(defun print-array (array)
  "Prints all non-nill values in the array."
  (let (ret)
    (loop for i1 from 0 to 1
	  do (loop for i2 from 0 to 2
		   do (loop for i3 from 0 to 1
			    do (loop for i4 from 0 to 2
				     when (aref array i1 i2 i3 i4)
				     do (format t "~%Index: (~a ~a ~a ~a)  Entry: ~a"
					           i1 i2 i3 i4 (aref array i1 i2 i3 i4))
				     collect (aref array i1 i2 i3 i4) into ret-val
				     finally (setf ret ret-val)))))
    ret))




(defparameter *p-successors-array*
  (let ((p-array (make-array '(2 3 2 3))))
    (mapcar #'(lambda (entry)
		(make-array-entry entry p-array))
	    *p-successors-table*)
    p-array))

(defparameter *i-successors-array*
  (let ((p-array (make-array '(2 3 2 3))))
    (mapcar #'(lambda (entry)
		(make-array-entry entry p-array))
	    *i-successors-table*)
    p-array))



;;;
;;;   TEST P and I SUCCESSOR functions

(defun test-p-successor (qvalue1 qvalue2 qspace)
  (test-successor (cdr qvalue1) (cdr qvalue2) qspace *p-successors-array*))

(defun test-i-successor (qvalue1 qvalue2 qspace)
  (test-successor (cdr qvalue1) (cdr qvalue2) qspace *i-successors-array*))


(defun test-successor (qval1 qval2 qspace succ-array)
  (let* ((entries   (combinations (list (qmag-index (qmag qval1))
					(qdir-index (qdir qval1))
					(qmag-index (qmag qval2))
					(qdir-index (qdir qval2))))))
    (loop for entry in entries
	  when (check-succ-array entry succ-array (qmag qval1) (qmag qval2) qspace)
	  do (return t))))


;;;
;;;  Evaluates the entry fpor INDEX in the ARRAY using QMAG1, QMAG2, and QSPACE.
;;;

(defun check-succ-array (index array qmag1 qmag2 qspace)
  (let ((entry (apply #'aref array index)))
    (when entry
      (progv '(qmag1 qmag2 qspace) (list qmag1 qmag2 qspace)
	(eval entry)))))




;;;  REPLACED BY THE CODE ABOVE
;;;

;;;-----------------------------------------------------------------------------
;;;  Test for P- and I-successors:  Is a pair of qvalues consistent with continuity?
;;;  P-successors go from a time-point to a time-interval:  t0 -> (t0,t1).
;;;-----------------------------------------------------------------------------

;;(defun test-P-successor (qvalue1 qvalue2 qspace)
;;  (let* ((qval1 (cdr qvalue1))
;;	 (qval2 (cdr qvalue2))
;;	 (qmag1 (qmag qval1))
;;	 (qmag2 (qmag qval2))
;;	 (qdirs1 (listify (qdir qval1)))
;;	 (qdirs2 (listify (qdir qval2)))
;;	 (qdir-combos (cross-product (list qdirs1 qdirs2))))
;;    (some #'(lambda (qdirs)            ; DJC
;;	      (check-p-successor qmag1 qmag2 (car qdirs) (cadr qdirs) qspace))
;;	  qdir-combos)))
;;
;;; DJC
;;(defun check-p-successor (qmag1 qmag2 qdir1 qdir2 qspace)
;;  (cond ((qmag-point-p qmag1)
;;	 (cond ((eql qdir1 'inc) (and (member qdir2 '(inc ign))
;;				      (qmag-interval-p qmag2)
;;				      (eql qmag1 (car qmag2))))
;;	       ((eql qdir1 'dec) (and (member qdir2 '(dec ign))
;;				      (qmag-interval-p qmag2)
;;				      (eql qmag1 (cadr qmag2))))
;;	       ((eql qdir1 'std) (cond ((eql qmag1 qmag2)
;;					(member qdir2 '(std ign)))
;;				       ((qmag-point-p qmag2) nil)
;;				       ((eql qmag1 (car qmag2))
;;					(member qdir2 '(inc ign)))
;;					 ((eql qmag1 (cadr qmag2))
;;					  (member qdir2 '(dec ign)))))
;;	       ((eql qdir1 'ign) (cond ((eql qmag1 qmag2)
;;					(member qdir2 '(std ign)))
;;				       ((qmag-point-p qmag2) nil)
;;				       ((eql qmag1 (car qmag2))
;;					(member qdir2 '(inc ign)))
;;				       ((eql qmag1 (cadr qmag2))
;;					(member qdir2 '(dec ign)))))))
;;	(t (cond ((member qdir1 '(std ign)) (qmags-intersect qmag1 qmag2 qspace))
;;		 ((eql qdir1 'inc) (and (equal qmag1 qmag2)
;;					(member qdir2 '(inc ign))))
;;		 ((eql qdir1 'dec) (and (equal qmag1 qmag2)
;;					(member qdir2 '(dec ign))))))))
;;
;;
;;;;;-----------------------------------------------------------------------------
;;;;;  I-successors go from a time-interval to a time-point:  (t0,t1) -> t1.
;;;;;-----------------------------------------------------------------------------
;;
;;(defun test-I-successor (qvalue1 qvalue2 qspace)
;;  (let* ((qval1 (cdr qvalue1))
;;	 (qval2 (cdr qvalue2))
;;	 (qmag1 (qmag qval1))
;;	 (qmag2 (qmag qval2))
;;	 (qdirs1 (listify (qdir qval1)))
;;	 (qdirs2 (listify (qdir qval2)))
;;	 (qdir-combos (cross-product (list qdirs1 qdirs2))))
;;    (some #'(lambda (qdirs)            ; DJC
;;	      (check-i-successor qmag1 qmag2 (car qdirs) (cadr qdirs) qspace))
;;	  qdir-combos)))
;;
;;(defun check-i-successor (qmag1 qmag2 qdir1 qdir2 qspace)
;;  (cond ((qmag-point-p qmag1)
;;	 (cond ((eql qdir1 'std) (and (equal qmag1 qmag2)
;;				      (member qdir2 '(std ign))))
;;	       ((eql qdir1 'ign) (and (equal qmag1 qmag2)
;;				      (member qdir2 '(std ign))))))
;;	(t (cond ((eql qdir1 'inc) (cond ((eql qmag2 (cadr qmag1))
;;					  (member qdir2 '(inc std ign)))
;;					 ((equal qmag2 qmag1)
;;					  (member qdir2 '(inc std ign)))
;;					 ((eql qmag2 (succ (car qmag1) qspace))
;;					  (member qdir2 '(std ign)))))
;;		 ((eql qdir1 'dec) (cond ((eql qmag2 (car qmag1))
;;					  (member qdir2 '(dec std ign)))
;;					 ((equal qmag2 qmag1)
;;					  (member qdir2 '(dec std ign)))
;;					 ((eql qmag2 (pred (cadr qmag1) qspace))
;;					  (member qdir2 '(std ign)))))
;;		 ((eql qdir1 'std) (and (equal qmag1 qmag2)
;;					(member qdir2 '(std ign))))
;;		 ((eql qdir1 'ign) (cond ((equal qmag2 qmag1) t)
;;					 ((qmag-interval-p qmag2) nil)
;;					 ((eql qmag2 (car qmag1))
;;					  (member qdir2 '(dec std ign)))
;;					 ((eql qmag2 (cadr qmag1))
;;					  (member qdir2 '(inc std ign)))
;;					 ((eql qmag2 (pred (cadr qmag1) qspace))
;;					  (member qdir2 '(std ign)))
;;					 ((eql qmag2 (succ (car qmag1) qspace))
;;					  (member qdir2 '(std ign)))))))))
;;
;;




;;; Create a symbol name string.
;;; Inputs:  prefix  - A symbol.
;;;          n       - An integer.
;;;          dash-p  - T if a dash should be in the name and NIL otherwise.
;;; Returns: A string (e.g., if prefix=S, n=3, and dash-p=T then return "S-3").
;;; Notes:   This function is used in place of (format nil "~a-~a" prefix n)
;;;          because format is INCREDIBLY SLOW.
;;; This version by BKay 3Feb93.
;;;
(defun gen-symbol-name (prefix n dash-p)
  (let* ((pstring (string prefix))
	 (plength (length pstring))
	 (dlength (if dash-p 1 0))
	 (nlength (if (zerop n) 1 (1+ (floor (log n 10)))))
	 (totlength  (+ plength dlength nlength))
	 s)
    (setq s (make-string totlength))
    (dotimes (i plength)
      (setf (schar s i) (schar pstring i)))
    (if dash-p
	(setf (schar s plength) #\-))
    (loop for pos from (1- totlength) downto (+ plength dlength)
	  for div = n
	  then (floor (/ div 10))
	  for dig = (rem div 10)
	  do
	  (setf (schar s pos) (digit-char dig)))
    s))


