;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-

;;;*****************************************************************************
;;;				  M O D E L S
;;;				      O F
;;;	     V A R I O U S   T Y P E S   O F   C O N T R O L L E R
;;;
;;;
;;; Pierre Fouche <fouche@cs.utexas.edu>
;;; August 1990
;;;
;;; In this file, you'll find models of systems described in the paper:
;;; 	"Qualitative Simulation of Feedback Control Systems"
;;; 	Pierre Fouche & Benjamin Kuipers
;;;	Proceedings of the IMACS International Symposium on Mathematical and
;;;	Intelligent Models in System Simulation
;;;
;;;*****************************************************************************


(defun describe-controllers ()
  (format *qsim-trace* "~%
In this catalog, you will find models of systems described in the paper:


	  \"Qualitative Simulation of Feedback Control Systems\"

		    Pierre Fouche & Benjamin Kuipers

	   Proceedings of the IMACS International Symposium on 
	Mathematical and Intelligent Models in System Simulation



These models are:

	+ A simple tank, without and with Q2
	+ An on-off controller, without and with Q2
	+ A proportional controller, with a comparative analysis of the gain,
	  without and with Q2
	+ A non-linear, proportional-integral controller, described in detail
	  in \"Reasoning about energy in qualitative simulation\",
	      Fouche & Kuipers
	      UT-TR-90-134
"
	  ))

;;;=============================================================================
;;; Parameters, functions and macros for envelope definitions
;;;=============================================================================

(defparameter ke1 1)
(defparameter ke2 ke1)
(defparameter ki1 1)
(defparameter ki2 ki1)

(defun f1     (x) (+ (* ke1 x) 1))
(defun f1-inv (y) (/ (- y 1) ke1))
(defun f2     (x) (+ (* ke2 x) 1))
(defun f2-inv (y) (/ (- y 1) ke2))

(defun g1     (x) (* ki1 x))
(defun g1-inv (y) (/ y ki1))
(defun g2     (x) (* ki2 x))
(defun g2-inv (y) (/ y ki2))

(defun square (y) (expt y 2))


;;; These functions used the old storage semantics for initial-ranges and 
;;; m-envelopes.  This has now been corrected.  BKay 13Mar91
;
;(defmacro set-qde-other-slot (qde slot clause)
;  `(setf (alookup ,slot (qde-other ,qde)) ,clause))

(defmacro set-envelopes (qde clause)
  `(setf (qde-m-envelopes ,qde) ,clause))
;  `(set-qde-other-slot ,qde 'm-envelopes ,clause))

(defmacro set-ranges (qde clause)
  `(setf (qde-initial-ranges ,qde) ,clause))
;  `(set-qde-other-slot ,qde 'initial-ranges ,clause))


;;;=============================================================================
;;; A simple tank
;;;=============================================================================

(define-qde tank
	    (text "A simple tank")
  
  (quantity-spaces
    (F1   (0 f* inf))				;inflow
    (F2   (0 f* inf))				;outflow
    (F    (minf 0 inf))				;netflow
    (L    (0 max))				;level in the tank
    (rL   (0 rmax))				;square root of L
    (V    (0 v* open))				;opening of the valve
    (V*rL (0 inf))				
    (K    (0 k1 inf))				;coefficient
    )
  
  (constraints
    ((add F F2 F1) (0 f* f*))			;mass balance
    ((d/dt L F))				
    ((M+ L rL) (0 0) (max rmax))
    ((mult V rL V*rL))
    ((mult K v*rL F2))
    
    ((constant K))
    ((constant V))
    ((constant F1)))
  
  (m-envelopes)
  (initial-ranges)
  
  (print-names (F1 "Inflow")
	       (F2 "Outflow")
	       (F "Netflow")
	       (L "Level")
	       (rL "Square root of level")
	       (V "Valve"))
  (layout
    (K 		V	)
    (L 		rL  	V*rL)
    (F   	F1  	F2  )))


;;;-----------------------------------------------------------------------------
;;; Init Function
;;;-----------------------------------------------------------------------------

(defun fill-tank ()
  (qsim-cleanup)
  (let* ((*check-quantitative-ranges* nil)
	 (std (make-new-state
		:from-qde tank
		:assert-values '((L (0 nil))
				 (f1 (f* std))
				(V (v* std))
				(k (k1 std))
			        ))))
    (qsim-display  (qsim std))))


(defun q2-fill ()
  (qsim-cleanup)
  (set-envelopes
    tank
    '(((M+ PC V)
       (upper-envelope f1)
       (upper-inverse  f1-inv)
       (lower-envelope f2)
       (lower-inverse  f2-inv))
      ((M+ L rL)
       (upper-envelope sqrt)
       (upper-inverse  square)
       (lower-envelope sqrt)
       (lower-inverse  square))))
  (set-ranges 
    tank
    '(((F1 f*) (.9 1.1))
      ((time t0) (0 0))
      ((k k1) (1.5 1.6))
      ((V open) (1 1))
      ((V v*) (.8 .8))
      ((L max) (2 2))))
  (let* ((*check-quantitative-ranges* t)
	 (std (make-new-state
		:from-qde tank
		:assert-values '((L (0 nil))
				 (f1 (f* std))
				 (V (v* std))
				 (k (k1 std)))
	       )))
    (qsim-display  (qsim std)
		   :layout '((K 	V	time)
			     (L 	rL  	V*rL)
			     (F   	F1  	F2  )))))



;;;=============================================================================
;;; A level-controlled tank
;;; with an On-Off Controller 
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; Transition functions
;;;-----------------------------------------------------------------------------

(defun hi-error (state)
  (let* ((e-qval (qval 'e state))
	 (e-qmag (qmag e-qval))
	 (e-qdir (qdir e-qval))
	 (v-qmag (qmag (qval 'v state))))
    (if	(and (lmark-p e-qmag)
	     (eq (lmark-name e-qmag) 'hi)
	     (eq e-qdir 'inc)
	     (eq v-qmag *zero-lmark*))
	(values t nil)
	(values nil t))))

(defun lo-error (state)
  (let* ((e-qval (qval 'e state))
	 (e-qmag (qmag e-qval))
	 (e-qdir (qdir e-qval))
	 (v-qmag (qmag (qval 'v state))))
    (if	(and (lmark-p e-qmag)
	     (eq (lmark-name e-qmag) 'lo)
	     (eq e-qdir 'dec)
	     (not (eq v-qmag *zero-lmark*)))
	(values t nil)
	(values nil t))))

(defun open-valve (state)
  (create-transition-state
		 :from-state state
		 :to-qde on-off-controller
		 :assert '((V (open std))
			   )
		 :inherit-qmag '(f1 l ls k a e)))

(defun close-valve (state)
  (create-transition-state
		 :from-state state
		 :to-qde on-off-controller
		 :assert '((V (0 std))
			   )
		 :inherit-qmag '(f1 l ls k a e)))


;;;-----------------------------------------------------------------------------
;;; The QDE...
;;;-----------------------------------------------------------------------------

(define-qde On-off-controller
  (text "A On-Off Controlled Tank")

  (quantity-spaces
  
    ;; Main variables
    (F1   (0 f1* inf))				;inflow
    (F2   (0 inf))				;outflow
    (F    (minf 0 inf))				;netflow
    (L    (0 lo l* hi max))			;level in the tank
    (rL   (0 rmax))				;square root of L
    (V    (0 open))				;opening of the valve
    (V*rL (0 inf))				
    (K    (0 k1 inf))				;coefficient
    
    ;; Control variables
    (LS   (0 l* inf))				;level set point
    (E    (minf lo 0 hi inf))			;error: L - LS
    (a    (0 on))				;open or closed loop
    )
  
  (constraints
    
    ;; System equations
    ((add F F2 F1))				;mass balance
    ((d/dt L F))				
    ((M+ L rL) (0 0) (max rmax))
    ((mult V rL V*rL))
    ((mult K v*rL F2))
    
    ;; Control equations
    ((add E LS L) (0 l* l*) (hi l* hi) (lo l* lo))	;error definition
    
    ((constant V))
    ((constant a))
    ((constant K))
    ((constant F1))
    ((constant LS)))

  (transitions (hi-error open-valve)
	       (lo-error close-valve))

  (m-envelopes)

  (initial-ranges)
 
  (print-names (E "Error")
	       (F1 "Inflow")
	       (F2 "Outflow")
	       (F "Netflow")
	       (L "Level")
	       (rL "Square root of level")
	       (LS "Set point")
	       (V "Opening of the valve")
	       (A "Controller on/off"))
  (layout
    	(nil	LS	V   )
	(E   	L 	rL  )
	(F   	F1  	F2  )))


;;;-----------------------------------------------------------------------------
;;; Initialization functions
;;;-----------------------------------------------------------------------------

(defun on-off-control ()
  (qsim-cleanup)
  (let* ((*check-quantitative-ranges* nil)
	 (start (make-new-state
		  :from-qde on-off-controller
		  :assert-values '((L (0 nil))
				   (V (0 std))
				   (f1 (f1* std))
				   (ls (l* std))
				   (k (k1 std))
				   (a (on std))
				   (e ((minf lo) nil))
				   ))))
    (qsim-display  (qsim start))))

(defun q2-on-off-control ()
  (qsim-cleanup)
  (set-envelopes
    on-off-controller
    '(((M+ L rL)
       (upper-envelope sqrt)
       (upper-inverse  square)
       (lower-envelope sqrt)
       (lower-inverse  square))))
  (set-ranges 
    on-off-controller
    '(((F1 f1*) (.9 1.1))
      ((LS l*) (1 1))
      ((L lo) (.9 .9))
      ((L max) (2 2))
      ((L hi) (1.1 1.1))
      ((time t0) (0 0))
      ((V open) (1 1))
      ((k k1) (2 2))
      ((A on) (1 1))))
  (let* ((*check-quantitative-ranges* t)
	 (start (make-new-state
		  :from-qde on-off-controller
		  :assert-values '((L (0 nil))
				   (V (0 std))
				   (f1 (f1* std))
				   (ls (l* std))
				   (k (k1 std))
				   (a (on std))
				   (e ((minf lo) nil))
				   ))))
    (qsim-display  (qsim start))))



;;;=============================================================================
;;; A level-controlled tank with a proportional controller 
;;; Comparative Analysis of the controller gain.
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; Transition function
;;;-----------------------------------------------------------------------------

(defparameter *control-type* nil)

(defun step-increase (std-state)
  (case *control-type* 
    (a=0 (setq *control-type* 'a=a1) 
	 (create-transition-state
	   :from-state std-state
	   :to-qde pc
	   :assert '((f1 (f+ std))
		     )
	   :inherit-qmag '(l ls k a V b)))
    (a=a1 (setq *control-type* 'a=a2)
	  (create-transition-state
	    :from-state std-state
	    :to-qde pc
	    :assert '((a (a1 std))
		      (f1 (f+ std))
		      (e (0 nil))
		      )
	    :inherit-qmag '(b ls k)))
    (a=a2 (setq *control-type* 'stop)
	  (create-transition-state
	    :from-state std-state
	    :to-qde pc
	    :assert '((a (a2 std))
		      (f1 (f+ std))
		      (e (0 nil))
		      )
	    :inherit-qmag '(b k ls)))
    (stop nil)))


;;;-----------------------------------------------------------------------------
;;; The QDE...
;;;-----------------------------------------------------------------------------

(define-qde PC
  (text "A P-controlled tank")

  (quantity-spaces
    
    ;; Main variables
    (F1   (0 f* f+ inf))			;inflow
    (F2   (0 f* f+ inf))			;outflow
    (F    (minf 0 inf))				;netflow
    (L    (0 inf))				;level in the tank
    (rL   (0 inf))				;square root of L
    (V    (0 inf))				;opening of the valve
    (V*rL (0 inf))				
    (K    (0 k1 inf))				;coefficient
    
    ;; Control variables
    (LS   (0 l* inf))				;level set point
    (E    (minf 0 inf))				;error: L-LS
    (a*E  (minf 0 inf))				;proportional control term
    (a    (0 a1 a2))				
    (b	  (minf 0 inf))				;linear control
    )
  
  (constraints
    
    ;; System equations
    ((add F F2 F1) (0 f* f*) (0 f+ f+))		;mass balance
    ((d/dt L F))				
    ((M+ L rL) (0 0) (inf inf))
    ((mult V rL V*rL))
    ((mult K v*rL F2))
    
    ;; Control equations
    ((add E LS L) )				;error definition
    ((mult a e a*e))
    ((add a*e b V))

    ((constant a))
    ((constant b))
    ((constant K))
    ((constant F1))
    ((constant LS)))

  (transitions ((f (0 std)) step-increase))

  (m-envelopes)
  (initial-ranges)
 
  (print-names (E "Error")
	       (F1 "Inflow")
	       (F2 "Outflow")
	       (F "Netflow")
	       (L "Level")
	       (rL "Square root of level")
	       (LS "Set point")
	       (a  "Gain")

	       (V "Opening of the valve"))
  (layout
    (a		V)
    (E   	L 	rL)
    (F   	F1  	F2)))


;;;-----------------------------------------------------------------------------
;;; Init Function
;;;-----------------------------------------------------------------------------

(defun pc ()
  (qsim-cleanup)
  (setf	*control-type* 'a=0)
  (let* ((*check-quantitative-ranges* nil)
	 (std (make-new-state
		:from-qde pc
		:assert-values '((E (0 std))
				 (f1 (f* std))
				 (ls (l* std))
				 (k (k1 std))
				 (a (0 std))
				 ))))
    (qsim-display  (qsim std))))


(defun q2-pc ()
  (qsim-cleanup)
  (set-envelopes
    pc
    '(((M+ PC V)
       (upper-envelope f1)
       (upper-inverse  f1-inv)
       (lower-envelope f2)
       (lower-inverse  f2-inv))
      ((M+ L rL)
       (upper-envelope sqrt)
       (upper-inverse  square)
       (lower-envelope sqrt)
       (lower-inverse  square))))
  (set-ranges 
    pc
    '(((F1 f*) (1 1))
      ((F1 f+) (1.2 1.2))
      ((LS l*) (1 1))
      ((time t0) (0 0))
      ((k k1) (2 2))
      ((A a1) (.9 1.1))
      ((A a2) (1.9 2.1))))
  (setf	*control-type* 'a=0)
  (let* ((*check-quantitative-ranges* t)
	 (std (make-new-state
		:from-qde pc
		:assert-values '((E (0 std))
				 (f1 (f* std))
				 (ls (l* std))
				 (k (k1 std))
				 (a (0 std))
				 ))))
    (qsim-display  (qsim std)
		   :layout 
		   '((a		V 	time)
		     (E   	L 	rL)
		     (F   	F1  	F2)))))


;;;=============================================================================
;;; A level-controlled tank 
;;;=============================================================================


(define-qde standard-PIC
  (text "A level-controlled tank")

  (quantity-spaces
    
    ;; Main variables
    (F1   (0 f1* f1+ inf))			;inflow
    (F2   (0 f2+ inf))				;outflow
    (F    (minf 0 inf))				;netflow
    (L    (0 l* inf))				;level in the tank
    (rL   (0 rL* inf))				;square root of L
;    (V    (0 v+ inf))				;opening of the valve
						;v+: position at new equilibrium	 
    (V    (0 inf))				;opening of the valve
    (V*rL (0 VrL+ inf))
    (K    (0 k1 inf))				;coefficient
    
    ;; Control variables
    (LS   (0 l* inf))				;level set point
    (E    (minf 0 inf))				;error: L-LS
    
    (PC   (minf 0 inf))				;proportional control term
    (IC   (minf 0 inf))				;integral control term
    
    ;(C    (minf inf))				;conservative term
    )
  
  (constraints
    
    ;; System equations
    ((add F F2 F1) (0 f2+ f1+))			;mass balance
    ((d/dt L F))				
    ((M+ rl L) (0 0) (rl* l*) (inf inf))
    ((mult V rL V*rL)
     ;(v+ rL* VrL+)		;valve equation
     )
    ((mult K v*rL F2) (k1 VrL+ f2+))
    
    ;; Control equations
    ((add E LS L) (0 l* l*))			;error definition
    ((d/dt IC E))
    ((add PC IC V) )				;PI control
    ((M+ PC E) (minf minf) (0 0) (inf inf))
   
    ;((M- C IC) (minf inf) (inf minf))

    ((constant K))
    ((constant F1))
    ((constant LS)))

  (energy-constraint (IC E  (- F1 (* K IC)) (- E)))
   
  ;(ignore-qdirs F F2 v*rL V)
  (unreachable-values (v 0))
  (phase-planes (f2 ic)(e ic)(v e))
  
  
  (print-names (E "Error")
	       (F1 "Inflow")
	       (F2 "Outflow")
	       (F "Netflow")
	       (L "Level")
	       (rL "Square root of level")
	       (LS "Set point")
	       (PC "Proportional action")
	       (IC "Integral action")
	       (V "Opening of the valve")
	       (C "Conservative term"))
  (layout
    (IC  PC  V)
    (E   nil L)
    (F   F1  F2)))



(define-qde standard-PIC-with-ign
  (text "A level-controlled tank")

  (quantity-spaces
    
    ;; Main variables
    (F1   (0 f1* f1+ inf))			;inflow
    (F2   (0 f2+ inf))				;outflow
    (F    (minf 0 inf))				;netflow
    (L    (0 l* inf))				;level in the tank
    (rL   (0 rL* inf))				;square root of L
;    (V    (0 v+ inf))				;opening of the valve
						;v+: position at new equilibrium	 
    (V    (0 inf))				;opening of the valve
    (V*rL (0 VrL+ inf))
    (K    (0 k1 inf))				;coefficient
    
    ;; Control variables
    (LS   (0 l* inf))				;level set point
    (E    (minf 0 inf))				;error: L-LS
    
    (PC   (minf 0 inf))				;proportional control term
    (IC   (minf 0 inf))				;integral control term
    
    ;(C    (minf inf))				;conservative term
    )
  
  (constraints
    
    ;; System equations
    ((add F F2 F1) (0 f2+ f1+))			;mass balance
    ((d/dt L F))				
    ((M+ rl L) (0 0) (rl* l*) (inf inf))
    ((mult V rL V*rL)
     ;(v+ rL* VrL+)		;valve equation
     )
    ((mult K v*rL F2) (k1 VrL+ f2+))
    
    ;; Control equations
    ((add E LS L) (0 l* l*))			;error definition
    ((d/dt IC E))
    ((add PC IC V) )				;PI control
    ((M+ PC E) (minf minf) (0 0) (inf inf))
   
    ;((M- C IC) (minf inf) (inf minf))

    ((constant K))
    ((constant F1))
    ((constant LS)))

  (energy-constraint (IC E  (- F1 (* K IC)) (- E)))
   
  (ignore-qdirs F F2 v*rL V)
  (unreachable-values (v 0))
  (phase-planes (f2 ic)(e ic)(v e))
  
  
  (print-names (E "Error")
	       (F1 "Inflow")
	       (F2 "Outflow")
	       (F "Netflow")
	       (L "Level")
	       (rL "Square root of level")
	       (LS "Set point")
	       (PC "Proportional action")
	       (IC "Integral action")
	       (V "Opening of the valve")
	       (C "Conservative term"))
  (layout
    (IC  PC  V)
    (E   nil L)
    (F   F1  F2)))



;;;-----------------------------------------------------------------------------
;;; Same QDE as before except for ignore-qdirs and curvature-at-steady clauses
;;;-----------------------------------------------------------------------------

(define-qde standard-PIC-with-acc
  (text "A level-controlled tank")

  (quantity-spaces
    
    ;; Main variables
    (F1   (0 f1* f1+ inf))			;inflow
    (F2   (0 f2+ inf))				;outflow
    (F    (minf 0 inf))				;netflow
    (L    (0 l* inf))				;level in the tank
    (rL   (0 rL* inf))				;square root of L
;    (V    (0 v+ inf))				;opening of the valve
						;v+: position at new equilibrium	 
    (V    (0 inf))				;opening of the valve
    (V*rL (0 VrL+ inf))
    (K    (0 k1 inf))				;coefficient
    
    ;; Control variables
    (LS   (0 l* inf))				;level set point
    (E    (minf 0 inf))				;error: L-LS
    
    (PC   (minf 0 inf))				;proportional control term
    (IC   (minf 0 inf))				;integral control term
    
    ;(C    (minf inf))				;conservative term
    )
  
  (constraints
    
    ;; System equations
    ((add F F2 F1) (0 f2+ f1+))			;mass balance
    ((d/dt L F))				
    ((M+ rl L) (0 0) (rl* l*) (inf inf))
    ((mult V rL V*rL)
     ;(v+ rL* VrL+)		;valve equation
     )
    ((mult K v*rL F2) (k1 VrL+ f2+))
    
    ;; Control equations
    ((add E LS L) (0 l* l*))			;error definition
    ((d/dt IC E))
    ((add PC IC V) )				;PI control
    ((M+ PC E) (minf minf) (0 0) (inf inf))
   
    ;((M- C IC) (minf inf) (inf minf))

    ((constant K))
    ((constant F1))
    ((constant LS)))

  (energy-constraint (IC E  (- F1 (* K IC)) (- E)))
   
  (curvature-at-steady
    ;; try the following ones when not using ignore-qdirs
     (v f)
     ;;(v (- f v))
     ;;(v (+ f (sd1 f)))
     ;;(f2 (* f (sd1 V)))
     (f2 f)
     )
  (unreachable-values (v 0))
  (phase-planes (f2 ic)(e ic)(v e))
  
  
  (print-names (E "Error")
	       (F1 "Inflow")
	       (F2 "Outflow")
	       (F "Netflow")
	       (L "Level")
	       (rL "Square root of level")
	       (LS "Set point")
	       (PC "Proportional action")
	       (IC "Integral action")
	       (V "Opening of the valve")
	       (C "Conservative term"))
  (layout
    (IC  PC  V)
    (E   nil L)
    (F   F1  F2)))


(defun pic (qde)
  (declare (special i1 step))
  (qsim-cleanup)
  (let* ((*perform-acc-analysis* (eq qde standard-pic-with-acc))
	 (*new-landmarks-at-initial-state* nil)
	 (*new-landmarks-on-transition* nil)
	 std)
    
    (setf (variable-no-new-landmarks-p (alookup 'f2 (qde-var-alist qde))) t
	  (variable-no-new-landmarks-p (alookup 'v*rl (qde-var-alist qde))) t
	  (variable-no-new-landmarks-p (alookup 'v (qde-var-alist qde))) t
	  
	  std (make-new-state
		:from-qde qde
		:assert-values '((E (0 std))
				 (f1 (f1* std))
				 (ls (l* std))
				 (k (k1 std))
				 ))
	  
	  (variable-no-new-landmarks-p (alookup 'f2 (qde-var-alist qde))) nil
	  (variable-no-new-landmarks-p (alookup 'v*rl (qde-var-alist qde))) nil
	  (variable-no-new-landmarks-p (alookup 'v (qde-var-alist qde))) nil
	  
	  step (create-transition-state
		 :from-state std
		 :to-qde qde
		 :assert '((f1 (f1+ std))
			   (e (nil inc))
			   )
		 :inherit-qmag '(l ic ls k)))
    
    (qsim-display  (qsim step)
		   :reference-states `((std ,std)))))

(defun pic-norm ()
  (pic standard-pic))


(defun pic-ign-qdir ()
  (pic standard-pic-with-ign))

(defun pic-acc ()
  (pic standard-pic-with-acc))






