;;; -*- Mode: LISP  -*-

;;Copyright (c) 1986 by John C. Hogge, The University of Illinois.
;; Modified for Lucid under IBM RT by H. Kim & K. Forbus
;;
;;File GRAPH.LISP of system Zgraph.

(in-package 'user)

;; This isn't supplied in LUCID environment, but is in their manual.
(defflavor property-list-mixin 
  ((property-list nil))
  ())

(defmethod (property-list-mixin :putprop) (value property) 
  (setf (getf property-list property) value))

(defmethod (property-list-mixin :get) (property &optional default)
  (getf property-list property default))

(defmethod (property-list-mixin :remprop) (property)
  (remf property-list property))

(DEFFLAVOR graph (name
		  type
		  (root-vertices NIL)
		  (vertices NIL)
		  (edges NIL)
		  (location-clumps NIL)
		  ;;how much we've zoomed
		  (x-scale-factor 1.0)
		  (y-scale-factor 1.0)
		  ;;how much we've panned 
		  (x-displacement 0.0)
		  (y-displacement 0.0)
		  ;; RT Change
		  (pane "display only"))
	   (PROPERTY-LIST-MIXIN) ;Used for storing misc. data for graph descriptions.

  :SETTABLE-INSTANCE-VARIABLES
  :GETTABLE-INSTANCE-VARIABLES
  :INITABLE-INSTANCE-VARIABLES
  (:REQUIRED-INIT-KEYWORDS :type)
;  (:DOCUMENTATION :COMBINATION
;   "Stores graphs.  Instance variables:
;TYPE: holds an instance of flavor GRAPH-TYPE, which is defined as our type of graph.
;This is a required init option.
;
;ROOT-VERTICES: list of root vertices, possibly generated by our graph type's
;default-root-finding form.  But if this default is NIL, we are assumed to get the roots on
;our own.
;
;VERTICES: list of VERTEX structs which comprise the graph.
;
;EDGES: list of EDGE structs stored in VERTICES.  We store this for efficient access to the
;edges.
;
;LOCATION-CLUMPS: groups of vertices and edges located close to each other.  Specifically,
;this is a list of the form:
;  ((XMIN . YMIN) (XMAX . YMAX) VERTICES-IN-THIS-CLUMP EDGES-IN-THIS-CLUMP)
;where the first two elements specify a world coordinate extent in which all vertex and edge
;structs in the last two elements lie.  This data structure is used to speed up :DRAW
;and the mouse-sensitivity code, since entire clumps can be ruled out for drawing or selection
;if their extent lies beyond the currently visible world coordinates.
;
;X-SCALE-FACTOR: modified during zooming.  Scales all graphics output that our output methods 
;do.
;
;Y-SCALE-FACTOR: ditto
;
;X-DISPLACEMENT: modified during real panning.  Translates all graphics output that our output
;methods do.
;
;Y-DISPLACEMENT: ditto")
)

#|
(DEFMETHOD (graph :AFTER :INIT) (IGNORE)
  "If no NAME is supplied or is NIL, set it to contain TYPE and a unique number.
If ROOT-VERTICES is non-NIL, include them in the name for further identification."
  (UNLESS (AND (BOUNDP name) name)
    (SETQ name
	  (IF root-vertices
	      ;;Be careful on length--graphs can have LOTS of roots.
	      (LET ((root-string (FORMAT NIL "~S" root-vertices))
		    (genname (SEND type :genname)))
		   (unless (stringp genname) (setq genname ":Genname call lost!"))
		(COND
		  ((> (STRING-LENGTH genname) 30.)
		   (IF (> (STRING-LENGTH genname) 40.)
		       (SUBSTRING genname 0 40.)
		       genname))
		  ;;10 is taken up by the text within the FORMAT string.
		  ((> (+ (STRING-LENGTH genname) 10. (STRING-LENGTH root-string)) 40.)
		   (FORMAT NIL "~a..."
			   (SUBSTRING (FORMAT NIL "~a, roots = ~a" genname root-string)
				      0 37.)))
		  (T
		   (FORMAT NIL "~a, roots = ~a" genname root-string))))
	      (SEND type :genname)))))
|#

(DEFMETHOD (graph :construct) ()
  (LET ((traversal-function (SEND type :traversal-function)))
    (UNLESS (FUNCTIONP traversal-function)
      (debug-print T "~%Warning: Graph Traversal Function ~s is not defined."))
    (IF (SEND type :traverse-recursively?)
	(SEND SELF :construct-recursively traversal-function)
	(SEND SELF :construct-non-recursively traversal-function))))


(DEFMETHOD (graph :construct-non-recursively) (traversal-function)
  (debug-print T "~%Computing graph from root vertices (non-recursively).  If this errs, ~
                    check traversal function ~s for ~S." traversal-function (SEND type :name))
  (SETQ traversal-function (get-compiled-function-or-die-trying traversal-function))
  ;;For each root vertex, create an entry in VERTICES of the form
  ;;(<root-vertex> . <VERTEX struct to hold its data and edges>)
  (LET (added-vertices)
    (LABELS
      ((apply-traversal-function (vertex)
	 (LET ((connections (FUNCALL traversal-function vertex))
	       (vstruct (make-vertex :location (CONS 0 0) :data vertex)))
	   (debug-print T "~%Edges from ~s = ~s" vertex connections)
	   ;;SETF required instead of initializing via :EDGE keyword because of
	   ;;hoky naming conflicts in MAKE-VERTEX.
	   (SETF (vertex-edges vstruct)
		 (LET (outgoing-edges)
		   (DOLIST (edge connections)
		     ;;Consistency check.  If we generate any connections to vertices which 
		     ;;are outside of the ROOT-VERTICES, ask user what to do about it. 
		     ;;(Method :CONSTRUCT-RECURSIVELY would just apply TRAVERAL-FUNCTION 
		     ;;recursively, here.)
		     (UNLESS (OR (MEMQ (CAR edge) root-vertices)
				 (MEMQ (CAR edge) added-vertices))
		       (CERROR
			 "Apply ~s recursively on tail vertex ~2* ~s."
		       "Application of traversal function ~s on vertex ~s generated edge ~s~%~
                        but tail vertex ~s is not among the root vertices."
			 (SEND type :traversal-function) vertex edge (CAR edge))
		       (PUSH (CAR edge) added-vertices)
		       (apply-traversal-function (CAR edge)))
		     ;;Make edge, push it onto vertex's edge list, and
		     ;;push it onto the graph's global edge list.
		     (PUSH (make-edge :vertex-2 (CAR edge) :data (LIST (CDR edge)))
			   outgoing-edges)
		     (PUSH (CAR outgoing-edges) edges))
		   outgoing-edges))
	   (PUSH (CONS vertex vstruct) vertices))))
      (DOLIST (root root-vertices)
	(apply-traversal-function root))))
  ;;Flesh out the EDGE structs.
  (DOLIST (entry vertices)
    (DOLIST (edge (vertex-edges (CDR entry)))
      (SETF (edge-vertex-1 edge) (CDR entry)
	    (edge-vertex-2 edge) (CDR (ASSOC (edge-vertex-2 edge) vertices :TEST #'EQ)))))
  ;;Convert VERTICES from temporary alist structure to a flat list.
  (DO* ((vs vertices (CDR vs)))
       ((NULL vs))
    (SETF (CAR vs) (CDAR vs)))
  ;;Set the CONNECTED-TO fields to contain vertices at the tail of outgoing edges.
  ;;CONNECTED-TO is an efficiency hack.
  (DOLIST (vertex vertices)
    (SETF (vertex-connected-to vertex) (MAPCAR #'(LAMBDA (edge) (edge-vertex-2 edge))
					       (vertex-edges vertex))))
  ;;Now add the incoming connections.
  (DOLIST (vertex vertices)
    (DOLIST (tail (vertex-connected-to vertex))
      (PUSHNEW vertex (vertex-connected-to tail))))

  (SEND SELF :merge-edges)  ;;Instead of doing this, we should merge up there.
  (SEND SELF :eliminate-self-loops))


(DEFMETHOD (graph :construct-recursively) (traversal-function)
  (debug-print T "~%Computing graph from root vertices (recusively).  If this errs, check ~
                    traversal function ~s for ~s" traversal-function (SEND type :name))
  (SETQ traversal-function (get-compiled-function-or-die-trying traversal-function))
  ;;Do the traversal.  These three are declared special so that GROW-GRAPH can access them.
  (LET ((visited-objects NIL)
	(collected-vertices NIL)
	(collected-edges NIL))
    (DECLARE (SPECIAL visited-objects collected-vertices collected-edges))
    ;;Construct the graph through a recursive traversal, starting with the root vertices.
    (DOLIST (root root-vertices)
      (grow-graph root traversal-function))
    ;;Store the results of the calls to GROW-GRAPH in our instance variables.
    (SETQ vertices collected-vertices
	  edges collected-edges))
  (SEND SELF :merge-edges)
  (SEND SELF :eliminate-self-loops))

(DEFUN grow-graph (object user-traversal-function)
  "Unless we've been called with OBJECT as argument previously, build onto the graph by
adding edges leading from OBJECT.  OBJECT is a Lisp object in the user's datastructure.  
Initially this function is called with a user-designated root object.  A list of these objects
and the use of recursive calls is how we build the rest of the graph.  

If you are trying to deal with a bug in your graph traversal function, try evaluating the
following template:

  (LET (visited-objects
        collected-vertices
        collected-edges)
    (DECLARE (SPECIAL visited-objects collected-vertices collected-edges))  
    (grow-graph 'INSERT-ONE-OF-YOUR-ROOT-OBJECTS-HERE 'INSERT-YOUR-TRAVERSAL-FUNCTION-NAME))"
  (DECLARE (SPECIAL visited-objects collected-vertices collected-edges))
  (UNLESS (MEMQ object visited-objects)
    (PUSH object visited-objects)
    (LET ((connected-objects-alist (FUNCALL user-traversal-function object))
	  ;;Look for a precreated struct for OBJECT, which would have been created in a
	  ;;previous (recursive) call.
	  ;;If there isn't one, make one and push it onto COLLECTED-VERTICES.
	  (vertex-struct (find-or-make-vertex-struct object)))
      
      ;;Let user see the process of generating vertices, so that if his function fails,
      ;;he'll have a clue.
      (debug-print T "~%Edges from ~s = ~s" object connected-objects-alist)
      
      ;;Add any vertices reachable from OBJECT via an edge in the graph.
      ;;We get an alist of these using the user-supplied graph traversal function.
      ;LOOP FOR (connected-object . edge-label) IN connected-objects-alist DO
      (dolist (entry connected-objects-alist)
         (let ((connected-object (car entry))
		(edge-label (cdr entry)))
	    (let* ((connected-object-vertex-struct (find-or-make-vertex-struct
						     connected-object))
		   (new-edge (make-edge :vertex-1 vertex-struct
					:vertex-2 connected-object-vertex-struct
					;;LISTed since one edge struct displays all
					;;relationships from one vertex to another.
					:data (LIST edge-label))))
	      (SETF (vertex-edges vertex-struct)
		    (CONS new-edge (vertex-edges vertex-struct)))
	      (PUSH new-edge collected-edges)
	      (PUSHNEW connected-object-vertex-struct (vertex-connected-to vertex-struct))
	      (PUSHNEW vertex-struct (vertex-connected-to connected-object-vertex-struct)))
	    (grow-graph connected-object user-traversal-function))))))

(DEFUN find-or-make-vertex-struct (object)
  "Look through special variable COLLECTED-VERTICES for a vertex struct for OBJECT.
If there is one, return it.  Otherwise create one and return it."
  (DECLARE (SPECIAL collected-vertices))
  (OR (FIND object collected-vertices
	    :TEST #'(LAMBDA (object struct) (EQ object (vertex-data struct))))
      (CAR (PUSH (make-vertex :location (CONS 0 0) :data object) collected-vertices))))

(DEFMETHOD (graph :merge-edges) ()
  ;;If more than one edge goes from one vertex to another, merge all of them into
  ;;one composite edge.  This edge's data slot will hold a list of all the edges' data
  ;;and these will be displayed together as the edge's label.  If this merging weren't
  ;;done, the crossover minimization would produce bogus results at times, and these edge
  ;;labels would overwrite each other.  NOTE: the success of this routine depends upon
  ;;DELETE-DUPLICATES implemented by deleting the first arg to :TEST.
  (DOLIST (vertex vertices)
    (SETF (vertex-edges vertex)
	  (DELETE-DUPLICATES (vertex-edges vertex)
			     :TEST #'(LAMBDA (possibly-edge-to-remove edge-to-keep)
				     ;;If VERTEX has two edges leading to the same vertex,
				     ;;Merge them into one and throw away the other.
				     (WHEN (EQ (edge-vertex-2 possibly-edge-to-remove)
					       (edge-vertex-2 edge-to-keep))
				       (SETF (edge-data edge-to-keep)
					     (NCONC (edge-data possibly-edge-to-remove)
						    (edge-data edge-to-keep)))
				       ;;Also delete from the local list of all edges.
				       (SETQ edges (DELQ possibly-edge-to-remove edges))
				       T))))))

(DEFMETHOD (graph :eliminate-self-loops) ()
  ;;Zgraph doesn't bother displaying self loops (edges going from one vertex to itself)
  ;;since it junks up the display on large graphs.  Instead, this method removes the
  ;;self loops but stores them incase the user wants to see them.
  (LET (self-loops)
    (DOLIST (vertex vertices)
      (WHEN (MEMQ vertex (vertex-connected-to vertex))	;fast test
	(SETF (vertex-connected-to vertex) (DELETE vertex (vertex-connected-to vertex))
	      (vertex-edges vertex) (DELETE-IF #'(LAMBDA (edge)
						   (WHEN (EQ (edge-vertex-2 edge) vertex)
						     (PUSH edge self-loops)
						     T))
					       (vertex-edges vertex)))))
    (SETQ edges (DELETE-IF #'(LAMBDA (edge)
			       (MEMQ edge self-loops))
			   edges))      
    (SEND SELF :PUTPROP self-loops :self-loops)))

(DEFUN line-degree (vertex)
  "For the purposes of minimizing crossovers, LINE-DEGREE
is the number of connections to/from other non-leaf vertices.
Connections to leaves don't affect crossovers.  If VERTEX is
a leaf, we return a line-degree of NIL."
  (LET ((connected-to (vertex-connected-to vertex)))
    (WHEN (CDR connected-to)
      (do* ((v (car connected-to) (car ct))
	    (ct (cdr connected-to) (cdr ct))
	    (cnt 0))
	   ((null v) cnt)
	 (when (CDR (vertex-connected-to v))
	   (1+ cnt))))))
;  (LET ((connected-to (vertex-connected-to vertex)))
;    (WHEN (CDR connected-to)
;      (LOOP FOR v IN connected-to
;	    COUNTING (CDR (vertex-connected-to v))))))

(DEFMACRO non-leaf? (vertex)
  "Returns non-NIL if VERTEX is a non-leaf of the graph."
  `(CDR (vertex-connected-to ,vertex)))

(DEFMACRO leaf? (vertex)
  "Returns non-NIL if VERTEX is a leaf of the graph."
  `(NULL (non-leaf? ,vertex)))

(DEFMETHOD (graph :description) (&OPTIONAL (stream *STANDARD-OUTPUT*))
  "Formats a description of the graph to STREAM."
  (LET ((number-of-vertices (LENGTH vertices)))
    ;;What else is useful?
    ;; RT Change
    (IF (equal stream *STANDARD-OUTPUT*)
    (format stream "~a
 Graph Type: ~s
 ~s vertices, ~s directed edges
 Approx. number of clipped vertices = ~s
 Number of self-loops = ~s"
	    name
	    (SEND type :name)
	    number-of-vertices
	    (LENGTH edges)
	    (SEND SELF :GET :clipped-vertex-count)
	    (LENGTH (SEND SELF :GET :self-loops)))
(FORMAT stream "~a
 Graph Type: ~s
 ~s vertices, ~s directed edges
 Approx. number of clipped vertices = ~s
 Number of self-loops = ~s"
	    name
	    (SEND type :name)
	    number-of-vertices
	    (LENGTH edges)
	    (SEND SELF :GET :clipped-vertex-count)
	    (LENGTH (SEND SELF :GET :self-loops))))))

(DEFMETHOD (graph :plot-vertices) ()
;  (DEFMETHOD (graph :plot-vertices) (&OPTIONAL (window *display-io*))
  "Dispatch the current plotting style messages, unless there are no vertices.
WINDOW is an instance of ZG:GRAPH-DISPLAY-PANE on which the graph will be drawn.
It and its REAL-WINDOW's sizes dictate the initial scale of the plotted graph.

Also sets instance variable LOCATION-CLUMPS to a list of clumps of vertices and edges of the
form
  ((XMIN . YMIN) (XMAX . YMAX) VERTICES-IN-THIS-CLUMP EDGES-IN-THIS-CLUMP)"
;  (LET ((*display-io* window)
;        (*graph-output* (SEND window :real-window)))
   (let ()	 
    (WHEN vertices
      (SEND SELF *graph-plotting-style*))
    (SEND SELF :plot-edges)
    (SETQ location-clumps
	  (clump-vertices-and-edges-by-location vertices edges))))
 

(DEFUN clump-vertices-and-edges-by-location (vertices edges
						      &OPTIONAL
						      (number-rows-of-clumps 10.0)
						      (number-columns-of-clumps 10.0))
  "Clumps vertices and edges into groups based on location.
First the world-coordinate extents of the vertices are calculated.  This provides
a rectangular boundary which is conceptually subdivided into 
NUMBER-ROWS-OF-CLUMPS x NUMBER-COLUMNS-OF-CLUMPS sectors.  VERTICES and EDGES are placed
in the sector which covers their territory.  This is used to speed up graph display (through
intelligent clipping operations) and mouse sensitivity."
  (WHEN vertices
    (MULTIPLE-VALUE-BIND (xmin ymin xmax ymax)
	;;Note that we don't need to look at edges to calculate extents.  Just vertices.
	(extents vertices) 
      (LET* ((clump-width (/ (- xmax xmin) number-rows-of-clumps))
	     (clump-height (/ (- ymax ymin) number-columns-of-clumps))
	     clumps)
	;;If we have a one-dimensional extent {as with (NULL (CDR vertices))}, make it infinitely
	;;wide/high instead of 0.
	(WHEN (ZEROP clump-width)
	  (SETQ clump-width most-positive-fixnum
		xmin (- (/ most-positive-fixnum 2.0))))
	(WHEN (ZEROP clump-height)
	  (SETQ clump-height most-positive-fixnum
		ymin (- (/ most-positive-fixnum 2.0))))
	(LABELS
	  ;;If there's already a clump made for location return it--otherwise make one.
	  ((get-or-make-clump (location)
	     (OR (get-clump (CAR location) (CDR location) clumps)
		 ;;This computation is fairly complex.   Say the graph's world coordinates
		 ;;start at -150 and our clump width/height are 100.  We want to generate
		 ;;clumps of the form
		 ;;(({-150+100i} . {-150+150j}) ({150+100i} . {150+150j}) {vertices} {edges}).
		 ;;where the CAR is the minimum location in the clump, the CDR is the max 
		 ;;location, which in this example is 100 more than the minimum 
		 ;;coordinates, and {vertices} and {edges} locations' lie 
		 ;;within the extents defined by the CAR and CDR.  An easier method is to 
		 ;;generate all possible clumps, add vertices to them, then weed out any 
		 ;;empty clumps, but that would cons more.
		 (LET ((clump-xmin (+ xmin (* clump-width (TRUNCATE (- (CAR location) xmin)
								    clump-width))))
		       (clump-ymin (+ ymin (* clump-height (TRUNCATE (- (CDR location) ymin)
								     clump-height)))))
		   ;;Add and return the clump
		   (CAR (PUSH (LIST (CONS clump-xmin clump-ymin)
				    (CONS (+ clump-xmin clump-width)
					  (+ clump-ymin clump-height))
				    NIL		;Vertices
				    NIL)	;Edges
			      clumps))))))
	  ;;Put all vertices into their clumps.
	  (DOLIST (vertex vertices)
	    (PUSH vertex (THIRD (get-or-make-clump (vertex-location vertex)))))
	  (DOLIST (edge edges)
	    (PUSH edge (FOURTH (get-or-make-clump (edge-misc edge)))))
	  clumps)))))

(DEFUN extents (vertices)
  "Returns the extents surrounding VERTICES."
  (LET* ((xmax (CAR (vertex-location (CAR vertices))))
	 (ymax (CDR (vertex-location (CAR vertices))))
	 (xmin xmax)
	 (ymin ymax))
    (DOLIST (vertex (CDR vertices))
      (LET ((location (vertex-location vertex)))
	(IF (> (CAR location) xmax)
	    (SETQ xmax (CAR location))
	    (IF (< (CAR location) xmin)
		(SETQ xmin (CAR location))))
	(IF (> (CDR location) ymax)
	    (SETQ ymax (CDR location))
	    (IF (< (CDR location) ymin)
		(SETQ ymin (CDR location))))))
    (VALUES xmin ymin xmax ymax)))

(DEFUN get-clump (x y clumps)
  "Returns the clump in CLUMPS representing a group of graphics objects whose combined
extents location X,Y lies within."
  (FIND-IF #'(LAMBDA (clump)
	       (AND (<= (CAAR clump) x (CAADR clump))
		    (<= (CDAR clump) y (CDADR clump))))
	   clumps))

(DEFMETHOD (graph :plot-edges) ()
  "Calculates locations for the directional pointers of edges, for use in mouse-sensitivity."
  (DOLIST (edge edges)
    (LET* ((from-point (vertex-location (edge-vertex-1 edge)))
	   (to-point (vertex-location (edge-vertex-2 edge))))
      (SETF (edge-misc edge)
	    (CONS (value-between (CAR from-point) (CAR to-point) .9)
		  (value-between (CDR from-point) (CDR to-point) .9))))))


;;
;; First method for plotting graphs: :PLOT-IN-ONE-BIG-CIRCLE
;;

(DEFUN decent-radius (number-of-non-leaf-vertices)
  "Returns a good radius to use for arranging a given number of vertices and
   edges in a circle."
  ;;Short edges look bad, so we impose a minimum edge length.
  ;;CIRCUMFERENCE  = 2 * PI * RADIUS so roughly
  ;;RADIUS = NUMBER-OF-VERTICES * {minimum edge length} / (2 * PI)
  (LET ((radius (/ (* number-of-non-leaf-vertices 500.) *2PI*)))
    ;;This takes into account the fact that :ARRANGE-VERTICES-IN-CIRCLE cuts
    ;;out a portion of the circle for the vertex leaves.
    (+ radius (* radius *percentage-radius-for-leaves-in-circular-arrangement*))))


(DEFMETHOD (graph :plot-in-one-big-circle) ()
  "This graph plotting method places non-leaf vertices of the graph in a circle.
 A circular arrangement won't always be optimum (wrt. crossovers), but it buys us several 
 things.  First, we can compute the number of crossovers symbolicly, so our computation in
 reducing crossovers is alot faster.  Second, it isn't clear that a tangled planar graph with
 few crossovers is easier to read than a graph arranged symetrically in a circle with more 
 crossovers.  Third, it's much easier to compute a circle of vertices, which is inherently a
 reasonable layout, than to compute a tangle of vertices which happens to come out planar or 
 close-to-planar.  Typically whenever I have to draw a graph for some reason, it's usually
 easiest to lay out and read the edges of the graph by placing the vertices in a circle.

The leaves of the graph are placed as sattelites to the circle around the vertices to which
they are connected."
  (LET (non-leaves)
    ;;Extract the leaves of the graph.
    (DOLIST (vertex vertices)
      (WHEN (non-leaf? vertex)
	(PUSH vertex non-leaves)))
    ;;Add to NON-LEAVES any leaf that has no connection with NON-LEAVES.  If we don't do 
    ;;this, isolated trees will not be plotted.
    (DOLIST (vertex vertices)
      (WHEN (leaf? vertex)
	(UNLESS (SOME #'(LAMBDA (connected-to) (MEMQ connected-to non-leaves))
		      (vertex-connected-to vertex))
	  (PUSH vertex non-leaves))))
    ;;Reorder NON-LEAVES, which will be displayed in a circle, so as to minimize crossovers.
    (SETQ non-leaves (SEND SELF :minimize-crossovers-for-circular-arrangement non-leaves))

    ;;Assign NON-LEAVES positions around the circle.
    (LET ((radius (decent-radius (LENGTH non-leaves))))
      (SEND SELF :arrange-vertices-in-circle non-leaves 0 0 radius radius))
    ;;Scale so user sees at least some of the graph.
    (SEND SELF :scale-for-initial-viewing)))


(DEFMETHOD (graph :scale-for-initial-viewing) (&OPTIONAL
						(real-window *graph-output*)
  					       ; (display-window *display-io*)
					       )
  "Sets the graph scale so that the user initially sees at least a part of the graph.
If there are more than a globally specified number of vertices, fits the graph into the 
output window; otherwise, fits it into the display pane.  If we don't do this, many graphs
will appear completely outside both the output window and the display pane."
  (MULTIPLE-VALUE-BIND (width height)
      (COND
	((> (LENGTH vertices)
	    *number-vertices-over-which-graphs-are-fit-onto-hidden-bit-array*)
	 ;;Not enough screen space!
	 (debug-print T "~%Fitting graph onto hidden bit array.  Pan to see portions not displayed on the window.")
	 (SEND real-window :INSIDE-SIZE))
	(T
	 (debug-print T "~%Fitting graph onto the display pane--all vertices will be visible.")
	 (display-pane-inside-size)
	; (SEND display-window :INSIDE-SIZE)
	 ))
    ;;If graph only has two vertices (connected by an edge usually), it looks better if we
    ;;let :SCALE-TO-FIT-WINDOW stretch one of the dimensions so as to lengthen the connecting
    ;;edge to fit snuggly into the window.  This is what the third argument specifies.
    ;;All other cases don't have this problem, since more than two vertices arranged in a
    ;;circle fit "snuggly" into a roughly square area (the window). 
    (SEND SELF :scale-to-fit-window width height (= (LENGTH vertices) 2))))


;;
;; Second (and default) method for plotting graphs: 
;; :PLOT-IN-CIRCLES-FOR-BI-CONNECTED-COMPONENTS
;;

(DEFMETHOD (graph :plot-in-circles-for-bi-connected-components) ()
  "This graph plotting method places certain bi-connected components of the graph in their 
 own circle.  Bi-connected components of graphs are those subgraphs which are connected such 
 that there is more than one path between each vertex in the subgraph.  In otherwords, 
 bi-connected components of a graph are each connected via only one edge.  This makes a good 
 grouping (perceptually) of the graph into subgraphs (with both connected and disconnected
 components, where any two subgraphs are connected by at most one edge, by the nature of
 bi-connected components).  There are undoubtedly other good groupings, such as 
 number-of-connections under some threshold.  However, I chose this grouping because of the
 availability of a fast algorithm for determining bi-connected components.

 Final note: certain bi-connected components are combined into one subgraph for ease of 
 handling.  These are those bi-connected components which share one vertex (rather than being
 connected by one edge."

  ;;This is how it works:
  ;;1. Generate the initial set of subgraphs from the bi-connected components of the graph.
  ;;   Merge some of these subgraphs together as described above (and below).
  ;;   Remove all edges connecting subgraphs from the graph, storing them for later addition.
  ;;2. Arrange each subgraph in its circle such that crossovers within the
  ;;   subgraph are minimized.  (The circular placements are figured at this point--not
  ;;   the actual coordinates of vertices.)
  ;;3. Add the edges connecting subgraphs back into the graph.
  ;;4. Use these edges to figure which subgraphs are connected together
  ;;   and a good screen area for each component, using a hexagonal grid arrangement.
  (MULTIPLE-VALUE-BIND (subgraphs connecting-edges)
      (extract-cycles vertices)

    ;;Step 1.

    ;;Create subgraphs for each connecting edge vertex which is in a tree.  For an example,
    ;;assume the following: edge (v6 . v12) is in the connecting edges.  V6 is in a 
    ;;bi-connected component.  V12 is not a leaf and is NOT in a bi-connected component 
    ;;(meaning it is part of a tree). We still want to remove (v6 . v12) as part of Step 1,
    ;;but since V12 isn't a leaf, we need to include it (and non-leaves attached to it) in a
    ;; new subgraph.
    (LABELS ((tree-vertex? (vertex)
	       (AND (non-leaf? vertex) (NOT (SOME #'(LAMBDA (subgraph) (MEMQ vertex subgraph))
						  subgraphs)))))
	(LET (tree-vertices tree-subgraphs)
	  (dolist (edge connecting-edges)
		(WHEN (tree-vertex? (CAR edge))
		  (PUSHNEW (CAR edge) tree-vertices))
		(WHEN (tree-vertex? (CDR edge))
		  (PUSHNEW (CDR edge) tree-vertices)))
	  
	;  (LOOP FOR edge IN connecting-edges
	;	WHEN (tree-vertex? (CAR edge))
	;	DO (PUSHNEW (CAR edge) tree-vertices)
	;	WHEN (tree-vertex? (CDR edge))
	;	DO (PUSHNEW (CDR edge) tree-vertices))
	  ;;Start out with a tree for each vertex and its immediate connected vertices which
	  ;;are among TREE-VERTICES (otherwise vertices in SUBGRAPHS  and leaf vertices would
	  ;;be included). Then destructively merge trees which share vertices, using NCOMBINE.
	  (SETQ tree-subgraphs
		(ncombine
		 (mapcar #'(lambda (vertex)
			     (cons vertex
				   (delete nil (mapcar #'(lambda (connected-to)
							   (when (memq connected-to tree-vertices)
							     connected-to)) 
						       (vertex-connected-to vertex)))))
			 tree-vertices)
;; John, I warned you about this! -- KDF
;		  (LOOP FOR vertex IN tree-vertices
;			COLLECT (CONS vertex
;				      (LOOP FOR connected-to IN (vertex-connected-to vertex)
;					    WHEN (MEMQ connected-to tree-vertices)
;					    COLLECT connected-to)))
		  :TEST #'EQ))
	  ;;Since this system doesn't handle trees specially, a quick way to have trees
	  ;;processed as if they were normal subgraphs (where all non-leaf vertices have
	  ;;non-NIL line-degree) is to double up one of each tree element's connected-to's,
	  ;;so that they are processed as non-leaves o.k. (they get a non-NIL line-degree).
	  ;;(The one we double up has to be a non-leaf.) Of course, this is just a way of
	  ;;quickly handling the problem, and the correct way is to print each of these trees
	  ;;in tree format, rather than doing the NCONC.  This would be fairly easy to fit
	  ;;in: allocate all trees space underneath the subgraph circles and print them
	  ;;downwards.
	  (DOLIST (subgraph tree-subgraphs)
	    (DOLIST (vertex subgraph)
	      (DOLIST (v (vertex-connected-to vertex))
		 (WHEN (non-leaf? v)
		   (PUSH v (vertex-connected-to vertex))
		   (RETURN NIL)))))
	  ;;Delete edges making up the trees from CONNECTING-EDGES, similar to when we
	  ;;deleted leaves from CONNECTING-EDGES above.  If we don't do this, edges of the
	  ;;tree will be treated as subgraph connectors in code below.
	  (SETQ connecting-edges
		(DELETE-IF #'(LAMBDA (edge) (AND (MEMQ (CAR edge) tree-vertices)
					       (MEMQ (CDR edge) tree-vertices)))
					    connecting-edges))
	  (SETQ subgraphs (NCONC subgraphs tree-subgraphs))))

    ;;Create a subgraph to hold any leaf that has no connection into a current subgraph.
    ;;If we don't do this, isolated trees will not be plotted.
    (LET (stray-leaves)
      (DOLIST (vertex vertices)
	(WHEN (leaf? vertex)
	  (UNLESS (SOME #'(LAMBDA (connected-to)
			  (SOME #'(LAMBDA (subgraph)
				  (MEMQ connected-to subgraph))
				subgraphs))
			(vertex-connected-to vertex))
	    (PUSH vertex stray-leaves))))
      (WHEN stray-leaves
	(PUSH stray-leaves subgraphs)))

    ;;We only want to remove edges connecting subgraphs, so that each subgraph can be plotted
    ;;in its own circle (using :MINIMIZE-CROSSOVERS-FOR-CIRCULAR-ARRANGEMENT).  Therefore,
    ;;we save edges leading to/from graph leaves from the edges we are about to remove.
    (SETQ connecting-edges
	  (DELETE-IF-NOT #'(LAMBDA (edge) (AND (non-leaf? (CAR edge)) (non-leaf? (CDR edge))))
			 connecting-edges))

    ;;Temporarily remove all subgraph-connecting edges from the graph.
    (DOLIST (edge connecting-edges)
      (LET ((v1 (CAR edge))
	    (v2 (CDR edge)))
	;;1 is efficient & important to the above kludge. 
	(SETF (vertex-connected-to v1) (DELQ v2 (vertex-connected-to v1) 1)
	      (vertex-connected-to v2) (DELQ v1 (vertex-connected-to v2) 1))))     
    ;;Combine any subgraphs (bi-connected components) which share a vertex.  For instance,
    ;;in the graph represented by
    ;;  (SETQ v1 '(v2 v3 v4) v2 '(v1 v3) v3 '(v1 v2)
    ;;        v4 '(v1 v5) v5 '(v4 v1))
    ;;V1 is shared by the (v1 v2 v3) bi-connected component and by the (v4 v5 v6)
    ;;bi-connected component. We want to combine these two connected components into one
    ;;circle; otherwise we'd have to share V1 between them, either by placing V1 arbitarily
    ;;in one or the other and connecting the two circles by more than one edge.
    ;;In case you aren't up on your bi-connected component theory, the following is a graph
    ;;with two bcc's which we *do* want display in two circles:
    ;;  (SETQ v1 '(v2 v3 v4) v2 '(v1 v3) v3 '(v1 v2)
    ;;        v4 '(v1 v5 v6) v5 '(v4 v6) v6 '(v4 v5)
    (SETQ subgraphs (ncombine subgraphs :TEST #'EQ))

    ;;Step #2.
     (do ((sgraphs subgraphs (cdr sgraphs)))
	 ((null  sgraphs))
       (setf (car sgraphs) (send self :minimize-crossovers-for-circular-arrangement (car sgraphs))))
;; Sigh.  Nobody believed me.
;    (LOOP FOR sgraphs ON subgraphs DO
;	  ;;Reorder NON-LEAVES, which will be displayed in a circle, so as to minimize
;	  ;;crossovers.
;	  (SETF (CAR sgraphs)
;		(SEND SELF :minimize-crossovers-for-circular-arrangement (CAR sgraphs))))

    ;;Step #3. Add back in the edges we removed from the graph in step 1.
    ;;This might not actually be needed for anything, but do it for any future changes
    ;;that depend on it.
    (DOLIST (edge connecting-edges)
      (LET ((v1 (CAR edge))
	    (v2 (CDR edge)))
	(SETF (vertex-connected-to v1) (CONS v2 (vertex-connected-to v1))
	      (vertex-connected-to v2) (CONS v1 (vertex-connected-to v2)))))
    ;;Step #4.
    (SEND SELF :place-subgraphs-on-a-hexagonal-grid subgraphs connecting-edges)
    ;;Scale so user sees at least some of the graph.
    (SEND SELF :scale-for-initial-viewing)))
										   
(DEFMETHOD (graph :place-subgraphs-on-a-hexagonal-grid) (subgraphs connecting-edges)
  ;;Use the set of connecting edges to figure out how to arrange the subgraphs so as to
  ;;minimize crossovers.  Since we're using circular areas for each component of the graph,
  ;;the optimum arrangement is a hexagonal grid.  (Picture a bunch of coins crowded
  ;;together on a table.)  So first we find a good arrangement of the components in a grid,
  ;;then rotate each component's circle to reduce crossovers caused by edges between
  ;;components.
  (LET* ((radius (decent-radius (do ((sgs subgraphs (cdr sgs))
				     (max 0) (len 0))
				    ((null sgs) max)
				  (setq len (length (car sgs)))
				  (if (> len max) (setq max len)))
		  
		  ;(LOOP FOR component IN subgraphs
		  ;		      MAXIMIZE (LENGTH component))
				))
	 most-popular
	 (most-connections 0))
    ;;Modify SUBGRAPHS to consist of elements
    ;;  (<subgraph> (<connected-to> <vertex> <vertex-to>)...)
    ;;where <connected-to> is a subgraph connected to <subgraph> through and edge at
    ;;<vertex> and <vertex-to>, where <vertex> is in <subgraph> and <vertex-to> is in
    ;;<connected-to>.
    (do ((sgraphs subgraphs (cdr sgraphs))
	 (subgraph nil))
	((null sgraphs))
      (setq subgraph (car sgraphs))
      (setf (car sgraphs)
	    (cons subgraph
		  (do ((ces connecting-edges (cdr ces))
		       (connecting-edge nil)
		       (our-edge nil) (result nil))
		      ((null ces) result)
		    (setq connecting-edge (car ces)
			  our-edge (if (memq (car connecting-edge) subgraph)
				       (list (car connecting-edge) (cdr connecting-edge))
				       (if (memq (cdr connecting-edge) subgraph)
					   (list (cdr connecting-edge) (car connecting-edge)))))
		    (when our-edge 
		      (push (cons (some #'(lambda (connected-to?)
					    (if (listp (car connected-to?))
						(when (memq (cadr our-edge) (car connected-to?))
						  (car connected-to?))
						(when (memq (cadr our-edge) connected-to?)
						  connected-to?))) subgraphs) our-edge) result))))))

;    (LOOP FOR sgraphs ON subgraphs
;	  FOR subgraph = (CAR sgraphs)
;	  DO
;      (SETF (CAR sgraphs)
;	    (CONS subgraph
;		  (LOOP FOR connecting-edge IN connecting-edges
;			FOR our-edge = (IF (MEMQ (CAR connecting-edge) subgraph)
;					   (LIST (CAR connecting-edge)
;						 (CDR connecting-edge))
;					   (IF (MEMQ (CDR connecting-edge) subgraph)
;					       (LIST (CDR connecting-edge)
;						     (CAR connecting-edge))))
;			WHEN our-edge
;			  ;;(CADR our-edge) = <vertex-to>
;			  COLLECT
;			    (CONS (SOME
;				    #'(LAMBDA (connected-to?)
;				      ;;Kludge.  In this loop we're modifying the 
;				      ;;elements of SUBGRAPHS, so any comparisons have
;				      ;;to be made against both the old and new format
;				      ;;of the list.  All to save consing.
;				      (IF (LISTP (CAR connected-to?))
;					  (WHEN (MEMQ (CADR our-edge) (CAR connected-to?))
;					    (CAR connected-to?))
;					  (WHEN (MEMQ (CADR our-edge) connected-to?)
;					    connected-to?)))
;				    subgraphs)
;				  our-edge)))))
    ;;Pick the subgraph having most connections to other subgraphs for the middle 
    ;;grid location. If there are no connections (as in a connected graph or where each
    ;;subgraph is disconnected), pick the subgraph with the most vertices.
    ;;We'll arrange the subgraphs it is connected to in adjacent locations.
    (DOLIST (entry subgraphs)
      (LET ((count (LENGTH (CDR entry))))
	(WHEN (< most-connections count)
	  (SETQ most-popular entry
		most-connections count))))
    (UNLESS most-popular
      (DOLIST (entry subgraphs)
	(LET ((count (LENGTH (CAR entry))))
	  (WHEN (< most-connections count)
	    (SETQ most-popular entry
		  most-connections count)))))
    ;;Put the most popular at the front of SUBGRAPHS.
    (SETQ subgraphs (CONS most-popular (DELETE most-popular subgraphs :TEST #'EQ)))
    ;;Make it the origin.
    (SETF (CDR most-popular) `(0 0 . ,(CDR most-popular)))
    
    ;;Make everyone else's coordinates unassigned.
    ;;This makes SUBGRAPHS consist of elements
    ;;(<subgraph> <x> <y> (<connected-to> <vertex> <vertex-to>)...)
    (DOLIST (entry (CDR subgraphs))
      (SETF (CDR entry) `(NIL NIL . ,(CDR entry))))
    
    ;;Assign relatively good coordinates to every subgraph.
    (DOLIST (entry subgraphs)
      (WHEN (CADR entry)
	(impose-ones-will-on-location-of-other-subgraphs entry subgraphs)))
    ;;Fit isolated subgraphs in somewhere.
    (assign-locations-near-origin-to-isolated-subgraphs subgraphs)
    ;;Hexagonalize the grid.  x := x + .5 * y;  y := y * {circlular constant}
    (DOLIST (entry subgraphs)
      (MULTIPLE-VALUE-BIND (x y)
	  (hexagonal-equivalents (CADR entry) (CADDR entry))
	(SETF (CADR entry) x)
	(SETF (CADDR entry) y)))

    (dolist (entry (cons most-popular subgraphs))
      (let* ((subgraph (car entry))
	     (x (cadr entry))
	     (y (caddr entry))
	     (diameter (* 2 radius))
	     (xmin (* diameter x))
	     (ymin (* diameter y)))
	(send self :arrange-vertices-in-circle
	      subgraph xmin ymin (+ xmin diameter) (+ ymin diameter))))
    
;    (LOOP FOR (subgraph x y . NIL) IN (CONS most-popular subgraphs)
;	  WITH diameter = (* 2 radius)
;	  FOR xmin =  (* diameter x)
;	  FOR ymin =  (* diameter y)
;	  DO (SEND SELF :arrange-vertices-in-circle
;		   subgraph xmin ymin (+ xmin diameter) (+ ymin diameter)))


))

(DEFUN hexagonal-equivalents (x y)
  ;;Note that the grid y values have to be reversed in translating the grid into
  ;;world coordinates, in which low values of Y occur at the top of the screen.
  ;;Also, .10 is an approximation which seems to work.  The real value is
  ;;(RADIUS - 1/2 * (2 * RADIUS * SIN (180/6) * COT (180/6))) / RADIUS, as
  ;;given by CRC manual Page 10.
  (VALUES (+ x  (* .5 y))
	  ;;Same as (- (- y (* .10 y)))
	  (- (* .10 y) y)))

(DEFVAR *neighbors-on-either-side-of-direction*
	     '(((-1 1) (-1 0) (0 1))  ;;These are in a specific order!!
	       ((0 1) (-1 1) (1 0))
	       ((1 0) (0 1) (1 -1))
	       ((1 -1) (1 0) (0 -1))
	       ((0 -1) (1 -1) (-1 0))
	       ((-1 0) (0 -1) (-1 1)))
  "In normal cardinal directions, North and West are the neighboring directions to North-West.
In this hexagonal grid, the relationships are similar.")

(DEFUN impose-ones-will-on-location-of-other-subgraphs (boss-subgraph subgraphs)
  (LET ((boss (CAR boss-subgraph)))
    (DOLIST (subgraph subgraphs)
      ;;When connected to boss and no coordinates have been assigned, let boss assign them.
      (LET ((connection (ASSQ (CAR subgraph) (CDDDR boss-subgraph))))
	(WHEN (AND connection (NULL (CADR subgraph)) (NOT (EQ boss (CAR subgraph))))
	  (LET* ((suggested-direction (vertex-direction-from-center boss (CADR connection)))
		 (opposite-direction (LIST (* (CAR suggested-direction) -1)
					   (* (CADR suggested-direction) -1))))
	    (MULTIPLE-VALUE-BIND (x y)
		(find-free-location
		  (CADR boss-subgraph) (CADDR boss-subgraph)
		  (CAR suggested-direction) (CADR suggested-direction) subgraphs)
	      ;;Set the connected subgraph's location to some closest free location in the
	      ;;suggested direction.
 	      (SETF (CADR subgraph) x
		    (CADDR subgraph) y)
	      ;;Rotate the connected subgraph till the connecting vertex faces the boss.
	      ;; IBM RT change
	      (do ((count 0 (1+ count))
		   (length (length (car subgraph))))
		  (nil)
		(when (= count length)
		  (setq opposite-direction (cadr (assoc opposite-direction 
							*neighbors-on-either-side-of-direction*
							:test #'EQUAL))
			count -1))
		(when (equal opposite-direction
			  (vertex-direction-from-center (car subgraph) (caddr connection)))
		   (return))
		(let ((old (car subgraph))
		      (new (nconc (cdar subgraph) (list (caar subgraph)))))
		  (dolist (subgraph subgraphs)
		    (dolist (connected-tos (cdddr subgraph) (cdr connected-tos))
		      (when (eq (car connected-tos) old)
			(setf (car connected-tos) new))))
		  (setf (car subgraph) new))))))))))

(DEFUN vertex-direction-from-center (subgraph vertex)
  "Returns qualitative description of the direction to VERTEX from the center of SUBGRAPH,
where VERTEX is in SUBGRAPH and SUBGRAPH is ordered so that the CAR will be assigned *3/2PI* 
degrees by :ARRANGE-VERTICES-IN-CIRCLE, and the rest are assigned increasing positive 
increments around the circle.  Returns a list (x y) where x and y are 0, 1 or, -1.  x= 1 
means Right, x= -1 means Left, y= 1 means up, y= -1 means down.  0 means no change in 
direction."
  (LET* ((length (LENGTH subgraph))
	 (percentage-of-circle-to-vertex (/ (- length (LENGTH (MEMQ vertex subgraph)))
					    length)))
    ;;The angles used are based on the assumption that we will hexagonalize (shift)
    ;;rightwards (any given row slides 1/2radius to the right over the row below it).
    ;;Picture (or lay out) a bunch of coins on a table.  Pick one in the middle as the
    ;;subgraph.  Then you'll see that the angles below correspond (roughly) to what you see
    ;;on the table.  For instance, there are 6 (not 8) cardinal directions from the coin.
    (COND
      ;;First 60 degrees = coin above and to the left of center coin
      ((<= percentage-of-circle-to-vertex 60/360) '(0 1))
      ;;First 60 degrees = coin to the left.
      ((<= percentage-of-circle-to-vertex 120/360) '(1 0))
      ;;First 60 degrees = coin down and to the left.
      ((<= percentage-of-circle-to-vertex 180/360) '(1 -1))
      ;;First 60 degrees = coin down and to the right.
      ((<= percentage-of-circle-to-vertex 240/360) '(0 -1))
      ;;First 60 degrees = coin to the right.
      ((<= percentage-of-circle-to-vertex 300/360) '(-1 0))
      ;;First 60 degrees = coin up and to the right.
      (T '(-1 1)))))


(DEFUN find-free-location (from-x from-y direction-x direction-y subgraphs)
  "Returns a free location close to FROM-X, FROM-Y in direction DIRECTION-X, DIRECTION-Y.
The direction values are -1, 0, and 1."
  (LET* ((neighboring-directions (CDR (ASSOC (LIST direction-x direction-y)
					     *neighbors-on-either-side-of-direction*
					     :TEST #'EQUAL)))
	 (first-neighbor (CAR neighboring-directions))
	 (second-neighbor (CADR neighboring-directions))
	 (location-queue (LIST (CONS (+ from-x direction-x) (+ from-y direction-y)))))
    ;;IBM RT change
    (do ((x (caar location-queue) (caar location-queue))
	 (y (cdar location-queue) (cdar location-queue)))
	(nil)
      (cond ((location-not-free? x y subgraphs)
	     (nconc location-queue
		    (list (cons (+ x direction-x) (+ y direction-y))
			  (cons (+ x (car first-neighbor)) (+ y (cadr first-neighbor)))
			  (cons (+ x (car second-neighbor)) (+ y (cadr second-neighbor)))))
	     (pop location-queue))
	    (t (return (values (caar location-queue) (cdar location-queue))))))))

(DEFUN assign-locations-near-origin-to-isolated-subgraphs (subgraphs)
  (LET ((all-subgraphs subgraphs)) ;;store head of list.
    (LABELS
      ((next-isolated-subgraph ()
	 ;;CDR down to next subgraph which hasn't been assigned coordinates.
	 ;;When we reach the end of the list, we've finished the assignment so exit function.
	 (DO ((subgraph (POP subgraphs) (POP subgraphs)))
	     (NIL)
	   (IF subgraph
	       (UNLESS (CADR subgraph)
		 (RETURN subgraph))
	       (RETURN-FROM assign-locations-near-origin-to-isolated-subgraphs NIL))))
       (assign-if-location-free (x y)
	 (UNLESS (location-not-free? x y all-subgraphs)
	   (LET ((isolated-subgraph (next-isolated-subgraph)))
	     (SETF (CADR isolated-subgraph) x
		   (CADDR isolated-subgraph) y)))))
      ;;Span out from origin in progressively larger hexagons.
      (DO* ((n 1 (1+ n))
	    (negative-n (- n) (- n)))
	   (NIL)
	;;Top hex side
	(DOTIMES (i (1+ n))
	  (assign-if-location-free (- i) n))
	;;Bottom hex side
	(DOTIMES (i (1+ n))
	  (assign-if-location-free i negative-n))
	;;Upper left hex side
	(DOTIMES (i n)
	  (assign-if-location-free negative-n i))
	;;Lower right hex side
	(DOTIMES (i n)
	  (assign-if-location-free n (- i)))
	;;Lower left hex side
	(do ((i 1 (1+ i)))
	    ((equal i n))
	  (assign-if-location-free (+ negative-n i) (- i)))
;	(LOOP FOR i FROM 1 TO (1- n) DO
;	  (assign-if-location-free (+ negative-n i) (- i)))
	;;Upper right hex side
;	(LOOP FOR i FROM 1 TO (1- n) DO
      	(do ((i 1 (1+ i)))
	    ((equal i n))
	  (assign-if-location-free (- n i) i))
      ))))

(DEFUN location-not-free? (x y subgraphs)
  "Returns NIL if location x,y isn't already taken by a subgraph; otherwise, returns the 
lucky subgraph."
  (SOME #'(LAMBDA (subgraph)
	  (WHEN (AND (CADR subgraph) (= (CADR subgraph) x) (= (CADDR subgraph) y))
	    subgraph))
	subgraphs))

(DEFUN ncombine (lists &KEY (test #'EQL))
  "Merges sublists of LISTS which share any elements, compared using TEST.
LISTS is munged destructively.
Example: (ncombine '((1 2 3 4) (2 5) (6 7 8))) => '((1 2 3 4 2 5) (6 7 8))"
  (LET ((merged-any? T))
;; IBM RT Change
    (do ()
	((not merged-any?))
      (setq merged-any? nil)
      (do ((slow-scanner lists (cdr slow-scanner)))
	  ((null slow-scanner))
	(do ((fast-scanner (cdr slow-scanner) (cdr fast-scanner)))
            ((null fast-scanner))
	  (when (intersection (car slow-scanner) (car fast-scanner) :test test)
	    (setq merged-any? t)
	    (nconc (car slow-scanner)
		   (nset-difference (car fast-scanner) (car slow-scanner) :test test))
	    (setf (car fast-scanner) nil))))))
  (DELETE-IF-NOT #'IDENTITY lists))


;;==========================================================================================
;; Brian Falkenhainer's code for plotting lattices.  
;; ****Ultimately, lattice detection within circles should be automated and this
;; used.
;; ****Loops infinitely on non-lattices.

(DEFMETHOD (graph :plot-lattice) ()
  (LET ((intercolumn-spacing 100)
	(interrow-spacing 30.0)
	(mark (list nil))
	roots lattice tmp)
    ;;Extract the roots of the graph.
    (DOLIST (vertex vertices)
      (WHEN (member (vertex-data vertex) root-vertices :test #'eq)
	(setf (vertex-misc vertex) (cons mark 0))      ;we've seen it and it's on level 0
	(PUSH vertex roots)))
    (setq roots (nreverse roots))
    ;;Do placement
    (setq lattice (list (cons 0 roots)))
    (do ((i 1 (1+ i))
	 (level roots next-level)
	 (next-level nil nil)			;next-level is collected while stepping through this level
	 (phantom-bit nil nil)			;phantoms fade away with successive levels.
	 (all-phantoms?)			;finished when level is all phantoms.
	 (children))
	(all-phantoms?)
      (setq all-phantoms? t)
      (dolist (current-node level)
	(cond ((eq current-node 'phantom)
	       (if phantom-bit (push 'Phantom next-level))   ;2 phantoms in a row -> one on next level
	       (setq phantom-bit (not phantom-bit)))
	      ((setq children (nreverse (mapcar #'edge-vertex-2 (vertex-edges current-node))))
	       (setq phantom-bit nil)
	       (setq all-phantoms? nil)
	       (dolist (c-node children)
		 (cond ((eq (car (vertex-misc c-node)) mark)
			(cond ((/= (cdr (vertex-misc c-node)) i)
				;; remove the node from the higher level and move it down to this level
			       (setq tmp (assoc (cdr (vertex-misc c-node)) lattice))
			       (setf (cdr tmp)  (delete c-node (cdr tmp)))
			       (setf (cdr (vertex-misc c-node)) i)
			       (push c-node next-level))))
		       (t (setf (vertex-misc c-node) (cons mark i))
			  (push c-node next-level)))))
	      ((push 'Phantom next-level))))	;parent has no children, insert phantom
      (setq next-level (nreverse next-level))
      (unless all-phantoms? (setq lattice (nconc lattice (list (cons i next-level))))))
    (do ((lattice (cdr lattice) (cdr lattice))
	 (current-level (cdar lattice) (cdar lattice))
	 (current-x 0 (+ current-x intercolumn-spacing))
	 current-y)
	((null current-level))
      (cond ((> (* (length current-level) interrow-spacing) 1000)
	     (setq current-y 0)
	     (setq interrow-spacing (/ 1000.0 (length current-level))))
	    ((setq current-y (- 500 (* (/ (length current-level) 2.0) interrow-spacing)))))
      (dolist (v current-level)
	(if (not (eq v 'phantom))
	    (setf (vertex-location v) (cons current-x current-y)))
	(incf current-y interrow-spacing)))
    (SEND SELF :scale-for-initial-viewing)))

;;==========================================================================================
;;
;;;; Arranging vertices in a circle so as to minimize crossovers.
;;
;;==========================================================================================

(DEFMETHOD (graph :minimize-crossovers-for-circular-arrangement) (non-leaves)
  "Reorders the vertices in NON-LEAVES so as to minimize crossovers when they are 
arranged (displayed) in a circle.  The optimal solution isn't always made--we go for speed
instead."  ;;Example where it isn't optimum: 4th Sample graph type, big-circle method.
  (COND
    ;;No minimization needed if less than four vertices.
    ((< (LENGTH non-leaves) 4)
     non-leaves)
    ;;Graph is too huge, so use a quicker algorithm (still experimental).
    ((OR
       (>= (LENGTH non-leaves) *too-many-non-leaves*)
       (>= (do ((nl non-leaves (cdr nl))
		(sum 0)
		(vertex nil))
	       ((null nl) sum)
	     (setf vertex (car nl))
	     (setf sum (+ sum (LENGTH (vertex-edges vertex)))))

          ; (LOOP FOR vertex IN non-leaves
	  ;	 SUMMING (LENGTH (vertex-edges vertex)))

	   *too-many-edges*))
     (SEND SELF :minimize-crossovers-for-large-graphs non-leaves))
    (T
     (LET ((stepping? (MEMQ :step-through-placement *graph-debug-actions*)))
       (debug-print T "~%Minimizing crossovers in a circular arrangement of vertices...")
       ;;Sort the non-leaves by decreasing line degree.  The idea is to 
       ;;start with a circle of large-degree non-leaves, then to keep adding
       ;;smaller-degree non-leaves where they will cause the least crossovers.
       (SETQ non-leaves
	     ;;OR is used incase leaves occur in NON-LEAVES.
	     (SORT non-leaves #'(LAMBDA (x y) (> (OR (line-degree x) 0)
						 (OR (line-degree y) 0)))))
       ;;Build up the circular arrangement in CIRCLE.
       ;;We use a circular list for easy handling.
       (LET ((circle (CIRCULAR-LIST (POP non-leaves) (POP non-leaves))))
	 (DOLIST (non-leaf non-leaves)
	   (WHEN stepping?
	     (SEND SELF :display-step (remove-list-circularity (copy-circular-list circle))
		   "~%Stepping through placement algorithm.   Added a non-leaf."))
	   (LET ((best-insertion-point NIL)
		 (least-crossovers 999999999999))
	     ;;Sees if inserting NON-LEAF at the CDR of POSITION produces a minimum of
	     ;;crossovers (so far).
	     (LABELS ((try-insertion (non-leaf position)
			(LET ((crossovers (count-crossovers-after-insertion
					    non-leaf position)))
			  (WHEN (< crossovers least-crossovers)
			    (SETQ least-crossovers crossovers
				  best-insertion-point position)))))
	       ;;IBM RT change
	       (do ((scanner  circle (cdr scanner)))
		   (nil)
		 (when (member (cadr scanner) (vertex-connected-to non-leaf))
		   (try-insertion non-leaf scanner)
		   (try-insertion non-leaf (cdr scanner)))
	       	 (when (eq (cdr scanner) circle) (return)))
;; Old code		    
;	       (LOOP FOR scanner ON circle
;		     WHEN (MEMQ (CADR scanner) (vertex-connected-to non-leaf))
;		       DO
;			 ;;Calculate # of crossovers produced by inserting non-leaf before the
;			 ;;connecting non-leaf.
;			 (try-insertion non-leaf scanner)
;			 ;;Calculate # of crossovers produced by inserting non-leaf after the
;			 ;;connecting non-leaf.
;			 (try-insertion non-leaf (CDR scanner))
;		     UNTIL (EQ (CDR scanner) circle))
	       )
	     (IF best-insertion-point
		 ;;Perform the insertion.
		 (SETF (CDR best-insertion-point)
		       (CONS non-leaf (CDR best-insertion-point)))
		 ;;No "optimum" insertion point was found, since NON-LEAF isn't connected
		 ;;to any of the vertices in CIRCLE.  (This can happen whenever several
		 ;;unconnected components are arranged in the same circle.)  Add it onto
		 ;;the front of the list, arbitrarily.
		 (SETF (CDR circle) (CONS non-leaf (CDR circle))))))
	 ;;Remove the list's circularity.
	 (remove-list-circularity circle)
	 ;;If we've been stepping, signal the end.
	 (debug-print T "done.")
	 (WHEN (MEMQ :step-through-placement *graph-debug-actions*)
	   (FORMAT T "~%Finished stepping through placement algorithm.~%"))
	 circle)))))

(DEFMETHOD (graph :minimize-crossovers-for-large-graphs) (non-leaves)
 (beep) ;Still need to remove (or update) global vars controlling when this is called from
 ;;User interface.  Can optimize the PUSHNEW below using cons trick.  Also assess this
 ;;method--spacing goes light->heavy as you go clockwise around circle.
  (LET ((stepping? (MEMQ :step-through-placement *graph-debug-actions*)))
    (debug-print T "~%Minimizing crossovers for *huge* circular arrangement of vertices...")
    ;;Sort the non-leaves by increasing line degree. (? or should it be decreasing, or does it
    ;;not matter?)
;    (SETQ non-leaves
;	  ;;OR is used incase leaves occur in NON-LEAVES.
;	  (SORT non-leaves #'(LAMBDA (x y) (< (OR (line-degree x) 0)
;					      (OR (line-degree y) 0)))))
    (LET (circle)
      (LABELS
	((add-me-and-my-neighbors (vertex neighbor-depth)
	   (PUSHNEW vertex circle)
	   (IF (ZEROP neighbor-depth)
	       (WHEN stepping?
		 (SEND SELF :display-step circle
		       "~%Stepping through huge graph placement algorithm.   Added a set of neighbors."))
	       ;;Else only add the neighbors which occur in NON-LEAVES, otherwise you get
	       ;;real strange results (when graph is broken up into subgraphs for display).
	       ;;We do the test here, instead of a MEMQ, for efficiency.
	       (DOLIST (neighbor (vertex-connected-to vertex))
		 (LET (present?)
		   (SETQ non-leaves (DELETE vertex non-leaves
					    :TEST #'(LAMBDA (v1 v2)
						    (WHEN (EQ v1 v2)
						      (SETQ present? T)))
					    :COUNT 1)) ;;We know there are no repeats.
		   (WHEN present?
		     (add-me-and-my-neighbors neighbor (1- neighbor-depth))))))))
;; IBM RT Change
	(do () ((null non-leaves))
	  (add-me-and-my-neighbors (pop non-leaves) 4))
;	(LOOP WHILE non-leaves DO
;	  (add-me-and-my-neighbors (POP non-leaves) 4))
	)
      (WHEN stepping?
	(FORMAT T "~%Finished stepping through placement algorithm.~%"))
      circle)))

(DEFUN count-crossovers-after-insertion (non-leaf circle)
  "Returns the count of crossovers added if NON-LEAF were inserted into CIRCLE at the CDR.
NON-LEAF is a vertex struct.  CIRCLE is a circular list of vertex structs."
  (LET ((connected-to (vertex-connected-to non-leaf))
	(start (CDR circle)))
    ;;For each connection (comprised of either an in-link, out-link, or both),
    ;;add the number of crossovers of the given in or out link.
    ;; IBM RT change
    (do ((scanner start (cdr scanner))
	 (sum 0))
	(nil)
      (when (memq (car scanner) connected-to)
	(setq sum (+ sum (count-connection-crossovers start scanner))))
      (when (eq scanner circle) (return sum)))
;    (LOOP FOR scanner ON start
;	  WHEN (MEMQ (CAR scanner) connected-to)
;	  SUMMING (count-connection-crossovers start scanner)
;	  UNTIL (EQ scanner circle))

    ))

(DEFUN count-connection-crossovers (vertices to-vertices)
  "Returns the number of connections between two sides of a circular list of vertices.
Both args are assumed to be pointers into the same circular list.  One side consists of the
vertices from pointer VERTICES to the vertex before (CAR TO-VERTICES).  The other side
consists of the rest of the vertices, except for (CAR TO-VERTICES)."
  (LET ((from-vertex (CAR vertices))
	(to-vertex (CAR to-vertices))
	(other-side (CDR to-vertices)))
    ;;For each vertex on one side of the circle, count the number of vertices on the OTHER
    ;;side of the circle to which it is connected.
    ;;IBM RT change
    (do ((this-side-vertex (car vertices) (car vts))
	 (vts (cdr vertices) (cdr vts))
	 (osum 0))
	((eq this-side-vertex to-vertex) osum)
      (setq osum (+ osum (do ((other-side-vertex (car other-side) (car os))
			      (os (cdr other-side) (cdr os))
			      (isum 0))
			     ((eq other-side-vertex from-vertex) isum)
			   (if (memq other-side-vertex
					      (vertex-connected-to this-side-vertex))
			       (incf isum))))))
;    (LOOP FOR this-side-vertex IN vertices
;	  UNTIL (EQ this-side-vertex to-vertex)
;	  SUMMING (LOOP FOR other-side-vertex IN other-side
;			UNTIL (EQ other-side-vertex from-vertex)
;			WHEN (memq-circular other-side-vertex
;					    (vertex-connected-to this-side-vertex))
;			SUMMING 1))
    ))

(DEFMETHOD (graph :display-step) (non-leaves format-string &REST format-args)
  "Handles one step while stepping through the crossover minimization algorithm.
Formats FORMAT-STRING with FORMAT-ARGS, displays NON-LEAVES, then calls the command
loop recursively, providing a means to jump out of the recursive level."
  (WHEN (MEMQ :print-graph-computation-messages *graph-debug-actions*)
    (APPLY #'FORMAT T format-string format-args))
  ;;Bind this to NIL so that: 1. We don't step through the metagraph (eg. infinite loop)
  ;;2. The creation of the metagraph isn't debug-printed, since that distracts the user
  ;;from the stepping.
;  (LET ((*graph-debug-actions* NIL))
    ;;NIL#1 = no vertices to highlite.  NIL#2 = don't redraw previously displayed graph
    ;;(we want to display successive portions of the graph we're creating, without
    ;;interrutption.)
;    (SEND *display-io* :recursively-display-vertices non-leaves NIL NIL))
  )



(DEFUN remove-list-circularity (list)
  "Removes the circularity from LIST."
  (do ((vs list (cdr vs)))
      ((null vs))
    (when (eq (cdr vs) list)
      (setf (cdr vs) nil)
      (return list))))
;  (LOOP FOR vs ON list DO
;	(WHEN (EQ (CDR vs) list)
;	  (SETF (CDR vs) NIL)
;	  (RETURN list)))


(DEFUN copy-circular-list (list)
  "Copies LIST, which is a circular list."
  (LET ((copy (CONS (CAR list)
		    (do ((vs (cdr list) (cdr vs))
			 (result nil))
			((eq vs list) (nreverse result))
		      (push (car vs) result))
;		    (LOOP FOR vs ON (CDR list)
;			  UNTIL (EQ vs list)
;			  COLLECT (CAR vs))
		    )))
    (SETF (CDR (LAST copy)) copy)))

(DEFUN memq-circular (item circular-list)
  (IF (EQ item (CAR circular-list))
      circular-list
      (do ((scanner (cdr circular-list) (cdr scanner)))
	  ((or (null scanner) (eq scanner circular-list)))
	(when (eq item (car scanner))
	  (return scanner)))
;      (LOOP FOR scanner ON (CDR circular-list)
;	    UNTIL (EQ scanner circular-list)
;	    DO
;	    (WHEN (EQ item (CAR scanner))
;	      (RETURN scanner)))
      ))


;;==========================================================================================
;;
;;;; Assignment of world coordinates to given non-leaf vertices and their connected leaves.
;;
;;==========================================================================================

(DEFMETHOD (graph :arrange-vertices-in-circle) (non-leaves x-min y-min x-max y-max)
  "Assign world coordinates to the vertices in LEAVES and NON-LEAVES so as to arrange them in 
a circle within the rectangular area specified by X-MIN, X-MAX, Y-MIN, and Y-MAX.  NON-LEAVES 
is assumed to be ordered in such a way that edge crossovers are minimized.  Leaf nodes to 
vertices in NON-LEAVES are plotted as satellites to the circle.  NON-LEAVES placed clockwise, 
starting at the top.  (Other parts of the program assume this.)

This method assumes (currently) that the X-MIN, X-MAX, Y-MIN, and Y-MAX specify a square 
region: the algorithm will use the square within the rectangle in any event.  It would be 
neat to generate ovals, instead of circles, when a non-square region is specified.  Ovals 
retain some of the neat computational properties of circles, but take up less space."
  ;;This means that any algorythm that splits up the window by group must try to produce
  ;;square subdivisions.
  (WHEN non-leaves
    (LET* ((origin (CONS (ROUND (value-between x-min x-max))
			 (ROUND (value-between y-min y-max))))
	   ;;Set radius to the minimum dimension the area of the window allocated for this 
	   ;;group.
	   (radius (ROUND (MIN (- (CAR origin) x-min) (- (CDR origin) y-min))))
	   (placement-angle (/ *2PI* (LENGTH non-leaves)))
	   ;;Start first vertex at top of circle.  This assures left/right symmetry.
	   (current-angle *3/2PI*)
	   (room-for-leaves (* radius
			       *percentage-radius-for-leaves-in-circular-arrangement*)))
      ;;Leave room for the leaves of the graph.
      (DECF radius room-for-leaves)
      ;;Assign NON-LEAVES positions around the circle.
      (DOLIST (vertex non-leaves)
	(SETF (vertex-location vertex)
	      (find-point origin radius current-angle))
	;;Assign any LEAVES attached to VERTEX sattelite positions away from the circle.
	(LET ((number-of-leaves
	       (do ((vs (vertex-connected-to vertex) (cdr vs))
		    (sum 0))
		   ((null vs) sum)
		 (if (leaf? (car vs)) (incf sum)))
;	       (LOOP FOR v IN (vertex-connected-to vertex)
;				      WHEN (leaf? v)
;				      SUMMING 1)
	       ))
	  (UNLESS (ZEROP number-of-leaves)
	    (LET (;;The more the leaves, the closer together they'll be squeezed.  But if 
		  ;;there's only a few, we use a miminum angle.
		  (leaf-placement-angle (MIN *1/8PI* (/ PI number-of-leaves)))
		  ;;First leaf is placed in same direction from center of circle as VERTEX.
		  (leaf-current-angle current-angle)
		  (x 0))
	      ;;Stagger the leaves outward from angle CURRENT-ANGLE.
	      (dolist (v (vertex-connected-to vertex))
		(when (leaf? v)
		  (incf x)
		  (setf (vertex-location v)
			(find-point (vertex-location vertex)
				    room-for-leaves leaf-current-angle))
		  (if (oddp x)
		      (incf leaf-current-angle (* x leaf-placement-angle))
		      (decf leaf-current-angle (* x leaf-placement-angle)))))
;	      (LOOP FOR v IN (vertex-connected-to vertex)
;		    WHEN (leaf? v) DO
;		    (INCF x)
;		    (SETF (vertex-location v)
;			  (find-point (vertex-location vertex)
;				      room-for-leaves leaf-current-angle))
;		    (IF (ODDP x)
;			(INCF leaf-current-angle (* x leaf-placement-angle))
;			(DECF leaf-current-angle (* x leaf-placement-angle))))

	      )))
	;;Next vertex angle.
	(INCF current-angle placement-angle)))))



;;==========================================================================================
;;
;;;; Geometric transformations on graphs.
;;
;;==========================================================================================


#|
(DEFUN-METHOD scale-x graph (x)
  "Used to convert a flonum world X coordinate into a fixnum screen coordinate,
taking into account the current X scale factor (which can be modified through zooming)
and displacement (modified through pans).
This is implemented as a method-function, rather than a method, for efficiency."
  (ROUND (* (+ x x-displacement) x-scale-factor)))

(DEFUN-METHOD scale-y graph (y)
  "Used to convert a flonum world Y coordinate into a fixnum screen coordinate,
taking into account the current Y scale factor (which can be modified through zooming).
and displacement (modified through pans).
This is implemented as a method-function, rather than a method, for efficiency."
  (ROUND (* (+ y y-displacement) y-scale-factor)))
|#

(DEFMETHOD (graph :scale-x) (x)
  "Same as SCALE-X; for external use."
  (ROUND (* (+ x x-displacement) x-scale-factor)))

(DEFMETHOD (graph :scale-y) (y)
  "Same as SCALE-Y; for external use."
  (ROUND (* (+ y y-displacement) y-scale-factor)))


(DEFMETHOD (graph :inverse-scale-x) (x)
  "Reverse transformation of SCALE-X."
  (- (/ x x-scale-factor) x-displacement))

(DEFMETHOD (graph :inverse-scale-y) (y)
  "Reverse transformation of SCALE-Y."
  (- (/ y y-scale-factor) y-displacement))
#|
(DEFMETHOD (graph :scale-relative) (x-scale y-scale)
  "Performs a relative scaling operation on the coordinates of the graph's vertices.
If you send this message with arguments .5 and .5, the graph's x and y extents will 
be half what they were.  The graph is not redrawn."
   (let
      ;; IBM RT change
      ((x-origin (+ x-displacement
	            (* (display-pane-inside-width) .5)))
       (y-origin (+ y-displacement
		    (* (display-pane-inside-height) .5))))
;      ((x-origin (+ (SEND *display-io* :x-displacement)
;		     (* (SEND *display-io* :INSIDE-WIDTH) .5)))
;	(y-origin (+ (SEND *display-io* :y-displacement)
;		     (* (SEND *display-io* :INSIDE-HEIGHT) .5))))

    ;;Set the scale factor.  All primitives drawn reference these through SCALE-X and
    ;;SCALE-Y. 
    (SETQ x-scale-factor (* x-scale-factor x-scale)
	  y-scale-factor (* y-scale-factor y-scale))
    ;;Increment the displacement (which is used also for real pans) so as to do
    ;;the zoom relative to the center of the screen.
    (DECF x-displacement (/ (- (* x-origin x-scale) x-origin) x-scale-factor))
    (DECF y-displacement (/ (- (* y-origin y-scale) y-origin) y-scale-factor))))
|#
(DEFMETHOD (graph :scale-relative) (x-scale y-scale)
  "Performs a relative scaling operation on the coordinates of the graph's vertices.
If you send this message with arguments .5 and .5, the graph's x and y extents will 
be half what they were.  The graph is not redrawn."

    ;;Set the scale factor.  All primitives drawn reference these through SCALE-X and
    ;;SCALE-Y. 
    (SETQ x-scale-factor (* x-scale-factor x-scale)
	  y-scale-factor (* y-scale-factor y-scale)))


(DEFMETHOD (graph :pan) (x-delta y-delta)
  "Performs an absolute panning operation on the coordinates of the graph's vertices.
The two args are screen coordinate distances."
  (INCF x-displacement (/ x-delta x-scale-factor))
  (INCF y-displacement (/ y-delta y-scale-factor)))

(DEFMETHOD (graph :reset-position-and-scale) ()
  "Undo all zooms and pans."
  ;;Reset the displacement
  (SETQ x-displacement 0
	y-displacement 0)
  ;;Reset the scale factors.
  (SETQ x-scale-factor 1.0
	y-scale-factor 1.0)
  ;;Set original scale (which tries to fit the graph into the viewing window).
  (SEND SELF :scale-for-initial-viewing))

(DEFMETHOD (graph :scale-to-fit-window) (pixel-width pixel-height
					 &OPTIONAL allow-stretching?
					 (xborder 30)
					 (yborder 30))
  "Sets the graph's scale factor so as fit all vertices within a window the size of the 
first two arguments.  ALLOW-STRETCHING?, if NIL, specifies that the X and Y scale factor
are to be identical.  For example, if the vertices of the graph are to be displayed in a
circle, differing X and Y scale factors would cause the circle to appear as an ellipse.
XBORDER and YBORDER specify margins to leave between the vertices and the edge of the window."
  (LET* ((xmax (CAR (vertex-location (CAR vertices))))
	 (ymax (CDR (vertex-location (CAR vertices))))
	 (xmin xmax)
	 (ymin ymax))
    (DOLIST (vertex (CDR vertices))
      (LET ((location (vertex-location vertex)))
	(IF (> (CAR location) xmax)
	    (SETQ xmax (CAR location))
	    (IF (< (CAR location) xmin)
		(SETQ xmin (CAR location))))
	(IF (> (CDR location) ymax)
	    (SETQ ymax (CDR location))
	    (IF (< (CDR location) ymin)
		(SETQ ymin (CDR location))))))
    (LET ((width (- xmax xmin))
	  (height (- ymax ymin)))
      (UNLESS (ZEROP width)
	;;Get a float, rather than rational, so that later arithmetic is faster.
	(SETQ x-scale-factor (FLOAT (/ (- pixel-width xborder xborder) width))))
      (UNLESS (ZEROP height)	
	(SETQ y-scale-factor (FLOAT (/ (- pixel-height yborder yborder) height)))))
    (UNLESS allow-stretching?
      (SETQ x-scale-factor (MIN x-scale-factor y-scale-factor)
	    y-scale-factor x-scale-factor))
    ;;This computation's a bit tricky.  We want to displace the graph such that its
    ;;top level corner (xmin, ymin) in world coordinates is mapped to screen coordinate
    ;;(xborder, yborder). XMIN and YMIN are world-coordinate values.  XBORDER and YBORDER are
    ;;screen-coordinate lengths. By dividing them by the scale factors, we get two
    ;;world-coordinate lengths.  The subtraction operation can be thought of as
    ;;(+ (- xmin) {border length in world coordinates}).
    (SETQ x-displacement (- (/ xborder x-scale-factor) xmin)
	  y-displacement (- (/ yborder y-scale-factor) ymin))))

(DEFMETHOD (graph :vertex-has-moved) (vertex)
  "Recomputes (or causes recomputation of) all data which must be updated whenever a vertex
has been relocated.  For instance, the slopes of edges must be recomputed."
  ;;Could be made more efficient by doing only the work required by changes in VERTEX,
  ;;but I wanted to hack this fast.  These operations are very fast anyway.
  vertex
  ;;Recompute edge slopes
  (DOLIST (edge edges)
    (SETF (edge-slope edge) (angle-for-point (vertex-location (edge-vertex-2 edge))
					     (vertex-location (edge-vertex-1 edge)))))
  ;;Recompute the positions of directional arrows.  This computation depends on the above.
  (SEND SELF :plot-edges)
  (SETQ location-clumps (clump-vertices-and-edges-by-location vertices edges)))


;;==========================================================================================
;;
;;;; Output of plotted graphs.
;;
;;==========================================================================================


(DEFMETHOD (graph :draw) (&OPTIONAL (window *graph-output*))
  "Draws a graph on WINDOW.
Assumes that the vertices of the graph have already been assigned coordinates."
  (LET ((*graph-output* window))
    (SEND *graph-output* :CLEAR-WINDOW)
    ;;Store edge line slopes unless they were computed in a previous :DRAW.
    (WHEN (AND edges (NULL (edge-slope (CAR edges))))
      (DOLIST (edge edges)
	(SETF (edge-slope edge) (angle-for-point (vertex-location (edge-vertex-2 edge))
						 (vertex-location (edge-vertex-1 edge))))))
;    (SEND SELF :draw-view-boundaries window)
    ;;Draw all edges.
    (SEND SELF :draw-edges window)
    (SEND SELF :draw-vertices window)))

(DEFMETHOD (graph :draw-vertices) (*graph-output*)
  "Draw graph vertices."
 (MULTIPLE-VALUE-BIND (width height)
     (SEND *graph-output* :INSIDE-SIZE)
   (LET* ((vertex-print-string-function (get-compiled-function-or-die-trying
					  (SEND (SEND SELF :type)
						:vertex-print-string-function)))
	  (display-xmin (SEND SELF :inverse-scale-x 0))
	  (display-ymin (SEND SELF :inverse-scale-y 0))
	  (display-xmax (+ display-xmin (/ width x-scale-factor)))
	  (display-ymax (+ display-ymin (/ height y-scale-factor)))
	  (clipped-vertex-count 0))
    (DOLIST (clump location-clumps)
      (IF (overlapping-extents?
	    (CAAR clump) (CDAR clump) (CAADR clump) (CDADR clump)
	    display-xmin display-ymin display-xmax display-ymax)
	  (DOLIST (vertex (THIRD clump))
	    (LET* ((vertex-location (vertex-location vertex))
		   (scaled-x (send self :scale-x (CAR vertex-location)))
		   (scaled-y (send self :scale-y (CDR vertex-location))))
	      (MULTIPLE-VALUE-BIND (text font)
		  (FUNCALL vertex-print-string-function vertex)

		;; RT Change 
		(if (send (send self :type) :vertex-label-font) 
		    (setf font (send (send self :type) :vertex-label-font))
		    (send (send self :type) :set-vertex-label-font font))
 
		;;Draw a circle as the vertex
		(SEND *graph-output* :DRAW-CIRCLE scaled-x scaled-y 3)
		;;Label the vertex using its DATA.
		;;Y is adjusted to keep from writing over the circle.
		(LET ((adjusted-y (- scaled-y 6)))  ;#+Symbolics 6 #+Explorer 12

		  ;; RT Change
		  (let ((vertex1 vertex)) 
		    (make-active-region
		     (make-region :x (- scaled-x 13) :y  adjusted-y
				  :width (+ (string-width text font) 2)
				  :height (+ (font-height font) 2))
		     :bitmap (send *graph-output* :window)

		     :mouse-enter-region
		     #'(lambda (viewport active-region mouse-event x y)
			 (declare (ignore mouse-event x y))
			 (bitblt-region (viewport-bitmap viewport) active-region
					(viewport-bitmap viewport) active-region
					boole-c1))

		     :mouse-exit-region
		     #'(lambda (viewport active-region mouse-event x y)
			 (declare (ignore mouse-event x y))
			 (bitblt-region (viewport-bitmap viewport) active-region
					(viewport-bitmap viewport) active-region
					boole-c1))
				
		     :mouse-right-down
		     #'(lambda (viewport active-region mouse-event x y)
			 (declare (ignore viewport active-region mouse-event x y))
			 (when (send *zgraph-display-pane* :description-pane-active?)
			   (send (send self :type) :handle-selection-of-object vertex1)))

		     :mouse-left-down
		     #'(lambda (viewport active-region mouse-event x y)
			 (declare (ignore viewport active-region mouse-event x y))
			 (send *zgraph-display-pane* :move-graphics-object vertex1))))
		       
		  (SEND *graph-output* :DRAW-STRING
			text
			(- scaled-x 12) adjusted-y
			;;Towards the right
			scaled-x adjusted-y
			;;Unknown arg, text font, and alu function (SETA will overwrite any 
			;;lines going through text).
			NIL font ;TV:ALU-SETA
			boole-1) ;; RT CHANGE
		  ))))
	  ;;Else
	  (INCF clipped-vertex-count (LENGTH (THIRD clump)))))
    ;;Store for graph description.
    (SEND SELF :PUTPROP clipped-vertex-count :clipped-vertex-count))))

(DEFUN overlapping-extents? (xmin1 ymin1 xmax1 ymax1 xmin2 ymin2 xmax2 ymax2)
  "Returns non-NIL if the two rectangular areas described by the arguments overlap."
  (NOT (OR (< xmax1 xmin2)
	   (< ymax1 ymin2)
	   (< xmax2 xmin1)
	   (< ymax2 ymin1))))


(DEFUN value-between (value1 value2 &OPTIONAL (percentage .5))
  "Returns a number between VALUE1 and VALUE2, specifically
  (VALUE2 - VALUE1) * PERCENTAGE + VALUE1.  Without arg, returns the average."
  (+ (* (- value2 value1) percentage) value1))

(DEFCONSTANT *error-delta* (/ PI 50))

;;This method is still the bottleneck for drawing/redrawing graphs.
;;The three main operations:
;;draw-line, draw-filled-in-sector, and draw-text seem to contribute equally.  Luckily, 
;;scaling operations don't contribute to the slowness.  Probably only these ways of speeding
;;up redraw:
;; 1. Clip out edges which are completely outside the output window.  (Mainly would
;;help on humungous graphs.) 
;; 2. Give each graph its own screen array.  Panning/zooming still
;;slow, but switching between graphs would redisplay very fast.  Expensive memory-wise?
;; 3. Get rid of the directional arrows, but add an arrow character onto the end of the
;;labels.  Pretty sharp, right?  Would only get rid of about 1/3 of the delay.
;;
(DEFCONSTANT *pi+.3* (+ PI .3)) ;;Precomputed for efficiency.
(DEFCONSTANT *pi-.3* (- PI .3))

;;(DEFUN draw-pointer(window head-x head-y direction length alu)
;;  (SEND window :DRAW-FILLED-IN-SECTOR head-x head-y length
;;  (- *pi-.3* direction) (- *pi+.3* direction) alu))
 
;;; RT CHANGE
(DEFUN draw-pointer (window head-x head-y direction edge)
  (let ((idx (round direction *1/4PI*))
	(xmin head-x)
	(xmax head-x)
	(ymin head-y)
	(ymax head-y)
        (x 0)
        (y 0))
    (dotimes (k 2)
      (setf x (+ head-x (aref *arrow* idx k 0)))
      (setf y (+ head-y (aref *arrow* idx k 1)))
      (if (> x xmax)
	  (setf xmax x)
	  (if (< x xmin)
	      (setf xmin x)))
      (if (> y ymax)
	  (setf ymax y)
	  (if (< y ymin)
	      (setf ymin y)))
      (when (send *zgraph-display-pane* :description-pane-active?)     
       (make-active-region
        (make-region :x (- xmin 1)  :y (- ymin 1)
		     :corner-x (+ xmax 2)  :corner-y (+ ymax 2))  

        :bitmap (send *graph-output* :window)

	:mouse-enter-region
	#'(lambda (viewport active-region mouse-event x y)
	    (declare (ignore mouse-event x y))
	    (bitblt-region (viewport-bitmap viewport) active-region
	  	           (viewport-bitmap viewport) active-region
			   boole-c1))

        :mouse-exit-region
	#'(lambda (viewport active-region mouse-event x y)
	    (declare (ignore mouse-event x y))
	    (bitblt-region (viewport-bitmap viewport) active-region
	                   (viewport-bitmap viewport) active-region
		           boole-c1))
				
        :mouse-right-down
        #'(lambda (viewport active-region mouse-event x y)
            (declare (ignore viewport active-region mouse-event x y))
            (send (send (send *zgraph-display-pane* :graph) :type)
		  :handle-selection-of-object edge))))

       (SEND window :DRAW-LINE head-x head-y x y))))

;;Used by method :DRAW-EDGES below.
;;#+Symbolics
;;(DEFUN-METHOD dress-edges graph (some-edges window)
(DEFMETHOD (graph :dress-edges) (some-edges window)
  (DOLIST (edge some-edges)
    (LET* ((from-point (vertex-location (edge-vertex-1 edge)))
	   (to-point (vertex-location (edge-vertex-2 edge)))
	   (direction (edge-slope edge))
	   (from-x (send self :scale-x (CAR from-point)))
	   (from-y (send self :scale-y (CDR from-point)))
	   (to-x (send self :scale-x (CAR to-point)))
	   (to-y (send self :scale-y (CDR to-point)))
	   (edge-print-string-function (get-compiled-function-or-die-trying
					 (SEND (SEND SELF :type)
					       :edge-print-string-function))))
      ;;Label the directed edge if it is long enough to bother with it.
      ;;For efficiency, the edge length is approximated.
      (WHEN (OR (> (ABS (- from-x to-x))
		   *smallest-edge-length-to-bother-labelling*)
		(> (ABS (- from-y to-y))
		   *smallest-edge-length-to-bother-labelling*))
	;;Draw label near the head vertex of the edge.  The rationale is that if an
	;;edge goes from V1 to V2 and another goes from V1 to V1, the edge labels
	;;won't overlap (though the edges do, since they are straight lines).
	;;This also puts the label near the feathers of the directional arrow 
	;;representing the edge.
	(LET ((x-offset 0)
	      (y-offset 0)
	      ;;The .80 (80%) specifies where along the edge to put the label.  Could
	      ;;make a nice parameter, since on certain unlucky graphs, lots of labels
	      ;;will overwrite each other (but the user could change the parameter).
	      ;;As is, it puts the label behind the arrow feathers, which are at 90%.
	      (label-x (ROUND (value-between from-x to-x .80)))
	      (label-y (ROUND (value-between from-y to-y .80))))
	  ;;Remember, windows are upside-down.  All these offsets were generated by
	  ;;fiddling with the values till they looked right.  They are a function of
	  ;;weirdness in the placement algorithm used by the :DRAW-STRING method.
	  ;;Placement for (<= *3/2PI* slope *7/4PI*) and (<= *7/4PI* slope *2*) is 
	  ;;fine, so they don't occur in the COND.
	  (COND
	    ;;Check to see if slope is close to infinite, since we want to draw 
	    ;;vertical and nearly vertical lines' labels downwards.
	    ((< (ABS (- direction *1/2pi*)) *error-delta*)
	     ;;There's a bug in :DRAW-STRING where text drawn exactly upwards is
	     ;;positioned way above the FROM-X, TO-X location.  The following works 
	     ;;around that by offsetting FROM-X by one.
	     (SETQ from-x (1- from-x) from-y 0))
	    ((< (ABS (- direction *3/2pi*)) *error-delta*)
	     (SETQ x-offset 3))
	    ((<= 0 direction *1/4PI*)
	     (SETQ y-offset -2 x-offset 4))
	    ((<= *1/4PI* direction *1/2PI*)
	     (SETQ x-offset 8))
	    ((<= *1/2PI* direction *3/4PI*)
	     (SETQ x-offset 2 y-offset 4))
	    ((<= *3/4PI* direction PI)
	     (SETQ y-offset -1 x-offset -3))
	    ((<= PI direction *5/4PI*)
	     (SETQ x-offset 2))
	    ((<= *5/4PI* direction *3/2PI*)
	     (SETQ x-offset 2)))
	  (MULTIPLE-VALUE-BIND (text font)
	      (FUNCALL edge-print-string-function edge)

	    ;; RT Change 
	    (if (send (send self :type) :edge-label-font) 
		(setf font (send (send self :type) :edge-label-font))
		(send (send self :type) :set-edge-label-font font))
 
	    ;;Appropriate offsets are calculated, so draw it.
	    (SEND window :DRAW-STRING
		  ;;Use the function designated for our graph-type for getting edge
		  ;;print strings.
		  text
		  (+ x-offset label-x) (+ y-offset label-y)
		  ;;This looks wrong, using the edge head (instead of the tail) in 
		  ;;directing the string, but the purpose of this is to have the 
		  ;;string drawn backward from LABEL-X, LABEL-Y.  Thus, as the user 
		  ;;zooms out, the labels will grow towards the head of the edge, 
		  ;;instead of immediately running into the tail of the edge (which 
		  ;;LABEL-X, LABEL-Y is close to).
		  (+ x-offset from-x) (+ y-offset from-y)
		  ;;Unknown arg, text font, and alu function (SETA will overwrite any 
		  ;;lines going through text).
		  NIL font ;TV:ALU-SETA
		  boole-1)))))
    ;;Draw directional indicators, using a filled pie-shape as an arrow, 8 pixels 
    ;;long.  PI-.3 and PI+.3 are slight increments/decrements of the value PI.  We 
    ;;subtract DIRECTION from these for two reasons:
    ;;1. In order to use filled-sectors as direction indicators, the DIRECTION has to 
    ;;be reversed (add *PI+.3* or *PI-.3*).
    ;;2. Since :DRAW-FILLED-IN-SECTOR is foolishly inconsistent with the rest of the 
    ;;graphics primitives, RADIANS are oriented counter-clockwise instead of clockwise
    ;;(subtract from 2*PI). These two are performed by subtracting from *PI+.3* or
    ;;*PI-.3*.
;   (draw-pointer window
;		  (scale-x (CAR (edge-misc edge))) (scale-y (CDR (edge-misc edge)))
;		  (edge-slope edge) 8. TV:ALU-SETA)
;;; RT CHANGE    
     (draw-pointer window
		   (send self :scale-x (CAR (edge-misc edge))) (send self :scale-y (CDR (edge-misc edge)))
		   (edge-slope edge) edge) 

    ))
#|
#+Explorer
(DEFUN-METHOD dress-edges graph (some-edges window)
  (DOLIST (edge some-edges)
    (LET* ((from-point (vertex-location (edge-vertex-1 edge)))
	   (to-point (vertex-location (edge-vertex-2 edge)))
	   (direction (edge-slope edge))
	   (from-x (scale-x (CAR from-point)))
	   (from-y (scale-y (CDR from-point)))
	   (to-x (scale-x (CAR to-point)))
	   (to-y (scale-y (CDR to-point)))
	   (edge-print-string-function (get-compiled-function-or-die-trying
					 (SEND (SEND SELF :type)
					       :edge-print-string-function))))
      ;;Label the directed edge if it is long enough to bother with it.
      ;;For efficiency, the edge length is approximated.
      (WHEN (OR (> (ABS (- from-x to-x))
		   *smallest-edge-length-to-bother-labelling*)
		(> (ABS (- from-y to-y))
		   *smallest-edge-length-to-bother-labelling*))
	;;Draw label near the head vertex of the edge.  The rationale is that if an
	;;edge goes from V1 to V2 and another goes from V1 to V1, the edge labels
	;;won't overlap (though the edges do, since they are straight lines).
	;;This also puts the label near the feathers of the directional arrow 
	;;representing the edge.
	(LET ((x-offset 0)
	      (y-offset 0)
	      ;;The .80 (80%) specifies where along the edge to put the label.  Could
	      ;;make a nice parameter, since on certain unlucky graphs, lots of labels
	      ;;will overwrite each other (but the user could change the parameter).
	      ;;As is, it puts the label behind the arrow feathers, which are at 90%.
	      (label-x (ROUND (value-between from-x to-x .80)))
	      (label-y (ROUND (value-between from-y to-y .80)))
	      ;;My quicky implementation of :DRAW-STRING gives us complete control over the
	      ;;drawing angle, so it will mindlessly draw strings backwards/upside down given the
	      ;;right angle.  Thus, it has an option for drawing the characters in reverse order.
	      ;;This var controls its use.
	      backwards?)
	  ;;Remember, windows are upside-down.  All these offsets were generated by
	  ;;fiddling with the values till they looked right.  They are a function of
	  ;;weirdness in the placement algorithm used by the :DRAW-STRING method.
	  (COND
	    ;;Check to see if slope is close to infinite, since we want to draw 
	    ;;vertical and nearly vertical lines' labels downwards.
	    ((< (ABS (- direction *1/2pi*)) *error-delta*)
	     (SETQ x-offset 3 from-y 0 backwards? T))
	    ((< (ABS (- direction *3/2pi*)) *error-delta*)
	     (SETQ x-offset 3))
	    ((<= 0 direction *1/4PI*)
	     (SETQ y-offset -4 x-offset 4 backwards? T))
	    ((<= *1/4PI* direction *1/2PI*)
	     (SETQ x-offset 4 y-offset -4 backwards? T))
	    ((<= *1/2PI* direction *3/4PI*)
	     (SETQ x-offset 2 y-offset 4))
	    ((<= *3/4PI* direction PI)
	     (SETQ y-offset -5 x-offset -3))
	    ((<= PI direction *5/4PI*)
	     (SETQ x-offset 2 y-offset -4))
	    ((<= *5/4PI* direction *3/2PI*)
	     (SETQ x-offset 3 y-offset -3))
	    ((<= *3/2PI* direction *7/4PI*)
	     (SETQ x-offset -4 y-offset -4 backwards? T))	;???
	    ((<= *7/4PI* direction *2PI*)
	     (SETQ y-offset -4 x-offset -2 backwards? T)))
	  (MULTIPLE-VALUE-BIND (text font)
	      (FUNCALL edge-print-string-function edge)
	    ;;Appropriate offsets are calculated, so draw it.
	    (SEND window :DRAW-STRING
		  ;;Use the function designated for our graph-type for getting edge
		  ;;print strings.
		  text
		  (+ x-offset label-x) (+ y-offset label-y)
		  ;;This looks wrong, using the edge head (instead of the tail) in 
		  ;;directing the string, but the purpose of this is to have the 
		  ;;string drawn backward from LABEL-X, LABEL-Y.  Thus, as the user 
		  ;;zooms out, the labels will grow towards the head of the edge, 
		  ;;instead of immediately running into the tail of the edge (which 
		  ;;LABEL-X, LABEL-Y is close to).
		  (+ x-offset from-x) (+ y-offset from-y)
		  ;;Unknown arg, text font, and alu function (SETA will overwrite any 
		  ;;lines going through text).
		  NIL font TV:ALU-SETA backwards?)))))
    ;;Draw directional indicators, using a filled pie-shape as an arrow, 8 pixels 
    ;;long.  PI-.3 and PI+.3 are slight increments/decrements of the value PI.  We 
    ;;subtract DIRECTION from these for two reasons:
    ;;1. In order to use filled-sectors as direction indicators, the DIRECTION has to 
    ;;be reversed (add *PI+.3* or *PI-.3*).
    ;;2. Since :DRAW-FILLED-IN-SECTOR is foolishly inconsistent with the rest of the 
    ;;graphics primitives, RADIANS are oriented counter-clockwise instead of clockwise
    ;;(subtract from 2*PI). These two are performed by subtracting from *PI+.3* or
    ;;*PI-.3*.
    (draw-pointer window
		  (scale-x (CAR (edge-misc edge))) (scale-y (CDR (edge-misc edge)))
		  (edge-slope edge) 8. TV:ALU-SETA)))
|#
(DEFMETHOD (graph :draw-edges) (*graph-output*)
  ;;First draw the edge line.  They go first so that labels will overwrite them.
  (DOLIST (edge edges)
    (LET ((from-point (vertex-location (edge-vertex-1 edge)))
	  (to-point (vertex-location (edge-vertex-2 edge))))
      (SEND *graph-output* :DRAW-LINE (send self :scale-x (CAR from-point))
	    (send self :scale-y (CDR from-point)) (send self :scale-x (CAR to-point))
	    (send self :scale-y (CDR to-point)))))
  ;;Second, draw the arrow direction indicators and labels.  We have the arrow coordinates
  ;;stored in clumps for fast mouse sensitivity, so use the clumps for faster display.
  ;;(Indicators which are outside of *GRAPH-OUTPUT* are clipped here.)  We also use these
  ;;coordinates to clip labels, even though the labels don't have a single point location.
  ;;(They are also offset from the coordinates).  This is justified by gained speed and it
  ;;doesn't result in any gross inaccuracies.
  (MULTIPLE-VALUE-BIND (width height)
      (SEND *graph-output* :INSIDE-SIZE)
    (LET* ((display-xmin (SEND SELF :inverse-scale-x 0))
	   (display-ymin (SEND SELF :inverse-scale-y 0))
	   (display-xmax (+ display-xmin (/ width x-scale-factor)))
	   (display-ymax (+ display-ymin (/ height y-scale-factor))))
      (DOLIST (clump location-clumps)
	(WHEN (overlapping-extents?
		(CAAR clump) (CDAR clump) (CAADR clump) (CDADR clump)
		display-xmin display-ymin display-xmax display-ymax)
	  (send self :dress-edges (FOURTH clump) *graph-output*))))))
#|
(DEFMETHOD (graph :draw-view-boundaries) (window)
  ;;Draw dashed line just inside WINDOW borders (unless feature is disabled.)
  ;;When the user fast-pans to the boundary of the hidden bit array,
  ;;the dashed line will provide a hint that further fast panning will cause
  ;;a real pan.  
  (WHEN *dashed-line-margin*
    (MULTIPLE-VALUE-BIND (low-x low-y high-x high-y)
	(SEND window :INSIDE-EDGES)
      ;;This var provides space between the actual boundary and the displayed boundary.
      ;;Without this space, the user would have no warning.
      (INCF low-x *dashed-line-margin*)
      (INCF low-y *dashed-line-margin*)
      (DECF high-x *dashed-line-margin*)
      (DECF high-y *dashed-line-margin*)
      (SEND window :DRAW-DASHED-LINE low-x low-y high-x low-y)
      (SEND window :DRAW-DASHED-LINE high-x low-y high-x high-y)
      (SEND window :DRAW-DASHED-LINE high-x high-y low-x high-y)
      (SEND window :DRAW-DASHED-LINE low-x high-y low-x low-y))))
|#
(COMPILE-FLAVOR-METHODS property-list-mixin graph)
  
;;==========================================================================================
;;
;;;; Code for finding the bi-connected components of a graph.
;;   This implements an efficient (order = number of vertices) algorithm described in
;;   Aho, Hopcroft, and Ullman (The Design and Analysis of Computer Algorithms).
;;
;;   Borrowed from Brian Falkenheimer and rehacked a bit for the Graph Displayer.
;;
;;==========================================================================================

(DEFVAR *stack* nil)
(DEFVAR *count* nil)

(DEFUN extract-cycles (vertices)
  "Search through the relational graph formed by the adjacency 
lists (vertex-connected-to vertex vertices)
and returns a list of the bi-connected-components of the graph as the first value
and a list of edges connecting them as the second value.  The first value is a list of lists
of vertices, one for each connected component.  The second value is a list of 
conses (vertex . vertex) refering to edges."
  (LET (*stack* *count* bi-con-components connecting-edges)
    ;;Use a copy, since NCONC mungs it below.
    (SETQ vertices (COPY-LIST vertices))
    ;;Initialize temporary storage for this algorithm's use.
    (DOLIST (vertex vertices)
      (SETF (vertex-misc vertex) (LIST NIL NIL NIL NIL)))
    
    (LOOP
          (setq *stack* NIL)
	  (setq *count* 1)
	  ;;CAR is just a random pick.
	 
;    (LOOP FOR *stack* = NIL
;	  FOR *count* = 1
;	  ;;CAR is just a random pick.
;	  DO
	  (SETQ bi-con-components (NCONC (searchb (CAR vertices) vertices NIL)
					 bi-con-components))
	  ;;Remove used vertices. DELETE for disconnected graphs.
	  (SETQ vertices (DELETE-IF #'(LAMBDA (v) (vertex-mark v)) vertices))
	  (UNLESS vertices
	    (RETURN NIL)))
    ;;Deallocate temporary storage.
    (DOLIST (vertex vertices)
      (SETF (vertex-misc vertex) NIL))
    (VALUES
     (do ((component (car bi-con-components) (car bcc))
	  (bcc (cdr bi-con-components) (cdr bcc))
	  (result nil))
	 ((null component) result)
       (cond ((cdr component)
	      (push (delete-duplicates (do ((edges (car component) (car ct))
					    (ct (cdr component) (cdr ct))
					    (iresult nil))
					   ((null edges) (nreverse iresult))
					 (push (car edges) iresult)
					 (push (cdr edges) iresult)))
		    result))
	     (t (push (car component) connecting-edges))))
; (LOOP FOR component IN bi-con-components
;       IF (CDR component)
;       COLLECT (DELETE-DUPLICATES (LOOP FOR edges IN component
;					COLLECT (CAR edges) COLLECT (CDR edges)))
;       ELSE DO (PUSH (CAR component) connecting-edges))
	    connecting-edges)))


(DEFUN searchb (v vertices bi-components)
  "Aho & Ullman's algorythm: depth-first search of the connected component of the graph 
which includes the given vertex V.  Returns the bi-connected-components of this portion of
the graph. The order of algorithm is the number of edges in the graph."
  (LET (cur-edge)
    (SETF (vertex-mark v) T
	  (vertex-dfnumber v) *count*
	  (vertex-low v)  *count*)
    (INCF *count*)
    (DOLIST (w (vertex-connected-to v))
      (SETQ cur-edge (CONS v w))
      (UNLESS (edge-in? cur-edge)
	(PUSH cur-edge *stack*))
      (COND ((NULL (vertex-mark w))
	     (SETF (vertex-father w)  v
		   bi-components (Searchb w vertices bi-components))
	     (IF (>= (vertex-low w)
		     (vertex-dfnumber v))
		 (PUSH (remove-bicomp cur-edge) bi-components))
	     (SETF (vertex-low v)
		   (MIN (vertex-low v)
			(vertex-low w))))
	    ((NULL (EQUAL w (vertex-father v)))
	     (SETF (vertex-low v)
		   (MIN (vertex-low v)
			(vertex-dfnumber w))))))
    bi-components))

(DEFUN edge-in? (edge)
  "Returns non-NIL if the edge already been inserted into the stack."
  (LET* ((vertex1 (CAR edge))
	 (vertex2 (CDR edge))
	 (dfnum1 (vertex-dfnumber vertex1))
	 (dfnum2 (vertex-dfnumber vertex2)))
    (AND (vertex-mark vertex2)
	 (OR (< dfnum1 dfnum2)
	     (AND (> dfnum1 dfnum2)
		  (EQ vertex2
		      (vertex-father vertex1)))))))

(DEFUN remove-bicomp (edge)
  "Remove the newly found bi-connected-component from the stack by
popping the stack down to the given edge."
  (LET (bi-comp stack-edge)
    (loop	
      (setq stack-edge (pop *stack*))
      (push stack-edge bi-comp)
      (when (or (null *stack*) (equal stack-edge edge))
	(return)))
;     (LET (bi-comp)
;    (LOOP FOR stack-edge = (POP *stack*)
;	  DO (PUSH stack-edge bi-comp)
;	  UNTIL (OR (NULL *stack*) (EQUAL stack-edge edge)))


    bi-comp))


;;
