;;; -*- Mode: LISP  -*-

;;Copyright (c) 1986 by John C. Hogge, The University of Illinois.
;;
;;File defs.lisp of system Zgraph.
;;
;; Modified for IBM RT running Lucid by H. Kim and Ken Forbus, University of Illinois
;;

(in-package 'user)

;;==========================================================================================
;;
;;;; Some graphics support code
;;
;;==========================================================================================

;;
;;Constants are used for efficiency:
;;
(DEFCONSTANT *1/8PI* (* PI 1/8))
(DEFCONSTANT *1/4PI* (* PI 1/4))
(DEFCONSTANT *1/2PI* (* PI 1/2))
(DEFCONSTANT *3/4PI* (* PI 3/4))
(DEFCONSTANT *5/4PI* (* PI 5/4))
(DEFCONSTANT *3/2PI* (* PI 3/2))
(DEFCONSTANT *7/4PI* (* PI 7/4))
(DEFCONSTANT *2PI* (* PI 2))

(DEFCONSTANT *arrow* (make-array '(9 2 2) :element-type 'integer :initial-contents
                                 '(((-5 -5) (-5 5))
                                   ((0 -7)  (-7 0))
				   ((5 -5)  (-5 -5))
				   ((0 -7)  (7  0))
				   ((5  5)  (5 -5))
				   ((0  7)  (7  0))
				   ((-5 5)  (5  5))
				   ((-7 0)  (0  7))
                                   ((-5 -5) (-5 5))))) 

(DEFUN angle-for-point (point &OPTIONAL (origin '(0 . 0)))
  "POINT is a cons (x . y).  Angle returned in radians, from ORIGIN to POINT."
  (LET* ((x (- (CAR point) (CAR origin)))
	 (y (- (CDR point) (CDR origin)))
	 (arctan (IF (ZEROP x)
		   *1/2PI*
		   (ATAN (/ (ABS y) (ABS x))))))
    (IF (MINUSP x)
	(IF (MINUSP y)
	    (+ PI arctan)  ;third quadrant
	    (- PI arctan)) ;second quadrant
	(IF (MINUSP y)
	    (- *2PI* arctan) ;fourth quadrant
	    arctan))))

(DEFUN find-point (from-point distance angle)
  "Locates a coordinate at a DISTANCE and ANGLE from FROM-POINT.  ANGLE is in radians.
Distance is in pixels.  Returns the located point: (x . y)."
  (CONS (+ (ROUND (* (COS angle) distance)) (CAR from-point))
	(+ (ROUND (* (SIN angle) distance)) (CDR from-point))))

(DEFUN multiple-of (x mult)
  (LET ((mod (mod x mult)))
    (IF (ZEROP mod)
	x
	(+ x (- mult mod)))))

;; All of the meta-window definition has been diked out.  Lucid's windows don't have
;; all the functionality, but are close enough for jazz.

;;==========================================================================================
;;
;;;; Global variables and constants.
;;
;;  Most of these are used by both the graph plotting code
;;  and the user interface code, so they appear here.
;;
;;==========================================================================================

(DEFVAR *graph-types* NIL
  "List of the types of graphs the Graph Displayer can display.
List entries are GRAPH-TYPE structs.")

(DEFVAR *graph-output* NIL  "The window to draw graphs on.")

(DEFVAR *menu-pane* NIL  "The window for menu pane.")

(DEFVAR *status-pane* NIL  "The window for status pane.")

(defvar *description-pane* nil)

(defvar *graph-status-pane* nil)

(defvar *documentation-pane* nil)

(defvar *dialog-window*)

(DEFVAR *display-io* NIL
  "The meta-window which displays what is drawn on *GRAPH-OUTPUT*.
The scaling code uses this to make sure that *GRAPH-OUTPUT* isn't shrunk smaller than the
meta window.")

(DEFVAR *description-output* NIL
  "The scroll window (if present) which should be used to display a description of
a selected vertex or edge.  If NIL, no such scroll window is available.")

(DEFVAR *graph-debug-actions* '(:print-graph-computation-messages)
  "List of extra things to do while computing a graph.  Refer to *ALL-GRAPH-DEBUG-ACTIONS*
for possibilities.")

(DEFVAR *all-graph-debug-actions*
	'(("Graph Computation Messages" . :print-graph-computation-messages)
	  ("Step Thru Placement" . :step-through-placement))
  "Possible elements *GRAPH-DEBUG-ACTIONS*. These are actions to do while computing graphs.
:PRINT-GRAPH-COMPUTATION-MESSAGES = print which vertices are reached from each vertex.
:STEP-THROUGH-PLACEMENT = display the graph at each stage of the placement algorythm.")

(DEFMACRO debug-print (destination control-string &REST args)
  `(WHEN (MEMQ :print-graph-computation-messages *graph-debug-actions*)
    (format ,destination ,control-string . ,args)))


(DEFVAR *all-graph-plotting-styles*
	'(("One Big Circle"
	   . :plot-in-one-big-circle)
	  ("Circles For Each Bi-Connected Component"
	   . :plot-in-circles-for-bi-connected-components)
	  ("Plot as a Lattice (use only for lattices!!)"
	   . :plot-lattice))
  "Alist of methods for plotting graphs in world coordinates.")

(DEFVAR *graph-plotting-style* :plot-in-circles-for-bi-connected-components
  "Current method to use of those in *all-graph-plotting-styles* in plotting graphs.")

(DEFVAR *default-graph-plotting-style* :plot-in-circles-for-bi-connected-components)

(DEFVAR *smallest-edge-length-to-bother-labelling* 100.
  "Any edges with length less than this won't be labelled.  This reduces clutter in the
graph display.")


(DEFVAR *too-many-non-leaves* 80.
  "If the graph has more leaves than this, don't try to minimize crossovers.")

(DEFVAR *too-many-edges* 400.
  "If the graph has more edges than this, don't try to minimize crossovers.")


(DEFVAR *percentage-radius-for-leaves-in-circular-arrangement* .30
  "Given a circle of radius R in which to place a given set of non-leaf
vertices and the leaves to which they are connected, this var determines
the radius of the inner circle on which to place the non-leaves.  The
leaves are placed outside of this inner circle.  INNER-CIRCLE-RADIUS =
OUTER * this variable")

(DEFVAR *gd* NIL "Bound in Graph Displayer command loop to the Graph Display Frame.")

(DEFVAR *default-pane-configuration* 'main
  "Specifies the configuration of window panes to use when instantiating ZGRAPH-DISPLAY-FRAME.
Can be either ZG:MAIN or ZG:TRIMMED.  The former is the default.  If you prefer TRIMMED,
feel free to SETQ this in your init file.")

(DEFVAR *number-vertices-over-which-graphs-are-fit-onto-hidden-bit-array* 18
  "If a graph has more than this number of vertices, it is fit onto the hidden bit array
when first displayed.  Otherwise, it is fit onto the display pane.")

(DEFVAR *dashed-line-margin* 10
  "Gap in pixels between inside edge of *GRAPH-OUTPUT* and the dashed line drawn to indicate
the viewing boundary.  If NIL, the dashed line isn't drawn at all.")


;;==========================================================================================
;;
;;;; Organization of vertices of a graph on a cartegion plane.
;;
;;==========================================================================================


(DEFSTRUCT (vertex
	     #+Explorer
	     (:PRINT-FUNCTION 
	       (LAMBDA (v stream IGNORE)
		 (FORMAT stream "#<VERTEX ~a>" (vertex-data v)))))
 "Holds a graph vertex.
  Slot LOCATION is a cons: ({x coordinate} . {y coordinate})
  Slot EDGES is a list of EDGE structs of edges incident from this vertex.
  Slot CONNECTED-TO is a list of vertices we are connected to via In and Out links (edges).
  DATA is LISP object associated with the vertex.
  MISC is a slot used internally for performing searches on the graph.  Currently it is used
only for the finding of bi-connected components."
  location
  edges
  connected-to
  data
  misc)

(DEFSTRUCT (edge
	     #+Explorer
	     (:PRINT-FUNCTION
	       (LAMBDA (e stream IGNORE)
		 (FORMAT stream "#<EDGE ~a->~a by ~a>"
			 (vertex-data (edge-vertex-1 e))
			 (vertex-data (edge-vertex-2 e))
			 (IF (CDR (vertex-data e)) (vertex-data e) (CAR (vertex-data e)))))))
  "Holds a graph edge.
  Slots VERTEX-1 and VERTEX-2 hold the endpoint vertices of this edge.
  SLOPE is used internally as the angle (in radians) of the vector from VERTEX-1 to VERTEX-2.
  DATA is a list of LISP objects associated with the edge, usually labels expressing any
relationships between the LISP objects (data) of the vertices the edge connects.  It is
important to note that this is a list of labels.  Zgraph combines any edges leading from one
vertex to another and displays a composite label.  In this way, the edge labels do not 
overwrite each other.  (It also makes edge-crossover minimization more accurate.)
  MISC is a slot used internally for storing label locations for doing mouse-sensitivity."
  vertex-1
  vertex-2
  slope
  data
  misc)

#|
;;Accessor macros which the following algorithm uses to store data in the MISC slot.
(DEFSUBST vertex-mark (vertex)
  (CAR (vertex-misc vertex)))

(DEFSUBST vertex-dfnumber (vertex)
  (CADR (vertex-misc vertex)))

(DEFSUBST vertex-low (vertex)
  (CADDR (vertex-misc vertex)))

(DEFSUBST vertex-father (vertex)
  (CADDDR (vertex-misc vertex)))

|#
(DEFMACRO vertex-mark (vertex)
  `(CAR (vertex-misc ,vertex)))

(DEFMACRO vertex-dfnumber (vertex)
  `(CADR (vertex-misc ,vertex)))

(DEFMACRO vertex-low (vertex)
  `(CADDR (vertex-misc ,vertex)))

(DEFMACRO vertex-father (vertex)
  `(CADDDR (vertex-misc ,vertex)))

(DEFUN get-compiled-function-or-die-trying (object)
  "OBJECT is either a compiled function or a symbol with a function definition.
If the latter and the definition is compiled, it is returned.  If it isn't compiled,
we compile it and return."
  (IF (TYPEP object 'COMPILED-FUNCTION)
      object
      (LET ((cf? (SYMBOL-FUNCTION object)))
	(cond ((TYPEP cf? 'COMPILED-FUNCTION) cf?)
	      (t ;else
		 (COMPILE object)
		 (SYMBOL-FUNCTION object))))))

(defun beep ())

(defun circular-list (&rest args)
  (setq args (copy-list args))
  (rplacd (last args) args)
  args)
 
(defun delq (item list &optional (n -1))
  (cond ((or (atom list) (zerop n)) list)
	((eq item (car list))
	 (delq item (cdr list) (1- n)))
	(t (rplacd list (delq item (cdr list) n)))))
