;;; -*- Mode: LISP; Syntax: Common-lisp; Base: 10; Package: ZG; Default-character-style: (:FIX :ROMAN :NORMAL) -*-

D,#TD1PsT[Begin using 006 escapes](1 0 (NIL 0) (NIL :BOLD-EXTENDED NIL) "CPTFONTB");;Copyright (c) 1986 by John C. Hogge, The University of Illinois.
;;
;;File GRAPH-TYPE.LISP of system Zgraph.

(2 0 (NIL 0) (NIL NIL NIL) "CPTFONT")(DEFFLAVOR graph-type
	((name (INTERN (FORMAT NIL "GRAPH-TYPE-~a" (LENGTH *graph-types*))
		       *package*))
	 (traversal-function NIL)
	 (default-root-finding-form NIL)
	 (instantiation-function 'default-instantiation-function)
	 (traverse-recursively? T)
	 (vertex-print-string-function 'default-vertex-print-string-function)
	 (edge-print-string-function 'default-edge-print-string-function)
	 (vertex/edge-description-function 'default-vertex/edge-description-function)
	 (vertex/edge-selection-handler 'default-vertex/edge-selection-handler)
	 (default-name-count 0))
	()
  :SETTABLE-INSTANCE-VARIABLES
1  ;;For use with TV:CHOOSE-VARIABLE-VALUES.
2  :SPECIAL-INSTANCE-VARIABLES
  :special-instance-variable-binding-methods
  (:DOCUMENTATION :COMBINATION
   "1Holds user-defined graph types.  Instances of this flavor are stored in instances
of flavor GRAPH as their defined graph type.  Instance variables:

NAME -- unique identifier

TRAVERSAL-FUNCTION -- function used to build graphs of this type.
  It is mapped onto a list of root vertices stored in DEFAULT-ROOT-FINDING-FORM2 1and
  should return an alist of entries of the form (TAIL-VERTEX . EDGE-LABEL), representing
  an outgoing edge from the vertex passed as arg.

DEFAULT-ROOT-FINDING-FORM -- form which is evaluated to obtain a list of root vertices to 
2  1pass to the TRAVERSAL-FUNCTION.  If NIL, it is assumed that each instance of flavor GRAPH
  will have their own set of roots.

INSTANTIATION-FUNCTION2 -- function called by the Zgraph user interface to get an instance of
  this graph type (really a flavor instance of ZG:GRAPH whose type is this ZG:GRAPH-TYPE).  
  The function takes one argument: the GRAPH-TYPE flavor instance itself.  The default
  ZG:1DEFAULT-INSTANTIATION-FUNCTION2, requests an identifying name from the user and allows him
  to edit DEFAULT-ROOT-FINDING-FORM.  An application might want to have their own
  instantiation function do such things as 1. not bother the user for the above info  2. set
  switches which control the operation of the DEFAULT-ROOT-FINDING-FORM, TRAVERSAL-FUNCTION,
  print string functions, etc.

1TRAVERSE-RECURSIVELY? -- flag indicating how the TRAVERSAL-FUNCTION is to be applied.  If
  NIL, the traversal function is applied only to each root vertex.  These roots are either
  provided by DEFAULT-ROOT-FINDING-FORM or by the user.  If the flag is non-NIL (the default),
  TRAVERSAL-FUNCTION is also applied recursively, in depth-first fashion, to any tail vertices
  found through previous applications.  For instance, application on V1 yields edges to
  V2 and V3, the traversal function will also be applied to V2 and V3.

VERTEX-PRINT-STRING-FUNCTION --  function which returns a string to output for labelling
  its one argument, a VERTEX struct.2  1The string is the first value returned, and font to
  display it in is the second value.  Default is ZG:DEFAULT-VERTEX-PRINT-STRING-FUNCTION.

EDGE-PRINT-STRING-FUNCTION --  function which returns a string to output for labelling
  its one argument, an EDGE struct.  The string is the first value returned, and font to
  display it in is the second value.  Default is ZG:DEFAULT-EDGE-PRINT-STRING-FUNCTION.

VERTEX/EDGE-DESCRIPTION-FUNCTION -- function which is called only in the Zgraph user 
2  1interface, when a user selects a vertex or edge to be described.  It should take a vertex 
2  1or edge2 1struct as first arg and a scroll window instance as second arg.  It is expected to
2 1 behave2 1pretty much like the default function (DEFAULT-VERTEX/EDGE-DESCRIPTION-FUNCTION) 
2  1does,2 1setting2 1the scroll window's scroll items (via message :SET-ITEMS) to a list of 
2  1one-line2 1strings2 1which, together, describe the vertex or edge.  The scroll window is very
2  1basic and2 1expects2 1the argument to :SET-ITEMS to be a list of scroll items of the form 
  ({FORMAT STRING} . {FORMAT ARGS}).  To display an item the scroll window simply applies
  FORMAT as follows:  (APPLY #'FORMAT {window} {item}).
  Pattern your function after DEFAULT-VERTEX/EDGE-SELECTION-HANDLER, modifying it to
  acommodate your graph type's vertices and edges.  The default function simply uses DESCRIBE
  to obtain a description of the data fields of vertices and edges.

VERTEX/EDGE-SELECTION-HANDLER -- function which handles selection of vertices and edges in
  displayed graphs.  (ZG:GRAPH-DISPLAY-PANE is set up to select vertices and edges when the
  user clicks MOUSE-M near them.)  The function takes two arguments: the VERTEX or EDGE struct
  selected by the user and the GRAPH flavor instance currently being displayed.  The default
  function, ZG:DEFAULT-VERTEX/EDGE-SELECTION-HANDLER, is set up to provide options for 
  inspecting the vertex or edge struct or its data field, setting symbol ZG:V to the data,
  and inspecting the results of applying the GRAPH's traversal function on the data.

DEFAULT-NAME-COUNT -- internal kruft used in :GENNAME.2"))


(DEFMETHOD (:INIT graph-type :AFTER) (IGNORE)
  "1Add SELF to the global list of graph types.2"
  (PUSH SELF *graph-types*))

(DEFMETHOD (:genname graph-type) ()
  "1Returns a string to use for the name of a graph of our type.
This is used as a default when instances of GRAPH are instantiated without a specified name.2"
  (FORMAT NIL "~a #~s" name (INCF default-name-count)))


(DEFUN default-vertex-print-string-function (vertex-struct)
  "1Default function for returning a string to output for labelling VERTEX structures.
The string is the first value returned, and font to display it in is the second value.
This default is set up as follows:
1. The VERTEX-DATA slot holds some LISP object found to be a vertex through
   Recursive applications of the user's graph traversal function.
2. This default function simply obtains a string from this object via the ~S format directive.
   You may wish to provide your own way of having vertices printed.
   For instance, if you don't want package names printed, bind *PACKAGE* to NIL
around the call to FORMAT.
3. Print all vertices using FONTS:TR8B.  You might want to instead use different fonts for
different types of vertices.2"
  (VALUES
    (FORMAT NIL "~s" (vertex-data vertex-struct))
;;;Changed to character style for 7.1 28.12.87 TYG
;;;    FONTS:TR8B))
    '(:Fix :bold :small)))

(DEFUN default-edge-print-string-function (edge-struct)
  "1Default function for returning a string to output for labelling EDGE structures.
The string is the first value returned, and font to display it in is the second value.
This default is set up as follows:
1. The EDGE-DATA slot holds a list of edge labels, usually of length one.  These labels
   are exactly what is returned in the CDRs of the alists returned by recursive applications
   of the user's graph traversal function.  If EDGE-DATA is of length greater than one,
   that means the traversal function dug up two different relationships going from the
   head vertex to the tail vertex of this edge.  If you design your own edge print function,
   you have to deal with EDGE-DATA much the way this function does.  However, you can
   do what you want on #2:
2. Zgraph takes the convention that if an element in EDGE-DATA is a string, it is to be used 
   as its own print string.  (In otherwords, it is output without double quotes.)  If it is
   NIL, the edge is not labelled.  If it is anything else, a string is obtained from it using
   the ~S format directive.  You may wish to set up your own convention.
3. Print all edges using FONTS:5x5.  You might want to instead use different fonts for
different types of edges.2"
  (LET ((edge-data (edge-data edge-struct)))
    (VALUES
      1;;If there is more than one relationship between the endpoint vertices,
2      1;;append them together and display.  Otherwise just display the one label.
2      (IF (CDR edge-data)
	  (WITH-OUTPUT-TO-STRING (stream)
	    (IF (SI:STRINGP (CAR edge-data))
		(FORMAT stream (CAR edge-data))
		(FORMAT stream "~S" (CAR edge-data)))
	    (DOLIST (label (CDR edge-data))
	      (IF (SI:STRINGP label)
		  (FORMAT stream " & ~A" label)
		  (WHEN label
		    (FORMAT stream " & ~S" label)))))
	  (IF (SI:STRINGP (CAR edge-data))
	      (CAR edge-data)
	      (WHEN (CAR edge-data)
		(FORMAT NIL "~S" (CAR edge-data)))))
;;;changed to character style for 7.1 conversion 28.12.87 TYG
;;;      FONTS:5X5)))
      '(:fix :roman :normal))))
(DEFUN default-vertex/edge-description-function (vertex-or-edge scroll-window)
  (LET* (1;;Get a description string.
2	 (description
	   (WITH-OUTPUT-TO-STRING (stream)
	     (LET ((*standard-output* stream))
	       (DESCRIBE (IF (TYPEP vertex-or-edge 'vertex)
			     (vertex-data vertex-or-edge)
			     (edge-data vertex-or-edge))
			 T))))
	 1;;Break up the string into a list of lines of the description: ("line 1" "line 2"...)
2	 (description-lines
	   (WITH-INPUT-FROM-STRING (stream description)
	     (LOOP FOR line = (READ-LINE stream NIL)
		   WHILE line
		   COLLECT line))))
    (LET ((separate-at (LENGTH (SEND scroll-window :ITEMS))))
      1;;Separate this description from previous ones.
2      (SEND scroll-window :APPEND-ITEM '("------------------------"))
      1;;Add in the lines of description.
      ;;The scroll window items are entries of the form ({format string} . {format args}).
      ;;We simply use "~a" to print out a description line.
2      (DOLIST (line description-lines)
	(SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
      1;;Scroll to top of this description.
2      (SEND scroll-window :scroll-absolute separate-at))))


(DEFMETHOD (:handle-selection-of-object graph-type) (vertex-or-edge-struct graph-display-pane)
  "1Handles mouse clicks on vertices and edges2 1of graphs of our type.
All this method does is to call whatever handler is stored in VERTEX/EDGE-SELECTION-HANDLER.2"
  1;;Pass the object the user clicked on and the display-pane, which the handler may need.
2  (FUNCALL vertex/edge-selection-handler vertex-or-edge-struct graph-display-pane))
	   

(DEFUN default-vertex/edge-selection-handler (vertex-or-edge-struct graph-display-pane)
  "1Pops up a menu of options whenever a vertex or edge is selected with the mouse.2"
  (DECLARE (SPECIAL v))
  (LET* ((graph (SEND graph-display-pane :graph))
	 (graph-type (SEND graph :type)))
    (COND
      ((TYPEP vertex-or-edge-struct 'vertex)
       (LET ((data (vertex-data vertex-or-edge-struct))
	     (options '(("Inspect Vertex Data" :inspect-data)
			("Inspect Vertex Structure" :inspect-vertex)
			("Set V to the Data" :set-v)
			("Inspect Results of Graph Traversal Function on Data" :apply)
			("Display Vertex (Recursively)" :display))))
	 1;;Don't add description option if user won't see the output.
2	 (WHEN (AND *description-output* (SEND *description-output* :exposed-p))
	   (PUSH '("Describe Vertex Data" :describe-vertex) options))
	 (CASE (TV:MENU-CHOOSE options
			       (si:string (FORMAT NIL "Options for vertex ~S" data))
			       '(:MOUSE)
			       (CAR options))
	   (:describe-vertex
	     (FUNCALL (SEND graph-type :vertex/edge-description-function)
		      vertex-or-edge-struct *description-output*))
	   (:inspect-data (INSPECT data))
	   (:inspect-vertex (INSPECT vertex-or-edge-struct))
	   (:set-v (SETQ v data)
		   (FORMAT T "~%~s is set to ~s." 'v data))
	   (:apply (INSPECT (FUNCALL (SEND graph-type :traversal-function) data)))
	   (:display (SEND graph-display-pane :recursively-display-vertices
			   1;;Include all neighboring vertices in the display.
2			   (CONS vertex-or-edge-struct
				 (vertex-connected-to vertex-or-edge-struct))
			   1;;But highlight the vertex which the user clicked on.
2			   (NCONS vertex-or-edge-struct))))))
      ((TYPEP vertex-or-edge-struct 'edge)
       (LET ((data (edge-data vertex-or-edge-struct))
	     (options '(("Inspect Edge Label" :inspect-data)
			("Inspect Edge Structure" :inspect-vertex)
			("Set V to the Edge Label" :set-v)
			("Display Edge (Recursively)" :display))))
	 1;;Don't add description option if user won't see the output.
2	 (WHEN (AND *description-output* (SEND *description-output* :exposed-p))
	   (PUSH '("Describe Edge Data" :describe-edge) options))
	 (CASE (TV:MENU-CHOOSE options
			       (si:string
				 (FORMAT NIL "Options for the edge from ~s to ~s"
					 (vertex-data (edge-vertex-1 vertex-or-edge-struct))
					 (vertex-data (edge-vertex-2 vertex-or-edge-struct))))
			       '(:MOUSE)
			       (CAR options))
	   (:describe-edge
	     (FUNCALL (SEND graph-type :vertex/edge-description-function)
		      vertex-or-edge-struct *description-output*))
	   (:inspect-data (INSPECT data))
	   (:inspect-vertex (INSPECT vertex-or-edge-struct))
	   (:set-v (SETQ v data)
		   (FORMAT T "~%~s is set to ~s." 'v data))
	   (:display (SEND graph-display-pane :recursively-display-vertices
			   (LIST (edge-vertex-1 vertex-or-edge-struct)
				 (edge-vertex-2 vertex-or-edge-struct))))))))))


(DEFVAR *graph-root-finding-form-edit-template* "
;;Edit the following default root-finding form, which is to be evaluated to return a list of
;;\"root\" vertices for graphs of this type, unless the user specifies different roots.
;; Test your code now, otherwise you'll be handling errors on the Graph Display Frame.

~s

;;Press CONTROL-Z to exit the window.  Press CONTROL-META-ABORT to blow it off.")

(DEFVAR *graph-root-finding-form-template* "
;;Hack together a default root-finding form, which is to be evaluated to return a list of
;;\"root\" vertices for graphs of this type, unless the user specifies different roots.
;; Test your code now, otherwise you'll be handling errors on the Graph Display Frame.

(LIST *root-vertex*) ;example

;;Press CONTROL-Z to exit the window.  Press CONTROL-META-ABORT to blow it off.")

(DEFUN default-instantiation-function (graph-type)
  "Default instantiation function for graph types,
called by the Zgraph user interface to get an instance of whatever graph type he has
selected to create.  GRAPH-TYPE is the flavor instance representing this chosen graph type.
This function is set up to request an identifying name from the user and allows him
to edit DEFAULT-ROOT-FINDING-FORM.  An application might want to have their own instantiation
function do such things as 1. not bother the user for the above info  2. set switches which
control the operation of the DEFAULT-ROOT-FINDING-FORM, TRAVERSAL-FUNCTION, print string
functions, etc."
  1;;Allow user to assign an identifying name and to override the default root-finding form
2  1;;(if there is one).
2  (LET (*name*
	(*form* (SEND graph-type :default-root-finding-form))
	*need-more-room*)
    (DECLARE (SPECIAL *name* *form* *need-more-room* *graph-root-finding-form-edit-template*))
    (TV:CHOOSE-VARIABLE-VALUES
      `((*name* ,(SI:STRING "Name")
		:DOCUMENTATION
		,(SI:STRING "Identifier for this graph instance.  Leave it NIL to accept a default name.")
		:STRING)
	(*form* ,(si:string "Root Finding Form")
		:DOCUMENTATION
		,(SI:STRING (FORMAT NIL "Form to EVAL to get a list of root ~
                                         vertices for this graph instance."))
		:SEXP)
	(*need-more-room*
	  ,(si:string "Need a large window to edit root-finding form?")
	  :DOCUMENTATION ,(si:string "Click on YES if you need a Zmacs window to edit the root-finding form.")
	  :BOOLEAN))
      :LABEL (SI:STRING "Move mouse over data fields.  Field descriptions will appear in who-line."))
    (WHEN *need-more-room*
      (SETQ *form*
	    (edit-lisp-form (IF *form*
				(FORMAT NIL *graph-root-finding-form-edit-template* *form*)
				*graph-root-finding-form-template*))))
    (debug-print T "~%Evaluating form to find roots of the graph.")
    (LET ((root-vertices (EVAL *form*)))
      (IF root-vertices
	  (debug-print T "~%Root vertices are: ~s" root-vertices)
	  (debug-print T "~%Warning, the default root finding form, ~s, evaluated to NIL."
		       *form*))
      (MAKE-INSTANCE 'graph
		     :name *name*
		     :type graph-type
		     :root-vertices root-vertices))))

(DEFMETHOD (:edit graph-type)
	   (&OPTIONAL
	     (cvv-label
	       "Move mouse over data fields.  Field descriptions will appear in who-line."))
  "1Edit the attributes of a graph.
This can be called to have the user initialize or modify a graph.2"
  (LET (*need-more-room*)
    (DECLARE (SPECIAL *need-more-room*))
    (TV:CHOOSE-VARIABLE-VALUES
      `((name ,(si:string "Name to refer to this Graph Type") :SEXP)
	(traversal-function ,(si:string "Traversal Function (name)")
          :DOCUMENTATION
	  ,(SI:STRING (FORMAT NIL "Returns an alist of the form ((tail-vertex . edge-label)...), given any vertex (the head of the edge) as arg."))
	  :SEXP)
	(default-root-finding-form ,(si:string "Default Root Finding Form")
          :DOCUMENTATION
	  ,(SI:STRING (FORMAT NIL "Form to EVAL to get a list of root ~
          vertices for any graph of this type.  NIL = query for this on every graph"))
	  :SEXP)
	(traverse-recursively? ,(si:string "Apply Traversal Function Recursively?")
          :DOCUMENTATION
	  ,(SI:STRING "If NIL, traversal function is only applied to root vertices.  Otherwise, it's applied recursively, depth-first.")
	  :BOOLEAN)
	(*need-more-room*
	  ,(si:string "Need a large window to edit root-finding form?")
	  :DOCUMENTATION ,(si:string "Click on YES if you need a Zmacs window to edit the root-finding form.")
	  :BOOLEAN)
	(vertex-print-string-function
	  ,(si:string "Vertex Print-String Function")
	  :DOCUMENTATION ,(si:string "Name of function for returning a print string and font to use in displaying a VERTEX struct.")
	  :SEXP)
	(edge-print-string-function
	  ,(si:string "Edge Print-String Function")
	  :DOCUMENTATION ,(si:string "Name of function for returning a print string and font to use in displaying a EDGE struct.")
	  :SEXP)
	(vertex/edge-description-function
	  ,(si:string "Vertex/Edge Description Function")
	  :DOCUMENTATION ,(si:string "Name of function for handling requests for description of vertices and edges.")
	  :SEXP)
	(vertex/edge-selection-handler
	  ,(si:string "Vertex/Edge Mouse Click Handler Function")
	  :DOCUMENTATION ,(si:string "Name of function for handling mouse clicks on graph vertices and edges.")
	  :SEXP))
      :LABEL (SI:STRING cvv-label))
    1;;Invoke editor if user requests it.
2    (WHEN *need-more-room*
      (SETQ default-root-finding-form
	    (edit-lisp-form (IF default-root-finding-form
				(FORMAT NIL *graph-root-finding-form-edit-template*
					default-root-finding-form)
				*graph-root-finding-form-template*))))
    1;;Handle special cases.  This could be real humorous to watch...
2    (COND
      ((NULL traversal-function)
       (BEEP)
       (SEND SELF :edit
	     "*** Traversal Function can not be NIL ***"))
      ((NOT (AND (SYMBOLP traversal-function)
		 (SYMBOLP edge-print-string-function)
		 (SYMBOLP vertex/edge-selection-handler)))
       (BEEP)
       (SEND SELF :edit
	     "*** All fields requesting \"functions\" should be symbols (or NIL for the 1st two) ***")))))


(DEFUN edit-lisp-form (editor-template &REST template-format-args)
  "1Gets a Lisp form from the user through the use of a pop-up editor window.
EDITOR-TEMPLATE is a format string used as a template form for the user to edit. 
TEMPLATE-FORMAT-ARGS are the respective FORMAT arguments.
Unfortunately, this function has no safegaurds against read errors.2"
  (READ-FROM-STRING
    (ZWEI:EDSTRING (SI:STRING (APPLY #'FORMAT NIL editor-template template-format-args))
		   NIL 0 1000. 700. ZWEI:*ZMACS-COMTAB*)))


(DEFMETHOD (:definition-form graph-type) (output-stream)
  "1Formats a string on OUTPUT-STREAM which would recreate SELF if read and evalled.2"
  (LET (*package*)
    (FORMAT output-stream "(MAKE-INSTANCE 'zg:graph-type :name '~s
 :traversal-function '~s
 :default-root-finding-form '~s
 :instantiation-function '~s
 :traverse-recursively? '~S
 :vertex-print-string-function '~s
 :edge-print-string-function '~s
 :vertex/edge-selection-handler '~s
 :vertex/edge-description-function '~s)"

	    name
	    traversal-function
	    default-root-finding-form
	    instantiation-function
	    traverse-recursively?
	    vertex-print-string-function
	    edge-print-string-function
	    vertex/edge-selection-handler
	    vertex/edge-description-function)))

(COMPILE-FLAVOR-METHODS graph-type)