;;; -*- Mode: LISP; Syntax: Common-lisp; Fonts: MEDFNT,TR12I; Package: USER -*-

;;;; Zgraph types for QPE
;;
;; Displaying total envisionments.
;;
;; Verticies are sclasses, classes of qualitatively distinct situations.
;; Edges are the limit hypotheses that connect them.

(defun tenv-vertex-print-string (sclass)
  (values (qpe::sclass-name (zg:vertex-data sclass))
	  (if (eq (qpe::sclass-duration (zg:vertex-data sclass)) :INSTANT)
	      fonts:tr8		;we want Instants to look different than others
	      fonts:tr8b)))

(defun tenv-edge-print-string (lh)
  (values (qpe::lh-name (car (zg:edge-data lh))) fonts:5x5)) 

(defun tenv-mouse-handler (v-or-e graph &rest ignore)
;;; TOO FEW ARGUMENTS HERE -- need to install menus to pop up a window
;;; that describes the situation and/or hypothesis.
  (zg:default-vertex/edge-selection-handler v-or-e graph))

(defun tenv-graph-traverser (v &optional (window nil) &aux answer)
  (dolist (env (qpe::sclass-assumptions v) answer)
    (dolist (pair (cdr (assoc env qpe::*next-state-table*)))
      (pushnew (cons (qpe::env-sclass (cdr pair)) (car pair))
	       answer
	       :test #'equal))))

(defvar *scroller-font-fixed?* nil)
(DEFVAR *tenv-graph-type-print-ds-values?* T)
(PUSH '("Toggle Printing of Ds values in TENV descriptions."
	:EVAL (SETQ *tenv-graph-type-print-ds-values?*
		    (NOT *tenv-graph-type-print-ds-values?*)))
      zg:*miscellaneous-command-menu*)

(defvar *tenv-graph-type-print-envs?* t)
(push '("Toggle printing of situations in TENV descriptions"
	:EVAL (SETQ *tenv-graph-type-print-envs?*
		    (not *tenv-graph-type-print-envs?*)))
      zg:*miscellaneous-command-menu*)

(defun tenv-v/e-description (part scroll-window &aux str description-lines)
  (IF (TYPEP part 'zg:vertex)
      (SETQ part (zg:vertex-data part))
      (SETQ part (CAR (zg:edge-data part))))
  (unless *scroller-font-fixed?*
    (funcall scroll-window :set-font-map (list fonts:hl10b)))
  (cond ((qpe::limit-hypothesis? part)
	 (setq str (with-output-to-string (foo)
		     (qpe::print-lh part nil foo))))
	(t ;; assume it is an sclass
	 (setq str (with-output-to-string (foo)
		     (qpe::print-sclass part foo)
		     (WHEN qpe::*tenv-graph-type-print-ds-values?*
		       (qpe::print-ds-values part foo))
		     (TERPRI foo)
		     (WHEN qpe::*tenv-graph-type-print-envs?*
		       (DOLIST (assumption (qpe::sclass-assumptions part))
			 (adb::print-env assumption foo)))))))
;;undefined	     (qpe::print-situation part foo)))))
  (setq description-lines (with-input-from-string (stream str)
			    (loop for line = (read-line stream nil)
				  while line
				  collect line)))
;;  (zl-user::send scroll-window :set-items
;;	(mapcar #'(lambda (line) (list "~A" line)) description-lines))
  ;;From ZG:DEFAULT-VERTEX/EDGE-DESCRIPTION-FUNCTION.  This appends the description
  ;;onto current scroll text.
  (LET ((separate-at (LENGTH (ZL-USER::SEND scroll-window :ITEMS))))
    ;;Separate this description from previous ones.
    (ZL-USER::SEND scroll-window :APPEND-ITEM '("------------------------"))
    ;;Add in the lines of description.
    ;;The scroll window items are entries of the form ({format string} . {format args}).
    ;;We simply use "~a" to print out a description line.
    (DOLIST (line description-lines)
      (ZL-USER::SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
    ;;Scroll to top of this description.
    (ZL-USER::SEND scroll-window :scroll-absolute separate-at)))

(declare (special qpe::*ps-vs* *inequalities*))

(defun tenv-find-root () (qpe::list-all-sclasses))

(zl:make-instance 'zg:graph-type
		     :name 'Total-Envisionment-Plot
		     :traversal-function 'tenv-graph-traverser
		     :default-root-finding-form '(tenv-find-root)
		     :vertex/edge-selection-handler 'tenv-mouse-handler	;;corrected this
		     :vertex/edge-description-function 'tenv-v/e-description
		     :vertex-print-string-function 'tenv-vertex-print-string
		     :edge-print-string-function 'tenv-edge-print-string)

;;;;; Plotting pure situations

;; Verticies are situations.
;; Edges are the limit hypotheses that connect them.

(defun tenv-sit-vertex-print-string (env)
  (values (format nil "~A" (zg:vertex-data env)) fonts:tr8b))

(defun tenv-sit-edge-print-string (lh)
  (values (qpe::lh-name (car (zg:edge-data lh))) fonts:5x5)) 

(defun tenv-sit-mouse-handler (v-or-e graph &rest ignore)
;;; TOO FEW ARGUMENTS HERE -- need to install menus to pop up a window
;;; that describes the situation and/or hypothesis.
  (zg:default-vertex/edge-selection-handler v-or-e graph))

(defun tenv-sit-graph-traverser (v &optional (window nil) &aux answer)
  (dolist (pair (cdr (assoc v qpe::*next-state-table*)) answer)
      (push (cons (cdr pair) (car pair)) answer)))

(defun tenv-sit-v/e-description (part scroll-window &aux str description-lines)
  (IF (TYPEP part 'zg:vertex)
      (SETQ part (zg:vertex-data part))
      (SETQ part (CAR (zg:edge-data part))))
  (unless *scroller-font-fixed?*
    (funcall scroll-window :set-font-map (list fonts:hl10b)))
  (cond ((qpe::limit-hypothesis? part)
	 (setq str (with-output-to-string (foo)
		     (qpe::print-lh part t foo))))
	(t ;; assume it is an sclass
	 (setq str (with-output-to-string (foo)
		     (qpe::print-env part foo)
		     (TERPRI foo)))))
;;undefined	     (qpe::print-situation part foo)))))
  (setq description-lines (with-input-from-string (stream str)
			    (loop for line = (read-line stream nil)
				  while line
				  collect line)))
;;  (zl-user::send scroll-window :set-items
;;	(mapcar #'(lambda (line) (list "~A" line)) description-lines))
  ;;From ZG:DEFAULT-VERTEX/EDGE-DESCRIPTION-FUNCTION.  This appends the description
  ;;onto current scroll text.
  (LET ((separate-at (LENGTH (ZL-USER::SEND scroll-window :ITEMS))))
    ;;Separate this description from previous ones.
    (ZL-USER::SEND scroll-window :APPEND-ITEM '("------------------------"))
    ;;Add in the lines of description.
    ;;The scroll window items are entries of the form ({format string} . {format args}).
    ;;We simply use "~a" to print out a description line.
    (DOLIST (line description-lines)
      (ZL-USER::SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
    ;;Scroll to top of this description.
    (ZL-USER::SEND scroll-window :scroll-absolute separate-at)))

(declare (special qpe::*ps-vs* *inequalities*))

(defun tenv-sit-find-root ()
  (let ((sits nil))
    (qpe::map-sclasses
      #'(lambda (sc) (setq sits (append (qpe::sclass-assumptions sc)
					sits))))
    sits))

(zl::make-instance 'zg:graph-type
		     :name 'Total-Envisionment-Plot-by-situation
		     :traversal-function 'tenv-sit-graph-traverser
		     :default-root-finding-form '(tenv-sit-find-root)
		     :vertex/edge-selection-handler 'tenv-sit-mouse-handler
		     ;;corrected this
		     :vertex/edge-description-function 'tenv-sit-v/e-description
		     :vertex-print-string-function 'tenv-sit-vertex-print-string
		     :edge-print-string-function 'tenv-sit-edge-print-string)

;;;; Graphing inequality relationships  
;;
;; The root finding form is (situation-inequalities <sit>)
;; It fetches all the inequalities that hold under the current
;; situation, and creates tables that the graph finding code
;; associated with ZGRAPH can manipulate.

;;; The tables take the following form:
;;; (<number> . <relations>)
;;; Arrows will be from the smaller to the larger quantity.
;;; In the case of equality, install arrows both ways.

(declare (special *inequalities* *index*))

(defvar *inequalities* nil)
(defvar *index* nil)
(defvar *index-counter* -1)

(defun find-index (item)
 (let ((entry (rassoc item *index* :test #'equal)))
    (unless entry
      (push (setq entry (cons (incf *index-counter*) item)) *index*))
    (car entry)))

(defun ineq-vertex-print-string (num)
  (values (adb::lprinc-string (cdr (assoc (zg:vertex-data num)
					     *index*))) fonts:5x5))

(defun ineq-edge-print-string (ineq)
  (values  (car (zg:edge-data ineq)) fonts:5x5))

(defun ineq-mouse-handler (v-or-e graph &rest ignore)
  ;;;Again, don't really know what to do with this yet.  So
  ;;;fake it.
  (zg:default-vertex/edge-selection-handler v-or-e graph))

(defun ineq-graph-traverser (v &optional (window nil) &aux answer)
  (dolist (ineq (cdr (assoc v *inequalities* :test #'equal)) 
		answer)
    (pushnew (cons (cadr ineq) (car ineq)) answer
	     :test #'equal)))

(defun find-inequalities (sit)
  ;;; Assumes sit is a legal tms::env.
  ;;; Table takes the form (<number> . arcs)
  ;;; Arcs take the form ("< or =", <n1> <n2>)
  (setq *inequalities* nil
	*index* nil
	*index-counter* -1)
  (dolist (gt (adb::fetch '((qpe::Greater-than ?n1 ?n2) . :TRUE) (list :IN sit)))
    (let ((key (find-index (caddr (car gt))))
	  (item (list 'qpe::< (find-index (cadr (car gt)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (lt (adb::fetch '((qpe::Less-than ?n1 ?n2) . :TRUE) (list :IN sit)))
    (let ((key (find-index (cadr (car lt))))
	  (item (list 'qpe::< (find-index (caddr (car lt)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (eq (adb::fetch '((qpe::Equal-to ?n1 ?n2) . :TRUE) (list :IN sit)))
    (let ((key (find-index (cadr (car eq))))
	  (item (list 'qpe::= (find-index (caddr (car eq)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry))))
    (let ((key (find-index (caddr (car eq))))
	  (item (list 'qpe::= (find-index (cadr (car eq)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry))))))

(defun ineq-v/e-description (part scroll-window &aux str description-lines)
  (IF (TYPEP part 'zg:vertex)
      (SETQ part (zg:vertex-data part))
      (SETQ part (CAR (zg:edge-data part))))
  (unless *scroller-font-fixed?*
    (funcall scroll-window :set-font-map (list fonts:hl10b)))
  ;;Fixed by Hogge--package for LPRINC-STRING. and string of wrong type.
  (setq description-lines
	(with-input-from-string (stream (STRING (adb::lprinc-string part)))
	  (loop for line = (read-line stream nil)
		while line
		collect line)))
  (LET ((separate-at (LENGTH (ZL-USER::SEND scroll-window :ITEMS))))
    (ZL-USER::SEND scroll-window :APPEND-ITEM '("------------------------"))
    ;;Add in the lines of description.
    ;;The scroll window items are entries of the form ({format string} . {format args}).
    ;;We simply use "~a" to print out a description line.
    (DOLIST (line description-lines)
      (ZL-USER::SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
    ;;Scroll to top of this description.
    (ZL-USER::SEND scroll-window :scroll-absolute separate-at)))

(defun ineq-find-roots (sit-env)
  (find-inequalities sit-env)
  (mapcar #'car *inequalities*))

(zl::make-instance 'zg:graph-type
		     :name 'Situation-Inequalities-Plot
		     :traversal-function 'ineq-graph-traverser
		     :default-root-finding-form '(ineq-find-roots 
						   (tms::e PICK-A-NUMBER))
		     :vertex/edge-selection-handler 'ineq-mouse-handler	;;corrected this
		     :vertex/edge-description-function 'ineq-v/e-description
		     :vertex-print-string-function 'ineq-vertex-print-string
		     :edge-print-string-function 'ineq-edge-print-string)

;;;; Graphing inequalities sans zero
(defun find-index-sans-zero (item)
  (if (eq item 'QPE::zero) 0
      (let ((entry (rassoc item *index* :test #'equal)))
	(unless entry
	  (push (setq entry (cons (incf *index-counter*) item)) *index*))
	(car entry))))

(defun ineq-vertex-print-string-sans-zero (num &aux entry sign number)
  (setq num (zg:vertex-data num)
	entry (cdr (assoc num *index*))
	sign (car entry)
	number (cdr entry))
  (values (format nil "~A:~A" sign (adb::lprinc-string number)) fonts:5x5))

(defun modify-index (&aux num number rel-to-zero sign)
  (dolist (pair *index*)
    (setq num (car pair)
	  number (cdr pair)
	  rel-to-zero
	  (cond ((car (rassoc (list 0) (cdr (assoc num *inequalities*)) :test #'equal)))
		((member num (mapcar #'cadr (cdr (assoc 0 *inequalities*)))) 'QPE::>))
	  sign (case rel-to-zero (QPE::> '+) (QPE::= '0) (QPE::< '-) (t "?")))
    (rplacd pair (cons sign number))))
	  
(defun ineq-graph-traverser-sans-zero (v &optional (window nil) &aux answer)
  (dolist (ineq (cdr (assoc v *inequalities* :test #'equal)) 
		answer)
    (unless (eq (cadr ineq) 0)
      (pushnew (cons (cadr ineq) (car ineq)) answer
	       :test #'equal))))

(defun find-inequalities-sans-zero (sit)
  ;;; Assumes sit is a legal tms:env.
  ;;; Table takes the form (<number> . arcs)
  ;;; Arcs take the form ("< or =", <n1> <n2>)
  (setq *inequalities* nil
	*index* nil
	*index-counter* 0)
  (dolist (gt (adb:fetch '((QPE::Greater-than ?n1 ?n2) . :TRUE) (list :IN sit)))
    (let ((key (find-index-sans-zero (caddr (car gt))))
	  (item (list 'QPE::< (find-index-sans-zero (cadr (car gt)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (lt (adb:fetch '((QPE::Less-than ?n1 ?n2) . :TRUE) (list :IN sit)))
    (let ((key (find-index-sans-zero (cadr (car lt))))
	  (item (list 'QPE::< (find-index-sans-zero (caddr (car lt)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (eq (adb:fetch '((QPE::Equal-to ?n1 ?n2) . :TRUE) (list :IN sit)))
    (let ((key (find-index-sans-zero (cadr (car eq))))
	  (item (list 'QPE::= (find-index-sans-zero (caddr (car eq)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry))))
    (let ((key (find-index-sans-zero (caddr (car eq))))
	  (item (list 'QPE::= (find-index-sans-zero (cadr (car eq)))))
	  (entry nil))
      (setq entry (assoc key *inequalities* :test #'equal))
      (unless entry (push (setq entry (list key)) *inequalities*))
      (rplacd entry (cons item (cdr entry))))))

(defun ineq-find-roots-sans-zero (sit-env &aux answer)
  (find-inequalities-sans-zero sit-env)
  (modify-index)
  (dolist (entry *inequalities* answer)
    (if (= (car entry) 0)
	(dolist (item (cdr entry))
	  (pushnew (cadr item) answer))
	(push (car entry) answer))))

(zl::make-instance 'zg:graph-type
	       :name 'ineq-graph-sans-zero
	       :traversal-function 'ineq-graph-traverser-sans-zero
	       :default-root-finding-form '(ineq-find-roots-sans-zero
					     (tms:e :PICK-A-NUMBER))
	       :vertex/edge-selection-handler 'ineq-mouse-handler	;;corrected this
	       :vertex/edge-description-function 'ineq-v/e-description
	       :vertex-print-string-function 'ineq-vertex-print-string-sans-zero
	       :edge-print-string-function 'ineq-edge-print-string)


;;;; Graphing ALL inequality relationships  
;;
;; The root finding form is *qnums*
;; The idea is to grab all inequality links, classifying them according to whether
;; they are in the assumed, derived, or resolution Qstate.
;; This should simplify debugging various algorithms.

;;; The tables take the following form:
;;; (<number> <ineq> <class>)
;;; Arrows will go both ways, since we are only expressing the fact that the
;;; two are compared.

(declare (special *inequalities* *index*))

(defvar *inequalities* nil)

(defun all-ineq-vertex-print-string (num)
  (values (qpe::qnum-string (zg:vertex-data num)) fonts:5x5))

(defun all-ineq-edge-print-string (ineq)
  (values (car (zg:edge-data ineq)) fonts:5x5))
					  
(defun all-ineq-mouse-handler (v-or-e graph &rest ignore)
  ;;;Again, don't really know what to do with this yet.  So
  ;;;fake it.
  (zg:default-vertex/edge-selection-handler v-or-e graph))

(defun all-ineq-graph-traverser (v &optional (window nil) &aux answer)
  (dolist (ineq-entry (cdr (assoc v *inequalities* :test #'eq)) 
		answer)
    (pushnew ineq-entry answer :test #'equal)))

(defvar *assumed-qstate* nil)

(defun find-all-inequalities (&optional (include-zero? t))
  ;;; Arcs take the form ("! or & or ? ", <n1> <n2>)
  (setq *inequalities* nil)
  (setq *assumed-qstate* 
	(remove-if-not #'qpe::fixed-ineq? qpe::*ineqs*))
  (dolist (ineq qpe::*ineqs*)
    (unless (and (not include-zero?)
		 (eq (qpe::ineq-n2 ineq) qpe::*zero-struct*))
    (let ((key (qpe::ineq-n1 ineq))
	  (item (cons (qpe::ineq-n2 ineq) (find-ineq-status ineq)))
	  (entry nil))
      (unless (and (string-equal (cdr item) "?")
		   (eq (car item) qpe::*zero-struct*)) 
	(setq entry (assoc key *inequalities* :test #'eq))
	(unless entry (push (setq entry (cons key nil)) *inequalities*))
	(setf (cdr entry) (cons item (cdr entry)))))))
  *inequalities*)

(proclaim '(special *assumed-qstate*))

(defun find-ineq-status (key)
  (cond ((member key *assumed-qstate* :test #'equal) "!")
	((member key qpe::*derived-qstate* :test #'equal) "&")
	(t "?")))

(defun all-ineq-v/e-description (part scroll-window &aux str description-lines)
  (IF (TYPEP part 'zg:vertex)
      (SETQ part (zg:vertex-data part))
      (SETQ part (CAR (zg:edge-data part))))
  (unless *scroller-font-fixed?*
    (funcall scroll-window :set-font-map (list fonts:hl10b)))
  ;;Fixed by Hogge--package for LPRINC-STRING. and string of wrong type.
  (setq description-lines
	(with-input-from-string (stream (STRING (adb::lprinc-string part)))
	  (loop for line = (read-line stream nil)
		while line
		collect line)))
  (LET ((separate-at (LENGTH (ZL-USER::SEND scroll-window :ITEMS))))
    (ZL-USER::SEND scroll-window :APPEND-ITEM '("------------------------"))
    ;;Add in the lines of description.
    ;;The scroll window items are entries of the form ({format string} . {format args}).
    ;;We simply use "~a" to print out a description line.
    (DOLIST (line description-lines)
      (ZL-USER::SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
    ;;Scroll to top of this description.
    (ZL-USER::SEND scroll-window :scroll-absolute separate-at)))

(defun all-ineq-find-roots ()
  (find-all-inequalities)
   qpe::*qnums*)

(zl::make-instance 'zg:graph-type
		     :name 'All-Inequalities-Plot
		     :traversal-function 'all-ineq-graph-traverser
		     :default-root-finding-form '(all-ineq-find-roots)
		     :vertex/edge-selection-handler 'all-ineq-mouse-handler
		     ;;corrected this
		     :vertex/edge-description-function 'all-ineq-v/e-description
		     :vertex-print-string-function 'all-ineq-vertex-print-string
		     :edge-print-string-function 'all-ineq-edge-print-string)

;;; This version flushes all isolated numbers.
(defun all-ineq-find-roots1 (&optional (include-zero? t))
  (find-all-inequalities include-zero?)
  (mapcar #'car *inequalities*))

(zl::make-instance 'zg:graph-type
		     :name 'All-Inequalities-Plot-sans-isolates
		     :traversal-function 'all-ineq-graph-traverser
		     :default-root-finding-form '(all-ineq-find-roots1)
		     :vertex/edge-selection-handler 'all-ineq-mouse-handler
		     ;;corrected this
		     :vertex/edge-description-function 'all-ineq-v/e-description
		     :vertex-print-string-function 'all-ineq-vertex-print-string
		     :edge-print-string-function 'all-ineq-edge-print-string)

;; This version doesn't include ZERO.
(zl::make-instance 'zg:graph-type
		     :name 'All-Inequalities-Sans-Zero
		     :traversal-function 'all-ineq-graph-traverser
		     :default-root-finding-form '(all-ineq-find-roots1 nil)
		     :vertex/edge-selection-handler 'all-ineq-mouse-handler
		     ;;corrected this
		     :vertex/edge-description-function 'all-ineq-v/e-description
		     :vertex-print-string-function 'all-ineq-vertex-print-string
		     :edge-print-string-function 'all-ineq-edge-print-string)

;;;; Graphing influences  
;;
;; The root finding form is (situation-influences <sit>)
;; It fetches all the direct and indirect influences  that hold under the current
;; situation, and creates tables that the graph finding code
;; associated with ZGRAPH can manipulate.

;;; The tables take the following form:
;;; (<number> . <relations>)
;;; For Qprop's, arrows go from the constrainer to the constrainee
;;; For I+'s, I-'s, arrows go from the rate contribution to the quantity directly influenced

(declare (special *influences* *index*))

(defvar *inequalities* nil)
(defvar *index* nil)
(defvar *index-counter* -1)

(defun influence-vertex-print-string (num)
      (values (adb::lprinc-string (cdr (assoc (zg:vertex-data num)
					     *index*))) fonts:5x5))

(defun influence-edge-print-string (ineq)
  (values  (car (zg:edge-data ineq)) fonts:5x5))

(defun influence-mouse-handler (v-or-e graph &rest ignore)
  ;;;Again, don't really know what to do with this yet.  So
  ;;;fake it.
  (zg:default-vertex/edge-selection-handler v-or-e graph))

(defun influence-graph-traverser (v &optional (window nil) &aux answer)
  (dolist (ineq (cdr (assoc v *influences* :test #'equal)) 
		answer)
    (pushnew (cons (cadr ineq) (car ineq)) answer
	     :test #'equal)))

(defun find-influences (the-sit)
  ;;; Assumes sit is either a situation env, :ALL or :IN. 
  ;;; Table takes the form (<number> . arcs)
  ;;; Arcs take the form ("Qprop+, Qprop-, I+, I-", <Quantity> <Quantity>)
  (let ((sit (case the-sit
	       ((:ALL :IN) nil)(t the-sit)))
	(condition (case the-sit
		     (:IN 'tms:in)(:ALL nil)(t (list ':IN the-sit)))))
  (setq *influences* nil
	*index* nil
	*index-counter* -1)
  ;;Index every quantity, to ensure that even uninfluenced quantities are
  ;;represented in the eventual graph.
  (dolist (q (qpe::gather-quantities)) (find-index q))
  (dolist (gt (adb::fetch '((qpe::Qprop ?q1 ?q2) . :TRUE) condition))
    (let ((key (find-index (caddr (car gt))))
	  (item (list 'qpe::Qprop (find-index (cadr (car gt)))))
	  (entry nil))
      (setq entry (assoc key *influences* :test #'equal))
      (unless entry (push (setq entry (list key)) *influences*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (gt (adb::fetch '((qpe::Qprop- ?q1 ?q2) . :TRUE) condition))
    (let ((key (find-index (caddr (car gt))))
	  (item (list 'qpe::Qprop- (find-index (cadr (car gt)))))
	  (entry nil))
      (setq entry (assoc key *influences* :test #'equal))
      (unless entry (push (setq entry (list key)) *influences*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (gt (adb::fetch '((qpe::I+ ?q1 ?q2) . :TRUE) condition))
    (let ((key (find-index (cadr (caddr (car gt)))))
	  (item (list 'qpe::I+ (find-index (cadr (car gt)))))
	  (entry nil))
      (setq entry (assoc key *influences* :test #'equal))
      (unless entry (push (setq entry (list key)) *influences*))
      (rplacd entry (cons item (cdr entry)))))
  (dolist (gt (adb::fetch '((qpe::I- ?q1 ?q2) . :TRUE) condition))
    (let ((key (find-index (cadr (caddr (car gt)))))
	  (item (list 'qpe::I- (find-index (cadr (car gt)))))
	  (entry nil))
      (setq entry (assoc key *influences* :test #'equal))
      (unless entry (push (setq entry (list key)) *influences*))
      (rplacd entry (cons item (cdr entry)))))))

(defun influence-v/e-description (part scroll-window &aux str description-lines)
  (IF (TYPEP part 'zg:vertex)
      (SETQ part (zg:vertex-data part))
      (SETQ part (CAR (zg:edge-data part))))
  (unless *scroller-font-fixed?*
    (funcall scroll-window :set-font-map (list fonts:hl10b)))
  ;;Fixed by Hogge--package for LPRINC-STRING. and string of wrong type.
  (setq description-lines
	(with-input-from-string (stream (STRING (adb::lprinc-string part)))
	  (loop for line = (read-line stream nil)
		while line
		collect line)))
  (LET ((separate-at (LENGTH (ZL-USER::SEND scroll-window :ITEMS))))
    (ZL-USER::SEND scroll-window :APPEND-ITEM '("------------------------"))
    ;;Add in the lines of description.
    ;;The scroll window items are entries of the form ({format string} . {format args}).
    ;;We simply use "~a" to print out a description line.
    (DOLIST (line description-lines)
      (ZL-USER::SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
    ;;Scroll to top of this description.
    (ZL-USER::SEND scroll-window :scroll-absolute separate-at)))

(defun influence-find-roots (sit-env)
  (find-influences sit-env)
  (mapcar #'car *index*))

(zl::make-instance 'zg:graph-type
		     :name 'Situation-Influences-Plot
		     :traversal-function 'influence-graph-traverser
		     :default-root-finding-form '(influence-find-roots 
						   (tms::e PICK-A-NUMBER))
		     :vertex/edge-selection-handler 'influence-mouse-handler
		     :vertex/edge-description-function 'influence-v/e-description
		     :vertex-print-string-function 'influence-vertex-print-string
		     :edge-print-string-function 'influence-edge-print-string)


;;;; Graphing Attainable Envisionments

(defvar *initial-conditions* nil)

;;; Here we return the sclasses that contain a situation which is consistent with the initial-conditions
;;;
(defun find-initial-sclasses (&aux sclasses)
  (dolist (sclass (qpe::list-all-sclasses))
    (if (some #'(lambda (sit)  (every #'(lambda (n) (tms::consistent-in? n sit))
				      *initial-conditions*))
	      (qpe::sclass-assumptions sclass))
	(push sclass sclasses)))
  sclasses)


(DEFUN Aenv-initiator (graph-type)
  "Instantiation function for Attainable Envisionment graph types"
  ;;Allow user to assign an identifying name and to override the default
  ;; root-finding form (if there is one).
  (LET (*name*
	(*form* (zl-user::SEND graph-type :default-root-finding-form)))

    (DECLARE (SPECIAL *name* *form*))
    (TV:CHOOSE-VARIABLE-VALUES
      `((*name* ,(SI:STRING "Name")
		:DOCUMENTATION
		,(SI:STRING
 "Identifier for this graph instance.  Leave it NIL to accept a default name.")
		:STRING)
	(*form* ,(si:string "Root Finding Form")
		:DOCUMENTATION
		,(SI:STRING (FORMAT NIL "Form to EVAL to get a list of root ~
                                         vertices for this graph instance."))
		:SEXP)
	)
      :LABEL (SI:STRING "Move mouse over data fields.  Field descriptions will appear in who-line."))

    (query-initial-conditions)			;get the initial conditions from the user
    (zg:debug-print T "~%Evaluating form to find roots of the graph.")
    (LET ((root-vertices (EVAL *form*)))
      (IF root-vertices
	  (zg:debug-print T "~%Root vertices are: ~s" root-vertices)
	  (zg:debug-print T "~%Warning, the default root finding form, ~s, evaluated to NIL."
		       *form*))
      (ZL::MAKE-INSTANCE 'zg:graph
		     :name *name*
		     :type graph-type
		     :root-vertices root-vertices))))


(defun query-initial-conditions ()
  (let ((inequalities (tv:multiple-choose
			(si:string "Select the Initial Conditions")
					  (get-qstate-menu-list)
					  `((:< ,(si:string "<"))
					    (:= ,(si:string "="))
					    (:> ,(si:string ">"))
					    (Dont-Care
					      ,(si:string "Don't Care"))))))
    (setq *initial-conditions* nil)
    (dolist (inequality inequalities)
      (qpe::put-ineq-prop (first inequality) (second inequality) :AENV-REL)
      (when (not (eq (second inequality) 'Dont-Care))
	(push (qpe::ineq-rel-node (first inequality) (second inequality))
	      *initial-conditions*)))))

;(defvar *initial-conditions-menu-list* nil)

(defun get-qstate-menu-list ()
  (mapcar #'(lambda (ineq)
	      (let ((rel (qpe::get-ineq-prop ineq :AENV-REL)))
		(list ineq (si:string (qpe::ineq-string ineq))
		      `((:< ,(eq rel :<))
			(:= ,(eq rel :=))
			(:> ,(eq rel :>))
			(Dont-Care ,(not (member rel '(:< := :>))))))))
		(qpe::gather-assumed-qstate)))

(zl::make-instance 'zg:graph-type
		     :name 'Attainable-Envisionment-Plot
		     :traversal-function 'tenv-graph-traverser
		     :default-root-finding-form '(find-initial-sclasses)
		     :instantiation-function 'Aenv-initiator
		     :vertex/edge-selection-handler 'tenv-mouse-handler	;;corrected this
		     :vertex/edge-description-function 'tenv-v/e-description
		     :vertex-print-string-function 'tenv-vertex-print-string
		     :edge-print-string-function 'tenv-edge-print-string)

;;;; Graphing the justification structure for a chosen fact

;;; Much of this is lifted from John Hogge's QPE debugger code

;;; (Thanks, Brian!)

(defvar *dependency-nodes-visited* nil
  "we need to clip circularities for lattice display")

(defvar *dp-show-labels* nil "flag to show a node's label under it in the graph")
(defvar *dp-clip-at-no-support* t
  "flag to halt inspection of unbelieved nodes (empty label)")
(defvar *dp-clip-at-assumption* t
  "flag to halt inspection of an assumption's justifications")
(defvar *dp-clip-at-false* t
 "flag to halt inspection of nodes with false truth condition")

(defun find-initial-tms-node ()
  (let ((items (mapcar #'(lambda (fact)
		   (let ((node (adb:get-tms-node (adb:referent fact))))
		     (cons (if (tms::node-status node)
			       (format nil "~A ~A" fact (tms::node-status node))
				       (format nil "~A" fact))
				   node)))
		       (fetch-candidate-initial-facts)))
	selection)
    (setq selection (tv:menu-choose items
			    (si:string
     "Facts matching your query. Choose one to explore")))
    (unless (or (null selection) (typep selection 'tms::node))
      (error "chosen item was not a tms node"))
    (list selection)))


;;; Query user to get a set of fact forms to eventually choose from (uses adb:fetch).
;;;  Also establishes user's graph drawing parameters (node clipping parms)
;;;
(defun fetch-candidate-initial-facts ()
  (let ((query-fact qpe::'(greater-than (A (Amount-of ?x)) (A (Amount-of ?y))))
	(truth-value :true)
	(fact-status nil)
	(translate? nil))
   (declare (special query-fact truth-value fact-status translate?))
   (tv:choose-variable-values			;query user for form
     `((query-fact ,(si:string "Form") :SEXP)
       (truth-value ,(si:string "Truth Value")
		    :CHOOSE (:true :false 1 0 -1 BT ?anything))
       (fact-status ,(si:string "Truth Status Constraint")
	   :CHOOSE (NIL tms:in tms:out :inness :consistency))
       ,(si:string "  NIL= no truth constraints")
       ,(si:string "  IN//OUT = believed//disbelieved")
       ,(si:string "  INNESS = Input an environment to check fact INness")
       ,(si:string
	  "  CONSISTENCY = Input an environment to check fact consistency.")
       (translate? ,(si:string "Translate Inequalities?") :boolean)
       ""
       ,(si:string "         == Graph Drawing Parameters ==")
       ""
       (*dp-show-labels* ,(si:string "Show node labels?") :boolean)
       (*dp-clip-at-no-support*
	 ,(si:string "Don't backtrace unsupported (empty label) nodes")
				:boolean)
       (*dp-clip-at-assumption*
	 ,(si:string "Don't backtrace nodes coming from assumptions")
				:boolean))
     :label (si:string "Input a fact whose status is to be queried"))
   (when (zl-user::memq fact-status '(:inness :consistency))
     (let ((selection (zl-user::prompt-and-read :number-or-nil
				       "environment number (or RETURN for menu): ")))
       (if selection
	   (unless (setq selection (tms:e selection))
		 (format t
		    "~%No Environment #~a--proceeding with no truth constraint."
			    selection))
	   (format t
		"~%No environment chosen--proceeding with no truth constraint."))
       (if selection
	   (setq fact-status
		 (if (eq fact-status :inness) (list :in selection) selection))
	   (setq fact-status nil))))
    ;; do actual fetch of the user provided form
   (nconc (adb:fetch (cons query-fact truth-value) fact-status)
	  (when (and translate?
		     (zl-user::memq (car query-fact) '(qpe::greater-than
					       qpe::less-than qpe::equal-to)))
	    (let ((reverse-op (case (car query-fact)
				(qpe::greater-than 'qpe::less-than)
				(qpe::less-than 'qpe::greater-than)
				(qpe::equal-to 'qpe::equal-to)))
		   (rest-of-form (cond ((listp (cdr query-fact))
					(reverse (cdr query-fact)))
				       (T (cdr query-fact)))))
	       (adb:fetch
		 (cons (cons reverse-op rest-of-form) truth-value)
		 fact-status))))))


(defun dependency-initiator (graph-type)
  "Instantiation function for TMS Dependency Lattice graph types"
  (let ((*form* (zl-user::send graph-type :default-root-finding-form)))
    (DECLARE (SPECIAL *form*))
    (setq *dependency-nodes-visited* nil)
    (zg:debug-print T "~%Evaluating form to find roots of the graph.")
    (let ((root-vertices (eval *form*)))
      (if root-vertices
	  (zg:debug-print T "~%Root vertices are: ~s" root-vertices)
	  (zg:debug-print T
	      "~%Warning, the default root finding form, ~s, evaluated to NIL."
			  *form*))
      (ZL::MAKE-INSTANCE 'zg:graph
		     :name nil
		     :type graph-type
		     :root-vertices root-vertices))))


;;; Given a tms node, return next nodes to grow from it (i.e., the current node's antecedents).
;;;   When about to grow into a node that is being reached a second time, grow it as (:bp node) so that
;;;   graph displayer will not recognize it as the same node seen before (will plot it again, but not grow it).
;;;
(defun dependency-graph-traverser (node &optional (window nil) &aux answer)
  (cond ((consp node)
			;one of my tags is at the car of (<tag> . justifications>)
	 (cond ((eq (car node) 'qpe::and)
		(mapcar #'(lambda (n)
			    (cond ((member n *dependency-nodes-visited* :test #'eq)
				   (cons (cons :bp n) " "))
				  ((cons n " "))))
			(cdr node)))
	       ((eq (car node) :bp) nil)
	       ((error "funny node: ~A" node))))
	((tms::assumption? node) nil)
	((and *dp-clip-at-assumption*
	      (tms::node-assumption node)) nil)	;don't grow assumption
	((and *dp-clip-at-no-support*
	      (eq (tms::node-status node) 'tms:out)) nil) ;empty label
	((and *dp-clip-at-false*
	      (eq (cdr (adb::lisp-form (tms::node-datum node))) :false)) nil)
	(t (push node *dependency-nodes-visited*)
	   (dolist (justification (tms::node-justifications node) answer)
	     (cond ((cddr justification)	;a conjunctive justification
		    (push (cons (cons 'qpe::and (cdr justification))
				(car justification))
			  answer))
		   ((cdr justification)
		    ;no cdr means it's an assertion - no justification
		    (if (member (second justification)
				*dependency-nodes-visited* :test #'eq)
			(push (cons (cons :bp (second justification))
				    (car justification))
			      answer)
			(push (cons (second justification) 
				    ;; WORK-AROUND FOR UNIQUE_VALUE bug
				    (if (listp (car justification))
					(caar justification)
					(car justification)))
			      answer))))))))


(defun dependency-vertex-print-string (tms-node)
  (macrolet ((node-string (node)
	       `(if *dp-show-labels*
		    (format nil "~A~%~A"
			    (substitute 126 5
				(adb::lprinc-string
				  (adb::lisp-form (tms::node-datum ,node))))
			    (tms::node-envs ,node))
		    (substitute 126 5
				(adb::lprinc-string
				  (adb::lisp-form (tms::node-datum ,node)))))))
    (let ((node (zg:vertex-data tms-node)))
      (cond ((consp node)
	     (cond ((eq (car node) :bp)
		    ;a backpointer node (we've already displayed it once)
		    (cond ((tms::assumption? (cdr node))
			   (values (format nil "ASSUMPTION ~A"
					   (tms::assumption-unique node))
				   fonts:tr8i))
			  ((tms::node-assumption (cdr node))
			   (values (node-string (cdr node)) fonts:tr8i))
			  ((null (cdar (tms::node-justifications (cdr node))))
			   ;no antecedents
			   (values (node-string (cdr node)) fonts:tr8b))
			  ((values (format nil "**~A**" (node-string (cdr node)))
				   fonts:tr8b))))
		   ((values (car node) fonts:tr8b))))
	    ;just print my tag at the car of form
	    ((tms::assumption? node)
	     (values (format nil "ASSUMPTION ~A"
			     (tms::assumption-unique node)) fonts:tr8i))
	    ((tms::node-assumption node)  (values (node-string node) fonts:tr8i))
	    ((values (node-string node) fonts:tr8b))))))

(defun dependency-edge-print-string (justification-type)
  (values (car (zg:edge-data justification-type)) fonts:5x5))


(defun dependency-v/e-description (part scroll-window)
  (let* ((description
	   (with-output-to-string (stream)
	     (let ((*standard-output* stream))
	       (describe (if (typep part 'zg:vertex)
			     (zg:vertex-data part)
			     (zg:edge-data part))
			 T))))
	 ;;Break up the string into a list of lines of the description: ("line 1" "line 2"...)
	 (description-lines
	   (with-input-from-string (stream description)
	     (loop for line = (read-line stream nil)
		   while line
		   collect line))))
    (LET ((separate-at (LENGTH (ZL-USER::SEND scroll-window :ITEMS))))
      (ZL-USER::SEND scroll-window :APPEND-ITEM '("------------------------"))
      ;;Add in the lines of description.
      ;;The scroll window items are entries of the form ({format string} . {format args}).
      ;;We simply use "~a" to print out a description line.
      (DOLIST (line description-lines)
	(ZL-USER::SEND scroll-window :APPEND-ITEM (LIST "~a" line)))
      ;;Scroll to top of this description.
      (ZL-USER::SEND scroll-window :scroll-absolute separate-at))))


(zl::make-instance 'zg:graph-type
	       :name 'dependency-structure-graph-type
	       :traversal-function 'dependency-graph-traverser
	       :default-root-finding-form '(find-initial-tms-node)
	       :instantiation-function 'Dependency-initiator
	       :vertex/edge-selection-handler 'tenv-mouse-handler	;;corrected this
	       :vertex/edge-description-function 'dependency-v/e-description
	       :vertex-print-string-function 'dependency-vertex-print-string
	       :edge-print-string-function 'dependency-edge-print-string)

