;;; -*- Mode: LISP; Syntax: Common-lisp; Base: 10; Fonts: CPTFONT,CPTFONTB; Package: ZG -*-

1;;Copyright (c) 1986 by John C. Hogge, The University of Illinois.
;;
;;File GRAPH.LISP of system Zgraph.

0(DEFFLAVOR graph (name
		  type
		  (root-vertices NIL)
		  (vertices NIL)
		  (edges NIL)
		  (location-clumps NIL)
		1  ;;how much we've zoomed
0		  (x-scale-factor 1.0)
		  (y-scale-factor 1.0)
		1  ;;how much we've panned 
0		  (x-displacement 0.0)
		  (y-displacement 0.0))
	   (SI:PROPERTY-LIST-MIXIN) 1;Used for storing misc. data for graph descriptions.
0  (:REQUIRED-INIT-KEYWORDS :type)
  :SETTABLE-INSTANCE-VARIABLES
  (:DOCUMENTATION :COMBINATION
   "1Stores graphs.  Instance variables:
TYPE: holds an instance of flavor GRAPH-TYPE, which is defined as our type of graph.
This is a required init option.

ROOT-VERTICES: list of root vertices, possibly generated by our graph type's
default-root-finding form.  But if this default is NIL, we are assumed to get0 1the roots on
our own.

VERTICES: list of VERTEX structs which comprise the graph.

EDGES: list of EDGE structs stored in VERTICES.  We store this for efficient access to the
edges.

LOCATION-CLUMPS: groups of vertices and edges located close to each other.0  1Specifically,
this is a list of the form:
  ((XMIN . YMIN) (XMAX . YMAX) VERTICES-IN-THIS-CLUMP EDGES-IN-THIS-CLUMP)
where the first two elements specify a world coordinate extent in which all vertex0 1and edge
structs in0 1the last two elements lie.  This data structure is used to speed up :DRAW
and the mouse-sensitivity code, since entire clumps can be ruled out0 1for drawing or selection
if their extent lies beyond the currently visible world0 1coordinates.

X-SCALE-FACTOR: modified during zooming.  Scales all graphics output that our output methods 
do.

Y-SCALE-FACTOR: ditto

X-DISPLACEMENT: modified during real panning.  Translates all graphics output that our0 1output
methods do.

Y-DISPLACEMENT: ditto0"))


(DEFMETHOD (graph :AFTER :INIT) (IGNORE)
  "1If no NAME is supplied0 1or is NIL, set it to contain TYPE and a unique number.
If ROOT-VERTICES is non-NIL, include them in the name for further identification.0"
  (UNLESS (AND (VARIABLE-BOUNDP name) name)
    (SETQ name
	  (IF root-vertices
	      1;;Be careful on length--graphs can have LOTS of roots.
0	      (LET ((root-string (FORMAT NIL "~S" root-vertices))
		    (genname (SEND type :genname)))
		(COND
		  ((> (STRING-LENGTH genname) 30.)
		   (IF (> (STRING-LENGTH genname) 40.)
		       (SUBSTRING genname 0 40.)
		       genname))
		  1;;10 is taken up by the text within the FORMAT string.
0		  ((> (+ (STRING-LENGTH genname) 10. (STRING-LENGTH root-string)) 40.)
		   (FORMAT NIL "~a..."
			   (SUBSTRING (FORMAT NIL "~a, roots = ~a" genname root-string)
				      0 37.)))
		  (T
		   (FORMAT NIL "~a, roots = ~a" genname root-string))))
	      (SEND type :genname)))))


(DEFMETHOD (graph :construct) ()
  (LET ((traversal-function (SEND type :traversal-function)))
    (UNLESS (FUNCTIONP traversal-function)
      (debug-print T "~%Warning: Graph Traversal Function ~s is not defined."))
    (IF (SEND type :traverse-recursively?)
	(SEND SELF :construct-recursively traversal-function)
	(SEND SELF :construct-non-recursively traversal-function))))


(DEFMETHOD (graph :construct-non-recursively) (traversal-function)
  (debug-print T "~%Computing graph from root vertices (non-recursively).  If this errs, ~
                    check traversal function ~s for ~S." traversal-function (SEND type :name))
  (SETQ traversal-function (get-compiled-function-or-die-trying traversal-function))
  1;;For each root vertex, create an entry in VERTICES of the form
0  1;;(<root-vertex> . <VERTEX struct to hold its data and edges>)
0  (LET (added-vertices)
    (LABELS
      ((apply-traversal-function (vertex)
	 (LET ((connections (FUNCALL traversal-function vertex))
	       (vstruct (make-vertex :location (CONS 0 0) :data vertex)))
	   (debug-print T "~%Edges from ~s = ~s" vertex connections)
	   1;;SETF required instead of initializing via :EDGE keyword because of
0	   1;;howky naming conflicts in MAKE-VERTEX.
0	   (SETF (vertex-edges vstruct)
		 (LET (outgoing-edges)
		   (DOLIST (edge connections)
		     1;;Consistency check.  If we generate any connections to vertices which 
0		     1;;are outside of the ROOT-VERTICES, ask user what to do about it. 
0		     1;;(Method :CONSTRUCT-RECURSIVELY would just apply TRAVERAL-FUNCTION 
0		     1;;recursively, here.)
0		     (UNLESS (OR (MEMQ (CAR edge) root-vertices)
				 (MEMQ (CAR edge) added-vertices))
		       (CERROR
			 "Apply ~s recursively on tail vertex ~2* ~s."
		       "Application of traversal function ~s on vertex ~s generated edge ~s~%~
                        but tail vertex ~s is not among the root vertices."
			 (SEND type :traversal-function) vertex edge (CAR edge))
		       (PUSH (CAR edge) added-vertices)
		       (apply-traversal-function (CAR edge)))
		     1;;Make edge, push it onto vertex's edge list, and
0		     1;;push it onto the graph's global edge list.
0		     (PUSH (make-edge :vertex-2 (CAR edge) :data (LIST (CDR edge)))
			   outgoing-edges)
		     (PUSH (CAR outgoing-edges) edges))
		   outgoing-edges))
	   (PUSH (CONS vertex vstruct) vertices))))
      (DOLIST (root root-vertices)
	(apply-traversal-function root))))
  1;;Flesh out the EDGE structs.
0  (DOLIST (entry vertices)
    (DOLIST (edge (vertex-edges (CDR entry)))
      (SETF (edge-vertex-1 edge) (CDR entry)
	    (edge-vertex-2 edge) (CDR (ASSOC (edge-vertex-2 edge) vertices :TEST #'EQ)))))
  1;;Convert VERTICES from temporary alist structure to a flat list.
0  (DO* ((vs vertices (CDR vs)))
       ((NULL vs))
    (SETF (CAR vs) (CDAR vs)))
  1;;Set the CONNECTED-TO fields to contain vertices at the tail of outgoing edges.
0  1;;CONNECTED-TO is an efficiency hack.
0  (DOLIST (vertex vertices)
    (SETF (vertex-connected-to vertex) (MAPCAR #'(LAMBDA (edge) (edge-vertex-2 edge))
					       (vertex-edges vertex))))
  1;;Now add the incoming connections.
0  (DOLIST (vertex vertices)
    (DOLIST (tail (vertex-connected-to vertex))
      (PUSHNEW vertex (vertex-connected-to tail))))

  (SEND SELF :merge-edges)  1;;Instead of doing this, we should merge up there.
0  (SEND SELF :eliminate-self-loops))


(DEFMETHOD (graph :construct-recursively) (traversal-function)
  (debug-print T "~%Computing graph from root vertices (recusively).  If this errs, check ~
                    traversal function ~s for ~s" traversal-function (SEND type :name))
  (SETQ traversal-function (get-compiled-function-or-die-trying traversal-function))
  1;;Do the traversal.  These three are declared special so that GROW-GRAPH can access them.
0  (LET ((visited-objects NIL)
	(collected-vertices NIL)
	(collected-edges NIL))
    (DECLARE (SPECIAL visited-objects collected-vertices collected-edges))
    1;;Construct the graph through a recursive traversal, starting with the root vertices.
0    (DOLIST (root root-vertices)
      (grow-graph root traversal-function))
    1;;Store the results of the calls to GROW-GRAPH in our instance variables.
0    (SETQ vertices collected-vertices
	  edges collected-edges))
  (SEND SELF :merge-edges)
  (SEND SELF :eliminate-self-loops))

(DEFUN grow-graph (object user-traversal-function)
  "1Unless we've been called with OBJECT as argument previously, build onto the graph by
adding edges leading from OBJECT.  OBJECT is a Lisp object in the user's datastructure.  
Initially this function is called with a user-designated root object.  A list of these objects
and the use of recursive calls is how we build the rest of the graph.  

If you are trying to deal with a bug in your graph traversal function, try evaluating the
following template:

  (LET (visited-objects
        collected-vertices
        collected-edges)
    (DECLARE (SPECIAL visited-objects collected-vertices collected-edges))  
    (grow-graph 'INSERT-ONE-OF-YOUR-ROOT-OBJECTS-HERE 'INSERT-YOUR-TRAVERSAL-FUNCTION-NAME)))0"
  (DECLARE (SPECIAL visited-objects collected-vertices collected-edges))
  (UNLESS (MEMQ object visited-objects)
    (PUSH object visited-objects)
    (LET ((connected-objects-alist (FUNCALL user-traversal-function object))
	  1;;Look for a precreated struct for OBJECT, which would have been created in a
0	  ;;1previous (recursive) call.
0	  1;;If there isn't one, make one and push it onto COLLECTED-VERTICES0.
	  (vertex-struct (find-or-make-vertex-struct object)))
      
      1;;Let user see the process of generating vertices, so that if his function fails,
0      1;;he'll have a clue.
0      (debug-print T "~%Edges from ~s = ~s" object connected-objects-alist)
      
      1;;Add any vertices reachable from OBJECT via an edge in the graph.
0      1;;We get an alist of these using the user-supplied graph traversal function.
0      (LOOP FOR (connected-object . edge-label) IN connected-objects-alist DO
	    (LET* ((connected-object-vertex-struct (find-or-make-vertex-struct
						     connected-object))
		   (new-edge (make-edge :vertex-1 vertex-struct
					:vertex-2 connected-object-vertex-struct
					1;;LISTed since one edge struct displays all
0					1;;relationships from one vertex to another.
0					:data (LIST edge-label))))
	      (SETF (vertex-edges vertex-struct)
		    (CONS new-edge (vertex-edges vertex-struct)))
	      (PUSH new-edge collected-edges)
	      (PUSHNEW connected-object-vertex-struct (vertex-connected-to vertex-struct))
	      (PUSHNEW vertex-struct (vertex-connected-to connected-object-vertex-struct)))
	    (grow-graph connected-object user-traversal-function)))))


(DEFUN find-or-make-vertex-struct (object)
  "1Look through special variable COLLECTED-VERTICES for a vertex struct for OBJECT.
If there is one, return it.  Otherwise create one and return it.0"
  (DECLARE (SPECIAL collected-vertices))
  (OR (FIND object collected-vertices
	    :TEST #'(LAMBDA (object struct) (EQ object (vertex-data struct))))
      (CAR (PUSH (make-vertex :location (CONS 0 0) :data object) collected-vertices))))


(DEFMETHOD (graph :merge-edges) ()
  1;;If more than one edge goes from one vertex to another, merge all of them into
0  1;;one composite edge.  This edge's data slot will hold a list of all the edges' data
0  1;;and these will be displayed together as the edge's label.  If this merging weren't
0  1;;done, the crossover minimization would produce bogus results at times, and these edge
0  1;;labels would overwrite each other.  NOTE: the success of this routine depends upon
0  1;;DELETE-DUPLICATES implemented by deleting the first arg to :TEST.
0  (DOLIST (vertex vertices)
    (SETF (vertex-edges vertex)
	  (DELETE-DUPLICATES (vertex-edges vertex)
			     :TEST #'(LAMBDA (possibly-edge-to-remove edge-to-keep)
				     1;;If VERTEX has two edges leading to the same vertex,
0				     1;;Merge them into one and throw away the other.
0				     (WHEN (EQ (edge-vertex-2 possibly-edge-to-remove)
					       (edge-vertex-2 edge-to-keep))
				       (SETF (edge-data edge-to-keep)
					     (NCONC (edge-data possibly-edge-to-remove)
						    (edge-data edge-to-keep)))
				       1;;Also delete from the local list of all edges.
0				       (SETQ edges (DELQ possibly-edge-to-remove edges))
				       T))))))

(DEFMETHOD (graph :eliminate-self-loops) ()
  1;;Zgraph doesn't bother displaying self loops (edges going from one vertex to itself)
0  1;;since it junks up the display on large graphs.  Instead, this method removes the
0  1;;self loops but stores them incase the user wants to see them.
0  (LET (self-loops)
    (DOLIST (vertex vertices)
      (WHEN (MEMQ vertex (vertex-connected-to vertex))	1;fast test
0	(SETF (vertex-connected-to vertex) (DELETE vertex (vertex-connected-to vertex))
	      (vertex-edges vertex) (DELETE-IF #'(LAMBDA (edge)
						   (WHEN (EQ (edge-vertex-2 edge) vertex)
						     (PUSH edge self-loops)
						     T))
					       (vertex-edges vertex)))))
    (SETQ edges (DELETE-IF #'(LAMBDA (edge)
			       (MEMQ edge self-loops))
			   edges))      
    (SEND SELF :PUTPROP self-loops :self-loops)))

(DEFUN line-degree (vertex)
  "1For the purposes of minimizing crossovers, LINE-DEGREE
is the number of connections to/from other non-leaf vertices.
Connections to leaves don't affect crossovers.  If VERTEX is
a leaf, we return a line-degree of NIL.0"
  (LET ((connected-to (vertex-connected-to vertex)))
    (WHEN (CDR connected-to)
      (LOOP FOR v IN connected-to
	    COUNTING (CDR (vertex-connected-to v))))))

(DEFMACRO non-leaf? (vertex)
  "1Returns non-NIL if VERTEX is a non-leaf of the graph.0"
  `(CDR (vertex-connected-to ,vertex)))

(DEFMACRO leaf? (vertex)
  "1Returns non-NIL if VERTEX is a leaf of the graph.0"
  `(NULL (non-leaf? ,vertex)))


(DEFMETHOD (graph :description) (&OPTIONAL (stream *STANDARD-OUTPUT*))
  "1Formats a description of the graph to STREAM.0"
  (LET ((number-of-vertices (LENGTH vertices)))
    1;;What else is useful?
0    (FORMAT stream "~a
 Graph Type: ~s
 ~s vertices, ~s directed edges
 Approx. number of clipped vertices = ~s
 Number of self-loops = ~s"
	    name
	    (SEND type :name)
	    number-of-vertices
	    (LENGTH edges)
	    (SEND SELF :GET :clipped-vertex-count)
	    (LENGTH (SEND SELF :GET :self-loops)))))



(DEFMETHOD (graph :plot-vertices) (&OPTIONAL (window *display-io*))
1  0"1Dispatch the current plotting style messages, unless there are no vertices.
WINDOW is an instance of ZG:GRAPH-DISPLAY-PANE on which the graph will be drawn.
It and its REAL-WINDOW's sizes dictate the initial scale of the plotted graph.

Also sets instance variable LOCATION-CLUMPS to a list of clumps of vertices0 1and edges of the
form
  ((XMIN . YMIN) (XMAX . YMAX) VERTICES-IN-THIS-CLUMP0 1EDGES-IN-THIS-CLUMP)0"
  (LET ((*display-io* window)
	(*graph-output* (SEND window :real-window)))
    (WHEN vertices
      (SEND SELF *graph-plotting-style*))
    (SEND SELF :plot-edges)
    (SETQ location-clumps
	  (clump-vertices-and-edges-by-location vertices edges))))

(DEFUN clump-vertices-and-edges-by-location (vertices edges
						      &OPTIONAL
						      (number-rows-of-clumps 10.0)
						      (number-columns-of-clumps 10.0))
  "1Clumps vertices0 1and edges into groups based on location.
First the world-coordinate extents of the vertices are calculated.  This provides
a rectangular boundary which is conceptually subdivided into 
NUMBER-ROWS-OF-CLUMPS x NUMBER-COLUMNS-OF-CLUMPS sectors.  VERTICES and EDGES0 1are placed
in the sector which covers their territory.  This is used to speed up0 1graph display (through
intelligent clipping operations) and mouse sensitivity.0"
  (WHEN vertices
    (MULTIPLE-VALUE-BIND (xmin ymin xmax ymax)
	1;;Note that we don't need to look at edges to calculate extents.  Just vertices.
0	(extents vertices) 
      (LET* ((clump-width (/ (- xmax xmin) number-rows-of-clumps))
	     (clump-height (/ (- ymax ymin) number-columns-of-clumps))
	     clumps)
	1;;If we have a one-dimensional extent {as with (NULL (CDR vertices))}, make it infinitely
0	1;;wide/high instead of 0.
0	(WHEN (ZEROP clump-width)
	  (SETQ clump-width most-positive-fixnum
		xmin (- (/ most-positive-fixnum 2.0))))
	(WHEN (ZEROP clump-height)
	  (SETQ clump-height most-positive-fixnum
		ymin (- (/ most-positive-fixnum 2.0))))
	(LABELS
	  1;;If there's already a clump made for location return it--otherwise make one.
0	  ((get-or-make-clump (location)
	     (OR (get-clump (CAR location) (CDR location) clumps)
		 1;;This computation is fairly complex.   Say the graph's world coordinates
0		 1;;start at -150 and our clump width/height are 100.  We want to generate
0		 1;;clumps of the form
0		 1;;(({-150+100i} . {-150+150j}) ({150+100i} . {150+150j}) {vertices} {edges}).
0		 1;;where the CAR is the minimum location in the clump, the CDR is the max 
0		 1;;location, which in this example is 100 more than the minimum 
0		 1;;coordinates, and {vertices} and {edges} locations' lie 
0		 1;;within the extents defined by the CAR and CDR.  An easier method is to 
0		 1;;generate all possible clumps, add vertices to them, then weed out any 
0		 1;;empty clumps, but that would cons more.
0		 (LET ((clump-xmin (+ xmin (* clump-width (TRUNCATE (- (CAR location) xmin)
								    clump-width))))
		       (clump-ymin (+ ymin (* clump-height (TRUNCATE (- (CDR location) ymin)
								     clump-height)))))
		   1;;Add and return the clump
0		   (CAR (PUSH (LIST (CONS clump-xmin clump-ymin)
				    (CONS (+ clump-xmin clump-width)
					  (+ clump-ymin clump-height))
				    NIL		1;Vertices
0				    NIL)	1;Edges
0			      clumps))))))
	  1;;Put all vertices into their clumps.
0	  (DOLIST (vertex vertices)
	    (PUSH vertex (THIRD (get-or-make-clump (vertex-location vertex)))))
	  (DOLIST (edge edges)
	    (PUSH edge (FOURTH (get-or-make-clump (edge-misc edge)))))
	  clumps)))))

(DEFUN extents (vertices)
  "1Returns the extents surrounding VERTICES0."
  (LET* ((xmax (CAR (vertex-location (CAR vertices))))
	 (ymax (CDR (vertex-location (CAR vertices))))
	 (xmin xmax)
	 (ymin ymax))
    (DOLIST (vertex (CDR vertices))
      (LET ((location (vertex-location vertex)))
	(IF (> (CAR location) xmax)
	    (SETQ xmax (CAR location))
	    (IF (< (CAR location) xmin)
		(SETQ xmin (CAR location))))
	(IF (> (CDR location) ymax)
	    (SETQ ymax (CDR location))
	    (IF (< (CDR location) ymin)
		(SETQ ymin (CDR location))))))
    (VALUES xmin ymin xmax ymax)))


(DEFUN get-clump (x y clumps)
  "1Returns the0 1clump0 1in CLUMPS representing a group of graphics objects whose combined
extents location0 1X,Y lies within.0"
  (FIND-IF #'(LAMBDA (clump)
	       (AND (<= (CAAR clump) x (CAADR clump))
		    (<= (CDAR clump) y (CDADR clump))))
	   clumps))

(DEFMETHOD (graph :plot-edges) ()
  "1Calculates locations for the directional pointers of edges, for use in mouse-sensitivity.0"
  (DOLIST (edge edges)
    (LET* ((from-point (vertex-location (edge-vertex-1 edge)))
	   (to-point (vertex-location (edge-vertex-2 edge))))
      (SETF (edge-misc edge)
	    (CONS (value-between (CAR from-point) (CAR to-point) .9)
		  (value-between (CDR from-point) (CDR to-point) .9))))))


1;;
;; First method for plotting graphs: :PLOT-IN-ONE-BIG-CIRCLE
;;

0(DEFUN decent-radius (number-of-non-leaf-vertices)
  "1Returns a good radius to use for arranging a given number of vertices and
edges in a circle.0"
1  ;;Short edges look bad, so we impose a minimum edge length.
  ;;CIRCUMFERENCE  = 2 * PI * RADIUS so roughly
  ;;RADIUS = NUMBER-OF-VERTICES * {minimum edge length} / (2 * PI)
0  (LET ((radius (/ (* number-of-non-leaf-vertices 500.) *2PI*)))
1    ;;This takes into account the fact that :ARRANGE-VERTICES-IN-CIRCLE cuts
    ;;out a portion of the circle for the vertex leaves.
0    (+ radius (* radius *percentage-radius-for-leaves-in-circular-arrangement*))))


(DEFMETHOD (graph :plot-in-one-big-circle) ()
  "1This graph plotting method places non-leaf vertices of the graph in a circle.
A circular arrangement won't always be optimum (wrt. crossovers), but it buys us several 
things.  First, we can compute the number of crossovers symbolicly, so our computation in
reducing crossovers is alot faster.  Second, it isn't clear that a tangled planar graph with
few crossovers is easier to read than a graph arranged symetrically in a circle with more 
crossovers.  Third, it's much easier to compute a circle of vertices, which is inherently a
reasonable layout, than to compute a tangle of vertices which happens to come out planar or 
close-to-planar.  Typically whenever I have to draw a graph for some reason, it's usually
easiest to lay out and read the edges of the graph by placing the vertices in a circle.

The leaves of the graph are placed as sattelites to the circle around the vertices to which
they are connected.0"
  (LET (non-leaves)
    1;;Extract the leaves of the graph.
0    (DOLIST (vertex vertices)
      (WHEN (non-leaf? vertex)
	(PUSH vertex non-leaves)))
    1;;Add0 1to0 1NON-LEAVES any leaf that has no connection with NON-LEAVES.  If we don't do 
0    1;;this, isolated trees will not be plotted.
0    (DOLIST (vertex vertices)
      (WHEN (leaf? vertex)
	(UNLESS (SOME #'(LAMBDA (connected-to) (MEMQ connected-to non-leaves))
		      (vertex-connected-to vertex))
	  (PUSH vertex non-leaves))))
    1;;Reorder NON-LEAVES, which will be displayed in a circle, so as to minimize crossovers.
0    (SETQ non-leaves (SEND SELF :minimize-crossovers-for-circular-arrangement non-leaves))
    1;;Assign NON-LEAVES positions around the circle.
0    (LET ((radius (decent-radius (LENGTH non-leaves))))
      (SEND SELF :arrange-vertices-in-circle non-leaves 0 0 radius radius))
    1;;Scale so user sees at least some of the graph.
0    (SEND SELF :scale-for-initial-viewing)))


(DEFMETHOD (graph :scale-for-initial-viewing) (&OPTIONAL
						(real-window *graph-output*)
						(display-window *display-io*))
  "1Sets the graph scale so that the user initially sees at least a part of the graph.
If there are more than a globally specified number of vertices, fits the graph into the 
output window;0 1otherwise, fits it into the display pane.  If we don't do this, many graphs
will appear completely outside both the output window and the display pane.0"
  (MULTIPLE-VALUE-BIND (width height)
      (COND
	((> (LENGTH vertices)
	    *number-vertices-over-which-graphs-are-fit-onto-hidden-bit-array*)
	 1;;Not enough screen space!
0	 (debug-print T "~%Fitting graph onto hidden bit array.  Pan to see portions not displayed on the window.")
	 (SEND real-window :INSIDE-SIZE))
	(T
	 (debug-print T "~%Fitting graph onto the display pane--all vertices will be visible.")
	 (SEND display-window :INSIDE-SIZE)))
    1;;If graph only has two vertices (connected by an edge usually), it looks better if we
0    1;;let :SCALE-TO-FIT-WINDOW stretch one of the dimensions so as to lengthen the connecting
0    1;;edge to fit snuggly into the window.  This is what the third argument specifies.
0    1;;All other cases don't have this problem, since more than two vertices arranged in a
0    1;;circle fit "snuggly" into a roughly square area (the window). 
0    (SEND SELF :scale-to-fit-window width height (= (LENGTH vertices) 2))))





1;;
;; Second (and default) method for plotting graphs: 
;; :PLOT-IN-CIRCLES-FOR-BI-CONNECTED-COMPONENTS
;;

0(DEFMETHOD (graph :plot-in-circles-for-bi-connected-components) ()
1  0"1This graph plotting method places certain bi-connected components of the graph in their 
own circle.  Bi-connected components of graphs are those subgraphs which are connected such 
that there is more than one path between each vertex in the subgraph.  In otherwords, 
bi-connected components of a graph are each connected via only one edge.  This makes a good 
grouping (perceptually) of the graph into subgraphs (with both connected and disconnected
components, where any two subgraphs are connected by at most one edge, by the nature of
bi-connected components).  There are undoubtedly other good groupings, such as 
number-of-connections under some threshold.  However, I chose this grouping because of the
availability of a fast algorithm for determining bi-connected components.

Final note: certain bi-connected components are combined into one subgraph for ease of 
handling.  These are those bi-connected components which share one vertex (rather than being
connected by one edge).0"

1  ;;This is how it works:
  ;;1. Generate the initial set of subgraphs from the bi-connected components of the graph.
  ;;   Merge some of these subgraphs together as described above (and below).
  ;;   Remove all edges connecting subgraphs from the graph, storing them for later addition.
  ;;2. Arrange each subgraph in its circle such that crossovers within the
  ;;   subgraph are minimized.  (The circular placements are figured at this point--not
  ;;   the actual coordinates of vertices.)
  ;;3. Add the edges connecting subgraphs back into the graph.
  ;;4. Use these edges to figure which subgraphs are connected together
  ;;   and a good screen area for each component, using a hexagonal grid arrangement.
0  (MULTIPLE-VALUE-BIND (subgraphs connecting-edges)
      (extract-cycles vertices)

    1;;Step 1.

    ;;Create subgraphs for each connecting edge vertex which is in a tree.  For an example,
    ;;assume the following: edge (v6 . v12)0 1is in the connecting edges.  V6 is in a 
0    1;;bi-connected component.  V12 is not a leaf and is NOT in a bi-connected component 
0    1;;(meaning it is part of a tree). We still want to remove (v6 . v12) as part of Step 1,
0    1;;but since V12 isn't a leaf, we need to include it (and non-leaves attached to it) in a
    ;; new subgraph.
0    (LABELS ((tree-vertex? (vertex)
	       (AND (non-leaf? vertex) (NOT (SOME #'(LAMBDA (subgraph) (MEMQ vertex subgraph))
						  subgraphs)))))
	(LET (tree-vertices tree-subgraphs)
	  (LOOP FOR edge IN connecting-edges
		WHEN (tree-vertex? (CAR edge))
		DO (PUSHNEW (CAR edge) tree-vertices)
		WHEN (tree-vertex? (CDR edge))
		DO (PUSHNEW (CDR edge) tree-vertices))
	  1;;Start out with a tree for each vertex and its immediate connected vertices which
0	1  ;;are among TREE-VERTICES (otherwise vertices in SUBGRAPHS  and leaf vertices would
0	  1;;be included). Then destructively merge trees which share vertices, using NCOMBINE.
0	  (SETQ tree-subgraphs
		(ncombine
		  (LOOP FOR vertex IN tree-vertices
			COLLECT (CONS vertex
				      (LOOP FOR connected-to IN (vertex-connected-to vertex)
					    WHEN (MEMQ connected-to tree-vertices)
					    COLLECT connected-to)))
		  :TEST #'EQ))
	1  ;;Since this system doesn't handle trees specially, a quick way to have trees
0	  1;;processed as if they were normal subgraphs (where all non-leaf vertices have
0	  1;;non-NIL line-degree) is to double up one of each tree element's connected-to's,
0	  1;;so that they are processed as non-leaves o.k. (they get a non-NIL line-degree).
0	  1;;(The one we double up has to be a non-leaf.) Of course, this is just a way of
0	  1;;quickly handling the problem, and the correct way is to print each of these trees
0	  1;;in tree format, rather than doing the NCONC.  This would be fairly easy to fit
0	  1;;in: allocate all trees space underneath the subgraph circles and print them
0	  1;;downwards.
0	  (DOLIST (subgraph tree-subgraphs)
	    (DOLIST (vertex subgraph)
	      (DOLIST (v (vertex-connected-to vertex))
		 (WHEN (non-leaf? v)
		   (PUSH v (vertex-connected-to vertex))
		   (RETURN NIL)))))
	  1;;Delete edges making up the trees from CONNECTING-EDGES, similar to when we
0	  1;;deleted leaves from CONNECTING-EDGES above.  If we don't do this, edges of the
0	  1;;tree will be treated as subgraph connectors in code below.
0	  (SETQ connecting-edges
		(DELETE-IF #'(LAMBDA (edge) (AND (MEMQ (CAR edge) tree-vertices)
					       (MEMQ (CDR edge) tree-vertices)))
					    connecting-edges))
	  (SETQ subgraphs (NCONC subgraphs tree-subgraphs))))

1    ;;Create a subgraph to hold any leaf that has no connection into a current subgraph.
    ;;If we don't do this, isolated trees will not be plotted.
0    (LET (stray-leaves)
      (DOLIST (vertex vertices)
	(WHEN (leaf? vertex)
	  (UNLESS (SOME #'(LAMBDA (connected-to)
			  (SOME #'(LAMBDA (subgraph)
				  (MEMQ connected-to subgraph))
				subgraphs))
			(vertex-connected-to vertex))
	    (PUSH vertex stray-leaves))))
      (WHEN stray-leaves
	(PUSH stray-leaves subgraphs)))

    1;;We only want to remove edges connecting subgraphs, so that each subgraph can be plotted
0    1;;in its own circle (using :MINIMIZE-CROSSOVERS-FOR-CIRCULAR-ARRANGEMENT).  Therefore,
0    1;;we save edges leading to/from graph leaves from the edges we are about to remove.
0    (SETQ connecting-edges
	  (DELETE-IF-NOT #'(LAMBDA (edge) (AND (non-leaf? (CAR edge)) (non-leaf? (CDR edge))))
			 connecting-edges))

    1;;Temporarily remove all subgraph-connecting edges from the graph.
0    (DOLIST (edge connecting-edges)
      (LET ((v1 (CAR edge))
	    (v2 (CDR edge)))
	1;;1 is efficient & important to the above kludge. 
0	(SETF (vertex-connected-to v1) (DELQ v2 (vertex-connected-to v1) 1)
	      (vertex-connected-to v2) (DELQ v1 (vertex-connected-to v2) 1))))
      
1    ;;Combine any subgraphs (bi-connected components) which share a vertex.  For instance,
    ;;in the graph represented by
    ;;  (SETQ v1 '(v2 v3 v4) v2 '(v1 v3) v3 '(v1 v2)
    ;;        v4 '(v1 v5) v5 '(v4 v1))
    ;;V1 is shared by the (v1 v2 v3) bi-connected component and by the (v4 v5 v6)
    ;;bi-connected component. We want to combine these two connected components into one
    ;;circle; otherwise we'd have to share V1 between them, either by placing V1 arbitarily
    ;;in one or the other and connecting the two circles by more than one edge.
    ;;In case you aren't up on your bi-connected component theory, the following is a graph
    ;;with two bcc's which we *do* want display in two circles:
    ;;  (SETQ v1 '(v2 v3 v4) v2 '(v1 v3) v3 '(v1 v2)
    ;;        v4 '(v1 v5 v6) v5 '(v4 v6) v6 '(v4 v5)
0    (SETQ subgraphs (ncombine subgraphs :TEST #'EQ))

1    ;;Step #2.
0    (LOOP FOR sgraphs ON subgraphs DO
	  1;;Reorder NON-LEAVES, which will be displayed in a circle, so as to minimize
0	  1;;crossovers.
0	  (SETF (CAR sgraphs)
		(SEND SELF :minimize-crossovers-for-circular-arrangement (CAR sgraphs))))

    1;;Step #3. Add back in the edges we removed from the graph in step 1.
    ;;This might not actually be needed for anything, but do it for any future changes
    ;;that depend on it.
0    (DOLIST (edge connecting-edges)
      (LET ((v1 (CAR edge))
	    (v2 (CDR edge)))
	(SETF (vertex-connected-to v1) (CONS v2 (vertex-connected-to v1))
	      (vertex-connected-to v2) (CONS v1 (vertex-connected-to v2)))))
    1;;Step #4.
0    (SEND SELF :place-subgraphs-on-a-hexagonal-grid subgraphs connecting-edges)
    1;;Scale so user sees at least some of the graph.
0    (SEND SELF :scale-for-initial-viewing)))


(DEFMETHOD (graph :place-subgraphs-on-a-hexagonal-grid) (subgraphs connecting-edges)
  1;;Use the set of connecting edges to figure out how to arrange the subgraphs so as to
0  1;;minimize crossovers.  Since we're using circular areas for each component of the graph,
0  1;;the optimum arrangement is a hexagonal grid.  (Picture a bunch of coins crowded
0  1;;together on a table.)  So first we find a good arrangement of the components in a grid,
0  1;;then rotate each component's circle to reduce crossovers caused by edges between
0  1;;components.
0  (LET* ((radius (decent-radius (LOOP FOR component IN subgraphs
				      MAXIMIZE (LENGTH component))))
	 most-popular
	 (most-connections 0))
    1;;Modify SUBGRAPHS to consist of elements
0    1;;  (<subgraph> (<connected-to> <vertex> <vertex-to>)...)
0    1;;where <connected-to> is a subgraph connected to <subgraph> through and edge at
0    1;;<vertex> and <vertex-to>, where <vertex> is in <subgraph> and <vertex-to> is in
0    1;;<connected-to>.
0    (LOOP FOR sgraphs ON subgraphs
	  FOR subgraph = (CAR sgraphs)
	  DO
      (SETF (CAR sgraphs)
	    (CONS subgraph
		  (LOOP FOR connecting-edge IN connecting-edges
			FOR our-edge = (IF (MEMQ (CAR connecting-edge) subgraph)
					   (LIST (CAR connecting-edge)
						 (CDR connecting-edge))
					   (IF (MEMQ (CDR connecting-edge) subgraph)
					       (LIST (CDR connecting-edge)
						     (CAR connecting-edge))))
			WHEN our-edge
			  1;;(CADR our-edge) = <vertex-to>
0			  COLLECT
			    (CONS (SOME
				    #'(LAMBDA (connected-to?)
				      1;;Kludge.  In this loop we're modifying the 
0				      1;;elements of SUBGRAPHS, so any comparisons have
0				      1;;to be made against both the old and new format
0				      1;;of the list.  All to save consing.
0				      (IF (LISTP (CAR connected-to?))
					  (WHEN (MEMQ (CADR our-edge) (CAR connected-to?))
					    (CAR connected-to?))
					  (WHEN (MEMQ (CADR our-edge) connected-to?)
					    connected-to?)))
				    subgraphs)
				  our-edge)))))
    1;;Pick the subgraph having most connections to other subgraphs for the middle 
0    1;;grid location. If there are no connections (as in a connected graph or where each
0    1;;subgraph is disconnected),0 1pick the subgraph with the most vertices.
0    1;;We'll arrange the subgraphs it is connected to in adjacent locations.
0    (DOLIST (entry subgraphs)
      (LET ((count (LENGTH (CDR entry))))
	(WHEN (< most-connections count)
	  (SETQ most-popular entry
		most-connections count))))
    (UNLESS most-popular
      (DOLIST (entry subgraphs)
	(LET ((count (LENGTH (CAR entry))))
	  (WHEN (< most-connections count)
	    (SETQ most-popular entry
		  most-connections count)))))
    1;;Put the most popular at the front of SUBGRAPHS.
0    (SETQ subgraphs (CONS most-popular (DELETE most-popular subgraphs :TEST #'EQ)))
    1;;Make it the origin.
0    (SETF (CDR most-popular) `(0 0 . ,(CDR most-popular)))
    
    1;;Make everyone else's coordinates unassigned.
0    1;;This makes SUBGRAPHS consist of elements
0    1;;(<subgraph> <x> <y> (<connected-to> <vertex> <vertex-to>)...)
0    (DOLIST (entry (CDR subgraphs))
      (SETF (CDR entry) `(NIL NIL . ,(CDR entry))))
    
    1;;Assign relatively good coordinates to every subgraph.
0    (DOLIST (entry subgraphs)
      (WHEN (CADR entry)
	(impose-ones-will-on-location-of-other-subgraphs entry subgraphs)))
    1;;Fit isolated subgraphs in somewhere.
0    (assign-locations-near-origin-to-isolated-subgraphs subgraphs)
    1;;Hexagonalize the grid.  x := x + .5 * y;  y := y * {circlular constant}
0    (DOLIST (entry subgraphs)
      (MULTIPLE-VALUE-BIND (x y)
	  (hexagonal-equivalents (CADR entry) (CADDR entry))
	(SETF (CADR entry) x)
	(SETF (CADDR entry) y)))
    
    (LOOP FOR (subgraph x y . NIL) IN (CONS most-popular subgraphs)
	  WITH diameter = (* 2 radius)
	  FOR xmin =  (* diameter x)
	  FOR ymin =  (* diameter y)
	  DO (SEND SELF :arrange-vertices-in-circle
		   subgraph xmin ymin (+ xmin diameter) (+ ymin diameter)))))


(DEFUN hexagonal-equivalents (x y)
  1;;Note that the grid y values have to be reversed in translating the grid into
0  1;;world coordinates, in which low values of Y occur at the top of the screen.
0  1;;Also, .10 is an approximation which seems to work.  The real value is
0  1;;(RADIUS - 1/2 * (2 * RADIUS * SIN (180/6) * COT (180/6))) / RADIUS, as
0  1;;given by CRC manual Page 10.
0  (VALUES (+ x  (* .5 y))
	  1;;Same as (- (- y (* .10 y)))
0	  (- (* .10 y) y)))

(DEFVAR *neighbors-on-either-side-of-direction*
	     '(((-1 1) (-1 0) (0 1))  1;;These are in a specific order!!
0	       ((0 1) (-1 1) (1 0))
	       ((1 0) (0 1) (1 -1))
	       ((1 -1) (1 0) (0 -1))
	       ((0 -1) (1 -1) (-1 0))
	       ((-1 0) (0 -1) (-1 1)))
  "1In normal cardinal directions, North and West are the neighboring directions to North-West.
In this0 1hexagonal grid, the relationships are similar.0")

(DEFUN impose-ones-will-on-location-of-other-subgraphs (boss-subgraph subgraphs)
  (LET ((boss (CAR boss-subgraph)))
    (DOLIST (subgraph subgraphs)
      1;;When connected to boss and no coordinates have been assigned, let boss assign them.
0      (LET ((connection (ASSQ (CAR subgraph) (CDDDR boss-subgraph))))
	(WHEN (AND connection (NULL (CADR subgraph)) (NEQ boss (CAR subgraph)))
	  (LET* ((suggested-direction (vertex-direction-from-center boss (CADR connection)))
		 (opposite-direction (LIST (* (CAR suggested-direction) -1)
					   (* (CADR suggested-direction) -1))))
	    (MULTIPLE-VALUE-BIND (x y)
		(find-free-location
		  (CADR boss-subgraph) (CADDR boss-subgraph)
		  (CAR suggested-direction) (CADR suggested-direction) subgraphs)
	      1;;Set the connected subgraph's location to some closest free location in the
0	1      ;;suggested direction.
0 	      (SETF (CADR subgraph) x
		    (CADDR subgraph) y)
	      1;;Rotate the connected subgraph till the connecting vertex faces the boss.
0	      (LOOP FOR count FROM 0
		    WITH length = (LENGTH (CAR subgraph))
		    1;;When we've tried all rotations, we're unlucky about the selected
0		1    ;;opposite direction.  (This will happen on subgraphs consisting of fewer
0		    1;;than 6 vertices.)  So try the direction next to the selected one.
0		    WHEN (= count length)
		    1;;Used Zetalisp ASSOC since Common Lisp one wasn't compiling right
0		1    ;;(despite use of :KEY #'EQUAL).
0		    DO (SETQ opposite-direction
			     (CADR
#+Symbolics
			       (SI:ASSOC opposite-direction
					 *neighbors-on-either-side-of-direction*)
#+Explorer
			       (ASSOC opposite-direction
				      *neighbors-on-either-side-of-direction*
				      :TEST #'EQUAL))
			     count -1)
		    UNTIL (EQUAL opposite-direction
				 (vertex-direction-from-center
				   (CAR subgraph) (CADDR connection)))
		    DO (LET ((old (CAR subgraph))
			     (new (NCONC (CDAR subgraph) (NCONS (CAAR subgraph)))))
			 (DOLIST (subgraph subgraphs)
			   (LOOP FOR connected-tos ON (CDDDR subgraph)
				 WHEN (EQ (CAR connected-tos) old)
				 DO (SETF (CAR connected-tos) new)))
			 (SETF (CAR subgraph) new))))))))))

(DEFUN vertex-direction-from-center (subgraph vertex)
  "1Returns qualitative description of the direction to VERTEX from the center of SUBGRAPH,
where VERTEX is in SUBGRAPH and SUBGRAPH is ordered so that the CAR will be assigned *3/2PI* 
degrees by :ARRANGE-VERTICES-IN-CIRCLE, and the rest are assigned increasing positive 
increments around the circle.  Returns a list (x y) where x and y are 0, 1 or, -1.  x= 1 
means Right, x= -1 means Left, y= 1 means up, y= -1 means down.  0 means no change in 
direction.0"
  (LET* ((length (LENGTH subgraph))
	 (percentage-of-circle-to-vertex (/ (- length (LENGTH (MEMQ vertex subgraph)))
					    length)))
1    ;;The angles used are based on the assumption that we will hexagonalize (shift)
0    1;;rightwards (any given row slides 1/2radius to the right over the row below it).
0    1;;Picture (or lay out) a bunch of coins on a table.  Pick one in the middle as the
0    1;;subgraph.  Then you'll see that the angles below correspond (roughly) to what you see
0    1;;on the table.  For instance, there are 6 (not 8) cardinal directions from the coin.
0    (COND
1      ;;First 60 degrees = coin above and to the left of center coin
0      ((<= percentage-of-circle-to-vertex 60/360) '(0 1))
1      ;;First 60 degrees = coin to the left.
0      ((<= percentage-of-circle-to-vertex 120/360) '(1 0))
1      ;;First 60 degrees = coin down and to the left.
0      ((<= percentage-of-circle-to-vertex 180/360) '(1 -1))
1      ;;First 60 degrees = coin down and to the right.
0      ((<= percentage-of-circle-to-vertex 240/360) '(0 -1))
1      ;;First 60 degrees = coin to the right.
0      ((<= percentage-of-circle-to-vertex 300/360) '(-1 0))
1      ;;First 60 degrees = coin up and to the right.
0      (T '(-1 1)))))


(DEFUN find-free-location (from-x from-y direction-x direction-y subgraphs)
  "1Returns a free location0 1close to FROM-X, FROM-Y in direction DIRECTION-X, DIRECTION-Y.
The direction values are -1, 0, and 1.0"
  (LET* ((neighboring-directions (CDR (ASSOC (LIST direction-x direction-y)
					     *neighbors-on-either-side-of-direction*
					     :TEST #'EQUAL)))
	 (first-neighbor (CAR neighboring-directions))
	 (second-neighbor (CADR neighboring-directions))
	 (location-queue (LIST (CONS (+ from-x direction-x) (+ from-y direction-y)))))
    (LOOP FOR x = (CAAR location-queue)
	  FOR y = (CDAR location-queue) DO
      (COND
	((location-not-free? x y subgraphs)
	 1;;Breadth-first search until a location is found.  We want to use breadth-first
0	 1;;so that we get the closest location with a 60 degree arc from the desired
0	 1;;direction.
0	 (NCONC location-queue
		(LIST (CONS (+ x direction-x) (+ y direction-y))
		      (CONS (+ x (CAR first-neighbor)) (+ y (CADR first-neighbor)))
		      (CONS (+ x (CAR second-neighbor)) (+ y (CADR second-neighbor)))))
	 (POP location-queue))
	1;;Location is free--return it.
0	(T
	 (RETURN (VALUES (CAAR location-queue) (CDAR location-queue))))))))


(DEFUN assign-locations-near-origin-to-isolated-subgraphs (subgraphs)
  (LET ((all-subgraphs subgraphs)) 1;;store head of list.
0    (LABELS
      ((next-isolated-subgraph ()
	 1;;CDR down to next subgraph which hasn't been assigned coordinates.
0	 1;;When we reach the end of the list, we've finished the assignment so exit function.
0	 (DO ((subgraph (POP subgraphs) (POP subgraphs)))
	     (NIL)
	   (IF subgraph
	       (UNLESS (CADR subgraph)
		 (RETURN subgraph))
	       (RETURN-FROM assign-locations-near-origin-to-isolated-subgraphs NIL))))
       (assign-if-location-free (x y)
	 (UNLESS (location-not-free? x y all-subgraphs)
	   (LET ((isolated-subgraph (next-isolated-subgraph)))
	     (SETF (CADR isolated-subgraph) x
		   (CADDR isolated-subgraph) y)))))
      1;;Span out from origin in progressively larger hexagons.
0      (DO* ((n 1 (1+ n))
	    (negative-n (- n) (- n)))
	   (NIL)
	1;;Top hex side
0	(DOTIMES (i (1+ n))
	  (assign-if-location-free (- i) n))
	1;;Bottom hex side
0	(DOTIMES (i (1+ n))
	  (assign-if-location-free i negative-n))
	1;;Upper left hex side
0	(DOTIMES (i n)
	  (assign-if-location-free negative-n i))
	1;;Lower right hex side
0	(DOTIMES (i n)
	  (assign-if-location-free n (- i)))
	1;;Lower left hex side
0	(LOOP FOR i FROM 1 TO (1- n) DO
	  (assign-if-location-free (+ negative-n i) (- i)))
	1;;Upper right hex side
0	(LOOP FOR i FROM 1 TO (1- n) DO
	  (assign-if-location-free (- n i) i))))))


(DEFUN location-not-free? (x y subgraphs)
  "1Returns NIL if location x,y isn't already taken by a subgraph; otherwise, returns the 
lucky subgraph.0"
  (SOME #'(LAMBDA (subgraph)
	  (WHEN (AND (CADR subgraph) (= (CADR subgraph) x) (= (CADDR subgraph) y))
	    subgraph))
	subgraphs))



(DEFUN ncombine (lists &KEY (test #'EQL))
  "1Merges sublists of LISTS which share any elements, compared using TEST.
LISTS is munged destructively.
Example: (ncombine '((1 2 3 4) (2 5) (6 7 8))) => '((1 2 3 4 2 5) (6 7 8))0"
  (LET ((merged-any? T))
    (LOOP WHILE merged-any? DO
	  (SETQ merged-any? NIL)
	  (LOOP FOR slow-scanner ON lists DO
		(LOOP FOR fast-scanner ON (CDR slow-scanner) DO
		1      ;;Need to use a non-consing INTERSECTP here.
0		      (WHEN (INTERSECTION (CAR slow-scanner) (CAR fast-scanner) :TEST test)
			(SETQ merged-any? T)
			(NCONC (CAR slow-scanner)
			       (NSET-DIFFERENCE (CAR fast-scanner) (CAR slow-scanner)
						:TEST test))
			1;;(CAR slow-scanner) has been merged with (CAR fast-scanner), so 
0			1;;setting it to NIL0 1takes it out of play.
0			(SETF (CAR fast-scanner) NIL))))))
1  0(DELETE-IF-NOT #'IDENTITY lists))



1;;==========================================================================================
;; Brian Falkenhainer's code for plotting lattices.  
;; ****Ultimately, lattice detection within circles should be automated and this
;; used.
;; ****Loops infinitely on non-lattices.

0(DEFMETHOD (graph :plot-lattice) ()
  (LET ((intercolumn-spacing 100)
	(interrow-spacing 30.0)
	(mark (list nil))
	roots lattice tmp)
    1;;Extract the roots of the graph.
0    (DOLIST (vertex vertices)
      (WHEN (member (vertex-data vertex) root-vertices :test #'eq)
	(setf (vertex-misc vertex) (cons mark 0))1      ;we've seen it and it's on level 0
0	(PUSH vertex roots)))
    (setq roots (nreverse roots))
    1;;Do placement
0    (setq lattice (list (cons 0 roots)))
    (do ((i 1 (1+ i))
	 (level roots next-level)
	 (next-level nil nil)			1;next-level is collected while stepping through this level
0	 (phantom-bit nil nil)			1;phantoms fade away with successive levels.
0	 (all-phantoms?)			1;finished when level is all phantoms.
0	 (children))
	(all-phantoms?)
      (setq all-phantoms? t)
      (dolist (current-node level)
	(cond ((eq current-node 'phantom)
	       (if phantom-bit (push 'Phantom next-level))   1;2 phantoms in a row -> one on next level
0	       (setq phantom-bit (not phantom-bit)))
	      ((setq children (nreverse (mapcar #'edge-vertex-2 (vertex-edges current-node))))
	       (setq phantom-bit nil)
	       (setq all-phantoms? nil)
	       (dolist (c-node children)
		 (cond ((eq (car (vertex-misc c-node)) mark)
			(cond ((/= (cdr (vertex-misc c-node)) i)
				1;; remove the node from the higher level and move it down to this level
0			       (setq tmp (assoc (cdr (vertex-misc c-node)) lattice))
			       (setf (cdr tmp)  (delete c-node (cdr tmp)))
			       (setf (cdr (vertex-misc c-node)) i)
			       (push c-node next-level))))
		       (t (setf (vertex-misc c-node) (cons mark i))
			  (push c-node next-level)))))
	      ((push 'Phantom next-level))))	1;parent has no children, insert phantom
0      (setq next-level (nreverse next-level))
      (unless all-phantoms? (setq lattice (nconc lattice (list (cons i next-level))))))

    (do ((lattice (cdr lattice) (cdr lattice))
	 (current-level (cdar lattice) (cdar lattice))
	 (current-x 0 (+ current-x intercolumn-spacing))
	 current-y)
	((null current-level))
      (cond ((> (* (length current-level) interrow-spacing) 1000)
	     (setq current-y 0)
	     (setq interrow-spacing (/ 1000.0 (length current-level))))
	    ((setq current-y (- 500 (* (/ (length current-level) 2.0) interrow-spacing)))))
      (dolist (v current-level)
	(if (not (eq v 'phantom))
	    (setf (vertex-location v) (cons current-x current-y)))
	(incf current-y interrow-spacing)))

    (SEND SELF :scale-for-initial-viewing)))

1;;==========================================================================================
;;
;;;; Arranging vertices in a circle so as to minimize crossovers.
;;
;;==========================================================================================

0(DEFMETHOD (graph :minimize-crossovers-for-circular-arrangement) (non-leaves)
  "1Reorders the vertices in NON-LEAVES so as to minimize crossovers when they are 
arranged (displayed) in a circle.  The optimal solution isn't always made--we go for speed
instead.0"  ;;Example where it isn't optimum: 4th Sample graph type, big-circle method.
  (COND
1    ;;No minimization needed if less than four vertices.
0    ((< (LENGTH non-leaves) 4)
     non-leaves)
    1;;Graph is too huge, so use a quicker algorithm (still experimental).
0    ((OR
       (>= (LENGTH non-leaves) *too-many-non-leaves*)
       (>= (LOOP FOR vertex IN non-leaves
		 SUMMING (LENGTH (vertex-edges vertex)))
	   *too-many-edges*))
     (SEND SELF :minimize-crossovers-for-large-graphs non-leaves))
    (T
     (LET ((stepping? (MEMQ :step-through-placement *graph-debug-actions*)))
       (debug-print T "~%Minimizing crossovers in a circular arrangement of vertices...")
       1;;Sort the non-leaves by decreasing line degree.  The idea is to 
0       1;;start with a circle of large-degree non-leaves, then to keep adding
0       1;;smaller-degree non-leaves where they will cause the least crossovers.
0       (SETQ non-leaves
	     1;;OR is used incase leaves occur in NON-LEAVES.
0	     (SORT non-leaves #'(LAMBDA (x y) (> (OR (line-degree x) 0)
						 (OR (line-degree y) 0)))))
       1;;Build up the circular arrangement in CIRCLE.
0       1;;We use a circular list for easy handling.
0       (LET ((circle (CIRCULAR-LIST (POP non-leaves) (POP non-leaves))))
	 (DOLIST (non-leaf non-leaves)
	   (WHEN stepping?
	     (SEND SELF :display-step (remove-list-circularity (copy-circular-list circle))
		   "~%Stepping through placement algorithm.   Added a non-leaf."))
	   (LET ((best-insertion-point NIL)
		 (least-crossovers 999999999999))
	     1;;Sees if inserting NON-LEAF at the CDR of POSITION produces a minimum of
0	     1;;crossovers (so far).
0	     (LABELS ((try-insertion (non-leaf position)
			(LET ((crossovers (count-crossovers-after-insertion
					    non-leaf position)))
			  (WHEN (< crossovers least-crossovers)
			    (SETQ least-crossovers crossovers
				  best-insertion-point position)))))
	       (LOOP FOR scanner ON circle
		     WHEN (MEMQ (CADR scanner) (vertex-connected-to non-leaf))
		       DO
			 1;;Calculate # of crossovers produced by inserting non-leaf before the
0			 1;;connecting non-leaf.
0			 (try-insertion non-leaf scanner)
			 1;;Calculate # of crossovers produced by inserting non-leaf after the
0			 1;;connecting non-leaf.
0			 (try-insertion non-leaf (CDR scanner))
		     UNTIL (EQ (CDR scanner) circle)))
	     (IF best-insertion-point
		 1;;Perform the insertion.
0		 (SETF (CDR best-insertion-point)
		       (CONS non-leaf (CDR best-insertion-point)))
		 1;;No "optimum" insertion point was found, since NON-LEAF isn't connected
0		 1;;to any of the vertices in CIRCLE.  (This can happen whenever several
0		 1;;unconnected components are arranged in the same circle.)  Add it onto
0		 1;;the front of the list, arbitrarily.
0		 (SETF (CDR circle) (CONS non-leaf (CDR circle))))))
	 1;;Remove the list's circularity.
0	 (remove-list-circularity circle)
	 1;;If we've been stepping, signal the end.
0	 (debug-print T "done.")
	 (WHEN (MEMQ :step-through-placement *graph-debug-actions*)
	   (FORMAT T "~%Finished stepping through placement algorithm.~%"))
	 circle)))))


(DEFMETHOD (graph :minimize-crossovers-for-large-graphs) (non-leaves)
 (beep) 1;Still need to remove (or update) global vars controlling when this is called from
0 1;;User interface.  Can optimize the PUSHNEW below using cons trick.  Also assess this
0 1;;method--spacing goes light->heavy as you go clockwise around circle.
0  (LET ((stepping? (MEMQ :step-through-placement *graph-debug-actions*)))
    (debug-print T "~%Minimizing crossovers for *huge* circular arrangement of vertices...")
    1;;Sort the non-leaves by increasing line degree. (? or should it be decreasing, or does it
0    1;;not matter?)
;0    (SETQ non-leaves
1;0	  1;;OR is used incase leaves occur in NON-LEAVES.
;0	  (SORT non-leaves #'(LAMBDA (x y) (< (OR (line-degree x) 0)
1;0					      (OR (line-degree y) 0)))))
    (LET (circle)
      (LABELS
	((add-me-and-my-neighbors (vertex neighbor-depth)
	   (PUSHNEW vertex circle)
	   (IF (ZEROP neighbor-depth)
	       (WHEN stepping?
		 (SEND SELF :display-step circle
		       "~%Stepping through huge graph placement algorithm.   Added a set of neighbors."))
	       1;;Else only add the neighbors which occur in NON-LEAVES, otherwise you get
0	       1;;real strange results (when graph is broken up into subgraphs for display).
0	       1;;We do the test here, instead of a MEMQ, for efficiency.
0	       (DOLIST (neighbor (vertex-connected-to vertex))
		 (LET (present?)
		   (SETQ non-leaves (DELETE vertex non-leaves
					    :TEST #'(LAMBDA (v1 v2)
						    (WHEN (EQ v1 v2)
						      (SETQ present? T)))
					    :COUNT 1)) 1;;We know there are no repeats.
0		   (WHEN present?
		     (add-me-and-my-neighbors neighbor (1- neighbor-depth))))))))
	(LOOP WHILE non-leaves DO
	  (add-me-and-my-neighbors (POP non-leaves) 4)))
      (WHEN stepping?
	(FORMAT T "~%Finished stepping through placement algorithm.~%"))
      circle)))

(DEFUN count-crossovers-after-insertion (non-leaf circle)
  "1Returns the count of crossovers added if NON-LEAF were inserted into CIRCLE at the CDR.
NON-LEAF is a vertex struct.  CIRCLE is a circular list of vertex structs.0"
  (LET ((connected-to (vertex-connected-to non-leaf))
	(start (CDR circle)))
    1;;For each connection (comprised of either an in-link, out-link, or both),
0    1;;add the number of crossovers of the given in or out link.
0    (LOOP FOR scanner ON start
	  WHEN (MEMQ (CAR scanner) connected-to)
	  SUMMING (count-connection-crossovers start scanner)
	  UNTIL (EQ scanner circle))))

(DEFUN count-connection-crossovers (vertices to-vertices)
  "1Returns the number of connections between two sides of a circular list of vertices.
Both args are assumed to be pointers into the same circular list.  One side consists of the
vertices from pointer VERTICES to the vertex before (CAR TO-VERTICES).  The other side
consists of the rest of the vertices, except for (CAR TO-VERTICES).0"
  (LET ((from-vertex (CAR vertices))
	(to-vertex (CAR to-vertices))
	(other-side (CDR to-vertices)))
    1;;For each vertex on one side of the circle, count the number of vertices on the OTHER
    ;;side of the circle to which it is connected.
0    (LOOP FOR this-side-vertex IN vertices
	  UNTIL (EQ this-side-vertex to-vertex)
	  SUMMING (LOOP FOR other-side-vertex IN other-side
			UNTIL (EQ other-side-vertex from-vertex)
			WHEN (memq-circular other-side-vertex
					    (vertex-connected-to this-side-vertex))
			SUMMING 1))))


(DEFMETHOD (graph :display-step) (non-leaves format-string &REST format-args)
  "1Handles one step while stepping through the crossover minimization algorithm.
Formats FORMAT-STRING with FORMAT-ARGS, displays NON-LEAVES, then calls the command
loop recursively, providing a means to jump out of the recursive level.0"
  (WHEN (MEMQ :print-graph-computation-messages *graph-debug-actions*)
    (APPLY #'FORMAT T format-string format-args))
  1;;Bind this to NIL so that: 1. We don't step through the metagraph (eg. infinite loop)
0  1;;2. The creation of the metagraph isn't debug-printed, since that distracts the user
0  1;;from the stepping.
0  (LET ((*graph-debug-actions* NIL))
    1;;NIL#1 = no vertices to highlite.  NIL#2 = don't redraw previously displayed graph
0    1;;(we want to display successive portions of the graph we're creating, without
0    1;;interrutption.
0    (SEND *display-io* :recursively-display-vertices non-leaves NIL NIL)))


(DEFUN remove-list-circularity (list)
  "1Removes the circularity from LIST.0"
  (LOOP FOR vs ON list DO
	(WHEN (EQ (CDR vs) list)
	  (SETF (CDR vs) NIL)
	  (RETURN list))))

(DEFUN copy-circular-list (list)
  "1Copies LIST, which is a circular list.0"
  (LET ((copy (CONS (CAR list) (LOOP FOR vs ON (CDR list)
				     UNTIL (EQ vs list)
				     COLLECT (CAR vs)))))
    (SETF (CDR (LAST copy)) copy)))

(DEFUN memq-circular (item circular-list)
  (IF (EQ item (CAR circular-list))
      circular-list
      (LOOP FOR scanner ON (CDR circular-list)
	    UNTIL (EQ scanner circular-list)
	    DO
	    (WHEN (EQ item (CAR scanner))
	      (RETURN scanner)))))


1;;==========================================================================================
@;;
1;;;; Assignment of world coordinates to given non-leaf vertices and their connected lea@ve1s.
@;;
1;;==========================================================================================

0(DEFMETHOD (graph :arrange-vertices-in-circle) (non-leaves x-min y-min x-max y-max)
  "1Assign world coordinates to the vertices in LEAVES and NON-LEAVES so as to arrange them in 
a circle within the rectangular area specified by X-MIN, X-MAX, Y-MIN, and Y-MAX.  NON-LEAVES 
is assumed to be ordered in such a way that edge crossovers are minimized.  Leaf nodes to 
vertices in NON-LEAVES are plotted as satellites to the circle.  NON-LEAVES placed clockwise, 
starting at the top.  (Other parts of the program assume this.)

This method assumes (currently) that the X-MIN, X-MAX, Y-MIN, and Y-MAX specify a square 
region: the algorithm will use the square within the rectangle in any event.  It would be 
neat to generate ovals, instead of circles, when a non-square region is specified.  Ovals 
retain some of the neat computational properties of circles, but take up less space.0"
  1;;This means that any algorythm that splits up the window by group must try to produce
  ;;square subdivisions.
0  (WHEN non-leaves
    (LET* ((origin (CONS (ROUND (value-between x-min x-max))
			 (ROUND (value-between y-min y-max))))
	   1;;Set radius to the minimum dimension the area of the window allocated for this 
0	   1;;group.
0	   (radius (ROUND (MIN (- (CAR origin) x-min) (- (CDR origin) y-min))))
	   (placement-angle (/ *2PI* (LENGTH non-leaves)))
	   1;;Start first vertex at top of circle.  This assures left/right symmetry.
0	   (current-angle *3/2PI*)
	   (room-for-leaves (* radius
			       *percentage-radius-for-leaves-in-circular-arrangement*)))
      1;;Leave room for the leaves of the graph.
0      (DECF radius room-for-leaves)
      1;;Assign NON-LEAVES positions around the circle.
0      (DOLIST (vertex non-leaves)
	(SETF (vertex-location vertex)
	      (find-point origin radius current-angle))
	1;;Assign any LEAVES attached to VERTEX sattelite positions away from the circle.
0	(LET ((number-of-leaves (LOOP FOR v IN (vertex-connected-to vertex)
				      WHEN (leaf? v)
				      SUMMING 1)))
	  (UNLESS (ZEROP number-of-leaves)
	    (LET (1;;The more the leaves, the closer together they'll be squeezed.  But if 
0		  1;;there's only a few, we use a miminum angle.
0		  (leaf-placement-angle (MIN *1/8PI* (/ PI number-of-leaves)))
		  1;;First leaf is placed in same direction from center of circle as VERTEX.
0		  (leaf-current-angle current-angle)
		  (x 0))
	      1;;Stagger the leaves outward from angle CURRENT-ANGLE.
0	      (LOOP FOR v IN (vertex-connected-to vertex)
		    WHEN (leaf? v) DO
		    (INCF x)
		    (SETF (vertex-location v)
			  (find-point (vertex-location vertex)
				      room-for-leaves leaf-current-angle))
		    (IF (ODDP x)
			(INCF leaf-current-angle (* x leaf-placement-angle))
			(DECF leaf-current-angle (* x leaf-placement-angle)))))))
	1;;Next vertex angle.
0	(INCF current-angle placement-angle)))))




1;;==========================================================================================
@;;
1;;;; Geometric transformations on graphs.
@;;
1;;==========================================================================================



0(DEFUN-METHOD scale-x graph (x)
  "1Used to convert a flonum world X coordinate into a fixnum screen coordinate,
taking into account the current X scale factor (which can be modified through zooming)
and displacement (modified through pans).
This is implemented as a method-function, rather than a method, for efficiency.0"
  (ROUND (* (+ x x-displacement) x-scale-factor)))

(DEFUN-METHOD scale-y graph (y)
  "1Used to convert a flonum world Y coordinate into a fixnum screen coordinate,
taking into account the current Y scale factor (which can be modified through zooming).
and displacement (modified through pans).
This is implemented as a method-function, rather than a method, for efficiency.0"
  (ROUND (* (+ y y-displacement) y-scale-factor)))

(DEFMETHOD (graph :scale-x) (x)
  "1Same as SCALE-X; for external use.0"
  (ROUND (* (+ x x-displacement) x-scale-factor)))

(DEFMETHOD (graph :scale-y) (y)
  "1Same as SCALE-Y; for external use.0"
  (ROUND (* (+ y y-displacement) y-scale-factor)))


(DEFMETHOD (graph :inverse-scale-x) (x)
  "1Reverse transformation of SCALE-X.0"
  (- (/ x x-scale-factor) x-displacement))

(DEFMETHOD (graph :inverse-scale-y) (y)
  "1Reverse transformation of SCALE-Y.0"
  (- (/ y y-scale-factor) y-displacement))


(DEFMETHOD (graph :scale-relative) (x-scale y-scale)
  "1Performs a relative scaling operation on the coordinates of the graph's vertices.
If you send this message with arguments .5 and .5, the graph's x and y extents will 
be half what they were.  The graph is not redrawn.0"
  (LET ((x-origin (+ (SEND *display-io* :x-displacement)
		     (* (SEND *display-io* :INSIDE-WIDTH) .5)))
	(y-origin (+ (SEND *display-io* :y-displacement)
		     (* (SEND *display-io* :INSIDE-HEIGHT) .5))))
    1;;Set the scale factor.  All primitives drawn reference these through SCALE-X and
    ;;SCALE-Y. 
0    (SETQ x-scale-factor (* x-scale-factor x-scale)
	  y-scale-factor (* y-scale-factor y-scale))
1    ;;Increment the displacement (which is used also for real pans) so as to do
    ;;the zoom relative to the center of the screen.
0    (DECF x-displacement (/ (- (* x-origin x-scale) x-origin) x-scale-factor))
    (DECF y-displacement (/ (- (* y-origin y-scale) y-origin) y-scale-factor))))


(DEFMETHOD (graph :pan) (x-delta y-delta)
  "1Performs an absolute panning operation on the coordinates of the graph's vertices.
The two args are screen coordinate distances0."
  (INCF x-displacement (/ x-delta x-scale-factor))
  (INCF y-displacement (/ y-delta y-scale-factor)))



(DEFMETHOD (graph :reset-position-and-scale) ()
  "1Undo all zooms and pans.0"
  1;;Reset the displacement
0  (SETQ x-displacement 0
	y-displacement 0)
1  ;;Reset the scale factors.
0  (SETQ x-scale-factor 1.0
	y-scale-factor 1.0)
  1;;Set original scale (which tries to fit the graph into the viewing window).
0  (SEND SELF :scale-for-initial-viewing))


(DEFMETHOD (graph :scale-to-fit-window) (pixel-width pixel-height
					 &OPTIONAL allow-stretching?
					 (xborder 30)
					 (yborder 30))
  "1Sets the graph's scale factor so as fit all vertices within a window the size of the 
first two arguments.  ALLOW-STRETCHING?, if NIL, specifies that the X and Y scale factor
are to be identical.  For example, if the vertices of the graph are to be displayed in a
circle, differing X and Y scale factors would cause the circle to appear as an ellipse.
XBORDER0 1and YBORDER specify margins to leave between the vertices and the edge of the window.0"
  (LET* ((xmax (CAR (vertex-location (CAR vertices))))
	 (ymax (CDR (vertex-location (CAR vertices))))
	 (xmin xmax)
	 (ymin ymax))
    (DOLIST (vertex (CDR vertices))
      (LET ((location (vertex-location vertex)))
	(IF (> (CAR location) xmax)
	    (SETQ xmax (CAR location))
	    (IF (< (CAR location) xmin)
		(SETQ xmin (CAR location))))
	(IF (> (CDR location) ymax)
	    (SETQ ymax (CDR location))
	    (IF (< (CDR location) ymin)
		(SETQ ymin (CDR location))))))
    (LET ((width (- xmax xmin))
	  (height (- ymax ymin)))
      (UNLESS (ZEROP width)
	1;;Get a float, rather than rational, so that later arithmetic is faster.
0	(SETQ x-scale-factor (FLOAT (/ (- pixel-width xborder xborder) width))))
      (UNLESS (ZEROP height)	
	(SETQ y-scale-factor (FLOAT (/ (- pixel-height yborder yborder) height)))))
    (UNLESS allow-stretching?
      (SETQ x-scale-factor (MIN x-scale-factor y-scale-factor)
	    y-scale-factor x-scale-factor))
    1;;This computation's a bit tricky.  We want to displace the graph such that its
0    1;;top level corner (xmin, ymin) in world coordinates is mapped to screen coordinate
0    1;;(xborder, yborder). XMIN and YMIN are world-coordinate values.  XBORDER and YBORDER are
0    1;;screen-coordinate lengths. By dividing them by the scale factors, we get two
0    1;;world-coordinate lengths.  The subtraction operation can be thought of as
0    1;;(+ (- xmin) {border length in world coordinates}).
0    (SETQ x-displacement (- (/ xborder x-scale-factor) xmin)
	  y-displacement (- (/ yborder y-scale-factor) ymin))))


(DEFMETHOD (graph :vertex-has-moved) (vertex)
  2"Recomputes (or causes recomputation of) all data which must be updated whenever a vertex
has been relocated.  For instance, the slopes of edges must be recomputed."
0  1;;Could be made more efficient by doing only the work required by changes in VERTEX,
0  1;;but I wanted to hack this fast.  These operations are very fast anyway.
0  vertex
  1;;Recompute edge slopes
0  (DOLIST (edge edges)
    (SETF (edge-slope edge) (angle-for-point (vertex-location (edge-vertex-2 edge))
					     (vertex-location (edge-vertex-1 edge)))))
  1;;Recompute the positions of directional arrows.  This computation depends on the above.
0  (SEND SELF :plot-edges)
  (SETQ location-clumps (clump-vertices-and-edges-by-location vertices edges)))


1;;==========================================================================================
@;;
1;;;; Output of plotted graphs.
@;;
1;;==========================================================================================


0(DEFMETHOD (graph :draw) (&OPTIONAL (window *graph-output*))
  "1Draws a graph0 1on WINDOW0.
1Assumes that the vertices of the graph have already been assigned0 1coordinates.0"
  (LET ((*graph-output* window))
    (SEND *graph-output* :CLEAR-WINDOW)
    1;;Store edge line slopes unless they were computed in a previous :DRAW.
0    (WHEN (AND edges (NULL (edge-slope (CAR edges))))
      (DOLIST (edge edges)
	(SETF (edge-slope edge) (angle-for-point (vertex-location (edge-vertex-2 edge))
						 (vertex-location (edge-vertex-1 edge))))))
    (SEND SELF :draw-view-boundaries window)
    1;;Draw all edges.
0    (SEND SELF :draw-edges window)
    (SEND SELF :draw-vertices window)))

(DEFMETHOD (graph :draw-vertices) (*graph-output*)
  "1Draw0 1graph vertices.0"
 (MULTIPLE-VALUE-BIND (width height)
     (SEND *graph-output* :INSIDE-SIZE)
   (LET* ((vertex-print-string-function (get-compiled-function-or-die-trying
					  (SEND (SEND SELF :type)
						:vertex-print-string-function)))
	  (display-xmin (SEND SELF :inverse-scale-x 0))
	  (display-ymin (SEND SELF :inverse-scale-y 0))
	  (display-xmax (+ display-xmin (/ width x-scale-factor)))
	  (display-ymax (+ display-ymin (/ height y-scale-factor)))
	  (clipped-vertex-count 0))
    (DOLIST (clump location-clumps)
      (IF (overlapping-extents?
	    (CAAR clump) (CDAR clump) (CAADR clump) (CDADR clump)
	    display-xmin display-ymin display-xmax display-ymax)
	  (DOLIST (vertex (THIRD clump))
	    (LET* ((vertex-location (vertex-location vertex))
		   (scaled-x (scale-x (CAR vertex-location)))
		   (scaled-y (scale-y (CDR vertex-location))))
	      (MULTIPLE-VALUE-BIND (text font)
		  (FUNCALL vertex-print-string-function vertex)
		1;;Draw a circle as the vertex
0		(SEND *graph-output* :DRAW-CIRCLE scaled-x scaled-y 3)
		1;;Label the vertex using its DATA.
0		1;;Y is adjusted to keep from writing over the circle.
0		(LET ((adjusted-y (- scaled-y #+Symbolics 6 #+Explorer 12)))
		  (SEND *graph-output* :DRAW-STRING
			text
			(- scaled-x 12) adjusted-y
			1;;Towards the right
0			scaled-x adjusted-y
			1;;Unknown arg, text font, and alu function (SETA will overwrite any 
0			1;;lines going through text).
0			NIL font TV:ALU-SETA)))))
	  1;;Else
0	  (INCF clipped-vertex-count (LENGTH (THIRD clump)))))
    1;;Store for graph description.
0    (SEND SELF :PUTPROP clipped-vertex-count :clipped-vertex-count))))

(DEFUN overlapping-extents? (xmin1 ymin1 xmax1 ymax1 xmin2 ymin2 xmax2 ymax2)
  "1Returns non-NIL if the two rectangular areas described by the arguments overlap.0"
  (NOT (OR (< xmax1 xmin2)
	   (< ymax1 ymin2)
	   (< xmax2 xmin1)
	   (< ymax2 ymin1))))


(DEFUN value-between (value1 value2 &OPTIONAL (percentage .5))
  "1Returns a number between VALUE1 and VALUE2, specifically
  (VALUE2 - VALUE1) * PERCENTAGE + VALUE1.  Without arg, returns the average.0"
  (+ (* (- value2 value1) percentage) value1))

(DEFCONSTANT *error-delta* (/ PI 50))

;;This method is still the bottleneck for drawing/redrawing graphs.
;;The three main operations:
;;draw-line, draw-filled-in-sector, and draw-text seem to contribute equally.  Luckily, 
;;scaling operations don't contribute to the slowness.  Probably only these ways of speeding
;;up redraw:
;; 1. Clip out edges which are completely outside the output window.  (Mainly would
;;help on humungous graphs.) 
;; 2. Give each graph its own screen array.  Panning/zooming still
;;slow, but switching between graphs would redisplay very fast.  Expensive memory-wise?
;; 3. Get rid of the directional arrows, but add an arrow character onto the end of the
;;labels.  Pretty sharp, right?  Would only get rid of about 1/3 of the delay.
;;
(DEFCONSTANT *pi+.3* (+ PI .3)) 1;;Precomputed for efficiency.
0(DEFCONSTANT *pi-.3* (- PI .3))

(DEFUN draw-pointer (window head-x head-y direction length alu)
  (SEND window :DRAW-FILLED-IN-SECTOR head-x head-y length
	(- *pi-.3* direction) (- *pi+.3* direction) alu))

1;;Used by method :DRAW-EDGES below.
0#+Symbolics
(DEFUN-METHOD dress-edges graph (some-edges window)
  (DOLIST (edge some-edges)
    (LET* ((from-point (vertex-location (edge-vertex-1 edge)))
	   (to-point (vertex-location (edge-vertex-2 edge)))
	   (direction (edge-slope edge))
	   (from-x (scale-x (CAR from-point)))
	   (from-y (scale-y (CDR from-point)))
	   (to-x (scale-x (CAR to-point)))
	   (to-y (scale-y (CDR to-point)))
	   (edge-print-string-function (get-compiled-function-or-die-trying
					 (SEND (SEND SELF :type)
					       :edge-print-string-function))))
      1;;Label the directed edge if it is long enough to bother with it.
0      1;;For efficiency, the edge length is approximated.
0      (WHEN (OR (> (ABS (- from-x to-x))
		   *smallest-edge-length-to-bother-labelling*)
		(> (ABS (- from-y to-y))
		   *smallest-edge-length-to-bother-labelling*))
	1;;Draw label near the head vertex of the edge.  The rationale is that if an
0	1;;edge goes from V1 to V2 and another goes from V1 to V1, the edge labels
0	1;;won't overlap (though the edges do, since they are straight lines).
0	1;;This also puts the label near the feathers of the directional arrow 
0	1;;representing the edge.
0	(LET ((x-offset 0)
	      (y-offset 0)
	      1;;The .80 (80%) specifies where along the edge to put the label.  Could
0	      1;;make a nice parameter, since on certain unlucky graphs, lots of labels
0	      1;;will overwrite each other (but the user could change the parameter).
0	      1;;As is, it puts the label behind the arrow feathers, which are at 90%.
0	      (label-x (ROUND (value-between from-x to-x .80)))
	      (label-y (ROUND (value-between from-y to-y .80))))
	  1;;Remember, windows are upside-down.  All these offsets were generated by
0	  1;;fiddling with0 1the values till they looked right.  They are a function of
0	  1;;weirdness in the placement algorithm used by0 1the :DRAW-STRING method.
0	  1;;Placement for (<= *3/2PI* slope *7/4PI*) and (<= *7/4PI* slope *2*) is 
0	  1;;fine, so they don't occur in the COND.
0	  (COND
	    1;;Check to see if slope is close to infinite, since we want to draw 
0	    1;;vertical and nearly0 1vertical lines' labels downwards.
0	    ((< (ABS (- direction *1/2pi*)) *error-delta*)
	     1;;There's a bug in :DRAW-STRING where text drawn exactly upwards is
0	     1;;positioned way above the FROM-X, TO-X location.  The following works 
0	     1;;around that by offsetting FROM-X by one.
0	     (SETQ from-x (1- from-x) from-y 0))
	    ((< (ABS (- direction *3/2pi*)) *error-delta*)
	     (SETQ x-offset 3))
	    ((<= 0 direction *1/4PI*)
	     (SETQ y-offset -2 x-offset 4))
	    ((<= *1/4PI* direction *1/2PI*)
	     (SETQ x-offset 8))
	    ((<= *1/2PI* direction *3/4PI*)
	     (SETQ x-offset 2 y-offset 4))
	    ((<= *3/4PI* direction PI)
	     (SETQ y-offset -1 x-offset -3))
	    ((<= PI direction *5/4PI*)
	     (SETQ x-offset 2))
	    ((<= *5/4PI* direction *3/2PI*)
	     (SETQ x-offset 2)))
	  (MULTIPLE-VALUE-BIND (text font)
	      (FUNCALL edge-print-string-function edge)
	    1;;Appropriate offsets are calculated, so draw it.
0	    (SEND window :DRAW-STRING
		  1;;Use the function designated for our graph-type for getting edge
0		  1;;print strings.
0		  text
		  (+ x-offset label-x) (+ y-offset label-y)
		  1;;This looks wrong, using the edge head (instead of the tail) in 
0		  1;;directing the string, but the purpose of this is to have the 
0		  1;;string drawn backward from LABEL-X, LABEL-Y.  Thus, as the user 
0		  1;;zooms out, the labels will grow towards the head of the edge, 
0		  1;;instead of immediately running into the tail of the edge (which 
0		  1;;LABEL-X, LABEL-Y is close to).
0		  (+ x-offset from-x) (+ y-offset from-y)
		  1;;Unknown arg, text font, and alu function (SETA will overwrite any 
0		  1;;lines going through text).
0		  NIL font TV:ALU-SETA)))))
    1;;Draw directional indicators, using a filled pie-shape as an arrow, 8 pixels 
0    1;;long.  PI-.3 and PI+.3 are slight increments/decrements of the value PI.  We 
0    1;;subtract DIRECTION from these for two reasons:
0    1;;1. In order to use filled-sectors as direction indicators, the DIRECTION has to 
0    1;;be reversed (add *PI+.3* or *PI-.3*).
0    1;;2. Since :DRAW-FILLED-IN-SECTOR is foolishly inconsistent with the rest of the 
0    1;;graphics primitives, RADIANS are oriented counter-clockwise instead of clockwise
0    1;;(subtract from 2*PI). These two are performed by subtracting from *PI+.3* or
0    1;;*PI-.3*.
0    (draw-pointer window
		  (scale-x (CAR (edge-misc edge))) (scale-y (CDR (edge-misc edge)))
		  (edge-slope edge) 8. TV:ALU-SETA)))
#+Explorer
(DEFUN-METHOD dress-edges graph (some-edges window)
  (DOLIST (edge some-edges)
    (LET* ((from-point (vertex-location (edge-vertex-1 edge)))
	   (to-point (vertex-location (edge-vertex-2 edge)))
	   (direction (edge-slope edge))
	   (from-x (scale-x (CAR from-point)))
	   (from-y (scale-y (CDR from-point)))
	   (to-x (scale-x (CAR to-point)))
	   (to-y (scale-y (CDR to-point)))
	   (edge-print-string-function (get-compiled-function-or-die-trying
					 (SEND (SEND SELF :type)
					       :edge-print-string-function))))
      1;;Label the directed edge if it is long enough to bother with it.
0      1;;For efficiency, the edge length is approximated.
0      (WHEN (OR (> (ABS (- from-x to-x))
		   *smallest-edge-length-to-bother-labelling*)
		(> (ABS (- from-y to-y))
		   *smallest-edge-length-to-bother-labelling*))
	1;;Draw label near the head vertex of the edge.  The rationale is that if an
0	1;;edge goes from V1 to V2 and another goes from V1 to V1, the edge labels
0	1;;won't overlap (though the edges do, since they are straight lines).
0	1;;This also puts the label near the feathers of the directional arrow 
0	1;;representing the edge.
0	(LET ((x-offset 0)
	      (y-offset 0)
	      1;;The .80 (80%) specifies where along the edge to put the label.  Could
0	      1;;make a nice parameter, since on certain unlucky graphs, lots of labels
0	      1;;will overwrite each other (but the user could change the parameter).
0	      1;;As is, it puts the label behind the arrow feathers, which are at 90%.
0	      (label-x (ROUND (value-between from-x to-x .80)))
	      (label-y (ROUND (value-between from-y to-y .80)))
	1      ;;My quicky implementation of :DRAW-STRING gives us complete control over the
0	1      ;;drawing angle, so it will mindlessly draw strings backwards/upside down given the
0	1      ;;right angle.  Thus, it has an option for drawing the characters in reverse order.
0	1      ;;This var controls its use.
0	      backwards?)
	  1;;Remember, windows are upside-down.  All these offsets were generated by
0	  1;;fiddling with0 1the values till they looked right.  They are a function of
0	  1;;weirdness in the placement algorithm used by0 1the :DRAW-STRING method.
0	  (COND
	    1;;Check to see if slope is close to infinite, since we want to draw 
0	    1;;vertical and nearly0 1vertical lines' labels downwards.
0	    ((< (ABS (- direction *1/2pi*)) *error-delta*)
	     (SETQ x-offset 3 from-y 0 backwards? T))
	    ((< (ABS (- direction *3/2pi*)) *error-delta*)
	     (SETQ x-offset 3))
	    ((<= 0 direction *1/4PI*)
	     (SETQ y-offset -4 x-offset 4 backwards? T))
	    ((<= *1/4PI* direction *1/2PI*)
	     (SETQ x-offset 4 y-offset -4 backwards? T))
	    ((<= *1/2PI* direction *3/4PI*)
	     (SETQ x-offset 2 y-offset 4))
	    ((<= *3/4PI* direction PI)
	     (SETQ y-offset -5 x-offset -3))
	    ((<= PI direction *5/4PI*)
	     (SETQ x-offset 2 y-offset -4))
	    ((<= *5/4PI* direction *3/2PI*)
	     (SETQ x-offset 3 y-offset -3))
	    ((<= *3/2PI* direction *7/4PI*)
	     (SETQ x-offset -4 y-offset -4 backwards? T))	;???
	    ((<= *7/4PI* direction *2PI*)
	     (SETQ y-offset -4 x-offset -2 backwards? T)))
	  (MULTIPLE-VALUE-BIND (text font)
	      (FUNCALL edge-print-string-function edge)
	    1;;Appropriate offsets are calculated, so draw it.
0	    (SEND window :DRAW-STRING
		  1;;Use the function designated for our graph-type for getting edge
0		  1;;print strings.
0		  text
		  (+ x-offset label-x) (+ y-offset label-y)
		  1;;This looks wrong, using the edge head (instead of the tail) in 
0		  1;;directing the string, but the purpose of this is to have the 
0		  1;;string drawn backward from LABEL-X, LABEL-Y.  Thus, as the user 
0		  1;;zooms out, the labels will grow towards the head of the edge, 
0		  1;;instead of immediately running into the tail of the edge (which 
0		  1;;LABEL-X, LABEL-Y is close to).
0		  (+ x-offset from-x) (+ y-offset from-y)
		  1;;Unknown arg, text font, and alu function (SETA will overwrite any 
0		  1;;lines going through text).
0		  NIL font TV:ALU-SETA backwards?)))))
    1;;Draw directional indicators, using a filled pie-shape as an arrow, 8 pixels 
0    1;;long.  PI-.3 and PI+.3 are slight increments/decrements of the value PI.  We 
0    1;;subtract DIRECTION from these for two reasons:
0    1;;1. In order to use filled-sectors as direction indicators, the DIRECTION has to 
0    1;;be reversed (add *PI+.3* or *PI-.3*).
0    1;;2. Since :DRAW-FILLED-IN-SECTOR is foolishly inconsistent with the rest of the 
0    1;;graphics primitives, RADIANS are oriented counter-clockwise instead of clockwise
0    1;;(subtract from 2*PI). These two are performed by subtracting from *PI+.3* or
0    1;;*PI-.3*.
0    (draw-pointer window
		  (scale-x (CAR (edge-misc edge))) (scale-y (CDR (edge-misc edge)))
		  (edge-slope edge) 8. TV:ALU-SETA)))

(DEFMETHOD (graph :draw-edges) (*graph-output*)
  1;;First draw the edge line.  They go first so that labels will overwrite them.
0  (DOLIST (edge edges)
    (LET ((from-point (vertex-location (edge-vertex-1 edge)))
	  (to-point (vertex-location (edge-vertex-2 edge))))
      (SEND *graph-output* :DRAW-LINE (scale-x (CAR from-point)) (scale-y (CDR from-point))
	    (scale-x (CAR to-point)) (scale-y (CDR to-point)))))
  1;;Second, draw the arrow direction indicators and labels.  We have the arrow coordinates
0  1;;stored in clumps for fast mouse sensitivity, so use the clumps for faster display.
0  1;;(Indicators which are outside of *GRAPH-OUTPUT* are clipped here.)  We also use these
0  1;;coordinates to clip labels, even though the labels don't have a single point location.
0  1;;(They are also offset from the coordinates).  This is justified by gained speed and it
0  1;;doesn't result in any gross inaccuracies.
0  (MULTIPLE-VALUE-BIND (width height)
      (SEND *graph-output* :INSIDE-SIZE)
    (LET* ((display-xmin (SEND SELF :inverse-scale-x 0))
	   (display-ymin (SEND SELF :inverse-scale-y 0))
	   (display-xmax (+ display-xmin (/ width x-scale-factor)))
	   (display-ymax (+ display-ymin (/ height y-scale-factor))))
      (DOLIST (clump location-clumps)
	(WHEN (overlapping-extents?
		(CAAR clump) (CDAR clump) (CAADR clump) (CDADR clump)
		display-xmin display-ymin display-xmax display-ymax)
	  (dress-edges (FOURTH clump) *graph-output*))))))



(DEFMETHOD (graph :draw-view-boundaries) (window)
  1;;Draw dashed line just inside WINDOW borders (unless feature is disabled.)
0  1;;When the user fast-pans to the boundary of the hidden bit array,
0  1;;the dashed line will provide a hint that further fast panning will cause
0  1;;a real pan.  
0  (WHEN *dashed-line-margin*
    (MULTIPLE-VALUE-BIND (low-x low-y high-x high-y)
	(SEND window :INSIDE-EDGES)
      1;;This var provides space between the actual boundary and the displayed boundary.
0      1;;Without this space, the user would have no warning.
0      (INCF low-x *dashed-line-margin*)
      (INCF low-y *dashed-line-margin*)
      (DECF high-x *dashed-line-margin*)
      (DECF high-y *dashed-line-margin*)
      (SEND window :DRAW-DASHED-LINE low-x low-y high-x low-y)
      (SEND window :DRAW-DASHED-LINE high-x low-y high-x high-y)
      (SEND window :DRAW-DASHED-LINE high-x high-y low-x high-y)
      (SEND window :DRAW-DASHED-LINE low-x high-y low-x low-y))))

(COMPILE-FLAVOR-METHODS graph)



  
1;;==========================================================================================
@;;
1;;;; Code for finding the bi-connected components of a graph.
;;   This implements an efficient (order = number of vertices) algorithm described in
;;   Aho, Hopcroft, and Ullman (The Design and Analysis of Computer Algorithms).
;;
;;   Borrowed from Brian Falkenheimer and rehacked a bit for the Graph Displayer.
@;;
1;;==========================================================================================

0(DEFVAR *stack*)
(DEFVAR *count*)

(DEFUN extract-cycles (vertices)
  "1Search through the relational graph formed by the adjacency 
lists (vertex-connected-to vertex vertices)
and returns a list of the bi-connected-components of the graph as the first value
and a list of edges connecting them as the second value.  The first value is a list of lists
of vertices, one for each connected component.  The second value is a list of 
conses (vertex . vertex) refering to edges.0"
  (LET (*stack* *count* bi-con-components connecting-edges)
    1;;Use a copy, since NCONC mungs it below.
0    (SETQ vertices (COPY-LIST vertices))
    1;;Initialize temporary storage for this algorithm's use.
0    (DOLIST (vertex vertices)
      (SETF (vertex-misc vertex) (LIST NIL NIL NIL NIL)))
    
    (LOOP FOR *stack* = NIL
	  FOR *count* = 1
	  1;;CAR is just a random pick.
0	  DO
	  (SETQ bi-con-components (NCONC (searchb (CAR vertices) vertices NIL)
					 bi-con-components))
	  1;;Remove used vertices. DELETE for disconnected graphs.
0	  (SETQ vertices (DELETE-IF #'(LAMBDA (v) (vertex-mark v)) vertices))
	  (UNLESS vertices
	    (RETURN NIL)))
    1;;Deallocate temporary storage.
0    (DOLIST (vertex vertices)
      (SETF (vertex-misc vertex) NIL))
    (VALUES (LOOP FOR component IN bi-con-components
		  IF (CDR component)
		  COLLECT (DELETE-DUPLICATES (LOOP FOR edges IN component
						   COLLECT (CAR edges) COLLECT (CDR edges)))
		  ELSE DO (PUSH (CAR component) connecting-edges))
	    connecting-edges)))


(DEFUN searchb (v vertices bi-components)
  "1Aho & Ullman's algorythm: depth-first search of the connected component of the graph 
which includes the given vertex V.  Returns the bi-connected-components of this portion of
the graph. The order of algorithm is the number of edges in the graph.0"
  (LET (cur-edge)
    (SETF (vertex-mark v) T
	  (vertex-dfnumber v) *count*
	  (vertex-low v)  *count*)
    (INCF *count*)
    (DOLIST (w (vertex-connected-to v))
      (SETQ cur-edge (CONS v w))
      (UNLESS (edge-in? cur-edge)
	(PUSH cur-edge *stack*))
      (COND ((NULL (vertex-mark w))
	     (SETF (vertex-father w)  v
		   bi-components (Searchb w vertices bi-components))
	     (IF (>= (vertex-low w)
		     (vertex-dfnumber v))
		 (PUSH (remove-bicomp cur-edge) bi-components))
	     (SETF (vertex-low v)
		   (MIN (vertex-low v)
			(vertex-low w))))
	    ((NULL (EQUAL w (vertex-father v)))
	     (SETF (vertex-low v)
		   (MIN (vertex-low v)
			(vertex-dfnumber w))))))
    bi-components))

(DEFUN edge-in? (edge)
  "1Returns non-NIL if the edge already been inserted into the stack.0"
  (LET* ((vertex1 (CAR edge))
	 (vertex2 (CDR edge))
	 (dfnum1 (vertex-dfnumber vertex1))
	 (dfnum2 (vertex-dfnumber vertex2)))
    (AND (vertex-mark vertex2)
	 (OR (< dfnum1 dfnum2)
	     (AND (> dfnum1 dfnum2)
		  (EQ vertex2
		      (vertex-father vertex1)))))))

(DEFUN remove-bicomp (edge)
  "1Remove the newly found bi-connected-component from the stack by
popping the stack down to the given edge.0"
  (LET (bi-comp)
    (LOOP FOR stack-edge = (POP *stack*)
	  DO (PUSH stack-edge bi-comp)
	  UNTIL (OR (NULL *stack*) (EQUAL stack-edge edge)))
    bi-comp))