" (c) 1992 Copyright (c) University of Washington
  Written by Tony Barrett.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to
  bug-ucpop@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The following code is used to implement a graphic interface to snlp
;;; that uses the clim graphics package.

(in-package "UCPOP")

(export '(play))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Variables and structures

(defvar *enable-vcr* nil)
(defvar *tape* nil)			; A SNLP run trace
(defvar *display* nil)			; Where to show the movie
(defvar *window* nil)
(defvar *max-open* 1)
(defvar *max-unsafe* 1)
(defvar *max-links* 1)
(defvar *max-steps* 1)
(defvar *tree* nil)
(defvar *movie-frame* nil)
(defvar *tree-scale* 8)
    
(defstruct (plan-tree (:print-function print-plan-tree))
  entry
  flaw
  children
  (displayed nil)
  (next nil)
  (prev nil)
  (xpos nil)
  (ypos nil))

(clim:define-presentation-type plan-picture ())
(clim:define-presentation-type problem-picture ())

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Defining the display interface

(clim::define-application-frame vcr
                          ()
  ()
  (:panes ((graph :application
                  :initial-cursor-visibility :off)
           (movie :application
                  :scroll-bars nil)
           (text :application)
           (menu :command-menu)
           (documentation :pointer-documentation)))
  (:layout ((main
             (:column 1
              (:column :rest
                (graph :rest)
                (menu :compute))
              (:column 3/10
                (:row :rest
                 (movie 1/4)
                 (text  3/4)))
              (documentation :compute)))
            )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Handy interface functions for using the program

(defun ENABLE-VCR ()
  (setf *enable-vcr* t
	*tape* nil))

(defun DISABLE-VCR ()
  (setf *enable-vcr* nil
	*tape* nil))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Record a single frame of the movie.
(defun VCR-ADD-FRAME (parent flaw children)
  (when *enable-vcr*
    (push (list parent flaw children) *tape*)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Load and display a tape using the clim graphics package.
(defun PLAY (message display)
  (setf *tree* (cons (compute-tree *tape*) message))
  (setf *movie-frame* nil)
  (when (null *display*)
    (setq *display* (clim::open-root-window :clx :host display)))
  (when (null *window*)
      (multiple-value-bind (left top right bottom)
          (size-vcr-frame *display* 0 0 800 600)
        (setq *window* 
	  (clim:make-application-frame 'vcr
				       :parent *display*
				       :left left :top top
				       :right right :bottom bottom))))
  (clim:run-frame-top-level *window*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Commands

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This macro is used to define command menu buttons
(defmacro DEFINE-VCR-BUTTON (name windows &body body)
  `(define-vcr-command 
       (,(intern (format nil "COM-~A-vcr" name)) :menu ,name) ()
     (let ,(mapcar #'(lambda (w)
		       `(,w (clim:get-frame-pane 
			     clim:*application-frame* (quote ,w))))
	    windows)
       ,@body)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; This command is used to make bugging a plan picture cause that
;;; plan to be displayed in the text window.
(define-vcr-command (com-display-plan) ((plan 'plan-picture))
  (display-a-plan plan)
  (when (or (plan-tree-next plan) (plan-tree-prev plan))
    (next-frame plan
		(clim:get-frame-pane clim:*application-frame* 'graph)
		(clim:get-frame-pane clim:*application-frame* 'movie))))
  
(clim:define-presentation-to-command-translator display-a-plan
    (plan-picture com-display-plan vcr)
  (object)
  `(,object))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; This command is used to make bugging a problem with a plan
;;; generate that plan's children (for that problem)
(define-vcr-command (com-choose-problem) ((problem 'problem-picture))
  (expand-prob (cadr problem) (caddr problem)))

(defun EXPAND-PROB (plan-entry bug)
  (let ((txt (clim:get-frame-pane clim:*application-frame* 'text))
        (graph (clim:get-frame-pane clim:*application-frame* 'graph))
        (children (new-plans (plan-tree-entry plan-entry) bug)))
    (setf (plan-tree-flaw plan-entry) bug
	  (plan-tree-children plan-entry)
	  (mapcar #'(lambda (x)
		      (make-plan-tree
		       :entry x
		       :flaw nil
		       :prev plan-entry
		       :next plan-entry))
		  children))
    (tree-node-pos plan-entry 
		   (plan-tree-xpos plan-entry)
		   (plan-tree-ypos plan-entry))
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry plan-entry graph))
    (clim:window-clear txt)
    (next-frame *movie-frame* 
		(clim:get-frame-pane clim:*application-frame* 'graph)
		(clim:get-frame-pane clim:*application-frame* 'movie))))

(clim:define-presentation-to-command-translator display-a-problem
    (problem-picture com-choose-problem vcr)
  (object)
  `(,object))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Animate backward from the current movie position.
(define-vcr-button "<<" (movie graph)
  (loop
    (unless (and *movie-frame* (plan-tree-prev *movie-frame*) 
		 (vcr-sleep movie)) 
      (return nil))
    (next-frame (plan-tree-prev *movie-frame*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Move the movie one step backward.
(define-vcr-button "<" (movie graph)
  (when (and *movie-frame* (plan-tree-prev *movie-frame*))
    (next-frame (plan-tree-prev *movie-frame*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Move the movie one step forward.
(define-vcr-button ">" (movie graph)
  (when (and *movie-frame* (plan-tree-next *movie-frame*))
    (next-frame (plan-tree-next *movie-frame*) graph movie)))
    
;;;;;;;;;;;;;;;;;;;;;;;;
;;; Animate forward from the current movie position.
(define-vcr-button ">>" (movie graph)
  (loop
    (unless (and *movie-frame* (plan-tree-next *movie-frame*) 
		 (vcr-sleep movie)) 
      (return nil))
    (next-frame (plan-tree-next *movie-frame*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; blow up the tree diagram
(define-vcr-button "Enlarge" (graph movie)
  (when *tree*
    (setf *tree-scale* (/ *tree-scale* 2))
    (clim:window-clear graph)
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry (car *tree*) graph))
    (next-frame *movie-frame* graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; shrink the tree diagram
(define-vcr-button "Shrink" (graph movie)
  (when *tree* 
    (setf *tree-scale* (* *tree-scale* 2))
    (clim:window-clear graph)
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry (car *tree*) graph))
    (next-frame *movie-frame* graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;
(define-vcr-button "Refresh" (graph movie text)
  (when *tree* 
    (clim:window-clear graph)
    (tree-node-pos (car *tree*) 0 0)
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry (car *tree*) graph))
    (clim:window-clear text)
    (clim:with-text-style ('(:sans-serif :roman :small) text)
      (clim:write-string (format nil (cdr *tree*)) text))
    (next-frame (car *tree*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; This command is used to exit the interface.
(define-vcr-button "Exit" (graph movie text)
  (clim:window-clear graph)
  (clim:window-clear movie)
  (clim:window-clear text)
  (setf *tree* nil
	*max-open* 1 *max-unsafe* 1 *max-links* 1 *max-steps* 1)
  (clim:frame-exit clim:*application-frame*))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Define  an interaction button
(clim:define-presentation-to-command-translator display-a-plan
    (plan-picture com-display-plan vcr)
  (object)
  `(,object))

(defun GRAPH-BUTTON (stream plan)
  (clim:with-output-as-presentation
      (:type 'plan-picture :object plan :stream stream :single-box t)
    (graph-plan stream plan :label (< *tree-scale* 16))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Define  an interaction button for problems
(clim:define-presentation-to-command-translator display-a-problem
    (problem-picture com-choose-problem vcr)  (object)
  `(,object))

(defun PROBLEM-BUTTON (stream problem)
  (clim:with-output-as-presentation
      (:type 'problem-picture :object problem :stream stream :single-box t)
    (clim:write-string (car problem) stream)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. plan-tree setup

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Compute the search tree from a tape
(defun COMPUTE-TREE (tape &aux (nodes nil))
  (dolist (entry tape)
    (dolist (kid (caddr entry))
      (setf *max-open* (max *max-open*   (plan-open kid))
	    *max-unsafe* (max *max-unsafe* (plan-unsafe kid))
	    *max-links* (max *max-links*  (length (plan-links kid)))
	    *max-steps* (max *max-steps*  (length (plan-steps kid)))))
    (push (make-plan-tree :entry    (car entry) 
			  :flaw     (cadr entry)
			  :children (caddr entry))
	  nodes))
  (do* ((p nil (car n))
	(n nodes (cdr n)))
      ((null n) nil)
    (setf (plan-tree-next (car n)) (when (cdr n) (cadr n))
	  (plan-tree-prev (car n)) p))
  (do ((ns nodes (cdr ns)))
      ((null ns) nil)
    (setf (plan-tree-children (car ns))
      (mapcar #'(lambda (child)
		  (let ((a (find child (cdr ns) 
				 :key #'plan-tree-entry :test #'eq)))
		    (unless a 
		      (setf a (make-plan-tree :entry child 
					      :flaw nil 
					      :children nil
					      :next (car ns)
					      :prev (car ns))))
		    a))
	      (plan-tree-children (car ns)))))
  (order-tree (car nodes))
  (tree-node-pos (car nodes) 0 0)
  (car nodes))

(defun PLAN-OPEN (p &aux (n 0))
  (dolist (f (plan-flaws p) n)
    (unless (unsafe-p f) (incf n))))

(defun PLAN-UNSAFE (p &aux (n 0))
  (dolist (f (plan-flaws p) n)
    (when (unsafe-p f) (incf n))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Order a tree so that the winning path is left justified
(defun ORDER-TREE (node)
  (cond ((plan-flaws (plan-tree-entry node))
         (dolist (c (plan-tree-children node))
           (when (order-tree c)
             (setf (plan-tree-children node)
               (cons c (remove c (plan-tree-children node)))
	       (plan-tree-displayed c) t)
             (return t))))
        (t t)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Compute search tree node display positions.
(defun TREE-NODE-POS (node x y)
  (setf (plan-tree-xpos node) x
	(plan-tree-ypos node) y)
  (if (plan-tree-children node)
      (dolist (c (plan-tree-children node) x)
	(setf x (tree-node-pos c x (+ y 200))))
    (+ x 200)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Draw the tree.
(defun DRAW-ENTRY (p stream)
  (clim:with-translation (stream (plan-tree-xpos p) (plan-tree-ypos p))
    (graph-button stream p))
  (when (and (plan-tree-displayed p) (plan-tree-children p))
    (clim:draw-line* stream
		     (+ (plan-tree-xpos p) 55) (+ (plan-tree-ypos p) 160)
		     (+ (plan-tree-xpos p) 55) (+ (plan-tree-ypos p) 180))
    (clim:draw-line* stream
     (+ (plan-tree-xpos p) 55) (+ (plan-tree-ypos p) 180)
     (+ 55 (plan-tree-xpos (car (last (plan-tree-children p)))))
     (+ (plan-tree-ypos p) 180))
    (dolist (c (plan-tree-children p)) 
      (clim:draw-line* stream
		       (+ (plan-tree-xpos c) 55) (- (plan-tree-ypos c) 20)
		       (+ (plan-tree-xpos c) 55) (plan-tree-ypos c))
      (draw-entry c stream))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Initialize the window when it is first created.
(defmethod initialize-instance :after ((frame vcr) &key)
  (let ((graph (clim:get-frame-pane frame 'graph))
	(movie (clim:get-frame-pane frame 'movie)))
    (clim:window-clear graph)
    (clim:window-clear movie)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Utility functions

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This routine is used to sleep between frames of an animation
;;;  it stops the animation if any button is pressed
(defun VCR-SLEEP (ws)
  (multiple-value-bind (gesture type)
      (clim:read-gesture :stream ws :timeout (/ 1 10))
    (declare (ignore gesture))
    (eq type :timeout)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This routine is used to resize the vcr window.
(defun SIZE-VCR-FRAME
    (root desired-left desired-top desired-width desired-height)
  (declare (values left top right bottom))
  (multiple-value-bind (left top right bottom)
      (clim:window-inside-edges root)
    (let ((desired-right (+ desired-left desired-width))
          (desired-bottom (+ desired-top desired-height)))
      (when (> desired-right right)
        (setf desired-right right
              desired-left (max left (- desired-right desired-width))))
      (when (> desired-bottom bottom)
        (setf desired-bottom bottom
              desired-top (max top (- desired-bottom desired-height))))
      (values desired-left desired-top desired-right desired-bottom))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Graphics routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This routine makes a histogram grapp of a plan.  The bars stand for
;;;  # open conditions, # links, # unsafe, # steps respectively.
(defun GRAPH-PLAN (stream node &key (fill nil) (label nil) &aux plan)
  (setf plan (plan-tree-entry node))
  (setf *max-open* (max *max-open* (plan-open plan))
	*max-links* (max *max-links* (length (plan-links plan)))
	*max-unsafe* (max *max-unsafe* (plan-unsafe plan))
	*max-steps* (max *max-steps* (length (plan-steps plan))))
  (flet ((draw-histo (fraction color)
	   (clim:draw-rectangle* stream 0 (* 110 (- 1 fraction)) 20 110
				 :ink color)))
    (cond ((plan-tree-displayed node)
	   (when (not fill)
	     (clim:draw-rectangle* stream 0 0 110 110 :ink clim::+background+)
	     (draw-histo (/ (plan-open plan) *max-open*) 
			 clim::+blue+)
	     (clim:with-translation (stream 30 0)
	       (draw-histo (/ (length (plan-links plan)) *max-links*) 
			   clim::+wheat+)
	       (clim:with-translation (stream 30 0)
		 (draw-histo (/ (plan-unsafe plan) *max-unsafe*) 
			     clim::+red+)
		 (clim:with-translation (stream 30 0)
		   (draw-histo (/ (length (plan-steps plan)) *max-steps*) 
			       clim::+green+)))))
	   (clim:draw-polygon* stream '(0 0 110 0 110 110 0 110)
			       :closed t :filled fill :line-thickness 1))
	  (t (clim:draw-polygon* stream '(0 0 110 0 110 110 0 110)
				 :closed t :filled nil :line-thickness 1)
	     (clim:draw-polygon* stream '(10 10 100 10 100 100 10 100)
				 :closed t :filled nil :line-thickness 1)))
    (when label
      (clim:draw-text stream (iconic-reason-label (plan-tree-entry node))
		      (clim:make-point 55 115) :align-x :center :align-y :top
		      :text-style '(:sans-serif :roman :tiny)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; display the next frame in the animation
(defun NEXT-FRAME (frame graph movie)
  (when (null (plan-tree-displayed frame))
    (setf (plan-tree-displayed frame) t) 
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry frame graph)))
  (clim:with-scaling (graph (/ 1 *tree-scale*))
    (when *movie-frame*
      (clim:with-translation (graph (plan-tree-xpos *movie-frame*) 
				    (plan-tree-ypos *movie-frame*))
	(graph-plan graph *movie-frame* :label (< *tree-scale* 16))))
    (setf *movie-frame* frame)
    (clim:with-translation (graph (plan-tree-xpos *movie-frame*) 
				  (plan-tree-ypos *movie-frame*))
      (graph-plan graph *movie-frame* :fill t :label (< *tree-scale* 16))))
  (multiple-value-bind (w h) (clim:window-inside-size movie)
    (clim:window-clear movie)
    (clim:with-scaling (movie (/ (min w h) 200))
      (clim:with-translation (movie 45 60)
	(graph-plan movie *movie-frame* :label t)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Print out na plan tree entry
(defun PRINT-PLAN-TREE (plan &optional (stream t) depth)
  (declare (ignore depth))
  (display-plan (plan-tree-entry plan) stream)
  (format stream "~%Reason: ~a" (reason (plan-tree-entry plan)) stream)
  (format stream "~%--------------------------------"))


(defun iconic-reason-label (plan)
  (let ((reason (cdr (assoc :reason (plan-other plan)))))
    (case (car reason)
     (:bogus "bogus")
     (:init "start")
     (:goal (format nil "G:~a" (caadr reason)))
     (:fact "fact")
     (:step
      (format nil "S:~a" 
	      (car (p-step-action 
		    (find (cadr reason) (plan-steps plan)
			  :key #'p-step-id)))))
     (:link
      (format nil "L:~a" (cadr reason)))
     (:cw-assumption "CWA")
     (:order
      (format nil "~a<~a" (cadr reason) (caddr reason))))))

(defun reason (plan)
  (let ((reason (cdr (assoc :reason (plan-other plan)))))
    (case (car reason)
      (:fact (format nil "Handle fact ~a" (cadr reason)))
      (:init "Initial plan")
      (:bogus "Remove bogus unsafe condition")
      (:goal 
       (format nil "Add goal [~a] for step [~d]" (cadr reason) (caddr reason)))
      (:step
       (format nil "Add step ~a to provide ~a"
	       (p-step-action 
		(find (cadr reason) (plan-steps plan)
		      :key #'p-step-id))
	       (caddr reason)))
      (:link
       (format nil "Link to step ~a for effect ~a" 
	       (cadr reason)
	       (caddr reason)))
      (:cw-assumption "Make a closed world assumption")
      (:order
       (format nil "Order step ~a before ~a" (cadr reason) (caddr reason))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 

(defun DISPLAY-A-PLAN (plan-entry)
  (declare (ignore ignore))
  (let* ((txt (clim:get-frame-pane clim:*application-frame* 'text))
         (plan (plan-tree-entry plan-entry))
         (steps (make-array (+ 1 (plan-high-step plan))))
         (order (top-sort (plan-ordering plan)
			  (plan-high-step plan)))
         (goal nil)
         (init nil)
	 (trace (assoc :trace (plan-other (plan-tree-entry plan-entry)))))
    (dolist (step-n (plan-steps plan))
      (cond
       ((eql (p-step-id step-n) :Goal)
        (setf goal step-n))
       ((eql (p-step-id step-n) 0)
        (setf init step-n))
       (t
        (setf (aref steps  (p-step-id step-n)) step-n))))
    (clim:window-clear txt)
    (clim:with-text-style ('(:sans-serif :roman :tiny) txt)
      (clim:write-string
       (format nil "~%Initial  : ~a~%"
               (mapcar #'(lambda (x)
                           (variable:bind-variable
                            x (plan-bindings plan)))
                       (effect-add
                        (car (p-step-add init)))))
       txt)
      (dotimes (i (plan-high-step plan))
        (display-step (aref steps (nth i order)) plan txt plan-entry))
      (display-step goal plan txt plan-entry)
      (clim:with-text-style ('(:sans-serif :roman :tiny) txt)
	(clim:write-string (format nil "~%Facts:") txt))
      (dolist (l (plan-flaws plan))
	(when (fact-p l)
	  (let ((msg (format nil " ~a"
			     (bind-variable (fact-condition l)
					    (plan-bindings plan)))))
	    (clim:with-text-style ('(:sans-serif :roman :tiny) txt)
	      (if (or (plan-tree-flaw plan-entry)
		      (eq (apply (fact-function l)
				 (mapcar 
				  #'(lambda (x) 
				      (bind-variable x (plan-bindings plan)))
				  (cdr (fact-condition l))))
			  :no-match-attempted))
                  (clim:write-string msg txt)
                (problem-button txt (list msg plan-entry l)))))))
      (unless (plan-flaws plan)
        (clim:write-string (format nil "~%Complete!") txt))
      (clim:write-string
       (format nil "~%Reason: ~a" (reason (plan-tree-entry plan-entry)))
       txt)
      (when trace 
	(clim:write-string (format nil "~%~a" (cdr trace))
			   txt)))))
  

;;;;;;;;;;;;;;;;
;;;
(defun display-step (step plan txt plan-entry)
  (if (numberp (p-step-id step))
      (clim:write-string
       (format nil "~%Step : ~20a   Created ~2a"
               (variable:bind-variable (p-step-action step)
                                       (plan-bindings plan))
               (p-step-id step))
       txt)
    (clim:write-string
     (format nil "~%~%Goal    : ~a" (p-step-precond step))
     txt))
  (dolist (l (plan-links plan))
    (when (eql (link-id2 l) (p-step-id step))
      (clim:write-string
       (format nil "~%              ~2a -> ~20a"
               (link-id1 l)
               (variable:bind-variable (link-condition l)
                                       (plan-bindings plan)))
       txt)
      (dolist (u (plan-flaws plan))
        (when (and (unsafe-p u) (eq l (unsafe-link u)))
          (let ((msg (format nil "<~a>"
                             (effect-id
                              (unsafe-clobber-effect u)))))
            (clim:with-text-style ('(:sans-serif :roman :tiny) txt)
	      (if (plan-tree-children plan-entry)
                  (clim:write-string msg txt)
                (problem-button txt (list msg plan-entry u)))))))))
  (dolist (l (plan-flaws plan))
    (when (and (openc-p l) (eql (openc-id l) (p-step-id step)))
      (let ((msg (format nil "~%              ?? -> ~20a"
                         (variable:bind-variable
                          (openc-condition l)
                          (plan-bindings plan)))))
        (clim:with-text-style ( '(:sans-serif :roman :tiny) txt)
          (if (plan-tree-children plan-entry)
              (clim:write-string msg txt)
            (problem-button txt (list msg plan-entry l))))))))


