" (c) 1990, 1991, 1992 Copyright (c) University of Washington
  Written by Tony Barrett, J Scott Penberthy, Stephen Soderland, and 
  Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-ucpop@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Program   : "UCPOP"
;;;
;;;  Description:
;;;  
;;;    These files contain the code for the planner UCPOP.  UCPOP
;;;    is a domain independent, partial order planner that handles
;;;    an expressive action representation (conditional effects,
;;;    universally quantified effects, and universally quantified
;;;    preconditions and goals).  UCPOP is both complete and sound.
;;;    The algorithm is described in:
;;;
;;;       Penberthy, J. S. and Weld, D., ``UCPOP: A Sound, Complete,
;;;       Partial-Order Planner for ADL,'' Third International
;;;       Conference on Knowledge Representation and Reasoning
;;;       (KR-92), Cambridge, MA, October 1992.
;;;
;;;    The representation of actions is a syntactic variant of a subset 
;;;    of Pednault's ADL.  See the BNF in the README file and the examples 
;;;    in domains.lisp
;;;
;;;    A plan includes a list of steps and a list of links between steps. 
;;;    Links are in the form  (id1 condition1 id2) where id1 is a step that
;;;    establishes condition1, which is in turn a precondition of step id2.
;;;    The plan also has a list of open conditions (not yet acheived) and a 
;;;    list of unsafe links (possibly clobbered by another step).  The other 
;;;    components of a plan are ordering constraints and bindings of variables.
;;;
;;;    Each iteration of the main control level creates possible next plans 
;;;    which are added to a priority queue.  New plans are created by adding
;;;    constraints to resolve unsafe links.  If there are no unsafe links,
;;;    new plans are created by adding a new step or new link to achieve
;;;    an open condition.  The plan is completed when there are no more unsafe
;;;    links or open conditions.  
;;;

(in-package "UCPOP")
(use-package "VARIABLE")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Outline of this file
;;;
;;;  1.   Handy interface for using the program
;;;  2.   Main control level of UCPOP
;;;  3.   Ranking partial plans
;;;  4.   Handling open conditions
;;;  4.1.   Adding new steps
;;;  4.2.   Adding links to old steps
;;;  4.3.   Adding links under closed world assumption
;;;  5.   Protecting causal links
;;;  5.1.   Resolving an unsafe link
;;;  5.2.   Detecting unsafety conditions
;;;  6.   handle partial orderings.
;;;  7.   Creating plan entries

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Handy interface for using the program

;;; This returns two values: a plan and a stat record
;;; If this is the top-level call, then both are printed.
(defun PLAN (initial
	     goals
	     &key
	     (rank-fun #'rank3)
	     (search-fun #'bestf-search))
  (multiple-value-bind (plan done? time q-len av-branch)
      (ucpop initial goals rank-fun search-fun)
    (values plan (make-stat :algo        "UCPOP"             
                            :date        (today)
                            :prob-num    1
                            :num-init    (length initial)       
                            :num-goal    (length goals)
                            :plan-len    (if plan (plan-high-step plan) 0)
                            :reached-max? (>= *nodes-visited* *search-limit*)
                            :complete?    done?
                            :time         time
                            :visited      *nodes-visited*     
                            :created      *plans-created*
                            :q-len        q-len
                            :ave-branch   (float av-branch)
                            :unify-count  *unify-count*
                            :rank-unifies *compute-rank-unifies*
                            :add-bindings *add-bind-count*)
	    )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Main control level of UCPOP

;;; Returns 5 values: final plan, t or nil denoting success, run time
;;; length of q at termination, average branching factor
(defun UCPOP (init goal rank-fun search-fun)
  (reset-stat-vars)			; clear globals for metering
  (test-wd goal nil)
  (let* ((init-time (get-internal-run-time))
	 (g (instantiate-term goal :goal))
	 (init-plan (tweak-plan 
		     nil
		     :reason '(:init)
		     :steps `(,(make-p-step :ID :Goal :precond g))  
		     :links nil
		     :unsafe nil
		     :open nil
		     :effects (list (make-effect :id 0 :add init))
		     :ordering nil
		     :bindings (new-bindings)
		     :high-step 0
		     :add-goal (make-openc :condition g
					   :id :goal))))
    (multiple-value-bind (plan bfactor qlength)
	(funcall search-fun init-plan
		 #'plan-refinements #'plan-test rank-fun *search-limit*)
      (values plan			; the plan itself
	      (and plan
		   (null (plan-unsafe plan))
		   (null (plan-open plan))) ; plan successfull?
	      (- (get-internal-run-time) init-time) ; time
	      qlength			; final length of the queue
	      bfactor))))		; average branching factor

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Goal test function for a search.
(defun PLAN-TEST (plan)
  (not (or (plan-unsafe plan)
	   (plan-open plan))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether one condition affects another.  This happens when one can
;;;  unify with the other or its negation.
(defmacro AFFECTS (cond1 cond2 bindings)
  `(unify (if (eq :not (car ,cond1)) (cadr ,cond1) ,cond1)
	  (if (eq :not (car ,cond2)) (cadr ,cond2) ,cond2)
	  ,bindings))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a list of one step refinements to the current plan.
;;;
;;;   Select one unsafe link and find all new plans that
;;; resolve it by separation, demotion, or promotion.  Separation is 
;;; adding bindings that prevent unifying of the condition with the
;;; added conditions of a possible clobbering step.  Demotion and
;;; promotion add ordering constraints so that a possibly clobbering
;;; step comes before the first step of the link or after the second
;;; step.
;;;   If there are no unsafe links, select an open condition to resolve
;;; by adding a new step or making a link to an existing step.  Adding 
;;; a new step or new link may cause a link to become unsafe (possibly 
;;; clobbered).
;;;
(defun PLAN-REFINEMENTS (plan)
  (if (plan-unsafe plan)
      (handle-unsafe (car (plan-unsafe plan)) plan)
      (handle-open (car (plan-open plan)) plan)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. ranking partial plans

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Perform an A* search on the number of causal links in a plan.
(defun RANK (plan)
  (let ((num-links (length (plan-links plan)))
	(open (length (plan-open plan))))
    (+ num-links open)))

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;;  To find a plan with the least searching.  Ranking is based on the
;;;  number of unsafe plus the number of goals to be resolved plus the
;;;  number of steps.  Including the number of steps prevents creating
;;;  long plans that add cycles of moves getting no closer to the goal.
(defun RANK3 (plan)
  (let ((num-steps (length (plan-steps plan)))
	(unsafe (length (plan-unsafe plan)))
	(open (length (plan-open plan))))
    (+ unsafe open num-steps)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Handling open conditions

(defun HANDLE-OPEN (open-cond plan)
  (cond ((eq (car (openc-condition open-cond)) :or)
	 (handle-or open-cond plan))
	(t (nconc (add-step open-cond plan)
		  (reuse-step open-cond plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Handling disjunction
(defun HANDLE-OR (goal plan &aux ret)
  (dolist (g (cdr (openc-condition goal)) (delete nil ret))
    (push (tweak-plan plan 
		      :reason `(:or ,goal)
		      :open (remove-1 goal (plan-open plan))
		      :add-goal (make-openc :condition g 
					    :id (openc-id goal)))
	  ret)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4.1. Adding new steps and reusing steps

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Returns a list of pairs of new plans and step#.  The new plan
;;; establishes the condition by adding a new step, instantiated 
;;; from a step in *templates* with post-conditions that unify with
;;; the condition. 
(defun ADD-STEP (open-cond plan)
  (let ((new-step-num (1+ (plan-high-step plan))))
    (new-link open-cond
	      (mapcar #'(lambda (templ) (instantiate-step templ new-step-num))
		      (get-opers (openc-condition open-cond)))
	      plan)))

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Quickly get those operators in the templates that match a condition
(defun GET-OPERS (condition &aux (ret nil))
  (labels ((test-templ (templ)
	     (dolist (e (p-step-add templ) nil)
	       (dolist (a (effect-add e))
		 (when (and (eql (car a) (car condition))
			    (or (not (eq (car a) :not))
				(eql (caadr a) (caadr condition))))
		   (return-from test-templ t))))))
    (dolist (templ *templates* ret)
      (when (test-templ templ)
	(push templ ret)))))
	
;;;;;;;;;;;;;;;;;;;
;;; 
(defun REUSE-STEP (open-cond plan)
  (let ((id (openc-id open-cond))
        (steps nil))
    (dolist (n (possibly-prior id plan))
      (push (if (eql n 0) :init
	      (dolist (s (plan-steps plan))
		(when (eql n (p-step-id s)) (return s))))
	    steps))
    (new-link open-cond steps plan)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4.2. Adding links to steps

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates new plans to achieve an open condition by creating a causal link.
;;;  If STEP is not nil then a new step is added to the plan.
;;;  Each effect tested to see if it has an add condition matching the 
;;;  open condition.  If so, a new plan is created with that link added.
;;;  Only the effects associated with STEPS are tried
(defun NEW-LINK (open-cond steps plan
		 &aux (new-plans nil) new-plan b)
  (let* ((condition (openc-condition open-cond))
	 (id (openc-id open-cond)))
    (dolist (step steps)
      (dolist (effect (cond ((eq :init step) (plan-effects plan))
			    ((> (p-step-id step) (plan-high-step plan))
			     (p-step-add step))
			    (t (plan-effects plan))))
	(when (= (if (eq :init step) 0 (p-step-id step)) (effect-id effect))
	  
	  (when (and (eq :init step) (eq (car condition) :not))
            (when (setf new-plan (new-cw-link open-cond id plan effect))
              (push new-plan new-plans)))
	  
	  (dolist (add-cond (effect-add effect))
	    (when (setf b (unify add-cond condition (plan-bindings plan)))
	      (let ((goal (peel-goal (car b) effect)))
		(when (and (> (effect-id effect) (plan-high-step plan))
			   (p-step-precond step))
		  (setf goal (if goal `(:and ,goal ,(p-step-precond step))
			       (p-step-precond step))))
		(setf new-plan
		  (tweak-plan 
		   plan
		   :reason (if (> (effect-id effect) (plan-high-step plan)) 
			       `(:step ,open-cond ,step ,effect ,add-cond) 
			     `(:link ,open-cond ,effect ,add-cond))
		   :effects (append (when (> (effect-id effect) 
					     (plan-high-step plan))
				      (p-step-add step))
				    (plan-effects plan))
		   :open (remove-1 open-cond (plan-open plan))
		   :steps (if (> (effect-id effect) (plan-high-step plan)) 
			      (cons step (plan-steps plan))
			    (plan-steps plan))
		   :links (cons (make-link :id1 (effect-id effect)
					   :condition condition
					   :id2 id)
				(plan-links plan))
		   :ordering (if (or (eql id :Goal) 
				     (eql (effect-id effect) '0))
				 (plan-ordering plan)
			       (cons (list (effect-id effect) id)
				     (plan-ordering plan)))
		   :bindings (add-bind (peel-binds (car b) effect)
				       (plan-bindings plan))
		   :add-goal (when goal (make-openc :condition goal
						    :id (effect-id effect)))
		   :high-step (if (> (effect-id effect) (plan-high-step plan))
				  (effect-id effect)
				(plan-high-step plan))))
		(when new-plan
		  (setf (plan-unsafe new-plan) 
		    (append (test-link new-plan (car (plan-links new-plan)))
			    (when (> (effect-id effect) (plan-high-step plan))
			      (test-effects new-plan (p-step-add step)))
			    (plan-unsafe new-plan)))
		  (push new-plan new-plans)
		  ))))))))
  new-plans)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4.3. Adding link to initial conditions for closed world model

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a link from a negated goal to the initial conditions
;;;  If a link can be added, a new plan is created with that link
;;;  A link will be added (from :not P to initial conditions) if
;;;
;;;   - for every initial condition Q, P does not unify with Q or
;;;   - for every initial condition Q with which P may unify, there
;;;       exists a binding constraint such that P and Q do not unify
;;;
;;;  If one of the above conditions holds, then a new plan is returned
;;;  with the link added, and any appropriate binding constraints added.

(defun NEW-CW-LINK (open-cond id plan effect)
  (let* ((condition (openc-condition open-cond))
         (bind-goals nil)
         (new-plan nil))
    
    (dolist (e (effect-add effect))
      (let ((b (unify (cadr condition) e (plan-bindings plan))))
	(when b
	  (setf b (car b))
	  (unless b (return-from NEW-CW-LINK nil))
	  (push (if (= 1 (length b))
		    `(:neq ,(caar b) ,(cdar b))
		  `(:or ,@(mapcar #'(lambda (x) 
				      `(:neq ,(car x) ,(cdr x)))
				  (car b))))
		bind-goals))))
    
    (setf new-plan
      (tweak-plan 
       plan
       :reason `(:link ,open-cond ,effect ,(effect-add effect))
       :open (remove open-cond (plan-open plan) :test #'eq)
       :links (cons (make-link :id1 0 
			       :condition condition 
			       :id2 id)
		    (plan-links plan))
       :add-goal (when bind-goals
		   (make-openc :condition `(:and ,@bind-goals)
			       :id (effect-id effect)))))
    
    (when new-plan
      (setf (plan-unsafe new-plan) 
	(append (test-link new-plan (car (plan-links new-plan)))
		(plan-unsafe new-plan))))
    new-plan))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Handling unsafe links

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Code used to handle a detected unsafety condition.  Returns a list
;;;  of plan refinements that resolve the threat.
(defun HANDLE-UNSAFE (unsafe-ln plan &aux binds)
  (if (and (my-member (unsafe-clobber-effect unsafe-ln) (plan-effects plan))
	   (setf binds (affects 
			(unsafe-clobber-condition unsafe-ln) ;if still exists..
			(link-condition (unsafe-link unsafe-ln))
			(plan-bindings plan)))
	   (my-member (effect-id (unsafe-clobber-effect unsafe-ln))
		      (possibly-between (link-id1 (unsafe-link unsafe-ln))
					(link-id2 (unsafe-link unsafe-ln))
					plan)))
      (nconc (disable unsafe-ln (car binds) plan)
	     (demote unsafe-ln plan)
	     (promote unsafe-ln plan))
    (list (tweak-plan plan
		      :reason `(:bogus ,unsafe-ln)
		      :unsafe (remove-1 unsafe-ln (plan-unsafe plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5.1. Resolving an unsafe link

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add goals to disable the errant effect.
;;;  Systematicity bug -- should add goals and anti goals O[2^preconds]
;;;                       (but only adds anti-goals O[preconds])

;;; *** Need to change for universals *** (Peel effects)

(defun DISABLE (unsafe-ln binds plan)
  (let* ((effect (unsafe-clobber-effect unsafe-ln))
	 (ord (nconc (when (< 0 (link-id1 (unsafe-link unsafe-ln)))
		       `((,(link-id1 (unsafe-link unsafe-ln)) 
			  ,(effect-id effect))))
		     (when (numberp (link-id2 (unsafe-link unsafe-ln)))
		       `((,(effect-id effect) 
			  ,(link-id2 (unsafe-link unsafe-ln)))))
		     (plan-ordering plan)))
	 (b (mapcar #'(lambda (x) `(:eq ,(car x) ,(cdr x)))
		    (peel-binds binds effect)))
	 (goal (if b `(:and ,@b ,(peel-goal binds effect))
		 (peel-goal binds effect)))
	 (p (tweak-plan 
	     plan 
	     :reason `(:disable ,unsafe-ln 
				,(effect-precond effect))
	     :ordering ord
	     :unsafe (remove-1 unsafe-ln (plan-unsafe plan))
	     :add-goal 
             (make-openc :condition `(:not ,goal)
			 :id (effect-id effect)))))
    (when p (list p))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes before the establishing step.  The output is a list of new
;;;  plans.
(defun DEMOTE (unsafe-ln plan)
  (let* ((clobber-id (effect-id (unsafe-clobber-effect unsafe-ln)))
	 (id (link-id1 (unsafe-link unsafe-ln)))
	 (demotable (my-member clobber-id (possibly-prior id plan))))
    (if demotable
	(list (tweak-plan plan :reason `(:demote ,unsafe-ln)
			  :unsafe (remove-1 unsafe-ln (plan-unsafe plan))
			  :ordering (cons (list clobber-id id)
					  (plan-ordering plan)))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes after the second step of the link.  The output is a list of new
;;;  plans.
(defun PROMOTE (unsafe-ln plan)
  (let* ((clobber-id (effect-id (unsafe-clobber-effect unsafe-ln)))
	 (link (unsafe-link unsafe-ln))
	 (id (link-id2 link))
	 (promotable (my-member id (possibly-prior clobber-id plan))))
    (if promotable
	(list (tweak-plan plan :reason `(:promote ,unsafe-ln)
			  :unsafe (remove-1 unsafe-ln (plan-unsafe plan))
			  :ordering (cons (list id clobber-id)
					  (plan-ordering plan)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5.2. Detecting unsafety conditions

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether a link is possibly clobbered by the delete conditions of any
;;;  steps possibly between the ID1 and ID2 of the link.  Returns nil if the 
;;;  link is safe, otherwise returns a list of (link clobber-id clobber-bind)
(defun TEST-LINK (plan link)
  (let ((new-unsafe nil)
	(bind2 (plan-bindings plan))
	(between-ids (intersection 
		      (possibly-prior (link-id2 link) plan)
		      (possibly-after (link-id1 link) plan))))
    (dolist (effect (plan-effects plan) new-unsafe)
      (when (my-member (effect-id effect) between-ids)
	(dolist (add-cond (effect-add effect))
	  (when (affects add-cond (link-condition link) bind2)
	    (push (make-unsafe :link link 
			       :clobber-effect effect 
			       :clobber-condition add-cond) 
		  new-unsafe)))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether a step possibly clobbers any link.  Returns nil if all
;;;  links are safe, otherwise returns a list of unsafe
;;;  Assumes that step is only constrained wrt start and a single other point.
;;;  This lets us only have to check if a link's source is prior to the step.
;;;
;;;  Warniing this assumes that all of the effects have the same id.
(defun TEST-EFFECTS (plan effects &aux (ret nil))
  (let ((prior (possibly-prior (effect-id (car effects)) plan))
	(after (possibly-after (effect-id (car effects)) plan)))
    (dolist (l (plan-links plan) ret)
      (when (and (my-member (link-id1 l) prior) 
		 (my-member (link-id2 l) after))
	(dolist (effect effects)
	  (dolist (c (effect-add effect))
	    (when (affects c (link-condition l) (plan-bindings plan))
	      (push (make-unsafe :link l 
				 :clobber-effect effect 
				 :clobber-condition c) 
		    ret))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. handle partial orderings.

(defun POSSIBLY-BETWEEN (s1 s2 plan)
  (intersection (possibly-after s1 plan) (possibly-prior s2 plan)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns list of step-id's of steps possibly prior to a given
;;; step.  Possibly prior always includes the initial conditions.
;;; First build a list of steps constrained to be not prior by
;;; the ordering constraints.  Then add to possibly prior all
;;; steps that aren't in the not-prior list.
(defun POSSIBLY-PRIOR (step-id plan)
  (when (not (eql step-id '0))
    (let ((table (make-list (1+ (plan-high-step plan))
			    :initial-element nil))
	  (poss-prior nil))
      (when (numberp step-id)
	(dolist (l (plan-ordering plan))
	  (push (cadr l) (nth (car l) table)))
	(do ((c (list step-id) (cdr c)))
	    ((null c) nil)
	  (when (listp (nth (car c) table))
	    (nconc c (nth (car c) table))
	    (setf (nth (car c) table) 'after))))
      (dotimes (n (1+ (plan-high-step plan)) poss-prior)
	(when (listp (car table))
	  (push n poss-prior))
	(pop table)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns list of step-id's of steps possibly after a given
;;; step.  Possibly after always includes the goal conditions.
;;; First build a list of steps constrained to be not prior by
;;; the ordering constraints.  Then add to possibly after all
;;; steps that aren't in the not-after list.
(defun POSSIBLY-AFTER (step-id plan)
  (when (not (eq step-id :goal))
    (let ((table (make-list (1+ (plan-high-step plan))
			    :initial-element nil))
	  (poss-after '(:goal)))
      (when (< 0 step-id)
	(dolist (l (plan-ordering plan))
	  (push (car l) (nth (cadr l) table)))
	(do ((c (list step-id) (cdr c)))
	    ((null c) nil)
	  (when (listp (nth (car c) table))
	    (nconc c (nth (car c) table))
	    (setf (nth (car c) table) 'before))))
      (dotimes (n (plan-high-step plan) poss-after)
	(when (listp (cadr table))
	  (push (1+ n) poss-after))
	(pop table)))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. Creating plan entries

;;;;;;;;;;;;;;;;
;;;  Compute the peelings of an effect caused by binding a universal variable.
;;;  The last effect returned is the actual unpeeled effect.
;;;
;;;  Assumption: the universal variables are always the car of a binding
(defun PEEL-GOAL (binds effect &aux (ret nil))
  (setf ret (copy-tree (effect-precond effect)))
  (dolist (b binds (if ret ret (effect-precond effect)))
    (when (my-member (car b) (effect-forall effect))
      (unless ret (setf ret (copy-tree (effect-precond effect))))
      (setf ret (nsubst (cdr b) (car b) ret)))))

(defun PEEL-BINDS (binds effect &aux (ret nil))
  (dolist (b binds ret)
    (unless (my-member (car b) (effect-forall effect)) (push b ret))))

;;;;;;;;;;;;;;;;
;;;  Create a modifixed version of PLAN
(defun TWEAK-PLAN (plan &key		; initially written by jsp
			reason
                        (steps :same)
                        (links :same)
                        (unsafe :same)
                        (open :same)
			(effects :same)
                        (ordering :same)
                        (bindings :same)
                        (high-step :same)
			(add-goal nil))
  "Return a new plan that is a copy of PLAN with the supplied
   instance variable tweaked."
  (flet ((tweak-it (keyword plan-accessor)
           (if (eq keyword :same)
               (funcall plan-accessor plan)
	     keyword)))
    (when (tweak-it bindings #'plan-bindings)
      (let ((plan1 (make-plan
		    :steps (tweak-it steps #'plan-steps)
		    :effects (tweak-it effects #'plan-effects)
		    :links (tweak-it links #'plan-links)
		    :unsafe (tweak-it unsafe #'plan-unsafe)
		    :open (tweak-it open #'plan-open)
		    :ordering (tweak-it ordering  #'plan-ordering)
		    :bindings (tweak-it bindings #'plan-bindings)
		    :high-step (tweak-it high-step #'plan-high-step)
		    :other (acons :reason reason nil))))
	(when add-goal
	  (setf plan1 (handle-and (list :and 
					(canonical (openc-condition add-goal)))
				  (openc-id add-goal)
				  plan1)))
	(when plan1
	  (push (cons :distance 
		      (if plan (1+ (cdr (assoc :distance (plan-other plan))))
			0))
		(plan-other plan1))
 	  (vcr-frame plan reason plan1)
	  )
	plan1))))

;;;;;;;;;;;;;;;;
;;; 
(defun HANDLE-AND (eqn id plan &aux (bs nil))
  (labels
      ((NEW-GOAL (n open links)
	 (when (equal n '(:not nil)) (return-from handle-and nil))
	 (dolist (o open) 
	   (when (eql (openc-id o) id)
	     (when (equal (openc-condition o) n) (return-from new-goal nil))
	     (when (negates (openc-condition o) n) (return-from handle-and nil))))
	 (dolist (l links) 
	   (when (eql id (link-id2 l))
	     (when (equal (link-condition l) n)
	       (return-from new-goal nil))
	     (when (negates (link-condition l) n) 
	       (return-from handle-and nil))))
	 t)
       (HANDLE-AND* (e)
	 (dolist (g (cdr e))
	   (cond ((eq (car g) :eq)
		  (push (cons (cadr g) (caddr g)) bs))
		 ((eq (car g) :neq)
		  (push (cons :not (cons (cadr g) (caddr g))) bs))
		 ((eq (car g) :and) 
		  (handle-and* g))
		 ((eq (car g) :forall)
		  (handle-and* (handle-forall g plan)))
		 ((eq (car g) :exists)
		  (handle-and* (handle-exists g)))
		 ((new-goal g (plan-open plan) (plan-links plan))
		  (push (make-openc :condition g :id id) (plan-open plan)))
		 (t nil)))))
    (handle-and* eqn)
    (setf bs (add-bind bs (plan-bindings plan)))
    (when bs
      (setf (plan-bindings plan) bs)
      plan)))

;;;;;;;;;;;;;;;;
;;;  See if one goal is a strict negation of the other.
(defun NEGATES (n1 n2)
  (let ((p1 (if (eq (car n1) :not) n1 n2))
	(p2 (if (eq (car n1) :not) n2 n1)))
    (and (eq (car p1) :not)
	 (equal p2 (cadr p1)))))

;;;;;;;;;;;;;;;;
;;; 
(defun HANDLE-FORALL (goal plan &aux temp)
  (let ((inits (effect-add 
		(car (last (plan-effects plan))))) ; assumes inits are last
	(opens (caddr goal)))
    (labels
	((handle* (vars alst)
	   (if (null vars) (push (v-sublis alst opens) temp)
	     (dolist (i inits)
	       (when (eq (car i) (caar vars))
		 (handle* (cdr vars) (acons (cadar vars) (cadr i) alst)))))))
      (handle* (cadr goal) nil))
    (cons :and temp)))

;;;;;;;;;;;;;;;;
;;;
(defun V-SUBLIS (alst e)
  (cond ((consp e) (cons (v-sublis alst (car e)) (v-sublis alst (cdr e))))
	((variable? e)
	 (let ((a (assoc e alst)))
	   (if a (cdr a) e)))
	(t e)))
	 
;;;;;;;;;;;;;;;;
;;; 
(defun HANDLE-EXISTS (goal)
  (let ((alst (mapcar #'(lambda (x) 
			  (cons (cadr x) 
				(uniquify-var (cadr x))))
		      (cadr goal))))
    (v-sublis alst `(:and ,(caddr goal) ,@(cadr goal)))))

;;;;;;;;;;;;;;;;
;;; 
(defun HANDLE-NOT (eqn)
  (let ((head (car eqn))
	(tail (cdr eqn)))
    (cond ((eq :eq head)  (cons :neq tail))
	  ((eq :neq head) (cons :eq tail))
	  ((eq :not head) (canonical (car tail)))
	  ((eq :or head)  (cons :and (mapcar #'handle-not tail)))
	  ((eq :and head) (cons :or (mapcar #'handle-not tail)))
	  ((eq :forall head) 
	   `(:exists ,(car tail) ,(handle-not (cadr tail))))
	  ((eq :exists head) 
	   `(:forall ,(car tail) ,(handle-not (cadr tail))))
	  (t (list :not eqn)))))

;;;;;;;;;;;;;;;;
;;; 
(defun CANONICAL (eqn)
  (cond ((eq :not (car eqn)) (handle-not (cadr eqn)))
	((or (eq (car eqn) :or) (eq (car eqn) :and))
	 (cons (car eqn) (mapcar #'canonical (cdr eqn))))
	((eq :forall (car eqn)) 
	 `(:forall ,(cadr eqn) ,(canonical (caddr eqn))))
	((eq :exists (car eqn)) 
	 `(:exists ,(cadr eqn) ,(canonical (caddr eqn))))
	(t eqn)))

;;;;;;;;;;;;;;;;
;;;  Make a plan and keep track of the total number of plans created.
(defun MAKE-PLAN (&rest args)
  (setq *plans-created* (+ 1 *plans-created*))
  (apply #'make-plan* args))

