(in-package 'spa)

;;;*******************************************************************
;;;  Here are some standard naive control functions.  There are 
;;;  three points where control functions guide the search:
;;;        1.  Decide which queue entry to consider next
;;;        2.  Decide which open/unsafe to extend next
;;;        3.  Decide which open/unsafe to retract next.
;;;
;;;  The first is passed as a parameter into ADAPT-PLAN.  
;;;  The second two are bound to variables *EXTEND-CONDITION-CHOOSER* 
;;;  and *RETRACT-CONDITION-CHOOSER* respectively.

;;;*********************************************************************
;;; Ranking functions:  take a queue entry (which is a direction plus 
;;; a plan) and returns a number.  The queue is processed in ascending order.
  
;;;*********************************************************************
;;; The default for CBR.  

(defun CRANK (e)
  (let* ((plan (qentry-iplan e))
         (num-steps (length (SNLP-plan-steps plan)))
         (unsafe (length (SNLP-plan-unsafe plan)))
         (open (length (SNLP-plan-open plan))))
    (floor (* (+ unsafe open (* 0.9 (- num-steps 2)))
              (if (eql (qentry-dir e) :retract) 120 80)))))

(defun CRANK0 (entry)
  (let* ((plan (qentry-iplan entry))
         (num-steps (length (SNLP-plan-steps plan)))
         (unsafe (length (SNLP-plan-unsafe plan)))
         (open (length (SNLP-plan-open plan))))
    (+ unsafe open num-steps -2)))

;;;*********************************************************************
;;; Experimental.  Might work well when the input plans tend to be 
;;; "bigger" than the library plans.  It seems that unsafe conditions
;;;  are not generally so bad, so not clear why they're weighted as 
;;;  heavily as they are. 

(defun refine-first-then-reduce-open (e)
  (let* ((plan (qentry-iplan e))
         (num-unsafe (length (SNLP-plan-unsafe plan)))
         (num-open (length (SNLP-plan-open plan)))
         (extend? (eql (qentry-dir e) ':extend)))
    (+ (* 20 num-unsafe) num-open (if (not extend?) 1000 0))))


(defun prefer-extends (qent)
  (let ((real-rank (* (if (eq ':extend (qentry-dir qent)) 1 500)
                      (or (snlp-plan-rank (qentry-iplan qent))
                          (rank (qentry-iplan qent))))))
    real-rank))

;;;*************************************************************************
;;;   Choosing what to extend or retract next.  Note that these have to be 
;;;   bound to the right globals to have any effect.

;;;  Here are two simple options for an extend chooser:  choose the first 
;;;  open or choose the first unsafe. 

(defun CHOOSE-UNSAFE-FIRST (plan)
  (cond
   ((snlp-plan-unsafe plan)
    (car (snlp-plan-unsafe plan)))
   ((snlp-plan-open plan)
    (choose-open-cond (snlp-plan-open plan)))
   (t NIL)))

(defun CHOOSE-OPEN-FIRST (plan)
  (cond
   ((snlp-plan-open plan)
    (choose-open-cond (snlp-plan-open plan)))
   ((snlp-plan-unsafe plan)
    (car (snlp-plan-unsafe plan)))
   (t NIL)))

;;;*************************************************************************
;;;  Default is to choose the first open
;;;*************************************************************************

;;; This belongs to Denise.... not sure why it's here, but 
;;; seems like a good idea to preserve it!
(defun CHOOSE-OPEN-COND (opens)
  (or (find-if #'(lambda (o) (cond-handsoff (open-condition o))) opens)
;     (find 'on opens :key #'(lambda (o) (car (cond-form (open-condition o)))))
      (car (last opens))))

(setf *extend-condition-chooser* 'choose-open-first)

;;;****************************************************************************
;;; Choosing for retraction

;;; Only one function at the moment: 
;;;   Prefer links over bindings over ordering decisions
;;; Retract choice of achieving on rather than clear if possible

(defun CHOICE-TO-RETRACT (ip)
  (let ((choice-list
	 (remove-if-not #'(lambda (d) (retractable-decision d ip))
			(snlp-plan-decisions ip))))
  (if *auto-choose*
      (auto-choose choice-list ip)
      (man-choose  choice-list ip))))

(defun AUTO-CHOOSE (choice-list ip)
  (debug-msg :retract "   ~2a Possible retractions" (length choice-list))
  (debug-progn :retract-o
    (dolist (c choice-list)
      (display-decision c ip)))
    
  ;; Always get rid of silly steps first, then links, ...
  (or (find :new-step choice-list :key #'decision-type)
      (find :new-link choice-list :key #'decision-type)
      (car choice-list)))

;;; requires utils/cmenu.lisp
;; Items are specified as (<printed form> <value returned>). 
(defun MAN-CHOOSE (choice-list ip)
  (format t "~%Retracting decision that lead to plan ~a" ip)
  (display-plan ip)
  (let ((menu nil))
    (dolist (b (decisions-of-type :separate choice-list))
      (push (list (display-decision b ip nil) b) menu))
    (dolist (o (decisions-of-type :promote choice-list))
      (push (list (display-decision o ip nil) o) menu))
    (dolist (o (decisions-of-type :demote choice-list))
      (push (list (display-decision o ip nil) o) menu))
    (dolist (l (decisions-of-type :new-link choice-list))
      (push (list (display-decision l ip nil) l) menu))
    (dolist (s (decisions-of-type :new-step choice-list))
      (push (list (display-decision s ip nil) s) menu))

    (utils::cmenu-choose menu "Choose what to Retract")))

(defun DECISIONS-OF-TYPE (type list)
  (remove-if-not #'(lambda (d) (eq (decision-type d) type)) list))


;;;*************************************************************************
;;; Set the default
;;;************************************************************************

(setf *retract-condition-chooser* 'choice-to-retract)

