" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Tony Barrett, Steve Hanks, Steve Soderland and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

(in-package 'spa)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Program   : "SNLPlan"
;;;  Author    : Stephen Soderland  
;;;              (modified & extended by Dan Weld and Tony Barrett)
;;;              (further modified by Denise Draper)
;;;  Date      : Spring 1992
;;;
;;;  Description:
;;;  
;;;    This program is  a  domain independent,  conjuctive, non-linear
;;; planner which   claims   to  be  both  complete  and  sound.   The
;;; representation of  conditions and actions  is  similar  to STRIPS,
;;; with each  step having  a list  of added conditions  and a list of
;;; deleted conditions.
;;;
;;; A plan includes a list of steps and a list of links between steps.
;;; Links are in the  form (id1 condition1  id2)  where id1 is  a step
;;; that establishes condition1, which  is in turn a  precondition  of
;;; step id2.  The plan also  has a list  of open conditions (not  yet
;;; acheived)  and  a  list  of  unsafe  links  (possibly clobbered by
;;; another step).  The  other  components  of   a plan are   ordering
;;; constraints and bindings of variables.
;;;
;;; Each  iteration of the  main  control level creates possible  next
;;; plans which are added to a priority  queue.  New plans are created
;;; by adding constraints  to resolve unsafe  links.   If there are no
;;; unsafe links, new plans  are created by  adding a new  step or new
;;; link to achieve an  open  condition.  The plan  is  completed when
;;; there are no more unsafe links or open conditions.
;;;  
;;;    SNLPlan    is adapted  from  a    non-linear  planner by  David
;;; McAllester.   As  the first phase  of  this project, I implemented
;;; Tweak, a   non-linear  planner described    by  David  Chapman  in
;;; Artificial Intelligence  1987.   Code from  my  Tweak program  was
;;; incorporated   into  SNLPlan.  The  data  structures  for variable
;;; bindings were later revised to enhance performance.
;;;

;;; Modification by Draper (supercedes mods by DSW) 3/92:
;;; In order to support retraction, the plan structure is supplemented
;;; by a  list of "decisions"  made (such as  add-step, promote, etc.)
;;; It  is these decisions which   get retracted.  Additionally, steps
;;; and links contain references  to  decisions  made on their behalf;
;;; this is used  to determine when  decisions may be  retracted.  See
;;; the comments in structs.lisp for more details.
;;;
;;; Implementation note: in an effort to  minimize consing, extensions
;;; to a plan  do not copy  the entire plan  structure, only that part
;;; which might be  changed.  {Note that most  of the time  lists  are
;;; only being consed onto, and so not changed.}  What can  be changed
;;; are:  the bindings   (constraints  structure), and  the  decisions
;;; references inside steps and links.  When the latter sort of change
;;; is made, we must copy  the step/link, and  the list which contains
;;; it.   To  make this slightly simpler  and more  robust, there is a
;;; special   version  of copy,  called   copy-plan-for-extend,  which
;;; automatically copies the link and step lists {not their contents},
;;; and optionally the bindings structure.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Possible bugs
;;; 1. After calls to unify, is it ok just to take the car of the resulting
;;;    list or does this throw away possible alternatives?
;;; 2. Are we violating systematicity by testing for unsafe links the
;;;    way we do?  In other words is it possible to add a constraint that
;;;    separates variable in one threatening step and later generate the
;;;    same (or equivalent because of an = constraint) constraint when
;;;    separating variables in another threatening step.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Outline of this file
;;; 
;;; (1. and 2. are now in cbr.lisp)
;;; 3. Creating a list of one step refinements to a partial plan
;;; 4. Resolving an unsafe link
;;; 5. Adding a step
;;; 6. Handling links

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Creating a list of one step refinements to a partial plan

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Creates a list of one step refinements to the current plan.
;;;
;;; Select one unsafe link and find all  new  plans that resolve it by
;;; separation, demotion, or promotion.  Separation is adding bindings
;;; that prevent unifying of the  condition with  the added conditions
;;; of a  possible   clobbering  step.  Demotion   and promotion   add
;;; ordering  constraints  so that  a possibly  clobbering  step comes
;;; before  the first step of the  link or after the  second step.
;;;
;;; The function *extend-condition-chooser* takes a partial plan and returns 
;;; the next condition to work on (either an open or an unsafe).  This 
;;; is supplied by the application.  Some examples and a default setting 
;;; appear in CHOOSE-FUNS.LISP

(defvar *extend-condition-chooser*)

(defun PLAN-REFINEMENTS (plan)
  (mapc #'check-unsafes! 
    (let ((next-condition (funcall *extend-condition-chooser* plan)))
      (cond
       ((unsafe-p next-condition) 
        (resolve-unsafe next-condition plan))
       ((open-p next-condition)
        (resolve-open next-condition plan))
       (t (error "Don't know how to refine ~d" next-condition))))))

;;;**************************************************************
;;;  Adding constraints (either in the process of closing open 
;;;  conditions or resolving unsafes) may cause other unsafes to 
;;;  go away.  Therefore we check each unsafe of each plan before 
;;;  before enqueueing it to see if all of its unsafe conditions 
;;;  are still problematic.  If not, we delete the corresponding 
;;;  condition.

;;;  Note:  all experimental evidence shows that this check isn't 
;;;  worth the effort (at least when the planner adopts the 
;;;  strategy of working on open conditions first).  Therefore 
;;;  I'm turning it off.

(defun CHECK-UNSAFES! (plan)
  (declare (ignore plan))
  (values))

;;; (defun CHECK-UNSAFES! (plan)
;;;  (let ((unsafes-to-keep '()))
;;;    (dolist (u (snlp-plan-unsafe plan))
;;;      (cond
;;;       ((really-unsafe? u plan)
;;;        (push u unsafes-to-keep))
;;;       (t (debug-msg :unsafe-check "Deleting unsafe ~a from plan ~a" u plan))))
;;;    (setf (snlp-plan-unsafe plan) unsafes-to-keep))
;;;  (values))
;;;
;;;(defun REALLY-UNSAFE? (u plan)
;;;  (let* ((clobbered-link (get-link (unsafe-link u) plan))
;;;         (clobbered-prop (link-condition clobbered-link))
;;;         (clobbering-step (unsafe-clobber-step u))
;;;         (plan-bindings (snlp-plan-bindings plan)))
;;;    (and (member clobbering-step 
;;;                 (possibly-between 
;;;                  (link-producer clobbered-link)
;;;                 (link-consumer clobbered-link)
;;;                  plan))
;;;         (some #'(lambda (prop) 
;;;                   (condxs-possibly-conflict? prop clobbered-prop plan-bindings))
;;;               (step-postcond (get-step clobbering-step plan))))))
;;;
;;;(defun CONDXS-POSSIBLY-CONFLICT? (prop1 prop2 bindings)
;;;  (not (eq :FAIL (unify (cond-form prop1) (cond-form prop2) bindings))))

;;;**********************************************

(defun RESOLVE-UNSAFE (unsafe plan)
  (let ((plan2 (remove-unsafe unsafe plan)))
    (debug-msg :extend "REFINE: Removing unsafe ~a: ~a => ~a" 
               unsafe plan plan2)
	(when *debug-save-tree* (setf *dbst-resolved* unsafe))
    (nconc (separate unsafe plan2)
	       (demote unsafe plan2)
	       (promote unsafe plan2))))

(defun RESOLVE-OPEN (open plan)
  (let ((plan2 (remove-open open plan)))
    (debug-msg :extend "REFINE: Removing open ~a: ~a => ~a" 
               open plan plan2)
	(when *debug-save-tree* (setf *dbst-resolved* open))
	(delete-if-not #'rtvar-ordering-ok
                   (nconc (add-step open plan2)
                          (new-link open plan2)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a new current plan with the specified unsafe link removed, and
;;;  returns the the new plan.
(defun REMOVE-UNSAFE (unsafe plan)
  (make-SNLP-plan
   :steps (SNLP-plan-steps plan)
   :links (SNLP-plan-links plan)
   :unsafe (remove unsafe  (SNLP-plan-unsafe plan) :test #'eq)
   :open (SNLP-plan-open plan) 
   :ordering (SNLP-plan-ordering plan)
   :bindings (SNLP-plan-bindings plan)
   :decisions (SNLP-plan-decisions plan)
   :high-step (SNLP-plan-high-step plan)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a new current plan with the specified open condition removed, and
;;;  returns the new plan
(defun REMOVE-OPEN (open plan)
  (make-SNLP-plan
   :steps (SNLP-plan-steps plan)
   :links (SNLP-plan-links plan)
   :unsafe (SNLP-plan-unsafe plan)
   :open (remove open (SNLP-plan-open plan) :test #'eq)
   :ordering (SNLP-plan-ordering plan)
   :bindings (SNLP-plan-bindings plan)
   :decisions (SNLP-plan-decisions plan)
   :high-step (SNLP-plan-high-step plan)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Resolving an unsafe link

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add constraints that prevent the bindings that
;;;  would clobber it.  The input is an UNSAFE structure, and the output is
;;;  a list of new plans.

(defun SEPARATE (unsafe plan)
  (let ((new-plans nil))
    (dolist (bind (unsafe-clobber-bind unsafe))
      (let ((neg-bind (negate-cf bind)))
	(when (and
	       (constraint-consistent? neg-bind (SNLP-plan-bindings plan))
	       (handsoff-consistent? bind (get-link (unsafe-link unsafe) plan)))
	  (debug-msg 0.5  "   * Separate with ~a" neg-bind) 
	  (let ((new-decision (make-decision
			       :type ':separate
			       :link (unsafe-link unsafe)
			       :step (unsafe-clobber-step unsafe)
			       :cf-list (list neg-bind)
			       :unsafe unsafe))
		(new-plan (copy-plan-for-extend plan :cs t)))

	    ;; add the new decision
	    (add-constraint! neg-bind (SNLP-plan-bindings new-plan))
	    (push new-decision (SNLP-plan-decisions new-plan))
	    ;; add the cross references for this decision
	    (add-new-protect new-plan (unsafe-link unsafe)
			     (decision-id new-decision))
	    (add-new-avoid new-plan (unsafe-clobber-step unsafe)
			   (decision-id new-decision))
	    ;;
	    (push new-plan new-plans)))))
    (values new-plans)))


;;; Return true if the proposed separation does not involve any
;;; variables from a handsoff link.  (i.e. the variables must be in
;;; the threatening link, or the unsafe link must not be a handsoff
;;; one).
(defun HANDSOFF-CONSISTENT? (clobber-bind unsafe-link)
  (or (not (cond-handsoff (link-condition unsafe-link)))
      (notany #'(lambda (v)
		  (or (eq (cf-var1 clobber-bind) v)
		      (eq (cf-var2 clobber-bind) v)))
	      (cond-vars (link-condition unsafe-link)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the
;;;  clobber step comes before the establishing step.  The output is a
;;;  list of (zero or one) plans.

(defun DEMOTE (unsafe plan)
  (let* ((clobber-id (unsafe-clobber-step unsafe))
         (estab-id   (link-producer (get-link (unsafe-link unsafe) plan)))
         (demotable  (member clobber-id (possibly-prior estab-id plan))))
    (debug-msg 1 "   Demote? step ~a" clobber-id)
    (cond
      (demotable
       (debug-msg 0.5  "   * Demoting step ~a < ~a" clobber-id estab-id)
       (let* ((new-ordering (make-ordering :pred clobber-id
					   :succ estab-id))
	      (new-decision (make-decision :type ':demote
					   :ordering (ordering-id new-ordering)
					   :link (unsafe-link unsafe)
					   :step clobber-id
					   :unsafe unsafe))
	      (new-plan (copy-plan-for-extend plan)))
	 (push new-ordering (SNLP-plan-ordering new-plan))
	 (push new-decision (SNLP-plan-decisions new-plan))
	 ;;
	 (add-new-protect new-plan (unsafe-link unsafe)
			  (decision-id new-decision))
	 (add-new-avoid new-plan clobber-id (decision-id new-decision))
	 ;;
	 (list new-plan)))
      (t nil))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes after the second step of the link.  The output is a list of
;;;  (zero or one) new plans.

(defun PROMOTE (unsafe plan)
  (let* ((clobber-id (unsafe-clobber-step unsafe))
         (use-id     (link-consumer (get-link (unsafe-link unsafe) plan)))
         (promotable (member use-id (possibly-prior clobber-id plan))))
    (debug-msg 1 "   Promote? step ~a ~a" clobber-id promotable)
    (cond
      (promotable
       (debug-msg 0.5 "   * Promoting step ~a" clobber-id)
       (let* ((new-ordering (make-ordering :pred use-id
					   :succ clobber-id))
	      (new-decision (make-decision :type ':promote
					   :ordering (ordering-id new-ordering)
					   :link (unsafe-link unsafe)
					   :step clobber-id
					   :unsafe unsafe))
	      (new-plan (copy-plan-quick plan)))
	 (push new-ordering (SNLP-plan-ordering new-plan))
	 (push new-decision (SNLP-plan-decisions new-plan))
	 ;;
	 (add-new-protect new-plan (unsafe-link unsafe)
			  (decision-id new-decision))
	 (add-new-avoid new-plan clobber-id (decision-id new-decision))
	 ;;
	 (list new-plan)))
      (t nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Adding a step

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Returns a list of new plans.  Each new plan establishes the
;;; condition by adding a new step, instantiated from a step in
;;; *templates* with post-conditions that unify with the condition. 
;;; Modified by DSW 1/91 to reduce consing

(defun ADD-STEP (open-cond plan)
  (let ((new-plans nil)
        (condition (open-condition open-cond))
        (id (open-step-id open-cond)))
    (unless (cond-handsoff condition)
      (debug-msg .5 "ADD-STEP: Trying to fill ~a" open-cond)
      (dolist (templ *templates*) 
	(debug-msg 1 "New step? ~a" (step-action (car templ)))
	(let* ((new-step-num (+ (snlp-plan-high-step plan) 1))
	       (new-step     (rename-step (car templ) new-step-num))
	       (step-cfs     (rename-term (cadr templ) new-step-num))
	       (templ-bind   (copy-cs (snlp-plan-bindings plan))))
	
	  (add-constraints! step-cfs templ-bind)
	
	  (dolist (post-cond (if *aggressive*
				 (nthcdr (step-effect-count new-step)
					 (step-postcond new-step))
			       (step-postcond new-step)))
	    (let ((new-bind (test-value-and-unify
			     condition post-cond templ-bind)))
	      (when (not (eq new-bind ':FAIL))
		(debug-msg 0.5 "* New Step ~a  New bindings ~a"
			   (step-action new-step) new-bind)
		(debug-msg 0 "S")
		;; adding a new step consists of two parts: adding the
		;; step itself, and adding the link between the new step
		;; and the open condition.  the decisions are separate
		;; because they can be retracted independently
		(let* ((step-decision (make-decision :type ':new-step
						     :step (step-id new-step)
						     :cf-list step-cfs))
		       (new-plan (copy-plan-for-extend plan)))

		  (push new-step (SNLP-plan-steps new-plan))
		  (push step-decision (SNLP-plan-decisions new-plan))
		  (setf (SNLP-plan-bindings new-plan) (copy-cs templ-bind))
		  (setf (SNLP-plan-high-step new-plan) (step-id new-step))
		  (insertf (SNLP-plan-open new-plan) (new-step-open new-step))
		
		  (make-new-link! (step-id new-step) condition id
				  new-bind new-plan)
		  ;; check to see if adding this new step clobbers
		  ;; any other steps
		  (dolist (link (SNLP-plan-links new-plan))
		    (insertf (SNLP-plan-unsafe new-plan)
			     (test-link new-plan link new-step)))
		
		  (push new-plan new-plans))))))))
    new-plans))

;;;;;;;;;;;;;;;;;;;
;;; Tests if two conditions have the same truth values,
;;; and if the propositions unify; returns :FAIL or diffs
(defun TEST-VALUE-AND-UNIFY (cond1 cond2 bindings)
  (if (eq (cond-value cond1) (cond-value cond2))
      (unify (cond-form cond1) (cond-form cond2) bindings)
      :FAIL))

;;;;;;;;;;;;;;;;;;;
(defun NEW-STEP-OPEN (step)
  (mapcar #'(lambda (pc) (make-open :condition pc :step-id (step-id step)))
	  (step-precond step)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. Handling links
  
;;;;;;;;;;;;;;;;;;;;;;;;
;;; Tests whether a link is possibly clobbered or superceded by the
;;; post conditions of any steps possibly between the ID1 and ID2 of
;;; the link.  Returns nil if the link is safe, otherwise returns a
;;; list of UNSAFE structures
  
(defun TEST-LINK (plan link &optional (st nil))
  (let ((new-unsafe nil)
        (between-ids (possibly-between (link-producer link) 
                                       (link-consumer link) plan))
        (real-step-list (if st (list st) (snlp-plan-steps plan))))
    (debug-msg 1 "Test link ~a with steps ~a" link between-ids)
    (dolist (step real-step-list)
      (when (member (step-id step) between-ids)
        (debug-msg 2 " Test step ~a" (step-id step))

 	;; attempt to unify the propositions of the post-conditions,
 	;; regardless of whether they are asserted true, false, or
 	;; unknown (if they disagree in value with the link, then
 	;; they directly threaten it; but check even if it _agrees_
 	;; with the link, for systematicity).

        (dolist (threat (step-postcond step))
          (debug-msg :test-link "   Testing condition ~a" threat)
          (let ((clobber-bind 
                 (unify (cond-form threat)
                        (cond-form (link-condition link))
                        (snlp-plan-bindings plan))))
            (when (not (eq clobber-bind ':FAIL))
              (push (make-unsafe :link         (link-id link)
                                 :clobber-step (step-id step) 
                                 :clobber-bind clobber-bind)
                    new-unsafe)
              (debug-msg 0.5 "   * New unsafe ~a "
			 (list link (step-id step) clobber-bind)))))))
    new-unsafe))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns a list of step# that are possibly prior to the
;;; step being acheived, but not before the establishing step.
(defun POSSIBLY-BETWEEN (estab-id id plan)
  (let ((not-after (list estab-id 0))
        (poss-between nil))

    ;; construct not-after list
    (do* ((queue (list estab-id) (cdr queue))
          (na-step estab-id (car queue)))
         ((null queue))
      (dolist (order (SNLP-plan-ordering plan))
	(when (eql na-step (ordering-succ order))
	  (setf not-after (cons (ordering-pred order) not-after))
	  (nconcf queue (list (ordering-pred order))))))

    (dolist (prior-id (possibly-prior id plan))
      (if (not (member prior-id not-after))
	  (push prior-id poss-between)))
    poss-between))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns list of step-id's of steps possibly prior to a given
;;; step.  Possibly prior always includes the initial conditions.
;;; First build a list of steps constrained to be not prior by
;;; the ordering constraints.  Then add to possibly prior all
;;; steps that aren't in the not-prior list.

(defun POSSIBLY-PRIOR (step-id plan)
  (cond
    ((init-step-id? step-id)
     '())
    (t (let ((not-prior (list step-id ':GOAL)))
         (do ((queue (list step-id)))
             ((null queue) (values))
           (let ((next-step (car queue)))
             (setq queue (cdr queue))
             (dolist (order (snlp-plan-ordering plan))
               (when (and (eql next-step (ordering-pred order))
                          (not (member (ordering-succ order) not-prior)))
                 (push (ordering-succ order) queue)
                 (push (ordering-succ order) not-prior)))))
         (let ((poss-prior '(0)))
           (dotimes (n (SNLP-plan-high-step plan))
             (if (not (member (+ n 1) not-prior))
                 (push (+ n 1) poss-prior)))
           poss-prior)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates new plans to achieve an open condition.  Each possibly prior step
;;;  is tested to see if it has an add condition matching the open condition.
;;;  If so, a new plan is created with that link added.
  
(defun NEW-LINK (open-cond plan)
  (if (cond-handsoff (open-condition open-cond))
      (add-new-handsoff-link open-cond plan)
      (add-new-ordinary-link open-cond plan)))

(defun ADD-NEW-ORDINARY-LINK (open-cond plan)
  (let* ((new-plans nil)
         (condition (open-condition open-cond))
         (id (open-step-id open-cond))
         (prior-ids (possibly-prior id plan)))
    (debug-msg 2 "Possibly prior steps ~a" prior-ids)
    (debug-msg .5 "NEW-LINK: Trying to fill ~a" open-cond)
    (dolist (step (SNLP-plan-steps plan))
      (when (member (step-id step) prior-ids)
        (debug-msg 1 "New link?  step ~a" (step-id step))
        (dolist (post-cond (if *aggressive*
			       (nthcdr (step-effect-count step)
				       (step-postcond step))
			       (step-postcond step)))
          (let ((new-bind (test-value-and-unify condition post-cond
						(SNLP-plan-bindings plan))))
	    (when (not (eq new-bind ':FAIL))
	      (let ((new-plan (copy-plan-for-extend plan :cs t)))
		(make-new-link! (step-id step) condition id
				new-bind new-plan)
                (debug-msg 0.5 " * New link ~a , new bindings ~a"
			   (car (SNLP-plan-links new-plan)) new-bind)
                (debug-msg 0 "L")     ; -- SH
		(push new-plan new-plans)))))))
    new-plans))

;;;;;;;;;;;;;;;;;;;

(defun ADD-NEW-HANDSOFF-LINK (open-cond plan)
  (let* ((condition (open-condition open-cond))
         (id (open-step-id open-cond))
	 (new-plan (copy-plan-for-extend plan)))
    (debug-msg .5 "HANDSOFF-LINK: Establishing ~a" open-cond)
    (make-new-link! 0 condition id nil new-plan)
    (list new-plan)))


;;;;;;;;;;;;;;;;;;;
;;; the grunt work of making a new link: we have to make a link, a
;;; decision for that link, an ordering for that link, and check to
;;; see if other steps threaten the new link

(defun MAKE-NEW-LINK! (p-step condx c-step bind plan)
  (let* ((ordering (if (or (init-step-id? p-step)
			   (goal-step-id? c-step))
		       nil
		     (make-ordering :pred p-step :succ c-step)))
	 (new-link (make-link :producer p-step 
			      :condition condx
			      :consumer c-step
			      :bindings bind
			      :ordering (if ordering (ordering-id ordering))))
	 (decision (make-decision :type ':new-link
				  :link (link-id new-link))))

    (add-constraints! bind (SNLP-plan-bindings plan))
    (when ordering
      (push ordering (SNLP-plan-ordering plan)))
    (push new-link (SNLP-plan-links plan))
    (push decision (SNLP-plan-decisions plan))
    (insertf (SNLP-plan-unsafe plan) (test-link plan new-link))

    ;; cross references
    (add-new-produce plan p-step (decision-id decision))
    (add-new-consume plan c-step (decision-id decision)))
  (values))

;;;;;;;;;;;;;;;;;;;
;;;
;;; Add ordering constraints which ensure that any use of
;;; an rtvar (i.e. any occurance in pre- or post-conditions
;;; of a step) occurs after the step which establishes the
;;; rtvar.
;;; This modifies and returns the plan if successful,
;;; else returns nil.

(defun RTVAR-ORDERING-OK (plan)
  plan)