(in-package 'spa)

;;;**********************************************************************
;;;
;;; These are utilities for fitting a plan's initial and goal conditions.
;;; We expect these to be called by the library routines themselves.
;;;
;;; Fitting library plan to current conditions:
;;;
;;;  Fixing initial conditions given 
;;;   If any INIT condition is in LIB-INITIAL but not REAL-INITIAL.
;;;     1.  Delete that condition from the plan's initial conditions.
;;;     2.  Change any link of the the form 
;;;           (1 INIT k)
;;;         to an open condition of the form (INIT k)
;;;
;;;  Fixing goal conditions
;;;   If form G is in the LIB-GOALS but not the REAL-GOALS
;;;      1. Delete any link that produces G for the GOAL step.
;;;   If form G is in the REAL-GOALS but not the LIB-GOALS
;;;      1.  Add an OPEN condition for G in the GOAL
;;;
;;; Note that we're assuming we have a copy of the plan at this point, 
;;; so we're side-effecting.

(defun fit-plan (plan initial goal)
  (let ((new-plan (copy-plan-completely plan)))
    (fit-plan! new-plan initial goal)
    new-plan))


(defun fit-plan! (plan new-inits new-goals)
  (flet ((ins (x) (instantiate x (snlp-plan-bindings plan)))
         (undo-link (l)
           (retract-decision
            (get-decision-establishing-link (link-id l) plan) plan)))
    ;; get rid of no-longer-correct links
    (dolist (l (snlp-plan-links plan))
      (if (init-step-id? (link-producer l))
          (unless (member (ins (link-condition l)) new-inits :test #'equalp)
            (undo-link l)
            (setf l nil)))
      (if (and l (goal-step-id? (link-consumer l)))
	  (unless (member (ins (link-condition l)) new-goals :test #'equalp)
	    (undo-link l))))
    ;; get rid of useless opens
    (setf (snlp-plan-open plan)
      (delete-if #'(lambda (o)
		     (and (goal-step-id? (open-step-id o))
			  (not (member (ins (open-condition o))
				       new-goals :test #'equalp))))
		 (snlp-plan-open plan)))
    ;; add new opens for new goal conditions
    (let ((ins-lib-goals (ins (snlp-plan-goal-conditions plan))))
      (dolist (new-goal new-goals)
	(unless (member new-goal ins-lib-goals :test #'equalp)
	  (push (make-open :condition new-goal
			   :step-id ':Goal)
		(snlp-plan-open plan)))))
    ;; finally, insert the new conditions
    (setf (snlp-plan-initial-conditions plan) new-inits)
    (setf (snlp-plan-goal-conditions plan) new-goals)))
