;;;*******************************************************************
;;; A domain theory is one or more identifying symbols plus a list of STEP 
;;; data structures.   The way to load one is 
;;;     (LOAD-DOMAIN-THEORY domain-name)
;;;

(in-package 'spa)

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This function is used to define plan steps of a domain theory.

;;; New 1/5/92 DSW
;;; Effects are just like props on the postcond list except this operator
;;; will not be used to achieve that prop
;;; *** See also define-step in domains.lisp
(defun DEFSTEP (&key action precond (handsoff nil) postcond 
		     (effect nil) (equals nil) (cost 0))
  (push (list (make-step
	       :action action
	       :precond (append (mapcar #'as-condx precond)
				(mapcar #'as-handsoff handsoff))
	       :postcond (mapcar #'as-condx (append effect postcond))
	       :cost cost
               :effect-count (length effect))
	      (mapcar #'as-cf equals))
	*templates*))


;;;********************

(defun load-sussman-ops ()
  ;; Define step for putting a block on the table.
  (defstep 
      :action '(newtower ?x)
      :precond  '((on ?X ?Z) (clear ?X))
      :postcond '((on ?X Table) (clear ?Z) (not (on ?X ?Z)))
      :equals '((not (?X ?Z)) (not (?X Table)) (not (?Z Table))))
   ;; Define step for placing one block on another.
  (defstep 
      :action '(puton ?X ?Y)
      :precond '((on ?X ?Z) (clear ?X) (clear ?Y))
      :postcond '((on ?X ?Y) (clear ?Z) (not (on ?X ?Z)))
      :equals '((not (?X ?Y)) (not (?X ?Z)) (not (?Y ?Z))
                (not (?X Table)) (not (?Y Table)))))
(defparameter *sa-blocksworld-names* '(sa-blocksworld sussman-anomaly blocksworld))

;;;****************************

(defun  load-regswap-ops ()
   ;; Define step for swapping two registers
   (defstep :action '(move ?sval ?rs ?rd ?dval)
     :precond  '((contains ?rs ?sval) (contains ?rd ?dval))
     :postcond '((contains ?rd ?sval) (not (contains ?rd ?dval)))
     :equals '((not (?rs ?rd)))))
(defparameter *regswap-names* '(regswap register-swap))

;;;****************************

(defun load-truckworld-ops ()
   (defstep
    :action '(move-fast ?A ?B)
    :precond '((truckloc ?A) (fuel-tank-full))
    :postcond '((truckloc ?B) (fuel-tank-empty)
	       (not (truckloc ?A) (not (fuel-tank-full))))
    :equals '((not (?A ?B))))
   (defstep
    :action '(move-slow-1 ?A ?B)
    :precond '((truckloc ?A) (fuel-tank-full) )
    :postcond '((truckloc ?B) (fuel-tank-medium)
	       (not (truckloc ?A)) (not (fuel-tank-full)))
    :equals '((not (?A ?B))))
   (defstep
    :action '(move-slow-2 ?A ?B)
    :precond '((truckloc ?A) (fuel-tank-medium) )
    :postcond '((truckloc ?B) (fuel-tank-empty)
	       (not (truckloc ?A)) (not (fuel-tank-medium)))
    :equals '((not (?A ?B))))
   (defstep
    :action '(fill-tank ?A)
    :precond '((truckloc ?A) (GasPumpat ?A))
    :postcond '((fuel-tank-full)
	       (not (fuel-tank-medium)) (not (fuel-tank-empty)))
    :equals '())
   (defstep
    :action '(grab-using-arm1 ?X)
    :precond '((small ?X) (in ?X ?A) (truckloc ?A))
    :postcond '((hold-bay1 ?X) (not (in ?X ?A)))
    :equals '((not (?X ?A))))
   (defstep
    :action '(grab-using-arm2 ?X)
    :precond '((big ?X) (in ?X ?A) (truckloc ?A))
    :postcond '((hold-bay2 ?X) (not (in ?X ?A)))
    :equals '((not (?X ?A))))
   (defstep
    :action '(putdown-using-arm1 ?X ?A) 
    :precond '( (small ?X) (hold-bay1 ?X) (truckloc ?A))
    :postcond '((in ?X ?A) (not (hold-bay1 ?X)))
    :equals '((not (?X ?A))))
   (defstep
    :action '(putdown-using-arm2 ?X ?A) 
    :precond '( (big ?X) (hold-bay2 ?X) (truckloc ?A))
    :postcond '((in ?X ?A) (not (hold-bay2 ?X)))
    :equals '((not (?X ?A)))))
(defparameter *truckworld-names* '(truckworld))

;*****

(defun load-rao-ops ()
   (defstep
    :action '(put-block-on-block ?x ?z ?y)
    :precond '((cleartop ?x) (cleartop ?y) (on ?x ?z))
    :postcond '((cleartop ?z) (on ?x ?y)
	       (not (cleartop ?y)) (not (on ?x ?z)))
    :equals '((not (?x table)) 
              (not (?y table)) 
              (not (?y ?z))
              (not (?x ?z))
              (not (?x ?y))))
   (defstep
    :action '(put-block-on-table ?x ?z TABLE)
    :precond '((cleartop ?x) (on ?x ?z))
    :postcond '((cleartop ?z) (on ?x TABLE)
	       (not (on ?x ?z)))
    :equals '((not (?z TABLE)) (not (?x TABLE)) (not (?x ?z)))))
(defparameter *rao-blocksworld-names* '(rao rao-experiments rao-blocksworld 
                                  alternative-blocksworld))

(defparameter *domains* 
  (list (list *sa-blocksworld-names* 'load-sussman-ops)
        (list *truckworld-names* 'load-truckworld-ops)
        (list *rao-blocksworld-names* 'load-rao-ops)
        (list *regswap-names* 'load-regswap-ops)))

;;;************************************************************************
;;; Loading a domain:  find the name, reset the templates.
;;; Note that we're not copying the templates.  I'm counting on the assumption
;;; that SNLP doesn't side effect these units.

(defparameter *current-domain-names* '())

(defun load-domain (domain-name)
  (let ((the-domain 
	 (find-if #'(lambda (domain-ent) 
		      (member domain-name (car domain-ent) :test #'eq))
		  *domains*)))
    (cond
      ((null the-domain) 
       (error "Couldn't find domain for ~a" domain-name))
      (t (setf *templates* '())
         (setf *current-domain-names* (car the-domain))
         (funcall (cadr the-domain))))))

(defmacro with-domain (domain-id &rest body)
  `(let* ((fun-to-call #'(lambda () ,@body))
		  (old-domain-name nil))
	 (setf old-domain-name (car *current-domain-names*))
	 (load-domain ,domain-id)
 	(prog1
      (funcall fun-to-call)
	  (load-domain old-domain-name))))

(defun domain-load-check (&optional (stream *standard-output*))
  (when (null *templates*)
    (format stream
      "WARNING:  No domain loaded.  Nothing interesting likely to happen.~%")))

(defun reset-domain ()
  (setf *templates* nil))
    
