(in-package 'spa)
;;;************************************************************
;;; Contents:
;;;   1) Debug-msg and friends (trace output)
;;;   2) Plan-stat  (statistics about the last planning run)
;;;   3) Time-stat  (general timing statistics tool)
;;;   4) Save-tree  (save entire search space for examination)
;;;   5) Check-plan (make sure a plan is consistent)
;;;
;;;************************************************************
;;; 1) Debugging functions
;;;
;;;    (add-debug &rest tags)
;;;       add TAGS to *TRACE*, and return value of *TRACE*
;;;    (un-debug &rest tags)
;;;       delete TAGS from *TRACE*, and return value of *TRACE*
;;;
;;;    (debug-msg tag &rest msgs)
;;;       if TAG appears on *TRACE*, print the message
;;;    (debug-display tag object)
;;;       if TAG appears on *TRACE*, display object
;;;    (debug-progn tag &body forms)
;;;       if TAG appears in *TRACE*, do the PROGN and return its value.

(defconstant *TRACE-TYPES* 
  '(0                                   ; "Plan at Current Node"
    0.5                                 ; ???
    1                                   ; * New step
    2                                   ; New step?
    :retract                            ; general retraction info
    :retract-o                          ; more detailes stuff for retraction
    :extend                             ; minimal info about CBR use of SNLP
    :q                                  ; what goes on and off CBR queue
    :q-d                                ; detailed info of what is dequeued
    :cs                                 ; adding contraints
    :fit                                ; when plans get fitted
    :planlib                            ; fetching library plans
    :unsafe-check                       ; discarding superfluous unsafe conditions
    ))

(defparameter *TRACE* '())
(defparameter *DEBUG-IO* *standard-output*)

(defun add-debug (&rest tags)
  (mapcar #'(lambda (tag) (pushnew tag *trace*)) tags)
  *trace*)

(defun un-debug (&rest tags)
  (cond
   ((null tags) (setq *trace* '()))
   (t (setq *trace*
	(delete-if #'(lambda (tag) (member tag *trace*)) *trace*))))
  *trace*)

(defun debug-msg (tag &rest msgs)
  (when (member tag *trace*)
    (fresh-line *debug-io*)
    (apply #'format (cons *debug-io* msgs))))

;;; Of course the right way to do this is to associate a display function
;;; with each structure...
(defun debug-display (tag object)
  (when (member tag *trace*)
    (cond ((snlp-plan-p object)
           (display-plan object *debug-io*))
          ((q-p object)
           (display-q object *debug-io*))
          ((qentry-p object)
           (display-qentry object *debug-io*))
          (t (print object))
          )))

(defmacro debug-progn (tag &rest forms)
  `(when (member ,tag *trace*)
     (progn ,@forms)))

;;;********************************************************************
;;; 2) Plan-stat

(defvar *plan-stat* nil)                ; plan-stat of last planning run
(defvar *planning-run* 0)
(defparameter *print-plan-stat* nil)      ; will print at end of planning run

(defstruct (plan-stat (:print-function print-plan-stat))
  (algo 'spa)
  (date (today))                        ; when performed
  (prob-num (incf *planning-run*))
  num-init                              ; how many initial conditions
  num-goal
  plan-len                              ; how many steps
  reached-max?                          ; terminated because of nodes?
  complete?                             ; planner successful
  time                                  ; internal cpu time
  visited                               ; nodes-visted
  created                               ; calls to make-plan
  q-len                                 ; queue len at termination
  ave-branch                            ; average branching factor
  )


(defun set-plan-stat (q ip fp)
  ;; args are terminal q, initial plan, final plan
  (let ((done-time (get-internal-run-time)))
    (setf *plan-stat*
      (make-plan-stat :num-init   (length (snlp-plan-initial-conditions ip))
		      :num-goal   (length (snlp-plan-goal-conditions ip))
		      :plan-len   (if fp (length (snlp-plan-steps fp)) 0)
		      :reached-max? (resource-limit-exceeded? q)
		      :complete?  (and fp (finished-plan? fp))
		      :time       (- done-time (q-cpu-time-created q))
		      :visited    (q-iplans-dequeued q)
		      :created    (q-iplans-enqueued q)
		      :q-len      (q-length q)
		      :ave-branch 0))	; --- bogus
    (when *print-plan-stat*
      (format t "~a" *plan-stat*))))
					

(defun print-plan-stat (s &optional (st t) ignore)
  (declare (ignore ignore))
  (format st "~%~6aRun~2a  (Init = ~2a ; Goals = ~a) => ~a (~a steps)"
          (plan-stat-algo s) (plan-stat-prob-num s)
	  (plan-stat-num-init s) (plan-stat-num-goal s)
          (if (plan-stat-complete? s) "Win " "Lose")
          (plan-stat-plan-len s))
  (format st "~%      Nodes  (V = ~4a; Q = ~4a; C = ~a)"
          (plan-stat-visited s) (plan-stat-q-len s) (plan-stat-created s))
  (format st "~%      CPU    ~9a       Branch    ~9a"
	  (plan-stat-time s) (plan-stat-ave-branch s)))

  
;;;  Get today's date
(defun today ()
  (let ((d (multiple-value-list (get-decoded-time))))
    (format nil "~a/~a ~a at ~a:~a:~a"
            (nth 4 d) (nth 3 d) (nth 5 d) (nth 2 d) (nth 1 d) (nth 0 d))))

;;;******************************************************************
;;; 3) Timing statistics:
;;;     (time-stat &rest messages)
;;; If variable *PRINT-TIME-STAT* is non-null, print the messages along
;;; with how long has occurred since the last call to STAT.
;;;
;;; Function also returns the amount of time elapsed since last call.

(defvar *last-time* (get-internal-run-time))
(defparameter *print-time-stat* t)

(defun time-stat (&rest messages)
  (let* ((abs-time (get-internal-run-time))
         (elapsed-time (- abs-time *last-time*)))
    (when *print-time-stat*
      (let ((format-string (car messages))
            (vars (cdr messages)))
        (fresh-line)
        (apply #'format
               (cons t
                     (cons (concatenate 'string 
                             "STAT: "
                             format-string
                             " -- absolute ~d elapsed ~d~%")
                           (append vars (list abs-time elapsed-time)))))))
    (setq *last-time* abs-time)
    elapsed-time))

;;;************************************************************
;;; 4) Save-tree
;;; If *debug-save-tree* is non-nil, the entire search space will be
;;; saved in the q-entries (see cstructs.lisp).  Each plan is given a
;;; name of the form IPnnn, and the symbol IPnnn is set to the plan,
;;; and the symbol QEnnn is set to its qentry.  To navigate though
;;; this structure, inspect whichever qentry you want.
;;;   Needed: some way to specialize this to only save parts
;;; of the search space.
;;;
;;;  A less radical version is *INTERN-PLAN-NAMES* which gives you 
;;;  the IPnnn bindings but does not save the queue's structure in 
;;;  the entry.

(defparameter *DEBUG-SAVE-TREE* nil)	; Use with caution!!
(defparameter *INTERN-PLAN-NAMES* nil)	
(defvar *dbst-last-plan* nil)           ; last plan dequeued
(defvar *dbst-most-retracted* nil)      ; highest plan on retraction hierarchy
(defvar *dbst-gensym* 0)
(defvar *dbst-resolved*)		; used for internal communication

;;; this is what is stuck into qentry-children.
;;; given a qentry, produce (list d qentry), where d
;;; is the pertinent decision relating to the production
;;; of qentry.  note that we fetch the add-step decision,
;;; if there was one.
(defun qd-list (qentry)
  (let* ((d-list (snlp-plan-decisions (qentry-iplan qentry)))
	 (d1 (car d-list))
	 (d2 (cadr d-list)))
    (cond ((null d1) (list nil qentry))
	  ((eq (decision-type d1) :new-link)
	   (cond ((null d2) (list d1 qentry))
		 ((eq (decision-type d2) :new-step)
		  (list d2 qentry))
		 (t (list d1 qentry))))
	  (t (list d1 qentry)))))

;;; make a new name for a plan + qentry.
;;; intern if *debug-save-tree* set

(defun set-names (qentry)
  (when (or *debug-save-tree* *intern-plan-names*)
    (incf *dbst-gensym*)
    (let* ((plan-name-string (format nil "IP~a" *dbst-gensym*))
           (plan-name (intern plan-name-string 'spa)))
      (setf (snlp-plan-id (qentry-iplan qentry)) plan-name)
      (set plan-name (qentry-iplan qentry))))
  (when *debug-save-tree* 
    (let* ((qe-name-string (format nil "QE~a" *dbst-gensym*))
           (qe-name (intern qe-name-string 'spa)))
      (set qe-name qentry))))

(defun clear-names ()
  (dotimes (i *dbst-gensym*)
    (makunbound (intern (format nil "IP~a" i) 'spa))
    (makunbound (intern (format nil "QE~a" i) 'spa)))
  (setf *dbst-gensym* 0))


;;;******************************************************************
;;; 5) Check-plan
;;; Check plans for static consistency -- i.e. that all the
;;; cross-references match, and that certain easily verifiable
;;; relationships hold

(defun check-plan (p &key (error nil))
  (let ((error-count 0))

    ;; steps
    (dolist (s (snlp-plan-steps p))
      (dolist (d (step-producing-decisions s))
	(unless (get-decision d p)
	  (format t "Step ~a references non-existent decision ~a~%" s d)
	  (incf error-count)))
      (dolist (d (step-consuming-decisions s))
	(unless (get-decision d p)
	  (format t "Step ~a references non-existent decision ~a~%" s d)
	  (incf error-count)))
      (dolist (d (step-avoiding-decisions s))
	(unless (get-decision d p)
	  (format t "Step ~a references non-existent decision ~a~%" s d)
	  (incf error-count)))
      (unless (or (eq (step-id s) ':Goal)
		  (<= (step-id s) (snlp-plan-high-step p)))
	(format t "Step-id of ~a greater than high-step (~a)~%"
		s (snlp-plan-high-step p))
	(incf error-count)))

    ;; links
    (dolist (l (snlp-plan-links p))
      (unless (get-step (link-producer l) p)
	(format t "Link ~a references non-existent step ~a~%"
		l (link-producer l))
	(incf error-count))
      (unless (get-step (link-consumer l) p)
	(format t "Link ~a references non-existent step ~a~%"
		l (link-consumer l))
	(incf error-count))
      (when (link-ordering l)
	(unless (get-ordering (link-ordering l) p)
	  (format t "Link ~a references non-existent ordering ~a~%"
		  l (link-ordering l))
	  (incf error-count)))
      (dolist (b (link-bindings l))
	(when (constraint-consistent? (negate-cf b)
				      (snlp-plan-bindings p))
	  (format t "Link ~a has bindings which do not hold in ~a~%"
		  l (snlp-plan-bindings p))
	  (incf error-count)))
      (dolist (d (link-protecting-decisions l))
	(let ((dd (get-decision d p)))
	  (unless dd
	    (format t "Link ~a references non-existent decision ~a~%"  l d)
	    (incf error-count))
	  (when dd
	    (unless (or (eq (decision-type dd) ':promote)
			(eq (decision-type dd) ':demote)
			(eq (decision-type dd) ':separate))
	      (format t "Link ~a references a non-protecting decision ~a~%" l dd)
	      (incf error-count))))))

    ;; unsafes
    (dolist (u (snlp-plan-unsafe p))
      (let ((l (get-link (unsafe-link u) p))
	    (s (get-step (unsafe-clobber-step u) p)))
	(unless l
	  (format t "Unsafe ~a references non-existent link ~a~%"
		  u (unsafe-link u))
	  (incf error-count))
	(unless s
	  (format t "Unsafe ~a references non-existent step ~a~%"
		  u (unsafe-clobber-step u))
	  (incf error-count))
	(when (and l s)
	  (let ((umatch 0))
	    (dolist (post (step-postcond s))
	      (let ((bind (test-value-and-unify
			   post (link-condition l)
			   (snlp-plan-bindings p))))
		(unless (eq bind ':FAIL)
		  (when (cf-list-equiv? bind (clobber-bind u))
		    (incf umatch)))))
	    (unless (> 0 umatch)
	      (format t "Unsafe ~a does not match bindings~%")
	      (incf error-count))))))

    ;; open
    (dolist (o (snlp-plan-open p))
      (let ((s (get-step (open-step-id o) p)))
	(unless s
	  (format t "Open ~a references non-existent step ~a~%"
		  o (open-step-id o))
	  (incf error-count))
	(when s
	  ;; have to instantiate these guys because fitted plans
	  ;; won't match exactly
	  (let ((prelist
		 (instantiate (step-precond s) (snlp-plan-bindings p))))
	    (unless
		(find (instantiate (open-condition o) (snlp-plan-bindings p))
		      prelist :test #'equalp)
	      (format t "Open ~a has bogus condition for step~%     ~a~%" o s)
	      (incf error-count))))))

    ;; ordering
    (dolist (o (snlp-plan-ordering p))
      (unless (get-step (ordering-pred o) p)
	(format t "Ordering ~a references non-existent step ~a~%"
		o (ordering-pred o))
	(incf error-count))
      (unless (get-step (ordering-succ o) p)
	(format t "Ordering ~a references non-existent step ~a~%"
		o (ordering-succ o))))

    ;; decisions
    (dolist (d (snlp-plan-decisions p))
      (let ((s (get-step (decision-step d) p))
	    (l (get-link (decision-link d) p))
	    (o (get-ordering (decision-ordering d) p)))
	(ecase (decision-type d)
	  ((:new-step)
	   (unless s
	     (format t "Decision ~a references non-existent step ~a~%"
		     d (decision-step d))
	     (incf error-count)))
	  ((:new-link)
	   (unless l
	     (format t "Decision ~a references non-existent link ~a~%"
		     d (decision-link d))
	     (incf error-count))
	   (when l
	     (let ((p-step (get-step (link-producer l) p))
		   (c-step (get-step (link-consumer l) p)))
	       (when p-step
		 (unless (member (decision-id d)
				 (step-producing-decisions p-step))
		   (format t "Step ~a doesn't have producing decision ~%     ~a~%"
			   p-step d)
		   (incf error-count)))
	       (when c-step
		 (unless (member (decision-id d)
				 (step-consuming-decisions c-step))
		   (format t "Step ~a doesn't have consuming decision ~%     ~a~%"
			   c-step d)
		   (incf error-count))))))
	  ((:promote :demote :separate)
	   (unless s
	     (format t "Decision ~a references non-existent step ~a~%"
		     d (decision-step d))
	     (incf error-count))
	   (unless l
	     (format t "Decision ~a references non-existent link ~a~%"
		     d (decision-link d))
	     (incf error-count))
	   (unless (or (eq (decision-type d) :separate) o)
	     (format t "Decision ~a references non-existent ordering ~a~%"
		     d (decision-ordering d))
	     (incf error-count))
	   (when s
	     (unless (member (decision-id d) (step-avoiding-decisions s))
	       (format t "Step ~a doesn't have avoiding decision ~%     ~a~%"
		       s d)
	       (incf error-count)))
	   (when l
	     (unless (member (decision-id d) (link-protecting-decisions l))
	       (format t "Link ~a doesn't have protecting decision ~%     ~a~%"
		       l d)
	       (incf error-count)))
	   (when (eq (decision-type d) :separate)
	     (dolist (b (decision-cf-list d))
	       (when (constraint-consistent?  (negate-cf b)
					      (snlp-plan-bindings p))
		 (format t "Separation ~a doesn't achieve ~a in ~a~%"
			 d b (snlp-plan-bindings p))
		 (incf error-count))))))))
  (when (and error (> 0 error-count))
    (error "Plan inconsistent"))
  (values)))
