(in-package 'spa)

;;; If the below is true, every CS created will be saved on the
;;; list, and then can be retrieved by ID number.  Use it with
;;; caution!
(defvar *save-cs* nil)
(defvar *cs-list* '())

;;; If *cs-paranoid* is true, all attempts to modify a CS will
;;; be checked for consistency (and an error invoked if failure)

(defvar *cs-paranoid* t)

;;;****************************************************************
;;;  OBJECTS --
;;;  objects are variables or constants

;;;  ------------------
;;;  Ordinary Variables

(defun o-variable? (thing)
  (and (symbolp thing)
       (char= #\? (elt (symbol-name thing) 0))))

(defun make-o-variable (&rest args)
  (intern (concatenate 'string "?" (transform-names args))))

;;;  ------------------
;;;  Run-Time Variables

(defun rt-variable? (thing)
  (and (symbolp thing)
       (char= #\! (elt (symbol-name thing) 0))))

(defun make-rt-variable (&rest args)
  (intern (concatenate 'string "!" (transform-names args))))

;;;  -------------------
;;;  Knowledge Variables

(defun k-variable? (thing)
  (and (symbolp thing)
       (char= #\@ (elt (symbol-name thing) 0))))

(defun make-k-variable (&rest args)
  (intern (concatenate 'string "@" (transform-names args))))

;;;  ---------
;;;  Constants

(defun constant? (thing)
  (and (symbolp thing)
       (not (null thing))
       (not (variable? thing))))

(defun plan-time-constant? (thing)
  "true if thing is a constant or run-time variable"
  (and (symbolp thing)
       (not (null thing))
       (let ((c1 (elt (symbol-name thing) 0)))
	 (not (or (char= #\? c1) (char= #\@ c1))))))

;;;  ---------

(defun variable? (thing)
  (and (symbolp thing)
       (let ((c1 (elt (symbol-name thing) 0)))
	 (or (char= c1 #\?) (char= c1 #\!) (char= c1 #\@)))))

(defun plan-time-variable? (thing)
  "true if thing is an ordinary or knowledge variable"
  (and (symbolp thing)
       (let ((c1 (elt (symbol-name thing) 0)))
	 (or (char= c1 #\?) (char= #\@ c1)))))

;;;  ---------

(defun variable-name (vble)
  (subseq (symbol-name vble) 1))

(defun variable-equal? (v1 v2)
  (eq v1 v2))

(defun make-same-variable (as-var &rest args)
  "make a variable of the same type as as-var"
  (intern (concatenate 'string
		       (symbol-name as-var)
		       (transform-names args))))

(defun transform-names (arg-list)
  (cond
    ((null arg-list) "")
    (t (concatenate 'string 
		    (transform-name (car arg-list))
		    (transform-names (cdr arg-list))))))

(defun transform-name (thing)
  (cond
    ((variable? thing) (variable-name thing))
    ((symbolp thing)   (symbol-name thing))
    ((stringp thing)   (string-upcase thing))
    ((numberp thing)   (format nil "~d" thing))
    (t (error "Don't know how to transform ~a" thing))))

;;;****************************************************************
;;;  VARSETS --
;;;
;;;  A VARSET represents an equivilence class of co-designating
;;;  variables.  The identity of a VARSET is ultimately determined
;;;  by the unique constant or run-time variable to which it is bound
;;;  (in fact, the constant and its VARSET will often be referred to
;;;  interchangeably).  At partial stages of computation, a VARSET
;;;  will generally contain only an aggregated set of co-designating
;;;  variables.  Such a VARSET (and its variables) will be called
;;;  'unbound.'  VARSETs also contain annotation indicating other
;;;  VARSETs with which they are necessarily-non-codesignating
;;;  (i.e. they cannot be merged).
;;;
;;;  =========== enclosed comments not implemented =================
;;;  Knowledge variables may always co-designate with run-time
;;;  variables, but if the caller requests, they will be restricted
;;;  from co-designating with constants or unbound VARSETs.
;;;
;;;  Run-time variables have the added restriction that they
;;;  cannot necessarily-non-codesignate with any other VARSET.
;;;  Additionally, at "run-time" rt-vars may be bound to any constant
;;;  (resulting in the merger of their respective VARSETs).
;;;  ===============================================================
;;;
;;;  VARSETs are represented as structures, which have the
;;;  following fields:
;;;    bound-const     -- the identifying constant or run-time
;;;                       variable, or nil if unbound
;;;    var-list        -- list of all co-designated vars (and constant)
;;;    ncd-list        -- an association list; the vars contain
;;;                       exemplar variables from VARSETs which cannot
;;;                       codesignate with this VARSET.  The cdrs are
;;;                       lists of CFs which established this ncd.
;;;                       These lists are symmetric: if v1 appears
;;;                       in v2's ncd-list, then v2 must appear in
;;;                       v1's ncd-list.
;;;    cd-list         -- list of all equality cf's used to construct
;;;                       VARSET (used to reconstruct the VARSET
;;;                       when a CF is retracted)
;;;    size            -- (+ (length var-list) (length ncd-list))
;;;                       used for deciding which way to merge varsets
;;;
;;;  {Implementation note: the reason for using exemplar variables
;;;   in the ncd-lists, instead of storing the VARSET itself, is that
;;;   when a constraint-set is copied, the ncd-lists would have to be
;;;   updated to contain the new VARSETs, whereas the exemplar variables
;;;   remain unchanged.  The cost is an extra variable lookup when
;;;   searching the ncd-list, but experience shows this is negligable
;;;   compared to the cost of making COPY-CS slow.}
;;;
;;;  Finally, a variable or constant by itself is interpreted as a
;;;  a varset with a single member and no constraints.


(defstruct (varset
	     (:print-function print-varset))
  (bound-const nil)
  (var-list nil)
  (ncd-list nil)
  (cd-list nil)
  (size 0))

(defun print-varset (self stream depth)
  (declare (ignore depth))
  (format stream "<VARSET ~a>" 
	  (or (varset-bound-const self) (varset-var-list self))))

(defun display-varset (self &optional (stream *debug-io*) (indent 0))
  (indent-stream stream indent)
  (format stream "Varset vars ~a~%" (varset-var-list self))
  (display-list  "ncds" (varset-ncd-list self) stream (+ indent 7))
  (indent-stream stream (+ indent 7))
  (format t  "cds~%")
  (indent-stream stream (+ indent 9))
  (format t "~a~%" (varset-cd-list self)))
  
(defun copy-varset (self)
  (make-varset
   :bound-const (varset-bound-const self)
   :var-list (copy-list (varset-var-list self))
   :ncd-list (mapcar #'copy-list (varset-ncd-list self))
   :cd-list (copy-list (varset-cd-list self))
   :size (varset-size self)))

(defun make-single-varset (ob)
  (make-varset
   :bound-const (if (plan-time-constant? ob) ob nil)
   :var-list (list ob)
   :size 1))

(defun make-double-varset (ob1 ob2 cf)
  (make-varset
   :bound-const (if (plan-time-constant? ob1)
		    ob1
		    (if (plan-time-constant? ob2)
			ob2
			nil))
   :var-list (list ob1 ob2)
   :cd-list (list cf)
   :size 2))


;;;****************************************************************
;;;  CS --
;;;  A constraint set contains a list of VARSETs and a hash table
;;;  mapping variables and constants onto their VARSET.

(defvar *cs-num* 0)          ; generate unique id's for CS's
(defvar *cs-table-size* 20)

(defstruct (cs
	     (:print-function print-cs)
	     (:constructor really-make-cs))
  (id          (incf *cs-num*))
  (var-table   (make-hash-table :size *cs-table-size* :test #'eq))
  (varset-list nil))

(defun make-cs ()
  (let ((the-cs (really-make-cs)))
    (when *save-cs* (push the-cs *cs-list*))
    (values the-cs)))

;;;****************************************************************
;;;  CS -- Utilities

(defun print-cs (self stream depth)
  (declare (ignore depth))
  (format stream "<CS ~d>" (cs-id self)))


(defun display-cs (cs &optional (stream *terminal-io*) (indent 0))
  (let ((real-cs (cs cs)))
    (if (not (cs-p real-cs))
        (error "~a doesn't seem to be a CS!" cs))
    (indent-stream stream indent)
    (format stream " CS ~d~% Varsets:~%" (cs-id real-cs))
    (dolist (vs (cs-varset-list real-cs))
      (display-varset vs stream (+ indent 3)))
    (format stream "~%")
    (values)))

(defun copy-cs (self)
  (let ((new-cs (make-cs)))
    (dolist (vs (cs-varset-list self))
      (install-new-varset (copy-varset vs) new-cs))
    (debug-msg :cs "Copy ~a to yield ~a" self new-cs)
    (values new-cs)))

(defun install-new-varset (vs cs)
  (push vs (cs-varset-list cs))
  (dolist (var (varset-var-list vs))
    (setf (gethash var (cs-var-table cs)) vs)))

;;; We can map ID numbers into CSs provided *SAVE-CS* is true

(defun cs (thing)
  (cond
   ((cs-p thing)  thing)
   ((integerp thing)
    (find thing *cs-list* :key #'cs-id))
   (t nil)))

(defun reset-cs-list ()
  (setf *cs-list* '()))

;;;****************************************************************
;;;  CS -- Queries

(defun var-varset (ob cs)
  "return the VARSET to which ob belongs, else ob"
  (or (gethash ob (cs-var-table cs))
      ob))

(defun var-ensure-varset (ob cs)
  "return the VARSET to which ob belongs, creating if necessary"
  (or (gethash ob (cs-var-table cs))
      (let ((new-vs (make-single-varset ob)))
	(install-new-varset new-vs cs)
	(values new-vs))))

(defun var-value (ob cs)
  "if ob is bound, return binding, else return ob"
  ;; in other words, return the "most specific" thing
  ;; we know about ob.
  (let ((vs (var-varset ob cs)))
    (if (varset-p vs)
	(or (varset-bound-const vs) ob)
	ob)))

;;; -------------

(defun cd-vars? (ob1 ob2 cs)
  "return true if the objects already co-designate"
  (eq (var-varset ob1 cs) (var-varset ob2 cs)))

(defun ncd-varsets (vs1 vs2 cs)
  "1} return true if vs1 and vs2 have been explicitly declared ncd
   2} return ncd listing for vs1 in vs2, if any"
  (find vs1 (varset-ncd-list vs2)
	:key #'(lambda (x) (var-varset (car x) cs))))

;;; -------------

;; return whether or not the constraint is consistent.
;; ============= not implemented ==================
;; the flag restrik indicates that it is not ok to
;; co-designate a k-var with a var or constant
;; ================================================

(defun constraint-consistent? (cf cs &key restrik)
  (if (cf-equals cf)
      (cd-consistent? (cf-var1 cf) (cf-var2 cf) cs :restrik restrik)
      (non-cd-consistent? (cf-var1 cf) (cf-var2 cf) cs :restrik restrik)))

(defun non-cd-consistent? (ob1 ob2 cs &key restrik)
  (declare (ignore restrik))
  (not (eq (var-varset ob1 cs) (var-varset ob2 cs))))

;; issue #2: do we allow rt-vars to non-cd with other vars?
;;
;;(defun non-cd-consistent? (ob1 ob2 cs &key restrik)
;;  (declare (ignore restrik))
;;  (and (not (eq (var-varset ob1 cs) (var-varset ob2 cs)))
;;       (not (rt-variable? (var-value ob1 cs)))
;;	 (not (rt-variable? (var-value ob2 cs)))))

(defun cd-consistent? (ob1 ob2 cs &key restrik)
  (let ((v1 (var-varset ob1 cs))
	(v2 (var-varset ob2 cs)))
    (cond ((eq v1 v2) t)
	  ((not (cd-types-ok? ob1 ob2 cs restrik)) nil)
	  ((and (varset-p v1) (varset-p v2))
	   (not (ncd-varsets v1 v2 cs)))
	  (t t))))

(defun cd-types-ok? (ob1 ob2 cs restrik)
  (declare (ignore restrik))
  (or (plan-time-variable? (var-value ob1 cs))
      (plan-time-variable? (var-value ob2 cs))))

;; issue #1: do we deal with k-variables here, or in search-control?
;;
;;(defun cd-types-ok? (ob1 ob2 cs restrik)
;;  (cond ((k-variable? ob1)
;;	 (case restrik
;;	   ((:free) t)
;;	   ((:const) (plan-time-constant? (var-value ob2 cs)))
;;	   ((:rtvar) (rt-variable? (var-value ob2 cs)))))
;;	((k-variable? ob2)
;;	 (case restrik
;;	   ((:free) t)
;;	   ((:const) (plan-time-constant? (var-value ob1 cs)))
;;	   ((:rtvar) (rt-variable? (var-value ob1 cs)))))
;;	(t
;;	 (or (plan-time-variable? (var-value ob1 cs))
;;	     (plan-time-variable? (var-value ob2 cs))))))

;; issue #2: do we allow rt-vars to non-cd with other vars
;; -- need to add a check that the other var has a null ncd-list

;;;**********************************************************
;;; I'm using these as entry points into my ranking function.
;;; PLEASE DON'T ELIMINATE THIS FUNCTIONALITY (AGAIN)!!!!!!!!!!!!

(defun necessarily-codesignate? (obj1 obj2 cs)
  (cd-vars? obj1 obj2 cs))

(defun necessarily-noncodesignate? (obj1 obj2 cs)
  (not (cd-consistent? obj1 obj2 cs)))

;;;****************************************************************
;;;  CS -- Construction

(defun add-constraints! (clist cs &key restrik)
  (dolist (cf clist)
    (add-constraint! cf cs :restrik restrik))
  (values))

(defun add-constraint! (cf cs &key restrik)
  "modify cs to incorporate constraint cf"
  (when *cs-paranoid*
    (unless (constraint-consistent? cf cs :restrik restrik)
      (error "add-constraint failure: ~a ~a" cf cs)))
  (debug-msg :cs "Add constraint ~a ~a ~a in ~a: ~a"
	     (cf-var1 cf) (if (cf-equals cf) "=" "<>") (cf-var2 cf)
	     cs (cs-varset-list cs))
  (if (cf-equals cf)
      (add-cd-constraint cf (cf-var1 cf) (cf-var2 cf) cs)
      (add-non-cd-constraint cf (cf-var1 cf) (cf-var2 cf) cs))
  (values))

;;; the following procedures _ASSUME_ that the constraints
;;; to be added are consistent.

(defun add-non-cd-constraint (constraint ob1 ob2 cs)
  (let* ((v1 (var-ensure-varset ob1 cs))
	 (v2 (var-ensure-varset ob2 cs))
	 (ncd1 (ncd-varsets v1 v2 cs))
	 (ncd2 (ncd-varsets v2 v1 cs)))
    (if ncd1
	(push constraint (cdr ncd1))
	(progn
	  (push (list ob2 constraint) (varset-ncd-list v1))
	  (incf (varset-size v1))))
    (if ncd2
	(push constraint (cdr ncd2))
	(progn
	  (push (list ob1 constraint) (varset-ncd-list v2))
	  (incf (varset-size v2))))
    (values)))

(defun add-cd-constraint (constraint ob1 ob2 cs)
  (let ((v1 (var-varset ob1 cs))
	(v2 (var-varset ob2 cs)))
    (cond ((eq v1 v2)
	   (when (varset-p v1)
	     (push constraint (varset-cd-list v1))))
	  ((not (varset-p v1))
	   (if (varset-p v2)
	       (merge-var-varset v1 v2 cs constraint)
	       (install-new-varset
		(make-double-varset ob1 ob2 constraint) cs)))
	  ((not (varset-p v2))
	   (merge-var-varset v2 v1 cs constraint))
	  ((< (varset-size v1) (varset-size v2))
	   (merge-varsets v1 v2 cs constraint))
	  (t
	   (merge-varsets v2 v1 cs constraint))))
  (values))

;;; -------------

(defun merge-var-varset (var vs cs constraint)
  (push var (varset-var-list vs))
  (incf (varset-size vs))
  (when (plan-time-constant? var)
    (setf (varset-bound-const vs) var))
  (push constraint (varset-cd-list vs))
  (setf (gethash var (cs-var-table cs)) vs)
  (values))

;;; -------------

(defun merge-varsets (v1 v2 cs constraint)
  ;; merge v1 into v2
  
  ;; 1} set the identifying constant
  (setf (varset-bound-const v2)
	(or (varset-bound-const v1) (varset-bound-const v2)))
  
  ;; 2} merge the var-lists
  (incf (varset-size v2) (length (varset-var-list v1)))
  (nconcf (varset-var-list v2) (varset-var-list v1))
  
  ;; 3} take care of the ncd-lists...
  (dolist (v1-other-ncd (varset-ncd-list v1))
    (let* ((other-vs (var-varset (car v1-other-ncd) cs))
	   (other-v1-ncd (ncd-varsets v1 other-vs cs))
	   (other-v2-ncd (ncd-varsets v2 other-vs cs))
	   (v2-other-ncd (ncd-varsets other-vs v2 cs)))
      ;; a} if the other guy had separate entries for v1 and v2,
      ;;    merge them and delete one
      (when other-v2-ncd
	(nconc other-v2-ncd (cdr other-v1-ncd))
	(deletef other-v1-ncd (varset-ncd-list other-vs))
	(decf (varset-size other-vs)))
      ;; b} if v2 already had other-vs on its list, merge v1's
      ;;    constraints.  else add other-vs to v2's ncd list
      (if v2-other-ncd
	  (nconc v2-other-ncd (cdr v1-other-ncd))
	  (progn
	    (push v1-other-ncd (varset-ncd-list v2))
	    (incf (varset-size v2))))))
  
  ;; 4} merge the cd constraint lists, and add the new one
  (nconcf (varset-cd-list v2) (varset-cd-list v1))
  (push constraint (varset-cd-list v2))
  
  ;; 5} remove v1 from cs
  (deletef v1 (cs-varset-list cs))

  ;; 6} tell all the vars in v1 that they now belong to v2
  (dolist (v (varset-var-list v1))
    (setf (gethash v (cs-var-table cs)) v2))

  (values))

;;;****************************************************************
;;;  CS -- Retraction

(defun delete-constraints! (cf-list cs)
  (let ((vs-list nil))
    ;; delete all non-cd-constraints
    (mapc #'(lambda (cf)
	      (when (not (cf-equals cf)) 
		(debug-msg :cs "delete ncd ~a~%" cf)
		(delete-non-cd-constraint! cf cs)))
	  cf-list)
    ;; delete cd-constraints, accumulating a list of affected varsets
    (setf vs-list
	  (mapcar #'(lambda (cf)
		      (when (cf-equals cf)
			(debug-msg :cs "delete cd ~a~%" cf)
			(delete-cd* cf cs)))
		  cf-list))
    ;; now re-construct those sets
    (reconstruct-varsets 
     (delete-duplicates (delete nil vs-list))
     cs)))


(defun delete-non-cd-constraint! (cf cs)
  (let* ((v1 (var-varset (cf-var1 cf) cs))
	 (v2 (var-varset (cf-var2 cf) cs))
	 (v1-ncdset (ncd-varsets v2 v1 cs))
	 (v2-ncdset (ncd-varsets v1 v2 cs)))
    (when *cs-paranoid*
      (unless (and v1-ncdset v2-ncdset)
	(error "attempt to delete non-existent constraint: ~a in ~a"
	       cf cs)))
    (delete-ncd* cf v1-ncdset v1)
    (delete-ncd* cf v2-ncdset v2)))

(defun delete-ncd* (cf ncd-set v)
  (setf (cdr ncd-set)
	(delete cf (cdr ncd-set) :count 1 :test #'cf-equiv?))
  ;; are there any ncd constraints left in this set?
  (when (= (length ncd-set) 1)
    (setf (varset-ncd-list v)
	  (delete ncd-set (varset-ncd-list v)))
    (decf (varset-size v))))


(defun delete-cd-constraint! (cf cs)
  (reconstruct-varset (delete-cd* cf cs) cs))

(defun delete-cd* (cf cs)
  (let ((v1 (var-varset (cf-var1 cf) cs)))
    (when *cs-paranoid*
      (unless (eq v1 (var-varset (cf-var2 cf) cs))
	(error "attempt to delete non-existent constraint: ~a in ~a"
	       cf cs)))
    (setf (varset-cd-list v1)
	  (delete cf (varset-cd-list v1) :count 1 :test #'cf-equiv?))
    (values v1)))


(defun reconstruct-varsets (v-list cs)
  (dolist (v v-list)
    (reconstruct-varset v cs)))

(defun reconstruct-varset (v cs)
  (debug-msg :cs "reconstruct ~a~%" v)
  (let ((cf-list nil))
    ;; 1} for each ncd-set we have with another varset, delete
    ;;    it from the _other_ guy {we are going to recreate
    ;;    it in step 4, because it might split, or need to have
    ;;    a different exemplar}.  Also accumulate list of ncd cf's
    (dolist (ncd-entry (varset-ncd-list v))
      (let* ((other-v (var-varset (car ncd-entry) cs))
	     (other-ncd (ncd-varsets v other-v cs)))
	(deletef other-ncd (varset-ncd-list other-v))
	(decf (varset-size other-v))
	(insertf cf-list (cdr ncd-entry))))

    ;; 2} delete the varset from cs
    (setf (cs-varset-list cs) (delete v (cs-varset-list cs)))
    
    ;; 3} tell all the vars in the varset that they only belong
    ;; to themselves.
    (dolist (var (varset-var-list v))
      (setf (gethash var (cs-var-table cs)) nil))

    ;; 4} and rebuild the constraints all over again.
    (add-constraints! (nconc (varset-cd-list v) cf-list) cs)))


;;;****************************************************************
;;; UNIFICATION --
;;; Attempt to unify two expressions, given a current constraint
;;; set.  Unify! will modify the constraint set, whereas unify
;;; will not.  In either case, the complete set of constraints
;;; is returned if unification is successful, else the symbol
;;; :FAIL  {Note: this is a change from the previous implementation,
;;; which returned only new constraints.}

(defun unify! (e1 e2 cs &optional bsf)
  (cond ((and (null e1) (null e2))  bsf)
	((and (atom e1) (atom e2))
	 (cond ((eq e1 e2) bsf)
	       ((cd-consistent? e1 e2 cs)
		(let ((new-cf (eq-cf e1 e2)))
		  (add-constraint! new-cf cs)
		  (cons new-cf bsf)))
	       (t ':FAIL)))
	((or (atom e1) (atom e2)) ':FAIL)
	((not (= (length e1) (length e2))) ':FAIL)
	(t (let ((new-diffs (unify! (car e1) (car e2) cs bsf)))
	     (if (eq new-diffs ':FAIL)
		 ':FAIL
	       (unify! (cdr e1) (cdr e2) cs new-diffs))))))


(defun unify (e1 e2 cs)
  ;; do a quick check to prevent unnecessary copying of cs --
  ;; this relies on the fact that all lists we will ever try
  ;; to unify are forms, and thus have constant car's
  (cond ((and (consp e1)
	      (or (not (consp e2))
		  (not (eq (car e1) (car e2)))
		  (not (= (length e1) (length e2)))))
	 ':FAIL)
	(t  (unify! e1 e2 (copy-cs cs)))))


;;;**************************************************************************
;;; This is the call from COPY.LISP.    In our context, operations
;;; always end up applying to the variables and constraints in
;;; the varsets.  In some cases, we know that we can optimize the
;;; the operation, so we do.  Otherwise, we go through and re-estabish
;;; the entire list of constraints, because their meanings may have
;;; changed some...

(defun operate-on-cs (cs form-fun keyword destructive?)
  (case keyword
    ((:copy :instantiate)
     (if destructive? cs (copy-cs cs)))
    ((:variabilize)
     (variabilize-cs cs form-fun destructive?))
    (t
     (let ((the-cs (if destructive? cs (copy-cs cs))))
       (let ((cf-lists
	      (mapcar #'(lambda (x)
			  (operate-on-varset-cfs x form-fun destructive?))
		      (cs-varset-list the-cs))))
	 ;; even if destructive? is nil, we still recreate
	 ;; the cs internals from scratch, because there is
	 ;; no telling what has been done to the constraints.

	 (setf (cs-var-table the-cs)
	       (make-hash-table :size *cs-table-size* :test #'eq))
	 (setf (cs-varset-list the-cs) nil)

	 (dolist (cf-list cf-lists)
	   (add-constraints! cf-list the-cs))

	 (values the-cs))))))

(defun operate-on-varset-cfs (vs form-fun destructive?)
  ;; return a list of cf's that have been operated on.
  ;; note that we have to prevent the ncd's from getting
  ;; counted twice, so we divide them according to which
  ;; varset var1 is in.
  (flet ((our-ncd-cf (cf)
	   (member (cf-var1 cf) (varset-var-list vs))))
    (let ((the-list nil))
      (dolist (ncd-set (varset-ncd-list vs))
	(insertf the-list
		 (operate-on (delete-if-not #'our-ncd-cf (cdr ncd-set))
			     form-fun destructive?)))
      (insertf the-list
	       (operate-on (varset-cd-list vs) form-fun destructive?))
      (values the-list))))


(defun variabilize-cs (cs form-fun destructive?)
  (let ((the-cs (if destructive? cs (copy-cs cs))))
    (dolist (vs (cs-varset-list the-cs))
      (variabilize-varset vs the-cs form-fun destructive?))
    (values the-cs)))

(defun variabilize-varset (vs cs form-fun destructive?)
  ;; run the operations over the constraints and variables...
  (setf (varset-var-list vs)
	(operate-on-forms (varset-var-list vs) form-fun destructive?))
  (setf (varset-cd-list vs)
	(operate-on (varset-cd-list vs) form-fun destructive?))
  (dolist (ncd-set (varset-ncd-list vs))
    (setf (car ncd-set) (operate-on-form (car ncd-set) form-fun destructive?))
    (setf (cdr ncd-set) (operate-on (cdr ncd-set) form-fun destructive?)))
  ;; ...and the constant.
  ;; if its changed, we have to update
  ;; the hash table, and check that it
  ;; is still constant
  (when (varset-bound-const vs)
    (let* ((old-const (varset-bound-const vs))
	   (new-const (operate-on-form old-const form-fun destructive?)))
      (unless (eq old-const new-const)
	(setf (gethash old-const (cs-var-table cs)) nil)
	(setf (gethash new-const (cs-var-table cs)) vs)
	(if (plan-time-constant? new-const)
	    (setf (varset-bound-const vs) new-const)
	    (setf (varset-bound-const vs) nil))))))
