;;;********************************************************************
(in-package 'spa)
;;;********************************************************************

;;;********************************************************************
;;;  Control functions specifically for the blocksworld---even more 
;;;  specifically for block-stacking problems that appear in Rao's 
;;;  thesis.  See also STANDARD-CONTOL-FUNS.LISP for how these things 
;;;  are used

;;;  This was a first cut at a ranking function---trying to penalize
;;;  those open conditions not true in the initial state.  It didn't 
;;;  work too well;  see the TR.

(defun bw-rank-fun (e)
  (let* ((plan (qentry-iplan e))
         (num-unsafe (length (SNLP-plan-unsafe plan)))
         (num-open (num-open-not-in-initial plan))
         (num-steps (length (snlp-plan-steps plan)))
         (num-ordering (length (snlp-plan-ordering plan)))
         (extend? (eql (qentry-dir e) ':extend)))
    (+ (* 100 num-steps)
       (* 20 num-open) 
       num-unsafe 
       (* -2 num-ordering)
       (if (not extend?) 1000 0))))

(defun num-open-not-in-initial (p)
  (let ((b  (snlp-plan-bindings p))
        (the-initial  (snlp-plan-initial-conditions p))
        (count 0))
    (dolist (o (snlp-plan-open p))
      (if (not (member (instantiate (open-condition o) b)
                       the-initial
                       :test #'equalp))
          (incf count)))
    count))


;;;;******************************************************
;;;  This is the most blocks-world-stacking-specific function 
;;;  I could think of.  We try to make the planner build the stack 
;;;  from the bottom up, assuming that the goal state is a stack and 
;;;  the goal conditions are ordered from top down, e.g. 
;;;     (ON A B) (ON B C) 
;;;  which means a step to fill the first form should be ordered
;;;  AFTER the step that fills the second.
;;;  

(defparameter *lge-open-real-penalty* 100)
(defparameter *lge-open-poss-penalty* 1000)

(defparameter *lge-unsafe-penalty* 1)

(defparameter *lge-rstep-poss-penalty* 20)
(defparameter *lge-rstep-real-penalty* 99999999)

(defparameter *lge-order-no-step-penalty* 100)
(defparameter *lge-order-poss-penalty* 20)
(defparameter *lge-order-real-penalty* 500)

(defun last-goal-earliest (qent)
  (setq *lge-rstep-poss-penalty* 20)
  (setq *lge-rstep-real-penalty* 500)
  (+  (standard-direction-score (qentry-dir qent))
      (lge-plan-ranker (qentry-iplan qent))))

(defun standard-direction-score (qdir)
  (if (not (eq qdir ':extend)) 900000 0))

;;;  Try to make this real simple:  take a huge hit for LGE order
;;;  violations, a smaller hit for number of steps, and a smaller 
;;;  still hit for number of opens and unsafes.

(defun lge-plan-ranker (plan)
  (let ((penalty 0))
    (multiple-value-bind (p r) (open-violations plan)
      (incf penalty (* p *lge-open-real-penalty*))
      (incf penalty (* r *lge-open-poss-penalty*)))
    (multiple-value-bind (no-step order-poss order-real) (lge-order-violations plan)
      (incf penalty (* no-step    *lge-order-no-step-penalty*))
      (incf penalty (* order-poss *lge-order-poss-penalty*))
      (incf penalty (* order-real *lge-order-real-penalty*)))
    (incf penalty  (* 10 (length (snlp-plan-steps plan))))
    (incf penalty  (* 5  (length (snlp-plan-open plan))))
    (incf penalty  (length (snlp-plan-unsafe plan)))
    penalty))

;;;****************************************************************************
;;; LGE says that goals are supposed to be achieved in reverse of the order 
;;; in which they appear.   Therefore if the goals are G1 ... Gn, then 
;;; the achieving step for Gi should precede the achieving steps for 
;;; G1, G2, ... Gi-1.  There are three possible violations for each pair 
;;; (Gk,Gi) where Gk precedes Gi in the goal list:  (1) either Gi or 
;;; Gk has no achieving step, (2) Gk's achieving step possibly precedes 
;;; Gi's, and (3) the same only necessarily.   Return counts for these three
;;; numbers.

(defun lge-order-violations (p)
  (let* ((the-goals (snlp-plan-goal-conditions p))
         (the-ordering (snlp-plan-ordering p))
         (ach-steps (mapcar #'(lambda (goal-form) 
                                (achieving-step-id goal-form 
                                                   (snlp-plan-links p)
                                                   (snlp-plan-bindings p)))
                            the-goals))
         (num-no-step 0)
         (num-possible 0)
         (num-certain 0))
     (do* ((goals the-goals (cdr goals))
           (asteps ach-steps (cdr asteps))
           (c-goal  (car goals)  (car goals))
           (c-astep (car asteps) (car asteps)))
          ((null goals))
       ;; (CDR GOALS) contains the goals that should *precede* C-GOAL
       ;; therefore each step id in (CDR ASTEPS) should precede c-step
         (dolist (preceding-astep (cdr asteps))
           (cond
            ((or (null c-astep) (null preceding-astep))
             (incf num-no-step))
            (t (let ((order (order-relation preceding-astep c-astep the-ordering)))
                 (cond
                  ((eq order :precedes))
                  ((eq order :unknown) (incf num-possible))
                  (t (incf num-certain))))))))
     (values num-no-step num-possible num-certain)))

(defun achieving-step-id (goal-form link-list bdgs)
  (let* ((inst-goal (instantiate-form goal-form bdgs))
         (ach-link (find-if #'(lambda (link) 
                                (and (eq (link-consumer link) ':goal)
                                     (equal inst-goal
                                            (instantiate-form (link-condition link)
                                                              bdgs))))
                            link-list)))
    (if ach-link 
        (link-producer ach-link)
        nil)))
  

;;;************************************************************************
;;; The entry point is ORDER-RELATION, which takes two steps and an 
;;; ordering, and produces one of :COINCIDES, :PRECEDES, :FOLLOWS, or :UNKNOWN.

(defun order-relation (sid-1 sid-2 ordering)
  (cond
    ((eql sid-1 sid-2) ':COINCIDES)
    ((eql sid-1 0)     ':PRECEDES)
    ((eql sid-1 ':GOAL) ':FOLLOWS)
    ((eql sid-2 0)     ':FOLLOWS)
    ((eql sid-2 ':GOAL) ':PRECEDES)
    (t (really-compute-order sid-1 sid-2 ordering))))

(defun really-compute-order (sid-1 sid-2 ordering)
  (cond
    ((find-path sid-1 sid-2 ordering) ':PRECEDES)
    ((find-path sid-2 sid-1 ordering) ':FOLLOWS)
    (t :UNKNOWN)))

;;;  This assumes no circularities in the ordering.  Will loop 
;;;  otherwise.

(defun find-path (sid-1 sid-2 ordering)
  (let ((sid1-orders (remove-if-not #'(lambda (o) (eql sid-1 (ordering-pred o)))
                                    ordering)))
    (cond
      ((null sid1-orders) NIL)
      ((some #'(lambda (o) (eq sid-2 (ordering-succ o))) sid1-orders) T)
      (T (some #'(lambda (sid) (find-path sid sid-2 ordering))
               (mapcar #'ordering-succ sid1-orders))))))

;;;************************************************************

(defun open-violations (plan)
  (let ((poss 0)
        (real 0))
    (dolist (open (snlp-plan-open plan))
      (case (compute-open-status open plan)
        ((:real-violation) (incf real))
        ((:possible-violation) (incf poss))))
    (values poss real)))

(defun compute-open-status (open plan)
  (cond-status (condx-to-list (open-condition open))
               (mapcar #'condx-to-list (snlp-plan-initial-conditions plan))
               (snlp-plan-bindings plan)))

(defun cond-status (open-cond initial-conds b)
  (cond
    ((null initial-conds) NIL)
    (t (or (form-status open-cond (car initial-conds) b)
           (cond-status open-cond (cdr initial-conds) b)))))

;;; Returning T means the forms are in agreement (no further checking for 
;;; contradictions).  Returning NIL means continue, returning 
;;; :REAL-VIOLATION or :POSSIBLE-VIOLATION stops and penalizes.

(defun form-status (oform iform b)
  (blocksworld-form-status oform iform b))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; NB -- This is all VERY blocksworld specific!!! It should 
;;; be segregated out somewhere.   Here are the cases:
;;;    open form       initial form      constraints      result
;;; ============================================================================
;;;    (cleartop ?x)    (cleartop ?a)     ?x CD  ?a             T
;;;    (cleartop ?x)    (cleartop ?a)     ?x NCD ?a             REAL-VIOLATION
;;;    (cleartop ?x)    (cleartop ?a)     otherwise             POSSIBLE-VIOLATION
;;;
;;;    (cleartop ?x)    (on ?a ?b)        ?x CD ?b              REAL-VIOLATION
;;;    (cleartop ?x)    (on ?a ?b)        ?x NCD ?b             NIL
;;;    (cleartop ?x)    (on ?a ?b)        otherwise             POSSIBLE-VIOLATION
;;;
;;;    (on ?x ?y)       (on ?a ?b)        ?x CD ?a, ?y CD ?b    T
;;;    (on ?x ?y)       (on ?a ?b)        ?y CD ?b, ?x NCD ?a   REAL-VIOLATION
;;;    (on ?x ?y)       (on ?a ?b)        ?y NCD ?b             NIL
;;;    (on ?x ?y)       (on ?a ?b)        otherwise             POSSIBLE-VIOLATION
;;;
;;;    (on ?x ?y)       (cleartop ?a)     ?a CD ?y              REAL-VIOLATION
;;;    (on ?x ?y)       (cleartop ?a)     ?a NCD ?y             NIL
;;;    (on ?x ?y)       (cleartop ?a)     otherwise             POSSIBLE-VIOLATION

(defun blocksworld-form-status (oform iform b)
  (cond
   ((and (eq (car oform) 'cleartop) (eq (car iform) 'cleartop))
    (cond
     ((necessarily-codesignate? (cadr oform) (cadr iform) b) T)
     ((necessarily-noncodesignate? (cadr oform) (cadr iform) b) :REAL-VIOLATION)
     (t :POSSIBLE-VIOLATION)))
   ((and (eq (car oform) 'cleartop) (eq (car iform) 'on))
    (cond
     ((necessarily-codesignate? (cadr oform) (caddr iform) b) :REAL-VIOLATION)
     ((necessarily-noncodesignate? (cadr oform) (caddr iform) b) NIL)
     (t :POSSIBLE-VIOLATION)))
   ((and (eq (car oform) 'on) (eq (car iform) 'on))
    (cond
     ((and (necessarily-codesignate? (cadr oform) (cadr iform) b)
           (necessarily-codesignate? (caddr oform) (caddr iform) b))   T)
     ((and (necessarily-noncodesignate? (cadr oform) (cadr iform) b)
           (necessarily-codesignate? (caddr oform) (caddr iform) b))   :REAL-VIOLATION)
     ((necessarily-noncodesignate? (caddr oform) (caddr iform) b) NIL)
     (t :POSSIBLE-VIOLATION)))
   ((and (eq (car oform) 'on) (eq (car iform) 'cleartop))
    (cond
     ((necessarily-codesignate? (caddr oform) (cadr iform) b)  :REAL-VIOLATION)
     ((necessarily-noncodesignate? (caddr oform) (cadr iform) b)  NIL)
     (t :POSSIBLE-VIOLATION)))
   (t NIL)))

;;;**********************************************************************
;;; Control information for choosing an extend (our experiments 
;;; use the standard retractor)
;;;
;;;**********************************************************************

;;; The idea is that we favor working "backwards" from the goal---
;;; first generating opens for the goal, then working on their 
;;; preconditions, and so on.   

;;; The way to use this function is to bind the variable 
;;; (setf *EXTEND-CHOOSE-FUN* #'lge-open-first)
;;; This is actually done in the code to run the experiments.

;;;  Once again two choices:  open first or unsafe first, but 
;;;  in either case prefer the "latest" option.

(defun lge-open-first (plan)
  (cond
    ((not (null (snlp-plan-open plan)))
     (find-closest-to-goal (snlp-plan-open plan) plan))
    (t (find-closest-to-goal (snlp-plan-unsafe plan) plan))))

(defun lge-unsafe-first (plan)
  (cond
    ((not (null (snlp-plan-unsafe plan)))
     (find-closest-to-goal (snlp-plan-unsafe plan) plan))
    (t (find-closest-to-goal (snlp-plan-open plan) plan))))

;;;***************************************************************

;;; First choose that group that are "closest" to the goal

(defun find-closest-to-goal (obj-list plan)
  (let* ((obj-out    (car obj-list))
         (best-dist  (obj-dist-to-goal obj-out plan))
         (best-index (obj-goal-index obj-out plan)))
    (dolist (obj (cdr obj-list))
      (let ((new-dist (obj-dist-to-goal obj plan)))
        (cond
          ((< new-dist best-dist)
           (setq obj-out obj)
           (setq best-dist new-dist)
           (setq best-index (obj-goal-index obj plan)))
          ((= new-dist best-dist)
           (let ((new-goal-index (obj-goal-index obj plan)))
             (when (> new-goal-index best-index)
               (setq obj-out obj)
               (setq best-dist new-dist)
               (setq best-index new-goal-index)))))))
    obj-out))

(defun obj-dist-to-goal (obj plan)
  (cond
    ((unsafe-p obj) (link-dist-to-goal (unsafe-link obj) plan))
    ((open-p   obj) (step-dist-to-goal (open-step-id obj) plan))))

(defun step-dist-to-goal (step-id plan)
  (cond
    ((eql step-id :goal) 0)
    (t (let* ((the-links (remove-if-not #'(lambda (l) (eql (link-producer l)
                                                           step-id))
                                        (snlp-plan-links plan))))
         (if (null the-links)
             999999
             (apply #'min (mapcar #'(lambda (l) (link-dist-to-goal l plan))
                                  the-links)))))))

(defun link-dist-to-goal (link plan)
  (let ((real-link (cond 
                    ((link-p link) link)
                    ((numberp link) (get-link link plan))
                    (t (error "Can't make a link out of ~a" link)))))
    (+ 1 (step-dist-to-goal (link-consumer real-link) plan))))

;****************

(defun obj-goal-index (obj plan)
  (let* ((instantiated-goal-form 
          (instantiate-form (condx-to-list (obj-goal-form obj plan))
                            (snlp-plan-bindings plan))))
    (position instantiated-goal-form
              (snlp-plan-goal-conditions plan) 
              :test #'(lambda (obj-form plan-condx) 
                        (equal obj-form (condx-to-list plan-condx))))))

(defun obj-goal-form (obj plan)
  (cond
    ((unsafe-p obj) (link-goal-form (unsafe-link obj) plan))
    ((open-p obj)   (open-goal-form obj plan))
    (t (error "Don't know how to find goal form for ~a" obj))))


(defun link-goal-form (link plan)
  (let ((real-link (cond
                    ((link-p link) link)
                    ((numberp link) (get-link link plan))
                    (t (error "Can't make a link out of ~a")))))
    (if (eq (link-consumer real-link) :goal)
        (link-condition real-link)
        (step-goal-form (link-consumer real-link) plan))))

(defun open-goal-form (open plan)
  (if (eq (open-step-id open) :goal)
      (open-condition open)
      (step-goal-form (open-step-id open) plan)))

(defun step-goal-form (step-id plan)
  (let ((the-links (step-links step-id plan)))
    (cond
      ((null the-links)  
       (error "Can't trace goal---no links for ~a in ~a" step-id plan))
      (t (link-goal-form (car the-links) plan)))))

(defun step-links (step-id plan)
  (remove-if-not #'(lambda (link) 
                     (eql step-id (link-producer link)))
                 (snlp-plan-links plan)))

