;;; TAQL Compiler, Version 3
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/taql-compiler.lisp
;;; Created August 20, 1988
;;; Modified by Erik Altmann, starting 8-89.
;;;
;;; This file implements the functions which translate TAQL Version 3 language
;;; constructs into Soar 5 productions.  The translation is invoked by
;;; simply evaluating a TAQL construct inside a Soar process after loading
;;; this compiler code.
;;;
;;; Useful little things to do:
;;;   . check common keywords with one function each, and spruce up
;;;     the error checking in the process.
;;;
;;; Known bugs/funnies:
;;; 
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 6-14-91 - gry - Added the :buffer-in-goal keyword to edit clauses.
;;;   This is an experimental (but advertised) keyword to attempt to
;;;   let users work around chunking problems that arise when editing
;;;   objects in higher goals.  This keyword lets one explicitly choose
;;;   which goal the edits will be buffered in.
;;;
;;; 6-12-91 - gry - Changed to support :actions and :sliding-actions
;;;   in apply-operator and result-superstate, and :actions in augment.
;;;
;;; 6-11-91 - gry - Added (duplicate-tc-warn [:yes | :no]) command, to
;;;   make it easier to find duplicate TC names.
;;;
;;; 6-11-91 - gry - Changed so that tc-name 'segment property is removed
;;;   when a TC is excised under (excise-task) or (excise SEGMENT-NAME).
;;;   See also segments.lisp.
;;;
;;; 5-29-91 - gry - Changed evaluate-object so that variables that begin
;;;   with "?" in :cycle-when conditions get replaced by different variables
;;;   in the two contexts the :cycle-when conditions are tested across.
;;;
;;; 5-21-91 - gry - changes propose-task-operator and propose-operator to
;;;   use pclass 'unknown' rather than 'sticky'.  Declaring them sticky
;;;   was resulting in Soar not printing retraction messages for those
;;;   productions.  There was an old comment from Erik in the code there
;;;   that they were being declared sticky because at one point, Soar was
;;;   not properly giving things in the transitive closure support.  Hopefully
;;;   that problem has been fixed.  Since Soar's operator-creation recognizer
;;;   is a little flaky (in my opinion), this could cause people problems in
;;;   some odd cases.  My new philosophy is to let Soar 'features' shine
;;;   through if there are any negative user-visible consequences of fixing
;;;   it, though.
;;;
;;;   Also changed pclass to 'unknown' in a few other places, to reduce the
;;;   number of places we use declare-p.  Search for 5-21-91.
;;;
;;;   Also moved defconstant for *taql-types* to types.lisp.
;;;
;;; 5-17-91 -- ema - added support for symbolic/numeric-value
;;;   "unknown".  two small changes in process-value; search for
;;;   "5-17-91".  see also taql-support.soar.
;;;
;;; 4-5-91 through 4-22-91 - gry - added space model support
;;;
;;; 2-7-91 through 3-13-91 - gry - Assorted changes to support the new
;;;   syntax and data models.  Assorted other fixes/cleanups made
;;;   along the way.  Changed all ,. to ,@ for safety.  If speed
;;;   becomes an issue maybe some of these can be changed back.  But
;;;   there weren't all that many places that still used ,..
;;;   Changed goal-test-group to put the attributes it used to put directly
;;;   on the goal into the new (goal ^taql-stuff*) object instead.  Eventually
;;;   all of the random goal augmentations should be moved there, to reduce
;;;   clutter.  The goal-test-group augmentations were a particular problem,
;;;   since their attribute names were task-specific, and hence could not
;;;   be excluded in print-data-model.
;;;
;;;   IMPORTANT CHANGE:  I changed the names of internal TAQL classes to end
;;;   in *.  This avoids conflicts in data models.  The affected classes are
;;;   add (add*), aug (aug*), control-spec-info (control-spec-info*),
;;;   control-stuff (control-stuff*), init-stuff (init-stuff*), and
;;;   map (map*).  Final-state-info did not change because that is
;;;   defined in the manual as something a user can explicitly create/test
;;;   if they want to do goal testing some way other than using
;;;   goal-test-group.
;;;
;;; 2-7-91 - gry - Fixed obscure bug in prepare-conditions.
;;;
;;; -----------------------------------------------------------------------
;;; 12-3-90 - gry - TAQL 3.1.3 (external) was frozen at this point.
;;; -----------------------------------------------------------------------
;;;
;;; 12-3-90 - gry - Fixed bug in verbose-aux -- ^ was not separated from
;;;   attribute name
;;;
;;; 11-15-90 - gry - defined check-sp-name, a function to see whether a symbol
;;;   is the name of a loaded production.  This is used in taql-support,
;;;   to give us some warning if someone changes the names of the default
;;;   productions we excise or modify.
;;;
;;; 10-12-90 - gry - Fixed bug in augment :new clauses.
;;;
;;; 9-29-90 - gry - Changed name of operator-program to operator-control.
;;;
;;; 9-14-90 - gry - Changed process-value to accept all of the symbolic values
;;;   listed in the manual.  Also changed to better disambiguate values that
;;;   are function calls (such as (compute <x> + <y>)) from values of the
;;;   form (RVALUE [:when (COND+)]*).
;;;
;;; 9-14-90 - gry - Added the sticky-taql-chunk-free-problem-spaces
;;;   function, which takes N args, each a problem space name, and
;;;   adds it to Soar's *chunk-free-problem-spaces* variable.  In addition,
;;;   after and excise-task, these spaces are again made chunk-free.
;;;
;;; 9-6-90 - gry - Distiguish default TCs from user TCs.  Default TCs
;;;   are those that appear between start-default/stop-default calls.
;;;   They are treated analogously to default productions in that they
;;;   persist after an excise-task.  Also, the :all and :types TC
;;;   specifiers only apply to user TCs by default.  Using them in
;;;   conjunction with the new :default keyword extends them to include
;;;   default TCs.  The :default keyword can appear anywhere in the
;;;   argument list, and applies to all occurrences of :all and :types
;;;   in the list.
;;;
;;; 9-5-90 - gry - added support for the TAQL data types.
;;;
;;; 9-5-90 - gry - added not-equal-ify to the section containing utility
;;;   routines.  It is used by the data type facility, but could be
;;;   generally useful.
;;;
;;; 9-2-90 - added operator-control to *taql-types*.
;;; 
;;; 8-17-90 - gry - bugs fixes, in state-edit and apply-operator-aux.
;;; 
;;; 8-16-90 - ema -
;;;   1. changed p-o to allow :op-id and :select-once-only together.
;;;   control-stuff* is now exclusive indifferent, so first  one
;;;   generated for an instance persists.
;;;   2. re-indented file, with tab-width 8.
;;;   
;;; 8-14-90 - gry - added the disable-in-taql function.  See the
;;;    taql-support.soar history entry for this date.
;;;
;;; 8-10-90 - ema/gry - fixed buggy wild-card parsing parse-state-copy-props.
;;; 
;;; 8-6-90:
;;;    Made the augment TC more consistent and flexible.  See the
;;;      TAQL 3.1.3 release notes.
;;;    Defined sticky-taql-trace-attributes, which is like Soar's
;;;      trace-attributes except that the attributes remain traced
;;;      after calls to excise-task.  This is useful for the attributes
;;;      that TAQL wants to have traced independent of task.
;;;
;;; 8-2-90:
;;;    Made various changes to support the new, improved
;;;      result-superstate TC.   See also taql-support.soar.
;;;    Made changes to add the :bind keyword to apply-operator and
;;;      result-superstate.
;;;    Changed prepare-conditions so that:
;;;      (1) CONTEXT-OBJ/no-id conditions can appear anywhere in a list of
;;;          conditions (not just in the first condition), even inside
;;;          (conjunctive) negations;
;;;      (2) a (conjunctive) negation can appear as the first, or only,
;;;          element of a :when clause.
;;;    Extended (edit :what) so that it can take any CONTEXT-OBJ as a value.
;;;    For all changes, search for "8-2-90" for details
;;;    Added :from keyword to :copy/:rename in propose-initial-state/space.
;;;
;;; -----------------------------------------------------------------------
;;; 7-30-90:  TAQL 3.1.2 (external) was frozen and released at this point.
;;; -----------------------------------------------------------------------
;;;
;;; 7-12-90 - 7-30-90 (GRY):  Convert TAQL to run in Soar 5.2.  Below is a
;;;   list of changes, see also taql-support.soar.
;;;
;;;   1. (import 'soar::lispsyntax)
;;;      7-20-90:  Removed, Soar has been fixed to export lispsyntax.
;;;   2. Explicitly make taql-compiler.lisp in the USER package, and switch to
;;;      the SOAR package and back when I need to redefine Soar things (like
;;;      compiled-sp).
;;;   3. Commented out compiled-sp fix until I know whether it works in 5.2.
;;;      7-20-90: The modified compiled-sp is now part of 5.2, so removed
;;;      TAQL's redefinition.
;;;   4. Changed state-copy compilation to correspond to Soar 5.2 default.soar.
;;;   5. Added ^dont-copy-anything flag to some default.soar productions, to
;;;      support :dont-copy (:any) in Soar 5.2.  Previously, the
;;;      absence of all copy flags resulted in not copying anything,
;;;      but in 5.2 the default is to copy all level-one attributes.
;;;   6. Change verbosity disabling to ^verbose false, enable with
;;;      - ^verbose false.  Note that the attribute name is now ^verbose,
;;;      not ^verbose*.
;;;   7. Change tpm so that it no longer prints out declare-p calls.  In 5.2,
;;;      (declare-p (pclass prod) prod) is no longer a no-op, as it
;;;      extends the class to all actions of the productions.  Thus it
;;;      is no longer safe to load productions printed by tpm and
;;;      expect them to run the same as when they were compiled.  You
;;;      have to reload the TAQL source to make sure it does the right
;;;      thing.
;;;   8. Declare-p calls must use second argument in soar::.
;;;      7-20-90:  Removed soar::, Soar now exports the right symbols.
;;;   9. Put adds* and rs-adds* on control-stuff instead of the goal.  With
;;;      rs-adds* on the goal, chunks won't give additions the right
;;;      (sticky) support.  There is still a problem that chunks won't be
;;;      recognized as operator-modifications if they don't test something
;;;      about the state, but hopefully I can get the powers that be to
;;;      change the support semantics.  Otherwise, I can hack the TAQL
;;;      compiler so that everything tests some meaningless state attribute.
;;;   10. Replaced @ by soar::@ when used as preference symbol, to work around
;;;       a Soar package bug.
;;;       7-20-90:  Removed soar::, the Soar bug has been fixed.
;;;   11. Changed taql-stats to use soar::soar-date.
;;;   12. Changed (goal ^tried) to (state ^tried-tied-operator), to correspond
;;;       to 5.2's default.soar.
;;;   13. Changed propose-space so that it forces a test of
;;;      (superstate ^dummy-att* true), so that chunks that modify
;;;      the operator will get O-support.  Hopefully, Soar 5.2's
;;;      support semantics will change soon, and I can take this out.
;;;      Note that this hack doesn't work when there isn't a
;;;      superstate.  If Soar hasn't changed by the time we have to
;;;      release, I'll have to come up with a better solution.  As it
;;;      stands, propose-space TCs might not apply for certain types
;;;      of (uncommon) impasses.
;;;   14. Changed state-edit so that it won't drop dummy-att* from the state
;;;       when the attribute in the drop spec is a variable.
;;;   15. Changed state copy semantics for more internal consistency (Soar 5.2
;;;       will use the same semantics).  The change is that it is now
;;;       illegal for :dont-copy (ATT-NAME*) to appear unless either
;;;       :copy (:all) or :copy-new (:all) is also given explicitly.
;;; 
;;; -----------------------------------------------------------------------
;;; 7-7-90:  TAQL 3.1.1 (external) was frozen and released at this point.
;;; -----------------------------------------------------------------------
;;;
;;; 7-7-90 - ema - fixed a bug in prepare-operator (was ignoring all
;;;   but one :substructure argument when many appeared).  fixed
;;;   up :terminate-when so it has to take a (possibly empty) list as
;;;   an argument, so now it can come before an edit clause...
;;;   
;;; 7-4-90 - ema - :cycle-when no longer assigns novalue
;;;   automatically (so an explicit value has to be given).  Also, in
;;;   a :what clause, all conditions (including :cycle-whens) are
;;;   conjoined to make the clause apply. 
;;; 
;;; 7-3-90 - gry - Changed taql-sp-aux to pprint the production it
;;;   tried to compile if Soar couldn't compile it.  Also added the
;;;   Soar patch to SP that allows us to detect this situation.
;;;
;;; 6-12-90 - ema - changed prefer to not go through the acceptable
;;;   preference when reconsider is given as a :value, because often
;;;   the acceptable won't be around, but it seems reasonable to
;;;   assume that the object will be selected.
;;;
;;; 6-10-90 - ema - changed :terminate-when to generate a reconsider
;;;   directly, rather than disable editing, because the disable was
;;;   being sticky (reason unknown; used to work).
;;;   
;;; 6-9-90 - ema - fixed:  tc-to-object-map for can now take multiple
;;;   maps; p-i-s no longer requires :space;  search for "6-9-90".
;;;
;;; 6-5-90 - ema - fixed :remove to take :all
;;; 
;;; 5-13-90 - ema - fixed a bug in result superstate, that was causing
;;;   :selective-remove to not generate any productions.
;;;
;;; 5-12-90 - ema - moved the temporary memory for creating the result
;;;   superstate to the supergoal, so that chunks preserve it.  now
;;;   if you replace a value with the same value (e.g. same constant),
;;;   then it works.  it DOESN'T work for chunks built in levels above
;;;   the supergoal, because they remember the reject as well as the
;;;   accept for the value, and includes in the friggin "preference
;;;   halo".  never did trust halos.  see .../soar/taql/prg/r-s-d-test.taql.
;;;   
;;; 5-2-90 - ema - undid:
;;; [4-30-90 - ema - tweaked the semantics of state copying.
;;;   :copy-new (ARG) now implies (copy (ALL - ARG), copy-new (ARG))]
;;;   
;;; 4-25-90 - ema - replaced variable-p from the current version in
;;;   the soar parser.  now, "<foo" won't be a variable (but it will
;;;   be a constant).
;;;   
;;; 4-20-90 - ema - :bind-object in evaluate-object now binds the copy
;;;   of the operator in the evaluation subgoal.  this could have gone
;;;   either way, but this way i didn't have to recode R1.
;;;   
;;; 4-16-90 - ema - did an overhaul of the evalution mechanism.
;;;   evaluate-when and evaluate-how are replaced by evaluate-object.
;;;   all the lookahead-state-copy stuff has been moved from
;;;   propose-space to evaluation-properties, which also has some new
;;;   functionality.  the new syntax and semantics will be found in the
;;;   release notes.  more information is also given taql-support.soar.
;;;
;;; 4-9-90 - ema - productions generated by init-state now test that
;;;   the current goal is not an evaluation subgoal.  this makes it
;;;   easy to get rid of the require preference for the initial state
;;;   and to implement leap, which installs a new state.  since
;;;   acceptable preferences are back-traced through, this shouldn't
;;;   affect chunking. 
;;;
;;; 3-29-90 - ema - Added ^default-operator-copy.
;;;
;;; 3-25-90 - ema - Changed goal-test-group to take a
;;;   final-state-object* from the goal, and augment that, rather than
;;;   every gtg production generating a new one.  This fixes a bug with
;;;   multiple tests or TCs instantiating.  For now, all augmentions of
;;;   the final state object are made parallel, including final-state-type*
;;;   (so a state can both succeed and fail; this is the user's problem).
;;;
;;; 3-22/23-90 - ema - Changed implementation of editing-control and
;;;   reconsidering.  All augmentations that control operators
;;;   (edit-enabled*, edit-from-subgoal-enabled*, reconsider*,
;;;   select-once-only*) are now on an object that is the value of the
;;;   ^control-stuff* attribute of the operator.  Reasons:  we were
;;;   already augmenting the operator with control stuff
;;;   (select-once-only*) because it's the most convenient, and there
;;;   might be more of that stuff; it means we can stop using the
;;;   operator as the value of control flags (the Soar bug still isn't
;;;   fixed); it means that chunks learned that implement operator
;;;   control are OC, so they're sticky, like the a-o and r-s
;;;   productions that work on control flags.
;;;   Also, reconsider* is now reconsider-disabled*, which means that
;;;   consecutive operator selections work now.  ^applied is now
;;;   funny, but the only place we use it is in evaluation, and that
;;;   will all change soon.  Search for "3-23".  For a discussion, see
;;;   /afs/cs/user/altmann/soar/taql/doc/reconsider.txt.
;;; 
;;; 8-mar-90 - gry - Changed goal testing so that a final-state operator
;;;   is proposed and required when there is a final-state* flag on the
;;;   state for the current goal.  This fixes assorted bugs involving
;;;   premature final-state detection.  It also makes the traces more
;;;   readable.  Because of this change, the state-inconsistent* flag is
;;;   no longer needed.  Productions generated by result-superstate are now
;;;   part of the implementation of the final-state operator.  Note that if
;;;   the user has an operator that is required at the same time as a goal
;;;   test applies, there will be a constraint-failure impasse.
;;;
;;; -----------------------------------------------------------------------
;;; 1-mar-90:  TAQL 3.1.0 (external) was frozen and released at this point.
;;; -----------------------------------------------------------------------
;;;
;;; 1-mar-90 - gry - added copyright notices, set public version flag, and
;;;   updated release date.  Also fixed a bug in prepare-operator.
;;;
;;; 2-27-90 - ema - changed the syntax of OPERATOR-SPEC to allow local
;;; :when conditions.  the new syntax is given in prepare-operator and
;;; the release notes.  to support this, the :substructure keyword now
;;; has to take an explicit list of CONDs or SOAR-ACTIONs, rather than
;;; multiple COND/SOAR-ACTION arguments.  changed the context object
;;; "task-operator" to "top-operator".
;;; 
;;; 2-26-90 - ema - added the :use keyword to propose-space.  added
;;; the appropriate top-context object as a value for :use for both
;;; p-i-s and p-sp.  so p-i-s takes
;;;         :use {<id> | superstate | top-state},
;;; and p-sp takes
;;;         :use {<id> | superspace | top-space}.
;;; The superspace value is of dubious use, but makes things uniform.
;;; I don't know about the hyphen in top-state/space, but it makes it
;;; consistent with the values of CONTEXT-OBJECT.
;;;
;;; 21-feb-90 - gry - 
;;;   1. Changed compiler to correspond to changes made to edit-enabled*,
;;;      etc, flag handling in taql-support.soar.  See the history entry in
;;;      taql-support.soar for this date.
;;;   2. Added support for sliding operators.  Edit clauses in an
;;;      apply-operator TC take a new :type keyword.  It is optional, and at
;;;      most one can appear.  It can take one of two values:  one-shot, or
;;;      sliding.  If not specified, the default is one-shot.  Edits in a
;;;      sliding edit clause can apply at any time the state is internally
;;;      consistent.  The only time the state is NOT internally consistent
;;;      is when there are pending buffered adds.  All "states" the operator
;;;      goes through during sliding edits (except for the last one) are
;;;      considered inconsistent with respect to other TCs (such as goal
;;;      tests).
;;;   3. Changed the :what keyword in edit clauses so that arbitrary objects
;;;      can be edited, not just those attached directly to the state.  The
;;;      new syntax is
;;;         :what {top-state | state | ({STATE-ATT | :none} CLASS-NAME [ID])
;;;      This is compatible with the old definition.  In the new form, if
;;;      :none is given in place of a state attribute, then the ID variable
;;;      must be given and must be bound somewhere in the conditions.  The
;;;      STATE-ATT form is retained both for compatibility and to provide a
;;;      convenient shorthand for the common case of editing first-level state
;;;      subobjects.
;;;
;;; 2-20 - ema - extended ATT-NAME (SOAR-ACTION+) to
;;; ATT-NAME {(SOAR-ACTION+)}+, for multi-attributes of objects.
;;; 
;;; 2-18 - ema - added a trojan horse to blow up TAQL on my birthday
;;; (today) in 1999.  :terminate-when's argument is now optional; if
;;; it's omitted, no reconsider it generated anywhere.  the
;;; synchronization flag for result-superstate edits is now
;;; ^edit-from-subgoal-enabled*, so that an operator application can
;;; be split (partially directly, partially in a subgoal).
;;; 
;;; 16-feb-90 - gry - Fixed bug in tpm -- it was calling TAQL's modified
;;; version of pclass instead of original-pclass when printing out declare-p's.
;;; Also changed tpm so that it always prints a declare-p, even if the pclass
;;; is NIL -- previously there was a Soar bug that kept pclass from accepting
;;; NIL as an argument.   Updated taql-sp to map a pclass of UNKNOWN to NIL,
;;; since this is what Soar's new version of pclass requires as an argument.
;;;
;;; 2-16-90 - ema - put tpm back in.  spm should do what tpm does
;;; (take :file, and print declare-ps), but wait on soar-bugs for
;;; that.  updated the release creation date to today.  replaced
;;; excise-old-instances with a call to the redefined excise.
;;;
;;; 2-15-90 - ema - changed goal-test-group productions to
;;; "not-sticky" from "unknown"
;;; 
;;; 2-13-90 - ema - fixed a bug i introduced into new-taql, when i
;;; integrated redef-soarfuns.lisp; added function
;;; "excise-old-instance", which introduces some redundant excising
;;; code.  search for "2-13".
;;;
;;; 2-12-90 - ema - added top context objects (top-goal, top-space,
;;; top-state, task-operator) as CONTEXT-OBJECTs.  added "top-state"
;;; as a legal value for ":what" in apply-operator.  moved memory for
;;; buffered adds from the state to the goal.  added ":terminate-when"
;;; as a keyword to apply-operator.  search for "2-12".
;;;
;;; 2-10-90 - ema - added the contents of redef-soarfuns.lisp, and
;;; removed taqlc and tpm.  fixed potential bug in TC-type-to-TC-names.
;;; extended ATT-NAME RVALUE to ATT-NAME {RVALUE}+ for creating
;;; multi-attributes.  search for "2-10". 
;;;
;;; 5-feb-90 -- GRY -- Extended taqlc to support unpbreak, unptrace,
;;;   full-matches, pm, spm, and spr.  These are all of the Soar interface
;;;   functions that can take production names as arguments that taqlc
;;;   didn't previously support.  (taqlc pclass ...) now prints more
;;;   meaningful output -- it prints the production names corresponding
;;;   to the pclasses it prints.
;;;
;;; 5-feb-90 -- GRY -- Added constants *num-separator* and *name-separator*
;;;   that define the separator symbols used in generated production names.
;;;   *num-separator* is the symbol that separates the number at the end of
;;;   a production name from the rest of the production name.
;;;   *name-separator* is the symbol that separates all other name components.
;;;   I changed the number separator from ~ to * because ~ doesn't read back
;;;   in properly in Allegro Common Lisp -- it is treated as a terminating
;;;   character.
;;;
;;; 2-5-90 - ema - added taql-version, taql-stats, taql-greeting.
;;; taql-stats is new for the interface.  added :select-once-only
;;; keyword for propose-operator (see defun "propose-operator").
;;; 
;;; 2-3-90 - ema
;;;   new production name format that includes TC names (see "defun
;;;   build-prefix").  also, r-s now really does take :space
;;;   optionally. 
;;;
;;;   1-30-90 -- GRY -- new TCs to complete PSCM, plus a few other changes:
;;;     - Added :use keyword to propose-initial-state and result-superstate
;;;     - Changed final state handling so that final state are specific
;;;       to a goal.  Necessary because the same state symbol could be in
;;;       several different goals at once.  See the release notes for the
;;;       new form of the final state signal.
;;;     - Added propose-superobjects TC
;;;     - Major changes to propose-space to support additional impasse types
;;;       and allow the same initialization keywords as propose-initial-state.
;;;       Eliminated initialize-space TC.
;;;     - Changes to prefer and compare.  Generalized to all context object
;;;       types, and preferences applicable in goals other than the one
;;;       the :when conds apply to.  Prefer allows reconsider as a value
;;;       for :value, and multiple :value keywords can be given.
;;;     - Added propose-task-state TC.  Taql-support now creates default
;;;       empty top state with worst preference, so that propose-task-state
;;;       can override.
;;;     - Modified propose-task-operator to take :space keyword.
;;;     - Generalized goal-test-group so that no :test keyword is required.
;;;       In this case, the group is satisfied if the :space and :when conds
;;;       are satisfied.
;;;     - Modified apply-operator so that reconsider preference is not made
;;;       unless at least one edit actually applies.  Also, we now allow
;;;       apply-operator TCs with no edits.  Not very useful maybe, but
;;;       it seems cleaner.  If no edits apply in apply-operator TCs for
;;;       the selected operator, the operator will no-change.
;;;     - Modified tpm so that it prints both the productions and a
;;;       declare-p call to define the production class of each production.
;;;       This makes tpm read/write compatible.
;;;     - Modified augment so that it forces the production it creates to
;;;       be non-sticky.  Also, added a :space keyword, and changed the
;;;       name it generates for its production -- it is now based on the
;;;       TC names.  Previously it just used augment*<gen-number>, which
;;;       was just too uninformative.
;;;
;;;   1-24-90 -- made value of edit-enabled* the operator, to hack
;;;   around a problem reported to soar-bugs on this date.
;;;   
;;;   1-18-90 -- :copy-new state copy bug in taql-support.soar.
;;;   
;;;   1-12-90 -- made TEST-NAME local to a TC instance
;;;
;;;   1-8-90 -- augment's actions now retract; spurious <id> in a
;;;   FIRST-ACTION detected; function/variable names reflect new
;;;   condition and action forms.
;;;   
;;;   1-4-90 -- atomic values for list-valued keywords detected (see
;;;   group-arguments)
;;;

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;;; -----------------------------------------------------------------------
;;
;; Below is a variety of useful utility routines.
;;
;;; -----------------------------------------------------------------------

(defvar *current-taql-name* nil
  "The name of the taql construct currently being processed.")

;; 2-3 - set as a side effect of a call to new-taql:
(defvar *current-taql-type* nil
  "The TC type of the construct currently being processed.")

;; 2-3:
(defvar *prod-name-prefix* nil
  "The prefix from which to derive all production names for the
current taql construct.")

(defvar *loaded-taqls* nil
  "A list of the names of currently loaded taql constructs.")

(defvar *default-taqls* nil
  "A list of the names of currently loaded default taql constructs.")

(defvar *taql-default-mode* nil
  "Non-nil if loaded TCs are defaults.  Set/reset by start-default/stop-default.")

(defvar *taql-warning-occurred* nil
  "Non-nil if a TAQL warning has been issued since the last call to reset-taql-warnings")

(defun reset-taql-warnings ()
  (setq *taql-warning-occurred* nil))

(defun taql-warning-occurred ()
  *taql-warning-occurred*)

(defun taql-warn (format-string &rest args)
  (taql-warn-common format-string args t))

(defun taql-warn2 (format-string &rest args)
  (taql-warn-common format-string args nil))

;; taql-warn-soft and taql-warn-soft2 are just like taql-warn and taql-warn2
;; except that they do not set *taql-warning-occurred* to non-nil.
;;
(defun taql-warn-soft (format-string &rest args)
  (taql-all-warn-common format-string args t nil))

(defun taql-warn-soft2 (format-string &rest args)
  (taql-all-warn-common format-string args nil nil))

(defun taql-warn-common (format-string args print-taql-name)
  (setq *taql-warning-occurred* t)
  (when (currently-expanding-pseudo-sp)
    (pseudo-sp-warning-occurred t))
  (taql-all-warn-common format-string args print-taql-name t))

(defun taql-all-warn-common (format-string args print-taql-name error)
  (fresh-line *error-output*)
  (if error
    (write-string "TAQL error: " *error-output*)
    ;; ELSE
    (write-string "TAQL warning: " *error-output*))
  (when print-taql-name
    (prin1 *current-taql-name* *error-output*)
    (write-string ".." *error-output*))
  (eval `(format *error-output* ',format-string
	  ,@(mapcar #'(lambda (arg) (list 'quote arg)) args)))
  (terpri *error-output*))

(defun all-tc-type-names ()
  (mapcar #'car *taql-types*))

(defun tc-type-name-p (arg)
  (assoc arg *taql-types*))

(defun default-tc-p (arg)
  (member arg *default-taqls*))

(defun collect (pred list)
  (mapcan #'(lambda (x)
              (if (funcall pred x)
                  (list x)
		  ;; ELSE
                  nil))
      list))

;;; 6-26-91 - gry
;;;
;;; Performance monitoring showed lots of time was going into variable-p.
;;; I changed it slightly from the original, and it runs nearly twice as
;;; fast now.  The original is commented out below.
;;;
;;; ;; 4-25-90 - ema - from the current soar parser:
;;; (defun variable-p (thing)
;;;   (and (symbolp thing)
;;;       (let* ((symbol-name (symbol-name thing))
;;; 	     (length-symbol-name (length (the string symbol-name))))
;;; 	(and (> length-symbol-name 2)
;;; 	    (char= (aref symbol-name 0) #\<)
;;; 	    (char= (aref symbol-name (1-
;;; 				      length-symbol-name)) #\>)))))
;;;
(defun variable-p (thing)
  (and (symbolp thing)
      (let* ((symbol-name (symbol-name thing))
	     (length-symbol-name (length (the string symbol-name))))

	(declare (type string symbol-name))
	(declare (type integer length-symbol-name))

	(and (> length-symbol-name 2)
	     (char= (char symbol-name 0) #\<)
	     (char= (char symbol-name (1- length-symbol-name)) #\>)))))

;; This assumes its argument satisfies variable-p.  It returns the string
;; that appears between < and > in the variable -- e.g.,
;; (variable-name-part '<abc>) --> "ABC".
;;
(defun variable-name-part (var)
  (let ((name (symbol-name var)))
    (subseq name 1 (1- (length name)))))

;;; 6-26-91 - gry
;;;
;;; Performance monitoring showed that a substantial fraction of time
;;; was going into makestring.  I've replaced it with an optimized version,
;;; and left the old definitions in as comments.
;;;
;;; (defun makestring (&rest components)
;;;   (apply #'concatenate
;;;       (cons 'string
;;; 	  (mapcar #'(lambda (x)
;;; 		      (write-to-string x :escape nil
;;; 			  :case :upcase))
;;; 	      components))))
;;;
(defun makestring (&rest components)
  (let ((component-strings nil)
	(total-length 0))

    (declare (type list component-strings))
    (declare (type integer total-length))

    (dolist (comp components)
      (let ((comp-string
	     (typecase comp
	       (symbol (symbol-name comp))
	       (string comp)
	       (t
		(write-to-string comp :escape nil)))))

	(declare (type string comp-string))

	(push (cons total-length comp-string) component-strings)
	(setq total-length (+ (length comp-string) total-length))))

    (let ((result-name (make-string total-length)))

      (declare (type simple-string result-name))

      (dolist (item (nreverse component-strings))
	(replace result-name (cdr item) :start1 (car item)))

      (nstring-upcase result-name))))

(defun makesym (&rest name-components)
  (intern (apply #'makestring name-components)))

(defun newsym (&rest name-components)
  (gentemp (apply #'makestring name-components)))

(defun restart-genvar ()
  (declare (special *taql-genvar-counter*))
  (setq *taql-genvar-counter* 0))

;;; 6-26-91 - gry
;;;
;;; Performance monitoring showed that a substantial fraction of time
;;; was going into makestring, much of it from the calls in genvar and
;;; genvar2.  I've replaced them with optimized versions that do not call
;;; makestring, and left the old definitions in as comments.
;;;
;;; (defun genvar (&optional (leader '=))
;;;   (declare (special *taql-genvar-counter*))
;;;   (when (variable-p leader)
;;;     (setq leader (variable-name-part leader)))
;;;   (intern (makestring '< leader '* (incf *taql-genvar-counter*) '>)))
;;;
;;; (defun genvar2 (&optional (leader '=))
;;;   (declare (special *taql-genvar-counter*))
;;;   (intern (makestring '<
;;; 		      (char (symbol-name leader)
;;; 			    (if (variable-p leader)
;;; 			      1
;;; 			      ;; ELSE
;;; 			      0))
;;; 		      '*
;;; 		      (incf *taql-genvar-counter*)
;;; 		      '>)))

(defun genvar (&optional (leader '=))
  (declare (special *taql-genvar-counter*))

  (let ((num-string (princ-to-string (incf *taql-genvar-counter*)))
	(leader-name (if (variable-p leader)
		       (variable-name-part leader)
		       ;; ELSE
		       (typecase leader
			 (symbol (symbol-name leader))
			 (string leader)
			 (t
			  (write-to-string leader :escape nil))))))

    (declare (type string num-string leader-name))

    (let* ((leader-length (length leader-name))
	   (result-length (+ 3 leader-length (length num-string))))

      (declare (type integer result-length leader-length))
    
      (let ((result-name (make-string result-length)))
	
	(declare (type simple-string result-name))
	
	(setf (schar result-name 0) #\<)
	(replace result-name leader-name :start1 1)
	(nstring-upcase result-name :start 1 :end (1+ leader-length))
	(setf (schar result-name (1+ leader-length)) #\*)
	(replace result-name num-string :start1 (+ 2 leader-length))
	(setf (schar result-name (1- result-length)) #\>)
      
	(intern result-name)))))

(defun genvar2 (&optional (leader '=))
  (declare (special *taql-genvar-counter*))
  (declare (type symbol leader))

  (let ((num-string (princ-to-string (incf *taql-genvar-counter*)))
	(prefix-char (if (variable-p leader)
		       (char (symbol-name leader) 1)
		       ;; ELSE
		       (char (symbol-name leader) 0))))

    (declare (type string num-string))
    (declare (type character prefix-char))

    (let ((result-length (+ 4 (length num-string))))

      (declare (type integer result-length))
    
      (let ((result-name (make-string result-length)))
	
	(declare (type simple-string result-name))
	
	(setf (schar result-name 0) #\<)
	(setf (schar result-name 1) prefix-char)
	(setf (schar result-name 2) #\*)
	(replace result-name num-string :start1 3)
	(setf (schar result-name (1- result-length)) #\>)
      
	(intern result-name)))))

(defun assign-context-vars ()
  (mapc #'(lambda (context-var)
            (set context-var
		(genvar (char (symbol-name context-var) 1))))
      *taql-context-variables*))

;; Find the variable corresponding to the specified context object, and get
;; its value (which is the id variable we are using to represent that
;; context object).
;;
(defun context-var (context-obj)
  (eval
   (do ((objs *taql-context-objects* (cdr objs))
	(vars *taql-context-variables* (cdr vars)))
       ((eq context-obj (car objs))
	(car vars)))))

(defun context-class (context-obj)
  (case context-obj
    ((goal problem-space state operator) context-obj)
    (object 'goal)
    (supergoal 'goal)
    (superspace 'problem-space)
    (superstate 'state)
    (superoperator 'operator)
    (top-goal 'goal)
    (top-space 'problem-space)
    (top-state 'state)
    (top-operator 'operator)
    (otherwise
      (error "INTERNAL TAQL COMPILER ERROR.  Case selector ~S fell through."
	  context-obj))))

;; input is a list of values, output is the same list suitable for an
;; action that creates a multi-attribute.  this is only probablematic
;; because of the comma as a preference separator.

(defun multify (att values)
  (mapcan
   #'(lambda (x)
       (list '^ att x '+ '&)) values))

;; 9-5-90:
;; Interleave <>'s with args.
;;
(defun not-equal-ify (args)
  (if args
    (cons '<> (cons (car args) (not-equal-ify (cdr args))))
    ;; ELSE
    nil))

(defun arg-list-to-TC-names (arg-list)
  (multiple-value-bind (soar-args taql-names)
      (mixed-args-to-soar-args arg-list nil)
    (declare (ignore soar-args))
    taql-names))

;; This function assumes everything in its argument list is the name of a
;; loaded TC.  taql-names may be either a single TC name or a list of TC
;; names.

(defun TC-names-to-sp-names (taql-names)
  (let ((tc-names (if (listp taql-names)
		    taql-names
		    (list taql-names))))
    (apply #'append
	   (mapcar #'(lambda (tc-name)
		       (get tc-name 'taql-productions))
		   tc-names))))

(defun TC-type-to-TC-names (type)
  (if (tc-type-name-p type)
      (get type 'taql-names)
      ;; ELSE
      ;; 2-10 - now explicitly returns nil:
      (progn
	(taql-warn2 (format nil "~S is not a valid TC type." type))
	nil)))

(defun arg-list-to-sp-names (arg-list)
  (multiple-value-bind (soar-args taql-names)
      (mixed-args-to-soar-args arg-list nil)
    (declare (ignore taql-names))
    soar-args))

;; Prints the productions generated for the TC names specified by its
;; argument list.

(defmacro taqlp (&body body)
  `(taqlp-aux ',body))

(defun taqlp-aux (body)
  (arg-list-to-sp-names body))

;; 7-18-90:  Changed to use soar::soar-date.
;;
(defun taql-stats ()
  (let* ((TC-count (length *loaded-taqls*))
	 (SP-count (length (TC-names-to-sp-names *loaded-taqls*)))
	 (default-TC-count (length *default-taqls*))
	 (default-SP-count (length (TC-names-to-sp-names *default-taqls*)))
	 (user-TC-count (- TC-count default-TC-count))
	 (user-SP-count (- SP-count default-SP-count)))
    
    (taql-version)
    
    (format t "~%TAQL statistics on ~A~%" (soar::soar-date))
    (format t "~A ~A ~A ~A ~A ~A ~A ~A~%"
	(lisp-implementation-type)
	(lisp-implementation-version)
	(machine-type)
	(machine-version)
	(machine-instance)
	(software-type)
	(software-version)
	(short-site-name))
    
    (format t "~%~A TCs (~A user, ~A default)~%    compiled into ~A productions (~A user, ~A default)"
	TC-count user-TC-count default-TC-count
	SP-count user-SP-count default-SP-count)

    t))

;; Print the productions generated from the specified TCs.

(defmacro tpm (&body body)
  `(tpm-aux ',body))

;; GRY - 16-feb-90 - Changed to work with Soar's new version of declare-p.
;; There was a bug in the old version that wouldn't let you declare a
;; production as unknown.  That is fixed now, so TPM always prints out
;; a declare-p.
;;
;; 7-16-90:
;;  Change tpm so that it no longer prints out declare-p calls.  In
;;  Soar 5.2, (declare-p (pclass prod) prod) is no longer a no-op, as it
;;  extends the class to all actions of the productions.  Thus it is no
;;  longer safe to load productions printed by tpm and expect them to
;;  run the same as when they were compiled.  You have to reload the
;;  TAQL source to make sure it does the right thing.
;;
(defun tpm-aux (body)
  (let ((file-supplied (eq (car body) :file))
	(filename nil)
	(taql-names nil))
    
    (if file-supplied
	(if (cdr body)
	    (progn
	      (setq filename (cadr body))
	      (setq body (cddr body)))
	    ;; ELSE
	    (taql-warn2 "tpm: No value given for :file keyword.")))
    
    (setq taql-names (arg-list-to-TC-names body))
    
    (cond (file-supplied
	   (with-open-file (ofile filename
			       :direction :output
			       :if-exists :supersede)
	     (let ((*standard-output* ofile))
	       (dolist (name taql-names)
		 (let ((prods (TC-names-to-SP-names name)))
		   (when prods
		     (format ofile "~%;;; TAQL construct ~S~%" name)
		     (dolist (prod prods)
		       (eval `(spm ,prod)))))))))
	(t
	 (dolist (name taql-names)
	   (let ((prods (TC-names-to-SP-names name)))
	     (dolist (prod prods)
	       (eval `(spm ,prod))))))))
  t)

;; Return the TC name corresponding to the production name given as
;; its argument. 

(defmacro ptaql (sp-name)
  `(ptaql-aux ',sp-name))

(defun ptaql-aux (sp-name)
  (do ((tps *loaded-taqls* (cdr tps)))
      ((or (null tps)
	   (member sp-name (TC-names-to-SP-names (car tps))))
       (if (null tps)
	   nil
	   ;; ELSE
	   (car tps)))))

;; Return a list of the names of loaded TCs of the specified types.

(defmacro taqln (&body body)
  `(arg-list-to-TC-names ',body))

;;; 6-11-91 - gry - Added duplicate-tc-warn.

(defvar *duplicate-tc-warn*
  nil
  "Non-nil means print warning message when a TC is loaded that has the same name as an existing TC.")

(defmacro duplicate-tc-warn (&body body)
  `(duplicate-tc-warn-aux ',body))

(defun duplicate-tc-warn-aux (arglist)
  (let ((arg (car arglist)))
    (cond
     ((null arglist)
      (if *duplicate-tc-warn*
	:yes
	;; ELSE
	:no))
     ((cdr arglist)
      (taql-warn2 "Usage: (duplicate-tc-warn [:yes | :no])")
      nil)
     ((member arg '(:yes :no))
      (setq *duplicate-tc-warn* (eql arg :yes))
      arg)
     (t
      (taql-warn2 "Usage: (duplicate-tc-warn [:yes | :no])")
      nil))))

;; Generate/excise a production that turns off/on verbose output.

(defvar *verbose-disable-prod*
  (newsym 'taq*disable-verbose*)
  "Name of the production that disables monitoring output")

(defmacro taql-verbose (&body body)
  `(taql-verbose-aux ',body))

;; 7-16-90:  Changed to use ^verbose false to disable verbosity, and
;;   - ^verbose false to enable.  This is for consistency with Soar 5.2.
;;   Note the lack of an asterisk on the end of ^verbose, unlike past
;;   versions.

(defun taql-verbose-aux (arglist)
  (let ((arg (car arglist))
	(switch (get *verbose-disable-prod* 'switch))
	(prod (get *verbose-disable-prod* 'prod)))
    
    ;; at first call, put a production on its name's plist, and note
    ;; the default value of :on on the name's plist:
    
    (when (not prod)
      (setq prod
	    (setf (get *verbose-disable-prod* 'prod)
		  `(sp ,*verbose-disable-prod*
		       (goal <top> ^ object nil)
		       -->
		       (goal <top> ^ verbose false))))
      (setq switch
	    (setf (get *verbose-disable-prod* 'switch) ':on)))
    
    (cond
      ((null arglist)
       switch)
      ((cdr arglist)
       (taql-warn2 "Usage:  (taql-verbose [:on | :off])")
       nil)
      ((eq arg ':off)
       (when (eq switch ':on)
	 (eval prod)
	 (setf (get *verbose-disable-prod* 'switch) arg))
       arg)
      ((eq arg ':on)
       (when (eq switch ':off)
	 (eval `(excise ,*verbose-disable-prod*))
	 (setf (get *verbose-disable-prod* 'switch) arg))
       arg)
      (t
       (taql-warn2 "Usage:  (taql-verbose [:on | :off])")
       nil))))

;; This macro disables a production for TAQL by changing it so that it
;; will only fire in goals that have "^disable-taql true".
;;
;; Calling form:
;;    (disable-in-taql production-name goal-identifier-variable *)
;;
;; Production-name is the name of the production to disable in TAQL.  The
;; remaining arguments are goal identifier variables that appear
;; in the production.  Disable-in-taql modifies the production so that
;; it tests (goal <goal-id> ^disable-taql true) for each of the goal
;; identifier variables in this list.
;;
;; Disable-for-taql does not evaluate its arguments.

(defmacro disable-in-taql (sp-name &rest goal-ids)
  `(disable-in-taql-aux ',sp-name ',goal-ids))

(defun disable-in-taql-aux (sp-name goal-ids)
  (let ((sp (get-sp-as-list sp-name)))
    
    (when (null sp)
      (error "INTERNAL TAQL ERROR: There is no production named ~S" sp-name))
    
    (when (null goal-ids)
      (error "INTERNAL TAQL ERROR: No goal ids specified for ~S" sp-name))
    
    (let ((sp-tail (member-if #'listp sp)))
      ;; We have to hunt for sp-tail because the production may or may
      ;; not have a production type such as "ELABORATE" after the
      ;; production name.
      
      (when (not
	     (subsetp goal-ids
		 sp-tail
		 :test #'(lambda (x y)
			   (and (listp y)
			       (eql (car y) 'goal)
			       (eql (cadr y) x)))))
	(error
	 "INTERNAL TAQL ERROR:  ~S is not a subset of the goal identifiers in ~S"
	 goal-ids sp-name))
      
      (eval `(excise ,sp-name))
      (eval `(,@(ldiff sp sp-tail)
	      ,@(mapcar #'(lambda (goal-id)
			    `(goal ,goal-id ^ disable-taql true))
		 goal-ids)
	      ,@sp-tail)))))

;; Get the SP form of the production named sp-name as a list.  We do this by
;; calling SPM on sp-name with *standard-output* rebound to a string stream,
;; then read from the resulting string to get the list representation of
;; the printed production.
;;
;; Returns nil if sp-name is not the name of a production.
;;
(defun get-sp-as-list (sp-name)
  (when (not (symbolp sp-name))
    (return-from get-sp-as-list nil))
  (with-input-from-string
      (stream (with-output-to-string
		  (*standard-output*)
		(eval (list 'spm sp-name))))
    (let ((read-val (read stream)))
      (if (not (listp read-val))
	  ;; No production named sp-name, spm printed something like
	  ;; "sp-name is not the name of a production"
	  nil
	  ;; ELSE
	  read-val))))

;;; 11/15/90 - gry
;;; Return T if sym is the name of a loaded production, else NIL.  Currently
;;; the best way we have of doing this is calling get-sp-as-list, but
;;; Soar should export a function users can call to do this.
;;;
(defun sp-name-p (sym)
  (if (get-sp-as-list sym)
    t
    nil))
    
;;; 11/15/90 - gry
;;; Cause an error break if sym is not the name of a loaded production.
;;;
(defun check-sp-name (sym)
  (when (not (sp-name-p sym))
    (error "INTERNAL TAQL ERROR: There is no production named ~S" sym)))

(defvar *taql-trace-attributes* nil
  "Maintained by the sticky-taql-trace-attributes function.")

;; 8-6-90:
;;   This function is like Soar's trace-attributes, except that the trace
;;   attributes defined with this function persist after calls to excise-task.
;;   To implement this, TAQL saves up all trace-attributes passed to this
;;   function in a list, and modifies the definition of excise-task so that it
;;   calls Soar's trace-attributes on this list.
;;
(defun sticky-taql-trace-attributes (a-list)
  (trace-attributes a-list)
  (dolist (pair a-list)
    (pushnew pair
	*taql-trace-attributes*
	:test #'equal))
  t)

(defvar *taql-chunk-free-problem-spaces* nil
  "Maintained by the sticky-taql-chunk-free-problem-spaces function.")

;; 9-14-90:
;;   This function adds problem spaces to Soar's *chunk-free-problem-spaces*
;;   variable, but does it in a way that these spaces will remain chunk-free
;;   after a call to (excise-task), which normally resets
;;   *chunk-free-problem-spaces* to nil.
;;   To implement this, TAQL saves up all space names passed to this
;;   function in a list, and modifies the definition of excise-task so that it
;;   adds these spaces to *chunk-free-problem-spaces*.
;;
(defun sticky-taql-chunk-free-problem-spaces (&rest space-names)
  (dolist (space-name space-names)
    (pushnew space-name *taql-chunk-free-problem-spaces*)
    (pushnew space-name *chunk-free-problem-spaces*))
  t)

;;; 2-10 - ema - took the following code from redef.soarfuns.lisp,
;;; written by GRY to redefine appropriate Soar interface functions and
;;; macros to behave properly in the context of TAQL.

;;; Below is code for redefining Soar interface functions so that they
;;; have approriate semantics when use in the context of TAQL.  The
;;; effected functions are: excise, excise-task, full-matches, pbreak,
;;; pclass, pi, pm, ptrace, restart-soar, smatches, spm, spr,
;;; unpbreak, and unptrace.  Most of these functions are just extended
;;; so that they can take TC names of arguments, which are expanded to
;;; the names of the productions they compile into before being passed
;;; to the original Soar interface function.  Some of these functions
;;; must do more than that when generalized to TAQL, however.  These
;;; functions are excise, excise-task, pclass, and restart-soar,
;;; Excise, excise-task, and restart-soar must all be extended to
;;; update some internal TAQL state variables.  Pclass must be
;;; extended to apply to multiple productions (because a TC can
;;; compile into multiple productions).  Soar's pclass function takes
;;; exactly one production name as an argument.  I change it so that
;;; it prints out one line for each production name it gets as an
;;; argument.  Each line prints the production name followed by its
;;; production class.

;; Add-to-end efficiently and destructively appends a list of new elements
;; to an existing list.  It takes two arguments:  new-elements, a list of
;; the new elements to place at the end of the list, and list-spec, which
;; specifies the list to operate on.  list-spec has the form (head . last),
;; where head points to the beginning of the list to be operated on, and
;; last points to the last cons cell in that list.  The null list can be
;; represented as either NIL or (NIL . NIL).  Add-to-end returns a specifier
;; of this same form, so that successive calls to add-to-end using the returned
;; value from the previous call provides an efficient way to append several
;; lists to the same list over a period of time.
;;
;; Add-to-end makes a copy of new-elements so that *it* will not be
;; destructively modified as a side effect.
;;
(defun add-to-end (new-elements list-spec)
  (cond ((null new-elements)
	 list-spec)
      (t
       (let ((copied-elements (copy-list new-elements)))
	 (cond ((null (car list-spec))
		(cons copied-elements
		    (last copied-elements)))
	     (t
	      (setf (cddr list-spec) copied-elements)
	      (cons (car list-spec)
		  (last copied-elements))))))))

;; Return non-nil if and only if name is the name of a loaded TC.
;;
;; 11-feb-91 - gry - Changed to make sure name is a symbol before calling get.
;;
;; 6-12-91 - gry - Frank has asked that I guarantee that this function will
;;   not go away.  So I hereby guarantee it.  I only guarantee that it
;;   returns non-nil for TC names, not anything with more meaning.
;;
(defun TC-name-p (name)
  (and (symbolp name)
       (get name 'taql-type)))

;; This function takes an argument list that may consist of TC or segment
;; specifiers mixed with other arguments.
;;
;; The syntax of an argument list is
;;
;;   NAME-SPEC* [:exclude SEGMENT-OR-TYPE-SPEC*] [:dont-exclude-taql]
;;
;;   NAME-SPEC ::= SEGMENT-OR-TYPE-SPEC | TC-NAME | OTHER-ARG
;;
;;   SEGMENT-OR-TYPE-SPEC ::= SEGMENT-SPEC | :types TC-TYPE-NAME+
;;                            | ([SEGMENT-SPEC] :types TC-TYPE-NAME+)
;;
;;   SEGMENT-SPEC ::= SEGMENT-GROUP-NAME | SEGMENT-NAME | :all | :latest
;;
;;   OTHER-ARG can be anything other than one of the things explicitly listed
;;   here.
;;
;; Error messages will be printed for illegal arguments, though there is no
;; special return value that lets you tell that an error occurred.
;;
;; The argument list expands to a set of Soar production names.  A TC-NAME
;; expands to the SPs the TC compiled into.  A SEGMENT-SPEC expands to
;; the names of productions compiled from TCs in the specified segment(s).
;; An argument of the form ([SEGMENT-SPEC] :types TC-TYPE-NAME+) expands
;; to the SPs compiled from TCs of the listed types (e.g. prefer, compare,
;; propose-operator) in the specified segment(s).  If the optional SEGMENT-SPEC
;; is missing, it defaults to all existing segments.
;; A SEGMENT-OR-TYPE-SPEC of the form ":types TC-TYPE-NAME+" is
;; equivalent to (:all :types TC-TYPE-NAME+).
;;
;; Any OTHER-ARGS are included verbatim in the expanded argument list.  If
;; the allow-non-TC-specs argument to this function is nil, then any OTHER-ARGS
;; that appear will rusult in an error message.
;;
;; If any SEGMENT-OR-TYPE-SPEC arguments appear anywhere after an :exclude
;; keyword, then the specified segments/types are excluded from the expansion
;; of and SEGMENT-OR-TYPE-SPECs appearing prior to the first :exclude
;; keyword.  Productions expanded from explicitly listed TC names, and
;; productions names appearing in OTHER-ARGs, are never excluded.
;;
;; Unless the :dont-exclude-taql is given, :exclude taql is treated
;; as an implicit argument.  This will exclude the items in TAQL's internal
;; segment.
;;
;; Six values are returned (all six values exclude anything excluded by
;; :exclude keywords):
;;
;;   1. A list that contains all OTHER-ARGS together with the names of all
;;      Soar productions specified by the other arguments.
;;   2. A list of all TCs names specified, either explicitly or by a
;;      SEGMENT-OR-TYPE-SPEC.
;;   3. A list of the names of segments named in the argument list for
;;      which no part of the segment was excluded.  Segments named only
;;      in (SEGMENT-SPEC :types TC-TYPE-NAME+) specs are not included in
;;      this list.  Only segments explicit named by an unqualified SEGMENT-SPEC
;;      are listed, and even then only if no part of the segment is excluded
;;      in an :exclude argument.
;;   4. A list of all OTHER-ARGS together with the names of productions
;;      expanded from explicit TC-NAME arguments
;;   5. A list of all explicit TC-NAME arguments
;;   6. A list of all implicitly specified TC names, expanded from
;;      SEGMENT-OR-TYPE-SPEC arguments.
;;
;; This function is used to extend Soar interface functions to apply to
;; TCs as well as Soar productions -- for example, it will convert a list
;; containing both SP and TC names into a list containing only SP names, which
;; can then be passed to a Soar interface function such as SPM or SMATCHES.
;; It also permits us to extend the Excise command to apply to entire
;; segments.
;; 
;; We try to preserve the order of the arguments that are not
;; SEGMENT-OR-TYPE-SPECs.  The first return value has the expansions of
;; any SEGMENT-OR-TYPE-SPECs at its beginning, followed by the expansions
;; of other arguments in the order they appeared.  This is an attempt to get
;; things like spr to work right (its optional final argument is a number).
;; Currently, all Soar commands that we redefine take any special arguments
;; at the end of their argument list, so this works.  To be really robust,
;; we'd have to preserve the order of all arguments, including
;; SEGMENT-OR-TYPE-SPECs.  But that is kind of complicated, in the face of
;; :exclude.
;;
;; The :default keyword supported in TAQL 3.1.3 is no longer supported.
;; It is superceded by :dont-exclude-taql.
;;
(defun mixed-args-to-soar-args (args allow-non-TC-specs)
  (prog ((segment-names nil)
	 (explicit-soar-args nil)
	 (explicit-tc-names nil)
	 (implicit-tc-names nil)
	 (including t)
	 (includes nil)
	 (excludes nil)
	 (dont-exclude-taql nil))

   loop
     (when (null args)
       (when (not dont-exclude-taql)
	 (multiple-value-setq (includes excludes)
	   (mixed-args-add-includes-excludes
	    'taql nil includes excludes)))
       (setq includes
	     (mixed-args-subtract-includes-excludes includes excludes))
       (setq implicit-tc-names
	     (expand-segment-args-to-tcs includes))
       (setq segment-names (collect #'atom includes))
       (return-from mixed-args-to-soar-args
	 (values (append (TC-names-to-sp-names implicit-tc-names)
			 (car explicit-soar-args))
		 (append (car explicit-tc-names) implicit-tc-names)
		 segment-names
		 (car explicit-soar-args)
		 (car explicit-tc-names)
		 implicit-tc-names)))
     
     (let ((arg (car args)))
       (cond
	 ((keywordp arg)
	  (case arg
	    (:all
	     (multiple-value-setq (includes excludes)
	       (mixed-args-add-includes-excludes
		:all including includes excludes)))
	    (:types
	     (let ((type-end (member-if-not #'tc-type-name-p (cdr args))))
	       (multiple-value-setq (includes excludes)
		 (mixed-args-add-includes-excludes
		  `(:all :types ,@(ldiff (cdr args) type-end))
		  including includes excludes))
	       (setq args type-end)
	       (go loop)))
	    (:default
	     (taql-warn2 "The :default keyword is not longer supported.  ~
                          See the description of the :dont-exclude-taql ~
                          keyword."))
	    (:dont-exclude-taql
	     (setq dont-exclude-taql t))
	    (:exclude
	     (setq including nil))
	    (otherwise
	      (taql-warn2
	       (format nil "~S is not a valid keyword in this context."
		   arg)))))
	 
	 ((or (listp arg)
	      (segment-spec-p arg))
	  (multiple-value-setq (includes excludes)
	    (mixed-args-add-includes-excludes
	     arg including includes excludes)))

	 ((TC-name-p arg)
	  (if including
	    (progn
	      (setq explicit-tc-names
		    (add-to-end (list arg) explicit-tc-names))
	      (setq explicit-soar-args
		    (add-to-end
		     (TC-names-to-sp-names (list arg))
		     explicit-soar-args)))
	    ;; ELSE
	    (taql-warn2 "Individual TCs cannot be excluded (~S)" arg)))
	 
	 (t
	  ;; Arg is not a TC specifier.
	  (if allow-non-TC-specs
	    (setq explicit-soar-args
		  (add-to-end (list arg) explicit-soar-args))
	    ;; ELSE
	    (taql-warn2 "No such taql construct as ~S" arg)))))
     
     (setq args (cdr args))
     (go loop)))

;; Seg-or-type-args is a list all of whose elements are either
;;
;;   1. A segment name  OR
;;   2. A list of the form (segment-name . taql-types)
;;
;; We assume all segment names and type names that appear in the arguments
;; actually exist.
;;
;; Return a list of the names of TCs that the arguments expand to.
;;
(defun expand-segment-args-to-tcs (seg-or-type-args)
  (let ((result nil))
    (dolist (arg seg-or-type-args)
      (let ((segment-tcs (segment-tcs (if (atom arg) arg (car arg)))))
	(when segment-tcs
	  (cond ((atom arg)
		 (setq result (add-to-end segment-tcs result)))
		(t
		 (dolist (tc-name segment-tcs)
		   (when (member (get tc-name 'taql-type) (cdr arg))
		     (setq result (add-to-end (list tc-name) result)))))))))

    (car result)))

;; Segment-or-type-spec must have one of the forms
;;
;;   1. SEGMENT-SPEC
;;   2. ([SEGMENT-SPEC] :types TC-TYPE-NAME+)
;;
;; In the second form, SEGMENT-SPEC defaults to all existing segments.
;;
;; If an error is detected, return nil.  Otherwise return a list of items
;; each of which has one of the forms
;;
;;   1. SEGMENT-NAME
;;   2. (SEGMENT-NAME . TC-TYPE-NAME+)
;;
;; where the SEGMENT-NAMEs are expended from the SEGMENT-SPEC.
;;
(defun expand-segment-or-type-spec (segment-or-type-spec)
  (when (and (consp segment-or-type-spec)
	     (eql (car segment-or-type-spec) :types))
    (setq segment-or-type-spec
	  (cons :all segment-or-type-spec)))

  (when (not (or (symbolp segment-or-type-spec)
		 (and (consp segment-or-type-spec)
		      (symbolp (car segment-or-type-spec))
		      (eql (cadr segment-or-type-spec) :types)
		      (cddr segment-or-type-spec)
		      (every #'tc-type-name-p (cddr segment-or-type-spec)))))
    (taql-warn2 "Expected a SEGMENT-SPEC or ([SEGMENT-SPEC] :types ~
                 TC-TYPE-NAME+), but found ~S"
		segment-or-type-spec)
    (return-from expand-segment-or-type-spec nil))

  (let ((segment-names (expand-segment-spec (if (atom segment-or-type-spec)
					      segment-or-type-spec
					      ;; ELSE
					      (car segment-or-type-spec)))))
    (if (atom segment-or-type-spec)
      segment-names
      ;; ELSE
      (let ((types (remove-duplicates (cddr segment-or-type-spec))))
	(mapcar #'(lambda (segment-name)
		    (cons segment-name types))
		segment-names)))))

;; Segment-or-type-spec must have one of the forms
;;
;;   1. SEGMENT-SPEC
;;   2. ([SEGMENT-SPEC] :types TC-TYPE-NAME+)
;;
;; In the second form, SEGMENT-SPEC defaults to all existing segments.
;;
;; Returns (values includes excludes) if it detects an error in its
;; arguments.  Otherwise it updates includes or excludes to contain
;; segment-of-type-spec (depending on the value of include-p),
;; and returns (values updated-includes updated-excludes).
;;
;; This may destructively modify includes or excludes.
;;
(defun mixed-args-add-includes-excludes (segment-or-type-spec
					 include-p includes excludes)

  (dolist (spec (expand-segment-or-type-spec segment-or-type-spec))
    (let ((seg-name (if (atom spec) spec (car spec))))
      (if include-p
	(cond ((member seg-name includes)
	       ;; Do nothing -- already all there
	       )
	      ((member-if #'(lambda (x) (and (listp x) (eql (car x) seg-name)))
			  includes)
	       (if (atom spec)
		 (setq includes
		       (cons seg-name
			     (delete-if #'(lambda (x)
					    (and (listp x)
						 (eql (car x) seg-name)))
					includes)))
		 ;; ELSE
		 (let ((item 
			(find-if #'(lambda (x)
				     (and (listp x) (eql (car x) seg-name)))
				 includes)))
		   (setf (cdr item) (union (cdr item) (cdr spec))))))
	      (t
	       (if (atom spec)
		 (push spec includes)
		 ;; ELSE we make a copy so that later destructive operations
		 ;; won't hurt us.
		 (push (copy-list spec) includes))))

	;; ELSE not include-p, add to excludes.

	(cond ((member seg-name excludes)
	       ;; Do nothing -- already all there
	       )
	      ((member-if #'(lambda (x) (and (listp x) (eql (car x) seg-name)))
			  excludes)
	       (if (atom spec)
		 (setq excludes
		       (cons seg-name
			     (delete-if #'(lambda (x)
					    (and (listp x)
						 (eql (car x) seg-name)))
					excludes)))
		 ;; ELSE
		 (let ((item 
			(find-if #'(lambda (x)
				     (and (listp x) (eql (car x) seg-name)))
				 excludes)))
		   (setf (cdr item) (union (cdr item) (cdr spec))))))
	      (t
	       (if (atom spec)
		 (push spec excludes)
		 ;; ELSE we make a copy so that later destructive operations
		 ;; won't hurt us.
		 (push (copy-list spec) excludes)))))))

  (values includes excludes))

;; Returns (values includes excludes) if it detects an error in its
;; arguments.  Otherwise it removes from includes anything listed in
;; excludes, and returns the resulting includes.  This may destructively
;; modify includes.
;;
(defun mixed-args-subtract-includes-excludes (includes excludes)
  (dolist (spec excludes)
    (if (atom spec)
      (setq includes
	    (delete-if #'(lambda (x)
			   (or (eql x spec)
			       (and (listp x)
				    (eql (car x) spec))))
		       includes))

      ;; ELSE spec has the form (SEGMENT-NAME . TYPE-NAME+)

      (let ((match (member-if #'(lambda (x)
				  (or (eql x (car spec))
				      (and (listp x)
					   (eql (car x) (car spec)))))
			      includes)))
	(setf (car match)
	      (cons (car spec)
		    (set-difference (if (atom (car match))
				      (all-tc-type-names)
				      ;; ELSE
				      (cdar match))
				    (cdr spec))))

	;; If the set of type becomes empty, remove the entry entirely.
	;;
	(if (null (cdar match))
	  (setq includes
		(delete-if #'(lambda (x)
			       (or (eql x (car spec))
				   (and (listp x)
					(eql (car x) (car spec)))))
			   includes))))))
  includes)

;; TAQLify-command takes the name of a function or macro that takes a list
;; of Soar production names as arguments, and redefines it so that it also
;; works when given a list of TC names.  The original command must be able
;; to accept multiple SP-name arguments -- essentially, this just replaces
;; each argument in the arg list that is a TC specifier with the names of
;; the Soar productions those TCs compile to.
;;
;; We only actually call the old Soar function if either there are some
;; arguments to pass to it, or there are no arguments passed to the command
;; in the first place.  We do this to correctly handle the case where only
;; TC specifiers are given as arguments, but these specifiers didn't yield
;; any production names.  If we went ahead and called the function anyways,
;; Soar would use a default argument, with confusing results.
;;
;; If taql-specific-func is specified, it must be a function that takes
;; a list of TC names that were specified in the arguments to the command.
;; This function will be called on the list of TC names after calling the
;; original Soar function.  This is useful for commands such as excise, for
;; which generalizing to TAQL means more than just applying the Soar command
;; to the appropriate productions (for excise, TAQL must additionally keep
;; track of what TCs are loaded, for example).
;;
(defun taqlify-command (command &optional taql-specific-func)
  (let* ((old-def (makesym 'original- command))
	 (new-def
	  (eval `(function
		  (lambda (&rest args)
		   (multiple-value-bind (soar-args taql-names)
		       (mixed-args-to-soar-args args t)
		     (prog1
			 (if (or soar-args (not args))
			     (eval (cons ',old-def soar-args)))
		       ,@(if taql-specific-func
			     `((funcall ',taql-specific-func
				taql-names))))))))))
    
    (redefine-once command new-def)
    nil))

;; Redefine command (the name of a function or macro) so that it behaves
;; according to new-def (a function (not the name but the function itself,
;; such as returned by FUNCTION or SYMBOL-FUNCTION)).
;; Before changing the definition, we copy the old definition to a symbol
;; whose name is "original-" prepended to the name of the command.  We only
;; actually do the redefinition if the original-<command> symbol doesn't
;; already have a function binding, to keep things from being redefined more
;; than once.  This is important if the new definition wants to call the old
;; definition (via calling original-<command>).
;;
;; If command was originally a function (not a macro), it is simply given
;; new-def as its new definition.  If command was originally a macro, its
;; new value will be a macro that passes all of its arguments to new-def.
;;
(defun redefine-once (command new-def)
  (let ((old-def (makesym 'original- command)))
    
    (if (fboundp old-def)
					; Don't redefine something twice
	(return-from redefine-once))
    
    (cond ((macro-function command)
	   (setf (macro-function old-def) (macro-function command))
	   (eval
	    `(defmacro ,command (&body body)
	      (list 'apply '',new-def (list 'quote body)))))
	((symbol-function command)
	 (setf (symbol-function old-def) (symbol-function command))
	 (setf (symbol-function command) new-def))
	(t
	 (error "INTERNAL TAQL COMPILER ERROR.  ~S has no function binding."
	     command)))
    
    nil))

;; If command was assigned a new definition by redefine-once, restore its
;; original definition, and make the original-<command> function undefined.
;;
(defun undo-redefine-once (command)
  (let ((orig-def (makesym 'original- command)))
    
    (if (not (fboundp orig-def))
	(return-from undo-redefine-once nil))
    
    (cond ((macro-function command)
	   (setf (macro-function command) (macro-function orig-def)))
	((symbol-function command)
	 (setf (symbol-function command) (symbol-function orig-def))))
    
    (fmakunbound orig-def))
  
  nil)

(defun taql-excise (&rest args)
  (multiple-value-bind (all-soar-args all-tc-names segment-names
				      explicit-soar-args explicit-tc-names
				      implicit-tc-names)
      (mixed-args-to-soar-args args t)
    (declare (ignore all-soar-args explicit-tc-names implicit-tc-names))

    (mapc #'(lambda (tc-name)
	      (taql-excise-tc tc-name 'call-excise segment-names))
	  all-tc-names)

    ;; Put this after the TC excises so that empty segments will get cleaned
    ;; up properly.  taql-excise-segments doesn't remove TCs, it just
    ;; does the other bookkeeping.
    ;;
    (when segment-names
      (taql-excise-segments segment-names))

    ;; See the comments in taqlify-command for an explanation of why this
    ;; call to original-excise must be embedded in the (when ...).
    ;;
    (when (or explicit-soar-args
	      (not args))
      (eval (cons 'original-excise explicit-soar-args)))))
    
;; The call-original-excise argument must only be nil when you
;; know that the productions it would excise are already excised (for example,
;; when called from taql-excise-task-extras).  Otherwise you can end up
;; with inconsistencies.
;;
;; 6-11-91 - gry:
;;
;; The forget-tc-was-in-segments argument
;; should be a list of segments that, if the TC was in that segment, then
;; TAQL will not retain the fact that it had been in that segment.  So, if
;; a TC with the same name is later reloaded, it will not be forced into its
;; old segment.  When a TC is excise as part of (excise-task) or
;; (excise SEGMENT-NAME), you want it to forget which segment the TC was in,
;; but otherwise you don't.
;;
(defun taql-excise-tc (tc-name call-original-excise
			       forget-tc-was-in-segments)

  (when (and (member tc-name *loaded-taqls*)
	     (not (member tc-name *default-taqls*)))
    (remove-tc-from-segment tc-name forget-tc-was-in-segments)
    (remove-tc-from-space-models tc-name)
    (let ((type (get tc-name 'taql-type)))
      ;; ema - remove type->name and name->type pointers:
      ;;
      (setf (get type 'taql-names)
	    (delete tc-name (get type 'taql-names)))
      (remprop tc-name 'taql-type))
    (when call-original-excise
      (eval (cons 'original-excise (get tc-name 'taql-productions))))
    (remprop tc-name 'taql-productions)
    (setq *loaded-taqls* (delete tc-name *loaded-taqls*)))
    
  t)

;; Taql-excise-task-extras is meant to be passed as the second argument to
;; taqlify when redefining EXCISE-TASK so that it does the right thing in
;; the context of a TAQL task.
;; It does the extra work required to excise a TAQL task beyond what Soar's
;; original excise-task would normally do.
;;
;; The argument here will be nil, as excise-task doesn't take any arguments.
;;
;; Excise-task does not affect default TCs.
;;
;; Any sticky-taql-trace-attributes are put back into force after the
;; excise-task.  We reverse the list because we build it in reverse
;; order of the way the user gave the attributes, which results in Soar
;; printing things in a different order than the user might have intended.
;;
;; 9-6-90 - gry - added support for user/default TC distinction.
;;
(defun taql-excise-task-extras (tc-names)
  (declare (ignore tc-names))
  
  (dolist (tc-name *loaded-taqls*)
    (taql-excise-tc tc-name nil (all-non-global-segment-names)))
  (data-model-excise-task-extras)
  (space-model-excise-task-extras)
  ;; Do this last so that empty segments will get cleaned up properly.
  ;;
  (segment-excise-task-extras)

  (trace-attributes (reverse *taql-trace-attributes*))
  (setq *chunk-free-problem-spaces*
	(union *chunk-free-problem-spaces*
	       *taql-chunk-free-problem-spaces*))
  t)

;;; For restart-soar, get rid of ALL TCs, default and user.
;;;
(defun taql-restart-soar-extras (tc-names)
  (declare (ignore tc-names))
  
  (setq *taql-default-mode* nil)
  (setq *taql-trace-attributes* nil)
  (setq *taql-chunk-free-problem-spaces* nil)
  (dolist (type (all-tc-type-names))
    (remprop type 'taql-names))
  (dolist (name *loaded-taqls*)
    (remprop name 'taql-productions)
    (remprop name 'taql-type))
  (setq *loaded-taqls* nil)
  (setq *default-taqls* nil)
  (setq *duplicate-tc-warn* nil)
  (init-taql)
  t)

(defun taql-start-default-extras (tc-names)
  (declare (ignore tc-names))

  (setq *taql-default-mode* t))

(defun taql-stop-default-extras (tc-names)
  (declare (ignore tc-names))

  (setq *taql-default-mode* nil))

;; Taql-pclass will become the new definition of PCLASS when it is redefined
;; to extend it to TAQL.  Soar's pclass only takes a single production name
;; as an argument, which is no good for TAQL, as a TC may compile into
;; several different productions.  The new definition below prints out
;; one line for each production name implied by it argument list.  Each line
;; gives a production name followed by its pclass.
;;
(defun taql-pclass (&rest args)
  (let ((soar-args (mixed-args-to-soar-args args t)))
    (dolist (sp-name soar-args)
      (format t "~%~A: ~A" sp-name (eval (list 'original-pclass sp-name))))
    nil))

;; This replaces the original definition or run.  If the space model has
;; changed since the user last called check-spaces, call check-spaces
;; before running.
;;
(defun taql-run (&rest args)
  (check-spaces-if-necessary)
  (eval (cons 'original-run args)))

;; This replaces the original definition or d.  If the space model has
;; changed since the user last called check-spaces, call check-spaces
;; before running.
;;
(defun taql-d (&rest args)
  (check-spaces-if-necessary)
  (eval (cons 'original-d args)))

;; Undo-redef returns all redefined Soar interface functions to their
;; original definitions.
;;
(defun undo-redefs ()
  (mapc #'undo-redefine-once
      '(full-matches pbreak pi pm ptrace smatches spm spr unpbreak
	unptrace excise excise-task restart-soar pclass start-default
	stop-default))
  nil)

;; Redefine the Soar functions that have different definitions when they
;; are extended to handle TAQL as well as Soar.
;;
;;   2-10 - ema - might be useful to supply a function that tells the
;;   user what commands are redefined, so there is some way to find
;;   out why things don't work anymore if the Soar interface changes.

(eval-when (load eval)
  (mapc #'taqlify-command
      '(full-matches pbreak pi pm ptrace smatches spm spr unpbreak
	unptrace))
  (redefine-once 'excise #'taql-excise)
  (taqlify-command 'excise-task #'taql-excise-task-extras)
  (taqlify-command 'start-default #'taql-start-default-extras)
  (taqlify-command 'stop-default #'taql-stop-default-extras)
  (redefine-once 'pclass #'taql-pclass)
  (redefine-once 'run #'taql-run)
  (redefine-once 'd #'taql-d)
  (taqlify-command 'restart-soar #'taql-restart-soar-extras))

;; The following functions are internal utilities called by the
;; functions that process individual TCs.

;; Updates the list of TC names, and updates plists of TC types and TC
;; names.
;;
;; 9-6-90 - gry - Added support for default TCs (as opposed to the user's
;;   task TCs).
;;
(defun new-taql (name type)
  (if (not (tc-type-name-p type))
      (error "INTERNAL TAQL COMPILER ERROR.  Unknown TC type ~S" type))
  (if (or (not (atom name))
	  (keywordp name))
      (progn
	(taql-warn2 "Illegal taql construct name: ~S" name)
	nil)
      ;; ELSE
      (progn
	(when (get name 'segment)
	  (eval `(begin-segment ,(get name 'segment))))
	(when (member name *loaded-taqls*)
	  ;; 6-11-91 - gry
	  (when *duplicate-tc-warn*
	    (let ((*current-taql-name* name))
	      (taql-warn-soft "Duplicate TC name.")))
	  (eval (list 'excise name)))
	;; side effect, but what the hey:
	(setq *current-taql-type* type)
	;; set up pointers from TC type to TC names, and back.
	(push name (get type 'taql-names)) ; type --> (names)
	(setf (get name 'taql-type) type)  ;  name --> type
	(push name *loaded-taqls*)
	(when *taql-default-mode*
	  (push name *default-taqls*))
	(add-tc-to-segment name)
	name)))

;; 2-3:
;; production-name ::= SPACE<>TAQL-ABBREV<>tc-name<>MESSAGE<>ID
;;   where:
;;     <> is the value of *{num,name}-separator*
;;     SPACE is value of :space, or "generic"
;;     TAQL-ABBREV is derived from *taql-types*
;;     MESSAGE is specific to the function of the production within
;;       the TC, and
;;     ID is a number that makes the name unique.
;;     
;;   returned is that up to and including "tc-name<>".  i used "<>"
;;   because it seems readable, and the parser takes it.
;;
;; 5-feb-90 -- GRY
;; Now use a constant that defines what symbol is used as a separator, so
;; that it is easier to change.  The separator between ID and the rest of
;; the name is the value of *num-separator*, and the separator for all other
;; components is the value of *name-separator*.

(defun build-prefix (space)
  (apply #'makesym
      (list
       (if space
	   space
	   'generic)
       *name-separator*
       (cadr (assoc *current-taql-type* *taql-types*))
       *name-separator*
       *current-taql-name*
       *name-separator*)))

;; This must be called every for every TC that is loaded.  It builds up the
;; inferred space model.  See the comments for add-tc-to-spaces in
;; space-model.lisp for more details.
;;
(defun add-current-tc-to-spaces (space-names prep-op-list
					     &optional (space-function nil))
  (let* ((bad-space (member-if-not #'symbolp space-names))
	 (operator-names
	  (let ((result nil))
	    (dolist (prep-op prep-op-list)
	      (when prep-op
		(pushnew (caar prep-op) result)))
	    result))
	(bad-op (member-if-not #'symbolp operator-names)))

    (cond (bad-space
	   (taql-warn "Problem-space name must be a symbol, but found ~S"
		      (car bad-space)))
	  (bad-op
	   (taql-warn "Operator name must be a symbol, but found ~S"
		      (car bad-op)))
	  (t
	   (add-tc-to-spaces space-names *current-taql-name*
			     *current-taql-type* operator-names
			     space-function)))))

(defmacro taql-sp (pclass &body body)
  `(taql-sp-aux ',pclass ',body))

;; GRY - 16-feb-90 - changed pclass unknown to map to pclass NIL, to work with
;; Soar's new version of declare-p.
;;
;; 9-5-90 - gry - added data type support
;;
(defun taql-sp-aux (pclass body)
  (if (taql-warning-occurred)
    nil
    ;; ELSE
    (let ((soar-pclass
           (case pclass
	     (sticky 'operator-application)
             (not-sticky 'miscellaneous)
             (unknown nil)
             (t
              (error "INTERNAL TAQL COMPILER ERROR.  Bad pclass: ~S"
                     pclass)
              (return-from taql-sp-aux nil)))))
      
      (expand-and-load-pseudo-sp-body body soar-pclass t))))

(defun expand-and-load-pseudo-sp-body (body soar-pclass isa-taql-sp)
  (multiple-value-bind (name lhs rhs)
      (expand-pseudo-sp (cons 'sp body))

    ;; Have to check again for warnings, in case anything showed up
    ;; during pseudo-sp expansion.
    ;;
    (when (taql-warning-occurred)
      (when (and (pseudo-sp-warning-occurred)
		 isa-taql-sp)
	;; Pprint the production, so that the user can try to figure out
	;; what went wrong.  If not isa-taql-sp, then presumably we were
	;; invoked from SP+, and the user already knows what the
	;; production looks like.
	(taql-warn "An error was detected in the following pseudo-production:")
	(pprint-sp `(sp ,name ,@lhs --> ,@rhs) *error-output*))
      (return-from expand-and-load-pseudo-sp-body nil))
    
    (when isa-taql-sp
      (push (car body) (get *current-taql-name* 'taql-productions)))

    (let ((the-production
	   `(sp ,name ,@lhs --> ,@rhs)))

      ;; If unknown, don't explicitly declare it that way, instead let
      ;; Soar figure it out for itself.
      ;;
      (when soar-pclass
	(eval `(declare-p ,soar-pclass ,name)))

      (let ((sp-return-value (eval the-production)))
	(cond (sp-return-value
	       ;; Loaded ok
	       t)
	      (t
	       (taql-warn
		"Soar error in trying to load this production:")
	       (pprint-sp the-production *error-output*)
	       nil))))))

;; This crudely pretty-prints a Soar production that may have errors in it.
;; It isn't great, but does better than just calling pprint.
;;
;; We assume that whatever else may be wrong with the arugment sp, it at
;; least has the form (sp SOMETHING SOMETHING* --> SOMETHING*).
;;
(defun pprint-sp (sp &optional (stream nil))
  (when (null stream)
    (setq stream *standard-output*))
  (multiple-value-bind (name lhs rhs)
      (split-production sp)
    (fresh-line stream)
    (format stream
	    "(~S ~S~{~%   ~S~}~% -->~{~%   ~S~})"
	    (car sp) name lhs rhs))
  t)

;; EXPORTED ROUTINE:
;; SP+ is like Soar's SP command, but it is for loading pseudo-productions
;; rather than real Soar productions.  It expands data type stuff (data
;; macros and directives), attribute paths, and structured value specs.
;; It also works together with the data-model code.
;;
;; This should not be used to process productions as part of compiling
;; a TAQL construct.  Use taql-sp (in taql-compiler.lisp) for that.
;; SP+ is primarily intended for people who want the advantages of
;; data types, type declarations and the new syntax, but don't want to
;; use TAQL.  (Plus, it is a convenient way for me to test this part
;; of the code.)
;;
;; I'm not sure I'll ever tell anyone about this or not, I'd rather they
;; used TAQL.
;;
(defmacro sp+ (&body body)
  `(sp+-aux ',body))

(defun sp+-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)

  (let* ((*current-taql-name* (car body)))
    (expand-and-load-pseudo-sp-body body nil nil)))

;; This takes a list of expressions and keywords and groups the expressions
;; according to type.  It returns a value of the form
;; (free-args . keyword-alist).  Free-args is the list of expressions that
;; are not keyowrd values, with left-to-right ordering preserved from the
;; input list.  keyword-alist is an association list.  The keys in the alist
;; are the keywords found in the input list.  The cdr of the entry for each
;; keyword is a list of values that were specified for that keyword (unlike
;; in standard Common Lisp keyword processing, the same keyword may appear
;; multiple times in the input list, and all values are unioned into an 
;; unordered list).  The value of a keyword in the input list is the
;; expression immediately following the keyword, if any.  If the keyword is
;; immediately followed by another keyowrd or the end of the list, the value
;; is nil.
;;
;;   free-args may be nil.  The cdr of an entry for a keyword will have one
;;   element for each time that keyword appeared (and will be nil if the
;;   keyword did not appear).  There is no way to distinguish the cases
;;   where (a) a keyword appeared without a value and (b) a keyword
;;   appeared with the value nil.  In either case, the value will appear
;;   as nil.
;; 
;;   1-4 - changed to detect if arglist is not a list.  if it
;;   isn't, a warning is generated, and a dummy list is returned that
;;   contains all the legal-keys (to avert cascading errors)
;;   8-feb-91 - changed to no longer return a dummy list in this case.
;;   The dummy list could also lead to cascading errors (illegal keyword
;;   combinations, illegal values, ...).  More importantly, the dummy list
;;   it was returning undid the nice invariant that a keyword appeared iff
;;   the cdr of its entry was non-nil.
;;
(defun group-arguments (arglist legal-keys &optional (where nil specified))
  (when (not (listp arglist))
    (taql-warn "argument must be a list: ~S" arglist)
    (return-from group-arguments nil))
  (do ((lis arglist)
       (free nil)
       (keylist nil))
      ((null lis) (cons (nreverse free) keylist))
    (cond ((keywordp (car lis))
	   (let ((value nil)
		 (keyword (car lis))
		 (entry (assoc (car lis) keylist)))
	     (if (not (member keyword legal-keys))
	       (taql-warn "Illegal keyword ~S appears in ~S"
			  keyword
			  (if specified where arglist)))
	     (cond ((keywordp (cadr lis))
		    (setq value nil)
		    (setq lis (cdr lis)))
		   (t
		    (setq value (cadr lis))
		    (setq lis (cddr lis))))
	     (cond (entry
		    (rplacd entry (cons value (cdr entry))))
		   (t
		    (setq keylist (acons keyword (list value) keylist))))))
	  (t
	   (push (car lis) free)
	   (setq lis (cdr lis))))))

(defun prepare-condition-sets (cond-sets)
  (apply #'append (mapcar #'prepare-conditions cond-sets)))

;; 7-feb-1991: Changed so that it will insert a linkage variable only
;;   if (a) the first element is a context-object and (b) the second
;;   element is a caret (^).  Previously, the (b) condition was that
;;   the second element was a variable.  This would cause it to treat
;;   a condition such as (state {<> <s1> <s2>} ^color red) improperly.
;;
;; 8-2-90: Modified to allow CONTEXT-OBJECT/no-id conditions anywhere in
;;   a list of conditions, not just as the first element.  Also eliminate
;;   the restriction that the first condition in a list must be a list.
;;   Removing this restrictions permits negations as the first element in
;;   the list.  This will also generate context linkage for conditions in
;;   (conjunctive) negations, such as - {(state ^color red)}.
;;
;;   We accumulate the context linkage and splice it into the list *after*
;;   we've processed all the conditions so that the linkage won't be
;;   inside any conjunctions ({}) that might be in the conditions.

(defun prepare-conditions (conds)
  (let ((copy-conds (copy-list conds))
	(context-objects nil))
    (do ((tail copy-conds (cdr tail)))
	((null tail)
	 (nconc (mapcan #'generate-context-linkage context-objects)
		copy-conds))
      (when (and (not (consp (car tail)))
		 (not (member (car tail) '(- { }))))
	(taql-warn "~S is not a valid COND.~%Possibly you are missing ~
                    a pair of parentheses around a list of CONDs."
		   (car tail))
	(return-from prepare-conditions nil))
      (when (and (consp (car tail))
		 (member (caar tail) *taql-context-objects*)
		 (member (cadar tail) '(^ -)))
	(pushnew (caar tail) context-objects)
	(setf (car tail)
	      (cons (context-class (caar tail))
		    (cons (context-var (caar tail))
			  (cdar tail))))))))

;;; 6-12-91 - gry - This is very similar to prepare-conditions.  One difference
;;; is that it returns two values:
;;;
;;;   (1) A list of conditions that will bind context objects in the argument
;;;       actions whose identifiers were not specified.
;;;   (2) The argument actions, with any missing context object identifiers
;;;       filled in (non-destructively).
;;;
;;; This is meant to be called on lists of Soar actions that are values of
;;; :actions and :sliding-actions keywords.
;;;
(defun prepare-actions (actions)
  (let ((copy-actions (copy-list actions))
	(context-objects nil))
    (do ((tail copy-actions (cdr tail)))
	((null tail)
	 (values (mapcan #'generate-context-linkage context-objects)
		 copy-actions))
      (when (not (consp (car tail)))
	(taql-warn "~S is not a valid SOAR-ACTION.~%Possibly you are missing ~
                    a pair of parentheses around a list of actions."
		   (car tail))
	(return-from prepare-actions (values nil nil)))
      (when (and (consp (car tail))
		 (member (caar tail) *taql-context-objects*)
		 (eql (cadar tail) '^))
	(pushnew (caar tail) context-objects)
	(setf (car tail)
	      (cons (context-class (caar tail))
		    (cons (context-var (caar tail))
			  (cdar tail))))))))

;; 2-12 - added linkage for top-context objects.

(defun generate-context-linkage (context-obj)
  (case context-obj
    (goal nil)
    (problem-space
      `((goal ,=goal ^ problem-space ,=problem-space)))
    (state
      `((goal ,=goal ^ state ,=state)))
    (operator
      `((goal ,=goal ^ operator ,=operator)))
    (supergoal
      `((goal ,=goal ^ object ,=object)))
    (object
      `((goal ,=goal ^ object ,=object)))
    (superspace
      `((goal ,=goal ^ object ,=object)
	(goal ,=object ^ problem-space ,=superspace)))
    (superstate
      `((goal ,=goal ^ object ,=object)
	(goal ,=object ^ state ,=superstate)))
    (superoperator
      `((goal ,=goal ^ object ,=object)
	(goal ,=object ^ operator ,=superoperator)))
    (top-goal
      `((goal ,=top-goal ^ object nil)))
    (top-space
      `((goal ,=top-goal ^ object nil ^ problem-space ,=top-space)))
    (top-state
      `((goal ,=top-goal ^ object nil ^ state ,=top-state)))
    (top-operator
      `((goal ,=top-goal ^ object nil ^ operator ,=top-operator)))
    (otherwise
      (error "INTERNAL TAQL COMPILER ERROR.  Case selector ~S fell through."
	  context-obj))))

;; 2-27-90 - ema:
;; parses operator specs and conditions (see note of 12-8).  result
;; is (info . parsed-spec), where info is a list (name . conds).
;; name is the name of the operator (or :no/any-name).  conds is a
;; list of parsed conditions, as specified by local :when clauses.
;; parsed-spec is a list that contains the spec for the operator (a
;; series of conditions or actions).  parsed-spec is nil if it would
;; only amount to class and id, so that parsed-spec can be spliced
;; right into a list of conditions/actions, resulting in either an
;; operator action/conditions sensible structure, or no
;; action/condition at all.
;; 
;;   12-8 - this is both OPERATOR-COND and OPERATATOR-SPEC.  so,
;;   there's no checking here for preference symbols.  also, 
;;   :no-name and :any-name are mapped onto the same thing, the
;;   absence of a name attribute.  probably want to clean this up
;;   later, but that involves two types of calls, one for a COND and
;;   one for a SPEC.
;;
;; :substructure is only actually legal in OPERATOR-SPECs (i.e. things that
;; create operators, rather than match against them), but we don't have a
;; way of checking this.  If :substructure is used in an OPERATOR-COND,
;; it will work -- it will act as though they were specified in a local
;; :when.  But the error messages for :substructure values assume that
;; they are in actions.  Also, :substructure values are not processed by
;; prepare-condition-sets, and so may not contain conditions that have
;; context-objects as the class and omit the identifier field.  The right
;; thing would be to add another argument to distinguish the situations
;; in which it is called.
;;
;; Early versions of TAQL allowed :substructure in conditions, and did not
;; accept the :when keyword.  So not checking this provides some backward
;; compatibility.  I wonder how many people still actually use
;; :substructure in conditions?

(defun prepare-operator (op-spec op-id)
  (if (atom op-spec)
      (list (cons op-spec nil) `(operator ,op-id ^ name ,op-spec))
      
      ;; ELSE op-spec = (OPNAME {ATT-NAME RVALUE*}*
      ;;                [:substructure (SOAR-ACTION+)]*
      ;;                [:when (COND+)]*)
      
      ;; When we group-arguments, only use the cdr of the op-spec.  The
      ;; car might be :no-name or :any-name, which we don't want to treat
      ;; the same way as we treat other keywords.
      
      (let* ((op-args
	      (group-arguments (cdr op-spec) '(:substructure :when)))
	     (substructure (assoc :substructure (cdr op-args)))
             (prefix (cons (car op-spec) (car op-args)))
	     ;; car of return value:
	     (the-car
	      (cons (car op-spec)
		    (prepare-condition-sets
		     (cdr (assoc :when (cdr op-args)))))))

	(when (member-if
	       #'(lambda (sub)
		   (if (member-if #'atom sub)
		     (taql-warn
		      "Argument to :substructure must be a list of actions: ~S"
		      sub)
		     ;; ELSE
		     nil))
	       (cdr substructure))
	  (setq substructure nil))
	
	;; :no-name if it's an OPERATOR-ACTION,
	;; :any-name if it's an OPERATOR-COND.
	(if (or (eql (car op-spec) ':no-name)
		(eql (car op-spec) ':any-name))
	    (if (cdr prefix)
		(cons the-car
		    (cons `(operator ,op-id ,@(cdr prefix))
			(apply #'append (cdr substructure))))
		;; ELSE the nameless op object has no structure:
		(cons the-car nil))
	    ;; ELSE the op object has at least a name:
	    (cons the-car
		(cons `(operator ,op-id ^ name ,@prefix)
		    (apply #'append (cdr substructure))))))))

;; For ATT-NAME {RVALUE+|(SOAR-ACTION+)+}, value specs are given a
;; parallel preference (in value-spec-actions).  Not so for the
;; form ATT-NAME SOAR-ACTION*, because there the programmer has
;; control. 

(defun action-spec-actions (class id action-spec)
  (cond
    ((atom action-spec)
     (taql-warn "Malformed new-object specification: ~S" action-spec) nil)
    
    ;; ACTION-SPEC ::= ATT-NAME {RVALUE+} |
    ;;                 ATT-NAME (SOAR-ACTION+)
    
    ((atom (car action-spec))
     (value-spec-actions class id (car action-spec) (cdr action-spec)))
    
    ;; ACTION-SPEC ::= ACTION-TAIL SOAR-ACTION*
    
    (t
     `((,class ,id ,@(car action-spec))
       ,@(cdr action-spec)))))

;; 13-feb-91 - gry
;; An RVALUE an atom (possibly a variable) or a function/data-macro
;; call or a tree-structured value specification.  A
;; function/data-macro call is a list starting with an atom that
;; doesn't contain "^".  A tree-structured value spec is a list that
;; contains "^" (currently, the class could be a data macro call -- a
;; list -- so this might not begin with an atom).  These tests suffice
;; to discriminate among syntactically correct options.  We might want
;; to do more sophisticated checking at some point to permit better
;; error detection.
;;
(defun rvalue-p (arg)
  (or (atom arg)                          ; a constant or variable
      (member '^ arg)                     ; a tree-structured value spec
      ;; If we get this far, we know arg doesn't
      ;; contain ^, so we don't have to check again.
      (atom (car arg))))                   ; a function/data-macro call

;; 13-feb-91 - gry
;; A list of Soar actions is a non-nil list of lists.    This tests suffices
;; to discriminate among syntactically correct options.  We might want
;; to do more sophisticated checking at some point to permit better
;; error detection.
;;
(defun soar-action-list-p (arg)
  (and (consp arg)
       (every #'listp arg)))

;; 2-10 - changed to allow {RVALUE}+ instead of just RVALUE. val-spec
;; is now (cdr action-spec), instead of (cadr action-spec).
;; 
;; 2-20 - changed to allow {(SOAR-ACTION+)}+.

(defun value-spec-actions (class id att val-spec)
  (cond
    
    ((null val-spec)
     (taql-warn "No values specified for attribute ~S" att)
     nil)

    ;; VALUE-SPEC ::= {RVALUE+}
    
    ((every #'rvalue-p val-spec)
     (list (cons class (cons id (multify att val-spec)))))
    
    ;; VALUE-SPEC ::= {(SOAR-ACTION+)}+.  Always generate the link variable.
    
    ((every #'soar-action-list-p val-spec)
     (let ((obj-list nil)
	   (id-list nil))
       (setq id-list
	   (mapcar
	    #'(lambda (x)
		;; x ::= (SOAR-ACTION+)
		(when (variable-p (cadar x))
		  (taql-warn
		   "Illegal first SOAR-ACTION.  Omit the identifier variable in:  ~S"
		   x))
		
		(let* ((gen-id (genvar2 (caar x))))
		  (push `((,(caar x) ,gen-id ,@(cdar x)) ,@(cdr x))
		      obj-list)
		  gen-id))
	    val-spec))
       (cons (cons class (cons id (multify att id-list)))
	     (apply #'nconc obj-list))))
    (t
     (taql-warn "Malformed value specification in: ~S" val-spec)
     nil)))

;; 8-2-90: Created (GRY).  OBJECT-SPECS are now used in several places,
;;   so have a single function for processing them.
;;
;; Process and OBJECT-SPEC, for example the value of the (edit :what)
;; keyword.
;;
;; Legal syntax: CONTEXT-OBJECT | ({ATT-NAME | :none} CLASS-NAME [ID])
;;
;; When the (ATT-NAME CLASS-NAME [ID]) form is used, the
;; relative-to-context-obj parameter specifies which context object
;; the attribute name is relative to.
;;
;; Returns two values.  The first value is a set of conditions that binds the
;; specified object.  The second value is a list of two items, (class id),
;; where class is the class of the specified object, and id is the identifier
;; variable that refers to it.
;;
;; If the input spec is illegal, it prints an error message and returns
;; values as if the input space was STATE.
;;
(defun process-object-spec (spec relative-to-context-obj)
  (when (and (not (member spec *taql-context-objects*))
	    (or (atom spec)
		(not (member (length spec) '(2 3)))
		(not (atom (car spec)))
		(and (keywordp (car spec))
		    (not (eql (car spec) :none)))
		(not (atom (cadr spec)))
		(and (eql (car spec) :none)
		    (null (cddr spec)))
		(and (cddr spec)
		    (not (variable-p (caddr spec))))))
    (taql-warn "Invalid object specifier: ~S" spec)
    (setq spec 'state))
  
  (when (and relative-to-context-obj
	    (not (member relative-to-context-obj *taql-context-objects*)))
    (error "INTERNAL TAQL ERROR:  Not a context object: ~S"
	relative-to-context-obj))
  
  (let ((bind-conds nil)
	(obj-spec nil))
    
    (cond
      ((member spec *taql-context-objects*)
       (setq bind-conds
	   (generate-context-linkage spec))
       (setq obj-spec
	   (list (context-class spec) (context-var spec))))
      (t
       (let* ((obj-att (car spec))
	      (obj-class (cadr spec))
	      (obj-id (if (cddr spec)
			  (caddr spec)
			  ;; ELSE
			  (genvar2 obj-class))))
	 (setq bind-conds
	     (if (eql obj-att :none)
		 nil
		 ;; ELSE
		 `(,@(generate-context-linkage relative-to-context-obj)
		   (,(context-class relative-to-context-obj)
		    ,(context-var relative-to-context-obj)
		    ^ ,obj-att ,obj-id))))
	 (setq obj-spec (list obj-class obj-id)))))
    
    (values bind-conds obj-spec)))

;;;1 -----------------------------------------------------------------------
;;
;; Below are the macros for each TAQL construction and related routines.
;;
;;; -----------------------------------------------------------------------

;; GRY - 29-jan-90 - This TC lets you put something in the state in the
;; top context.  A good thing to put there is information about the
;; initial and desired task states.  The implementation space for the task
;; operator can then look at these, and its implementation chunks will
;; depend on the initial and desired states.

(defmacro propose-task-state (&body body)
  `(propose-task-state-aux ',body))

(defun propose-task-state-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'propose-task-state))
         (args (group-arguments (cdr body) '(:space :when :new)
		   'construct))
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
         (outer-when-conds (prepare-condition-sets (cdr (assoc :when
							    (cdr args))))) 
         (news (assoc :new (cdr args)))
	 (init-stuff-id (genvar 'i))
         (tc-id (makesym (get-internal-real-time) '*)))
    
    ;; space is optional:
    (when (cddr space)
      (taql-warn "At most one :space argument must appear.")
      (setq space nil))
    
    (if (car args)
	(taql-warn "All arguments must be values of keywords."))
    
    (when (and (not outer-when-conds) (not space))
      ;; Set up default conditions.
      (setq space '(:space top-space)))
    
    (add-current-tc-to-spaces (cdr space) nil)

    (gen-init-productions 'state space outer-when-conds nil nil news
	nil tc-id init-stuff-id)))


;; 3-23 - ema - modified to create the control-stuff* augmentation on
;; the task operator.  see propose-operator for more.

;; 4-9 - ema - task ops are now implicitly select-once-only.  this
;; removes the special case in taql-support for rejecting the task op.

(defmacro propose-task-operator (&body body)
  `(propose-task-operator-aux ',body))

(defun propose-task-operator-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body)
				  'propose-task-operator))
         (args (group-arguments (cdr body) '(:op :when :space) 'construct))
         (op (assoc :op (cdr args)))
	 (op-id (genvar 'o))
	 (c-id (genvar 'c))
	 (prep-op nil)
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 ;; if conditions or space are provided, use them, else default to
	 ;; problem-space ^name top-space:
         (conds
	  (cond
	    ((prepare-condition-sets (cdr (assoc :when (cdr args)))))
	    (space nil)
	    (t
	     (setq space '(:space top-space))
	     nil))))
    
    ;; space is optional:
    
    (when (cddr space)
      (taql-warn "At most one :space argument must appear.")
      (setq space nil))
    
    (when (or (not op) (cddr op))
      (taql-warn "Exactly one :op argument must appear:  ~S" op)
      (setq op '(:op junk)))
    
    (if (car args)
        (taql-warn "All arguments must be values of keywords."))
    
    (setq prep-op (prepare-operator (cadr op) op-id))

    (add-current-tc-to-spaces (cdr space) (list prep-op))
    
    (let ((*proposed-operator-id* op-id))
      (eval
       ;; 3-29 - ema - this might not have to be sticky, because there
       ;; are no augs of augs, but at time of writing there are bugs
       ;; with OC support, so make explicitly sticky: 
       ;; 5-21-91 - gry - if this is explicitly declared sticky, Soar doesn't
       ;; trace retractions of the rules generated.  So we'll hope the support
       ;; problem Erik mentions above is gone.
       `(taql-sp unknown ,(newsym *prod-name-prefix*)
	  ,@(when space
	      `((goal ,=goal ^ problem-space ,=problem-space)
		(problem-space ,=problem-space ^ name ,(cadr space))))
	  ,@(cdar prep-op)
	  ,@conds
	  -->
	  (goal ,=goal ^ operator ,op-id)
	  (operator ,op-id ^ control-stuff* ,c-id)
	  (control-stuff* ,c-id ^ select-once-only* not-yet)
	  ,@(cdr prep-op)))))
  t)

;; 3-28 - removed state copy declarations.

(defmacro propose-space (&body body)
  `(propose-space-aux ',body))

(defun propose-space-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'propose-space))
         (args (group-arguments (cdr body)
		   '(:space :function :when :copy :rename :new :use)
		   'construct))
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 (function (assoc :function (cdr args)))
	 (parsed-function nil)
         (copies (assoc :copy (cdr args)))
         (renames (assoc :rename (cdr args)))
         (news (assoc :new (cdr args)))
	 (use (assoc :use (cdr args)))
	 (local-error nil)
         (outer-when-conds (prepare-condition-sets
			    (cdr (assoc :when (cdr args))))))
    
    (when (or (not function) (cddr function))
      (taql-warn "Exactly one :function argument must appear.")
      (setq function '(:function (apply operator junk))))
    
    ;; because of all the interactions, set local-error t and do
    ;; something about that at the end:
    
    (when (and (not use)
	      (or (cddr space)               ; ":space" given more than once
		  (not (cadr space))         ; not given, or ":space nil"
		  (not (atom (cadr space))))) ; non-atomic arg
      (taql-warn
       "Either :use or exactly one atomic :space argument must appear: ~S"
       space)
      (setq local-error t))
    
    (when (cddr use)
      (taql-warn "At most one :use argument must appear.")
      (setq local-error t))
    
    (when (and use (or copies renames news))
      (taql-warn ":Use cannot be used with :copy, :rename, or :new.")
      (setq local-error t))
    
    (when (not (or use space))
      (taql-warn ":At least one of :use or :space must appear.")
      (setq local-error t))
    
    (when (and use space)
      (taql-warn "At most one of :use and :space can appear.")
      (setq local-error t))
    
    (when (car args)
      (taql-warn "All arguments must be values of keywords.")
      (setq local-error t))
    
    (when local-error
      (setq copies nil renames nil news nil space nil)
      (setq use (list :use (genvar 'junk))))
    
    (setq parsed-function (parse-space-function (cadr function)))
    
    (when parsed-function
      (let* ((common-conds
	      (nconc
	       (gen-space-function-conds parsed-function)
	       outer-when-conds))
	     (tc-id (makesym (get-internal-real-time) '*))
	     (map-id (genvar 'mp))
	     (init-stuff-id (genvar 'i))
	     (conds nil)
	     (space-id nil))
	
	(add-current-tc-to-spaces (cdr space) nil parsed-function)
    
	(cond
	  ;; :use superspace:
	  ((and use (eql (cadr use) 'superspace))
	   (setq conds (nconc
			`((goal ,=goal ^ object ,=object)
			  (goal ,=object ^ problem-space ,=superspace))
			common-conds))
	   (setq space-id =superspace))
	  
	  ;; :use top-space:
	  ((and use (eql (cadr use) 'top-space))
	   (setq conds (nconc
			`((goal ,=top-goal ^ object nil
			   ^ problem-space ,=top-space))
			common-conds))
	   (setq space-id =top-space))
	  
	  ;; :use <id>:
	  ((and use (variable-p (cadr use)))
	   (setq conds common-conds)
	   (setq space-id (cadr use)))
	  
	  ;; bad argument:
	  (use
	   (taql-warn
	    "Value of :use must be 'superspace', 'top-space', or a variable.")
	   (setq use (list :use (genvar 'junk)))
	   (setq conds common-conds)
	   (setq space-id (genvar 'junk)))
	  
	  ;; :use not given:
	  (t
	   (setq conds common-conds)
	   (setq space-id nil)))
	
	;; productions that put user-specified augmentations on the
	;; space (news, copies, renames):
	
	(gen-init-productions 'problem-space space conds
	    copies renames news space-id tc-id init-stuff-id)
	
	;; one production to put state-copy and init-stuff stuff on
	;; the space, if this is a new problem-space:
	
	(if (not use)
	    (let ((space-id (genvar 'p)))
	      (eval
	       `(taql-sp unknown ,(newsym *prod-name-prefix*)
		 ,@conds
		 (goal ,=goal ^ init-stuff* ,init-stuff-id)
		 (init-stuff* ,init-stuff-id ^ type problem-space)
		 -->
		 (goal ,=goal ^ problem-space ,space-id)
		 (problem-space ,space-id ^ name ,(cadr space)
		  ;; 7-13-90: These copy specs are the defaults in Soar 5.2,
		  ;; so we don't need to create them any more.
		  ;; 3-29: operator copying:
		  ;; ^ default-state-copy yes ^ default-operator-copy yes
		  ;; 4-24: this is rejected if evaluation-properties
		  ;; is given:
		  ;; ^ all-attributes-at-level one
		  )
		 (init-stuff* ,init-stuff-id ^ tc-ids ,tc-id + &
		  ;; 6-9-90 - ema - can now have multiple:
		  ^ tc-to-object-map ,map-id + &)
		 (map* ,map-id ^ tc-id ,tc-id ^ object ,space-id))))))))
  t)

(defun parse-space-function (func)
  (let ((parsed-subfunc nil))
    (cond ((and (listp func)
		(eql (first func) 'apply)
		(eql (second func) 'operator)
		(third func)
		(null (cdddr func)))
	   
	   ;; Form is "apply operator OPERATOR-COND"
	   
	   `(apply operator (operator ,(third func))))
	  
	  ((and (listp func)
		(member (first func) '(propose select))
		(member (second func) '(initial-state operator))
		(eql (third func) 'for)
		(fourth func)
		(null (cddddr func))
		(atom (fourth func)))
	   
	   ;; Form: "{propose|select} {initial-state|operator} for SPACE-NAME"
	   
	   `(,(first func) ,(if (eql (second func) 'initial-state)
			      'state
			      (second func))
	     (space ,(fourth func))))
	  
	  ((and (listp func)
		(member (first func) '(propose select))
		(eql (second func) 'space)
		(eql (third func) 'that)
		(eql (fourth func) 'will))
	   
	   ;; Form is "{propose|select} space that will FUNCTION"
	   
	   (setq parsed-subfunc (parse-space-function (cddddr func)))
	   
	   (if parsed-subfunc
	     (cons (first func)
		   (cons 'problem-space parsed-subfunc))
	     ;; ELSE just return nil, error message was printed by recursive
	     ;;      call
	     nil))
	  
	  (t
	   (taql-warn "Malformed value for :function keyword.")
	   nil))))

(defun gen-space-function-conds (func)
  (do* ((last-goal-id =goal new-goal-id)
	(new-goal-id =object (genvar 'g))
	(last-obj-type nil (cadr func))
        (func func (cddr func))
	(conds nil conds))
      ((not (cdr func))
       ;; We are at the last element, which will be either
       ;; (operator OPERATOR-COND) or (space SPACE-NAME).
       (nconc (reverse conds)
	   ;; 7-23-90: Force a test of some aspect of the
	   ;; state, so that chunks that modify the
	   ;; operator will get the right support in Soar
	   ;; 5.2.  The dummy-att* attribute is put on
	   ;; every state by taql-support.soar.
	   ;; Hopefully this is a temporary thing, and
	   ;; Soar will change its definition of operator
	   ;; modification to something more useful.
	   (if (eql last-obj-type 'operator)
	       (let ((s-id-var (cond ((eql last-goal-id =goal)
				      =state)
				     ((eql last-goal-id =object)
				      =superstate)
				     (t
				      (genvar 's)))))
		 `((goal ,last-goal-id ^ state ,s-id-var)
		   (state ,s-id-var ^ dummy-att* true)))
	       ;; ELSE
	       nil)
	   (if (eql (caar func) 'operator)
	     (let* ((op-id-var (cond ((eql last-goal-id =goal)
				      =operator)
				     ((eql last-goal-id =object)
				      =superoperator)
				     (t
				      (genvar 'o))))
		    (prep-op (prepare-operator (cadar func) op-id-var)))
	       `((goal ,last-goal-id ^ operator ,op-id-var)
		 ,@(cdar prep-op)
		 ,@(cdr prep-op)))
	     ;; ELSE space
	     (let ((space-id-var (cond ((eql last-goal-id =goal)
					=problem-space)
				       ((eql last-goal-id =object)
					=superspace)
				       (t
					(genvar 'p)))))
	       `((goal ,last-goal-id ^ problem-space ,space-id-var)
		 (problem-space ,space-id-var ^ name ,(cadar func)))))))
    
    (let* ((impasse
	    (case (car func)
	      (apply 'no-change)
	      (propose 'no-change)
	      (select 'tie)
	      (t
	       (error
		"TAQL INTERNAL ERROR: case selector ~S fell through"
		(car func)))))
	   (attribute
	    (cond ((eql impasse 'tie) (cadr func))
		  ((eql (car func) 'apply) 'operator)
		  (t ; (car func) is propose
		   (case (cadr func)
		     (operator 'state)
		     (state 'problem-space)
		     (problem-space 'goal)
		     (t
		      (error
		       "TAQL INTERNAL ERROR: case selector ~S fell through"
		       (cadr func))))))))
      (push
       `(goal ,last-goal-id ^ object ,new-goal-id
	      ^ impasse ,impasse ^ attribute ,attribute)
       conds))))

;; creates and makes acceptable a new state when any :when clause
;; given matches.  an empty state is created if:
;;   the outer :when clause is absent or satisfied, and
;;   no action clauses are given.
;; if an action clause is given but isn't satisfied, no state is
;; created because the conjunction of inner and outer whens isn't
;; satisfied.
;; 
;;   is this still right?
;;
;; GRY - 26-jan-90 - added :use keyword
;; 
;; 2-26 - ema - now, if :use is given, the conditions and the id of
;; the state to use are bound here.  this localizes all parsing of
;; the use stuff in two places, here and in propose-space.

(defmacro propose-initial-state (&body body)
  `(propose-initial-state-aux ',body))

(defun propose-initial-state-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'propose-initial-state))
         (args (group-arguments (cdr body) '(:space :when :copy :rename :new
					     :use)
		   'construct))
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
         (outer-when-conds (prepare-condition-sets (cdr (assoc :when
							    (cdr args))))) 
         (copies (assoc :copy (cdr args)))
         (renames (assoc :rename (cdr args)))
         (news (assoc :new (cdr args)))
	 (use (assoc :use (cdr args)))
	 (init-stuff-id (genvar 'i))
         (tc-id (makesym (get-internal-real-time) '*)))
    
    (when (cddr space)
      (taql-warn "At most one :space argument must appear.")
      (setq space '(:space junk)))
    
    (when (cddr use)
      (taql-warn "At most one :use argument must appear.")
      (setq use nil))
    
    (when (and use (or copies renames news))
      (taql-warn ":Use cannot be used with :copy, :rename, or :new.")
      (setq use nil))
    
    (if (car args)
        (taql-warn "All arguments must be values of keywords."))
    
    ;; 2-26:
    (let ((conds nil)
	  (state-id nil))

      (add-current-tc-to-spaces (cdr space) nil)
      
      (cond
	;; :use superstate:
	((and use (eql (cadr use) 'superstate))
	 (setq conds (nconc
		      `((goal ,=goal ^ object ,=object)
			(goal ,=object ^ state ,=superstate))
		      outer-when-conds))
	 (setq state-id =superstate))
	
	;; :use top-state:
	((and use (eql (cadr use) 'top-state))
	 (setq conds (nconc
		      `((goal ,=top-goal ^ object nil ^ state ,=top-state))
		      outer-when-conds))
	 (setq state-id =top-state))
	
	;; :use <id>:
	((and use (variable-p (cadr use)))
	 (setq conds outer-when-conds)
	 (setq state-id (cadr use)))
	
	;; bad argument:
	(use
	 (taql-warn "Value of :use must be 'superstate' or a variable.")
	 (setq use nil))
	
	;; :use not given:
	(t
	 (setq conds outer-when-conds)
	 (setq state-id nil)))
      
      (gen-init-productions 'state space conds
	  copies renames news state-id tc-id init-stuff-id))))

;; GRY - 29-jan-90 - This all used to be a part of propose-initial-state.
;; I split it out and generalized it so that it can be used by both
;; propose-initial-state and propose-space.
;;
;;   2-26 - ema - moved construction of conditions when :use is
;;   given to the callee.  use is now nil when :use is absent,
;;   otherwise use is the id of the state/space to use.  this is part
;;   of adding :use to propose-space.
;;
;; 8-2-90:  Added :from keyword in :copy/:rename.  Default is superstate.

(defun gen-init-productions (type space common-conds copies renames news
				use tc-id init-id)
  
  ;; sanity check:  use and the others are mutually exclusive, but
  ;; one or the other must be non-nil; type must be known.
  
  (when (or (and use (or copies renames news))
	    (not (or (eql type 'state) (eql type 'problem-space))))
    (error "TAQL INTERNAL ERROR: gen-init-productions"))
  
  ;; no action keywords given, :use not given either; create an empty
  ;; state if the outer
  ;; when is satisfied (the outer when is "satisfied" if it is null):
  
  (if (and (eql type 'state)
	  (not use)
	  (null copies)
	  (null renames)
	  (null news))
      (init-edit 'state nil nil nil tc-id 'augs
	  'empty-state space common-conds nil init-id nil nil))
  
  ;; 2-26:
  ;; :Use keyword is given, use the specified state/space:
  
  (when use
    (if (eql type 'state)
	(init-edit 'state nil nil nil tc-id 'augs
	    'use-existing-state space common-conds use init-id nil nil)
	;; ELSE type = problem-space
	(init-edit 'problem-space nil nil nil tc-id 'augs
	    'use-existing-space space common-conds use init-id nil nil)))
  
  ;; :use not given.  create a production for each action clause:
  
  (dolist (new1 (cdr news))
    (let* ((new (group-arguments new1 '(:when)))
	   (inner-when-conds
	    (prepare-condition-sets (cdr (assoc :when (cdr new))))))
      
      (init-edit type nil nil (car new) tc-id 'augs
	  'new-att space (append common-conds inner-when-conds) nil
	  init-id nil nil)))
  
  (dolist (copy1 (cdr copies))
    (let* ((copy (group-arguments copy1 '(:when :copy-new :from)))
	   (conds 
	    (append common-conds
		(prepare-condition-sets (cdr (assoc :when (cdr copy))))))
	   (from (assoc :from (cdr copy)))
	   (from-obj-conds nil)
	   (from-obj-spec nil)
	   (copy-new (assoc :copy-new (cdr copy))))
      
      (when (cddr from)
	(taql-warn "At most one :from argument may be specified: ~S"
	    copy1)
	(setq from nil))
      
      (when (not from)
					; Default is superstate.
	(setq from '(:from superstate)))
      
      (multiple-value-setq (from-obj-conds from-obj-spec)
	(process-object-spec (cadr from) 'superstate))
      
      ;; no values for either :copy or its :copy-new subkeyword:
      
      (when (not (or (car copy) (cadr copy-new)))
	(taql-warn
	 ":copy contains no arguments: ~S" copy1)
	(setq copy '((junk)))
	(setq copy-new nil))
      
      ;; :copy given, but no value or nil value or non-list value:
      
      (when (and (car copy) (not (consp (car copy))))
	(taql-warn
	 ":copy takes a list of attributes: ~S" copy1)
	(setq copy '((junk))))
      
      ;; :copy-new given, but no value or nil value:
      
      (when (and copy-new (null (cadr copy-new)))
	(taql-warn ":copy-new was given no arguments: ~S" copy1)
	(setq copy-new nil))
      
      ;; :copy-new given a value, but value is not a list:
      
      (when (and copy-new (not (consp (cadr copy-new))))
	(taql-warn ":copy-new takes a list of attributes: ~S" copy1)
	(setq copy-new nil))
      
      ;; copy augmentations from superstate:
      
      (if (car copy)
	  (init-edit type (car copy) (car copy) nil tc-id 'augs 'copy
	      space conds nil init-id from-obj-conds from-obj-spec))
      
      ;; duplicate augmentations from superstate:
      
      (if copy-new
	  (init-edit type (cadr copy-new) (cadr copy-new) nil
	      tc-id 'dups 'copy-new
	      space conds nil init-id from-obj-conds from-obj-spec))))
  
  (dolist (rename1 (cdr renames))
    (let* ((rename (group-arguments rename1 '(:when :copy-new :from)))
	   (conds
	    (append common-conds
		(prepare-condition-sets (cdr (assoc :when (cdr rename))))))
	   (atts (car rename))
	   (from (assoc :from (cdr rename)))
	   (from-obj-conds nil)
	   (from-obj-spec nil)
	   (copy-new (assoc :copy-new (cdr rename))))
      
      (when (cddr from)
	(taql-warn "At most one :from argument may be specified: ~S"
	    rename1)
	(setq from nil))
      
      (when (not from)
					; Default is superstate.
	(setq from '(:from superstate)))
      
      (multiple-value-setq (from-obj-conds from-obj-spec)
	(process-object-spec (cadr from) 'superstate))
      
      (when (not (or (car rename) (cadr copy-new)))
	(taql-warn
	 ":rename contains no arguments: ~S" rename1)
	(setq rename '((junk junk2)))
	(setq copy-new nil))
      
      (when (and (car rename) (not (consp (car rename))))
	(taql-warn
	 ":rename takes a list of attributes: ~S" rename)
	(setq rename '((junk junk2))))
      
      (when (and copy-new (not (cadr copy-new)))
	(taql-warn ":copy-new was given no arguments: ~S" rename1)
	(setq copy-new nil))
      
      (when (and copy-new (not (consp (cadr copy-new))))
	(taql-warn ":copy-new takes a list of attributes: ~S" rename1)
	(setq copy-new nil))
      
      ;; process from/to pairs in argument to :rename:
      
      (do ((rest-atts atts (cddr rest-atts))
	   (from-atts nil)
	   (to-atts nil))
	  ((null (cdr rest-atts))
	   (if rest-atts
	       (taql-warn
		":rename list must contain even number of attributes: ~S"
		rename1)
	       ;; ELSE, one production for each from/to pair:
	       (init-edit type from-atts to-atts nil tc-id 'augs 'rename
		   space conds nil init-id from-obj-conds from-obj-spec)))
	
	(push (car rest-atts) from-atts)
	(push (cadr rest-atts) to-atts))
      
      ;; process from/to pairs in argument to :copy-new:
      
      (if copy-new
	  (do ((rest-atts (cadr copy-new) (cddr rest-atts))
	       (from-atts nil)
	       (to-atts nil))
	      ((null (cdr rest-atts))
	       (if rest-atts
		   (taql-warn
		    ":copy-new list must contain even number of atts: ~S"
		    rename1)
		   ;; ELSE, one production for each from/to pair:
		   (init-edit type from-atts to-atts nil tc-id 'dups
		       'rename-new space conds nil init-id
		       from-obj-conds from-obj-spec)))
	    
	    (push (car rest-atts) from-atts)
	    (push (cadr rest-atts) to-atts)))))
  t)

;; generate one production for each augmentation, so that the
;; construct can apply partially. 
;; 
;; to prevent productions from reinstantiating later,
;; init-creation-enabled << problem-space state >> is tested.  it is
;; rejected in taql-support, once a space (or state) is actually selected.
;; this is mainly for efficiency, but also for safety, if intermediate
;; results are produced.
;;
;; these productions must be op-apps, because they retract when the
;; enable flag goes away, and they might create substructure.
;;
;; GRY - 29-jan-90 - This function was originally just for states. I
;; generalized it to handle space initialization as well.  Not all of the
;; arguments to this function are used in both the space and state cases.
;; The new "type" argument must be either Problem-space or State.
;;
;; ema, 2-26 - dropped the assumption that use <> NIL -> type = state.
;; when use is non-nil, type is used directly as an attribute, of
;; which use is the value.
;;
;; 4-9 - ema - added test for (- ^ evaluate-state-op*), which is the
;; signal that this is an evaluation subgoal.
;;
;; 8-2-90:  Added from-obj-conds and from-obj-spec parameters as part
;;   of implementation of :from keyword in :copy/:rename.

(defun init-edit (type from-specs to-specs action-spec tc-id aug-type
		     descriptive-part space when-conds use init-id
		     from-obj-conds from-obj-spec)
  (let* ((aug-id (genvar 'a))
	 (name-base (makesym *prod-name-prefix* descriptive-part
			*num-separator*))
	 (space-cond
	  (if (and (eql type 'state) space)
	      `((goal ,=goal ^ problem-space ,=problem-space)
		(problem-space ,=problem-space ^ name ,(cadr space)))))
	 (common-conds
	  `((goal ,=goal ^ init-creation-enabled* ,type)
	    (goal ,=goal ^ init-stuff* ,init-id)
	    ;; 4-9 - ema 
	    (goal ,=goal - ^ evaluate-state-op*)
	    (init-stuff* ,init-id ^ type ,type)
	    ,@when-conds))
	 (common-act
	  `((init-stuff* ,init-id ^ ,aug-type ,aug-id + &)
	    ,@(if (eql type 'state)
		  ;; For the Space case, this action is just done once when
		  ;; the problem space is created, and we don't need to do
		  ;; it again here.
		  `((init-stuff* ,init-id ^ tc-ids ,tc-id + &)))
	    (aug* ,aug-id ^ tc-id* ,tc-id))))
    
    (cond
      
      ;; one production for each (from,to) pair:
      
      ((and from-specs to-specs (not action-spec))
       (do ((from-spec-rest from-specs (cdr from-spec-rest))
	    (to-spec-rest to-specs (cdr to-spec-rest))
	    (prods nil))
	   ((or (null from-spec-rest) (null to-spec-rest))
	    (if (or from-spec-rest to-spec-rest)
		(error "INTERNAL TAQL ERROR, init-edit, do"))
	    (mapc #'eval prods))
	 
	 (let* ((from-spec (car from-spec-rest))
		(to-spec (car to-spec-rest))
		(from-obj-class (car from-obj-spec))
		(from-obj-id (cadr from-obj-spec))
		(val-var (genvar2 from-spec)))
	   
	   (push `(taql-sp sticky ,(newsym name-base)
		   ,@common-conds
		   ,@from-obj-conds
		   ,@space-cond
		   (,from-obj-class ,from-obj-id ^ ,from-spec ,val-var)
		   -->
		   ,@common-act
		   (aug* ,aug-id ^ ,to-spec ,val-var + &))
	       prods))))
      
      ;; only get one action-spec at a time as a parameter:
      
      ((and action-spec (not from-specs) (not to-specs))
       (eval 
	`(taql-sp sticky ,(newsym name-base)
	  ,@common-conds
	  ,@space-cond
	  -->
	  ,@common-act
	  ,@(action-spec-actions 'aug* aug-id action-spec))))
      
      ;; 2-26:
      ;; this is the case where a :use keyword was given to indicate
      ;; some pre-existing object.  "type" must be "problem-space" or
      ;; "state".  
      
      (use
       (eval 
	`(taql-sp sticky ,(newsym name-base)
	  ,@common-conds
	  ,@space-cond
	  -->
	  (goal ,=goal ^ ,type ,use))))
      
      ;; this is the case where there are no initial augmentations for
      ;; the state, and no :use was given, but the outer :when condition
      ;; matched.  produces a new empty state.  note that an empty
      ;; space is produced by the one production generated directly
      ;; by propose-space. 
      
      ((eql type 'state)
       (eval 
	`(taql-sp sticky ,(newsym name-base)
	  ,@common-conds
	  ,@space-cond
	  -->
	  ,@common-act)))))
  t)

;; 3-23 - ema - modified to create the control-stuff* augmentation for
;; operators.  all TAQL operators must be born with
;; this augmentation when the operator is proposed, because
;; taql-support assumes it will be there.  if an
;; existing operator is being proposed (with :op-id), then it inherits
;; whatever properties are controlled by the user, like
;; :select-once-only.  so control keywords like that can only be given
;; when the operator is first proposed.
;;
;;   8-16-90 - ema - changed to allow :op-id and :select-once-only,
;;   so that :s-o-o applies equally to ops built up by problem-solving.
;;   we already took care of this problem for ops proposed by p-su, by
;;   generating ^control-stuff* whenever there isn't one.

(defmacro propose-operator (&body body)
  `(propose-operator-aux ',body))

(defun propose-operator-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'propose-operator))
         (args (group-arguments
		(cdr body) '(:space :op :when :op-id
			     :select-once-only) 'construct))
	 (select-once (assoc :select-once-only (cdr args)))
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
         (ops (assoc :op (cdr args)))
         (op-id (assoc :op-id (cdr args)))
	 (op-id-var nil)
	 (prep-op-list nil)
	 (c-id (genvar 'c))
         (outer-whens (prepare-condition-sets (cdr (assoc :when (cdr
								 args)))))) 

    ;; We have to do a more complicated test here because if
    ;; :select-once-only with no arguments, select-once will be
    ;; (:select-once-only nil).
    ;;
    (when (and select-once
	       (or (cadr select-once)
		   (cddr select-once)))
      (taql-warn ":select-once-only takes no arguments: ~S"
	  select-once)
      (setq select-once nil))
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))
    
    (when (or (and ops op-id)           ; both :op and :op-id given
	      (and (not ops) (not op-id))   ; neither given
	      (and op-id (cddr op-id))) ; :op-id given more than once
      (taql-warn "One of :op-id or {:op ... }+ must appear.")
      (setq op-id nil)
      (setq ops '(:op junk)))
    
    (when (and op-id (not (variable-p (cadr op-id))))
      (taql-warn "Argument to :op-id must be a variable: ~S" op-id)
      (setq op-id (list :op-id (genvar 'junk))))
    
    (if (car args)
	(taql-warn "All arguments must be values of keywords."))
    
    (cond
      (op-id
       ;; use the id given to :op-id:
       (setq op-id-var (cadr op-id))
       ;; get prepare-operator to return nil:
       (setq ops '(:op (:no-name))))
      (t
       (setq op-id-var (genvar 'o))))
    
    (dolist (op-spec (cdr ops))
      (let* ((prep-op (prepare-operator op-spec op-id-var))
	     ;; *proposed-operator-id*, when bound to the identifier variable
	     ;; of an operator object, causes the data model code to check
	     ;; that all of that operator's required attributes were in the
	     ;; production.  So we don't want to bind this to the identifier
	     ;; if it came from :op-id.
	     ;;
	     (*proposed-operator-id* (if op-id
				       nil
				       ;; ELSE
				       op-id-var)))

	(push prep-op prep-op-list)

	(eval
	 ;; 3-29: sticky to ensure support for augs of augs.  this
	 ;; isn't working properly in Soar.
	 ;; 5-21-91 - gry - if this is explicitly declared sticky, Soar doesn't
	 ;; trace retractions of the rules generated.  So we'll hope the
	 ;; support problem Erik mentions above is gone.
	 `(taql-sp unknown ,(newsym *prod-name-prefix*)
	   (goal ,=goal ^ problem-space ,=problem-space ^ state ,=state)
	   ,@(when space
	       (list `(problem-space ,=problem-space ^ name
				     ,(cadr space))))
	   ,@(cdar prep-op)
	   ,@outer-whens
	   -->
	   (goal ,=goal ^ operator ,op-id-var)
	   
	   ;; 8-16-90:  indifferent, so the previous control-stuff*
	   ;; sticks and an instance keeps its control properties:
	   ,@(if select-once
		 `((operator ,op-id-var ^ control-stuff* ,c-id + =)
		   (control-stuff* ,c-id ^ select-once-only* not-yet)))
	   
	   ;; if :op is given but empty, (cdr prep-op) is nil, so only the 
	   ;; op's id is generated:
	   ,@(cdr prep-op)))))

    (add-current-tc-to-spaces (cdr space) prep-op-list))

  t)

;; maps the names of soar 5 preferences onto their symbols.
;; all binary preferences are represented except parallel, which
;; doesn't apply to context objects.
;;
;; GRY - 29-jan-90 - Extended compare to all object types (problem space,
;; state, operator).  Also, the preferences created can now be made in
;; a different goal than the conditions match.  This supports, for example,
;; resolving tie impasses from a subgoal without using the selection space.

(defmacro compare (&body body)
  `(compare-aux ',body))

(defun compare-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'compare))
         (args (group-arguments (cdr body)
		   '(:space :op1 :op2 :relation :when :for-goal :object-type
		     :object1 :object2)
		   'construct)) 
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
         (op1 (assoc :op1 (cdr args)))
         (prep-op1 nil)
         (op1-id (genvar 'o))
         (op2 (assoc :op2 (cdr args)))
         (prep-op2 nil)
         (op2-id (genvar 'o))
	 (object1 (assoc :object1 (cdr args)))
	 (object2 (assoc :object2 (cdr args)))
	 (object-type (assoc :object-type (cdr args)))
	 (for-goal (assoc :for-goal (cdr args)))
	 (for-goal-id nil)
         (relation (assoc :relation (cdr args)))
	 (binary-prefs '((better >) (worse <) (indifferent =)))
	 (relation-symbol nil)
         (conds (prepare-condition-sets (cdr (assoc :when (cdr args))))))
    
    (when (cddr for-goal)
      (taql-warn "At most one :for-goal argument may appear.")
      (setq for-goal nil))
    
    (when (and for-goal
	      (not (eql (cadr for-goal) 'supergoal))
	      (not (variable-p (cadr for-goal))))
      (taql-warn "The value of :for-goal must be 'supergoal' or a variable.")
      (setq for-goal nil))
    
    (cond ((null for-goal)
	   (setq for-goal-id =goal))
	((eql (cadr for-goal) 'supergoal)
	 (setq for-goal-id =object))
	(t ; an identifier variable
	 (setq for-goal-id (cadr for-goal))))
    
    (when (cddr object-type)
      (taql-warn "At most one :object-type argument may appear.")
      (setq object-type '(:object-type operator)))
    
    (when (and object-type (not (member (cadr object-type)
				    '(problem-space state operator))))
      (taql-warn "The value of :object-type must be one of 'problem-space', 'state', 'operator'.")
      (setq object-type '(:object-type operator)))
    
    (when (or (and (not op1) (not object1))
	      (and op1 object1))
      (taql-warn "Exactly one of :op1 and :object1 must appear.")
      (setq op1 '(:op1 junk))
      (setq object-type '(:object-type operator))
      (setq object1 nil))
    
    (when (or (and (not op2) (not object2))
	      (and op2 object2))
      (taql-warn "Exactly one of :op2 and :object2 must appear.")
      (setq op2 '(:op2 junk))
      (setq object-type '(:object-type operator))
      (setq object2 nil))
    
    (when (cddr op1)
      (taql-warn "At most one :op1 argument may appear.")
      (setq op1 '(:op1 junk)))
    
    (when (cddr op2)
      (taql-warn "At most one :op2 argument may appear.")
      (setq op2 '(:op2 junk)))
    
    (when (and op1 object-type (not (eql (cadr object-type) 'operator)))
      (taql-warn "When :op1 and :object-type are both given, :object-type must be 'operator'.")
      (setq object-type '(:object-type operator)))
    
    (when (and op2 object-type (not (eql (cadr object-type) 'operator)))
      (taql-warn "When :op2 and :object-type are both given, :object-type must be 'operator'.")
      (setq object-type '(:object-type operator)))
    
    (when (and (not op1) (not op2) (not object-type))
      (taql-warn ":Object-type must be specified when neither :op1 nor :op2 is specified.")
      (setq object-type '(:object-type operator)))
    
    (when (and (or op1 op2) (not object-type))
      (setq object-type '(:object-type operator)))
    
    (when (cddr object1)
      (taql-warn "At most one :object1 argument may appear.")
      (setq object1 (list :object1 (genvar 'junk))))
    
    (when (cddr object2)
      (taql-warn "At most one :object2 argument may appear.")
      (setq object2 (list :object2 (genvar 'junk))))
    
    (when (and object1 (not (variable-p (cadr object1))))
      (taql-warn "The value of :object1 must be a variable.")
      (setq object1 (list :object1 (genvar 'junk))))
    
    (when (and object2 (not (variable-p (cadr object2))))
      (taql-warn "The value of :object2 must be a variable.")
      (setq object2 (list :object2 (genvar 'junk))))
    
    (when op1
      (setq prep-op1 (prepare-operator (cadr op1) op1-id)))
    
    (when op2
      (setq prep-op2 (prepare-operator (cadr op2) op2-id)))
    
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))
    
    (when (or (not relation) (cddr relation))
      (taql-warn "Exactly one :relation argument must appear.")
      (setq relation '(:relation indifferent)))
    
    (when (not (member (cadr relation)
		   (mapcar #'car binary-prefs)))
      (taql-warn
       "Invalid :relation specification:  ~S" (cadr relation))
      (setq relation '(:relation indifferent)))
    
    (setq relation-symbol (assoc (cadr relation) binary-prefs))
    
    (if (car args)
	(taql-warn "All arguments must be values of keywords."))
    
    (add-current-tc-to-spaces (cdr space) nil)

    (let ((prod-name (newsym
		      *prod-name-prefix*
		      (makesym (case (cadr object-type)
				 (problem-space 'space)
				 (t (cadr object-type)))
			  '-pref *num-separator*)))
	  (obj1 (if op1 op1-id (cadr object1)))
	  (obj2 (if op2 op2-id (cadr object2))))
      
      (eval
       `(taql-sp unknown ,prod-name
	 ,@(cond ((null for-goal)
		  ;; Relative to =goal, no additional conds needed
		  nil
		  )
	       ((eql (cadr for-goal) 'supergoal)
		`((goal ,=goal ^ object ,=object)))
	       (t
		;; An identifier variable, no additional conds needed
		;; because we assume it is already bound in the
		;; user-specified conditions.
		nil
		))
	 ,@(if space
	       `((goal ,=goal ^ problem-space ,=problem-space)
		 (problem-space ,=problem-space ^ name ,(cadr space))))
	 (goal ,for-goal-id ^ ,(cadr object-type) ,obj1 +
	  ^ ,(cadr object-type) { <> ,obj1 ,obj2 } +)
	 ,@(if op1 (cdar prep-op1))
	 ,@(if op1 (cdr prep-op1))
	 ,@(if op2 (cdar prep-op2))
	 ,@(if op2 (cdr prep-op2))
	 ,@conds
	 -->
	 (goal ,for-goal-id ^ ,(cadr object-type) ,obj1
	  ,(cadr relation-symbol) ,obj2)))))
  t)


;; as above, values of the :value keyword are now mapped onto the
;; corresponding soar 5 symbols.  all unary preferences are
;; represented except reconsider (which is generated at the right
;; time by the compiler) and parallel, which doesn't apply to context
;; objects.
;;
;; GRY - 29-jan-90 - Extended prefer to all object types (problem space,
;; state, operator).  Also, the preferences created can now be made in
;; a different goal than the conditions match.  This supports, for example,
;; rejecting the superoperator, and resolving tie impasses from a subgoal
;; without using the selection space.  Finally, reconsider is now supported,
;; mainly so that problem spaces can be reconsidered.  
;; 
;; GRY - 30-jan-90 - Extended to allow multiple :value keywords.

(defmacro prefer (&body body)
  `(prefer-aux ',body))

(defun prefer-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'prefer))
	 (args (group-arguments (cdr body)
		   '(:space :op :value :when :object :object-type :for-goal)
		   'construct)) 
	 (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 (op (assoc :op (cdr args)))
	 (prep-op nil)
	 (op-id (genvar 'o))
	 (values (assoc :value (cdr args)))
	 (value-symbols nil)
	 (object (assoc :object (cdr args)))
	 (object-type (assoc :object-type (cdr args)))
	 (for-goal (assoc :for-goal (cdr args)))
	 (for-goal-id nil)
	 (unary-prefs '((best >) (worst <) (indifferent =) (reconsider @)
			(reject -) (require !) (prohibit ~)))
	 (conds (prepare-condition-sets (cdr (assoc :when (cdr args))))))
    
    (when (cddr for-goal)
      (taql-warn "At most one :for-goal argument may appear.")
      (setq for-goal nil))
    
    (when (and for-goal
	      (not (eql (cadr for-goal) 'supergoal))
	      (not (variable-p (cadr for-goal))))
      (taql-warn "The value of :for-goal must be 'supergoal' or a variable.")
      (setq for-goal nil))
    
    (cond ((null for-goal)
	   (setq for-goal-id =goal))
	((eql (cadr for-goal) 'supergoal)
	 (setq for-goal-id =object))
	(t ; an identifier variable
	 (setq for-goal-id (cadr for-goal))))
    
    (when (cddr object-type)
      (taql-warn "At most one :object-type argument may appear.")
      (setq object-type '(:object-type operator)))
    
    (when (and object-type (not (member (cadr object-type)
				    '(problem-space state operator))))
      (taql-warn "The value of :object-type must be one of 'problem-space', 'state', 'operator'.")
      (setq object-type '(:object-type operator)))
    
    (when (or (and (not op) (not object))
	      (and op object))
      (taql-warn "Exactly one of :op and :object must appear.")
      (setq op '(:op junk))
      (setq object-type '(:object-type operator))
      (setq object nil))
    
    (when (cddr op)
      (taql-warn "At most one :op argument may appear.")
      (setq op '(:op junk)))
    
    (when (and op object-type (not (eql (cadr object-type) 'operator)))
      (taql-warn "When :op and :object-type are both given, :object-type must be 'operator'.")
      (setq object-type '(:object-type operator)))
    
    (when (and (not op) (not object-type))
      (taql-warn ":Object-type must be specified when :op is not specified.")
      (setq object-type '(:object-type operator)))
    
    (when (and op (not object-type))
      (setq object-type '(:object-type operator)))
    
    (when (cddr object)
      (taql-warn "At most one :object argument may appear.")
      (setq object (list :object (genvar 'junk))))
    
    (when (and object (not (variable-p (cadr object))))
      (taql-warn "The value of :object must be a variable.")
      (setq object (list :object (genvar 'junk))))
    
    (when op
      (setq prep-op (prepare-operator (cadr op) op-id)))
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))
    
    (when (not values)
      (taql-warn "At least one :value argument must appear.")
      (setq values '(:value reject)))
    
    (let ((found-bad-one nil))
      (dolist (value (cdr values))
	(when (not (member value
		       (mapcar #'car unary-prefs)))
	  (taql-warn "Invalid :value: ~S " value)
	  (setq found-bad-one t)))
      (if found-bad-one
	  (setq values '(:value reject))))
    
    (setq value-symbols
	(mapcar #'(lambda (value)
		    (cadr (assoc value unary-prefs)))
	    (cdr values)))
    
    (if (car args)
	(taql-warn "All arguments must be values of keywords."))

    (add-current-tc-to-spaces (cdr space) nil)
    
    (let ((prod-name (newsym
		      *prod-name-prefix*
		      (makesym (case (cadr object-type)
				 (problem-space 'space)
				 (t (cadr object-type)))
			  '-pref *num-separator*))))
      (eval
       `(taql-sp unknown ,prod-name
	 ,@(cond ((null for-goal)
		  ;; Relative to =goal, no additional conds needed
		  nil)
	       ((eql (cadr for-goal) 'supergoal)
		`((goal ,=goal ^ object ,=object)))
	       (t
		;; An identifier variable, no additional conds needed
		;; because we assume it is already bound in the
		;; user-specified conditions.
		nil))
	 ,@(if space
	       `((goal ,=goal ^ problem-space ,=problem-space)
		 (problem-space ,=problem-space ^ name ,(cadr space))))
	 (goal ,for-goal-id ^ ,(cadr object-type)
	  ,(if op op-id (cadr object))
	  ;; 6-12-90 - ema - if @ is a value, assume the object is
	  ;; selected (can't count on acceptable)
	  ,@(if (not (member 'reconsider values))
		'(+)))
	 ,@(if op (cdar prep-op))
	 ,@(if op (cdr prep-op))
	 ,@conds
	 -->
	 (goal ,for-goal-id
	  ,@(mapcan #'(lambda (value-sym)
			(list '^ (cadr object-type)
			    (if op op-id (cadr object))
			    value-sym))
	     value-symbols))))))
  t)

;; goal tests as follows:  for each test clause, a production is
;; built that contains the conditions of the test, and elaborates the
;; goal with ^test type, where att is the name of the test, and val is
;; the type of the group (created anew, if not given).  if two
;; productions elaborate the same attribute, that's a disjunctive
;; test.  one other producion is generated, which tests for all ^test type
;; pairs on the goal, and elaborates the state with  ^final-state* <finfo>.
;; <finfo> is an object:
;;     (final-state-info <finfo> ^goal-id <gid>
;;          ^final-state-type* <type>
;;          ^goal-group-name* <name>)
;; If the state has a ^final-state* object whose ^goal-id matches the
;; ^goal-id* attribute of a goal the state is in, then the final state
;; has been reached in that goal.  ^final-state-type* and ^goal-group-name*
;; are the group name and type, if these are given in the TC (the latter two
;; augs communicate with result-superstate).
;;
;;   3-11-91 - gry - Move the goal-test attributes that used to be hung
;;     directly on the goal into the new (goal ^taql-stuff*) object.
;;
;;   28-jan-90 - GRY - made final state info specific to the goal the state
;;     is in, as the same state could appear in several different goals.
;;     Generalized so that no :test keyword need appear.  In this case,
;;     the group is satisfied if the :space and :when conds are satisfied.
;;
;;   1-12 - ^test is now newsymmed, so that different TCs can't
;;   satisfy each other's tests.

(defmacro goal-test-group (&body body)
  `(goal-test-group-aux ',body))

(defun goal-test-group-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'goal-test-group))
	 (tc-instance (car body))
	 (args (group-arguments (cdr body)
		   '(:space :when :test :group-type :group-name) 'construct))
	 (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 (outer-when-conds (prepare-condition-sets (cdr (assoc :when
							    (cdr args)))))
	 (tests (assoc :test (cdr args)))
	 (testname-atts nil)
	 (type (assoc :group-type (cdr args)))
	 (group-name (assoc :group-name (cdr args)))
	 (group-id nil)
	 (final-state-obj-id (genvar 'f))
	 (taql-stuff-id (genvar 't))
	 (goal-id (genvar 'g))
	 (prods nil))
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))  
    
    (when (cddr type)
      (taql-warn "At most one :group-type argument may appear.")
      (setq type nil))
    
    (when (and type (not (member (cadr type) '(success failure))))
      (taql-warn "Value of :group-type must be one of success, failure.")
      (setq type nil))
    
    (when (cddr group-name)
      (taql-warn "At most one :group-name argument may appear.")
      (setq group-name nil))
    
    (if (car args)
	(taql-warn "All arguments must be values of keywords."))
    
    ;; if no type is supplied, make up a unique one:
    
    (setq group-id (if group-name
		       (cadr group-name)
		       (makesym (get-internal-real-time) '*)))

    (add-current-tc-to-spaces (cdr space) nil)
    
    (dolist (test (cdr tests))
      (let* ((prep-test (group-arguments test '(:when)))
	     (conds (prepare-condition-sets
		     (cdr (assoc :when (cdr prep-test))))))
	
	(when (null conds)
	  (taql-warn "At least one :when argument must be specified: ~S" test)
	  (setq conds '((goal <g> ^ junky junk))))
	
	(when (or (null (car prep-test)) (cdar prep-test))
	  (taql-warn "Exactly one test name must be specified per test: ~S"
	      test)
	  (setq prep-test (cons '(junk) (cdr prep-test))))
	
	;; include TC-NAME in the test name, to make the test name
	;; local to a TC:
	(push (makesym tc-instance '* (caar prep-test) '*) testname-atts)
	
	;; these are elaboration productions, so they should not be
	;; sticky:
	;; 5-21-91 - gry - changed pclass from not-sticky to unknown, since
	;; Soar should always classify these right anyways.
	
	(push
	 `(taql-sp unknown ,(newsym *prod-name-prefix*
				'satisfies- (caar prep-test) *num-separator*)
	   (goal ,=goal ^ taql-stuff* ,taql-stuff-id)
	   ,@(if space
		 `((goal ,=goal ^ problem-space ,=problem-space)
		   (problem-space ,=problem-space ^ name ,(cadr space))))
	   ,@outer-when-conds
	   ,@conds
	   -->
	   ;; Make it a parallel attribute because the same
	   ;; test name could appear in different groups, with
	   ;; more than one of them being satisfied at once.
	   (taql-stuff* ,taql-stuff-id ^ ,(car testname-atts) ,group-id + &))
	 prods)))
    
    (mapc #'eval
	(cons
	 ;; 5-21-91 - gry - changed pclass from not-sticky to unknown, since
	 ;; Soar should always classify these right anyways.
	 `(taql-sp unknown ,(newsym *prod-name-prefix* 'goal-done
				*num-separator*)
	   (goal ,=goal ^ state ,=state)
	   (goal ,=goal ^ goal-id* ,goal-id)
	   
	   ;; 3-25: get the final-state-object from the goal, so
	   ;; that if multiple tests or TCs apply, they all create
	   ;; preferences for the same object:
	   (goal ,=goal ^ final-state-object* ,final-state-obj-id)
	   
	   ,@(if testname-atts
		 `((goal ,=goal ^ taql-stuff* ,taql-stuff-id)
		   (taql-stuff* ,taql-stuff-id
		    ,@(mapcan #'(lambda (testname-att)
				  (list '^ testname-att group-id))
		       testname-atts)))
		 ;; ELSE no :test clauses, so apply the test to the state
		 ;; itself, rather than the conjunction of explicit tests.
		 `(,@(if space
			 `((goal ,=goal ^ problem-space ,=problem-space)
			   (problem-space ,=problem-space
			    ^ name ,(cadr space))))
		   ,@outer-when-conds))
	   -->
	   ;; 4-9:  multiple fs-objects can occur if a state is a
	   ;; final state for different goals.
	   (state ,=state ^ final-state* ,final-state-obj-id + &)
	   (final-state-info ,final-state-obj-id ^ goal-id ,goal-id
	    ,@(if group-name
		  (list '^ 'test-group-name* group-id '+ '&))
	    
	    ;; 3-25: if multiple types, deal with it at run-time; may
	    ;; be intentional?
	    ,@(if type
		  (list '^ 'final-state-type* (cadr type) '+ '&))))
	 prods)))
  t)

;; in 3.0, augments are retractable.  should this be explicit
;; in the syntax?
;;
;;   this becomes "elaborate" as soon as soar is packaged properly,
;;   to avert the conflict in the name space.
;;
;;   the actions should be given in a keyword clause, for uniformity,
;;   and it would be nice replace FIRST-ACTION with some other
;;   syntax, because this is the only place it's a terminal.
;;
;; 8-6-90: Augment made more consistent and flexible.  See the TAQL 3.1.3
;;   release notes.
;;
;; 6-12-91 - gry - Changed to support the :actions keyword

(defmacro augment (&body body)
  `(augment-aux ',body))

(defun augment-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'augment))
	 (args (group-arguments (cdr body) '(:when :space :what :actions
					     :new :copy :rename) 'construct))
	 (conds (prepare-condition-sets (cdr (assoc :when (cdr args)))))
	 (space (assoc :space (cdr args)))
	 (what (assoc :what (cdr args)))
	 (news (assoc :new (cdr args)))
	 (copies (assoc :copy (cdr args)))
	 (renames (assoc :rename (cdr args)))
	 (actions (assoc :actions (cdr args)))
	 (augment-what nil)
	 (obj-spec nil)
	 (common-conds nil)
	 (common-action-conds nil)
	 (*prod-name-prefix* (build-prefix (cadr space))))
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))
    
    (when (cddr what)
      (taql-warn "At most one :what argument may appear.")
      (setq what '(:what state)))
    
    (when (and (not what) (or news copies renames))
      (taql-warn "A :what keyword must appear when :new, :copy, or :rename is given.")
      (setq what '(:what state)))
    
    (when (not (or news copies renames actions))
      (taql-warn "At least one of :new, :copy, :rename, :actions must appear."))
    
    (if (car args)
        (taql-warn "All arguments must be values of keywords."))
    
    (add-current-tc-to-spaces (cdr space) nil)

    (when what
      (multiple-value-setq (augment-what obj-spec)
	  (process-object-spec (cadr what) 'state)))
    
    (setq common-action-conds
	(append (if space
		    `((goal ,=goal ^ problem-space ,=problem-space)
		      (problem-space ,=problem-space ^ name ,(cadr space)))
		    ;; ELSE
		    nil)
	    conds))

    (setq common-conds (append common-action-conds augment-what))
    
    (dolist (action1 (cdr actions))
      (let* ((action (group-arguments action1 '(:when)))
	     (inner-when-conds
	      (prepare-condition-sets (cdr (assoc :when (cdr action))))))
	
	(gen-augment-prod nil (append inner-when-conds common-action-conds)
	    'soar-actions nil nil nil nil (car action))))
    
    (dolist (new1 (cdr news))
      (let* ((new (group-arguments new1 '(:when)))
	     (inner-when-conds
	      (prepare-condition-sets (cdr (assoc :when (cdr new))))))
	
	(gen-augment-prod obj-spec (append inner-when-conds common-conds)
	    'new nil nil (car new) nil nil)))
    
    (dolist (copy1 (cdr copies))
      (let* ((copy (group-arguments copy1 '(:when :from)))
	     (inner-when-conds
	      (prepare-condition-sets (cdr (assoc :when (cdr copy)))))
	     (from (assoc :from (cdr copy)))
	     (from-obj-conds nil)
	     (from-obj-spec nil))
	
	(when (cddr from)
	  (taql-warn "At most one :from argument may be specified: ~S"
	      copy1)
	  (setq from nil))
	
	(when (not from)
	  ;; Default is state.
	  (setq from '(:from state)))
	
	(multiple-value-setq (from-obj-conds from-obj-spec)
	  (process-object-spec (cadr from) 'state))
	
	;; no values for either :copy attrbutes
	
	(when (null (car copy))
	  (taql-warn
	   ":copy contains no attribute names: ~S" copy1)
	  (setq copy '((junk))))
	
	;; :copy given, but no value or nil value or non-list value:
	
	(when (and (car copy) (not (consp (car copy))))
	  (taql-warn
	   ":copy takes a list of attributes: ~S" copy1)
	  (setq copy '((junk))))
	
	;; copy augmentations from state:
	
	(if (car copy)
	    (gen-augment-prod
	     obj-spec
	     (append from-obj-conds inner-when-conds common-conds)
	     'copy (car copy) (car copy) nil from-obj-spec nil))))
    
    (dolist (rename1 (cdr renames))
      (let* ((rename (group-arguments rename1 '(:when :from)))
	     (inner-when-conds
	      (prepare-condition-sets (cdr (assoc :when (cdr rename)))))
	     (atts (car rename))
	     (from (assoc :from (cdr rename)))
	     (from-obj-conds nil)
	     (from-obj-spec nil))
	
	(when (cddr from)
	  (taql-warn "At most one :from argument may be specified: ~S"
	      rename1)
	  (setq from nil))
	
	(when (not from)
	  ;; Default is superstate.
	  (setq from '(:from state)))
	
	(multiple-value-setq (from-obj-conds from-obj-spec)
	  (process-object-spec (cadr from) 'state))
	
	(when (null (car rename))
	  (taql-warn
	   ":rename contains no attribute names: ~S" rename1)
	  (setq rename '((junk junk2))))
	
	(when (and (car rename) (not (consp (car rename))))
	  (taql-warn
	   ":rename takes a list of attributes: ~S" rename)
	  (setq rename '((junk junk2))))
	
	;; process from/to pairs in argument to :rename:
	
	(do ((rest-atts atts (cddr rest-atts))
	     (from-atts nil)
	     (to-atts nil))
	    ((null (cdr rest-atts))
	     (if rest-atts
		 (taql-warn
		  ":rename list must contain even number of attributes: ~S"
		  rename1)
		 ;; ELSE, one production for each from/to pair:
		 (gen-augment-prod
		  obj-spec
		  (append from-obj-conds inner-when-conds common-conds)
		  'rename from-atts to-atts nil from-obj-spec nil)))
	  
	  (push (car rest-atts) from-atts)
	  (push (cadr rest-atts) to-atts)))))
  
  t)

;;; 6-12-91 - gry - Changed to support the :actions keyword
;;;
(defun gen-augment-prod (obj-spec common-conds descriptive-part
			    from-specs to-specs action-spec from-obj-spec
			    soar-actions)
  
  (let* ((name-base (makesym *prod-name-prefix* descriptive-part
			*num-separator*))
	 (obj-class (car obj-spec))
	 (obj-id (cadr obj-spec)))
    
    (when from-specs   ; from/to-atts (copy/rename)
      
      (do ((from-spec-rest from-specs (cdr from-spec-rest))
	   (to-spec-rest to-specs (cdr to-spec-rest))
	   (prods nil))
	  ((or (null from-spec-rest) (null to-spec-rest))
	   (if (or from-spec-rest to-spec-rest)
	       (error "INTERNAL TAQL ERROR, from/to-atts length mismatch"))
	   (mapc #'eval prods))
	
	(let* ((from-spec (car from-spec-rest))
	       (to-spec (car to-spec-rest))
	       (from-obj-class (car from-obj-spec))
	       (from-obj-id (cadr from-obj-spec))
	       (val-var (genvar2 from-spec)))
	  
	  (push `(taql-sp not-sticky ,(newsym name-base)
		  ,@common-conds
		  (,from-obj-class ,from-obj-id ^ ,from-spec ,val-var)
		  -->
		  (,obj-class ,obj-id ^ ,to-spec ,val-var + &))
	      prods))))
    
    (when soar-actions     ; :actions
      (multiple-value-bind (linkage-conds modified-actions)
	  (prepare-actions soar-actions)
	(eval
	 `(taql-sp not-sticky ,(newsym name-base)
	    ,@common-conds
	    ,@linkage-conds
	    -->
	    ,@modified-actions))))

    (when action-spec     ; :new
      
      (eval 
       `(taql-sp not-sticky ,(newsym name-base)
	 ,@common-conds
	 -->
	 ,@(action-spec-actions obj-class obj-id action-spec))))))

;; 2-12 - added "top-state" as a value for ":what".  added support for
;; ":terminate-when". 
;;
;; 8-2-90: apply-operator-aux now takes a flag telling it whether to
;;   generate code for apply-operator or result-superstate.

(defmacro apply-operator (&body body)
  `(apply-operator-aux ',body nil))

;; 8-mar-90 - gry - Changed so that the generated productions are part of
;; the implementation of the TAQL final-state operator.
;;
;; 8-2-90: Merged with apply-operator-aux.
;;
(defmacro result-superstate (&body body)
  `(apply-operator-aux ',body t))

;; 8-2-90: apply-operator-aux now takes a flag telling it whether to
;;   generate code for apply-operator or result-superstate.  Changed code
;;   so that it can handle either TC.  Also added support for the :bind
;;   keyword.
;;
;; 9-5-90: added support for the :directive keyword
;;
;; 6-12-91 - gry - Added support for :actions and :sliding-actions keywords.

(defun apply-operator-aux (body result-superstate-flag)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body)
					(if result-superstate-flag
					  'result-superstate
					  'apply-operator)))
	 (args (group-arguments (cdr body)
		(if result-superstate-flag
		  '(:space :when :terminate-when :group-name :group-type
			   :use :bind :actions :sliding-actions)
		  ;; ELSE
		  '(:op :space :when :terminate-when :use :bind
			:actions :sliding-actions))
		'construct))
	 (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 (op (if result-superstate-flag
	       '(:op (final-state ^ type* taql-op))
	       ;; ELSE
	       (assoc :op (cdr args))))
	 (group-name (assoc :group-name (cdr args)))
	 (group-type (assoc :group-type (cdr args)))
	 (actions (assoc :actions (cdr args)))
	 (sliding-actions (assoc :sliding-actions (cdr args)))
	 (use (assoc :use (cdr args)))
	 (bind (assoc :bind (cdr args)))
	 (edits nil)
	 (terminate-when (assoc :terminate-when (cdr args)))
	 (terminate-when-conds-p nil) ; reconsider if :tw conds
	 (reconsider-edit-p nil) ; reconsider from edit clause
	 (outer-when-conds
	  (prepare-condition-sets (cdr (assoc :when (cdr args))))))

    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))
	
    (when (or (not op) (cddr op))
      (taql-warn "Exactly one :op argument must appear.")
      (setq op '(:op junk)))

    (when (cddr group-type)
      (taql-warn "At most one :group-type argument may appear.")
      (setq group-type nil))
	
    (when (cddr group-name)
      (taql-warn "At most one :group-name argument may appear.")
      (setq group-name nil))

    (when (and group-type
	       (not (member (cadr group-type) '(success failure))))
      (taql-warn "Value of :group-type must be one of success, failure.")
      (setq group-type nil))
	
    (when (cddr use)
      (taql-warn "At most one :use argument must appear.")
      (setq use nil))

    (when (and use
	       (not result-superstate-flag)
	       (not (variable-p (cadr use))))
      (taql-warn
       "Value of :use must be a variable.")
      (setq use nil))

    (when (and use
	       result-superstate-flag
	       (not (eql (cadr use) 'superstate))
	       (not (eql (cadr use) 'final-state))
	       (not (variable-p (cadr use))))
      (taql-warn
       "Value of :use must be 'superstate', 'final-state', or a variable.")
      (setq use nil))

    ;; figure out when to reconsider, if at all:
	
    (cond (terminate-when ; :terminate-when (COND+) or :terminate-when ()
	   (setq reconsider-edit-p nil)
	   (setq terminate-when-conds-p nil)
	   ;; A valueless :terminate-when may be followed by an edit
	   ;; clause, tricking the parser into thinking the edit
	   ;; was the value of the :terminate-when.  Fix things up
	   ;; here.  The TAQL syntax doesn't officially allow a
	   ;; valueless terminate-when, but group-arguments doesn't
	   ;; allow us to distinguish between a valueless keyword
	   ;; and a keyword with a nil value.  If edit was a keyword
	   ;; this wouldn't happen.
	   (dolist (term-when (cdr terminate-when))
	     (if (and (listp term-when)
		      (eq 'edit (car term-when)))
	       (push term-when (car args))
	       ;; ELSE
	       (if term-when
		 (setq terminate-when-conds-p
		       (append
			(prepare-conditions term-when)
			terminate-when-conds-p))))))
	  (t ; :terminate-when absent
	   (setq reconsider-edit-p t)
	   (setq terminate-when-conds-p nil)))

    (dolist (arg (car args))
      (cond ((atom arg)
	     (taql-warn "Edit expected, ~S found." arg))
	    ((eq (car arg) 'edit)
	     (push arg edits))
	    (t
	     (taql-warn "Edit expected, ~S found." arg))))
	
    (when (and use (or edits actions sliding-actions))
      (taql-warn ":Use and edit/action clauses cannot appear in the same TC.")
      (setq use nil))
	
    (when (and (not terminate-when) (not use) (not edits)
	       (not actions) (not sliding-actions))
      (taql-warn
       "Either :terminate-when, :use or at least one edit/action clause must be given.")
      (setq use '(:use superstate)))

    (let* ((prep-op (prepare-operator (cadr op) =operator))
	   (final-obj-id (genvar 'f))
	   (goal-id (genvar 'g))
	   (all-conds
	    `((goal ,=goal ^ operator ,=operator)
	      ,@(if space
		  `((goal ,=goal ^ problem-space ,=problem-space)
		    (problem-space ,=problem-space ^ name ,(cadr space))))
	      ,@(when (or group-type group-name)
		  `((goal ,=goal ^ goal-id* ,goal-id ^ state ,=state)
		    (state ,=state ^ final-state* ,final-obj-id)
		    (final-state-info ,final-obj-id
		      ^ goal-id ,goal-id
		      ,@(when group-type
			  `(^ final-state-type* ,(cadr group-type)))
		      ,@(when group-name
			  `(^ test-group-name* ,(cadr group-name))))))
	      ,@(cdar prep-op)
	      ,@(cdr prep-op)
	      ,@outer-when-conds)))

      (add-current-tc-to-spaces (cdr space) (list prep-op))

      (setq all-conds
	    (append
	     (process-bind-keyword (cdr bind) all-conds *prod-name-prefix*
				   result-superstate-flag)
	     all-conds))

      (if terminate-when-conds-p
	(eval
	 ;; 2-12: not sticky, because this has to retract in
	 ;; response to changes in the state by IO:
	 ;;
	 ;;   6-10-90 - ema - ha, that's a laugh.  when this fires,
	 ;;   it blows away edit-enabled* like it was never there.
	 ;; 5-21-91 - gry - changed pclass from not-sticky to unknown, since
	 ;; Soar should get this right now that we generate the @ directly.
		   
	 `(taql-sp unknown ,(newsym *prod-name-prefix*
				       'terminate-when *num-separator*)
	    ;; 3-23: added
	    ;; 7-26-90: Commented out, reconsider pref now generated directly.
	    ;;(operator ,=operator ^ control-stuff* ,c-id)
			 
	    ,@(if result-superstate-flag
		`((goal ,=goal ^ object ,=object)
		  (goal ,=object ^ operator ,=superoperator))
		;; ELSE
		`((goal ,=goal ^ operator ,=operator)))
	    ,@all-conds
	    ,@terminate-when-conds-p
	    -->
	    ;; 3-23:
	    ;; 6-10-90: reconsider the operator directly, because the
	    ;; reconsider has to be retractable; see comment above.
	    ;;(control-stuff* ,c-id ^ reconsider-disabled* true -)
	    ,(if result-superstate-flag
	       `(goal ,=object ^ operator ,=superoperator @)
	       ;; ELSE
	       `(goal ,=goal ^ operator ,=operator @)))))
	  
      (when use
	(state-edit nil nil nil nil nil nil
           ;; For :use, we only pass a partial production name, as more than
	   ;; one production will be generated.
           (makesym *prod-name-prefix*
		    'use- (if (variable-p (cadr use))
			    'existing-state
			    (cadr use)) *name-separator*)
	   all-conds
	   reconsider-edit-p nil
	   result-superstate-flag
	   (cadr use)
	   nil nil nil nil nil))
    
      (dolist (action1 (cdr actions))
	(let* ((action (group-arguments action1 '(:when)))
	       (inner-when-conds
		(prepare-condition-sets (cdr (assoc :when (cdr action))))))

	  (state-edit nil nil nil nil nil nil
		      (newsym *prod-name-prefix* 'actions *num-separator*)
		      (append all-conds inner-when-conds)
		      reconsider-edit-p nil result-superstate-flag
		      nil nil nil nil (car action) nil)))
    
      (dolist (action1 (cdr sliding-actions))
	(let* ((action (group-arguments action1 '(:when)))
	       (inner-when-conds
		(prepare-condition-sets (cdr (assoc :when (cdr action))))))

	  (state-edit nil nil nil nil nil nil
		      (newsym *prod-name-prefix* 'sliding-actions *num-separator*)
		      (append all-conds inner-when-conds)
		      nil t result-superstate-flag
		      nil nil nil nil (car action) nil)))

      (dolist (edit edits)
	(process-edit edit all-conds reconsider-edit-p
		      result-superstate-flag))))
  t)

;; 8-2-90:
;; Process the :bind keyword in an apply-operator or result-superstate TC.
;;
;; It takes these arguments:
;;    bind-values:  A list of the values of the :bind keywords.
;;    conds:  all-conds from apply-operator-aux.
;;    prod-name-prefix:  A prefix string for the name of the generated
;;       production.
;;    result-superstate-flag:  A boolean indicating that is non-nil when
;;       the TC is a result-superstate TC (otherwise it is apply-operator).
;;
;; It returns a list of conditions that should be added to all other
;; productions generated by the TC.  These extra conditions bind the
;; identifiers specified by the :bind keywords.  If an error is detected,
;; it returns NIL.
;;
;; Process-bind-keyword also creates an operator application production
;; that elaborate's the operator's control-stuff with the bindings of the
;; specified variables.  This production must apply on the first elaboration
;; cycle after the operator is selected, at the same time the control flags
;; are added to control-stuff.  I had originally thought to use an operator
;; modification production to create the bindings (i.e. test the operator's
;; acceptable preferences, so that the bindings will be there by the time
;; it is installed).  But there are a number of problems with that, the most
;; serious of which would have prohibited using :bind in result-superstate.

(defun process-bind-keyword (bind-values conds prod-name-prefix
				result-superstate-flag)
  (when (null bind-values)
    (return-from process-bind-keyword nil))
  
  (let ((funny-value (member-if-not #'variable-p bind-values)))
    (when funny-value
      (taql-warn "Value of :bind must be a variable: ~S" (car funny-value))
      (return-from process-bind-keyword nil)))
  
  (let* ((c-id (genvar 'c))
	 (bind-actions nil)
	 (bind-conds nil)
	 (control-stuff
	  (if result-superstate-flag
	      `((goal ,=goal ^ object ,=object)
		(goal ,=object ^ operator ,=superoperator)
		(operator ,=superoperator ^ control-stuff* ,c-id))
	      ;; ELSE TC is apply-operator
	      `((operator ,=operator ^ control-stuff* ,c-id)))))
    
    (dolist (var bind-values)
      ;; We make the indifferent preference for the binding for two reasons.
      ;; First, we could have a chunk transfer to create bindings in a
      ;; situation where the other result-creation chunks do not transfer.
      ;; In such cases, we don't want to get new bindings in the subgoal.
      ;; We could get around this problem in another way: by testing that
      ;; the bindings don't already exist before creating them.  But there
      ;; is a second problem:  the production that creates the bindings could
      ;; have multiple instantiations, and we only want to get one binding.
      (setq bind-actions
	  `(^ ,(makesym *current-taql-name* '- var)
	    ,(genvar (char (symbol-name var) 1)) + =
	    ,@bind-actions))
      (setq bind-conds
	  `(^ ,(makesym *current-taql-name* '- var) ,var
	    ,@bind-conds)))
    
    (eval
     `(taql-sp sticky ,(newsym prod-name-prefix 'bind *num-separator*)
       ,@conds
       ,@control-stuff
       -->
       (control-stuff* ,c-id ,@bind-actions)))
    
    `(,@control-stuff
      (control-stuff* ,c-id ,@bind-conds))))

;; processes a single edit clause.  called by apply-operator-aux.
;;
;;   2-12 - added "top-state" as value for ":what".  
;;   reconsider-op is passed to state-edit.
;;
;;   21-feb-90 - added support for sliding operators (:type keyword)
;;   sliding edits do not cause reconsiders to be generated, so that
;;   even is reconsider-op is passed in non-nil, it will be reset to nil
;;   if the edit clause is :type sliding.  Also changed :what keyword to
;;   allow editing of arbitrary objects.  The new syntax is
;;       :what {top-state | state | ({STATE-ATT | :none} CLASS-NAME [ID])
;;   This is compatible with the old definition.  In the new form, if
;;   :none is given in place of a state attribute, then the ID variable
;;   must be given and must be bound somewhere in the conditions.  The
;;   STATE-ATT form is retained both for compatibility and to provide a
;;   convenient shorthand for the common case of editing first-level state
;;   subobjects.
;;
;; 8-2-90: Added result-superstate-flag argument, added support for
;;   the new :copy keyword, and changed the semantics of the :rename
;;   keyword.
;;
;; 9-5-90: Added support for the :directive keyword.
;;
;; 6-14-91 - gry - Changed to support the :buffer-in-goal keyword.
;;
(defun process-edit (body outer-conds reconsider-op result-superstate-flag)
  (let* ((args (group-arguments (cdr body)
				'(:what :when :replace :copy :rename :remove
					:new :type :directive :buffer-in-goal)
				'construct)) 
	 (what (assoc :what (cdr args)))
	 (type (assoc :type (cdr args)))
	 (sliding-op nil)
	 (replaces (assoc :replace (cdr args)))
	 (copies (assoc :copy (cdr args)))
	 (renames (assoc :rename (cdr args)))
	 (removes (assoc :remove (cdr args)))
	 (news (assoc :new (cdr args)))		 
         (directives (assoc :directive (cdr args)))
	 (buffer-in-goal (assoc :buffer-in-goal (cdr args)))
	 (buffer-goal-value nil)
	 (obj-spec nil)
	 (editing-what nil)
	 (common-conds
	  (append
	   outer-conds
	   ;; the :when clause local to this "edit:"
	   (prepare-condition-sets (cdr (assoc :when (cdr args)))))))

    (when (not type)
      (setq type '(:type one-shot)))

    (when (cddr type)
      (taql-warn "At most one :type argument may appear.")
      (setq type '(:type one-shot)))

    (when (not (member (cadr type) '(one-shot sliding)))
      (taql-warn "Value of :type must be one of one-shot, sliding.")
      (setq type '(:type one-shot)))

    (setq sliding-op (eql (cadr type) 'sliding))
    ;; Sliding edits do not generate reconsiders.
    (if sliding-op
      (setq reconsider-op nil))
	
    (when (cddr buffer-in-goal)
      (taql-warn "At most one :buffer-in-goal argument may appear.")
      (setq buffer-in-goal nil))

    (when (and buffer-in-goal
	       (not (or (member (cadr buffer-in-goal)
				'(goal supergoal top-goal))
			(variable-p (cadr buffer-in-goal)))))
      (taql-warn "The value of :buffer-in-goal must be 'goal', 'supergoal', ~
                  'top-goal', or the identifier variable of a bound goal ~
                  object, not ~S"
		 (cadr buffer-in-goal))
      (setq buffer-in-goal nil))

    (when (and (null replaces) (null renames) (null removes) (null news)
	       (null copies) (null directives))
      (taql-warn
       "At least one of :replace, :copy, :rename, :remove, :new, :directive must appear.")
      (setq removes '(:remove (junk))))
	
    (when (cddr what)
      (taql-warn "At most one :what argument may be specified.")
      (setq what '(:what state)))
	
    (multiple-value-setq (editing-what obj-spec)
      (process-object-spec (cadr what) 'state))

    (setq buffer-goal-value (cadr buffer-in-goal))

    (dolist (replace1 (cdr replaces))
      (let* ((replace (group-arguments replace1 '(:when :by)))
	     (attval (car replace))
	     (by (assoc :by (cdr replace)))
	     (inner-when-conds (prepare-condition-sets
				(cdr (assoc :when (cdr replace))))))
		
	(when (or (not by) (cddr by))
	  (taql-warn "Exactly one :by argument must be specified: ~S"
		     replace1)
	  (setq by '(:by junk)))
		
	(when (or (null attval)
		  (cddr attval)
		  (not (atom (car attval)))
		  (not (atom (cadr attval))))
	  (taql-warn "Illegal attribute specification: ~S" replace1)
	  (setq attval '(junk)))
		
	(let ((from-spec (list (car attval)
			       (if (cdr attval)
				 (cadr attval)
				 (genvar2 (car attval)))))
	      (to-spec (list (car attval) (cadr by)))
	      (add-prod (newsym *prod-name-prefix* 'edit*replace*post-add
				*num-separator*))
	      (drop-prod (newsym *prod-name-prefix* 'edit*replace*drop
				 *num-separator*)))
		  
	  ;; replace semantics:  if LVALUE is given, make the add
	  ;; contingent on it.  otherwise, add is independent.
		  
	  (cond
	   ((cdr attval)
	    (state-edit from-spec to-spec t 'value-spec
			editing-what obj-spec add-prod
			(append common-conds inner-when-conds)
			reconsider-op sliding-op result-superstate-flag
			nil nil nil nil nil buffer-goal-value))
			
	   ;; no LVALUE; generate a separate production for
	   ;; doing the add:
			
	   (t
	    (state-edit nil to-spec nil 'value-spec 
			editing-what obj-spec add-prod
			(append common-conds inner-when-conds)
			reconsider-op sliding-op result-superstate-flag
			nil nil nil nil nil buffer-goal-value)
			 
	    (state-edit from-spec nil t 'value-spec 
			editing-what obj-spec drop-prod
			(append common-conds inner-when-conds)
			reconsider-op sliding-op result-superstate-flag
			nil nil nil nil nil buffer-goal-value))))))
	
    (dolist (copy1 (cdr copies))
      (let* ((copy (group-arguments copy1 '(:when :from :remove)))
	     (from (assoc :from (cdr copy)))
	     (remove (assoc :remove (cdr copy)))
	     (atts (car copy))
	     (from-obj-conds nil)
	     (from-obj-spec nil)
	     (inner-when-conds (prepare-condition-sets
				(cdr (assoc :when (cdr copy))))))
	
	(when (cddr from)
	  (taql-warn "At most one :from argument may be specified: ~S"
		     copy1)
	  (setq from nil))

	(when (not from)
	  (setq from '(:from state)))
		
	(multiple-value-setq (from-obj-conds from-obj-spec)
	  (process-object-spec (cadr from) 'state))
		
	(when (member-if #'(lambda (x)
			     (and (not (eql x 'source))
				  (not (eql x 'target))))
			 (cdr remove))
	  (taql-warn
	   "Value of :remove in :copy must be one of 'source', 'target': ~S"
	   copy1)
	  (setq remove nil))
		
	(when (null atts)
	  (taql-warn ":copy must specify at least one attribute: ~S"
		     copy1)
	  (setq atts '(junk)))

	(dolist (att atts)
	  (let* ((attvar (genvar 'c))
		 (att-spec (list att attvar))
		 (drop-p (member 'source (cdr remove))))

	    (when (not (atom att))
	      (taql-warn "Attribute name in :copy must be a symbol: ~S" att))

	    (state-edit att-spec att-spec drop-p 'att-name
			editing-what obj-spec
			(newsym *prod-name-prefix*
				(if drop-p
				  'edit*copy*drop&post
				  'edit*copy*post)
				*num-separator*)
			(append common-conds inner-when-conds)
			reconsider-op sliding-op result-superstate-flag nil
			from-obj-conds from-obj-spec nil nil buffer-goal-value)

	    (when (member 'target (cdr remove))
	      (state-edit att-spec nil t 'att-name
			  editing-what obj-spec
			  (newsym *prod-name-prefix* 'edit*copy*drop-target
				  *num-separator*)
			  (append common-conds inner-when-conds)
			  reconsider-op sliding-op result-superstate-flag nil
			  nil nil nil nil buffer-goal-value))))))
	
    (dolist (rename1 (cdr renames))
      (let* ((rename (group-arguments rename1 '(:when :from :remove)))
	     (from (assoc :from (cdr rename)))
	     (remove (assoc :remove (cdr rename)))
	     (att-pairs (car rename))
	     (from-obj-conds nil)
	     (from-obj-spec nil)
	     (inner-when-conds (prepare-condition-sets
				(cdr (assoc :when (cdr rename))))))
	
	(when (cddr from)
	  (taql-warn "At most one :from argument may be specified: ~S"
		     rename1)
	  (setq from nil))

	(when (not from)
	  (setq from '(:from state)))
		
	(multiple-value-setq (from-obj-conds from-obj-spec)
	  (process-object-spec (cadr from) 'state))

	(when (member-if #'(lambda (x)
			     (and (not (eql x 'source))
				  (not (eql x 'target))))
			 (cdr remove))
	  (taql-warn
	   "Value of :remove in :rename must be one of 'source', 'target': ~S"
	   rename1)
	  (setq remove nil))
		
	(when (null att-pairs)
	  (taql-warn ":rename must have at least one attribute pair: ~S"
		     rename1)
	  (setq att-pairs '(junk1 junk2)))

	(do ((rest-atts att-pairs (cddr rest-atts)))
	    ((null (cdr rest-atts))
	     (if rest-atts
	       (taql-warn
		":rename list must contain even number of attributes: ~S"
		rename1)))
	  (let* ((from-att (car rest-atts))
		 (to-att (cadr rest-atts))
		 (attvar (genvar 'r))
		 (from-spec (list from-att attvar))
		 (to-spec (list to-att attvar))
		 (drop-p (member 'source (cdr remove))))

	    (when (not (atom from-att))
	      (taql-warn "Attribute name in :rename must be a symbol: ~S"
			 from-att))
	    (when (not (atom to-att))
	      (taql-warn "Attribute name in :rename must be a symbol: ~S"
			 to-att))

	    (state-edit from-spec to-spec drop-p 'att-name
			editing-what obj-spec
			(newsym *prod-name-prefix*
				(if drop-p
				  'edit*rename*drop&post
				  'edit*rename*post)
				*num-separator*)
			(append common-conds inner-when-conds)
			reconsider-op sliding-op result-superstate-flag nil
			from-obj-conds from-obj-spec nil nil buffer-goal-value)

	    (when (member 'target (cdr remove))
	      (state-edit to-spec nil t 'att-name
			  editing-what obj-spec
			  (newsym *prod-name-prefix* 'edit*rename*drop-target
				  *num-separator*)
			  (append common-conds inner-when-conds)
			  reconsider-op sliding-op result-superstate-flag nil
			  nil nil nil nil buffer-goal-value))))))

    (dolist (new1 (cdr news))
      (let* ((new (group-arguments new1 '(:when)))
	     (action-spec (car new))
	     (inner-when-conds (prepare-condition-sets
				(cdr (assoc :when (cdr new))))))

	(state-edit nil action-spec nil 'action-spec
		    editing-what obj-spec 
		    (newsym *prod-name-prefix* 'edit*new*post *num-separator*)
		    (append common-conds inner-when-conds)
		    reconsider-op sliding-op result-superstate-flag
		    nil nil nil nil nil buffer-goal-value)))
    
    (dolist (directives1 (cdr directives))
      (let* ((directive-spec (group-arguments directives1 '(:when)))
             (directive-list (car directive-spec))
             (inner-when-conds (prepare-condition-sets
                                (cdr (assoc :when (cdr directive-spec))))))

        (state-edit nil nil nil nil
		    editing-what obj-spec 
		    (newsym *prod-name-prefix* 'edit*directive *num-separator*)
		    (append common-conds inner-when-conds)
		    reconsider-op sliding-op result-superstate-flag nil nil nil
		    directive-list nil buffer-goal-value)))
	
    (dolist (remove1 (cdr removes))
      (let* ((remove (group-arguments (cdr remove1) '(:when)))
	     (attval (cons (car remove1) (car remove)))
	     (inner-when-conds (prepare-condition-sets
				(cdr (assoc :when (cdr remove))))))
		
	(when (or (null attval)
		  (cddr attval)
		  (not (atom (car attval)))
		  (not (atom (cadr attval))))
	  (taql-warn "Illegal target attribute specification: ~S" remove1)
	  (setq attval '(junk)))
		
	(let* ((att-var (if (eq (car attval) ':all)
			  (genvar 'a)
			  (car attval)))
	       (val-var (if (cdr attval)
			  (cadr attval)
			  (genvar 'v)))			  
	       (drop-spec (list att-var val-var)))
		  
	  (state-edit drop-spec nil t 'att-name 
		      editing-what obj-spec
		      (newsym *prod-name-prefix* 'edit*remove*drop *num-separator*)
		      (append common-conds inner-when-conds)
		      reconsider-op sliding-op result-superstate-flag
		      nil nil nil nil nil buffer-goal-value)))))
  t)

;; generates a single production to edit a state, for the
;; apply-operator TC.  called by process-edit.  only reject
;; ^reconsider-disabled* if reconsider-op is t.  if sliding-op is t,
;; reconsider-op is nil. 
;;
;;   2-12 - added support for top-state as a value of :what.  moved
;;   ^adds* to goal, and made production explicitly "sticky".
;;   added boolean reconsider-op, as support for :terminate-when.
;;
;;   21-feb-90 - added boolean sliding-op, to support sliding operators.
;;      Also changed to edit arbitrary objects, given
;;      editing-what = 'arbitrary-object and obj-spec = (att class id),
;;      where att may be :none.
;;
;;   7-18-90:  Moved adds* to operator's control-stuff, for consistency
;;      with other control things, and most important because rs-adds*
;;      can't be on the goal if chunks are to have the correct support
;;      (they must give changes to the adds structures O-support).
;;
;; 8-2-90:
;;   Changed to take editing-what as a list of conditions that binds
;;     the id of the object to be edited, and obj-spec as a list (class id)
;;     specifying the class and identifer variable of the object to be
;;     edited.
;;   Added result-superstate-flag argument, for differential
;;     processing depending on whether we are inside an apply-operator
;;     or result-superstate TC.
;;   Added use-state parameter, to support the new :use keyword.
;;   Changed compilation of drops so that if an attribute to be dropped
;;     is a variable, it will not remove dummy-att* from the state.
;;   Added from-obj-conds, from-obj-spec parameters, to allow
;;     source and target objects to be different.  If the source and target
;;     objects are the same, these two arguments can be nil.  Changed
;;     drop-spec to drop-p, a boolean saying whether to drop the attribute
;;     specified by from-spec.
;;
;; 9-5-90:  Added support for the :directive keyword.
;;
;; 6-12-91 - gry - Added support for :actions and :sliding-actions keywords
;;
;; 6-14-91 - gry - Changed to support the :buffer-in-goal keyword.
;;
(defun state-edit (from-spec to-spec drop-p spec-type
			     editing-what obj-spec prod-name conds
			     reconsider-op sliding-op
			     result-superstate-flag use-state
			     from-obj-conds from-obj-spec
			     directives soar-actions buffer-in-goal)

  (when (not from-obj-spec)
    (setq from-obj-spec obj-spec))

  (let* ((add-id (genvar 'a))
	 (aug-id (genvar 'u))
	 (obj-class (car obj-spec))
	 (obj-id (cadr obj-spec))
	 (from-obj-class (car from-obj-spec))
	 (from-obj-id (cadr from-obj-spec))
	 (superop-c-id (genvar 'c))
	 (subop-c-id (genvar 'c))
	 (buffer-c-id (cond (buffer-in-goal
			     (case buffer-in-goal
			       (goal subop-c-id)
			       (supergoal superop-c-id)
			       (t (genvar 'c))))
			    (result-superstate-flag
			     superop-c-id)
			    (t
			     subop-c-id)))
	 (linkage-conds nil)
	 (modified-actions nil)

	 (control-stuff
	  `((goal ,=goal ^ operator ,=operator)
	    (operator ,=operator ^ control-stuff* ,subop-c-id)
	    ;; Always need the subop control-stuff, but in some cases
	    ;; we may need the control-stuff from other operators as well.
	    ,@(if result-superstate-flag
		`((goal ,=goal ^ object ,=object)
		  (goal ,=object ^ operator ,=superoperator)
		  (operator ,=superoperator
			    ^ control-stuff* ,superop-c-id)))
	    ,@(if buffer-in-goal
		(case buffer-in-goal
		  (goal
		   ;; Already bound subop control-stuff
		   nil)
		  (supergoal
		   (if result-superstate-flag
		     ;; Already bound superop control-stuff
		     nil
		     ;; ELSE
		     `((goal ,=goal ^ object ,=object)
		       (goal ,=object ^ operator ,=superoperator)
		       (operator ,=superoperator
				 ^ control-stuff* ,superop-c-id))))
		  (top-goal
		   `((goal ,=top-goal ^ object nil ^ operator ,=top-operator)
		     (operator ,=top-operator
			       ^ control-stuff* ,buffer-c-id)))
		  (t
		   (let ((buffer-op-id (genvar 'o)))
		     `((goal ,buffer-in-goal ^ operator ,buffer-op-id)
		       (operator ,buffer-op-id
				 ^ control-stuff* ,buffer-c-id))))))))

	 ;; 21-feb-90 - For sliding operators, edit clauses can apply at any
	 ;; time, as long as the state is internally consistent.  The state
	 ;; is NOT internally consistent when there are buffered adds that
	 ;; haven't been processed.  We don't need to test - ^adds* in the
	 ;; non-sliding case, because only one wave of edits gets to apply.
	 (synch-start (append
		       control-stuff
		       (if sliding-op
			 (if (not soar-actions)
			   `((control-stuff* ,buffer-c-id - ^ adds*)))
			 ;; ELSE, reconsider-op is nil also.
			 (if result-superstate-flag
			   `((control-stuff* ,superop-c-id
					    ^ edit-from-subgoal-enabled* true))
			   `((control-stuff* ,subop-c-id
					    ^ edit-enabled* true))))))
	
	 ;; 21-feb-90 - don't need to reject edit-enabled* here any more,
	 ;; see taql-support.soar, 21-feb-90.
	 (synch-stop
	  (let ((synch-stop-common
		 (if reconsider-op
		   ;; 3-23:
		   `((control-stuff* ,subop-c-id
				    ^ reconsider-disabled* true -))
		   ;; ELSE
		   nil)))
	    (if result-superstate-flag
	      (cons
	       `(control-stuff* ,superop-c-id
			       ^ edit-from-subgoal-enabled* true -)
	       synch-stop-common)
	      ;; ELSE
	      synch-stop-common))))

    (when use-state
      (let ((prod-name-1 (newsym prod-name 'goal-done *num-separator*))
	    (prod-name-2 (newsym prod-name 'state-pref *num-separator*)))
		
	;; Use-state can be superstate, final-state, or a variable that will
	;; be bound to a state indentifier at run time.  The first case occurs
	;; when the superstate is already the
	;; result state (as specified by the :use keyword).  This
	;; happens when the operators in the subspace have been
	;; making changes to superobjects all along.  All we have to do in this
	;; case is the synch-stop actions.
	;; In the final-state and variable cases, we must do the synch-stop
	;; actions as well as displace the current superstate and replace
	;; it with the specified result state.  We only want to displace
	;; the current superstate if the state we want as a result is actually
	;; a different state object.

	;; The above comments apply to :use used in the context of
	;; result-superstate.  It can also be used in the context of
	;; apply-operator, in which case only the variable case applies.
		
	;; these don't become op-apps by themselves, but they
	;; must be sticky:
	(eval
	 `(taql-sp sticky ,prod-name-1
		   ,@conds
		   ,@synch-start
		   -->
		   ,@synch-stop))
		
	(when (or (eql use-state 'final-state)
		  (variable-p use-state))
	  (let ((new-state (if (eql use-state 'final-state)
			     =state
			     use-state))
		(target-goal (if result-superstate-flag =object =goal))
		(old-state (if result-superstate-flag =superstate =state)))
							   
	    ;; We don't test the synch-start below because it really isn't
	    ;; necessary, and doing so would cause the state preferences to
	    ;; retract when the edit-from-subgoal-enabled* flag was removed.
	    ;; Make sure we have the superoperator bound, so that synch-stop
	    ;; will work.
	    (eval
	     `(taql-sp sticky ,prod-name-2
		       ,@conds
		       ,@control-stuff
		       ;; 4-9 - ema - removed negative test, because it was
		       ;; chunking unlinked conditions:
		       (goal ,target-goal ^ state { <> ,new-state ,old-state })
		       -->
		       ,@synch-stop
		       (goal ,target-goal ^ state ,old-state - @
			     ^ state ,new-state +))
	     )))))

    (when (not use-state)
      (when soar-actions
	(multiple-value-setq (linkage-conds modified-actions)
	    (prepare-actions soar-actions)))

      (eval
       `(taql-sp sticky ,prod-name
	   ,@synch-start
	   ,@conds
	   ,@editing-what
	   ,@linkage-conds
	   ,@from-obj-conds
	   ;; remove, rename (bind the value):
	   ,@(if from-spec
	       `((,from-obj-class ,from-obj-id
				  ^ ,@(if (and drop-p
					       (variable-p (car from-spec)))
					`({ <> dummy-att* ,(car from-spec) })
					;; ELSE
					(list (car from-spec)))
				  ,(cadr from-spec))))
	   -->
	   ,@synch-stop
	   
	   ;; just add the data type directives as-is, they will be processed
	   ;; later
	   
	   ,@directives

	   ,@modified-actions
	   
	   ;; Replace, remove, rename (drops are done directly):
	   ,@(if drop-p
	       `((,from-obj-class ,from-obj-id
				  ^ ,(car from-spec) ,(cadr from-spec) -)))
	   
	   ;; replace, rename, new (post adds to be done interpretively):
	   ,@(if to-spec
	       `((control-stuff* ,buffer-c-id ^ adds* ,add-id + &)
		 (add* ,add-id ^ class ,obj-class ^ id ,obj-id ^ aug ,aug-id + &)
		 ,@(case spec-type
		     ;; rename (att given):
		     (att-name
		      `((aug* ,aug-id ^ ,(car to-spec) ,(cadr to-spec) + &)))
		     ;; replace:
		     (value-spec 
		      (value-spec-actions 'aug* aug-id (car to-spec)
					  (cdr
					   to-spec)))
		     ;; new:
		     (action-spec
		      (action-spec-actions 'aug* aug-id to-spec))
		     (t
		      (error
		       "TAQL INTERNAL ERROR: case selector ~S fell through"
		       spec-type))))))))))

;; 8-mar-90 - gry - Changed so that the generated productions are part of
;; the implementation of the TAQL final-state operator.
;;
(defmacro propose-superobjects (&body body)
  `(propose-superobjects-aux ',body))

(defun propose-superobjects-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'propose-superobjects))
	 (args (group-arguments (cdr body)
		   '(:space :when :group-type :group-name
		     :object-type :object)
		   'construct))
	 (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 (outer-when-conds (prepare-condition-sets
			    (cdr (assoc :when (cdr args)))))
	 (objects (assoc :object (cdr args)))
	 (object-type (assoc :object-type (cdr args)))
	 (goal-id (genvar 'g))
	 (final-obj-id (genvar 'f))
	 (group-type (assoc :group-type (cdr args)))
	 (group-name (assoc :group-name (cdr args))))
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))	
    
    (when (cddr group-type)
      (taql-warn "At most one :group-type argument may appear.")
      (setq group-type nil))
    
    (when (and group-type (not (member (cadr group-type) '(success failure))))
      (taql-warn "Value of :group-type must be one of success, failure.")
      (setq group-type nil))
    
    (when (cddr group-name)
      (taql-warn "At most one :group-name argument may appear.")
      (setq group-name nil))
    
    (when (or (not object-type) (cddr object-type))
      (taql-warn "Exactly one :object-type argument must appear.")
      (setq object-type '(:object-type state)))
    
    (when (not (member (cadr object-type) '(problem-space state operator)))
      (taql-warn "Value of :object-type must be one of problem-space, state, operator.")
      (setq object-type '(:object-type state)))
    
    (when (not (cdr objects))
      (taql-warn "At least one :object keyword must appear.")
      (setq objects (list :object (genvar 'junk))))
    
    (when (member-if-not #'variable-p (cdr objects))
      (taql-warn "Value of :object must be a variable.")
      (setq objects (list :object (genvar 'junk))))
    
    (if (car args)
	(taql-warn "All arguments must be values of keywords."))
    
    (add-current-tc-to-spaces (cdr space) nil)

    (let* ((prod-name
	    (newsym *prod-name-prefix*
		'*propose-super
		(if (eql (cadr object-type) 'problem-space)
		    'space
		    (cadr object-type)) '<>)))
      (eval
       `(taql-sp unknown ,prod-name
	 (goal ,=goal ^ state ,=state ^ object ,=object)
	 (goal ,=goal ^ operator ,=operator)
	 (operator ,=operator ^ name final-state
	  ^ type* taql-op)
	 ,@(when (or group-type group-name)
	     `((goal ,=goal ^ goal-id* ,goal-id)
	       (state ,=state ^ final-state* ,final-obj-id)
	       (final-state-info ,final-obj-id
		^ goal-id ,goal-id
		,@(when group-type
		    `(^ final-state-type* ,(cadr group-type)))
		,@(when group-name
		    `(^ test-group-name* ,@(cdr group-name))))))
	 ,@(if space
	       `((goal ,=goal ^ problem-space ,=problem-space)
		 (problem-space ,=problem-space ^ name ,(cadr space))))
	 ,@outer-when-conds
	 -->
	 ,@(mapcar #'(lambda (object-id)
		       `(goal ,=object ^ ,(cadr object-type) ,object-id +))
	    (cdr objects))))))
  t)

;; calls process-what once for each :what clause, and process-value
;; for each :numeric/symbolic-value clause.

(defmacro evaluate-object (&body body)
  `(evaluate-object-aux ',body))

(defun evaluate-object-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'evaluate-object))
         (args (group-arguments (cdr body)
		   '(:space :what :numeric-value :symbolic-value :bind-object
		     :when)
		   'construct))
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
         (whats (assoc :what (cdr args)))
         (numeric-values (assoc :numeric-value (cdr args)))
         (symbolic-values (assoc :symbolic-value (cdr args)))
         (outer-bind-object (assoc :bind-object (cdr args)))
         (outer-when-conds
	  (prepare-condition-sets (cdr (assoc :when (cdr args)))))
	 (lookahead-evals nil)
	 (direct-evals nil))
    
    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space '(:space junk)))
    
    (when (not whats) 
      (taql-warn "At least one :what argument must appear.")
      (setq whats '(:what operator)))
    
    (when (and outer-bind-object (not (variable-p (cadr outer-bind-object))))
      (taql-warn "Value of :bind-object must be a variable.")
      (setq outer-bind-object nil))
    
    (if (car args)
        (taql-warn "All arguments must be values of keywords."))
    
    ;; process-what returns the OBJECT
    ;;
    ;;   plus a bunch of other goo consed into a list, that was being
    ;;   used to tell if return-when and cycle-when were present, but
    ;;   now i don't think is being used for anything.
    
    (dolist (what1 (cdr whats))
      (let* ((status
	      (process-what what1 space
		  outer-when-conds outer-bind-object))
	     (what-type (car status)))
	
	(cond
	  ((eq what-type 'lookahead-state)
	   (setq lookahead-evals t))
	  ((or (eq what-type 'problem-space)
	       (eq what-type 'state)
	       (eq what-type 'operator))
	   (setq direct-evals t))
	  (t
	   (error "INTERNAL TAQL ERROR, evaluate-object-aux")))))
    
    (when (not (or numeric-values symbolic-values))
      (taql-warn
       "No :numeric-value or :symbolic-value statements given.")
      (setq numeric-values '(:numeric-value 42)))

    (add-current-tc-to-spaces (cdr space) nil)
    
    ;; first do numeric, then symbolic values:
    (dolist (value-group (list numeric-values symbolic-values))
      
      ;; loop over values of type fixed in outer loop:
      (dolist (value1 (cdr value-group))
	
	(process-value value1 space
	    outer-when-conds outer-bind-object
	    (car value-group) direct-evals lookahead-evals))))
  t)

;; generates a production for the value of one :what clause.  this
;; production enables an evaluation take place for that object, by
;; raising a flag on the operator.  in the selection space, the
;; operator is evaluate-object, and is already selected.  in the
;; evaluation subgoal, the operator is evaluate-state, and is
;; elaborated before it is selected.  a support production proposes
;; evaluate-state once it is enabled, AND has a value.
;;
;; also generates code for returning an object early, when the
;; conjunction of :return-when clauses is satisfied.
;; 
;; what1 ::= OBJECT | (OBJECT [:when ... ] ... )
;;
;; returns (list OBJECT cycle-when-conds return-when-conds).
;;
;;   4-11 - added :cycle-when

(defun process-what (what1 space outer-when-conds outer-bind-object)
  (let* ((what (if (atom what1)
		   (list (list what1))
		   (group-arguments what1
		       '(:when :return-when :bind-object :cycle-when))))
	 (what-object (caar what))
	 (local-conds
	  (prepare-condition-sets (cdr (assoc :when (cdr what)))))
	 (return-when-conds
	  (prepare-condition-sets (cdr (assoc :return-when (cdr what)))))
	 (local-bind-object (assoc :bind-object (cdr what)))
	 (cycle-when-conds (assoc :cycle-when (cdr what)))
	 (bind-object nil)
	 (space-conds
	  (if space
	      `((goal ,=goal ^ problem-space ,=problem-space)
		(problem-space ,=problem-space ^ name ,(cadr space)))
	      ()))
	 (what-values '(lookahead-state operator state problem-space)))
    
    (when (cddr local-bind-object)
      (taql-warn "At most one local :bind-object argument may appear.")
      (setq local-bind-object nil))
    
    (when (and local-bind-object (not (variable-p (cadr local-bind-object))))
      (taql-warn "Value of :bind-object must be a variable.")
      (setq local-bind-object nil))
    
    (when (not (member what-object what-values))
      (taql-warn "Illegal value for :what: ~S" what-object)
      (setq what `(:what ,(car what-values))))
    
    (setq bind-object
	(cond (local-bind-object) (outer-bind-object) (t nil)))
    
    (cond
      
      ;; generate enabling productions for :what lookahead-state.
      
      ((eq what-object 'lookahead-state)
       
       (let* ((ev-state-id (genvar 'o))
	      ;; implicitly always test the object in lookahead, to
	      ;; get some context into the chunks:
	      (op-id (if bind-object
			 (cadr bind-object)
			 (genvar 'o)))
	      (bind-object-conds
	       ;; 4-20-90 - ema - bind copy of operator, rather than
	       ;; original, only because it saves having to recode R1.
	       ;; 7-19-90: Change (goal ^tried) to (state ^tried-tied-operator)
	       `((goal ,=goal ^ state ,=state)
		 (state ,=state ^ tried-tied-operator ,op-id))))
	 
	 ;; enable evaluation, by raising a flag on evaluate-state
	 ;; (unless cycle-when or return-when condtions are given;
	 ;; want those conjoined with other conditions to control
	 ;; flag).
	 
	 (if (not (or cycle-when-conds return-when-conds))
	     (eval
	      ;; a state elaboration, so not sticky:
	      `(taql-sp not-sticky
		,(newsym *prod-name-prefix*
		  'what*lookahead *num-separator*)
		(goal ,=goal ^ evaluate-state-op* ,ev-state-id)
		(operator ,ev-state-id ^ name evaluate-state)
		,@space-conds
		,@bind-object-conds
		,@outer-when-conds
		,@local-conds
		-->
		(operator ,ev-state-id ^ evaluation-type final))))
	 
	 (if cycle-when-conds
	     (let* ((goal-id (genvar 'g))
		    (classes '(goal problem-space state operator))
		    (newer (prepare-condition-sets
			    (replace-?-variables
			     (copy-tree (cdr cycle-when-conds)))))
		    (class (if newer
			       (caar (cadr cycle-when-conds))
			       'state))	; error condition
		    (obj-id (genvar2 class))
		    (older nil))
	       
	       ;; 7-4-90 - ema - haven't worked out what happens if we
	       ;; let the class be other than one of GPSO, so for now
	       ;; restrict to those, and also enforce being able to
	       ;; use our own variables to duplicate the conditions.
	       
	       (when (not (member class classes))
		 (taql-warn "Illegal class, first COND of :cycle-when"))
	       
	       (when (not (member (cadar (cadr cycle-when-conds)) '(^ -)))
		 (taql-warn "First COND for :cycle-when must omit ID."))
	       
	       ;; duplicate the cycle-when condition to test in
	       ;; another context:
	       
	       (setq older
		   `( (goal { <> ,=goal ,goal-id } ^ ,class ,obj-id)
		     (,class ,obj-id ,@(cdaadr cycle-when-conds))
		     ,@(cdadr cycle-when-conds)))
	       
	       (eval
		`(taql-sp not-sticky
		  ,(newsym *prod-name-prefix*
		    'what*lookahead*detect-cycle *num-separator*)
		  (goal ,=goal ^ evaluate-state-op* ,ev-state-id
		   ^ operator ,=operator)
		  (operator ,ev-state-id ^ name evaluate-state)
		  ,@space-conds
		  ,@bind-object-conds
		  ,@outer-when-conds
		  ,@newer
		  ,@older
		  -->
		  (goal ,=goal ^ operator ,=operator @)
		  (operator ,ev-state-id ^ evaluation-type final
		   ;; 7-4-90 - ema:
		   ;; ^ numeric-value novalue
		   ;; so we can print out the states:
		   ^ older-state* ,obj-id + = ^ newer-state* ,=state)))))
	 
	 ;; return early, from lookahead:
	 (if return-when-conds
	     (let* ((sg-id (genvar 's))
		    (leap-id (genvar 'o)))
	       (eval
		;; sticky, 
		`(taql-sp sticky
		  ,(newsym *prod-name-prefix*
		    'what*lookahead*return *num-separator*)
		  (goal ,=goal ^ object ,=object ^ state ,=state
		   ^ leap-op* ,leap-id)
		  (goal ,=object ^ object ,sg-id)
		  ,@space-conds
		  ,@bind-object-conds
		  ,@outer-when-conds
		  ,@return-when-conds
		  -->
		  (goal ,sg-id ^ operator ,leap-id + >)
		  (operator ,leap-id ^ state ,=state)))))))
      
      ;; generate enabling productions when :what isn't lookahead-state.
      
      ((not (eq what-object 'lookahead-state))
       
       (let* ((goal-id (genvar 'g))
	      (state-id (genvar 's))
	      (ev-obj-id (genvar 'o))
	      (eval-id (genvar 'e))
	      (object-id (if bind-object
			     (cadr bind-object)
			     (genvar 'o)))
	      (bind-object-conds
	       `((goal ,goal-id ^ state ,state-id ^ operator ,ev-obj-id)
		 (state ,state-id ^ evaluation ,eval-id)
		 (operator ,ev-obj-id ^ object ,object-id ^ type evaluation)
		 (evaluation ,eval-id ^ object ,object-id))))
	 
	 ;; enable evaluation, by raising a flag on evaluate-object:
	 (eval
	  `(taql-sp unknown
	    ,(newsym *prod-name-prefix*
	      'what*object *num-separator*)
	    ;; task context is supercontext:
	    (goal ,goal-id ^ object ,=goal ^ operator ,ev-obj-id)
	    (operator ,ev-obj-id ^ type evaluation)
	    ,@space-conds
	    ,@(if bind-object
		  `(,@bind-object-conds))
	    ,@outer-when-conds
	    ,@local-conds
	    -->
	    (operator ,ev-obj-id ^ evaluation-type final)))
	 
	 ;; return early, from selection space:
	 (if return-when-conds
	     (eval
	      `(taql-sp unknown
		,(newsym *prod-name-prefix* what-object
		  'what*object*return *num-separator*)
		;; have to bind object, to create a pref for it:
		(goal ,goal-id ^ object ,=goal)
		,@space-conds
		,@bind-object-conds
		,@outer-when-conds
		,@return-when-conds
		-->
		(goal ,=goal ^ ,what-object ,object-id))))))
      
      ;; should've trapped this:
      (t 
       (error "INTERNAL TAQL ERROR.  evalute-object-aux, cond.")))
    
    ;; return the value of what, and cycle-when-conds as a flag.
    (list what-object cycle-when-conds return-when-conds))
  )

(defun replace-?-variables (s-expr)
  (sublis (mapcar #'(lambda (var)
		      (cons var (genvar (variable-name-part var))))
		  (find-?-variables s-expr))
	  s-expr))

(defun find-?-variables (s-expr)
  (cond ((consp s-expr)
	 (union (find-?-variables (car s-expr))
		(find-?-variables (cdr s-expr))))
	((?-variable-p s-expr)
	 (list s-expr))
	(t nil)))

(defun ?-variable-p (thing)
  (and (variable-p thing)
       (char= (aref (symbol-name thing) 1) #\?)))

;; generates productions for value clauses.  these put their value in
;; a set when local and global conditions are satisfied.  support
;; productions copy values from the set to an evaluation object when a
;; signal to evaluate has been generated by a :what production.  the
;; set is stored on the operator, either evaluate-object or
;; evaluate-state. 
;;
;; if there's at least one each of direct-op-eval and lookahead-op-eval,
;; then for each value we have to generate production to assign
;; that value in the selection space, and one to assign that value
;; in the evaluation subgoal.  this provides maximum flexibility.
;; for instance, a user can  
;; supply direct evaluation knowledge for an operator in certain
;; cases, and have it fall through to lookahead search in other cases.
;; one of the generated productions will apply.
;; 
;; value1 ::= RVALUE OR (RVALUE [:when (COND+)]*)
;;
;; 5-17-91 - ema - added "unknown" as a legal RVALUE. 
;;
;; 9-14-90 - gry - Changed to accept all of the symbolic values listed
;;   in the manual.  Also changed to better disambiguate values that are
;;   function calls (such as (compute <x> + <y>)) from values of the
;;   form (RVALUE [:when (COND+)]*).
;;
(defun process-value (value1 space outer-when-conds outer-bind-object
			 value-type direct-evals lookahead-evals)
  (let* ((legal-symbolic-values
	  '(success failure partial-success
	    partial-failure novalue
	    required-success prohibit-failure lose win
	    draw indifferent
	    unknown			; 5-17-91 - ema
	    ))
	 ;; (keyword production-name-component operator-attribute)
	 (symbol-map '((:symbolic-value symbolic symbolic-value)
		       (:numeric-value numeric numeric-value)))
	 (value (cond
		 ((atom value1)
		  (list (list value1)))
		 ((member '^ value1)   ; a tree-structured value spec
		  (taql-warn
		   "The value of an evaluation cannot be an object: ~S"
		   value1)
		  (list (list (genvar 'junk))))
		 (t
		  ;; ELSE we have to try to decide whether the value is
		  ;; a function call, such as (compute <x> + <y>), or
		  ;; a value spec of the form (RVALUE [:keywords ...]).
		  (let ((garg (group-arguments value1
					       '(:when :bind-object))))
		    (cond ((or (cdr garg) ; there were keywords
			       ;; A function call can't start with a number,
			       ;; variable, list, or (we assume) one of the
			       ;; symbols that can be a symbolic-value.
			       (numberp (caar garg))
			       (variable-p (caar garg))
			       (listp (caar garg))
			       (member (caar garg) legal-symbolic-values))
			   (when (or (null (car garg))
				     (cdar garg))
			     (taql-warn "Exactly one value must be given for each ~S keyword: ~S"
					value-type value1)
			     (setq garg (list (list (genvar 'junk)))))
			   garg)
			  (t      ; assume it is a function call
			   (list garg)))))))
	 (the-value (caar value))
	 (local-conds (prepare-condition-sets
		       (cdr (assoc :when (cdr value)))))
	 (local-bind-object (assoc :bind-object (cdr value)))
	 (goal-id (genvar 'g))
	 (op-id (genvar 'o))
	 (bind-object nil)
	 (space-conds
	  (if space
	      `((goal ,=goal ^ problem-space ,=problem-space)
		(problem-space ,=problem-space ^ name ,(cadr space)))
	      ()))
	 (val-aug
	  ;; 5-17-91 - ema - added special case for "unknown":
	  (if (eq the-value 'unknown)
	      `(^ unknown-value true)
	      ;; ELSE a normal value:
	      `(^ ,(caddr (assoc value-type symbol-map)) ,the-value))))

    (when (cddr local-bind-object)
      (taql-warn "At most one local :bind-object argument may appear.")
      (setq local-bind-object nil))
    
    (when (and local-bind-object (not (variable-p (cadr local-bind-object))))
      (taql-warn "Value of :bind-object must be a variable.")
      (setq local-bind-object nil))
    
    (when (and (eql value-type :symbolic-value)
	       (not (variable-p the-value))
	       (not (member the-value legal-symbolic-values))
	       (not (listp the-value))) ; function call
      (taql-warn "Illegal value for ~S: ~S" value-type value1))
    
    (when (and (eql value-type :numeric-value)
	       (not (variable-p the-value))
	       (not (numberp the-value))
	       (not (eql the-value 'novalue))
	       (not (listp the-value))) ; function call
      (taql-warn "Illegal value for ~S: ~S" value-type value1))
    
    (setq bind-object
	(cond
	  (local-bind-object
	   (cadr local-bind-object))
	  (outer-bind-object
	   (cadr outer-bind-object))
	  (t nil)))
    
    (when lookahead-evals
      
      ;; we're in the evaluation subgoal; instantiate
      ;; evaluate-state with the value.  I-support the value,
      ;; so that we don't have to mask inconsistent states.
      ;; if the value disappears when evaluate-state is
      ;; selected, i guess it no-changes.
      ;;
      ;; the task context is the current context.
      
      (eval
       `(taql-sp not-sticky
	 ,(newsym *prod-name-prefix*
	   (cadr (assoc value-type symbol-map))
	   '*lookahead-state *num-separator*)
	 (goal ,=goal ^ evaluate-state-op* ,op-id)
	 (operator ,op-id ^ name evaluate-state)
	 ,@(if bind-object
	       ;; 4-20-90 - ema - bind copy of operator:
	       ;; 7-19-90: (goal ^tried) to (state ^tried-tied-operator)
	       `((goal ,=goal ^ state ,=state)
		 (state ,=state ^ tried-tied-operator ,bind-object)))
	 ,@space-conds
	 ,@outer-when-conds
	 ,@local-conds
	 -->
	 (operator ,op-id ,@val-aug))))
    
    (when direct-evals
      
      (if (not bind-object)
	  (setq bind-object (genvar 'o)))
      
      ;; we're in the selection space, applying
      ;; evaluate-object.  task context is supercontext.
      
      (eval
       `(taql-sp unknown
	 ,(newsym *prod-name-prefix* 
	   (cadr (assoc value-type symbol-map))
	   *num-separator*)
	 ;; task context is supercontext:
	 (goal ,goal-id ^ object ,=goal ^ operator ,op-id)
	 (operator ,op-id ^ type evaluation)
	 ,@(if bind-object
	       `((operator ,op-id ^ object ,bind-object)))
	 ,@space-conds
	 ,@outer-when-conds
	 ,@local-conds
	 -->
	 (operator ,op-id ,@val-aug))))))

;; modifies the ^desired object, and attaches state copy directives
;; to the problem space.  calls parse-state-copy-props.
;;
;; to prevent stupid chunks, evaluation properties are
;; installed only once, on the first instance of the problem-space.
;; this requires op-apps that test for the existence of ^default* true.
;; also means that properties can be changed only
;; once per instance of a space.

(defmacro evaluation-properties (&body body)
  `(evaluation-properties-aux ',body))

(defun evaluation-properties-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)
  
  (let* ((*current-taql-name* (new-taql (car body) 'evaluation-properties))
         (args (group-arguments (cdr body)
		   '(:space :when :better :if-multiple :lookahead)
		   'construct))
         (space (assoc :space (cdr args)))
	 (*prod-name-prefix* (build-prefix (cadr space)))
         (better (assoc :better (cdr args)))
	 (if-multiple (assoc :if-multiple (cdr args)))
	 (lookahead-args (assoc :lookahead (cdr args)))
	 ;; lookahead-args ::= (:lookahead (:monotonic arg
	 ;;                         :copy (arg1 arg2 ...)
	 ;;                         :copy-new (arg1 arg2 ...)
	 ;;                         :dont-copy (arg1 arg2 ...)))
	 (lookahead
	  (if lookahead-args
	      (group-arguments (cadr lookahead-args)
		  '(:monotonic :copy :dont-copy :copy-new))
	      nil))
	 (monotonic (assoc :monotonic (cdr lookahead)))
         (conds (prepare-condition-sets (cdr (assoc :when (cdr args)))))
	 (if-multiple-values '(pessimistic optimistic all))
	 (better-values '(higher lower))
	 (monotonic-values '(increasing decreasing))
	 (d-id (genvar 'd)))
    
    (when (or (not space) (cddr space))
      (taql-warn "Exactly one :space argument must be specified.")
      (setq space '(:space junk)))
    
    (when (and (not better) (not monotonic) (not if-multiple)
	      (not lookahead-args))
      (taql-warn
       "At least one of :better, :if-multiple, or :lookahead must be given.")
      (setq better `(:better ,(car better-values))))
    
    (when (and better (not (member (cadr better) better-values)))
      (taql-warn "Value of :better must be one of:  ~S " better-values)
      (setq better `(:better ,(car better-values))))
    
    (when (cddr better)
      (taql-warn "At most one instance of :better is allowed:  ~S" better)
      (setq better `(:better ,(car better-values))))
    
    (when (and if-multiple
	      (not (member (cadr if-multiple) if-multiple-values)))
      (taql-warn "Value of :if-multiple must be one of:  ~S "
	  if-multiple-values)
      (setq if-multiple `(:if-multiple ,(car if-multiple-values))))
    
    (when (cddr if-multiple)
      (taql-warn "At most one instance of :if-multiple is allowed: ~S "
	  if-multiple)
      (setq if-multiple `(:if-multiple ,(car if-multiple-values))))
    
    (when (cddr lookahead-args)
      (taql-warn "At most one instance of :lookahead is allowed:  ~S"
	  lookahead-args)
      (setq lookahead nil))
    
    (when (and monotonic
	      (not (member (cadr monotonic) monotonic-values)))
      (taql-warn ":monotonic value must be one of ~S " monotonic-values)
      (setq monotonic `(:monotonic ,(car monotonic-values))))
    
    (when (cddr monotonic)
      (taql-warn "At most one instance of :monotonic allowed:  ~S" monotonic)
      (setq monotonic `(:monotonic ,(car monotonic-values))))
    
    (if (car args)
        (taql-warn "All arguments must be values of keywords."))
    
    (add-current-tc-to-spaces (cdr space) nil)

    ;; each of these productions turns off the default* true, so only
    ;; one properties declaration can be made.  default values for
    ;; these properties are assumed to have worst preferences.
    
    (if better 
	(eval
	 `(taql-sp sticky
	   ,(newsym *prod-name-prefix* 'better *num-separator*)
	   (goal ,=goal ^ problem-space ,=problem-space ^ desired ,d-id)
	   ;; 4-16: assume that defaults have worse prefs:
	   (desired ,d-id ^ default* true)
	   (problem-space ,=problem-space ^ name ,(cadr space))
	   ,@conds
	   -->
	   (desired ,d-id ^ better ,(cadr better) ^ default* true -))))
    
    (if monotonic
	(eval
	 `(taql-sp sticky
	   ,(newsym *prod-name-prefix* 'monotonic *num-separator*)
	   (goal ,=goal ^ problem-space ,=problem-space ^ desired ,d-id)
	   (desired ,d-id ^ default* true)
	   (problem-space ,=problem-space ^ name ,(cadr space))
	   ,@conds
	   -->
	   (desired ,d-id ^ monotonic ,(cadr monotonic) ^ default* true -))))
    
    (if if-multiple
	(eval
	 `(taql-sp sticky
	   ,(newsym *prod-name-prefix* 'if-multiple *num-separator*)
	   (goal ,=goal ^ problem-space ,=problem-space ^ desired ,d-id)
	   (desired ,d-id ^ default* true)
	   (problem-space ,=problem-space ^ name ,(cadr space))
	   ,@conds
	   -->
	   (desired ,d-id ^ if-multiple ,(cadr if-multiple)
	    ^ default* true -))))
    
    ;; treat the state copy keywords as a group:
    
    (if lookahead
	(let ((actions (parse-state-copy-props lookahead)))
	  (if actions
	      (eval
	       `(taql-sp sticky
		 ,(newsym *prod-name-prefix* 'lookahead *num-separator*)
		 (goal ,=goal ^ problem-space ,=problem-space ^ desired ,d-id)
		 (desired ,d-id ^ default* true)
		 (problem-space ,=problem-space ^ name ,(cadr space))
		 ,@conds
		 -->
		 (desired ,d-id ^ default* true -)
		 ,@actions))))))
  t)

;; 3-29, ema.
;; called by evaluate-properties, to parse state copy properties.
;;
;; input:
;;
;;   larg ::= ( (<stuff-that-shouldn't-be-there>)
;;				(:copy (arg1 arg2 ...))
;;				(:copy-new (arg1 arg2 ...))
;;				(:dont-copy (arg1 arg2 ...)))
;;
;; outputs:
;; 7-13-90: Modified to return list of actions instead of list of attributes
;;   to add to the problem space, because in Soar 5.2 the
;;   ^all-attributes-at-level one augmentation has to go on the GOAL.
;;                         
;; 7-13-90: Added support for the ^dont-copy-anything flag (for the
;;   :dont-copy (:any) case).
;;
;; 7-20-90: Move the all-attributes-at-level one attribute back to the problem
;;   space, and add a taql-support production that copies it to the goal.
;;   This is because the production the state-copy specs get compiled into
;;   fires in the goal where the problem space is first installed, and not
;;   in the lookahead goal.  So putting the ^all-attributes-at-level one
;;   attribute on the goal would put it in the wrong place, and this
;;   seemed like the easiest fix.
;;
;; 8-2-90: Changed semantics so that :copy only defaults to :all when
;;   neither :copy-new nor :dont-copy is given.  Also allowed some reasonable
;;   keyword combinations that weren't supported before.  See the release
;;   notes for TAQL 3.1.3.
;;
;; semantics are given in the release-notes.

(defun parse-state-copy-props (larg)
  (let ((copy (assoc :copy (cdr larg)))
	(copy-new (assoc :copy-new (cdr larg)))
	(dont-copy (assoc :dont-copy (cdr larg)))
	(space-augs nil))
    
    (when (or (cddr copy)
	      (cddr copy-new)
	      (cddr dont-copy))
      (taql-warn
       "At most one each of :copy/:copy-new/:dont-copy may be given: ~S "
       (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (setq copy (cadr copy))
    (setq copy-new (cadr copy-new))
    (setq dont-copy (cadr dont-copy))
    
    (when (or (not (listp copy))
	      (not (listp copy-new))
	      (not (listp dont-copy)))
      (taql-warn "Value of :copy/:copy-new/:dont-copy must be a list: ~S "
	  (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (when (or (and (member :all copy) (cdr copy))
	      (and (member :all copy-new) (cdr copy-new))
	      (and (member :any dont-copy) (cdr dont-copy)))
      (taql-warn "Value :all cannot appear with other arguments: ~S "
	  (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (when (or (member-if-not #'atom copy)
	      (member-if-not #'atom copy-new)
	      (member-if-not #'atom dont-copy))
      (taql-warn "Value of :copy/:copy-new/:dont-copy must be a list of symbols: ~S"
	  (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (when (and (eq :any (car dont-copy))
	      (or copy copy-new))
      (taql-warn ":dont-copy (:any) cannot appear with :copy or :copy-new: ~S"
	  (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (when (and dont-copy
	      (not (eq :any (car dont-copy)))
	      (not (or (eq :all (car copy))
		       (eq :all (car copy-new)))))
      (taql-warn ":dont-copy (ATT-NAME+) can only appear along with :copy (:all) or :copy-new (:all): ~S"
	  (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (when (not (member nil
		   (cdr (member nil
			    (list (eq :all (car copy))
				(eq :all (car copy-new))
				(eq :any (car dont-copy)))))))
      ;; If there aren't at least two nils in the list, then at least two
      ;; wildcards were given.  This is not allowed.
      (taql-warn
       "At most one of :copy/:copy-new/:dont-copy can be given a wildcard value: ~S"
       (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (when (or (intersection copy copy-new)
	      (intersection copy dont-copy)
	      (intersection dont-copy copy-new))
      (taql-warn "The same attribute name cannot appear in more than one of :copy/:copy-new/:dont-copy: ~S"
	  (cdr larg))
      (setq copy nil copy-new nil dont-copy nil))
    
    (if (car larg)
	(taql-warn "All arguments to :lookahead must be values of keywords"))
    
    (when copy
      (if (eq :all (car copy))
	  (setq space-augs
	      (append '(^ all-attributes-at-level one) space-augs))
	  ;; ELSE
	  (setq space-augs
	      (append (multify 'one-level-attributes copy)
		  space-augs))))
    
    (when copy-new
      (if (eq :all (car copy-new))
	  (setq space-augs
	      (append '(^ all-attributes-at-level two) space-augs))
	  ;; ELSE
	  (setq space-augs
	      (append (multify 'two-level-attributes copy-new)
		  space-augs))))
    
    (when dont-copy
      (if (eq :any (car dont-copy))
	  (setq space-augs
	      (append '(^ dont-copy-anything true) space-augs))
	  ;; ELSE
	  (setq space-augs
	      (append (multify 'dont-copy dont-copy)
		  space-augs))))
    
    (if space-augs
	`((problem-space ,=problem-space ,@space-augs)))))

(eval-when (compile load eval)
  (soarsyntax))
