;;; Simple Menu Enhancements for GNU Emacs
;;;
;;; Version 1.1.1
;;;  3-oct-91 -FER TAB and M- replace "   " and ^[ in full help descriptions.
;;; 16-Sep-91 -FER better help display
;;; 6-12-91 - unbelievably better key search in sm-find-binding
;;; 6-11-91 - even more robust key search in sm-find-binding
;;; 6-10-91 - more robust key search in sm-find-binding
;;; Version 1.1
;;; 6-5-91 - added ability to show esc-x commands in help
;;; 5-27-91 - added ability to show esc-x commands after command completion
;;; 2 may 91 added (require 'cl) reported by dfreuden@govt.shearson.com,
;;;   ne201ph@prism.gatech.edu (Halvorson,Peter J), rayv@revenge.sps.mot.com 
;;;   (Ray Voith), & Sara.Kalvala@computer-lab.cambridge.ac.uk
;;; 30 may 91 - posted to gnu.emacs.sources version 1.0

;;; COPYRIGHT and WARNINGS
;;;
;;; GNU Emacs is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY.  No author or distributor
;;; accepts responsibility to anyone for the consequences of using it
;;; or for whether it serves any particular purpose or works at all,
;;; unless he says so in writing.  Refer to the GNU Emacs General Public
;;; License for full details.
;;;
;;; Everyone is granted permission to copy, modify and redistribute
;;; GNU Emacs, but only under the conditions described in the
;;; GNU Emacs General Public License.   A copy of this license is
;;; supposed to have been given to you along with GNU Emacs so you
;;; can know your rights and responsibilities.  It should be in a
;;; file named COPYING.  Among other things, the copyright notice
;;; and this notice must be preserved on all copies.
;;;
;;; Copyright (C) 1991 Frank Ritter.  Same license as above.
;;; Updated versions (if any) are available from the author or via ftp:
;;; from the elisp archive on tut.cis.ohio-state.edu as file
;;;  pub/gnu/emacs/elisp-archive/interfaces/simple-menu2.el.Z
;;;
;;; Initially based on code posted by Chris Ward.
;;;        Texas Instruments 
;;;        (cward@houston.sc.ti.com)       (214) 575-3128
;;;        (X.400: /ADMD=MCI/PRMD=TI/C=US/G=Chris/S=Ward MCI_Mail_ID #4418566)
;;; and posted comments on Chris's code by Erik Hennum (Erik@informix.com)

;;; I've completely rewritten the Chris Ward's menu system to suit my
;;; needs.  It is a simple tty based menu system for providing a limited
;;; number of choices in an extensible way.  I use it daily (well, not
;;; really, I now use the keystroke equivalents it teaches), but the point
;;; is that it is robust enough to put out.  I have cut most of the GNU
;;; commands from the menus, the present package is offered more for
;;; applications, but I would be happy to paste stuff people send me.  At
;;; the bottom of the code, I provide a sample set of menus.

;;;	OVERVIEW/INTRODUCTION
;;; 
;;; Simple-menu is a way to provide simple menus, rather reminiscent of
;;; the menus provided by the PDP software of McClellend & Rumelhart.  With
;;; the simple menus defined here for gnu-emacs, an initial menu of
;;; commands is displayed in the message line by calling run-menu on a
;;; previously defined menu.  The user types the first letter of an item to 
;;; select it, and a command gets executed, or a sub-menu is entered.
;;; Often you will bind the top menu call to a key.
;;;
;;; The prompt that is displayed includes a reminder that help is available  
;;; by typing ``?''.  (Help is also available by typing ^h or SPC.)
;;; 
;;; Simple menus are defined with def-menu.  This takes a menu-name, an
;;; title, an intro help comment (ie.: "Pick a command"), and a list of
;;; items to be put on the menu.  Each  menu item is a list with 2 
;;; components: 1) a display string, and 2) the command corresponding 
;;; to the string.  The first word is put in the menu, the first letter in
;;; the string is used as the selector for the item (case insensitive),
;;; and the whole string is used in the help display.  
;;; Def-menu and sm-add-to-menu allow you add commands to menus after they have
;;; been created.
;;;
;;; For example, the menu item:
;;; 
;;; ("Redraw         Redraw the screen."   recenter)
;;; 
;;; would create the item Redraw in the menu, and the letter R would
;;; select it.  In the help display, the full string would appear, along
;;; with any keybindings for the command in the local buffer, in this case
;;; the help line would look like 
;;; 
;;; Redraw         Redraw the screen. (C-l)
;;; 
;;; The command given as the second argument can be either: 1) a simple
;;; function name, 2) a list to eval, or 3) a menu name (symbol).  If you
;;; want two commands there, wrap them in a progn because the internals of
;;; the program use each list position.  The command should not display 
;;; a value with message as its result.
;;;
;;; If there is only one menu item, it is executed when the menu is run.
;;; After an item is selected and sucessfully completed, a possible keybinding
;;; or call via meta-X is displayed if possible.
;;;
;;;  Here's an example:
;;; 
;;; (def-menu simple-menu
;;;   "Choose a simple command"
;;;   "Here are some simple commands to choose from:"
;;;  (("Add 2 + 2      Add 2+2 and print it out for me."
;;;    (progn (message (format "The Answer is %s." (+ 2 2)))
;;;           (sleep-for 2)))
;;;   ("Redraw         Redraw the screen." recenter)
;;;   ("Simple menu    Recurse and run this darn menu again." simple-menu)))
;;; 
;;; Run-menu will start up the menu.  ^g will abort the menu.
;;; eg.
;;; (run-menu 'simple-menu)
;;; Binding this to a key makes the menu more usable.
;;; 
;;; I will NOT maintain it in the traditional sense (mostly a note to myself to
;;; get back to the thesis), but I will 1) incorporate changes that are
;;; useful to me, 2) fix bugs that you notice that would bother my
;;; application, and 3) incorporate good stuff you post me.
;;; 
;;; I am willing to answer questions if things aren't clear on how to get
;;; started.  
;;; 
;;; possible bugs/misfeatures:
;;; * The command should not display a value with message as its result.


(require 'cl)
(provide 'simple-menu)


;;; 
;;; 	I.	Variables and constants 
;;;

(defvar sm-default-function 'sm-cant-do-this
  "*Default function to call if a menu items doesn't have a function 
assigned to it.")

;; uses main help buffer, used to be *MENU Help*
(defconst help-buffer "*Help*")

(defconst simple-menu-help-string "(?):")

(defconst sm-default-help-header "Commands in the")
(defconst sm-default-help-for-help 
  "? or ^h or space to display this text at the first prompt.")
(defconst sm-default-help-footer "^G or space-bar to quit this menu now.
 First letter of the line to choose a command.")
(defconst CR "
")

; menus have the following fields:
;  prompt - the string used as the prompt before the choices
;  full-prompt - the string put in the message line
;  items - the list of items
;  prompt-header  - header (leading string) for the command line
;  help-header - header for the help buffer


;;;
;;; 	II.	Creating functions
;;;
;; menus are symbols, 
;; the raw items are stored under the plist 'items
;; the list that is displayed is stored in their value, 
;;    it is made by calling sm-menu-ized-items on the items, 
;; the prompt-header is under the 'prompt-header property
;; the help-header   is under the 'help-header prop.


(defun sm-menu-p (poss-menu)
 "Return t if item is a simple-menu."
 (and (boundp poss-menu)
      poss-menu
      (get poss-menu 'items)
      (get poss-menu 'prompt-header)
      (get poss-menu 'help-header)
      t))

(defun sm-def-menu (name prompt help-header items)
 "Define a menu object"
 ;; check for errors on the way in and massage args
 (if (not (symbolp name)) 
     (error (format "%s, the first arg must be a symbol." name)))
 (cond 
   ( (get name 'items) ;it's been created already
     (sm-add-to-menu name items) 
     (put name 'prompt-header prompt)
     (put name 'help-header help-header))
   (t  ;; doit
     (put name 'items items)
     (set name (sm-menu-ized-items items))
     (put name 'prompt-header prompt)
     (put name 'help-header help-header)
     t)) )

(fset 'def-menu 'sm-def-menu)

(defun sm-add-to-menu (menu items)
  "Add to NAME the list of ITEMS."
  (mapcar (function (lambda (x) (sm-add-to-menu-item menu x)))
          items))

(defun sm-add-to-menu-item (menu item)
  (let ( (old-items (get menu 'items)) )
   (cond ( (member item old-items) )
         (t 
           (put menu 'items (append old-items items))
           (set menu (sm-menu-ized-items (get menu 'items)))
           (put menu 'full-prompt nil)))
  ))

(fset 'add-to-menu 'sm-add-to-menu)


;;;
;;;		Running functions
;;;
;;; The cursor-in-echo-area doesn't work on pmaxen with X windows,
;;; we don't know why.

(defun sm-run-menu (amenu)
 "Present a menu"
 ;; get & present the prompt
 (if (= (length (eval amenu)) 1)
     (sm-eval-single-menu amenu)
 (let ((prompt (get amenu 'prompt-header))
       (full-prompt (get amenu 'full-prompt))
       (old-window (selected-window))
       (items (eval amenu))    )
  (if (not (string= prompt "")) (setq prompt (concat prompt ": ")))
  (if full-prompt
      (message full-prompt)
      (progn
        ;; this makes a full prompt, & saves it for later use
        (mapcar (function (lambda (x) (setq prompt (concat prompt x " "))))
                (mapcar 'first-word items))
        (setq prompt (concat prompt simple-menu-help-string))
        (put amenu 'full-prompt prompt)
        (message prompt)))
  ;; read it in & process char choice
  (let ( (cursor-in-echo-area t)
         (echo-keystrokes 0) )
  (setq opt (read-char)) )
  (setq opt (downcase opt))
  (if (or (= opt ?\C-h) (= opt ??)  (= opt ? ))
      (setq opt (downcase (sm-pop-up-help amenu))))
  (sm-eval-menu amenu opt)
 )))

(fset 'run-menu 'sm-run-menu)


;;;
;;; 	III. 	Helper functions 
;;; 

(defun sm-eval-menu (amenu opt)
 "Find in AMENU the command corresponding to OPT."
 (let ( (items (eval amenu))
        (current-key-map (current-local-map))
        (command nil) )
  (while items
     (setq item (pop items))
     (cond ( (and (null (third item))
                  (= opt (second item)))
             (setq command t)
             (error "Menu item \"%c\" not implemented yet." opt))
           ( (and (third item)
                  (= opt (third item)))
             (setq items nil)
             (setq command (second item))
             (cond ;; its a command
                   ((and (not (listp command)) (fboundp command))
                    (call-interactively command)
		    (sm-note-function-key command current-key-map))
                   ;; it is something to eval
                   ((listp command)
                    (eval command))
                   ;; it is another menu, you hope...
                   (t (sm-run-menu command))))))
  (if (not command) ; no match
      (progn (message (format "%c did not match a menu name" opt))
             (beep)))      ;note we lost
))

(defun sm-eval-single-menu (amenu)
 "Run in AMENU the single only command."
 (let* ( (item (first (eval amenu)))
         (command (second item)) 
         (current-key-map (current-local-map)) )
   (cond ;; its a command
        ((and (not (listp command)) 
              (fboundp command))
         (call-interactively command)
         (sm-note-function-key command current-key-map))
        ;; it is something to eval
        ((listp command)
         (eval command))
        ;; it is another menu, you hope...
        (t (sm-run-menu command)))
   (if (not command) ; no match
       (progn (message (format "%c did not match a menu name" opt))
              (beep)))     ;note we lost
))

(defun sm-make-help (help-header name items)
 "Make a help string for a simple menu."
 ;; this is a bit sloppy about how to make it....
 (let ((header nil) (result ""))
  (setq result
        (concat result
               (cond ((string= "" help-header)
                      (format "%s %s:%s" sm-default-help-header name CR CR))
                     (t (concat help-header ":" CR CR)))))
  (mapcar
     (function 
       (lambda (x) 
          (let ((bind-thing (sm-find-binding (car (cdr x))))
                (help-string (car x)) )
           (setq result (format "%s %s " result help-string))
           (if bind-thing
               (if (> (+ (length bind-thing) (length help-string)) fill-column)
                   (setq result 
                         (format "%s\n\t\t\t (%s)" result bind-thing))
                   (setq result 
                         (format "%s (%s)" result bind-thing))))
           (setq result (concat result CR))           )))
      items)
  (setq result (concat result CR " " sm-default-help-for-help ))
  (setq result (concat result CR " " sm-default-help-footer))
  result))

(defun sm-find-binding (function &optional map)
 "Finds a keybinding for function if it can."
 (if (not (symbolp function)) 
      nil
 ;; else
 (if (not map) (setq map (current-local-map)))
 (let ((initial-result
        (cond
         ((car (where-is-internal function map)))
         ;; check escape map too
         ( (sm-find-esc-binding function) )
         ( (fboundp function)
           ;; this assumes that function in interactive
           (format "ESC-X %s" function)))))
   (if (not (stringp initial-result))
       nil
     (if (string= "\t" initial-result)
         (setq initial-result "TAB"))
     (if (string= "\C-[" (substring initial-result 0 1))
         (setq initial-result (format "M-%s" (substring initial-result 1))))
     initial-result))))

(defun sm-find-esc-binding (function)
  "Finds a keybinding of FUNCTION just on the local escape map (if any)."
  (let* ( (local-map (current-local-map))
          (esc-map (if local-map
                       (lookup-key (current-local-map) "")))
          (esc-key (if esc-map
                       (where-is-internal function esc-map))) )
  (if esc-key
      (concat "M-" (car esc-key)))))

(defun sm-menu-ized-items (items)
 "Strips the first letter off and makes it the third item for ease and speed."
 (mapcar (function (lambda (x)
            (append (sm-setup-menu-item x)
                    (list (string-to-char (first-letter x))))))
         items))

(defun sm-setup-menu-item (x)
 "Setup the menu item X, which should have a string and symbol or listp.
If it doesn't, add a dummy function call."
 (cond ( (and (listp x)
              (stringp (car x))
              (or (symbolp (car (cdr x))) (listp (car (cdr x)))))
          x)
       ( (and (listp x)         ;given a null function
              (stringp (car x))
              (null (car (cdr x))))
         (append x (list sm-default-function)))
       (t (error "Bad menu item: %s" x)))
)

(defun sm-pop-up-help (menu)
  "Display the full documentation of MENU."
  ;; changed to work on menu items.
  (let ((opt nil) (opt-key 'beep) (full-prompt (get menu 'full-prompt))
        (help-info 
           (cond ((get menu 'help))
                 ((put menu 'help (sm-make-help (get menu 'help-header)
                                                menu
                                                (get menu 'items))))
                 (t "not documented")))  )
    (save-window-excursion
      (switch-to-buffer help-buffer)
      (erase-buffer)
      (insert help-info)
      (goto-char (point-min))
      (while (not (equal opt-key 'self-insert-command))
        (message full-prompt)
        (setq opt (read-key-sequence nil))
        (setq opt-key (lookup-key (current-global-map) opt))
        (if (memq opt-key 
                  (append 
                   (if (not (pos-visible-in-window-p (point-min)))
                       '(scroll-up))
                   (if (not (pos-visible-in-window-p (point-max)))
                       '(scroll-down))
                   '(next-line previous-line forward-line forward-char 
                     backward-char keyboard-quit scroll-right scroll-left)))
            (call-interactively opt-key)
          (bury-buffer help-buffer))))
    (string-to-char opt)))

(defun sm-note-function-key (command keymap)
 "Note to the user any keybindings for Command"
 (let ( (key-binding (sm-find-binding command keymap)) )
  (if key-binding
      (message (format "%s is also bound to \"%s\"."
	                command key-binding))) ))


;;;
;;; 	IV.	Utilities
;;; 

;; (first-word '("asdf" fun1))
;; (first-letter '("Asdf" fun1))

(defun sm-cant-do-this ()
  (message "No function to do this menu item yet."))

(defun first-word (menu-item)
 "return the first word of the first part (a string) of MENU-ITEM"
 (let ((string  (car menu-item)))
  (substring string 0 (string-match " " string))))

(defun first-letter (menu-item)
 "return the first letter of the first part (a string) of MENU-ITEM"
 (let ((string  (first-word menu-item)))
    (downcase (substring string 0 1))))


;;;
;;;	V.	Menus for emacs
;;;

(def-menu 'emacs-menu
  "Emacs commands"
  "Menu of plain Emacs commands"
 '(("Windows      Manipulate multiple window settings."   emacs-windows-menu)
   ("Modify       Change your editing environment."       emacs-modify-menu)
   ("Block menu   Perform operations on blocks (regions) of text." emacs-block-menu)
))

(def-menu  'emacs-block-menu
  "Block Option"
  "Displays menu of block commands to chose from"
 '(("Align    Adjust all lines in region Left, Right, or Centered." 
        emacs-align-menu)
  ("Eval     Evaluate region as a Lisp expression."           eval-region)
  ("Fill     Fill each paragraph in the region."              fill-region)
  ("Indent   Indent region according to major mode."          indent-region)
  ("Lower    Convert all characters in region to lowercase."  downcase-region)
  ("Narrow   Narrow scope of edit to region."                 narrow-to-region)
  ("Spell    Check spelling of all words in region."          spell-region)
  ("Upcase   Convert all characters in region to uppercase."  upcase-region)
  ))

(def-menu 'emacs-modify-menu
  "Modify Option"
  "Modify editing environment options are"
 '(("Keys     Locally rebind a key to a function."      local-set-key)
   ("Mode     Change current major/minor mode."         emacs-mode-menu)
   ("Options  Change environmental variable values."    (edit-options))
   ("Save     Save current options settings to a file."
             (message "Modify Save not implemented yet."))
   ("Tabs     Modify tab stop locations."               edit-tab-stops))  )

(def-menu 'emacs-windows-menu
  ""
  "Displays menu of window commands to chose from"
 '(("Buffers  Change to buffers menu."                       emacs-buffer-menu) 
  ("Compare  Compare text in current window with text in next window."
    compare-windows)  
  ("Delete   Remove current window from the display."               delete-window)
  ("Find     Find another buffer and change current window to it."  select-window)
  ("Split    Divide current window Vertically or Horizontally."
   (progn
    (while (not (or (= opt ?h) (= opt ?v)))
      (message "Split window: Horizontally Vertically ")
      (setq opt (downcase (read-char))))
    (if (= opt ?h) 
        (call-interactively 'split-window-horizontally)
        (call-interactively 'split-window-vertically))   ))
  ("Other    Change to next window."                      other-window)
  ("1        Make current window the only one visible."   (delete-other-windows))
  ("+        Increase lines in current window."           (do-window-sizing))
  ("-        Decrease lines in current window."           (do-window-sizing))
  ("<        Increase columns in current window."         (do-window-sizing))
  (">        Decrease columns in current window."         (do-window-sizing))))

(defun do-window-sizing ()
 ;; is opt passed down?
 (while (or (= opt ?+) (= opt ?-) (= opt ?>) (= opt ?<))
   (message "Change window size press '+', '-', '<', '>', or space to quit.")
   (if (= opt ?+) (enlarge-window 1))
   (if (= opt ?-) (shrink-window 1))
   (if (= opt ?>) (enlarge-window-horizontally 1))
   (if (= opt ?<) (shrink-window-horizontally 1))
   (setq opt (read-char))))


(def-menu 'emacs-buffer-menu
  ""
  "Displays menu of buffer commands to chose from"
 '(("Delete   Kill current buffer."               kill-buffer)
  ("Edit     Edit another buffer."               switch-to-buffer)
  ("File     Change to use File menu."           files-menu)
  ("List     List current buffers and status."   list-buffers)
  ("Other    Switch to buffer in other window."  switch-to-buffer-other-window)
  ("Spell    Check spelling for current buffer." ispell-buffer)
  ("Toggle   Toggle current buffer read-only status." toggle-read-only)
  ("Window   Change to Windows menu."                 windows-menu)))

(def-menu 'emacs-mode-menu
  "Mode"
  "Displays menu of known major and minor modes to chose from"
 '(("1  [pfe-mode] Use PFE emulation and keyboard layout."   (pfe-mode))
  ("A  [edit-abbrevs-mode] Major mode for editing list of abbrev definitions."
     (edit-abbrevs-mode))
  ("C  [c-mode] Major mode for editing C language source files."   (c-mode))
  ("D  [normal-mode] Default to normal mode for current file."  (normal-mode))
  ("F  [fortran-mode] Major mode for editing FORTRAN source files."  
    (fortran-mode))
  ("G  [emacs-lisp-mode] Major mode for editing GNU Emacs lisp source files."
     (emacs-lisp-mode))
  ("I  [lisp-interaction-mode] Major mode for typing/evaluating Lisp forms."
     (lisp-interaction-mode))
  ("L  [lisp-mode] Major mode for editing LISP code other than Emacs Lisp."
    (lisp-mode))
  ("O  [outline-mode] Major mode for editing outlines with selective display."
     (outline-mode))
  ("P  [picture-mode] Use quarter-plane screen model to edit."  (picture-mode))
  ("T  [text-mode] Major mode for editing regular text files." (text-mode))
  ("X  [tex-mode] Major mode for editing files of input for TeX or LaTeX."
     (tex-mode))
  ("Z  [fundamental-mode] Major mode not specialized for anything."
    (fundamental-mode))))

(def-menu 'emacs-align-menu
  "Align Option"
  "Displays menu of region alignment commands to chose from:"
 '(("Center   Center all lines in region between left margin and fill column."
     center-region)
  ("Justify  Fill each paragraph between left margin and fill column."
     (fill-region (point) (mark) t))
  ("Left     Adjust lines to start in a specific column."
    (progn (setq opt 
                 (read-input "Align left at column: " (int-to-string left-margin)))
           (setq opt (string-to-int opt))
           (message (format "Align left at column %d." opt))
           (indent-rigidly (point) (mark) opt)))
  ("Right    Ajdust lines to end in a specific column if possible."
     (progn (setq opt (read-input "Align right at column: " 
                                  (int-to-string left-margin)))
            (setq opt (string-to-int opt))
            (message (format "Align right at column %d." opt))
            (right-flush-region (point) (mark) opt)))
  ("Tab      Indent each line in region relative to line above it." indent-region)
  ))
