;;; -*- Mode: Text -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;;; File            : taql-template.el
;;; Author          : Frank Ritter
;;; Created On      : Fri Jul  6 13:59:51 1990
;;; Last Modified By: Frank Ritter
;;; Last Modified On: Tue Sep 24 22:31:31 1991
;;; Update Count    : 143
;;; 
;;; PURPOSE
;;; 	Changes to template mode and its associated menu and symbol modes
;;;  to make it more useful for soar.
;;;
;;; TABLE OF CONTENTS
;;;	i.	New require(s)
;;;	ii.  	New template mode variables
;;; 	I.  	Change to sym-read-string
;;;     II. 	Change to menu-mode in menu.el
;;; 	III.    tpl-insert-template
;;;	IV. 	Menu-set-local-keys
;;;     ;V. 	menu-help-message [will go away since unused on 1 sept.]
;;;	VI.     Tpl-insert-selection
;;;	VII.	Tpl-initialize-scan
;;;	VIII.	Symbol-mode
;;;	IX.	menu-exit-recursive-exit...strips leading spaces now
;;; 	X.	Menu-next-match
;;;	XI.	Tpl-insert-selection
;;;	XII.	Make-sub-selection-template
;;;	XIII.	Tpl-y-or-n-p 
;;;	XIV.	Tpl-insert-repetition
;;;	XV.	tpl-build-template-list
;;;	XVI.	sym-set-local-keys
;;;	XVII.	make-symbol-mode-syntax-table
;;;	XVIII.	symbol-dabbrev-expand
;;;     XVIV.	tpl-unscan
;;;	XX.	tpl-fix-syntax
;;;
;;; (C) Copyright 1990, Frank Ritter, all rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; * should save where textenter goes in case pointer gets repositioned
;;; * should auto expand optional String types when chosen non-optional
;;;
;;; * should have variable tpl-expand-optional-after-query


;;;
;;;	i.	New require(s)
;;;
;;; most sites should have dabbrev.el, but we'll now carry it around
;;; with us, and require it here.
;;;

(require 'new-dabbrevs)

;;;
;;;	ii.  New template mode variables
;;;

(defconst sym-prompt-help " (CR ends, ^G quits, TAB auto completes)")

(defvar tpl-sub-selection-name "-clauses"
  "name to append to regular token name when creating a sub-selection template.")

(defvar tpl-look-in-buffer-on-errorp nil
  "if template is not found, should program query user to check buffers?")

(defvar tpl-look-in-file-on-errorp nil
  "if template is not found, should program query user to check?") 

(defconst tpl-menu-title-header "Pick a " 
  "default menu header for tpl menus.")

(defconst tpl-menu-title-footer " (CR to select)" 
  "default menu footer for tpl menus.")

(defvar length-tpl-begin-optional nil
  "how long in characters the tpl-begin-optional *really* is.")

(defvar length-tpl-pattern-optional nil
  "how long in characters the tpl-pattern-optional *really* is.")

(defvar length-tpl-begin-placeholder nil
  "how long in characters the tpl-begin-placeholder *really* is.")

;;;  Menu constants
;;;

(defconst menu-mode-nickname "*Menu*" 
    "Description of menu mode, and the first thing the on the line")

(defconst menu-mode-help-reference 
    (substitute-command-keys "\\[describe-mode] help")
    "Description of how to get help.")

(defconst menu-mode-quit-help " ^G quits."
    "How to exit w/o selection")



;;;
;;;	ii.  New values for old variables
;;;

;;; These variables have to be set before entering code...

(setq tpl-begin-placeholder "{")
(setq tpl-end-placeholder "}")
(setq length-tpl-begin-placeholder 1)
(setq tpl-literal-whitespace t)
(setq tpl-display-begin "==>")
(setq tpl-display-end "<==")
(setq sym-start-display "==>")
(setq sym-end-display "<==")

;; tests:
;(string-match  "\\(<\\|\\[\\)" "[")
;(string-match  "\\?" "d")

(setq tpl-pattern-optional "\\?")

;; the tpl-pattern-optional is really just "?" (1 long), the \\ is needed 
;; to quote it.
(setq length-tpl-pattern-optional 1)




;;; 
;;;  	I.  Change to sym-read-string (prompt original)
;;; change in symbol.el:
;;;  moved improved prompt also onto mode-line.
;;;  now checks for syntax-table

(defun sym-read-string (prompt original)
  "Read characters and insert them at point.  First arg PROMPT is a
    message to prompt the user.  Second arg ORIGINAL is an initial
    string to use if all input is deleted."
 (let (save-mode save-name save-keymap sym-input sym-valid-response
       save-mode-line-format (save-syntax-table (syntax-table)) )
   (setq prompt (concat prompt sym-prompt-help))
   ;; Initialize state
   (setq sym-start (point))
   (setq sym-end-marker (make-marker))
   (set-marker sym-end-marker (point))
   (setq sym-original original)
   ;; Save mode variables
   (setq save-mode major-mode)
   (setq save-name mode-name)
   (setq save-keymap (current-local-map))
   (setq save-mode-line-format mode-line-format)  ;-fer
   ;; this sets up syntax and more
   (symbol-mode prompt)
   (message "%s " prompt)
   ;; Wait for user's creation
   (setq sym-input original)
   (unwind-protect
     (progn
        (setq sym-valid-response nil)
        (while (not sym-valid-response)
          (recursive-edit)
	  ;; Pick up created string
          (setq sym-input
                (buffer-substring sym-start (marker-position sym-end-marker)))
          (sym-reposition-point)
          ;; If invalid try again
          (if (and (boundp 'sym-check-validity-hook)
                   sym-check-validity-hook)
              (progn
                 (setq sym-valid-response
                 (funcall sym-check-validity-hook sym-input)))
              ;; else
              (setq sym-valid-response t)) ))
        ;; Restore display string
       (if (< sym-start (marker-position sym-end-marker))
           (progn
            (goto-char sym-start)
            (delete-backward-char (length sym-start-display))
            (goto-char (marker-position sym-end-marker))
            (delete-char (length sym-end-display))))
       ;; Restore mode variables
       (setq mode-line-format save-mode-line-format)  ;-fer
       (setq major-mode save-mode)
       (setq mode-name save-name)
       (use-local-map save-keymap))
       (set-syntax-table save-syntax-table)
       ;; Return string entered
       sym-input ))


;;;
;;;	II.		Menu-mode
;;;
;;;  Added title and default-item.
;;;

(defun menu-mode (&optional title default-item)
  "Major mode for selecting an item from a menu.
     Like View Mode, but with only the following commands:
     Space, C-n		menu-next-item
     Del, C-p		menu-previous-item
     a,b,...,z		menu-next-match
     C-h m 		describe-mode  (help)
     C-g		aborts menu choice
     Return, Linefeed, 
     \\[exit-recursive-edit] 		exits.
   Returns the line selected."
  ;; title is used in the mode line
  (interactive)
  (let ( (window-min-height 2))  ;allows small windows to be built
  (or menu-mode-map (menu-set-local-keys))
  (use-local-map menu-mode-map)  (setq major-mode 'menu-mode)
  (setq mode-name "Menu")        (setq case-fold-search t)
  (setq menu-last-input "")
  (setq mode-line-format (menu-mode-line-format title))
  (message "Position on selection and exit with Return (or ESC-^c).")
  (setq menu-choice nil)
  (cond ( (not (equal "" default-item))
          (goto-char (point-min))
          (re-search-forward (concat "^*" default-item) nil t)))
  (unwind-protect
     (recursive-edit))
  menu-choice))

(defvar soar-whitespace 
        (list (string-to-char " ")  (string-to-char "	")))


;;;
;;; 	III.   tpl-insert-template
;;;
;;;   changed so that it cleans up after itself.  It's also probably
;;; the place to handle non-template matches gracefully.
;;;

(defun tpl-insert-template (tpl-name)
 "Insert the template TPL-NAME at point."
 (let (display-string template start template-type looking)
  ;; Display selected template
  (setq display-string (concat tpl-display-begin tpl-name tpl-display-end))
  (insert-before-markers display-string)
  (backward-char (length display-string))
  (setq looking t)
  (while looking				; Find template.
    (setq template (tpl-find-template tpl-name))
    (if template
        (progn
	  (setq looking nil)		; Insert template
	  (delete-char (length display-string))
	  (setq start (point))
	  (setq template-type (tpl-token-type template))
	  (cond
	    ((equal template-type tpl-sequence-type)
             (tpl-unscan template)
	     (tpl-find-expansion-destination start (point))
	     (cond ((< tpl-ask-expansion-depth 0)
                    (tpl-delete-placeholders-in-region start (point)))
                   ((> tpl-ask-expansion-depth 0)
                    (expand-placeholders-in-region start (point))) ))
	    ((equal template-type tpl-selection-type)
	     (tpl-insert-selection template))
	    ((equal template-type tpl-repetition-type)
		(tpl-insert-repetition template))
	    ((equal template-type tpl-function-type)
		(tpl-insert-function template))
	    ((equal template-type tpl-string-type)
		(tpl-unscan template)) ))
	; Else report failure
	(progn
	  (if (and tpl-look-in-buffer-on-errorp
                   (y-or-n-p "Cannot find template---look in a buffer? "))
	      (progn (setq looking nil)
                     (tpl-insert-string-from-buffer tpl-name display-string) )
	   ;; else
	   (if (and tpl-look-in-file-on-errorp
                    (y-or-n-p "Cannot find template---load a template file? "))
	       (progn (save-some-buffers)
		      (load-tpl-file))
	      ;; else
	      (progn
		(setq looking nil)
                ;; clean up -fer
                (setq a display-string)
                (delete-char (length display-string))
                (setq display-string 
                      (concat tpl-begin-placeholder tpl-name 
                              tpl-end-placeholder))
                (insert-before-markers display-string)
		(error "Gave up looking for template.")))))
	))))



;;;
;;;	IV. 	tpl-expand-textlong-type

(defun tpl-expand-textlong-type (name)
  "Expand the textlong placeholder at point called NAME."
  (let (start display-string save-buffer new-string start-column)
					; Highlight placeholder
    (setq start (point))
    (re-search-forward tpl-pattern-placeholder)
    (delete-region start (point))
    (setq display-string (concat tpl-display-begin name tpl-display-end))
    (insert-before-markers display-string)
    (backward-char (length display-string))
    (setq start (point))		; Save current location
    (save-window-excursion		; Prepare buffer
      (setq save-buffer (buffer-name))
      (switch-to-buffer-other-window tpl-textlong-buffer)
      (erase-buffer)
      (shrink-window 5)
      ;; Wait for return from recursive edit
      (message (substitute-command-keys
		"Type replacement and exit with \\[exit-recursive-edit]"))
      ;; change here, -fer makes how to get out clearer
      (setq mode-line-format ( tpl-textlong-buffer-mode-line-format))
      (recursive-edit)
					; Get new value and insert
      (setq new-string (buffer-substring (point-min) (point-max)))
      (set-buffer save-buffer)
      (delete-windows-on tpl-textlong-buffer)  ) ; save-window-excursion
    (bury-buffer tpl-textlong-buffer)
    ;; Return to proper location
    (goto-char start)
    (delete-char (length display-string))
    (setq start-column (current-column))
    (setq start (point))
    (insert-before-markers new-string)
    (indent-rigidly start (point) start-column))  )

(defun tpl-textlong-buffer-mode-line-format ()
  "set up the mode-line for a textlong buffer"
  ;; note good cog eng. here in that we make the command to get out
  ;; always visible
  ;; also simplified the mode line considerably
  (list
   "" 
   ;mode-line-modified mode-line-buffer-identification "   " 
   " Creating  {"
   ;; name is dynamically bound in Tpl-expand-textlong-type
   'name
   "}  of type %10b "  
   global-mode-string
   (substitute-command-keys
        "( Type \\[exit-recursive-edit] to exit, \\[describe-mode] for help )")
   ))

(save-window-excursion
    (set-buffer tpl-textlong-buffer)
    (setq mode-line-format (tpl-textlong-buffer-mode-line-format)))


;;; 
;;;	V. 	Menu-mode-line-format
;;;
;;; New mode-line that tells the user more about what's going on.
;;;
;;; used to have:
;;;--**-emacs[plateo.soar]: Menu                   [(Menu Fill)]----Top-----------
;;; would like to have
;;; Select a {taql-construct} SPC,^N next item; A-z auto match; CR select;^G quit

(defun menu-mode-line-format (&optional title)
  "set up the mode-line for a menu buffer"
  (if (not title) (setq title  menu-mode-nickname))
  ;; note good cog eng. here in that we make the command to get out
  ;; always visible, etc.     (menu-mode-line-format)
  (list     ""
     title  "  ("
     menu-mode-help-reference ", " 
     (substitute-command-keys menu-mode-quit-help) 
     ")"
     "----"  '(-3 . "%p")
     "-%-"
  ))



;;;
;;; 	VI.	Tpl-insert-selection
;;;
;;; Leading spaces on plain items in selection templates are now preserved.
;;; Leading spaces on template names are not.

(defun tpl-insert-selection (template)
  "Insert a template at point using the selection type TEMPLATE."
 (let* (start stop size choice choice-template choice-list
        (window-min-height 2) ; let menus offer just two selections
        (save-buffer (buffer-name))
        (token-name (tpl-token-name template))
        (menu-title (concat tpl-menu-title-header
                            tpl-begin-placeholder token-name
                            tpl-end-placeholder
                            tpl-menu-title-footer))
        (display-string (concat tpl-display-begin token-name tpl-display-end)) )
    ;; Highlight placeholder
    (insert-before-markers display-string)
    (backward-char (length display-string))
    ;; Prepare menu buffer
    (save-window-excursion
      (switch-to-buffer-other-window tpl-menu-buffer)
      (erase-buffer)
      ;; Build the menu
      (tpl-unscan template)
      ;; Size the window
      (goto-char (point-max))
      (setq stop (point))
      (goto-char (point-min))
      (setq start (point))
      (setq size (1+ (count-lines start stop)))
      (setq size (max size window-min-height))
      (if (< size (window-height))
	  (shrink-window (- (window-height) size)))
      ;; Allow user to view and select
      (setq choice (menu-mode menu-title nil))
      (set-buffer save-buffer)
      (delete-windows-on tpl-menu-buffer))
    (bury-buffer tpl-menu-buffer)
    (delete-char (length display-string))
    ;; Insert choice as template or string
    (if choice
	(progn
	  (setq choice-list 
                (tpl-parse-choice (string-trim soar-whitespace choice)))
	  (setq choice-template (nth 1 choice-list))
	  (if choice-template
	      (tpl-insert-template choice-template)
	    ;; else insert the raw string selection
	    (insert-before-markers choice)))
        ;; else insert placeholder
        (progn
	  (setq display-string (concat tpl-begin-placeholder
                                       (tpl-token-name template)
				       tpl-end-placeholder))
          (insert-before-markers display-string)
          (backward-char (length display-string))
          (error "Quit.")) )     )  )



;;;
;;;	IV. 	Menu-set-local-keys
;;;
;;;  added that - gets searched for too.
;;;

(defun menu-set-local-keys ()
  "Create key-map for Menu Mode."
  (setq menu-mode-map (make-keymap))
  (suppress-keymap menu-mode-map)
  (define-key menu-mode-map "\C-g" 'menu-abort-recursive-edit)
  (define-key menu-mode-map "\C-j" 'menu-exit-recursive-edit) ; LFD
  (define-key menu-mode-map "\C-m" 'menu-exit-recursive-edit) ; CR
  (define-key menu-mode-map "\e\C-c" 'menu-exit-recursive-edit)
  (define-key menu-mode-map " "     'menu-next-item)
  (define-key menu-mode-map "\C-n" 'menu-next-item)
  (define-key menu-mode-map "\C-p" 'menu-previous-item)
  (define-key menu-mode-map "\177" 'menu-previous-item) ; DEL
  (define-key menu-mode-map "?" 'describe-mode)
  (define-key menu-mode-map "-" 'menu-next-match)
  (define-key menu-mode-map "a" 'menu-next-match)
  (define-key menu-mode-map "b" 'menu-next-match)
  (define-key menu-mode-map "c" 'menu-next-match)
  (define-key menu-mode-map "d" 'menu-next-match)
  (define-key menu-mode-map "e" 'menu-next-match)
  (define-key menu-mode-map "f" 'menu-next-match)
  (define-key menu-mode-map "g" 'menu-next-match)
  (define-key menu-mode-map "h" 'menu-next-match)
  (define-key menu-mode-map "i" 'menu-next-match)
  (define-key menu-mode-map "j" 'menu-next-match)
  (define-key menu-mode-map "k" 'menu-next-match)
  (define-key menu-mode-map "l" 'menu-next-match)
  (define-key menu-mode-map "m" 'menu-next-match)
  (define-key menu-mode-map "n" 'menu-next-match)
  (define-key menu-mode-map "o" 'menu-next-match)
  (define-key menu-mode-map "p" 'menu-next-match)
  (define-key menu-mode-map "q" 'menu-next-match)
  (define-key menu-mode-map "r" 'menu-next-match)
  (define-key menu-mode-map "s" 'menu-next-match)
  (define-key menu-mode-map "t" 'menu-next-match)
  (define-key menu-mode-map "u" 'menu-next-match)
  (define-key menu-mode-map "v" 'menu-next-match)
  (define-key menu-mode-map "w" 'menu-next-match)
  (define-key menu-mode-map "x" 'menu-next-match)
  (define-key menu-mode-map "y" 'menu-next-match)
  (define-key menu-mode-map "z" 'menu-next-match)
)



;;;
;;;  	V. 	Menu-help-message
;;;
;;; Provide a help message when called.
;;;
;
;(defconst menu-mode-next-item-help " SPC,^N next item;" 
;   "How to go forward")
;
;(defconst menu-mode-previous-item-help " ^P back;"
;   "Help on how to go back")
;   ;; can we assume users can figure it out.
;
;(defconst menu-mode-auto-match-help " a-Z searchs;" 
;   "Help on auto search")
;
;(defconst menu-mode-return-help " CR selects;"
;   "How to return a selection")
;
;(defconst menu-mode-help-line
;    (substitute-command-keys 
;    (concat menu-mode-help-intro menu-mode-next-item-help 
;            menu-mode-previous-item-help
;            menu-mode-auto-match-help 
;            menu-mode-return-help 
;            menu-mode-quit-help)))

;;;
;;;	VI.     Expand-placeholders-in-region
;;;

(setq tpl-expand-optional-after-first-query t)
(defvar tpl-pattern-required "!")
(defvar tpl-begin-required (concat tpl-begin-placeholder
				     tpl-pattern-required))

(defun expand-placeholders-in-region (start stop)
  "Expand each placeholder in the region between START and STOP."
 (interactive "r")
 (let (stop-marker save)
   (goto-char start)
   (setq stop-marker (make-marker))
   (set-marker stop-marker stop)
   ;; (The check for out-of-bounds is needed for a placeholder at
   ;;   the end of the region.)
   (while (and (< (point) (marker-position stop-marker))
               (re-search-forward
		 tpl-pattern-placeholder (marker-position stop-marker) t))
     (re-search-backward tpl-pattern-placeholder)
     (cond ((looking-at tpl-begin-optional)
            (expand-optional-placeholder))
           ((looking-at tpl-begin-required)
            (tpl-expand-required-placeholder))
           ((or (< tpl-expansion-depth tpl-ask-expansion-depth)
		(tpl-y-or-n-p "Expand? "))
            (setq tpl-expansion-depth (1+ tpl-expansion-depth))
	    (unwind-protect
	       (tpl-expand-placeholder (marker-position stop-marker))
	       (setq tpl-expansion-depth (1- tpl-expansion-depth))) )
           (t (re-search-forward tpl-pattern-placeholder)))
      ) ; while (re-search-forward...)
    (if (< (point) (marker-position stop-marker))
	(goto-char (marker-position stop-marker)))
    (set-marker stop-marker nil)) )

(defun expand-optional-placeholder ()
  "expand an optional placeholder appropriately"
 (if (or (equal t tpl-keep-optional-placeholders)
         (and tpl-keep-optional-placeholders
              (tpl-y-or-n-p "Keep optional placeholder? ")))
     (progn
      (delete-char length-tpl-begin-optional)
      (insert-before-markers tpl-begin-placeholder)
      (re-search-backward tpl-begin-placeholder)
      (if (or (< tpl-expansion-depth tpl-ask-expansion-depth)
              tpl-expand-optional-after-first-query
      	      (tpl-y-or-n-p "Expand? "))
          (progn (setq tpl-expansion-depth (1+ tpl-expansion-depth))
                 (unwind-protect
                    (tpl-expand-placeholder (marker-position stop-marker))
                    (setq tpl-expansion-depth (1- tpl-expansion-depth))))
          ;; else
          (re-search-forward tpl-pattern-placeholder)))
    ;; else
    (progn (setq save (point))
         (re-search-forward tpl-pattern-placeholder)
         (delete-region save (point))
         (if (tpl-blank-line)
             (delete-blank-lines)))) )

(defun tpl-expand-required-placeholder ()
  "expand an required placeholder appropriately"
  ;; is this stop marker a kludge, or will it work? -fer 29/nov/90
  (let (stop-marker)
    (setq stop-marker (make-marker))
    (set-marker stop-marker stop)
  (delete-char (length tpl-begin-required))
  (insert-before-markers tpl-begin-placeholder)
  (re-search-backward tpl-begin-placeholder)
  (setq tpl-expansion-depth (1+ tpl-expansion-depth))
  (unwind-protect                            
    (tpl-expand-placeholder (marker-position stop-marker))
    (setq tpl-expansion-depth (1- tpl-expansion-depth)))))

;;;
;;;	VII.	 Tpl-initialize-scan
;;;

(defun tpl-initialize-scan ()
  "Initialize environment for scan."
  ;; Make all characters non-symbols
  (tpl-fix-syntax tpl-begin-placeholder)
  (tpl-fix-syntax tpl-end-placeholder)
  (tpl-fix-syntax tpl-sep-placeholder)
  (tpl-fix-syntax tpl-pattern-optional)
  ;; Build composite patterns
  (setq tpl-begin-optional 
        (concat tpl-begin-placeholder tpl-pattern-optional))
  (setq length-tpl-begin-optional
        (+ length-tpl-begin-placeholder length-tpl-pattern-optional))
  (setq tpl-destination-placeholder 
        (concat tpl-begin-placeholder tpl-destination-symbol
		tpl-end-placeholder))
  (setq tpl-pattern-placeholder
        (concat tpl-begin-placeholder  "\\(" tpl-pattern-optional "\\)?"
                ; also allow matching to required items
                "\\(" tpl-pattern-required "\\)?"
		tpl-pattern-symbol
                "\\(" tpl-sep-placeholder tpl-pattern-symbol "\\)?"
		tpl-end-placeholder))
  ;; Build lexical patterns
  (setq lex-patterns
        (list
	   (tpl-make-pattern tpl-placeholder-type tpl-pattern-placeholder)
	   (tpl-make-pattern tpl-whitespace-type tpl-pattern-whitespace)
	   (tpl-make-pattern tpl-word-type tpl-pattern-word)
	   (tpl-make-pattern tpl-punctuation-type tpl-pattern-punctuation)
	   (tpl-make-pattern tpl-other-type tpl-pattern-other)	   ))
  (setq string-patterns
	  (list  (tpl-make-pattern tpl-string-type tpl-pattern-string) ))
  (setq tpl-newline-token
	  (tpl-make-token tpl-terminal-type tpl-newline-type nil))  )


;;;
;;;	VIII.	Symbol-mode
;;;
;;; now uses syntax table so you can avoid matching ==> and <==.
;;;

(defun symbol-mode (&optional prompt)
  "Major mode for inserting symbols in place.  Most editing commands
still work except that newline or return will exit and the symbol will be
inserted.  The following commands are available:
  TAB  		attempt to auto-complete last identifier typed
  RETURN	exit normally, inserting the symbol"
  ;; Meant to be called from sym-read-string, which initializes.
  (interactive)
  (or symbol-mode-map
     (sym-set-local-keys))
  (setq mode-line-format prompt)
  (use-local-map symbol-mode-map)
  (setq major-mode 'symbol-mode)
  (make-symbol-mode-syntax-table)
  (setq mode-name "Symbol") )



;;;
;;;	IX. 	Strip-leading-whitespace
;;;
;;;  Strips leading spaces off what is passed
;;;  This can go aqay if unused after 1 sept. 90
	
(defun strip-leading-whitespace (item)
  "strip the whitespace in front of ITEM"
  (let ( (first-blank  (string-match tpl-pattern-whitespace item)) )
  (cond ( (and first-blank
               (= 0 first-blank))
          (substring item (match-end 0) nil))
        ( t item))))



;;;
;;; 	X.	Menu-next-match
;;;
;;; Modified to accept whitespace in front of items

(defvar menu-next-match-leader "^[ ]*"
  "allowable leading text on menu items.")

;; this has a problem with searching for the first thing on the menu
;; when over the first thing

(defun menu-next-match ()
  "Find the next item with last-input-char leading character."
  (interactive)
  (let (stop search-string)
    (setq menu-last-input
	  (concat menu-last-input (char-to-string last-input-char)))
    (setq search-string (concat menu-next-match-leader menu-last-input))
    (beginning-of-line)
    (cond ((not (re-search-forward search-string nil t))
           (setq stop (point))
           (beginning-of-buffer)
           (if (not (re-search-forward search-string stop stop))
               (progn (message "No match for that character!")
                      (ding)
                      (setq menu-last-input "")))))    ))



;;;
;;;	XI.	Tpl-insert-selection
;;;
;;;  Small change handles spaces in front correctly.
;;;

;(defun tpl-insert-selection (template)
;  "Insert a template at point using the selection type TEMPLATE."
;  (let (save-buffer start stop size choice choice-template choice-list
;		    display-string)
;    ;; Highlight placeholder
;    (setq display-string (concat tpl-display-begin (tpl-token-name template)
;                                 tpl-display-end))
;    (insert-before-markers display-string)
;    (backward-char (length display-string))
;    ;; Prepare menu buffer
;    (save-window-excursion
;      (setq save-buffer (buffer-name))
;      (switch-to-buffer-other-window tpl-menu-buffer)
;      (erase-buffer)
;      ;; Build the menu
;      (tpl-unscan template)
;      ;; Size the window
;      (goto-char (point-max))
;      (setq stop (point))
;      (goto-char (point-min))
;      (setq start (point))
;      (setq size (1+ (count-lines start stop)))
;      (setq size (max size window-min-height))
;      (if (< size (window-height))
;	  (shrink-window (- (window-height) size)))
;      ;; Allow user to view and select
;      (setq choice (menu-mode))
;      (set-buffer save-buffer)
;      (delete-windows-on tpl-menu-buffer))
;    (bury-buffer tpl-menu-buffer)
;    (delete-char (length display-string))
;    ;; Insert choice as template or string
;    (if choice
;	(progn
;	  (setq choice-list 
;                 (tpl-parse-choice (string-trim soar-whitespace choice)))
;	  (setq choice-template (nth 1 choice-list))
;	  (if choice-template
;	      (tpl-insert-template choice-template)
;	    ; else
;	    (insert-before-markers choice)))
;         ;; else insert placeholder
;         (progn
;           (setq display-string 
;                 (concat tpl-begin-placeholder (tpl-token-name template)
;                         tpl-end-placeholder))
;           (insert-before-markers display-string)
;           (backward-char (length display-string))
;           (error "Quit."))  )    )  )


;;;
;;;	XII.	Make-sub-selection-template
;;;
;;; Make selection templates of the major clauses in template with
;;; name NAME.  This should be called once at the end of each release
;;; in a buffer that already has soar-mode loaded.

(defun make-sub-selection-templates ()  ; (make-sub-selection-templates)
 "Create a file of sub-clause selection templates for major templates in taql
mode."
 (interactive)
   (let ( (top-level-templates 
          '("propose-space" "propose-initial-state" "propose-task-state"
            "propose-task-operator" "propose-operator" "prefer"
            "compare" "operator-control" "apply-operator" "result-superstate"
            "propose-superobjects" "goal-test-group"
            "evaluate-object" "evaluation-properties"
            "augment"))
          (work-buffer (get-buffer-create "*taql-work*"))  )
   (set-buffer work-buffer)
   (taql-mode -1)
   (taql-mode)
   (auto-fill-off-mode)
   (erase-buffer)
   (insert ";;; File of taql-sub-clauses.\n\n") 
   (while top-level-templates
      (insert-selection-template (pop top-level-templates)))
 ))

(defun insert-selection-template (name)
 "Insert a selection template form of the major clauses in template with
name NAME."
 (let* ( (token (tpl-find-template name))
         (start-column (current-column))
         (save-hook auto-fill-hook)
         (line-list (tpl-token-value token))
         (begin-template (point))
         in-template
         token-list line    )
   (setq auto-fill-hook nil)
   ;; Unscan template
   (if token
       (progn (beginning-of-line nil)
              (insert-before-markers "\nTemplate ")
              (insert-before-markers name)
              (insert-before-markers tpl-sub-selection-name)
              (insert-before-markers " Selection\n")
              (insert-before-markers ":begin\n")))
   (while line-list
     (setq line (pop line-list))
     (setq token-list (tpl-line-tokens line))
     (setq in-template nil)
     (while token-list
       (setq token (pop token-list))
       (cond (in-template 
              (if (string= tpl-end-placeholder (tpl-token-value token))
                  (setq in-template nil)
                (cond ( (member (tpl-token-name token) '(placeholder word other))
                        (insert-before-markers (tpl-token-value token))
                        ;(tpl-token-selection-name token) ;; seems to not
                        ; have placeholders anymore, 6-18-91 - fer
                  ))))
             ((not in-template)
              (if (string= tpl-begin-placeholder (tpl-token-value token))
                  (setq in-template t))) ))
     (if line-list
         (newline)) )
   (insert-before-markers ":end\n")
   ;; Reset auto-fill-hook
   (setq auto-fill-hook save-hook) ))

(defun tpl-token-selection-name (token)
 "name to print for a token for inclusion in a selection menu."
 (concat (tpl-token-name (tpl-parse-placeholder (tpl-token-value token)))
         ":"))

;;;
;;;	XIII.		Tpl-y-or-n-p 
;;;
;;; Also takes space as yes and DEL as no.
;;;

(defvar tpl-y-or-n-p-help "(y [SPC] or n [DEL]) ")

(defun tpl-y-or-n-p (msg)
  "Display MSG and await positive ('y') or negative ('n') response.
Differs from 'y-or-n-p' in that it leaves the cursor in the active
window, rather than moving to the mode-line."
  (let (answered prompt reply result)
    (setq answered nil)  
    (setq prompt (concat msg tpl-y-or-n-p-help))
    (while (not answered)
      (message prompt)
      (setq reply (read-char))
      (cond
       ((char-equal reply ?y)
	(setq answered t)
	(setq result t))
       ((char-equal reply ? )
	(setq answered t)
	(setq result t))
       ((char-equal reply ?n)
	(setq answered t)
	(setq result nil))
       ((char-equal reply 12)  ;^L
        (recenter))
       ((or (char-equal reply ?\13)   ; this is CR
            (char-equal reply ?\177)) ; this is DEL
	(setq answered t)
	(setq result nil))
       (t (ding)
          (setq prompt (concat "Please respond 'y' or 'n'.  "
		                msg tpl-y-or-n-p-help)))))
    result ))


;;;
;;;	XIV.	Tpl-insert-repetition
;;;
;;; modified to allow for required expansion

(defun tpl-insert-repetition (template)
  "Insert at point instances of the repetition type TEMPLATE."
  (let (start template-name column)
    (setq start (point))
    (setq column (current-column))
    (setq template-name (tpl-token-name template))
					; Insert first instance
    (tpl-unscan template)
    (re-search-backward tpl-pattern-placeholder)
    (delete-region start (point))
    (cond  ((looking-at tpl-begin-required)
            (tpl-expand-required-placeholder))
           (t (tpl-expand-placeholder nil))  )
					; Insert more instances
    (while (tpl-y-or-n-p (concat "More instances of " template-name "?"))
    (tpl-unscan template column)
    (cond ((> tpl-ask-expansion-depth 0)
	   (re-search-backward tpl-pattern-placeholder)
           (cond  ((looking-at tpl-begin-required)
                   (tpl-expand-required-placeholder))
                  (t (tpl-expand-placeholder nil))  ))))))



;;;
;;;	XV.	tpl-build-template-list
;;;
;;; slightly quieter version, no note if templates not found
;;; allows you to pass in the mode to use, e.g. a minor mode name

(defun tpl-build-template-list (&optional mode-to-use)
  "Build template-list, using mode-passed in or current major mode."
  (let (mode-entry template-list)
    (setq tpl-local-template-list
	  (list (tpl-mode-templates
		 (tpl-mode-match 'generic tpl-global-template-list))))
    ;; Use loaded templates if available
    (setq template-list
	  (tpl-mode-templates
	   (tpl-mode-match (or mode-to-use major-mode) tpl-global-template-list)))
    (if template-list
	(setq tpl-local-template-list
	      (cons template-list tpl-local-template-list))
      ;; else
      (progn
	(setq mode-entry (tpl-mode-match major-mode tpl-auto-template-alist))
	(if mode-entry
            (load-tpl-library (tpl-mode-file mode-entry) major-mode)
	  ; else  -fer no message if going into taql mode
	  (or taql-mode (message "No templates found for this mode.")))))
    (if tpl-rebuild-all-templates-template
	(tpl-make-all-templates-template))    ) )


;;;
;;;	XVI.	sym-set-local-keys
;;;
;;; set so that esc-/ does auto completion correctly in symbol mode.
;;;

(defun sym-set-local-keys ()
  "Create key-map for Symbol Mode."
  (let (empty-keymap)
    (setq empty-keymap (make-keymap))
    (setq symbol-mode-map empty-keymap)
    (suppress-keymap symbol-mode-map t)

  (define-key symbol-mode-map "\177" 'sym-delete-backward-char) ; DEL

  ;; Control keys
  (define-key symbol-mode-map "\C-a" 'sym-position-start)
  (define-key symbol-mode-map "\C-b" 'sym-backward-char)
					; ^C is a prefix
  (define-key symbol-mode-map "\C-d" 'sym-delete-char)
  (define-key symbol-mode-map "\C-e" 'sym-position-end)
  (define-key symbol-mode-map "\C-f" 'sym-forward-char)
  (define-key symbol-mode-map "\C-g" 'sym-abort-recursive-edit)
					; ^H is okay
  (define-key symbol-mode-map "\C-i" 'symbol-dabbrev-expand) ; TAB
  (define-key symbol-mode-map "\C-j" 'exit-recursive-edit) ; LFD
  (define-key symbol-mode-map "\C-k" 'sym-kill-line)
					; ^L is okay
  (define-key symbol-mode-map "\C-m" 'exit-recursive-edit) ; CR
					; ^N is okay
  (define-key symbol-mode-map "\C-o" 'sym-illegal-command)
					; ^P is okay
  (define-key symbol-mode-map "\C-q" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-r" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-s" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-t" 'sym-transpose-chars)


  (define-key symbol-mode-map "\C-u" 'sym-illegal-command)
					; ^V is okay
  (define-key symbol-mode-map "\C-w" 'sym-illegal-command)
					; ^X is a prefix
  (define-key symbol-mode-map "\C-y" 'sym-illegal-command)
					; ^Z is okay

					; ^X prefix keys
  (define-key symbol-mode-map "\C-x\t" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-x\C-l" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-x\C-o" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-x\C-t" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-x\C-u" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-xg" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-xi" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-xk" 'sym-illegal-command)
  (define-key symbol-mode-map "\C-x\177" 'sym-illegal-command) ; ^X DEL

					; ESC prefix keys
  (define-key symbol-mode-map "\e\C-b" 'sym-backward-word)
  (define-key symbol-mode-map "\e\C-c" 'exit-recursive-edit)
  (define-key symbol-mode-map "\e\C-f" 'sym-forward-word)
  (define-key symbol-mode-map "\e\C-k" 'sym-illegal-command)
  (define-key symbol-mode-map "\e\C-o" 'sym-illegal-command)
  (define-key symbol-mode-map "\e\C-s" 'sym-illegal-command)
  (define-key symbol-mode-map "\e\C-t" 'sym-illegal-command)
  (define-key symbol-mode-map "\e\C-w" 'sym-illegal-command)
  (define-key symbol-mode-map "\e\C-\\" 'sym-illegal-command)
  (define-key symbol-mode-map "\e/"  'symbol-dabbrev-expand) ; same as TAB
  (define-key symbol-mode-map "\e " 'sym-illegal-command) ; ESC SPACE
  (define-key symbol-mode-map "\e%" 'sym-illegal-command)
  (define-key symbol-mode-map "\e(" 'sym-illegal-command)
  (define-key symbol-mode-map "\e)" 'sym-illegal-command)
  (define-key symbol-mode-map "\e;" 'sym-illegal-command)
  (define-key symbol-mode-map "\e<" 'sym-position-start)
  (define-key symbol-mode-map "\e>" 'sym-position-end)
  (define-key symbol-mode-map "\e^" 'sym-illegal-command)
  (define-key symbol-mode-map "\eb" 'sym-backward-word)
  (define-key symbol-mode-map "\ed" 'sym-kill-word)
  (define-key symbol-mode-map "\ef" 'sym-forward-word)
  (define-key symbol-mode-map "\eg" 'sym-illegal-command)
  (define-key symbol-mode-map "\ei" 'sym-illegal-command)
  (define-key symbol-mode-map "\ej" 'sym-illegal-command)
  (define-key symbol-mode-map "\ek" 'sym-illegal-command)
  (define-key symbol-mode-map "\eq" 'sym-illegal-command)
  (define-key symbol-mode-map "\et" 'sym-illegal-command)
  (define-key symbol-mode-map "\ey" 'sym-illegal-command)
  (define-key symbol-mode-map "\ez" 'sym-illegal-command)
  (define-key symbol-mode-map "\e\177" 'sym-backward-kill-word) ; ESC DEL

					; Self-inserting keys

  (define-key symbol-mode-map " " 'sym-self-insert-command) ; SPACE
  (define-key symbol-mode-map "!" 'sym-self-insert-command)
  (define-key symbol-mode-map "\042" 'sym-self-insert-command) ; DOUBLE QUOTE
  (define-key symbol-mode-map "#" 'sym-self-insert-command)
  (define-key symbol-mode-map "$" 'sym-self-insert-command)
  (define-key symbol-mode-map "%" 'sym-self-insert-command)
  (define-key symbol-mode-map "&" 'sym-self-insert-command)
  (define-key symbol-mode-map "\047" 'sym-self-insert-command) ; SINGLE QUOTE
  (define-key symbol-mode-map "(" 'sym-self-insert-command)
  (define-key symbol-mode-map ")" 'sym-self-insert-command)
  (define-key symbol-mode-map "*" 'sym-self-insert-command)
  (define-key symbol-mode-map "+" 'sym-self-insert-command)
  (define-key symbol-mode-map "," 'sym-self-insert-command)
  (define-key symbol-mode-map "-" 'sym-self-insert-command)
  (define-key symbol-mode-map "." 'sym-self-insert-command)
  (define-key symbol-mode-map "/" 'sym-self-insert-command)

  (define-key symbol-mode-map "0" 'sym-self-insert-command)
  (define-key symbol-mode-map "1" 'sym-self-insert-command)
  (define-key symbol-mode-map "2" 'sym-self-insert-command)
  (define-key symbol-mode-map "3" 'sym-self-insert-command)
  (define-key symbol-mode-map "4" 'sym-self-insert-command)
  (define-key symbol-mode-map "5" 'sym-self-insert-command)
  (define-key symbol-mode-map "6" 'sym-self-insert-command)
  (define-key symbol-mode-map "7" 'sym-self-insert-command)
  (define-key symbol-mode-map "8" 'sym-self-insert-command)
  (define-key symbol-mode-map "9" 'sym-self-insert-command)

  (define-key symbol-mode-map ":" 'sym-self-insert-command)
  (define-key symbol-mode-map ";" 'sym-self-insert-command)
  (define-key symbol-mode-map "<" 'sym-self-insert-command)
  (define-key symbol-mode-map "=" 'sym-self-insert-command)
  (define-key symbol-mode-map ">" 'sym-self-insert-command)
  (define-key symbol-mode-map "?" 'sym-self-insert-command)
  (define-key symbol-mode-map "@" 'sym-self-insert-command)

  (define-key symbol-mode-map "A" 'sym-self-insert-command)
  (define-key symbol-mode-map "B" 'sym-self-insert-command)
  (define-key symbol-mode-map "C" 'sym-self-insert-command)
  (define-key symbol-mode-map "D" 'sym-self-insert-command)
  (define-key symbol-mode-map "E" 'sym-self-insert-command)
  (define-key symbol-mode-map "F" 'sym-self-insert-command)
  (define-key symbol-mode-map "G" 'sym-self-insert-command)
  (define-key symbol-mode-map "H" 'sym-self-insert-command)
  (define-key symbol-mode-map "I" 'sym-self-insert-command)
  (define-key symbol-mode-map "J" 'sym-self-insert-command)
  (define-key symbol-mode-map "K" 'sym-self-insert-command)
  (define-key symbol-mode-map "L" 'sym-self-insert-command)
  (define-key symbol-mode-map "M" 'sym-self-insert-command)
  (define-key symbol-mode-map "N" 'sym-self-insert-command)
  (define-key symbol-mode-map "O" 'sym-self-insert-command)
  (define-key symbol-mode-map "P" 'sym-self-insert-command)
  (define-key symbol-mode-map "Q" 'sym-self-insert-command)
  (define-key symbol-mode-map "R" 'sym-self-insert-command)
  (define-key symbol-mode-map "S" 'sym-self-insert-command)
  (define-key symbol-mode-map "T" 'sym-self-insert-command)
  (define-key symbol-mode-map "U" 'sym-self-insert-command)
  (define-key symbol-mode-map "V" 'sym-self-insert-command)
  (define-key symbol-mode-map "W" 'sym-self-insert-command)
  (define-key symbol-mode-map "X" 'sym-self-insert-command)
  (define-key symbol-mode-map "Y" 'sym-self-insert-command)
  (define-key symbol-mode-map "Z" 'sym-self-insert-command)

  (define-key symbol-mode-map "[" 'sym-self-insert-command)
  (define-key symbol-mode-map "\134" 'sym-self-insert-command) ; BACKSLASH
  (define-key symbol-mode-map "]" 'sym-self-insert-command)
  (define-key symbol-mode-map "^" 'sym-self-insert-command)
  (define-key symbol-mode-map "_" 'sym-self-insert-command)
  (define-key symbol-mode-map "`" 'sym-self-insert-command)

  (define-key symbol-mode-map "a" 'sym-self-insert-command)
  (define-key symbol-mode-map "b" 'sym-self-insert-command)
  (define-key symbol-mode-map "c" 'sym-self-insert-command)
  (define-key symbol-mode-map "d" 'sym-self-insert-command)
  (define-key symbol-mode-map "e" 'sym-self-insert-command)
  (define-key symbol-mode-map "f" 'sym-self-insert-command)
  (define-key symbol-mode-map "g" 'sym-self-insert-command)
  (define-key symbol-mode-map "h" 'sym-self-insert-command)
  (define-key symbol-mode-map "i" 'sym-self-insert-command)
  (define-key symbol-mode-map "j" 'sym-self-insert-command)
  (define-key symbol-mode-map "k" 'sym-self-insert-command)
  (define-key symbol-mode-map "l" 'sym-self-insert-command)
  (define-key symbol-mode-map "m" 'sym-self-insert-command)
  (define-key symbol-mode-map "n" 'sym-self-insert-command)
  (define-key symbol-mode-map "o" 'sym-self-insert-command)
  (define-key symbol-mode-map "p" 'sym-self-insert-command)
  (define-key symbol-mode-map "q" 'sym-self-insert-command)
  (define-key symbol-mode-map "r" 'sym-self-insert-command)
  (define-key symbol-mode-map "s" 'sym-self-insert-command)
  (define-key symbol-mode-map "t" 'sym-self-insert-command)
  (define-key symbol-mode-map "u" 'sym-self-insert-command)
  (define-key symbol-mode-map "v" 'sym-self-insert-command)
  (define-key symbol-mode-map "w" 'sym-self-insert-command)
  (define-key symbol-mode-map "x" 'sym-self-insert-command)
  (define-key symbol-mode-map "y" 'sym-self-insert-command)
  (define-key symbol-mode-map "z" 'sym-self-insert-command)

  (define-key symbol-mode-map "{" 'sym-self-insert-command)
  (define-key symbol-mode-map "|" 'sym-self-insert-command)
  (define-key symbol-mode-map "}" 'sym-self-insert-command)
  (define-key symbol-mode-map "~" 'sym-self-insert-command)

  ) ; let
)


;;;
;;;	XVII.	make-symbol-mode-syntax-table
;;;

;; set up new syntax table
(defvar symbol-mode-syntax-table nil)

(defun make-symbol-mode-syntax-table ()
  (if symbol-mode-syntax-table
      nil
      (progn
        (setq symbol-mode-syntax-table
             (make-syntax-table lisp-mode-syntax-table))
        (modify-syntax-entry ?* " " symbol-mode-syntax-table)
        ;; not all lisp modes set these
       ;; treating them as delimiters could get screwed up with
       ;;  preferences, we'll 
       ;; have to assume that users can keep track of variable names
        (modify-syntax-entry ?< "." symbol-mode-syntax-table)
        (modify-syntax-entry ?> "." symbol-mode-syntax-table)
        ;; . is now a seperator for block names in taql 3.14
        (modify-syntax-entry ?. "." symbol-mode-syntax-table)
        ;;(modify-syntax-entry ?= " " symbol-mode-syntax-table)
        ;; use the table so the mods go into symbol mode table only
        (set-syntax-table symbol-mode-syntax-table)
        (tpl-initialize-scan)))
  (set-syntax-table symbol-mode-syntax-table) ;; use a table you can control
)



;;;
;;;	XVIII.	symbol-dabbrev-expand
;;;
;;; Expand dabbrev in a way that avoids matching ==> and <==.
;;;

(defun symbol-dabbrev-expand (&optional arg)
  "Call dabbrev-expand for symbol-mode."
  (interactive)
  (let ( (dabbrevs-min-limit 3)   ;dynamically bound!, size of prompt
         (dabbrevs-insert-before t) )
  (dabbrev-expand arg)))


;;;
;;;     XVIV.	tpl-unscan
;;;
;;; Added the ability to return a list of strings used.
;;;

(defun tpl-unscan (token &optional column)
  "Insert at point the values of tokens in the tree rooted by TOKEN.
Optional second argument COLUMN specifies where to indent rigidly.
Default is the current column.
If column is -1, then just return a list of the strings."
  (let (begin-template start-column token-list line-list line
	real-column-p
	(results nil) (partial-results nil)
	(commented nil)
        save-hook)
    ;; Save auto-fill-hook and reset
    (setq save-hook auto-fill-hook)
    (if (not tpl-fill-while-unscanning)
	(setq auto-fill-hook nil))
    ;; Unscan template
    (setq begin-template (point))
    (if column
	(setq start-column column)
      ;; else
      (setq start-column (current-column)))
    (setq real-column-p (>= start-column 0))
    (setq line-list (tpl-token-value token))
    ;; loop over each line
    (while line-list
      (setq partial-results nil)
      (setq commented nil)
      (setq line (car line-list))
      (setq line-list (cdr line-list))
      (if real-column-p
	  (if (= tpl-comment-level (tpl-line-indent line))
	      (indent-to comment-column)
              ; else
	    (indent-to (+ start-column (tpl-line-indent line)))))
      (setq token-list (tpl-line-tokens line))
      ;; loop over each part of the line
      (while token-list
	(setq token (car token-list))
	(setq token-list (cdr token-list))
	;(debug "tpl-unscan token:" token)
	(if real-column-p
	    (insert-before-markers (tpl-token-value token))
          (if (and (not commented)
		   (member (tpl-token-name token) '(word other)))
              (push (tpl-token-value token) partial-results)
	    (if (member (tpl-token-name token) '(punctuation))
		(setq commented t))))
	  )
      ;; done with line
      (if (and real-column-p line-list)
	  (newline))
      (if (not real-column-p)
	  (push (mapconcat '(lambda(x) x) (reverse partial-results) "")
		results))
      ) ;done with while line-list
    (if (and (boundp 'template-unscan-hook)
	     template-unscan-hook)
	(funcall template-unscan-hook begin-template (point) start-column))
    ;; Reset auto-fill-hook
    (setq auto-fill-hook save-hook)
    (if (not real-column-p)
	(reverse results)))   )


;;;
;;;	XX.	tpl-fix-syntax
;;;

(defun tpl-fix-syntax (string &optional new-type)
  "Change syntax of chars in STRING from (word/symbol/quote) to punctuation."
  (let (char)
    (while (> (length string) 0)
      (setq char (string-to-char string))
      (setq string (substring string 1))
      (if (or (equal (char-syntax char) ? )
	      (equal (char-syntax char) ?_)
	      (equal (char-syntax char) ?'))
	  (modify-syntax-entry char
                               (or new-type
                                   ".   "))      )    )  ))


(message "Taql-mode is great.")
