;;;; -*- Mode: Emacs-Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; 
;;;; File            : taql.el|2.1/
;;;; Author          : Frank Ritter
;;;; Created On      : Fri Jul  6 14:37:53 1990
;;;; Last Modified By: Frank Ritter
;;;; Last Modified On: Thu Sep 26 15:20:32 1991
;;;; Update Count    : 229
;;;; 
;;; TAQL stuff for GNU Emacs   
;;; Copyright (C) 1989         Frank Ritter    ritter@cs.cmu.edu
;;; 


;;;
;;;                   Table of Contents
;;; 
;;;	i.	Disclaimer
;;;	ii.	Abstract
;;;	iii.	.emacs Initializations
;;;	iv.	Initialization: variables
;;;	iv.	Initialization: loads
;;;	vi.	Initialization: Key bindings
;;;
;;;	I.	taql-insert-construct
;;;	IB.	taql-add-clause
;;;	IC.	taql-fixup-construct
;;;	II.	Helper functions
;;;	III.	Help for TAQL constructs
;;;	IV.	Taql-mode (sets up a sub-mode)
;;;	V.	taql-bug (sets up mail message)
;;;
;;;	N-1.	Wrap up (load any patches, run hooks and provide yourself)
;;;	N.	Known bugs and suggestion list


;;;
;;;	i.	Disclaimer
;;;
;;; This program is public domain software;  you can redistribute it and/or 
;;; modify it as you wish.  Many of its parts, however, are covered under the
;;; GNU copyleft agreement.
;;;
;;; This program is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along
;;; with this program; if not, write to the Free Software Foundation, Inc.,
;;; 675 Mass Ave, Cambridge, MA 02139, USA.
;;; 
;;; This package includes stuff for TAQL editing.
;;; It does not assume that soar-mode is available.
;;;; 
;;;; (C) Copyright 1990, Frank Ritter, all rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;
;;;	ii.	Abstract
;;;   
;;;
;;; This file provides stuff to help editing taql.  First off, it allows
;;; menu based insertion of complete templates for all TAQL constructs.
;;; It is currently implemented as a command and also could be bound to a key.
;;;   (C-c C-t, the C-c can be changed to C-6 or C-z)
;;; The user types "M-x taql-insert-construct" and is presented with a menu of
;;; TAQL constructs.  If the point (emacs cursor location) is over a current
;;; template name or clause name, it is presented as the default.  The menu is
;;; implemented as a scrollable buffer, and includes some help comments.  The
;;; user selects the template by positioning the cursor on the template's
;;; name, and hitting <cr>.  The expanded template is inserted in the buffer
;;; and expansion is done through queries and the user typing names in.
;;;   The user can interupt this process at any time by typing ^G, and restart
;;; it with M-x taql-expand-construct, or esc-E.  After expansion is complete,
;;; the user can add additional clauses by M-x taql-add-clause, C-c C-a, and 
;;; cleanup the construct in a limited way (remove extra spaces, reformat) with
;;; M-x taql-fixup-construct, C-c C-f.  
;;;
;;;  Full documentation is available in taql-mode.doc in the manuals directory 
;;; underneath the directory where this file resides.
;;;
;;;    These expansions could be implemented as abbreviations, so advanced users
;;; could access some of them with "^X '".  (There appears to be a bug that
;;; precludes full use of abbreviations with a "-" in them, such as
;;; "propose-space".)  So let me know if you need them.
;;;
;;;================================================================
;;; to do/Bug list: (fixed up for release as announcement only)
;;; * taql-debug statements are not provided, such as taqlp
;;; * Grammar is not complete
;;; * should provide check for duplicate tc names
;;; *  would like last problem space name as default (?how to do, what to do)
;;;    


;;;
;;;	iii.	.emacs Initializations
;;;
;;; In order to use taql.el, you will need to call the commands.
;;; by loading taql-mode-defaults.el, or by  cutting and pasting them
;;; in your .emacs file, and then not calling the taql-mode-defaults.el file.
;;; taql-mode-defaults is also called by soar-mode-defaults.el


;;;
;;;	iv.	Initialization: user variables
;;;  The *'s here indicate that users might want to set thses variables.
;;;  They also point out the variables as such to the help system.
;;;

;; defines the version and the name of the template file to load
(defvar taql-version "taql314"
  "*The version of TAQL used, with numbers run together.")

;; relative pathname off of soar-mode
(defvar taql-ilisp-subdirectory "ilisp/4.11"
  "Subdirectory to taql-mode-home-directory that ilisp lives in.")

(defvar taql-mode-load-hook nil
  "*Hook run at the end of loading this file.  This is not a good place to put
keybindings in your .emacs, use taql-mode-hook.")

(defvar taql-mode-hook nil
  "*Gets called each time taql-mode is.  Good place for keybinding resetting.")

(defvar taql-clause-indention 2
  "*The number of spaces to indent each clause.")

(defvar taql-pull-closing-paren-up t
  "*Bring the closing paren up to the next to last line. Default is t.")

;; This works if soar-mode is loaded first.
(defvar use-soar-mode-if-available t
  "*Put taql-mode buffers into soar-mode too if possible.")

(defvar taql-load-patches t "*Load the latest patches if they exist.")

(defvar taql-mode-home-directory
       "/afs/cs/project/soar/5.2/emacs/taql/2.1"
       "*Where taql-mode and its files live.")

(defvar taql-insert-construct-use-menu t
  "*Use a menu to get the construct name to insert with taql-insert-construct.
Default is t; nil results in query with autocompletion of name.")

(defvar taql-max-column-correction 3 
  "*Don't correct columns out of whack more than this, assume the 
user wants them that way.")

(defvar taql-start-first-column t "*When t (the default), always start
an TC insertion in the first column.")

(defvar taql-cr-after-insert t "*When t (the default), insert a CR
after a tc when inserting it.")


;;;
;;;	v.	Non-user Variables
;;;

(setq tpl-load-path
      (list nil (concat taql-mode-home-directory "/template")))

(defconst taql-insert-template-query
  "TC to insert (? for complete list): ")

(defvar taql-template-names nil
  "List of templates for completing-read in taql-insert-construct.")

;; build menu and abbrev table
(defvar *taql-menu-buffer* nil "where the taql constructs live")

;; the code goes down quite a ways, so  make sure the stack is big enough.
;; If the user has it set to more, fine, don't pull it down.
(cond ( (< max-lisp-eval-depth 400)
        (setq max-lisp-eval-depth 400) ))

(defvar taql-mode nil
   "Non-nil value indicates buffer is in taql mode.  Automatically 
    local to all buffers when taql-mode is called.")
(make-variable-buffer-local 'taql-mode) 

;; add to minor mode alist
(if (not (assq 'taql-mode minor-mode-alist))
    (setq minor-mode-alist
          (cons '(taql-mode " TAQL") minor-mode-alist)))

(defvar taql-bugs-to "soar-bugs@cs.cmu.edu" "Who to send bug reports to.")

(defvar taql-mode-version "2.1" "The version of the taql-mode.")
(defvar taql-syntax-table nil "Syntax table just for taql-mode.")
(defvar pre-taql-map nil "Previous keymap before taql-mode was called.")
(defvar pre-taql-syntax-table 
  "Previous syntax table before taql-mode was called.")

;;;
;;;	vi.	Initialization: Soar-mode variables that taql-mode uses
;;;

(defvar soar-command-prefix "\C-c" "*Prefix sequence for Soar commands.")
(defvar soar-menu-prefix "\C-m" "Prefix for putting up the menu.")

;; We want soar mode to know if taql is here to do taql things
(setq soar-taql-mode-on t)


;;;
;;;	vii.	Initialization: loads
;;;

(setq load-path
      (cons (concat taql-mode-home-directory "/template")
      (cons (concat taql-mode-home-directory "/taql")
      (cons (concat taql-mode-home-directory "/utilities")
      (cons (concat taql-mode-home-directory "/" taql-ilisp-subdirectory)
            ;for new version of taql.el
      (cons (concat taql-mode-home-directory "/ilisp")     
            ;for released versions
            load-path))))))

(require 'cl)

;; this sets up keys for us, ilisp not required...
(require 'soar-ilisp-keymap-changes "soar-ilisp-keymap-changes")

;; this includes string-trim and other misc. funs
(load "soar-misc" nil t)

(require 'goto-manual)
(if (not (boundp 'man-manual-homes))
    (setq man-manual-homes
          (list (concat taql-mode-home-directory "/manuals/")))
    (setq man-manual-homes
          (append man-manual-homes
                  (list (concat taql-mode-home-directory "/manuals/")))))

;; simple menu code
(require 'simple-menu)
(load "taql-simple-menus")

;; until gnu catches up with us!
;(require 'dabbrevs)  is this needed to back up new-dabbrevs?
(require 'new-dabbrevs)


;;; 
;;;	vii.2	Load template mode
;;;

;; gently set the load-path, for safety (set it), gently so users can
;; override it.

(defvar tpl-load-path
      (list (concat taql-mode-home-directory "/template")))

(require 'template)

;; reworks the template mode to be nicer
(load "taql-template")

;; load path has been set in the code...
;;  loads "{taql-version}tpl.el
(load-tpl-library taql-version 'taql-mode)



;;;
;;;	viii.	Initialization: Key bindings
;;;
;;;  We may have trouble finding the mode-map.  Initial user may not have 
;;; a soar-mode map, others may have an old version.
;;;
;;; When you change stuff here, also change 
;;; stuff in soar-mode/../keybindings.txt

(defun taql-set-keys (mode-map)
 "set the keys in map to work taql."
 (cond ((not mode-map)
        (message "Couldn't find mode-map for TAQL functions.")
        (sit-for 3))
       (t  (let ((ilisp-prefix soar-command-prefix))
            ;; here we use dynamic binding
           (ilisp-defkey mode-map "\C-t" 'taql-insert-construct)
           ;; C-SPC C-e is taken by ilisp, esc-e is forw-sent, not useful in lisp.
           (define-key mode-map  "\M-e" 'taql-expand-construct)
           (ilisp-defkey mode-map "\C-a" 'taql-add-clause) 
           ;; C-SPC C-r is taken by ilisp, C-f is not for fix 
           (ilisp-defkey mode-map "\C-f" 'taql-fixup-construct)
           (ilisp-defkey mode-map soar-menu-prefix 'run-soar-menu) ))))


;;; 
;;;	I.	taql-insert-construct 
;;;

(defun taql-insert-construct ()
  "*Insert a taql construct into the current buffer, leaving point
and mark set around the new TC."
  (interactive)
  (if taql-start-first-column (beginning-of-line))
  (if taql-cr-after-insert 
      (progn   (insert CR)  (backward-char 1)))
  (if taql-insert-construct-use-menu
      (tpl-insert-template "taql-construct")
      (tpl-insert-template
        (completing-read taql-insert-template-query
           (taql-get-template-names)
	   nil 'require-match)))
  )

(defun taql-get-template-names ()
  ; chaches its results
  (if taql-template-names
      taql-template-names
    (setq taql-template-names
	  (mapcar '(lambda (x) (list x))
	          (tpl-unscan (tpl-find-template "taql-construct") -1)))))

;;; 
;;;	IB.	taql-add-clause 
;;;

(defun taql-add-clause () "Insert another taql clause into the current
taql construct."  
  (interactive) 
  (let ( (template-name (soar-extract-TC-type)) )
    (if (not (= (current-column) taql-clause-indention))
        (let ( (inset (- taql-clause-indention (1- (current-column)))) )
          (if (> inset 0)
              (progn (beginning-of-line)
                     (insert-spaces (1- inset)))
            (progn (beginning-of-line)
                   (insert-spaces inset)
                   (forward-char taql-clause-indention)))))
  (tpl-insert-template (concat template-name tpl-sub-selection-name)) ))


;;; 
;;;	IC.	taql-fixup-construct 
;;;
;;;  Cleans up one parenthesis problem, and moves :keywords back to indent 
;;;  amount if they are in lesser columns.
;;;

(defvar end-of-tc (make-marker)
  "Remembers where search should end.")

(defun taql-fixup-construct () 
 "Clean up the parenthesis and such in a TC."
  (interactive) 
  (let ((begin-of-tc (make-marker)) (end-of-tc (make-marker)) )
  ;; save your place 
  (save-excursion 
    (end-of-defun)
    (re-search-backward ")" 0 t)
    (set-marker end-of-tc (point))
    (beginning-of-defun) 
    (set-marker begin-of-tc (point)))
  (save-excursion
    (taql-cleanup-parens begin-of-tc end-of-tc)
    (taql-cleanup-keywords begin-of-tc end-of-tc)
    (taql-cleanup-closing-paren begin-of-tc end-of-tc))
  ;; could now go and replace double-spaces with single spaces, but let's 
  ;; leave that as a 'feature', it only happens between arg/value pairs 
  (message "Finished fixing-up TAQL construct.")))

(defun taql-cleanup-parens (begin-of-tc end-of-tc)
  (save-excursion
  (let (cut-object start end)
    ;; clean up each paren @ end of line of tabs and spaces 
    (goto-char begin-of-tc)
    (while (re-search-forward "^[ \t]*)" end-of-tc t) 
      ;;leave alone if on left edge 
      (if (not (= (current-column) 1)) 
          (progn ;; cut it 
            (setq start (- (point) 1)) 
            (setq end (save-excursion (end-of-line) (point))) 
            (setq cut-object (buffer-substring start end)) 
            (beginning-of-line) 
            (delete-region (point) (+ 1 end)) 
            ;; get the cr & paste it on end 
            (forward-line -1)
            (end-of-line) 
            (insert cut-object) )) ))))

(defun taql-cleanup-keywords (begin-of-tc end-of-tc)
  ;; clean up :keywords @ front of line of tabs and spaces 
  (save-excursion 
   (goto-char begin-of-tc)   
   ;; move the end-of-tc pointer to before any edit clauses
   ;; a smarter way to do this would be to do the code after edits too...
   (if (re-search-forward "^[ ]*(edit" end-of-tc t)
       (setq end-of-tc (point)))
   (goto-char begin-of-tc)   
   ;; you can let extra spaces match, cause you won't move unless less than
   ;; desired colmn width.
   (while (re-search-forward "^[ ]*:" end-of-tc t)
     (let ( (inset (- taql-clause-indention (1- (current-column)))) )
     ;; heuristic: dont' change things off from the desired column alot,
     ;; they are part of subclauses (should/could parse clauses to do it right)
       (if (or (> inset taql-max-column-correction)
               (< inset (- taql-max-column-correction)))
           nil
         ;; else do all this
         (beginning-of-line)
         (insert-spaces inset)
         (end-of-line) )))    ))

(defun taql-cleanup-closing-paren (begin-of-tc end-of-tc)
  ;; do the closing paren
  (if taql-pull-closing-paren-up
      (save-excursion
        (goto-char end-of-tc)
        (cond ( (= 0 (current-column)) ;just paren and you
                (forward-char -1)
                (delete-char 1) )))   ))

(defun insert-spaces (n)
  "Insert N spaces."
  (if (> n 0)
     (while (> n 0)
       (insert " ")
       (setq n (1- n)))
     (while (< n 0)
       (delete-char 1)
       (setq n (1+ n)))
   ))


;;; 
;;;	II.	Helper functions 
;;; 
;;;

(defun taql-expand-clause () 
  "*Expand the TC clause you are over."
  (interactive) 
  (cond ((looking-at tpl-begin-placeholder)) 
        (t (re-search-backward tpl-begin-placeholder))) 
  (expand-placeholder))


(defun taql-expand-construct (&optional arg)
  "*Expand the all clauses within the TC you are in.  If ARG, 
expand just the clause you are over."
  (interactive "P") 
  (if arg 
      (taql-expand-clause)
      (let (start)
         (beginning-of-defun)
         (setq start (point))
         (push-mark (point) 'no-display-mark-set)
         (end-of-defun)
         (expand-placeholders-in-region start (point))) ))

(defun soar-extract-TC-type () 
 "return the type of TC the point is in."  
  (save-excursion 
    (beginning-of-defun) 
    (forward-char 1)
    (buffer-substring (point) (progn (forward-sexp) (point)))))


;;; 
;;;	III.	Help for TAQL Constructs 
;;;

;; taql-help provides functions that help can recognize under the existing ^h f 
;; functionality, and which when called insert a template.

;(load "taql-help")


;;;
;;;	IV.	Taql-mode 
;;;
;;; This does the appropriate things when a buffer is setup to edit taql 
;;; things.  Loads the taql items for this buffer, mostly setting up the
;;; template mode without setting template mode on.  Much of the code is
;;; taken from the function template-mode.

(defun taql-mode (&optional arg) 
"Toggle taql mode (a minor mode) on and off.  If an argument is passed,
turn mode on if argument is positive, otherwise turn mode off.
Taql-mode is designed for editing TAQL constructs (TCs) as templates
to be filled in.

There are just a few commands:

\\[run-soar-menu] will provide you with a menu of commands and show
you their keybindings.

\\[run-soar-menu] d will let you read a manual from our fine library
of related Soar and TAQL manuals.

\\[taql-insert-construct] will insert a TC template to fill in.

\\[taql-expand-construct] will expand all the clauses in the TC the cursor is
in.  If given an argument (i.e. preceeded by \\[universal-argument]), it will
only expand the clause the cursor is in, or the nearest clause.

\\[taql-add-clause] will insert into the current TC a clause you select 
from an appropriate menu.

\\[taql-fixup-construct] will fixup the current TC, e.g. fixup the parenthesis.

More information about the major mode taql-mode is running under is available
by typing \\[describe-mode].

More information on taql-mode is available from the manual, either on-line
under the command menu, or by printing out the postscript file in manual
directory.
"
 (interactive "P")
 (cond (arg (if (numberp arg)
                (if (> arg 0)
                    (taql-mode-on)
                    (taql-mode-off))
                (if (listp arg) ; assume it is (4)
                    (taql-mode-on)
                    (taql-mode-off))))
       (t (if taql-mode
             (taql-mode-off)
             (taql-mode-on)))) )


(defun taql-mode-on ()
 "Set up things for TAQL when soar-mode gets called."
 ;; set up soar-mode if available and requested
 (if (and use-soar-mode-if-available
          (fboundp 'soar-mode))
     (soar-mode)
     (if use-lisp-mode-if-available
         (lisp-mode)))
 ;; normally only major modes do this, but we are special
 (setq pre-taql-map (current-local-map))
 (setq pre-taql-syntax-table (syntax-table))
 ;; set up the auto-completion buffer
 (let ( (file-name (buffer-name)) )
   (setq sym-completion-buffer (concat "id-" file-name))
   (if tpl-save-identifier-file
       (find-file-noselect sym-completion-buffer)
       (get-buffer-create sym-completion-buffer))
   (bury-buffer sym-completion-buffer))
 ;; build templates, taken from tplhelper, tpl-build-template-list
 (setq taql-syntax-table (copy-syntax-table (syntax-table)))
 (set-syntax-table taql-syntax-table)
 (tpl-initialize-scan)
 (tpl-build-template-list 'taql-mode)
 (setq tpl-local-template-list
       (list (tpl-mode-templates 
                 (tpl-mode-match 'taql-mode tpl-global-template-list))))
 ;; set up keymap
 ;; every taql buffer will not have the same taql-keymap, nor
 ;; share the keymap it replaces
 (cond ( (or (not (boundp 'taql-mode-map))
             (not taql-mode-map))
         (let ((current-local-map (or pre-taql-map
                                      (make-sparse-keymap))))
         (make-local-variable 'taql-mode-map)
         (setq taql-mode-map (copy-keymap current-local-map))
         (make-local-variable 'pre-taql-map) )))
 (taql-set-keys taql-mode-map)
 (use-local-map taql-mode-map)
 ;; apparently loading templates requires the "." syntax for :
 (tpl-fix-syntax tpl-sep-placeholder "_")
 ;; finally, run hook
 (run-hooks 'taql-mode-hook)
 (setq taql-mode t))

(defun taql-mode-off ()
  ;; should reset keybindings, etc.
 (let* ( (file-name (buffer-name))
         (id-buffer (concat "id-" file-name)))
   (if (bufferp id-buffer)
       (kill-buffer id-buffer))
   ;; reset keybindings and syntax table
   (use-local-map pre-taql-map)
   (set-syntax-table pre-taql-syntax-table)
   (setq taql-mode nil)))


;;;
;;;	V.	taql-bug (sets up mail message)
;;;
;;; Create way to send bugs in with all variables noted.
;;;

(defun taql-bug ()
 "Generate an taql-mode bug report."
 (interactive)
 (message "Setting up taql-bug...")
 (let ( (ilisp-bugs-to taql-bugs-to) )
  (if (fboundp 'ilisp-bug)
      (ilisp-bug)
      (progn
         (mail)
         (insert taql-bugs-to)))
  (save-excursion  ;ilisp-bug will leave you on subject
   (taql-bug-header))
  (message "Insert your problem.")))

(defun taql-bug-header ()
    (search-forward "Type C-c C-c to send")
    (forward-line 1)
    (insert 
       "\n;;;;;;;;;;;;;;;;;;;;;;;;; TAQL STATE INFO ;;;;;;;;;;;;;;;;;;;;;;;")
    (insert 
      (format "\nWindow System: %s %s" window-system window-system-version))
   (insert (format "\nTAQL templates Version: %s" taql-version))
   (insert (format "\ntaql-mode version: %s" taql-mode-version))
   (insert (format "\ntaql-mode-home-directory: %s" taql-mode-home-directory))
   (insert (format "\nload-path: %s" load-path))
   (insert (format "\ntaql-mode-load-hook: %s" taql-mode-load-hook))
   (insert (format "\ntaql-mode-hook: %s" taql-mode-hook))
   (insert (format "\nGnu version: %s\n" (emacs-version)))
)



;;;
;;;	N-1.	Wrap up (load patches, run hooks and provide yourself)
;;;

(let ((patch-file (concat taql-mode-home-directory 
                          "/latest-taql-mode-patches")))
  (if (and taql-load-patches (file-exists-p patch-file))
      (load patch-file)))

(run-hooks 'taql-mode-load-hook)

(provide 'taql-mode)

(message "Done loading taql mode for TAQL version %s" taql-version)


;;;
;;;	N.	Known bugs and suggestions
;;;

;;; * what to do with negation template?  pitch it? make user go back,
;;;   or continue to force typing lots of spaces.

