;;; TAQL Compiler, Documentation String module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/taql/tc-doc-strings.lisp
;;; Created November 26, 1990.
;;;
;;; CONTENTS:
;;;
;;; This file contains documentation strings for the macros that
;;; define TAQL constructs.  The strings are constants, and the
;;; documentation properties of the TCs are setfed to be those
;;; constants, so doc strings can be added without changing the TAQL
;;; source.  For a table of contents, search for "CONTENTS".
;;;
;;; GENESIS:
;;; 
;;; The contents (except for this header) were compiled from an ASCII
;;; version of the 3.1.3 manual, the 3.1.4 release notes, and the
;;; contents of ~altmann/taql/state-copy-semantics.txt.  Compilation
;;; was done with the script ~altmann/bin/gendocs and some manual
;;; post-processing.  Look in gendocs for more details.  "~" expands
;;; to /afs/cs.cmu.edu/user/altmann/.   
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 9-4-91 - gry - Updated to describe previously-undocumented features
;;;   of TAQL 3.1.4.
;;;
;;; 7-13-91 - ema - recreated using gendocs as follows:
;;; 	(1) Each string consists of the manual entry for that TC,
;;; 	    as generated by scribe -f.
;;;     (2) The variable *release-notes-3.1.4* contains the release notes.
;;;     (3) The entry for evaluation-properties includes the
;;;         state-copy semantics table from the file
;;;         state-copy-semantics.txt, derived from release-notes.3.1.3,
;;;         because the .doc table generated by scribe is unusable.
;;;
;;; 3-24-91 - ema - added setfs for documentation properties. 
;;; 
;;; 11-26-90 - ema - Created as follows:
;;; 	1.  Each string consists of the manual entry for that TC,
;;; 	    as generated by scribe -f.
;;; 	2.  Strings for TCs that have changed in 3.1.3 are augmented
;;; 	    with "NOTE:  see *release-notes.3.1.3* for changes"
;;;     3.  The variable *release-notes-3.1.3* contains the release notes.
;;;     4.  The entry for evaluation-properties includes the
;;;         state-copy semantics table from the 3.1.3 release notes,
;;;         because the .doc table generated by scribe is unusable.
;;;     5.  The entry for operator-control is the file
;;;         operator-control.txt.

;;; ^^^ Insert file header above ^^^
;;;
;;; TC doc strings, generated Sat Jul 13 19:13:54 EDT 1991
;;;
;;; CONTENTS:
;;; 
;;; 1. defconstants that define TC doc strings
;;; 2. release notes defconstant ("RELEASE-NOTES")
;;; 3. setfs for the TC's documentation properties ("SETF")
;;;

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;;; This eval-when encloses everything in the remainder of the file:
;;;
(eval-when (compile load eval)

(defconstant *propose-task-operator-doc-string* "
4.3.2. propose-task-operator
  Syntax:
          (propose-task-operator NAME
             :space SPACE
             :op OPERATOR-SPEC
             [:when (COND+)]*)

  Semantics: Pose a task to Soar as an operator in the given space.

  If neither :space nor :when are specified, the task operator applies  in  the
top  problem space, which TAQL selects automatically at run time.  The state in
the top space is  usually  the  one  proposed  by  propose-task-state  (Section
4.2.2).    A  problem  space  to  apply  the task operator can be proposed with
propose-space (Section 4.1).

  The task operator can be terminated with a goal-test-group TC  (Section  4.6)
that  applies  in the top space.  If no such TC is supplied, TAQL automatically
terminates the task operator when a final state  is  reached  in  the  subgoal.
TAQL  rejects  the  task  operator  once  it  is terminated, and selects a halt
operator in its place.  The halt operator prints the identifier  of  the  final
task state, and halts Soar.


  Example:
          (propose-task-operator top*pto*r1 :op r1)



")
(defconstant *propose-task-state-doc-string* "
4.2.2. propose-task-state
  Syntax:
          (propose-task-state NAME
             [:space SPACE]
             [:when (COND+)]*
             [:new (ACTION-SPEC
                     [:when (COND+)]*)]*)

  Semantics:  Defines the contents of the state in the top level context.  This
is the state that an operator defined by propose-task-operator (Section  4.3.2)
will  apply  to.  Together, the task state and the task operator should contain
all the information necessary to specify  a  particular  task.    If  no  state
proposal  TC  applies  in the top context, TAQL automatically supplies an empty
top state.

  The :new keyword specifies the contents  of  the  state,  and  has  the  same
semantics  as  it  does  in the propose-initial-state TC.  The :space and :when
keywords constrain the situations in which the TC applies.  If  they  are  both
omitted,  the  value  of :space defaults to top-space.  This designates the top
problem space, which TAQL selects automatically at run-time.


  Example:
          (propose-task-state r1-top*pts
              :new ((^configuration <config> ^order <order>
                     ^ub-length-left <six-hundred> ^ub-load-left <nineteen>
                     ^modules-to-configure <m1>)
                     ...))



")
(defconstant *Problem-doc-string* "
4.1. Problem space proposal
  Syntax:
          (propose-space NAME
             [:space SPACE]
             :function (FUNCTION)
             [:when (COND+)]*
             [:copy (ATT-NAME*
                     [:from OBJECT-SPEC]
                     [:copy-new (ATT-NAME+)]
                     [:when (COND+)]*)]*
             [:rename ({ATT-NAME ATT-NAME}*
                     [:from OBJECT-SPEC]
                     [:copy-new ({ATT-NAME ATT-NAME}+)]
                     [:when (COND+)]*)]*
             [:new (ACTION-SPEC
                     [:when (COND+)]*)]*
             [:use {superspace | top-space | ID}])
          FUNCTION ::= apply operator OPERATOR-COND |
                       {propose | select} {initial-state | operator} for SPACE
                       {propose | select} space that will FUNCTION

  Semantics:  Propose  a  problem  space  with  name  SPACE,  under   specified
conditions.    The :function keyword specifies the type of impasse situation in
which the space will be proposed.  The definition of FUNCTION is  recursive  in
the  case  of  proposing spaces that propose or select spaces that perform some
other function.  For example, the function of a space could be to  select  from
among  multiple  competing  application  spaces  for  an  operator.   Often the
function will simply be to propose or apply an operator.

  The :copy, :rename, and :new keywords are for initializing the problem  space
with   static   information.      Their  semantics  are  the  same  as  in  the
propose-initial-state TC.  Because they are used more commonly in that context,
they  are discussed in detail in Section 4.2.  The source object for copies and
renames is the superstate, unless another object  is  specified  with  a  :from
clause.

  If  an  instance  of  SPACE  occurs  more that once in the context stack, for
example  during  lookahead  search,  it  is  initialized  only  on  the   first
occurrence,  with  the  values  carrying  over  to other occurrences.  The :use
keyword allows pre-existing problem-space objects to be proposed.   It  permits
no  modification of attributes of the existing space, so the only other keyword
that can be given with it is :function.


  Example:
          (propose-space unassigned-bp*ps
              :function (apply operator propose-configuration)
              :space unassigned-bp)



")
(defconstant *propose-initial-state-doc-string* "
4.2.1. propose-initial-state
  Syntax:
          (propose-initial-state NAME
             [:space SPACE]
             [:when (COND+)]*
             [:copy (ATT-NAME*
                     [:from OBJECT-SPEC]
                     [:copy-new (ATT-NAME+)]
                     [:when (COND+)]*
                     )]*
             [:rename ({ATT-NAME ATT-NAME}*
                     [:from OBJECT-SPEC]
                     [:copy-new ({ATT-NAME ATT-NAME}+)]
                     [:when (COND+)]*
                     )]*
             [:new (ACTION-SPEC [:when (COND+)]*)]*
             [:use {superstate | top-state | ID}])

  Semantics: Propose an initial state for problem space SPACE,  and  initialize
it  with  a  set  of  augmentations.    The  state  should  contain all dynamic
information in the problem space.  If SPACE is not given, the TC can  apply  in
any problem space.

  Propose-initial-state  provides  ways  to  copy  augmentations to the initial
state, ways to duplicate existing objects, and  ways  to  create  entirely  new
augmentations.

  Arguments  of  the  :copy keyword are attributes to be copied from the source
object along with their values.  The source object is specified  by  the  :from
keyword,  and  defaults to the superstate.<If the (ATT-NAME CLASS [ID]) form of
OBJECT-SPEC is used, ATT-NAME is taken to be an attribute of  the  superstate.>
Arguments  of  the :copy-new keyword are attributes whose values are objects to
duplicate .  Objects attached to  the  state  should  be  duplicated  from  the
supercontext  if they will be edited in the current space by apply-operator TCs
(Section 4.4;  object  duplication  is  further  discussed  in  that  section).
(Objects  to  duplicate  for  the  initial  state  in an evaluation subgoal are
specified with the evaluation-properties TC (Section 4.8.2)).

  The :rename keyword takes an even-length list of attribute names,  which  are
processed in pairs.  If the first four attributes are ATT1 ATT2 ATT3 ATT4, then
the values of ATT1 and ATT3 on the source object are copied to attributes named
ATT2  and  ATT4  in the new state.  The source object is specified by the :from
keyword, as for :copy.  The :copy-new keyword takes the same kind  of  argument
list, but duplicates the objects attached to ATT1 and ATT3.

  The  :new  keyword allows new augmentations to be added to the initial state.
The augmentations may be entirely new objects, or may  be  constructed  out  of
pre-existing structure by carrying values over from the conditions.

  The :use keyword causes TAQL to use the specified value as the initial state,
instead of creating a new state.  If an identifier is  specified,  it  must  be
bound  somewhere in the conditions, and must identify a state object.  The :use
keyword cannot be given together with the :copy, :rename, or :new.
  The conditions constrain the circumstances under which the augmentations  are
created.  Conditions at the top-level of the TC apply to :use and to all :copy,
:rename, and :new clauses in the TC.  Conditions within a  :copy,  :rename,  or
:new clause apply only to the augmentations specified in that clause (including
arguments of :copy-new).

  An initial state is created and made acceptable only if at least some of  the
specified  conditions are matched.  If multiple propose-initial-state TCs match
under the same circumstances, each will create an initial  state  and  make  it
acceptable, possibly resulting in a state-tie impasse.


  Example:
          (propose-initial-state conf-mods*pis
              :space configure-modules
              :copy (backplane ub-load-left
                  :copy-new (power-available configuration))
              :rename (current-module current-node)
              :new (phase process-node))


")
(defconstant *propose-operator-doc-string* "
4.3.1. propose-operator
  Syntax:
          (propose-operator NAME
             {{:op OPERATOR-SPEC}+ | :op-id ID}
             [:space SPACE]
             [:when (COND+)]*
             [:select-once-only])
  Semantics: Propose an operator when the specified conditions  are  satisfied.
If  the  :space  keyword is present, the TC applies only in that problem space.
Several :op clauses may be given, if they all specify OPERATOR-SPECs.  This  is
useful  for  proposing  multiple  instances  of  the same operator, because the
parameters that distinguish the instances can be bound in :when  clauses  local
to  each  :op  clause.    If  :op-id is given, it can be given at most once, it
cannot appear along with :op, and the associated ID must be bound in the global
conditions of the TC.  The object with that ID is proposed as the operator.

  The  :select-once-only  keyword  signals  that  any  instance  of an operator
proposed by that TC will be rejected after it is selected and reconsidered.  If
several  instances  of  an  operator are proposed, then each has a chance to be
selected once.  The keyword takes no arguments.


  Example:
          (propose-operator conf-boards*po*board-into-slot
              :op (board-into-slot ^board <board> ^slot <slot>)
              :space configure-boards
              :when ((state ^current-slot <slot> ^module <mod>)
                     (module <mod> ^board <board>)))


")
(defconstant *compare-doc-string* "
4.5.2. compare
  Syntax:
          (compare NAME
             [:space SPACE]
             [:when (COND+)]*
             [:for-goal {supergoal | ID}]
             [:object-type {problem-space | state | operator}]
             [:object1 ID]
             [:object2 ID]
             [:op1 OPERATOR-COND]
             [:op2 OPERATOR-COND]
             :relation {better | worse | indifferent})

  Semantics: Create a single binary  preference  that  compares  two  specified
objects  for  a  specified  goal.   The :space and :when keywords constrain the
context in  which  the  TC  applies.    The  :relation  keyword  specifies  the
preference type.

  The  keywords  :for-goal,  :object-type,  :object1 and :object2, and :op1 and
:op2 are analogs of those in the prefer TC (Section 4.5.1), extended to  binary
relations.   For problem-spaces or states, :object-type, :object1, and :object2
must all be given.  For operators, exactly one of :object1 or :op1, and exactly
one  of  :object2  and  :op2, must be specified.  If :object-type is given when
either :op1 or :op2 is given, its value must be operator.


  Example:
          (compare conf-boards*c*next-slot-worse-board-slot
              :space configure-boards
              :op1 next-slot
              :relation worse
              :op2 board-into-slot)


")
(defconstant *prefer-doc-string* "
4.5.1. prefer
  Syntax:
          (prefer NAME
             [:space SPACE]
             [:when (COND+)]*
             [:for-goal {supergoal | ID}]
             [:object-type {problem-space | state | operator}]
             [:object ID]
             [:op OPERATOR-COND]
             {:value {best | worst | indifferent | reject | require | prohibit 

  Semantics:  Create  a  set  of  unary preferences for a specified object in a
specified goal.  The :space and :when keywords constrain the context  in  which
the  TC applies.  The object and goal affected by the preferences are specified
with a combination of the keywords :for-goal, :object-type, :object, and :op.

  If :for-goal is given, it specifies the goal in which the preference is to be
created.    If  the  value  is an identifier, it must be bound somewhere in the
conditions.  If the value is supergoal, it indicates the goal above the context
in which the TC applies.  If :for-goal is omitted, the preference is created in
the same goal as the TC applies.

  Exactly one of :op and :object must  be  given.    The  :object-type  keyword
specifies  one  of  problem-space, state, or operator.  It is required when the
object of the preference is a problem space or state, but may  be  omitted  for
operators  if the :op keyword is given.  The value of :object is the identifier
of the object, and must be bound in the conditions.  The :op keyword provides a
shorthand  for  the  common  case  of  operator  preferences.  If :op is given,
:object-type must specify operator if it is present.

  A preference is created for each :value clause given.  Usually,  the  set  of
preferences  is created when their object is proposed.  (However, if reconsider
is one of the values, then all the preferences are created only when the object
is  actually  selected,  because  reconsider  is  only  meaningful for selected
objects.)  These semantics make it possible to define one  problem  space  that
performs  search control on behalf of another.  For example, if an operator tie
arises in space P , space P  could resolve the impasse as follows:
                 a         b
      A propose-space TC  proposes  P ,  with  the  function  of  selecting
                                     b
    operators  for  P .  P  determines what the competing operators are (by
                     a    b
    examining the ^item augmentations of the goal), and problem  solves  to
    determine  the preference relations among them.  Periodically, a prefer
    TC will apply to create preferences  for  operators  in  the  supergoal
    (using  :for-goal  supergoal).   Eventually, enough preferences will be
    created to resolve the impasse.

  Unlike other subgoals, those for tie impasses have no explicit  final  state,
because Soar resolves the impasse when it can choose among the tied candidates.
Thus P  in the example above requires no goal-test-group TC.
      b

  Example:
          (prefer conf-boards*p*board-slot*reject*slot-full
              :space configure-boards
              :value reject
              :op (board-into-slot ^slot <slot>)
              :when ((state ^configuration <conf>)
                     (configuration <conf> ^configured <cfd>)
                     (board-slot <cfd> ^slot <slot>)))


")
(defconstant *Goal-doc-string* "
4.6. Goal testing
  Syntax:
          (goal-test-group NAME
             [:space SPACE]
             [:when (COND+)]*
             [:group-name GROUP-NAME]
             [:group-type {success | failure}]
             [:test (TEST-NAME {:when (COND+)}+)]*)

  Semantics:  Declare  that  a  final state has been reached in a problem space
when specified tests have been satisfied.  If the :space  keyword  is  present,
the  TC  applies  only  in that problem space.  Conditions specified at the top
level of the TC apply to all :test clauses.

  The simplest way to express a test is  with  conditions  given  in  top-level
:when  clauses.    If no :test keyword is given, then a final state is declared
when all top-level conditions are simultaneously satisfied in the given  space.
Mechanisms for conjunctive and disjunctive goal tests are described below.

  When  a  final  state  is  declared,  the state is labeled with GROUP-NAME if
:group-name is given, and typed with  success  or  failure  if  :group-type  is
given.     These  labels  are  for  constraining  and  communicating  with  the
result-superstate and propose-superobjects TCs (Section 4.7).  Soar's lookahead
search  mechanism  also  recognizes  the  type  of  a final state reached in an
evaluation subgoal.  It  translates  success  into  best  preferences  for  the
operators that created the state, and failure into reject preferences.

  When  a  goal  test  is  satisfied,  TAQL  proposes and selects a final-state
operator.  TCs that generate subgoal results (Section 4.7) do so as part of the
application  of  this operator.  This guarantees that subgoal results are based
on a consistent final state, and also improves the readability of traces.    If
any  user-defined  operator  has  a  require  preference  in  a  final state, a
constraint-failure impasse will arise between it and the final-state operator.

  The primitive goal test operation: The goal-test-group TC is really  a  macro
that  provides convenient support for a more primitive problem space operation.
TAQL's goal test primitive is a symbol that can  be  placed  on  the  state  to
communicate  to  other  TCs  that the state is final.  The symbol consists of a
^final-state*  attribute  whose  value   is   an   object   with   class   name
final-state-info.    The  object  must  be augmented with a ^goal-id* attribute
whose value is the same as the ^goal-id* attribute of the  goal  in  which  the
state  is  final.    (The  ^goal-id*  augmentation  of  the  goal  is generated
automatically by TAQL.)  This mechanism is necessary because  a  state  can  be
used  in  multiple goals but doesn't have to be a final state in all of them at
once.

  The object attached to ^final-state* can be augmented with ^final-state-type*
success  or  ^final-state-type*  failure,  to communicate the type of the final
state to Soar's lookahead search mechanism and to  TCs  that  generate  subgoal
results.

  Conjunctive and disjunctive goal tests: Each test in a group is identified by
its TEST-NAME (which is local to that TC).  If multiple test clauses  have  the
same name, they represent the same test, and that test is satisfied if at least
one of the clauses is satisfied.  This  feature  allows  for  disjunctive  goal
tests.    The  group  as  a  whole  is  satisfied  when  all distinct tests are
satisfied.  This feature provides for conjunctive goal tests.  All tests  in  a
conjunction  have  to  be satisfied simultaneously to signal that a final state
has been reached.

  In the  example  below,  the  goal-test-group  is  satisfied  when  both  the
backplane-into-box  and modules-into-backplane tests are satisfied, but each of
these tests can be satisfied by either of two situations.


  Example:
          (goal-test-group conf-bp*gtg
              :group-type success
              :space configure-bp
              :test
                 (backplane-into-box
                    :when ((state ^backplane <bp> ^configuration <conf>)
                           (configuration <conf> ^configured <c1> <c2>)
                           (back-box <c1> ^backplane <bp>)
                           (back-back <c2> ^backplane2 <bp>)))
              :test
                 (backplane-into-box
                    :when ((state ^backplane <bp> ^configuration <conf> - ^prev
                           (configuration <conf> ^configured <c1>)
                           (back-box <c1> ^backplane <bp>)))

              :test
                 (modules-into-backplane
                    :when ((state ^backplane <bp> ^configuration <conf>)
                           (configuration <conf> ^configured <c1>)
                           (mod-back <c1> ^backplane <bp>)))
              :test
                 (modules-into-backplane
                    :when ((state ^backplane <bp>)
                           (backplane <bp> ^name ub-rep))))
")
(defconstant *apply-operator-doc-string* "
4.4.1. apply-operator
  Syntax:
          (apply-operator NAME
             :op OPERATOR-COND
             [:space SPACE]
             [:when (COND+)]*
             [:terminate-when ([COND+])]*
             [:use ID]
             [:bind ID]*
             [EDIT]*)
          EDIT ::=

          (edit :what OBJECT-SPEC
             [:type {sliding | one-shot}]
             [:when (COND+)]*
             [:replace (ATT-NAME [LVALUE] :by VALUE-SPEC [:when (COND+)]*)]*
             [:copy (ATT-NAME+ [:from OBJECT-SPEC]
                               [:when (COND+)]*
                               [:remove {source | target}]*)]*
             [:rename ({ATT-NAME ATT-NAME}+ [:from OBJECT-SPEC]
                               [:when (COND+)]*
                               [:remove {source | target}]*)]*
             [:remove ({ATT-NAME | :all} [LVALUE] [:when (COND+)]*)]*
             [:new (ACTION-SPEC [:when (COND+)]*)]*
             [:directive (DIRECTIVE+ [:when (COND+)]*)]*
             )

  General  semantics:  Specify  what objects to edit when applying a particular
operator, and the circumstances under which the edits should take  place.    If
SPACE is given, the TC applies only in that problem space.

  An alternative way to add information to an object is with state entailments,
using the augment TC (Section 4.9).

  The optional :use keyword takes the identifier variable of a state object  as
its  value.    It  specifies  that the operator's result state is the specified
state.  The state identifier variable must be bound in the conditions.  In  any
apply-operator  TC,  either exactly one :use clause or one or more edit clauses
must be given.

  The optional :bind keyword binds the specified identifer variable  to  a  new
symbol.   The specified variable can be used throughout the TC.  This is useful
when you want to create a new object  that  must  be  referred  to  in  several
different  edit  clauses, for example when two newly-created objects must point
to each other.

  The edit clause: Editing operations are specified with edit  clauses.    Each
edit  clause  applies  to objects designated by the :what keyword.  Local :when
clauses given at the top  level  of  the  edit  clause  apply  to  all  editing
operations  in that clause.  These conditions are conjoined with any conditions
specified at the top level of the apply-operator TC.

  The :what clause specifies which object the editing operations given  in  the
edit clause apply to.

  An  edit  clause supports five editing operations: (1) replacing the value of
an attribute with a new value; (2) copying an attribute  from  another  object;
(3)  renaming  an attribute; (4) removing an attribute or a particular value of
an attribute; and  (5)  adding  a  new  augmentation.    The  source  attribute
specification  for  :replace  has two parts: an attribute name, and an optional
LVALUE constraining which value of a (multi-)attribute is  affected.    If  the
LVALUE  is  a  variable,  it  must  be  bound somewhere in the conditions.  The
attribute specification for :remove is similar, except that  the  keyword  :all
may  be used in place of an attribute name.  :All indicates that all attributes
should be removed, subject to any constraints imposed by the LVALUE.  The  :new
keyword  can specify brand new augmentations.  It has the same semantics as the
:new keyword used in the propose-initial-state TC  described  in  Section  4.2.
Variable attribute names may be used as long as the variable is bound somewhere
in the conditions of the TC.

  The :copy and :rename  keywords  are  used  for  moving  attributes  from  an
existing object to the result state.  They differ only in that for a :copy, the
name of the resulting  attribute  is  the  same  as  the  name  of  the  source
attribute,  while  for  a  :rename  the  source  and target attributes can have
different names.  For a :rename, the attributes  are  listed  in  source/target
pairs,  with  the source attribute listed first in the pair.  The :from keyword
specifies the source object for the copy or rename, and defaults to the current
state.    If  :remove  source  is  specified, the source attributes are removed
during the copying or renaming.  If :remove target is specified,  any  existing
values  for  the  target attributes on the current state are removed before the
new values are created.

  The :directive keyword is for use with TAQL's data  type  facility.    It  is
described in Section 5.0.1 (page 32).

  Semantics of adding, dropping, and replacing: The five editing operations can
be decomposed into adds and drops of augmentations.  When the collected set  of
adds  and  the  collected  set  of  drops  overlap, the adds dominate.  This is
useful, for example, when all  the  values  of  a  multi-attribute  are  to  be
replaced with a new value which may or may not be in the old set.

  The  semantics  of  :replace are that if no LVALUE is given, the drop and the
add are independent, so the attribute does not have to exist for the add to  be
effective.    This means an attribute does not have to be initialized before it
is replaced.  However, if :replace is given an LVALUE, then the attribute  must
exist before it can be replaced.

  Operator termination: The apply-operator TC provides several options for when
to  terminate  an  operator.    Reconsider  preferences  are  generated  either
automatically   when  the  state  is  edited,  or  under  explicitly  specified
conditions,  or  not  at  all.    The  mechanism  to  control   this   is   the
:terminate-when keyword and its values.

  If  :terminate-when  is absent, then the TC generates a reconsider preference
only if a  non-sliding  (see  below)  edit  clause  actually  applies.    If  a
:terminate-when  clause  is given with a null list as a value, no reconsider is
generated from anywhere in the TC.

  If at least one :terminate-when clause is given, and has a list of conditions
as  a  value,  then  a  reconsider  is  generated  only if those conditions are
satisfied when all edit clauses have  finished  applying.    No  preference  is
generated as a result of edit clauses applying.  The termination conditions are
conjoined with those specified by global :when clauses.

  If no reconsider is generated by a particular TC  at  run-time,  an  operator
application  impasse will arise (unless another TC applies to the same operator
and generates a reconsider).  Such an  impasse  terminates  when  a  reconsider
preference  is  generated from the subgoal (as when, for example, a final state
is reached), or when working memory  is  changed  to  satisfy  the  termination
conditions (either from the subgoal, or by IO).

  Sliding  edits: By default, TAQL constrains edit clauses to apply in parallel
to the state as it was when the operator was selected, and prevents  them  from
applying  again.    This  synchronous,  atomic model of operator application is
convenient and often sufficient.   However,  it  limits  interaction  with  the
environment   through  IO,  and  disguises  Soar  5's  underlying  symbol-level
computational model (Laird, Congdon, Altmann and Swedlow, 1990).

  A variation on this model allows edits to apply whenever their conditions are
satisfied.    Sliding  edits  can  create  a  sequence  of intermediate states,
starting from the state as it was when the operator was selected, and ending in
the  operator's  result  state.   The result state is that reached when no more
sliding edits can apply and when the operator is terminated.  (A  sliding  edit
never generates a reconsider preference, even if :terminate-when is absent from
the TC.)

  Intermediate states are inconsistent with  respect  to  other  problem  space
operations.    For  example,  a  goal-test-group  TC  will  not  apply to these
intermediate states, if it does not also apply to the result state.

  An edit clause is declared sliding when  that  is  the  value  of  its  :type
keyword.   If :type is omitted, the value defaults to one-shot, which specifies
the synchronous, atomic model described above.  The specified type  applies  to
all editing operations within the edit clause that contains it.

  Using  duplicate  objects: If it applies, an edit clause always destructively
modifies the object that it edits.  Therefore, whenever an object is edited  in
a  subspace,  and  the  object  is  linked  to  the supercontext, there are two
effects: (1) chunks will be created (if learning is on),  and  (2)  the  change
will  be  visible  in  the  supercontext.   To avoid these effects (if they are
undesirable), objects should  be  duplicated  before  they  are  first  edited.
Directives  for  creating duplicate objects can be given when the initial state
is proposed with the propose-initial-state TC (Section 4.2).  When objects will
be edited in an evaluation subgoal (during lookahead search), a list of objects
to duplicate can  be  specified  with  the  evaluation-properties  TC  (Section
4.8.2).


  Example:
          (apply-operator conf-box*ao*get-box
              :space configure-box
              :op (get-box ^instance <box>)

              (edit :what (order order)
                  :remove (ordered <box>))

              (edit :what state
                  :replace (system-units-used
                            :by ((number ^value (compute <nsuu> + <nbxs>)))
                            :when ((state ^system-units-used <suu> ^box-sur <bx
                                   (number <suu> ^value <nsuu>)
                                   (number <bxs> ^value <nbxs>)))
                  :replace (current-box :by <box>)

              (edit :what (power-available power)
                  :remove (:all)
                  :new (<type> <amount>
                        :when ((box <box> ^power-available <power>)
                               (power <power> ^<type> <amount>)))

                  :new (<type> <amount>
                        :when ((box <box> ^section <sec>)
                               (section <sec> ^number 1 ^power-available <power
                               (power <power> ^<type> <amount>)))))



")
(defconstant *Entailment-doc-string* "
4.9. Entailment
  Syntax:
          (augment NAME
             [:space SPACE]
             [:when (COND+)]*
             :what OBJECT-SPEC
             [:new (ACTION-SPEC [:when (COND+)]*)]*
             [:copy (ATT-NAME+
                         [:from OBJECT-SPEC]
                         [:when (COND+)]*)]*
             [:rename ({ATT-NAME ATT-NAME}+
                         [:from OBJECT-SPEC]
                         [:when (COND+)]*)]*)

  Semantics:  Augment  some  object  with  entailments('Entailments' are called
'refinements' in the Soar  5.2  User's  Manual  (Laird,  Congdon,  Altmann  and
Swedlow, 1990).) based on existing information.

  Entailments  remain  part  of an object until the conditions under which they
were created  are  no  longer  satisfied.    At  that  point,  entailments  are
automatically  removed  from  the  object.   This effect cascades, so that if a
chain of augment TCs are linked by their conditions and actions, then  all  the
TCs will remove their entailments if the first TC no longer matches.

  Entailments  are  useful  for making simple inferences that no longer hold if
the information they are based on leaves working memory.  Permanent changes  to
working memory should be made by operator application (Section 4.4).  Note that
if entailments are the basis for edits performed by operator  application,  the
operator  application  edits remain even if the entailments are removed at some
later point.

  The :what keyword specifies the object the entailments should be  placed  on.
The  :new,  :copy  and :rename keywords specify what entailments to add.  These
keywords have the same  semantics  as  they  do  in  operator-application  edit
clauses (see Section 4.4.1), with the exception that here, :copy and :rename do
not accept an embedded :remove keyword.


  Example:
          (augment generic*a*state*get-cars-on-siding-past*recognize-done
              :when ((state - ^source))
              :when ((state ^track <tsiding> ^strategy get-cars-on-siding-past)
                     (track <tsiding> ^position <siding> ^car none)
                     (position <siding> ^name siding))
              :what state
              :new (succeeded get-cars-on-siding-past))
")
(defconstant *result-superstate-doc-string* "
4.7.1. result-superstate
  Syntax:
          (result-superstate NAME
             [:space SPACE]
             [:group-name {GROUP-NAME | ID}]
             [:group-type {success | failure | ID}]
             [:when (COND+)]*
             [:terminate-when ([COND+])]*
             [:use {superstate | final-state | ID]]
             [:bind ID]*
             [EDIT]*[See Section 4.4.1 for the syntax
and semantics of EDIT.])

  Semantics: Specify how to create the result state for  an  operator  that  is
applied in a subgoal.

  The  result-superstate  TC is nearly identical to the apply-operator TC.  The
differences are:

   1. Result-superstate does not accept  an  :op  keyword.    Instead,  it
      always specifies how to apply TAQL's final-state operator.

   2. Result-superstate  accepts  :group-type  and  :group-name  keywords.
      Their semantics are described later in this section.

   3. The :use keyword can take the values final-state and  superstate  as
      well  as  an  identifier  variable.  The value final-state specifies
      that the final state of the subspace is to be  used  as  the  result
      state of the superoperator.  The value superstate specifies that the
      superstate is already the result state of the superoperator, and  no
      further changes need to be made.

  The  edit  clauses  in  the result-superstate TC must make the changes to the
superstate that are required to produce the superoperator's result state.  Most
commonly  this will be done with (edit :what superstate ...) clauses, though in
some cases other objects will be edited (such as subobjects of the superstate).

  TAQL does not permit the final-state operator to be implemented in a subgoal.
It  must  always  be implemented directly by result-superstate.  If TAQL gets a
no-change on the final-state operator, it uses the  superstate  as  the  result
state  of  the  superoperator,  and  does  not build a chunk over the no-change
impasse.  This is convenient in spaces where sometimes there  are  results  for
result-superstate to return, but sometimes no further changes need to be made.

  In  some situations, when you would otherwise not need a result-superstate TC
at all, you can use one that specifies :use superstate.  This declares that the
superstate is already the result state, and no further changes need to be made.
TAQL will not get an impasse on the final-state operator when  :use  superstate
is specified.

  Coordinating  result-superstate and goal-test-group: The result-superstate TC
takes effect only after a final state in the current space  has  been  reached.
Tests  to  recognize  final  states  are  specified  by  the goal-test-group TC
(Section 4.6).  Goal-test-group can optionally label a final state with a  name
that  identifies  the group, and with a type that is either success or failure.
These  labels  can  be  tested  in  result-superstate  with   :group-name   and
:group-type.    If a result-superstate TC tests a name or a type, it will apply
only if the final state is  labeled  with  that  name  or  type.    If  neither
:group-name  nor  :group-type  is  specified, then result-superstate applies as
soon as any final state is reached.  If the value of :group-name or :group-type
is  a  variable,  then that variable is bound to the value of the group-name or
group-type, respectively, of the final state.


  Example:
          (result-superstate conf-mod*rs*failure
              :space configure-module
              :group-type failure
              :group-name <group-name>

              (edit :what superstate
                     :new (exit-cond-modules <group-name>)))


")
(defconstant *evaluation-properties-doc-string* "
4.8.2. evaluation-properties
  Syntax:
          (evaluation-properties NAME
             :space SPACE
             [:when (COND+)]*
             [:better {higher | lower}]
             [:if-multiple { pessimistic | optimistic | all }]
             [:lookahead ( :monotonic { increasing | decreasing }
                           :copy ({:all | ATT-NAME*})
                           :dont-copy ({:any | ATT-NAME*})
                           :copy-new ({:all | ATT-NAME*}))])

  Semantics: Declares properties of  the  evaluation  function  for  the  given
space, and how state-copying will be carried out in an evaluation subgoal where
that space is selected.  Properties can be given only once for  any  particular
instance  of  a  problem  space.    In  the  absence  of  declarations, default
properties are supplied.

  The :better keyword  indicates  whether  higher  or  lower  numbers  indicate
greater  worth.    The  default  value  is  higher.    The :if-multiple keyword
indicates how to choose among multiple  numeric  values  for  an  object.    If
pessimistic  is  given, then of all values associated with an object, the worst
is chosen.  Optimistic is analogous.  If  all  is  specified,  then  TAQL  will
assign all values to the object.  If multiple values are assigned to an object,
and they overlap with  values  of  other  objects,  the  selection  space  will
generate  preferences  that  will  result  in  a  conflict  impasse between the
objects.  See the Soar manual (Laird, Congdon, Altmann and Swedlow,  1990)  for
details.

  The  :lookahead  keyword  introduces  a  set  of  subkeywords  that  describe
properties of lookahead evaluation using the given space.    If  :lookahead  is
absent,  the  effects  of omitting the individual subkeywords are implied.  The
subkeywords are describe in the following two subsections.

  Monotonic evaluation functions:  The  :monotonic  keyword  declares  that  if
consecutive lookahead states in an evaluation subgoal are evaluated, the values
will increase or decrease monotonically.  This  allows  TAQL  to  terminate  an
evaluation  subgoal  once  a state is reached whose value is no better than the
value of some operator that has been evaluated previously.   If  :monotonic  is
omitted, the evaluation function is not assumed to be monotonic.

  State-copy properties: The :copy, :copy-new, and :dont-copy keywords describe
how to transfer augmentations from the tie-context state to the duplicate state
in  the  evaluation  subgoal.    :Copy  specifies  which  augmentations to copy
directly to the duplicate state, and :copy-new specifies which augmentations to
duplicate onto the duplicate state.  :Dont-copy interacts with the other two to
inhibit the transfer of attributes.

  Several considerations are relevant to what objects to copy and how  to  copy
them:

  Efficiency      Copying  only  attributes that are needed reduces the cost of
                  the operation.

  State entailment
                  State  entailments  are  simple  inferences based on existing
                  data in the state, created with augment  TCs  (Section  4.9).
                  They  should  retract  automatically  if the state changes so
                  that they are no longer relevant.  However, if an  entailment
                  is  copied to a duplicate state, it could incorrectly persist
                  after the duplicate state is  modified,  if  the  tie-context
                  state remains the same.

  Editing subobjects
                  Operators in the  evaluation  subgoal  might  modify  objects
                  attached  to the state.  These objects must be duplicated, so
                  that their tie-context  versions  remain  intact.    This  is
                  further  discussed  in  the context of apply-operator Section
                  4.4.1.


  The semantics of the three state-copy keywords are given in  Table  2.    The
first  column  of the table gives all legal combinations of the three keywords,
and the second column gives the corresponding semantics.  The metasymbols  used
in the table are:
                    ARGS ::= a set of attributes
                    ALL  ::= all attributes on the original state
                    NONE ::= the empty set
                    +    ::= set union
                    -    ::= set difference


  All  keyword combinations other than those listed in Table 2 are illegal.  In
table entries that have both ARGS1 and ARGS2, it is  an  error  for  ARGS1  and
ARGS2 to overlap.


    ---------------------------------------------------------------------------
    Keywords                         Semantics
    --------                         ---------

    NO KEYWORDS                      copy ALL
    dont-copy (:any)                 Don't copy any attributes
    copy ARGS                        copy ARGS
    copy-new ARGS                    copy-new ARGS
    copy ARGS1, copy-new ARGS2       copy ARGS1, copy-new ARGS2
    copy (:all)                      copy ALL
    copy-new (:all)                  copy-new ALL
    copy (:all), copy-new ARGS       copy(ALL - ARGS), copy-new ARGS  
    
    copy (:all), dont-copy ARGS      copy(ALL - ARGS)
    copy (:all), copy-new ARGS1,     copy(ALL - ARGS1 - ARGS2), copy-new ARGS1
         dont-copy ARGS2
    copy-new (:all), copy ARGS       copy ARGS, copy-new(ALL - ARGS)
    copy-new (:all), dont-copy ARGS  copy-new(ALL - ARGS)
    copy-new (:all), copy ARGS1,     copy ARGS1, copy-new(ALL - ARGS1 - ARGS2)
         dont-copy ARGS2
    ---------------------------------------------------------------------------

                    Table 2:  State-copy keyword semantics


  Example:
          (evaluation-properties unassigned-bp*ep
              :space unassigned-bp
              :better lower
              :lookahead
                  (:monotonic increasing
                   :copy (:all)
                   :dont-copy (backplane-succeeded-operator)
                   :copy-new (configuration order power-available)))
")
(defconstant *evaluate-object-doc-string* "
4.8.1. evaluate-object
  Syntax:
          (evaluate-object[This TC is intended to provide a complete,
flexible, and localized interface to the selection space.  In common
use, instances are unlikely to be as bewildering as the syntax
specification (with the possible exception of the example given
here).] NAME
             [:space SPACE]
             [:when (COND+)]*
             [:bind-object ID]
             {:what {OBJECT | (OBJECT
                                 [:when (COND+)]*
                                 [:bind-object ID]
                                 [:cycle-when (COND+)]
                                 [:return-when (COND+)]*)}}+
             [:numeric-value {NVALUE | (NVALUE
                                          [:when (COND+)]*
                                          [:bind-object ID])}]+
             [:symbolic-value {SVALUE | (SVALUE
                                          [:when (COND+)]*
                                          [:bind-object ID])}]+)
          OBJECT    ::= problem-space | state | operator | lookahead-state
          NVALUE    ::= novalue | INTEGER | <variable> | (function arg1 arg2 ..
          SVALUE(This gives the entire legal set provided by Soar.  The
                    re  ommonly u ed.)
first three values a::=cnovalue |ssuccess | failure |
                        partial-success | partial-failure |
                        required-success | prohibit-failure |
                        lose | win | draw | indifferent | <variable> | (functio

  Semantics:  Specifies  what  objects  are  to   be   evaluated   under   what
circumstances,  and what values to assign to them.  This is TAQL's interface to
Soar's selection space.  The semantics most commonly exercised are described in
the  next  four  paragraphs.    The  extended  semantics  are  described in the
remaining subsections.

  The :what keyword specifies  what  object  to  evaluate.    If  it  specifies
problem-space,  state, or operator, a value is assigned to each evaluate-object
operator that evaluates an object of that type, as  soon  as  the  operator  is
selected.    (The evaluate-object operator is proposed by Soar in the selection
space.)

  If :what specifies lookahead-state, then the evaluate-object operator applies
in an evaluation subgoal.  As soon as the :what clause and a value clause apply
in the subgoal, TAQL  proposes  an  evaluate-state  operator  to  evaluate  the
lookahead  state.    Conditions  local  to  the :what clause determine when the
clause applies.  If no local conditions are given, the clause applies  as  soon
as  the  operator  being tried out is terminated.  Multiple :what clauses apply
disjunctively, so that a state can  be  evaluated  under  alternative  sets  of
conditions.

  The  particular  value  that  an  object receives is determined by the set of
value clauses (:numeric-value and :symbolic-value).  Conditions local  to  each
clause  serve  to decide which objects get which values, and to bind values for
use in an evaluation function.  At least one value  clause  must  apply  to  an
object for the evaluate-object operator to terminate.

  The  :bind-object keyword provides access to the object being evaluated.  Its
value must be a variable.  If a problem space,  state,  or  operator  is  being
evaluated,  TAQL  binds  ID to the identifier of the object.  If an operator is
being evaluated by lookahead, TAQL binds ID to the first operator  selected  in
the evaluation subgoal.  (This is a copy of the operator being evaluated.)

  The  novalue  value:  The special value novalue can be either a symbolic or a
numeric value.  An object assigned novalue is made worse than all objects it is
tied with.  If all objects in a selection subgoal evaluate to novalue, and that
subgoal is below another selection subgoal, then the operator  being  evaluated
in  the  higher  subgoal  also  evaluates  to  novalue.   If there is no higher
selection subgoal, then all of the novalue objects are made indifferent to each
other.

  Novalue  is  intended  for use during lookahead search.  It expresses nothing
more than the desire to backtrack.  As part of these  semantics,  no  chunk  is
built when novalue is assigned to an object.

  Cycle  detection:  The  :cycle-when keyword allows for detection of cycles in
multi-level lookahead search.  The conditions are tested pairwise across goals.
If they are satisfied in any pair of contexts, then the :what clause applies.

  The  variables  in  the  conditions are the same in both contexts, forcing an
equality test.  This works correctly for testing static  objects  and  constant
values.    However,  if  objects  involved  in  the  test are duplicated across
contexts, their identifiers will change and the conditions will not recognize a
cycle.

  More  on  lookahead  evaluation: This subsection presents an unordered set of
facts about  TAQL's  treatment  of  lookahead  search,  including  some  obcure
pitfalls.

  If  the  problem  space  has  a  :monotonic  evaluation  property,  then  the
evaluate-state operator is selected  automatically  after  each  task  operator
applies  in  the  evaluation  subgoal,  independent  of  any  :what clause.  If
evaluate-state indicates a cutoff, then TAQL creates a preference that  removes
the  operator  being  evaluated  from the tied set.  If no cutoff is indicated,
problem solving continues until the next  opportunity  for  cutoff  arises,  or
until  a  :what  clause  and  a  value  clause apply simultaneously.  Monotonic
evaluation functions are discussed in the context of the  evaluation-properties
TC (Section 4.8.2).

  If multiple value clauses apply when evaluation is enabled, the tie is broken
according  to  the  if-multiple  evaluation  property  (Section  4.8.2).     If
evaluate-object  is  selected  and  no  value  clause  applies,  Soar impasses.
Usually, one value clause will apply whenever a :what clause applies.

  If at any  time  during  lookahead  search  a  final  state  is  reached  (as
determined  by a goal test), and the state has a type, the type is converted to
the corresponding symbolic value (success or failure).  The subgoal  terminates
and  that value is returned as the value of the operator being evaluated.  If a
final state is reached and does not have  a  type,  then  the  subgoal  returns
novalue.

  The  evaluate-state operator is automatically required when a :what lookahead
clause and a value clause apply, which can be any time after the first operator
in  the subgoal is selected.  Because evaluate-state is proposed with a require
preference, a constraint-failure impasse will arise if a user-supplied operator
is required at the same time.

  Exploiting  serendipity:  The  :return-when keyword provides for declaring an
object as best, thereby immediately breaking the tie above the selection space.
When  the  specified  conditions  are  satisfied, the object being evaluated is
selected in the tie context.

  If the value of :what is lookahead-state, the  object  selected  in  the  tie
context  is  the  internal TAQL operator leap, instantiated with the state from
the context that satisfied the :return-when  conditions.    The  leap  operator
replaces the state in the tie context with its parameter state.

  The  :return-when  keyword is intended for returning a successful final state
from deep in multi-level lookahead search.  If, instead, an explicit goal  test
is  used  to  recognize  a successful final state in an evaluation subgoal, the
symbolic  value  success  is  passed   recursively   back   through   all   the
evaluate-object  operators on the context stack, resulting in a best preference
for the very first operator  that  was  being  evaluated.    That  operator  is
selected  in  the  context  in  which it tied.  Unless accurately remembered in
chunks, all the processing that went to produce the final state is lost.


  Example:
          (evaluate-object unassigned-bp*eo
              :space unassigned-bp
              :what (lookahead-state
                     :when ((state ^exit-cond-modules end-of-list)))
              :what (lookahead-state
                     :when ((state ^exit-cond-modules kmc11-configured)))
              :what (lookahead-state
                     :when ((state ^previous-bp <pbp> ^backplane-succeeded-oper
                            (backplane <pbp> - ^name 4-slot - ^name 9-slot)))
              :what (lookahead-state
                     :bind-object <op>
                     :when ((state ^backplane-failed-operator <op>)))
              :symbolic-value (failure
                               :bind-object <op>
                               :when ((state ^backplane-failed-operator <op>)))
              :numeric-value ((compute <nsuu> + (100 * <nbps>) + (1000 * <nboxe
                              :bind-object <op>
                              :when ((state - ^backplane-failed-operator <op>
                              ^system-units-used <suu>
                              ^backplanes-ordered <bps>
                              ^boxes-ordered <boxes>)
                                     (number <suu> ^value <nsuu>)
                                     (number <bps> ^value <nbps>)
                                     (number <boxes> ^value <nboxes>))))


")
(defconstant *propose-superobjects-doc-string* "
4.7.2. propose-superobjects
  Syntax:
          (propose-superobjects NAME
             [:space SPACE]
             [:group-name {GROUP-NAME | ID}]
             [:group-type {success | failure | ID}]
             [:when (COND+)]*
             :object-type {problem-space | state | operator}
             {:object ID}+)

  Semantics: Propose a set of problem spaces, initial states, or operators  for
the  supergoal as the result of problem solving in the specified space.  The TC
applies after a final state has been reached in the subspace, as recognized  by
a  goal-test-group  TC (Section 4.6).  The :group-name and :group-type keywords
are the same as for result-superstate (Section 4.7.1).

  The :object keywords specify the identifiers of the objects to be proposed in
the  superspace.    The  identifier  variables  must  be  bound  in  the  :when
conditions.   The  :object-type  keyword  specifies  whether  the  objects  are
problem-spaces, states (to be considered as initial states), or operators.


  Example:
          (propose-superobjects propose-move*psu
              :space propose-move
              :group-type success
              :object-type operator
              :object <o>
              :when ((state ^move-op <o>)))
")
(defconstant *operator-control-doc-string* "
4.5.3. operator-control
  Novice users are advised to skip this section.
  Syntax:
          (operator-control NAME
             [:space SPACE]
             [:when (COND+)]*
             :control-spec CONTROL-SPEC)

          CONTROL-SPEC ::= OP-COND | (MODIFIER+ CONTROL-SPEC+)

          MODIFIER ::= CONTROL-SPEC-TYPE | CONTROL-SPEC-MODIFIER |
                       OPERATOR-MODIFIER

          CONTROL-SPEC-TYPE ::= :seq | :or | :loop

          CONTROL-SPEC-MODIFIER ::= :swp | :optional | :when (COND+) |
                                    :exit-when (COND+) | :loop-name SYMBOL

          OPERATOR-MODIFIER ::= :ind | :ra | :oot

  Semantics: The operator-control TC provides a simple way to specify  operator
control   in  situations  where  operator  sequencing  is  largely  procedural.
Operator  control-specs  are  similar  to  regular  expressions  defining   the
allowable  operator  sequences  in  a  problem  space.  The 'characters' in the
regular expression are TAQL OP-CONDs, conditions that match against  operators.
Operators  that  are  inconsistent  with  the  regular expression are rejected.
Actually, operator control-specs are somewhat  more  complicated  and  powerful
than true regular expressions, but this is still a useful way to think of them.

  The  :when  and  :space  keywords  in  the  TC  specify  when  the associated
control-spec should be activated.  Thus  a  control-spec  need  not  always  be
active  starting  from  the  initial  state  in  a  space.  Once activated, the
control-spec remains in effect until either  it  has  run  to  completion,  the
exit-control-spec  operator  is  selected,  or  a final state is reached in the
problem space.  It is an error for more than one control-spec to be  active  at
once,  but  several  control-specs  may be activated over the course of problem
solving in a space.

  Once a control-spec has been activated, TAQL rejects  all  proposed  opertors
that  are inconsistent with the operator sequencing defined by the control-spec
(except for interrupt operators, described below).    The  operator  sequencing
defined by a control-spec depends on its CONTROL-SPEC-TYPE, as follows:

   1. If the control-spec has the form OP-COND, then only operators
      matching the OP-COND can be selected, and after one of them has been
      selected the control-spec exits.


   2. If the control-spec has the form (:seq <op-spec-1> ... <op-spec-n>),
      then the control-specs <op-spec-1> through <op-spec-n> are activated
      one after the other.


   3. If the control-spec has the form (:or <op-spec-1> ... <op-spec-n>),
      then the sequence of operators applied is consistent with one or
      more of the control-specs <op-spec-1> through <op-spec-n>.


   4. If the control-spec has the form
      (:loop <op-spec-1> ... <op-spec-n>), then TAQL will repeatedly
      select operator sequences consistent with one of <op-spec-1> through
      <op-spec-n> until the loop exits (see below for a description of the
      ways a loop can exit).  (:loop <op-spec-1> ... <op-spec-n>) is
      entirely equivalent to (:loop (:or <op-spec-1> ... <op-spec-n>)).

  A  control-spec  of  the form (MODIFIER+ CONTROL-SPEC+) must aways specify at
least one modifier keyword, and the keywords must appear at  the  beginning  of
the list.  A control-spec that does not begin with a keyword will be assumed to
be an OP-COND.  At most one of :seq/:or/:loop can be used at once, and if  none
is given :seq is assumed.

  Control-spec-modifiers:   The   :swp,   :optional,   :when,  :exit-when,  and
:loop-name keywords modify control-spec behavior as follows:

     :swp            :Swp  is  short  for  'select  when   proposed',   and   a
                     control-spec with this modifier will be activated when its
                     initial operator is proposed (a control-spec may have more
                     than one initial operator, for example either x or y could
                     be the initial operator of (:or x y)).   For  example,  in
                     the  control-spec  (:or  (:seq  a b) (:seq :swp c d)), the
                     (:seq :swp c d) branch will be taken when  operator  c  is
                     proposed,   even  if  operator  a  is  proposed  as  well.
                     However, if a is proposed but not c, then it will take the
                     (:seq  a  b)  branch.   If multiple :swp branches could be
                     active at the same time, there will be  a  tie  among  the
                     initial  operators  of  those  branches  (unless  there is
                     sufficient additional search control knowledge to  make  a
                     selection among them).

     :optional       The   control-spec  is  entered  if  any  of  its  initial
                     operators are proposed, else it is skipped.  Also,  if  an
                     optional  control-spec  is  followed  in  a  sequence by a
                     select-when-proposed control-spec, and the latter control-
                     spec  could  be  entered  (one of its initial operators is
                     proposed), then  the  optional  control-spec  is  skipped.
                     Loops are always implicitly optional.
     :when           If   a   control-spec   specifies  :when  conditions,  the
                     control-spec will only be activated when those  conditions
                     hold.   The conditions constrain only the selection of the
                     control-spec's initial operator, and are not inherited  by
                     all sub-control-specs.  :When implies :optional.

     :exit-when      If   the   :exit-when   conditions   are   true  when  the
                     control-spec would normally be entered,  the  control-spec
                     will be skipped.  For a loop, when the conditions are true
                     at the beginning of a loop iteration, the loop will  exit.
                     :Exit-when implies :optional.

     :loop-name      This keyword can only be used in loops, and its value must
                     be a non-nil, non-variable symbol.   See  the  section  on
                     exiting loops, below, for an explanation of its use.

  Operator-modifiers:

     :ind            :Ind  is  short  for  'indifferent'.   It states that when
                     multiple  initial   operators   are   proposed   for   the
                     control-spec,  they  are  indifferent.    For  example, in
                     (:or :ind a (:seq b c)), all proposed instances of a and b
                     will  be  indifferent  as  the  initial  operator  of  the
                     control-spec.  In the absence of :ind (or other additional
                     search  control knowledge), there would be a tie among all
                     the instance of a and b proposed.

     :ra             :Ra is short for 'reject after'.  It can only be used in a
                     control-spec  whose  only  element  is  an OP-COND, and it
                     causes the selected operator instance to never be selected
                     again after it has applied.

     :oot            :Oot is short for 'one of type'.  It can only be used in a
                     control-spec whose only element is an OP-COND, and once  a
                     matching  operator has been selected and applied, no other
                     operator matching that OP-COND will be selected as long as
                     there is an active control-spec.

  Exiting a loop: There are five ways a loop can terminate:

   1. A final state is reached in the problem space.

   2. The   loop   control-spec  gave  :exit-when  conditions,  and  those
      conditions were satisfied at the beginning of  the  loop  iteration.
      Or,   the   loop  control-spec  gave  :when  conditions,  and  those
      conditions  were  not  satisfied  at  the  beginning  of  the   loop
      iteration.

   3. At  the  beginning  of a loop interation, none of the loop's initial
      operators are proposed.

   4. A control-spec with the :swp keyword follows the loop,  and  at  the
      beginning  of  a  loop  iteration,  one  of  the :swp control-spec's
      initial operators is proposed.

   5. An exit-loop directive associated with the loop applies (see below).

  Aside from reaching a final state in the problem space  and  exiting  via  an
exit-loop  directive,  a loop will only exit when it is at the beginning of one
of its iterations.

  TAQL provides two compile-time directives, exit-loop and next-iteration, that
can appear wherever operators can appear in a control-spec.  Exit-loop causes a
loop to exit, and next-iteration causes control to skip to the beginning of the
next  loop iteration.  To cause an exit or skip in an outer loop from within an
inner loop, use :loop-name to name the outer loop, and give that  name  as  the
^loop-name parameter of exit-loop or next-iteration in the inner loop.

  For  example,  suppose  we  have  operators  inc-count1  and  inc-count2 that
increment the state  attributes  count1  and  count2,  respectively,  operators
reset-count1 and reset-count2 that set count1 and count2 to zero, respectively,
and another operator named other-operator.  Then the control-spec
          :control-spec
              (:seq reset-count1
                  (:loop :loop-name inc-loop
                         :when ((state ^count1 < 2))
                         (:seq inc-count1
                               reset-count2
                               (:loop :when ((state ^count2 < 2))
                              (:seq inc-count2
                                  (:when ((state ^count1 2 ^count2 1))
                                      (next-iteration
                                          ^loop-name inc-loop))))
                               other-operator)))

  will execute this sequence of operators:
          reset-count1
              inc-count1
              reset-count2
                  inc-count2
                  inc-count2
              other-operator
              inc-count1
              reset-count2
                  inc-count2          M note the incomplete loop iteration
              inc-count1
              reset-count2
                  inc-count2
                  inc-count2
              other-operator
  Interrupts: It is possible  to  interrupt  a  control-spec  and  select/apply
operators  that  are  not part of the control-spec.  Any proposed operator that
has an ^interrupt true augmentation will be eligible for selection and will  be
treated as a select-when-proposed operator.  Control-spec execution will resume
when no more interrupt operators are proposed.

  It is possible to halt and deactivate a control-spec at any time by proposing
the exit-control-spec operator as an interrupt.

  Operator control examples:
          (operator-control run-control-spec*control-spec-2
              :space run-control-spec
              :when ((superoperator ^control-spec 2))
              :control-spec (:seq (:or :ind a b) f))

  Below  are  several  example  control-specs  along  with  descriptions of the
operator sequences they will induce.  Assume that we  have  operators  named  a
through  f,  and that one instance of each is proposed.  Operator f is the only
one that does anything we will be concerned with M it makes  the  space's  goal
test  become  satisfied.    Suppose  also  that we have operators inc-count and
exit-count-loop.  Inc-count is always proposed, and increments a counter on the
state,  which  is initially zero.  Exit-count-loop is proposed when the counter
reaches four, and when it applies it satisfies the space's goal test.

(:seq (:or :ind a b) f)
                First apply either a or b, choosing one indifferently, and then
                apply f.  If the :ind was  left  out,  there  would  be  a  tie
                between operators a and b.


(:seq a (:loop :ind (:seq (:ra b) c) (:ra d)) f)
                First a, then both b/c and d (in either order), and finally  f.
                Because  of  the :ra's (reject-after) inside the loop, the loop
                exits from exhaustion after both b/c and d have applied.


(:or (:seq a b) (:seq :swp a f))
                First a, then f.  It should not get a tie between b and f after
                a applies, because the :swp forces it to choose  the  a/f  path
                once a has been selected..  If the :swp was not there, it would
                get a tie between b and f.


(:seq (:loop inc-count) (:swp exit-count-loop))
                Apply   inc-count  four  times,  then  exit-count-loop.    When
                exit-count-loop is proposed (when the count reaches four),  the
                :swp associated with it forces the loop to exit.


(:seq (:optional a) (:optional b) (:swp f))
                Apply f.  The :swp associated with f causes  the  two  optional
                specs in the sequence to be skipped.
")
;;;
;;; RELEASE-NOTES:
;;;

(defconstant *release-notes.3.1.4* "
 Release notes for TAQL 3.1.4

 Gregg Yost and Erik Altmann

 Started 20 March 1991
 Last modified 4 September 1991

 --------------------------------------------------------------------

 TAQL 3.1.4 runs in Soar 5.2.0 with the 12 November 1990 (or later) patch file.

 A complete manual accompanied the TAQL 3.1.3 release.  These notes detail the
 differences between TAQL 3.1.4 and TAQL 3.1.3.

 Contents:

         0. Summary of incompatibilities with TAQL 3.1.3.
     U   1. Introduced the notion of code segments.
     U   2. Added problem space declarations and space models.
     U   3. Added type declarations and type checking.
     Ui  4. New condition/action syntax.
     ui  5. Changed argument syntax for Soar and TAQL interface commands.
     U   6. Tracing production components while loading.
     u   7. Added symbolic value 'unknown' to the evaluate-object TC.
      i  8. Changed the 'verbose' command to 'taql-verbose'.
     u   9. Enhanced :cycle-when in evaluate-object.
     u  10. Added duplicate TC warning option.
     u  11. Added an easier way to load TAQL libraries.
     u  12. Added :buffer-in-goal keyword to edit clauses.
     u  13. Added :actions and :sliding-actions keywords to apply-operator and
            result-superstate.
     u  14. Added :actions keyword to the augment TC.

        Appendix 1:  TAQL Problem Space Models
	Appendix 2:  TAQL Data Models

     Legend:
       U = upgrade
       I = incompatibility
       Case of prefix approximately indicates magnitude.

 ---------------------------

 0. Summary of incompatibilities with TAQL 3.1.3.

   - The :default keyword to TAQL and Soar interface functions has been
     replaced.  See item 5 below.

   - Attribute names containing periods will be interpreted as attribute
     paths.  See item 4 below.

   - TAQL's 'verbose' command has been renamed 'taql-verbose'.  See item 8
     below.

 1. Introduced the notion of program segments.

    TAQL provides a way to assign group together related TCs and
    declarations into subsections of a TAQL program.  These groupings
    are called 'segments'.  Two new commands, 'begin-segment' and
    'end-segment', are used to define a segment.  A third command,
    'segment-group', is used to assign a name to a group of segments or
    segment groups.

    A. (begin-segment [SEGMENT-NAME])

      This command defines the beginning of a segment.  Everything
      that appears before the next begin-segment or end-segment is
      part of the named segment.  SEGMENT-NAME must be an unquoted symbol.
      Segments cannot be nested: if begin-segment is called again
      before end-segment is called to close the current segment, the
      current segment is closed before entering the new segment.
      However, you can reopen a segment after it has been closed just
      by issuing another begin-segment command for it.

      All code that is not part of some other segment is part of a
      predefined segment named USER.

      With no arguments, begin-segment just returns the name of the
      current segment.  Otherwise, begin-segment returns the name
      of the newly-entered segment, or nil if an error was detected.

      There is a segment named TAQL that contains declarations
      for objects that TAQL uses internally, TAQL's runtime support
      TCs and productions, and other internal TAQL components.

      There is a segment named SOAR that contains declarations for
      standard Soar objects, including both architectural objects like
      goals and impasses and objects that are used in default.soar
      (such as evaluations).  It also includes the declarations for
      objects associated with SoarIO and the function package (which
      are used by TAQL's text and expression data types, respectively).

    B. (end-segment)

      Close the current segment.  Everything appearing between
      end-segment and the next begin-segment is treated as part
      of the USER segment.

    C. (segment-group [SEGMENT-GROUP-NAME [SEGMENT-SPEC+]])

      SEGMENT-SPEC ::= SEGMENT-GROUP-NAME | SEGMENT-NAME | :all | :latest

      Segment-group allows the user to assign a name to a set of segments.
      Segment groups can contain arbitrary sets of segments, and the same
      segment can be part of several different groups.

      With no arguments, it returns a list of defined segment group names.

      With one argument, it returns a list of the segments in the named group.

      With more than one argument, it defines the first argument to be
      a group name for the segments and segment-groups named in the
      remaining arguments.  The remaining arguments must be
      already-defined segment or group names.  The keyword :all stands
      for all currently loaded segments, and the keyword :latest
      stands for the segment something was added to most recently
      (initially the user segment).

 2. Added problem space declarations and space models.

    The space model facility contains commands for declaring problem
    spaces, printing the problem space structure inferred from a loaded
    system, and checking for inconsistencies between declared and inferred
    problem space information.  A full description appears later in this
    document.

 3. Added type declarations and type checking.

    We call this TAQL's data model facility.  A full description appears
    later in this document.

 4. New condition/action syntax.

    TAQL now permits two more concise ways to specify Soar conditions
    and actions in a TC.  The only incompatibility with the previous
    syntax is that attribute names that contain a period are now
    interpreted differently.

    The new syntax has two components (both of which can be used in either
    conditions or actions):

    A. Attribute paths

       If an attribute name has one or more periods in it, the parts
       separated by periods are interpreted as the attribute names in
       a path through subobjects to the attributes value.  For
       example,

          (state ^perceptions.object.name <name>)

       would expand to the following (possibly with different identifier
       variables):

          (state ^perceptions <p*17>)
          (perceptual-data <p*17> ^object <o*23>)
          (object-info <o*23> ^name <name>)

      The class names are obtained from type declarations in the data model.
      It is an error to use this form of syntax if the value types for
      attributes in the path have not been declared.

      Attribute paths can only appear immediately after an ^.  For example,
      (state ^ << a.b c.d >> <value>) and (state ^ {<> <x> a.b} <value>)
      are both illegal.  Currently, attribute paths can't appear as negated
      attributes (e.g. (state - ^x.y 2)), though they can appear in
      conjunctive negations (e.g. -{(state ^x.y 2)}).

    B. Structured value specifications

       When the value of an attribute is an object, the TAQL programmer
       can provide a list describing that object in place of an identifier
       variable linking to that object.  For example,

          (state ^perceptions (^object (^name <name> ^ontop table)))

       would expand to something like

          (state ^perceptions <p*17>)
          (perceptual-data <p*17> ^object <o*23>)
          (object-info <o*23> ^name <name> ^ontop table)

       A structured value specification can optionally contain class name
       and/or identifier fields.  For example

          (state ^perceptions (vision-data ^object (<b> ^name <name>
                                                        ^ontop table))
                 ^gripper-not-holding <b>)

       would expand to something like

          (state ^perceptions <p*17> ^gripper-not-holding <b>)
          (vision-data <p*17> ^object <b>)
          (object-info <b> ^name <name> ^ontop table)

       If the class is omitted, it must be inferred from type
       declarations as for attribute paths.  If the class and/or
       identifier appears, it can be anything ordinary Soar syntax
       allows in these positions (including << ... >> or { ... }).

       This kind of subobject specification cannot appear within { ... } or
       << ... >>, as for attribute paths.

       If the TAQL programmer specifies only one of the class and identifier
       fields in a structured value specification, there is an ambiguity
       in the syntax as to whether it is a class or an identifier.  TAQL
       resolves the ambiguity by assuming that it is an indentifier if
       it is a variable, otherwise a class.  Thus if one wants to use this
       syntax with a variable class, one must also specify the identifier
       variable.

    C. General comments and programming hints

       1. When using a structured value specification or attribute path
          in an operator object, it is good practice to place them
          after the ^name attribute.  When TAQL must consult the data
          model to fill in class information, it needs to know the
          operator name first so that it can look up operator-specific
          declarations.

       2. Structured value specifications are preferable to attribute paths
          in several situations.

            - When you need to test several attributes of the same subobject.
            - When you need to bind the identifier variable of the subobject
              for use elsewhere.
            - When you have must specify the class of the subobject because
              you have not declared it.

          Attribute paths and structured value specifications can be mixed.
          That is, a structured value specification can be the value of an
          attribute path, and a structured value specification can contain
          attribute paths.

 5. Changed argument syntax for Soar and TAQL interface commands.

    The NAME-LIST arguments formerly accepted by Soar and TAQL commands
    (see Section 6.8 of the TAQL 3.1.3 manual) has changed.  The new
    syntax is:

      NAME-LIST ::= NAME-SPEC* [:exclude SEGMENT-OR-TYPE-SPEC*]
                               [:dont-exclude-taql]

      NAME-SPEC ::= SEGMENT-OR-TYPE-SPEC | TC-NAME | OTHER-ARG

      SEGMENT-OR-TYPE-SPEC ::= SEGMENT-SPEC
                               | :types TC-TYPE-NAME+
                               | ([SEGMENT-SPEC] :types TC-TYPE-NAME+)

      OTHER-ARG can be anything other than one of the things explicitly
      listed here, for example a Soar production name.

    The argument list specifies a set of Soar production names.  A
    TC-NAME specifies the SPs the TC compiled into.  A SEGMENT-SPEC
    specifies the names of productions compiled from TCs in the
    indicated segment(s).  An argument of the form
    ([SEGMENT-SPEC] :types TC-TYPE-NAME+) specifies the SPs compiled
    from TCs of the listed TC types (e.g. prefer, compare, propose-operator)
    in the indicated segment(s).  If the optional SEGMENT-SPEC is missing, it
    defaults to all existing segments.  A SEGMENT-OR-TYPE-SPEC of the form
    ':types TC-TYPE-NAME+' is equivalent to (:all :types TC-TYPE-NAME+).

    Any OTHER-ARGS are passed on to the interface command being invoked.

    If any SEGMENT-OR-TYPE-SPEC arguments appear anywhere after an :exclude
    keyword, then the specified segments/types are excluded from
    SEGMENT-OR-TYPE-SPECs appearing prior to the first :exclude
    keyword.  Productions expanded from explicitly listed TC names, and
    productions names appearing in OTHER-ARGs, are never excluded.

    Unless the :dont-exclude-taql is given, the TAQL segment is excluded.
    This excludes internal TAQL items that most users would not want to see.

    The :default keyword supported in TAQL 3.1.3 is no longer supported.
    It is superceded by :dont-exclude-taql.  This is the only
    incompatibility in the new form of interface command arguments.

 6. Tracing production components while loading.

    The new trace-load command allows users to indicate that they
    should be notified whenever TAQL loads a TC that compiles to a
    Soar production containing selected classes, attributes, or
    values.  The untrace-load command turns off such load tracing.

      (trace-load [ CLASS-TRACE [ ATT-TRACE [ VALUE-TRACE ] ] ])

      (untrace-load)

    CLASS-TRACE and ATT-TRACE must be either symbolic class/attribute names,
    the symbol *unknown*, or one of the symbols * or ?.  In addition,
    CLASS-TRACE may have the form 'operator*<op-name>', which specifies
    operator objects for the operator named <op-name>.  'Operator**unknown*'
    specifies operator objects whose names TAQL cannot uniquely determine.
  
    VALUE-TRACE must be a primitive value, one of the symbols * or ?, or
    a list of the form (primitive TYPE-NAME*), (object TYPE-NAME*), or
    (function FUNCTION-NAME*).  The semantics of these patterns are
    described below.
  
    ATT-TRACE and VALUE-TRACE default to * when not given.  There is
    no default for CLASS-TRACE -- trace-load with no arguments prints
    out the currently active trace specifications.
  
    The effect of trace-load is to print a trace message when the items
    matching the trace specification are encountered in a production
    when it is loaded.  This can be very convenient when using
    print-data-model to print the data model inferred from a large production
    set.  Trace-load can be used to locate the TCs that selected classes,
    attributes, and values appeared in.
  
    The symbol *unknown* stands for classes, attributes, and value types
    that TAQL could not determine uniquely.  For example, TAQL treats
    variable classes and attributes as *unknown*.
  
    The symbols * and ? are wildcards.  The difference is that for *, the
    associated component need not exist to match, while for ? it does.
    For example, (trace-load state color *) will print a trace message
    every time TAQL encounters the color attribute used in a state object,
    even if no value was given.  It will only print the class and attribute
    name in the trace message.  On the other hand, (trace-load state color ?)
    will only print a trace message if some value appear for the color
    attribute on a state.  It will print the class, attribute, and value.
  
    ATT-TRACE and VALUE-TRACE default to * when not given.  There is
    no default for CLASS-TRACE -- trace-load with no arguments prints
    out the currently active trace specifications.
  
    There are three special value trace specifiers:
  
      (primitive TYPE-NAME*) -- This matches primitive values that
          are of one of the specified primitive types.  If no type
          name are listed, then all primitive values match.
  
      (object TYPE-NAME*) -- This matches structured values that
          are of one of the specified types.  If no type names are
          listed, then all structured values match.
  
      (function FUNCTION-NAME*) -- This matches values that are function
          calls (e.g. compute) to functions named in the list.  If no
          function names are given, then all function calls match.
  
    Untrace-load removes all current trace specifications.

    USAGE HINTS:

       A common reason to use trace-load is to find out which TCs *unknown*
       values came from in an inferred data model that you have printed out.
       Often, perfectly correct programs will have some *unknown* items in
       their inferred data model.  However, it is a good idea to always
       check these entries items manually, as TAQL cannot perform type
       checking on *unknown* items.

       Issuing the following four trace-load commands and then reloading
       your program will cause TAQL to print messages identifying every
       TC that contributed to an *unknown* item in the inferred data model:

          (trace-load *unknown*)
	  (trace-load * *unknown*)
	  (trace-load * * *unknown*)
	  (trace-load * * (object *unknown*))

 7. The evaluate-object TC now takes a new SVALUE, 'unknown'.  If
    'unknown' is the value of a :symbolic-value clause, and that is the
    only value clause that applies, then evaluate-state will impasse
    if it is selected.  (Evaluate-state is selected when a value
    clause and a :what clause apply at the same time.)  This allows
    Soar to problem-solve in a subgoal to come up with an evaluation. 

    To return an evaluation from a subgoal, the 'evaluation' object on
    the evaluate-state operator must be explicitly given a
    ^symbolic-value or ^numeric-value attribute.  For example, the
    following clauses could appear in a result-superstate or
    apply-operator TC in the subgoal:

        :when ( [conditions for failure] )
        (edit :what (:none evaluation <e>)
            :when ((superoperator ^evaluation <e>))
            :new (symbolic-value failure))

 8. Changed the 'verbose' command to 'taql-verbose'.

    We changed the name of the 'verbose' command to 'taql-verbose' to
    avoid name conflicts with other packages in some Lisps.

 9. Enhanced :cycle-when in evaluate-object.

    The :cycle-when keyword in evaluate-object specifies a set of
    conditions that should be matched across all pairs of goals in a
    lookahead search.  If there is a pair of goals for which they
    match, that pair defines a state cycle in the lookahead search.
    Previously, if the conditions contained variables, TAQL used
    identical variables in both of the goal contexts.  This kept
    :cycle-when from detecting cycles when the conditions involved
    state substructure that was copied during lookahead -- the
    identifiers changed as a result of copying, so the same variables
    would not match across contexts.  In TAQL 3.1.4 it is possible to
    solve this problem.  TAQL will replace any variables in the
    :cycle-when conditions that begin with '?'  (for example, <?block>)
    by separate variables in the two contexts the conditions are matched
    across.  Thus users can specify which variables should be identical
    in the two contexts, and which should be different.

 10. Added duplicate TC warning option.

    There is a new command that controls whether or not TAQL will print
    a warning message when a TC with the same name as a already-loaded
    TC is loaded.  This command makes it much easier to detect duplicate
    TC names when a file is loaded.  Its syntax is:

       (duplicate-tc-warn [:yes | :no])

    An argument of :yes turns on warnings, and an argument of :no turns
    them off.  Initially, (duplicate-tc-warn :no) is in effect.

    With no arguments, duplicate-tc-warn reports the current setting of
    the option.

 11. Added an easier way to load TAQL libraries.

    TAQL provides a number of TC libraries.  Previously, these libraries
    were rather painful to load -- the user had to explicitly load them
    from the directory that is the value of *taql-library-directory-pathname*.
    Now there is an easier way:

       (use-taql-library LIBRARY-NAME)

    loads the library indicated by LIBRARY-NAME, where LIBRARY-NAME is
    the name of one of the libraries in the library directory.  Currently,
    it must be one of EXPR, LIST, or TEXT.  If the named library has already
    been loaded, it is not loaded again.

 12. Added :buffer-in-goal keyword to edit clauses.

    WARNING:  This is an experimental keyword.  We're not entirely sure
    how it will work out yet, and it may change or disappear in future
    releases.  But, if you are having certain chunking problems with TAQL
    operator applications, read on.  Novices who are not familiar with
    the add buffers TAQL uses in applying operators should definitely
    skip this section.

    Edit clauses (i.e., (edit ...)) in apply-operator and result-superstate
    now take a new keyword, whose syntax is

       [:buffer-in-goal { goal | supergoal | top-goal | ID }]

    It specifies in which goal the edits in the edit clause should buffer
    their adds.  The default is 'goal' for apply-operator and 'supergoal'
    for result-superstate.  If an ID is given, it must be the identifier
    variable of a goal object bound elsewhere in the TC's conditions.

    We think (remember, this is experimental) that the right goal to
    buffer adds in is the highest goal in which the object being edited
    exists in.  That way, all the chunks learned from editing the object
    will preserve buffering.  If an object exists in a goal higher than
    the one buffering occurs in, the chunks for that goal will not
    buffer the edits, because the buffer only exists in the subgoal.
    This can lead to undesired differences between how a TAQL program runs
    before and after learning.  Unfortunately, the TAQL compiler cannot
    in general determine which is the highest goal an object exists in,
    so instead we provide this new keyword to let users specify it
    manually.

    So, if you're having chunking problems of the sort described here,
    give this keyword a try, and let us know how it works.

 13. Added :actions and :sliding-actions keywords to apply-operator and
     result-superstate.

    The syntax of these keywords is

      [:actions (SOAR-ACTION+ [:when (COND+)]*)]+
      [:sliding-actions (SOAR-ACTION+ [:when (COND+)]*)]*

    The intent of these keywords is to let people specify operator applications
    directly in terms of raw Soar right-hand-side actions.  This allows a
    number of things that TAQL did not formerly support, such as editing
    multiple objects with a single keyword and using attribute preferences
    other than + and -.

    Actions specified in an :actions keyword are like one-shot edits --
    they are constrained to only apply in the first wave of editing
    after an operator is selected.  :Sliding-actions are like sliding
    edits -- they can apply whenever their conditions match, and are useful
    for operators that apply over multiple elaboration cycles.

    The productions generated for :actions and :sliding-actions are
    not subject to the add buffering TAQL uses in edit clauses.  A negative
    consequence of this is that productions generated from :sliding-actions
    don't mix well with edit clauses for the same operator -- the
    :sliding-actions productions could fire on an inconsistent state where
    any removes specified in the edits have taken effect, but the buffered
    adds have not yet been installed.

    The SOAR-ACTIONs in :actions and :sliding-actions can contain the
    following TAQL extensions:

       - New TAQL syntax:  attribute pathnames and structured value
         specifications.

       - Data type extensions:  data macros and data type directives.

    Also, SOAR-ACTIONs in :actions and :sliding-actions are subject to the
    same automatic context-object linkage as TAQL conditions are.  So, if
    the class field of an action is the name of a context object, and the
    identifier variable is omitted, TAQL will automatically generate additional
    conditions to link the action to the specified context object.

    Example:

       ;; Move the monkey to <place>.  The move operator is only proposed
       ;; for locations other than the monkey's current location, so
       ;; we'll never have <place> = <old-place>.
       ;;
       (apply-operator mab*move
         :space mab
	 :op (move ^to <place>)
	 :actions ((state ^at <old-place> - <place> +)
	           :when ((state ^at <old-place>))))

 14. Added :actions keyword to the augment TC.

    An :actions keyword (with the same syntax as in apply-operator) has been
    added to the augment TC.  It allows one to create entailments on
    multiple objects using a single keyword, and also allows one to
    entail all attribute preferences, not just +.  And, in general, it
    allows you to create any entailments that Soar lets you create.

    Example:

       (augment paint*a*state*default-color-is-red
         :space paint
	 :actions ((state ^color red + < )))

Appendix 1:  TAQL Problem Space Models

I. PROBLEM SPACE DECLARATIONS

   A. Syntax

      (defspace SPACE-NAME
        :initial-state-proposal       IMPLEMENTATION-TYPES
    :entailment                   IMPLEMENTATION-TYPES
        :goal-testing                 IMPLEMENTATION-TYPES
        :result-returning             IMPLEMENTATION-TYPES
        :selection                    IMPLEMENTATION-TYPES
        :evaluation                   IMPLEMENTATION-TYPES
        :default-operator-proposal    IMPLEMENTATION-TYPES
        :default-operator-application IMPLEMENTATION-TYPES
        :operators            none | ( SPACE-OPERATOR-DECL+ )
    )

      IMPLEMENTATION-TYPES ::= none | direct | subspace
                                    | ( {direct | subspace}+ )

      SPACE-OPERATOR-DECL ::= OPERATOR-NAME |
                              ( OPERATOR-NAME [:propose {none | direct}]
                                  [:apply IMPLEMENTATION-TYPES] )

   B. Semantics

      1. Declarations that appear between start-default/stop-default will
         remain in effect after an excise-task.  Other declarations
     are removed by excise-task.  Excising a segment (see Section 5
     of these release notes) will excise declarations in that
     segment.  The excise-space-model command (see Section V)
     excises just the space model information in a segment.
     (excise SEGMENT-SPEC) and excise-space-model will not remove
     space declarations that appeared between start-default and
     stop-default.

      2. A redeclaration of a space overwrites the previous declaration
         and issues a warning message.  The updated declaration is installed
     in the same segment as the old declaration, and the current segment
     is set to that segment.

      3. Most of the keyword arguments to defspace correspond to the PSCM
         roles a TC can play in problem solving.  Page 2 of the TAQL 3.1.3
     User Manual gives the mapping between TC types and their PSCM roles.
     The value of each of these keywords declares how that role is
     implemented:

       'none' if no TCs implement the role
       'direct' if the role is implemented directly in TCs
       'subspace' if the role is implemented by TCs in a subspace
       '(direct subspace)' if the role is implemented in both of these ways

     TAQL does not provide special mechanisms for doing entailment,
     goal-testing, result-returning and evaluation is subspaces (though
     operators applied in subgoals may contribute to any of these roles).
     Therefore 'subspace' is not accepted as a value for the keywords
     corresponding to these four roles.

     The :operators, :default-operator-proposal, and
     :default-operator-application keywords do not correspond directly
     to PSCM roles, and are described below.

      4. The :operators keyword declares the operators that are
         defined or used by the space.  Its value is either 'none'
     if there are no operators, or a list of operator declarations.

     An operator declaration consists of either just the operator name
     or a list giving the operator name along with specifications of
     how it is proposed and/or applied (specified by the :propose and
     :apply keywords, respectively).  :Apply accepts the values
     'none', 'direct', and 'subspace', as for the other keywords.
         :Propose accepts only 'none' and 'direct', since TAQL does not
     provide a way to define operator proposal subspaces that are
     specific to a particular operator.  If the space's operators are
     proposed in a subspace, 'subspace' must be one of the values of the
     :default-operator-proposal keyword.

     If the :propose or :apply keywords are not specified for an operator,
     their values default to the values of the :default-operator-proposal
     and :default-operator-application keywords, respectively.

II. INFERRED SPACE MODELS

   As TCs are loaded, TAQL builds an internal model of the problem spaces
   in a system.  It records which TCs belong to a problem space (as indicated
   by the TC's :space keyword) and the PSCM roles that they play.  It also
   records the connections among problem spaces, as defined by any
   propose-space TCs.

   The print-space-model (see Section III) and print-space-graph (see
   Section VI) commands print out the inferred models.  The
   check-spaces command (see Section IV) checks for inconsistencies
   between declared and inferred information for a space.

   Print-space-model prints information in the form of defspace declarations
   that users can insert into their TAQL source code.  We expect that
   a common mode of operation will be for users to first write their code
   without declarations.  After loading the program the first time,
   they will print out the inferred models and inspect them to see if
   their system contains all of the parts that they think it should.
   Then they will insert the printed defspace information as declarations
   in their source code.  When the system is loaded again, users can use
   the check-spaces command to compare declared and inferred information
   to detect errors and inconsistencies (TAQL calls check-spaces automatically
   whenever the user runs their system, if the space model has changed since
   check-spaces was last called).

   Every space model has an associated list of 'home segments'.  The home
   segment for a declared space model is the segment the defspace appeared
   in.  For inferred space models, every segment that contains a TC that
   directly plays some role for the space is a home segment for that space's
   model.  The concept of home segments is important for space model commands
   that can take segment names as arguments, such as print-space-model and
   excise-space-model.  See Section III for more details.

   If TAQL cannot determine what space a TC belongs to, the TC is placed
   in the inferred space model for a space named *unknown*.  Two situations
   where TAQL can't determine a TC's space are (1) when the TC doesn't have
   a :space keyowrd and (2) when the value of the :space keyword is a
   variable.
III. PRINTING PROBLEM SPACE MODELS

    TAQL provides a way to print out inferred and/or declared space model
    information for selected segments:  the print-space-model command.
    Print-space-model displays its results in the form of declaration
    commands.  Therefore, one can load a TAQL program (with no declarations),
    print out the inferred space models, and then include the printed
    information in the TAQL program as a declaration for when it is loaded
    in the future.  The advantage of explicitly declaring the information
    is that TAQL is able to do more extensive error checks when declarations
    are given.

    A. (print-space-model SPACE-MODEL-SPEC-ARGS)

           SPACE-MODEL-SPEC-ARGS ::= MODEL-ITEMS-SPEC*
                                 [:exclude MODEL-ITEMS-SPEC*]
                                     [:dont-exclude-taql]
                                     [:dont-exclude-soar]
                     [:no-tc-names]

           MODEL-ITEMS-SPEC ::= ( [ MODEL-TYPE [ ITEM-PART [ SEGMENT-PART ]]] )

           MODEL-TYPE ::= inferred | declared | all

       ITEM-PART ::= :all | SPACE-NAME | ( SPACE-NAME+ )

           SEGMENT-PART ::= SEGMENT-SPEC | ( SEGMENT-SPEC+ )

       For a definition of SEGMENT-SPEC, see Section 1 of these release notes.

       Print-space-model prints the space models specified by its arguments.
       Each MODEL-ITEMS-SPEC in the argument list indicates a set of
       space models, as follows:

         A MODEL-ITEMS-SPEC consists of a list of three items.  The
     first item specifies the type of model: inferred, declared,
     or all (both inferred and declared).  It defaults to 'all'.
     The second item specifies what problem spaces to include
     models for.  It defaults to :all.  The third argument
     constrains the set of segments from which model information
     is gathered.  It defaults to :all.  Only problem spaces that
     have one of these segments as a home segment (see Section II)
     will be included.

       Print-space-model prints all of the space models specified by
       MODEL-ITEM-SPECs that appear before the first :exclude keyword,
       excluding any models that are also specified by MODEL-ITEM-SPECs
       that appear after the first :exclude keyword.

       If both inferred and declared models are specified for the same space,
       the two models are merged and printed as a single model.

       If no MODEL-ITEMS-SPECs are given before an :exclude keyword,
       it defaults to (all :all :all).  Information on spaces from the
       TAQL and Soar segments is always excluded unless the
       :dont-exclude-taql or :dont-exclude-soar keywords are given,
       respectively.  This keeps it from printing internal information
       that most users probably aren't interested in.

       Normally, for inferred space models, print-space-model prints the
       names of the TCs in that space as comments under the appropriate
       defspace keyword.  The :no-tc-names argument suppresses this.
IV. ERROR CHECKING

   The (check-spaces) command checks for inconsistencies between the
   declared and inferred space models for a problem space.  It doesn't
   take any arguments.

   A. Inconsistencies result in warnings -- TAQL will load a production if
      the only errors it contains are inconsistencies with declarations.

   B. Only explicitly declared information is used for type checking.  TAQL
      can't warn you about problems if you don't declare your spaces.

   C. Whenever the user runs their system and some inferred space model has
      changed since check-spaces was last invoked, check-spaces is called
      automatically before the run begins.

   D. One interface functions affects what check-spaces warns about:

      1. (all-spaces-declared {:yes | :no} TYPE-NAME+)

     When :yes is given, check-spaces will issue a warning when
     there is inferred problem space information for a space that
     has not been declared.  When :no is given, TAQL will not warn
     about such spaces.  (all-space-declared :no) is in effect
     when TAQL starts up.  It is good practice to not rely on this
     option having its default setting, because a previously-loaded
     file may have changed it.  You should always declare explicitly
     whether or not all of your problem spaces are declared.

V. EXCISING SPACE MODELS

   The excise-space-model command removes selected space models.  It
   will not remove type information for declarations that appeared
   between start-default and stop-default.  Such declarations can only
   be removed by restarting TAQL.

   Its syntax is

      (excise-space-model MODEL-ITEMS-SPEC*)

   where MODEL-ITEMS-SPEC is as defined for print-space-model (see
   Section III).  The defaults are the same as for print-space-model,
   except that :exclude is not meaningful here.

VI. PRINTING PROBLEM SPACE CONNECTIONS

    As described in Section II, TAQL keeps track of the connections among
    problem spaces, as defined by propose-space TCs.  Users can print
    out this information using the print-space-graph command:

      (print-space-graph [TOP-SPACE-NAME])

    Print-space-graph presents its output in the form of a watch 0 trace,
    with impasses standing for the problem space connections.  It begins
    with the system's top space, which is named by the TOP-SPACE-NAME
    argument and defaults to 'top-space'.

    When the same problem space or operator appears more than once in
    the graph, the spaces beneath it are only printed once.  When
    subspaces are omitted for a later occurrence, it prints '[...]'
    following the name of the item.

    A problem space's operators are printed in alphabetical order, and may
    have no relation to the order in which they actually apply during a run.

    When there are missing propose-space TCs, there might not be a single
    graph rooted at the top space that includes every space in the system.
    In such a case, print-space-graph prints separate graphs for each
    disconnected component, until all connections have been printed.
    The check-spaces command doesn't check for spaces that are unreachable
    from the top space, so it is a good idea to use print-space-graph to
    make sure everything is connected up the way you think it should be.

VII. SUMMARY OF SPACE MODEL INTERFACE COMMANDS

   Section I

     defspace

   Section III

     print-space-model

   Section IV

     check-spaces
     all-space-declared

   Section V

     excise-space-model

   Section VI

     print-space-graph

Appendix 2:  TAQL Data Models

I. TYPE DECLARATIONS

   A. Syntax

      (defobject TYPE-NAME
        [:global {yes | no}]
        [:all-info-declared {yes | no}]
    ATTRIBUTE-DECL*)

      (defobject-merge TYPE-NAME
        [:global {yes | no}]
        [:all-info-declared {yes | no}]
        ATTRIBUTE-DECL*)

      (defoperator OPERATOR-NAME
        [:global {yes | no}]
        [:all-info-declared {yes | no}]
    ATTRIBUTE-DECL*)

      (defoperator-merge OPERATOR-NAME
        [:global {yes | no}]
        [:all-info-declared {yes | no}]
        ATTRIBUTE-DECL*)

      (use-type-defs
        [:global {yes | no}]
        [:all-info-declared {yes | no}]
    { TYPE-NAME | (SEGMENT-NAME TYPE-NAME+) }+ )

      (defenumeration TYPE-NAME VALUE+)

         - VALUE is a non-variable symbol, number, or string

      (defprimitive TYPE-NAME PREDICATE-NAME)

         - PREDICATE-NAME is the name of a function of one argument that can
       be applied to an attribute's (non-variable) value.  It must return
       non-nil if and only if its argument is to be considered a member
       of the type being declared.

      ATTRIBUTE-DECL ::= ATT-NAME
                       | (ATT-NAME [:type { TYPE-NAME | (TYPE-NAME+) }]
                           [:optional])

         - The form where you just give the attribute name is for when you
           don't want to declare the value type.

     - The :optional keyword is only meaningful in defoperator and
       defoperator-merge.

   B. Semantics

      1. Declarations that appear between start-default/stop-default will
         remain in effect after an excise-task.  Other declarations
     are removed by excise-task.  Excising a segment (see the Section 5
     of these release notes) excises declarations in that segment.
     The excise-data-model command (See Section IV) excises just
     the data model information in a segment.

     (excise SEGMENT-SPEC) and excise-data-model will not remove type
     declarations that appeared between start-default and stop-default.

      2. A declaration does not overwrite or merge with a previous declaration
         for that type.  Multiple declarations result in an error message.
     (Except for defobject-merge and defoperator-merge.  See below.)

      3. The defobject-merge command adds new attribute declarations to a
         previosuly declared object type.  If the object type has not already
     been declared, defobject-merge is equivalent to defobject.

      4. Giving defobject or defobject-merge an :all-info-declared
         keyword is equivalent to calling all-info-declared with the
     appropriate argument immediately after the declaration is
     loaded.  See Section II.D.3.

      5. If the value of the :type keyword in an ATTRIBUTE-DECL is a list,
         then the attribute's values may be of any of the named types.

      6. Defoperator and defoperator-merge are like defobject and
         defobject-merge, but declare attributes for specific operators
     (based on the value of their ^name attribute).  In addition to
     declaring attributes for individual operators, one can declare
     attributes for the generic operator type using
     (defobject-merge operator ...).  If an attribute is declared
     both for a specific operator and for operators in general,
     the specific declaration takes precedence.

     In defoperator and defoperator-merge, an ATTRIBUTE-DECL can contain
     the keyword :optional.  This declares that the named attribute need
     not be present when the operator is proposed.  By default, attributes
     declared for specific operators must be present when an operator
     object is proposed.  TAQL will issue a warning whenever a
     propose-operator or propose-task-operator TC creates and proposes
     an operator object that is missing a required attribute.

      7. TAQL predeclares one primitive type, named Primitive.  Every
         non-variable symbol, number and string is of type Primitive.
         It has an associated type predicate function named primitive-p.

      8. A type can have both local and global versions, as indicated by
         the value of the :global keyword (yes or no).  A global type
     declaration is accessible in all segments.  A local type declaration
     is only in effect in the segment that it appears in.  A local type
     declaration inherits any global attribute declarations that
     are in effect for the same type at declaration time.

     Local types are useful for object classes that are used in multiple
     spaces with different meanings -- state objects are a prime example.
     We expect that for multi-space systems, TAQL users will define a
     segment corresponding to each problem space, and will put a local
     state declaration in that segment declaring only the attributes
     used in that segment.  This will allow TAQL to catch errors it would
     miss if all state attributes from all segments were declared globally.

      9. The use-type-defs command imports types into the current segment
         from some other segment.  An argument of the form TYPE-NAME
     specifies the global definition of the type.  An argument of the
     form (SEGMENT-NAME TYPE-NAME+) specifies the local definitions of
     the named types in the named segment.  A TYPE-NAME of the form
     'operator*<op-name>', where <op-name> is the name of an operator,
     specifies the operator declaration for the named operator.

     Use-type-defs in effect does a defobject-merge or defoperator-merge
     in the current segment for the named types, taking the contents
     of the declarations from the segments they are imported from.
     The :global and :all-info-declared keywords to use-type-defs
     have the same meaning as they do for defobject and defoperator.

II. TYPE CHECKING

   A. Type mismatches result in warnings -- TAQL will load a production if
      the only errors it contains are inconsistencies with type declarations.

   B. Only explicitly declared information is used for type checking.  TAQL
      also builds an internal model of the types/objects actually used in
      a loaded program, but this inferred model is not used for type
      checking.  However, the programmer can print this inferred model and
      later insert it as an explicit declaration.  See the description
      of print-data-model in Section III.

   C. TAQL will issue a warning message when a value-type declaration for
      an attribute doesn't match the type used in a TC.

   D. Three interface functions affect the warning messages that are generated
      during type checking:

      1. (all-types-declared {:yes | :no})
      
         a. A :yes argument causes TAQL to issue a warning when it encounters
        an object whose type has not been declared.  A :no argument
        inhibits such warnings.

         b. The default setting is :no.  Programs should not rely on the
        default, however, because a previously-loaded file may have
        overridden the default.

      2. (all-operators-declared {:yes | :no})
      
         a. A :yes argument causes TAQL to issue a warning when it encounters
        an operator object with a name that has not been declared using
        defoperator.  A :no argument inhibits such warnings.

         b. The default setting is :no.  Programs should not rely on the
        default, however, because a previously-loaded file may have
        overridden the default.

      3. (all-info-declared {:yes | :no}
                            { TYPE-NAME | (SEGMENT-NAME TYPE-NAME+) }+ )

         a. TYPE-NAME specifies the global definition of the named type.
        (SEGMENT-NAME TYPE-NAME+) specifies the local definitions of the
        named types in the named segment.

         b. It is an error if one of the listed types is undeclared or
        declared primitive.

     d. When :no is given, TAQL will NOT issue a warning when a TC uses
        an attribute for one of the named types that was not declared
        as a legal attribute for that type.

        This is the default for the global goal, problem-space, state
        and operator types.

     c. When :yes is given, TAQL will issue a warning when a TC uses an
        attribute for one of the named types that was not declared
        for that type.

        This is the default for declared types other than the
        global goal, problem-space, state and operator types.

        TAQL currently includes global declarations for several types
        for which all-info-declared is :yes.  These types are

           boolean, column, digit, evaluation, final-state-info, fraction,
           impasse, integer, list, mapping, real, set, status,
           term and text.

            Most of these types appear in TAQL's data types (lists,
        sets, expressions and text -- see Section 5 of the TAQL 3.1.3
        manual).  If users use objects of these types with attributes
        other than the ones TAQL declares, they will get warning unless
        they either

                1. Set all-info-declared to :no for those types, OR
        2. Use defobject-merge to extend the global definitions
           for those types, declaring the other attributes that
           they use, OR
        3. Use defobject or defobject-merge to define local versions
           of those types declaring the other attributes that they use.

            Users can examine TAQL's declarations for these types by
        using the print-data-model command (see Section III).
        TAQL declares these types in either the TAQL or SOAR
        segment.  For example, to print the declaration for the list
        type, use

           (print-data-model (declared list (taql soar))
                             :dont-exclude-taql :dont-exclude-soar)

            TAQL also globally declares these operators:

               evaluate-object, evaluate-state, exit-control-spec,
           exit-loop, final-state, halt, leap, next-iteration
           and set-up-result-object.

     e. It is good practice to not rely on the defaults mentioned above,
        because a previously-loaded file may have overridden the default.
            (In particular, TAQL's libraries or support productions may
        change the settings for certain types.)  When you rely on a
        particular setting, you should declare it explicitly.

     f. A TYPE-NAME passed to all-info-declared can have the form
        'operator*<op-name>', where <op-name> is the name of an operator.
        This causes all-info-declared to apply to the defoperator
        information for the named operator.

III. PRINTING DATA MODELS

    TAQL maintains both inferred and declared data model information.
    Declared information is information given in type declarations (see
    Section I).  Inferred information is accumulated as TCs are loaded.
    The kinds of information that are inferred are the same kinds of
    information that can be declared:  what classes are used, what
    attributes are used with those classes, and what value types those
    attributes take.

    TAQL provides a way to print out inferred and/or declared data model
    information for selected segments:  the print-data-model command.
    Print-data-model displays its results in the form of declaration
    commands.  Therefore, one can load a TAQL program (with no declarations),
    print out the inferred data model, and then include the printed
    information in the TAQL program as a declaration for when it is loaded
    in the future.  The advantage of explicitly declaring the information
    is that TAQL is able to do more extensive error checks when declarations
    are given.

    A. (print-data-model DATA-MODEL-SPEC-ARGS)

           DATA-MODEL-SPEC-ARGS ::= MODEL-ITEMS-SPEC*
                                [:exclude MODEL-ITEMS-SPEC*]
                                    [:dont-exclude-taql]
                                    [:dont-exclude-soar]
                    [:infer-globals {yes | no}]

           MODEL-ITEMS-SPEC ::= ( [ MODEL-TYPE [ ITEM-PART [ SEGMENT-PART ]]] )

           MODEL-TYPE ::= inferred | declared | all

       ITEM-PART ::= :all | TYPE-NAME | ( TYPE-NAME+ )

           SEGMENT-PART ::= SEGMENT-SPEC | ( SEGMENT-SPEC+ )

       For a definition of SEGMENT-SPEC, see Section 1 of these release notes.

       Print-data-model prints the type information specified by its arguments.
       Each MODEL-ITEMS-SPEC in the argument list indicates a set of
       data models, as follows:

         A MODEL-ITEMS-SPEC consists of a list of three items.  The
     first item specifies the type of model: inferred, declared,
     or all (both inferred and declared).  It defaults to 'all'.
     The second item specifies what data types to include
     information for.  It defaults to :all.  The third argument
     constrains the set of segments from which model information
     is gathered.  It defaults to :all.  Data type declarations
     reside in the segment they appeared in, and the inferred data
     model for each segment contains information inferred from TCs
     in that segment.

       Print-data-model prints all of the type information specified by
       MODEL-ITEM-SPECs that appear before the first :exclude keyword,
       excluding any information that is also specified by MODEL-ITEM-SPECs
       that appear after the first :exclude keyword.  Information is excluded
       on a per-attribute basis.  For example, suppose that the inferred
       model for type T1 in segment S1 has attributes A and B, and that
       the local declaration for T1 in segment S1 has attributes A and
       C.  Then (print-data-model (inferred T1 S1) :exclude (declared T1 S1)
       would print only attribute B for T1 in S1.

       When type information is excluded, it is usually excluded only
       from items for the same type in the same segment.  However, if
       the excluded information is a global declaration, it will be
       excluded from any items for the same type, regardless of
       segment.  Because of this feature, doing

          (print-data-model (inferred) :exclude (declared))

       will print only the inferred items that are not covered by either
       a local or global declaration.  This is a very convenient way to
       find out what new declarations you have to add as you extend your
       system.

       If both inferred and declared models are specified for the same type,
       the two models are merged and printed as a single model.

       When the :infer-global argument is 'yes' (the default),
       print-data-model tries to guess at what inferred information
       represents global types, and prints this information as global
       declarations rather than as local to the segment it was
       inferred from.  Print-data-model assumes that inferred
       information about type T1 in segment S1 is global when T1 is
       not declared locally in S1 and one of the following conditions holds:

         1. T1 has a global definition OR
     2. There is inferred information about T1 in more than one segment
        that it is not declared locally in.

       When :infer-globals is 'no', print-data-model prints all inferred
       information as being local to the segment it was inferred from.

       If no MODEL-ITEMS-SPECs are given before an :exclude keyword,
       it defaults to (all :all :all).  Information on types from the
       TAQL and Soar segments is always excluded unless the
       :dont-exclude-taql or :dont-exclude-soar keywords are given,
       respectively.  This keeps it from printing internal information
       that most users probably aren't interested in.

       More precisely, if :dont-exclude-taql is absent, then

         :exclude (declared :all taql)
              (all (aug* control-stuff* link-index* taql-stuff*) :all)

       are implicit arguments.  This keeps it from printing information
       on objects that TAQL uses internally.  This is the default, since
       normally the user will want to see only the objects that appear
       explicitly in their TCs.

       And, if :dont-exclude-soar is absent, then

         :exclude (declared :all soar)

       is an implicit argument.  This keeps it from printing information
       on standard Soar objects that the user probably doesn't want to
       see (predefined goal and impasse augmentations, for example).

       The symbol *unknown* will sometimes appear as a class, attribute or
       value type name when printing a data model that includes inferred
       information.  When *unknown* appears as a class or attribute name,
       it means that TAQL could not uniquely determine the name of that
       class or attribute (for example, because it was a variable rather
       than a constant).  When *unknown* appears as the value type for
       an attribute, it means that TAQL was able to infer that the value
       of that attribute was a structured object (not a primitive value),
       but was unable to infer the class of that object.  TAQL will not
       accept declarations that contain *unknown*, so when TAQL prints
       such information it is for informational purposes only.

       In print-data-model, a TYPE-NAME can be a symbol of the form
       'operator*<op-name>'.  This specifies the data model information
       specific to the operator named <op-name>.  For example,
         (print-data-model (all operator*move-tile))
       will print out defoperator information for the move-tile operator,
       accumulated from both inferred and declared information in all
       segments.

       Print-data-model prints information for global goals, problem-spaces,
       states and operators using defobject-merge.  For all other object
       types it uses defobject.  A user may wish to change these when
       inserting the printed information as a declaration.

       USAGE HINTS:

         - (print-data-model (inferred) :exclude (declared))
       is a useful way to find out what new declarations you have to
       add when you extend a system that already has some type
       declarations in it.  It is also a useful when you want to check
       your inferred data models for things like misspelled attribute
       names, because for a system that already has declarations it
       will print out much less information than
       (print-data-model (inferred)) would.

         - TAQL includes a global declaration for some state attributes
           that it uses.  So in the absence of local state declarations,
           print-data-model will assume that all inferred state
           information is global.  You can get it to treat the inferred
           information as local in two ways:

             1. Use (print-data-model (all state) :infer-globals no) OR
         2. Insert empty local state declarations
            (i.e. (defobject state :all-info-declared no))
            in each segment where the inferred state information
            should be local.  This is enough to keep print-data-model from
            assuming that the inferred state information in those segments
            is global.  Later, replace these empty declarations with the
            information printed by (print-data-model).

IV. EXCISING DATA MODELS

   The excise-data-model command removes type information from selected
   data models.  It will not remove type information for declarations
   that appeared between start-default and stop-default.  Such
   declarations can only be removed by restarting TAQL.

   Its syntax is

      (excise-data-model MODEL-ITEMS-SPEC*)

   where MODEL-ITEMS-SPEC is as defined for print-data-model (see
   Section III).  The defaults are the same as for print-data-model,
   except that :exclude is not meaningful here.

V. SUMMARY OF DATA MODEL INTERFACE COMMANDS

   Section I

     defobject
     defobject-merge
     defoperator
     defoperator-merge
     defenumeration
     defprimitive
     use-type-defs

   Section II

     all-types-declared
     all-operators-declared
     all-info-declared

   Section III

     print-data-model

   Section IV

     excise-data-model
")
;;;
;;; SETFS:
;;;

(setf (documentation 'propose-task-operator 'function)
    *propose-task-operator-doc-string*)
(setf (documentation 'propose-task-state 'function)
    *propose-task-state-doc-string*)
(setf (documentation 'propose-space 'function)
    *Problem-doc-string*)
(setf (documentation 'propose-initial-state 'function)
    *propose-initial-state-doc-string*)
(setf (documentation 'propose-operator 'function)
    *propose-operator-doc-string*)
(setf (documentation 'compare 'function)
    *compare-doc-string*)
(setf (documentation 'prefer 'function)
    *prefer-doc-string*)
(setf (documentation 'goal-test-group 'function)
    *Goal-doc-string*)
(setf (documentation 'apply-operator 'function)
    *apply-operator-doc-string*)
(setf (documentation 'augment 'function)
    *Entailment-doc-string*)
(setf (documentation 'result-superstate 'function)
    *result-superstate-doc-string*)
(setf (documentation 'evaluation-properties 'function)
    *evaluation-properties-doc-string*)
(setf (documentation 'evaluate-object 'function)
    *evaluate-object-doc-string*)
(setf (documentation 'propose-superobjects 'function)
    *propose-superobjects-doc-string*)
(setf (documentation 'operator-control 'function)
    *operator-control-doc-string*)

) ;;; Close eval-when (compile load eval), opened near beginning of file.

(eval-when (compile load eval)
  (soarsyntax))
