
/****************************************************************************
 *
 * MODULE:  utility.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Routines of general use.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed lock declarations and references in 
 *                    conditional compilation block. Also enclosed defn.
 *                    for ops_init_profile() and ops_badlock()
 *                    (#ifndef UNIPROC_VERSION)
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Use new lock macros from psm_locks.h.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    23 Oct 88 V3.1  Dirk Kalp
 *                    Added calls to "ops_run_stats" and "ops_memory_stats" routines.
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf.
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    28 Aug 86       Dirk Kalp
 *    22 Aug 86       Dirk Kalp
 *    31 Jul 86 V1.0  Dirk Kalp
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include "global.h"


/* Exported routines:
 *    char    *ops_malloc(bytesize)
 *    char    *ops_realloc(ptr, bytesize)
 *    boolean  ops_equiv_strings(str1, str2)
 *    int      ops_length(str)
 *    void     ops_fatal(str)
 *    void     ops_warn(str)
 *    void     ops_init_time()
 *    double   ops_compute_utime();
 *    double   ops_compute_stime();
 *    void     ops_print_timing_info()
 *    void     ops_init_profile()
 *    void     ops_get_profile()
 *    void     ops_badlock()
 *
 */



/* Imported Routines:
 *    From system:
 *       malloc
 *       realloc
 *       exit
 *    From gensymbol.c:
 *       ops_symid_lookup
 *    From shmem.c:
 *       ops_ShMalloc
 *       ops_memory_stats
 *    From psm_locks.c
 *       ops_alloc_and_init_lock()
 *
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char   *ops_ShMalloc();       /* Imported from shmem.c. */
extern void    ops_memory_stats();   /* Imported from shmem.c. */
extern void    ops_run_stats();      /* Imported from rhsrtn.c. */
extern char   *realloc();            /* Imported from system. */
extern symptr  ops_symid_lookup();   /* Imported from gensymbol.c. */

#ifndef UNIPROC_VERSION
extern LockPtr ops_alloc_and_init_lock(); /* Imported from psm_locks.c. */
#endif


/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
double ops_compute_utime();
double ops_compute_stime();
void   ops_fatal();


#ifndef UNIPROC_VERSION
static LockPtr    MonitorLock;        /* Lock for writing out process profiles. */
#endif

   

char
*ops_malloc(bytesize)
   int bytesize;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate memory in shared address space.
 *
 * Parameters:
 *    bytesize - size in bytes to allocate.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the memory.
 *
 * Calls:
 *    "ops_ShMalloc" in shmem.c.
 *    "ops_fatal" in this module.
 *
 * Called by:
 *    many callers.
 *
 *-------------------------------------------------------------------------*/
{
   char *ptr;
   
   if ((ptr = ops_ShMalloc(bytesize)) == NULL)
      ops_fatal("ops_malloc: out of memory.");
   else
      return(ptr);
}



char
*ops_realloc(ptr, bytesize)
   char *ptr;
   int bytesize;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate memory.
 *
 * Parameters:
 *    ptr      - ptr to the memory.
 *    bytesize - size in bytes to allocate.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the memory.
 *
 * Calls:
 *    "realloc" in system.
 *    "ops_fatal" in this module.
 *
 * Called by:
 *    many callers.
 *
 *-------------------------------------------------------------------------*/
{
#ifdef UNIPROC_VERSION
   if ((ptr = realloc(ptr, bytesize)) == NULL)
      ops_fatal("ops_realloc: out of memory.");
   else
      return(ptr);
#else
   ops_fatal("ops_realloc does not work for the shared-memory version\n");
#endif

}



boolean
ops_equiv_strings(str1, str2)
   string str1, str2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Determine if 2 string are equivalent.
 *
 * Parameters:
 *    str1, str2 - the 2 strings string.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    TRUE if the same and FALSE otherwise.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "symname_lookup" in "gensymbol.c".
 *
 *-------------------------------------------------------------------------*/
{
   while (*str1)
      if (*str1 != *str2)
        return(FALSE);
      else { str1++; str2++; }
   
   if (*str2)
      return(FALSE);
   else
      return(TRUE);
}



int
ops_length(str)
   string str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Find the length of a string.
 *
 * Parameters:
 *    str - a string.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The length of the string.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "make_string" in "gensymbol.c".
 *
 *-------------------------------------------------------------------------*/
{
   int  n;
   
   n = 0;                    /* Does NOT count the NULL byte at the end. */
   while (*str++) n++;
   return(n);
}



void
ops_fatal(str)
   string str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Print an error message and exit the program.
 *
 * Parameters:
 *    str - the error message.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "exit" in system and "ops_symid_lookup" in "gensymbol.c".
 *
 * Called by:
 *    many callers.
 *
 *-------------------------------------------------------------------------*/
{
symptr psym;

   /* flg_ruleid will not hold a valid OpsVal until the initial
    * production is fired.
    */
   if (symbolp(flg_ruleid))
     {
      psym = ops_symid_lookup(val2sym(flg_ruleid));
      fprintf(fp_err, "  ** FATAL ERROR in Production %s\n ", psym->SymName);
     }
   else
      fprintf(fp_err , " *** FATAL ERROR in initialization \n");
      
   fprintf(fp_err, "   %s\n", str);
   
   exit(1);
}



void
ops_warn(str)
   string str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Print an warning message but don't exit the program.
 *
 * Parameters:
 *    str - the error message.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "ops_symid_lookup" in "gensymbol.c".
 *
 * Called by:
 *    many callers.
 *
 *-------------------------------------------------------------------------*/
{
symptr psym;

   /* flg_ruleid will not hold a valid OpsVal until the initial
    * production is fired.
    */
   if (symbolp(flg_ruleid))
     {
      psym = ops_symid_lookup(val2sym(flg_ruleid));
      fprintf(fp_err, "* WARNING in Production %s\n", psym->SymName);
     }
   else
      fprintf(fp_err, "* WARNING in initialization.\n");
      
   fprintf(fp_err, "   %s\n", str);
}


void
ops_init_time()
{
   g_match_started = FALSE;
   g_rhs_utime   = g_rhs_stime   = 0.0;
   g_match_utime = g_match_stime = 0.0;
   g_load_utime  = g_load_stime  = 0.0;

   getrusage(RUSAGE_SELF, g_time_init);
   g_init_utime = ops_compute_utime(NULL, g_time_init);
   g_init_stime = ops_compute_stime(NULL, g_time_init);

}


double
ops_compute_utime(beg, end)
    struct rusage *beg, *end;
{
    struct timeval butime, eutime;
    double         bt, et;

    eutime = end->ru_utime;
    et = eutime.tv_sec + (double) eutime.tv_usec / (double) 1000000;

    if (beg == NULL)  return(et);

    butime = beg->ru_utime;
    bt = butime.tv_sec + (double) butime.tv_usec / (double) 1000000;
    return(et - bt);
}

double
ops_compute_stime(beg, end)
    struct rusage *beg, *end;
{
    struct timeval bstime, estime;
    double         bt, et;

    estime = end->ru_stime;
    et = estime.tv_sec + (double) estime.tv_usec / (double) 1000000;

    if (beg == NULL)  return(et);

    bstime = beg->ru_stime;
    bt = bstime.tv_sec + (double) bstime.tv_usec / (double) 1000000;
    return(et - bt);
}


void
ops_print_timing_info()
{
   fprintf(fp_out, "\n");
   fprintf(fp_out, "Init:     User_Time: %9.4f sec, Sys_Time: %9.4f sec, Tot_Time: %9.4f sec\n",
   		g_init_utime, g_init_stime, (g_init_utime+g_init_stime));
   fprintf(fp_out, "Match:    User_Time: %9.4f sec, Sys_Time: %9.4f sec, Tot_Time: %9.4f sec\n",
   		g_match_utime, g_match_stime, (g_match_utime+g_match_stime));
   fprintf(fp_out, "RHS:      User_Time: %9.4f sec, Sys_Time: %9.4f sec, Tot_Time: %9.4f sec\n",
   		g_rhs_utime, g_rhs_stime, (g_rhs_utime+g_rhs_stime));

   fprintf(fp_out, "\n");
   fprintf(fp_out, "Load:     User_Time: %9.4f sec, Sys_Time: %9.4f sec, Tot_Time: %9.4f sec\n",
   		g_load_utime, g_load_stime, (g_load_utime+g_load_stime));

   fprintf(fp_out, "\n");
   ops_run_stats(fp_out);
   ops_memory_stats(fp_out);

   fprintf(fp_out, "\n\n");
   fflush(fp_out);
}

#ifndef UNIPROC_VERSION
void
ops_init_profile()
{
   MonitorLock = ops_alloc_and_init_lock();
}
#endif

void
ops_get_profile()
{
   char num[8];
   char *ptr;

   Test_Then_Lock(MonitorLock);

   ptr = (char *) malloc(128);
   strcpy(ptr, "mon.out");
   sprintf(num, "%d", ProcessID);
   strcat(ptr, num);

/*   monitor(0); */
   if (NumProc > 1)  rename("mon.out", ptr);

   Release(MonitorLock);
}

#ifndef UNIPROC_VERSION
void
ops_badlock()
{
int *z;
  fprintf(fp_err, " ************   Bad Lock Encountered  *************** \n");
  z = 0;
 *z = 1;   /* Nil pointer to force a segment fault. */
}
#endif
