
/****************************************************************************
 *
 * MODULE:  shmem.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    This file contains the allocator for the shared memory, and various
 *    assoiated initialization routines.
 *
 * Exports:
 *    ops_InitShMem();
 *    unsigned *ops_ShMalloc(int nbytes);
 *
 * Notes:
 *    When using multiple processes special care should be taken so
 *    that the initialization of shared memory for first process is finished
 *    before the initialization for the next process is begun.  This is
 *    because the processes share the value *ShMemFillPtr, and incorrect
 *    allocation will take place if it is being incremented by multiple 
 *    processes.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed all lock initializations and references in
 *                     conditional compilation blocks (#ifndef UNIPROC_VERSION)
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Use new lock macros from psm_locks.h.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    23 Oct 88 V3.2  Dirk Kalp
 *                    Fixed up error msgs. Added "ops_memory_stats" routine.
 *    21 Sep 88 V3.1  Dirk Kalp
 *                    Allow size in Kbytes of shared memory to be specified
 *                    at run time.
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf.
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *                    Added this module header.     
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 *
 ****************************************************************************/


#include "global.h"

#ifdef MACH_SHARED_MEMORY_VERSION
#include <mach.h>
#endif

#ifdef UMAX_SHARED_MEMORY_VERSION
#include <parallel.h>
#endif



/* Exported routines:
 *    void      ops_InitShMem(mem_size)
 *    unsigned *ops_ShMalloc(nbytes)
 *    void      ops_touch_pages()
 *    void      ops_memory_stats(fp_x)
 *
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_fatal
 *
 *    From system:
 *       vm_allocate
 *       vm_inherit
 *
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern void     ops_fatal();           /* Imported from utility.c. */


/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
/* There are none! */


   
static int ShMemSize = 1023*2048;  /* Size in words of initial allocation. */

static unsigned  *ShMemEndPtr;     /* Pointer to end of shared mem block. */
static unsigned  *ShMemFillPtr;    /* Pointer to end of already used shared mem */

#ifndef UNIPROC_VERSION
static LockPtr   ShMallocLockPtr;  /* Ptr to lock for allocating shared mem   */
#endif

static int *Start_Addr;   /* For use by ops_touch_pages(). */
extern int *TouchCount;




void
ops_InitShMem(mem_size)
   int mem_size;   /* Number of K bytes of shared memory to allocate. */
{
   int           *start_addr;      /* Holds addr of first chunk of shared mem. */

#ifdef MACH_SHARED_MEMORY_VERSION
   kern_return_t  ret;

   ShMemSize = mem_size * (1024 / sizeof(int));   /* Keep allocation in terms of words. */

   ret = vm_allocate(task_self(), &start_addr, (ShMemSize)*sizeof(int), TRUE);
   if (show_debug)  fprintf(fp_dbug, "InitShMem Allocate: ret = %d, start_addr = 0x%x.\n", ret, (int) start_addr);
   if (ret != KERN_SUCCESS)
     {
      fprintf(fp_err, "*** Shared memory Allocate failed in InitShMem for %d bytes.\n", ShMemSize*sizeof(int));
      fprintf(fp_err, "*** Mach kernel error code returned is %d.\n", ret);
      ops_fatal("Probable cause is not enough physical memory on your system.\n");
     }
	
   ret = vm_inherit(task_self(), start_addr, (ShMemSize)*sizeof(int), VM_INHERIT_SHARE);
   if (show_debug)  fprintf(fp_dbug, "InitShMem Inherit: ret = %d.\n", ret);
   if (ret != KERN_SUCCESS)  ops_fatal("Shared memory Inherit failed in InitShMem.\n");
#endif


#ifdef UMAX_SHARED_MEMORY_VERSION
   int  size;

   ShMemSize = mem_size * (1024 / sizeof(int));   /* Keep allocation in terms of words. */

   size = ShMemSize * sizeof(int);

   start_addr = (int *) share(0,size);
   if (start_addr == NULL)  ops_fatal("InitShMem: share failed. \n");
   if (show_debug)  fprintf(fp_dbug, "InitShMem Allocate: start_addr = 0x%x.\n", (int) start_addr);
#endif



#ifdef  UNIPROC_VERSION
   ShMemSize = mem_size * (1024 / sizeof(int));   /* Keep allocation in terms of words. */

   start_addr = (int *) malloc(LongwordSize * (ShMemSize));
   if (start_addr == NULL)  ops_fatal("InitShMem: Requested amount of memory is not available.\n");
   if (show_debug)   fprintf(fp_dbug, "InitShMem malloc for single process private memory completed: start_addr = 0x%x.\n", (int) start_addr);
#endif

   if (show_debug)  fflush(fp_dbug);

   Start_Addr = start_addr;  /* For use by ops_touch_pages(). */


   /* Fix forever the shared mem locations that contain global pointers needed
    * across all processes.
    */
   ShMemEndPtr      = (unsigned *) start_addr;
   ShMemFillPtr     = (unsigned *) (start_addr + 1);  /* in actual, adds 4 */

#ifndef UNIPROC_VERSION
   ShMallocLockPtr  = (LockPtr)    (start_addr + 2);  /* in actual, adds 8 */
   *ShMallocLockPtr = 0;		/* The lock is initialized to 0, i.e., open */
#endif

   /* Set initial values for the global shared locations. All contain
    * pointers except the lock.
    */
   *ShMemEndPtr     = (unsigned) (start_addr + ShMemSize);  /* in actual, adds 4*ShMemSize */
   *ShMemFillPtr    = (unsigned) (start_addr + 3);          /* in actual, adds 12 */


  
}



unsigned 
*ops_ShMalloc(nbytes)
int nbytes;
{
/* This procedure allocates nbytes of shared memory that is alligned to a
 * longword boundary.  If the shared memory is exhausted, the program
 * terminates.  This allocator is designed with idea that the storage once
 * allocated will not be freed.  (Of course, the program may internally
 * maintain free lists of data objects.
 */
unsigned *adr;
 
  /* First obtain lock so that nobody else is touching shared memory */
  Test_Then_Lock(ShMallocLockPtr);
  
  adr = (unsigned *) *ShMemFillPtr;
  *ShMemFillPtr = (unsigned) (adr + (nbytes + LongwordSize - 1)/LongwordSize); 
  /* assume that *ShMemFillPtr points to longword boundary, and make sure that
   * it points to longword boundary after the allocation. */
   
  if (*ShMemFillPtr >= *ShMemEndPtr)
    ops_fatal("Shared mem exhausted. Try to allocate a bigger block at startup with -m switch. \n");

  Release(ShMallocLockPtr);
  
  return(adr);
  
}


void
ops_touch_pages()
{
   int *end_addr, *dummy;

   end_addr = Start_Addr + ShMemSize - 1;

   for (dummy = Start_Addr; dummy <= end_addr; dummy += 2048)  *dummy = *dummy;

  /* Borrow existing lock to protect TouchCount access. */
  Test_Then_Lock(ShMallocLockPtr);
  
   (*TouchCount)--;

   Release(ShMallocLockPtr);
}


void
ops_memory_stats(fp_x)
   FILE *fp_x;
{
   int bytes_allocated, bytes_used;

   bytes_allocated = ShMemSize*sizeof(int);
   bytes_used      = (int) *ShMemFillPtr - (int) Start_Addr;

   fprintf(fp_x, "Shared Memory Usage:   %d bytes allocated, %d bytes used, %5.2f%% usage.\n", 
                   bytes_allocated, bytes_used, ((bytes_used*100.0)/bytes_allocated));
}
