
/****************************************************************************
 *
 * MODULE:  psm_locks.h
 *
 ****************************************************************************
 *
 * Abstract:
 *    Declarations for locks and lock operations:
 *       The basic mechanism used throughout the system to provide
 *       synchonization and access to shared data is simple spin-locks.
 *       These simple locks are provided by the macros "Test_Then_Lock(lock)"
 *       and "Release(lock)" defined in this module. The "lock" parameter to
 *       these is a variable of type "LockPtr" (defined here also). A "lock"
 *       of this type is allocated and initialized by a call to the routine
 *       "ops_alloc_and_init_lock" in module "psm_locks.c". 
 *    
 *       For locks used to protect access to the hash table buckets for
 *       tokens in the Rete network, we provide a more complex locking
 *       scheme for providing access to a hash table "line". A "line" is a
 *       pair of corresponding buckets (buckets with the same hash index)
 *       from the left and right hash tables. Instead of associating a simple
 *       lock with each line in the hash table to give exclusive access to the
 *       line, we use a complex lock with each line that implements a
 *       "multiple-reader-single-writer" protocol for match processes that
 *       want to access that line. This permits several tokens to be
 *       processed in the same line at the same time. Special macros are
 *       defined in this module that provide the operations necessary to
 *       implement these complex hash table locks. These operations are
 *       invoked only in the module "matchasm.c". The array of locks (1 lock
 *       required per bucket) and a corresponding array of counters needed
 *       to implement the protocol are allocated and initialized by a call
 *       to the routine "ops_alloc_and_init_hashtable_locks" in module
 *       "psm_locks.c".
 *       
 *       The locking macros define the lock operations as machine dependent
 *       function calls. These functions are implemented as assembly coded
 *       routines which are built using the basic interlocked test-and-set
 *       instruction available on the machine. These routines are found in
 *       the module "<machine-type>_locks.s" where, currently "machine-type"
 *       can be either "VAX" or "ENCORE".
 *
 *       In order to port the system to a different shared memory multi-
 *       processor, the assembly coded routines would have to be provided.
 *       This is not too difficult for the simple locks since they require
 *       only 2 routines of 4-5 instructions. The complex hash table locks
 *       require, however, correspondingly complex assembly routines. But it
 *       is possible to just use the same simple locks for the hash tables
 *       instead. In order to make this easy, we have provided an alternative
 *       set of definitions for the complex lock macros that translate into
 *       the simple style locking operations. These alternative macros can
 *       be obtained by first commenting out the define in this module for
 *       "COMPLEX_HASH_TABLE_LOCKS" before compiling the system.
 *
 *    We have also provide a set of definitions for the locking macros that
 *    result in no source code, i.e., null macros. These get incorporated
 *    when the conditional compilation variable UNIPROC_VERSION is defined.
 *    With this we get a uniprocessor version of the system that is fully
 *    C coded and thus machine independent.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.3  Dirk Kalp
 *                    Provided conditional compilation variable to allow
 *                    selection of simple or complex stlye hash table locks.
 *                    Complex locks are the default.
 *                    Also added a lot of documentation.
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed the declaration of Lockptr in a conditional
 *                    compilation block (#ifndef UNIPROC_VERSION)
 *                    defined a new conditional compilation block to set
 *                    all locking macros to null for the uniproc version
 *                    (#ifdef UNIPROC_VERSION)
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Update for CParaOPS5 Version 4.0 P4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Create for CParaOPS5 Version 3.0 P4.2.
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include "version.h"



#ifndef UNIPROC_VERSION

typedef char	   *LockPtr;	/* We will store only 0 or 1 for lock value. */
				/* 0 => Lock Open,  1 => Lock Closed         */

#define LOCKSIZE   sizeof(char)


#define COMPLEX_HASH_TABLE_LOCKS  1    /* Comment out this for simple hash table locks. */

#endif


/* #define SYS_DEBUG  1     Define here or in "version.h" for system debugging use only. */



/*----------------------------------------------------------------------------
 *
 *  UNIPROC_VERSION:  Macros to define lock operations are all defined as
 *                    null for the uniprocessor version. The uniprocessor
 *                    version is entirely C-based and without machine
 *                    dependencies.
 *
 *--------------------------------------------------------------------------*/
#ifdef UNIPROC_VERSION

#define Test_Then_Lock(lock)
#define Release(lock)
#define Lock(lock)

#define LockAccess_LeftRead_RightWrite(bucket)
#define Release_LeftRead(bucket)
#define Release_RightWrite(bucket)

#define LockAccess_LeftWrite_RightRead(bucket)
#define Release_LeftWrite(bucket)
#define Release_RightRead(bucket)

#endif   /* UNIPROC_VERSION */





/*----------------------------------------------------------------------------
 *
 *  VAX_VERSION:  Macros to define lock operations. The parameter "lock"
 *                is of type "LockPtr". The parameter "bucket" is an "int".
 *                See module "VAX_locks.s" for assembly routines.
 *
 *--------------------------------------------------------------------------*/
#ifdef VAX_VERSION

/* Operations for simple locks.
 */
#define Test_Then_Lock(lock)    VAX_Test_Then_Lock(lock)
#define Release(lock)           VAX_Release(lock)
#define Lock(lock)              VAX_Lock(lock)



/* Operations for locks on the hash table based token memories.
 */
#ifdef COMPLEX_HASH_TABLE_LOCKS

/* These implement complex multiple-reader-single-writer
 * locking protocol.
 */
#define LockAccess_LeftRead_RightWrite(bucket)    VAX_LA_LR_RW_HT(bucket)
#define Release_LeftRead(bucket)                  VAX_Rel_LR_HT(bucket)
#define Release_RightWrite(bucket)                VAX_Rel_RW_HT(bucket)

#define LockAccess_LeftWrite_RightRead(bucket)    VAX_LA_LW_RR_HT(bucket)
#define Release_LeftWrite(bucket)                 VAX_Rel_LW_HT(bucket)
#define Release_RightRead(bucket)                 VAX_Rel_RR_HT(bucket)

#else

/* These implement simple locking protocol. (The macro definitions here are
 * a bit of a hack: we should clean things up by hiding "tokHTlock" from
 * "matchasm.c". But we'll do that another day.)
 */
#define LockAccess_LeftRead_RightWrite(bucket)    VAX_Test_Then_Lock(&tokHTlock[bucket])
#define Release_LeftRead(bucket)                  VAX_Release(&tokHTlock[bucket])
#define Release_RightWrite(bucket)                /* null */

#define LockAccess_LeftWrite_RightRead(bucket)    VAX_Test_Then_Lock(&tokHTlock[bucket])
#define Release_LeftWrite(bucket)                 /* null */
#define Release_RightRead(bucket)                 VAX_Release(&tokHTlock[bucket])

#endif   /* COMPLEX_HASH_TABLE_LOCKS */

#endif   /* VAX_VERSION */




/*----------------------------------------------------------------------------
 *
 *  ENCORE_VERSION:  Macros to define lock operations. The parameter "lock"
 *                   is of type "LockPtr". The parameter "bucket" is an "int".
 *                   See module "ENCORE_locks.s" for assembly routines.
 *
 *--------------------------------------------------------------------------*/
#ifdef ENCORE_VERSION

/* Operations for simple locks.
 */
#define Test_Then_Lock(lock)    ENCORE_Test_Then_Lock(lock)
#define Release(lock)           ENCORE_Release(lock)
#define Lock(lock)              ENCORE_Lock(lock)



/* Operations for locks on the hash table based token memories.
 */
#ifdef COMPLEX_HASH_TABLE_LOCKS

/* These implement complex multiple-reader-single-writer
 * locking protocol.
 */
#define LockAccess_LeftRead_RightWrite(bucket)    ENCORE_LA_LR_RW_HT(bucket)
#define Release_LeftRead(bucket)                  ENCORE_Rel_LR_HT(bucket)
#define Release_RightWrite(bucket)                ENCORE_Rel_RW_HT(bucket)

#define LockAccess_LeftWrite_RightRead(bucket)    ENCORE_LA_LW_RR_HT(bucket)
#define Release_LeftWrite(bucket)                 ENCORE_Rel_LW_HT(bucket)
#define Release_RightRead(bucket)                 ENCORE_Rel_RR_HT(bucket)

#else

/* These implement simple locking protocol. (The macro definitions here are
 * a bit of a hack: we should clean things up by hiding "tokHTlock" from
 * "matchasm.c". But we'll do that another day.)
 */
#define LockAccess_LeftRead_RightWrite(bucket)    ENCORE_Test_Then_Lock(&tokHTlock[bucket])
#define Release_LeftRead(bucket)                  ENCORE_Release(&tokHTlock[bucket])
#define Release_RightWrite(bucket)                /* null */

#define LockAccess_LeftWrite_RightRead(bucket)    ENCORE_Test_Then_Lock(&tokHTlock[bucket])
#define Release_LeftWrite(bucket)                 /* null */
#define Release_RightRead(bucket)                 ENCORE_Release(&tokHTlock[bucket])

#endif   /* COMPLEX_HASH_TABLE_LOCKS */

#endif   /* ENCORE_VERSION */





/*----------------------------------------------------------------------------
 *
 *  LOCKING ROUTINES:
 *    Below are pseudo-coded routines that define the routines implemented in
 *    <machine-type>_locks.s:
 *
 *----------------------------------------------------------------------------
 *
 *
 *    /# Simple spin locks:
 *     #    These next routines give pseudo-code to show how spin locks
 *     #    are implemented. Test_Then_Lock implements a 2 stage "test" then
 *     #    "test-and-set". This is done so that we spin waiting in the
 *     #    cache to "test" for the lock free and only attempt to perform
 *     #    the interlocked "test-and_set" over the memeory bus when we
 *     #    think it has a chance of succeeding. This helps to reduce bus
 *     #    contention.
 *     #/
 *
 *    Test_Then_Lock(lockptr)
 *       LockPtr lockptr;
 *    {
 *       for (;;)
 *          if (*lockptr == 0)
 *             if (INTERLOCKED_TESTandSET(lockptr) == 0)  return;
 *    }
 *    
 *    
 *    Lock(lockptr)    /# Previously used only to measure effect of bus contention. #/
 *       LockPtr lockptr;
 *    {
 *       for (;;)
 *          if (INTERLOCKED_TESTandSET(lockptr) == 0)  return;
 *    }
 *    
 *    
 *    Release(lockptr)
 *       LockPtr lockptr;
 *    {
 *       INTERLOCKED_CLEAR(lockptr);
 *    }
 *    
 *    
 *    char
 *    INTERLOCKED_TESTandSET(lockptr)
 *       LockPtr lockptr;
 *    {
 *       char test_value;
 *    
 *       HARDWARE INTERLOCK on Memory[lockptr]
 *          test_value = *lockptr;
 *          *lockptr = 1;
 *       RELEASE HARDWARE INTERLOCK
 *       return(test_value);
 *    }
 *    
 *    
 *    INTERLOCKED_CLEAR(lockptr)
 *       LockPtr lockptr;
 *    {
 *       HARDWARE INTERLOCK on Memory[lockptr]
 *          *lockptr = 0;
 *       RELEASE HARDWARE INTERLOCK
 *    }
 *    
 *    
 *    
 *    
 *----------------------------------------------------------------------------
 *
 *    
 *    /# Complex hash table locks:
 *     #    These next 6 routines implement the multiple-reader-
 *     #    single-writer protocol for a given pair of left and
 *     #    right hash table buckets. Each left and right pair (i.e.,
 *     #    line in hash table as discussed above) has an associated 
 *     #    32-bit counter LeftRightCounter defined in "psm_locks.c". 
 *     #    The leftmost 16 bits form a counter to control access to
 *     #    the left bucket and right 16 bits are a counter to control
 *     #    access to the right bucket. A 16-bit counter can have the
 *     #    following states:
 *     #        0 => No matcher process is either reading or writing
 *     #             the associated bucket.
 *     #       -1 => The associated bucket is locked for write access
 *     #             by a single matcher process; no other writers or
 *     #             readers are permitted access.
 *     #       >0 => The associated bucket is locked for read only
 *     #             access by any number of matcher processes; writers
 *     #             are not permitted access; the value of the counter
 *     #             indicates the number of readers present.
 *     #    To make the pseudo code below easier to read, we will use
 *     #    LeftCounter[bucket] and RightCounter[bucket] to designate
 *     #    the LeftRightCounter[bucket]. (Note that we use a single
 *     #    32-bit data object instead of 2 separate 16-bit counters
 *     #    because the assembly code is more efficient using this
 *     #    arrangement.)
 *     #
 *     #    A given matcher process will always require write access
 *     #    to one bucket in the pair and read access to the opposite
 *     #    bucket when processing a task. The write access is needed
 *     #    for only a short time to add/delete the token to/from the
 *     #    bucket. The read access in the opposite bucket is more
 *     #    lengthy since the matcher process must look at each token
 *     #    stored at the bucket and see if it matches the token just
 *     #    added/deleted in the other bucket.
 *     #
 *     #    The hash table lock associated with a bucket pair is a
 *     #    simple style lock used to control access to the pair of
 *     #    16-bit counters. Once a matcher process acquires the lock
 *     #    to a counter pair, it can then examine the counters to see
 *     #    if they are in the right state to allow the matcher to use
 *     #    the bucket pair. If so, the matcher modifies the counters
 *     #    appropriately to indicate his use of the bucket pair and
 *     #    releases the lock.
 *     #
 *     #    Thus, for example, to add a token to a right bucket and
 *     #    search its corresponding left bucket, the matcher first
 *     #    locks the bucket counters. If LeftCounter[bucket] >= 0
 *     #    (i.e., unused or other readers in left bucket) and if
 *     #    RightCounter[bucket] == 0 (i.e., no one in right bucket),
 *     #    then increment the LeftCounter to indicate another reader
 *     #    and set the RightCounter to -1 to indicate exclusive write
 *     #    access in the right bucket. Then the matcher releases the
 *     #    lock on the bucket counters and proceeds with the task.
 *     #    As soon as he finishes the add/delete in the right bucket,
 *     #    he resets the RightCounter to release his exclusive use of 
 *     #    the right bucket. When he completes the task of matching on 
 *     #    the tokens in the left bucket, he gets the counter lock again 
 *     #    and decrements the LeftCounter to release his read use of the
 *     #    left bucket.
 *     #/
 *
 *
 *    LockAccess_LeftRead_RightWrite(bucket)
 *       int bucket;
 *    /######################################################################
 *     #
 *     # Obtain read access in left bucket and write access in right bucket.
 *     #
 *     #####################################################################/
 *    {
 *       for (;;)
 *         {
 *          /# First see if the counters are in an acceptable state before
 *           # we even bother to lock them.
 *           #/
 *          if ((*LeftCounter[bucket] >= 0)  && (*RightCounter[bucket] == 0))
 *            {
 *    	       Test_Then_Lock(tokHTlock[bucket]);
 *             /# OK, now test their state under locked access.
 *              #/
 *    	       if ((*LeftCounter[bucket] >= 0)  && (*RightCounter[bucket] == 0))
 *    	         {
 *                /# Counters in acceptable state so update them to indicate
 *                 # our required access.
 *                 #/
 *    	          *RightCounter[bucket] = -1;    /# Writing in the right bucket. #/
 *    	          *LeftCounter[bucket]++;        /# Reading in the left bucket. #/
 *                Release(tokHTlock[bucket]);
 *    	          return;
 *    	         }
 *             /# Counters not in acceptable state so release lock and try again.
 *              #/
 *    	       Release(tokHTlock[bucket]);
 *            }
 *         } /# endfor #/
 *    }
 *    
 *    
 *    Release_LeftRead(bucket)
 *       int bucket;
 *    /######################################################################
 *     #
 *     # Release read access in the left bucket.
 *     #
 *     #####################################################################/
 *    {
 *       Test_Then_Lock(tokHTlock[bucket]);
 *       *LeftCounter[bucket]--;    /# One less reader in the left bucket. #/
 *       Release(tokHTlock[bucket]);
 *    }
 *    
 *    
 *    Release_RightWrite(bucket)
 *       int bucket;
 *    /######################################################################
 *     #
 *     # Release write access in the right bucket.
 *     #
 *     #####################################################################/
 *    {
 *       *RightCounter[bucket] = 0;   /# No need to lock access since no contention for write owner. #/
 *    }
 *    
 *    
 *       
 *    LockAccess_LeftWrite_RightRead(bucket)
 *       int bucket;
 *    /######################################################################
 *     #
 *     # Obtain read access in left bucket and write access in right bucket.
 *     #
 *     #####################################################################/
 *    {
 *       for (;;)
 *         {
 *          /# First see if the counters are in an acceptable state before
 *           # we even bother to lock them.
 *           #/
 *          if ((*RightCounter[bucket] >= 0)  && (*LeftCounter[bucket] == 0))
 *            {
 *    	       Test_Then_Lock(tokHTlock[bucket]);
 *             /# OK, now test their state under locked access.
 *              #/
 *    	       if ((*RightCounter[bucket] >= 0)  && (*LeftCounter[bucket] == 0))
 *    	         {
 *                /# Counters in acceptable state so update them to indicate
 *                 # our required access.
 *                 #/
 *    	          *LeftCounter[bucket] = -1;    /# Writing in the left bucket. #/
 *    	          *RightCounter[bucket]++;      /# Reading in the right bucket. #/
 *                Release(tokHTlock[bucket]);
 *    	          return;
 *    	         }
 *             /# Counters not in acceptable state so release lock and try again.
 *              #/
 *    	       Release(tokHTlock[bucket]);
 *            }
 *         } /# endfor #/
 *    }
 *    
 *    
 *    Release_LeftWrite(bucket)
 *       int bucket;
 *    /######################################################################
 *     #
 *     # Release write access in the left bucket.
 *     #
 *     #####################################################################/
 *    {
 *       *LeftCounter[bucket] = 0;   /# No need to lock access since no contention for write owner. #/
 *    }
 *    
 *    
 *    Release_RightRead(bucket)
 *       int bucket;
 *    /######################################################################
 *     #
 *     # Release read access in the right bucket.
 *     #
 *     #####################################################################/
 *    {
 *       Test_Then_Lock(tokHTlock[bucket]);
 *       *RightCounter[bucket]--;    /# One less reader in the right bucket. #/
 *       Release(tokHTlock[bucket]);
 *    }
 *    
 *
 *--------------------------------------------------------------------------*/
