/****************************************************************************
 *
 * MODULE:  matchasm.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Routines needed for the Rete match. The bulk of the Rete network code is
 *    in a C module generated by the CParaOPS5 compiler from the productions
 *    of the application program. Program independent routines needed to
 *    perform the match are contained here in "matchasm.c". The main loop
 *    for the forked Rete matcher processes is here (in "ops_MatcherCode").
 *    This main loop searches the task queue(s) for node tasks in the Rete
 *    network code to execute during each OPS5 Match-Resolve-Act (MRA) cycle.
 *    Other routines here manage the queues, add/delete tokens in the hash
 *    table based node memories, manage memory for token data structures,
 *    dispatch the node tasks, and provide generic routines that the beta
 *    node tasks call upon to perform common parts of their tasks.
 *
 *    The routines contained here were, in the previous ParaOPS5 system,
 *    coded either in assembly language of the target machine or required
 *    hand editing of the assembly code generated for them by the C compiler,
 *    hence the original name for this module. There are some additional
 *    routines used for match that are contained in the module "match.c".
 *    Those routines never required any machine level hacking to make them
 *    work for ParaOPS5.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.3  Dirk Kalp
 *                    Added a lot of documentation.
 *                    Fix bug in "ops_PopTaskQueueN": Task was being removed from
 *                    queue before active bit in *BitVector was set. This made
 *                    control process think that match was done.
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed lock declarations and lock references in
 *                    conditional compilation blocks (#ifndef UNIPROC_VERSION)
 *                    Also enclosed the declaration of the routine "show_counts"
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Use new lock macros from psm_locks.h.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf. Use global free lists for
 *                    alpha and beta cells - added new routines "alpha_alloc",
 *                    "beta_4_alloc", "beta_32_alloc", "beta_128_alloc", and
 *                    "beta_256_alloc" to accomodate this. Added additional
 *                    local free lists to accomodate various sized beta cells.
 *                    Redefined alpha and beta cells in global.h. Cleaned up
 *                    a few comments in "rmem" and "lmem".
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    23 Sep 86	V1.0  Created by axg
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include "global.h"



/* Exported routines:
 *    void  ops_rightand_input(nodeid, and_testpass_nodeid)
 *    void  ops_leftand_input(nodeid, betalev, and_testpass_nodeid)
 *    int   ops_rightnot_input(nodeid, not_test_nodeid, not_pass_nodeid)
 *    int   ops_leftnot_input(nodeid, betalev, not_test_nodeid)
 *    void  ops_MatcherCode()
 *    void  ops_PushTaskQueue()
 *    void  ops_PushTaskQueueN(queue)
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 *       ops_fatal
 *    From match.c:
 *       ops_TaskQueueOverflow
 */


/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char *ops_malloc();              /* Imported from utility.c. */
extern void  ops_fatal();               /* Imported from utility.c. */
extern void  ops_TaskQueueOverflow();   /* Imported from match.c. */



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
void    ops_PushTaskQueueN();
boolean PopTaskQueueN();




static alpha_cell  *free_acell = NULL;      /* Free list for alpha cells. */
static beta_cell   *free_4_bcell = NULL;    /* Free list for beta cells. */
static beta_cell   *free_32_bcell = NULL;   /* Free list for beta cells. */
static beta_cell   *free_128_bcell = NULL;  /* Free list for beta cells. */
static beta_cell   *free_256_bcell = NULL;  /* Free list for beta cells. */

static alpha_cell  *temp_free_acell = NULL;  /* Temporary free list for alpha cells. */
static beta_cell   *temp_free_bcell = NULL;  /* Temporary free list for beta cells.  */
static alpha_cell  *temp_free_acell_last = NULL;  /* Last cell on temporary free list for alpha cells. */
static beta_cell   *temp_free_bcell_last = NULL;  /* Last cell on temporary free list for beta cells.  */

static tq_cell     *free_tqcell = NULL;      /* Free list for alpha cells. */


#ifndef COMPLEX_HASH_TABLE_LOCKS
/* This is a hack to allow us to easily provide simple style hash
 * table locks with the macros in "psm_locks.h". Some day we should
 * clean up this to hide "tokHTlock" completely in "psm_locks.c".
 * But for now, we need the declaration below, so that the compiler
 * can compute the array access correctly.
 */
extern char  *tokHTlock;
#endif



/* This stuff in the ifdef is hacked in just for debugging use.
 */
#ifdef  SYS_DEBUG
#ifndef UNIPROC_VERSION
extern char *tokHTlock;
#ifdef COMPLEX_HASH_TABLE_LOCKS
extern int  *LeftRightCounter;
#endif
static void
show_counts()
{
   int i;

   for (i=0; i<MEM_HASHTABLE_SIZE; i++)
      if (LeftRightCounter[i])  if (show_debug) printf("SHOW: HKey=%d LeftRight[%d] = 0x%x\n", HKey, i, LeftRightCounter[i]);

   for (i=0; i<NumQueues; i++)
      if (*tq_lock[i])  printf("tqlockSHOW: tq_lock[%d] = 0x%x\n", i, tq_lock[i]);
}
#endif   /* UNIPROC_VERSION */
#endif   /* SYS_DEBUG */



static alpha_cell* alpha_alloc()
{
   alpha_cell *cell;

   if (*Global_Free_Acell)
     {
      Test_Then_Lock(Free_Acell_Lock);
      if (*Global_Free_Acell)
        {   
         cell = *Global_Free_Acell;
         *Global_Free_Acell = cell->next;
         Release(Free_Acell_Lock);
	 return(cell);
	}
      Release(Free_Acell_Lock);
     }


   /* Allocate a new one if none on global free list. */
   cell = (alpha_cell *) ops_malloc(sizeof(alpha_cell));
   return(cell);
}



static beta_cell* beta_4_alloc()
{
   beta_cell *cell;

   if (*Global_Free_4_Bcell)
     {
      Test_Then_Lock(Free_4_Bcell_Lock);
      if (*Global_Free_4_Bcell)
        {   
         cell = *Global_Free_4_Bcell;
         *Global_Free_4_Bcell = cell->next;
         Release(Free_4_Bcell_Lock);
	 return(cell);
	}
      Release(Free_4_Bcell_Lock);
     }


   /* Allocate a new one if none on global free list. We allocate enough for
    * the first 4 fields and then only enough for 4 elements of the token array.
    * This is done in order to save space.
    */
   cell = (beta_cell *) ops_malloc(3*sizeof(int) + PointerSize + 4*sizeof(int));
   return(cell);
}



static beta_cell* beta_32_alloc()
{
   beta_cell *cell;

   if (*Global_Free_32_Bcell)
     {
      Test_Then_Lock(Free_32_Bcell_Lock);
      if (*Global_Free_32_Bcell)
        {   
         cell = *Global_Free_32_Bcell;
         *Global_Free_32_Bcell = cell->next;
         Release(Free_32_Bcell_Lock);
	 return(cell);
	}
      Release(Free_32_Bcell_Lock);
     }


   /* Allocate a new one if none on global free list. We allocate enough for
    * the first 4 fields and then only enough for 32 elements of the token array.
    * This is done in order to save space.
    */
   cell = (beta_cell *) ops_malloc(3*sizeof(int) + PointerSize + 32*sizeof(int));
   return(cell);
}



static beta_cell* beta_128_alloc()
{
   beta_cell *cell;

   if (*Global_Free_128_Bcell)
     {
      Test_Then_Lock(Free_128_Bcell_Lock);
      if (*Global_Free_128_Bcell)
        {   
         cell = *Global_Free_128_Bcell;
         *Global_Free_128_Bcell = cell->next;
         Release(Free_128_Bcell_Lock);
	 return(cell);
	}
      Release(Free_128_Bcell_Lock);
     }


   /* Allocate a new one if none on global free list. We allocate enough for
    * the first 4 fields and then only enough for 128 elements of the token array.
    * This is done in order to save space.
    */
   cell = (beta_cell *) ops_malloc(3*sizeof(int) + PointerSize + 128*sizeof(int));
   return(cell);
}



static beta_cell* beta_256_alloc()
{
   beta_cell *cell;

   if (*Global_Free_256_Bcell)
     {
      Test_Then_Lock(Free_256_Bcell_Lock);
      if (*Global_Free_256_Bcell)
        {   
         cell = *Global_Free_256_Bcell;
         *Global_Free_256_Bcell = cell->next;
         Release(Free_256_Bcell_Lock);
	 return(cell);
	}
      Release(Free_256_Bcell_Lock);
     }


   /* Allocate a new one if none on global free list. We allocate enough for
    * the first 4 fields and then only enough for 256 elements of the token array.
    * This is done in order to save space.
    */
   cell = (beta_cell *) ops_malloc(3*sizeof(int) + PointerSize + 256*sizeof(int));
   return(cell);
}



/* Allocate a new alpha_cell for a beta node's right memory.
 * Used by ops_rmem in this module.
 */
#define new_alpha(new)		\
   if (free_acell)		\
     {				\
      new = free_acell;		\
      free_acell = free_acell->next;	\
     }				\
   else				\
      new = alpha_alloc();



static void
ops_free_alpha()
{
   if (temp_free_acell_last != NULL)
     {
       temp_free_acell_last->next = free_acell;
       free_acell  = temp_free_acell;

       temp_free_acell = temp_free_acell_last = NULL;
     }
}

/* Return an alpha_cell to the free list.  Used by ops_rmem. */
#define temp_free_alpha(acell)  	\
   if (temp_free_acell_last == NULL)						\
     { temp_free_acell = temp_free_acell_last = acell; acell->next = NULL; }	\
   else										\
     { acell->next = temp_free_acell; temp_free_acell  = acell; }


static beta_cell* new_beta(toksize)
   int toksize;
{
/*   Allocate a new beta_cell for a beta node's left memory.
 *   Used by ops_lmem in this module.
 */
   beta_cell *new;
   
   if (toksize <= 4)
     {
      if (free_4_bcell)
        {
         new = free_4_bcell;
         free_4_bcell = free_4_bcell->next;
        }
      else
         new = beta_4_alloc();
     }
   else if (toksize <= 32)
     {
      if (free_32_bcell)
        {
         new = free_32_bcell;
         free_32_bcell = free_32_bcell->next;
        }
      else
         new = beta_32_alloc();
     }
   else if (toksize <= 128)
     {
      if (free_128_bcell)
        {
         new = free_128_bcell;
         free_128_bcell = free_128_bcell->next;
        }
      else
         new = beta_128_alloc();
     }
   else
     {
      if (free_256_bcell)
        {
         new = free_256_bcell;
         free_256_bcell = free_256_bcell->next;
        }
      else
         new = beta_256_alloc();
     }

   return(new);
}



static void
ops_free_beta() 
{
   int        toksize;
   beta_cell *ptr;

   while (temp_free_bcell)
     {
      ptr = temp_free_bcell->next;
      toksize = temp_free_bcell->toksize;

      if (toksize <= 4)
        {
         temp_free_bcell->next = free_4_bcell;
         free_4_bcell  = temp_free_bcell;
	}
      else if (toksize <= 32)
        {
         temp_free_bcell->next = free_32_bcell;
         free_32_bcell  = temp_free_bcell;
	}
      else if (toksize <= 128)
        {
         temp_free_bcell->next = free_128_bcell;
         free_128_bcell  = temp_free_bcell;
	}
      else
        {
         temp_free_bcell->next = free_256_bcell;
         free_256_bcell  = temp_free_bcell;
	}

      temp_free_bcell = ptr;
     }

   temp_free_bcell_last = NULL;
}



static void temp_free_beta(bcell) 
   beta_cell *bcell;
{
 /* Return a beta_cell to the free list.  Called by ops_lmem. */

   if (temp_free_bcell_last == NULL)
     { temp_free_bcell = temp_free_bcell_last = bcell; bcell->next = NULL; }
   else
     { bcell->next = temp_free_bcell; temp_free_bcell  = bcell; }
}




static tq_cell* 
tq_cell_alloc()
{
   tq_cell *cell;

   if (*Global_Free_TQcell)
     {
      Test_Then_Lock(Free_TQcell_Lock);
      if (*Global_Free_TQcell)
        {   
         cell = *Global_Free_TQcell;
         *Global_Free_TQcell = cell->next;
         Release(Free_TQcell_Lock);
	 return(cell);
	}
      Release(Free_TQcell_Lock);
     }


   /* Allocate a new one if none on global free list. */
   cell = (tq_cell *) ops_malloc(sizeof(tq_cell));
   return(cell);
}


/* Allocate a new tq_cell for ops_PushTaskQueueN
 */
#define new_tq_cell(new)		\
   if (free_tqcell)			\
     {					\
      new = free_tqcell;		\
      free_tqcell = free_tqcell->next;	\
     }					\
   else					\
      new = tq_cell_alloc();


/* PopTaskQueue puts a tq_cell on the free list.
 */
#define free_tq_cell(tqcell)	\
   tqcell->next = free_tqcell;	\
   free_tqcell  = tqcell;




 
static int
ops_rmem(nodeid)
   register int nodeid;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add/delete a token to/from a beta node's right token memory hash table
 *    bucket. The token is built from a single wme just added/deleted to/from
 *    working memory as a result of an RHS action. Also set up a pointer to
 *    the corresponding left token memory hash table bucket. The left bucket
 *    has tokens that must be matched against the right token.
 *
 * Parameters:
 *    nodeid  - the node for which the token is being added/deleted.
 *
 * Global Parameters:
 *    HKey   - hash table position where token ought to be stored.
 *    curwme - pointer to the wme just added/deleted.
 *    curdir - tells whether to add/delete.
 *
 * Side Effects on Global Parameters:
 *    rtokHT[HKey] - the right token built from curwme is added/deleted at this bucket.
 *    R6           - set to point at token just added/deleted in bucket rtokHT[HKey].
 *    R7           - set to point to bucket ltokHT[HKey], the opposite token memory.
 *
 * Returns:
 *    0 => continue with the execution of the right beta node.
 *    1 => abort the execution of the right beta node (because we found that we
 *         had a conjugate token that cancelled our token).
 * Calls:
 *    "new_alpha" and "temp_free_alpha" in this module.
 *    locking macros in "psm_locks.h."
 *
 * Called by:
 *    The generic Rete match routines "ops_rightand_input" and "ops_rightnot_input"
 *    in this module.
 *
 *-------------------------------------------------------------------------*/
{
   register int hkey;
   register alpha_cell *acell;
   register alpha_cell *pred;


   hkey = HKey;        /* Put it in a register. */
   

   /* Lock the hash table for reading in the left
    * bucket and writing in the right bucket.
    */
   LockAccess_LeftRead_RightWrite(hkey);  /* Implements complex hash table locks. */
   /* Test_Then_Lock(tokHTlock[hkey]); ***** Old locking scheme ***/


   /* Now add/delete the current wme to the rmem.
    */
   if (curdir == DIRIN)
     {
      /* First see if token is the conjugate rmem. */
      if (rtokHTconj[hkey] != NULL)
        {
         acell = rtokHTconj[hkey];
         pred  = NULL;
         while (acell)
           {
            if ((acell->pWme == curwme) && (acell->nodeid == nodeid))
               break;
            else
              {
               pred = acell;
               acell = acell->next;
              }
           }
         
         if (acell != NULL)
           {
	    /* OK, we found the token in the conjugate memory so we just
	     * remove it from there and return with an indication that
	     * no further processing of the right beta node task is required.
	     */
            if (pred)
               pred->next = acell->next;
            else
               rtokHTconj[hkey] = acell->next;
   
            temp_free_alpha(acell);

            /* Release write access in right bucket, read access in left
	     * bucket is released when node task exits in PopTaskQueueN.
	     */
            Release_RightWrite(hkey); 

	    return(1);   /* i.e., LeaveBetaTask */
           }
        }
   
      /* Token not found in conjugate mem so add it to normal right
       * token memory hash bucket. Also set up global pointers R6
       * and R7 for use by the right beta node task when rmem returns.
       */
      new_alpha(acell);  /* New alpha cell to hold 'curwme'. */
      acell->pWme   = curwme;
      acell->nodeid = nodeid;
      acell->next   = rtokHT[hkey];
      rtokHT[hkey] = acell;
      R6.AlphaPtr = acell;
      R7.BetaPtr  = ltokHT[hkey];

      /* Release write access in right bucket, read access in left
       * bucket is released when node task exits in PopTaskQueueN.
       */
      Release_RightWrite(hkey); 

      return(0);   /* i.e., continue with right beta node */
     }
   else  /* DIROUT */
     {
      /* First look for token in the normal rmem. */
      if (rtokHT[hkey] != NULL)
        {
         acell = rtokHT[hkey];
         pred  = NULL;
         while (acell)
           {
            if ((acell->pWme == curwme) && (acell->nodeid == nodeid))
               break;
            else
              {
               pred = acell;
               acell = acell->next;
              }
           }
            
         if (acell != NULL)
           {
            /* OK, token was found in the normal right token memory bucket,
	     * so we remove the token from the bucket and set up the global
	     * pointers R6 and R7 for use by the right beta node task upon
	     * return from rmem.
	     */
            if (pred)
               pred->next = acell->next;
            else
               rtokHT[hkey] = acell->next;
   
            temp_free_alpha(acell);
            R6.AlphaPtr = acell;
	    R7.BetaPtr  = ltokHT[hkey];

            /* Release write access in right bucket, read access in left
	     * bucket is released when node task exits in PopTaskQueueN.
	     */
            Release_RightWrite(hkey); 

	    return(0);   /* i.e., continue with the right beta node */
           }
	}
   
      /* Token not found so add it to the conjugate right token memory
       * and return with and indication that no further processing is
       * needed by the right beta node task.
       */
      /* fprintf(fd_dbug, "RightMem:  Delete being processed before insert. \n"); */
      /* fflush(fd_dbug); */
      new_alpha(acell);  /* New alpha cell to hold 'curwme'. */
      acell->pWme   = curwme;
      acell->nodeid = nodeid;
      acell->next   = rtokHTconj[hkey];
      rtokHTconj[hkey] = acell;

      /* Release write access in right bucket, read access in left
       * bucket is released when node task exits in PopTaskQueueN.
       */
      Release_RightWrite(hkey); 

      return(1);   /* i.e., LeaveBetaTask */
     }

}






static int
ops_lmem(nodeid, betalev)
   register int nodeid;
   int betalev;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add/delete a token to/from a beta node's left token memory hash table
 *    bucket. The token is a list of wmes that match the condition elements
 *    that the node joins. The token is built from a left part and a right part
 *    passed down from the left and right memories of the predecessor node in
 *    the network. These 2 parts are supplied in the global parameters R9 and R8.
 *    If the predecessor node is not another beta node, then the token we build
 *    cosists of a single wme, curwme, which was just added/deleted to/from
 *    working memory as a result of an RHS action. Also set up a pointer to
 *    the corresponding right token memory hash table bucket. The right bucket
 *    has tokens that must be matched against the left token.
 *
 * Stack Parameters:
 *    nodeid  - the node for which the token is being added/deleted.
 *    betalev - the beta level of the node tells the length of tokens in
 *              the left memory; corresponds to the depth of the node in the
 *              Rete network and, thus, now many condition elements the node
 *              joins.
 *
 * Global Parameters:
 *    HKey     - hash table position where token ought to be stored.
 *    R9,R8 or - the token to be added/deleted is built from a left part and a
 *    curwme     right part (pointed to respectively by R9 and R8) or from a
 *               single wme (curwme) just added/deleted in working memory.
 *    curdir   - tells whether to add/delete the token.
 *
 * Side Effects on Global Parameters:
 *    ltokHT[HKey] - the left token built from (R9,R8)/curwme is added/deleted at this bucket.
 *    R7           - set to point at token just added/deleted in bucket ltokHT[HKey].
 *    R6           - set to point to bucket rtokHT[HKey], the opposite token memory.
 *
 * Returns:
 *    0 => continue with the execution of the left beta node.
 *    1 => abort the execution of the left beta node (because we found that we
 *         had a conjugate token that cancelled our token).
 *
 * Calls:
 *    "new_beta" and "temp_free_beta" in this module.
 *    locking macros in "psm_locks.h."
 *
 * Called by:
 *    The generic Rete match routines "ops_leftand_input" and "ops_leftnot_input"
 *    in this module.
 *
 *-------------------------------------------------------------------------*/
{

   register int hkey;
   register beta_cell  *r7;
   register beta_cell *pred;
   /*register*/ int k;


   hkey = HKey;        /* Put it in a register. */


   /* Lock the hash table for writing in the left
    * bucket and reading in the right bucket.
    */
   LockAccess_LeftWrite_RightRead(hkey);  /* Implements complex hash table locks. */
   /* Test_Then_Lock(tokHTlock[hkey]); ***** Old locking scheme ***/


   /* Now add/delete the token to the node's lmem.  */
   if (curdir == DIRIN)
     {
      /* First look for it in the conjugate token memory. */
      r7 = ltokHTconj[hkey];
      pred  = NULL;
      while (r7) 
        {
	    if (r7->nodeid == nodeid)
	      {
   	    	for (k = betalev-2; k >= 0; --k)
      	    	   if (r7->token[k] != R9.BetaPtr->token[k])
                   {
            	     pred  = r7;
            	     r7 = r7->next;
	    	     goto contin;
                   }

		if (betalev == 1)
		  {if (r7->token[0] == curwme)  break;}
		else
		  {
		   if (R8.AlphaPtr != NULL)
		      {if (R8.AlphaPtr->pWme == r7->token[betalev-1])  break;}
		   else
		      {if (r7->token[betalev-1] == NULL)  break;}
		  }

                pred  = r7;
            	r7 = r7->next;
	        goto contin;
	      }
	    else { pred = r7; r7 = r7->next; }
	    contin:;
        } /* endwhile */
         
      if (r7 != NULL)
        { /* Now take conjugate token out of the memory. */
         if (pred)
            pred->next = r7->next;
         else
           ltokHTconj[hkey] = r7->next;
         
         temp_free_beta(r7);

         /* Release write access in left bucket, read access in right
          * bucket is released when node task exits in PopTaskQueueN.
	  */
         Release_LeftWrite(hkey); 

         return(1);   /* i.e., LeaveBetaTask */
        }

      /* Didn't find it in conjugate mem so go ahead and add it to the normal mem. */
      r7 = new_beta(betalev);  /* New beta cell to hold token. */
      
      /* Stuff r7 with contents of tokens in passed in registers R8 and R9.  */
      for (k = betalev-2; k >= 0; --k)
         r7->token[k] = R9.BetaPtr->token[k];

      /* if betalev == 1 then curwme ought to be loaded directly;
       * else if R8.AlphaPtr==NULL then NULL ought to be loaded;
       * else R8.AlphaPtr->pWme has to be loaded;
       */
      if (betalev == 1) r7->token[0] = curwme;
      else if (R8.AlphaPtr != NULL) r7->token[betalev - 1] = R8.AlphaPtr->pWme;
      else r7->token[betalev - 1] = NULL;

      r7->toksize  = betalev;
      r7->refcount = 0;               /* Only needed for NOT_NODEs. */
      r7->nodeid   = nodeid;
      r7->next     = ltokHT[hkey];
      ltokHT[hkey] = r7;
      R6.AlphaPtr  = rtokHT[hkey];   /* Set up pointer to right memory for this betalev. */
      R7.BetaPtr   = r7;             /* Point to token with R7 global. */


      /* Release write access in left bucket, read access in right
       * bucket is released when node task exits in PopTaskQueueN.
       */
      Release_LeftWrite(hkey); 

      return(0);  /* i.e., continue with beta node */
     }
   else  /* DIROUT */
     {
      /* Find token in node's lmem and remove it.       */

      r7 = ltokHT[hkey];
      pred  = NULL;
      while (r7) 
        {
	    if (r7->nodeid == nodeid)
	      {
   	    	for (k = betalev-2; k >= 0; --k)
      	    	   if (r7->token[k] != R9.BetaPtr->token[k])
                   {
            	     pred  = r7;
            	     r7 = r7->next;
	    	     goto contin1;
                   }

		if (betalev == 1)
		  {if (r7->token[0] == curwme)  break;}
		else
		  {
		   if (R8.AlphaPtr != NULL)
		      {if (R8.AlphaPtr->pWme == r7->token[betalev-1])  break;}
		   else
		      {if (r7->token[betalev-1] == NULL)  break;}
		  }

                pred  = r7;
            	r7 = r7->next;
	        goto contin1;
	      }
	    else { pred = r7; r7 = r7->next; }
	    contin1:;
        } /* endwhile */
         
      if (r7 != NULL)
        { /* Now take token out of the memory. */
         if (pred)
            pred->next = r7->next;
         else
           ltokHT[hkey] = r7->next;
         
         temp_free_beta(r7);   /* Does not destroy 'refcount' in 'r7'. */
         R6.AlphaPtr  = rtokHT[hkey];   /* Set up pointer to right memory for this betalev. */
         R7.BetaPtr   = r7;             /* Point to token with R7 global. */

         /* Release write access in left bucket, read access in right
          * bucket is released when node task exits in PopTaskQueueN.
          */
         Release_LeftWrite(hkey); 

         return(0);   /* i.e., continue with beta node */
        }

      /* Token to be deleted was not in the normal lmem so just add it the
       * conjugate lmem and leave the beta node task.
       */
/*
      fprintf(fd_dbug, "LeftMem:  Delete being processed before insert. \n");
      fflush(fd_dbug);
*/      
      r7 = new_beta(betalev);  /* New beta cell to hold conjugate token. */
      
      /* Stuff r7 with contents of tokens in passed in registers R8 and R9.  */
      for (k = betalev-2; k >= 0; --k)
         r7->token[k] = R9.BetaPtr->token[k];

      /* if betalev == 1 then curwme ought to be loaded directly;
       * else if R8.AlphaPtr==NULL then NULL ought to be loaded;
       * else R8.AlphaPtr->pWme has to be loaded;
       */
      if (betalev == 1) r7->token[0] = curwme;
      else if (R8.AlphaPtr != NULL) r7->token[betalev - 1] = R8.AlphaPtr->pWme;
      else r7->token[betalev - 1] = NULL;


      r7->toksize  = betalev;
      r7->refcount = 0;               /* Only needed for NOT_NODEs. */
      r7->nodeid   = nodeid;
      r7->next     = ltokHTconj[hkey];
      ltokHTconj[hkey] = r7;

      /* Release write access in left bucket, read access in right
       * bucket is released when node task exits in PopTaskQueueN.
       */
      Release_LeftWrite(hkey); 

      return(1);  /* i.e., LeaveBetaTask */
     } /* if DIROUT */
}








void
ops_rightand_input(nodeid, and_testpass_nodeid)
   int nodeid, (*and_testpass_nodeid)();
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is a generic piece of code in the Rete network that
 *    implements part of the processing for a token flowing into the right
 *    input of an AND node in the network. A token entering a right input
 *    is essentially just some wme which has just been added/deleted to/from
 *    working memory. The node and the token it must process constitute the 
 *    basic unit of work in the Rete network and correspond to a task
 *    in the parallel implementation of the Rete match.
 *
 *    The actions performed by an AND node to handle a token from its right
 *    input are to add/delete the token to/from the node's right memory
 *    and then test to see if the token matches any tokens in the node's left
 *    memory. If a left token matches, a new token is created from the left
 *    and right tokens and is passed to the successor node in the network.
 *    This routine implements that generic processing and is called by every
 *    "right-and-node" task. Before this routine is called, some task specific
 *    processing is done to provide the parameters to this routine and set up
 *    some global variables to correspond to the particular task.
 *
 * Parameters:
 *    nodeid              - the unique ID assigned to the AND node.
 *    and_testpass_nodeid - the address of the specific routine that holds
 *                          the code for performing the match against a
 *                          token from the left memory; this same routine
 *                          will also pass along joined tokens that matched
 *                          to the successor node; the code for this routine
 *                          has been generated in a file by the CParaOPS5
 *                          compiler from the productions.
 *
 * Global Parameters:
 *    curwme - the wme being input to the AND node on its right input.
 *    curdir - indicates whether the wme is to be added or deleted.
 *    HKey   - the key that gives the hash bucket where right memory tokens
 *             are stored for the AND node; the key is based on the nodeid
 *             and the curwme (from its fields that are bound to equal variables).
 *
 * Environment, Actions, and Side Effects:
 *    Execution of this routine is preceded by several steps that begin the
 *    processing of the specific "right-and-node" task. It all begins in
 *    in "PopTaskQueueN" (in this module) where the task is dispatched. The 
 *    control block for the task is used to set up some of the global
 *    parameters above (curwme and curdir) and also supplies the address
 *    of the node specific routine that is invoked in "PopTaskQueueN".
 *    That routine sets up HKey and the parameters to "ops_rightand_input",
 *    our generic routine here. The side effects of the processing here are
 *    to add/delete a token in the node's right memory and perform the match
 *    against the opposite memory. This may result in tokens built and passed
 *    down to the successor node in the network. Tasks for successor nodes
 *    are queued by calls to "ops_PushTaskQueueN" (in this module) invoked as part
 *    of the processing in the routine at address "and_testpass_nodeid".
 *    "ops_PushTaskQueueN" relies on the globals succdir, R7, R6, and R0 in order
 *    to build the task control block for a successor node task.
 *
 * Side Effects on Globals:
 *    rtokHT[HKey] or  - the token is added/deleted to the hash table bucket
 *    rtokHTconj[HKey]   of the node's right memory.
 *    succdir          - the direction tag for tokens that get passed to the
 *                       successor node is set here; direction is add or delete.
 *    R7               - points to a token in the left hash bucket.
 *    R6               - points to the token added/deleted in the right hash bucket.
 *    R0               - gets set to the address of the successor node task.
 *
 * Calls:
 *    the task specific routine whose address is passed in the parameter
 *    "and_testpass_nodeid"; this routine is in the code file produced by
 *    the CParaOPS5 compiler.
 *    "ops_rmem" in this module.
 *
 * Returns:
 *    Nothing.
 *
 * Called by:
 *    A routine in the Rete network code that implements a task corresponding
 *    to a token flowing into right input of an AND node in the Rete network.
 *    The calling routine is part of the code file generated by the CParaOPS5
 *    compiler from the production system program.
 *
 *-------------------------------------------------------------------------*/
{
   /* For AND nodes, the tag on tokens constructed and passed to the
    * successor node will be the same as that for the token entering
    * the AND node here. So we set up succdir here.
    */
   succdir = curdir;

   /* Now add/delete the token to the node's right memory. Sets up the
    * globals R6 and R7 which respectively point to the token just
    * added/deleted and the opposite (or left) hash table bucket.
    */
   if (ops_rmem(nodeid))  return;    /* We return here only if "ops_rmem"   */
                                     /* finds we are dealing with conjugate */
				     /* tokens; in this case no more        */
				     /* processing is needed.               */

   /* Now run down the list of tokens in the left hash bucket to
    * test for matches with the right token. A pair of matching tokens
    * gets passed to (i.e., queued as a task for) the successor node.
    * The node specific routine addressed by "and_testpass_nodeid"
    * tests for a match between a left token and the right token
    * and does the queueing of a successor node task if a token pair
    * satisfies the match tests.
    */
   while (R7.BetaPtr)
     {
      if (R7.BetaPtr->nodeid == nodeid)  (*and_testpass_nodeid)();
      R7.BetaPtr = R7.BetaPtr->next;
     }

   /* Depending on the locking scheme used for the hash table token
    * memories, the processing in "ops_rmem" may have required that
    * the left hash bucket remain locked for read only access during
    * our processing here. When the call stack is unwound and the
    * node specific task returns to the dispatch point in "PopTaskQueueN",
    * the read only lock would have to be released.
    */
}


void
ops_leftand_input(nodeid, betalev, and_testpass_nodeid)
   int nodeid, betalev, (*and_testpass_nodeid)();
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is a generic piece of code in the Rete network that
 *    implements part of the processing for a token flowing into the left
 *    input of an AND node in the network. The node and the token it must
 *    process constitute the basic unit of work in the Rete network and
 *    correspond to a task in the parallel implementation of the Rete match.
 *    A token entering a left input is a list of wmes that constitute a
 *    partial instantiation of a production. If the list consists of say
 *    N wmes, then the token is a partial instantiation that matches the
 *    first N condition elements of the production. If N = 1, the token is
 *    just a single wme which was just added/deleted to/from working memory.
 *    For N > 1, the token was built from a left token and a right token
 *    joined together and passed down from the node memories of the predecessor
 *    node. The handling of these 2 cases for N involves some small differences
 *    in the use of the global parameters during the processing for the node
 *    task which will be noted below.
 *
 *    The actions performed by an AND node to handle a token from its left
 *    input are to add/delete the token to/from the node's left memory
 *    and then test to see if the token matches any tokens in the node's right
 *    memory. If a right token matches, a new token is created from the left
 *    and right tokens and is passed to the successor node in the network.
 *    This routine implements that generic processing and is called by every
 *    "left-and-node" task. Before this routine is called, some task specific
 *    processing is done to provide the parameters to this routine and set up
 *    some global variables to correspond to the particular task.
 *
 * Parameters:
 *    nodeid              - the unique ID assigned to the AND node.
 *    betalev             - the length of tokens in the left memory; corresponds
 *                          to the depth of the AND node in the Rete network.
 *    and_testpass_nodeid - the address of the specific routine that holds
 *                          the code for performing the match against a
 *                          token from the right memory; this same routine
 *                          will also pass along joined tokens that matched
 *                          to the successor node; the code for this routine
 *                          has been generated in a file by the CParaOPS5
 *                          compiler from the productions.
 *
 * Global Parameters:
 *    R9,R8 or - the token being input to the AND node on its left input; if
 *    curwme     N = 1 (i.e., the betalev), the node is at the top of the
 *               network and the token is just the curwme; if N > 1, then the
 *               token consists of a left part and a right part passed down
 *               from the respective left and right memories of the predecessor
 *               node; the token comes into the AND node here with R9 pointing
 *               to the left part and R8 pointing to the right part; we concatenate
 *               these 2 parts to form a new left token to be added/deleted to/from
 *               the left memory of the AND node.
 *    curdir   - indicates whether the token is to be added or deleted.
 *    HKey     - the key that gives the hash bucket where left memory tokens
 *               are stored for the AND node; the key is based on the nodeid
 *               and the token (from wme fields that are bound to equal variables).
 *
 * Environment, Actions, and Side Effects:
 *    Execution of this routine is preceded by several steps that begin the
 *    processing of the specific "left-and-node" task. It all begins in
 *    in "PopTaskQueueN" (in this module) where the task is dispatched. The 
 *    control block for the task is used to set up some of the global
 *    parameters above (R9,R8/curwme and curdir) and also supplies the address
 *    of the node specific routine that is invoked in "PopTaskQueueN".
 *    That routine sets up HKey and the parameters to "ops_leftand_input",
 *    our generic routine here. The side effects of the processing here are
 *    to add/delete a token in the node's left memory and perform the match
 *    against the opposite memory. This may result in tokens built and passed
 *    down to the successor node in the network. Tasks for successor nodes
 *    are queued by calls to "ops_PushTaskQueueN" (in this module) invoked as part
 *    of the processing in the routine at address "and_testpass_nodeid".
 *    "ops_PushTaskQueueN" relies on the globals succdir, R7, R6, and R0 in order
 *    to build the task control block for a successor node task.
 *
 * Side Effects on Globals:
 *    ltokHT[HKey] or  - the token is added/deleted to the hash table bucket
 *    ltokHTconj[HKey]   of the node's left memory.
 *    succdir          - the direction tag for tokens that get passed to the
 *                       successor node is set here; direction is add or delete.
 *    R7               - points to the token added/deleted in the left hash bucket;
 *                       this token is the one formed from the left and right parts
 *                       described by R9,R8 or curwme in the Global Parameters above.
 *    R6               - points to a token in the right hash bucket.
 *    R0               - gets set to the address of the successor node task.
 *
 * Calls:
 *    the task specific routine whose address is passed in the parameter
 *    "and_testpass_nodeid"; this routine is in the code file produced by
 *    the CParaOPS5 compiler.
 *    "ops_lmem" in this module.
 *
 * Returns:
 *    Nothing.
 *
 * Called by:
 *    A routine in the Rete network code that implements a task corresponding
 *    to a token flowing into left input of an AND node in the Rete network.
 *    The calling routine is part of the code file generated by the CParaOPS5
 *    compiler from the production system program.
 *
 *-------------------------------------------------------------------------*/
{
   /* For AND nodes, the tag on tokens constructed and passed to the
    * successor node will be the same as that for the token entering
    * the AND node here. So we set up succdir here.
    */
   succdir = curdir;

   /* Now add/delete the token to the node's left memory. Sets up the
    * globals R7 and R6 which respectively point to the token just
    * added/deleted and the opposite (or right) hash table bucket.
    */
   if (ops_lmem(nodeid, betalev))  return;    /* We return here only if "ops_lmem"   */
                                              /* finds we are dealing with conjugate */
				              /* tokens; in this case no more        */
				              /* processing is needed.               */

   /* Now run down the list of tokens in the right hash bucket to
    * test for matches with the left token. A pair of matching tokens
    * gets passed to (i.e., queued as a task for) the successor node.
    * The node specific routine addressed by "and_testpass_nodeid"
    * tests for a match between the left token and a right token
    * and does the queueing of a successor node task if a token pair
    * satisfies the match tests.
    */
   while (R6.AlphaPtr)
     {
      if (R6.AlphaPtr->nodeid == nodeid)  (*and_testpass_nodeid)();
      R6.AlphaPtr = R6.AlphaPtr->next;
     }

   /* Depending on the locking scheme used for the hash table token
    * memories, the processing in "ops_lmem" may have required that
    * the right hash bucket remain locked for read only access during
    * our processing here. When the call stack is unwound and the
    * node specific task returns to the dispatch point in "PopTaskQueueN",
    * the read only lock would have to be released.
    */
}




int
ops_rightnot_input(nodeid, not_test_nodeid, not_pass_nodeid)
   int nodeid, (*not_pass_nodeid)();
   boolean (*not_test_nodeid)();
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is a generic piece of code in the Rete network that
 *    implements part of the processing for a token flowing into the right
 *    input of a NOT node in the network. A token entering a right input
 *    is essentially just some wme which has just been added/deleted to/from
 *    working memory. The node and the token it must process constitute the 
 *    basic unit of work in the Rete network and correspond to a task
 *    in the parallel implementation of the Rete match.
 *
 *    The actions performed by a NOT node to handle a token from its right
 *    input are to add/delete the token to/from the node's right memory
 *    and then test to see if the token matches any tokens in the node's left
 *    memory. If a left token matches, a new token may be created and passed
 *    down to the successor node. This new token consists of the matching
 *    left token and a NULL right token (since we are dealing with a NOT node
 *    here). Also, the new token is sent only if the reference count on the
 *    left token has the correct value when the right token is added/deleted.
 *    The reference count on a left token indicates how many tokens in the
 *    right memory currently match the left token. This routine implements the
 *    above generic processing and is called by every "right-not-node" task.
 *    Before this routine is called, some task specific processing is done to
 *    provide the parameters to this routine and set up some global variables
 *    to correspond to the particular task.
 *
 * Parameters:
 *    nodeid              - the unique ID assigned to the NOT node.
 *    not_test_nodeid     - the address of the specific routine that holds
 *                          the code for performing the match against a
 *                          token from the left memory; the code for this
 *                          routine has been generated in a file by the CParaOPS5
 *                          compiler from the productions.
 *    not_pass_nodeid     - the address of the specific routine that holds
 *                          the code for passing a new token to the successor
 *                          node when a left token matches the right token;
 *                          the code for this routine has been generated in a
 *                          file by the CParaOPS5 compiler from the productions.
 *
 * Global Parameters:
 *    curwme - the wme being input to the NOT node on its right input.
 *    curdir - indicates whether the wme is to be added or deleted.
 *    HKey   - the key that gives the hash bucket where right memory tokens
 *             are stored for the NOT node; the key is based on the nodeid
 *             and the curwme (from its fields that are bound to equal variables).
 *
 * Environment, Actions, and Side Effects:
 *    Execution of this routine is preceded by several steps that begin the
 *    processing of the specific "right-not-node" task. It all begins in
 *    in "PopTaskQueueN" (in this module) where the task is dispatched. The 
 *    control block for the task is used to set up some of the global
 *    parameters above (curwme and curdir) and also supplies the address
 *    of the node specific routine that is invoked in "PopTaskQueueN".
 *    That routine sets up HKey and the parameters to "ops_rightnot_input",
 *    our generic routine here. The side effects of the processing here are
 *    to add/delete a token in the node's right memory and perform the match
 *    against the opposite memory. This may result in tokens built and passed
 *    down to the successor node in the network. Tasks for successor nodes
 *    are queued by calls to "ops_PushTaskQueueN" (in this module) invoked as part
 *    of the processing in the routine at address "not_pass_nodeid".
 *    "ops_PushTaskQueueN" relies on the globals succdir, R7, R6, and R0 in order
 *    to build the task control block for a successor node task.
 *
 * Side Effects on Globals:
 *    rtokHT[HKey] or  - the token is added/deleted to the hash table bucket
 *    rtokHTconj[HKey]   of the node's right memory.
 *    ltokHT[HKey]     - tokens in the left memory may have their refcount updated.
 *    succdir          - the direction tag for tokens that get passed to the
 *                       successor node is set here; direction is add or delete.
 *    R7               - points to a token in the left hash bucket.
 *    R6               - points to the token added/deleted in the right hash bucket.
 *    R0               - gets set to the address of the successor node task.
 *
 * Calls:
 *    the task specific routines whose addresses are passed in the parameters
 *    "not_test_nodeid" and "not_pass_nodeid"; this routine is in the code file
 *    produced by the CParaOPS5 compiler.
 *    "ops_rmem" in this module.
 *
 * Returns:
 *    An integer code that tells if we are dealing with conjugate tokens (code=1)
 *    or not (code=0).
 *
 * Called by:
 *    A routine in the Rete network code that implements a task corresponding
 *    to a token flowing into right input of a NOT node in the Rete network.
 *    The calling routine is part of the code file generated by the CParaOPS5
 *    compiler from the production system program.
 *
 *-------------------------------------------------------------------------*/
{
   boolean tests_satisfied;
   DataPtr Tmp;

   /* For "right-not-nodes", the tag on tokens constructed and passed to the
    * successor node will be the opposite as that for the token entering
    * the NOT node here. So we set up succdir here.
    */
   succdir = (curdir == DIROUT) ? DIRIN : DIROUT;

   /* Now add/delete the token to the node's right memory. Sets up the
    * globals R6 and R7 which respectively point to the token just
    * added/deleted and the opposite (or left) hash table bucket.
    */
   if (ops_rmem(nodeid))             /* We return here only if "ops_rmem"   */
      return(EXIT_NODE /* 1 */);     /* finds we are dealing with conjugate */
				     /* tokens; in this case no more        */
				     /* processing is needed.               */

   /* Now run down the list of tokens in the left hash bucket to
    * test for matches with the right token. The node specific
    * routine addressed by "not_test_nodeid" tests if a left token
    * matches the right token. Since we are dealing with a NOT node,
    * the action taken on a successful match is a bit more complex
    * than AND nodes. The token sent to the successor node consists
    * of the left token concatenated with a NULL right token (instead
    * of the matching right token). Also the successor node receives
    * the token only if the reference count on the left token has the
    * correct value with respect to the direction tag of the right token.
    * The reference count on the left token indicates how many tokens in
    * the right memory currently match the left token. The other node
    * specific routine, addressed by "not_pass_nodeid", handles sending
    * the new token to the successor node by queueing a node task for the
    * successor node.
    */
   while (R7.BetaPtr)
     {
       if (R7.BetaPtr->nodeid == nodeid)
        {
	 if (not_test_nodeid == NULL)   /* No tests to perform. */
	    tests_satisfied = TRUE;
	 else
	    tests_satisfied = (*not_test_nodeid)();

	 if (tests_satisfied)
	   {
            /* A left token matches the right token so send a token to
	     * the successor if the refcount has the right value. The
	     * new token will consist of the left token (pointed to by R7)
	     * concatenated with a NULL right token. "ops_PushTaskQueueN"
	     * requires R7 and R6 as pointers to the respective left and
	     * right parts of the new token.
	     */
	    if (curdir == DIRIN)
	       {
		/* We send a new token down only when the refcount on
		 * the matching left token goes from 0 to 1 as a result
		 * of adding the right token to the right memory.
		 */
		if (++R7.BetaPtr->refcount == 1)
		  {
	           Tmp.Ptr = R6.Ptr;  R6.Ptr = NULL;
	           (*not_pass_nodeid)();
	           R6.Ptr = Tmp.Ptr;
		  }
	       }
	    else  /* DIROUT */
	       {
		/* We send a new token down only when the refcount on
		 * the matching left token goes from 1 to 0 as a result
		 * of deleting the right token from the right memory.
		 */
		if (--R7.BetaPtr->refcount == 0)
		  {
	           Tmp.Ptr = R6.Ptr;  R6.Ptr = NULL;
	           (*not_pass_nodeid)();
	           R6.Ptr = Tmp.Ptr;
		  }
	       }
	   }
	}
      R7.BetaPtr = R7.BetaPtr->next;
     }

   return(CONTINUE_NODE /* 0 */);

   /* Depending on the locking scheme used for the hash table token
    * memories, the processing in "ops_rmem" may have required that
    * the left hash bucket remain locked for read only access during
    * our processing here. When the call stack is unwound and the
    * node specific task returns to the dispatch point in "PopTaskQueueN",
    * the read only lock would have to be released.
    */
}





int
ops_leftnot_input(nodeid, betalev, not_test_nodeid)
   int nodeid, betalev;
   boolean (*not_test_nodeid)();
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is a generic piece of code in the Rete network that
 *    implements part of the processing for a token flowing into the left
 *    input of a NOT node in the network. The node and the token it must
 *    process constitute the basic unit of work in the Rete network and
 *    correspond to a task in the parallel implementation of the Rete match.
 *    A token entering a left input is a list of wmes that constitute a
 *    partial instantiation of a production. If the list consists of say
 *    N wmes, then the token is a partial instantiation that matches the
 *    first N condition elements of the production. If N = 1, the token is
 *    just a single wme which was just added/deleted to/from working memory.
 *    For N > 1, the token was built from a left token and a right token
 *    joined together and passed down from the node memories of the predecessor
 *    node. The handling of these 2 cases for N involves some small differences
 *    in the use of the global parameters during the processing for the node
 *    task which will be noted below.
 *
 *    The actions performed by a NOT node to handle a token from its left
 *    input are to add/delete the token to/from the node's left memory
 *    and then test to see if the token matches any tokens in the node's right
 *    memory. If the left token is being deleted, then we do not need to do
 *    any matching against the right memory since the refcount of the left
 *    token tells us already how many right tokens match it. We don't care
 *    about what those right tokens are since they are not needed when we
 *    construct a new token to pass down to the successor of the NOT node.
 *    Such a new token will consist of the left token and a NULL right token.
 *    If the left token is being added, then we match against the right memory
 *    in order to find how many right tokens match and set the refcount of the
 *    left token. For a "left-not-node" task, a new token is constructed and
 *    passed down only if the refcount on the left token is found to be 0.
 *    This routine implements the above  generic processing and is called by every
 *    "left-not-node" task. Before this routine is called, some task specific
 *    processing is done to provide the parameters to this routine and set up
 *    some global variables to correspond to the particular task.
 *
 * Parameters:
 *    nodeid              - the unique ID assigned to the NOT node.
 *    betalev             - the length of tokens in the left memory; corresponds
 *                          to the depth of the NOT node in the Rete network.
 *    not_test_nodeid     - the address of the specific routine that holds
 *                          the code for performing the match against a
 *                          token from the left memory; the code for this
 *                          routine has been generated in a file by the CParaOPS5
 *                          compiler from the productions.
 *
 * Global Parameters:
 *    R9,R8 or - the token being input to the NOT node on its left input; if
 *    curwme     N = 1 (i.e., the betalev), the node is at the top of the
 *               network and the token is just the curwme; if N > 1, then the
 *               token consists of a left part and a right part passed down
 *               from the respective left and right memories of the predecessor
 *               node; the token comes into the NOT node here with R9 pointing
 *               to the left part and R8 pointing to the right part; we concatenate
 *               these 2 parts to form a new left token to be added/deleted to/from
 *               the left memory of the NOT node.
 *    curdir   - indicates whether the token is to be added or deleted.
 *    HKey     - the key that gives the hash bucket where left memory tokens
 *               are stored for the NOT node; the key is based on the nodeid
 *               and the token (from wme fields that are bound to equal variables).
 *
 * Environment, Actions, and Side Effects:
 *    Execution of this routine is preceded by several steps that begin the
 *    processing of the specific "left-not-node" task. It all begins in
 *    in "PopTaskQueueN" (in this module) where the task is dispatched. The 
 *    control block for the task is used to set up some of the global
 *    parameters above (R9,R8/curwme and curdir) and also supplies the address
 *    of the node specific routine that is invoked in "PopTaskQueueN".
 *    That routine sets up HKey and the parameters to "ops_leftnot_input",
 *    our generic routine here. The side effects of the processing here are
 *    to add/delete a token in the node's left memory and perform the match
 *    against the opposite memory. This may result in a token built and passed
 *    down to the successor node in the network. Tasks for successor nodes
 *    are queued by calls to "ops_PushTaskQueueN" (in this module) invoked as part
 *    of the final processing in the calling routine. The caller uses the
 *    refcount of the left token to determine if a new token should be passed.
 *    "ops_PushTaskQueueN" relies on the globals succdir, R7, R6, and R0 in order
 *    to build the task control block for a successor node task.
 *
 * Side Effects on Globals:
 *    ltokHT[HKey] or  - the token is added/deleted to the hash table bucket
 *    ltokHTconj[HKey]   of the node's left memory; the token will have its
 *                       refcount set by the processing here.
 *    succdir          - the direction tag for tokens that get passed to the
 *                       successor node is set here; direction is add or delete.
 *    R7               - points to the token added/deleted in the left hash bucket;
 *                       this token is the one formed from the left and right parts
 *                       described by R9,R8 or curwme in the Global Parameters above.
 *    R6               - points to a token in the right hash bucket.
 *    R0               - gets set to the address of the successor node task.
 *
 * Calls:
 *    the task specific routine whose address is passed in the parameter
 *    "not_test_nodeid"; this routine is in the code file produced by
 *    the CParaOPS5 compiler.
 *    "ops_lmem" in this module.
 *
 * Returns:
 *    An integer code that tells if we are dealing with conjugate tokens (code=1)
 *    or not (code=0).
 *
 * Called by:
 *    A routine in the Rete network code that implements a task corresponding
 *    to a token flowing into left input of an NOT node in the Rete network.
 *    The calling routine is part of the code file generated by the CParaOPS5
 *    compiler from the production system program.
 *
 *-------------------------------------------------------------------------*/
{
   boolean tests_satisfied;

   /* For "left-not-nodes", the tag on tokens constructed and passed to the
    * successor node will be the same as that for the token entering
    * the NOT node here. So we set up succdir here.
    */
   succdir = curdir;

   /* Now add/delete the token to the node's left memory. Sets up the
    * globals R7 and R6 which respectively point to the token just
    * added/deleted and the opposite (or right) hash table bucket.
    */
   if (ops_lmem(nodeid, betalev))    /* We return here only if "ops_lmem"   */
      return(EXIT_NODE /* 1 */);     /* finds we are dealing with conjugate */
				     /* tokens; in this case no more        */
				     /* processing is needed.               */

   /* If the direction is delete, then the left token is already in the
    * node's left memory and thus its refcount shows how many tokens
    * in the right memory match it. Thus we don't need to match against
    * the right memory. (Remember that the new token sent to the successor
    * node is built from the left token and a NULL right token.) So we
    * return to the caller who checks the refcount and passes a new token
    * to the successor if refcount is 0.
    */
   if (curdir == DIROUT)  return(CONTINUE_NODE /* 0 */);

   /* Now run down the list of tokens in the right hash bucket to
    * test for matches with the left token. The node specific
    * routine addressed by "not_test_nodeid" tests if a right token
    * matches the left token. For "left-not-nodes", we only need to find
    * out how many tokens in the right memory match the left memory
    * since the token sent to the successor node consists of the left
    * token concatenated with a NULL right token (instead of the
    * matching right token). Also the successor node receives
    * the token only if the reference count on the left token has the
    * value 0. We return to the caller who deals with the details of this.
    */
   while (R6.AlphaPtr)
     {
      if (R6.AlphaPtr->nodeid == nodeid)
        {
	 if (not_test_nodeid == NULL)   /* No tests to perform. */
	    tests_satisfied = TRUE;
	 else
	    tests_satisfied = (*not_test_nodeid)();

	 if (tests_satisfied)  R7.BetaPtr->refcount++;
        }
      R6.AlphaPtr = R6.AlphaPtr->next;
     }

   /* Return to the caller who continues with the node task by
    * checking refcount and, if 0, queues up a task for the
    * successor node.
    */
   return(CONTINUE_NODE /* 0 */);

   /* Depending on the locking scheme used for the hash table token
    * memories, the processing in "ops_lmem" may have required that
    * the right hash bucket remain locked for read only access during
    * our processing here. When the call stack is unwound and the
    * node specific task returns to the dispatch point in "PopTaskQueueN",
    * the read only lock would have to be released.
    */
}






void
ops_MatcherCode()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This is the main control loop of the Rete match. Matcher processes,
 *    when forked at system startup, enter this routine and loop indefinitely
 *    looking for and executing node tasks from the task queue(s) and exit
 *    only at program termination when the control process sets the
 *    "end_of_run_flag". When multiple task queues are used, a matcher process
 *    is assigned a particular queue to/from which the matcher pushes/pops
 *    tasks. If a matcher cannot find a task in its assigned queue, it
 *    searches the other queues round robin for a task to execute. Each
 *    matcher process also is assigned a bit in the activity "BitVector"
 *    which shows whether or not the matcher is engaged in executing tasks
 *    or is idle. A matcher sets its activity bit whenever it pops a task
 *    and resets it only when it cannot find another available task in any of
 *    the queues. When the match phase is over for a given OPS5 MRA cycle,
 *    the task queues are empty, the "BitVector" is 0, and all the matchers
 *    execute the idle loop where they remain until the control process
 *    signals them for a new cycle by setting the "BitVector".
 *
 *    The control process does not execute "ops_MatcherCode" except for
 *    certain top level commands when the program is being run interactively 
 *    from the command interpreter or unless no dedicated match processes have
 *    been forked at system startup.
 *
 * Parameters:
 *    None.
 *
 * Global Parameters:
 *    tq_index[q] - each task queue, "q", has an index which tells how many
 *                  tasks are in the queue.
 *
 * Environment:
 *    The control process is executing the basic OPS5 MRA cycle for the
 *    production system program. It waits for the match to be completed,
 *    performs conflict resolution to select a production instantiation
 *    to fire, and then fires the production actions which generate new
 *    tasks in the queue to provoke the match on the next cycle.
 *
 * Calls:
 *    "PopTaskQueueN", "ops_free_alpha", and "ops_free_beta" in this module.
 *    Locking marcos from "psm_locks.h".
 *
 * Returns:
 *    Nothing.
 *
 * Called by:
 *    "ops_start_processes", "ops_main", and "wait_for_rete"  in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   register int i, q;

LookForNewTask:
   /* Make a pass over the queues looking for a task. Begin at an assigned
    * queue and then look round robin. If one is found, go pop it and
    * execute it.
    */
   for (i = NumQueues, q = MyPrimaryQueue; i > 0; i--, q = (q + 1) % NumQueues)
     {
      if (*tq_index[q] >= 0)
        {
	 if (PopTaskQueueN(q))  goto LookForNewTask;
	}
     }

CheckBit:
   /* If we get here, we didn't find any task in the queues, so make sure
    * our active bit is reset.
    */
   if (*BitVector & ProcessIDBit)
     {
      Test_Then_Lock(tb_lock);
      *BitVector &= ~ProcessIDBit;
      Release(tb_lock);
     }

CheckIdle:
   /* If no match process is actively working on a task, just loop here
    * until something new comes along. Control process will signal new
    * work by setting BitVector.
    */
   while (*BitVector == 0)
     {
      ops_free_alpha();
      ops_free_beta();
      if (ProcessID == 0)  return;  /* Control Process */
      if (*end_of_run_flag)  return;
     }

   /* There's work to do so go to it.
    */
   goto LookForNewTask;
}




void
ops_PushTaskQueue()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called only from the Rete network code in the file
 *    generated by the CParaOPS5 compiler from the productions. The call
 *    here is simply used to convert a generic PushTaskQueue into one that
 *    will cause the task to be pushed to the queue assigned to the matcher
 *    process queueing the task.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    A matcher process generates a new task while executing a node task.
 *
 * Calls:
 *    "ops_PushTaskQueueN" in this module.
 *
 * Returns:
 *    Nothing.
 *
 * Called by:
 *    Rete match code.
 *
 *-------------------------------------------------------------------------*/
{
   ops_PushTaskQueueN(MyPrimaryQueue);
}



void
ops_PushTaskQueueN(queue)
   register int queue;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Push a node task into the specified queue. Note that a queue is
 *    is implemented as a LIFO data structure.
 *
 * Parameters:
 *    queue - identifies the queue where the task should be pushed.
 *
 * Global Parameters:
 *    tq_queue[queue] - the queue that gets the task.
 *    tq_index[queue] - index into the queue where last task was added.
 *    tq_lock[queue]  - lock for the queue.
 *    succdir         - direction (add/delete) for the node task.
 *    R7              - the left part of the token for the node task.
 *    R6              - the right part of the token for the node task.
 *    R0              - the address in the Rete code of the node task.
 *
 * Environment:
 *    New tasks are created either by the control process or one of the
 *    matcher processes. The control process creates a task as the result
 *    of executing a RHS action that adds/deletes a wme to/from working
 *    memory. This results in the ROOT node task being queued with the wme
 *    as its input. In this case, R6 points to the wme and R7 is NULL. A
 *    matcher process creates a task as a result of executing one of the
 *    node tasks from the queue(s). The new task is for the successor node.
 *    If the successor node is directly below the ROOT in the Rete network,
 *    then we also have R6 pointing to a wme and R7 NULL. Otherwise, for
 *    nodes lower in the network, R7 points to a token from the left
 *    memory of the parent of the successor node and R6 points to a token
 *    from the parent right memory. These left and right parts are the result
 *    of a successful match just performed by the matcher process executing
 *    the parent node task. The parent task supplies succdir and R0 also.
 *
 * Calls:
 *    "ops_TaskQueueOverFlow" in "match.c".
 *    Locking macros in "psm_locks.h".
 *    "new_tq_cell" in this module.
 *
 * Returns:
 *    Nothing.
 *
 * Called by:
 *    "ops_PushTaskQueue" in this module.
 *    "ops_eval_rete" in the Rete match code.
 *
 *-------------------------------------------------------------------------*/
{
   register tq_cell *ptask;
   register int     index;

   /* Get a new task control block and fill it.
    */
   new_tq_cell(ptask);
   ptask->direction = succdir;
   ptask->routine   = R0;
   ptask->lpart.Ptr = R7.Ptr;
   ptask->rpart.Ptr = R6.Ptr;

   /* Lock the queue while we add the task.
    */
   Test_Then_Lock(tq_lock[queue]);
   if ((index = ++*tq_index[queue]) == QueueSize)
      ops_TaskQueueOverflow(queue);
   else
      *(task_queue[queue] + index) = ptask;
   Release(tq_lock[queue]);
}



static
boolean
PopTaskQueueN(queue)
   register int queue;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine removes a task from a queue and dispatches it. It first
 *    locks the queue, makes sure a task is queued, removes the task, unloads
 *    the task control block, and then builds the call to the node task. The
 *    node task delivers a return code which is used here to identify the
 *    type of node task just executed. We need that information so that
 *    read-only locks held on left or right token memory hash table buckets
 *    for the execution of beta nodes can be released.
 *
 * Parameters:
 *    queue - the queue from which to get the task.
 *
 * Environment:
 *    The match phase is active during the MRA cycle.
 *
 * Calls:
 *    The node task popped from the queue.
 *    Locking macros in "psm_locks.h".
 *    "free_tq_cell" in this module.
 *
 * Returns:
 *    FALSE if we found the queue empty when we checked it under lock.
 *    TRUE if we popped a task and executed it.
 *
 * Called by:
 *    "ops_MatcherCode" in this module.
 *
 *-------------------------------------------------------------------------*/
{

   register int *hkey;

   register tq_cell *ptask;
   register int     index;
   register int     ret_code;


   hkey = &HKey;        /* Put it in a register. */

   Test_Then_Lock(tq_lock[queue]);             /* Lock queue during critical section. */
   index = *tq_index[queue];                   /* Now it's OK to get the queue index. */
   if (index < 0)
     {
      Release(tq_lock[queue]);                 /* Empty queue so just unlock and go home. */
      return(FALSE);
     }
   else
     {
      /* NOTE: It is ESSENTIAL that the "if" statement (labelled 1 at left)
       *       precede the statement (labelled 2 at left) that modifies the
       *       task queue.
       */
      /* MUST make sure our active bit is set before we remove the task
       * from the queue so that the control process does not see
       * the queue empty and our bit clear at the same time.
       * Otherwise we could get disaster with control process
       * thinking that match has reached quiescence.
       */
/*1*/ if ((*BitVector & ProcessIDBit) == 0)
/*.*/   {
/*.*/    Test_Then_Lock(tb_lock);
/*.*/	 *BitVector = *BitVector | ProcessIDBit;
/*.*/	 Release(tb_lock);
/*.*/	}
/*.*/ ptask = *(task_queue[queue] + index);    /* Get task from queue. */
/*2*/ (*tq_index[queue])--;                    /* Update the queue index. */
      Release(tq_lock[queue]);                 /* Now it's OK to unlock queue. */

      curdir = ptask->direction;               /* Fill in the globals at leisure, i.e., */
      R0     = ptask->routine;                 /* we're out of the critical section.    */
      R9.Ptr = ptask->lpart.Ptr;
      R8.Ptr = ptask->rpart.Ptr;
      curwme = R8.WmePtr;  /* At times, just a bogus assign. */


#ifdef  SYS_DEBUG
if (show_debug) printf("PopTQN: before calling node task\n");
#endif 

      ret_code = (*R0)();  /* Invoke the node task. */

#ifdef  SYS_DEBUG
if (show_debug) printf("PopTQN: after calling node task\n");
#endif 

#ifdef  SYS_DEBUG
#ifndef UNIPROC_VERSION
if (show_debug) show_counts();
#endif 
#endif 

      /* Find out what kind of node task we just executed and perform
       * required cleanup.
       * NOTE: This would be a good place to do some task profiling.
       */
      switch (ret_code)
        {
	 case LEFT_LEAVE_BETA_TASK:
            /* Finally the left beta node task releases read access
	     * on the right bucket of the token memory hash table.
	     */
	    Release_RightRead(*hkey);
	    /* Release(tokHTlock(hkey));  * Old locking scheme. */
	    break;

	 case RIGHT_LEAVE_BETA_TASK:
            /* Finally the right beta node task releases read access
	     * on the left bucket of the token memory hash table.
	     */
	    Release_LeftRead(*hkey);
	    /* Release(tokHTlock(hkey));  * Old locking scheme. */
	    break;

	 case LEAVE_ROOT_TASK:
            /* nothing special needed */
	    break;

	 case LEAVE_PNODE_TASK:
            /* nothing special needed */
	    break;

	 default:
            ops_fatal("Bogus return code from node task.");
	    break;
	}

      free_tq_cell(ptask);

      return(TRUE);    /* Return to "ops_MatcherCode" to look for another task. */
     }
}

