
/****************************************************************************
 *
 * MODULE:  gensymbol.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Routines to create, maintain, and access the run-time symbol table.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *                    Allow user to specify the size for the symbol table
 *                    with -y cmd line switch.
 *    14 Feb 89 V4.1  Dirk Kalp
 *                    Init new fields added to definition of a symbol table
 *                    record for support of standard OPS5 printing of wmes.
 *                    Fixed bug in "make_string" routine.
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *     1 Oct 88 V3.1  Dirk Kalp
 *                    Init new fields added to definition of a symbol table
 *                    record for support of OPS5 top level "matches" and "call".
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf. Fix bug in "make_string"
 *                    Add routine ops_send_att_bindings.
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    28 Aug 86       Dirk Kalp
 *    22 Aug 86       Dirk Kalp
 *    31 Jul 86
 *    27 Jul 86
 *    27 Jul 86
 *    23 Jul 86
 *    22 Jul 86 V1.0  Dirk Kalp
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include <stdio.h>
#include "global.h"



/* Exported routines:
 *    void    ops_init_symbols()
 *    symptr  ops_symid_lookup(SymbolId)
 *    symptr  ops_symname_lookup(symname)
 *    symptr  ops_new_symbol(str)
 *    OpsVal  ops_intern(str)
 *    void    ops_syminstall(OpsNames, OpsSymbols)
 *    void    ops_bind_names(opsboundnames, opsbindings)
 *    void    ops_send_att_bindings(fp_x)
 *    int     ops_get_litval(val)
 *    int     ops_sym_binding(SymbolId)
 *    OpsVal  ops_make_symbol()
 *    string  ops_pname(atom)
 *
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 *       ops_realloc
 *       ops_fatal
 *       ops_length
 *       ops_equiv_strings
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char    *ops_malloc();          /* Imported from utility.c. */
extern char    *ops_realloc();         /* Imported from utility.c. */
extern void     ops_fatal();           /* Imported from utility.c. */
extern int      ops_length();          /* Imported from utility.c. */
extern boolean  ops_equiv_strings();   /* Imported from utility.c. */



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
string make_string();


   

int  Default_MaxSymbols = 4096;   /* Initial size to allocate for SymbolTable. */

#define SYMTAB_INCREMENT    1024  /* Increment when symbol table must grow. */

#define MINSYMIDNUMBER      0     /* The smallest valid symbol ID. */

#define INVALIDSYMID       -1     /* An invalid symbol ID. */

static symptr  *SymbolTable;      /* Run-time symbol table. It's indexed */
                                  /* by SymId. The size of the array is  */
				  /* allocated at run-time.              */

static int     SizeSymbolTable;   /* Tells size of array above. */

static Symbol  NextSymIdNumber;   /* The next SymId to be assigned. */



#define SIZE_HASHTABLE      1009

static symptr  *HashedSymbolTable;  /* Provides access to symbols */
                                    /* via their string names.    */


/* old: static int **OpsBoundNames; */        /* Used to get access to attribute bindings. */


/* Globals in the .c file generated by the OPS5 compiler, "cops5".
 */
extern int     OpsBoundNames[];
extern int     NumBindings;
extern int     OpsBindings[];
extern char   *OpsNames[];
extern int     SymCount;
extern OpsVal  OpsSymbols[];



void
ops_init_symbols()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Init the run-time symbol table. Called once at system start up.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "ops_rt_init" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   int i;

   SizeSymbolTable = Default_MaxSymbols + MINSYMIDNUMBER;
   SymbolTable  = (symptr *) ops_malloc(SizeSymbolTable * sizeof(symptr));
   NextSymIdNumber = MINSYMIDNUMBER;
   
   HashedSymbolTable = (symptr *) ops_malloc(SIZE_HASHTABLE * sizeof(symptr));
   for (i = 0; i < SIZE_HASHTABLE; i++)   HashedSymbolTable[i] = NULL;
   for (i = 0; i < SizeSymbolTable; i++)  SymbolTable[i] = NULL;
}






static
symptr
new_symrec()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate storage for a new symbol table record and initialize it.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new symbol table record..
 *
 * Calls:
 *    "ops_malloc" in "utility.c".
 *
 * Called by:
 *    "ops_syminstall" and "ops_make_symbol" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   symptr new;
   
   new = (symrec *) ops_malloc(sizeof(symrec));

   new->AttPtrList = NULL;             /* NULL means it's not a production name. */
   new->is_vecatt  = FALSE;            /* FALSE means it's not a vector attribute. */
   new->Func_addr  = NULL;             /* NULL means it's not an external function name. */
   new->NodeidList = NULL;             /* NULL means it's not a production name. */
   new->OpsBind    = INVALIDBINDING;   /* i.e., no binding. */
   new->SymId      = INVALIDSYMID;
   new->SymName    = NULL;
   new->next       = NULL;
   
   return(new);
}



static
int
comp_hash(name)
   string name;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Calculate the hash key for a string name. The formula for the key is:
 *        key = (sum of chars in name) mod HASHTABLESIZE.
 *
 * Parameters:
 *    name - the string name.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The key into the hash table for the name.
 *    (i.e., 0 <= key <= HASHTABLESIZE-1)
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "lookup_sym_addr" and "sym_addr" in this module.
 *
 *-------------------------------------------------------------------------*/
{
  int key;

   key = 0;
   while (*name != '\0')
     {
      key = key + *name;
      name++;
     }
         
   key = key % SIZE_HASHTABLE;
   
   return(key);
}



static
add_symbol(psym)
   symptr psym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add a symbol to the run-time symbol table.
 *
 * Parameters:
 *    psym - pointer to the symbol to be added.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "comp_hash" in this module.
 *
 * Called by:
 *    "ops_syminstall", "intern", and "ops_make_symbol" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   int key;

   /* First see if SymbolTable has any room left.
    */
   if (psym->SymId >= SizeSymbolTable)
     {
      fprintf(fp_dbug, "ops_add_symbol: symid is %d\n", psym->SymId);  fflush(fp_dbug);     
      SizeSymbolTable += SYMTAB_INCREMENT;
      SymbolTable = (symptr *) ops_realloc(SymbolTable, SizeSymbolTable * sizeof(symptr));
     }

   /* Put new entry in symbol table.
    */
   SymbolTable[psym->SymId] = psym;

   /* Now install symbol in the HashedSymbolTable. Put new one at
    * head of its hash bucket.  NOTE: The sequence of the three stmts
    * below is critical in the multiprocessor version, where the RHS
    * evaluation process is adding symbols to the symbol table in shared
    * memory while match is going on.
    */
   key = comp_hash(psym->SymName);
   psym->next = HashedSymbolTable[key];
   HashedSymbolTable[key] = psym;
}



symptr
ops_symid_lookup(SymbolId)
   Symbol SymbolId;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Find the entry for a symbol in the run-time symbol table. 
 *
 * Parameters:
 *    SymbolId - the unique ID number assigned to a symbol.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the symbol table entry if the symbol is entered in the
 *    SymbolTable; Null if symbol has not been entered yet.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "ops_bind_names" and "pname" in this module.
 *    many other callers.
 *
 *-------------------------------------------------------------------------*/
{

   if (SymbolId > SizeSymbolTable)
      return(NULL);
   else
      return(SymbolTable[SymbolId]);
}



symptr
ops_symname_lookup(symname)
   string symname;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Find the entry for a symbol in the run-time symbol table. 
 *
 * Parameters:
 *    symname - the symbol name to find.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the symbol table entry if the symbol is entered in the
 *    SymbolTable; Null if symbol has not been entered yet.
 *
 * Calls:
 *    "ops_equiv_strings" in "utility.c", "comp_hash" in this module..
 *
 * Called by:
 *    Many callers.
 *
 *-------------------------------------------------------------------------*/
{
   int    key;
   symptr psym;

   key = comp_hash(symname);   /* Compute the key into the hash table. */

   if (HashedSymbolTable[key] == NULL)
     {
      /* No symbol table records at this bucket so name not entered.
       */
      return(NULL);
     }
   else
     {
      /* Hash table bucket is not empty so see if our name is
       * linked on the list at this bucket.
       */
      psym = HashedSymbolTable[key];
      while (psym)
        {
         if (ops_equiv_strings(symname, psym->SymName))
            return(psym);   /* Return with the address of the entry. */
         else
            psym = psym->next;
        }
       
      /* Looked through the whole list at this hash bucket and didn't find
       * the name so just return NULL.
       */
      return(NULL);
     }
}


symptr
ops_new_symbol(str)
   string str;
{
   symptr psym;

   psym = new_symrec();
   psym->SymName = str;
   psym->SymId   = NextSymIdNumber++;    /* Bump the next ID. */
   add_symbol(psym);

   return(psym);
}

OpsVal
ops_intern(str)
   string str;
{
   symptr psym;

   psym = ops_symname_lookup(str);

   if (psym == NULL)  psym = ops_new_symbol(str);

   return(sym2val(psym->SymId));
}





static int  *ops_sym;
static int  ops_sym_cnt;

static
void
show_ops_symbols()
{
   int i;

   fprintf(fp_dbug, "*** OPS_SYMBOLS ARRAY of SIZE %d\n", SymCount);
   for (i=0; i<SymCount; i++)  fprintf(fp_dbug, "%d ", val2sym(OpsSymbols[i]));
   fprintf(fp_dbug, "\n");
}

static
void
show_symtab()
{
   int i;

   fprintf(fp_dbug, "*** SYMBOL TABLE    NextSymIdNumber=%d\n", NextSymIdNumber);
   for (i = 0; i < NextSymIdNumber; i++)
     {
      fprintf(fp_dbug, "ID=%d  Bind=%d  %s \n", SymbolTable[i]->SymId,
             SymbolTable[i]->OpsBind, SymbolTable[i]->SymName);
     }
}


void
ops_syminstall()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine initializes the 'ops_symbols' table before the production
 *    system begins to run. Each word entry in 'ops_symbols' is associated
 *    with one of the ASCIIZ stored strings in 'ops_names'. The value stored
 *    in the entry is the unique symbol ID number assigned to the symbol. We
 *    assign that value here. The assigned value is stored left-shifted 1 bit
 *    with the low bit set. This same representation is used when an attribute
 *    field of a working memory element has a value which is a string name
 *    instead of an integer. An integer is stored in a wme field in the same
 *    way except that the low bit is clear.
 *
 *    Also, this routine enters each symbol in the run-time symbol table. We
 *    know that the symbol names stored at 'ops_names' are unique among
 *    themselves since the compiler has ensured this.
 *
 * Note:
 *    Some of the names in 'ops_names' may already be in the symbol table if
 *    they happen to coincide with any of the internal names entered by the
 *    initialization code at startup.
 *
 * Parameters:
 *    OpsNames   - pointer to the 'ops_names' area. This area is part of the
 *                 data segment produced as a result of the compilation of the
 *                 productions. Contained in this area are the ASCIIZ
 *                 representations of the string names used in a given set of
 *                 productions.
 *    OpsSymbols - pointer to 'ops_symbols' table.
 *
 * Environment:
 *    The symbol table has entries of the standard internal names stored by
 *    the startup routine 'ops_rt_init' in "rhsrtn.c".
 *
 * Calls:
 *    "ops_intern", "new_symrec" and "add_symbol" in this module.
 *
 * Called by:
 *    the Rete match assembly code.
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   int symcount;

   for (symcount = 0; symcount < SymCount; symcount++)
     {
      /* Allocate a new symbol table record, fill it in, and put it in
       * the symbol table.
       */
      psym = new_symrec();
      psym->SymName = OpsNames[symcount];
      psym->SymId   = NextSymIdNumber++;   /* Also bump the next ID. */
      add_symbol(psym);

      /* Store the OpsVal representation of the symbol ID into the
       * 'OpsSymbols' table.
       */
      OpsSymbols[symcount] = sym2val(psym->SymId);
     }

   /* Define internal symbols */
   symcrlf   = ops_intern("| C R L F |");
   symrjust  = ops_intern("| R J U S T |");
   symtabto  = ops_intern("| T A B T O |");
   symnil    = ops_intern("nil");
   syminf    = ops_intern("inf");
   symeof    = ops_intern("end-of-file");
   symin     = ops_intern("in");
   symout    = ops_intern("out");
   symaccept = ops_intern("accept");
   symwrite  = ops_intern("write");
   symtrace  = ops_intern("trace");
}



void
ops_bind_names()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Obtain the bindings for attribute names and record them with their
 *    corresponding entries in the symbol table.
 *
 * Parameters:
 *    opsboundnames - pointer to the 'ops_bound_names' table. An entry in
 *                    this table points to the entry in 'ops_symbols' that
 *                    is associated with the attribute name.
 *    opsbindings   - pointer to 'opsbindings' table. An entry here is the
 *                    binding assigned by the compiler to the associated
 *                    attribute name.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "ops_symid_lookup" in this module.
 *
 * Called by:
 *    the Rete match assembly code.
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   int    i;

   for (i = 0; i < NumBindings; i++)
     {
      psym = ops_symid_lookup(val2sym(OpsSymbols[OpsBoundNames[i]]));
      if (psym == NULL)
        {
         fprintf(fp_err, "ops_bind_names: symid %d invalid    OpsBoundNames Index = %d\n",
	                 val2sym(OpsSymbols[OpsBoundNames[i]]), i);
	 fflush(fp_err);     
        }      
      
      psym->OpsBind = OpsBindings[i];
     }
}



void
ops_send_att_bindings(fp_x)
   FILE *fp_x;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Send attribute bindings to a file.
 *
 * Parameters:
 *    fp_x - the file to write to.
 *
 * Environment:
 *    OpsBoundNames is a global pointing to the table built by the compiler.
 *    Caller has opened file for writing.
 *
 * Calls:
 *    "ops_symid_lookup" in this module.
 *
 * Called by:
 *    
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   int    *opsboundnames;

   opsboundnames = OpsBoundNames;   
   while (*opsboundnames > 0)
     {
      psym = ops_symid_lookup(val2sym(*opsboundnames));
      fprintf(fp_x, "%s %d ", psym->SymName, psym->OpsBind);
      opsboundnames++;
     }
}




int
ops_get_litval(val)
   OpsVal val;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Get the literal value of an OpsVal.
 *
 * Parameters:
 *    val - an OpsVal which represents either an integer or Symbol.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    If it's a Symbol, then return the binding if the Symbol is an
 *    attribute and 0 if not. If it's an integer, return the integer value.
 *
 * Calls:
 *    "ops_sym_binding" in this module.
 *
 * Called by:
 *    "ops_litval" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   if (symbolp(val))
      return(ops_sym_binding(val2sym(val)));
   else
      return(val2int(val));
}



int
ops_sym_binding(SymbolId)
   Symbol SymbolId;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Get the binding of a symbol.
 *
 * Parameters:
 *    SymbolId - the ID representing the symbol.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The binding if the Symbol is an attribute and 0 if not.
 *
 * Calls:
 *    "ops_fatal" in "utility.c".
 *    "ops_symid_lookup" in this module.
 *
 * Called by:
 *    "ops_get_litval" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   
   psym = ops_symid_lookup(SymbolId);

   if (psym == NULL)
      ops_fatal("ops_sym_binding: symbol lookup failed on SymId.");

   if (psym->OpsBind == INVALIDBINDING)
     {
      fprintf(fp_err, "Ops_sym_binding: Symbol %s is not an attribute name\n",
             psym->SymName);
                   
      return(0);
     }
   else
      return(psym->OpsBind);
}




OpsVal
ops_make_symbol()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Create a new symbol and add it to the symbol table..
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The Symbol represented as an OpsVal.
 *
 * Calls:
 *    "new_symrec", "make_string" and "add_symbol" in this module.
 *
 * Called by:
 *    "ops_genatom" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   
   psym = new_symrec();
   
   psym->SymId   = NextSymIdNumber;
   psym->SymName = make_string("$$OPS_", NextSymIdNumber);
   
   add_symbol(psym);
   
   return(sym2val(NextSymIdNumber++));    /* Also bump the next ID. */
}



static
string
make_string(prefix, cnt)
   string prefix;
   int    cnt;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    X.
 *
 * Parameters:
 *    prefix - a string prefix for a name.
 *    cnt    - and integer to append to the prefix.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "ops_length" and "ops_malloc" in "utility.c".
 *
 * Called by:
 *    "make_symbol" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   string str;
   int    n, len, plen;
   
   plen = ops_length(prefix);
   
   n = cnt;
   len = 1;
   while (n = n/10)  len++;
   len = len + plen + 1;
   
   str = (string) ops_malloc(len);
   
   for (n=0; n<plen; n++)  str[n] = prefix[n];
   
   for (n=len-2; n>=plen; n--)
     {
      str[n] = '0' + (cnt % 10);
      cnt = cnt / 10;
     }
   str[len-1] = 0;   /* NULL byte. */
   
   return(str);
}




string
ops_pname(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Get the string name for a Symbol.
 *
 * Parameters:
 *    atom - a symbol ID represented as an OpsVal.
 *
 * Environment:
 *    Symbol names known at compile time are stored in the 'ops_names'
 *    area allocated in the data segment of the Rete assembly code.
 *
 * Returns:
 *    A pointer to the string name.
 *
 * Calls:
 *    "ops_symid_lookup" in this module.
 *
 * Called by:
 *    "ops_dorjust" and "wratom" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   
   psym = ops_symid_lookup(val2sym(atom));

   return(psym->SymName);
}

