/****************************************************************************
 *
 * MODULE:  conres.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    This module maintains the conflict set and performs conflict
 *    resolution.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed the lock declarations and the lock 
 *                    initializations within conditional compilation
 *                    blocks (#ifndef UNIPROC_VERSION)
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Use new lock macros from psm_locks.h.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *    14 Feb 89 V4.1  Dirk Kalp
 *                    Add use of ConflictSet_Tail to get CS printed in
 *                    increasing order of arrival (i.e., reverse of way
 *                    it's printed out now for top-level "cs" cmd).
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    23 Oct 88 V3.2  Dirk Kalp
 *                    Modified form of printout in "ops_dumpallcs", "dumpcs",
 *                    and "ops_print_cs" routines.
 *    24 Sep 88 V3.1  Dirk Kalp
 *                    Added parameter to "ops_resolve" so top level inquiry
 *                    from user interface can be handled.
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf. Added global free list
 *                    for cs_cells so memory can be reclaimed upon system
 *                    reinitialization. Added routine "ops_reinit_conflictset".
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *     3 Aug 86	      Anoop Gupta
 *    31 Jul 86
 *    23 Jul 86 V1.0  Dirk Kalp
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include "global.h"



/* Exported routines:
 *    void     ops_init_conflict_set()
 *    void     ops_reinit_conflict_set()
 *    void     ops_modifycs(pnameID, cecount, spcount, rhsaddr, token, dir)
 *    void     ops_resolve(refract)
 *    void     ops_dumpallcs(fp_x)
 *    boolean  ops_check_csempty()
 *    void     ops_print_cs(fp_x, psym)
 *
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 *    From wmemory.c:
 *       ops_get_timetag
 *       ops_write_wme
 *    From gensymbol.c:
 *       ops_symid_lookup
 *    From psm_locks.c:
 *       ops_alloc_and_init_lock
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char   *ops_malloc();        /* Imported from utility.c. */
extern symptr  ops_symid_lookup();  /* Imported from gensymbol.c. */
extern void   *ops_write_wme();     /* Imported from wmemory.c. */

#ifndef UNIPROC_VERSION
extern LockPtr ops_alloc_and_init_lock();   /* Imported from psm_locks.c. */
#endif

   

typedef struct cs_struct
        {
        OpsVal           ruleid;            /* OpsVal corresponding to rule name. */
        int              cecount;           /* Number of condition elements. */
        int              poscecount;        /* Number of positive cond elems. */
        int              spcount;           /* Specificity of production. */
        int              rhs;               /* Address (just an index) of the threaded code. */
        wmeptrvec        wmes;              /* Vector of pointers to wmes. */
        int              ttags[MAXCECOUNT]; /* Time tags of wmes. */
	int		 firecount;	    /* number of times rule has fired */
        struct cs_struct *flink;
        struct cs_struct *blink;
        }
        cs_cell, *ptr_cs_cell;



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
ptr_cs_cell new_cs_cell();
ptr_cs_cell find_cs_cell();




#ifndef UNIPROC_VERSION
static LockPtr      CS_Lock;             /* Lock for modifying conflict set */
#endif

static ptr_cs_cell *ConflictSet;         /* Header for the conflict set.    */
static ptr_cs_cell *ConflictSet_Tail;    /* Tail of the conflict set. */
static ptr_cs_cell *CSXDeletesList;      /* Header for conflict-set extra deletes list */

static ptr_cs_cell  CSFreeList;          /* Header for the local free list.       */

static ptr_cs_cell *Global_CSFreeList;   /* Header (in shared memory) for global free list. */

#ifndef UNIPROC_VERSION
static LockPtr      Global_CSFree_Lock;  /* Lock for global free list. */
#endif




void
ops_init_conflict_set()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Init the conflict set. Called only once at system start up.
 *    Should be called only by process P0 before forking off other match processes.
 *
 * Environment:
 *    Nothing special.
 *
 * Called by:
 *    "ops_rt_init" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
    ConflictSet             = (ptr_cs_cell *) ops_malloc(PointerSize);
    ConflictSet_Tail        = (ptr_cs_cell *) ops_malloc(PointerSize);
    CSXDeletesList          = (ptr_cs_cell *) ops_malloc(PointerSize);
    CR_Strategy             = (int *)         ops_malloc(PointerSize);

#ifndef UNIPROC_VERSION
    CS_Lock                 = ops_alloc_and_init_lock();
    Global_CSFree_Lock      = ops_alloc_and_init_lock();
#endif

    Global_CSFreeList       = (ptr_cs_cell *) ops_malloc(PointerSize);

   *ConflictSet             = NULL;
   *ConflictSet_Tail        = NULL;
   *CSXDeletesList          = NULL;
    CSFreeList              = NULL;
   *CR_Strategy             = MEA;
   *Global_CSFreeList       = NULL;
}


void
ops_reinit_conflict_set()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize the Conflict Set after a system reinit.
 *
 * Environment:
 *    Nothing special.
 *
 * Called by:
 *    "ops_reinit" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   ptr_cs_cell ptr, lastptr;

   if (*ConflictSet)
     {
/*      ptr = *ConflictSet; */
/*      while (ptr)  { lastptr = ptr;  ptr = ptr->flink; } */
/*      lastptr->flink = *Global_CSFreeList; */
      (*ConflictSet_Tail)->flink = *Global_CSFreeList;
      *Global_CSFreeList = *ConflictSet;
      *ConflictSet = NULL;
      *ConflictSet_Tail = NULL;
     }

   if (*CSXDeletesList)
     {
      ptr = *CSXDeletesList;
      while (ptr)  { lastptr = ptr;  ptr = ptr->flink; }
      lastptr->flink = *Global_CSFreeList;
      *Global_CSFreeList = *CSXDeletesList;
      *CSXDeletesList = NULL;
     }
}



static
ptr_cs_cell
alloc_cell()
{
   ptr_cs_cell cell;

   if (CSFreeList)
     {
      cell = CSFreeList;
      CSFreeList = cell->flink;
      return(cell);
     }

   if (*Global_CSFreeList)
     {
      Test_Then_Lock(Global_CSFree_Lock);
      if (*Global_CSFreeList)
        {   
         cell = *Global_CSFreeList;
         *Global_CSFreeList = cell->flink;
         Release(Global_CSFree_Lock);
	 return(cell);
	}
      Release(Global_CSFree_Lock);
     }


   /* Allocate a new one if none on any free list. */
   cell = (ptr_cs_cell) ops_malloc(sizeof(cs_cell));
   return(cell);
}



void
ops_modifycs(pnameID,
         cecount,
         spcount,
         rhsaddr,
         token,
         dir
        )
   OpsVal    pnameID;
   int       cecount;
   int       spcount;
   int       rhsaddr;   /* Changed from char *rhsaddr for CParaOPS5. */
   wmeptrvec token;
   int       dir;
{
   ptr_cs_cell cell, dcell;
   int i, cnt;


   if (dir == DIRIN)
     {

      cell = new_cs_cell(pnameID,
                         cecount,
                         spcount,
                         rhsaddr,
                         token,
                         dir
                        );
      

      Test_Then_Lock(CS_Lock);

      if (*CSXDeletesList != NULL)
        {
         dcell = find_cs_cell(*CSXDeletesList, pnameID, token, cecount);
	 if (dcell != NULL)
	   {
	    excise(CSXDeletesList, dcell);
            Release(CS_Lock);
	    /* Free the cell that we had allocated */
   	    cell->flink = CSFreeList;
	    CSFreeList = cell;
	    return;
	   }
	}

      /* Add it to the conflict set. */
      cell->flink = *ConflictSet;
      cell->blink = NULL;
      if (*ConflictSet)  (*ConflictSet)->blink = cell;
      *ConflictSet = cell;
      if (*ConflictSet_Tail == NULL)  *ConflictSet_Tail = cell;
     }
   else  /* DIROUT */
     {
      Test_Then_Lock(CS_Lock);

      if (*ConflictSet != NULL)
        {
         cell = find_cs_cell(*ConflictSet, pnameID, token, cecount);
	 if (cell != NULL)
	   {
	    excise(ConflictSet, cell);  /* NOTE: excise does not destroy cell->blink. */
	    if (*ConflictSet_Tail == cell)  *ConflictSet_Tail = cell->blink;
            Release(CS_Lock);
	    return;
	   }
	}

      cell = new_cs_cell(pnameID,
                         cecount,
                         spcount,
                         rhsaddr,
                         token,
                         dir
                        );
      
      /* Add it to the deletes list. */
      cell->flink = *CSXDeletesList;
      cell->blink = NULL;
      if (*CSXDeletesList)  (*CSXDeletesList)->blink = cell;
      *CSXDeletesList = cell;
/*
fprintf(fp_dbug, "ConflRes: An extra delete encountered. RuleID = %d \n", pnameID);
dumpcs(fp_dbug, cell);
fflush(fp_dbug);
*/
     }

   Release(CS_Lock);
}



static
ptr_cs_cell
new_cs_cell(pnameID,
            cecount,
            spcount,
            rhsaddr,
            token,
            dir
           )
   OpsVal    pnameID;
   int       cecount;
   int       spcount;
   int       rhsaddr;   /* Changed from char *rhsaddr for CParaOPS5. */
   wmeptrvec token;
   int       dir;
{
   ptr_cs_cell cell;
   int i, cnt;
   
   cell = alloc_cell(); /* Get a conflict set cell. */
   cell->ruleid    = pnameID;
   cell->cecount   = cecount;
   cell->spcount   = spcount;
   cell->rhs       = rhsaddr;
   cell->firecount = 0;
   cnt = 0;
   for (i = 0; i < cecount; i++)
     {
      cell->wmes[i] = token[i];
      if (token[i] != NULL)  cnt++;
      if (dir == DIRIN) cell->ttags[i] = ops_get_timetag(token[i]);
     }
   cell->poscecount = cnt;
   if (dir == DIRIN) sort_ttags(cell->ttags, cecount, *CR_Strategy);
   return(cell);
}


static
sort_ttags(ttags, cnt, strategy)
   int ttags[MAXCECOUNT];
   int cnt;
   int strategy;
{
   register int *pi, *pj;
   int i, j;
   int start;
   int swap;
   
   if (strategy == LEX)
      start = 0;
   else
      start = 1;
   
   for (i = start, pi = ttags+start; i <= (cnt-2); i++, pi++)
      for (j = (i+1), pj = ttags+i+1; j <= (cnt-1); j++, pj++)
         if (*pi < *pj)
           {
            swap = *pi;
            *pi = *pj;
            *pj = swap;
           }
}



static
ptr_cs_cell
find_cs_cell(cslist, pnameID, token, cecount)
   ptr_cs_cell cslist;
   OpsVal      pnameID;
   wmeptrvec   token;
   int         cecount;
{
   ptr_cs_cell cell;
   int         i;
   
   cell = cslist;
   while (cell)
     {
      if (cell->ruleid == pnameID)
        {
         i = 0;
         while ((i < cecount) && (cell->wmes[i] == token[i]))  i++;
         if (i == cecount)  return(cell);
        }
      cell = cell->flink;
     }
   
  return(NULL);

}



static
excise(cslistptr, cell)
   ptr_cs_cell *cslistptr;
   ptr_cs_cell cell;
{
   ptr_cs_cell pred, succ;
   
   pred = cell->blink;
   succ = cell->flink;
   
   if (pred)
      pred->flink = succ;
   else
      *cslistptr = succ;
   if (succ)  succ->blink = pred;
   
   cell->flink = CSFreeList;
   CSFreeList  = cell;
}



/***
 ***  Ops_resolve selects the dominant instantiation in the conflict
 ***  set.  The selected instantiation is removed from the conflict
 ***  set.  It sets the global var flgpfire to the selected inst if 
 ***  it could get an instantiation; NULL if the conflict set is empty.
 ***
 ***  8/3/86:  axg:  Changed so that the best instantiation is not deleted
 ***  from the conflict set.  It will be deleted when one of the constituent
 ***  wmes is deleted.  Howwever, we mark its firecount field to 1, so that
 ***  it will not participate in normal resolve process.
 ***
 ***  9/24/88: Dirk: Added parameter to ops_resolve to indicate if we are just
 ***  doing a conflict resolution inquiry on behalf of the top level user
 ***  interface cmd, resolve. In this case we are only trying to find out
 ***  which instantiation currently dominates and are not going to fire and
 ***  refract it.
 ***/
void
ops_resolve(refract)
   boolean refract;  /* TRUE -> perform CR and refract, FALSE -> just inquire. */
{
   ptr_cs_cell best;           /* The dominant element to those seen. */
   ptr_cs_cell try;            /* The next element to examine. */
   int         b_wmecnt;       /* How many wmes best contains. */
   int         t_wmecnt;       /* How many wems try contains. */
   int         tag_limit;      /* How many time tags to examine. */
   int         b_ttag;         /* A time tag from best. */
   int         t_ttag;         /* A time tag from try. */
   boolean     found_dominant;
   int         i;
   
/*
 * Note that we do not need to lock the conflict-set while executing
 * this procedure because only process P0 will be accessing the
 * conflict-set at this time.
 */


   /* Scan the set to find the first non refracted instantiation. */
   best = *ConflictSet;
   while ((best != NULL) && (best->firecount > 0)) best = best->flink;

   if (best == NULL)
     {
      if (refract)
        {
         flgpfire = NULL;
         fprintf(fp_dbug, "\n$$$$$$$$$$ NO PRODUCTIONS LEFT TO FIRE IN CONFLICT SET $$$$$$$$$$\n");
	}
      else
        {
         /* Just doing CR for top level resolve inquiry, so print out the msg.
          */
         fprintf(fp_dbug, "\n$$$$$$$$$$ NO PRODUCTIONS TO FIRE IN CONFLICT SET $$$$$$$$$$\n");
	}
      return;
     }

   /* Scan the set to find the best instantiation. */
   for (try = best->flink; try != NULL; try = try->flink)
     {
      if (try->firecount > 0) continue;   /* try is a refracted instantiaion */
	 
      b_wmecnt = best->poscecount;
      t_wmecnt = try->poscecount;
      
      tag_limit = ((b_wmecnt < t_wmecnt) ? b_wmecnt : t_wmecnt);
      found_dominant = FALSE;
      for (i = 0; i < tag_limit; i++)
        {
         b_ttag = best->ttags[i];
         t_ttag = try->ttags[i];
         if (b_ttag != t_ttag)
           {
            found_dominant = TRUE;
            if (b_ttag < t_ttag)  best = try;
            break;
           }
        }
      
      if (!found_dominant)
        {
         if (b_wmecnt != t_wmecnt)
           {
            if (b_wmecnt < t_wmecnt)  best = try;
           }
         else
           {
            if (best->spcount < try->spcount)  best = try;
           }
        }
     }
   
   if (refract)
     {   
      /* Put the chosen instantiation into the global variables.
       */
      flgpfire         = best->rhs;
      flg_len_instance = best->cecount;
      flg_instance     = (wmeptrvec *) best->wmes;
      flg_ruleid       = best->ruleid;
   
      best->firecount  = 1;      /* mark the instantiation refracted */
     }
   else
     {
      /* Just doing CR for top level resolve inquiry, so print out
       * the instantiation.
       */
      dumpcs(fp_out, best);
     }

}




static
dumpcs(fp_x, cs)
   FILE        *fp_x;
   ptr_cs_cell cs;
{
   int       i;
   wmeptr   *wm;
   symptr    psym;

   psym  = ops_symid_lookup(val2sym(cs->ruleid));
   if (cs->firecount)  fprintf(fp_x, "**REFRACTED** ");
   fprintf(fp_x, "%s:   NumberCEs: %d  Specificity: %d", psym->SymName,
           cs->cecount, cs->spcount);

   if (*CR_Strategy == MEA)
      fprintf(fp_x, "   MEA ordered WMEs:");
   else
      fprintf(fp_x, "   LEX Ordered WMEs:");
   for (i = 0; i < cs->poscecount; i++)  fprintf(fp_x, " %d", cs->ttags[i]);
   fprintf(fp_x, " \n");

   for (i = 0; i < cs->cecount; i++)
     {
      fprintf(fp_x, "    CE_%d:  ", i+1);
      if (cs->wmes[i])
        {
         ops_write_wme(fp_x, cs->wmes[i]);
        }
      else
	 fprintf(fp_x, "nil\n");
     }
}



void
ops_dumpallcs(fp_x)
   FILE *fp_x;
{
   ptr_cs_cell ptr;

   fprintf(fp_x, "\n**** Dump Conflict Set (refracted instantiations marked): ****\n\n");

   ptr = *ConflictSet_Tail;   /* 2/14/89: Changed from *ConflictSet to get reverse printing of CS. */
   while(ptr)
     {
      dumpcs(fp_x, ptr);
      fprintf(fp_x, "\n");
      ptr = ptr->blink;       /* 2/14/89: Changed from flink to get reverse printing of CS. */
     }
     
   fprintf(fp_x, "**** End of Conflict Set ****\n");
   fflush(fp_x);
}



boolean
ops_check_csempty()
   /* This function checks if Conflict set is empty. */
{
   if (*ConflictSet == NULL)  return(TRUE);  else  return(FALSE);	
}



void
ops_print_cs(fp_x, psym)
   FILE   *fp_x;
   symptr psym;
{
   register ptr_cs_cell cs;
   register OpsVal      pnameid;
   register int         i;

   pnameid = sym2val(psym->SymId);
   cs = *ConflictSet_Tail;   /* 2/14/89: Changed from *ConflictSet to get reverse printing of CS. */
   while (cs)
     {
      if (cs->ruleid == pnameid)
        {
         fprintf(fp_x, "   (");
         for (i = 0; i < cs->cecount; i++)
           {
            if (i > 0)  fprintf(fp_x, " ");
            if (cs->wmes[i])
               fprintf(fp_x, "%d", cs->wmes[i]->wme[WMETIMETAG]);
            else
	       fprintf(fp_x, "nil", i);
	   }
	  if (cs->firecount)  fprintf(fp_x, ")   REFRACTED\n");  else  fprintf(fp_x, ")\n");
	}
      cs = cs->blink;       /* 2/14/89: Changed from flink to get reverse printing of CS. */
     }
}

