
/****************************************************************************
 *
 * MODULE:  literal.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Process the declarations in an OPS5 program.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    12 May 89 V2.0 Dirk Kalp
 *                   Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    14 Feb 89 V4.1 Dirk Kalp
 *                   Modified "assign_scalars" to save the list of vector
 *                   attributes needed to add support for standard OPS5 run-time
 *                   printing of wmes.
 *                   Modified "literalize" to call "reverse_att" to put the
 *                   list of attributes, b_attptr, in reverse order from that
 *                   given in the parsing. This is in order to get bindings
 *                   assigned to the attributes in a manner closer to Lisp-
 *                   based OPS5 so that printing of wmes is more uniform
 *                   in the C and Lisp versions. Also modified "add_conflict"
 *                   for same reason. Note however that the binding assignments
 *                   still do not correspond exactly between the 2 versions.
 *    24 Oct 88 V4.0 Dirk Kalp
 *                   Release of ParaOPS5 Version 4.0.
 *     1 Oct 88 V3.1 Dirk Kalp
 *                   Add routine "add_to_ext_list" to support the OPS5 top
 *                   level "call" action.
 *    13 Aug 88 V3.0 Dirk Kalp
 *                   No changes.
 *    25 May 88 V2.0 Dirk Kalp
 *                   Updated to consolidate Vax and Encore versions.
 *    12 Sep 86      Dirk Kalp
 *    10 Sep 86      Dirk Kalp
 *    16 May 86
 *    14 May 86
 *     1 May 86
 *    29 Apr 86
 *    14 Mar 86
 *    13 Mar 86
 *    10 Mar 86
 *     9 Mar 86 V1.0  Dirk Kalp
 *                    Put together from previous version created by
 *                    Quei-Len Lee.
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/

 
   
#include "defs.h"



/* Imported Routines:
 *    From ops5.c:
 *       opserror
 *       opswarn
 *       putstring
 *       puttab
 *       puteol
 *       bomb_out
 *
 *    From gencode.c:
 *       newinst
 *       codeouts
 *       codeouti
 *       codeoutl
 *
 *    From printops.c:
 *       printsym1
 *
 *    From system:
 *       malloc
 */

/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
/* There are none. */



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
string       newname();
symptr       new_symptr();
symptr       lookup_sym_addr();
symptr       sym_addr();
symptr       new_attname();
symptr       test_cname();
cname_ptr    new_cnameptr();
symptr       new_pname();
sym_attptr   new_attlist_rec();
boolean      not_dup_att();
boolean      not_in_confset();





init_decl() 
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize global variables before the parse and compilation of
 *    declarations is begun.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The parser has not yet begun.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "main" in "ops5.c".
 *
 *-------------------------------------------------------------------------*/
{
   int i;

   free_attptr = NULL;
   b_attptr    = NULL;
   e_attptr    = NULL;
   
   b_cname_ptr = NULL;
   e_cname_ptr = NULL;
   
   for (i = 0; i < HASHTABSIZE; i++)
      hashtable[i] = NULL;

   g_last_buc = FIRST_OPSBIND;
   for (i=0; i < BUCKETSIZE; i++)
      bucket[i]= NULL;

   ExternalList = NULL;   /* Init the list used to collect up external names. */
}




int
strlength(str)
   string str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Determine the length of a string. This includes the byte of storage
 *    required for the terminating null character. The empty string thus
 *    has length 1.
 *
 * Parameters:
 *    str - the string.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The length of the string.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "newname" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   int size;

   size = 1;    /* Count the terminating null character. */
   
   while (*str != '\0')  {size++; str++;}
   
   return(size);
}



strcopy(new,str)
   string new, str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Copy a string to a new area of storage.
 *
 * Parameters:
 *    new - the address where the string is to be copied.
 *    str - the address of the string to copy.
 *
 * Environment:
 *    The string to copy is terminated by a null character and the caller
 *    has ensured adequate storage for it beginning at new.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "newname" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   while (*new++ = *str++);
}



int
strcompare(str1, str2)
   string str1, str2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Determine if 2 strings are identical.
 *
 * Parameters:
 *    str1, str2 - the 2 strings to compare..
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    0 if the strings are equal; otherwise a non-zero number.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "lookup_sym_addr", "sym_addr", and "not_dup_att" in this module.
 *    "find_equiv_varname" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   while (*str1 == *str2)
     if (*str1 == '\0')
        return (0); 
     else
       { str1++; str2++;}
       
   return(*str1 - *str2);
}



string
newname(str)
   string str;  
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate storage for a new copy of a string name and copy it there.
 *
 * Parameters:
 *    str - the string name to copy.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new copy of the string.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *    "strcopy" in this module.
 *
 * Called by:
 *    "sym_addr" in this module.
 *    "newnode", "cmp_cevar", and "cmp_new_var" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   string new;

   if ((new = (string)malloc(strlength(str))) == NULL)
     {
      opserror("newname:  no more memory...");  puteol();
      bomb_out();
     }
     
   strcopy(new, str);
   return(new);
}




symptr
new_symptr()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new symbol table record and initialize it.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new symbol table record.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "sym_addr" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   symptr new;

   if ((new = (symtab *)malloc(sizeof(symtab))) == NULL)
     {
      opserror("new_symptr:  no more memory...");  puteol();
      bomb_out();
     }
     
   new->symname    = NULL;
   new->is_vec     = FALSE;
   new->is_pname   = FALSE;
   new->is_cname   = FALSE;
   new->is_att     = FALSE;
   new->is_fname   = FALSE;
   new->valid_bind = FALSE;
   new->opsbind    = 0;
   new->attlist    = NULL;
   new->conflist   = NULL;
   new->offset     = -1;
   new->next       = NULL;

   return(new);
}




int
comp_hash(name)
   string name;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Calculate the hash key for a string name. The formula for the key is:
 *        key = (sum of chars in name) mod HASHTABLESIZE.
 *
 * Parameters:
 *    name - the string name.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The key into the hash table for the name.
 *    (i.e., 0 <= key <= HASHTABLESIZE-1)
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "lookup_sym_addr" and "sym_addr" in this module.
 *
 *-------------------------------------------------------------------------*/
{
  int key;

   key = 0;
   while (*name != '\0')
     {
      key = key + *name;
      name++;
     }
         
   key = key % HASHTABSIZE;
   
   if (debug)  printf("comp_hash: %s %d\n", name, key);
   
   return(key);
}



symptr
lookup_sym_addr(name)
   string name;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine finds the symbol table entry for the string name symbol
 *    or else reports it empty. The symbol table records are stored
 *    in a hash table. Names that hash to the same bucket are stored in a
 *    linked list at that bucket. Duplicate entries in the symbol table for
 *    the same name will not occur since the routine 'sym_addr' that adds
 *    new entries prevents that.
 *
 * Parameters:
 *    name - the string name of a symbol.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The address of the symbol table record for this name or NULL if it
 *    does not exist.
 *
 * Calls:
 *    "comp_hash" and "strcompare" in this module.
 *
 * Called by:
 *    "cmp_tab" in "cmplhs.c".
 *    "gen_atmoffset" in "gencode.c".
 *
 *-------------------------------------------------------------------------*/
{
   int    key;
   symptr ptr;

   key = comp_hash(name);   /* Compute the key into the hash table. */

   if (hashtable[key] == NULL)
     {
      /* No symbol table records at this bucket so name not entered.
       */
      if (debug)  printf("hashtable[%d] = NULL\n", key);
      return(NULL);
     }
   else
     {
      /* Hash table bucket is not empty so see if our name is
       * linked on the list at this bucket.
       */
      ptr = hashtable[key];
      while (ptr)
        {
         if (strcompare(name, ptr->symname) == 0)
           {
            /* Found name in symbol table.
             */
            if (debug)
               printf("symaddr: old symbol key= %d %s\n", key, ptr->symname);
            return(ptr);   /* Return with the address of the entry. */
           }
         ptr = ptr->next;
        }
       
      /* Looked through the whole list at this hash bucket and didn't find
       * the name so just return NULL.
       */
      return(NULL);
     }
}



symptr
sym_addr(name)
   string name;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine finds the symbol table entry for the string name symbol
 *    or else creates an entry for it. The symbol table records are stored
 *    in a hash table. Names that hash to the same bucket are stored in a
 *    linked list at that bucket. Duplicate entries in the symbol table for
 *    the same name will not occur since, here, we always look through the
 *    symbol table records linked at the bucket for a new name to see if the
 *    new name is already there. 
 *
 * Parameters:
 *    name - the string name of a symbol.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The address of the symbol table record for this name.
 *
 * Calls:
 *    "comp_hash", "new_symptr", "newname", and "strcompare" in this module.
 *
 * Called by:
 *    "new_attname", "test_cname", "new_pname", "vec_att", and "extern_dec"
 *    in this module.
 *    "add_anyatm" and "cmp_constant" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   int    key;
   symptr ptr, lastptr;

   key = comp_hash(name);   /* Compute the key into the hash table. */

   if (hashtable[key] == NULL)
     {
      /* No symbol table records at this bucket so create a new one
       * for the name and connect it here.
       */
      hashtable[key] = new_symptr();
      hashtable[key]->symname = newname(name);
      if (debug)  printf("hashtable[%d] = %s\n", key, hashtable[key]->symname);
      return(hashtable[key]);
     }
   else
     {
      /* Hash table bucket is not empty so see if our name is already
       * linked on the list at this bucket.
       */
      ptr=hashtable[key];
      while (ptr)
        {
         if (strcompare(name, ptr->symname) == 0)
           {
            /* Found name already in symbol table.
             */
            if (debug)
               printf("symaddr: old symbol key= %d %s\n", key,ptr->symname);
            return(ptr);   /* Return with the address of the entry. */
           }
         lastptr = ptr;
         ptr = ptr->next;
        }
       
      /* Looked through the whole list at this hash bucket and didn't find
       * the name so just create a new symbol table record for the name and
       * add it to the end of the list. Lastptr from the loop above already
       * points to the end of the list.
       */
      ptr = new_symptr();    /* Get new symbol table record. */
      lastptr->next = ptr;   /* Add it to the end of the list. */
      ptr->symname = newname(name);  /* Copy name out of YACC value stack. */
      return(ptr);
     }
}




symptr
new_attname(att)
   attptr att;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Retrieve or create the symbol table entry for an attribute name.
 *
 * Parameters:
 *    att - a pointer to a record holding the attribute name.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the symbol table record for the attribute name.
 *
 * Calls:
 *    "sym_addr" in this module.
 *
 * Called by:
 *    "literalize" and "literal" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   symptr new;

   new = sym_addr(att->attname);
   new->is_att = TRUE;
   return(new);
}



symptr
test_cname(cname)
   str_arr cname;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine retrieves or creates the symbol table entry for a class
 *    name. If the name already appears in the symbol table, we check to see
 *    if it was entered as a class name in a previous Literalize declaration.
 *    If so, we'll print an error message and just continue to process the
 *    remainder of the declarations looking for other errors.
 *
 * Parameters:
 *    cname - the class name.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "sym_addr" in this module.
 *    "opserror", "puttab", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    "literalize" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   symptr sym;

   /* Get the symbol table entry for the name, "cname". We may have to
    * create a new entry.
    */
   sym = sym_addr(&cname[0]);
    
   /* Now check to see if this name has been previously used as a class
    * name in an earlier Literalize declaration.   
    */
   if (sym->is_cname == TRUE) 
     {
      opserror("class name redefined : ");
      puttab();  putstring(&cname[0]);  puteol();
      /* Just continue to process declarations. */
     }

   return(sym);
}



cname_ptr
new_cnameptr()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate and initialize storage for a new cname_list record.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new cname_list record.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "add_cname_list" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   cname_ptr new;

   if ((new = (cname_list *)malloc(sizeof(cname_list))) == NULL)
     {
      opserror("new_cnameptr:  no more memory...");  puteol();
      bomb_out();
     }
   
   /* Init the record.
    */  
   new->cname   = NULL;
   new->attlist = NULL;
   new->next    = NULL;
   
   return(new);
}



add_cname_list (sym)
  symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add a new class name to the global cname_list. The identity of the
 *    class name is represented in the list by a record holding the address
 *    of the symbol table entry for the class name. The new record is added
 *    to the end of the cname_list.
 *
 * Parameters:
 *    sym - the address of the symbol table entry for the class name.
 *
 * Environment:
 *    The global cname_list is maintained with globals 'b_cname_ptr' and
 *    'e_cname_ptr' which point to the beginning and end of the list.
 *
 * Calls:
 *    "new_cnameptr" in this module.
 *
 * Called by:
 *    "literalize" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   cname_ptr new;

   /* First get a new record to add to the cname_list.
    */
   new = new_cnameptr();
   
   /* Now store the address of the symbol table entry for the new class
    * name into that record.
    */
   new->cname = sym;
   
   /* Now tack the new record to the end of the cname_list.
    */
   if (b_cname_ptr == NULL)
     {
      b_cname_ptr = new;
      e_cname_ptr = new;
     }
   else
     {
      e_cname_ptr->next = new;
      e_cname_ptr = new;
     }
}



symptr
new_pname(pname)
   str_arr pname;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Create a symbol table table entry for the name of a production and
 *    return a pointer to it.
 *
 * Parameters:
 *    pname - the string name of a production (name is stored in YACC value
 *            stack).
 *
 * Environment:
 *    We are compiling productions now instead of declarations.
 *
 * Returns:
 *    A pointer to the symbol table entry for the name.
 *
 * Calls:
 *    "sym_addr" in this module.
 *
 * Called by:
 *    "cmp_production" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   symptr new;
    
   new = sym_addr(pname);
   new->is_pname = TRUE;
   return(new);
}




sym_attptr
new_attlist_rec(sym)
   symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new sym_att record to hold the address of the symbol table
 *    entry for an attribute name.
 *
 * Parameters:
 *    sym - the address of the symbol table entry for the attribute name.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "literalize", "add_bucket", and "add_vec_attlist" in this module.
 *    "new_opssym" in "gencode.c".
 *
 *-------------------------------------------------------------------------*/
{  
   sym_attptr rec;
 
   if ((rec = (sym_att *)malloc(sizeof(sym_att))) == NULL)
     {
      opserror("new_attlist_rec:  no more memory...");  puteol();
      bomb_out();
     }
  
   rec->attsym = sym;
   rec->next = NULL;
   return(rec);
}




literalize(class_name)
   str_arr class_name;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine completes processing of a Literalize declaration. The
 *    attribute names in the declaration have been assembled into a list
 *    by prior actions of the parser. 
 *
 * Parameters:
 *    class_name - the string name of the class passed by the parser and
 *                 stored in the YACC value stack.
 *
 * Environment:
 *    The attributes for the class have been collected up in a list from
 *    earlier actions of the parser. The beginning and end of the list are
 *    pointed to by the globals 'b_attptr' and 'e_attptr'. Another global
 *    list is maintained through the pointers 'b_cname_ptr' and 'e_cname_ptr'
 *    and holds a record for each class name defined by a Literalize
 *    declaration.
 *
 * Calls:
 *    "opserror", "putstring", "puttab", and "puteol" in "ops5.c".
 *    "test_cname", "add_cname_list", "new_attname", "new_attlist_rec",
 *    and "mark_conflict" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   symptr     csym;       /* Points to symbol table entry for class name. */
   symptr     asym;       /* Ptr to symbol table entry for an attribute. */
   attptr     att;        /* Ptr to an att_list record. */
   sym_attptr symatt;     /* Tmp ptr to a new sym_att record. */
   sym_attptr b_symatt;   /* Head ptr to new version of att list we build. */
   sym_attptr e_symatt;   /* Tail ptr to new version of att list we build. */
   
   if (have_compiled_p)  
     {
      /* Declarations may not appear in the source after we've begun to
       * compile the productions.
       */
      opserror("literalize declaration after p...");
      puttab();  putstring(class_name);  puteol();
     }

   /* Now get the symbol table entry for the class name - we may have to
    * create one if the name itself is not already entered. If the name is
    * already entered, we also check that it has not been previously used
    * as a class name (it may have been entered as an attribute name as OPS5
    * permits the same name to be used both as an attribute name and a class
    * name).
    */
   csym = test_cname(class_name);  

   /* Now enter the class name into the global list of class names by storing
    * its symbol table address in the list.
    */
   add_cname_list(csym); /* The global 'e_cname_ptr' points to the new one. */

   /* Now set some fields in the symbol table entry.
    */
   csym->is_cname = TRUE;       /* Mark it as a class name. */
   reverse_att();               /* First reverse the b_attptr list of attributes. */
   csym->attlist  = b_attptr;   /* Attach the attribute list built earlier */
                                /* in the parse of the declaration.        */

   /* Now take the attribute list for this class and build symbol table
    * entries for each of the attribute names. Recall that when the attribute
    * list was formed earlier in the parse of the declaration (refer to
    * "add_attr_list"), we did not enter the names in the symbol table. Nor
    * did we copy the attribute name (stored in the YACC value stack by
    * the parser) into permanent storage. So we must also do that here since
    * the storage in the YACC value stack for each of the names will be lost
    * when the stack pointer is decremented in the reduction made by the
    * parser.
    *
    * Another thing we will do here is to build another version of the
    * attribute list. Instead of each element of the list holding the name
    * of an attribute, it will hold the address of the symbol table entry
    * for that name. This new list will be hooked onto the record in the
    * global class name list for this class name (pointed to by 'e_cname_ptr').
    */
   att = b_attptr;    /* Get a local pointer to att list for loop below. */
   b_attptr = NULL;   /* Disconnect attribute list from global pointers. */
   e_attptr = NULL;
   b_symatt = NULL;   /* Init pointer to list we will now build below. */
   while (att)
     {
      /* Find or create symbol table entry for this attribute name.
       * Side-effect here also creates permanent storage for the name.
       */
      asym = new_attname(att); 
      /* Do what "add_attr_list" left undone with respect to storage
       * for the name.
       */
      att->attname = asym->symname; 
      
      /* Build a new sym_att record to hold the address of the symbol
       * table entry for the attribute. Then add that record to the new
       * form of attribute list we are constructing.
       */
      symatt = new_attlist_rec(asym);
      if (b_symatt == NULL) 
         b_symatt = e_symatt = symatt;
      else
        {
         e_symatt->next = symatt;
         e_symatt = symatt;
        }
      
      /* OK, get the next attribute.
       */
      att = att->next;
     
     } /*end while */
     
   /* Now, at last, 'b_symatt' points to the new version of the attribute
    * list just built above for this class. Hook this list onto the
    * cname_list record for this class in the global list of class names.
    * Recall that the record for this class is pointed to by 'e_cname_ptr'.
    */
   e_cname_ptr->attlist = b_symatt;
   
   /* Finally, using this new version of the attribute list for the class,
    * add every attribute in the list to the conflict set of each attribute
    * in the list.
    */
   mark_conflict(b_symatt);

}


reverse_att()     /* Called only by "literalize" to reverse the b_attptr list. */
{
   attptr att, succ;

   att = b_attptr;
   b_attptr = e_attptr = NULL;
   while (att)
     {
      succ = att->next;
      if (b_attptr == NULL)
        {
	 b_attptr = e_attptr = att;
	 att->next = NULL;
	}
      else
        {
	 att->next = b_attptr;
	 b_attptr = att;
	}
      att = succ;
     }
}



add_attr_list(attr_name)
   str_arr attr_name;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Build a list of attributes from a Literalize, Vector-Attribute, or
 *    External declaration by adding the new attribute obtained from the
 *    parser to the global list of attributes being built for the
 *    declaration currently being processed. We don't build symbol table
 *    entries here for the names - we'll do that later when the parser
 *    recognizes the kind of declaration being compiled. Instead, we just
 *    build a list of names.
 *
 * Parameters:
 *    attr_name - the string name of the attribute. Storage for the string
 *                is in the YACC value stack.
 *
 * Environment:
 *    The list being built is referenced by the globals 'b_attptr' and
 *    'e_attptr' which point to the beginning and end of the list. 
 *
 * Calls:
 *    "not_dup_att" and "new_attptr" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   /* Add attr_name to attribute list but only after determining that
    * it's not already on the list (i.e., discard duplicate attribute
    * names in the declaration).
    */
   if (not_dup_att(attr_name))
     {
      new_attptr();   /* Add an att_list record to end of the list. */
      e_attptr->attname = attr_name;   /* Put a pointer to the name there. */
                                       /* Note that storage is still in    */
                                       /* YACC value stack.                */
     }
}



boolean
not_dup_att(att)
   string att;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Check if an attribute name already appears in the attribute list
 *    under construction for a declaration.
 *
 * Parameters:
 *    att - the string name of the attribute.
 *
 * Environment:
 *    The head of the attribute list is pointed to by the global 'b_attptr'.
 *
 * Returns:
 *    TRUE if the attribute is not already in the attribute list.
 *
 * Calls:
 *    "strcompare" in this module.
 *    "opswarn", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    "add_attr_list" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
   attptr ptr;

   ptr = b_attptr;
   while (ptr)
     {
      if (strcompare(att, ptr->attname) == 0)
        { 
         opswarn("Duplicate attribute name found : "); 
         putstring(ptr->attname);  puteol();
         return(FALSE);
        }
      ptr = ptr->next;
     }
     
   return(TRUE);
}




new_attptr ()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate and initialize a new att_list record for a new attribute and
 *    add it to the end of the global attribute list being built for the
 *    declaration currently being compiled. 
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The attribute list is pointed to by the globals 'b_attptr' and
 *    'e_attptr'.
 *
 * Calls:
 *    "malloc" system call.
 *    'opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "add_attr_list" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   attptr rec;

   /* Look for an att_list record on the free list or else allocate a
    * new one.
    */
   if (free_attptr == NULL)
     {
      if ((rec = (att_list *)malloc(sizeof(att_list))) == NULL)
        {
         opserror("new_attptr:  no more memory...\n");  puteol();
         bomb_out();
        }
     }
   else
     {
      rec = free_attptr;
      free_attptr =  free_attptr->next;
     }
     
   /* Init the new record.
    */
    rec->attname = NULL;
    rec->num     = 0;
    rec->next    = NULL;

   /* Put it at the end of the global attribute list.
    */
   if (b_attptr == NULL)
     { b_attptr = rec;  e_attptr = rec; }
   else
     { e_attptr->next = rec;  e_attptr = rec; }
}



free_allatt()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Move the att_list records from the global attribute list to the free
 *    list.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The global attribute list is pointed to by 'b_attptr' and 'e_attptr'.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "literal", "vec_att", and "extern_dec" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   /* Just attach the global list at the front of the free list.
    */
   if (b_attptr != NULL)
     {
      if (debug)  printf("free all attribute pointers on att list\n");
      e_attptr->next = free_attptr;   /* End of list onto free list head. */
      free_attptr = b_attptr;         /* Set new free list head. */
     }
     
   /* Now make global list empty.
    */
   b_attptr = NULL;
   e_attptr = NULL;
}




mark_conflict(b_symatt)
   sym_attptr b_symatt;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine performs the last step (except for assigning bindings
 *    to each attribute name corresponding to a field index in a working
 *    memory element; this gets done just before the first production is
 *    compiled) in the processing of a Literalize declaration. The
 *    attributes of the class specified in the Literalize declaration are
 *    represented now in a linked list of records where each record contains
 *    a field holding the address of the symbol table entry for that
 *    attribute name. A pointer to the beginning of that list is passed as
 *    a parameter here. Using this list, we must take each attribute in the
 *    list and add to its conflict set all the other attributes in the list.
 *    The conflict set for a given attribute represents all the other
 *    attributes whose bindings to field element indices must be made
 *    relative to the given attribute. That is to say, no member of the
 *    conflict set may have the same binding as the given attribute. (It is
 *    possible, however, for members of the conflict set to have the same
 *    bindings since the conflict set for a given attribute may consist of
 *    attributes from more than 1 class - recall that the same attribute
 *    may appear in more than 1 Literalize declaration.) The conflict set
 *    for a given attribute is also represented as a linked list of records.
 *    Here too, each record contains a field holding the address of the
 *    symbol table entry for the conflicting attribute. A pointer to the
 *    beginning of the conflict list for the given attribute is stored as
 *    a field in the symbol table entry for the given attribute.
 *
 *    What we must do then in this routine is to take each attribute in
 *    the list passed here (headed by 'b_symatt') and add to its conflict
 *    set every other attribute in the list. Before actually adding
 *    something to the conflict set, we must first check that the conflicting
 *    attribute is not already in the conflict set under consideration. This
 *    would be the case if the given attribute and the conflicting one had
 *    appeared together in a previous Literalize declaration (eg., 
 *    Literalize(foo a b c)  Literalize(bar a b d) ).
 *
 * Parameters:
 *    b_symatt - the beginning of a list of records that represents all
 *               the attributes of the class from the Literalize declaration
 *               currently being compiled. Each record in the list represents
 *               a unique attribute since earlier processing of attribute
 *               for the class has ensured this (i.e., "add_attr_list"
 *               detects duplicate occurrences of the same attribute name
 *               and eliminates them (for example, in Literalize(foo a b c
 *               a d), the class foo has only 4 attributes)..
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "not_in_confset" and "add_conflict" in this module.
 *
 * Called by:
 *    "literalize" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr given_att; /* Refers to att whose conf list we are building. */
   sym_attptr conf_att;  /* Refers to att to put in conf list of given_att. */

   given_att = b_symatt;   /* Begin with first att in the list. */
   while (given_att)       /* Build conflict set for each att in list. */
     {
      conf_att = b_symatt;
      while (conf_att)   /* Add every other att in list to conf set of */
        {                /* given_att.                                 */
         /* Make sure we're not trying to put the attribute into its
          * own conflict set.
          */
         if (conf_att != given_att)
           {
            /* Add conf_att attribute to conflict set unless it's already
             * there.
             */
            if (not_in_confset(given_att->attsym, conf_att->attsym))
               add_conflict(given_att->attsym, conf_att->attsym);
           }
         conf_att = conf_att->next;
        }
      given_att = given_att->next;
     }
}



boolean
not_in_confset(sym, confsym)
   symptr sym, confsym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Check if an attribute is already in the conflict set of another.
 *
 * Parameters:
 *    sym     - the attribute whose conflict set we must check.
 *    confsym - the attribute we're looking for in sym's conflict set.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    TRUE if the attribute is already in the other's conflict set.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "mark_conflict" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   conf_ptr cptr;

   cptr = sym->conflist;   /* Start with 1st member of sym's conf list. */
   while (cptr)            /* Look through sym's whole conf list. */
     {
      if (cptr->conf == confsym)  return(FALSE);
      cptr = cptr->next;
     }
  
   return(TRUE);   /* It's not in the conflict set. */
}



add_conflict(sym, confsym)
   symptr sym, confsym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add an attribute to the conflict set of another attribute.
 *
 * Parameters:
 *    sym     - the attribute whose conflict set we wish to add to.
 *    confsym - the attribute to add to sym's conflict set.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "mark_conflict" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   conf_ptr rec;   /* The new record to create and add to conflict list.*/
   conf_ptr cptr, last;  /* Added 2/14/89 to make binding assign close to Lisp OPS5. */

   /* Allocate a record to add to sym's conflict list.
    */
   if ((rec = (conf_list *)malloc(sizeof(conf_list))) == NULL)
     {
      opserror("add_conflict: no more memory... \n");  puteol();
      bomb_out();
     }
  
   rec->conf = confsym;


   /* 2/14/89: This is what we used to do:                */   
   /* Put new attribute at head of conflict list for sym.
    */
   /* rec->next = sym->conflist; */
   /* sym->conflist = rec;       */


   /* 2/14/89: This is what we do now to get bindings close to Lisp OPS5:
    */
   /* Put new attribute at end of conflict list for sym.
    */
   cptr = sym->conflist;
   if (cptr)
     {
      while (cptr)  {last = cptr; cptr = cptr->next;}
      last->next = rec;
     }
   else
      sym->conflist = rec;
   rec->next = NULL;
}




literal ()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Complete processing of a Literal declaration. The attributes given
 *    in the declaration are in a list pointed to by 'b_attptr'. Each
 *    attribute name must be entered (or found) in the symbol table and
 *    marked as an attribute and assigned the binding specified for it
 *    in the declaration. The binding will not be marked as valid here
 *    since we can't determine its acceptability until all declarations
 *    have been compiled and the full conflict set is known for the
 *    attribute.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The attributes are assembled in a list pointed to by 'b_attptr'.
 *
 * Calls:
 *    "opserror", "puttab", "putstring", "puteol", and "opswarn" in "ops5.c".
 *    "free_allatt" and "new_attname" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   attptr att;    /* Points to an attribute in the global list. */
   symptr sym;    /* Points to the symbol table entry for an attribute. */

   if (have_compiled_p)  
     {
      /* Declarations may not appear in the source after we've begun to
       * compile the productions.
       */
      opserror("literal declaration after p...");  puteol();
      free_allatt();   /* Free storage for the list of names. */
      return;          /* Continue with compilation to detect other errors. */
     }

   att = b_attptr;     /* Point to head of list of attribute names. */
   if (att == NULL)
     {
      opswarn("Empty literal declaration");  puteol();
     }
   else
     {
      while (att)
        {
         /* Find or create symbol table entry for this attribute name.
          * Side-effect here also creates permanent storage for the name.
          */
         sym = new_attname(att);
         if (sym->opsbind > 0)
           {
            opswarn("An attribute appears again in a literal statement.");
            puteol();
           }
         /* Store the specified binding although we can't validate it
          * until later. (i.e. sym->valid_bind does not get set here)
          */
         sym->opsbind = att->num;
         att = att->next;
        }
      free_allatt();   /* List storage no longer needed. */
     }
}



add_literal(att, val)
   str_arr att;
   int     val;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine builds a list of the attributes in a Literal declaration.
 *    The linked list of records built is pointed to by the global 'b_attptr'.
 *    The global 'e_attptr' points to the end of the list where the record
 *    for the new attribute will be added. The attribute name and the number
 *    being assigned to it are stored in the record.
 *
 * Parameters:
 *    att - string name of the attribute. Storage is in YACC value stack.
 *    val - the index number being assigned to the attribute.
 *
 * Environment:
 *    The attribute list is pointed to by globals 'b_attptr' and 'e_attptr'.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *    "new_attptr" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   if ((val < FIRST_OPSBIND) || (val > MAX_OPSBIND))
     {
      opserror("Number assigned to attribute is out of range, <1 or >127");
      puteol();
      return;   /* Continue compilation looking for other errors. */
     }

   new_attptr();   /* Allocate a new record at end of list. */
   
   e_attptr->attname = att;    /* Fill in the record with the attribute. */ 
   e_attptr->num = val;        /* Fill in the binding specified. */
}




vec_att()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine completes processing of a Vector-Attribute declaration.
 *    The vector attribute names in the declaration have been assembled into
 *    a list by prior actions of the parser. The head of the list is pointed
 *    to by the global 'b_attptr'. Here we create symbol table entries for
 *    the names and mark them as vector attributes.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The vector attribute names are in the global attribute list pointed
 *    to by b_attptr.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *    "free_allatt" and "sym_addr" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   attptr att;
   symptr sym;

   if (have_compiled_p)  
     {
      /* Declarations may not appear in the source after we've begun to
       * compile the productions.
       */
      opserror("vector-attribute declaration after p...");  puteol();
      free_allatt();   /* Free storage for the list of names. */
      return;          /* Continue with compilation to detect other errors. */
     }

   att = b_attptr;     /* Point to head of list of names. */
   if (att == NULL)
     {
      opswarn("Empty vector-attribute declaration");  puteol();
      }
   else
     {
      while (att)
        {
         sym = sym_addr(att->attname);
         sym->is_vec = TRUE;
	 /* sym->is_att = TRUE; will occur in the Literal/Literalize decl. */
         att = att->next;
        }
      free_allatt();   /* List storage no longer needed. */
     }
}




extern_dec()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine completes processing of an External declaration.
 *    The external function names in the declaration have been assembled into
 *    a list by prior actions of the parser. The head of the list is pointed
 *    to by the global 'b_attptr'. Here we create symbol table entries for
 *    the names and mark them as external function names.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The external function names are in the global attribute list pointed
 *    to by b_attptr.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *    "free_allatt" and "sym_addr" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   attptr att;
   symptr sym;

   att = b_attptr;     /* Point to head of list of names. */
   if (att == NULL)
     {
      opswarn("Empty external declaration"); puteol();
      }
   else
     {
      while (att)
        {
         sym = sym_addr(att->attname);
         sym->is_fname = TRUE;
         add_to_ext_list(sym);
         att = att->next;
        }
      free_allatt();   /* List storage no longer needed. */
     }
}



add_to_ext_list(sym)
   symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Keep a global list of all the external names.
 *
 * Parameters:
 *    sym - symbol table entry for an external name.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "extern_dec" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   elist_ptr eptr;

   if ((eptr = (elist_rec *)malloc(sizeof(elist_rec))) == NULL)
     {
      opserror("add_to_ext_list: no more memory... \n");  puteol();
      bomb_out();
     }
   
   eptr->ext_fname = sym;
   eptr->next = ExternalList;
   ExternalList = eptr;
}




assign_scalars()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called after all declarations have been compiled and
 *    just after the parser has recognized the beginning of the first
 *    production. Here we must now assign valid bindings to all the
 *    attributes for each class and determine that bindings assigned to
 *    attributes explicitly with Literal declarations are also acceptable
 *    with respect to the conflict set for each of those attributes. We do
 *    this by taking the global list of class names (the cname_list) pointed
 *    to by 'b_cname_ptr' and processing each class in the list. For each
 *    class in the list we assign unique bindings to each attribute in that
 *    class. If the attribute is a vector attribute (and remember only 1 is
 *    permitted in each class), then we defer assigning it a binding until
 *    all non-vector attributes in all classes have been assigned bindings.
 *    At that time, each vector attribute is assigned a binding that is 1
 *    greater than the largest binding assigned to an attribute in any of
 *    the classes in which the vector attribute appears. Equivalently, this
 *    means that all the attributes on a vector attribute's conflict list
 *    must have a lower binding than that vector attribute.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    All declarations have been compiled and the global list of class
 *    names along with associated attribute lists is pointed to by
 *    'b_cname_ptr'. The beginning of the first production has just been
 *    recognized by the parser.
 *
 * Calls:
 *    "add_vec_attlist", "assign_scalars2", and "assign_vec_binding" in
 *    this module.
 *    "opserror", "puttab", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    "cmp_production" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   cname_ptr  cnameptr;      /* Points to a class in the class list. */
   sym_attptr aptr;          /* Points to an attribute of a class. */
   sym_attptr vec_attlist;   /* Heads list of all the vector atts found. */
   symptr     sym;           /* Points to the symbol table entry of an att. */
   boolean    found_vecatt;  /* Tells when we find a vector att in a class. */

   vec_attlist = NULL;       /* No vector atts found yet. */
   cnameptr = b_cname_ptr;   /* Point to the list of classes. */
   while (cnameptr)          /* Assign att bindings for each class in list. */
     {
      found_vecatt = FALSE;       /* No vector att found in this class yet. */
      aptr = cnameptr->attlist;   /* Get the att list for this class. */
      while (aptr)                /* Process each att in the attlist. */
        {
         sym = aptr->attsym;      /* Get symbol table entry for the att. */
         
         /* If its not a vector att, we can assign a binding right away;
          * otherwise, we defer the assignment till later.
          */
         if (!sym->is_vec)
            assign_scalar2(sym);
         else
           {
            /* Got a vector att so make sure its the only one in the class.
             */
            if (found_vecatt)
              {
               opserror("More than 1 vector attribute in a class.");
               puteol();
              }
            else
              {
               /* Remember that we've now found a vector att in this class
                * and put the vector att on a list so we can deal with it
                * later.
                */
               found_vecatt = TRUE;
               add_vec_attlist(sym, &vec_attlist);
              }
           }
         
         aptr = aptr->next;   /* Get next att for this class. */
        
        } /* end while */
      
      cnameptr = cnameptr->next;   /* Get next class on list. */
     
     } /* end while */

   /* Save the list of vector atts for later use in building assembly file.
    */
   g_vecatt_list = vec_attlist;
     
   /* Now that we've assigned bindings to all the simple attributes, we
    * can next assign bindings to all the vector attributes found above.
    */
   while (vec_attlist)
     {
      assign_vec_binding(vec_attlist->attsym);
      vec_attlist = vec_attlist->next;
     }


   /* Finally, deal with all the other attributes that were declared in
    * Literal statements but were not part of any Literalize declaration.
    * Also detect any Vector-Attribute declarations that did not appear
    * in a Literal/Literalize declaration.
    */
   assign_finish();

} /* end assign_scalars */



assign_scalar2(sym)
   symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    We want to assign a binding to a given attribute. We will give the
 *    attribute the lowest binding available between 2 and 127 inclusive
 *    which does not conflict with a binding already held by some other
 *    attribute on the given attribute's conflict list. Recall that the
 *    conflict list for a given attribute is a list of all other attributes
 *    whose bindings must be unique relative to the given attribute since
 *    the conflict list consists of all the atributes from all the classes
 *    in which the given attribute appears. Remember that this does not
 *    mean that none of the attributes in the conflict list may have the
 *    same binding; it means that none of the attributes in the conflict
 *    list may have the same binding as the given attribute. (Thus, for
 *    example, with Literalize(foo a x) Literalize(bar b x) Literalize(
 *    baz c x), a,b,c are on x's conflict listand x could have the binding
 *    3 while a,b,c could have the same binding of 2.)
 *
 *    If the given attribute already has a binding assigned, it may have
 *    been due to an explicit assignment from a Literal declaration. In
 *    this case, we still must check the conflict list of the given
 *    attribute to make sure that Literal did not erroneously assign the
 *    same binding to a member of the given attribute's conflict set. We
 *    could not make this determination during the earlier processing of
 *    the Literal declaration since we cannot determine the full conflict
 *    set until all Literalize declarations are processed.
 *
 * Parameters:
 *    sym - pointer to the symbol table entry for the attribute whose
 *          binding is to be set.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *    "printsym1" in "printops.c".
 *    "add_bucket" in this module.
 *
 * Called by:
 *    "assign_scalars" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
   opsnum   num;
   opsnum   bigbind;
   int      i;
   conf_ptr cptr;
   boolean  bind[MAX_OPSBIND+1];

   /* First check if the attribute already has been assigned a binding.
    * If so, then either this attribute appears in another class which
    * has already received attribute binding assignments from "assign_scalar2"
    * or else this attribute was given a binding by a Literal declaration.
    */
   if (sym->opsbind > 0)
     {
      if (sym->valid_bind)
         /* The assigned binding has already been validated by
          * "assign_scalar2".
          */
         return;
      else
        {
         /* Binding was assigned by a Literal declaration so check its
          * validity.
          */
         num = sym->opsbind;
         cptr = sym->conflist;
         while (cptr)
           {
            if (cptr->conf->opsbind == num)
              {
               opserror("Literal gave same binding to 2 attributes in same class.");
               puteol();
               return;
              }
            cptr = cptr->next;
           }
         sym->valid_bind = TRUE;
	 add_bucket(sym);
         return;
        }
     }
   else
     {
      /* A binding has not yet been assigned and validated, i.e., the
       * att appeared in a Literalize but not in a Literal statement.
       * For an att declared in a Literalize, we reserve the first
       * binding for the class name and thus begin att bindings at
       * 1 beyond FIRST_OPSBIND. 
       */
      for (i = FIRST_OPSBIND+1; i <= MAX_OPSBIND; i++)  bind[i] = FALSE;
      
      /* Record all the bindings assigned to members of the attribute's
       * conflict set and remember the largest one.
       */
      bigbind = FIRST_OPSBIND+1;
      cptr = sym->conflist;
      while (cptr)
        {
         num = cptr->conf->opsbind;
         if (num >= FIRST_OPSBIND)  bind[num] = TRUE;
         if (num > bigbind)  bigbind = num;
         cptr = cptr->next;
        }
      
      /* Find the first available binding not assigned to a member of
       * the conflict set.
       */
      i = FIRST_OPSBIND+1;
      while (i <= bigbind)
        {
         if (!bind[i])
            break;
         else
            i++;
        }
       
      /* Assign the binding making sure it doesn't exceed the max.
       */ 
      if (i <= MAX_OPSBIND)
        {
         sym->opsbind = i;
         sym->valid_bind = TRUE;
         add_bucket(sym);
        }
      else
        {
         opserror("No binding assigned since class has too many attributes.");
         puteol();
        }

      if (debug)  { printf("Assign_scalar2:\n");  printsym1(sym); }
     }
}



assign_finish()
{
   int    i;
   symptr sym;

   for (i = 0; i < HASHTABSIZE; i++)
     {
      sym = hashtable[i];
      while (sym)
        {
	 if ((sym->is_att) && (!sym->valid_bind))
	   {
	    /* Att declared in a Literal statement but not in a Literalize.
	     */
	    sym->valid_bind = TRUE;
	    add_bucket(sym);
	   }

	 if ((sym->is_vec) && (!sym->is_att))
	   {
	    /* Vector att was not declared in a Literalize or Literal.
	     */
            opserror("Vector att did not appear in any Literalize or Literal declaration: ");
            putstring(sym->symname);
	    puteol();
	   }
	 
         sym = sym->next;
	} /* end while */
     }
}



add_bucket(att)
   symptr att;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Attach an attribute to the bucket corresponding to its assigned
 *    binding.
 *
 * Parameters:
 *    att - the attribute to be added to a bucket.
 *
 * Environment:
 *    The 'bucket' array and 'g_last_buc' are maintained here.
 *
 * Calls:
 *    "new_attlist_rec" in this module.
 *
 * Called by:
 *    "assign_scalar2" and "assign_vec_binding" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr buc;
   opsnum     num;

   buc = new_attlist_rec(att);
   
   num = att->opsbind;
   
   if (bucket[num])
     { 
      buc->next = bucket[num];   /* Put new one at head of */
      bucket[num] = buc;         /* list at this bucket.   */
     }
   else
      bucket[num] = buc;
      
   if (g_last_buc < num)  g_last_buc = num;
}



add_vec_attlist(vsym, veclist)
   symptr     vsym;
   sym_attptr *veclist;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add a vector attribute to a list of all vector attributes encountered
 *    in "assign_scalars" as it assigns bindings. The bindings for vector
 *    attributes will be delayed till last by "assign_scalars".
 *
 * Parameters:
 *    vsym    - pointer to the symbol table entry of the vector attribute.
 *    veclist - the address of the pointer to the list of vector attributes.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "new_attlist_rec" in this module..
 *
 * Called by:
 *    "assign_scalars" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr vptr;
   
   if (*veclist == NULL)
     {
      *veclist = new_attlist_rec(vsym);
      return;
     }
   else
     {
      vptr = *veclist;
      while (vptr)
        {
         if (vsym == vptr->attsym)
           {
            /* The vector attribute is already on the list so just return.
             */
            return;
           }
         vptr = vptr->next;
        }
      
      /* Vector attribute is not in the list so put it at the head of the
       * list.
       */
      vptr = new_attlist_rec(vsym);
      vptr->next = *veclist;
      *veclist = vptr;
     }
}



assign_vec_binding(vsym)
   symptr vsym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Assign a binding to a vector attribute. The binding must be bigger
 *    than that of any attribute on its conflict list.
 *
 * Parameters:
 *    vsym - pointer to the symbol table entry for the vector attribute 
 *           whose binding is to be set.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *    "add_bucket" in this module.
 *
 * Called by:
 *    "assign_scalars" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
   opsnum   num;
   opsnum   bigbind;
   conf_ptr cptr;

   
   /* Find the biggest binding in its conflict set.
    */
   bigbind = FIRST_OPSBIND - 1;
   cptr = vsym->conflist;
   while (cptr)
     {
      num = cptr->conf->opsbind;
      if (num > bigbind)  bigbind = num;
      cptr = cptr->next;
     }
     
   if (vsym->opsbind > 0)
     {
      /* The vector attribute was assigned a binding by a Literal
       * declaration, so validate it.
       */
      if (vsym->opsbind <= bigbind)
        {
         opserror("Literal assigned too small a binding for vector att.");
         puteol();
         return;
        }
      else
        {
         vsym->valid_bind = TRUE;
         add_bucket(vsym);
        }
     }
   else
     {
      /* The vector attribute has no binding so give it a binding 1
       * greater than the largest in its conflict set. Make sure the
       * binding assigned is bigger than FIRST_OPSBIND in this case.
       */
      if (bigbind < FIRST_OPSBIND)
         num = FIRST_OPSBIND + 1;
      else
         num = bigbind + 1;
      if (num > MAX_OPSBIND)
        {
         opserror("No binding in range could be assigned to vector att.");
         puteol();
        }
      else
        {
         vsym->opsbind = num;
         vsym->valid_bind = TRUE;
         add_bucket(vsym);
        }
     }
}

