
/****************************************************************************
 *
 * MODULE:  cmprhs.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    This module produces "threaded" code for the right hand sides (RHS)
 *    of productions. The routines here deal mostly with variables used
 *    and/or created in the RHS of a production. 
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.2 Anurag Acharya
 *                   Modified "note_rhs()" so that "include ../rhs/global.h"
 *                   is generated only once. This is needed since the include
 *                   temporary file inclfile is writtento in an append mode.
 *                   Added the routine "write_ext_func_list()". This routine
 *                   generates the extern declarartions for all the external 
 *                   routines in the program and writes them to inclfile.
 *                   It takes over this fucntionality from "write_ext_list()"
 *                   in gencode.c
 *                   Added a call to write_ext_func_list() to note_rhs().
 *                   This ensures that the extern declarations will be placed
 *                   before any reference to the routines themselves.
 *                   These changes fix the bug reported by dean grannes
 *                   who couldn't get external routine calls to work.
 *    12 May 89 V2.0 Dirk Kalp
 *                   Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    21 Oct 88 V3.2  Dirk Kalp
 *                    Change data segment pseudo ops in routine "init_rhs".
 *     1 Oct 88 V3.1  Dirk Kalp
 *                    Added routine "extfunc" to be called by parser when
 *                    external function name is used in RHS.
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    No changes.
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    22 Aug 86       Dirk Kalp
 *    19 Aug 86
 *     1 May 86
 *    29 Apr 86
 *    27 Feb 86 V1.0  Dirk Kalp
 *                    Put together from previous version created by
 *                    Quei-Len Lee.
 *
 * Copyright (c) 1986, 1987, 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/

 
   
#include "defs.h"


/* Imported Routines:
 *    From ops5.c:
 *       opserror
 *       putstring
 *       puteol
 *
 *    From literal.c:
 *       newname
 *
 *    From cmplhs.c:
 *       find_equiv_varname
 *       newvar
 *
 *    From gencode.c:
 *       newinst
 *       codeoutc
 *       codeouts
 *       codeouti
 *       codeoutl
 *       outeol
 */

/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern string      newname();                /* Imported from literal.c. */
extern varptr      find_equiv_varname();     /* Imported from cmplhs.c. */    
extern varptr      newvar();                 /* Imported from cmplhs.c. */    
extern symptr      lookup_sym_addr();        /* Imported from literal.c. */    



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
void write_ext_func_list();






note_rhs()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called just before we begin to parse and compile the
 *    right-hand side of a production into threaded code. If this is the
 *    first production in the system, then we write out the pseudo-ops in
 *    the header to mark the beginning of a data segment for the threaded
 *    code for the right-hand sides. For each production we write out the
 *    production name as a label to mark the start in the data segment of
 *    the threaded code for the right-hand side of the production. The
 *    subsequent actions of the parser in the syntax directed translation
 *    of each production's right-hand side will result in the threaded code
 *    being produced and added to this data segment. 
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The left hand-side of a production has just been compiled and we
 *    are about to begin to compile the right-hand side.
 *
 * Calls:
 *    "codeouts", "codeoutc", and "outeol" in "gencode.c";
 *    "write_ext_func_list" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 * Log:
 *   9/29/89 : Anurag Acharya
 *     added call to write_ext_func_list()
 *     enclosed the statements to generate the include and extern
 *        declarations in an if block so that it executes only once
 *        in the entire compilation (rather than once per output file.
 *
 *-------------------------------------------------------------------------*/
{
   flag_rhs = TRUE;   /* Flag that we're now working on an RHS. */
   
   /* Write the pseudo-ops to mark the start of the data segment for
    * the threaded code.
    */
   if (have_cmprhs == FALSE)
     {
      have_cmprhs = TRUE;          /* We only do this once per output file. */

      /* 9/29/89 : Anurag Acharya
       * since the include file is only appended to, we need to generate the 
       * the include statement only once. The same goes for the external
       * routine declarations.
       */
      if (g_num_rete_roots == 0)
	{
	  fprintf(fp_incl, "#include %c../rhs/global.h%c\n\n", '"', '"');
	  write_ext_func_list();
	}

      /* 9/29/89 : Anurag Acharya
       * generate the extern declarations for all the right hand side action
       * rouitnes.
       */

      fprintf(fp_sub, "OpsVal OpsSymbols[];\n\n");
      fprintf(fp_sub, "extern int ops_symcon(), ops_tab(), ops_variable(), opsret();\n"); /*ops_default*/
      fprintf(fp_sub, "extern int ops_bmake(), ops_emake(), ops_remove(), ops_bmodify(), ops_emodify(), ops_halt();\n");
      fprintf(fp_sub, "extern int ops_reset(), ops_fcall(), ops_genatom(), ops_bind(), ops_cbind(), ops_call(), ops_write();\n");
      fprintf(fp_sub, "extern int ops_openfile(), ops_closefile(), ops_default(), ops_tabto(), ops_varibletab(), ops_rval();\n");
      fprintf(fp_sub, "extern int ops_fixcon(), ops_litval(), ops_substr(), ops_crlf(), ops_rjust(), ops_accept(), ops_acceptline();\n");
      fprintf(fp_sub, "extern int ops_ucall(), ops_add(), ops_sub(), ops_mult(), ops_div(), ops_mod();\n");
     }
   
   /* Newline and then write the production name as a label.
    */  
   fprintf(fp_sub, "\n/* %s */\nThreadedCodeElement p%d[] = {",
                   g_pname, g_pcount);
   first_inst = TRUE;    /* Tell newinst() to not output a ',' before the 1st rhs code. */
}


/**********************************************************************
 *                                                                    *
 *  write_ext_func_list()                                             *
 *     It is necessary to separate the declaration of the external    *
 *     routines themselves from the declaration of the external       *
 *     routines' symbol names -- since OpsSymbols isn't generated     *
 *     till much later. This routine will generate just the extern    *
 *     declarations for the routines declared "external".             *
 *                                                                    *
 *     Environment :                                                  *
 *         Assumes the global variable "ExternalList" has a list      *
 *         of data structures that conatin the info about all the     *
 *         external routines. writes to the file "fp_incl"            *
 *                                                                    *
 *     Calls:                                                         *
 *          nothing                                                   *
 *                                                                    *
 *     Called by :                                                    *
 *          note_rhs in this module                                   *
 *                                                                    *
 *     Log:                                                           *
 *         9/29/89 : Anurag Acharya                                   *
 *            created.                                                *
 **********************************************************************/

static void 
write_ext_func_list()
{
  elist_ptr eptr;

  fprintf(fp_incl,"\n");
  for (eptr = ExternalList; eptr != NULL; eptr = eptr->next)
   fprintf(fp_incl, "extern int %s();\n", eptr->ext_fname->symname);

  fprintf(fp_incl,"\n");
}

init_bvar()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize globals used to assign bindings to RHS variables
 *    created with BIND and CBIND actions.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The compilation of a new production has just begun.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "init_p" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   g_bvarnum = 0;
   g_cbvarnum = 0;
}

  
      
new_bvar(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Create a new variable bound in a BIND action in the RHS and add it to
 *    the global list headed by 'g_bvar'.
 *
 * Parameters:
 *    var - the string name of the variable stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newvar" in "cmplhs.c".
 *    "newname" in "literal.c".
 *
 * Called by:
 *    "bvarnum" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   varptr new;

   new = newvar();
   new->varname = newname(var);
   new->snum    = g_bvarnum;
   new->next    = g_bvar;
   if (g_bvar)  g_bvar->prev = new;
   g_bvar = new;
}



new_cbvar(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Create a new variable bound in a CBIND action in the RHS and add it to
 *    the global list headed by 'g_cbvar'.
 *
 * Parameters:
 *    var - the string name of the variable stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newvar" in "cmplhs.c".
 *    "newname" in "literal.c".
 *
 * Called by:
 *    "cbvarnum" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   varptr new;

   new = newvar();
   new->varname = newname(var);
   new->cenum   = g_cbvarnum;
   new->next    = g_cbvar;
   if (g_cbvar)  g_cbvar->prev = new;
   g_cbvar = new;
}



bvarnum(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Generate the "threaded" code to specify the variable bound in a
 *    BIND action in the RHS.
 *
 * Parameters:
 *    var - the string name of the variable stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "find_equiv_varname" in "cmplhs.c".
 *    "newinst" and "codeouti" in "gencode.c".
 *    "new_bvar" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   varptr old;

   old = find_equiv_varname(var, g_bvar);
   
   if (old)
     {
      newinst();  codeouti(old->snum);
     }
   else
     {
      g_bvarnum--;
      new_bvar(var); 
      newinst();  codeouti(g_bvarnum);
     }
}



cbvarnum(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Generate the "threaded" code to specify the variable bound in a
 *    CBIND action in the RHS.
 *
 * Parameters:
 *    var - the string name of the variable stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "find_equiv_varname" in "cmplhs.c".
 *    "newinst" and "codeouti" in "gencode.c".
 *    "new_cbvar" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   varptr old;

   old = find_equiv_varname(var, g_cbvar);
   
   if (old)
     {
      newinst();  codeouti(old->cenum);
     }
   else
     {
      g_cbvarnum--;
      new_cbvar(var); 
      newinst();  codeouti(g_cbvarnum);
     }
}




ele_var(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called to handle a variable used in the right-hand
 *    side of the current production to designate a working memory element.
 *    The variable should have appeared earlier as a condition element
 *    variable in the left-hand side of the production or as a variable
 *    in a CBIND action in the list of actions for the RHS. If the variable
 *    has no prior occurrence in either of these, then we have an error.
 *
 * Parameters:
 *    var - the string name of a variable stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    An index of the binding for the variable. If the variable bound to
 *    a ce from the LHS, then the index is a positive number representing
 *    that ce. If the variable has been bound with CBIND, then the index
 *    is a negative number representing ???????
 *
 * Calls:
 *    "find_equiv_varname" in "cmplhs.c".
 *    "opserror", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{  
   varptr old;

   /* First see if the variable has a binding from a CBIND action.
    */
   old = find_equiv_varname(var, g_cbvar);
   
   /* If not, see if it has a binding to a condition element in the LHS.
    */
   if (old == NULL)  old = find_equiv_varname(var, g_cevar);
   
   if (old)
      return(old->cenum);   /* Return an index of the binding. */
   else
     {
      opserror("no such element variable : ");  putstring(var);
      putstring("  in rule : ");                putstring(g_pname);
      puteol();
    }
}



int
ele_num(n)
   int n;   /* The n-th positive ce in the LHS. */
{
   int i, countdown;

   countdown = n;
   i = FIRST_CE_INDEX;
   while ((i <= g_cecount) && (countdown > 0))
     {
      if (ce_type[i] == POSITIVE_CE)  countdown--;
      i++;
     }

   if (countdown == 0)
      return(i-1);
   else
     {
      opserror("Invalid element designator number.");
      printf("Designator is %d.\n", n);
      bomb_out();
     }
}





varnum(var)
   string var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Generate the "threaded" code to designate an attribute variable used
 *    in an RHS action. The variable is bound either to a variable created
 *    with a previous BIND action in the production or to an attribute
 *    variable associated with the left hand side of the production.
 *
 * Parameters:
 *    var - the string name of the variable stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "find_equiv_varname" in "cmplhs.c".
 *    "newinst" and "codeouti" in "gencode.c".
 *    "opserror", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    "cmp_var" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   varptr old;

   /* First see if the variable was bound using BIND.
    */
   old = find_equiv_varname(var, g_bvar);
   
   if (old)
     {
      /* It was created by BIND so write the assigned binding (it's a
       * negative number) to the code output file.
       */
      newinst();  codeouti(old->snum);
     }
   else
     {
      old = find_equiv_varname(var, g_var);
      if (old)
        {
         /* When the variable is associated with an LHS value, write the
          * binding info (which wme in the token and which field of the wme)
          * to the code output file.
          */
         newinst();  codeouti(old->cenum);  
         newinst();  codeouti(old->snum);
        }
      else
        {
         opserror("no such variable : ");  putstring(var);
         putstring("  in rule : ");        putstring(g_pname);
         puteol();
        }
     }
}







extfunc(fname)
   str_arr fname;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Check that a external function name used in the RHS has been declared.
 *    Also write the name into the threaded code.
 *
 * Parameters:
 *    fname - the string name of the external function stored in the YACC
 *            value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "lookup_sym_addr" in "literal.c".
 *
 * Called by:
 *    parser in "ops5.yacc".
 *
 *-------------------------------------------------------------------------*/
{
   symptr fsym;

   fsym = lookup_sym_addr(fname);
   if (fsym != NULL)
     {
      if (fsym->is_fname)  { codeoutsx(fname); return; }
     }

   /* Name not found or is not declared external.
    */
   opserror("Undeclared external function : ");  putstring(fname);
   putstring("  in rule : ");        putstring(g_pname);
   puteol();
}

