
/****************************************************************************
 *
 * MODULE:  cmplhs.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Compile the left hand sides of OPS5 productions into a Rete network.
 *    This module also performs some of the translation of the right hand
 *    sides into "threaded" code. The details of the compilation of the
 *    left hand sides (LHS) of productions and the construction of the
 *    network follow below.
 *
 *
 *    Compiling LHS's of Productions:
 *
 *    The LHS's of the productions in a given production system are compiled
 *    into a network of nodes linked together in  a fashion which represents
 *    the conjunction of all the attribute terms in a given condition element
 *    and the subsequent conjunction of all the condition elements in a given
 *    production to effectively represent the instantiation of the production.
 *    The general structure is one in which each condition element is
 *    represented by a linked list of 1-input nodes representing the attribute
 *    terms of the condition element. The conjunction of the attribute terms
 *    is implicitly represented by the son-link of the predecessor term (i.e.
 *    node) to the single input of its successor term (i.e., node). [Note: Our
 *    compiler does no preprocessing of the productions or condition elements
 *    to optimize the ordering of attribute tests within a condition element
 *    or ordering of condition elements. Presumably this is left in the hands
 *    of the applications programmer. We simply look for structural similarity
 *    between condition elements in their initial sequence of terms and share
 *    network structure on that basis. Thus (ce1 ^a1 1 ^a2 2 ^a3 3) and
 *    (ce2 ^a3 3 ^a2 2 ^a1 1) are separate and distinct network branches.]
 *    Thus in order for a working memory element to match the condition
 *    element, it must pass each attribute test in turn at the nodes in the
 *    chain.
 *
 *    The conjunction of the condition elements is represented in the network
 *    in pairs. Instead of joining together all of the n condition elements
 *    of a given production to an n-input node, we represent the conjunction
 *    by joining a pair of condition elements  together with a 2-input node
 *    (we refer to these as beta nodes) and then, using another 2-input node,
 *    tieing the last join with the next condition element, etc. This
 *    requires n-1 2-input nodes for n distinct condition elements.
 *
 *    We also use structural similarity (as defined elsewhere) between
 *    condition elements in the same or different productions to reduce
 *    duplication of condition element branches or sub-branches in the
 *    network. This implies that some nodes might have a son-link to more
 *    than one successor node at the point where 2 or more condition elements
 *    diverge in their structural similarity. Such multiple son-links from
 *    a node may also exist when an identical condition element appears in
 *    several productions and is thus joined to perhaps a different subsequent
 *    condition element in each production. We refer to all the sons of a
 *    node as brothers. In order to represent this abstraction of parent
 *    node with an arbitrary number of sons-links, we will add a brother
 *    link to our node data structure and use a single son-link of the
 *    parent to point to the first son in a list of sons linked by their
 *    brother-links. In effect, the son-link moves down in the network while
 *    the brother link moves horizontally.
 *
 *    In actuality, we will have 2 brother-links in each node to distinguish
 *    between "left" and "right" brothers. This distinction is necessary (as
 *    we will see) because the son-link from a parent node may go to a 1-input
 *    node or may go to the left or right input of a 2-input node.
 *
 *    The network is represented using the following basic 1-input nodes:
 *        ROOT_NODE
 *        P_NODE
 *        T_NODE
 *        ANY_NODE
 *        RI_NODE
 *    and 2-input (beta) nodes:
 *        AND_NODE
 *        NOT_NODE
 *
 *    A single ROOT_NODE exists in the network and represents the starting
 *    point in the network where a new or modified working memory element
 *    is passed in for inspection by the nodes in the match phase. The sons
 *    of the ROOT_NODE are the first term (node) of each of the distinct
 *    conditions elements in all the productions of the production system.
 *    As brothers, these first term nodes are linked together by their
 *    brother links. Since we do not have to distinguish between left and
 *    right inputs for 1-input nodes, we adopt the convention of using the
 *    left brother link to connect them together and to represent the fact
 *    that they all have the same parent.
 *
 *    The T_NODE and ANY_NODE are the 1-input nodes used for the terms in the
 *    condition elments and respectively represent the common attribute tests
 *    and the attribute tests against lists of constants. These are the
 *    primary nodes that make up the network branch representing a condition
 *    element.
 *
 *    The P_NODE is used as a terminal node in the system and represents the
 *    instantiation of its corresponding production.
 *
 *    The RI_NODE is used as a special marker node to identify a condition
 *    element branch in the network as the right input of a beta node.
 *    Prior to linking a condition element branch to the right input of
 *    a beta node, we add the RI_NODE to the end of the branch and then
 *    link it to the beta node. The RI_NODEs are necessary to mark the right
 *    input branches to beta nodes so that we can properly chain left and
 *    right brother lists for beta nodes whenever a particular network branch
 *    serves as both a right input for one beta node and a left input for
 *    another. Without the RI_NODE, we do not know whether to chain from the
 *    left or right of the new beta node, i.e., we do not know how the parent
 *    of the RI_NODE was previously linked into the network - we only know
 *    that it points to a node. Thus the RI_NODE serves as the mechanism for
 *    changing the role of the network branch from a left input to a right
 *    input. Whenever an RI_NODE appears in a left-brother chain, it
 *    effectively announces that the network branch terminated by the parent
 *    of the RI_NODE is a right input to the beta nodes below the RI_NODE.
 *    These sons of the RI_NODE are all chained together by right brother
 *    links. A few obvious (hopefully the above discussion has made it so)
 *    facts about RI_NODEs can be stated:
 *        1. An RI_NODE will never appear more than once in a given
 *           left-brother chain. (Any beta nodes below in the network
 *           that require the branch above the RI_NODE as a right input
 *           will be sons of that RI_NODE and will be chained together in
 *           a right-brother list.)
 *        2. RI_NODEs (or any other 1-input nodes) never appear in
 *           right-brother chains. (By convention, we treat the single
 *           input of a 1-input node as a left input.)
 *
 *    A beta node is a 2-input node with its inputs designated as left and
 *    right inputs. We construct the network for the LHS of a production by
 *    building a network branch for the first condition element and building
 *    a branch for the second condition element and then joining them with
 *    a beta node with the first and second condition elements as left and
 *    right inputs respectively. The next condition element will have a new
 *    branch created for it (assume no structural similarity in the condition
 *    elements here) and will serve as the right input to the next beta node
 *    created to join the new condition element with the previously joined
 *    condition elements. Thus the previous beta node will serve as the left
 *    input of the new beta node. Successive condition elements are joined
 *    in this fashion.
 *
 *    From this model, we see that the right input of a beta node is always
 *    a node which represents a single condition element and that the left
 *    input may be either a node link representing a single condition element
 *    (i.e., the first condition element of a production) or a link
 *    representing several joined condition elements (i.e., the predecessor
 *    node is another beta node).
 *
 *    From this we can also see that beta nodes will always serve as left
 *    inputs to succeeding beta nodes or as the single input to a P_NODE -
 *    the 1-input terminating node that represents the instantiation of a
 *    production. And thus, beta nodes are never right inputs to another
 *    beta node (although they might be right or left brothers to another
 *    beta node which is something entirely different).
 *
 *    There are 2 types of beta nodes - AND_NODEs and NOT_NODEs. The
 *    AND_NODE and  NOT_NODE are used to join a condition element or
 *    negated condition element respectively as the right input with the
 *    network branch at its left input which is either the first condition
 *    element of a production or a link from a prvious beta node that has
 *    joined the earlier condition elements in the production. Also
 *    associated with the beta nodes are left and right node memories
 *    which hold the working memory element tokens passed through the
 *    network. Each of the elements of a right memory will always represent
 *    a single working memory element that matches the condition element
 *    represented by the right input branch of the network. Each of the
 *    elements of the left memory, on the other hand, consists of a list of
 *    working memory elements where each working memory element satisfies
 *    the corresponding condition element represented in the network branch
 *    joined together to the left input.
 *
 *
 *
 *    Remarks on Right and Left Brother Chains:
 *
 *    For 1-input nodes, the left-brother link is used exclusively and the
 *    right-brother link is never used. This is strictly a matter of
 *    convention that we have chosen since a 1-input node needs only one
 *    chain to represent the connection to its single input from the parent
 *    node above it.
 *
 *    For the 2-input beta nodes,the right-brother link is used for linking
 *    beta nodes that take the same right input token. By our convention
 *    above, the right-brother chain has only beta nodes in it and the first
 *    beta node in the chain will have an RI_NODE as the parent to its right
 *    input. As mentioned earlier, the RI_NODE serves as a distinguished node
 *    that tells us when an input to a beta node is a right input and thus
 *    allows us to correctly choose the right-brother link when we add
 *    brother nodes to this chain of beta nodes. The beta nodes in a
 *    right-brother chain will differ in the following ways:
 *        - the nodes may be of different type: AND_NODE or NOT_NODE.
 *        - the nodes may have different variable bindings or tests to perform
 *          in order to satisfy the token(s) joined in from the left input.
 *        - the left inputs may be different.
 *
 *    Left-brother chains, on the other hand, may contain all beta nodes, all
 *    1-input nodes, or a mixture of both. The 1-input nodes in a chain
 *    indicate that they share the same parent (i.e., the same network branch)
 *    as input while the beta nodes in the chain share the same parent as their
 *    left inputs. Beta nodes and 1-input nodes will be mixed in the same chain
 *    only whenever the parent node is a simple 1-input node and thus
 *    represents, for the beta nodes, the first condition element in a
 *    production (since all left inputs to subsequent beta nodes for a given
 *    production have a previous beta node as parent). The 1-input nodes in
 *    a mixed chain serve to extend the network branch of a condition element
 *    that has the same initial branch represented by the parent node (which,
 *    as stated above, represents the first condition element in some
 *    production). The 1-input node in the chain thus represents either the
 *    next attribute term for some longer (though structurally similar
 *    condition element) or, in the case that it's an RI_NODE, represents that
 *    this same condition element is a right input to a beta node below in the
 *    network as well as a left input to beta nodes in the mixed chain.
 *
 *
 *
 *    Strategy for Building the Network:
 *
 *    The correct strategy for building the network in order to use structural
 *    similarity to share network branches and nodes (and thus reduce the size
 *    of the network) is based on a process of looking for an equivalent node
 *    already in the network that represents the node we need to add. The
 *    equivalence is based on both function and position of the node in the
 *    network.
 *
 *    The basic strategy is to look for an equivalent node in the proper
 *    brother chain to represent our new node or else just add a new node
 *    in the network at the proper place. For adding 1-input nodes, we simply
 *    check the parent node's son-link. If the son-link is not NULL, we look
 *    at the son and all its left brothers to see if an equivalent node
 *    exists. If no equivalent node is found, we simply add the new node as
 *    the son of the parent and fix the new node's left link to point to any
 *    son-chain previously there. In either case, the new node or the
 *    equivalent one, if found, becomes the new parent node. For each
 *    condition element, we begin with the ROOT_NODE as the first parent
 *    node. Thus all the sons of the ROOT_NODE are linked in a left-brother
 *    chain and each brother represents the first term of a condition element
 *    in one of the productions in the production system. Actually,this last
 *    statement is not quite true. It is possible that some of the nodes in
 *    the ROOT_NODE's son-list are also RI_NODEs and beta nodes. This will
 *    occur whenever the first condition element in a production has no
 *    1-input nodes in the network. An example condition element would be
 *    (^a1 <x> ^a2 <y>). The result is that the first beta node for this
 *    production will be in the ROOT_NODE's son-list. If any subsequent
 *    condition element in this production is of the same form (i.e., has
 *    only variable bindings in its terms), then the RI_NODE which joins it
 *    as the right input to the network branch for this production will also
 *    be in the son-list of the ROOT_NODE.
 *
 *    Adding 2-input nodes to the network is a little more complicated. We
 *    first have to add an RI_NODE to the right input branch (or find one
 *    already there). Then we have to look for a beta node that has the same
 *    left and right inputs as well as the same variable bindings and tests.
 *    This involves looking through left-brother and right-brother chains
 *    and finding a common intersection. We perform the search as described
 *    below.
 *
 *    We take the parent nodes of the left and right (an RI_NODE) inputs
 *    to our new beta node and check their son-links. If either is NULL,
 *    then no equivalent node exists. Otherwise, we look at the son of the
 *    parent from the left and all his left brothers for an equivalent beta
 *    node. If we find one, then we stop and look at the son of the parent
 *    (RI_NODE) from the right and all his right brothers to see if the beta
 *    node we found in the left chain is also in the right chain. If it is,
 *    then we have found our equivalent node. If not, then we go back to the
 *    left chain and continue searching for another equivalent beta node.
 *    (Remember that the left-brother chain may have several beta nodes that
 *    are equivalent in every respect except for their right inputs.) If we
 *    find another, we check the right chain again in the same way. We
 *    continue in this way until we've searched through the whole left chain
 *    or have found the equivalent beta node. If we find an equivalent beta
 *    node, we simply set a pointer to it so we can use it as the left input
 *    to the next beta node or as the input to a subsequent P_NODE. If,
 *    instead, we must add a new beta node, we set up the same pointer for
 *    the same reasons after we have linked the new beta node into the
 *    network. For any case in which we must add a new beta node, it would
 *    be linked into the network using the following rules:
 *        - if the left parent node's son-link is NULL, simply make it point
 *          to the new beta node; otherwise, set the new beta node's
 *          left-brother link to point to the parent node's son and then
 *          change the parent's son-link to point to the new beta node.
 *        - for the right side, use the above rule with the word "right"
 *          substituted for "left".
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.3 Dirk Kalp
 *                   Add ParaOPS5 4.4 bug fix for multiple output files.
 *    25 Jun 89 V4.0 Dirk Kalp
 *                   Update with ParaOPS5 4.3.
 *    12 May 89 V2.0 Dirk Kalp
 *                   Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    29 Sep 89 V4.4 Dirk Kalp
 *                   Fix up test on "g_pcount" in "cmp_production" so multiple output
 *                   files are broken correctly according to "g_max_rules_in_rete".
 *    25 Jun 89 V4.3 Dirk Kalp
 *                   Added missing "/" to comment in "init_p". It had caused
 *                   call to "init_bvar" to be commented out resulting in bug
 *                   in the OPS "bind" action.
 *                   Change all compiler generated routine names to have
 *                   "ops_" prefix. This is to ensure no conflicts with
 *                   system or user routine names when user programs are
 *                   linked.
 *    24 Oct 88 V4.0 Dirk Kalp
 *                   Release of ParaOPS5 Version 4.0.
 *    21 Oct 88 V3.1 Dirk Kalp
 *                   Reset "have_cmprhs" in "cmp_production" so that data
 *                   segment pseudo ops get written at head of threaded code
 *                   for a new output file.
 *    13 Aug 88 V3.0 Dirk Kalp
 *                   No changes.
 *    25 May 88 V2.0 Dirk Kalp
 *                   Updated to consolidate Vax and Encore versions.
 *    12 Sep 86      Dirk Kalp
 *    22 Aug 86      Dirk Kalp
 *    19 Aug 86
 *     9 Jul 86
 *    14 May 86
 *    12 May 86
 *     1 May 86
 *    29 Apr 86
 *    14 MAR 86
 *    27 Feb 86
 *    26 Feb 86
 *    25 Feb 86
 *    24 Feb 86
 *    20 Feb 86
 *    19 Feb 86
 *    18 Feb 86
 *    17 Feb 86
 *     3 Feb 86 V1.0  Dirk Kalp
 *                    Put together from previous version created by
 *                    Quei-Len Lee.
 *
 * Copyright (c) 1986, 1987, 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/

 
   
#include "defs.h"


/* Imported Routines:
 *    From ops5.c:
 *       opserror
 *       opswarn
 *       putstring
 *       puttab
 *       puteol
 *       bomb_out
 *
 *    From literal.c:
 *       lookup_sym_addr
 *       sym_addr
 *       new_pname
 *       newname
 *       strcompare
 *       assign_scalars
 *
 *    From cmprhs.c:
 *       init_bvar
 *       varnum
 *
 *    From gencode.c:
 *       newinst
 *       codeouts
 *       codeouti
 *       codeoutl
 *       codeoutf
 *       symoffset
 *
 *    From printops.c:
 *       printnet
 *       printvar
 *       printsym1
 *       write_symtab
 *
 *    From system:
 *       malloc
 *       free
 */

/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern symptr  lookup_sym_addr();    /* Imported from literal.c. */
extern symptr  sym_addr();           /* Imported from literal.c. */
extern symptr  new_pname();          /* Imported from literal.c. */
extern string  newname();            /* Imported from literal.c. */    
extern string  strcompare();         /* Imported from literal.c. */    



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
boolean      dup_anyatm();
anyatm_ptr   newanyatm();
nodeptr      newnode();
nodeptr      alloc_node();
t_nodeptr    new_t();
any_nodeptr  new_any();
p_nodeptr    new_p();
beta_nodeptr new_beta();
varptr       find_equiv_varname();
varptr       newvar();
tests_ptr    add_test();
tests_ptr    newtest();
nodeptr      link_left();
nodeptr      find_equiv_1input_node();
boolean      equiv_1input();
boolean      equiv_atmlists();
nodeptr      link_both();
nodeptr      find_equiv_beta_node();
boolean      equiv_beta();
boolean      equiv_testlists();
plist_ptr    new_prod();





init_cmp()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize global variables before the parse and compilation of the
 *    productions are begun.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The parser has not yet begun.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "main" in "ops5.c".
 *
 *-------------------------------------------------------------------------*/
{

    g_offset     = 0;
    g_opssym     = NULL;
    g_lastsym    = NULL;

    g_pcount     = 0;
    ProductionList = NULL;

    g_cevar      = NULL;
    g_var        = NULL;
    g_curvar     = NULL;
    g_bvar       = NULL;
    g_cbvar      = NULL;

    freevar      = NULL;
    freenode     = NULL;
    freetnode    = NULL;
    freepnode    = NULL;
    freebetanode = NULL;
    freeanynode  = NULL;
    free_anyatm  = NULL;
    free_test    = NULL;
    
    have_compiled_p = FALSE;
    have_cmprhs     = FALSE;
    first_inst   = TRUE;
}




make_bottom_node()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called just before compilation of the first production
 *    begins. It creates the ROOT_NODE as the first node in the Rete network
 *    to be constructed for the set of productions. The global, g_root, is
 *    assigned to permanently point to this root node in the network.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The OPS5 declarations have just been processed and the beginning of
 *    the first production has just been recognized by the parser.
 *
 * Calls:
 *    "newnode" in this module.
 *
 * Called by:
 *    "cmp_production" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   g_root = newnode(ROOT_NODE,0,0,0,0);  
}




cmp_production(pname)
   str_arr pname;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Set up to compile the next production. If this is first production,
 *    then we also perform a few initial actions such as creating the
 *    root node for the network and assigning bindings to attributes.
 *
 * Parameters:
 *    pname - the string name of the production supplied by the parser from
 *            the input and stored in the YACC value stack.
 *
 * Environment:
 *    All the declarations have been processed by the parser and the symbol
 *    table has been built prior to the first call of this routine.
 *
 * Calls:
 *    "make_bottom_node" and "init_p" in this module
 *    "assign_scalars" and "new_pname" in "literal.c"
 *    "opserror", "puteol", and "putstring" in "ops5.c"
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   symptr sym;

   if (csw.errcnt)
     {
      opserror("error found in declarations ...exit"); puteol();
      bomb_out();
     }

   if (have_compiled_p == FALSE)
     {
      init_gen_rete();
      make_bottom_node();      /* Make the root node. */
      if (debug)  { printf("Before assign:\n"); write_symtab(); }
      assign_scalars();        /* Assign bindings to attributes. */
      if (debug)  { printf("After assign:\n"); write_symtab(); }
      have_compiled_p = TRUE;  /* Set flag so we do the above only once. */
     }

   if ( (g_pcount > 0) && (((g_pcount) % g_max_rules_in_rete) == 0))
     {
      gen_rete(TRUE);        /* TRUE indicates that from cmp_prod fn */
      make_bottom_node();    /* Make the root node. */
      have_cmprhs = FALSE;   /* Reset so that .data section pseudo op gets written to new file. */
     }

   init_p();                    /* Init globals for a new production. */
   sym = new_pname(pname);      /* Put pname in the symbol table. */
   g_pname = sym->symname;      /* Put production name in global. */
   g_prod = new_prod(sym);      /* Set up to collect list of beta nodes for this prod. */

   putstring(g_pname);
   if ((g_pcount%10) == 0) putstring("\n");
}



init_p()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Prepare to compile a new production by initializing global variables
 *    and data structures.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "init_bvar" in "cmprhs.c".
 *
 * Called by:
 *    "cmp_production" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   flag_rhs       = FALSE;  /* Haven't started on the RHS yet. */
   g_pcount++;              /* Count 1 more production. */
   g_cecount      = 0;      /* No ce's encountered yet in this production. */
   g_lastbranch   = NULL;   /* No network branches built yet for this */
                            /* production.                            */
   g_betalev      = 1;      /* Depth of beta nodes always starts at 1. */
   g_featurecount = 0;      /* We will count total number of attributes */
                            /* in the production.                       */

   init_bvar();              /* Init globals associated with RHS variables. */
}



cmp_p()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Finish the compilation of the production. Allocate a new node for the
 *    network. Unless this production is identical (ridiculous but possible)
 *    to a previous production, we need a new node to represent it. (Almost
 *    as ridiculous would be a production which had the same network structure
 *    as a previous one but with a different p_name - in which case we would
 *    still need a new P_NODE and would link it by its left-brother link to
 *    its twin.) In this final step in compiling the production, we store into
 *    the nodeinfo record the p_name and a count of the features (attribute
 *    tests against constants and variables in the condition elements).
 *    Finally, we link the P_NODE into the network and then free storage
 *    in the variable lists built for the production.
 *
 *    Note: The case where the LHS of a production has no nodes for attributes
 *          in its condition elements (eg.,((^a <x> ^b <y>) (^a <y> ^b <x>)) )
 *          will be handled correctly by "cmp_ce" and result in an RI_NODE and
 *          beta node as sons of the ROOT_NODE and the P_NODE as son of the
 *          beta node. Also the degenerate case of an empty LHS will also be
 *          handled correctly with a P_NODE being attached as a son of the
 *          ROOT_NODE. Such a production would be instantiated each time a new
 *          or modified working memory element is added to the system. Thus we
 *          see that while the sons of the ROOT_NODE are normally expected to
 *          be T_NODEs and ANY_NODEs, we may also expect to see sons that are
 *          RI_NODEs, beta nodes, or P_NODEs.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    All the condition elements for this production have been compiled and
 *    installed into the network with condition elements joined by the
 *    appropriate beta nodes.
 *
 * Calls:
 *    "newnode", "link_new_1input_node", and "free_var" in his module.
 *    "putstring" in "ops5.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr   new;

   fprintf(fp_sub, "};\n");    /* Write end of line for initialized array of threaded code. */

   new = newnode(P_NODE,0,0,0,0);   /* Get a new P_NODE and set pname */
                                    /* and feature count.             */

   /* Put production with its beta nodes onto list with other productions.
    */
   g_prod->next = ProductionList;
   ProductionList = g_prod;

   link_new_1input_node(new);       /* Link it into network. */
   putstring("*");

   /* Now do housecleaning of data structures linked into the variable
    * lists for the condition elements built by the compilation of the
    * production. These data structures are no longer needed since
    * we've built a representation of them in the node network in compiling
    * the production.
    */
   free_var(GVAR);     /* Free the variable list. */
   free_var(CEVAR);    /* Free the condition element variable list. */
   free_var(BVAR);     /* Free the variable list for BIND in RHS. */
   free_var(CBVAR);    /* Free the variable list for CBIND in RHS. */
}




init_ce()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Prepare to compile the next condition element in the current production.
 *    We compile the condition element by setting a global pointer, g_last, to
 *    the root of the network and attempt to find a condition element already
 *    represented in the network that is identical (structurally if not
 *    literally) to this condition element or to find one that at least shares
 *    some initial similarity. Our attempt here to take advantage of
 *    structural similarity is restricted only to condition elements that
 *    begin with the "same" terms. As soon as the condition element differs
 *    from any already in the network, we diverge in the construction of the
 *    network branch for the new condition element by forming a left-brother
 *    link at the node of difference and proceed to complete the network
 *    branch for the condition element by building new nodes for the remaining
 *    terms. Thus if 2 condition elements are identical except for their third
 *    terms, they will share network nodes only for their first 2 terms and
 *    diverge on different branches afterwards.
 *
 *    If the condition element has no "similar" term as any other condition
 *    element already in the network, then the new condition element is linked
 *    into the network by creating a new node for its first term and setting
 *    the son-link of the root node to point to it. The left-brother link of
 *    the new node is then set to point to what the root formerly pointed to
 *    (which, except for the unusual cases referred to in "cmp_p", should be
 *    the first term of some condition element previously encountered in this
 *    production or an earlier compiled production). In general, the root
 *    points to the first term of some condition element; and the left-brother
 *    of that term points to the first term of another condition element; and
 *    the left-brother of that term points to the first term of ....  Thus to
 *    properly search through the network for a "similar" condition element,
 *    we must take care to examine the left-brother chain at the point of
 *    divergence to see if any other previously compiled condition element
 *    which also diverged at this point might indeed provide a match for the
 *    current condition element. For example, look at ( (a b c d) (a b x y)
 *    (a b c m n) ).
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   g_cecount++;          /* Count 1 more ce in this production. */

   /* Mark what kind of ce this is. */
   if (g_flag_negce)
      ce_type[g_cecount] = NEGATIVE_CE;
   else
      ce_type[g_cecount] = POSITIVE_CE;

   g_last   = g_root;    /* Point to root where we begin our search for a   */
                         /* ce already in the network with same first term. */
   g_subnum = 0;         /* Init the subelement number index of the wme. */
   /* g_curvar = NULL;  Not needed since this is a side-effect of cmp_beta.*/

   if (debugnet)   printf("init_ce lastnode : %x\n", g_last);
}



set_flag_negce()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Just set a flag showing that the parser found the condition element
 *    to be compiled next is a negated condition element.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   g_flag_negce = TRUE;
}



cmp_ce()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Finish compilation of the current condition element by linking it into
 *    the network as the right input of a new beta node. The left input to
 *    the new beta node will be the last network branch formed by the
 *    conjunction of the earlier condition elements in the current production.
 *    This last branch is pointed to by the global, g_lastbranch, and will
 *    (usually) be the beta node added the last time "cmp_ce" was called for
 *    this production. G_lastbranch will be nil when the first condition
 *    element (actually the first non-empty ce if we allow for the degenerate
 *    case) of the production is compiled. After "cmp_ce" has been called for
 *    this first ce, g_lastbranch will point to the last 1-input node in the
 *    network branch for that ce. Then when "cmp_ce" is called for the second
 *    ce, the first and second ce's will be joined by a beta node. On the
 *    subsequent calls to "cmp_ce" for this production, g_lastbranch will
 *    always point to the latest beta node added to form the new branch. Note
 *    again that adding a node may simply mean finding an existing node that
 *    has the same function and position in the network as the required node.
 *    Also note that we haven't discussed above the special cases where the
 *    first ce in a production has no network branch beyond the root node. An
 *    example here is ( (^a <x> ^b <y>) (......... ) ). In this case,
 *    g_lastbranch will point to the root node after the first call to
 *    "cmp_ce" since the first ce has no 1-input nodes associated with it.
 *    Another interesting case with respect to node connections is given by
 *    ( (^a <x> ^b <y>) (^c <m> ^d <n>) (......) ). Here we get a beta node
 *    whose left input is from the root node and whose right input is an
 *    RI_NODE (as usual) whose input comes from the root node.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    G-lastbranch points to the last network branch formed by conjunction of
 *    the earlier condition elements in this production. G_last points to the
 *    last node in the network representing the last term in the condition
 *    element currently being compiled. G-curvar points to the list of the
 *    variables encountered in this condition element.
 *
 * Calls:
 *    "cmp_nobeta", "cmp_and", and "cmp_not" in this module.
 *    "printvar" in "printops.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   if (debugnet)
     {
      printf("g_curvar: \n");
      if (g_curvar)  printvar(g_curvar);
      printf("g_var: \n");
      if (g_var)  printvar(g_var);
      printf("\n");
     }

   /* If the condition element is empty, we simply exit. The ce is empty if
    * it has no nodes beyond the root node and no variables were encountered.
    */
   if ( (g_last == g_root) && (g_curvar == NULL) )  return;
   
   if (g_lastbranch == NULL)
      cmp_nobeta();          /* i.e., this is first ce of this production. */
   else if (g_flag_negce)
      cmp_not();
   else
      cmp_and();
      
   /* free_var(CURVAR); */  /* Effectively done in cmp_beta. */
} /* end cmp_ce */




init_lhsterm()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Called by the parser just before beginning to reduce the next attribute
 *    term (the condition element name is also treated like any attribute) in
 *    the current condition element. The preparation here is to initialize
 *    and set defaults.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   g_curtest = TEST_EQ;   /* TEST_EQ is the default if no explicit  */
                          /* predicate appears in the parsed input. */
   g_subnum++;            /* Default is just to increment the subelement */
                          /* number in case no explicit subelement index */
                          /* is given in the parsed input.               */
   g_anyatm  = NULL;      /* This will hold the list of constants atoms  */
                          /* if the lhs-value is given as a disjunction. */
}



init_test()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Called by the parser just before beginning to reduce the next restriction
 *    term (the condition element name is also treated like any attribute) in
 *    the current condition element. The preparation here is to initialize
 *    and set defaults.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   g_curtest = TEST_EQ;   /* TEST_EQ is the default if no explicit  */
                          /* predicate appears in the parsed input. */
}




cmp_tab(att)
   str_arr att;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called by the parser upon recognizing an attribute
 *    name in the input. Here we convert the name into an index for the
 *    field of a working memory element that the name has been bound to
 *    from the earlier processing of the Literal and Literalize declarations
 *    in the OPS5 program. The index number bound to this attribute name
 *    is found in the symbol table record associated with this name. We must
 *    look up the name 'att' in the symbol table and see if it is registered
 *    as an attribute name (i.e., is sym->is_att == TRUE). If so, check its
 *    binding for a proper value and store it in the global, 'g_subnum', for
 *    later use in the processing of this attribute. If 'att' is not in the
 *    symbol table or is not registered as an attribute name, then it must
 *    never have appeared in a Literalize or Literal declaration, so report
 *    an error.
 *
 * Parameters:
 *    att - the attribute name string stored in YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "lookup_sym_addr" in "literal.c".
 *    "opserror", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *---------------------------------------------------------------------------
 *
 * Abstract2:
 *    This routine is also used to process the attribute names occurring
 *    in the RHS (right hand side) of a production. In addition to
 *    performing the above mentioned evaluation of the index, we also
 *    must generate the threaded code for the RHS to handle the attribute.
 *
 * Parameters:
 *    att - the attribute name string stored in YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "lookup_sym_addr" in "literal.c".
 *    "opserror", "putstring", and "puteol" in "ops5.c".
 *    "newinst", "codeouts", and "codeouti" in "gencode.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   symptr sym;

              
   /*----------------------------------------------------------------------*/
   /*                 LHS and RHS common code:                             */
   /*----------------------------------------------------------------------*/
   
   sym = lookup_sym_addr(att);
   
   /* If the name was in the symbol table, then see if its registered as
    * an attribute name and get its index binding and check it.
    */
   if (sym)
     {
      if (sym->is_att)
        {
         g_subnum = sym->opsbind;
         if ((g_subnum < FIRST_OPSBIND) || (g_subnum > MAX_OPSBIND))
           {
            opserror("attribute name has an invalid index binding.");
            putstring(att);  puteol();
            if (debug)  printsym1(sym);
            return;
           }
        }
      else
        {
         opserror("Name not registered as an attribute name.");
         putstring(att);  puteol();
         return;
        }
     }
   else
     {
      opserror("no such attribute name ever declared.");
      putstring(att);  puteol();
      return;
     }
     
       
   /*----------------------------------------------------------------------*/
   /*                            RHS only:                                 */
   /*----------------------------------------------------------------------*/

   if (flag_rhs)
     {
      newinst();  codeouts("_ops_tab");
      newinst();  codeouti(g_subnum);
     }
} /* end cmp_tab */



cmp_tabno(att)
   int att;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called by the parser upon recognizing a number in
 *    the input that represents an explicit attribute index to a working
 *    memory element (eg., ^3 rock). Here we range check the index and
 *    store it in the global, g_subnum, for later use in the processing
 *    of this attribute in the LHS.
 *
 * Parameters:
 *    att - an explicit attribute index number.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *---------------------------------------------------------------------------
 *
 * Abstract2:
 *    This routine is also used to process the attribute index occurring
 *    in the RHS (right hand side) of a production. In addition to
 *    performing the above range check of the index, we also must generate
 *    the threaded code for the RHS to handle the attribute.
 *
 * Parameters:
 *    att - an explicit attribute index number.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror", "putstring", and "puteol" in "ops5.c".
 *    "newinst", "codeouts", and "codeouti" in "gencode.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
     
       
   /*----------------------------------------------------------------------*/
   /*                 LHS and RHS common code:                             */
   /*----------------------------------------------------------------------*/

   if ((att < FIRST_OPSBIND) || (att > MAX_OPSBIND)) 
     {
      opserror("tab must be a number >= 1 or <= 127. in :"); 
      putstring(g_pname); puteol();
      return;
     }

              
   /*----------------------------------------------------------------------*/
   /*                 LHS and RHS specific code:                           */
   /*----------------------------------------------------------------------*/
   
   if (flag_rhs)
     {
      newinst();  codeouts("_ops_tab");
      newinst();  codeouti(att);
     }
   else
      g_subnum = att;
      
} /* end cmp_tabno */




cmp_any()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine completes the processing of an lhs-value given by a
 *    disjunction of literals. The literals in the disjunction have been
 *    recognized and assembled on a list headed by the global, g_anyatm.
 *    Here we create a new 1-input node to perform the test for the
 *    disjunction and link the node into the network.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newnode" and "link_new_1input_node" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   nodeptr new;

   new = newnode(ANY_NODE, 0, 0, 0, 0);   /* Gets list form g_anyatm. */
   link_new_1input_node(new);
}



add_anystr(str)
   str_arr str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Process the next literal string in the sequence of constants from a
 *    disjunction. The literal is put into the symbol table and also on a
 *    list headed by g_anyatm. The literal is put at the head of that list
 *    but only after first ensuring that the literal is not already on the
 *    list. This elimination of duplicates will make it easier when looking
 *    for an equivalent ANY_NODE at some later time.
 *
 * Parameters:
 *    str - the string name of a literal atom in the disjunction. Storage
 *          for the string is in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "dup_anyatm" and "newanyatm" in this module.
 *    "sym_addr" in "literal.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   anyatm_ptr new;
   symptr     sym;
   
   /* Put the literal atom in the symbol table and return its
    * symbol table address. Note: Str may already be in the
    * symbol table.
    */
   sym = sym_addr(str);

   /* Build a record for the new atom.
    */
   new = newanyatm();
   new->is_int       = FALSE;
   new->aval.any_ptr = sym;

   
   /* Now check if the literal atom is already in our list of
    * literals for this disjunction; i.e., we want to remove
    * duplicates such as in << A B C B D >>.
    */
   if (!dup_anyatm(new))
     {
      new->next = g_anyatm;
      g_anyatm  = new;
     }
   else
     {
      /* The atom is already on the 'g_anyatm' list so just return
       * the record to the free list and exit.
       */
      new->next   = free_anyatm;
      free_anyatm = new;
     }
} /* end add_anystr */



add_anynum(num)
   int num;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Process the next constant integer in the sequence of constants from a
 *    disjunction. The atom is put onto a list headed by g_anyatm. The atom
 *    is put at the head of that list but only after first ensuring that the
 *    atom is not already on the list. This elimination of duplicates will
 *    make it easier when looking for an equivalent ANY_NODE at some later
 *    time.
 *
 * Parameters:
 *    num - the integer value of a constant atom in the disjunction.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "dup_anyatm" and "newanyatm" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   anyatm_ptr new;
   
   /* Build a record for the new atom.
    */
   new = newanyatm();
   new->is_int       = TRUE;
   new->aval.any_int = num;

   
   /* Now check if the integer atom is already in our list of
    * atoms for this disjunction; i.e., we want to remove
    * duplicates such as in << A 17 C 17 D >>.
    */
   if (!dup_anyatm(new))
     {
      new->next = g_anyatm;
      g_anyatm  = new;
     }
   else
     {
      /* The atom is already on the 'g_anyatm' list so just return
       * the record to the free list and exit.
       */
      new->next   = free_anyatm;
      free_anyatm = new;
     }
} /* end add_anynum */



boolean
dup_anyatm(new)
   anyatm_ptr new;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Look to see if an atom is already on the list of atoms for the
 *    disjunction currently being compiled. The list is headed by
 *    g_anyatm.
 *    Note: For atoms that are string constants, we only need to compare
 *          the symbol table record addresses for the literals and not the
 *          actual literal strings since the symbol table does not store
 *          duplicate names (re: "sym_addr").
 *
 * Parameters:
 *    new - pointer to the anyatm record holding the atom.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    True if this atom is already in the 'g_anyatm' list.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "add_anystr" and "add_anynum" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   anyatm_ptr ptr;

   ptr = g_anyatm;   /* Point to the disjunction list. */
   while (ptr)
     {
      if (new->is_int && ptr->is_int)
        {
         if (new->aval.any_int == ptr->aval.any_int)
            return(TRUE);   /* Found atom in our current disjunction list. */
        }

      if ((!new->is_int) && (!ptr->is_int))
        {
         if (new->aval.any_ptr == ptr->aval.any_ptr)
            return(TRUE);   /* Found atom in our current disjunction list. */
        }

      ptr = ptr->next;
     }
   
   return(FALSE);   /* Atom was not in the g_anyatm list. */
}



anyatm_ptr
newanyatm()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate an anyatm_rec record to hold a new atom for a disjunction.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the anyatm_rec allocated.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *    "malloc" system call.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   anyatm_ptr new;

   if (free_anyatm)
     {
      new = free_anyatm;
      free_anyatm = free_anyatm->next;
      return(new);
     }
     
   if ((new = (anyatm_rec *)malloc(sizeof(anyatm_rec))) == NULL) 
     {
      opserror("newanyatm:  no more memory ...");
      puteol();
      bomb_out();
     }
   else
      return(new);
}




note_test(test)
  scalar test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Record the predicate that the parser discovered in the input relating
 *    the attribute to its value.
 *
 * Parameters:
 *    test - the predicate relating the attribute to its value.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
    g_curtest = test;
}



cmp_constant(atm)
   str_arr atm;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called by the parser when it recognizes a symbolic
 *    string constant as the lhs-value for an attribute in a condition
 *    element. We have already recognized the predicate associated with
 *    the attribute and its string value and have it stored in the global,
 *    g_curtest. Here we must check that that predicate is valid to use
 *    with a string constant and then construct and link into the network
 *    branch for the current condition element a new T_NODE which performs
 *    the test of the attribute against the string constant. We also put
 *    the string constant into the symbol table.
 *
 * Parameters:
 *    atm - the string constant stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "sym_addr" in "literal.c".
 *    "newnode" and "link_new_1nput_node" in this module.
 *    "opswarn", "puttab", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *---------------------------------------------------------------------------
 *
 * Abstract2:
 *    This routine is also used to process symbolic string constants that
 *    appear in the RHS (right hand side) of a production. It generates the
 *    threaded code to handle the symbolic constant.
 *
 * Parameters:
 *    atm - the string constant stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "sym_addr" in "literal.c".
 *    "newinst", "codeouts", and "symoffset" in "gencode.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   nodeptr new;
   opsval val;
   symptr sym;

       
   /*----------------------------------------------------------------------*/
   /*                     LHS and RHS common code:                            */
   /*----------------------------------------------------------------------*/
   
   /* Retrieve or create a symbol table entry for the string constant.
    */
   sym = sym_addr(atm);



   /*----------------------------------------------------------------------*/
   /*                            RHS only:                                 */
   /*----------------------------------------------------------------------*/
   
   if (flag_rhs)
     {
      newinst();  codeouts("_ops_symcon");
      newinst();  codeoutopssym(sym);
      return;
     }

       
   /*----------------------------------------------------------------------*/
   /*                            LHS only:                                 */
   /*----------------------------------------------------------------------*/

   if ((g_curtest == TEST_EQ) || (g_curtest == TEST_NE) || 
       (g_curtest == TEST_XX))
     {
      val.str = sym->symname;
      new = newnode(T_NODE, g_curtest, ATM, val, 0);
      link_new_1input_node(new);
     }
   else
     {
      opserror("non-numeric constant after numeric predicate :");
      puttab();  putstring(g_pname);  puteol();
     }
} /* end cmp_constant */



cmp_fixnum(fixnum)
   int fixnum;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called by the parser when it recognizes an integer
 *    number as the lhs-value for an attribute in a condition element. We
 *    have already recognized the predicate associated with the attribute
 *    and its numeric value and have it stored in the global, g_curtest.
 *    Here we must construct and link into the network branch for the
 *    current condition element a new T_NODE which performs the test of
 *    the attribute against the numeric constant.
 *
 * Parameters:
 *    fixnum - the integer number passed by the parser.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newnode" and "link_new_1nput_node" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *---------------------------------------------------------------------------
 *
 * Abstract2:
 *    This routine is also used to process integer number constants that
 *    appear in the RHS (right hand side) of a production. It generates the
 *    threaded code to handle the numeric constant.
 *
 * Parameters:
 *    fixnum - the integer number passed by the parser.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newinst", "codeouts", "codeoutl" in "gencode.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   nodeptr new; 
   opsval  val;

       
   /*----------------------------------------------------------------------*/
   /*                                 RHS:                                 */
   /*----------------------------------------------------------------------*/

   if (flag_rhs)
     {
      newinst();  codeouts("_ops_fixcon"); 
      newinst();  codeoutl(fixnum);
      return;
     }

       
   /*----------------------------------------------------------------------*/
   /*                                 LHS:                                 */
   /*----------------------------------------------------------------------*/
 
   val.num = fixnum;
   new = newnode(T_NODE, g_curtest, NUM, val, 0);
   link_new_1input_node(new);    

} /* end cmp_fixnum */


   

nodeptr
newnode(ntype, ntest, ttype, field1, field2)
   scalar ntype, ntest, ttype; 
   opsval field1, field2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Create a new 1 or 2-input node, initialize it, and set its
 *    characteristics based upon the parameters as well as certain global
 *    variables.
 *
 * Parameters:
 *    ntype  - the node type (ROOT, RI, T, ANY, AND, NOT, P).
 *    ntest  - the predicate type for T_NODE (TEST_EQ, _NE, _LT, _LE, _GT,
 *             _GE, _XX).
 *    ttype  - the type of the attribute value for T_NODE (NUM, ATM, S).
 *                 Note: S => value is bound to internal variable.
 *    field1 - used only for T_NODE and depends upon the ttype:
 *                 NUM => integer
 *                 ATM => string (already in the symbol table)
 *                 S   => the subelement index of the attribute that the
 *                        T_NODE represents.
 *    field2 - used only for T_NODE of ttype=S and denotes the subelement
 *             index of the attribute compared against by the attribute
 *             represented by the T_NODE (i.e., for ttype=S, field1 and
 *             field2 index the attributes, bound by the common internal
 *             variable, which are to be compared).
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new node created.
 *
 * Calls:
 *    "alloc_node" in this module.
 *    "newname" in "literal.c".
 *    "printnet" in "printops.c".
 *
 * Called by:
 *    "cmp_p", "cmp_any", "cmp_constant", "cmp_fixnum", "cmp_flonum",
 *    "make_bottom_node", "cmp_new_eq_var, "cmp_old_eq_var" and "build_beta"
 *    in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr     new;
   t_nodeptr   tn;
   any_nodeptr any;
   p_nodeptr   pn;

   /* Allocate and init a node of the proper type. This also includes
    * a nodeinfo record for certain nodes.
    */
   new = alloc_node(ntype);

   /* Now fill in the nodeinfo record using parameters and globals.
    */
   switch (ntype)
     {
      case AND_NODE:
      case NOT_NODE:
      case RI_NODE:
      case ROOT_NODE:
         break;       /* These nodes have no nodeinfo record or else the */
                      /* caller will fill it in separately.              */
         
      case P_NODE:
         pn = new->nodeinfo.pnode;
         pn->pname   = g_pname;
	 pn->pindex  = g_pcount;
         pn->testcnt = g_featurecount;
         pn->cecount = g_cecount;
         break;
         
      case ANY_NODE:
         any = new->nodeinfo.anynode;
         any->anyatm_list = g_anyatm;
         any->snum        = g_subnum;
         break;
         
      case T_NODE:
         new->nodename = ntest + ttype;
         tn = new->nodeinfo.tnode;
         tn->pos = g_subnum;
         switch (ttype)
           {
            case NUM:
               tn->utype     = INTNUM;
               tn->value.num = field1.num;
               break;
               
            case ATM:
               tn->utype     = STRING;
               tn->value.str = field1.str;
               break;
                      
            case S:
               tn->utype      = SUBNUM;
               tn->pos        = field1.snum;  /* Can't just use g_subnum    */
                                              /* here. re: "cmp_new_eq_var" */
               tn->value.snum = field2.snum;
               break;
               
           } /* end switch */

     } /* end switch */

   if (debugnet)  {printf("Newnode:\n"); printnet(new);}
     
   return(new);
}



nodeptr
alloc_node(kind)
   scalar kind;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new node of the indicated type and initialize it. If the
 *    node requires a nodeinfo record, then allocate and init it too.
 *
 * Parameters:
 *    kind - the type of node to allocate.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the node allocated.
 *
 * Calls:
 *    "init_tnode", "init_anynode", "init_pnode", "init_betanode",
 *    "new_t", "new_any", "new_p" and "new_beta" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *    "malloc" system call.
 *
 * Called by:
 *    "newnode" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr new;
     
   new = NULL;
   
   /* If we already have a node on a free list, allocate it from there.
    * Nodes on a free list will already have nodeinfo records attached
    * as required so we just init them too.
    */
   switch (kind)
     {
      case ROOT_NODE:
      case RI_NODE: 
         if (freenode)
           {
            new = freenode;
            freenode = freenode->son;
           }
         break;
         
      case T_NODE:
         if (freetnode)
           {
            new = freetnode;
            freetnode = freetnode->son;
            init_tnode(new->nodeinfo.tnode);
           }
         break;
         
      case AND_NODE:
      case NOT_NODE:
         if (freebetanode)
           {
            new = freebetanode;
            freebetanode = freebetanode->son;
            init_betanode(new->nodeinfo.betanode);
           }
         break;
         
      case ANY_NODE:
         if (freeanynode)
           {
            new = freeanynode;
            freeanynode = freeanynode->son;
            init_anynode(new->nodeinfo.anynode);
           }
         break;
         
      case P_NODE:
         if (freepnode)
           {
            new = freepnode;
            freepnode = freepnode->son;
            init_pnode(new->nodeinfo.pnode);
           }
         break;
         
      default:
         opserror("alloc_node: internal error - no such node name "); puteol();
         bomb_out();
         break;
     } /* end switch */


   /* If we couldn't get a node from the free list, then we must allocate
    * new memory.
    */
   if (new == NULL)
     {
      if ((new = (noderec *)malloc(sizeof(noderec))) == NULL)
        {
         opserror("alloc_node: no more memory... "); puteol();
         bomb_out();
        }
      else
        {
         /* Now allocate and init a nodeinfo record where required.
          */
         switch (kind)
           {
            case T_NODE:
               new->nodeinfo.tnode = new_t();
               break;
           
            case ANY_NODE:
               new->nodeinfo.anynode = new_any();
               break;
           
            case AND_NODE:
            case NOT_NODE:
               new->nodeinfo.betanode = new_beta();
               break;
           
            case P_NODE:
               new->nodeinfo.pnode = new_p();
               break;
           } /*end switch */
        }
     }


   new->nodename = kind;    /* "newnode" will modify this for T_NODEs. */
   new->brothl   = NULL;
   new->brothr   = NULL;
   new->son      = NULL;
   
   return(new);
   
} /* end alloc_node */



init_tnode(tn)
   t_nodeptr tn;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize the nodeinfo record for a newly allocated T_NODE.
 *
 * Parameters:
 *    tn - the nodeinfo record for a T_NODE.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "alloc_node" and "new_t" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   tn->pos       = 0;
   tn->utype     = 0;
   tn->value.str = 0;   /* Init the largest variant in the union. */
}



init_anynode(any)
   any_nodeptr any;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize the nodeinfo record for a newly allocated ANY_NODE.
 *
 * Parameters:
 *    any - the nodeinfo record for a ANY_NODE.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "alloc_node" and "new_any" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   any->snum        = 0;
   any->anyatm_list = NULL;
}



init_pnode(pn)
   p_nodeptr pn;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize the nodeinfo record for a newly allocated P_NODE.
 *
 * Parameters:
 *    pn - the nodeinfo record for a P_NODE.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "alloc_node" and "new_p" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   pn->pname   = NULL;
   pn->rtn     = NULL;
   pn->testcnt = 0;
   pn->cecount = 0;
}



init_betanode(bn)
   beta_nodeptr bn;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize the nodeinfo record for a newly allocated beta node.
 *
 * Parameters:
 *    bn - the nodeinfo record for a beta node.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "alloc_node" and "new_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   bn->nodeid       = -1;
   bn->betalev      = 0;
   bn->testlist     = NULL;
   bn->right_rtn    = NULL;
   bn->left_rtn     = NULL;
   bn->testpass_rtn = NULL;
   bn->test_rtn     = NULL;
   bn->pass_rtn     = NULL;
}



t_nodeptr
new_t()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new nodeinfo record for a T_NODE and initialize it.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to a new nodeinfo record for a T_NODE.
 *
 * Calls:
 *    "malloc" system call.
 *    "init_tnode" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "alloc_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   t_nodeptr new;

   if ((new = (t_noderec *)malloc(sizeof(t_noderec))) == NULL)
     {
      opserror("new_t: no more memory... ");  puteol();
      bomb_out();
     }
   else
     {
      init_tnode(new);
      return(new);
     }
}



any_nodeptr
new_any()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new nodeinfo record for an ANY_NODE and initialize it.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to a new nodeinfo record for an ANY_NODE.
 *
 * Calls:
 *    "malloc" system call.
 *    "init_anynode" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "alloc_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   any_nodeptr new;

   if ((new = (any_noderec *)malloc(sizeof(any_noderec))) == NULL)
     {
      opserror("new_any: no more memory... ");  puteol();
      bomb_out();
     }
   else
     {
      init_anynode(new);
      return(new);
     }
}



p_nodeptr
new_p()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new nodeinfo record for a P_NODE and initialize it.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to a new nodeinfo record for a P_NODE.
 *
 * Calls:
 *    "malloc" system call.
 *    "init_pnode" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "alloc_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   p_nodeptr new;

   if ((new = (p_noderec *)malloc(sizeof(p_noderec))) == NULL)
     {
      opserror("new_pnode: no more memory... ");  puteol();
      bomb_out();
     }
   else
     {
      init_pnode(new);
      return(new);
     }
}



beta_nodeptr
new_beta()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new nodeinfo record for a beta node and initialize it.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to a new nodeinfo record for a beta node.
 *
 * Calls:
 *    "malloc" system call.
 *    "init_betanode" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "alloc_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr new;

   if ((new = (beta_noderec *)malloc(sizeof(beta_noderec))) == NULL)
     {
      opserror("new_beta: no more memory... ");  puteol();
      bomb_out();
     }
   else
     {
      init_betanode(new);
      return(new);
     }
}




free_node(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine puts a node on a list of free nodes according to the
 *    type of the node. If the node is an ANY_NODE, the list of records
 *    attached to the node to hold the literals in the disjunction is
 *    disconnected from the node and placed on a separate free list for
 *    these data structures. Similarly, if the node is a beta node, the
 *    records comprising the testlist attached to the node are also put
 *    on their own separate free list.
 *
 * Parameters:
 *    node - the node to put on a free list.
 *
 * Environment:
 *    The node being freed has no connection into the network with any
 *    other node. Thus we do not need to worry about unlinking the node
 *    from the network.
 *
 * Calls:
 *    "free_atmlist" and "free_testlist" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "link_left" and "link_both" in this module.
 *
 *-------------------------------------------------------------------------*/
{
    if (debugnet)  printf("freenode: %x \n", node);

   switch (node->nodename)
     {
      case RI_NODE:
      case ROOT_NODE:
         node->son = freenode;
         freenode  = node;
         break;
         
      case P_NODE:
         node->son = freepnode;
         freepnode = node;
         break;
         
      case ANY_NODE:
         free_atmlist(node->nodeinfo.anynode->anyatm_list);
         node->son   = freeanynode;
         freeanynode = node;
         break;
         
      case AND_NODE:
      case NOT_NODE:
         free_testlist(node->nodeinfo.betanode->testlist);
         node->son    = freebetanode;
         freebetanode = node;
         break;
         
      default:
         /* Only type left is a T_NODE. However, these have the test type
          * added into the nodename so we make a token examination here
          * to see if there is a nodeinfo record attached.
          */
         if (node->nodeinfo.tnode == NULL)
           {
            opserror("free_node: internal error..."); puteol();
            bomb_out();
           }
         node->son = freetnode;
         freetnode = node;
    }
}



free_atmlist(atmlist)
   anyatm_ptr atmlist;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Take a list of records holding the constants in a disjunction for an
 *    ANY_NODE and put them on a free list for future use.
 *
 * Parameters:
 *    atmlist - the list of records holding the constants in a disjunction.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "free_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   anyatm_ptr atm;

   while (atmlist)
     {
      /* Take an atom record from the head of the list.
       */
      atm = atmlist;
      atmlist = atmlist->next;
      
      /* Put record at head of free list.
       */
      atm->next   = free_anyatm;
      free_anyatm = atm;
     }
}



free_testlist(testlist)
   tests_ptr testlist;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Take a list of records holding the variable binding tests for a
 *    beta node and put them on a free list for future use.
 *
 * Parameters:
 *    testlist - the list of records holding the variable binding tests.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "free_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   tests_ptr t1, t2;

   t1 = testlist;
   while (t1)
     {
      t2 = t1->next;
      t1->test   = 0;
      t1->l_wme  = 0;
      t1->l_snum = 0;
      t1->r_snum = 0;
      t1->next   = free_test;
      free_test  = t1;
      t1 = t2;
     }
}




cmp_cevar(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Process a condition element variable by simply putting it on a global
 *    list, g_cevar, for the current production. The list of ce variables
 *    will be used later by the right hand side of the production. Before
 *    putting the variable on the list, look to see if it's already on the
 *    list. If so, then the variable appeared earlier associated with
 *    another ce and this new occurrence is an error.
 *
 * Parameters:
 *    var - variable name string stored in the YACC value stack.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "find_equiv_varname" and "newvar" in this module.
 *    "newname" in "literal.c".
 *    "opserror", "puttab", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{
   varptr old, new;

   old = find_equiv_varname(var, g_cevar);
   if (old)
     {
      opserror("condition element variable used twice : ");
      puttab(); putstring(var); puteol();
     }
   else
     {
      new = newvar();
      new->varname = newname(var);
      new->cenum   = g_cecount;
      new->next    = g_cevar;
      if (g_cevar)  g_cevar->prev = new;
      g_cevar = new;
     }
}



cmp_var(var)
   str_arr var;
/*---------------------------------------------------------------------------
 *
 * Abstract1:
 *    This routine processes each attribute variable as it is found in the
 *    condition element currently being compiled. The predicate associated
 *    with the variable has been previously recognized by the parser and
 *    is stored in g_curtest. Variables found in the current ce are placed
 *    on the g_curvar list. Before adding a variable to the g_curvar list,
 *    we first look through the list to see if a var of the same name
 *    exists there already. If we find one and its associated predicate is
 *    equality (TEST_EQ), then the new occurrence of the variable is not
 *    put on the g_curvar list. Instead, we build a T_NODE to add to the
 *    network branch for the current ce to represent this variable binding
 *    internal to the ce. The T_NODE will be used to perform the comparison,
 *    according to g_curtest, of the attribute bound to this latest occurrence
 *    of the variable with the attribute bound to the earlier occurrence.
 *
 *    If the variable we found on g_curvar does not have a TEST_EQ predicate,
 *    then we do 1 of 2 things depending upon the predicate associated with
 *    this new occurrence of the variable. If the new occurrence of the
 *    variable has a predicate other than TEST_EQ, we simply put it onto the
 *    g_curvar list. If the new occurrence has a TEST_EQ predicate, then we
 *    add it to the g_curvar list but only after removing every other
 *    occurrence of the variable on g_curvar and building a T_NODE for each.
 *    Each T_NODE will be used to perform the comparison, according to the
 *    predicate associated with the old occurrence, of the attribute bound
 *    to the old occurrence with the attribute bound to the latest occurrence
 *    of the variable.
 *
 *    Note: Duplicate or redundant T_NODEs may occur in the network branch
 *          for a ce since we make no attempt to detect that here or earlier
 *          in the compilation. This could occur for T_NODEs that perform
 *          internal variable binding tests as well as those that test the
 *          value of an attribute. These redundancies would result from
 *          careless programming by the user, eg. (^a > <x> .... ^a > <x>).
 *          Also we do not attempt to detect absurd constructions such as
 *          (^a < <x> ^a > <x>).
 *
 * Parameters:
 *    var - the variable name string stored in the YACC value stack.
 *
 * Environment:
 *     Nothing special.
 *
 * Calls:
 *    "find_equiv_varname", "cmp_old_eq_var", "cmp_new_eq_var", and
 *    "cmp_new_var" in this module.
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *---------------------------------------------------------------------------
 * Abstract2:
 *    This routine is also used to process the attribute variables from
 *    the RHS (right hand side) of a production. It generates the threaded
 *    code for the RHS to handle the variable.
 *
 * Parameters:
 *    var - the variable name string stored in the YACC value stack.
 *
 * Environment:
 *    Flag_rhs has been set by the parser to indicate that we are compiling
 *    a variable in the RHS of a production.
 *
 * Calls:
 *    "newinst" and "codeouts" in "gencode.c".
 *    "varnum" in "cmprhs.c".
 *
 * Called by:
 *    the parser routine "yyparse" in "y.tab.c".
 *
 *-------------------------------------------------------------------------*/
{ 
   varptr old;
       
   /*----------------------------------------------------------------------*/
   /*                                 RHS:                                 */
   /*----------------------------------------------------------------------*/
   
   if (flag_rhs)
     {
      newinst();
      codeouts("_ops_variable");
      varnum(var);
      return;
     }
     
       
   /*----------------------------------------------------------------------*/
   /*                                 LHS:                                 */
   /*----------------------------------------------------------------------*/
   
   /* First see if the variable has already been encountered in this ce,
    * i.e., is it on the g_curvar list.
    */
   if (debug || debugnet)  {printf("Cmp_var: Before find_eq: var: %s\n", var);}
   old = find_equiv_varname(var, g_curvar);
   if (debug || debugnet)  {printf("Cmp_var: After find_eq: old: %x\n", old);}
   
   if (old)
     {
      /* Found it on g_curvar list, so check associated predicate.
       */
      if (old->test == TEST_EQ)
         /* Build a T_NODE to do the comparison of the attributes bound
          * to the new and old occurrences of the variable.
          */
         cmp_old_eq_var(old, g_curtest);
      else if (g_curtest == TEST_EQ)
         /* The new variable has a TEST_EQ predicate so go off and remove
          * all other occurrences of variable from the g_curvar list,
          * building T_NODEs for each.
          */
         cmp_new_eq_var(var, old);
      else 
         /* Just put the latest occurrence of variable on the g_curvar list.
          */
         cmp_new_var(var, g_curtest);
     }
   else
     {
      if (debug || debugnet)  {printf("Cmp_var: Before Cmp_new_var: %s\n", var);}
      cmp_new_var(var, g_curtest);  /* Add new variable to g_curvar list. */
     }

} /* end cmp_var */



cmp_new_var(var, test)
   string var;
   scalar test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add to the g_curvar list a new variable associated with the current
 *    attribute in the condition element being compiled. The index of the
 *    attribute is held in the global g_subnum.
 *
 * Parameters:
 *    var  - the variable name string stored in the YACC value stack.
 *    test - the predicate associated with this variable.
 *
 * Environment:
 *    The index of the attribute associated with the variable is held
 *    in g_subnum.
 *
 * Calls:
 *    "newvar" in this module.
 *    "newname" in "literal.c".
 *
 * Called by:
 *    "cmp_var" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   varptr new;

   if (debug || debugnet)  {printf("Cmp_new_var: Begin: var %s\n", var);}
   
   new = newvar();
   if (debug || debugnet)  {printf("Cmp_new_var: After newvar: new %x\n", new);}
   new->varname = newname(var);  /* Create permanent storage for string. */
   if (debug || debugnet)  {printf("Cmp_new_var:After name : varname %s\n", new->varname);}
   new->test    = test;
   new->cenum   = g_cecount;
   new->snum    = g_subnum;
   new->next    = g_curvar;
   if (debug || debugnet)  {printf("Cmp_new_var: Before if: g_curvar %x\n", g_curvar);}
   if (g_curvar)  g_curvar->prev = new;
   g_curvar = new;

   if (debug || debugnet)  {printf("Cmp_new_var: new: %x\n", new); printvar(new);}
}



cmp_new_eq_var(var, old)
   string var;
   varptr old;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    The variable, var, associated with the current attribute in the
 *    condition element being compiled has a predicate of TEST_EQ. Add
 *    this variable to the g_curvar list but only after first removing
 *    from g_curvar all other occurrences of the variable. These other
 *    occurrences of the variable are associated with attributes
 *    encountered in the ce prior to the current attribute and each has
 *    a predicate different from TEST_EQ. We will build a T_NODE for each
 *    of these old occurrences to perform the attribute comparison specified
 *    by its predicate between it and the current occurrence of the variable.
 *    The T_NODEs are added to the network branch under construction for
 *    the current ce. 
 *
 * Parameters:
 *    var - the variable name string stored in the YACC value stack.
 *    old - a previous occurrence of the variable on the g_curvar list.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    itself recursively, if necessary.
 *    "delvar", "find_equiv_varname", "cmp_new_var", "newnode", and
 *    "link_new_1input_node" in this module.
 *
 * Called by:
 *    itself recursively, if necessary.
 *    "cmp_var" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   varptr  next;
   nodeptr new;
   opsval  val1, val2;
  
   /* Remove old variable from g_curvar list.
    * Note: Delvar does not destroy any info in "old" that we need below.
    */
   delvar(old, CURVAR);
   
   /* See if there is another occurrence on g_curvar.
    */
   next = find_equiv_varname(var, g_curvar);
   if (next)
      cmp_new_eq_var(var, next);  /* Recursive call to remove it too. */
   else
      cmp_new_var(var, TEST_EQ);  /* Finally, add the new occurrence of */
                                  /* the variable to the g_curvar list. */
   
   /* Now build the T_NODE for the old occurrence.
    */
   val1.snum = old->snum;
   val2.snum = g_subnum;
   new = newnode(T_NODE, old->test,  S, val1, val2);
   link_new_1input_node(new);
}



cmp_old_eq_var(old, test)
   varptr old;
   scalar test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Create a T_NODE that compares 2 attributes bound to the same variable
 *    in the current condition element. The old occurrence of the variable
 *    (represented by "old") has an associated predicate of TEST_EQ and is
 *    on the g_curvar list. The latest occurrence of the variable is
 *    associated with the attribute currently being compiled which is the
 *    attribute referenced by the subelement index stored in g_subnum. The
 *    predicate associated with this latest occurrence of the variable is
 *    passed here in the parameter "test".
 *
 * Parameters:
 *    old  - a variable, having a predicate of TEST_EQ, that is already on
 *           the g_curvar list.
 *    test - the predicate associated with a new occurrence of the variable.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newnode" and "link_new_1input_node" in this module.
 *
 * Called by:
 *    "cmp_var" in this module
 *
 *-------------------------------------------------------------------------*/
{ 
   nodeptr new;
   opsval  val1, val2;

   val1.snum = g_subnum;  /* Index of attribute we're building T_NODE for. */
   val2.snum = old->snum; /* Index of att that T_NODE will compare against. */
   new = newnode(T_NODE, test, S, val1, val2);
   link_new_1input_node(new);
}




varptr
find_equiv_varname(var, vlist)
   string  var;
   varptr  vlist;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Look for a variable with the same name as "var" in the variable list
 *    "vlist". Return a pointer to it, if found; otherwise return NULL.
 *
 * Parameters:
 *    var   - the string name of a variable.
 *    vlist - a list of variables.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to a variable in the list or NULL.
 *
 * Calls:
 *    "strcompare" in "literal.c".
 *
 * Called by:
 *    "cmp_cevar", "cmp_var", "cmp_new_eq_var", and "cmp_beta" in this
 *    module.
 *    "ele_num", "var_num", "bvarnum", and "cbvarnum" in "cmprhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   if (debug || debugnet)  {printf("Find_eq: vlist: %x var %s\n", vlist, var);}   
   while (vlist)
     {
     if (strcompare(var, vlist->varname) == 0)
        return(vlist);
     else
        vlist = vlist->next;
     }
 
   return(NULL);
}


  
varptr
newvar()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Obtain and initialize a new record for holding a variable.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the record allocated.
 *
 * Calls:
 *    "malloc" system call.
 *    "initvar" in this module.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "cmp_cevar" and "cmp_new_var" in this module.
 *    "new_bvar" and "new_cbvar" in "cmprhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   varptr new;

   if (freevar == NULL)
     {
      if ((new = (varrec *)malloc(sizeof(varrec))) == NULL)
        {
         opserror("allocvar: no more memory..."); puteol();
         bomb_out();
        }
     }
   else
     {
      new = freevar;
      freevar = freevar->next;
     }
     
   new->varname = NULL;
   new->test    = 0;
   new->cenum   = 0;
   new->snum    = 0;
   new->prev    = NULL;
   new->next    = NULL;

   return(new);
}



free_var(kind)
   scalar kind;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Free all the variable records on the indicated variable list and put
 *    them on the free list, freevar. Also release any storage allocated
 *    to the varname field that held the string name of the variable in the
 *    record.
 *
 * Parameters:
 *    kind - the kind of variable list.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "delvar" in this module.
 *
 * Called by:
 *    "cmp_p" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
   varptr ptr, next;

   switch (kind)
     {
      case CEVAR:   ptr = g_cevar;  break;
      case GVAR:    ptr = g_var;    break;
      case CURVAR:  ptr = g_curvar; break;
      case BVAR:    ptr = g_bvar;   break;
      case CBVAR:   ptr = g_cbvar;  break;
     }
     
   while (ptr)
     {
      next = ptr->next;
      delvar(ptr, kind);       /* Also frees storage for varname. */
      ptr = next;
     }
}



delvar(var, kind)
   varptr var;
   scalar kind;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Remove the variable from the indicated variable list, free storage
 *    allocated to its variable name field (varname), and put the record
 *    on the free list (freevar).
 *
 * Parameters:
 *    var  - the variable to be removed from an active variable list.
 *    kind - the kind of variable list.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "unlink_var" in this module.
 *    "free" system call.
 *
 * Called by:
 *    "free_var", "cmp_new_eq_var", and "cmp_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   unlink_var(var, kind);
   free(var->varname);
   var->next = freevar;
   if (freevar)  freevar->prev = var;
   freevar = var;
}



unlink_var(var, kind)
   varptr var;
   scalar kind;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Unlink the variable from the indicated variable list. If the variable
 *    is at the head of a list, fix up the global list head pointer. Also
 *    fix up the "next" and "prev" pointers in the list.
 *
 * Parameters:
 *    var  - the variable to be unlinked.
 *    kind - the kind of variable list.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "promote_var" and "delvar" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   varptr pred, succ;

   pred = var->prev;
   succ = var->next;
   
   /* See if the variable is at the head of the list and, if so, fix up
    * the global list head pointer.
    */
   switch (kind)
     {
      case CEVAR:
         if (var == g_cevar)  g_cevar = succ;
         break;
         
      case GVAR:
         if (var == g_var)  g_var = succ;
         break;
         
      case CURVAR:
         if (var == g_curvar)  g_curvar = succ;
         break;
         
      case BVAR:
         if (var == g_bvar)  g_bvar = succ;
         break;
         
      case CBVAR:
         if (var == g_cbvar)  g_cbvar = succ;
         break;
     } /* end switch */

   /* Fix up the hole in the list.
    */
   if (pred)  pred->next = succ;
   if (succ)  succ->prev = pred;
  
   /* Disassociate var from the list.
    */
   var->prev = NULL;
   var->next = NULL;
}




plist_ptr
new_prod(sym)
   symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new list header to point to all beta nodes for a production.
 *
 * Parameters:
 *    sym - symbol table entry for production name.
 *
 * Environment:
 *    The production is about to be compiled into its Rete network tree.
 *
 * Returns:
 *    A pointer to the new list header.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "cmp_production" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   plist_ptr plptr;

   if ((plptr = (plist_rec *)malloc(sizeof(plist_rec))) == NULL)
     {
      opserror("new_prod: no more memory... ");  puteol();
      bomb_out();
     }
   else
     {
      plptr->prodsym    = sym;
      plptr->blist_head = NULL;
      plptr->blist_tail = NULL;
      plptr->next       = NULL;
      return(plptr);
     }
}



add_to_blist(next_bnode)
   nodeptr next_bnode;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add the next beta node to the list of beta nodes that comprise the
 *    production currently being compiled.
 *
 * Parameters:
 *    next_bnode - the next beta node in the Rete network for the production.
 *
 * Environment:
 *    The production is being compiled into its Rete network tree.
 *
 * Returns:
 *    Nothing.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "cmp_production" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   blist_ptr bptr;

   if ((bptr = (blist_rec *)malloc(sizeof(blist_rec))) == NULL)
     {
      opserror("add_to_blist: no more memory... ");  puteol();
      bomb_out();
     }
   else
     {
      bptr->bnode = next_bnode;
      bptr->next  = NULL;

      if (g_prod->blist_head == NULL)
         g_prod->blist_head = bptr;          /* The first beta node. */
      else
         g_prod->blist_tail->next = bptr;    /* Succeeding ones go at end of list. */

      g_prod->blist_tail = bptr;
     }
}




cmp_nobeta()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Compile the first condition element in a production. A beta node is not
 *    needed here since there is nothing to join to it.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "cmp_beta" in this module.
 *
 * Called by:
 *    "cmp_ce" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   cmp_beta(0);   /* 0 parameter says don't make a beta node. */
}


cmp_not()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Compile a negated condition element by representing it in the network
 *    as a right input to a NOT_NODE type beta node. The new beta node joins
 *    the negated condition element with the left input which is the
 *    conjunction of earlier condition elements in this production.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "cmp_beta" in this module.
 *
 * Called by:
 *    "cmp_ce" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   cmp_beta(NOT_NODE);
   g_flag_negce = FALSE;   /* Reset the flag for negative ce. */
}



cmp_and()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Compile a condition element by representing it in the network as the
 *    right input to an AND_NODE type beta node. The new beta node joins
 *    the condition element with the left input which is the conjunction
 *    of earlier condition elements in this production.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "cmp_beta" in this module.
 *
 * Called by:
 *    "cmp_ce" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   cmp_beta(AND_NODE);
}




cmp_beta(kind)
   scalar kind;   /* 0, AND_NODE, or NOT_NODE */
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is the final step in the compilation of the current
 *    condition element. In this final step, we must process the variables
 *    associated with the condition element attributes and build a test list
 *    that specifies the bindings and comparisons to be made with variables of
 *    the same name that appeared both in this condition element and in the
 *    earlier condition elements of the current production. Having done
 *    this, we then build a beta node to join the network branch of the
 *    current condition element with the branch formed by the conjunction
 *    of the earlier condition elements of this production. The test list
 *    will be attached to this beta node.
 *
 *    In the earlier steps of the compilation of this condition element, the
 *    attribute variables encountered in the condition element were placed
 *    in a list headed by g_curvar. Associated with each variable is the
 *    test predicate (eg., =, >, <,...) and the binding information. From
 *    this earlier processing, any variable bindings internal to the
 *    condition element have been handled with the appropriate T_NODEs being
 *    added to the network branch under construction for the condition
 *    element. (Note that an internal binding occurs when the variable is
 *    used more than once in the condition element and at least one of those
 *    occurrences involves the predicate test for equality, TEST_EQ. For
 *    example, (ce1 ^a <x> ^b {< <x>} ^c {>= <x>}) has internal bindings for
 *    attributes b and c while (ce2 ^p {<= <x> ^q {> <x>} ^r {< <x>}) has
 *    none.) All occurrences of a variable for which no T_NODE was built
 *    are contained in the g_curvar list. We must process that list here in
 *    the following way:
 *        First we look to see if the variable on the g_curvar list has
 *        already appeared in a prior condition element in this production.
 *        The list headed by g_var contains the initial occurrences of each
 *        variable in the production. (Recall too that OPS5 requires that
 *        the initial occurrence of a variable in a production must have an
 *        associated predicate of TEST_EQ.) If the variable also appears on
 *        g_var, then we simply add a test for the variable to our test list
 *        for the condition element. If the variable is not on g_var, then
 *        this is its first occurrence in the production. Its associated
 *        predicate should be TEST_EQ (or else we have an OPS5 usage error)
 *        and we simply put the variable on the g_var list.
 *
 *
 *    For the first condition element of a production, the variables on the
 *    g_curvar list will each be the first occurrence of that variable in
 *    the production and these all should have associated predicates of
 *    TEST_EQ. Thus there will be no test list here since all variables are
 *    simply moved to the g_var list. Furthermore, we do not build a beta
 *    node for the first condition element since we have nothing to join
 *    to it.
 *
 *    Since negated condition elements cannot introduce new variables that
 *    have external bindings in a later condition element, we expect that
 *    variables on g_curvar for a NOT_NODE are either bound to an occurrence
 *    of the variable in a previous positive condition element or else are
 *    simply internally bound within the negated condition element. Thus no
 *    variables on g_curvar will be promoted to g_var in this case.
 *
 * Parameters:
 *    kind - the kind of beta node we are to construct. A 0 here means to
 *           not construct any beta node at all for the case when we are
 *           compiling the first condition element in a production.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "find_equiv_varname", add_test", delvar", "promote_var", and
 *    "build_beta" in this module.
 *    "opserror", "puttab", "putstring", and "puteol" in "ops5.c". 
 *
 * Called by:
 *    "cmp_nobeta", "cmp_and", and "cmp_not" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   varptr    vptr, old;
   tests_ptr tlist;

   /* Process the variables on g_curvar list and build the test list
    * for the beta node.
    */
   tlist = NULL;
   while (g_curvar)  /* Process g_curvar list till it's exhausted. */
     {
      vptr  = g_curvar;   /* Get variable at head of list. */
      
      /* First look to see if the variable has appeared in an earlier
       * condition element and is thus on the g_var list.
       */
      old = find_equiv_varname(vptr->varname, g_var);
      
      if (old)
        {
         tlist = add_test(tlist, vptr, old);
         delvar(vptr, CURVAR);   /* Delete var from g_curvar list. */
        }
      else
        {
         /* Variable does not appear on g_var so this is its first
          * occurrence in the production. Its predicate must be TEST_EQ
          * or else we have an OPS5 usage error.
          */
         if (vptr->test == TEST_EQ)
           {
            /* Predicate is OK so promote it to g_var list unless
             * we have a NOT_NODE.
             */
            if (kind != NOT_NODE)
               promote_var(vptr);  /* Put it on g_var list. */
            else
              {
               /* Variable must have been just internal to negative ce
                * so just dump it.
                */
               delvar(vptr, CURVAR);
              }
           }
         else
           {
            /* Variable's first occurrence was not TEST_EQ.
             */
            delvar(vptr, CURVAR);
            opserror("illegal predicate for first occurrence");  puttab();
            putstring(vptr->varname);  puteol();
            /* Don't bomb_out, just continue. */
           }
        }
     } /* endwhile */
     
   if (kind != 0)  build_beta(kind, tlist);

   g_lastbranch = g_last;

   if (debugnet)
      printf("cmp_beta: lastnode and lastbranch : %x\n", g_last);

   /* free_var(CURVAR) not needed since we have emptied the g_curvar
    * list above.
    */
  
} /* end cmp_beta */



tests_ptr
add_test(list, new, old)
  tests_ptr list;
  varptr    new, old;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine builds the test list associated with a beta node for
 *    external variable bindings. It builds a new variable binding test
 *    that relates the new occurrence of the variable to its old (i.e.,
 *    first) occurrence in a previous condition element. Before adding
 *    the new test, we look at the test list to see if the new one already
 *    has a duplicate on the list.
 *
 * Parameters:
 *    list - the current test list
 *    new  - the new occurrence of the variable
 *    old  - the first occurrence of the variable (from g_var)
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new test list (it may be returned unchanged).
 *
 * Calls:
 *    "newtest" in this module.
 *
 * Called by:
 *    "cmp_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   tests_ptr tptr;
   
   /* First look for an equivalent test already on the list. 
    */
   tptr = list;
   while (tptr)
     {
      /* Remember that BETA was previously added to the "test" field of
       * the test records already on the list.
       */
      if ((new->test + BETA == tptr->test) &&
          (old->cenum == tptr->l_wme) &&
          (old->snum == tptr->l_snum) &&
          (new->snum == tptr->r_snum))
        {
         return(list);   /* Found an equivalent test on the list */
        }                /* so just return the list unchanged.   */
        
      tptr = tptr->next;
     }
     
   /* We didn't find an equivalent test so add a new variable binding
    * test to the list.
    */
   g_featurecount++; 
   tptr = newtest();           /* Get a new tests_rec. */
   tptr->test   = new->test + BETA;
   tptr->l_wme  = old->cenum;  /* The wme in the token from the left mem  */
                               /* will be the one corresponding to the ce */
                               /* in which the variable was first bound.  */
                               
   tptr->l_snum = old->snum;   /* The subelement number in that wme to   */
                               /* which the attribute variable is bound. */
                               
   tptr->r_snum = new->snum;   /* The subelement number of the new binding */
                               /* for the variable in the wme from the     */
                               /* right mem. Tokens in the right mem are   */
                               /* just single wme's that match the ce      */
                               /* joined from the right by the beta node.  */
                               
   tptr->next   = list;        /* Put the test at the head of the list. */
   
   return(tptr);               /* Return the new list. */  

} /* end add_test */



tests_ptr
newtest()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new tests_rec record. We don't bother to init the record
 *    since the caller will fill it in right away.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to a new tests-rec.
 *
 * Calls:
 *    "malloc" system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "add_test" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   tests_ptr new;
  
   if (free_test)    /* First look for a structure on the free list. */
     {
      new = free_test;
      free_test = free_test->next;
      return(new);
     }
     
   if ((new = (tests_rec *)malloc(sizeof(tests_rec))) == NULL)
     {
      opserror("newtest: no more memory..."); puteol();
      bomb_out();
     }
   else
      return(new);
}



promote_var(var)
   varptr var;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Move a variable from the g_curvar list to the g_var list. The first
 *    occurrence of a given attribute variable in a production is always
 *    placed on the g_var list. The associated predicate for the variable
 *    must be TEST_EQ or else we have a violation of OPS5 rules.
 *
 * Parameters:
 *    var - the variable to be promoted to the g_var list.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "unlink_var" in this module.
 *
 * Called by:
 *    "cmp_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   unlink_var(var, CURVAR);  /* Unlink var from g_curvar list. */
   var->next = g_var;        /* Link var to g_var list. */
   if (g_var)  g_var->prev = var;
   g_var = var;
}



build_beta(kind, tlist)
   scalar    kind;
   tests_ptr tlist;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Construct a new beta node and link it into the network to join the
 *    condition element just compiled with the conjunction of previously
 *    compiled condition elements in the current production. First get a
 *    new beta node, attach the test list for external variable bindings,
 *    and set the beta level. Then get an RI_NODE and attach it to the
 *    end of the network branch built for the ce just compiled. This marks
 *    the current ce as the right input to the beta node with the left
 *    input coming from the join of the earlier ce's in the production.
 *    Finally link the new beta node into the network to join the two
 *    network branches.
 *    Note: The network linking routines will look for equivalent nodes
 *          already in the network that can be used before actually
 *          adding any new nodes.
 *
 * Parameters:
 *    kind  - the type (AND or NOT) beta node we are building.
 *    tlist - the list of external variable bindings.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "newnode", "link_new_1input_node", and "link_new_beta_node" in
 *    this module.
 *
 * Called by:
 *    "cmp_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr new, ri;

   new = newnode(kind, 0, 0, 0, 0);
   new->nodeinfo.betanode->testlist = tlist;
   new->nodeinfo.betanode->betalev  = g_betalev++;

   ri = newnode(RI_NODE,0,0,0,0);
   link_new_1input_node(ri);

   link_new_beta_node(new);
}

 


link_new_1input_node(new)
   nodeptr new;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Used only to link the 1-input nodes - T_NODE, ANY_NODE, RI_NODE, and
 *    P_NODE - into the network. If it's a T_NODE or ANY_NODE, then the
 *    count of features must be incremented for this production. The node
 *    will be linked into the network as the son of the last node linked
 *    into the branch for the condition element currently being compiled.
 *    This last node in the current branch is pointed to by g_last. If this
 *    last node already has a son (i.e., due to structural similarity of a
 *    previously compiled condition element), then we first look at that son
 *    and all its left brothers to see if any node among them is equivalent
 *    to the new node we want to insert. If we find an equivalent node, then
 *    we do not attach the new node and, instead, simply share the equivalent
 *    node for the condition element branch we are building and set g_last to
 *    point to this node. If we do not find an equivalent node, then we
 *    attach the new node to the branch by making it the first son of the
 *    last node and linking any previous sons of the last node as left
 *    brothers of the new node. We then set g_last to point to this new node.
 *    In effect, if a node has more than one son, then we represent them in
 *    the network by making the parent's son-link point to the youngest son
 *    and having all the other sons linked by their left-brother links. For
 *    the 1-input nodes, the right-brother links are never used.
 *
 * Parameters:
 *    new - the new node to be added.
 *
 * Environment:
 *    G_last points to the parent of the node we are adding.
 *
 * Calls:
 *    "link_left" in this module.
 *
 * Called by:
 *    "cmp_p", "cmp_any", "cmp_constant", "cmp_fixnum", "cmp_flonum",
 *    "cmp_new_eq_var", "cmp_old_eq_var", and "build_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   if ((new->nodename != P_NODE) && (new->nodename != RI_NODE)) 
      g_featurecount++;
   g_last = link_left(g_last, new);
   if (debugnet)  printf("link_new_1input_node: lastnode : %x\n", g_last);
}



nodeptr
link_left(pred, succ)
   nodeptr pred, succ;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Link the new 1-input node succ as the son of the existing node pred.
 *    If pred already has 1 or more sons, then it will have its son-link
 *    pointing to the youngest son and all the other sons will be linked by
 *    their left-brother pointers originating with the youngest son. So if
 *    pred already has sons, we will first try to find an equivalent son in
 *    the list of brothers. If one is found, then we do not need the new succ
 *    node and can simply share the equivalent node; otherwise, we link the
 *    new son at the head of the left-brother list and set pred's son-link
 *    to point to it.
 *
 * Parameters:
 *    pred - the existing predecessor node.
 *    succ - the new successor node (if needed). It's a 1-input node.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new node, succ, if added to the network. If an
 *    equivalent node already in the network is found, then return a
 *    pointer to it instead.
 *
 * Calls:
 *    "find_equiv_1input_node", "attach_left", and "free_node" in this module.
 *    "printnet" in "printops.c".
 *
 * Called by:
 *    "link_new_1input_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr son_list;   /* Will point to the list of sons of pred. */
   nodeptr twin;       /* Will point to node equivalent to succ (if found) */
                       /* in the son_list.                                 */
 
   son_list = pred->son;   /* This may be Nil. */
   
   if ((twin = find_equiv_1input_node(succ, son_list)) == NULL)
     {
      /* Didn't find an equivalent node, so make succ a son of pred.
       */
      attach_left(pred, succ);
      if (debugnet)
        {
         printf("Link_left: show pred and succ:\n");
         printnet(pred);  printnet(succ);
        }
      return(succ);
     }
   else
     {
      /* Found a node equivalent to succ that is alreay a son of pred.
       */
      if (debugnet)
        {
         printf("Link_left: show pred & twin:");
         printnet(pred);  printnet(twin);
        } 
      free_node(succ);  /* Free the new node data structure we didn't need. */
      return(twin);
     }

}



attach_left(pred, succ)
   nodeptr pred, succ;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Make succ a son of pred. The parent may already have several sons - if
 *    so, the parent's son-link points to the youngest son and that son's
 *    left-brother link points to the next youngest son, etc. Succ will now
 *    become the youngest son.
 *
 * Parameters:
 *    pred - the existing predecessor node.
 *    succ - the new successor node.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "link_left" and "link_both" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   /* If there are other brothers, then link succ in as the youngest.
    */
   if (pred->son)  succ->brothl = pred->son;
  
   pred->son = succ;   /* Make pred point to the youngest son. */
}


  
attach_right(pred, succ)
   nodeptr pred, succ;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Used only to make the connection for the right input of the new beta
 *    node succ. Pred will always be an RI_NODE and we will make succ its son.
 *    If pred already has sons, they are linked by right-brother links with
 *    the youngest son at the head of the right-brother chain and pointed to
 *    by pred. Succ will now be linked in as the youngest son. All the sons
 *    on the right-brother chain are beta nodes which take the network branch
 *    above the RI_NODE, pred, as their right inputs.
 *
 * Parameters:
 *    pred - the predecessor node (better be an RI_NODE).
 *    succ - the successor node (should always be a beta node).
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "link_both" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   if (pred->nodename != RI_NODE)
     {
      opserror("attach_right:  internal error...");  puteol();
      bomb_out();
     }
   else
     {
      /* Make succ the youngest son. */
      if (pred->son)  succ->brothr = pred->son;
      
      pred->son = succ;   /* Make pred point to the youngest son. */
     }
}



nodeptr
find_equiv_1input_node (new_son, son_list)
   nodeptr new_son, son_list;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Before adding a new 1-input node to the network, we first check to see
 *    if, due to structural similarity, an equivalent node does not already
 *    exist. For an equivalent node to exist, it must be both identical in
 *    function and already located in the network as a son of the parent node
 *    to which it is being attached. Son_list points to the parent node's
 *    list (it may be empty) of existing sons. Since we are dealing with
 *    1-input nodes (T, ANY, P, and RI), the list is a chain of left-brothers.
 *    (However, it is possible that beta nodes may indeed be linked into this
 *    left-brother chain.)
 *
 * Parameters:
 *    new_son  - The new son node we need to add or find. It is a 1-input
 *               node.
 *    son_list - The list of brothers where an equivalent to new_son might
 *               already exist or where new_son will be added.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    NULL or an equivalent node to new_son.
 *
 * Calls:
 *    "equiv_1input" in this module.
 *
 * Called by:
 *    "link_left" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   while (son_list)
     {
      if ( equiv_1input(new_son, son_list) )
         return(son_list);    /* Return the equivalent son. */
      else
         son_list = son_list->brothl;   /* Look at the next son. */
     }
     
   return(NULL);   /* We didn't find an equivalent node. */
}



boolean
equiv_1input(node1, node2)
   nodeptr node1, node2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    See if a 1-input node is equivalent to a node from the left-brother
 *    chain to which the 1-input node will belong. The 1-input node may be
 *    a T_NODE, ANY_NODE, P_NODE, or RI_NODE.
 *    Note: Remember that for a T_NODE, the nodename holds the predicate
 *          plus the type (ATM, NUM, S) of T_NODE rather than the value
 *          T_NODE.
 *
 * Parameters:
 *    node1 - the first node is a 1-input node.
 *    node2 - the second node is a node from a left-brother chain.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "equiv_atmlists" in this module.
 *
 * Called by:
 *    "find_equiv_1input_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   t_nodeptr   tn1, tn2;
   any_nodeptr an1, an2;
   p_nodeptr   pn1, pn2;

   if (debugnet)
     {
      printf("Equiv_1input: node1 and node2:\n");
      printnet(node1);  printnet(node2);
     }
     
   /* First see if they are the same kind of nodes.
    */
   if (node1->nodename != node2->nodename)  return(FALSE);

   /* At this point, we also know that both are 1-input nodes.
    */

   switch (node1->nodename)
     {
      case TEQA: case TNEA: case TXXA:
      case TEQN: case TNEN: case TGTN: case TGEN:
      case TLTN: case TLEN: case TXXN:
      case TEQS: case TNES: case TGTS: case TGES:
      case TLTS: case TLES: case TXXS:
         tn1 = node1->nodeinfo.tnode;
         tn2 = node2->nodeinfo.tnode;
         if (tn1->pos != tn2->pos)
            return(FALSE);
         else if (tn1->utype != tn2->utype)
            return(FALSE);
         else
           {
            switch (tn1->utype)
              {
               case INTNUM:
                  if (tn1->value.num != tn2->value.num)  return(FALSE);
                  break;
                
               case STRING:
                  /* Only need to compare pointers to strings stored in
                   * symbol table since symbol table does not store
                   * duplicates.
                   */
                  if (tn1->value.str != tn2->value.str)  return(FALSE);
                  break;
                   
               case SUBNUM:
                  if (tn1->value.snum != tn2->value.snum)  return(FALSE);
                  break;
              } /* end switch */
           } /* endif */
         return(TRUE);
         
      case ANY_NODE:
         an1 = node1->nodeinfo.anynode;
         an2 = node2->nodeinfo.anynode;
         if (an1->snum != an2->snum)
            return(FALSE);
         else
            return(equiv_atmlists(an1->anyatm_list, an2->anyatm_list));
         
      case RI_NODE:
         return(TRUE);  /* Has no nodeinfo record; only nodename had to */
                        /* be tested.                                   */
      case P_NODE:
         pn1 = node1->nodeinfo.pnode;
         pn2 = node2->nodeinfo.pnode;
         if (pn1->pname != pn2->pname)   /* Only need to compare pointers */
            return(FALSE);               /* into the symbol table.        */
         else if (pn1->testcnt != pn2->testcnt)
            return(FALSE);
         else
            return(TRUE);
         
     } /* end switch */
}



boolean
equiv_atmlists(list1, list2)
   anyatm_ptr list1, list2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Compare 2 lists of constants from 2 ANY_NODEs. The 2 lists are equal
 *    if they have the same number of elements and if every constant in
 *    one list is contained in the other. The lists may be ordered
 *    differently (eg. << A B C >> vs << C B A >>) since no lexicographic
 *    ordering is imposed earlier in the compilation of the disjunctions.
 *    We assume here that a given list contains no duplicates (guaranteed
 *    by "add_anystr" and "add_anynum"). Also, for constants that are strings,
 *    we do not have to compare actual strings - we only need to compare the
 *    addresses of their symbol table entries since the symbol table does not
 *    store duplicates (re: "sym_addr" in "literal.c").  
 *
 * Parameters:
 *    list1, list2 - the lists of constant atoms associated with 2 ANY_NODEs.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    TRUE if the 2 lists are equivalent, otherwise FALSE.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "equiv_1input" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   anyatm_ptr ptr1, ptr2;
   
   /* First check the length of the 2 lists.
    */
   ptr1 = list1;  ptr2 = list2;
   while (ptr1)
     {
      if (ptr2 == NULL)  return(FALSE);
      ptr1 = ptr1->next;  ptr2 = ptr2->next;
     }
   if (ptr2 != NULL)  return(FALSE);
   
   /* Now check if every element in list1 is in list2.
    */
   ptr1 = list1;
   while (ptr1)
     {
      ptr2 = list2;
      while (ptr2)
        {
         if (ptr1->is_int && ptr2->is_int)
           {
            if (ptr1->aval.any_int == ptr2->aval.any_int)
               break;   /* Found it. */
           }

         if ((!ptr1->is_int) && (!ptr2->is_int))
           {
            if (ptr1->aval.any_ptr == ptr2->aval.any_ptr)
               break;   /* Found it. */
           }

         ptr2 = ptr2->next;
        }
      if (ptr2 == NULL)  return(FALSE);   /* Didn't find atom in list2. */
      ptr1 = ptr1->next;
     }
   
   return(TRUE);   /* The 2 lists are equivalent. */
}




link_new_beta_node(new)
   nodeptr new;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    A beta node is needed to join together the current condition element
 *    branch as the right input with the left input branch which represents
 *    the conjunction of all the prior condition elements in the current
 *    production. The right input branch has already been terminated with an
 *    RI_NODE as required for the right parent of a beta node. This right
 *    parent is also pointed to by "g_last". The left parent is pointed to
 *    by "g_lastbranch" and will be one of the following:
 *        - a beta node representing the conjunction of all the previous
 *          ce's in this production.
 *        - a T_NODE or ANY_NODE representing the last term in the initial
 *          (and non-empty) ce of this production.
 *        - the ROOT_NODE if the first non-empty ce of this production has
 *          no 1-input nodes in the network. An example here is a
 *          production with first ce as (^a <x> ^b <y>).
 *
 *    The new beta node may not actually be needed if we can find an
 *    equivalent beta node already in the network. We let the "link_both"
 *    routine handle that determination.
 *
 *    After completing the linking in the network, "g_last" will point to
 *    this beta node as the last node 'added' and "g_lastbranch" will point
 *    to it as the last branch joined in the compilation of this production.
 *
 * Parameters:
 *    new - the new beta node to be linked into the network.
 *
 * Environment:
 *    G_lastbranch points to the left input and g_last points to the right
 *    input of the beta node.
 *
 * Calls:
 *    "link_both" in this module.
 *
 * Called by:
 *    "build_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   g_last = link_both(g_lastbranch, g_last, new);
   g_lastbranch = g_last;
   add_to_blist(g_last);
   if (debugnet)
      printf("link_new_beta_node: lastnode and lastbranch : %x\n", g_last);
}



nodeptr
link_both(l_parent, r_parent, new_beta)
   nodeptr l_parent, r_parent, new_beta;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    We must make a network connection with l_parent and r_parent as the
 *    left and right inputs to a new beta node. Before actually adding and
 *    linking the new beta node supplied,we first look in the network for
 *    an existing beta node that is equivalent both functionally and
 *    positionally to the new one we wish to add. This requires looking at
 *    all the left-brother chained sons of the left parent and then at all
 *    the right-brother chained sons of the right parent until the same
 *    beta node with all the required characteristics is found on both chains.
 *    Effectively, this means that such an equivalent node will have the same
 *    left and right inputs required by our new beta node and will also be
 *    identical in its variable bindings and tests between its left and right
 *    inputs. If such an equivalent node is found, then we can simply
 *    deallocate the new beta node and return a pointer to the equivalent
 *    node. Otherwise, we link the new beta node into the network. The
 *    possibility of an equivalent node is due to the sharing of network
 *    branches and nodes resulting from structural similarity among the
 *    condition elements of all the productions in a system.
 *
 * Parameters:
 *    l_parent - the left parent node.
 *    r_parent - the right parent node (an RI_NODE).
 *    new_beta - a new beta node to join the 2 above nodes.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the beta node, new_beta, or else its equivalent, if found.
 *
 * Calls:
 *    "find_equiv_beta_node", "free_node", "attach_left", and "attach_right"
 *    in this module.
 *    "printnet" in "printops.c".
 *
 * Called by:
 *    "link_new_beta_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr beta_twin;    /* The equivalent beta node (if found). */   

   if (debugnet)
     {
      printf("link_both:  left right succ\n");
      printnet(l_parent); printnet(r_parent); printnet(new_beta);
      printf("end link_both parameter\n");
     }
    
   beta_twin = find_equiv_beta_node(l_parent->son, r_parent->son, new_beta);
   if (beta_twin)
     {
      /* Found an equivalent node.
       */
      free_node(new_beta);   /* Deallocate the new beta node. */
      return(beta_twin);     /* Just return a ptr to the equiv node. */
     }
   else
     {
      attach_left(l_parent, new_beta);   /* Connect left input to beta. */
      attach_right(r_parent, new_beta);  /* Ditto for right input. */
      if(debugnet)
        {
         printnet(l_parent); printnet(r_parent); printnet(new_beta);
        }
      return(new_beta);   /* Return ptr to new beta node just added. */
     }
}



nodeptr
find_equiv_beta_node(l_chain, r_chain, new_beta)
   nodeptr l_chain, r_chain, new_beta;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Search for a beta node that exists in both chains and is equivalent
 *    to new_beta. If not found, return NULL.
 *
 * Parameters:
 *    l_chain  - a left-brother chain to search.
 *    r_chain  - a right-brother chain to search.
 *    new_beta - the beta node we are looking for.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the equivalent beta node (if found) or else NULL.
 *
 * Calls:
 *    "equiv_beta" in this module.
 *
 * Called by:
 *    "link_both" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr l, r;
   
   /* If either chain is empty, then no equivalent node is possible.
    */
   if ((l_chain == NULL) || (r_chain == NULL))  return(NULL);
   
   l = l_chain;
   while (l)    /* Look at each brother in the left chain. */
     {
      r = r_chain;
      while (r)    /* Compare against each brother in the right chain. */
        {
         if (l == r)
           {
            /* Found same node on both chains. If it's the one we want,
             * return with it.
             * Note: Since the right chain only holds beta nodes, equiv_beta
             *       knows that its second parameter is also a beta node.
             */
            if (equiv_beta(new_beta, l))  return(l);
           }
         r = r->brothr;
        }
      l = l->brothl;
     }
     
   return(NULL);
}



boolean
equiv_beta(bnode1, bnode2)
   nodeptr bnode1, bnode2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine checks if two beta nodes are equivalent. Both
 *    are assumed to be at the same beta level in the network so here
 *    we must check that they are both the same kind of beta node and
 *    that they have equivalent test lists for external attribute
 *    variable bindings.
 *
 * Parameters:
 *    bnode1 - a beta node.
 *    bnode2 - the  beta node at the same beta level in the network that we
 *             want to compare to bnode1.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    TRUE if the nodes are equivalent and FALSE otherwise.
 *
 * Calls:
 *    "equiv_testlists" in this module.
 *    "printnet" in "printops.c".
 *
 * Called by:
 *    "find_equiv_beta_node" in this module.
 *
 *-------------------------------------------------------------------------*/
{

   if (debugnet)
     {
      printf("equiv_beta: nodename1 is: %d nodename2 is: %d\n",
             bnode1->nodename, bnode2->nodename);
      printnet(bnode1);
      printnet(bnode2);
      printf("end equiv_beta parameter\n");
     }
     
   /* First check that both are the same kind of beta node.
    */
   if (bnode1->nodename != bnode2->nodename)  return(FALSE);
   
   /* Just return the result of checking the testlists.
    */
   return(equiv_testlists(bnode1->nodeinfo.betanode->testlist,
                          bnode2->nodeinfo.betanode->testlist));
}



boolean
equiv_testlists(list1, list2)
   tests_ptr list1, list2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Compare 2 test lists. They are the same if they have the same number
 *    of elements (i.e., test_rec's) and each element in one list appears
 *    in the other. We are guaranteed that there are no duplicate tests
 *    in either list. This is done earlier in the compilation of each list
 *    (re: "add_test" in this module).
 *
 * Parameters:
 *    list1, list2 - the 2 test lists to compare.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    TRUE if the test lists are equivalent and FALSE otherwise.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "equiv_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   tests_ptr t1, t2;
   
   /* First check the length of the 2 lists.
    */
   t1 = list1;  t2 = list2;
   while (t1)
     {
      if (t2 == NULL)  return(FALSE);
      t1 = t1->next;  t2 = t2->next;
     }
   if (t2 != NULL)  return(FALSE);
   
   /* Now check that every element in list1 is in list2.
    */
   t1 = list1;
   while (t1)
     {
      t2 =list2;
      while (t2)
        {
         if ((t1->test == t2->test) && (t1->l_wme == t2->l_wme) &&
             (t1->l_snum == t2->l_snum) && (t1->r_snum == t2->r_snum))
            break;   /* Found it. */
         else
            t2 = t2->next;
        }
      if (t2 == NULL)  return(FALSE);   /* Didn't find one in list2. */
      t1 = t1->next;
     }
     
   return(TRUE);   /* The 2 lists are equivalent. */
}

print_lhs_stats()
/* This routine prints out the lhs stats at the end of compilation
 */
{

printf("\t----------- END OF CODE GENERATION -----------\n\n");
printf("Number of productions     = %d\n", g_pcount);
printf("Number of two-input nodes = %d\n", nodeid_cnt);
printf("\n");
fflush(stdout);
}

